#include <core/math.h>

#include <Export/Skeleton.h>
#include <Misc/GenCrc.h>
#include <Misc/HelperFuncs.h>
#include "PartialAnimEditor.h"		// For RemoveNonPartialAnimBones

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

void DumpHierarchy(INode* node)
{ FUNC_ENTER("DumpHierarchy"); 
	char buf[2048];
	sprintf(buf, "DumpHierarchy: %s (", (char*)node->GetName());

	// Concatenate parents into string
	INode* parent = node->GetParentNode();

	while(parent)
	{
		strcat(buf, (char*)parent->GetName());
		strcat(buf, ",");

		parent = parent->GetParentNode();
	}
	
	strcat(buf, ")\n");
	OutputDebugString(buf);

	int nKids = node->NumberOfChildren();
	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		DumpHierarchy(child);
	}
}

void CSkeletonData::AddIgnoreException(INode* node)
{ FUNC_ENTER("CSkeletonData::AddIgnoreException"); 
	if (node)
		mp_ignoreExceptions[m_numExceptions++] = node;
}

void CSkeletonData::print_name_table()
{ FUNC_ENTER("CSkeletonData::print_name_table"); 
    DebugPrint( "Name table:\n" );

    for ( int i = 0; i < m_numNodes; i++ )
    {
        MaxAssert( mp_nodes[i] );
        DebugPrint( "Name %d %s\n", i, mp_nodes[i]->GetName() );
    }
};

INode* CSkeletonData::GetRootBone( INode* pNode )
{ FUNC_ENTER("CSkeletonData::GetRootBone"); 
	if (pRootNode)
		return pRootNode;

	INode* root = GetTrueRootBone(pNode);
	CStr name = root->GetName();

	return GetNaturalRootBone( root, pNode );
}

// Determines if a child node is a child of the given node (pNode)
bool CSkeletonData::HasChild(INode* pNode, INode* pChildNode)
{ FUNC_ENTER("CSkeletonData::HasChild"); 
	// base case
	if (pNode == pChildNode)
		return true;

	int nKids = pNode->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = pNode->GetChildNode(i);
			
		if (HasChild(child, pChildNode))
			return true;
	}

	return false;
}

// This computes the depth of the tree from the given node
int CSkeletonData::GetNodeDepth(INode* pNode)
{ FUNC_ENTER("CSkeletonData::GetNodeDepth"); 
	int nKids = pNode->NumberOfChildren();

	// Base case
	if (nKids == 0)
		return 1;

	int childDepth;
	int maxDepth    = 0;

	for(int i = 0; i < nKids; i++)
	{
		INode* child = pNode->GetChildNode(i);
		childDepth = GetNodeDepth(child);

		if (childDepth > maxDepth)
			maxDepth = childDepth;
	}

	return maxDepth + 1;
}

INode* CSkeletonData::GetNaturalRootBone( INode* pNode, INode* pHintNode )
{ FUNC_ENTER("CSkeletonData::GetNaturalRootBone"); 
	// NOTE: This presents a problem if two or more siblings have valid names
	//       in which case the child with the largest tree depth should be used
	//       (In all present examples this doesn't apply since the nodes at this level
	//        just have one child, so not adding overhead for this now.  May need to change through)
	//
	// History:
	// 4-15-03: Updated to account for tree depth and hint node   aml

	CStr name = pNode->GetName();
	name.toLower();

	// Control root is auto accepted
	if (strstr(name, "control") && strstr(name, "root"))
		return pNode;

	////////////////////////////////////
	if (!strstr(name, "dummy") &&
		!strstr(name, "control"))
		return pNode;
	////////////////////////////////////////

	// Find the first bone from the root that doesn't encounter any of our name restrictions
	// and has the greatest depth in the tree or contains our hint node as a child
	int    nKids           = pNode->NumberOfChildren();
	int    curDepth;
	int    maxDepth        = 0;
	int    nodeWithHintCnt = 0;
	INode* maxDepthNode    = NULL;
	INode* nodeWithHint    = NULL;

	for(int i = 0; i < nKids; i++)
	{
		INode* child      = pNode->GetChildNode(i);
		INode* nodeResult = GetNaturalRootBone(child, pHintNode);

		if (nodeResult)
		{
			if (pHintNode)
			{
				if (HasChild(nodeResult, pHintNode))
				{
					nodeWithHint = nodeResult;
					break;
				}
			}

			curDepth = GetNodeDepth(nodeResult);

			if (curDepth > maxDepth)
			{
				maxDepth     = curDepth;
				maxDepthNode = nodeResult;
			}
		}
	}

	if (nodeWithHint)
	{
		CStr nrName = nodeWithHint->GetName();
		return nodeWithHint;
	}

	if (maxDepthNode)
	{
		CStr nrName = maxDepthNode->GetName();
		return maxDepthNode;
	}

	// Nodes containing dummy are not accepted
	if (strstr(name, "dummy"))
		return NULL;

	// Nodes containing control are not accepted (unless control_root)
	if (strstr(name, "control"))
		return NULL;

	return pNode;
}

INode* CSkeletonData::GetTrueRootBone( INode* pNode )
{ FUNC_ENTER("CSkeletonData::GetTrueRootBone"); 
    MaxAssert( pNode );
    MaxAssert( pNode->GetParentNode() );

	// If we're forcing a particular node as root it should be considered
	// the root for all calls
	if (pRootNode)
		return pRootNode;

    if ( pNode->GetParentNode()->GetParentNode() == NULL )
    {
        // the root object's parent should be "Scene Root".
        // The Scene Root's parent should be NULL.
        return pNode;
    }
    else
    {
        return GetTrueRootBone( pNode->GetParentNode() );
    }
}

void CSkeletonData::recursive_add_node( INode* pNode, bool bParentIgnored )
{ FUNC_ENTER("CSkeletonData::recursive_add_node"); 
    MaxAssert( pNode );

	TSTR strNodeNameLC = pNode->GetName();
	strNodeNameLC.toLower();

	if (strstr(strNodeNameLC, "brow") ||
		strstr(strNodeNameLC, "bone_head"))
	{
		int zzz;
		zzz = 0;
	}

	bool isRoot;

	INode* nodeParent = pNode->GetParentNode();

	if (nodeParent && pNode != pRootNode)
	{
		if (nodeParent == GetCOREInterface()->GetRootNode())
			isRoot = true;
		else
			isRoot = false;
	}
	else
		isRoot = true;

	// Bones should only be included if they start with "bone" or "cloth"
	// (We must also always include the root as the current format relies on it)
//	if (strstr(strNodeNameLC, "bone") == (char*)strNodeNameLC ||
//		strstr(strNodeNameLC, "cloth") == (char*)strNodeNameLC ||
//		isRoot)

	// Reject names that start out with the term "dummy"
	// Reject any cameras attached to the skeleton
	if ((strstr(strNodeNameLC, "dummy") != (char*)strNodeNameLC || isRoot) && 
		 pNode->EvalWorldState(0).obj->SuperClassID() != CAMERA_CLASS_ID)
	{
		// ignore nodes with physique meshes, and leaf nodes, and any nodes that have "IK_" in them...
		if ( !FindPhysiqueModifier( pNode ) && !strstr( pNode->GetName(), "IK_" ) )
	    {
			if (!strstr(strNodeNameLC, "control_") ||
				isRoot)
			{
				if ( (bIgnoreLeafs  && pNode->NumberOfChildren() > 0) || !bIgnoreLeafs )
				{
					MaxAssert( m_numNodes < vMAX_NODES );

					if (bParentIgnored)
					{
						// This node can only be added if it exists within the exception list
						for(int i = 0; i < m_numExceptions; i++)
						{
							if (mp_ignoreExceptions[i] == pNode)
							{
								mp_nodes[m_numNodes++] = pNode;
								break;
							}
						}

						char strBuf[256];
						sprintf(strBuf, "Threw out bone '%s'\n", (char*)pNode->GetName());
						OutputDebugString(strBuf);
					}
					else
						mp_nodes[m_numNodes++] = pNode;
				}
			}
		}
	}

	int nKids = pNode->NumberOfChildren();

    for ( int i = 0; i < nKids; i++ )
    {
		INode* child = pNode->GetChildNode(i);

		if (child != ignoreBranch)
			recursive_add_node( child, bParentIgnored );
		else
			recursive_add_node( child, true );
    }
}

void CSkeletonData::BuildSkeleton(INode* pNode)
{ FUNC_ENTER("CSkeletonData::BuildSkeleton"); 
	m_numNodes = 0;
	pRootNode  = NULL;

	DebugPrint( "Starting with node %s\n", pNode->GetName() );

	// should be the "Dummy_Scale" node for skaters
	if (bTreatAsRoot)
		pRootNode = pNode;
	else
		pRootNode = GetRootBone( pNode );

	DebugPrint( "Root node is %s\n", pRootNode->GetName() );

#ifdef DUMP_HIERARCHY
	DumpHierarchy(pRootNode);
#endif

	// given any node in a particular hierarchy,
	// this will walk up the tree until it finds the root
	// ("Dummy_Scale" for skaters), then build a table
	// of children from there.  It will ignore
	// any Physiqued children

	if (bForceOneNode)
	{
		m_numNodes  = 1;
		mp_nodes[0] = pNode;
	}
	else
		recursive_add_node(pRootNode);

	//print_name_table();

	print_debug_file();
}

CSkeletonData::CSkeletonData( INode* pNode, INode* ignoreBranch, bool bIgnoreLeafs, bool bForceOneNode, bool bTreatAsRoot)
{ FUNC_ENTER("CSkeletonData::CSkeletonData"); 
	this->bIgnoreLeafs  = bIgnoreLeafs;
	this->bForceOneNode = bForceOneNode;
	this->bTreatAsRoot  = bTreatAsRoot;
	this->ignoreBranch  = ignoreBranch;

	// Assign mask (by default all bones exist unless set otherwise)
	for(int i = 0; i < vMAX_NODES / 32; i++)
		m_mask[i] = 0xFFFFFFFF;

	m_numExceptions = 0;

	if (pNode)
		BuildSkeleton(pNode);
}

CSkeletonData::CSkeletonData( INode* pNode, bool bIgnoreLeafs, bool bForceOneNode, bool bTreatAsRoot)
{ FUNC_ENTER("CSkeletonData::CSkeletonData"); 
	this->bIgnoreLeafs  = bIgnoreLeafs;
	this->bForceOneNode = bForceOneNode;
	this->bTreatAsRoot  = bTreatAsRoot;
	this->ignoreBranch  = NULL;

	// Assign mask (by default all bones exist unless set otherwise)
	for(int i = 0; i < vMAX_NODES / 32; i++)
		m_mask[i] = 0xFFFFFFFF;

	m_numExceptions = 0;

	if (pNode)
		BuildSkeleton(pNode);
}

INode* CSkeletonData::GetBone( int index )
{ FUNC_ENTER("CSkeletonData::GetBone"); 
	MaxAssert( index >= 0 && index < m_numNodes );

	return mp_nodes[index];
}

int CSkeletonData::GetCount()
{ FUNC_ENTER("CSkeletonData::GetCount"); 
	return m_numNodes;
}

int CSkeletonData::GetBoneIndex( INode* pNode )
{ FUNC_ENTER("CSkeletonData::GetBoneIndex"); 
    for ( int i = 0; i < m_numNodes; i++ )
    {
        if ( pNode == mp_nodes[i] )
        {
            return i;
        }
    }
                   
    // node not found
    MaxAssert( 0 );
    return -1;
}

int CSkeletonData::GetBoneIndexNoAssert( INode* pNode )
{ FUNC_ENTER("CSkeletonData::GetBoneIndex"); 
    for ( int i = 0; i < m_numNodes; i++ )
    {
        if ( pNode == mp_nodes[i] )
        {
            return i;
        }
    }
                   
    // node not found
    return -1;
}

unsigned int CSkeletonData::GetBoneName( int i )
{ FUNC_ENTER("CSkeletonData::GetBoneName"); 
	MaxAssert( i >= 0 && i < GetCount() );

	return GenerateCRC( mp_nodes[i]->GetName() );
}

unsigned int CSkeletonData::GetParentName( int i )
{ FUNC_ENTER("CSkeletonData::GetParentName"); 
	MaxAssert( i >= 0 && i < GetCount() );

	if ( GetBone(i) == GetRootBone( GetBone(i) ) )
	{
		// if we're the root node, then the parent is 0
		return 0;
	}
	else
	{
		return GenerateCRC( mp_nodes[i]->GetParentNode()->GetName() );
	}
}

unsigned int CSkeletonData::GetFlipName( int i )
{ FUNC_ENTER("CSkeletonData::GetFlipName"); 
	MaxAssert( i >= 0 && i < GetCount() );

	TCHAR* pCurrentBoneName = GetBone(i)->GetName();
	TSTR neutral_name = GetBone(i)->GetName();
	BOOL is_left = FALSE;
	BOOL is_right = FALSE;
	TCHAR* p_temp;
	TSTR left_name;
	TSTR right_name;

	if ( p_temp = strstr(pCurrentBoneName, "left_"))
	{
		// compensate for "left_" not being at the beginning of the string
		neutral_name = p_temp;
		neutral_name = neutral_name.Substr( strlen("left_"), strlen(pCurrentBoneName) - strlen("left_") );
		is_left = TRUE;
		left_name = TSTR("left_") + neutral_name;
		right_name = TSTR("right_") + neutral_name;
	}
	else if ( (p_temp = strstr(pCurrentBoneName, "_L")) &&
		      (p_temp == pCurrentBoneName + (strlen(pCurrentBoneName) - 2)) )
	{
		// compensate for "_L" not being at the end of the string
		neutral_name = pCurrentBoneName;
		neutral_name[neutral_name.Length() - 2] = '\0';
		is_left = TRUE;
		left_name = neutral_name + "_L";
		right_name = neutral_name + "_R";
	}
	else if ( p_temp = strstr(pCurrentBoneName, "right_"))
	{
		// compensate for "right_" not being at the beginning of the string
		neutral_name = p_temp;
		neutral_name = neutral_name.Substr( strlen("right_"), strlen(pCurrentBoneName) - strlen("right_") );
		is_right = TRUE;
		left_name = TSTR("left_") + neutral_name;
		right_name = TSTR("right_") + neutral_name;
	}
	else if ( (p_temp = strstr(pCurrentBoneName, "_R")) &&
		      (p_temp == pCurrentBoneName + (strlen(pCurrentBoneName) - 2)) )
	{

		neutral_name = pCurrentBoneName;
		neutral_name[neutral_name.Length() - 2] = '\0';
		is_right = TRUE;
		left_name = neutral_name + "_L";
		right_name = neutral_name + "_R";
	}

	// if left-hand part
	if (is_left)
	{
		for ( int i = 0; i < GetCount(); i++ )
		{
			TCHAR* pBoneName = GetBone(i)->GetName();
			if ( strstr(pBoneName, right_name.data()) )
			{
				return GenerateCRC(pBoneName);
			}
		}
		return 0;
	}
	else if (is_right)
	{
		for ( int i = 0; i < GetCount(); i++ )
		{
			TCHAR* pBoneName = GetBone(i)->GetName();
			if ( strstr(pBoneName, left_name.data()) )
			{
				return GenerateCRC(pBoneName);
			}
		}
		return 0;
	}
	else
	{
		return 0;
	}
}

unsigned int CSkeletonData::GetChecksum()
{ FUNC_ENTER("CSkeletonData::GetChecksum"); 
	unsigned int checksum = (unsigned int)GetCount();

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetBoneName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetParentName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		checksum ^= GetFlipName( i );
	}

	return checksum;
}

void CSkeletonData::print_debug_file( void )
{ FUNC_ENTER("CSkeletonData::print_debug_file"); 
	unsigned int checksum = GetChecksum();
	DebugPrint("checksum = %08x\n", checksum);

	unsigned int numBones = GetCount();
	DebugPrint("numBones = %08x\n", numBones);

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		char parentName[256];
		for ( int j = 0; j < GetCount(); j++ )
		{
			if ( GetBone(i) == GetRootBone( GetBone(i) ) )
			{
				// if we're the root node, then the parent is 0
				strcpy( parentName, "Root" );
			}
			else
			{
				strcpy( parentName, mp_nodes[i]->GetParentNode()->GetName() );
			}
		}
		int flipIndex = -1;

		DebugPrint("bone %d name %s %08x parent = %08x (%s)\n", i, GetBone(i)->GetName(),
			GetBoneName(i),
			GetParentName(i),
			parentName);
	}
}

void CSkeletonData::SetPartialAnimState(int boneID, bool bActive)
{ FUNC_ENTER("CSkeletonData::SetPartialAnimState"); 
	assert(boneID < GetCount());

	if (bActive)
		m_mask[boneID / 32] |= (1 << (boneID % 32));
	else
		m_mask[boneID / 32] &= ~(1 << (boneID % 32));
}

bool CSkeletonData::GetPartialAnimState(int boneID)
{ FUNC_ENTER("CSkeletonData::GetPartialAnimState"); 
	return (m_mask[boneID / 32] & (1 << (boneID % 32)));
}

int CSkeletonData::WriteSKAFormat( FILE* pFile )
{ FUNC_ENTER("CSkeletonData::WriteSKAFormat"); 
	MaxAssert( pFile );

	unsigned int size = 0;

	unsigned int checksum = GetChecksum();
	fwrite(&checksum,sizeof(unsigned int),1,pFile);
	size += sizeof(unsigned int);

	unsigned int numBones = GetCount();
	fwrite(&numBones,sizeof(unsigned int),1,pFile);
	size += sizeof(unsigned int);

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
		fwrite(&boneName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
		fwrite(&parentName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
		fwrite(&flipName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

#ifdef ANIMFMTVERSION3
	int nMask = ((GetCount() - 1) / 32) + 1;
	
	// The mask chunk should only be written out if this is a partial anim
	// (i.e. not all the bits in the mask are set)
	bool bIsPartialAnim = false;

	for( i = 0; i < nMask; i++ )
	{
		if (m_mask[i] != 0xFFFFFFFF)
			bIsPartialAnim = true;
	}

	if (bIsPartialAnim)
	{
		fwrite(&numBones, sizeof(unsigned int), 1, pFile);

		for ( i = 0; i < nMask; i++)
		{
			fwrite(&m_mask[i], sizeof(unsigned int), 1, pFile);
			size += sizeof(unsigned int);
		}
	}
#endif

	// return number of bytes written
	return size;
}

int CSkeletonData::WriteSKAFormat( unsigned char* pData )
{ FUNC_ENTER("CSkeletonData::WriteSKAFormat"); 
	MaxAssert( pData );

	unsigned int size = 0;

	unsigned int checksum = GetChecksum();
	memcpy(pData, &checksum, sizeof(unsigned int));
	pData += sizeof(unsigned int);
	size  += sizeof(unsigned int);
	
	unsigned int numBones = GetCount();
	memcpy(pData, &numBones, sizeof(unsigned int));
	pData += sizeof(unsigned int);
	size  += sizeof(unsigned int);
	
	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
		memcpy(pData, &boneName, sizeof(unsigned int));
		pData += sizeof(unsigned int);
		size  += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
		memcpy(pData, &parentName, sizeof(unsigned int));
		pData += sizeof(unsigned int);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
		memcpy(pData, &flipName, sizeof(unsigned int));
		pData += sizeof(unsigned int);
		size += sizeof(unsigned int);
	}

#ifdef ANIMFMTVERSION3
	int nMask = ((GetCount() - 1) / 32) + 1;

	// The mask chunk should only be written out if this is a partial anim
	// (i.e. not all the bits in the mask are set)
	bool bIsPartialAnim = false;

	for( i = 0; i < nMask; i++ )
	{
		if (m_mask[i] != 0xFFFFFFFF)
			bIsPartialAnim = true;
	}

	if (bIsPartialAnim)
	{
		memcpy(pData, &numBones, sizeof(unsigned int));
		pData += sizeof(unsigned int);
		size  += sizeof(unsigned int);

		for ( i = 0; i < nMask; i++ )
		{
			memcpy(pData, &m_mask[i], sizeof(unsigned int));
			pData += sizeof(unsigned int);
			size += sizeof(unsigned int);
		}
	}
#endif

	// return number of bytes written
	return size;
}

int CSkeletonData::Write( FILE* pFile )
{ FUNC_ENTER("CSkeletonData::Write"); 
	return WriteSKAFormat( pFile );
}

int CSkeletonData::Write( unsigned char* pData )
{ FUNC_ENTER("CSkeletonData::Write"); 
	return WriteSKAFormat( pData );
}

#define SKEFILE_VERSION 2

int CSkeletonData::WriteSKEFormat( FILE* pFile )
{ FUNC_ENTER("CSkeletonData::WriteSKEFormat"); 
	MaxAssert( pFile );

	unsigned int size = 0;

	if ( SKEFILE_VERSION >= 2 )
	{
		// version 2 (THPS5+) of the SKE
		// format has a magic number as the checksum
		// (as well as the version number and flags)
		unsigned int checksum = CRCD(0x222756d5,"skeleton");
		fwrite(&checksum,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);

		unsigned int versionNumber = SKEFILE_VERSION;
		fwrite(&versionNumber,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);

		unsigned int flags = 0;		// flags field is currently unused...
		fwrite(&flags,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}
	else
	{
		// version 1 (THPS3-4) of the SKE file has
		// only the skeleton checksum, which we
		// end up ignoring anyway
		unsigned int checksum = GetChecksum();
		fwrite(&checksum,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	unsigned int numBones = GetCount();
	fwrite(&numBones,sizeof(unsigned int),1,pFile);
	size += sizeof(unsigned int);

	int i;

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int boneName = GetBoneName( i );
		fwrite(&boneName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int parentName = GetParentName( i );
		fwrite(&parentName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

	for ( i = 0; i < GetCount(); i++ )
	{
		unsigned int flipName = GetFlipName( i );
		fwrite(&flipName,sizeof(unsigned int),1,pFile);
		size += sizeof(unsigned int);
	}

////////////////////////////////////////////////////////////
	// TODO: Insert Partial anim mask field

	if ( SKEFILE_VERSION >= 2 )
	{
		// Write out the neutral pose data...
		for ( i = 0; i < GetCount(); i++ )
		{
			Mth::Quat myQuat;
			Mth::Vector myVector;
			GetNeutralPose( i, &myQuat, &myVector );
			fwrite(&myQuat,sizeof(Mth::Quat),1,pFile);
			size += sizeof(Mth::Quat);	
			fwrite(&myVector,sizeof(Mth::Vector),1,pFile);
			size += sizeof(Mth::Vector);	
		}
	}

	// return number of bytes written
	return size;
}

void CSkeletonData::GetNeutralPose( int i, Mth::Quat* pQuat, Mth::Vector* pVector )
{ FUNC_ENTER("CSkeletonData::GetNeutralPose"); 
	// This function grabs the default pose at frame 0.
	// The default pose needs to be the same as the one
	// used to build the skinned mesh.

	Matrix3 tm;
	Matrix3 parentTM;
	INode*	node = mp_nodes[i];
	INode*  parentNode = node->GetParentNode();

	// get the rotation
	tm=node->GetNodeTM(0);	
	tm.NoScale();

	if ( i == 0 )
	{
		tm.RotateX( DegToRad(-90.0f) );
	}

	// transform into local space
	parentNode = node->GetParentNode();
	parentTM = parentNode->GetNodeTM(0);
	parentTM.NoScale();
	tm = tm * Inverse(parentTM);

	Quat quat=Quat(tm);

	// make sure they're all anims start off with a positive w.
	if (quat.w<0.0f)
	{
		quat.x=-quat.x;
		quat.y=-quat.y;
		quat.z=-quat.z;
		quat.w=-quat.w;
	}

	*pQuat = Mth::Quat(quat.x,quat.y,quat.z,quat.w);

	// now get the translation
	tm=node->GetNodeTM(0);
	tm.NoScale();
	
	// transform into local space
	parentNode = node->GetParentNode();
	parentTM = parentNode->GetNodeTM(0);
	parentTM.NoScale();
	tm = tm * Inverse(parentTM);
	
	Point3 trans=tm.GetTrans();

	// Integrated LF fix per Gary (aml 6-25-03)
	if ( i == 0 )
	{
		float temp;
		temp = trans[Y];
		trans[Y]=trans[Z];
		trans[Z]=-temp;
	}
	// --- End Fix

	// the coordinate system doesn't 
	// need to be swapped
	*pVector = Mth::Vector(trans[X],trans[Y],trans[Z],trans[W]);
}

void CSkeletonData::RemoveNonPartialAnimBones(PartialAnimSet* pPartialAnimSet)
{ FUNC_ENTER("CSkeletonData::RemoveNonPartialAnimBones"); 
#ifdef ANIMFMTVERSION3
	// In version 3 we no longer remove bones from the skeleton but simply write the
	// bone set out to the mask field and store 0 keys for those bones
	for(int i = 0; i < m_numNodes; i++)
	{
		if (pPartialAnimSet->IsExportableBone(mp_nodes[i]->GetName()))
			m_mask[i / 32] |= (1 << (i % 32));
		else
			m_mask[i / 32] &= ~(1 << (i % 32));
	}

#else
	for(int i = 0; i < m_numNodes; i++)
	{
		// If the bone isn't exportable it needs to be removed from the sequence
		if (!pPartialAnimSet->IsExportableBone(mp_nodes[i]->GetName()))
		{
			m_numNodes--;
			memcpy(&mp_nodes[i], &mp_nodes[i + 1], sizeof(INode*) * m_numNodes);
			i--;
		}
	}
#endif
}
