#include "FuncEnter.h"

/*
	ScriptOptimize.cpp
	This runs through the exported node array and converts it to a group
	of shared substructures as per Mick's Ruby script
  
	5-29-03
*/

#include "ScriptOptimize.h"
#include "../PropEdit/ParseFuncs.h"
#include "../misc/GenCrc.h"
#include "max.h"
#include <string.h>
#include "path.h"
#include "../PropEdit/PropBufGen.h"
#include "ExportOptions.h"
#include "SceneExportOptions.h"
#include "ModelExportOptions.h"
#include "../appdata.h"
#include <time.h>
#include "Next.h"
#include "UserConfig.h"

extern HINSTANCE  hInstance;
extern Interface* gInterface;

#define DEFLIST_BUFSIZE  (1024 * 1024 * 5)		// 5 Meg
#define SECS_PER_DAY     86400                  // 60 sec->min * 60 min->hrs * 24 hrs->days

ScriptOptimizer::ScriptOptimizer() : lookupTable(8)
{ FUNC_ENTER("ScriptOptimizer::ScriptOptimizer"); 
	level = OPTLEVEL_DEFINITIONS_ONLY;
}

void ScriptOptimizer::StoreQNUpdateTime()
{ FUNC_ENTER("ScriptOptimizer::StoreQNUpdateTime"); 
	ReferenceTarget* pScene = gInterface->GetScenePointer();

	time_t curtime;
	time(&curtime);

	void* pTimeChunk = malloc(sizeof(time_t));
	memcpy(pTimeChunk, &curtime, sizeof(time_t));

	pScene->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_LAST_QN_EXPORT_DATE);
	pScene->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_LAST_QN_EXPORT_DATE, sizeof(time_t), pTimeChunk);
}

bool ScriptOptimizer::QNNeedsFullUpdate(int nDays)
{ FUNC_ENTER("ScriptOptimizer::QNNeedsFullUpdate"); 
	ReferenceTarget* pScene = gInterface->GetScenePointer();

	AppDataChunk* appdata = pScene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_LAST_QN_EXPORT_DATE);

	if (!appdata || !appdata->data)
		return true;

	time_t* pExportTime = (time_t*)appdata->data;
	time_t  curtime;
	time_t  elapsedtime;
	time(&curtime);

	elapsedtime = curtime - (*pExportTime);

	if (elapsedtime / SECS_PER_DAY >= nDays)
		return true;

	// Ensure that the definition file exists.  If it doesn't a full update must be performed
	ExportOptions* eo = GetExportOptions();
	
	CStr path = getenv(APP_ENV);

	if (eo->m_ExportType == ExportOptions::vEXPORT_MODEL)
	{
		ModelExportOptions meo;
		GetModelExportOptions(&meo);
		
		path += MODEL_QN_PATH;

		if (strlen(meo.m_ModelDir) > 0)
			path += meo.m_ModelDir + CStr("\\");

		path += meo.m_ModelName;

		// This is now going to dump as a .txt file instead of a .qn so that we
		// can append it to the end of the actual level QN without qcompall compiling both
		path += CStr("\\") + CStr("ncomp_") + CStr(meo.m_ModelName) + ".txt";
	}
	else
	{
		// eo.m_ExportType == ExportOptions::vEXPORT_SCENE
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);
		
		path += LEVEL_PATH;
		path += seo.m_SceneName;

		// This is now going to dump as a .txt file instead of a .qn so that we
		// can append it to the end of the actual level QN without qcompall compiling both
		path += CStr("\\") + CStr("ncomp_") + CStr(seo.m_SceneName) + ".txt";
	}

	FILE* fpTest = fopen(path, "r");
	if (!fpTest)
		return true;

	fclose(fpTest);

	return false;
}

void ScriptOptimizer::OptimizeScripts(char* filename)
{ FUNC_ENTER("ScriptOptimizer::OptimizeScripts"); 
	// Open the level script file and reparse the node array
	// Pos, Angles, Name are always considered unique
	// TriggerScript is also always considered unique at the end of the sequence
	// as is Links

	fileBuf  = "";
	fileBuf2 = "";

	fp = fopen(filename, "rb");
	//char lineBuf[1024];
	long offset;

	// Determine file size
	fseek(fp, 0, SEEK_END);
	offset = ftell(fp);
	fseek(fp, 0, SEEK_SET);

	// Load the entire node array into memory for faster processing
	// (leave the file open so we can dump the scripts after our nodearray replacements)
	char* pBuf = (char*)malloc(offset + 1);
	fread(pBuf, offset, 1, fp);
	fclose(fp);

	// Ensure the end of the file is a properly terminated string
	pBuf[offset] = 0;

	char* pEndPos = strstr(pBuf, "TriggerScripts =");
	assert(pEndPos);
	char  cEnd = *pEndPos;
	*pEndPos = 0;

	fileBuf = pBuf;
	*pEndPos = cEnd;
	fileBuf2 = pEndPos;

	/*
	while(fgets(lineBuf, 1023, fp) && !strstr(lineBuf, "TriggerScripts ="))
		fileBuf += lineBuf;

	// Load the remainder of the .qn file after the node array
	while(fgets(lineBuf, 1023, fp))
		fileBuf2 += lineBuf;
	*/

	fclose(fp);

	defFilename = GetQNOptimizePath();

	// Allocate memory for the replacement list that we will immediately dump out to the end of the QN file
	char* bufDefList  = (char*)malloc(DEFLIST_BUFSIZE);
	int   sizeDefList = 0;
	bool  bNewExport  = false;

	// If we're set to use periodic compression determine the compression level that should be used for this pass
	if (level == OPTLEVEL_PERIODIC_COMPRESSION)
	{
		SceneExportOptions seo;

		if (QNNeedsFullUpdate(seo.m_PeriodicDays))
		{
			level      = OPTLEVEL_LOW_COMPRESSION;
			bNewExport = true;
		}
		else
			level = OPTLEVEL_HIGH_COMPRESSION;
	}

	switch(level)
	{
	case OPTLEVEL_DEFINITIONS_ONLY:
		defFilename += GetQNOptimizeName();
		LoadDefs(defFilename, bufDefList, &sizeDefList);
		PerformDefReplace();
		break;

	case OPTLEVEL_LOW_COMPRESSION:
		{
			//defFilename += QNOPTIMIZE_USRNAME;
			
			// Generated file will reside with scene
			ExportOptions* eo = GetExportOptions();
			
			CStr path = getenv(APP_ENV);

			if (eo->m_ExportType == ExportOptions::vEXPORT_MODEL)
			{
				ModelExportOptions meo;
				GetModelExportOptions(&meo);
				
				path += MODEL_QN_PATH;

				if (strlen(meo.m_ModelDir) > 0)
					path += meo.m_ModelDir + CStr("\\");

				path += meo.m_ModelName;

				// This is now going to dump as a .txt file instead of a .qn so that we
				// can append it to the end of the actual level QN without qcompall compiling both
				path += CStr("\\") + CStr("ncomp_") + CStr(meo.m_ModelName) + ".txt";
			}
			else
			{
				// eo.m_ExportType == ExportOptions::vEXPORT_SCENE
				SceneExportOptions seo;
				GetSceneExportOptions(&seo);
				
				path += LEVEL_PATH;
				path += seo.m_SceneName;

				// This is now going to dump as a .txt file instead of a .qn so that we
				// can append it to the end of the actual level QN without qcompall compiling both
				path += CStr("\\") + CStr("ncomp_") + CStr(seo.m_SceneName) + ".txt";
			}

			PerformLowCompress(path, bufDefList, &sizeDefList);
			PerformDefReplace();
		}
		break;

	case OPTLEVEL_HIGH_COMPRESSION:
		{
			// Generated file will reside with scene
			ExportOptions* eo = GetExportOptions();
			
			CStr path = getenv(APP_ENV);

			if (eo->m_ExportType == ExportOptions::vEXPORT_MODEL)
			{
				ModelExportOptions meo;
				GetModelExportOptions(&meo);
				
				path += MODEL_QN_PATH;

				if (strlen(meo.m_ModelDir) > 0)
					path += meo.m_ModelDir + CStr("\\");

				path += meo.m_ModelName;

				// This is now going to dump as a .txt file instead of a .qn so that we
				// can append it to the end of the actual level QN without qcompall compiling both
				path += CStr("\\") + CStr("ncomp_") + CStr(meo.m_ModelName) + ".txt";
			}
			else
			{
				SceneExportOptions seo;
				GetSceneExportOptions(&seo);
				
				path = getenv(APP_ENV);
				path += LEVEL_PATH;
				path += seo.m_SceneName;

				// This is now going to dump as a .txt file instead of a .qn so that we
				// can append it to the end of the actual level QN without qcompall compiling both
				path += CStr("\\") + CStr("ncomp_") + CStr(seo.m_SceneName) + ".txt";
			}

			LoadDefs(path, bufDefList, &sizeDefList);
			PerformDefReplace();
		}

		/*
		defFilename += QNOPTIMIZE_USRNAME;
		PerformHighCompress(defFilename);
		PerformDefReplace();
		*/
		break;
	}

	// Now write the modified data to the file
	fp = fopen(filename, "wb");
	
	if (!fp)
	{
		char strErr[256];
		sprintf(strErr, "Failed to open '%s' to write optimized .qn", (char*)filename);
		MessageBox(gInterface->GetMAXHWnd(), strErr, "Failed to write qn", MB_ICONWARNING|MB_OK);
		free(bufDefList);
		return;
	}

	//fileBuf = ReplaceStr(fileBuf, LF, CRLF);
	//fileBuf2 = ReplaceStr(fileBuf2, LF, CRLF);

	fwrite(fileBuf, fileBuf.Length(), 1, fp);
	fwrite(fileBuf2, fileBuf2.Length(), 1, fp);
	//fprintf(fp, fileBuf);
	//fprintf(fp, fileBuf2);

	// Dump the definition file out to the end of the QN
	char LF[2], CRLF[3];
	LF[0]   = 10;
	LF[1]   = 0;
	CRLF[0] = 13;
	CRLF[1] = 10;
	CRLF[2] = 0;

	int len = strlen(bufDefList);

	//CStr defList = ReplaceStr(bufDefList, LF, CRLF);
	CStr defList = bufDefList;
	sizeDefList = defList.Length();
	fwrite((char*)defList, sizeDefList, 1, fp);

	fclose(fp);
	free(bufDefList);

	if (bNewExport)
		StoreQNUpdateTime();
}

void ScriptOptimizer::PerformDefReplace()
{ FUNC_ENTER("ScriptOptimizer::PerformDefReplace"); 
	Link<ReplaceRecord>* link = replaceList.GetHead();
	char LF[2], LFCR[3], CRLF[3], TAB[2];
	LF[0] = 10;
	LF[1] = 0;
	LFCR[0] = 10;
	LFCR[1] = 13;
	LFCR[2] = 0;
	TAB[0] = 9;
	TAB[1] = 0;
	CRLF[0] = 13;
	CRLF[1] = 10;
	CRLF[2] = 0;

	//fileBuf = ReplaceStr(fileBuf, LF, LFCR);

	// Scan through the file buffer (nodearray) and replace any buffer occurances with their structure
	while(link)
	{
		//fileBuf = ReplaceStrNoTab(fileBuf, link->data.buffer, CStr(TAB) + link->data.name + CStr(LF));
		fileBuf = ReplaceStr(fileBuf, link->data.buffer, CStr(TAB) + link->data.name + CStr(CRLF));

		link = link->next;
	}

	//fileBuf = ReplaceStr(fileBuf, LFCR, LF);
}

bool ScriptOptimizer::LoadDefs(char* filename, char* buf, int* size)
{ FUNC_ENTER("ScriptOptimizer::LoadDefs"); 
	if (size)
		*size = 0;

	replaceList.Clear();
	
	//FILE* fpNodeCompress = fopen(filename, "r");
	FILE* fpNodeCompress = fopen(filename, "rb");

	if (!fpNodeCompress)
	{
		char strErr[256];
		sprintf(strErr, "Failed to open compress definition file '%s'.", filename);
		MessageBox(gInterface->GetMAXHWnd(), strErr, "QN Compress Pass Aborted", MB_ICONWARNING|MB_OK);
		return false;
	}

	char lineBuf[1024];
	char token[256];
	int  pos = 0;
	char eol[3];
	eol[0] = 13;
	eol[1] = 10;
	eol[2] = 0;
//	eol[0] = 10;
//	eol[1] = 0;


	while(fgets(lineBuf, 1023, fpNodeCompress))
	{
		// Skip to a line that contains '='
		if (!strstr(lineBuf, "="))
			continue;

		ReplaceRecord replaceRecord;

		// Read identifier
		GetToken(lineBuf, &pos, token, " \r\n");
		StripToken(token);
		replaceRecord.name = token;

		// Skip ahead to block
		GetToken(lineBuf, &pos, token, " \r\n");		// Skip '=' 
		FileAdvance(fpNodeCompress, lineBuf, &pos, '{');

		char* closeBracket;
		while (!(closeBracket = strstr(lineBuf,"}")))
		{
			if (!fgets(lineBuf,1023, fpNodeCompress))
			{
				MessageBox(gInterface->GetMAXHWnd(), "End of compress definition file reached without finding definition end '}'", "Missing close definition", MB_ICONWARNING|MB_OK);
				fclose(fpNodeCompress);
				return false;
			}

			pos = 0;
			//replaceRecord.buffer += GetRemainLinePartial(lineBuf, &pos) + CStr(eol);
			replaceRecord.buffer += GetRemainLineExact(lineBuf, &pos) + CStr(eol);
		}

		if (closeBracket)
		{
			*strrchr(replaceRecord.buffer, '}') = 0;

			// Add entry to the replace list
			replaceList.Add(&replaceRecord);
		}

		pos = 0;
	}

	// If loading to an external buffer reload the whole file at once
	if (buf)
	{
		fseek(fpNodeCompress, 0, SEEK_END);
		long offset = ftell(fpNodeCompress);
		fseek(fpNodeCompress, 0, SEEK_SET);
		fread(buf, offset, 1, fpNodeCompress);

		if (size)
			*size = offset;

		buf[offset] = 0;
	}

	fclose(fpNodeCompress);

	return true;
}

void ScriptOptimizer::PerformLowCompress(char* filename, char* buf, int* size)
{ FUNC_ENTER("ScriptOptimizer::PerformLowCompress"); 
	LinkList<CStr> blockList;
	CStr blockBuf;
	int  storePos;
	int  pos    = 0;
	int  endpos = fileBuf.Length();
	int  currec = 0;

	char LF[2], LFCR[3], TAB[2], CRLF[3];
	LF[0] = 10;
	LF[1] = 0;
	LFCR[0] = 10;
	LFCR[1] = 13;
	LFCR[2] = 0;
	TAB[0] = 9;
	TAB[1] = 0;
	CRLF[0] = 13;
	CRLF[1] = 10;
	CRLF[2] = 0;

	if (size)
		*size = 0;

	replaceList.Clear();

	// Since we can garuntee the exact node array output format and sequence
	// (since we generated it) we can safely assume the ordering of the data

	pProgress = new ProgressBar(hInstance,gInterface->GetMAXHWnd(),"Building optimal .qn node compress file", 0, endpos);

	while(pos <= endpos)
	{
		pProgress->SetVal(pos);

		// Advance to block start and read contents
		if (!StrAdvance(fileBuf, &pos, '{'))
			break;

		while(!strstr(GetRemainLinePartial(fileBuf, &pos), "Name =")) {}
		GetRemainLinePartial(fileBuf, &pos);

		//GetRemainLinePartial(fileBuf, &pos);	// { eol
		//GetRemainLinePartial(fileBuf, &pos);	// Node #
		//GetRemainLinePartial(fileBuf, &pos);	// Pos =
		//GetRemainLinePartial(fileBuf, &pos);	// Angles =
		//GetRemainLinePartial(fileBuf, &pos);	// Name =

		/// Relevant block info (read to }, skin TriggerLink and Links =
		storePos = pos;
		StrAdvance(fileBuf, &pos, '}');

		if (pos - storePos == 0)
			break;

		blockBuf = fileBuf.Substr(storePos, pos - storePos);

		blockList.Clear();	// new
		PropBufGen::BuildList(&blockList, blockBuf, false);

		// Remove any TriggerScript or Links lines
		Link<CStr>* link = blockList.GetHead();
		Link<CStr>* linkNext;

		char  bufClassName[1024];
		char* classBuf;
		CStr  className;

		while(link)
		{
			linkNext = link->next;
			
			if (classBuf = strstr(link->data, "Class ="))
			{
				sscanf(classBuf, "Class = %s", bufClassName);
				className = bufClassName;
			}

			if (link->data.Length() == 0 ||
				strstr(link->data, "TriggerScript =") ||
				strstr(link->data, "Links=") ||
				strstr(link->data, "ncomp_"))
				blockList.Remove(link);

			link = linkNext;
		}

		ReplaceRecord replaceRec;
		char recID[40];
		currec++;
		_itoa(currec,recID,10);

		replaceRec.buffer = PropBufGen::BuildBuf(&blockList);
		//replaceRec.buffer = ReplaceStr(PropBufGen::BuildBuf(&blockList), TAB, "");

		//if (replaceRec.buffer.Length() > 0)
		//	replaceRec.buffer[replaceRec.buffer.Length()-1] = 0;

		replaceRec.name   = CStr("ncomp_") + className + CStr("_") + CStr(recID);
		replaceRec.buffer = ReplaceStr(replaceRec.buffer, LF, CRLF);
		
		// Replaces must be ran on this buffer to ensure there are no subsets
		Link<ReplaceRecord>* repLink = replaceList.GetHead();

		while(repLink)
		{
			replaceRec.buffer = ReplaceStr(replaceRec.buffer, repLink->data.buffer, "");
			repLink = repLink->next;
		}

		// The buffer must occur at least 2 times in the node array for us to add it as a seperate struct
		if (CountStr(fileBuf, replaceRec.buffer) > 1)
			replaceList.AddUnique(&replaceRec);
	}

	delete pProgress;

	// Dump the auto generated replacement list out to a file (if appropriate)
	if (filename)
	{
		//FILE* fp = fopen(filename, "w");
		FILE* fp = fopen(filename, "wb");

		if (!fp)
		{
			char strErr[256];
			sprintf(strErr, "Failed to open '%s' for writting.  Operation Aborted.", (char*)filename);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Failed to write replace file", MB_ICONWARNING|MB_OK);
			return;
		}

		Link<ReplaceRecord>* link = replaceList.GetHead();

		while(link)
		{
			//fprintf(fp, "%s = {\n", link->data.name);
			fprintf(fp, "%s = {", (char*)link->data.name);
			fwrite(CRLF, 2, 1, fp);
			//fprintf(fp, "%s", (char*)ReplaceStr(link->data.buffer, LFCR, LF));
			fprintf(fp, "%s", (char*)link->data.buffer);
			//fprintf(fp, "}\n\n");
			fprintf(fp, "}");
			fwrite(CRLF, 2, 1, fp);
			fwrite(CRLF, 2, 1, fp);

			link = link->next;
		}

		fclose(fp);
	}

	if (buf)
	{
		char sbuf[2048];
		int  len;
		Link<ReplaceRecord>* link = replaceList.GetHead();

		while(link)
		{
			sprintf(sbuf, "%s = {", (char*)link->data.name);
			len = strlen(sbuf);
			memcpy(buf, sbuf, len);
			buf     += len;
			if (size)
				(*size) += len;


			memcpy(buf, CRLF, 2);
			buf     += 2;
			if (size)
				(*size) += 2;

			sprintf(sbuf, "%s", (char*)link->data.buffer);
			len = strlen(sbuf);
			memcpy(buf, sbuf, len);
			buf     += len;
			if (size)
				(*size) += len;

			sprintf(sbuf, "}");
			memcpy(buf, sbuf, 1);
			buf     += 1;
			if (size)
				(*size) += 1;

			memcpy(buf, CRLF, 2);
			buf     += 2;
			if (size)
				(*size) += 2;

			memcpy(buf, CRLF, 2);
			buf     += 2;
			if (size)
				(*size) += 2;

			link = link->next;
		}
	}
}

void ScriptOptimizer::PerformHighCompress(char* filename, char* buf, int* size)
{ FUNC_ENTER("ScriptOptimizer::PerformHighCompress"); 

}
