/*
	ObjAnimExport.cpp
	This is adapted from the camera exporter to export information for animation of objects within the level
*/

#include <Engine/Engine.h>
#include "ObjAnimExport.h"
#include <Sk/Gamenet/ExportMsg.h>
#include "../Misc/GenCrc.h"
#include "../Misc/Util.h"
#include <io.h>
#include <sys/stat.h>
#include <direct.h>
#include "GenExport.h"
#include "../../../../include/AnimExportFmt.h"
#include "Resource.h"
#include "../UI/ListView.h"
#include "../PropEdit/ParseFuncs.h"
#include "path.h"
#include "Appdata.h"
#include "SceneExportOptions.h"
#include "../UI/OKtoAll.h"
#include "SceneExport.h"
#include <process.h>
#include "../misc/mth.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

#define WRITE_ACCESS           0x02
#define DEFAULT_COMPRESS       0.999700f
#define DEFAULT_TCOMPRESS      1.000000f
#define CAMERA_CRC             0xc4e311fa

#define USERSCRIPT_NAME		   "[User]"

static ObjectExporter theObjectExporter;
ObjectExporter* GetObjExporter() { FUNC_ENTER("GetObjExporter");  return &theObjectExporter; }

extern Interface* gInterface;

struct ObjExportSettings
{
	bool bDebug;		// Output debug information
	bool bCompress;		// Compress the keyframe data
	bool bNoDbls;		// Compress 2 key sequences to 1 key
	bool bExportQN;		// Export a QN file containing all exported Objects
	char strPrefix[256];// Export prefix name
	bool bExportAll;	// True if all Objects should be exported
	bool bCompressTime;	// Time should be compressed (no floats)   // Now Always On
	bool bCleanup;		// Deletes unsed Object files from the output directory
	bool bArchive;		// Archives the Object animation *.ska files
};

void GetNodeList(INodeTab& nodes,INode* root=NULL)
{ FUNC_ENTER("GetNodeList"); 
	if (!root)
		root = gInterface->GetRootNode();

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* node = root->GetChildNode(i);
		GetNodeList(nodes,node);
	}

	if (root != gInterface->GetRootNode())
		nodes.Append(1,&root);
}

// This function will encode a list of strings so that it's suitable for incorporation into a datachunk
void* EncodeStringList(LinkList<CStr>* list,int* length)
{ FUNC_ENTER("EncodeStringList"); 
	int   len  = 0;
	int   slen;
	char* mem  = NULL;
	char* pos  = NULL;

	if (list->GetSize()==0)
	{
		*length = 0;
		return NULL;
	}

	Link<CStr>* curnode = list->GetHead();

	// Compute the total amount of memory required to store all the strings in retrievable form
	while(curnode)
	{
		// String size plus the length of the string
		len += curnode->data.Length()+1+sizeof(int);
		curnode = curnode->next;
	}

	// Now that we've determined the necessary amount of memory we can allocate it
	// and copy the data in sequence
	mem = (char*)malloc(len);
	pos = mem;

	curnode = list->GetHead();

	while(curnode)
	{
		slen = curnode->data.Length()+1;

		memcpy(pos,&slen,sizeof(int));
		pos += sizeof(int);

		memcpy(pos,(char*)curnode->data,slen);
		pos += slen;

		curnode = curnode->next;
	}

	*length = len;

	return mem;
}

// This function will decode a list of strings from memory (suitable for processing from an appdata chunk)
void DecodeStringList(LinkList<CStr>* list,void* buf,int len)
{ FUNC_ENTER("DecodeStringList"); 
	char* pos = (char*)buf;
	char* end = pos + len;
	char* strBuf;
	int   slen;

	list->Clear();

	while(pos < end)
	{
		slen = *((int*)pos);
		strBuf = new char[slen+1];
		pos += sizeof(int);

		strcpy(strBuf,pos);

		pos += slen;

		list->Add(&CStr(strBuf));

		delete [] strBuf;
	}
}

CStr DecodeStringBuffer(void* buf, int len)
{ FUNC_ENTER("DecodeStringBuffer"); 
	char* pos = (char*)buf;
	char* end = pos + len;
	int   slen;
	char* strBuf;
	CStr  outBuf;

	while(pos < end)
	{
		slen = *((int*)pos);
		strBuf = new char[slen+1];

		pos += sizeof(int);

		strcpy(strBuf,pos);

		pos += slen;

		outBuf += CStr(strBuf) + CStr("\n");
		delete [] strBuf;
	}

	return outBuf;
}

bool ContainsSystem(AppDataChunk* appdata,CStr sysName)
{ FUNC_ENTER("ContainsSystem"); 
	if (appdata && appdata->data)
	{
		LinkList<CStr> list;
		DecodeStringList(&list,appdata->data,appdata->length);

		// Scan the decoded string list to see if it contains the system
		Link<CStr>* curnode = list.Find(&sysName);

		if (curnode)
			return true;
	}

	return false;
}

void GetExportNodes(char* sysName,INodeTab& nodes,INode* root=NULL)
{ FUNC_ENTER("GetExportNodes"); 
	if (!root)
	{
		nodes.ZeroCount();
		root = gInterface->GetRootNode();
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* node = root->GetChildNode(i);
		GetExportNodes(sysName,nodes,node);
	}

	if (root != gInterface->GetRootNode())
	{
		AppDataChunk* appdataFlag = root->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

		if (appdataFlag && appdataFlag->data)
		{
			// The appdata should contain the name of the system
			if (ContainsSystem(appdataFlag,sysName))
				nodes.Append(1,&root);
		}
	}
}

void GetAnimSystems(LinkList<CStr>* list,INode* root=NULL)
{ FUNC_ENTER("GetAnimSystems"); 
	if (!root)
	{
		root = gInterface->GetRootNode();
		list->Clear();
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		GetAnimSystems(list,child);
	}

	if (root == gInterface->GetRootNode())
		return;

	AppDataChunk* appdata = root->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

	if (appdata && appdata->data)
	{
		LinkList<CStr> slist;
		DecodeStringList(&slist,appdata->data,appdata->length);

		Link<CStr>* curnode = slist.GetHead();

		while(curnode)
		{
			list->AddToTailUnique(&curnode->data);
			curnode = curnode->next;
		}
	}
}

/////////// ObjAddKeyDlg  ////////////////////////////////////////////

ObjAddKeyDlg::ObjAddKeyDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_CAMOPTADDKEY),hwndParent)
{ FUNC_ENTER("ObjAddKeyDlg::ObjAddKeyDlg"); 
	this->ip=ip;

	scriptEditor = new RTFEditor(hInstance,GetDlgItem(hwnd,IDC_RICHEDIT1));
	scriptEditor->InitScriptDB();

	fpOK=NULL;
	pOKData=NULL;
	bAbort=false;
	bScriptsLoaded=false;

	IEdit=GetICustEdit(GetDlgItem(hwnd,IDC_EDITTIME));
	IEditFile=GetICustEdit(GetDlgItem(hwnd,IDC_EDITFILE));
	ISpin=GetISpinner(GetDlgItem(hwnd,IDC_SPINTIME));

	ISpin->LinkToEdit(GetDlgItem(hwnd,IDC_EDITTIME),EDITTYPE_FLOAT);
}

ObjAddKeyDlg::~ObjAddKeyDlg()
{ FUNC_ENTER("ObjAddKeyDlg::~ObjAddKeyDlg"); 
	ReleaseISpinner(ISpin);
	ReleaseICustEdit(IEdit);
	ReleaseICustEdit(IEditFile);

	delete scriptEditor;
}

CStr ObjAddKeyDlg::GetScriptFile()
{ FUNC_ENTER("ObjAddKeyDlg::GetScriptFile");  
	if (scriptFile.Length()==0)
		return CStr("[None]");

	return scriptFile; 
}

void ObjAddKeyDlg::Show()
{ FUNC_ENTER("ObjAddKeyDlg::Show"); 
	MSDlgWindow::Show();

	if (!bScriptsLoaded)
	{
		Refresh();
		bScriptsLoaded=true;
	}
}

BOOL ObjAddKeyDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("ObjAddKeyDlg::DlgProc"); 
	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (scriptEditor->ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			;//EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case CC_SPINNER_CHANGE:
		SpinChange();
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDOK:
			Hide();
			if (fpOK)
				fpOK(this,pOKData);
			
			scriptFile="";

			return TRUE;

		case IDCANCEL:
			Hide();
			return TRUE;

		case IDC_RESETSCRIPT:
			SetScript("");
			SetScriptCode("");
			scriptEditor->AllowChanges(TRUE);
			return TRUE;

		case IDC_REFRESH:
			Refresh();
			return TRUE;

		case IDC_EDITFILE:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				{
					char buf[256];
					IEditFile->GetText(buf,256);
					if (strlen(buf)==0)
					{
						scriptEditor->AllowChanges(TRUE);
						scriptEditor->SetText("");
					}

					DisableAccelerators();

					break;
				}
			}

		case IDC_SCRIPTLIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				SelChange();
				return TRUE;
			}
		}
	}

	return FALSE;
}

bool ObjAddKeyDlg::MatchContent(FILE* fp,char* wildcard,char* buf,int* line)
{ FUNC_ENTER("ObjAddKeyDlg::MatchContent"); 
	char  tempBuf[512];

	if (!fp)
		return false;

	while(!feof(fp))
	{
		fgets(tempBuf,510,fp);
		(*line)++;

		CStr strBuf=tempBuf;
		strBuf.toLower();

		CStr strWildcard=wildcard;
		strWildcard.toLower();

		if (MatchPattern(strBuf,strWildcard))
		{
			tempBuf[511]='\0';
			strcpy(buf,tempBuf);
			return true;
		}

		// Check for cancel
		MSG msg;

		if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
		{
			if (msg.message==WM_LBUTTONDOWN)
			{
				bAbort=true;
				return false;
			}
		}
	}
	
	return false;
}

void ObjAddKeyDlg::Refresh()
{ FUNC_ENTER("ObjAddKeyDlg::Refresh"); 
	CStr strPath=getenv(APP_ENV);
	strPath+=SCRIPT_PATH;
	strPath=strPath.Substr(0,strPath.Length()-1);

	int count=CountFiles(strPath,"*.q");
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETRANGE,0,MAKELPARAM(0,count));
	SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,0,0);
	SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_RESETCONTENT,0,0);
	scriptFileDB.Clear();

	bAbort=false;
	Find(strPath,"*.q","script *");
}


int ObjAddKeyDlg::CountFiles(char* sdir,char* prefix)
{ FUNC_ENTER("ObjAddKeyDlg::CountFiles"); 
	_finddata_t fdata;
	int         count=0;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				count+=CountFiles(strPath,prefix);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[1024];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
				count++;
		}

		bContinue=_findnext(handle,&fdata);
	}

	_findclose(handle);

	return count;
}

void ObjAddKeyDlg::Find(char* sdir,char* prefix,char* search)
{ FUNC_ENTER("ObjAddKeyDlg::Find"); 
	_finddata_t fdata;

	_chdir(sdir);
	long handle=_findfirst("*.*",&fdata);
	int  bContinue=-1;

	if (handle)
		bContinue=0;

	while(bContinue==0)
	{
		// Recurse subdirectories
		if (fdata.attrib & _A_SUBDIR)
		{
			if (strcmp(fdata.name,".")!=0 &&
				strcmp(fdata.name,"..")!=0)
			{
				char strPath[1024];
				strcpy(strPath,sdir);
				strcat(strPath,"\\");
				strcat(strPath,fdata.name);

				Find(strPath,prefix,search);
				_chdir(sdir);
			}
		}
		else
		{
			char strPath[2048];
			strcpy(strPath,sdir);
			strcat(strPath,"\\");
			strcat(strPath,fdata.name);

			if (MatchPattern(CStr(fdata.name),CStr(prefix)))
			{
				FILE* fp;
				char buf[1024]="";
				int  line=0;
				fp=fopen(fdata.name,"r");

				while(MatchContent(fp,search,buf,&line))
				{
					char strNew[1024];
					char strNum[256];
					_itoa(line,strNum,10);

					/*
					strcpy(strNew,strPath);
					strcat(strNew," (");
					strcat(strNew,strNum);
					strcat(strNew,")        ");
					strcat(strNew,buf);
					*/

					CStr strBuf=buf;
					strBuf.toLower();

					int pos=Instr(strBuf,"script ");
					if (pos!=-1 && pos<strBuf.Length())
					{
						strcpy(strNew,buf+pos+7);

						// Terminate after first word
						for(int i=0;i<strlen(strNew);i++)
						{
							if ((strNew[i]==' ' &&  !LineContainsAll(strNew,' ')) ||
								strNew[i]=='\t' ||
								strNew[i]=='\n' ||
								strNew[i]=='\r')
							{
								strNew[i]='\0';
								break;
							}
						}

						CStr str=StripLeft(strNew);

						if (str.Length()>0)
						{
							int index=SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_ADDSTRING,0,(LPARAM)(char*)str);
						
							// Add filename to the newly added entry
							char path[512];
							_getcwd(path,511);

							CStr strFilename=CStr(path)+CStr("\\")+CStr(fdata.name);
							Link<CStr>* link=scriptFileDB.Add(&strFilename);
							SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)link);
						}
					}

					// Check for cancel
					MSG msg;

					if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
					{
						if (msg.message==WM_LBUTTONDOWN)
							return;
					}
				}

				fclose(fp);

				if (bAbort)
					return;

				// Check for cancel
				MSG msg;

				if (PeekMessage(&msg,GetDlgItem(hwnd,IDC_CANCEL),0,0,PM_REMOVE))
				{
					if (msg.message==WM_LBUTTONDOWN)
					{
						_findclose(handle);
						return;
					}
				}

				// Update progress bar
				int val=SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_GETPOS,0,0);
				SendDlgItemMessage(hwnd,IDC_PROGBAR,PBM_SETPOS,(WPARAM)val+1,0);
			}
		}

		bContinue=_findnext(handle,&fdata);
	}

	_findclose(handle);
}

void ObjAddKeyDlg::SetOKCB(void(*Func)(ObjAddKeyDlg*,void*),void* pData)
{ FUNC_ENTER("ObjAddKeyDlg::SetOKCB"); 
	fpOK=Func;
	pOKData=pData;
}

void ObjAddKeyDlg::SetLimits(int min,int max)
{ FUNC_ENTER("ObjAddKeyDlg::SetLimits"); 
	ISpin->SetLimits(min,max);
}
	
int ObjAddKeyDlg::GetTime()
{ FUNC_ENTER("ObjAddKeyDlg::GetTime"); 
	return IEdit->GetFloat();
}

void ObjAddKeyDlg::SetTime(int time)
{ FUNC_ENTER("ObjAddKeyDlg::SetTime"); 
	ISpin->SetValue(time,TRUE);
}

CStr ObjAddKeyDlg::GetScript()
{ FUNC_ENTER("ObjAddKeyDlg::GetScript"); 
	char buf[256];
	IEditFile->GetText(buf,256);
	return CStr(buf);
}

CStr ObjAddKeyDlg::GetScriptCode()
{ FUNC_ENTER("ObjAddKeyDlg::GetScriptCode"); 
	return scriptEditor->GetText();	
}

void ObjAddKeyDlg::SetScript(CStr strScript)
{ FUNC_ENTER("ObjAddKeyDlg::SetScript"); 
	IEditFile->SetText(strScript);
}

void ObjAddKeyDlg::SetScriptCode(CStr strScriptCode)
{ FUNC_ENTER("ObjAddKeyDlg::SetScriptCode"); 
	scriptEditor->SetText(strScriptCode);
	if (strScriptCode==CStr(""))
		scriptEditor->AllowChanges(TRUE);
}

void ObjAddKeyDlg::SelChange()
{ FUNC_ENTER("ObjAddKeyDlg::SelChange"); 
	char buf[256];

	int index=SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETCURSEL,0,0);
	SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

	// Set the script name
	IEditFile->SetText(buf);

	// Set the script filename
	Link<CStr>* link=(Link<CStr>*)SendDlgItemMessage(hwnd,IDC_SCRIPTLIST,LB_GETITEMDATA,(WPARAM)index,0);
	scriptFile=link->data;

	// Load the script into the script editor window
	if (strlen(buf)>0)
	{
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->GoToScriptDefinition(buf);
		scriptEditor->AllowChanges(FALSE);
	}
	else
	{
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->SetText("");
	}
}

void ObjAddKeyDlg::SpinChange()
{ FUNC_ENTER("ObjAddKeyDlg::SpinChange"); 
	char     buf[256];
	int      val       = ISpin->GetIVal();
	Interval animrange = ip->GetAnimRange();
	
	sprintf(buf,"%f",((float)val-(animrange.Start()/GetTicksPerFrame()))/(float)GetFrameRate());

	SetDlgItemText(hwnd,IDC_TIME,buf);

	ip->SetTime(val*GetTicksPerFrame());
}

float ObjAddKeyDlg::GetTimeSec()
{ FUNC_ENTER("ObjAddKeyDlg::GetTimeSec"); 
	char buf[256];
	GetDlgItemText(hwnd,IDC_TIME,buf,255);

	return atof(buf);
}

/////////// ObjAnimExportDlg  //////////////////////////////////////////

FILE* fpDbgOutput;

ObjAnimExportDlg::ObjAnimExportDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip,ExportOptionsDlg* expopt) :
	ResizeDlgWindow(hInstance,MAKEINTRESOURCE(IDD_OBJANIMOPTIONS),hwndParent)
{ FUNC_ENTER("ObjAnimExportDlg::ObjAnimExportDlg"); 
	this->ip=ip;
	this->expopt = expopt;

	scriptEditor = new RTFEditor(hInstance,GetDlgItem(hwnd,IDC_RICHEDIT1));
	scriptEditor->InitScriptDB();

	// Assign cam specific defaults
	Interval interval=ip->GetAnimRange();
	objsettings_defaults.start          = interval.Start()/GetTicksPerFrame();
	objsettings_defaults.end            = interval.End()/GetTicksPerFrame();
	objsettings_defaults.fRotTolerance  = DEFAULT_COMPRESS;
	objsettings_defaults.fTranTolerance = DEFAULT_TCOMPRESS;

	lastItem=0;

	bLockobjList=false;
	bLockKeyList=false;
	bLockNodeSelect=false;
	bLockLoadSave=false;
	bLockSelChange=false;
	//bSwapCoordSystem=false;		// Swap Object export to game coordinate system
	bSwapCoordSystem=false;		// Swap Object export to game coordinate system

	lastCamNode=NULL;

	// Create the AddKey Dialog
	pAddKeyDlg=new ObjAddKeyDlg(hInstance,hwnd,ip);
	pAddKeyDlg->SetOKCB(AddKeyCB,this);

	// Register for notification on MAX Selection change
	// Install notification callback
#ifndef DISABLE_NOTIFICATIONS
	int rVal=RegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register Selection Callback","Object Exporter",MB_ICONSTOP|MB_OK);

	rVal = RegisterNotification(CloseExportDlg,this,NOTIFY_FILE_PRE_OPEN);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register open Callback","Object Exporter",MB_ICONSTOP|MB_OK);

	rVal = RegisterNotification(CloseExportDlg,this,NOTIFY_SYSTEM_PRE_RESET);

	if (!rVal)
		MessageBox(NULL,"Couldn't Register reset Callback","Object Exporter",MB_ICONSTOP|MB_OK);
#endif

	// Initialize ListView control
	lview=new ListView(GetDlgItem(hwnd,IDC_KEYLIST));
	objList=new ListView(GetDlgItem(hwnd,IDC_OBJLIST));

	objList->AddColumn("Object Name",100);
	objList->AddColumn("MAX Name",100);
	objList->AddColumn("Start",50);
	objList->AddColumn("End",50);

	lview->AddColumn("KeyFrame");
	lview->AddColumn("Script",150);
	lview->AddColumn("KeyTime");
	lview->AddColumn("ScriptFile",270);

	// Connect custom edit/spinners
	editQErr    = GetICustEdit(GetDlgItem(hwnd,IDC_QERR));
	editTErr    = GetICustEdit(GetDlgItem(hwnd,IDC_TERR));
	editStart   = GetICustEdit(GetDlgItem(hwnd,IDC_SFRAMEEDIT));
	editEnd     = GetICustEdit(GetDlgItem(hwnd,IDC_EFRAMEEDIT));
	editKeyTime = GetICustEdit(GetDlgItem(hwnd,IDC_KEYTIMEEDIT));
	editSearch  = GetICustEdit(GetDlgItem(hwnd,IDC_EDITSEARCH));

	spinQErr    = GetISpinner(GetDlgItem(hwnd,IDC_QERRS));
	spinTErr    = GetISpinner(GetDlgItem(hwnd,IDC_TERRS));
	spinStart   = GetISpinner(GetDlgItem(hwnd,IDC_SFRAMESPIN));
	spinEnd     = GetISpinner(GetDlgItem(hwnd,IDC_EFRAMESPIN));
	spinKeyTime = GetISpinner(GetDlgItem(hwnd,IDC_KEYTIMESPIN));

	// Attach spinners to the edit boxes
	spinQErr->SetLimits(-5.000000f,5.000000f);
	spinTErr->SetLimits(-5.000000f,5.000000f);
	spinQErr->SetScale(0.000001f);
	spinTErr->SetScale(0.000001f);
	
	spinKeyTime->SetLimits(0.0f,999999.0f);
	spinKeyTime->SetScale(1.0f);

	GetRange();

	spinQErr->LinkToEdit(GetDlgItem(hwnd,IDC_QERR),EDITTYPE_FLOAT);
	spinTErr->LinkToEdit(GetDlgItem(hwnd,IDC_TERR),EDITTYPE_FLOAT);
	
	spinStart->LinkToEdit(GetDlgItem(hwnd,IDC_SFRAMEEDIT),EDITTYPE_INT);
	spinEnd->LinkToEdit(GetDlgItem(hwnd,IDC_EFRAMEEDIT),EDITTYPE_INT);

	spinKeyTime->LinkToEdit(GetDlgItem(hwnd,IDC_KEYTIMEEDIT),EDITTYPE_INT);

	spinQErr->SetValue(DEFAULT_COMPRESS,TRUE);
	spinTErr->SetValue(DEFAULT_TCOMPRESS,TRUE);

	spinStart->SetValue(interval.Start()/GetTicksPerFrame(),TRUE);
	spinEnd->SetValue(interval.End()/GetTicksPerFrame(),TRUE);

	// Set Defaults
	CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);

	SetResizeModes(6,
				   IDC_QERR,		RSW_SCALE_X,	0,0,0,0,
				   IDC_TERR,		RSW_SCALE_X,	0,0,0,0,
				   IDC_SFRAMEEDIT,	RSW_SCALE_X,	0,0,0,0,
				   IDC_EFRAMEEDIT,	RSW_SCALE_X,	0,0,0,0,
				   IDC_PREFIX,		RSW_SCALE_X,	0,0,0,0,
				   IDC_EDITNAME,	RSW_SCALE_X,	0,0,0,0);

	GetObjSysData();
	//GetObjData();
	Show();
}

ObjAnimExportDlg::~ObjAnimExportDlg()
{ FUNC_ENTER("ObjAnimExportDlg::~ObjAnimExportDlg"); 
	// Unregister MAX selection change notification
	UnRegisterNotification(MAXSelChange,this,NOTIFY_SELECTIONSET_CHANGED);
	UnRegisterNotification(CloseExportDlg,this,NOTIFY_FILE_PRE_OPEN);
	UnRegisterNotification(CloseExportDlg,this,NOTIFY_SYSTEM_PRE_RESET);

	if (lview)
		delete lview;

	if (pAddKeyDlg)
		delete pAddKeyDlg;

	// Unregister MAX UI controls
	ReleaseISpinner(spinQErr);
	ReleaseISpinner(spinTErr);
	ReleaseISpinner(spinStart);
	ReleaseISpinner(spinEnd);
	ReleaseISpinner(spinKeyTime);
	ReleaseICustEdit(editQErr);
	ReleaseICustEdit(editTErr);
	ReleaseICustEdit(editStart);
	ReleaseICustEdit(editEnd);
	ReleaseICustEdit(editKeyTime);
	ReleaseICustEdit(editSearch);

	delete scriptEditor;

	//fprintf(fpDbgOutput,"Exit ::~ObjectExporter()\n");
	//fclose(fpDbgOutput);
}

void ObjAnimExportDlg::CloseExportDlg(void *param, NotifyInfo *info)
{ FUNC_ENTER("ObjAnimExportDlg::CloseExportDlg"); 
	OutputDebugString("HANDLER: CloseExportDlg\n");
	ObjAnimExportDlg* pthis = (ObjAnimExportDlg*)param;
	pthis->Hide();

	if (pthis->expopt)
	{
		// Emulate pressing of the OK button on the export options dialog
		SendMessage(pthis->expopt->GetHWND(),WM_COMMAND,MAKEWPARAM(IDOK,0),(LPARAM)GetDlgItem(pthis->expopt->GetHWND(),IDOK));
	}
}

void ObjAnimExportDlg::Show()
{ FUNC_ENTER("ObjAnimExportDlg::Show"); 
	LoadSettings();

	AcquireSelSet();

	GetObjSysData();
	//GetObjData();
	SelChange(TRUE);
	
	MSDlgWindow::Show();
}

void ObjAnimExportDlg::Hide()
{ FUNC_ENTER("ObjAnimExportDlg::Hide"); 
	if (selSet.Count()==1)
		SaveToNode(GetCurSystem(),selSet[0]);

	SaveSettings();
	MSDlgWindow::Hide();
}

void ObjAnimExportDlg::UpdateScript()
{ FUNC_ENTER("ObjAnimExportDlg::UpdateScript"); 
	int index = lview->GetSel();

	if (index!=-1)
	{
		Link<CStr>* link = (Link<CStr>*)lview->GetItemData(index);
		if (link)
			link->data = scriptEditor->GetText();
	}
}

BOOL ObjAnimExportDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("ObjAnimExportDlg::DlgProc"); 
	NMHDR* nmhdr;

	ResizeDlgWindow::DlgProc(hwnd,msg,wParam,lParam);

	switch(msg)
	{
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case EN_CHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_RICHEDIT1:
				UpdateScript();
				break;

			case IDC_EDITSEARCH:
				UpdateAllObjSel();
				break;
			}
			break;
		}
		break;
	}

	if (LOWORD(wParam)==IDC_RICHEDIT1)
		if (scriptEditor->ProcMessage(hwnd,msg,wParam,lParam))
			return TRUE;

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			;//EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_CLOSE:
		Hide();
		return TRUE;

	case WM_NOTIFY:
		nmhdr=(NMHDR*)lParam;

		switch(nmhdr->code)
		{
		// Always allow inlist editing of the labels
		case LVN_BEGINLABELEDIT:
			return FALSE;

		case LVN_ENDLABELEDIT:
			UpdateLabel((NMLVDISPINFO*)lParam);
			return TRUE;
		}

		switch(nmhdr->idFrom)
		{
		case IDC_OBJLIST:
			if (!bLockobjList)
			{
				switch(nmhdr->code)
				{
				case LVN_ITEMCHANGED:
					CamSelChange((LPNMLISTVIEW)lParam);
					return TRUE;
				}
			}

			return FALSE;

		case IDC_KEYLIST:
			switch(nmhdr->code)
			{
			case LVN_ITEMCHANGED:
				ListViewUpdate((LPNMLISTVIEW)lParam);
				return TRUE;
			}

			return FALSE;
		}

		return FALSE;

	case CC_SPINNER_CHANGE:
		switch(LOWORD(wParam))
		{
		case IDC_SFRAMESPIN:
		case IDC_EFRAMESPIN:
			if (!bLockSelChange)
				UpdateRange();
			return TRUE;

		case IDC_KEYTIMESPIN:
			UpdateKeyTime();
			return TRUE;
		}

		return FALSE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_ANIMSYSLIST:
			AnimSysSelChange();
			return TRUE;
		
		case IDC_ADDSYS:
			AddSystem();
			return TRUE;

		case IDC_REMOVESYS:
			RemoveSystem();
			return TRUE;

		case IDC_DONE:
			Hide();
			return TRUE;

		case IDC_EXPORT:
			if (!DoExport())
			{
				MessageBox(NULL,"Export Failed!","Object Exporter",MB_ICONSTOP|MB_OK);
			}
			else
			{
				//SendNetMessage();
			}

			return TRUE;

		case IDC_ADDOBJ:
			AddObj(GetCurSystem());
			return TRUE;

		case IDC_REMOVEOBJ:
			RemoveObj(GetCurSystem());
			return TRUE;

		case IDC_ADDSCRIPT:
			AddScript();
			return TRUE;

		case IDC_DELSCRIPT:
			DeleteScript();
			return TRUE;

		case IDC_GETRANGE:
			GetRange();
			return TRUE;

		case IDC_RESORT:
			ResortList();
			return TRUE;

		case IDC_SFRAMEEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateRange();
				return TRUE;
			}
			return FALSE;

		case IDC_EFRAMEEDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateRange();
				return TRUE;
			}
			return FALSE;

		case IDC_EDITNAME:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateName();
				return TRUE;
			}
		}
	}

	return FALSE;
}

void ObjAnimExportDlg::MAXSelChange(void *param,NotifyInfo *info)
{ FUNC_ENTER("ObjAnimExportDlg::MAXSelChange"); 
	OutputDebugString("HANDLER: MAXSelChange\n");
	ObjAnimExportDlg* pthis=(ObjAnimExportDlg*)param;
	
	if (!pthis->bLockNodeSelect)
		pthis->SelChange();
}

CStr ObjAnimExportDlg::GetSaveFile()
{ FUNC_ENTER("ObjAnimExportDlg::GetSaveFile"); 
	OPENFILENAME ofn;
	char filename[256]="";

	// Fill out the arduously long openfilename struct
	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=hwnd;
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Animation Files (*.ska)\0*.ska\0All Files (*.*)\0*.*\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=0;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Enter Filename to save object data as";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt="*.ska";
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);
	return CStr(filename);
}

void ObjAnimExportDlg::GetObjData(char* sysName)
{ FUNC_ENTER("ObjAnimExportDlg::GetObjData"); 
	bLockSelChange=true;
	objList->Clear();

	SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_RESETCONTENT,0,0);

	// Add all the nodes in the scene and add them to the object list
	INodeTab nodes;
	GetNodeList(nodes);

	int count = nodes.Count();

	for(int i=0;i<count;i++)
	{
		// Only add if the node is a LevelObject or GameObject
		CStr className;
		if (!nodes[i]->GetUserPropString(CStr("Class"),className))
			if (!nodes[i]->GetUserPropString(CStr("class"),className))
				continue;

		if (className != CStr("LevelObject") &&
			className != CStr("GameObject"))
			continue;

		int index = SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_ADDSTRING,0,(LPARAM)(char*)nodes[i]->GetName());

		ObjInfo newObjInfo;
		newObjInfo.node = nodes[i];
		Link<ObjInfo>* objlink = objDB.Add(&newObjInfo);

		SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)objlink);

		// Only add the object to the export list if it's flagged as such
		AppDataChunk* appdataFlag = nodes[i]->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);
		
		CStr name = nodes[i]->GetName();
		
		if (appdataFlag && appdataFlag->data)
		{
			LinkList<CStr> sysList;

			DecodeStringList(&sysList,appdataFlag->data,appdataFlag->length);

			if (sysList.Find(&CStr(sysName)))
				AddObj(objlink);
		}
	}

	objList->Sort(Sort);
	LoadObjSelection();
	bLockSelChange=false;
}

void ObjAnimExportDlg::GetObjSysData()
{ FUNC_ENTER("ObjAnimExportDlg::GetObjSysData"); 
	// Find all the animation systems used within the scene and add them to the
	// animation system list

	SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_RESETCONTENT,0,0);

	LinkList<CStr> systems;
	GetAnimSystems(&systems);

	Link<CStr>* curnode = systems.GetHead();

	while(curnode)
	{
		SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_ADDSTRING,0,(LPARAM)(char*)curnode->data);
		curnode = curnode->next;
	}
}

void ObjAnimExportDlg::AddObj(Link<ObjInfo>* objlink)
{ FUNC_ENTER("ObjAnimExportDlg::AddObj"); 
	// Acquire the object name and range
	ObjSpecificExportSettings* settings;
	AppDataChunk* appdata;
	INode* node = objlink->data.node;
	int index;

	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

	if (appdata)
	{
		char strStart[256],strEnd[256];

		settings=(ObjSpecificExportSettings*)appdata->data;
		_itoa(settings->start,strStart,10);
		_itoa(settings->end,strEnd,10);

		Interval interval=ip->GetAnimRange();

		if (strlen(strStart)==0)
			_itoa(interval.Start()*GetTicksPerFrame(),strStart,10);

		if (strlen(strEnd)==0)
			_itoa(interval.End()*GetTicksPerFrame(),strEnd,10);

		CStr name=node->GetName();

		index=objList->AddItem(settings->strUserName);
		objList->AddSubItem(node->GetName());
		objList->AddSubItem(strStart);
		objList->AddSubItem(strEnd);
	}
	else
	{
		char strStart[256],strEnd[256];

		Interval interval=ip->GetAnimRange();

		_itoa(interval.Start()/GetTicksPerFrame(),strStart,10);
		_itoa(interval.End()/GetTicksPerFrame(),strEnd,10);

		CStr name=node->GetName();

		index=objList->AddItem(node->GetName());
		objList->AddSubItem(node->GetName());
		objList->AddSubItem(strStart,2);
		objList->AddSubItem(strEnd,3);

		SetDlgItemText(hwnd,IDC_EDITNAME,(char*)node->GetName());
	}

	objList->SetItemData(index,(DWORD)objlink);
}

void ObjAnimExportDlg::SaveObjSelection()
{ FUNC_ENTER("ObjAnimExportDlg::SaveObjSelection"); 
	// Go through the list and mark the selection state of all objects
	int size=objList->GetCount();

	for(int i=0;i<size;i++)
	{
		Link<ObjInfo>* objLink=(Link<ObjInfo>*)objList->GetItemData(i);

		if (!objLink || !objLink->data.node)
			return;

		INode* node=objLink->data.node;

		node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID);
		bool* pSel=(bool*)malloc(sizeof(bool));

		*pSel=objList->IsSelected(i);
		node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID,sizeof(bool),pSel);
	}
}

void ObjAnimExportDlg::LoadObjSelection()
{ FUNC_ENTER("ObjAnimExportDlg::LoadObjSelection"); 
	// Go through the list and mark the selection state of all objects
	int size=objList->GetCount();

	for(int i=0;i<size;i++)
	{
		Link<ObjInfo>* objLink=(Link<ObjInfo>*)objList->GetItemData(i);

		if (!objLink || !objLink->data.node)
			return;

		INode* node=objLink->data.node;

		AppDataChunk* appdata;
		appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID);

		if (appdata && (*((bool*)appdata->data))==true)
			objList->Select(i);
		else
			objList->UnSelect(i);
	}
}

void ObjAnimExportDlg::CancelCB(void* pData)
{ FUNC_ENTER("ObjAnimExportDlg::CancelCB"); 
	ObjAnimExportDlg* pthis=(ObjAnimExportDlg*)pData;
	pthis->bAbort=true;
}

bool ObjAnimExportDlg::DoExport()
{ FUNC_ENTER("ObjAnimExportDlg::DoExport"); 
	SaveSettings();
	Apply();

	ObjectExporter* objExp = GetObjExporter();

	if (objExp->DoExport(vOBJECT_EXPORT))
		MessageBox(hwnd,"Export Completed.","Export Objects",MB_ICONWARNING|MB_OK);

	return true;
}

bool ObjAnimExportDlg::ObjectExists(char* name)
{ FUNC_ENTER("ObjAnimExportDlg::ObjectExists"); 
	CStr strName = name;
	strName.toLower();

	strName = ReplaceStr(strName,".ska","");

	int count = objList->GetCount();

	for(int i=-0;i<count;i++)
	{
		char bufName[256];
		objList->GetItem(i,bufName,255,0);

		CStr strBufName = bufName;
		strBufName.toLower();

		if (strBufName == strName)
			return true;
	}

	return false;
}

void ObjAnimExportDlg::Cleanup(char* path)
{ FUNC_ENTER("ObjAnimExportDlg::Cleanup"); 
	char curdir[1024];
	_getcwd(curdir, 1023);
	_chdir(path);

	_finddata_t fileinfo;

	long fhandle = _findfirst("*.ska",&fileinfo);

	if (!ObjectExists(fileinfo.name))
		DeleteFile(fileinfo.name);

	while(_findnext(fhandle,&fileinfo)==0)
	{
		if (!ObjectExists(fileinfo.name))
			DeleteFile(fileinfo.name);
	}

	_findclose(fhandle);
	_chdir(curdir);
}

void ObjAnimExportDlg::Archive(char* path)
{ FUNC_ENTER("ObjAnimExportDlg::Archive"); 
	char curdir[1024];
	_getcwd(curdir, 1023);
	_chdir(path);

	char zpath[256];
	CStr strZipName;
	
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	strZipName  = seo.m_SceneName;
	strZipName += "Cams.zip";

	char* args[5];

	DeleteFile(strZipName);

	sprintf( zpath, "%s..\\..\\..\\bin\\win32\\pkzip25.exe", path );
	args[0] = zpath;
	args[1] = "-add";
	args[2] = (char*)strZipName;
	//args[3] = "*.ska";
	args[4] = NULL;

	// Launch pkzip25 to archive the .ska files in the
	_finddata_t fileinfo;

	long fhandle = _findfirst("*.ska",&fileinfo);

	if (ObjectExists(fileinfo.name))
	{
		args[3] = fileinfo.name;
		//spawnvp( _P_WAIT, args[0], args );
		ExecuteCommand( args );
	}

	while(_findnext(fhandle,&fileinfo)==0)
	{
		if (ObjectExists(fileinfo.name))
		{
			args[3] = fileinfo.name;
			//spawnvp( _P_WAIT, args[0], args );
			ExecuteCommand( args );
		}
	}

	_chdir(curdir);
}

void ObjAnimExportDlg::SelChange(bool bNoApply)
{ FUNC_ENTER("ObjAnimExportDlg::SelChange"); 
	UpdateScript();

	if (bLockSelChange)
		return;

	if (!bNoApply)
		Apply();
	
	bLockobjList=true;

	// Unselect everything in the Object list
	objList->UnSelectAll();

	int numSelNodes=ip->GetSelNodeCount();

	for(int i=0;i<numSelNodes;i++)
	{
		INode* node=ip->GetSelNode(i);

		Object* obj=node->EvalWorldState(0).obj;

		// Highlight the selected Object
		// Scan through the listview window to find our node

		int count=objList->GetCount();
		for(int i=0;i<count;i++)
		{
			Link<ObjInfo>* link=(Link<ObjInfo>*)objList->GetItemData(i);

			if (link->data.node==node)
			{
				objList->Select(i);
				lastCamNode=node;
				break;
			}
		}
	}

	AcquireSelSet();
	if (selSet.Count()==1)
		LoadFromNode(GetCurSystem(),selSet[0]);
	else
	{
		scriptDB.Clear();
		lview->Clear();
		scriptEditor->SetText("");
	}

	bLockobjList=false;
}

void ObjAnimExportDlg::CamSelChange(NMLISTVIEW* pNotify)
{ FUNC_ENTER("ObjAnimExportDlg::CamSelChange"); 
	UpdateScript();

	if (bLockSelChange)
		return;

	if (pNotify->uChanged & LVIF_STATE)
	{
		Apply();
		bLockNodeSelect=true;

		// Select/Unselect Object that changed highlighted in the list
		Link<ObjInfo>* link;
		link=(Link<ObjInfo>*)objList->GetItemData(pNotify->iItem);

		if (objList->IsSelected(pNotify->iItem))
			ip->SelectNode(link->data.node,0);
		else
			ip->DeSelectNode(link->data.node);

		ip->ForceCompleteRedraw();

		AcquireSelSet();
		if (selSet.Count()==1)
			LoadFromNode(GetCurSystem(),selSet[0]);
		else
		{
			scriptDB.Clear();
			lview->Clear();
			scriptEditor->SetText("");
		}

		// Send a network update message if necessary
		/*
		if (selSet.Count()==1 && IsDlgButtonChecked(hwnd,IDC_SENDNET)==BST_CHECKED)
		{
			Net::MsgViewObjSetCamAnimFile msg;
			char bufName[256];
			char bufPrefix[256];
			GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

			objList->GetItem(pNotify->iItem,bufName,255);

			CStr sceneName = GetSceneName();
			CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
			CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

			// First verify that the animation actually exists
			FILE* fp;
			fp = fopen(absFileName,"rb");

			if (!fp)
			{
				int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected Object doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

				if (bResult == IDYES)
				{
					ExportSingleCam(objList->GetSelItem());
				}
				else
				{
					bLockNodeSelect=false;
					return;
				}
			}
			else
				fclose(fp);

			strcpy(msg.m_Filename,(char*)fileName);
			msg.m_checksum=GenerateCRC(bufName);

			gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ),
											 &msg );
		}
		*/
	}

	bLockNodeSelect=false;
}

void ObjAnimExportDlg::AddScript()
{ FUNC_ENTER("ObjAnimExportDlg::AddScript"); 
	if (selSet.Count()>1)
	{
		MessageBox(hwnd,"You may only have one Object selected when adding a script key.","Multiple Objects Selected",MB_ICONWARNING|MB_OK);
		return;
	}

	Interval interval=ip->GetAnimRange();

	pAddKeyDlg->Show();
	pAddKeyDlg->SetLimits((int)interval.Start()/GetTicksPerFrame(),(int)interval.End()/GetTicksPerFrame());
	pAddKeyDlg->SetScript("");
	pAddKeyDlg->SetScriptCode("");
}

void ObjAnimExportDlg::DeleteScript()
{ FUNC_ENTER("ObjAnimExportDlg::DeleteScript"); 
	int index=lview->GetSel();
	lview->DeleteItem(index);
}

void ObjAnimExportDlg::AddKeyCB(ObjAddKeyDlg* pDlg,void* pData)
{ FUNC_ENTER("ObjAnimExportDlg::AddKeyCB"); 
	char buf[256];

	ObjAnimExportDlg* pthis=(ObjAnimExportDlg*)pData;

	_itoa(pDlg->GetTime(),buf,10);

	int index = pthis->lview->AddItem(buf);

	CStr scriptName = pDlg->GetScript();

	if (scriptName == CStr("") ||
		scriptName == CStr(USERSCRIPT_NAME))
	{
		pthis->lview->AddSubItem(USERSCRIPT_NAME);

		float timesec=pDlg->GetTimeSec();
		sprintf(buf,"%f",timesec);

		pthis->lview->AddSubItem(buf);
		pthis->lview->AddSubItem("N/A");

		Link<CStr>* link = pthis->scriptDB.AddToTail(&pDlg->GetScriptCode());
		pthis->lview->SetItemData(index,(DWORD)link);
	}
	else
	{
		pthis->lview->AddSubItem((char*)pDlg->GetScript());

		float timesec=pDlg->GetTimeSec();
		sprintf(buf,"%f",timesec);

		pthis->lview->AddSubItem(buf);
		pthis->lview->AddSubItem((char*)pDlg->GetScriptFile());
	}
}

void ObjAnimExportDlg::ListViewUpdate(NMLISTVIEW* pNotify)
{ FUNC_ENTER("ObjAnimExportDlg::ListViewUpdate"); 
	if (pNotify->uNewState & LVIS_SELECTED)
	{
		char buf[256];

		lview->GetItem(pNotify->iItem,buf,255);

		// Set current animation frame
		ip->SetTime(atof(buf)*GetTicksPerFrame());
	}

	// Update the displayed keytime
	char keytime[256];
	lview->GetItem(pNotify->iItem,keytime,255,0);
	bLockKeyList = true;
	spinKeyTime->SetValue(atoi(keytime),TRUE);
	bLockKeyList = false;

	// Update the displayed script
	char filename[1024];
	lview->GetItem(pNotify->iItem,filename,1023,3);

	lview->SetSel(pNotify->iItem);

	if (CStr(filename) == CStr("N/A"))
	{
		Link<CStr>* link = (Link<CStr>*)lview->GetItemData(pNotify->iItem);
		scriptEditor->AllowChanges(TRUE);
		scriptEditor->SetText(link->data);
	}
	else
	{
		char scriptname[1024];
		lview->GetItem(pNotify->iItem,scriptname,1023,1);
		scriptEditor->GoToScriptDefinition(scriptname);
		scriptEditor->AllowChanges(FALSE);
	}
}

void ObjAnimExportDlg::SaveToNode(char* strSys,INode* node)
{ FUNC_ENTER("ObjAnimExportDlg::SaveToNode"); 
	if (bLockLoadSave)
		return;

	bLockLoadSave=true;

	if (!node)
	{
		bLockLoadSave=false;
		return;
	}

	CStr name=node->GetName();

	node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);
	//node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	ObjSpecificExportSettings* settings=(ObjSpecificExportSettings*)malloc(sizeof(ObjSpecificExportSettings));

	settings->fRotTolerance=editQErr->GetFloat();
	settings->fTranTolerance=editTErr->GetFloat();
	settings->start=editStart->GetInt();
	settings->end=editEnd->GetInt();
	
	char bufName[256];
	GetDlgItemText(hwnd,IDC_EDITNAME,bufName,255);
	strObjName=bufName;
	strcpy(settings->strUserName,(char*)strObjName);

	LinkList<CStr> listStream;

	// Read in any old data
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	Link<CStr>* linkSys = NULL;

	if (appdata && appdata->data)
	{
		DecodeStringList(&listStream,appdata->data,appdata->length);

		// Search for the ID of our anim system
		Link<CStr>* curanim = listStream.GetHead();

		while(curanim)
		{
			if (curanim->data == CStr(strSys))
			{
				linkSys = curanim->next;
				break;
			}

			if (!curanim->next)
				break;

			curanim = curanim->next->next;
		}

		node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);
	}

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID,sizeof(ObjSpecificExportSettings),settings);

	CStr strStream;
	char buf[2048];
	char frameNum[256];
	char scriptName[512];
	char scriptFile[1024];
	
	int count=lview->GetCount();

	// Build data stream
	for(int i=0;i<count;i++)
	{
		lview->GetItem(i,frameNum,255);
		lview->GetItem(i,scriptName,511,1);
		lview->GetItem(i,scriptFile,1023,3);

		sprintf(buf,"%s\n%s\n%i\n",scriptName,scriptFile,atoi(frameNum));
		strStream+=buf;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			Link<CStr>* bufLink = (Link<CStr>*)lview->GetItemData(i);

			char buf2[256];
			_itoa(bufLink->data.Length(),buf2,10);
			strcat(buf2,"\n");

			// Add length to the stream
			strStream+=buf2;
			strStream+=bufLink->data;
		}
	}

	// Now add the stream to the DB
	if (linkSys)
		linkSys->data = strStream;
	else
	{
		listStream.Add(&CStr(strSys));
		listStream.Add(&strStream);
	}

	int   len;
	void* pData = EncodeStringList(&listStream,&len);

	node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID,len,pData);

	bLockLoadSave=false;
}

void ObjAnimExportDlg::LoadFromNode(char* strSys,INode* node)
{ FUNC_ENTER("ObjAnimExportDlg::LoadFromNode"); 
	if (bLockLoadSave)
		return;

	bLockLoadSave=true;

	lview->Clear();
	scriptEditor->SetText("");
	scriptDB.Clear();

	if (!node)
	{
		bLockLoadSave=false;
		return;
	}

	AppDataChunk* appdata;

	ObjSpecificExportSettings* settings;
	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

	if (appdata)
		settings=(ObjSpecificExportSettings*)appdata->data;
	else
	{
		settings=&objsettings_defaults;
		strcpy(settings->strUserName,(char*)node->GetName());
	}

	spinQErr->SetValue(settings->fRotTolerance,TRUE);
	spinTErr->SetValue(settings->fTranTolerance,TRUE);

	spinStart->SetValue(settings->start,TRUE);
	spinEnd->SetValue(settings->end,TRUE);

	strObjName=settings->strUserName;
	SetDlgItemText(hwnd,IDC_EDITNAME,(char*)strObjName);
	SendDlgItemMessage(hwnd,IDC_EDITNAME,EM_SETSEL,(WPARAM)strObjName.Length(),(LPARAM)strObjName.Length());
	//objList->AddSubItem(settings->strUserName,0,lastItem);

	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	if (!appdata)
	{
		bLockLoadSave=false;
		return;
	}

	// Acquire the appropriate buffer and length for this animation system
	LinkList<CStr> listStream;
	DecodeStringList(&listStream,appdata->data,appdata->length);
	
	Link<CStr>* curanim = listStream.GetHead();
	Link<CStr>* linkSys = NULL;

	while(curanim)
	{
		if (curanim->data == CStr(strSys))
		{
			linkSys = curanim->next;
			break;
		}

		if (!curanim->next)
			break;

		curanim = curanim->next->next;
	}

	if (!linkSys)
	{
		bLockLoadSave=false;
		return;
	}

	int   len = linkSys->data.Length();
	char* buf = linkSys->data;

	int   frameNum;
	char  sframeNum[256];
	char  scriptName[512];
	char  scriptFile[1024];
	char  frameTime[256];
	int   extlen;				// Length of extended data for attached user scripts
	char  token[1024];
	int   pos = 0;

	while(1)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptFile,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);
		
		//sscanf(buf,"%s\n%s\n%i\n%i\n",scriptName,scriptFile,&frameNum,&extlen);

		// Compute frame time
		Interval animrange = ip->GetAnimRange();
		sprintf(frameTime,"%f",((float)frameNum-(animrange.Start()/GetTicksPerFrame()))/(float)GetFrameRate());
		sprintf(sframeNum,"%i",frameNum);

		// Add this entry to listview
		int index = lview->AddItem(sframeNum);
		lview->AddSubItem(scriptName,1);
		lview->AddSubItem(frameTime,2);
		lview->AddSubItem(scriptFile,3);

		//buf+=Instr(buf,"\n")+1;
		//buf+=Instr(buf,"\n")+1;
		//buf+=Instr(buf,"\n")+1;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			char* sbuf;
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);
			
			sbuf = new char[extlen+1];

			for(int i=0;i<extlen;i++)
				sbuf[i]=buf[pos++];

			sbuf[i]='\0';
			
			Link<CStr>* link = scriptDB.Add(&CStr(sbuf));
			lview->SetItemData(index,(DWORD)link);

			delete sbuf;
		}
	}

	bLockLoadSave=false;
}

void ObjAnimExportDlg::SaveSettings()
{ FUNC_ENTER("ObjAnimExportDlg::SaveSettings"); 
	ObjExportSettings* settings=(ObjExportSettings*)malloc(sizeof(ObjExportSettings));

	ReferenceTarget* scene=ip->GetScenePointer();
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG)==BST_CHECKED)
		settings->bDebug=true;
	else
		settings->bDebug=false;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION)==BST_CHECKED)
		settings->bCompress=true;
	else
		settings->bCompress=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS)==BST_CHECKED)
		settings->bNoDbls=true;
	else
		settings->bNoDbls=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTQN)==BST_CHECKED)
		settings->bExportQN=true;
	else
		settings->bExportQN=false;

	if (IsDlgButtonChecked(hwnd,IDC_EXPALL)==BST_CHECKED)
		settings->bExportAll=true;
	else
		settings->bExportAll=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPRESSEDTIME)==BST_CHECKED)
		// Always on from now on
		settings->bCompressTime=true;
//	else
//		settings->bCompressTime=false;

	if (IsDlgButtonChecked(hwnd,IDC_CLEANUP)==BST_CHECKED)
		settings->bCleanup=true;
	else
		settings->bCleanup=false;

	if (IsDlgButtonChecked(hwnd,IDC_ARCHIVE)==BST_CHECKED)
		settings->bArchive=true;
	else
		settings->bArchive=false;

	GetDlgItemText(hwnd,IDC_PREFIX,settings->strPrefix,255);

	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID,sizeof(ObjExportSettings),settings);

	SaveObjSelection();
}

void ObjAnimExportDlg::LoadSettings()
{ FUNC_ENTER("ObjAnimExportDlg::LoadSettings"); 
	ObjExportSettings* settings;
	AppDataChunk* appdata;

	ReferenceTarget* scene=ip->GetScenePointer();

	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);
	
	if (appdata)
	{
		settings=(ObjExportSettings*)appdata->data;

		if (settings->bDebug)
			CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_UNCHECKED);

		if (settings->bCompress)
			CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_UNCHECKED);

		if (settings->bNoDbls)
			CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_COMPDBLS,BST_UNCHECKED);

		if (settings->bExportQN)
			CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_UNCHECKED);

//		if (settings->bCompressTime)
//			CheckDlgButton(hwnd,IDC_COMPRESSEDTIME,BST_CHECKED);
//		else
//			CheckDlgButton(hwnd,IDC_COMPRESSEDTIME,BST_UNCHECKED);

		SetDlgItemText(hwnd,IDC_PREFIX,settings->strPrefix);

		if (settings->bExportAll)
			CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPALL);
		else
			CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPSEL);

		if (settings->bCleanup)
			CheckDlgButton(hwnd,IDC_CLEANUP,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_CLEANUP,BST_UNCHECKED);

		if (settings->bArchive)
			CheckDlgButton(hwnd,IDC_ARCHIVE,BST_CHECKED);
		else
			CheckDlgButton(hwnd,IDC_ARCHIVE,BST_UNCHECKED);
	}
	else
	{
		CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_COMPDBLS,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_OUTPUTQN,BST_CHECKED);		// Output QN set as default
		CheckRadioButton(hwnd,IDC_EXPSEL,IDC_EXPALL,IDC_EXPALL);
		
		CheckDlgButton(hwnd,IDC_CLEANUP,BST_UNCHECKED);
		CheckDlgButton(hwnd,IDC_ARCHIVE,BST_UNCHECKED);
	}
}

void ObjAnimExportDlg::AcquireSelSet()
{ FUNC_ENTER("ObjAnimExportDlg::AcquireSelSet"); 
	int numNodes=ip->GetSelNodeCount();

	selSet.ZeroCount();

	for(int i=0;i<numNodes;i++)
	{
		INode* node=ip->GetSelNode(i);
		selSet.Append(1,&node);
	}
}

void ObjAnimExportDlg::Apply()
{ FUNC_ENTER("ObjAnimExportDlg::Apply"); 
	int count=selSet.Count();

	if (count>1)
	{
		scriptDB.Clear();
		lview->Clear();
		return;
	}

	for(int i=0;i<count;i++)
	{
		// Apply the listview contents (scripted keys) to each node
		SaveToNode(GetCurSystem(),selSet[i]);
	}
}

void ObjAnimExportDlg::GetRange()
{ FUNC_ENTER("ObjAnimExportDlg::GetRange"); 
	Interval interval=ip->GetAnimRange();
	spinStart->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());
	spinEnd->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());
}

void ObjAnimExportDlg::UpdateRange()
{ FUNC_ENTER("ObjAnimExportDlg::UpdateRange"); 
	// Determine current selection
	int count=objList->GetCount();
	//int index;
	int selCount=0;

	/*
	for(int i=0;i<count;i++)
	{
		if (objList->IsSelected(i))
		{
			index=i;
			selCount++;
		}
	}

	if (selCount>1)
	{
		MessageBox(hwnd,"You can only update the frame range of one Object at a time.","Multiple Objects Selected",MB_ICONWARNING|MB_OK);
		return;
	}
	*/

	char strStart[256];
	char strEnd[256];

	int start=editStart->GetInt();
	int end=editEnd->GetInt();

	_itoa(start,strStart,10);
	_itoa(end,strEnd,10);

	// This now affects the entire system
	for(int i = 0; i < count; i++)
	{
		//if (objList->IsSelected(i))
		{
			objList->AddSubItem(strStart, 2, i);
			objList->AddSubItem(strEnd, 3, i);
		}
	}
}

void ObjAnimExportDlg::UpdateLabel(NMLVDISPINFO* dispinfo)
{ FUNC_ENTER("ObjAnimExportDlg::UpdateLabel"); 
	int index=dispinfo->item.iItem;

	if (dispinfo->item.pszText)
	{
		objList->AddSubItem(dispinfo->item.pszText,0,index);
		strObjName=dispinfo->item.pszText;
		lastItem=index;
		SetDlgItemText(hwnd,IDC_EDITNAME,dispinfo->item.pszText);
	}
}

void ObjAnimExportDlg::UpdateName()
{ FUNC_ENTER("ObjAnimExportDlg::UpdateName"); 
	int index=objList->GetSelItem();
	char buf[256];
	GetDlgItemText(hwnd,IDC_EDITNAME,buf,255);

	objList->AddSubItem(buf,0,index);
	int buflen=strlen(buf);
}

int ObjAnimExportDlg::Sort(DWORD dw1,DWORD dw2,ListView* pList,void* pExtData)
{ FUNC_ENTER("ObjAnimExportDlg::Sort"); 
	Link<ObjInfo>* objLink1=(Link<ObjInfo>*)dw1;
	Link<ObjInfo>* objLink2=(Link<ObjInfo>*)dw2;

	if (!objLink1 || !objLink2)
		return 0;

	INode* node1=objLink1->data.node;
	INode* node2=objLink2->data.node;

	AppDataChunk *appdata1, *appdata2;

	appdata1=node1->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);
	appdata2=node2->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

	if (!appdata1 || !appdata2)
		return 0;
	
	ObjSpecificExportSettings* settings1=(ObjSpecificExportSettings*)appdata1->data;
	ObjSpecificExportSettings* settings2=(ObjSpecificExportSettings*)appdata2->data;
	
	CStr strName1=settings1->strUserName;
	CStr strName2=settings2->strUserName;

	// Switch both to lower case so sorting is case-insensitive
	strName1.toLower();
	strName2.toLower();

	if (strName1<strName2)
		return -1;

	if (strName1==strName2)
		return 0;

	return 1;
}

void ObjAnimExportDlg::ResortList()
{ FUNC_ENTER("ObjAnimExportDlg::ResortList"); 
	Apply();
	objList->Sort(Sort);
}

bool ObjAnimExportDlg::ExportSingleCam(int i)
{ FUNC_ENTER("ObjAnimExportDlg::ExportSingleCam"); 
	/*
	FILE* fpQN = NULL;

	if (i==-1)
		return false;

	Apply();

	// Get UI Settings //////////////////////////////////////////////////////
	bool bCompress, bOneFrame, bDebug, bCompressedTime;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION)==BST_CHECKED)
		bCompress=true;
	else
		bCompress=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS)==BST_CHECKED)
		bOneFrame=true;
	else
		bOneFrame=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG)==BST_CHECKED)
		bDebug=true;
	else
		bDebug=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPRESSEDTIME)==BST_CHECKED)
		// Always on from now on
		bCompressedTime=true;
//	else
//		bCompressedTime=false;

	float errorQ=editQErr->GetFloat();
	float errorT=editTErr->GetFloat();

	/////////////////////////////////////////////////////////////////////////

	bAbort=false;

	// Build path
	CStr strSceneName=GetSceneName();
	char bufPrefix[256];
	char bufName[256];
	CStr strFile;
	CStr strPath=getenv(APP_ENV);
	strPath+=CStr(CAM_PATH)+strSceneName;

	// Attempt to create the directory (incase it doesn't exist)
	_mkdir(strPath);

	strPath+="\\";
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);
	strPath+=bufPrefix;

	objList->GetItem(i,bufName,255,0);

	strFile=strPath+CStr(bufName)+CStr(".ska");

	// Get start and end range for animation
	char strStart[256],strEnd[256];

	objList->GetItem(i,strStart,255,2);
	objList->GetItem(i,strEnd,255,3);

	if (atoi(strEnd)<atoi(strStart))
	{
		delete progBar;

		if (fpQN)
			fclose(fpQN);

		MessageBox(hwnd,"Your end export frame is less than your start export frame.\nOperation Aborted.","Export Frames Invalid",MB_ICONWARNING|MB_OK);
		return false;
	}

	ObjInfo* pObjInfo=&(((Link<ObjInfo>*)objList->GetItemData(i))->data);

	if (!Export(strFile,pObjInfo,atoi(strStart),atoi(strEnd),
			    errorQ,errorT,
				bCompress,bOneFrame,bDebug,bCompressedTime))
	{
		if (fpQN)
			fclose(fpQN);

		delete progBar;
		return false;
	}

	if (fpQN)
		fprintf(fpQN,"\tLoadAsset \"%s\" %s\n",(char*)strFile,bufName);

	return true;
	*/

	return true;
}

void ObjAnimExportDlg::SendNetMessage(int i)
{ FUNC_ENTER("ObjAnimExportDlg::SendNetMessage"); 
	bLockNodeSelect = true;

	if (i == -1)
		i = objList->GetSelItem();

	Net::MsgViewObjSetCamAnimFile msg;
	char bufName[256];
	char bufPrefix[256];
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

	objList->GetItem(i,bufName,255);

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	// First verify that the animation actually exists
	FILE* fp;
	fp = fopen(absFileName,"rb");

	if (!fp)
	{
		int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected Object doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

		if (bResult == IDYES)
		{
			ExportSingleCam(objList->GetSelItem());
		}
		else
		{
			bLockNodeSelect=false;
			return;
		}
	}
	else
		fclose(fp);

	strcpy(msg.m_Filename,(char*)fileName);
	msg.m_checksum=GenerateCRC(bufName);

	//gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ), &msg,
	//								 Net::NORMAL_PRIORITY, 	Net::QUEUE_SEQUENCED );

	Net::MsgDesc msgdesc;
	msgdesc.m_Id       = Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE;
	msgdesc.m_Length   = sizeof( Net::MsgViewObjSetCamAnimFile );
	msgdesc.m_Data     = &msg;
	msgdesc.m_Priority = Net::NORMAL_PRIORITY;
	msgdesc.m_Queue    = Net::QUEUE_DEFAULT;
	msgdesc.m_GroupId  = Net::GROUP_ID_DEFAULT;
	msgdesc.m_Singular = false;
	msgdesc.m_Delay    = i * 250;

	gClient->EnqueueMessageToServer( &msgdesc );
	
	bLockNodeSelect = false;
}

void ObjAnimExportDlg::BuildTargetFile(int i)
{ FUNC_ENTER("ObjAnimExportDlg::BuildTargetFile"); 
	bLockNodeSelect = true;

	if (i == -1)
		i = objList->GetSelItem();

	char bufName[256];
	char bufPrefix[256];
	GetDlgItemText(hwnd,IDC_PREFIX,bufPrefix,255);

	objList->GetItem(i,bufName,255);

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(bufPrefix)+CStr(bufName)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	// First verify that the animation actually exists
	FILE* fp;
	fp = fopen(absFileName,"rb");

	if (!fp)
	{
		int bResult = MessageBox(gInterface->GetMAXHWnd(),"The selected Object doesn't exist.  Would you like to export it?","Couldn't send network update",MB_ICONQUESTION|MB_YESNO);

		if (bResult == IDYES)
		{
			ExportSingleCam(objList->GetSelItem());
		}
		else
		{
			bLockNodeSelect=false;
			return;
		}
	}
	else
		fclose(fp);

	// Execute animation converter
	char  ancPath[256];
	char* args[4];
	CStr  strFileCmd = CStr("-f") + absFileName;
	sprintf( ancPath, "%s\\bin\\win32\\animconv.exe", getenv(APP_ENV) );

	args[0] = ancPath;
	args[1] = strFileCmd;
	switch( GetQuickviewPlatform())
	{
		case vQUICKVIEW_XBOX:
			args[2] = "-px";
			break;
		case vQUICKVIEW_PS2:
			args[2] = "-pp";
			break;
		case vQUICKVIEW_NGC:
			args[2] = "-pg";
			break;
		case vQUICKVIEW_PCVIEW:
			args[2] = "-pc";
			break;
	}
	args[3] = NULL;

	//int result = ( spawnvp( _P_WAIT, ancPath, args ) == 0 );
	int result = ExecuteCommand( args );
	
	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];

		sprintf( path, "%s.xbx", absFileName );
		sprintf( copy_path, "%s\\bin\\win32\\copyx.bat", getenv(APP_ENV) );
		args[0] = copy_path;
		args[1] = path;
		args[2] = NULL;

		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );
	}
	else if( GetQuickviewPlatform() == vQUICKVIEW_NGC )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];
		char* proj_root, *path_ptr;

		proj_root = getenv( APP_ENV );

		sprintf( path, "%s.ngc", absFileName );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );
	}
	bLockNodeSelect = false;
}

void ObjAnimExportDlg::UpdateKeyTime()
{ FUNC_ENTER("ObjAnimExportDlg::UpdateKeyTime"); 
	if (bLockKeyList)
		return;

	int sel = lview->GetSel();

	if (sel==-1)
		return;

	int val = spinKeyTime->GetIVal();
	char buf[256];
	sprintf(buf,"%i",val);
	lview->AddSubItem(buf,0,sel);
}

void ObjAnimExportDlg::AddObj(char* sysName)
{ FUNC_ENTER("ObjAnimExportDlg::AddObj"); 
	// Go through the object list and add everything selected
	int count = SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<ObjInfo>* objlink = (Link<ObjInfo>*)SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_GETITEMDATA,(WPARAM)i,0);
			
			if (objlink->data.node)
			{
				LinkList<CStr> sysList;

				// Get the appdata chunk and add
				AppDataChunk* appdata = objlink->data.node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

				if (appdata && appdata->data)
					DecodeStringList(&sysList,appdata->data,appdata->length);

				// Don't add if the entry already exists
				Link<CStr>* link = sysList.Find(&CStr(sysName));

				if (link)
					continue;

				// Add marker flag
				sysList.Add(&CStr(sysName));
				
				int   len;
				void* pData = EncodeStringList(&sysList,&len);

				objlink->data.node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);
				objlink->data.node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED,len,pData);

				// Add data to the object list only if it doesn't already exist
				if (objList->Find(objlink->data.node->GetName(),1)  == -1)
					AddObj(objlink);
			}
		}
	}
}

void ObjAnimExportDlg::RemoveObj(char* sysName)
{ FUNC_ENTER("ObjAnimExportDlg::RemoveObj"); 
	int nObjs = objList->GetCount();

	for(int i=0;i<nObjs;i++)
	{
		if (objList->IsSelected(i))
		{
			Link<ObjInfo>* objlink = (Link<ObjInfo>*)objList->GetItemData(i);
			objlink->data.node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

			objList->DeleteItem(i);
			i=0;						// Restart from begining as deleting an item will reorder the list
		}
	}
}

void ObjAnimExportDlg::UpdateAllObjSel()
{ FUNC_ENTER("ObjAnimExportDlg::UpdateAllObjSel"); 
	char wildcard[256];
	char buf[256];
	editSearch->GetText(wildcard,255);

	strcat(wildcard,"*");

	int count = SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_GETTEXT,(WPARAM)i,(LPARAM)buf))
			if (MatchPattern(CStr(buf),CStr(wildcard)))
				SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)i);
			else
				SendDlgItemMessage(hwnd,IDC_ALLOBJLIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);
	}
}

CStr ObjAnimExportDlg::GetCurSystem()
{ FUNC_ENTER("ObjAnimExportDlg::GetCurSystem"); 
	char buf[256];
	int index = SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_GETCURSEL,0,0);

	SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

	return CStr(buf);
}

void ObjAnimExportDlg::AddSystem()
{ FUNC_ENTER("ObjAnimExportDlg::AddSystem"); 
	// Open system name dialog to name the new system
	SystemNameDlg* sysNameDlg = new SystemNameDlg(hInstance,hwnd);

	sysNameDlg->Show();
	CStr name = sysNameDlg->GetName();

	delete sysNameDlg;

	int index = SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_ADDSTRING,0,(LPARAM)(char*)name);
	SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_SETCURSEL,(WPARAM)index,0);

	GetObjData(name);
}

void ObjAnimExportDlg::RemoveSystem(CStr sysName,INode* root)
{ FUNC_ENTER("ObjAnimExportDlg::RemoveSystem"); 
	// Go through all the nodes and remove any references to the given system
	if (root == NULL)
	{
		root = ip->GetRootNode();
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		RemoveSystem(sysName,child);
	}

	if (root == ip->GetRootNode())
		return;

	LinkList<CStr> sysList;

	AppDataChunk* appdata = root->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

	if (appdata && appdata->data)
	{
		LinkList<CStr> sysList;
		DecodeStringList(&sysList,appdata->data,appdata->length);

		Link<CStr>* link = sysList.Find(&sysName);
		
		// If this node is part of the given animation system, remove it
		if (link)
			sysList.Remove(link);

		// Now get rid of the data chunk and reencode
		root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);

		int len;
		void* pData = EncodeStringList(&sysList,&len);

		root->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED,len,pData);
	}

	// In addition to removing the EXPORTFLAGGED chunk data for the system
	// we also need to remove the existance of any scripts related to the system from
	// the vNAPP_OBJECT_EXPORT_ID chunk
	appdata = root->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	if (appdata && appdata->data)
	{
		LinkList<CStr> listStream;
		DecodeStringList(&listStream,appdata->data,appdata->length);

		Link<CStr>* curlink = listStream.GetHead();

		while(curlink)
		{
			if (curlink->data == CStr(sysName))
			{
				// Found a script chunk for the system being removed
				// Dump it!
				if (curlink->next)
					listStream.Remove(curlink->next);

				listStream.Remove(curlink);
				break;
			}

			if (!curlink->next)
				break;

			curlink = curlink->next->next;
		}
	}
}

void ObjAnimExportDlg::RemoveSystem()
{ FUNC_ENTER("ObjAnimExportDlg::RemoveSystem"); 
	int index = SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_GETCURSEL,0,0);
	
	if (index == -1)
		return;

	CStr sysName = GetCurSystem();
	RemoveSystem(sysName);

	// Remove the system from the system list
	SendDlgItemMessage(hwnd,IDC_ANIMSYSLIST,LB_DELETESTRING,(WPARAM)index,0);
}

void ObjAnimExportDlg::AnimSysSelChange()
{ FUNC_ENTER("ObjAnimExportDlg::AnimSysSelChange"); 
	Apply();
	CStr name = GetCurSystem();
	GetObjData(name);
	objList->UnSelectAll();
	lview->UnSelectAll();
	ip->ClearNodeSelection();
	lview->Clear();
}

///////////////////// Object Exporter
ObjectExporter::ObjectExporter()
{ FUNC_ENTER("ObjectExporter::ObjectExporter"); 
	bSwapCoordSystem=false;		// Swap Object export to game coordinate system
	bNoParentTransform=true;	// Objects should not be exported with respect to their parents
}

ObjectExporter::~ObjectExporter()
{ FUNC_ENTER("ObjectExporter::~ObjectExporter"); 
}

bool ObjectExporter::ExportNode(FILE* fpExport,FILE* fpDebug,INode* node,int numExtFrames,
								float errorQ,float errorT, bool bUseCompression, bool bCompressTime, bool bOneFrame)
{ FUNC_ENTER("ObjectExporter::ExportNode"); 
	int fps3DS=GetFrameRate();								// Frames per second MAX is set to
	Interval animRange    = gInterface->GetAnimRange();

	int num3DSFrames = end-start;
	int reqFrames    = num3DSFrames*OUTPUT_FPS/fps3DS;		// Number of frames at the game's frame rate

	if (skeleton)
		delete skeleton;
	//skeleton=new CSkeletonData(node,false);
	rootbone=node;

	this->bCompressTime = bCompressTime;

	INode* root=node;
	//int numNodes=skeleton->GetCount();
	int numNodes=skeleton->GetCount();

	GetAnim(root,start,end,errorQ,errorT,NULL,bUseCompression);
	
	if (bOneFrame)
		RemoveDblKeys(numNodes,num3DSFrames);

	// Compute total QFrames and TFrames
	int QKeys=0;
	int TKeys=0;
	int i;

	for(i=0;i<numNodes;i++)
	{
		QKeys+=numCompressedQFrames[i];
		TKeys+=numCompressedTFrames[i];		
	}
	
	// Output the stored data in THPS4 format
	//CSkeletonData* skeleton=new CSkeletonData(node);

	// Output file header
	SHeader header;
	header.version=ANIMFILE_VERSION;
	header.flags=INTERMEDIATE_FORMAT | OBJECTANIM;
	
	if (!bUseCompression)
		header.flags|=UNCOMPRESSED_FORMAT;

	if (bCompressTime)
		header.flags|=COMPRESSEDTIME;

	// Objects always had pre rotated root, assign the flag for it
	if (bRotateRoot)
		header.flags|=PREROTATEDROOT;

	header.flags|=CONTAINS_CAMDATA;

	header.skeletonName=GenerateCRC("OBJECT");			// hard-coded default for now
	header.duration=(float)num3DSFrames/(float)fps3DS;
	header.numBones=numNodes;
	header.numQKeys=QKeys;
	header.numTKeys=TKeys;

	header.numUserDefinedKeys=numExtFrames;

	if (!header.Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	// Output using CSkeleton instead
	if (!skeleton->Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	///////////////////////////////////////////////////

	// Output compressed rotation data
	SQData sqdata(numNodes,QKeys);
	int QKeyPos=0;

	for(i=0;i<numNodes;i++)
	{
		sqdata.numKeysPerBone[i]=numCompressedQFrames[i];
		
		for(int j=0;j<numCompressedQFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
				float ftmp = (compQframes[curFrame].time * (float)OUTPUT_FPS);
				char buf[256];
				sprintf(buf,"Pre float is: %f\n",ftmp);
				OutputDebugString(buf);
				*val = Round(ftmp);
				
				//if ( (*val % 2) == 1 )
				//	assert(0);
			}
			else
				sqdata.theQKeys[QKeyPos].time = compQframes[curFrame].time;

			sqdata.theQKeys[QKeyPos].qx   = compQframes[curFrame].q[X];
			sqdata.theQKeys[QKeyPos].qy   = compQframes[curFrame].q[Y];
			sqdata.theQKeys[QKeyPos].qz   = compQframes[curFrame].q[Z];
			sqdata.theQKeys[QKeyPos].real = compQframes[curFrame].q[W];

			if (fpDebug)
			{
				if (bCompressTime)
				{
					char sbuf[256];
					sprintf(sbuf,"---> Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
					OutputDebugString(sbuf);
					
					sprintf(sbuf,"---> actual time: %f\n",compQframes[curFrame].time);
					sprintf(sbuf,"---> new time: %f\n",compQframes[curFrame].time * 60.0f);
					OutputDebugString(sbuf);
				}
				else
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%f\n",i,j,compQframes[curFrame].time);

				fprintf(fpDebug,"Q[node=%i frame=%i].qx=%f\n",i,j,compQframes[curFrame].q[X]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qy=%f\n",i,j,compQframes[curFrame].q[Y]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qz=%f\n",i,j,compQframes[curFrame].q[Z]);
				fprintf(fpDebug,"Q[node=%i frame=%i].real=%f\n\n",i,j,compQframes[curFrame].q[W]);
			}

			QKeyPos++;
		}
	}

	if (!sqdata.Write(fpExport))
		return false;

	// Output compressed translation data
	STData stdata(numNodes,TKeys);
	int TKeyPos=0;

	for(i=0;i<numNodes;i++)
	{
		stdata.numKeysPerBone[i]=numCompressedTFrames[i];
		
		for(int j=0;j<numCompressedTFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&stdata.theTKeys[TKeyPos].time;
				float ftmp = compTframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);

				//if ( *val % 2 == 1 )
				//	assert(0);
			}
			else
				stdata.theTKeys[TKeyPos].time = compTframes[curFrame].time;

			stdata.theTKeys[TKeyPos].tx   = compTframes[curFrame].t[0];
			stdata.theTKeys[TKeyPos].ty   = compTframes[curFrame].t[1];
			stdata.theTKeys[TKeyPos].tz   = compTframes[curFrame].t[2];

			if (fpDebug)
			{
				if (bCompressTime)
					fprintf(fpDebug,"T[node=%i frame=%i].time=%i\n",i,j,*((int*)&stdata.theTKeys[TKeyPos].time));
				else
					fprintf(fpDebug,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);

				fprintf(fpDebug,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				fprintf(fpDebug,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				fprintf(fpDebug,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);

				char test[256];
				sprintf(test,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				OutputDebugString(test);
				sprintf(test,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);
				OutputDebugString(test);
			}

			TKeyPos++;
		}
	}

	if (!stdata.Write(fpExport))
		return false;

	delete skeleton;
	skeleton=NULL;

	return true;
}

int ObjectExporter::GetKeyCount(char* buf)
{ FUNC_ENTER("ObjectExporter::GetKeyCount"); 
	int   len=strlen(buf);

	int   frameNum;
	int   extlen;				// Length of extended data for attached user scripts
	char  token[1024];
	int   pos = 0;
	char  scriptName[1024];
	int   count = 0;

	while(1)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;
	
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);
		
		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);		
			pos+=extlen;
		}

		count++;
	}

	return count;
}

//MCozzini 7-4-03
//purpose of this method is to move to the correct position where the key scriot are stored
//otherwise 0 key scripts are returned everytime
char* ObjectExporter::MoveToKeyScriptData(char* source)
{ FUNC_ENTER("ObjectExporter::MoveToKeyScriptData"); 
	int num;
	//get the numer of bytes to skip
	memcpy(&num,source,sizeof(int));
	
	//add the number of bytes to skip plus 2 time the space of an INT, so it's possible to read
	//the info of the key scripts
	return (char*)  (source + num + 2*sizeof(int));

}

int ObjectExporter::GetNumScriptKeys(INode* node)
{ FUNC_ENTER("ObjectExporter::GetNumScriptKeys"); 
	AppDataChunk* appdata;
	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	if (!appdata)
		return 0;

	//MCozzini 7-4-03
	char* buf=MoveToKeyScriptData((char*)appdata->data);
	return GetKeyCount(buf);
}

GenKey<ObjScriptKey>* ObjectExporter::GetObjScriptKeys(INode* node,int* numKeys)
{ FUNC_ENTER("ObjectExporter::GetObjScriptKeys"); 
	GenKey<ObjScriptKey>* ObjScriptKeys;

	// Get the node for the current Object
	if (!node)
	{
		if (numKeys)
			*numKeys=0;

		return NULL;
	}
	
	AppDataChunk* appdata;
	appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

	if (!appdata)
	{
		if (numKeys)
			*numKeys=0;

		return NULL;
	}

	int   len=appdata->length;

	//MCozzini 7-4-03
	char* buf=MoveToKeyScriptData((char*)appdata->data);

	int   frameNum;
	char  scriptName[512];
	char  scriptFile[1024];
	char  token[1024];

	int   numItems=GetKeyCount(buf);
	int   extlen;
	int   pos = 0;

	if (numKeys)
		*numKeys=numItems;

	if (numItems==0)
		return NULL;

	ObjScriptKeys=new GenKey<ObjScriptKey>[numItems];

	for(int i=0;i<numItems;i++)
	{
		if (!GetToken(buf,&pos,token,"\n"))
			break;

		strcpy(scriptName,token);
		
		if (!GetToken(buf,&pos,token,"\n"))
			break;
	
		strcpy(scriptFile,token);

		if (!GetToken(buf,&pos,token,"\n"))
			break;

		frameNum = atoi(token);

		// Compute frame time
		Interval animrange = gInterface->GetAnimRange();

		ObjScriptKeys[i].time = (frameNum-(animrange.Start()/GetTicksPerFrame())/(float)GetFrameRate());
		
		ObjScriptKeys[i].key.name   =  scriptName;
		ObjScriptKeys[i].key.file   =  scriptFile;
		ObjScriptKeys[i].key.frame  =  frameNum;

		if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
		{
			GetToken(buf,&pos,token,"\n");
			extlen = atoi(token);		
			pos+=extlen;
		}
	}

	return ObjScriptKeys;
}

/*
bool ObjectExporter::Export(char* filename,LinkList<ObjInfo>* objDB,int start,int end,
							float errorQ,float errorT,
							bool bCompress,bool bOneFrame, bool bDebug, bool bCompressTime)
{
	FILE *fpExport=NULL,*fpDebug=NULL;
	bool bRemoveAll=false;

	// Export the Object
	if (ObjInfo)
	{
retry:
		fpExport=fopen(filename,"wb");

		if (!fpExport)
		{
			// Check the file attributes to see if the file is read-only
			if (_access(filename,WRITE_ACCESS)!=0)
			{
				if (bRemoveAll)
					goto removeFlag;

				char ErrMsg[256];
				int  rVal;
				sprintf(ErrMsg,"The file '%s' does not have write access enabled.\nWould you like to manually remove the Read-Only flag on all items?\n",(char*)filename);
				rVal=MessageBox(gInterface->GetMAXHWnd(),ErrMsg,"Export Object",MB_YESNOCANCEL|MB_ICONWARNING);

				if (rVal==IDYES)
				{
					bRemoveAll=true;

removeFlag:
					if (_chmod(filename, _S_IREAD | _S_IWRITE)==-1)
					{
						MessageBox(NULL,"An error occurred while changing the file's permission settings!","Export Object",MB_ICONSTOP|MB_OK);
						return false;
					}

					goto retry;
				}

				if (rVal==IDNO)
					goto removeFlag;
				
				return false;
			}

			return false;
		}

		// Acquire the script keys
		GenKey<ObjScriptKey>* ObjScriptKeys;
		int numObjScriptKeys;
		int numExtKeys;

		ObjScriptKeys = GetObjScriptKeys(ObjInfo,&numObjScriptKeys);
		numExtKeys = numObjScriptKeys;

		// Open debug file if requested
		if (bDebug)
			fpDebug=fopen("c:\\debug.txt","w");

		// Export the Object position data
		this->start = start;
		this->end   = end+1;

		if (!ExportNode(fpExport,fpDebug,ObjInfo->node,numExtKeys,
			            errorQ, errorT, bCompress, bCompressTime, bOneFrame))
		{
			fclose(fpExport);

			if (ObjScriptKeys)
				delete [] ObjScriptKeys;

			return false;
		}

		// Now write out the change script keys
		for(int i=0;i<numObjScriptKeys;i++)
		{
			if (bCompressTime)
			{
				// Script time was always the keyframe
				// represent as int instead of float
				int* val = (int*)&ObjScriptKeys[i].time;
				*val = ObjScriptKeys[i].time;
			}

			if (fpDebug)
			{
				fprintf(fpDebug,"%i\n",i);

				if (bCompressTime)
					fprintf(fpDebug,"Time: %i\n",*((int*)&ObjScriptKeys[i].time));
				else
					fprintf(fpDebug,"Time: %f\n",ObjScriptKeys[i].time);

				fprintf(fpDebug,"Name: %s (0x%x)\n",(char*)ObjScriptKeys[i].key.name,GenerateCRC((char*)ObjScriptKeys[i].key.name));

				char buf[256];
				sprintf(buf,"Time: %f\n",ObjScriptKeys[i].time);
				OutputDebugString(buf);

				sprintf(buf,"Name: %s (0x%x)\n",(char*)ObjScriptKeys[i].key.name,GenerateCRC((char*)ObjScriptKeys[i].key.name));
				OutputDebugString(buf);
			}

			CStr name;

			if (ObjScriptKeys[i].key.name == CStr(USERSCRIPT_NAME))
			{
				char buf[256];
				_splitpath(filename,NULL,NULL,buf,NULL);
				name += buf;
				name += "Frame";
				sprintf(buf,"%i",ObjScriptKeys[i].key.frame);
				name += buf;
			}
			else
			{
				name = ObjScriptKeys[i].key.name;
			}

			unsigned long crc = GenerateCRC((char*)name);

			if (!RunScriptKey(ObjScriptKeys[i].time,GenerateCRC((char*)name)).Write(fpExport))
			{
				if (ObjScriptKeys)
					delete [] ObjScriptKeys;

				return false;
			}
		}

		// Close debug file
		if (fpDebug)
			fclose(fpDebug);

		fclose(fpExport);

		// Free allocated keys
		if (ObjScriptKeys)
			delete [] ObjScriptKeys;

		return true;
	}

	return false;
}
*/


void ObjectExporter::GetFrameRange( int& min, int& max)
{ FUNC_ENTER("ObjectExporter::GetFrameRange"); 
	Link<NodeData>* curNode = nodeList.GetHead();

	if (curNode)
	{
		min = curNode->data.settings.start;
		max = curNode->data.settings.end;

		while(curNode)
		{
			if (curNode->data.settings.start < min)
				min = curNode->data.settings.start;

			if (curNode->data.settings.end > max)
				max = curNode->data.settings.end;

			curNode = curNode->next;
		}
	}
	else
	{
		min = 0;
		max = 0;
	}
}

// Count all the Q frames in the entire export
int ObjectExporter::GetNumQFrames()
{ FUNC_ENTER("ObjectExporter::GetNumQFrames"); 
	Link<NodeData>* curNode = nodeList.GetHead();
	int total = 0;

	while(curNode)
	{
		for(int i=0;i<curNode->data.numNodes;i++)
			total += curNode->data.numCompressedQFrames[i];

		curNode = curNode->next;
	}

	return total;
}

// Count all the T frames in the entire export
int ObjectExporter::GetNumTFrames()
{ FUNC_ENTER("ObjectExporter::GetNumTFrames"); 
	Link<NodeData>* curNode = nodeList.GetHead();
	int total = 0;

	while(curNode)
	{
		for(int i=0;i<curNode->data.numNodes;i++)
			total += curNode->data.numCompressedTFrames[i];

		curNode = curNode->next;
	}

	return total;
}

int ObjectExporter::GetNumScriptKeys( INodeTab& nodes )
{ FUNC_ENTER("ObjectExporter::GetNumScriptKeys"); 
	// Go through all the nodes and count up the number of contained script keys
	int count = nodes.Count();
	int total = 0;

	for(int i=0;i<count;i++)
		total += GetNumScriptKeys( nodes[i] );
	
	return total;
}

void ObjectExporter::DumpDebug(NodeData* nd)
{ FUNC_ENTER("ObjectExporter::DumpDebug"); 
	FILE* fp = fopen("c:\\OEdebug.txt", "a");
	int h, i;
	int curQFrame = 0;
	int curTFrame = 0;

	fprintf(fp, "Object: %s\n", (char*)nd->node->GetName());

	for(h = 0; h < nd->numNodes; h++)
	{
		fprintf(fp, "Node: %i\n", h);

		for(i = 0; i < nd->numCompressedQFrames[h]; i++)
		{
			fprintf(fp, "#: %i  time: %f  qx: %f  qy: %f  qz: %f  real: %f\n", i,
																			   nd->QFrames[curQFrame].time,
																			   nd->QFrames[curQFrame].q[X],
																			   nd->QFrames[curQFrame].q[Y],
																			   nd->QFrames[curQFrame].q[Z],
																			   nd->QFrames[curQFrame].q[W]);

			curQFrame++;
		}

		for(i = 0; i < nd->numCompressedTFrames[h]; i++)
		{
			fprintf(fp, "#: %i  time: %f  tx: %f  ty: %f  tz: %f\n", i,
																	 nd->TFrames[curTFrame].time,
																	 nd->TFrames[curTFrame].t[X],
																	 nd->TFrames[curTFrame].t[Y],
																	 nd->TFrames[curTFrame].t[Z]);

			curTFrame++;
		}
	}

	fclose(fp);
}

Link<ObjectExporter::NodeData>* ObjectExporter::AddNodeData(INode* node, ObjSpecificExportSettings* settings, bool bCompress, bool bOneFrame, void (*PreCompCB)(void*), void* pData)
{ FUNC_ENTER("ObjectExporter::AddNodeData"); 
	// Now acquire the animation data and store it in the nodelist
	Link<NodeData>* nd = nodeList.Add();

	nd->data.settings = *settings;

	// This forces the animation exporter to not descend more than one node
	nd->data.numNodes = 1;

	nd->data.node = node;

	CStr name = node->GetName();

	GetAnim(node,settings->start,settings->end,settings->fRotTolerance,settings->fTranTolerance,NULL,bCompress,
			&nd->data.numCompressedQFrames, &nd->data.QFrames,
			&nd->data.numCompressedTFrames, &nd->data.TFrames,
			&nd->data.numNodes,
			NULL,
			PreCompCB,
			pData);

#ifdef DEBUG_NODE_DATA
	DumpDebug(&nd->data);
#endif

	if (bOneFrame)
	{
		int num3DSFrames = settings->end - settings->start;
		
		RemoveDblKeys(nd->data.numNodes,num3DSFrames,
			          nd->data.QFrames,
					  nd->data.TFrames,
					  nd->data.numCompressedQFrames,
					  nd->data.numCompressedTFrames);
	}

	return nd;
}

void ObjectExporter::BuildNodeData(INodeTab& nodes,bool bCompress,bool bOneFrame)
{ FUNC_ENTER("ObjectExporter::BuildNodeData"); 
	ObjSpecificExportSettings  default_settings;
	
	Interval interval = gInterface->GetAnimRange();

	default_settings.start          = interval.Start()/GetTicksPerFrame();
	default_settings.end            = interval.End()/GetTicksPerFrame();
	default_settings.fRotTolerance  = DEFAULT_COMPRESS;
	default_settings.fTranTolerance = DEFAULT_TCOMPRESS;

	nodeList.Clear();

	int count = nodes.Count();

	for(int i=0;i<count;i++)
	{
		// Acquire the object specific data for this node
		ObjSpecificExportSettings* settings;
		AppDataChunk* appdata;
		appdata=nodes[i]->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

		// If the Object doesn't have any data attached we will assume it to use the defaults
		if (appdata)
			settings=(ObjSpecificExportSettings*)appdata->data;
		else
		{
			settings=&default_settings;
			strcpy(settings->strUserName,(char*)nodes[i]->GetName());
		}

		// Now acquire the animation data and store it in the nodelist
		Link<NodeData>* nd = nodeList.Add();

		nd->data.settings = *settings;

		// This forces the animation exporter to not descend more than one node
		nd->data.numNodes = 1;

		nd->data.node = nodes[i];

		GetAnim(nodes[i],settings->start,settings->end,settings->fRotTolerance,settings->fTranTolerance,NULL,bCompress,
			    &nd->data.numCompressedQFrames, &nd->data.QFrames,
				&nd->data.numCompressedTFrames, &nd->data.TFrames,
				&nd->data.numNodes);

		if (bOneFrame)
		{
			int num3DSFrames = settings->end - settings->start;
			
			RemoveDblKeys(nd->data.numNodes,num3DSFrames,
			              nd->data.QFrames,
						  nd->data.TFrames,
						  nd->data.numCompressedQFrames,
						  nd->data.numCompressedTFrames);
		}
	}
}


bool ObjectExporter::DoExport( ExportType type )
{ FUNC_ENTER("ObjectExporter::DoExport"); 
	// Assign default settings	Interval interval=gInterface->GetAnimRange();

	ObjExportSettings default_exportSettings;

	default_exportSettings.bDebug        = false;
	default_exportSettings.bCompress     = true;
	default_exportSettings.bNoDbls       = true;
	default_exportSettings.bExportQN     = true;
	default_exportSettings.bArchive      = true;
	default_exportSettings.bCleanup      = false;
	default_exportSettings.bCompressTime = true;
	default_exportSettings.bExportAll    = true;
	default_exportSettings.bExportQN     = false;
	default_exportSettings.bNoDbls       = true;
	default_exportSettings.strPrefix[0]  = 0;

	// Get the Object export settings
	ObjExportSettings* exportSettings;
	ReferenceTarget* scene=gInterface->GetScenePointer();

	AppDataChunk* appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);
	
	// If there are no Object export settings in the scene, use the defaults
	if (appdata)
		exportSettings=(ObjExportSettings*)appdata->data;
	else
		exportSettings=&default_exportSettings;

	// Go through all the animation systems and export all of them
	LinkList<CStr> systems;
	GetAnimSystems(&systems);

	int count = systems.GetSize();

	for(int cursys=0;cursys<count;cursys++)
	{
		INodeTab nodes;
		GetExportNodes(systems[cursys],nodes);
		int numObjs=nodes.Count();

		// Determine the file path
		CStr strSceneName=GetSceneName();
		CStr strFile;
		CStr strPath=getenv(APP_ENV);

		if (strPath==CStr(""))
		{
			char strErr[256];
			sprintf(strErr,"Your '%s' environment variable isn't set.  Please set it to continue.\nOperation aborted.",APP_ENV);

			MessageBox(gInterface->GetMAXHWnd(),strErr,"Object Exporter",MB_ICONWARNING|MB_OK);
			return false;
		}

		strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;

		// Attempt to create the directory (incase it doesn't exist)
		_mkdir(strPath);

		strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;
		strPath+="\\";
		strPath+=exportSettings->strPrefix;

		CStr strFileName = strPath + systems[cursys] + ".ska";

		//////////////////////////////////////////////////////////////////
		BuildNodeData(nodes,exportSettings->bCompress,exportSettings->bNoDbls);
		
		FILE* fp = fopen(strFileName,"wb");

		if (!fp)
		{
			char buf[256];
			sprintf(buf,"Couldn't open '%s' for output.");
			MessageBox(gInterface->GetMAXHWnd(),buf,"ObjAnim Export",MB_ICONWARNING|MB_OK);
			return false;
		}

		if (exportSettings->bDebug)
			fpDebug = fopen("c:\\debug.txt","w");

		// Output file header
		SHeader header;
		header.version=ANIMFILE_VERSION;
		header.flags=INTERMEDIATE_FORMAT | OBJECTANIM;
		
		if (!exportSettings->bCompress)
			header.flags|=UNCOMPRESSED_FORMAT;

		if (exportSettings->bCompressTime)
			header.flags|=COMPRESSEDTIME;

		// Objects always had pre rotated root, assign the flag for it
		if (bRotateRoot)
			header.flags |= PREROTATEDROOT;

		GetFrameRange(start,end);

		int num3DSFrames = end-start;
		int fps3DS       = GetFrameRate();
			
		header.skeletonName=GenerateCRC("OBJECTS");			// hard-coded default for now
		header.duration=(float)num3DSFrames/(float)fps3DS;
		header.numBones = numObjs;
		header.numQKeys = GetNumQFrames();
		header.numTKeys = GetNumTFrames();

		header.numUserDefinedKeys = GetNumScriptKeys(nodes);

		if (!header.Write(fp))
			return false;

		// Build CRC  (There are no parent or flip names in this situation and ^= 0 has no effect so we can
		//             safely ignore them)
		unsigned int uiCRC = (unsigned int)numObjs;

		for(int i=0;i<numObjs;i++)
		{
			uiCRC ^= GenerateCRC(nodes[i]->GetName());
		}

		// Output Skeleton data
		fwrite(&uiCRC,sizeof(unsigned int),1,fp);		// CRC

		fwrite(&numObjs,sizeof(unsigned int),1,fp);		// Write out number of bones (number of objects)

		// Bone names/Object names
		for(i=0;i<numObjs;i++)
		{
			unsigned int uiCRC = GenerateCRC(nodes[i]->GetName());

			// Write skeleton data fields (mostly N/A)
			fwrite(&uiCRC,sizeof(unsigned int),1,fp);
		}

		// Pad out unused fields (Parent Names, Flip Names)
		for(i=0;i<numObjs*2;i++)
		{
			unsigned int uiEmpty = 0;
			fwrite(&uiEmpty,sizeof(unsigned int),1,fp);
		}

	/////////////

		// Output compressed rotation data
		SQData sqdata(numObjs,header.numQKeys);
		int QKeyPos=0;

		for(i=0;i<nodeList.GetSize();i++)
		{
			sqdata.numKeysPerBone[i]=nodeList[i].numCompressedQFrames[0];
			
			for(int j=0;j<nodeList[i].numCompressedQFrames[0];j++)
			{
				//int curFrame=j+i*reqFrames;
				int curFrame=j;

				// If the compress time flag is specified we need to multiply time time by the FPS
				// rate to convert it into the frame number as opposed to the time into the animation
				if (exportSettings->bCompressTime)
				{
					// This is still going to get dumped out as a float, so we need to make sure
					// the compiler makes the representation of the float match an int
					// We can do this since they're both 4 bytes
					int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
					float ftmp = (nodeList[i].QFrames[curFrame].time * (float)OUTPUT_FPS);
					char buf[256];
					sprintf(buf,"Pre float is: %f\n",ftmp);
					OutputDebugString(buf);
					*val = Round(ftmp);
					
					//if ( (*val % 2) == 1 )
					//	assert(0);
				}
				else
					sqdata.theQKeys[QKeyPos].time = nodeList[i].QFrames[curFrame].time;

				sqdata.theQKeys[QKeyPos].qx   = nodeList[i].QFrames[curFrame].q[X];
				sqdata.theQKeys[QKeyPos].qy   = nodeList[i].QFrames[curFrame].q[Y];
				sqdata.theQKeys[QKeyPos].qz   = nodeList[i].QFrames[curFrame].q[Z];
				sqdata.theQKeys[QKeyPos].real = nodeList[i].QFrames[curFrame].q[W];

				if (fpDebug)
				{
					if (exportSettings->bCompressTime)
					{
						char sbuf[256];
						sprintf(sbuf,"---> Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
						fprintf(fpDebug,"Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
						OutputDebugString(sbuf);
						
						Quat q;
						q.x = sqdata.theQKeys[QKeyPos].qx;
						q.y = sqdata.theQKeys[QKeyPos].qy;
						q.z = sqdata.theQKeys[QKeyPos].qz;
						q.w = sqdata.theQKeys[QKeyPos].real;
						float rot[3];
						QuatToEuler(q, rot);
						fprintf(fpDebug,"Euler: X: %f Y: %f Z: %f\n", rot[0], rot[1], rot[2]);

						sprintf(sbuf,"---> actual time: %f\n",nodeList[i].QFrames[curFrame].time);
						sprintf(sbuf,"---> new time: %f\n",nodeList[i].QFrames[curFrame].time * 60.0f);
						OutputDebugString(sbuf);
					}
					else
						fprintf(fpDebug,"Q[node=%i frame=%i].time=%f\n",i,j,nodeList[i].QFrames[curFrame].time);

					fprintf(fpDebug,"Q[node=%i frame=%i].qx=%f\n",i,j,nodeList[i].QFrames[curFrame].q[X]);
					fprintf(fpDebug,"Q[node=%i frame=%i].qy=%f\n",i,j,nodeList[i].QFrames[curFrame].q[Y]);
					fprintf(fpDebug,"Q[node=%i frame=%i].qz=%f\n",i,j,nodeList[i].QFrames[curFrame].q[Z]);
					fprintf(fpDebug,"Q[node=%i frame=%i].real=%f\n\n",i,j,nodeList[i].QFrames[curFrame].q[W]);
				}

				QKeyPos++;
			}
		}

		if (!sqdata.Write(fp))
			return false;

		// Output compressed translation data
		STData stdata(numObjs,header.numTKeys);
		int TKeyPos=0;

		for(i=0;i<nodeList.GetSize();i++)
		{
			stdata.numKeysPerBone[i]=nodeList[i].numCompressedTFrames[0];
			
			for(int j=0;j<nodeList[i].numCompressedTFrames[0];j++)
			{
				//int curFrame=j+i*reqFrames;
				int curFrame=j;

				if (exportSettings->bCompressTime)
				{
					// This is still going to get dumped out as a float, so we need to make sure
					// the compiler makes the representation of the float match an int
					// We can do this since they're both 4 bytes
					int* val = (int*)&stdata.theTKeys[TKeyPos].time;
					float ftmp = nodeList[i].TFrames[curFrame].time * (float)OUTPUT_FPS;
					*val = Round(ftmp);

					//if ( *val % 2 == 1 )
					//	assert(0);
				}
				else
					stdata.theTKeys[TKeyPos].time = nodeList[i].TFrames[curFrame].time;

				stdata.theTKeys[TKeyPos].tx   = nodeList[i].TFrames[curFrame].t[0];
				stdata.theTKeys[TKeyPos].ty   = nodeList[i].TFrames[curFrame].t[1];
				stdata.theTKeys[TKeyPos].tz   = nodeList[i].TFrames[curFrame].t[2];

				if (fpDebug)
				{
					if (exportSettings->bCompressTime)
						fprintf(fpDebug,"T[node=%i frame=%i].time=%i\n",i,j,*((int*)&stdata.theTKeys[TKeyPos].time));
					else
						fprintf(fpDebug,"T[node=%i frame=%i].time=%f\n",i,j,nodeList[i].TFrames[curFrame].time);

					fprintf(fpDebug,"T[node=%i frame=%i].tx=%f\n",i,j,nodeList[i].TFrames[curFrame].t[0]);
					fprintf(fpDebug,"T[node=%i frame=%i].ty=%f\n",i,j,nodeList[i].TFrames[curFrame].t[1]);
					fprintf(fpDebug,"T[node=%i frame=%i].tz=%f\n\n",i,j,nodeList[i].TFrames[curFrame].t[2]);

					char test[256];
					sprintf(test,"T[node=%i frame=%i].time=%f\n",i,j,nodeList[i].TFrames[curFrame].time);
					OutputDebugString(test);
					sprintf(test,"T[node=%i frame=%i].tx=%f\n",i,j,nodeList[i].TFrames[curFrame].t[0]);
					OutputDebugString(test);
					sprintf(test,"T[node=%i frame=%i].ty=%f\n",i,j,nodeList[i].TFrames[curFrame].t[1]);
					OutputDebugString(test);
					sprintf(test,"T[node=%i frame=%i].tz=%f\n\n",i,j,nodeList[i].TFrames[curFrame].t[2]);
					OutputDebugString(test);
				}

				TKeyPos++;
			}
		}

		if (!stdata.Write(fp))
			return false;

	///////////////////////////////////////

		for(int s=0;s<nodeList.GetSize();s++)
		{
			// Acquire the script keys
			GenKey<ObjScriptKey>* ObjScriptKeys;
			int numObjScriptKeys;
			int numExtKeys;

			ObjScriptKeys = GetObjScriptKeys(nodeList[s].node,&numObjScriptKeys);
			numExtKeys = numObjScriptKeys;

			// Now write out the change script keys
			for(int i=0;i<numObjScriptKeys;i++)
			{
				if (exportSettings->bCompressTime)
				{
					// Script time was always the keyframe
					// represent as int instead of float
					int* val = (int*)&ObjScriptKeys[i].time;
					*val = ObjScriptKeys[i].time;
				}

				if (fpDebug)
				{
					fprintf(fpDebug,"%i\n",i);

					if (exportSettings->bCompressTime)
						fprintf(fpDebug,"Time: %i\n",*((int*)&ObjScriptKeys[i].time));
					else
						fprintf(fpDebug,"Time: %f\n",ObjScriptKeys[i].time);

					fprintf(fpDebug,"Name: %s (0x%x)\n",(char*)ObjScriptKeys[i].key.name,GenerateCRC((char*)ObjScriptKeys[i].key.name));

					char buf[256];
					sprintf(buf,"Time: %f\n",ObjScriptKeys[i].time);
					OutputDebugString(buf);

					sprintf(buf,"Name: %s (0x%x)\n",(char*)ObjScriptKeys[i].key.name,GenerateCRC((char*)ObjScriptKeys[i].key.name));
					OutputDebugString(buf);
				}

				CStr name;

				if (ObjScriptKeys[i].key.name == CStr(USERSCRIPT_NAME))
				{
					char buf[256];
					_splitpath(strFileName,NULL,NULL,buf,NULL);
					name += buf;
					name += "Frame";
					sprintf(buf,"%i",ObjScriptKeys[i].key.frame);
					name += buf;
				}
				else
				{
					name = ObjScriptKeys[i].key.name;
				}

				unsigned long crc = GenerateCRC((char*)name);

				if (!RunScriptKey(ObjScriptKeys[i].time,GenerateCRC((char*)name)).Write(fp))
				{
					if (ObjScriptKeys)
						delete [] ObjScriptKeys;

					return false;
				}
			}

			// Free allocated keys
			if (ObjScriptKeys)
				delete [] ObjScriptKeys;
		}

		// Close debug file
		if (fpDebug)
			fclose(fpDebug);

		fclose(fp);
	}	// end cursys loop

	if (fpDebug)
	{
		fclose(fpDebug);
		fpDebug = NULL;
	}

	return true;
}

bool ObjectExporter::Save()
{ FUNC_ENTER("ObjectExporter::Save"); 
	return true;
}

CStr ObjectExporter::GetSceneName()
{ FUNC_ENTER("ObjectExporter::GetSceneName"); 
	// This is temporary, to get the scene name
	const int vMAX_LENGTH_SCENE_NAME = 64;
	const int vVERSION  = 0x0002;
	const int vVERSION2 = 0x0003;
	const int vVERSION3 = 0x0004;
	const int vVERSION4 = 0x0005;
	//const int vVERSION5 = 0x0006;

	ReferenceTarget* scene = gInterface->GetScenePointer();
	char scene_name[vMAX_LENGTH_SCENE_NAME];
	char* options_data;
	AppDataChunk* pad;
	int version;

	pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_EXPORT_OPTIONS_ID );

	if (pad)
	{
		options_data = (char*) pad->data;

		version=*((int*)options_data);
		if (version == vVERSION || version == vVERSION2 || version == vVERSION3 || version >= vVERSION4 )
		{
			options_data += sizeof( int );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			options_data += sizeof( bool );
			memcpy( scene_name, options_data, vMAX_LENGTH_SCENE_NAME );
		}
		else
		{
			MessageBox(gInterface->GetMAXHWnd(),"WARNING!  An unrecognized SceneExportOptions data chunk version was detected.\nUnable to retrieve scene name.","Version out of date",MB_ICONWARNING|MB_OK);
			return CStr("default");
		}

		return CStr(scene_name);
	}

	return CStr("default");
}

bool ObjectExporter::NodeSelected(INode* node)
{ FUNC_ENTER("ObjectExporter::NodeSelected"); 
	AppDataChunk* appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID);
	
	if (!appdata || !appdata->data)
		return false;
	
	return *((bool*)appdata->data);
}

bool ObjectExporter::ExportQN(FILE* fp)
{ FUNC_ENTER("ObjectExporter::ExportQN"); 
	if (!fp)
		return false;

	// Get the Object export settings
	ObjExportSettings* exportSettings;
	ReferenceTarget* scene=gInterface->GetScenePointer();

	AppDataChunk* appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);
	
	// If there are no Object export settings in the scene, use the defaults
	CStr strPrefix;
	
	if (appdata)
	{
		exportSettings=(ObjExportSettings*)appdata->data;
		strPrefix=exportSettings->strPrefix;
	}
	else
		strPrefix="";

	// Determine the file path
	CStr strSceneName=GetSceneName();
	CStr strFile;
	CStr strPath=getenv(APP_ENV);

	if (strPath==CStr(""))
	{
		char strErr[256];
		sprintf(strErr,"Your '%s' environment variable isn't set.  Please set it to continue.\nOperation aborted.",APP_ENV);

		MessageBox(gInterface->GetMAXHWnd(),strErr,"Object Exporter",MB_ICONWARNING|MB_OK);
		return false;
	}

	//strPath=CStr(getenv(APP_ENV))+CStr(CAM_PATH)+strSceneName;
	strPath=CStr(CAMREL_PATH)+strSceneName;
	strPath+="\\";
	strPath+=strPrefix;

	// Write .qn header
	fprintf(fp,"script LoadObjectAnims\n");


	// Process all of the animation systems within the scene
	LinkList<CStr> systems;
	GetAnimSystems(&systems);

	int count = systems.GetSize();
	int i;

	for(int cursys=0;cursys<count;cursys++)
	{
		strFile=strPath+systems[cursys]+CStr(".ska");
		fprintf(fp,"\tLoadAsset \"%s\" %s\n",(char*)strFile,(char*)systems[cursys]);
		/*
		INodeTab nodes;
		GetExportNodes(systems[cursys],nodes);

		int numObjs=nodes.Count();

		for(i=0;i<numObjs;i++)
		{
			CStr   objName;
			INode* node=nodes[i];

			ObjSpecificExportSettings* settings;
			AppDataChunk* appdata;
			appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

			// If the Object doesn't have any data attached we will assume it to use the defaults
			if (appdata)
			{
				settings=(ObjSpecificExportSettings*)appdata->data;
				objName=settings->strUserName;
			}
			else
			{
				objName=node->GetName();
			}

			// Determine the Object animation filename
			strFile=strPath+objName+CStr(".ska");

			// We now have the Object name, write a line for it to the .qn file
			fprintf(fp,"\tLoadAsset \"%s\" %s\n",(char*)strFile,(char*)objName);
		}
		*/
	}

///////////////////////////////////
	// Write .qn footer
	fprintf(fp,"endscript\n\n");

	for(cursys=0;cursys<count;cursys++)
	{
		INodeTab nodes;
		GetExportNodes(systems[cursys],nodes);

		int numObjs=nodes.Count();

		// Now generate the user defined Object scripts
		for(i=0;i<numObjs;i++)
		{
			INode* node=nodes[i];
			
			// Get the user name for the Object (export name)
			CStr name;
			appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);

			if (appdata)
			{
				name = ((ObjSpecificExportSettings*)appdata->data)->strUserName;
			}
			else
			{
				name = node->GetName();
			}

			appdata=node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);

			if (!appdata)
				continue;

			int   len=appdata->length;
			char* buf=(char*)appdata->data;

			char  sframeNum[256];
			char  scriptName[512];
			int   extlen;				// Length of extended data for attached user scripts
			char  token[1024];
			int   pos = 0;

			while(1)
			{
				if (!GetToken(buf,&pos,token,"\n"))
					break;

				strcpy(scriptName,token);
				
				if (!GetToken(buf,&pos,token,"\n"))	// scriptFile
					break;
				
				if (!GetToken(buf,&pos,token,"\n"))
					break;

				strcpy(sframeNum,token);

				if (CStr(scriptName)==CStr(USERSCRIPT_NAME))
				{
					char* sbuf;
					GetToken(buf,&pos,token,"\n");
					extlen = atoi(token);
					
					sbuf = new char[extlen+1];

					for(int i=0;i<extlen;i++)
						sbuf[i]=buf[pos++];

					sbuf[i]='\0';
					
					CStr sname, strBuf;
					sname = name;
					sname += "Frame";
					sname += sframeNum;

					strBuf = ReplaceStr(sbuf,"\r\n","\n");

					fprintf(fp,"script %s\n",(char*)sname);
					fprintf(fp,"%s\n",(char*)strBuf);
					fprintf(fp,"endscript\n\n");

					delete sbuf;
				}
			}
		}
//////////////////////////
	}

	return true;
}

void ObjectExporter::SendNetMessage(char* objname,int delay)
{ FUNC_ENTER("ObjectExporter::SendNetMessage"); 
	Net::MsgViewObjSetCamAnimFile msg;

	CStr sceneName = GetSceneName();
	CStr fileName  = CStr(CAMREL_PATH)+sceneName+CStr("\\")+CStr(objname)+".ska";
	CStr absFileName = CStr(getenv(APP_ENV)) + CStr("\\data\\")+fileName;

	strcpy(msg.m_Filename,(char*)fileName);
	msg.m_checksum=GenerateCRC(objname);

	//gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE, sizeof( Net::MsgViewObjSetCamAnimFile ), &msg,
	//								 Net::NORMAL_PRIORITY, 	Net::QUEUE_SEQUENCED );

	Net::MsgDesc msgdesc;
	msgdesc.m_Id       = Net::vMSG_ID_VIEWOBJ_SET_CAMANIM_FILE;
	msgdesc.m_Length   = sizeof( Net::MsgViewObjSetCamAnimFile );
	msgdesc.m_Data     = &msg;
	msgdesc.m_Priority = Net::NORMAL_PRIORITY;
	msgdesc.m_Queue    = Net::QUEUE_DEFAULT;
	msgdesc.m_GroupId  = Net::GROUP_ID_DEFAULT;
	msgdesc.m_Singular = false;
	msgdesc.m_Delay    = delay;

	gClient->EnqueueMessageToServer( &msgdesc );
}

void ObjectExporter::DoQBRUpdate()
{ FUNC_ENTER("ObjectExporter::DoQBRUpdate"); 
	/*
	CStr camFileQN = CStr("..\\") + camFile;
	CStr camFileQB = getenv(APP_ENV);
	
	// Strip from the last backslash since we're going back a directory
	// The game doesn't understand ..
	char* pos = strrchr(camFileQB,'\\');
	if (!pos)
		return;

	*pos=0;

	camFileQB += CStr("\\") + camFile;
	
	camFileQB[camFileQB.Length()-1] = 'b';

	FILE* fp = fopen(camFileQN,"w");
	if (fp)
	{
		theObjectExporter.ExportQN(fp);
		fclose(fp);
	}
	*/

	Exporter* exporter = GetSceneExporter();
	exporter->DoExport( vSCRIPT_EXPORT );

	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	CStr dirPath = getenv(APP_ENV);
	dirPath += CStr(QN_PATH);
	dirPath += seo.m_SceneName + CStr("\\") + seo.m_SceneName + ".qn";

	// Execute qcomp to build .qb
	char  path[256];
	char* args[3];
	sprintf( path, "%s\\bin\\win32\\qcomp.exe", getenv(APP_ENV) );

	args[0] = path;
	args[1] = dirPath;
	args[2] = NULL;

	//spawnvp( _P_WAIT, path, args );
	ExecuteCommand( args );

	dirPath[dirPath.Length()-1] = 'b';

	// Execute rq to send to game
	sprintf( path, "%s\\bin\\win32\\rq.exe", getenv(APP_ENV) );

	args[0] = path;
	args[1] = dirPath;
	args[2] = NULL;

	//spawnvp( _P_WAIT, path, args );
	ExecuteCommand( args );
}

////////////// SystemNameDlg

SystemNameDlg::SystemNameDlg(HINSTANCE hInstance,HWND hwndParent) :
	ModalDlgWindow(hInstance,MAKEINTRESOURCE(IDD_SYSNAME),hwndParent)
{ FUNC_ENTER("SystemNameDlg::SystemNameDlg"); 
}

SystemNameDlg::~SystemNameDlg()
{ FUNC_ENTER("SystemNameDlg::~SystemNameDlg"); 
	ReleaseICustEdit(IEdit);
}

BOOL SystemNameDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("SystemNameDlg::DlgProc"); 
	switch(msg)
	{
	case WM_INITDIALOG:
		IEdit = GetICustEdit(GetDlgItem(hwnd,IDC_SYSNAME));
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDCANCEL:
		case IDOK:
			{
				char buf[256];
				IEdit->GetText(buf,255);
				name = buf;

				EndDialog(hwnd,0);
				return TRUE;
			}			
		}
	}

	return FALSE;
}

void SystemNameDlg::SetName(CStr name)
{ FUNC_ENTER("SystemNameDlg::SetName"); 
	IEdit->SetText(name);
}

CStr SystemNameDlg::GetName()
{ FUNC_ENTER("SystemNameDlg::GetName"); 
	return CStr(name);
}

//////////////// End SystemNameDlg
