#include "FuncEnter.h"

#include <next.h>
#include <appdata.h>
#include <Export/ExportOptions.h>
#include <Export/ModelExportOptions.h>
#include <Export/TextureExporter.h>
#include "../UI/PropList.h"

static PropList* pLODPropList = NULL;
extern HINSTANCE hInstance;

static INT_PTR CALLBACK ModelExportOptionsDlgProc(
		HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("ModelExportOptionsDlgProc"); 
	static ModelOptionsDlg *dlg = NULL;
	char   buf[256];

	switch(msg) 
	{
		case CC_SPINNER_CHANGE:
		{
			// If the number of LOD levels has changed we need to add another field
			// to the LOD property list
			
			switch(LOWORD(wParam))
			{
			case IDC_LODLEVELSSPIN:
				ISpinnerControl* iLODLevelSpin = GetISpinner( GetDlgItem( hWnd, IDC_LODLEVELSSPIN));
				
				if (iLODLevelSpin)
				{
					int nLODLevels = iLODLevelSpin->GetIVal();

					pLODPropList->SaveProperties();
					pLODPropList->DestroyUI();
					pLODPropList->Clear();

					for(int i = 0; i < nLODLevels; i++)
					{
						char buf[80];
						_itoa(i + 1, buf, 10);

						CStr fieldName = CStr("Distance_") + buf;
						pLODPropList->AddSpinEdit(fieldName, 0.0f, 99999.0f, 0.1f, "Distance at which this level should be activated");
					}

					pLODPropList->BuildUI();
					pLODPropList->RestoreProperties();
				}
				return TRUE;
			}
			break;
		}

		case WM_INITDIALOG:
		{
			ModelExportOptions options;
			ICustEdit* model_name_ctrl;
			ICustEdit* model_dir_ctrl;
			
			// Initialize the LOD Distance Property List
			if (pLODPropList)
				delete pLODPropList;

			pLODPropList = new PropList(hInstance, 1);
			pLODPropList->Attach(GetDlgItem(hWnd, IDC_LODPROPLIST));
			pLODPropList->HasApply(false);

			dlg = (ModelOptionsDlg*) lParam;

			GetModelExportOptions( &options );		

			model_name_ctrl = GetICustEdit( GetDlgItem( hWnd, IDC_MODEL_NAME ));
			model_dir_ctrl  = GetICustEdit( GetDlgItem( hWnd, IDC_MODEL_DIR ));
			model_name_ctrl->SetText( options.m_ModelName );
			model_dir_ctrl->SetText( options.m_ModelDir );
			ReleaseICustEdit( model_name_ctrl );			
			ReleaseICustEdit( model_dir_ctrl );

			ICustEdit*       model_lod_lower_cap       = GetICustEdit( GetDlgItem( hWnd, IDC_LODCAPB ) );
			ICustEdit*       model_lod_num_levels      = GetICustEdit( GetDlgItem( hWnd, IDC_LODLEVELS ) );
			ISpinnerControl* model_lod_lower_cap_spin  = GetISpinner( GetDlgItem( hWnd, IDC_LODCAPBSPIN ) );
			ISpinnerControl* model_lod_num_levels_spin = GetISpinner( GetDlgItem( hWnd, IDC_LODLEVELSSPIN ) );

			model_lod_lower_cap_spin->LinkToEdit( GetDlgItem( hWnd, IDC_LODCAPB ), EDITTYPE_FLOAT);
			model_lod_num_levels_spin->LinkToEdit( GetDlgItem( hWnd, IDC_LODLEVELS ), EDITTYPE_INT);

			model_lod_lower_cap_spin->SetLimits(0.0f, 100.0f);
			model_lod_num_levels_spin->SetLimits(0,100);

			model_lod_lower_cap_spin->SetValue(options.m_LODLowerLimit, TRUE);
			model_lod_num_levels_spin->SetValue(options.m_LODNumLevels, TRUE);

			ReleaseICustEdit( model_lod_lower_cap );
			ReleaseICustEdit( model_lod_num_levels );
			ReleaseISpinner( model_lod_lower_cap_spin );
			ReleaseISpinner( model_lod_num_levels_spin );

			if( options.m_Optimized )
			{
				SendDlgItemMessage( hWnd, IDC_OPTIMIZE_GEOMETRY, BM_SETCHECK, BST_CHECKED, 0 );
			}			

			if( options.m_ExportQN )
			{
				SendDlgItemMessage( hWnd, IDC_EXPORTQN, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_DirWarn )
			{
				SendDlgItemMessage( hWnd, IDC_DIRWARN, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_ExportToSceneDir )
			{
				SendDlgItemMessage( hWnd, IDC_DUMPTOSCENE, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_IncSkelData )
			{
				SendDlgItemMessage( hWnd, IDC_INCSKELETON, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_FlipVerts )
			{
				SendDlgItemMessage( hWnd, IDC_FLIPVERTS, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_AlwaysLOD )
			{
				SendDlgItemMessage( hWnd, IDC_ALWAYSLOD, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_LODProgressive )
			{
				SendDlgItemMessage( hWnd, IDC_LODPROGRESSIVE, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_ExportByName )
			{
				SendDlgItemMessage( hWnd, IDC_EXPORTBYNAME, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_NoNormals )
			{
				SendDlgItemMessage( hWnd, IDC_NONORMALS, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_DumpDebugData )
			{
				SendDlgItemMessage( hWnd, IDC_DUMPDEBUG, BM_SETCHECK, BST_CHECKED, 0 );
			}

			if( options.m_AllowPropOverrides )
			{
				SendDlgItemMessage( hWnd, IDC_ALLOWOVERRIDES, BM_SETCHECK, BST_CHECKED, 0 );
			}

			// Assign LOD Distances
			for(int i = 0; i < options.m_LODNumLevels; i++)
			{
				char buf[80];
				_ltoa(options.m_LODDistances[i], buf, 10);
				pLODPropList->SetValue(i, buf);
			}

			switch( options.m_ExportMode )
			{
			case ModelExportOptions::vEXPORT_ALL:
				// RETURN EXPORT OPTS: Check
				break;

			case ModelExportOptions::vEXPORT_VISIBLE:

				break;

			case ModelExportOptions::vEXPORT_SELECTION:

				break;

			case ModelExportOptions::vEXPORT_GROUP:

				break;
			}

			if (options.m_IncSkelData)
				EnableWindow(GetDlgItem(hWnd,IDC_FLIPVERTS),TRUE);
			else
				EnableWindow(GetDlgItem(hWnd,IDC_FLIPVERTS),FALSE);

			ShowWindow( hWnd, SW_SHOW );
			break;
		}

		case WM_COMMAND:
		{
			switch (LOWORD(wParam)) 
			{
				case IDC_INCSKELETON:
					if (IsDlgButtonChecked(hWnd,IDC_INCSKELETON)==BST_CHECKED)
						EnableWindow(GetDlgItem(hWnd,IDC_FLIPVERTS),TRUE);
					else
						EnableWindow(GetDlgItem(hWnd,IDC_FLIPVERTS),FALSE);
					return TRUE;

				case IDOK:
				{
					ModelExportOptions options;
					ICustEdit* model_name_ctrl;
					ICustEdit* model_dir_ctrl;

					model_name_ctrl = GetICustEdit( GetDlgItem( hWnd, IDC_MODEL_NAME ));
					model_name_ctrl->GetText( buf, 255 );
					options.m_ModelName = buf;
					model_dir_ctrl = GetICustEdit( GetDlgItem( hWnd, IDC_MODEL_DIR ));
					model_dir_ctrl->GetText( buf, 255 );
					options.m_ModelDir = buf;
					ReleaseICustEdit( model_name_ctrl );
					ReleaseICustEdit( model_dir_ctrl );

					ICustEdit* model_lod_lower_cap  = GetICustEdit( GetDlgItem( hWnd, IDC_LODCAPB ) );
					ICustEdit* model_lod_num_levels = GetICustEdit( GetDlgItem( hWnd, IDC_LODLEVELS ) );

					options.m_LODLowerLimit = model_lod_lower_cap->GetFloat();
					options.m_LODNumLevels  = model_lod_num_levels->GetInt();

					ReleaseICustEdit( model_lod_lower_cap );
					ReleaseICustEdit( model_lod_num_levels );

					options.m_Optimized          = SendDlgItemMessage( hWnd, IDC_OPTIMIZE_GEOMETRY, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ExportQN           = SendDlgItemMessage( hWnd, IDC_EXPORTQN, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_DirWarn            = SendDlgItemMessage( hWnd, IDC_DIRWARN, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ExportToSceneDir   = SendDlgItemMessage( hWnd, IDC_DUMPTOSCENE, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_IncSkelData        = SendDlgItemMessage( hWnd, IDC_INCSKELETON, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_FlipVerts          = SendDlgItemMessage( hWnd, IDC_FLIPVERTS, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_AlwaysLOD          = SendDlgItemMessage( hWnd, IDC_ALWAYSLOD, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_LODProgressive     = SendDlgItemMessage( hWnd, IDC_LODPROGRESSIVE, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_ExportByName       = SendDlgItemMessage( hWnd, IDC_EXPORTBYNAME, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_NoNormals          = SendDlgItemMessage( hWnd, IDC_NONORMALS, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_DumpDebugData      = SendDlgItemMessage( hWnd, IDC_DUMPDEBUG, BM_GETCHECK, 0, 0 ) == BST_CHECKED;
					options.m_AllowPropOverrides = SendDlgItemMessage( hWnd, IDC_ALLOWOVERRIDES, BM_GETCHECK, 0, 0 ) == BST_CHECKED;

					// Assign LOD level distances
					options.SetNumLODLevels(options.m_LODNumLevels);

					for(int i = 0; i < options.m_LODNumLevels; i++)
					{
						CStr value;
						pLODPropList->GetValue(i, value);

						options.m_LODDistances[i] = atof(value);
					}

					SetModelExportOptions( &options );

					if (pLODPropList)
					{
						delete pLODPropList;
						pLODPropList = NULL;
					}

					DestroyModelOptionsDialog();
					break;
				}
				case IDCANCEL:
				{
					if (pLODPropList)
					{
						delete pLODPropList;
						pLODPropList = NULL;
					}

					DestroyModelOptionsDialog();
					break;
				}				
				default:
					break;
			}
			break;
		}		
		default:
			return FALSE;
	}
	return TRUE;
}

void	ModelOptionsDlg::Show( void )
{ FUNC_ENTER("ModelOptionsDlg::Show"); 
	m_wnd = CreateDialogParam(
		hInstance,
		MAKEINTRESOURCE(IDD_MODEL_EXPORT_OPTIONS),
		gInterface->GetMAXHWnd(),
		ModelExportOptionsDlgProc,
		(LPARAM)this);	
	TSTR title = "Model Options";
	SetWindowText( m_wnd, title );
	gInterface->RegisterDlgWnd( m_wnd );
}

void	ModelOptionsDlg::Hide( void )
{ FUNC_ENTER("ModelOptionsDlg::Hide"); 
	gInterface->UnRegisterDlgWnd( m_wnd );
	DestroyWindow( m_wnd );	
}

void GetModelExportOptions( ModelExportOptions* options )
{ FUNC_ENTER("GetModelExportOptions"); 
	char* options_data;
	ReferenceTarget *scene = gInterface->GetScenePointer();
	AppDataChunk* pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_MODEL_EXPORT_OPTIONS_ID );

	// Return the export options if they exist
	if( pad && pad->data )
	{
		int version;

		options_data = (char *) pad->data;
		version = *options_data;

		//if( version != ModelExportOptions::vVERSION_NUMBER )
		if ( version < 0x0002)
		{
			// for now, don't be backward compatible. This will force the dialog
			// to use the default values
			scene->RemoveAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_MODEL_EXPORT_OPTIONS_ID );			
		}
		else if (version < 0x000C)
		{
			char model_name[ModelExportOptions::vMAX_LENGTH_MODEL_NAME];
			char model_dir[ModelExportOptions::vMAX_LENGTH_DIR_NAME];
			char model_group[ModelExportOptions::vMAX_LENGTH_GROUP_NAME];
			
			memcpy( &options->m_Version, options_data, sizeof( int ));
			options_data += sizeof( int );
			memcpy( &options->m_Optimized, options_data, sizeof( bool ));
			options_data += sizeof( bool );
			memcpy( model_name, options_data, ModelExportOptions::vMAX_LENGTH_MODEL_NAME );
			options_data += ModelExportOptions::vMAX_LENGTH_MODEL_NAME;
			options->m_ModelName = model_name;
			memcpy( &options->m_ExportQN, options_data, sizeof( bool ));
			options_data += sizeof( bool );
			
			if (version >= 0x0003)
			{
				memcpy( &options->m_DirWarn, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x0004)
			{
				memcpy( &options->m_ExportToSceneDir, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x0005)
			{
				memcpy( model_dir, options_data, ModelExportOptions::vMAX_LENGTH_DIR_NAME );
				options_data += ModelExportOptions::vMAX_LENGTH_DIR_NAME;
				options->m_ModelDir = model_dir;
			}

			if (version >= 0x0006)
			{
				memcpy( &options->m_IncSkelData, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x0007)
			{
				memcpy( &options->m_FlipVerts, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x0008)
			{
				memcpy( &options->m_ExportMode, options_data, sizeof( int ));
				options_data += sizeof( int );

				memcpy( model_group, options_data, ModelExportOptions::vMAX_LENGTH_GROUP_NAME);
				options_data += ModelExportOptions::vMAX_LENGTH_GROUP_NAME;
				options->m_GroupName = model_group;
			}

			if (version >= 0x0009)
			{
				memcpy( &options->m_AlwaysLOD, options_data, sizeof( bool ));
				options_data += sizeof( bool );

				memcpy( &options->m_LODLowerLimit, options_data, sizeof( float ));
				options_data += sizeof( float );

				memcpy( &options->m_LODNumLevels, options_data, sizeof( int ));
				options_data += sizeof( int );

				memcpy( &options->m_LODProgressive, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x000A)
			{
				memcpy( &options->m_ExportByName, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}

			if (version >= 0x000B)
			{
				memcpy( &options->m_NoNormals, options_data, sizeof( bool ));
				options_data += sizeof( bool );
			}
		}
		else
		{
			options->Retrieve(options_data);
		}
	}	
}

void SetModelExportOptions( ModelExportOptions* options )
{ FUNC_ENTER("SetModelExportOptions"); 
	//char model_name[ModelExportOptions::vMAX_LENGTH_MODEL_NAME];
	//char model_dir[ModelExportOptions::vMAX_LENGTH_DIR_NAME];
	//char model_group[ModelExportOptions::vMAX_LENGTH_GROUP_NAME];
	char* options_data;	
	AppDataChunk* pad;
	ReferenceTarget *scene;
	//int version;

	ModelExportOptions old_options;
	GetModelExportOptions(&old_options);

	assert( options );

	scene = gInterface->GetScenePointer();	
	pad = scene->GetAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_MODEL_EXPORT_OPTIONS_ID );

	if( pad )
	{
		// Old version needs thrown out first, since we require more space
		scene->RemoveAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_MODEL_EXPORT_OPTIONS_ID );

		options_data = (char*) pad->data;
	}

	options_data = (char*)malloc(options->GetSize());
	options->Store(options_data);
	scene->AddAppDataChunk( vNEXT_CLASS_ID, 
		                    GUP_CLASS_ID, 
							vNAPP_MODEL_EXPORT_OPTIONS_ID,
							options->GetSize(), 
							(void*)options_data );

	// If the model name or export modes relating to the model name have changed then the
	// texture dictionary must be flagged as dirty so that the texture dictionary is rewritten
	// out to the new directory/name
	if (old_options.m_ModelName    != options->m_ModelName ||
		old_options.m_ModelDir     != options->m_ModelDir  ||
		old_options.m_ExportByName != options->m_ExportByName)
	{
		ITextureExporter* tex_exp = GetTextureExporter();
		tex_exp->ResetInitialExport();
	}


	/*
	// Yikes!  MAX uses a different memory manager must use malloc  aml
	//options_data = new char[GetSizeModelExportOptions()];
	char* pDataStart = options_data = (char*)malloc(GetSizeModelExportOptions());
	memset(options_data, 99999, GetSizeModelExportOptions());
	scene->AddAppDataChunk( vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_MODEL_EXPORT_OPTIONS_ID, 
		GetSizeModelExportOptions(), (void*) options_data );	

	version = ModelExportOptions::vVERSION_NUMBER;
	memcpy( options_data, &version, sizeof( int ));
	options_data += sizeof( int );
	memcpy( options_data, &options->m_Optimized, sizeof( bool ));
	options_data += sizeof( bool );
	strcpy( model_name, options->m_ModelName );
	memcpy( options_data, model_name, ModelExportOptions::vMAX_LENGTH_MODEL_NAME );
	options_data += ModelExportOptions::vMAX_LENGTH_MODEL_NAME;	
	memcpy( options_data, &options->m_ExportQN, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_DirWarn, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportToSceneDir, sizeof( bool ));
	options_data += sizeof( bool );
	strcpy( model_dir, options->m_ModelDir );
	memcpy( options_data, model_dir, ModelExportOptions::vMAX_LENGTH_DIR_NAME );
	options_data += ModelExportOptions::vMAX_LENGTH_DIR_NAME;
	memcpy( options_data, &options->m_IncSkelData,sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_FlipVerts,sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportMode,sizeof( int ));
	options_data += sizeof( int );
	strcpy( model_group, options->m_GroupName );
	memcpy( options_data, model_group, ModelExportOptions::vMAX_LENGTH_GROUP_NAME );
	options_data += ModelExportOptions::vMAX_LENGTH_GROUP_NAME;
	memcpy( options_data, &options->m_AlwaysLOD,sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_LODLowerLimit, sizeof( float ));
	options_data += sizeof( float );
	memcpy( options_data, &options->m_LODNumLevels, sizeof( int ));
	options_data += sizeof( int );
	memcpy( options_data, &options->m_LODProgressive, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_ExportByName, sizeof( bool ));
	options_data += sizeof( bool );
	memcpy( options_data, &options->m_NoNormals, sizeof( bool ));
	options_data += sizeof( bool );
	*/
}

int	GetSizeModelExportOptions( void )
{ FUNC_ENTER("GetSizeModelExportOptions"); 
	int size;

	size = 0;
	
	size += sizeof( int );	//	int		m_Version;
	size += sizeof( bool );	//	bool	m_Optimized;
	size += ModelExportOptions::vMAX_LENGTH_MODEL_NAME;	//	TSTR	m_ModelName;
	size += sizeof( bool );	//  bool    m_ExportQN
	size += sizeof( bool ); //  bool    m_DirWarn
	size += sizeof( bool ); //  bool    m_ExportToSceneDir
	size += ModelExportOptions::vMAX_LENGTH_DIR_NAME;	//  TSTR    m_ModelDir;
	size += sizeof( bool ); //  bool    m_IncSkelData
	size += sizeof( bool ); //  bool    m_FlipVerts
	size += sizeof( int );  //  int     m_ExportMode      v8
	size += ModelExportOptions::vMAX_LENGTH_GROUP_NAME;	//  TSTR    m_GroupName;
	size += sizeof( bool );	//  bool    m_AlwaysLOD       v9
	size += sizeof( float );//  float   m_LODLowerLimit   v9
	size += sizeof( int );  //  int     m_LODNumLevels    v9
	size += sizeof( bool ); //  bool    m_LODProgressive  v9
	size += sizeof( bool ); //  bool    m_ExportByName    v10
	size += sizeof( bool ); //  bool    m_NoNormals       v11

	return size;
}

ModelExportOptions::ModelExportOptions( void )
{ FUNC_ENTER("ModelExportOptions::ModelExportOptions"); 
	m_Optimized          = true;	
	m_ModelName          = "default";
	m_ExportQN           = true;
	m_DirWarn            = true;
	m_ExportToSceneDir   = false;
	m_ModelDir           = "";
	m_IncSkelData        = false;
	m_FlipVerts          = false;
	m_ExportMode         = 0;
	m_GroupName          = "";
	m_AlwaysLOD          = false;
	m_LODLowerLimit      = 50.0f;
	m_LODNumLevels       = 5;
	m_LODProgressive     = false;
	m_ExportByName       = false;
	m_NoNormals          = false;
	m_LODDistances       = (float*)malloc(sizeof(float) * m_LODNumLevels);
	m_DumpDebugData      = false;
	m_AllowPropOverrides = true;
}

ModelExportOptions::~ModelExportOptions( void )
{ FUNC_ENTER("ModelExportOptions::~ModelExportOptions"); 
	if (m_LODDistances)
		free(m_LODDistances);
}

void ModelExportOptions::SetNumLODLevels(unsigned int nLevels)
{ FUNC_ENTER("ModelExportOptions::SetNumLODLevels"); 
	m_LODDistances = (float*)realloc(m_LODDistances, sizeof(float) * nLevels);
	m_LODNumLevels = nLevels;
}

int  ModelExportOptions::GetSize()
{ FUNC_ENTER("ModelExportOptions::GetSize"); 
	int size;

	size = 0;
	
	size += sizeof( int );	//	int		m_Version;
	size += sizeof( bool );	//	bool	m_Optimized;
	size += GetStringSize(m_ModelName);
	//size += ModelExportOptions::vMAX_LENGTH_MODEL_NAME;	//	TSTR	m_ModelName;
	size += sizeof( bool );	//  bool    m_ExportQN
	size += sizeof( bool ); //  bool    m_DirWarn
	size += sizeof( bool ); //  bool    m_ExportToSceneDir
	size += GetStringSize(m_ModelDir);
	//size += ModelExportOptions::vMAX_LENGTH_DIR_NAME;	//  TSTR    m_ModelDir;
	size += sizeof( bool ); //  bool    m_IncSkelData
	size += sizeof( bool ); //  bool    m_FlipVerts
	size += sizeof( int );  //  int     m_ExportMode      v8
	size += GetStringSize(m_GroupName);
	//size += ModelExportOptions::vMAX_LENGTH_GROUP_NAME;	//  TSTR    m_GroupName;
	size += sizeof( bool );	//  bool    m_AlwaysLOD       v9
	size += sizeof( float );//  float   m_LODLowerLimit   v9
	size += sizeof( int );  //  int     m_LODNumLevels    v9
	size += sizeof( bool ); //  bool    m_LODProgressive  v9
	size += sizeof( bool ); //  bool    m_ExportByName    v10
	size += sizeof( bool ); //  bool    m_NoNormals       v11
	size += sizeof( float ) * m_LODNumLevels;	// float   m_LODDistances[]  v12
	size += sizeof( bool );	//  bool    m_DumpDebugData   v13
	size += sizeof( bool );	//  bool    m_AllowPropOverrides  v14

	return size;
}

void ModelExportOptions::Store(void* data)
{ FUNC_ENTER("ModelExportOptions::Store"); 
	unsigned char* pos = (unsigned char*)data;

	m_Version = vVERSION_NUMBER;
	Write(&pos, sizeof(int),  &m_Version);
	Write(&pos, sizeof(bool), &m_Optimized);
	WriteString(&pos, m_ModelName);
	Write(&pos, sizeof(bool), &m_ExportQN);
	Write(&pos, sizeof(bool), &m_DirWarn);
	Write(&pos, sizeof(bool), &m_ExportToSceneDir);
	WriteString(&pos, m_ModelDir);
	Write(&pos, sizeof(bool), &m_IncSkelData);
	Write(&pos, sizeof(bool), &m_FlipVerts);
	Write(&pos, sizeof(int),  &m_ExportMode);
	WriteString(&pos, m_GroupName);
	Write(&pos, sizeof(bool), &m_AlwaysLOD);
	Write(&pos, sizeof(float), &m_LODLowerLimit);
	Write(&pos, sizeof(int), &m_LODNumLevels);
	Write(&pos, sizeof(bool), &m_LODProgressive);
	Write(&pos, sizeof(bool), &m_ExportByName);
	Write(&pos, sizeof(bool), &m_NoNormals);
	
	for(int i = 0; i < m_LODNumLevels; i++)
		Write(&pos, sizeof(float), &m_LODDistances[i]);

	Write(&pos, sizeof(bool), &m_DumpDebugData);
	Write(&pos, sizeof(bool), &m_AllowPropOverrides);
}

void ModelExportOptions::Retrieve(void* data)
{ FUNC_ENTER("ModelExportOptions::Retrieve"); 
	unsigned char* pos = (unsigned char*)data;

	Read(&pos, sizeof(int), &m_Version);
	Read(&pos, sizeof(bool), &m_Optimized);
	GetString(&pos, m_ModelName);
	Read(&pos, sizeof(bool), &m_ExportQN);
	Read(&pos, sizeof(bool), &m_DirWarn);
	Read(&pos, sizeof(bool), &m_ExportToSceneDir);
	GetString(&pos, m_ModelDir);
	Read(&pos, sizeof(bool), &m_IncSkelData);
	Read(&pos, sizeof(bool), &m_FlipVerts);
	Read(&pos, sizeof(int), &m_ExportMode);
	GetString(&pos, m_GroupName);
	Read(&pos, sizeof(bool), &m_AlwaysLOD);
	Read(&pos, sizeof(float), &m_LODLowerLimit);
	Read(&pos, sizeof(int), &m_LODNumLevels);
	Read(&pos, sizeof(bool), &m_LODProgressive);
	Read(&pos, sizeof(bool), &m_ExportByName);
	Read(&pos, sizeof(bool), &m_NoNormals);
	
	for(int i = 0; i < m_LODNumLevels; i++)
		Read(&pos, sizeof(float), &m_LODDistances[i]);

	if (m_Version >= 0x000D)
		Read(&pos, sizeof(bool), &m_DumpDebugData);

	if (m_Version >= 0x000E)
		Read(&pos, sizeof(bool), &m_AllowPropOverrides);
}
