#include "FuncEnter.h"

#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <Export/ModelExport.h>
#include <Export/ScriptExport.h>
#include <Export/TextureExporter.h>
#include <Export/Strip/Strip.h>
#include <Misc/GenCrc.h>
#include <Misc/Util.h>
#include <process.h>
#include "../UI/OKtoAll.h"
#include "../misc/mdir.h"
#include "skeleton.h"
#include <modstack.h>	// MAXSDK: Modifier stack
#include <ifnpub.h>		// MAXSDK: Function Publishing
#include "../misc/maxutil.h"
#undef __TRIG__
#include "trig.h"
#include "ExportOptions.h"

#define MULTIRES_CLASS_ID  Class_ID(0x6a9e4c6b, 0x494744dd)
#define PB_VERTEXCOUNT     1
#define PB_VERTEXPERCENT   2
#define PB_GENERATE        10

#define MODEL_KEYFRAMED    "Model Keyframed???"

extern Interface* gInterface;

class ModelExporter : public IModelExporter
{
public:
	NxModel*		ExtractModelData( Tab <INode *>& exportable_nodes, bool skinned, float scale, INode* rootBone = NULL, bool bCutscene = false );

	bool	SpawnPostExportProcessor( void );
	void	EnumerateExportableNodes( void );
	bool	SaveTextures( char* modelName = NULL );
	bool	SaveModel( char* modelName = NULL );
	bool	SaveGeometry( IoUtils::CVirtualOutputFile &file );

	bool    BuildModelSkelData(NxModel* model,INode* node);
	int		FindBone(NxModel* model,unsigned long crc);

	bool	ExportIndividualModelSet(char* modelName = NULL);

	// From Exporter
	bool	DoExport( ExportType type );
	bool	Save( void );

	bool	CheckForKeys(INode* node);

	inline  void SetExportOptions(ModelExportOptions* expopt) { FUNC_ENTER("ModelExporter::SetExportOptions");  m_export_options = *expopt; }
	void    SetExportModel(NxModel* model) { FUNC_ENTER("ModelExporter::SetExportModel");  m_model = model; }

private:

	Modifier*	EvalLODData( INode* node );
	void		EvalLODData( Modifier* mod, INode* node, NxObject* nxobj );
	void		enumerate_visible_nodes( INode* root );

	Tab <INode *>		m_exportable_nodes;	
	ModelExportOptions	m_export_options;
	NxModel*			m_model;
	CStr                strModelName;				// The name of the active model being exported
};

static	ModelExporter	s_model_exporter;

IModelExporter*	GetModelExporter( void )
{ FUNC_ENTER("GetModelExporter"); 
	return &s_model_exporter;
}

DWORD WINAPI ModelExportProgressFunc(LPVOID arg) 
{ FUNC_ENTER("ModelExportProgressFunc"); 
    return(0);
}

int		ModelExporter::FindBone(NxModel* model,unsigned long crc)
{ FUNC_ENTER("ModelExporter::FindBone"); 
	int nBones = model->nBones;

	for(int i=0;i<nBones;i++)
	{
		if (model->bones[i].crc == crc)
			return i;
	}

	return -1;
}

void DumpMatrix(char* name, Mth::Matrix mat)
{ FUNC_ENTER("DumpMatrix"); 
	// Don't dump unless the file already exists
	FILE* fp = fopen("c:\\modeldbg.txt","r");

	if (!fp)
		return;

	fclose(fp);

	fp = fopen("c:\\modeldbg.txt","a");

	fprintf(fp,"New Matrix (%s):\n",name);

	for(int i=0;i<4;i++)
	{
		for(int j=0;j<4;j++)
		{
			fprintf(fp,"%f ",mat[i][j]);
		}

		fprintf(fp,"\n");
	}

	fprintf(fp,"\n");

	fclose(fp);
}

void DumpMatrix(char* name, Matrix3 mat)
{ FUNC_ENTER("DumpMatrix"); 
	// Don't dump unless the file already exists
	FILE* fp = fopen("c:\\modeldbg.txt","r");

	if (!fp)
		return;

	fclose(fp);
	
	fp = fopen("c:\\modeldbg.txt","a");

	fprintf(fp,"New Matrix (%s):\n",name);

	for(int i=0;i<4;i++)
	{
		Point3 row;
		row = mat.GetRow(i);

		fprintf(fp,"%f %f %f",row.x,row.y,row.z);

		fprintf(fp,"\n");
	}

	fprintf(fp,"\n");

	fclose(fp);
}

bool	ModelExporter::BuildModelSkelData(NxModel* model,INode* node)
{ FUNC_ENTER("ModelExporter::BuildModelSkelData"); 
	ModelExportOptions meo;
	GetModelExportOptions(&meo);

	MessageBoxReset(MODEL_KEYFRAMED);

	if (!meo.m_IncSkelData)
		return false;

	// Allocate memory for bone data
	if (model->m_Objects.Count()==0)
		return false;

	// All objects in the model should be connected in the hierarchy
	// (at least to the root anyway, so it should be safe to pick the first one)
	CSkeletonData csData(node,false);
	CStr name = node->GetName();

	int nKids = node->NumberOfChildren();

	model->nBones = csData.GetCount();

	model->bones = new NxModel::BoneDesc[model->nBones];

	// Now go through all the bones and build up a list of corresponding parent matricies
	for(int i=0;i<model->nBones;i++)
	{
		INode* bone = csData.GetBone(i);
		CStr   name = bone->GetName();

		Control* TMController  = bone->GetTMController();
		Control* PosController;
		Control* RotController;

		if (TMController)
		{
			PosController = TMController->GetPositionController();
			RotController = TMController->GetRotationController();
			
			if (PosController->NumKeys() > 0 ||
				RotController->NumKeys() > 0)
			{
				char strErr[256];
				sprintf(strErr, "WARNING!  The node '%s' contains keyframed data.  Do you want to continue exporting the model?", (char*)name);

				ReportWarning( strErr );
				int bResult = MessageBoxAll(gInterface->GetMAXHWnd(), strErr, MODEL_KEYFRAMED, MB_ICONWARNING| MB_OKCANCEL);

				if (bResult == IDNO)
					return false;
			}
		}

		model->bones[i].crc = GenerateCRC(name);

		Matrix3 tm = bone->GetNodeTM(0);

		// The position will be off if the user has moved the pivot point
		// Compensate for moved pivot points
		Point3 pivotOffset = bone->GetObjOffsetPos();

		Point3 trans = tm.GetTrans();

		// Rotate the root bone
		if (i==0)
			tm.RotateX(DegToRad(-90));

		// Swap coordSystem
		//tm.RotateX(DegToRad(-90));

		tm.NoScale();
		INode* parentNode = bone->GetParentNode();
		CStr   parentName = parentNode->GetName();

		Matrix3 parentTM;

		if (parentNode == gInterface->GetRootNode())
			parentTM.IdentityMatrix();
		else
			parentTM = parentNode->GetNodeTM(0);

		parentTM.NoScale();
		DumpMatrix(parentName + " PARENT",parentTM);
		
		DumpMatrix(name + "Orig TM",tm);

		tm = tm * Inverse(parentTM);

		DumpMatrix(name + "Post TM",tm);

		/*
		quat=Quat(tm);

		// make sure they're all anims start off with a positive w.
		// (only do this on frame 0!)
		if (quat.w<0.0f)
		{
			quat.x=-quat.x;
			quat.y=-quat.y;
			quat.z=-quat.z;
			quat.w=-quat.w;
		}

		prevQuat = quat;
		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		*/
		
		// MAX is right-handed as well, so this should be OK in theory
		model->bones[i].mat.Ident();
		
		Point3 row;
		row = tm.GetRow(0);
		model->bones[i].mat[Mth::RIGHT][0] = row.x;
		model->bones[i].mat[Mth::RIGHT][1] = row.y;
		model->bones[i].mat[Mth::RIGHT][2] = row.z;
		
		row = tm.GetRow(1);
		model->bones[i].mat[Mth::UP][0] = row.x;
		model->bones[i].mat[Mth::UP][1] = row.y;
		model->bones[i].mat[Mth::UP][2] = row.z;

		row = tm.GetRow(2);
		model->bones[i].mat[Mth::AT][0] = row.x;
		model->bones[i].mat[Mth::AT][1] = row.y;
		model->bones[i].mat[Mth::AT][2] = row.z;

		row = tm.GetRow(3);
		model->bones[i].mat[Mth::POS][0] = row.x;
		model->bones[i].mat[Mth::POS][1] = row.y;
		model->bones[i].mat[Mth::POS][2] = row.z;

		DumpMatrix(name,model->bones[i].mat);
	}

	return true;
}

bool	ModelExporter::SaveGeometry( IoUtils::CVirtualOutputFile &file )
{ FUNC_ENTER("ModelExporter::SaveGeometry"); 
	int i, j, k, num_obj, num_faces, num_verts;

	// TODO: Hierarchy matrix save

	num_obj = m_model->m_Objects.Count();

	file.Write((const char*) &num_obj, sizeof( int ));
	gInterface->ProgressStart(_T("Exporting Geometry"), TRUE, 
							ModelExportProgressFunc, NULL );
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxFace* face_list;
		NxObject* object;
		Point3 center;
		float radius;
		unsigned long checksum;

		object = m_model->m_Objects[i];
		vert_list = object->m_Verts;
		face_list = object->m_Faces;
		num_faces = object->m_NumFaces;
		num_verts = object->m_NumVerts;

		
		// Write out the object's checksum		
		checksum = GenerateCRC( object->m_Name );
		file.Write((const char*) &checksum, sizeof( unsigned long ));		

		if (object->m_Flags & NxObject::mINVISIBLE)
		{
			int zzz;
			zzz=0;
		}

		// Write out the flags which describe the object's properties
		object->m_Flags |= NxObject::mHASVERSIONINFO;

		if (object->m_Flags & NxObject::mINVISIBLE)
		{
			int zzz;
			zzz=0;
		}

		if (m_model->nBones > 0)
			object->m_Flags |= NxObject::mSKELETALMODEL;

		file.Write((const char*) &object->m_Flags, sizeof( int ));

		// Write out the index of the object's parent matrix
		if (object->m_Flags & NxObject::mSKELETALMODEL)
		{
			int index = FindBone(m_model,checksum);
			file.Write((const char*) &index, sizeof( int ));
		}

		// Write out the number of sets of UVs
		file.Write((const char*) &object->m_NumUVSets, sizeof( int ));

		// Write out bounding box/sphere data
		file.Write((const char*) &object->m_BoundingBox.pmin, sizeof( Point3 ));
		file.Write((const char*) &object->m_BoundingBox.pmax, sizeof( Point3 ));

		center = object->m_BoundingBox.Center();
		radius = (( object->m_BoundingBox.pmax - object->m_BoundingBox.pmin ).Length()) / 2.0f;
		file.Write((const char*) &center, sizeof( Point3 ));
		file.Write((const char*) &radius, sizeof( float ));

		file.Write((const char*) &num_verts, sizeof( int ));
		for( j = 0; j < num_verts; j++ )
		{
			NxVertex* vert;			
			
			vert = &vert_list[j];

			// Write out first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					file.Write((const char*) &vert->m_TexCoord[k].x, sizeof( float ));
					file.Write((const char*) &vert->m_TexCoord[k].y, sizeof( float ));
				}
			}

			// Write out colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				file.Write((const char*) &vert->m_Color.r, sizeof( float ));
				file.Write((const char*) &vert->m_Color.g, sizeof( float ));
				file.Write((const char*) &vert->m_Color.b, sizeof( float ));
				file.Write((const char*) &vert->m_Color.a, sizeof( float ));
			}				

			// Write out normals
			if( object->m_Flags & ( NxObject::mNORMALS | NxObject::mSS_NORMALS ))
			{					
				file.Write((const char*) &vert->m_Normal.x, sizeof( float ));
				file.Write((const char*) &vert->m_Normal.y, sizeof( float ));
				file.Write((const char*) &vert->m_Normal.z, sizeof( float ));
			}

			// Write out weighting info
			if( object->m_Flags & NxObject::mSKINNED )
			{
				file.Write((const char*) &vert->m_Weight[0], sizeof( float ));
				file.Write((const char*) &vert->m_Weight[1], sizeof( float ));
				file.Write((const char*) &vert->m_Weight[2], sizeof( float ));

				file.Write((const char*) &vert->m_WeightedIndex[0], sizeof( int ));
				file.Write((const char*) &vert->m_WeightedIndex[1], sizeof( int ));
				file.Write((const char*) &vert->m_WeightedIndex[2], sizeof( int ));	
				
				if( object->m_Flags & NxObject::mHAS4WEIGHTS )
				{
					file.Write((const char*) &vert->m_Weight[3], sizeof( float ));
					file.Write((const char*) &vert->m_WeightedIndex[3], sizeof( int ));
				}
			}

			// Write out vc wibble data
			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				file.Write((const char*) &vert->m_WibbleIndex, sizeof( char ));
				file.Write((const char*) &vert->m_WibbleOffset, sizeof( char ));	
			}

			// Write out positions				
			file.Write((const char*) &vert->m_Pos.x, sizeof( float ));
			file.Write((const char*) &vert->m_Pos.y, sizeof( float ));
			file.Write((const char*) &vert->m_Pos.z, sizeof( float ));		
		}

		file.Write((const char*) &num_faces, sizeof( int ));
		for( j = 0; j < num_faces; j++ )
		{
			// Write out the material checksum for this face
			file.Write((const char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Write out the face normal
			file.Write((const char*) &face_list[j].m_Normal, sizeof( Point3 ));

			// Versioning data isn't saved with NxObject.  So we are using this flag
			// to indicate the addition of CAS Face data
			face_list[j].m_FaceFlags |= mFD_CASFACEFLAGSEXIST;
			face_list[j].m_MinFaceFlags |= mFD_CASFACEFLAGSEXIST;

			// Write out the face flags
			file.Write((const char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Write out the minimalist face flags
			file.Write((const char*) &face_list[j].m_MinFaceFlags, sizeof( FlagType ));

			// Write out the CAS face flags
			file.Write((const char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));

			// Write out the vertex indices
			file.Write((const char*) face_list[j].m_Vertex, sizeof( int ) * 3 );
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Write out CAS face removal flags
			file.Write((const char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		// Write out KBias if appropriate
		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Write out the KBias
			file.Write((const char*)&object->m_KBias,sizeof( float ));
		}

		// Write out LOD data if appropriate
		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Write out the LOD info
			file.Write((const char*)&object->m_LODVersion,sizeof(unsigned int));
			file.Write((const char*)&object->m_LODFlags,sizeof(int));

			if (object->m_LODFlags & NxObject::mMASTER)
			{
				file.Write((const char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));
				file.Write((const char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
			}

			if (object->m_LODFlags & NxObject::mSLAVE)
			{
				file.Write((const char*)&object->m_LODSlave.m_masterCRC,sizeof(int));
			}
		}

		if (object->m_Flags & NxObject::mHASINTLODINFO)
		{
			file.Write((const char*)&object->m_LODLevels,sizeof(int));
			
			for(int lodlev = 0; lodlev < object->m_LODLevels; lodlev++)
			{
				file.Write((const char*)&object->m_LODinfo[lodlev].numFaces, sizeof(int));

				file.Write((const char*)object->m_LODinfo[lodlev].faces,
					       sizeof(NxLODFace) * object->m_LODinfo[lodlev].numFaces);

#ifndef DISABLE_USER_LODDIST_SUPPORT
				if (object->m_Flags & NxObject::mHASLODDISTS)
					file.Write((const char*)&object->m_LODinfo[lodlev].distance, sizeof(float));
#endif
			}
		}

		if (object->m_Flags & NxObject::mBILLBOARD)
		{
			file.Write((const char*)&object->m_BillboardType, sizeof(NxObject::BillboardType));
			file.Write((const char*)&object->m_BillboardOrigin, sizeof(Point3));
			file.Write((const char*)&object->m_PivotPos, sizeof(Point3));
			file.Write((const char*)&object->m_PivotAxis, sizeof(Point3));
		}

		// Write out version info
		file.Write((const char*)&object->m_Version,sizeof(unsigned int));

		// Write out relative info v1
		file.Write((const char*)&object->m_ParentCRC,sizeof(unsigned long));
		file.Write((const char*)&object->m_NumChildren,sizeof(int));
		file.Write((const char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			//file.close();
			return false;
		}
		gInterface->ProgressUpdate( i * 100 / num_obj );
	}

	// Write out model matrix hierarchy information
	// This is only read if an object(s) were read in with the mSKELETALMODEL flag
	if (m_model->nBones > 0)
	{
		int version = NxModel::MODEL_VERSION;
		file.Write((const char*) &version, sizeof( int ));

		file.Write((const char*)&m_model->nBones,sizeof( int ));
		file.Write((const char*)m_model->bones,sizeof(NxModel::BoneDesc) * m_model->nBones);
	}

	gInterface->ProgressEnd();
	
	return true;
}

bool	ModelExporter::SaveModel( char* modelName )
{ FUNC_ENTER("ModelExporter::SaveModel"); 
	//fstream file;
	IoUtils::CVirtualOutputFile file;
	TSTR mdl_name, mdl_dir, base_name;
	char* project_root;
	int version;
	DWORD attribs;
	bool sky_export;
	
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		if (modelName)
			base_name = modelName;
		else
			base_name = m_export_options.m_ModelName;
	}	

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		mdl_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name;
	}
	else
	{
		if( m_export_options.m_ModelDir == TSTR( "" ))
		{
			mdl_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name;
		}
		else
		{
			mdl_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name;
		}
	}

	//CreateDirectory( mdl_dir, NULL );	
	MDir(mdl_dir);	// Make directory hierarchy
	CreateDirectory( "c:/temp", NULL );	// for image processing

	mdl_name = mdl_dir + TSTR( "/" ) + base_name + TSTR( ".mdl" );
	attribs = GetFileAttributes( mdl_name );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			int result;
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", mdl_name );
			result = MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( mdl_name, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = mdl_name;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				attribs = GetFileAttributes( mdl_name );
				if(( attribs & FILE_ATTRIBUTE_READONLY ) || ( attribs == -1 ))
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Model",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}

	
	//file.open( mdl_name, ios::out | ios::binary );
	file.Init( vMAX_MODEL_FILE_SIZE );

	version = NxMaterial::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	file.Write((const char*) &version, sizeof( int ));
	sky_export = false;
	file.Write((const char*) &sky_export, sizeof( bool ));

	if( SaveMaterials( file, m_model ) == false )
	{
		//file.close();
		file.Uninit();
		return false;
	}

	if( SaveGeometry( file ) == false )
	{
		//file.close();
		file.Uninit();
		return false;	
	}	

	//file.close();
	file.Save( mdl_name );
	return true;	
}

bool	ModelExporter::SpawnPostExportProcessor( void )
{ FUNC_ENTER("ModelExporter::SpawnPostExportProcessor"); 
	int result;
	char *args[5];
	char path[_MAX_PATH], conv_path[_MAX_PATH];	
	TSTR scene_name, scene_dir, base_name, tex_name;
	char* project_root;
	
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	/*if( m_export_type == vQUICK_VIEW )
	{
		base_name = "quick";
	}
	else*/
	{
		if (strModelName.Length() > 0)
			base_name = strModelName;
		else
			base_name = m_export_options.m_ModelName;;
	}	

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name;
	}
	else
		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name;

	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".mdl" );
	tex_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".tex" );

	sprintf( path, "-f%s", scene_name );
	sprintf( conv_path, "%s\\bin\\win32\\sceneconv.exe", project_root );

	args[0] = conv_path;
	switch( GetQuickviewPlatform())
	{
		case vQUICKVIEW_XBOX:
			args[1] = "-px";
			break;
		case vQUICKVIEW_PS2:
			args[1] = "-pp";
			break;
		case vQUICKVIEW_NGC:
			args[1] = "-pg";
			break;
		case vQUICKVIEW_PCVIEW:
			args[1] = "-pc";
			break;
	}
	args[2] = path;	

	if (UseSceneConvOptimize())
	{
		args[3] = "-o";
		args[4] = NULL;
	}
	else
		args[3] = NULL; 

	//result = spawnv( _P_WAIT, args[0], args );
	result = ExecuteCommand( args );

	if (result < 0)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Sceneconv has failed.  If you are exporting without textures you should re-export with textures.", "SceneConv Failed", MB_ICONSTOP|MB_OK);
		return false;
	}

	if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
	{
		char copy_path[_MAX_PATH];

		sprintf( path, "%s.xbx", scene_name );
		sprintf( copy_path, "%s\\bin\\win32\\copyx.bat", project_root );
		args[0] = copy_path;
		args[1] = path;

		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );

		sprintf( path, "%s.xbx", tex_name );
		args[1] = path;
		
		//result = spawnv( _P_WAIT, args[0], args );
		result = ExecuteCommand( args );
	}
	else if( GetQuickviewPlatform() == vQUICKVIEW_NGC )
	{
		char path[_MAX_PATH], copy_path[_MAX_PATH];
		char* proj_root, *path_ptr;

		proj_root = getenv( APP_ENV );

		// We need to copy the .scn.ngc file to the ndata directory
		sprintf( path, "%s.ngc", scene_name );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );

		// We need to copy the .scn.ngc.gd file to the ndata directory
		sprintf( path, "%s.ngc.gd", scene_name);
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );

		// We need to copy the .tex.ngc file to the ndata directory
		sprintf( path, "%s.ngc", tex_name );
		path_ptr = strstr( strlwr( path ), strlwr( proj_root ));
		if( path_ptr )
		{
			path_ptr += strlen( proj_root );
			path_ptr += strlen( "\\data" );
		}
		sprintf( copy_path, "%s\\ndata%s", path_ptr );		
		CopyFile( path, copy_path, false );		
	}

	return true;
}

bool	ModelExporter::SaveTextures( char* modelName )
{ FUNC_ENTER("ModelExporter::SaveTextures"); 
	TSTR texfile_name, usgfile_name, base_name;
	char* project_root;	
	ITextureExporter* tex_exp;
	
	tex_exp = GetTextureExporter();
	project_root = getenv( APP_ENV );
	if( project_root == NULL )
	{
		char strWarnBuf[256];
		sprintf(strWarnBuf,"You must set up your %s environment variable", APP_ENV);
		MessageBox( gInterface->GetMAXHWnd(), strWarnBuf,
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if (modelName)
	{
		base_name = modelName;
	}
	else
	{
		if( m_export_type == vQUICK_VIEW ||
			m_export_type == vQUICK_VIEW_NOTEX)
		{
			base_name = "quick";
		}
		else
		{
			base_name = m_export_options.m_ModelName;
		}
	}

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".tex" );
		usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".usg" );
	}
	else
	{
		if( m_export_options.m_ModelDir == TSTR( "" ))
		{
			texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR("/") + base_name + TSTR( ".tex" );
			usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + base_name + TSTR( ".usg" );
		}
		else
		{
			texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".tex" );
			usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".usg" );
		}
	}
	
	if (m_export_options.m_DirWarn)
	    if (!tex_exp->VerifyTexturePaths( TSTR( "models/" ) + base_name ))
			return false;

	return tex_exp->SaveTextureDictionary( texfile_name, usgfile_name );
}

bool	ModelExporter::Save( void )
{ FUNC_ENTER("ModelExporter::Save"); 
	if (strModelName.Length() > 0)
		return( SaveModel(strModelName) && SaveTextures(strModelName) );
	else
		return( SaveModel() && SaveTextures() );
}

bool	ModelExporter::ExportIndividualModelSet(char* modelName)
{ FUNC_ENTER("ModelExporter::ExportIndividualModelSet"); 
	// Write out a .qn file for the model
	IScriptExporter*  scriptExp = GetScriptExporter();
	ITextureExporter* tex_exp;

	tex_exp = GetTextureExporter();
	tex_exp->Reset();	

	// Dump model .qn
	if (m_export_options.m_ExportQN)
	{
		SceneExportOptions options;
		GetSceneExportOptions(&options);
		//options.m_SceneName=m_export_options.m_ModelName;

		//scriptExp->ExportScriptsFromSet(m_export_options.m_ModelName,m_exportable_nodes,true);
		if (modelName)
			scriptExp->ExportScriptsFromSet(modelName, m_exportable_nodes, true, modelName);
		else
			scriptExp->ExportScripts(options, true, modelName);
	}
	

	if( m_exportable_nodes.Count() == 0 )
	{
		// Nothing to export
		return false;
	}

	// Check (and warn) for duplicate object names
	int i,j;

	for( i = 0; i < m_exportable_nodes.Count(); i++)
	{
		for( j = 0; j < m_exportable_nodes.Count(); j++ )
		{
			if (j != i)
			{
				CStr name1 = m_exportable_nodes[i]->GetName();
				CStr name2 = m_exportable_nodes[j]->GetName();

				if( name1 == name2 )
				{
					char msg[1024];

					sprintf( msg, "Multiple objects are named %s. This will cause problems.  Aborting export.", (char*)m_exportable_nodes[i]->GetName() );
					MessageBoxAll( gInterface->GetMAXHWnd(), msg, "Multiple named objects!", MB_OK|MB_TASKMODAL);
					ReportWarning( msg );
					return false;
				}
			}
		}
	}

	VerifyMemory();

	// At this point, we have gathered a list of nodes which represent the exportable geometry.
	// Now parse these nodes and store relavent data in our own generic database
	m_model = ExtractModelData( m_exportable_nodes, false, 1.0f );

	VerifyMemory();

	// If something went wrong or it was cancelled, don't continue
	if( m_model == NULL )
	{
		return false;
	}

	// Load the textures needed for the current export set
	tex_exp->LoadTextureData();

	// Here output the model and its materials, either via the network or to disk
	// ModelName is used by SaveModel.  Had to keep as a member rather than param to keep compat with Exporter class
	if (modelName)
		strModelName = modelName;
	else
		strModelName = "";

	if( Save() )
	{		
		if( SpawnPostExportProcessor())
		{
			// Here do quickview stuff
			if( m_export_type == vRESET_AND_RELOAD ||
				m_export_type == vRESET_AND_RELOAD_NOTEX)
			{
				char exec_str[128];
				char *viewer_root;

				if( viewer_root = getenv( APP_ENV ))
				{			
					char viewer_app[_MAX_DIR + _MAX_FNAME];

					sprintf( viewer_app, "%s\\build\\NGPSgnu\\%s", viewer_root, vVIEWER_APP_NAME );
					sprintf( exec_str, "ps2run -r %s", viewer_app );
					WinExec( exec_str, SW_HIDE );
				}
			}
			else if( m_export_type == vQUICK_VIEW ||
				     m_export_type == vQUICK_VIEW_NOTEX)
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				if (modelName)
					sprintf( msg.m_Filename, modelName );
				else
					sprintf( msg.m_Filename, m_export_options.m_ModelName );

				if( GetQuickviewPlatform() == vQUICKVIEW_PS2 )
				{
					Net::MsgDesc msgdesc;
					msgdesc.m_Id     = Net::vMSG_ID_QUICKVIEW;
					msgdesc.m_Length = sizeof( Net::MsgQuickview );
					msgdesc.m_Data   = &msg;

					gClient->EnqueueMessageToServer( &msgdesc );
				}
				else if( GetQuickviewPlatform() == vQUICKVIEW_XBOX )
				{
					char* server_ip;

					server_ip = getenv( vXBOX_SERVER_IP_VARIABLE );
					if( server_ip == NULL )
					{
						MessageBox( gInterface->GetMAXHWnd(), "You must set up your XBOX_VIEWER_IP environment variable if you want real-time updates",
										"Warning", MB_OK|MB_TASKMODAL);		
					}
					else
					{
						gClient->SendMessageTo( Net::vMSG_ID_QUICKVIEW, sizeof( Net::MsgQuickview ), &msg,
												inet_addr( server_ip ), Net::vEXPORT_COMM_PORT, 0 );						
					}
				}								
				else if( GetQuickviewPlatform() == vQUICKVIEW_NGC )
				{
					char* server_ip;

					server_ip = getenv( vNGC_SERVER_IP_VARIABLE );
					if( server_ip == NULL )
					{
						MessageBox( gInterface->GetMAXHWnd(), "You must set up your NGC_VIEWER_IP environment variable if you want real-time updates",
										"Warning", MB_OK|MB_TASKMODAL);		
					}
					else
					{
						char msg_data[Net::Manager::vMAX_PACKET_SIZE];
						bool result;
						char* byte_size;
						unsigned short msg_len;

						msg_len = sizeof( Net::MsgQuickview );
						byte_size = (char*) &msg_len;
						msg_data[0] = Net::vMSG_ID_QUICKVIEW;
						msg_data[2] = *byte_size;
						byte_size++;
						msg_data[1] = *byte_size;
						memcpy( &msg_data[3], &msg, msg_len );					
						result = gClient->SendTo( inet_addr( server_ip ), Net::vEXPORT_COMM_PORT, 
								msg_data, msg_len + Net::Manager::vMSG_HEADER_LENGTH_WITH_SIZE, 0 );
					}
				}								
			}
		}
	}

	return true;
}

bool	ModelExporter::DoExport( ExportType type )
{ FUNC_ENTER("ModelExporter::DoExport"); 
	CStr export_name;

	// Ensure that an object is select if doing skeletal animation export
	if (m_export_options.m_IncSkelData && !m_export_options.m_ExportByName)
	{
		if (gInterface->GetSelNodeCount()==0)
		{
			MessageBox(gInterface->GetMAXHWnd(),"You must select the model object you are exporting if you are exporting with skeletal animation data.","Model Skeletal Animation Export",MB_ICONWARNING|MB_OK);
			return false;
		}
	}

	// Reset all warning mesages
	MessageBoxResetAll();

	GetModelExportOptions( &m_export_options );
	m_export_type = type;

	// If we're exporting objects by name then only one object (and its children) will be included
	// in the export set per file
	if (m_export_options.m_ExportByName)
	{
		// Scan through all the nodes in the scene and build up a list of the nodes that should be exported
		// for each individual model file
		INodeTab nodeList;
		GetAllNodes(nodeList);

		for(int i = 0; i < nodeList.Count(); i++)
		{
			CStr strExportName;
			
			//if (!nodeList[i]->GetUserPropString("ModelExportName", strExportName))
			if (!nodeList[i]->GetUserPropString("ExportName", strExportName))
				continue;

			if (strExportName.Length() == 0)
				continue;

			// Add this node and it's children to the list
			m_exportable_nodes.ZeroCount();
			GetAllNodes(m_exportable_nodes, nodeList[i]);

			//if (!ExportIndividualModelSet())
			//	return false;

			if (!ExportIndividualModelSet(strExportName))
				return false;

			/*
			// True prevents the texture dictionary from getting reset
			if (i == 0)
			{
				if (!ExportIndividualModelSet(strExportName, false))
					return false;
			}
			else
			{
				if (!ExportIndividualModelSet(strExportName, true))
					return false;
			}
			*/
		}
	}
	else
	{
		EnumerateExportableNodes();
		
		if (!ExportIndividualModelSet())
			return false;
	}

	return true;
}

NxModel*		ModelExporter::ExtractModelData( Tab <INode *>& exportable_nodes, bool skinned, float scale, INode* rootBone, bool bCutscene )
{ FUNC_ENTER("ModelExporter::ExtractModelData"); 
	int i;
	INode* node;
	NxModel* model;
	int num_exportable_nodes;
	model = new NxModel;
	model->m_Skinned = skinned;
	model->m_SkinRootBone = rootBone;
	model->m_DynamicallyLit = true;

	VerifyMemory();

	num_exportable_nodes = exportable_nodes.Count();

	gInterface->ProgressStart(_T("Enumerating exportable objects"), TRUE, 
							ModelExportProgressFunc, NULL );
	

	for( i = 0; i < num_exportable_nodes; i++ )
	{
		NxObject* object;

		node = exportable_nodes[i];
		CStr nodeName = CStr("Enum Nodes Exporting: ") + node->GetName() + "\n";
		OutputDebugString(nodeName);

		//object = model->CreateObjectFromNode( node, scale );

		////  Objects with the class LevelObject should be in local coordinates /////////
		CStr class_name, lclass_name;

		if( !node->GetUserPropString( "Class", class_name ))
		{
			node->GetUserPropString( "class", class_name );
		}

		lclass_name = class_name;
		lclass_name.toLower();

		// Need to apply MultiRes LOD modifier (if necessary) at this point so
		// that CreateObjectFromNode evaluates the mesh with MultiRes
		//Modifier* pMultiRes = EvalLODData( node );

		// Determine if object should be exported using local coordinate system
		// If we're exporting with skeleton data everything must export to local coords
		
		// Determine which parameters to use for extracting MultiRes LOD data
		ULONG flags = 0;
		
		ExportOptions* export_options = GetExportOptions();

		if (export_options->m_ExportType == ExportOptions::vEXPORT_SKIN)
			flags = GeoDatabase::EVALLOD_SKIN;
		else
			flags = GeoDatabase::EVALLOD_MODEL;
		
		// Standard model exporter shouldn't swap, done by animconv
		//flags |= GeoDatabase::NOCOORD_SWAP;

		// Don't export normals if the user has choosen not to
		if (m_export_options.m_NoNormals)
			flags |= GeoDatabase::NONORMALS;

		if (bCutscene)
		{
			flags |= GeoDatabase::LOCAL_COORDS | GeoDatabase::OFFSET_PIVOT | GeoDatabase::OFFSET_ROTATION | GeoDatabase::NOCOORD_SWAP;

			CStr name = exportable_nodes[0]->GetName();
			Quat q = exportable_nodes[0]->GetObjOffsetRot();

			float rot[3];
			QuatToEuler(q, rot);

			int zzz;
			zzz = 0;
		}

		if (m_export_options.m_IncSkelData)
		{
			ULONG flags = GeoDatabase::LOCAL_COORDS | GeoDatabase::SKEL_EXPORT | GeoDatabase::OFFSET_PIVOT | GeoDatabase::NOCOORD_SWAP;

			if (export_options->m_ExportType == ExportOptions::vEXPORT_SKIN)
				flags |= GeoDatabase::EVALLOD_SKIN;
			else
				flags |= GeoDatabase::EVALLOD_MODEL;

			if (m_export_options.m_FlipVerts)
				flags |= GeoDatabase::FLIP_VERTS;

			VerifyMemory();

			object = model->CreateObjectFromNode( node, scale, flags );
		}
		else
		{
			if (lclass_name == CStr("levelobject"))
				object = model->CreateObjectFromNode( node, scale, GeoDatabase::LOCAL_COORDS |
				                                                   flags);
			else
				object = model->CreateObjectFromNode( node, scale, flags);
		}
		/////////////////////////////////////////////////////////////////////////////////

		if( object )
		{
			model->m_Objects.Append( 1, &object );		
//			EvalLODData( pMultiRes, node, object );

			// Dump the model data out to a debug file if appropriate
			ModelExportOptions meo;
			GetModelExportOptions(&meo);

			if (meo.m_DumpDebugData)
				object->FileDump("c:\\objdump.txt");
		}
		
		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			delete model;
			return NULL;
		}
		gInterface->ProgressUpdate( i * 100 / num_exportable_nodes );
	}

	// Build up the hierarchial model data
	if (exportable_nodes.Count()>0 && m_export_options.m_IncSkelData)
	{
		int selCount = gInterface->GetSelNodeCount();
		
		if (selCount > 0)
		{
			INode* node = gInterface->GetSelNode(0);
			
			if (!BuildModelSkelData(model,node))
				return NULL;
		}
		else
		{
			MessageBox(gInterface->GetMAXHWnd(),"Select the model that you wish to export.","No model selected for export",MB_ICONWARNING|MB_OK);
			//BuildModelSkelData(model,exportable_nodes[0]);
		}

		/*
		// Find the first exportable node that has more than 1 child
		int cnt = exportable_nodes.Count();

		for(int i=0;i<cnt;i++)
		{
			if (exportable_nodes[i]->NumberOfChildren()>0)
			{
				BuildModelSkelData(model,exportable_nodes[0]);
				break;
			}
		}
		*/
	}

	gInterface->ProgressEnd();	
	
	return model;
}

void	ModelExporter::enumerate_visible_nodes( INode* root )
{ FUNC_ENTER("ModelExporter::enumerate_visible_nodes"); 
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		// Add only visible, non-frozen nodes
		if(	( child->IsHidden() == false ) &&
			( child->IsFrozen() == false ))
		{
			if( IsExportable( child ))
			{
				m_exportable_nodes.Append( 1, &child );			
			}
		}

		// Recursively add all visible children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_visible_nodes( child );
		}
	}
}

void	ModelExporter::EnumerateExportableNodes( void )
{ FUNC_ENTER("ModelExporter::EnumerateExportableNodes"); 
	m_exportable_nodes.ZeroCount();
	
	enumerate_visible_nodes( gInterface->GetRootNode());	
}

void	ModelExporter::EvalLODData(Modifier* mod, INode* node, NxObject* nxobj)
{ FUNC_ENTER("ModelExporter::EvalLODData"); 
	if (!mod || !node || !nxobj)
		return;

	IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);

	// Go through and evaluate the object at different LOD levels
	
	float start = m_export_options.m_LODLowerLimit;
	float step  = (100.0f - start) / (float)m_export_options.m_LODNumLevels;

	int idx = 0;
	Interval interval = FOREVER;

	float origper, origcount;
	pblk->GetValue(PB_VERTEXPERCENT, 0, origper, interval);
	interval = FOREVER;
	pblk->GetValue(PB_VERTEXCOUNT, 0, origcount, interval);

	if (m_export_options.m_LODProgressive)
	{
		Object* obj = node->EvalWorldState(0).obj;
		if (obj->CanConvertToType(triObjectClassID))
		{
			// Determine the number of verticies within the mesh
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
			Mesh& mesh = triObj->GetMesh();

			int nverts = mesh.numVerts;

			if (triObj != obj)
				triObj->DeleteThis();

			nxobj->m_LODLevels = nverts;
			nxobj->m_LODinfo   = new NxLODInfo[nverts];

			for(int cvert = nverts; cvert > 0; cvert--)
			{
				pblk->SetValue(PB_VERTEXCOUNT, 0, cvert);

				Object* obj = node->EvalWorldState(0).obj;
				if (obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

					Mesh& mesh = triObj->GetMesh();

					nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
					nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

					for(int cface = 0; cface < mesh.numFaces; cface++)
					{
						// Converts from MAX Vertex indicies to the indicies of the expanded verts
						nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
						nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
						nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];

						Point3 vert;
						
						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

						vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_MaxVert];
						
						if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_Pos)
							MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);
					}

					if (triObj != obj)
						triObj->DeleteThis();

					idx++;
				}
			}
		}
	}
	else
	{
		nxobj->m_LODLevels = m_export_options.m_LODNumLevels;
		nxobj->m_LODinfo   = new NxLODInfo[m_export_options.m_LODNumLevels];

		for(float per = 100.0f; per >= start; per -= step)
		{
			pblk->SetValue(PB_VERTEXPERCENT, 0, per);

			Object* obj = node->EvalWorldState(0).obj;
			if (obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

				Mesh& mesh = triObj->GetMesh();

				nxobj->m_LODinfo[idx].numFaces = mesh.numFaces;
				nxobj->m_LODinfo[idx].faces = new NxLODFace[mesh.numFaces];

				for(int cface = 0; cface < mesh.numFaces; cface++)
				{
					// Converts from MAX Vertex indicies to the indicies of the expanded verts
					nxobj->m_LODinfo[idx].faces[cface].v[0] = nxobj->m_MaxVerts[mesh.faces[cface].v[0]];
					nxobj->m_LODinfo[idx].faces[cface].v[1] = nxobj->m_MaxVerts[mesh.faces[cface].v[1]];
					nxobj->m_LODinfo[idx].faces[cface].v[2] = nxobj->m_MaxVerts[mesh.faces[cface].v[2]];

					Point3 vert;
/*					
					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[0]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[1]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);

					vert = mesh.verts[nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_MaxVert];
					
					if (vert != nxobj->m_Verts[nxobj->m_LODinfo[idx].faces[cface].v[2]].m_Pos)
						MessageBox(gInterface->GetMAXHWnd(),"Failed to match vert!", "Vert match failed", MB_OK);
*/
				}

				if (triObj != obj)
					triObj->DeleteThis();

				idx++;
			}
		}
	}

	nxobj->m_Flags |= NxObject::mHASINTLODINFO;

	pblk->SetValue(PB_VERTEXPERCENT, 0, origper);
	pblk->SetValue(PB_VERTEXCOUNT, 0, origcount);
}

Modifier*	ModelExporter::EvalLODData(INode* node)
{ FUNC_ENTER("ModelExporter::EvalLODData"); 
	// Check if the node has a MultiRes modifier applied to it
	Object* obj = node->GetObjectRef();

	if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		int nMods = dobj->NumModifiers();

		for(int j = 0; j < nMods; j++)
		{
			Modifier* mod = dobj->GetModifier(j);

			// Check if this modifier is a MultiRes modifier
			if (mod->ClassID() == MULTIRES_CLASS_ID)
			{
				//EvalLODData(mod, node, nxobj);
				return mod;
			}
		}
	}

	// If we didn't find a MultiRes Modifier we'll add one
	if (m_export_options.m_AlwaysLOD)
	{
		Modifier* mod = (Modifier*)CreateInstance(OSM_CLASS_ID, MULTIRES_CLASS_ID);
		ModContext* mc = new ModContext(new Matrix3(1), NULL, NULL);

		IDerivedObject* dobj;

		if (obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			dobj = (IDerivedObject*)obj;
			dobj->AddModifier(mod, mc, dobj->NumModifiers());
		}
		else
		{
			dobj = CreateDerivedObject(obj);
			dobj->AddModifier(mod, mc, dobj->NumModifiers());
			node->SetObjectRef(dobj);
		}

		Object* cobj = node->EvalWorldState(0).obj;

		IParamBlock2* pblk = ((Animatable*)mod)->GetParamBlock(0);
		pblk->SetValue(PB_GENERATE, 0, 1);	// Force MultiRes Generate Function
		//EvalLODData(mod, node, nxobj);
		return mod;
	}

	return NULL;
}

bool	ModelExporter::CheckForKeys(INode* node)
{ FUNC_ENTER("ModelExporter::CheckForKeys"); 
	if (node->NumKeys() > 0)
		return true;

	return false;
}

NxModel::NxModel( void )
{ FUNC_ENTER("NxModel::NxModel"); 
	NxMaterial default_mat;

	m_Materials.AddToTail( &default_mat );

	nBones = 0;
	bones = NULL;
}

NxModel::~NxModel()
{ FUNC_ENTER("NxModel::~NxModel"); 
	if (bones)
		delete [] bones;
}
