/*
	AnimExporter.cpp
	Animation Exporter
	1-11-01
*/

#include "AnimExporter.h"
#include <direct.h>			// for _mkdir
#include <io.h>				// for _access, _chmod
#include <process.h>
#include <sys/stat.h>		// for access flags

#include "../../../../include/AnimExportFmt.h"
#include "../Misc/GenCrc.h"
#include "../Misc/mth.h"
#include "../Misc/util.h"
#include "AnimTol.h"
#include "UI/OkToAll.h"
#include "CutsceneExportDlg.h"
//#include "trig.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

#define PI  ((float)3.1415926535)
#define TWOPI ((float)6.283185307)
#define HALFPI ((float)1.570796326794895)

#define DEG_TO_RAD (PI/(float)180.0)
#define RAD_TO_DEG ((float)180.0/PI)
#define DegToRad(deg) (((float)deg)*DEG_TO_RAD)
#define RadToDeg(rad) (((float)rad)*RAD_TO_DEG)

#define  WRITE_ACCESS     0x02
#define  REMOVE_DBLKEYS
//#define  USE_EULERFIX

#define  DUMP_PRECOMPRESS_TFRAMES

static double RtQuatDiff( Mth::Quat& p_in1, Mth::Quat& p_in2 )
{ FUNC_ENTER("RtQuatDiff"); 
	double f_cos_theta =	( p_in1[X] * p_in2[X] ) + ( p_in1[Y] * p_in2[Y] ) +
							( p_in1[Z] * p_in2[Z] ) + ( p_in1[W] * p_in2[W] );

	bool   b_obtuse_theta = ( f_cos_theta < 0.0 );

	if( b_obtuse_theta )
	{
		// Flip one of the quaternions.
		f_cos_theta = ( f_cos_theta < -1.0 ) ? 1.0 : -f_cos_theta;
		p_in2[X] =-p_in2[X];
		p_in2[Y] =-p_in2[Y];
		p_in2[Z] =-p_in2[Z];
		p_in2[W] =-p_in2[W];
    }
    else
    {
		f_cos_theta = ( f_cos_theta > 1.0 ) ? 1.0 : f_cos_theta;
    }

	return f_cos_theta;
}

int AnimExporter::IsQLinear( const Gfx::SAnimQFrame* p_data, unsigned int start, unsigned int end, double error )
{ FUNC_ENTER("AnimExporter::IsQLinear"); 
	Mth::Quat at_start_frame;
	Mth::Quat at_end_frame;
	double lastFrameDiff = 0;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return LINEAR_TRUE;

	at_start_frame	= p_data[start].q;
	at_end_frame	= p_data[end].q;
	for( i = start + 1; i < end; ++i )
	{
		Mth::Quat at_this_frame				= p_data[i].q;
		Mth::Quat at_prev_frame             = p_data[i-1].q;
		Mth::Quat interpolated_at_this_frame;
		double diff_at_this_frame;
		double thisFrameDiff;

		interpolated_at_this_frame = Mth::Slerp(at_start_frame,
			                                    at_end_frame,
										        (float)(i - start) / (float)(end - start));

		//RpSkinAnimQuatSlerp( &at_start_frame, &at_end_frame, (float)( i - start ) / (float)( end - start ), &interpolated_at_this_frame );
		
		diff_at_this_frame			= RtQuatDiff( at_this_frame, interpolated_at_this_frame );

		thisFrameDiff = RtQuatDiff( at_this_frame, at_prev_frame);

		// If the sign changes from this frame to the next we need to consider the sequence non-linear
		// (thus causing a key to get added for it)
		if ( bFixReverseKeyInterpolation &&
			((lastFrameDiff < 0 && thisFrameDiff > 0) ||
			 (lastFrameDiff > 0 && thisFrameDiff < 0)))
		{
			nReverseQKeys++;
			fprintf(fpKeyRevDebug, "QRevKey: %i/%i (%i)\n", start, end, i);
			return i;
		}

		// The diff is measured in the scale [0.0,1.0], with 1.0 meaning an exact match.
		if( diff_at_this_frame < error )
			return LINEAR_FALSE;

		lastFrameDiff = thisFrameDiff;
	}
	return LINEAR_TRUE;
}

int AnimExporter::IsFloatLinear( const GenKey<float>* vals, unsigned int start, unsigned int end, double error )
{ FUNC_ENTER("AnimExporter::IsFloatLinear"); 
	double at_start_frame;
	double at_end_frame;
	double delta;
	double lastFrameDiff = 0;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return LINEAR_TRUE;

	// Scan through and calculate the difference between each inbetween point and the line
	// joining start and end points. Too large a difference results in this set of points no longer being
	// considered linear.
	at_start_frame	= vals[start].key;
	at_end_frame	= vals[end].key;
	delta			= at_end_frame - at_start_frame;

	for( i = start + 1; i < end; ++i )
	{
		double at_this_frame			= vals[i].key;
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );
		double thisFrameDiff;

		thisFrameDiff = at_this_frame - vals[i-1].key;

		// If the sign changes from this frame to the next we need to consider the sequence non-linear
		// (thus causing a key to get added for it)
		if ( bFixReverseKeyInterpolation &&
			((lastFrameDiff < 0 && thisFrameDiff > 0) ||
			 (lastFrameDiff > 0 && thisFrameDiff < 0)))
		{
			nReverseFKeys++;
			fprintf(fpKeyRevDebug, "FRevKey: %i/%i (%i)\n", start, end, i);
			return i;
		}

		if( diff_at_this_frame > error )
			return LINEAR_FALSE;

		lastFrameDiff = thisFrameDiff;
	}

	return LINEAR_TRUE;
}

int AnimExporter::IsTLinear( const Gfx::SAnimTFrame* p_data, unsigned int start, unsigned int end, double error_x, double error_y, double error_z )
{ FUNC_ENTER("AnimExporter::IsTLinear"); 
	double at_start_frame;
	double at_end_frame;
	double delta;
	double lastFrameDiff;
	unsigned int i;

	// If just a two key run, always linear.
	if(( end - start ) <= 1 )
		return LINEAR_TRUE;

	// Scan through and calculate the 'x' axis difference between each inbetween point and the line
	// joining start and end points. Too large a difference results in this set of points no longer being
	// considered linear.
	at_start_frame	= p_data[start].t[X];
	at_end_frame	= p_data[end].t[X];
	delta			= at_end_frame - at_start_frame;

	lastFrameDiff   = 0;

	for( i = start + 1; i <= end; ++i )
	{
		double at_this_frame			= p_data[i].t[X];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );
		double thisFrameDiff;

		//char buf[256];
		//sprintf(buf, "Key[X]: %f\n", at_this_frame);
		//OutputDebugString(buf);

		thisFrameDiff = at_this_frame - p_data[i-1].t[X];

		// If the sign changes from this frame to the next we need to consider the sequence non-linear
		// (thus causing a key to get added for it)
		if ( bFixReverseKeyInterpolation &&
			((lastFrameDiff < 0 && thisFrameDiff > 0) ||
			 (lastFrameDiff > 0 && thisFrameDiff < 0)))
		{
			nReverseTKeys++;
			fprintf(fpKeyRevDebug, "TRevKey: %i/%i (%i)\n", start, end, i);
			//OutputDebugString("^^ SIGN CHANGE ^^ (Not linear)\n");

			return i;
		}

		if( diff_at_this_frame > error_x )
		{
			//OutputDebugString("^^^ Diff outside of error tol ^^^\n");
			return LINEAR_FALSE;
		}

		lastFrameDiff = thisFrameDiff;
	}

	at_start_frame	= p_data[start].t[Y];
	at_end_frame	= p_data[end].t[Y];
	delta			= at_end_frame - at_start_frame;
	
	lastFrameDiff   = 0;

	for( i = start + 1; i <= end; ++i )
	{
		double at_this_frame			= p_data[i].t[Y];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );
		double thisFrameDiff;

		thisFrameDiff = p_data[i-1].t[Y] - at_this_frame;

		//char buf[256];
		//sprintf(buf, "Key[Y]: %f\n", at_this_frame);
		//OutputDebugString(buf);

		// If the sign changes from this frame to the next we need to consider the sequence non-linear
		// (thus causing a key to get added for it)
		if ( bFixReverseKeyInterpolation &&
			((lastFrameDiff < 0 && thisFrameDiff > 0) ||
			 (lastFrameDiff > 0 && thisFrameDiff < 0)))
		{
			nReverseTKeys++;
			fprintf(fpKeyRevDebug, "TRevKey: %i/%i (%i)\n", start, end, i);
			//OutputDebugString("^^ SIGN CHANGE ^^ (Not linear)\n");

			return i;
		}

		if( diff_at_this_frame > error_y )
		{
			//OutputDebugString("^^^ Diff outside of error tol ^^^\n");
			return LINEAR_FALSE;
		}

		lastFrameDiff = thisFrameDiff;
	}

	at_start_frame	= p_data[start].t[Z];
	at_end_frame	= p_data[end].t[Z];
	delta			= at_end_frame - at_start_frame;

	lastFrameDiff   = 0;
	
	for( i = start + 1; i <= end; ++i )
	{
		double at_this_frame			= p_data[i].t[Z];
		double linear_at_this_frame		= at_start_frame + (((double)( i - start ) / (double)( end - start )) * delta );
		double diff_at_this_frame		= fabs( at_this_frame - linear_at_this_frame );
		double thisFrameDiff;

		//char buf[256];
		//sprintf(buf, "Key[Z]: %f\n", at_this_frame);
		//OutputDebugString(buf);

		thisFrameDiff = p_data[i-1].t[Z] - at_this_frame;

		// If the sign changes from this frame to the next we need to consider the sequence non-linear
		// (thus causing a key to get added for it)
		if ( bFixReverseKeyInterpolation &&
			((lastFrameDiff < 0 && thisFrameDiff > 0) ||
			 (lastFrameDiff > 0 && thisFrameDiff < 0)))
		{
			nReverseTKeys++;
			fprintf(fpKeyRevDebug, "TRevKey: %i/%i (%i)\n", start, end, i);
			//OutputDebugString("^^ SIGN CHANGE ^^ (Not linear)\n");

			return i;
		}

		if( diff_at_this_frame > error_z )
		{
			//OutputDebugString("^^^ Diff outside of error tol ^^^\n");
			return LINEAR_FALSE;
		}

		lastFrameDiff = thisFrameDiff;
	}

	//OutputDebugString("^^^ Linear ^^^\n");
	return LINEAR_TRUE;
}

int AnimExporter::CompressRawQFrameData( Gfx::SAnimQFrame* p_newdata, Gfx::SAnimQFrame* p_data, int num_keys, double error, ProgressBar* pbar )
{ FUNC_ENTER("AnimExporter::CompressRawQFrameData"); 
//	Dbg_MemberFunction;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			p_newdata[i]=p_data[i];

		return num_keys;
	}

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	if (pbar)
	{
		pbar->SetCaption("Compressing Q Frames");
		pbar->SetRange(0, num_keys);
	}

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		if (pbar)
			pbar->SetVal(current_index);

		// Continue stepping along a straight line...
		if( IsQLinear( p_data, current_index, next_index, error ) )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				p_newdata[newdata_write_index++]=p_data[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++]=p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++]=p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	p_newdata[newdata_write_index++]=p_data[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;
}

int AnimExporter::CompressRawTFrameData( Gfx::SAnimTFrame* p_newdata, Gfx::SAnimTFrame* p_data, int num_keys, double error, ProgressBar* pbar )
{ FUNC_ENTER("AnimExporter::CompressRawTFrameData"); 
//	Dbg_MemberFunction;

	double min_x =  10000.0;
	double max_x = -10000.0;
	double min_y =  10000.0;
	double max_y = -10000.0;
	double min_z =  10000.0;
	double max_z = -10000.0;

	double error_x;
	double error_y;
	double error_z;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml
	unsigned int i;

	if (pbar)
	{
		pbar->SetRange(0, num_keys);
		pbar->SetCaption("Compressing T Frames");
	}

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			p_newdata[i]=p_data[i];

		return num_keys;
	}

	// Scan through the data, calculating min and max values...
	for( i = 0; i < num_keys; ++i )
	{
		if( p_data[i].t[X] > max_x )
			max_x = p_data[i].t[X];

		if( p_data[i].t[X] < min_x )
			min_x = p_data[i].t[X];

		if( p_data[i].t[Y] > max_y )
			max_y = p_data[i].t[Y];

		if( p_data[i].t[Y] < min_y )
			min_y = p_data[i].t[Y];

		if( p_data[i].t[Z] > max_z )
			max_z = p_data[i].t[Z];

		if( p_data[i].t[Z] < min_z )
			min_z = p_data[i].t[Z];
	}

	// ...and the error as a percentage of the range.
	error_x = ( fabs( max_x - min_x ) * error ) / 100.0;
	error_y = ( fabs( max_y - min_y ) * error ) / 100.0;
	error_z = ( fabs( max_z - min_z ) * error ) / 100.0;

	// Limit error from getting too small (in runs with very little or no variation).
	if( error_x < 0.005 )
		error_x = 0.005f;
	if( error_y < 0.005 )
		error_y = 0.005f;
	if( error_z < 0.005 )
		error_z = 0.005f;

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		if (pbar)
			pbar->SetVal(current_index);

		// Continue stepping along a straight line...
		int lmTLinear = IsTLinear( p_data, current_index, next_index, error_x, error_y, error_z );
		if( lmTLinear == LINEAR_TRUE )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				p_newdata[newdata_write_index++] = p_data[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// If lmTLinear is greater than 0 that means there was a sign change
		// at the given frame
		/*
		if( lmTLinear > 0 )
		{
			// A key must be added for the frame that caused the reverse in interpolation
			// as well as the key before that frame
			p_newdata[newdata_write_index++] = p_data[lmTLinear-1];
			p_newdata[newdata_write_index++] = p_data[lmTLinear];
			current_index = lmTLinear;
			next_index = current_index + 1;
			continue;
		}
		*/

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++] = p_data[current_index];
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			p_newdata[newdata_write_index++] = p_data[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	//p_data[next_write_index++] = p_data[num_keys - 1];
	p_newdata[newdata_write_index++] = p_data[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;	// aml
}

int AnimExporter::CompressFloatKeys( GenKey<float>* newvals, GenKey<float>* vals, int num_keys, double Serror )
{ FUNC_ENTER("AnimExporter::CompressFloatKeys"); 
//	Dbg_MemberFunction;

	double min =  10000.0;
	double max = -10000.0;

	double error;

	int num_linear_keys;
	int current_index;
	int next_index;
	int next_write_index;
	int newdata_write_index=0;	// aml
	unsigned int i;

	// Need at least 3 points to be able to compress data.
	if(( num_keys ) < 3 )
	{
		for(int i=0;i<num_keys;i++)
			newvals[i]=vals[i];

		return num_keys;
	}

	// Scan through the data, calculating min and max values...
	for( i = 0; i < num_keys; ++i )
	{
		if( vals[i].key > max )
			max = vals[i].key;

		if( vals[i].key < min )
			min = vals[i].key;
	}

	// ...and the error as a percentage of the range.
	error = ( fabs( max - min ) * Serror ) / 100.0;

	// Limit error from getting too small (in runs with very little or no variation).
	if( error < 0.005 )
		error = 0.005f;

	num_linear_keys = 0;
	current_index = 0;
	next_index = 1;

	// The compressed frames are just written over the existing frames (since the data will never
	// be larger). This index keeps track of the next 'write' index.
	next_write_index = 0;
	
	while( current_index < ( num_keys - 1 ))
	{
		unsigned int run_length;

		// Continue stepping along a straight line...
		if ( IsFloatLinear( vals, current_index, next_index, error ) )
		{
			++next_index;
			if( next_index >= num_keys )
			{
				// Need to write out a key where the current index is.
				++num_linear_keys;
				//p_data[next_write_index++] = p_data[current_index];
				newvals[newdata_write_index++] = vals[current_index];	// aml
				break;
			}
			else
				continue;
		}

		// ...until a curve is found.
		--next_index;
		run_length = next_index - current_index;		

		if( run_length > 1 )
		{
			// A linear section of at least 3 keys was found.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			newvals[newdata_write_index++] = vals[current_index];
			current_index = next_index;
			++next_index;
			continue;
		}
		else
		{
			// Just write out the key.
			++num_linear_keys;
			//p_data[next_write_index++] = p_data[current_index];
			newvals[newdata_write_index++] = vals[current_index];	// aml
			current_index = next_index;
			++next_index;
			continue;
		}
	}

	// Now just need to write out the end key.
	++num_linear_keys;
	//p_data[next_write_index++] = p_data[num_keys - 1];
	newvals[newdata_write_index++] = vals[num_keys - 1];	// aml

	//return num_linear_keys;
	return newdata_write_index;	// aml
}

AnimExporter::AnimExporter()
{ FUNC_ENTER("AnimExporter::AnimExporter"); 
	Qframes            = NULL;
	Tframes            = NULL;
	compQframes        = NULL;
	compTframes        = NULL;
	skeleton           = NULL;
	pelvisbone         = NULL;
	bRemoveReadOnly    = false;
	bCheckOut          = false;
	bSwapCoordSystem   = false;
	bNoParentTransform = false;

	bRotateRoot        = false;
	bTestRotate        = false;
	bCompressTime      = false;

	bFixReverseKeyInterpolation = false;

	numCompressedQFrames=NULL;
	numCompressedTFrames=NULL;

	nReverseTKeys       = 0;
	nReverseQKeys       = 0;
	nReverseFKeys       = 0;

	fpKeyRevDebug = fopen("c:\\keyrevdbg.txt", "a");
}

AnimExporter::~AnimExporter()
{ FUNC_ENTER("AnimExporter::~AnimExporter"); 
	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (compQframes)
		FreeFrames(&compQframes);

	if (compTframes)
		FreeFrames(&compTframes);

	if (skeleton)
		delete skeleton;
}

INode* AnimExporter::GetRoot(INode* node)
{ FUNC_ENTER("AnimExporter::GetRoot"); 
	assert(skeleton);
	return skeleton->GetRootBone(node);

	/*
	assert(node!=NULL);
	CStr name=node->GetName();


	INode* lastNode=node;

	while(node && !node->IsRootNode())
	{
		lastNode=node;
		node=node->GetParentNode();

		CStr name2=node->GetName();
	}
	

	// node should be the scene root
	return lastNode;
	*/
}

int AnimExporter::CountNodes(INode* node)
{ FUNC_ENTER("AnimExporter::CountNodes"); 
	assert(skeleton);
	return skeleton->GetCount();

	/*	
	CSkeletonData* skel=new CSkeletonData(node);
	int count=skel->GetCount();
	delete skel;
	*/
	
	/*
	int numChildren=node->NumberOfChildren();
	int count;

	if (!node->IsHidden())
		count=1;
	else
		count=0;

	for(int i=0;i<numChildren;i++)
	{
		INode* child=node->GetChildNode(i);

		if (!child->IsHidden())
			count+=CountNodes(node->GetChildNode(i));
	}

	return count;
	*/
}

bool AnimExporter::ExportNodes(INode* node)
{ FUNC_ENTER("AnimExporter::ExportNodes"); 
	int fps3DS;								// Frames per second MAX is set to
	int num3DSFrames;						// Number of frames @ the selected 3DS FPS
	int reqFrames;							// Number of frames required for game's FPS rate

	// Don't export the scene root, but export it's children
	if (node->IsRootNode())
	{
		int numChildren=node->NumberOfChildren();
		
		for(int i=0;i<numChildren;i++)
			if (!ExportNodes(node->GetChildNode(i)))
				return false;

		return true;
	}

	fps3DS=GetFrameRate();
	num3DSFrames=end-start+1;
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;

	Gfx::SAnimQFrame Qframe;
	Gfx::SAnimTFrame Tframe;

	// Dump out animation frames at game FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		/*
		if (node==rootbone)
		{
			tm.IdentityMatrix();
			parentTM.IdentityMatrix();
		}
		else
		*/

		// Get TM for Quats
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bRotateRoot)
			{
				if (node==rootbone)
				{
					tm.RotateX(DegToRad(-90));
				}
			}

			if (bSwapCoordSystem)
			{	
				tm.RotateX(DegToRad(-90));
				//tm.RotateY(DegToRad(-90));
			}

			tm.NoScale();
			parentNode = node->GetParentNode();

			if (!bNoParentTransform)
			{
				if (bSwapCoordSystem)
				{
					//parentTM = node->GetObjectTM(curFrame*GetTicksPerFrame());
					parentTM = node->GetNodeTM(curFrame*GetTicksPerFrame());
				}
				else
					parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}

		Quat    quat=Quat(tm);
		Qframe.q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		//Qframe.time=(float)i/(float)OUTPUT_FPS;
		Qframe.time=(float)i/(float)fps3DS;

		// Get TM for Translations
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bSwapCoordSystem)
			{
				//tm.RotateX(DegToRad(-90));
			}

			tm.NoScale();

			if (!bNoParentTransform)
			{
				parentNode = node->GetParentNode();
				parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}

		Point3 trans=tm.GetTrans();

		if (bSwapCoordSystem)
		{
			Tframe.t[X]=trans[X];
			Tframe.t[Y]=trans[Z];
			Tframe.t[Z]=-trans[Y];
		}
		else
		{
			Tframe.t[0]=trans[0];
			Tframe.t[1]=trans[1];
			Tframe.t[2]=trans[2];
		}

		//Tframe.time=(float)i/(float)OUTPUT_FPS;
		Tframe.time=(float)i/(float)fps3DS;
			
		// Dump transform to file
		fwrite(&Qframe.q, sizeof(Mth::Quat),1,fpExport);
		fwrite(&Qframe.time, sizeof(float),1,fpExport);
		fwrite(&Qframe.bFlip, sizeof(bool),3,fpExport);
		
		//fwrite(&Qframe,sizeof(Gfx::SAnimQFrame),1,fpExport);

		fwrite(&Tframe.t, sizeof(float), 3, fpExport);
		fwrite(&Tframe.time, sizeof(float), 1, fpExport);

		//fwrite(&Tframe,sizeof(Gfx::SAnimTFrame),1,fpExport);
	}

	// Dump children
	for(i=0;i<node->NumberOfChildren();i++)
		if (!ExportNodes(node->GetChildNode(i)))
			return false;

	return true;
}

bool AnimExporter::Export(INode* root,char* Filename,char* ModelName,int start,int end)
{ FUNC_ENTER("AnimExporter::Export"); 
	this->start=start;
	this->end=end;

	fpExport=fopen(Filename,"wb");

	if (!fpExport)
		return false;

	int numTransforms=CountNodes(root);
	fwrite(&numTransforms,sizeof(int),1,fpExport);  // Number of Transforms
	ExportNodes(root);

	fclose(fpExport);
	return true;
}

bool AnimExporter::GetAnim(INode* root,int start,int end,float errorQ,float errorT,TolData* tdata,bool bCompress,
						   int**        numCompressedQFrames, Gfx::SAnimQFrame** compQFrames,
						   int**        numCompressedTFrames, Gfx::SAnimTFrame** compTFrames,
						   int*         numNodes,
						   ProgressBar* pbar,
						   void (*PreCompCB)(void*),
						   void* pPreCompData,
						   bool  bExportFromPelvis)
{ FUNC_ENTER("AnimExporter::GetAnim"); 
	if (skeleton)
		rootbone = skeleton->GetRootBone(root);

	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (numCompressedQFrames)
		delete [] numCompressedQFrames;

	if (numCompressedTFrames)
		delete [] numCompressedTFrames;

	if (*numNodes != 1)
		*numNodes = CountNodes(root);

	if (pbar)
	{
		char strBuf[256];
		sprintf(strBuf, "Exporting %s", (char*)root->GetName(), start, end);
		pbar->SetCaption(strBuf);

		pbar->SetRange(0, *numNodes);
	}

	int fps3DS=GetFrameRate();						// 3DS MAX Frame rate
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	this->start     = start;
	this->end       = end;
	this->numFrames = num3DSFrames;
	this->numNodes  = *numNodes;

	// Allocate space for the uncompressed animation data
	Qframes=new Gfx::SAnimQFrame[*numNodes*num3DSFrames];
	Tframes=new Gfx::SAnimTFrame[*numNodes*num3DSFrames];

	// Allocate space for the compressed animation data

	// Compiler not liking *compQFrames = new Gfx::SAnimQFrame[*numNodes*num3DSFrames]
	// (thinks it's incorrect type) reassigning to a local var works ????
	Gfx::SAnimQFrame** pcompQframes = compQFrames;
	Gfx::SAnimTFrame** pcompTframes = compTFrames;

	CStr name = root->GetName();
	if (name == CStr("Control_Root_S"))
	{
		int zzz;
		zzz = 0;
	}

	*pcompQframes = new Gfx::SAnimQFrame[*numNodes*num3DSFrames];
	*pcompTframes = new Gfx::SAnimTFrame[*numNodes*num3DSFrames];

	*numCompressedQFrames=new int[*numNodes];
	*numCompressedTFrames=new int[*numNodes];

	this->start=start;
	this->end=end;

	/*
	if (*numNodes == 1)
		GetQFrames(Qframes,root,bExportFromPelvis,true,pbar);
	else
		GetQFrames(Qframes,root,bExportFromPelvis,false,pbar);
	*/

	if (*numNodes == 1)
		GetQTFrames(Qframes, Tframes, root, bExportFromPelvis,true,pbar);
	else
		GetQTFrames(Qframes, Tframes, root, bExportFromPelvis,true,pbar);

	// Fix Flipped Euler rotations
	for(int i=0;i<*numNodes;i++)
	{
		INode* node;

		if (*numNodes != 1)
			node=skeleton->GetBone(i);
		else
			node=root;

		CStr name=node->GetName();

		FixEulerFlips(Qframes+i*num3DSFrames,num3DSFrames);
	}

	if (*numNodes == 1)
		GetTFrames(Tframes,root,bExportFromPelvis,true,pbar);
	else
		GetTFrames(Tframes,root,bExportFromPelvis,false,pbar);

	// Perform custom pre compression step
	if (PreCompCB)
		PreCompCB(pPreCompData);

	for(int curNode=0;curNode<*numNodes;curNode++)
	{
		if (pbar)
			pbar->SetVal(curNode);

		if (skeleton && !skeleton->GetPartialAnimState(curNode))
		{
			(*numCompressedQFrames)[curNode]=0;
			(*numCompressedTFrames)[curNode]=0;
			continue;
		}

		if (bExportFromPelvis && (curNode == 0 || curNode == 1))
		{
			(*numCompressedQFrames)[curNode] = 1;
			(*numCompressedTFrames)[curNode] = 1;

			compQframes[curNode*num3DSFrames].q[X]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[Y]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[Z]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[W]     = 1.0f;
			compQframes[curNode*num3DSFrames].time     = 0.0f;
			compQframes[curNode*num3DSFrames].bFlip[0] = false;
			compQframes[curNode*num3DSFrames].bFlip[1] = false;
			compQframes[curNode*num3DSFrames].bFlip[2] = false;
			compQframes[curNode*num3DSFrames].flags    = 0;
			
			compTframes[curNode*num3DSFrames].t[0]  = 0.0f;
			compTframes[curNode*num3DSFrames].t[1]  = 0.0f;
			compTframes[curNode*num3DSFrames].t[2]  = 0.0f;
			compTframes[curNode*num3DSFrames].time  = 0.0f;
			compTframes[curNode*num3DSFrames].flags = 0;

			Qframes[curNode*num3DSFrames] = compQframes[curNode*num3DSFrames];
			Tframes[curNode*num3DSFrames] = compTframes[curNode*num3DSFrames];
			continue;
		}
		

		if (!bCompress)
		{
			(*numCompressedQFrames)[curNode]=num3DSFrames;
			(*numCompressedTFrames)[curNode]=num3DSFrames;
		}
		else
		{
			// Handle variable compression entries
			INode* node;
			
			if (*numNodes != 1)
				node = skeleton->GetBone(curNode);
			else
				node = root;

			CStr   strClass;
			float  varErrorQ, varErrorT;

			if (!node->GetUserPropString("Class",strClass))
				node->GetUserPropString("class",strClass);

			TolRecord tdataFind;
			tdataFind.name = node->GetName();
			Link<TolRecord>* trLink;

			if (tdata)
			{
				trLink = tdata->DB.Find(&tdataFind);
			}
			else
				trLink=NULL;

			if (trLink)
			{
				if (trLink->data.bHasRot)
				{
					varErrorQ = trLink->data.valRot;
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
				}

				if (trLink->data.bHasTran)
				{
					varErrorT = trLink->data.valTran;
				}
				else
				{
					varErrorT = errorT;
				}
			}
			else
			{
				/* Leaving this in for now, just in case */
				if (strClass==CStr("VarAnimTolerance"))
				{
					node->GetUserPropFloat("RotTolerance",varErrorQ);
					node->GetUserPropFloat("TransTolerance",varErrorT);

					// Scale the rotation compression value appropriately
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
					varErrorT = errorT;
				}
			}

#ifdef DUMP_PRECOMPRESS_TFRAMES
			// Dump T frames
			FILE* dpc_fpT = fopen("c:\\TFrames.txt", "w");
			int dpc_iT;

			for(dpc_iT = 0; dpc_iT < num3DSFrames; dpc_iT++)
			{
				fprintf(dpc_fpT, "%i:  Time: %f  tx: %f  ty: %f  tz: %f\n", dpc_iT,
																	   (Tframes+(curNode*num3DSFrames))[dpc_iT].time,
																	   (Tframes+(curNode*num3DSFrames))[dpc_iT].t[X],
																	   (Tframes+(curNode*num3DSFrames))[dpc_iT].t[Y],																	   
																	   (Tframes+(curNode*num3DSFrames))[dpc_iT].t[Z]);
			}

			fclose(dpc_fpT);
#endif

#ifdef DUMP_PRECOMPRESS_QFRAMES
			// Dump Q frames
			FILE* dpc_fpQ = fopen("c:\\QFrames.txt","w");
			int dpc_iQ;

			for(dpc_iQ = 0; dpc_iQ < num3DSFrames; dpc_iQ++)
			{
				fprintf(dpc_fpQ, "%i:  Time: %f  qx: %f  qy: %f  qz: %f  real: %f\n", dpc_iQ,
					                                                             (Qframes+(curNode*num3DSFrames))[dpc_iQ].time,
																				 (Qframes+(curNode*num3DSFrames))[dpc_iQ].q.x,
																				 (Qframes+(curNode*num3DSFrames))[dpc_iQ].q.y,
																				 (Qframes+(curNode*num3DSFrames))[dpc_iQ].q.z,
																				 (Qframes+(curNode*num3DSFrames))[dpc_iQ].q.w);
			}

			fclose(dpc_fpQ);
#endif

			(*numCompressedQFrames)[curNode]=CompressRawQFrameData( (*pcompQframes)+(curNode*num3DSFrames), Qframes+(curNode*num3DSFrames), num3DSFrames, varErrorQ, pbar );
			(*numCompressedTFrames)[curNode]=CompressRawTFrameData( (*pcompTframes)+(curNode*num3DSFrames), Tframes+(curNode*num3DSFrames), num3DSFrames, varErrorT, pbar );
		
			// Ensure that the last frame of animation is ALWAYS exported
			/*	
			Gfx::SAnimQFrame* qframes = (*pcompQframes)+(curNode*num3DSFrames);
			Gfx::SAnimTFrame* tframes = (*pcompTframes)+(curNode*num3DSFrames);

			int nQFrames = (*numCompressedQFrames)[curNode];
			int nTFrames = (*numCompressedTFrames)[curNode];

			float finalTime = ((float)end-(float)start+1.0f) / GetFrameRate();

			// If the last keyframe isn't at the final time, add the uncompressed key
			if (qframes[nQFrames-1].time < finalTime)
			{
				qframes[nQFrames] = Qframes[curNode*num3DSFrames-1];
				(*numCompressedQFrames)[curNode]++;
			}

			if (tframes[nTFrames-1].time < finalTime)
			{
				tframes[nTFrames] = Tframes[curNode*num3DSFrames-1];
				(*numCompressedTFrames)[curNode]++;
			}
			*/
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	if (bExportFromPelvis)
	{
		int boardBone;

		for(int i = 0; i < skeleton->GetCount(); i++)
		{
			CStr boneName = skeleton->GetBone(i)->GetName();
			boneName.toLower();

			if (boneName == CStr("bone_board_root"))
			{
				boardBone = i;
				break;
			}
		}
		
		// Copy the contents of the 2nd node (should be pelvis) into the first node and zero the 2nd node
		int curNode, curFrame;

		for(curNode = 0; curNode < *numNodes; curNode++)
		{
			(*numCompressedQFrames)[0] = (*numCompressedQFrames)[boardBone];

			for(curFrame = 0; curFrame < (*numCompressedQFrames)[boardBone]; curFrame++)
			{
				compQframes[0 * num3DSFrames + curFrame] = compQframes[boardBone * num3DSFrames + curFrame];
			}

			numCompressedTFrames[0] = numCompressedTFrames[boardBone];

			for(curFrame = 0; curFrame < (*numCompressedTFrames)[boardBone]; curFrame++)
			{
				compTframes[0 * num3DSFrames + curFrame] = compTframes[boardBone * num3DSFrames + curFrame];
			}
		
			Qframes[curNode*num3DSFrames] = compQframes[curNode*num3DSFrames];
			Tframes[curNode*num3DSFrames] = compTframes[curNode*num3DSFrames];
		}

		(*numCompressedQFrames)[boardBone] = 1;
		(*numCompressedTFrames)[boardBone] = 1;

		compQframes[boardBone * num3DSFrames].q[X]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[Y]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[Z]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[W]     = 1.0f;
		compQframes[boardBone * num3DSFrames].time     = 0.0f;
		compQframes[boardBone * num3DSFrames].bFlip[0] = false;
		compQframes[boardBone * num3DSFrames].bFlip[1] = false;
		compQframes[boardBone * num3DSFrames].bFlip[2] = false;
		compQframes[boardBone * num3DSFrames].flags    = 0;
			
		compTframes[boardBone * num3DSFrames].t[0]  = 0.0f;
		compTframes[boardBone * num3DSFrames].t[1]  = 0.0f;
		compTframes[boardBone * num3DSFrames].t[2]  = 0.0f;
		compTframes[boardBone * num3DSFrames].time  = 0.0f;
		compTframes[boardBone * num3DSFrames].flags = 0;

		Qframes[boardBone * num3DSFrames] = compQframes[boardBone * num3DSFrames];
		Tframes[boardBone * num3DSFrames] = compTframes[boardBone * num3DSFrames];
	}
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////

	if (!bCompress)
	{
		for(int x=0;x<(*numNodes)*num3DSFrames;x++)
		{
			(*pcompQframes)[x]=Qframes[x];
			(*pcompTframes)[x]=Tframes[x];
		}
	}

	if (*numCompressedQFrames>0 ||
		*numCompressedTFrames>0)
		return true;

	return false;
}

bool AnimExporter::GetAnim(INode* root,
						   int start,
						   int end,
						   float errorQ,
						   float errorT,
						   TolData* tdata,
						   bool bCompress,
						   ProgressBar* pbar,
						   void (*PreCompCB)(void*),
						   void* pPreCompData,
						   bool bExportFromPelvis)
{ FUNC_ENTER("AnimExporter::GetAnim"); 
	if (skeleton)
		rootbone = skeleton->GetRootBone(root);

	if (Qframes)
		FreeFrames(&Qframes);

	if (Tframes)
		FreeFrames(&Tframes);

	if (compQframes)
		FreeFrames(&compQframes);

	if (compTframes)
		FreeFrames(&compTframes);

	if (numCompressedQFrames)
		delete [] numCompressedQFrames;

	if (numCompressedTFrames)
		delete [] numCompressedTFrames;

	int numNodes=CountNodes(root);					// Number of nodes (bones)	
	int fps3DS=GetFrameRate();						// 3DS MAX Frame rate
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	this->start     = start;
	this->end       = end;
	this->numFrames = num3DSFrames;
	this->numNodes  = numNodes;

	if (pbar)
	{
		char strBuf[256];
		sprintf(strBuf, "Exporting %s", (char*)root->GetName(), start, end);
		pbar->SetCaption(strBuf);

		pbar->SetRange(0, numNodes);
	}

	// Allocate space for the uncompressed animation data
	Qframes=new Gfx::SAnimQFrame[numNodes*num3DSFrames];
	Tframes=new Gfx::SAnimTFrame[numNodes*num3DSFrames];

	// Allocate space for the compressed animation data
	compQframes=new Gfx::SAnimQFrame[numNodes*num3DSFrames];
	compTframes=new Gfx::SAnimTFrame[numNodes*num3DSFrames];

	numCompressedQFrames=new int[numNodes];
	numCompressedTFrames=new int[numNodes];

	this->start=start;
	this->end=end;

	/*
	GetQFrames(Qframes,root,bExportFromPelvis,false,pbar);

	if (pbar && pbar->WasCanceled())
		return false;

	// Fix Flipped Euler rotations
	for(int i=0;i<numNodes;i++)
	{
		INode* node=skeleton->GetBone(i);
		CStr name=node->GetName();

		FixEulerFlips(Qframes+i*num3DSFrames,num3DSFrames);
	}

	GetTFrames(Tframes,root,bExportFromPelvis,false,pbar);
	*/

	GetQTFrames(Qframes,Tframes,root,bExportFromPelvis,false,pbar);

	if (pbar && pbar->WasCanceled())
		return false;

	// Fix Flipped Euler rotations
	for(int i=0;i<numNodes;i++)
	{
		INode* node=skeleton->GetBone(i);
		CStr name=node->GetName();

		FixEulerFlips(Qframes+i*num3DSFrames,num3DSFrames);
	}
	//////////////////////////////////////////////////////////////////////////

	// Perform custom pre compression step
	if (PreCompCB)
		PreCompCB(pPreCompData);

	for(int curNode=0;curNode<numNodes;curNode++)
	{
		///
		if (skeleton && !skeleton->GetPartialAnimState(curNode))
		{
			numCompressedQFrames[curNode]=0;
			numCompressedTFrames[curNode]=0;
			continue;
		}
		///

		if (bExportFromPelvis && (curNode == 0 || curNode == 1 /*|| curNode == 50*/))
		{
			numCompressedQFrames[curNode] = 1;
			numCompressedTFrames[curNode] = 1;

			compQframes[curNode*num3DSFrames].q[X]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[Y]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[Z]     = 0.0f;
			compQframes[curNode*num3DSFrames].q[W]     = 1.0f;
			compQframes[curNode*num3DSFrames].time     = 0.0f;
			compQframes[curNode*num3DSFrames].bFlip[0] = false;
			compQframes[curNode*num3DSFrames].bFlip[1] = false;
			compQframes[curNode*num3DSFrames].bFlip[2] = false;
			compQframes[curNode*num3DSFrames].flags    = 0;
			
			compTframes[curNode*num3DSFrames].t[0]  = 0.0f;
			compTframes[curNode*num3DSFrames].t[1]  = 0.0f;
			compTframes[curNode*num3DSFrames].t[2]  = 0.0f;
			compTframes[curNode*num3DSFrames].time  = 0.0f;
			compTframes[curNode*num3DSFrames].flags = 0;

			Qframes[curNode*num3DSFrames] = compQframes[curNode*num3DSFrames];
			Tframes[curNode*num3DSFrames] = compTframes[curNode*num3DSFrames];
			//Tframes[curNode*num3DSFrames] = Tframes[curNode*num3DSFrames];
			continue;
		}


		if (!bCompress)
		{
			numCompressedQFrames[curNode]=num3DSFrames;
			numCompressedTFrames[curNode]=num3DSFrames;
		}
		else
		{
			// Handle variable compression entries
			INode* node = skeleton->GetBone(curNode);
			CStr   strClass;
			float  varErrorQ, varErrorT;

			if (!node->GetUserPropString("Class",strClass))
				node->GetUserPropString("class",strClass);

			TolRecord tdataFind;
			tdataFind.name = node->GetName();
			Link<TolRecord>* trLink;

			if (tdata)
			{
				trLink = tdata->DB.Find(&tdataFind);
			}
			else
				trLink=NULL;

			if (trLink)
			{
				if (trLink->data.bHasRot)
				{
					varErrorQ = trLink->data.valRot;
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
				}

				if (trLink->data.bHasTran)
				{
					varErrorT = trLink->data.valTran;
				}
				else
				{
					varErrorT = errorT;
				}
			}
			else
			{
				/* Leaving this in for now, just in case */
				if (strClass==CStr("VarAnimTolerance"))
				{
					node->GetUserPropFloat("RotTolerance",varErrorQ);
					node->GetUserPropFloat("TransTolerance",varErrorT);

					// Scale the rotation compression value appropriately
					varErrorQ = 0.999700f+0.000300f*varErrorQ;
				}
				else
				{
					varErrorQ = errorQ;
					varErrorT = errorT;
				}
			}

			numCompressedQFrames[curNode]=CompressRawQFrameData( compQframes+(curNode*num3DSFrames), Qframes+(curNode*num3DSFrames), num3DSFrames, varErrorQ, pbar );
			numCompressedTFrames[curNode]=CompressRawTFrameData( compTframes+(curNode*num3DSFrames), Tframes+(curNode*num3DSFrames), num3DSFrames, varErrorT, pbar );

			// RETURN HERE!
			// Ensure that the last frame of animation is ALWAYS exported		
			/*
			Gfx::SAnimQFrame* qframes = compQframes+(curNode*num3DSFrames);
			Gfx::SAnimTFrame* tframes = compTframes+(curNode*num3DSFrames);

			int nQFrames = numCompressedQFrames[curNode];
			int nTFrames = numCompressedTFrames[curNode];

			float finalTime = ((float)end-(float)start+1.0f) / GetFrameRate();

			// If the last keyframe isn't at the final time, add the uncompressed key
			if (qframes[nQFrames-1].time < finalTime)
			{
				qframes[nQFrames] = Qframes[curNode*num3DSFrames-1];
				qframes[nQFrames].time = finalTime;
				numCompressedQFrames[curNode]++;
			}

			if (tframes[nTFrames-1].time < finalTime)
			{
				tframes[nTFrames] = Tframes[curNode*num3DSFrames-1];
				tframes[nTFrames].time = finalTime;
				numCompressedTFrames[curNode]++;
			}
			*/
		}
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	if (!bCompress)
	{
		for(int x=0;x<numNodes*num3DSFrames;x++)
		{
			compQframes[x]=Qframes[x];
			compTframes[x]=Tframes[x];
		}
	}

	/*
	if (bExportFromPelvis)
	{
		INode* node0 = skeleton->GetBone(0);
		INode* node2 = skeleton->GetBone(1);
		
		CStr name0 = node0->GetName();
		CStr name2 = node2->GetName();

		int boardBone;

		INode* boneRoot   = skeleton->GetRootBone(NULL);		// Root bone should have already been determined
		INode* bonePelvis = FindPelvisBone(boneRoot);
		
		// Find the board bone
		for(int i = 0; i < skeleton->GetCount(); i++)
		{
			CStr boneName = skeleton->GetBone(i)->GetName();
			boneName.toLower();

			if (boneName == CStr("bone_board_root"))
			{
				boardBone = i;
				break;
			}
		}

		// OK, at this point the board is at pos = oldRoot * Board
		// we need the board with respect to our newly created root
		// so pos = (Inverse(oldRoot) * Board)

		for(curFrame = 0; curFrame < numCompressedQFrames[boardBone]; curFrame++)
		{
			Mth::Matrix gameMatrix;
			Matrix3 mat;

			compQframes[boardBone * num3DSFrames + curFrame].q.GetMatrix(gameMatrix);

			// Convert to 3DS matrix
			mat.SetRow(0, Point3(gameMatrix.GetRight()[0], gameMatrix.GetRight()[1], gameMatrix.GetRight()[2]));
			mat.SetRow(1, Point3(gameMatrix.GetUp()[0], gameMatrix.GetUp()[1], gameMatrix.GetUp()[2]));
			mat.SetRow(2, Point3(gameMatrix.GetAt()[0], gameMatrix.GetAt()[1], gameMatrix.GetAt()[2]));
			mat.SetRow(3, Point3(gameMatrix.GetPos()[0], gameMatrix.GetPos()[1], gameMatrix.GetPos()[2]));

			// Recompute matrix
			Matrix3 oldRootMatrix = boneRoot->GetNodeTM(0);
			Matrix3 pelvisMatrix  = bonePelvis->GetNodeTM(0);
			mat = mat * Inverse(oldRootMatrix) * Inverse(pelvisMatrix);
			
			// Convert recomputed matrix back into a quaternion and reassign
			Quat quat(mat);

			compQframes[boardBone * num3DSFrames + curFrame].q[X] = quat.x;
			compQframes[boardBone * num3DSFrames + curFrame].q[Y] = quat.y;
			compQframes[boardBone * num3DSFrames + curFrame].q[Z] = quat.z;
			compQframes[boardBone * num3DSFrames + curFrame].q[W] = quat.w;

			//compQframes[boardBone * num3DSFrames + curFrame].q[X] = 0.0f;
			//compQframes[boardBone * num3DSFrames + curFrame].q[Y] = 0.0f;
			//compQframes[boardBone * num3DSFrames + curFrame].q[Z] = 0.0f;
			//compQframes[boardBone * num3DSFrames + curFrame].q[W] = 1.0f;
		}

		for(curFrame = 0; curFrame < numCompressedTFrames[boardBone]; curFrame++)
		{
			Mth::Matrix gameMatrix;
			Matrix3 mat;

			mat.IdentityMatrix();
			mat.SetTrans(Point3(compTframes[boardBone * num3DSFrames + curFrame].t[X],
				                compTframes[boardBone * num3DSFrames + curFrame].t[Y],
								compTframes[boardBone * num3DSFrames + curFrame].t[Z]));

			// Recompute matrix
			Matrix3 oldRootMatrix = boneRoot->GetNodeTM(0);
			Matrix3 pelvisMatrix  = bonePelvis->GetNodeTM(0);
			mat = mat * Inverse(oldRootMatrix) * Inverse(pelvisMatrix);
			
			compTframes[boardBone * num3DSFrames + curFrame].t[X] = mat.GetTrans().x;
			compTframes[boardBone * num3DSFrames + curFrame].t[Y] = mat.GetTrans().y;
			compTframes[boardBone * num3DSFrames + curFrame].t[Z] = mat.GetTrans().z;

			//compTframes[boardBone * num3DSFrames + curFrame].t[X] = 1000;
			//compTframes[boardBone * num3DSFrames + curFrame].t[Y] = 1000;
			//compTframes[boardBone * num3DSFrames + curFrame].t[Z] = 1000;

			//compTframes[boardBone * num3DSFrames + curFrame].t[X] = 0.0f;
			//compTframes[boardBone * num3DSFrames + curFrame].t[Y] = 0.0f;
			//compTframes[boardBone * num3DSFrames + curFrame].t[Z] = 0.0f;
		}

		/*
		// Copy board bone to root bone
		int curNode, curFrame;

		for(curNode = 0; curNode < numNodes; curNode++)
		{
			numCompressedQFrames[0] = numCompressedQFrames[boardBone];

			for(curFrame = 0; curFrame < numCompressedQFrames[boardBone]; curFrame++)
			{
				compQframes[0 * num3DSFrames + curFrame] = compQframes[boardBone * num3DSFrames + curFrame];
			}

			numCompressedTFrames[0] = numCompressedTFrames[boardBone];

			for(curFrame = 0; curFrame < numCompressedTFrames[boardBone]; curFrame++)
			{
				compTframes[0 * num3DSFrames + curFrame] = compTframes[boardBone * num3DSFrames + curFrame];
			}
		
			Qframes[curNode*num3DSFrames] = compQframes[curNode*num3DSFrames];
			Tframes[curNode*num3DSFrames] = compTframes[curNode*num3DSFrames];
		}

		numCompressedQFrames[boardBone] = 1;
		numCompressedTFrames[boardBone] = 1;

		compQframes[boardBone * num3DSFrames].q[X]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[Y]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[Z]     = 0.0f;
		compQframes[boardBone * num3DSFrames].q[W]     = 1.0f;
		compQframes[boardBone * num3DSFrames].time     = 0.0f;
		compQframes[boardBone * num3DSFrames].bFlip[0] = false;
		compQframes[boardBone * num3DSFrames].bFlip[1] = false;
		compQframes[boardBone * num3DSFrames].bFlip[2] = false;
		compQframes[boardBone * num3DSFrames].flags    = 0;
			
		compTframes[boardBone * num3DSFrames].t[0]  = 0.0f;
		compTframes[boardBone * num3DSFrames].t[1]  = 0.0f;
		compTframes[boardBone * num3DSFrames].t[2]  = 0.0f;
		compTframes[boardBone * num3DSFrames].time  = 0.0f;
		compTframes[boardBone * num3DSFrames].flags = 0;

		Qframes[boardBone * num3DSFrames] = compQframes[boardBone * num3DSFrames];
		Tframes[boardBone * num3DSFrames] = compTframes[boardBone * num3DSFrames];
		*/
	//}

	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////
	/////////////////////////////////////////////////////////////////////////////////////////////////////////

	if (numCompressedQFrames>0 ||
		numCompressedTFrames>0)
		return true;




	return false;
}

void AnimExporter::BuildNodeQTFrames(Gfx::SAnimQFrame* qframes,
									 Gfx::SAnimTFrame* tframes,
									 INode* node,
									 bool   bExportFromPelvis,
									 ProgressBar* pbar)
{ FUNC_ENTER("AnimExporter::BuildNodeQTFrames");
	int num3DSFrames;
	int reqFrames;

	int fps3DS=GetFrameRate();
	num3DSFrames=end-start+1;
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;

	if (pbar)
		pbar->SetRange(0, num3DSFrames);

	Quat prevQuat;

	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3  tm,parentTM;
		Quat     quat,quatParent;
		Interval interval=FOREVER;
		
		if (pbar)
		{
			char strBuf[256];
			sprintf(strBuf, "%s (%i/%i)", (char*)node->GetName(), i, num3DSFrames);
			pbar->SetCaption(strBuf);
			pbar->SetVal(i);

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}

		Control* TMController = node->GetTMController();
		Control* rot          = TMController->GetRotationController();
		Control* parentRot    = NULL;

		INode*   parentNode   = node->GetParentNode();

		if (bExportFromPelvis)
		{
			if (parentNode == rootbone)
				parentNode = pelvisbone;
		}

		TMController = parentNode->GetTMController();

		if (TMController)
			parentRot = TMController->GetRotationController();

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// If there's no rotation controller, grab the rotation from the node TM
		if (!rot)
		{
			// Build the game's structures for output
			if (bSwapCoordSystem)
			{
				//tm=node->GetObjectTM(curFrame*GetTicksPerFrame());
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
				tm.RotateX(DegToRad(-90));
				//tm.RotateY(DegToRad(-90));
			}
			else
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());	
			
			tm.NoScale();
			parentNode = node->GetParentNode();

			if (bExportFromPelvis)
			{
				if (parentNode == rootbone)
					parentNode = pelvisbone;
			}

			if (!bNoParentTransform)
			{
				if (bSwapCoordSystem)
				{
					//parentTM=parentNode->GetObjectTM(curFrame*GetTicksPerFrame());
					parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
				}
				else
					parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}
		else
		{
			// A rotation controller exists, grab the rotation
			rot->GetValue(curFrame*GetTicksPerFrame(),&quat,interval,CTRL_ABSOLUTE);
			interval=FOREVER;

			quat.MakeMatrix(tm);

			if (parentRot)
			{
				parentRot->GetValue(curFrame*GetTicksPerFrame(),&quatParent,interval,CTRL_ABSOLUTE);
				quatParent.MakeMatrix(parentTM);
			}
			else
			{
				// No parent rotation controller (use identity)
				//parentTM=Matrix3(1);

				parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
			}

			tm=tm*node->GetObjectTM(curFrame*GetTicksPerFrame());
			parentTM=parentTM*parentNode->GetObjectTM(curFrame*GetTicksPerFrame());

			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
			parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bSwapCoordSystem)
				tm.RotateX(DegToRad(-90));
			
			tm.NoScale();

			if (!bNoParentTransform)
			{
				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}

		quat=Quat(tm);

		if ( i != 0 )
		{
			quat.MakeClosest(prevQuat);
		}
		else
		{
			// make sure they're all anims start off with a positive w.
			// (only do this on frame 0!)
			if (quat.w<0.0f)
			{
				quat.x=-quat.x;
				quat.y=-quat.y;
				quat.z=-quat.z;
				quat.w=-quat.w;
			}
		}

		prevQuat = quat;

		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		qframes[i].time= (float)i/(float)fps3DS;
		//qframes[i].time= (float)start/(float)fps3DS + (float)i/(float)fps3DS;

		// TODO: Consider bSwapCoordSystem 
		Point3 trans = tm.GetTrans();
		tframes[i].t[X] = trans[X];
		tframes[i].t[Y] = trans[Y];
		tframes[i].t[Z] = trans[Z];

		tframes[i].time=(float)i/(float)fps3DS;
	}

}


///// temp
void AnimExporter::BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node,bool bExportFromPelvis,ProgressBar* pbar)
{ FUNC_ENTER("AnimExporter::BuildNodeQFrames"); 
	int num3DSFrames;
	int reqFrames;

	int fps3DS=GetFrameRate();
	num3DSFrames=end-start+1;
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;

	if (pbar)
		pbar->SetRange(0, num3DSFrames);

	Quat prevQuat;

	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3  tm,parentTM;
		Quat     quat,quatParent;
		Interval interval=FOREVER;
		
		if (pbar)
		{
			char strBuf[256];
			sprintf(strBuf, "%s (%i/%i)", (char*)node->GetName(), i, num3DSFrames);
			pbar->SetCaption(strBuf);
			pbar->SetVal(i);

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}

		Control* TMController = node->GetTMController();
		Control* rot          = TMController->GetRotationController();
		Control* parentRot    = NULL;

		INode*   parentNode   = node->GetParentNode();

		if (bExportFromPelvis)
		{
			if (parentNode == rootbone)
				parentNode = pelvisbone;
		}

		TMController = parentNode->GetTMController();

		if (TMController)
			parentRot = TMController->GetRotationController();

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// If there's no rotation controller, grab the rotation from the node TM
		if (!rot)
		{
			// Build the game's structures for output
			if (bSwapCoordSystem)
			{
				//tm=node->GetObjectTM(curFrame*GetTicksPerFrame());
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
				tm.RotateX(DegToRad(-90));
				//tm.RotateY(DegToRad(-90));
			}
			else
				tm=node->GetNodeTM(curFrame*GetTicksPerFrame());	
			
			tm.NoScale();
			parentNode = node->GetParentNode();

			if (bExportFromPelvis)
			{
				if (parentNode == rootbone)
					parentNode = pelvisbone;
			}

			if (!bNoParentTransform)
			{
				if (bSwapCoordSystem)
				{
					//parentTM=parentNode->GetObjectTM(curFrame*GetTicksPerFrame());
					parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
				}
				else
					parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}
		else
		{
			// A rotation controller exists, grab the rotation
			rot->GetValue(curFrame*GetTicksPerFrame(),&quat,interval,CTRL_ABSOLUTE);
			interval=FOREVER;

			quat.MakeMatrix(tm);

			if (parentRot)
			{
				parentRot->GetValue(curFrame*GetTicksPerFrame(),&quatParent,interval,CTRL_ABSOLUTE);
				quatParent.MakeMatrix(parentTM);
			}
			else
			{
				// No parent rotation controller (use identity)
				//parentTM=Matrix3(1);

				parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
			}

			tm=tm*node->GetObjectTM(curFrame*GetTicksPerFrame());
			parentTM=parentTM*parentNode->GetObjectTM(curFrame*GetTicksPerFrame());

			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
			parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bSwapCoordSystem)
				tm.RotateX(DegToRad(-90));
			
			tm.NoScale();

			if (!bNoParentTransform)
			{
				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}
		}

		////////////////////////////////////////////
		if (bTestRotate)
		{
			if (node==rootbone)
			{
				tm.RotateX(DegToRad(135));
				tm.RotateZ(DegToRad(135));
			}
		}
		////////////////////////////////////////////

		quat=Quat(tm);

		if ( i != 0 )
		{
			quat.MakeClosest(prevQuat);
		}
		else
		{
			// make sure they're all anims start off with a positive w.
			// (only do this on frame 0!)
			if (quat.w<0.0f)
			{
				quat.x=-quat.x;
				quat.y=-quat.y;
				quat.z=-quat.z;
				quat.w=-quat.w;
			}
		}

		prevQuat = quat;

		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		qframes[i].time= (float)i/(float)fps3DS;
		//qframes[i].time= (float)start/(float)fps3DS + (float)i/(float)fps3DS;
	}
}

/*
// Fill out an array of QFrames for a single node for all frames at the game's FPS rate
void AnimExporter::BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node)
{
	int num3DSFrames;							// Number of frames @ the selected 3DS FPS
	int reqFrames;								// Number of frames required for game's FPS rate

	int fps3DS=GetFrameRate();					// 3DS MAX Frame rate
	num3DSFrames=end-start;						// Number of frames in animation at 3DS MAX's frame rate
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	// Dump out animation frames at game's FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		if (bSwapCoordSystem)
		{
			//tm=node->GetObjectTM(curFrame*GetTicksPerFrame());
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());
			tm.RotateX(DegToRad(-90));
			//tm.RotateY(DegToRad(-90));
		}
		else
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());	
		
		tm.NoScale();
		parentNode = node->GetParentNode();

		if (bSwapCoordSystem)
		{
			//parentTM=parentNode->GetObjectTM(curFrame*GetTicksPerFrame());
			parentTM=parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
		}
		else
			parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());

		parentTM.NoScale();
		tm = tm * Inverse(parentTM);

		Quat    quat=Quat(tm);

		if (quat.w<0.0f)
		{
			quat.x=-quat.x;
			quat.y=-quat.y;
			quat.z=-quat.z;
			quat.w=-quat.w;
		}

		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		//qframes[i].time=(float)i/(float)OUTPUT_FPS;
		qframes[i].time=(float)i/(float)fps3DS;
	}
}
*/

// Creates a single TFrame for a single node for the given frame at the game's FPS rate
void AnimExporter::BuildNodeTFrames(Gfx::SAnimTFrame* tframes,INode* node,bool bExportFromPelvis,ProgressBar* pbar)
{ FUNC_ENTER("AnimExporter::BuildNodeTFrames"); 
	int num3DSFrames;							// Number of frames @ the selected 3DS FPS
	int reqFrames;								// Number of frames required for game's FPS rate

	int fps3DS=GetFrameRate();					// 3DS MAX Frame rate
	num3DSFrames=end-start+1;						// Number of frames in animation at 3DS MAX's frame rate
	reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	if (pbar)
		pbar->SetRange(0, num3DSFrames);

	// Dump out animation frames at game's FPS rate
	//for(int i=0;i<reqFrames;i++)
	for(int i=0;i<num3DSFrames;i++)
	{
		Matrix3 tm;
		Matrix3 parentTM;
		INode*  parentNode;
		
		if (pbar)
		{
			char strBuf[256];
			sprintf(strBuf, "%s (%i/%i)", (char*)node->GetName(), i, num3DSFrames);
			pbar->SetCaption(strBuf);
			pbar->SetVal(i);

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}

		// Compute what the current MAX frame would be at our frame rate
		int curFrame=i;//*num3DSFrames/reqFrames;
		curFrame+=start;

		// Build the game's structures for output
		/*
		if (node==rootbone)
		{
			tm.IdentityMatrix();
			parentTM.IdentityMatrix();
		}
		else
		*/
		{
			tm=node->GetNodeTM(curFrame*GetTicksPerFrame());

			if (bRotateRoot)
			{
				if (node==rootbone)
				{
					tm.RotateX(DegToRad(-90));
				}
			}


			if (bSwapCoordSystem)
			{
				//tm.RotateX(DegToRad(-90));
			}

			tm.NoScale();
			if (!bNoParentTransform)
			{
				parentNode = node->GetParentNode();

				if (bExportFromPelvis)
				{
					if (parentNode == rootbone)
						parentNode = pelvisbone;
				}

				parentTM = parentNode->GetNodeTM(curFrame*GetTicksPerFrame());
				parentTM.NoScale();
				tm = tm * Inverse(parentTM);
			}

			////////////////////////////////////////////
			if (bTestRotate)
			{
				if (node==rootbone)
				{
					tm.RotateX(DegToRad(90));
					tm.RotateZ(DegToRad(135));
				}
			}
			////////////////////////////////////////////
		}

		Point3 trans=tm.GetTrans();

		if (bSwapCoordSystem)
		{
			tframes[i].t[X]=trans[X];
			tframes[i].t[Y]=trans[Z];
			tframes[i].t[Z]=-trans[Y];
		}
		else
		{
			tframes[i].t[X]=trans[X];
			tframes[i].t[Y]=trans[Y];
			tframes[i].t[Z]=trans[Z];
		}

		//tframes[i].time=(float)i/(float)OUTPUT_FPS;
		tframes[i].time=(float)i/(float)fps3DS;
		//tframes[i].time=(float)start/(float)fps3DS + (float)i/(float)fps3DS;
	}
}

// Builds a list of SAnimQFrames for each rotation in the model
// at the game's FPS rate
void AnimExporter::GetQFrames(Gfx::SAnimQFrame* qframes,INode* root,bool bExportFromPelvis,bool bOneNode,ProgressBar* pbar)
{ FUNC_ENTER("AnimExporter::GetQFrames"); 
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate


	if (!bOneNode)
	{
		int numBones=skeleton->GetCount();
		for(int i=0;i<numBones;i++)
		{
			INode* node=skeleton->GetBone(i);

			// Only bones that are in the partial animation set need be processed
			// bones that aren't in the set will be discarded later
			if (skeleton->GetPartialAnimState(i))
				BuildNodeQFrames(qframes+num3DSFrames*i,node,bExportFromPelvis,pbar);

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}
	}
	else
		BuildNodeQFrames(qframes,root,bExportFromPelvis,pbar);
}

// Builds a list of SAnimTFrames for each translation in the model
// at the game's FPS rate
void AnimExporter::GetTFrames(Gfx::SAnimTFrame* tframes,INode* root,bool bExportFromPelvis,bool bOneNode,ProgressBar* pbar)
{ FUNC_ENTER("AnimExporter::GetTFrames"); 
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate


	if (!bOneNode)
	{
		int numBones=skeleton->GetCount();
		for(int i=0;i<numBones;i++)
		{
			INode* node=skeleton->GetBone(i);

			// Only bones that are in the partial animation set need be processed
			// bones that aren't in the set will be discarded later
			if (skeleton->GetPartialAnimState(i))
				BuildNodeTFrames(tframes+num3DSFrames*i,node,bExportFromPelvis,pbar);

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}
	}
	else
		BuildNodeTFrames(tframes,root,bExportFromPelvis,pbar);
}

void AnimExporter::GetQTFrames(Gfx::SAnimQFrame* qframes,
							   Gfx::SAnimTFrame* tframes,
							   INode*            root,
							   bool              bExportFromPelvis,
							   bool              bOneNode,
							   ProgressBar*      pbar)
{ FUNC_ENTER("AnimExporter::GetQTFrames");
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate


	if (!bOneNode)
	{
		int numBones=skeleton->GetCount();
		for(int i=0;i<numBones;i++)
		{
			INode* node=skeleton->GetBone(i);

			// Only bones that are in the partial animation set need be processed
			// bones that aren't in the set will be discarded later
			if (skeleton->GetPartialAnimState(i))
			{
				BuildNodeQTFrames(qframes+num3DSFrames*i,
				                  tframes+num3DSFrames*i,
								  node,
								  bExportFromPelvis,
								  pbar);
			}

			// Handle possibility of user pressing cancel
			if (pbar && pbar->WasCanceled())
				return;
		}
	}
	else
		BuildNodeQFrames(qframes,root,bExportFromPelvis,pbar);
}

void AnimExporter::FreeFrames(Gfx::SAnimQFrame** frame)
{ FUNC_ENTER("AnimExporter::FreeFrames"); 
	if (frame)
		delete [] *frame;

	*frame=NULL;
}

void AnimExporter::FreeFrames(Gfx::SAnimTFrame** frame)
{ FUNC_ENTER("AnimExporter::FreeFrames"); 
	if (frame)
		delete [] *frame;

	*frame=NULL;
}

int AnimExporter::GetNodeName(CStr* strNames,INode* node,int ID)
{ FUNC_ENTER("AnimExporter::GetNodeName"); 
	if (node->IsHidden())
		ID--;
	else
		strNames[ID]=node->GetName();

	int numChildren=node->NumberOfChildren();

	INode* newNode;

	for(int i=0;i<numChildren;i++)
	{
		newNode=node->GetChildNode(i);
		ID=GetNodeName(strNames,newNode,++ID);
	}

	return ID;
}


int AnimExporter::GetParentName(CStr* strNames,INode* node,int ID)
{ FUNC_ENTER("AnimExporter::GetParentName"); 
	INode* parent=node->GetParentNode();

	if (node->IsHidden())
		ID--;
	else
		if (parent && !parent->IsHidden())
			strNames[ID]=parent->GetName();
		else
			strNames[ID]="";

	int numChildren=node->NumberOfChildren();

	INode* newNode;

	for(int i=0;i<numChildren;i++)
	{
		newNode=node->GetChildNode(i);
		ID=GetParentName(strNames,newNode,++ID);
	}

	return ID;
}

void AnimExporter::GetNodeNames(CStr* strNames,INode* root)
{ FUNC_ENTER("AnimExporter::GetNodeNames"); 
	int numChildren=root->NumberOfChildren();
	int ID=0;

	INode* node;

	if (!root->IsHidden())
		strNames[ID]=root->GetName();
	else
		ID=-1;

	for(int i=0;i<numChildren;i++)
	{
		node=root->GetChildNode(i);
		ID=GetNodeName(strNames,node,++ID);
	}
}

void AnimExporter::GetParentNames(CStr* strNames,INode* root)
{ FUNC_ENTER("AnimExporter::GetParentNames"); 
	int numChildren=root->NumberOfChildren();
	int ID=0;

	INode* node;
	INode* parent=root->GetParentNode();

	if (!root->IsHidden())
	{
		if (parent && !parent->IsHidden())
			strNames[ID]=root->GetName();
		else
			strNames[ID]="";
	}
	else
		ID=-1;


	for(int i=0;i<numChildren;i++)
	{
		node=root->GetChildNode(i);
		ID=GetParentName(strNames,node,++ID);
	}
}

void AnimExporter::HideDummy(INode* node)
{ FUNC_ENTER("AnimExporter::HideDummy"); 
	CStr name=node->GetName();

	if (name==CStr("Dummy_scale"))
		node->Hide(TRUE);

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		HideDummy(newnode);
	}
}

void AnimExporter::UnhideDummy(INode* node)
{ FUNC_ENTER("AnimExporter::UnhideDummy"); 
	CStr name=node->GetName();

	if (name==CStr("Dummy_scale"))
		node->Hide(FALSE);

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		UnhideDummy(newnode);
	}
}

void AnimExporter::UnhideDummies()
{ FUNC_ENTER("AnimExporter::UnhideDummies"); 
	INode* node=gInterface->GetRootNode();

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		UnhideDummy(newnode);
	}
}

void AnimExporter::HideDummies()
{ FUNC_ENTER("AnimExporter::HideDummies"); 
	INode* node=gInterface->GetRootNode();

	int numChildren=node->NumberOfChildren();

	INode* newnode;

	for(int i=0;i<numChildren;i++)
	{
		newnode=node->GetChildNode(i);
		HideDummy(newnode);
	}
}

bool AnimExporter::ApproxEqual(float v1,float v2)
{ FUNC_ENTER("AnimExporter::ApproxEqual"); 
	if (v1 > v2-0.00005f &&
		v1 < v2+0.00005f)
		return true;

	return false;
}

void AnimExporter::RemoveDblKeys(int numNodes,int numFrames)
{ FUNC_ENTER("AnimExporter::RemoveDblKeys"); 
	for(int i=0;i<numNodes;i++)
	{
		int curFrame=i*numFrames;

		if (numCompressedQFrames[i]==2)
		{
			if (ApproxEqual(compQframes[curFrame].q[X],compQframes[curFrame+1].q[X]) &&
				ApproxEqual(compQframes[curFrame].q[Y],compQframes[curFrame+1].q[Y]) &&
				ApproxEqual(compQframes[curFrame].q[Z],compQframes[curFrame+1].q[Z]) &&
				ApproxEqual(compQframes[curFrame].q[W],compQframes[curFrame+1].q[W]))
			{
				//compQframes[curFrame].time=compQframes[curFrame+1].time;
				numCompressedQFrames[i]=1;
			}
		}

		if (numCompressedTFrames[i]==2)
		{
			if (ApproxEqual(compTframes[curFrame].t[0],compTframes[curFrame+1].t[0]) &&
			    ApproxEqual(compTframes[curFrame].t[1],compTframes[curFrame+1].t[1]) &&
				ApproxEqual(compTframes[curFrame].t[2],compTframes[curFrame+1].t[2]))
			{
				//compTframes[curFrame].time=compTframes[curFrame+1].time;
				numCompressedTFrames[i]=1;
			}
		}
	}
}

void AnimExporter::RemoveDblKeys(int numNodes, int numFrames,
								 Gfx::SAnimQFrame* compQframes,
								 Gfx::SAnimTFrame* compTframes,
								 int* numCompressedQFrames,
								 int* numCompressedTFrames)
{ FUNC_ENTER("AnimExporter::RemoveDblKeys"); 
	for(int i=0;i<numNodes;i++)
	{
		int curFrame=i*numFrames;

		if (numCompressedQFrames[i]==2)
		{
			if (compQframes[curFrame].q[X]==compQframes[curFrame+1].q[X] &&
				compQframes[curFrame].q[Y]==compQframes[curFrame+1].q[Y] &&
				compQframes[curFrame].q[Z]==compQframes[curFrame+1].q[Z] &&
				compQframes[curFrame].q[W]==compQframes[curFrame+1].q[W])
			{
				//compQframes[curFrame].time=compQframes[curFrame+1].time;
				numCompressedQFrames[i]=1;
			}
		}

		if (numCompressedTFrames[i]==2)
		{
			if (compTframes[curFrame].t[0]==compTframes[curFrame+1].t[0] &&
			    compTframes[curFrame].t[1]==compTframes[curFrame+1].t[1] &&
				compTframes[curFrame].t[2]==compTframes[curFrame+1].t[2])
			{
				//compTframes[curFrame].time=compTframes[curFrame+1].time;
				numCompressedTFrames[i]=1;
			}
		}
	}
}

bool AnimExporter::ExportAnim(HWND hwnd,INode* node,
							  char* Filename,
							  char* ModelName,
							  int start,
							  int end,
							  float errorQ,
							  float errorT,
							  TolData* tdata,
							  bool bOneFrame,
							  bool bDebug,
							  bool bUseCompression,
							  bool bCompressTime,
							  bool bRotateRoot,
							  PartialAnimSet* pPartialAnim)
{ FUNC_ENTER("AnimExporter::ExportAnim"); 
	int fps3DS=GetFrameRate();						// Frames per second MAX is set to
	int num3DSFrames=end-start+1;					// Number of frames in animation at 3DS MAX's frame rate
	int reqFrames=num3DSFrames*OUTPUT_FPS/fps3DS;	// Number of frames at the game's frame rate

	this->bCompressTime = bCompressTime;
	this->bRotateRoot   = bRotateRoot;

	if (skeleton)
		delete skeleton;
	skeleton=new CSkeletonData(node);

	if (pPartialAnim)
		skeleton->RemoveNonPartialAnimBones(pPartialAnim);

	rootbone=skeleton->GetRootBone(node);

	//UnhideDummies();

	INode* root=GetRoot(node);
	int numNodes=CountNodes(root);

	if (!root)
		return false;

	CStr name2=node->GetName();
	CStr name=root->GetName();

	// Write out animation file
	fpExport=fopen(Filename,"wb");

	if (!fpExport)
	{
		if (bRemoveReadOnly)
		{
			if (_access(Filename,WRITE_ACCESS)!=0)
			{
				if (_chmod(Filename, _S_IREAD | _S_IWRITE)==-1)
				{
					MessageBox(NULL,"Couldn't change file attribute setting from read only.","Export Animations",MB_ICONSTOP|MB_OK);
					return false;
				}				
			}			
		}
		else if( bCheckOut )
		{
			char *args[4];
			int process_result;

			args[0] = "p4";
			args[1] = "edit";
			args[2] = Filename;
			args[3] = NULL;
			//process_result = spawnvp( _P_WAIT, args[0], args );
			process_result = ExecuteCommand( args );
			if (_access(Filename,WRITE_ACCESS)!=0)
			{
				MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Animations",MB_ICONSTOP|MB_OK);
				return false;
			}
		}
		else
		{
			// Check the file attributes to see if the file is read-only
			if (_access(Filename,WRITE_ACCESS)!=0)
			{
				char ErrMsg[256];
				int  rVal;
				sprintf(ErrMsg,"The file '%s' does not have write access enabled.\nWould you like to manually remove the Read-Only flag on all items?\nOr would you like to check this file out?",(char*)Filename);
				rVal=MessageBoxAll(hwnd,ErrMsg,"Export Animations",MB_CHECKOUT_UNLOCK);

				if (rVal==IDWRITABLE)
				{
					bRemoveReadOnly=true;
					rVal=IDNO;

					if (_chmod(Filename, _S_IREAD | _S_IWRITE)==-1)
					{
						MessageBox(NULL,"An error occurred while changing the file's permission settings!","Export Animations",MB_ICONSTOP|MB_OK);
						return false;
					}
				}
				else if( rVal == IDCHECKOUT )
				{
					char *args[4];
					int process_result;

					bCheckOut = true;
					rVal=IDNO;					

					args[0] = "p4";
					args[1] = "edit";
					args[2] = Filename;
					args[3] = NULL;
					//process_result = spawnvp( _P_WAIT, args[0], args );
					process_result = ExecuteCommand( args );
					if (_access(Filename,WRITE_ACCESS)!=0)
					{
						MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Animations",MB_ICONSTOP|MB_OK);
						return false;
					}
				}
				else
				{
					return false;
				}				
			}			
		}

		fpExport=fopen(Filename,"wb");
		if (!fpExport)
			return false;
	}

	GetAnim(root,start,end,errorQ,errorT,tdata,bUseCompression);
	
	if (bOneFrame)
		RemoveDblKeys(numNodes,num3DSFrames);

	// Compute total QFrames and TFrames
	int QKeys=0;
	int TKeys=0;
	int i;

	for(i=0;i<numNodes;i++)
	{
		QKeys+=numCompressedQFrames[i];
		TKeys+=numCompressedTFrames[i];		
	}
	
	// Output the stored data in THPS4 format
	//CSkeletonData* skeleton=new CSkeletonData(node);

	// Output file header
	SHeader header;
	header.version=ANIMFILE_VERSION;
	header.flags=INTERMEDIATE_FORMAT;
	
	if (!bUseCompression)
		header.flags|=UNCOMPRESSED_FORMAT;

	if (bRotateRoot)
		header.flags|=PREROTATEDROOT;

	if (bCompressTime)
		header.flags|=COMPRESSEDTIME;

	if (pPartialAnim)
		header.flags|=PARTIALANIM;

	header.skeletonName=skeleton->GetChecksum();			// hard-coded default for now
	header.duration=(float)(num3DSFrames-1)/(float)fps3DS;
	header.numBones=numNodes;
	header.numQKeys=QKeys;
	header.numTKeys=TKeys;
	header.numUserDefinedKeys=0;

	if (!header.Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	// Output Skeleton data

	// Output using CSkeleton instead
	if (!skeleton->Write(fpExport))
	{
		delete skeleton;
		skeleton=NULL;
		return false;
	}

	///////////////////////////////////////////////////

	// Output compressed rotation data
	SQData sqdata(numNodes,QKeys);
	int QKeyPos=0;

	FILE* fpDebug=NULL;

	if (bDebug)
	{
		fpDebug=fopen("c:\\debug.txt","w");

		fprintf(fpDebug,"Duration: %f\n",header.duration);
		fprintf(fpDebug,"NumTKeys: %i\n",header.numTKeys);
		fprintf(fpDebug,"NumQKeys: %i\n",header.numQKeys);

		if (!fpDebug)
			MessageBox(NULL,"Failed to open c:\\debug.txt","Debug output",MB_OK|MB_ICONWARNING);
	}

	for(i=0;i<numNodes;i++)
	{
		sqdata.numKeysPerBone[i]=numCompressedQFrames[i];

		if (bDebug)
		{
			CStr name=skeleton->GetBone(i)->GetName();
			fprintf(fpDebug,"Node: %s\n",(char*)name);
		}

		for(int j=0;j<numCompressedQFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
				float ftmp = compQframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);
			}
			else
				sqdata.theQKeys[QKeyPos].time = compQframes[curFrame].time;

			sqdata.theQKeys[QKeyPos].qx   = compQframes[curFrame].q[X];
			sqdata.theQKeys[QKeyPos].qy   = compQframes[curFrame].q[Y];
			sqdata.theQKeys[QKeyPos].qz   = compQframes[curFrame].q[Z];
			sqdata.theQKeys[QKeyPos].real = compQframes[curFrame].q[W];

			if (bDebug)
			{
				float EulerAng[3],Deg1[3];
				Quat quat = Quat(sqdata.theQKeys[QKeyPos].qx,
					             sqdata.theQKeys[QKeyPos].qy,
					             sqdata.theQKeys[QKeyPos].qz,
					             sqdata.theQKeys[QKeyPos].real);
				
				QuatToEuler(quat,EulerAng);
				
				Deg1[X]=RadToDeg(EulerAng[X]);
				Deg1[Y]=RadToDeg(EulerAng[Y]);
				Deg1[Z]=RadToDeg(EulerAng[Z]);

				if (bCompressTime)
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%i\n",i,j,*((int*)&sqdata.theQKeys[QKeyPos].time));
				else
					fprintf(fpDebug,"Q[node=%i frame=%i].time=%f\n",i,j,compQframes[curFrame].time);

				fprintf(fpDebug,"Q[node=%i frame=%i].qx=%f\tEulerX: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[X],EulerAng[X],Deg1[X],compQframes[curFrame].bFlip[X]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qy=%f\tEulerY: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[Y],EulerAng[Y],Deg1[Y],compQframes[curFrame].bFlip[Y]);
				fprintf(fpDebug,"Q[node=%i frame=%i].qz=%f\tEulerZ: %f\tDeg: %f\tFlip: %i\n",i,j,compQframes[curFrame].q[Z],EulerAng[Z],Deg1[Z],compQframes[curFrame].bFlip[Z]);
				fprintf(fpDebug,"Q[node=%i frame=%i].real=%f\n\n",i,j,compQframes[curFrame].q[W]);
			}

			QKeyPos++;
		}
	}

	if (!sqdata.Write(fpExport))
		return false;

	// Output compressed translation data
	STData stdata(numNodes,TKeys);
	int TKeyPos=0;

	for(i=0;i<numNodes;i++)
	{
		stdata.numKeysPerBone[i]=numCompressedTFrames[i];
		
		if (bDebug)
		{
			CStr name=skeleton->GetBone(i)->GetName();
			fprintf(fpDebug,"Node: %s\n",(char*)name);
		}

		for(int j=0;j<numCompressedTFrames[i];j++)
		{
			//int curFrame=j+i*reqFrames;
			int curFrame=j+i*num3DSFrames;

			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&stdata.theTKeys[TKeyPos].time;
				float ftmp = compTframes[curFrame].time * (float)OUTPUT_FPS;
				*val = Round(ftmp);
			}
			else
				stdata.theTKeys[TKeyPos].time = compTframes[curFrame].time;
	
			stdata.theTKeys[TKeyPos].tx   = compTframes[curFrame].t[0];
			stdata.theTKeys[TKeyPos].ty   = compTframes[curFrame].t[1];
			stdata.theTKeys[TKeyPos].tz   = compTframes[curFrame].t[2];

			if (bDebug)
			{
				if (bCompressTime)
					fprintf(fpDebug,"T[node=%i frame=%i].time=%i\n",i,j,*((int*)&stdata.theTKeys[TKeyPos].time));
				else
					fprintf(fpDebug,"T[node=%i frame=%i].time=%f\n",i,j,compTframes[curFrame].time);

				fprintf(fpDebug,"T[node=%i frame=%i].tx=%f\n",i,j,compTframes[curFrame].t[0]);
				fprintf(fpDebug,"T[node=%i frame=%i].ty=%f\n",i,j,compTframes[curFrame].t[1]);
				fprintf(fpDebug,"T[node=%i frame=%i].tz=%f\n\n",i,j,compTframes[curFrame].t[2]);
			}

			TKeyPos++;
		}
	}

	if (bDebug)
		fclose(fpDebug);

	if (!stdata.Write(fpExport))
		return false;

	fclose(fpExport);

	//HideDummies();

	delete skeleton;
	skeleton=NULL;

	return true;
}

unsigned int AnimExporter::GetChecksum(char* Filename)
{ FUNC_ENTER("AnimExporter::GetChecksum"); 
	FILE* fp=fopen(Filename,"rb");

	if (!fp)
		return 0;
	
	unsigned int checksum;

	SHeader header;
	if (!header.Read(fp))
	{
		fclose(fp);
		return 0;
	}

	// The next 4 bytes in the file should be the skeleton checksum
	if (!fread(&checksum,sizeof(unsigned int),1,fp))
	{
		fclose(fp);
		return 0;
	}

	fclose(fp);
	return checksum;
}

// This copy method will sample the from controller and smooth out all flips
// Nikolai 1-15-99

// This is based off the code in \MAXSDK\SAMPLES\CONTROLLERS\EULRCTRL.CPP
// in EulerRotation::Copy(Control *from)
// This should be done BEFORE compression
//#define THRESHHOLD 3.140350877   // Allow rotations of up to 179 degrees per frame (not quite PI)
#define THRESHHOLD 1.0

void AnimExporter::FixEulerFlips(Gfx::SAnimQFrame* pdata,int numKeys)
{ FUNC_ENTER("AnimExporter::FixEulerFlips"); 
	// Don't fix Euler flips, so we can check this in
#ifndef USE_EULERFIX
	return;
#endif

	if (numKeys<2)
		return;

	/*
	for(int curKey=0;curKey<numKeys;curKey++)
	{
		float EulerAng[3];

		Quat quat = Quat(pdata[curKey].q[X],
						 pdata[curKey].q[Y],
						 pdata[curKey].q[Z],
						 pdata[curKey].q[W]);

		QuatToEuler(quat,EulerAng);
		
		EulerAng[X]=RadToDeg(EulerAng[X]);
		EulerAng[Y]=RadToDeg(EulerAng[Y]);
		EulerAng[Z]=RadToDeg(EulerAng[Z]);

		int i;
		i=0;
	}
	*/

	Quat qPrev;
	Quat qCurr;

	float   eaCurr[3];
	float   eaPrev[3];
	float   EulerAng[3] = {0,0,0};
	Matrix3 tmPrev,tmCurr;

	// Fill out a MAX Quat with our quat
	qPrev.x = pdata[0].q[X];
	qPrev.y = pdata[0].q[Y];
	qPrev.z = pdata[0].q[Z];
	qPrev.w = pdata[0].q[W];
	
	float dEuler[3],f;	

	// Here we sample over the time range, to detect flips
	for(int curKey=1;curKey<numKeys;curKey++)
	{
		//from->GetValue(time,&qCurr,iv);
		qCurr.x = pdata[curKey].q[X];
		qCurr.y = pdata[curKey].q[Y];
		qCurr.z = pdata[curKey].q[Z];
		qCurr.w = pdata[curKey].q[W];
		
		QuatToEuler(qPrev,eaPrev);
		QuatToEuler(qCurr,eaCurr);

		float prevDeg[3],currDeg[3];		// Switch to degrees for DEBUG
		prevDeg[0]=RadToDeg(eaPrev[0]);
		prevDeg[1]=RadToDeg(eaPrev[1]);
		prevDeg[2]=RadToDeg(eaPrev[2]);

		currDeg[0]=RadToDeg(eaCurr[0]);
		currDeg[1]=RadToDeg(eaCurr[1]);
		currDeg[2]=RadToDeg(eaCurr[2]);

		// The Euler/Quat ratio is the relation of the angle difference in Euler space to 
		// the angle difference in Quat space. If this ration is bigger than PI the rotation 
		// between the two time steps contains a flip

		qPrev.MakeMatrix(tmPrev);
		qCurr.MakeMatrix(tmCurr);

		f = GetEulerMatAngleRatio(tmPrev,tmCurr,eaPrev,eaCurr,EULERTYPE_XYZ);	

		bool bFlip[3];
		bFlip[X]=false;
		bFlip[Y]=false;
		bFlip[Z]=false;

		QuatToEuler(qPrev,eaPrev);
		QuatToEuler(qCurr,eaCurr);

		if ( f > PI )
		//if (1)
		{
			for(int j=0;j<3;j++)
			{
				// find the sign flip (Over 110 degrees for a frame, we flip)
				if ((eaPrev[j]<0 && eaCurr[j]>0) ||
					(eaPrev[j]>0 && eaCurr[j]<0))
				{
					// The values should be close together omiting the sign
					float fval=fabs(RadToDeg(eaCurr[j])+RadToDeg(eaPrev[j]));
					
					// But the distance should be far away from the other if it was flipped
					float fdist=fabs(RadToDeg(eaPrev[j])-RadToDeg(eaCurr[j]));

					GetEulerMatAngleRatio(tmPrev,tmCurr,eaPrev,eaCurr,EULERTYPE_XYZ);

					if (fval<35 && fdist>100)
					{
						char strErr[256];
						sprintf(strErr,"Frame: %i  Flip Axis: %i\nPrevDeg: %f  CurrDeg: %f\nDiff: %f",curKey,j,RadToDeg(eaPrev[j]),RadToDeg(eaCurr[j]),fval);

						MessageBox(NULL,strErr,"Debug",MB_OK);

						//EulerAng[j] = eaCurr[j];// * -1.0f;	// Flip it back
						EulerAng[j] = (2*PI - (float) (fabs(eaCurr[j]) + fabs(eaPrev[j]))) * (eaPrev[j] > 0 ? 1 : -1);
						bFlip[j]=true;
					}
					else
						EulerAng[j] = eaCurr[j];
				}
			}
		}
		else
		{
			for(int j=0 ; j < 3 ; j++)
			{
				dEuler[j] = eaCurr[j]-eaPrev[j];
				EulerAng[j] += dEuler[j];
			}
		}
		
		/*
		if(  f > PI)
		{
			// We found a flip here
			for(int j=0 ; j < 3 ; j++)
			{				
				// find the sign flip :
				if(fabs((eaCurr[j]-eaPrev[j])) < 2*PI-THRESHHOLD )
					dEuler[j] = eaCurr[j]-eaPrev[j];
				else
				{
					char strErr[256];
					sprintf(strErr,"Frame: %i  Flip Axis: %i\nPrevDeg: %f  CurrDeg: %f\nDiff: %f",curKey,j,RadToDeg(eaPrev[j]),RadToDeg(eaCurr[j]),RadToDeg(f));

					MessageBox(NULL,strErr,"Debug",MB_OK);

					// unflip the flip
					dEuler[j] = (2*PI - (float) (fabs(eaCurr[j]) + fabs(eaPrev[j]))) * (eaPrev[j] > 0 ? 1 : -1);
				}
				
				EulerAng[j] += dEuler[j];
			}
		}
		else
		{
			// Add up the angle difference
			for(int j=0 ; j < 3 ; j++)
			{
				dEuler[j] = eaCurr[j]-eaPrev[j];
				EulerAng[j] += dEuler[j];
			}
		}
		*/

		// Write out our new key
		Quat qTemp;
		EulerToQuat(EulerAng,qTemp,EULERTYPE_XYZ);
		pdata[curKey].q[X] = -qTemp.x;
		pdata[curKey].q[Y] = -qTemp.y;
		pdata[curKey].q[Z] = -qTemp.z;
		pdata[curKey].q[W] = -qTemp.w;
		pdata[curKey].bFlip[X]=bFlip[X];
		pdata[curKey].bFlip[Y]=bFlip[Y];
		pdata[curKey].bFlip[Z]=bFlip[Z];
		qPrev = qCurr;
	}
}
