#include "FuncEnter.h"

/*
	AnimStore.cpp
	A utility plugin to allow the storing of a node's anim transforms
	and the reapplication thereof to another model while retaining 
	hierarchial transform data from deleted nodes

    aml - 10-18-02
*/

#include "AnimStore.h"
#include "Resource.h"
#include "../UI/NodeSelect.h"
#include "../misc/maxutil.h"

#define SUBANIM_ROTX  0
#define SUBANIM_ROTY  1
#define SUBANIM_ROTZ  2

extern HINSTANCE hInstance;
extern Interface* gInterface;

HWND        hwndAnimStore;
NodeSelect  *nsPreserve;
NodeSelect  *nsAppPoint;

AnimStoreClassDesc theAnimStoreClassDesc;

ClassDesc2* GetAnimStoreClassDesc( void )
{ FUNC_ENTER("GetAnimStoreClassDesc"); 
	return &theAnimStoreClassDesc;
}

AnimStore::AnimStore()
{ FUNC_ENTER("AnimStore::AnimStore"); 
	nsPreserve = NULL;
	nsAppPoint = NULL;
}

AnimStore::~AnimStore()
{ FUNC_ENTER("AnimStore::~AnimStore"); 

}

CStr GetLoad()
{ FUNC_ENTER("GetLoad"); 
	OPENFILENAME ofn;
	char filename[256]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=gInterface->GetMAXHWnd();
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Transform Files (*.tsf)\0*.tsf\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Load captured transforms...";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".tsf");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetOpenFileName(&ofn);

	return CStr(filename);
}

CStr GetSave()
{ FUNC_ENTER("GetSave"); 
	OPENFILENAME ofn;
	char filename[256]="";

	ofn.lStructSize=sizeof(ofn);
	ofn.hwndOwner=gInterface->GetMAXHWnd();
	ofn.hInstance=hInstance;
	ofn.lpstrFilter="Transform Files (*.tsf)\0*.tsf\0\0";
	ofn.lpstrCustomFilter=NULL;
	ofn.nMaxCustFilter=0;
	ofn.nFilterIndex=0;
	ofn.lpstrFile=filename;
	ofn.nMaxFile=256;
	ofn.lpstrFileTitle=NULL;
	ofn.nMaxFileTitle=128;
	ofn.lpstrInitialDir=NULL;
	ofn.lpstrTitle="Save captured transforms as...";
	ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
	ofn.nFileOffset=0;
	ofn.nFileExtension=0;
	ofn.lpstrDefExt=TEXT(".tsf");
	ofn.lCustData=0;
	ofn.lpfnHook=NULL;
	ofn.lpTemplateName=NULL;

	GetSaveFileName(&ofn);

	return CStr(filename);
}

enum ContType
{
	CONTTYPE_AXISX,
	CONTTYPE_AXISY,
	CONTTYPE_AXISZ,
};

bool FindTime(Tab<TimeValue>& times, TimeValue timeval)
{ FUNC_ENTER("FindTime"); 
	int nSize = times.Count();

	for(int i = 0; i < nSize; i++)
	{
		if (timeval == times[i])
			return true;
	}

	return false;
}

static int TimeCompare(const void* time1, const void* time2)
{ FUNC_ENTER("TimeCompare"); 
	TimeValue* ptime1 = (TimeValue*)time1;
	TimeValue* ptime2 = (TimeValue*)time2;

	if (*ptime1 < *ptime2)
		return -1;

	if (*ptime1 > *ptime2)
		return 1;

	// Must be equal
	return 0;
}

void AddAnimNode()
{ FUNC_ENTER("AddAnimNode"); 
	int nNodes = gInterface->GetSelNodeCount();

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = gInterface->GetSelNode(i);

		int idx = SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_ADDSTRING, 0, (LPARAM)(char*)node->GetName());
		SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)node);
	}
}

void RemoveAnimNode()
{ FUNC_ENTER("RemoveAnimNode"); 
	int idx = SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_GETCURSEL, 0, 0);
	
	if (idx != LB_ERR)
		SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_DELETESTRING, idx, 0);
}

// Builds up node times based solely on the Rotation controller's keytimes
// and uses the nodelist in the rollout for combining times as opposed to
// using combined time of the child nodes
bool BuildTimes(Tab<TimeValue>& times, INode* nodeStart)
{ FUNC_ENTER("BuildTimes"); 
	if (!nodeStart)
		return false;

	Control* cTM = nodeStart->GetTMController();

	if (cTM)
	{
		Tab<TimeValue> newtimes;
		Control* cRot = cTM->GetRotationController();

		if (cRot)
		{
			Interval intervalForever = FOREVER;
			cRot->GetKeyTimes(newtimes, intervalForever, KEYAT_ROTATION);

			// Now we need to add the times contained in the newtimes table into
			// the times in the original time table if the newtimes don't duplicate
			// the original times
			int nSize = newtimes.Count();

			for(int i = 0; i < nSize; i++)
			{
				if (!FindTime(times, newtimes[i]))
				{
					times.Append(1, &(newtimes[i]));
				}
			}
		}
	}

	return true;
}

bool BuildTimesList(Tab<TimeValue>& times, INode* nodeStart)
{ FUNC_ENTER("BuildTimesList"); 
	// Build the times up for the node that we're capturing
	BuildTimes(times, nodeStart);

	// Build the times up for the relevant key nodes in the list
	int nItems = SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < nItems; i++)
	{
		INode* node = (INode*)SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_GETITEMDATA, (WPARAM)i, 0);
		BuildTimes(times, node);
	}

	times.Sort(TimeCompare);
	return true;
}

bool BuildTimesCont(Tab<TimeValue>& times, INode* nodeStart, ContType type)
{ FUNC_ENTER("BuildTimesCont"); 
	INode*   node = nodeStart;
	
	if (!node)
		return false;

	Control* cTM  = node->GetTMController();

	if (cTM)
	{
		Control* cRot = cTM->GetRotationController();
		
		if (cRot)
		{
			Control* cRotAxis;

			switch(type)
			{
			case CONTTYPE_AXISX:
				cRotAxis = cRot->GetXController();
				break;

			case CONTTYPE_AXISY:
				cRotAxis = cRot->GetYController();
				break;

			case CONTTYPE_AXISZ:
				cRotAxis = cRot->GetZController();
				break;

			default:
				cRotAxis = cRot->GetXController();
				break;
			}

			if (cRotAxis)
			{
				Tab<TimeValue> newtimes;
				Interval       intervalForever = FOREVER;
				cRotAxis->GetKeyTimes(newtimes, intervalForever, KEYAT_ROTATION);

				// Now we need to add the times contained in the newtimes table into
				// the times in the original time table if the newtimes don't duplicate
				// the original times
				
				int nSize = newtimes.Count();

				for(int i = 0; i < nSize; i++)
				{
					int time = newtimes[i] / GetTicksPerFrame();

					if (!FindTime(times, newtimes[i]))
					{
						times.Append(1, &(newtimes[i]));
					}
				}
			}
		}
	}

	return true;
}

bool BuildTimesList(Tab<TimeValue>& times, INode* nodeStart, ContType type)
{ FUNC_ENTER("BuildTimesList"); 
	// Build the times up for the node that we're capturing
	BuildTimesCont(times, nodeStart, type);

	// Build the times up for the relevant key nodes in the list
	int nItems = SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < nItems; i++)
	{
		INode* node = (INode*)SendDlgItemMessage(hwndAnimStore, IDC_KEYNODELIST, LB_GETITEMDATA, (WPARAM)i, 0);
		BuildTimesCont(times, node, type);
	}

	times.Sort(TimeCompare);
	return true;
}

bool BuildTimes(Tab<TimeValue>& times, INode* nodeStart, ContType type)
{ FUNC_ENTER("BuildTimes"); 
	INode*   node = nodeStart;
	
	if (!node)
		return false;

	Control* cTM  = node->GetTMController();

	if (cTM)
	{
		Control* cRot = cTM->GetRotationController();
		
		if (cRot)
		{
			Control* cRotAxis;

			switch(type)
			{
			case CONTTYPE_AXISX:
				cRotAxis = cRot->GetXController();
				break;

			case CONTTYPE_AXISY:
				cRotAxis = cRot->GetYController();
				break;

			case CONTTYPE_AXISZ:
				cRotAxis = cRot->GetZController();
				break;

			default:
				cRotAxis = cRot->GetXController();
				break;
			}

			if (cRotAxis)
			{
				Tab<TimeValue> newtimes;
				Interval       intervalForever = FOREVER;
				cRotAxis->GetKeyTimes(newtimes, intervalForever, KEYAT_ROTATION);

				// Now we need to add the times contained in the newtimes table into
				// the times in the original time table if the newtimes don't duplicate
				// the original times
				
				int nSize = newtimes.Count();

				for(int i = 0; i < nSize; i++)
				{
					if (!FindTime(times, newtimes[i]))
					{
						times.Append(1, &(newtimes[i]));
					}
				}
			}
		}
	}

	int nKids = node->NumberOfChildren();
	INode* child;

	for(int i = 0; i < nKids; i++)
	{
		child = node->GetChildNode(i);
		BuildTimes(times, child, type);
	}

	return true;
}

// We'll do this with quaternions now instead of Euler angles
void AnimSave2()
{ FUNC_ENTER("AnimSave2"); 
	INode* node = nsPreserve->GetNode();

	if (!node)
	{
		MessageBox(gInterface->GetMAXHWnd(), "No valid start chain node was selected.", "AnimLoad", MB_ICONWARNING|MB_OK);
		return;		
	}


}

void AnimLoad2()
{ FUNC_ENTER("AnimLoad2"); 
	INode* nodeApply = nsAppPoint->GetNode();

	// Build up rotation transforms
	// These should be combined between the start and end chain nodes
	// And computed for each time in which a keyframe exists
	FILE*  fp;
	INode* curnode;
	
	Tab<TimeValue>  times;
	int             nTimes;
	Matrix3         tm;
	Interval        intervalForever = FOREVER;
	Control* cTM  = nodeApply->GetTMController();

	if (!cTM)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to retrieve Transform Controller","AnimLoad", MB_ICONSTOP|MB_OK);
		return;
	}

	Control* cRot = cTM->GetRotationController();

	if (!cRot)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to retrieve Rotation Controller","AnimLoad", MB_ICONSTOP|MB_OK);
		return;
	}

	// [We're going to rely on ROTX, ROTY, ROTZ and the chain having uniform samples]
	// This assumption is causing problems.  We need to evaluate all the controllers up to an
	// end point and combine their times.  We'll then add the samples at all the combined times
	// to the transform file by requesting the rotation controller to evaluate those times.

	curnode = nodeApply;

	CStr filename = GetSave();

	if (filename.Length() == 0)
		return;

	fp = fopen(filename, "wb");

	if (!fp)
	{
		char sErr[256];
		sprintf(sErr, "Failed to open transform file '%s' for writting.", (char*)filename);
		MessageBox(gInterface->GetMAXHWnd(), sErr, "AnimSave", MB_ICONWARNING|MB_OK);
		return;
	}

	// Dump keys
	BuildTimesList(times, nodeApply);

	nTimes = times.Count();

	fwrite(&nTimes, sizeof(int), 1, fp);

	int i;

	for(i = 0; i < nTimes; i++)
	{
		Quat val;
		Interval intervalForever = FOREVER;
		
		fwrite(&times[i], sizeof(TimeValue), 1, fp);
		cRot->GetValue(times[i], &val, intervalForever, CTRL_ABSOLUTE);
	}

	fclose(fp);
}

void AnimSave()
{ FUNC_ENTER("AnimSave"); 
	// Grab the transform data for the specified chain of nodes
	INode* node    = nsPreserve->GetNode();

	if (!node)
	{
		MessageBox(gInterface->GetMAXHWnd(), "No valid start chain node was selected.", "AnimLoad", MB_ICONWARNING|MB_OK);
		return;
	}

	// Build up rotation transforms
	// These should be combined between the start and end chain nodes
	// And computed for each time in which a keyframe exists
	FILE*  fp;
	INode* curnode;
	
	Tab<TimeValue>  timesX, timesY, timesZ, times;
	int             nTimesX, nTimesY, nTimesZ;
	Matrix3         tm;
	Interval        intervalForever = FOREVER;
	Control* cTM  = node->GetTMController();

	if (!cTM)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to retrieve Transform Controller","AnimLoad", MB_ICONSTOP|MB_OK);
		return;
	}

	Control* cRot = cTM->GetRotationController();

	if (!cRot)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to retrieve Rotation Controller","AnimLoad", MB_ICONSTOP|MB_OK);
		return;
	}

	// [We're going to rely on ROTX, ROTY, ROTZ and the chain having uniform samples]
	// This assumption is causing problems.  We need to evaluate all the controllers up to an
	// end point and combine their times.  We'll then add the samples at all the combined times
	// to the transform file by requesting the rotation controller to evaluate those times.

	Control* cRotX = cRot->GetXController();
	Control* cRotY = cRot->GetYController();
	Control* cRotZ = cRot->GetZController();

	curnode = node;

	CStr filename = GetSave();

	if (filename.Length() == 0)
		return;

	fp = fopen(filename, "wb");

	if (!fp)
	{
		char sErr[256];
		sprintf(sErr, "Failed to open transform file '%s' for writting.", (char*)filename);
		MessageBox(gInterface->GetMAXHWnd(), sErr, "AnimSave", MB_ICONWARNING|MB_OK);
		return;
	}

	// Dump keys
	BuildTimesList(timesX, node, CONTTYPE_AXISX);
	BuildTimesList(timesY, node, CONTTYPE_AXISY);
	BuildTimesList(timesZ, node, CONTTYPE_AXISZ);
	//BuildTimesList(times, node, CONTTYPE_AXISX);
	//BuildTimesList(times, node, CONTTYPE_AXISY);
	//BuildTimesList(times, node, CONTTYPE_AXISZ);

	//BuildTimes(timesX, node, CONTTYPE_AXISX);
	//BuildTimes(timesY, node, CONTTYPE_AXISY);
	//BuildTimes(timesZ, node, CONTTYPE_AXISZ);

	// After the time list has been built up we'll need to sort it
	//times.Sort(TimeCompare);
	timesX.Sort(TimeCompare);
	timesY.Sort(TimeCompare);
	timesZ.Sort(TimeCompare);

	for(int j = 0; j < times.Count(); j++)
	{
		int time = times[j] / GetTicksPerFrame();
		char buf[256];
		sprintf(buf, "Time: %i\n", time);
		OutputDebugString(buf);
	}

	//nTimes  = times.Count();
	nTimesX = timesX.Count();
	nTimesY = timesY.Count();
	nTimesZ = timesZ.Count();

	int i;

	/*
	for(i = 0; i < nTimes; i++)
	{
		float    valX, valY, valZ;
		Interval intervalForever = FOREVER;

		fwrite(&times[i], sizeof(TimeValue), 1, fp);

		cRotX->GetValue(times[i], &valX, intervalForever, CTRL_ABSOLUTE);
		cRotY->GetValue(times[i], &valY, intervalForever, CTRL_ABSOLUTE);
		cRotZ->GetValue(times[i], &valZ, intervalForever, CTRL_ABSOLUTE);

		fwrite(&valX, sizeof(float), 1, fp);
		fwrite(&valY, sizeof(float), 1, fp);
		fwrite(&valZ, sizeof(float), 1, fp);
	}
	*/

	fwrite(&nTimesX, sizeof(int), 1, fp);

	for(i = 0; i < nTimesX; i++)
	{
		float val;
		Interval intervalForever = FOREVER;
		
		fwrite(&timesX[i], sizeof(TimeValue), 1, fp);
		cRotX->GetValue(timesX[i], &val, intervalForever, CTRL_ABSOLUTE);

		fwrite(&val, sizeof(float), 1, fp);
	}

	fwrite(&nTimesY, sizeof(int), 1, fp);

	for(i = 0; i < nTimesY; i++)
	{
		fwrite(&timesY[i], sizeof(TimeValue), 1, fp);

		float val;
		Interval intervalForever = FOREVER;
		cRotY->GetValue(timesY[i], &val, intervalForever, CTRL_ABSOLUTE);

		fwrite(&val, sizeof(float), 1, fp);
	}

	fwrite(&nTimesZ, sizeof(int), 1, fp);

	for(i = 0; i < nTimesZ; i++)
	{
		fwrite(&timesZ[i], sizeof(TimeValue), 1, fp);

		float val;
		Interval intervalForever = FOREVER;
		cRotZ->GetValue(timesZ[i], &val, intervalForever, CTRL_ABSOLUTE);

		fwrite(&val, sizeof(float), 1, fp);
	}

	fclose(fp);
}

void AnimLoad()
{ FUNC_ENTER("AnimLoad"); 
	INode* nodeApply = nsAppPoint->GetNode();

	if (!nodeApply)
	{
		MessageBox(gInterface->GetMAXHWnd(), "You must select a node to apply the transform to", "AnimLoad", MB_ICONWARNING|MB_OK);
		return;
	}

	CStr filename = GetLoad();

	FILE* fp = fopen(filename, "rb");

	if (!fp)
	{
		char sErr[256];
		sprintf(sErr,"Failed to open transform file '%s' for reading", (char*)filename);
		MessageBox(gInterface->GetMAXHWnd(), sErr, "AnimLoad", MB_ICONWARNING|MB_OK);
		return;
	}

	int nTimesX, nTimesY, nTimesZ;
	int nTimes;
	TimeValue curtime;
	float     val;
	//float     valX, valY, valZ;

	Control* cTM   = nodeApply->GetTMController();
	Control* cRot  = cTM->GetRotationController();

/*
	Control* cRotX = (Control*)cTM->SubAnim(SUBANIM_ROTX);
	Control* cRotY = (Control*)cTM->SubAnim(SUBANIM_ROTY);
	Control* cRotZ = (Control*)cTM->SubAnim(SUBANIM_ROTZ);
*/

	Control* cRotX = cRot->GetXController();
	Control* cRotY = cRot->GetYController();
	Control* cRotZ = cRot->GetZController();

	int i;

	fread(&nTimes, sizeof(int), 1, fp);

	/*
	SuspendAnimate();
	AnimateOn();

	for(i = 0; i < nTimes; i++)
	{
		fread(&curtime, sizeof(TimeValue), 1, fp);
		fread(&valX, sizeof(float), 1, fp);
		fread(&valY, sizeof(float), 1, fp);
		fread(&valZ, sizeof(float), 1, fp);

		cRotX->SetValue(curtime, &valX, 1, CTRL_RELATIVE);
		cRotY->SetValue(curtime, &valY, 1, CTRL_RELATIVE);
		cRotZ->SetValue(curtime, &valZ, 1, CTRL_RELATIVE);
	}

	ResumeAnimate();
	*/

	fread(&nTimesX, sizeof(int), 1, fp);

	for(i = 0; i < nTimesX; i++)
	{
		fread(&curtime, sizeof(TimeValue), 1, fp);
		fread(&val, sizeof(float), 1, fp);

		SuspendAnimate();
		AnimateOn();
		cRotX->SetValue(curtime, &val, 1, CTRL_RELATIVE);

		//if (i % 2)
		//	AnimateOff();

		ResumeAnimate();
	}

	fread(&nTimesY, sizeof(int), 1, fp);

	int nTicks = GetTicksPerFrame();

	for(i = 0; i < nTimesY; i++)
	{
		fread(&curtime, sizeof(TimeValue), 1, fp);
		fread(&val, sizeof(float), 1, fp);

		SuspendAnimate();
		AnimateOn();
		cRotY->SetValue(curtime, &val, 1, CTRL_RELATIVE);

		//if (i % 2)
		//	AnimateOff();
		
		ResumeAnimate();
	}

	fread(&nTimesZ, sizeof(int), 1, fp);

	for(i = 0; i < nTimesZ; i++)
	{
		fread(&curtime, sizeof(TimeValue), 1, fp);

		//if (curtime >= 1533 * 160 && curtime <= 1001 * 160)
		if (curtime >= 1530 * 160 && curtime <= 1540 * 160)
		{
			int zzz;
			zzz = 0;
		}

		fread(&val, sizeof(float), 1, fp);

		SuspendAnimate();
		AnimateOn();
		cRotZ->SetValue(curtime, &val, 1, CTRL_RELATIVE);

		//if (i % 2)
		//	AnimateOff();
		
		ResumeAnimate();
	}

	fclose(fp);
}

BOOL CALLBACK DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("DlgProc"); 
	switch(msg)
	{
	case WM_COMMAND:
		{
			switch(LOWORD(wParam))
			{
			case IDC_ANIMSAVE:
				AnimSave();				
				return TRUE;

			case IDC_ANIMLOAD:
				AnimLoad();
				return TRUE;

			case IDC_ADDNODE:
				AddAnimNode();
				return TRUE;

			case IDC_REMOVENODE:
				RemoveAnimNode();
				return TRUE;
			}
		}
	}

	return FALSE;
}

void AnimStore::BeginEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("AnimStore::BeginEditParams"); 
	this->ip = ip;

	REGISTERCLASS(NodeSelect, hInstance);

	nsPreserve = new NodeSelect(hInstance);
	nsAppPoint = new NodeSelect(hInstance);

	hwndAnimStore = ip->AddRollupPage(hInstance, MAKEINTRESOURCE(IDD_ANIMSTORE), DlgProc, "Animation Store Util",(LPARAM)this);

	nsPreserve->Attach(GetDlgItem(hwndAnimStore, IDC_NSPRESERVENODE));
	nsAppPoint->Attach(GetDlgItem(hwndAnimStore, IDC_NSAPPPOINT));
}

void AnimStore::EndEditParams(Interface *ip,IUtil *iu)
{ FUNC_ENTER("AnimStore::EndEditParams"); 
	ip->DeleteRollupPage(hwndAnimStore);

	if (nsPreserve)
	{
		delete nsPreserve;
		nsPreserve = NULL;
	}

	if (nsAppPoint)
	{
		delete nsAppPoint;
		nsAppPoint = NULL;
	}
}

void AnimStore::SelectionSetChanged(Interface *ip,IUtil *iu)
{ FUNC_ENTER("AnimStore::SelectionSetChanged"); 

}

