/*
	PropList.h
	Property List Window Class
	
	Adam Lippmann
	adam@neversoft.com
*/

#ifndef __PROPLIST__
#define __PROPLIST__

//#include "IntDynTab.h"
#include "../misc/llist.h"
#include "ToolTip.h"
#include "max.h"
#include "strclass.h"

class SliderNum;
class SpinEdit;
class ScriptSelect;
class NodeSelect;
class MultiList;
class FlagSelect;
class ColorPopup;

enum PropColor
{
	PROPCOLOR_GRAY,
	PROPCOLOR_RED,
	PROPCOLOR_PURPLE,
	PROPCOLOR_GREEN,
	PROPCOLOR_BLUE,
	PROPCOLOR_WHITE,
};

enum PropType
{
	PROPTYPE_UNDEFINED,
	PROPTYPE_EDIT,
	PROPTYPE_LIST,
	PROPTYPE_EXTLIST,
	PROPTYPE_SLIDER,
	PROPTYPE_SCRIPT,
	PROPTYPE_FILE,
	PROPTYPE_COLOR,
	PROPTYPE_CHECK,
	PROPTYPE_SPINEDIT,
	PROPTYPE_STATIC,
	PROPTYPE_SLIDERNUM,
	PROPTYPE_NODE,
	PROPTYPE_MULTI,
	PROPTYPE_FLAGS,
	PROPTYPE_MTL,
	PROPTYPE_COLORPOPUP,
};

struct ListData
{
	LinkList<CStr>		    list;	// Selection list
	int                     curSel;	// Current selection in the list

	ListData() { curSel=0; }
};

struct SliderData
{
	int min;
	int max;
};

struct FileData
{
	CStr               FileName;	// Name of currently selected file
	char               Fltr[512];	// Filter string (max 512 bytes)
	bool               bSelected;	// True if a file has been selected

	FileData()
	{
		bSelected=false;
	}
};

struct ColorData
{
	IColorSwatch*      IColor;
	COLORREF           DefColor;
	CStr               DlgText;

	ColorData()
	{
		IColor   = NULL;
		DefColor = 0;
	}

	ColorData(ColorData& right)
	{
		IColor   = NULL;
		DefColor = right.DefColor;
		DlgText  = right.DlgText;
	}

	ColorData& operator = (ColorData& right)
	{
		IColor   = NULL;
		DefColor = right.DefColor;
		DlgText  = right.DlgText;

		return *this;
	}
};

struct Property
{
	HWND hwndTxt;				// Window containing property name text
	HWND hwndFld;				// Window containing property modification interface (Field)

	CStr strPropName;			// Name of the property
	CStr strDispName;			// The name that is actually displayed for this property
								// (if 0 length, displays actual property name)

	CStr strToolTip;			// Tool tip info for the property
	CStr strExtData;			// External data set by the user

	void    *pdata;				// Data specific to the specified property
								// NOTE: These should ALWAYS be allocated with new
	PropType  ptype;
	PropColor color;

	bool      bWasUpdated;		// True if this property was updated
	DWORD     flags;			// User flags
	DWORD     userData;
	bool      bVisible;			// True if the property should be displayed currently
								// Used for multi-prop lists

	Property()
	{ 
		flags = 0;
		pdata = NULL; 
		ptype = PROPTYPE_UNDEFINED;
		color = PROPCOLOR_GRAY;
		bWasUpdated = false;
		bVisible = true;
		strToolTip = CStr("No Tooltip help defined.\nGo Define Some!"); 
	}

	~Property()
	{ 
		// Color swatches don't follow the normal MAX setup and their
		// interface must be released at termination
		ColorData* cdata;

		if (pdata && ptype==PROPTYPE_COLOR)
		{
			cdata=(ColorData*)pdata;
			
			if (cdata->IColor)
				ReleaseIColorSwatch(cdata->IColor);
		}

		if (pdata) 
			delete pdata; 
	}

	bool operator== (Property& right)
	{
		if (strPropName == right.strPropName)
			return true;

		return false;
	}
};

struct PropVal
{
	int  index;
	CStr value;
};

class PropList
{
	HINSTANCE                   hInstance;					// The application's hInstance
	
	HWND                        hwndParent;					// The parent of the property list window
	HWND                        hwndApply;					// The apply button at the bottom of the property list
	HWND                        hwndScroll;					// The vertical scrollbar window

	HBRUSH                      hBGBrush;					// Background color brush used to paint the
															// child property name windows

	HBRUSH                      hBGBrushR;
	HBRUSH                      hBGBrushG;
	HBRUSH                      hBGBrushB;
	HBRUSH                      hBGBrushP;
	HBRUSH						hBGBrushW;

	WNDCLASS                    wndclass;					// The class of the property list popup window

	RECT rectRefTXT;										// Reference Property Text window dimensions
	RECT rectRefFLD;										// Reference Property Field window dimensions

	int TXTWidth,TXTHeight;									// Width/Height of Reference Text window
	int FLDWidth,FLDHeight;									// Width/Height of Reference Field window
	int FLDWidthS;											// Field width with scrollbar taken into account

	int width,height;										// Width and height of the window
	int origWidth;											// Original width (no scrollbar)

	ToolTip ttip;											// Tooltip manager

	CStr DlgText;											// Popup dialog text

	int                        numCols;						// Number of columns
	int                        x,y;							// X,Y coordinate of window (have to reposition due
															// to ATI's MultiMonitor stuff.  Argh!)

	bool                       bHasSliders;					// True if the property list contains slider controls
	bool                       bAttachedWindow;				// True if property list window was embedded on another window
	bool                       bFloatingWindow;				// True if floating property window
	bool                       bHasApply;					// True if property list has an Apply button
	bool                       bSetValLocked;				// True if SetValue cannot be called

	int                        lastModID;					// The ID of the last property modified
	int                        curIndex;					// The current index of items to be added

	DWORD                      userData;					// User defined value stored with the property list

	Link<Property>*            linkAddPos;					// The position that new properties will be added to
															// (Usually the end of the list)

	void(*fpChange)(PropList*,void*);						// Callback function called when the property list is updated
	void* pChangeData;										// Callback retained data

	void(*fpApply)(PropList*,void*);						// Callback function to call when apply is pressed
	void* pApplyData;										// Callback retained data

	void(*fpExtList)(CStr,CStr,void*);						// Callback function called when a list value is added
	void* pExtListData;										// Callback retained data

	void(*fpDblClick)(PropList*,int,void*);					// Callback function for property text double click
	void* pDblClickData;									// Callback retained data

	void(*fpFocus)(PropList*,int,void*);					// Callback function for property focus acquisition
	void *pFocusData;										// Callback retained data

															// Userdefined WndProc intercept for prop text child window
	LRESULT(*fpSProc)(PropList*,HWND,UINT,WPARAM,LPARAM,void*);
	void* pSProcData;										// Callback retained data

	WNDPROC OldStaticProc;

	HWND BuildStaticUI(int index);							// Construct User Static control interface
	HWND BuildEditUI(int index);							// Construct User Edit control interface
	HWND BuildSpinEditUI(int index);						// Construct User Spin Edit control interface
	HWND BuildListUI(int index,ListData *data);				// Construct User List control interface
	HWND BuildExtListUI(int index,ListData *data);			// Construct User Extensible List control interface
	HWND BuildSliderUI(int index,SliderData *data);			// Construct User Slider control interface
	HWND BuildFileUI(int index);							// Construct User File control interface
	HWND BuildColorUI(int index,ColorData *data);			// Construct User Color control interface
	HWND BuildCheckUI(int index);							// Construct User check control interface
	HWND BuildScriptUI(int index);							// Construct User script control interface
	HWND BuildSliderNumUI(int index);						// Construct User slider number control interface
	HWND BuildNodeUI(int index);							// Construct User node control interface
	HWND BuildMultiUI(int index);							// Construct User multi control interface
	HWND BuildFlagsUI(int index);							// Construct User flag select control interface
	HWND BuildMtlUI(int index);								// Construct User Material select control interface
	HWND BuildColorPopup(int index);						// Construct User ColorPopup control interface

	// Subclassed function for static property name labels
	static LRESULT CALLBACK StaticProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	// Redirects windows messages so they can be aware of the specific class instance that relates to them
	static LRESULT CALLBACK RedirectWndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	void OpenFileDlg(HWND hwnd,CStr* File,CStr* FileTitle,char* Title,char* Filter);

	void ProcExtLists();									// Processes extensible lists and calls the callback func
															// if a new value has been added

	void UpdateUI(int scrollpos);							// Called on scroll position change
	int  UpdateScroll(HWND scrollBar,int code,int pos);		// Update scroll bar
	BOOL ProcColor(HWND hwndCtl);

	int  GetXFLDPos(int index);								// Positioning functions that take into account columns
	int  GetXTXTPos(int index);
	int  GetYFLDPos(int index);
	int  GetYTXTPos(int index);

	void RestoreInternalValues();							// Restores calls to SetProperty before UI was built
	void SetScrollRange();

	int FindProp(Link<Property>* prop);

protected:
	HWND                        hwnd;						// The Property list window
	
	LinkList<Property>          props;						// The property information base
	LinkList<CStr>              savedValues;				// Saved values
	LinkList<PropVal>           propVals;					// Property values added prior to UI being built

	int  GetVisiblePropCount();

	// Class specific DlgProc
	virtual LRESULT WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

public:
	PropList(HINSTANCE hInstance,HWND hwdParent=0,int x=0,int y=0,int width=300,int height=263,CStr DlgText="");
	PropList(HINSTANCE hInstance,int cols);
	~PropList();

	PropList& operator = (PropList& right);

	void       Attach(HWND hwndThis);

	void       Clear();										// Clears contents
	void       DeleteProp(int index,bool bDelSaved=true);

	void          AddStatic(CStr propName,CStr ToolTip="");
	void          AddEdit(CStr propName,CStr ToolTip="");
	SpinEdit*     AddSpinEdit(CStr propName,float min,float max,float incr=1.0f,CStr ToolTip="");
	void          AddSlider(CStr propName,int min,int max,CStr ToolTip="");
	ListData*     AddList(CStr propName,CStr ToolTip="");
	ListData*     AddExtList(CStr propName,CStr ToolTip="");
	FileData*     AddFile(CStr propName,CStr ToolTip="");
	ColorData*    AddColor(CStr propName,CStr ToolTip="",COLORREF Color=0,CStr DlgText="");
	void          AddCheck(CStr propName,CStr ToolTip="");
	ScriptSelect* AddScript(CStr propName,CStr ToolTip="");
	SliderNum*    AddSliderNum(CStr propName,int min=0,int max=100,CStr ToolTip="");
	NodeSelect*   AddNode(CStr propName,CStr ToolTip="");
	MultiList*    AddMulti(CStr propName,CStr ToolTip="");
	FlagSelect*   AddFlags(CStr propName,CStr ToolTip="");
	ColorPopup*   AddColorPopup(CStr propName,int min,int max,CStr ToolTip="");
	void          AddMtl(CStr propName,CStr ToolTip="");
	void          AddProperty(Property* prop);

	int        GetCurIndex();
	int        ResetIndexToEnd();
	inline int NumProps()    { return props.GetSize(); }

	PropType   GetType(int index);
	bool       GetValue(int index,CStr& propVal);
	bool       GetValue(CStr name,CStr& propVal);
	CStr       GetName(int index);
	void       SetName(int index,CStr name);
	int        GetPropFromHWnd(HWND hwnd);

	bool       SetColor(int index,PropColor color);
	PropColor  GetColor(int index);
	bool       SetValue(int index,CStr strValue);
	bool       SetValue(CStr strName,CStr strValue);
	void       SetFlags(int index,int flags);
	DWORD      GetFlags(int index);
	CStr       GetDesc(int index);
	void       SetDesc(int index,CStr str);
	
	void       SetUserData(int index,CStr str);
	CStr       GetUserData(int index);
	void       SetUserVal(int index,DWORD val);
	DWORD      GetUserVal(int index);


	bool       WasUpdated(int index);
	bool       WasUpdated();
	void       ResetUpdateStatus();
	void       SetMod(int index,bool bVal);

	void       BuildUI();							// Construct the user interface child windows
	void       DestroyUI();							// Deletes UI contents for updating
	void       SaveValues();
	void       RestoreValues();

	void	   GetValues(LinkList<CStr>* list);		// Adds all values in the proplist to the given link list
	void       SetValues(LinkList<CStr>* list);		// Sets proplist values from a linklist
	void       ClearValues();						// Sets all UI controls to value ""

	void       SetApplyCB(void(*Func)(PropList*,void*),void* pData);
	void       SetChangeCB(void(*Func)(PropList*,void*),void* pData);
	void       SetExtListCB(void(*Func)(CStr propName,CStr propValue,void*),void* pData);
	void       SetDblClickCB(void(*Func)(PropList*,int,void*),void* pData);
	void       SetStaticCB(LRESULT(*Func)(PropList*,HWND,UINT,WPARAM,LPARAM,void*),void* pData);
	void       SetFocusCB(void(*Func)(PropList*,int,void*),void* pData);

	void       HasApply(bool bValue);

	int         GetReqHeight(int numProps);				// Computes the required height to see the
														// specified number of properties

	inline int  GetLastMod() { return lastModID; }

	inline void Show() { ShowWindow(hwnd,SW_SHOW); }
	inline void Hide() { ttip.Close(); ShowWindow(hwnd,SW_HIDE); }

	inline bool IsVisible() { return IsWindowVisible(hwnd) ? true : false; }

	inline bool HasSliders() { return bHasSliders; }
	inline void CloseTips()  { ttip.Close(); }

	void SetPos(int x,int y);
	void SetAddPos(int i);
	void RemoveProp(int id);

	void SetScrollVal(int val);
	int  GetScrollVal();

	bool PropExists(CStr name);

	inline void  SetUserData(DWORD val) { userData = val;  }
	inline DWORD GetUserData()          { return userData; }

	// Gets the position of the property list window
	int         GetX();
	int         GetY();
	inline void SetX(int x) { this->x=x; }
	inline void SetY(int y) { this->y=y; }
	inline void SetWidth(int width)   { this->width=width; }
	inline void SetHeight(int height) { this->height=height; }
	inline HWND GetHWnd()   { return hwnd; }
};


#endif
