#include <next.h>
#include <image/quant.h>


#define	RED	1
#define	GREEN	2
#define BLUE	3
#define ALPHA	4

#define MAXDEPTH 5
#define MAXCOLOR 256

#define NxColorRealLengthSq(a)	((a)->red*(a)->red + (a)->green*(a)->green + (a)->blue*(a)->blue + (a)->alpha*(a)->alpha)
#define	NxColorRealFromNxColor(o,i)										\
{                                                                        \
    (o)->red   =                                                         \
        (((float)(((i)->red))) * (   (float)((1.0/255.0))));           \
    (o)->green =                                                         \
        (((float)(((i)->green))) * ( (float)((1.0/255.0))));           \
    (o)->blue  =                                                         \
        (((float)(((i)->blue))) * (  (float)((1.0/255.0))));           \
    (o)->alpha =                                                         \
        (((float)(((i)->alpha))) * ( (float)((1.0/255.0))));           \
}                                                                        \

#define NxColorRealScale(o,a,scale)										\
{                                                                        \
    (o)->red   = (((a)->red) * (   scale));                              \
    (o)->green = (((a)->green) * ( scale));                              \
    (o)->blue  = (((a)->blue) * (  scale));                              \
    (o)->alpha = (((a)->alpha) * ( scale));                              \
}                                                                        \

#define NxColorRealAdd(o,a,b)                                        \
{                                                                        \
    (o)->red   = (((a)->red) + (   (b)->red));                           \
    (o)->green = (((a)->green) + ( (b)->green));                         \
    (o)->blue  = (((a)->blue) + (  (b)->blue));                          \
    (o)->alpha = (((a)->alpha) + ( (b)->alpha));                         \
}                                                                        \

#define NxColorRealSub(o,a,b)                                        \
{                                                                        \
    (o)->red   = (((a)->red) - (   (b)->red));                           \
    (o)->green = (((a)->green) - ( (b)->green));                         \
    (o)->blue  = (((a)->blue) - (  (b)->blue));                          \
    (o)->alpha = (((a)->alpha) - ( (b)->alpha));                         \
}                                                                        \

typedef unsigned long OctantMap;

/* local */
static OctantMap    splice[256];

/*************************************************************************/
static LeafNode    *
InitLeaf(LeafNode * Leaf)
{
    assert( Leaf );

    Leaf->palIndex = -1;
    Leaf->weight = 0.0f;
    Leaf->ac.red = 0.0f;
    Leaf->ac.green = 0.0f;
    Leaf->ac.blue = 0.0f;
    Leaf->ac.alpha = 0.0f;
    Leaf->m2 = 0.0f;
    
	return Leaf;
}

/*************************************************************************/
static BranchNode  *
InitBranch(BranchNode * Branch)
{
    int                 i;

    assert( Branch );

    for (i = 0; i < 16; i++)
    {
        Branch->dir[i] = (OctNode *)NULL;
    }

    return Branch;
}

/*************************************************************************/
static OctNode     *
CreateCube( void )//RwFreeList * fl)
{
    OctNode            *cube;

    //assert(fl);

    cube = new OctNode;//(OctNode *) RwFreeListAlloc(fl);
    return(cube);
}

/*************************************************************************/
static              OctantMap
GetOctAdr(NxRgbaColor * c)
{
    int                 cs = 8 - MAXDEPTH;

    assert(c);

    return((splice[c->red >> cs] << 3) | (splice[c->green >> cs] << 2) |
             (splice[c->blue >> cs] << 1) | (splice[c->alpha >> cs] << 0));
}

/*************************************************************************/
static OctNode     *
AllocateToLeaf(NxPalQuant * pq, OctNode * root, OctantMap Octs, int depth)
{

    assert(pq);
    assert(root);

    /* return leaf */
    if (depth == 0)
    {
        return(root);
    }

    /* populate branch */
    if (!root->Branch.dir[Octs & 15])
    {
        OctNode            *node;

        node = CreateCube();//(pq->cubefreelist);
        root->Branch.dir[Octs & 15] = node;
        if (depth == 1)
        {
            InitLeaf(&node->Leaf);
        }
        else
        {
            InitBranch(&node->Branch);
        }
    }

    return(AllocateToLeaf
             (pq, root->Branch.dir[Octs & 15], Octs >> 4, depth - 1));
}

/*************************************************************************/
void
NxPalQuantAddImage(NxPalQuant * pq, NxTexture * texture, float weight)
{
    int             ColShift;
    int             width, height, stride;
    unsigned char		*pixels;
    unsigned char       *palette;

    assert(pq);
    assert(texture);

    ColShift = 8 - MAXDEPTH;

    pixels = (unsigned char*) texture->GetTexelData( 0 );
    palette = (unsigned char*) texture->GetPaletteData();
    height = texture->GetHeight( 0 );

    switch (texture->GetBpp())
    {
        case 4:
		{
			stride = texture->GetBytesPerRow( 0 );
			while (height--)
            {
                unsigned char            *linePixels = pixels;

                width = texture->GetWidth( 0 );
                while (width--)
                {
                    //NxRgbaColor             *color = &palette[*linePixels];
					NxRgbaColor             color;
                    NxColorReal          rColor;
                    OctNode            *leaf;
                    OctantMap           Octs;
					unsigned char byte, texel;


					byte = *linePixels;
					if( width & 1 )
					{
						texel = byte & 0xF;
					}
					else
					{
						texel = ( byte & 0xF0 ) >> 4;
					}
					switch( texture->GetPaletteBpp())
					{
						case 24:
						{
							NxRgbColor* entry = (NxRgbColor *) &palette[ texel * 3 ];
							color.red = entry->red;
							color.green = entry->green;
							color.blue = entry->blue;
							color.alpha = 255;
							break;
						}
						case 32:
						{
							NxRgbaColor* entry = (NxRgbaColor*) &palette[ texel * 4 ];
							
							color = *entry;
							break;
						}
						default:
						{
							assert( 0 );
							break;
						}
					}
					

                    /* build down to leaf */
                    Octs = GetOctAdr(&color);
                    leaf = AllocateToLeaf(pq, pq->root, Octs, MAXDEPTH);

                    NxColorRealFromNxColor(&rColor, &color);
                    leaf->Leaf.weight += weight;
                    NxColorRealScale(&rColor, &rColor, weight);
                    NxColorRealAdd(&leaf->Leaf.ac, &leaf->Leaf.ac,
                                  &rColor);
                    leaf->Leaf.m2 += weight*NxColorRealLengthSq(&rColor);

					if( !( width % 2 ))
					{
						linePixels++;
					}
                }

                pixels += stride;
            }
            break;
		}
        case 8:
        {
			stride = texture->GetWidth( 0 );
            while (height--)
            {
                unsigned char            *linePixels = pixels;

                width = texture->GetWidth( 0 );
                while (width--)
                {
                    //NxRgbaColor             *color = &palette[*linePixels];
					NxRgbaColor             color;
                    NxColorReal          rColor;
                    OctNode            *leaf;
                    OctantMap           Octs;

					switch( texture->GetPaletteBpp())
					{
						case 24:
						{
							NxRgbColor* entry = (NxRgbColor *) &palette[ *linePixels * 3 ];
							color.red = entry->red;
							color.green = entry->green;
							color.blue = entry->blue;
							color.alpha = 255;
							break;
						}
						case 32:
						{
							NxRgbaColor* entry = (NxRgbaColor*) &palette[ *linePixels * 4 ];
							
							color = *entry;
							break;
						}
						default:
						{
							assert( 0 );
							break;
						}
					}
					

                    /* build down to leaf */
                    Octs = GetOctAdr(&color);
                    leaf = AllocateToLeaf(pq, pq->root, Octs, MAXDEPTH);

                    NxColorRealFromNxColor(&rColor, &color);
                    leaf->Leaf.weight += weight;
                    NxColorRealScale(&rColor, &rColor, weight);
                    NxColorRealAdd(&leaf->Leaf.ac, &leaf->Leaf.ac,
                                  &rColor);
                    leaf->Leaf.m2 += weight*NxColorRealLengthSq(&rColor);

                    linePixels++;
                }

                pixels += stride;
            }
            break;
        }

        default:
			break;
    }    
}

/*************************************************************************/

/*************************************************************************/
static void
assignindex(OctNode * root, NxRgbaColor * origin, int depth, box * region,
            int palIndex)
{
    int                 width, dr, dg, db, da, dR, dG, dB, dA;
    int                 i;

    assert(origin);
    assert(region);

    if (!root)
        return;

    width = 1 << depth;

    dr = origin->red - region->col1.red;
    dg = origin->green - region->col1.green;
    db = origin->blue - region->col1.blue;
    da = origin->alpha - region->col1.alpha;
    if (dr >= 0 || dg >= 0 || db >= 0 || da >= 0)
    {
        return;
    }

    dR = region->col0.red - origin->red;
    dG = region->col0.green - origin->green;
    dB = region->col0.blue - origin->blue;
    dA = region->col0.alpha - origin->alpha;
    if (dR >= width || dG >= width || dB >= width || dA >= width)
    {
        return;
    }

    /* wholly inside region and a leaf? */
    if (dr <= -width && dg <= -width && db <= -width && da <= -width)
        if (dR <= 0 && dG <= 0 && dB <= 0 && dA <= 0)
            if (depth == 0)
            {
                root->Leaf.palIndex = (unsigned char) palIndex;
                return;
            }

    /* try children */
    depth--;
    for (i = 0; i < 16; i++)
    {
        NxRgbaColor              suborigin;

        suborigin.red = origin->red + (((i >> 3) & 1) << depth);
        suborigin.green = origin->green + (((i >> 2) & 1) << depth);
        suborigin.blue = origin->blue + (((i >> 1) & 1) << depth);
        suborigin.alpha = origin->alpha + (((i >> 0) & 1) << depth);

        assignindex(root->Branch.dir[i], &suborigin, depth, region, palIndex);
    }    
}

/*************************************************************************/

/* Assign palIndex to leaves */
static void
nAssign(int palIndex, OctNode * root, box * cube)
{
    NxRgbaColor              origin;

    assert(root);
    assert(cube);

    origin.red = 0;
    origin.green = 0;
    origin.blue = 0;
    origin.alpha = 0;
    assignindex(root, &origin, MAXDEPTH, cube, palIndex);    
}

/*************************************************************************/
static void
addvolume(OctNode * root, NxRgbaColor * origin, int depth, box * region,
          LeafNode * volume)
{
    int                 width, dr, dg, db, da, dR, dG, dB, dA;
    int                 i;

    assert(origin);
    assert(region);

    if (!root)
        return;

    width = 1 << depth;

    dr = origin->red - region->col1.red;
    dg = origin->green - region->col1.green;
    db = origin->blue - region->col1.blue;
    da = origin->alpha - region->col1.alpha;
    if (dr >= 0 || dg >= 0 || db >= 0 || da >= 0)
    {
        return;
    }

    dR = region->col0.red - origin->red;
    dG = region->col0.green - origin->green;
    dB = region->col0.blue - origin->blue;
    dA = region->col0.alpha - origin->alpha;
    if (dR >= width || dG >= width || dB >= width || dA >= width)
    {
        return;
    }

    /* wholly inside region? */
    if (dr <= -width && dg <= -width && db <= -width && da <= -width)
        if (dR <= 0 && dG <= 0 && dB <= 0 && dA <= 0)
#ifndef CACHEWEIGHTS
            if (depth == 0)    /* we need to visit each leaf */
#endif
            {
                volume->weight += root->Leaf.weight;
                NxColorRealAdd(&volume->ac, &volume->ac, &root->Leaf.ac);
                volume->m2 += root->Leaf.m2;
                return;
            }

    /* try children */
    depth--;
    for (i = 0; i < 16; i++)
    {
        NxRgbaColor              suborigin;

        suborigin.red = origin->red + (((i >> 3) & 1) << depth);
        suborigin.green = origin->green + (((i >> 2) & 1) << depth);
        suborigin.blue = origin->blue + (((i >> 1) & 1) << depth);
        suborigin.alpha = origin->alpha + (((i >> 0) & 1) << depth);

        addvolume(root->Branch.dir[i], &suborigin, depth, region, volume);
    }    
}

/*************************************************************************/

/* Compute sum over a box of any given statistic */
static LeafNode    *
nVol(LeafNode * Vol, OctNode * root, box * cube)
{
    NxRgbaColor              origin;

    assert(root);
    assert(cube);

    origin.red = 0;
    origin.green = 0;
    origin.blue = 0;
    origin.alpha = 0;
    InitLeaf(Vol);
    addvolume(root, &origin, MAXDEPTH, cube, Vol);

    return(Vol);
}

/*************************************************************************/

/* Compute the weighted variance of a box */

/* NB: as with the raw statistics, this is really the variance * size */
static              float
nVar(OctNode * root, box * cube)
{
    LeafNode            Node;

    assert(root);
    assert(cube);

    nVol(&Node, root, cube);

    return(Node.m2 - (NxColorRealLengthSq(&Node.ac) / Node.weight));
}

/*************************************************************************/

/* We want to minimize the sum of the variances of two subboxes.
 * The sum(c^2) terms can be ignored since their sum over both subboxes
 * is the same (the sum for the whole box) no matter where we split.
 * The remaining terms have a minus sign in the variance formula,
 * so we drop the minus sign and MAXIMIZE the sum of the two terms.
 */
static              float
nMaximize(OctNode * root, box * cube, int dir, int * cut,
          LeafNode * whole)
{
    box                 infcube;
    LeafNode            left, right;
    float              maxsum, val, lastval;
    int             i;

    assert(root);
    assert(cube);
    assert(cut);
    assert(whole);

    lastval = maxsum = 0.0f;
    *cut = -1;
    infcube = *cube;
    switch (dir)
    {
        case RED:
            for (i = cube->col0.red; i < cube->col1.red; i++)
            {
                infcube.col1.red = (unsigned char) i;

                nVol(&left, root, &infcube);
                NxColorRealSub(&right.ac, &whole->ac, &left.ac);
                right.weight = whole->weight - left.weight;
                if ((left.weight > 0.0f) && (right.weight > 0.0f))
                {
                    val = NxColorRealLengthSq(&left.ac) / left.weight;
                    val += NxColorRealLengthSq(&right.ac) / right.weight;

                    if (val > maxsum)
                    {
                        maxsum = val;
                        *cut = i;
                    }
                    else if (val < lastval)
                    {
                        /* we've past the peak */
                        break;
                    }

                    lastval = val;
                }
            }
            break;

        case GREEN:
            for (i = cube->col0.green; i < cube->col1.green; i++)
            {
                infcube.col1.green = (unsigned char) i;

                nVol(&left, root, &infcube);
                NxColorRealSub(&right.ac, &whole->ac, &left.ac);
                right.weight = whole->weight - left.weight;
                if ((left.weight > 0.0f) && (right.weight > 0.0f))
                {
                    val = NxColorRealLengthSq(&left.ac) / left.weight;
                    val += NxColorRealLengthSq(&right.ac) / right.weight;

                    if (val > maxsum)
                    {
                        maxsum = val;
                        *cut = i;
                    }
                    else if (val < lastval)
                    {
                        /* we've past the peak */
                        break;
                    }

                    lastval = val;
                }
            }
            break;

        case BLUE:
            for (i = cube->col0.blue; i < cube->col1.blue; i++)
            {
                infcube.col1.blue = (unsigned char) i;

                nVol(&left, root, &infcube);
                NxColorRealSub(&right.ac, &whole->ac, &left.ac);
                right.weight = whole->weight - left.weight;
                if ((left.weight > 0.0f) && (right.weight > 0.0f))
                {
                    val = NxColorRealLengthSq(&left.ac) / left.weight;
                    val += NxColorRealLengthSq(&right.ac) / right.weight;

                    if (val > maxsum)
                    {
                        maxsum = val;
                        *cut = i;
                    }
                    else if (val < lastval)
                    {
                        /* we've past the peak */
                        break;
                    }

                    lastval = val;
                }
            }
            break;

        case ALPHA:
            for (i = cube->col0.alpha; i < cube->col1.alpha; i++)
            {
                infcube.col1.alpha = (unsigned char) i;

                nVol(&left, root, &infcube);
                NxColorRealSub(&right.ac, &whole->ac, &left.ac);
                right.weight = whole->weight - left.weight;
                if ((left.weight > 0.0f) && (right.weight > 0.0f))
                {
                    val = NxColorRealLengthSq(&left.ac) / left.weight;
                    val += NxColorRealLengthSq(&right.ac) / right.weight;

                    if (val > maxsum)
                    {
                        maxsum = val;
                        *cut = i;
                    }
                    else if (val < lastval)
                    {
                        /* we've past the peak */
                        break;
                    }

                    lastval = val;
                }
            }
            break;
    }

    return(maxsum);
}

/*************************************************************************/
static              bool
nCut(OctNode * root, box * set1, box * set2)
{
    int             cutr, cutg, cutb, cuta;
    float              maxr, maxg, maxb, maxa;
    LeafNode            whole;

    assert(root);
    assert(set1);
    assert(set2);

    nVol(&whole, root, set1);
    maxr = nMaximize(root, set1, RED, &cutr, &whole);
    maxg = nMaximize(root, set1, GREEN, &cutg, &whole);
    maxb = nMaximize(root, set1, BLUE, &cutb, &whole);
    maxa = nMaximize(root, set1, ALPHA, &cuta, &whole);

    /* did we find any splits? */
    if ( (maxr == 0.0f) &&
         (maxg == 0.0f) &&
         (maxb == 0.0f) &&
         (maxa == 0.0f) )
    {
        return(FALSE);
    }

    *set2 = *set1;

    /* NEED TO CHECK FOR ALPHA TOO */

    if (maxr >= maxg)
    {
        if (maxr >= maxb)
        {
            if (maxr >= maxa)
            {
                set1->col1.red = set2->col0.red = (unsigned char) cutr;
            }
            else
            {
                set1->col1.alpha = set2->col0.alpha = (unsigned char) cuta;
            }
        }
        else
        {
            if (maxb >= maxa)
            {
                set1->col1.blue = set2->col0.blue = (unsigned char) cutb;
            }
            else
            {
                set1->col1.alpha = set2->col0.alpha = (unsigned char) cuta;
            }
        }
    }
    else if (maxg >= maxb)
    {
        if (maxg >= maxa)
        {
            set1->col1.green = set2->col0.green = (unsigned char) cutg;
        }
        else
        {
            set1->col1.alpha = set2->col0.alpha = (unsigned char) cuta;
        }
    }
    else if (maxb >= maxa)
    {
        set1->col1.blue = set2->col0.blue = (unsigned char) cutb;
    }
    else
    {
        set1->col1.alpha = set2->col0.alpha = (unsigned char) cuta;
    }


    return(TRUE);
}

/*************************************************************************/
#ifdef CACHEWEIGHTS
static LeafNode    *
CalcNodeWeights(OctNode * root, int depth)
{
    LeafNode           *Leaf;
    int                 i;

    Leaf = NULL;
    if (root)
    {
        Leaf = &root->Leaf;

        /* is it a branch? */
        if (depth > 0)
        {
            InitLeaf(Leaf);
            for (i = 0; i < 16; i++)
            {
                LeafNode           *SubNode =
                    CalcNodeWeights(root->Branch.dir[i], depth - 1);

                if (SubNode)
                {
                    Leaf->weight += SubNode->weight;
                    NxColorRealAdd(&Leaf->ac, &Leaf->ac, &SubNode->ac);
                    Leaf->m2 += SubNode->m2;
                }
            }
        }
    }

    return(Leaf);
}
#endif

/*************************************************************************/
static int
CountLeafs(OctNode * root, int depth)
{
    int                 i, n;

    n = 0;
    if (root)
    {
        /* is it a branch? */
        if (depth > 0)
        {
            for (i = 0; i < 16; i++)
            {
                n += CountLeafs(root->Branch.dir[i], depth - 1);
            }
        }
        else
        {
            n = 1;
        }
    }

    return(n);
}

/*************************************************************************/
static int
CountNodes(OctNode * root, int depth)
{
    int                 i, n;

    n = 0;
    if (root)
    {
        n = 1;

        /* is it a branch? */
        if (depth > 0)
        {
            for (i = 0; i < 16; i++)
            {
                n += CountNodes(root->Branch.dir[i], depth - 1);
            }
        }
    }

    return(n);
}

/*************************************************************************/

/* 
 * Note the use of 255.9999f value when generating the scale variable.
 * This is used instead of 255.0f to prevent rounding errors.
 */

#define node2pal(rgb, node)                                              \
{                                                                        \
    int             quantize;                                        \
    float              scale = ( ((node)->weight > 0) ?                 \
                                  (255.9999f / (node)->weight) :         \
                                  (float) 0 );                          \
                                                                         \
    quantize = (int) ((node)->ac.red * scale);                       \
    (rgb)->red = (unsigned char) quantize;                                     \
    quantize = (int) ((node)->ac.green * scale);                     \
    (rgb)->green = (unsigned char) quantize;                                   \
    quantize = (int) ((node)->ac.blue * scale);                      \
    (rgb)->blue = (unsigned char) quantize;                                    \
    quantize = (int) ((node)->ac.alpha * scale);                     \
    (rgb)->alpha = (unsigned char) quantize;                                   \
}                                                                        \

/*************************************************************************/
int
NxPalQuantResolvePalette(NxRgbaColor * palette, int maxcols, NxPalQuant * pq)
{
    int             numcols, uniquecols, i, k, next;
    
#if (defined(RWDEBUG))
    if (256 < maxcols)
    {
        RWMESSAGE(("256 < %d == maxcols", (int) maxcols));
    }
#endif /* (defined(RWDEBUG)) */


    assert(palette);
    assert(maxcols <= 256);
    assert(pq);

    numcols = maxcols;
    uniquecols = CountLeafs(pq->root, MAXDEPTH);
    if (numcols > uniquecols)
    {
        numcols = uniquecols;
    }

#ifdef CACHEWEIGHTS
    /* cache weightings at every node */
    CalcNodeWeights(pq->root, MAXDEPTH);
#endif

    /* divide and conquer */
    pq->Mcube[0].col0.red = 0;
    pq->Mcube[0].col0.green = 0;
    pq->Mcube[0].col0.blue = 0;
    pq->Mcube[0].col0.alpha = 0;
    pq->Mcube[0].col1.red = 1 << MAXDEPTH;
    pq->Mcube[0].col1.green = 1 << MAXDEPTH;
    pq->Mcube[0].col1.blue = 1 << MAXDEPTH;
    pq->Mcube[0].col1.alpha = 1 << MAXDEPTH;
    pq->Mvv[0] = nVar(pq->root, &pq->Mcube[0]);
    for (i = 1; i < numcols; i++)
    {
        float               maxvar;

        /* find best box to split */
        next = -1;
        maxvar = 0.0f;
        for (k = 0; k < i; k++)
        {
            if (pq->Mvv[k] > maxvar)
            {
                maxvar = pq->Mvv[k];
                next = k;
            }
        }

        /* stop if we couldn't find a box to split */
        if (next == -1)
        {
            break;
        }

        /* split box */
        if (nCut(pq->root, &pq->Mcube[next], &pq->Mcube[i]))
        {
            /* volume test ensures we won't try to cut one-cell box */
            pq->Mvv[next] = nVar(pq->root, &pq->Mcube[next]);
            pq->Mvv[i] = nVar(pq->root, &pq->Mcube[i]);
        }
        else
        {
            /* don't try to split this box again */
            pq->Mvv[next] = 0.0f;
            i--;
        }
    }

    /* extract the new palette */
    for (k = 0; k < maxcols; k++)
    {
        if (k < numcols)
        {
            LeafNode            Node;

            nAssign(k, pq->root, &pq->Mcube[k]);
            nVol(&Node, pq->root, &pq->Mcube[k]);
            node2pal(&palette[k], &Node);
        }
        else
        {
            palette[k].red = 0;
            palette[k].green = 0;
            palette[k].blue = 0;
            palette[k].alpha = 0;
        }
    }

    return(numcols);
}

/*************************************************************************/
static              unsigned char
GetIndex(OctNode * root, OctantMap Octs, int depth)
{
    unsigned char             result;

    assert(root);

    if (depth == 0)
    {
        result = root->Leaf.palIndex;
    }
    else
    {
        result = GetIndex(root->Branch.dir[Octs & 15], Octs >> 4, depth - 1);
    }

    return(result);
}

void
NxPalQuantMatchImage(unsigned char * dstpixels, int dststride, int dstdepth,
                     NxPalQuant* pq, NxTexture* texture)
{
    unsigned int            width, x, height, stride, maxcol;
    unsigned char            *pixels, *dstLinePixels;
    unsigned char             index;
    OctantMap           Octs;

    assert(dstpixels);
    assert(pq);
    assert(texture);

    //stride = NxTextureGetStride(img);
    pixels = (unsigned char*) texture->GetTexelData( 0 );//NxTextureGetPixels(img);
    maxcol = 1 << dstdepth;

    /* store two pixels per byte if the destination is packed
     * (i.e is a raster not an image) */
	switch( texture->GetBpp())
	{
		case 4:
		{
            unsigned char* palette = (unsigned char*) texture->GetPaletteData();//NxTextureGetPalette(img);
            height = texture->GetHeight( 0 );//NxTextureGetHeight(img);
			stride = texture->GetBytesPerRow( 0 );
			while (height--)
            {
                unsigned char            *srcLinePixels = pixels;

                dstLinePixels = dstpixels;

                width = texture->GetWidth( 0 );//NxTextureGetWidth(img);
                for (x = 0; x < width; x++)
                {
					NxRgbaColor color;
					unsigned char byte, texel;
                    
					byte = *srcLinePixels;
					if( x & 1 )
					{
						texel = ( byte & 0xF0 ) >> 4;
					}
					else
					{
						texel = ( byte & 0x0F );
					}
					switch( texture->GetPaletteBpp())
					{
						case 24:
						{
							NxRgbColor* entry = (NxRgbColor *) &palette[ texel * 3 ];
							color.red = entry->red;
							color.green = entry->green;
							color.blue = entry->blue;
							color.alpha = 255;
							break;
						}
						case 32:
						{
							NxRgbaColor* entry = (NxRgbaColor*) &palette[ texel * 4 ];
							
							color = *entry;
							break;
						}
						default:
						{
							assert( 0 );
							break;
						}
					}
                    /* map to new index */
                    Octs = GetOctAdr(&color);
                    index = GetIndex(pq->root, Octs, MAXDEPTH);
                    assert(index < maxcol);

                    if ((x/*+raster->nOffsetX*/) & 1)
                    {
                       *dstLinePixels &= 0x0F;
                       *dstLinePixels |= (index & 0x0F) << 4;
                        dstLinePixels++;
                    }
                    else
                    {
                       *dstLinePixels &= 0xF0;
                       *dstLinePixels |= index & 0x0F;
                    }

					if( x & 1 )
					{
						srcLinePixels++;
					}
                }

                pixels += stride;
                dstpixels += dststride;
            }
            break;
        }        
		case 8:
        {
            unsigned char* palette = (unsigned char*) texture->GetPaletteData();//NxTextureGetPalette(img);
			stride = texture->GetWidth( 0 );

            height = texture->GetHeight( 0 );//NxTextureGetHeight(img);
            while (height--)
            {
                unsigned char            *srcLinePixels = pixels;

                dstLinePixels = dstpixels;

                width = texture->GetWidth( 0 );//NxTextureGetWidth(img);
                while (width--)
                {
					NxRgbaColor color;
                    
					switch( texture->GetPaletteBpp())
					{
						case 24:
						{
							NxRgbColor* entry = (NxRgbColor *) &palette[ *srcLinePixels++ * 3 ];
							color.red = entry->red;
							color.green = entry->green;
							color.blue = entry->blue;
							color.alpha = 255;
							break;
						}
						case 32:
						{
							NxRgbaColor* entry = (NxRgbaColor*) &palette[ *srcLinePixels++ * 4 ];
							
							color = *entry;
							break;
						}
						default:
						{
							assert( 0 );
							break;
						}
					}

                    /* map to new index */
                    Octs = GetOctAdr(&color);
                    index = GetIndex(pq->root, Octs, MAXDEPTH);
                    assert(index < maxcol);
                    *dstLinePixels++ = index;
                }

                pixels += stride;
                dstpixels += dststride;
            }
            break;
        }
        
		default:
			assert( 0 );        
    }    
}

/*************************************************************************/
bool
NxPalQuantInit(NxPalQuant * pq)
{
    int                 i, j, maxval;

    assert(pq);

    /* lookup mapping (8) bit-patterns to every 4th bit b31->b00 (least to most) */
    maxval = 1 << MAXDEPTH;
    for (i = 0; i < maxval; i++)
    {
        OctantMap           mask = 0;

        for (j = 0; j < MAXDEPTH; j++)
        {
            mask |= (i & (1 << j)) ? (1 << ((MAXDEPTH - 1 - j) * 4)) : 0;
        }
        splice[i] = mask;
    }

    pq->Mcube = (box *) calloc(sizeof(box), MAXCOLOR);
    pq->Mvv = (float *) calloc(sizeof(float), MAXCOLOR);

    //pq->cubefreelist = RwFreeListCreate(sizeof(OctNode), 64, 0);
    pq->root = CreateCube();//(pq->cubefreelist);
    InitBranch(&pq->root->Branch);

    return(TRUE);
}

/*************************************************************************/
static void
DeleteOctTree(NxPalQuant * pq, OctNode * root, int depth)
{
    int                 i;

    assert(pq);

    if (root)
    {
        /* is it a branch? */
        if (depth > 0)
        {
            for (i = 0; i < 16; i++)
            {
                DeleteOctTree(pq, root->Branch.dir[i], depth - 1);
            }
        }

        //RwFreeListFree(pq->cubefreelist, root);
    }
}

/*************************************************************************/
void
NxPalQuantTerm(NxPalQuant * pq)
{

    assert(pq);

    DeleteOctTree(pq, pq->root, MAXDEPTH);
    pq->root = (OctNode *)NULL;

    //RwFreeListDestroy(pq->cubefreelist);
    free( pq->Mvv );
    free( pq->Mcube );
}

bool	NxPalettizeMips( NxTexture** mips, int levels )
{
	int i, num_colors;
	unsigned char* pal_1, *pal_2;
	unsigned char* final_pal, *dst;
	bool needs_quant;
	NxPalQuant pal_quant;
	NxRgbaColor mip_pal[256];
	NxTexture* src_image;

	assert( levels <= NxTexture::vMAX_MIP_LEVELS );
	assert( mips && *mips );

	// First, make sure we even need to perform the quantization
	needs_quant = false;
	pal_1 = (unsigned char*) mips[0]->GetPaletteData();	
	for( i = 1; i < levels; i++ )
	{	
		pal_2 = (unsigned char*) mips[i]->GetPaletteData();
		if( memcmp( pal_1, pal_2, mips[0]->GetPaletteDataSize()))
		{
			needs_quant = true;
			break;
		}
	}
	
	// All mips involved already have the same palette. Don't do anything
	if( needs_quant == false )
	{		
		for( i = 1; i < levels; i++ )
		{
			mips[0]->SetWidth( i, mips[i]->GetWidth( 0 ));
			mips[0]->SetHeight( i, mips[i]->GetHeight( 0 ));
			mips[0]->SetTexelData( i, (char*) mips[i]->GetTexelData( 0 ));
		}
		return true;
	}

	if( !NxPalQuantInit( &pal_quant ))
	{
		return false;
	}
	
	// Add all of the images' pixels as input
	for( i = 0; i < levels; i++ )
	{
		NxPalQuantAddImage( &pal_quant, mips[i], (float)(1 << (i << 1)));
	}
	
	// Create a palette for all of the mips
	num_colors = NxPalQuantResolvePalette( mip_pal, 1 << mips[0]->GetBpp(), &pal_quant );
	
	// Fill in our new palette with the results
	final_pal = new unsigned char[ mips[0]->GetPaletteDataSize() ];
	dst = final_pal;
	for( i = 0; i < num_colors; i++ )
	{
		*dst++ = mip_pal[i].red;
		*dst++ = mip_pal[i].green;
		*dst++ = mip_pal[i].blue;
		if( mips[0]->GetPaletteBpp() == 32 )
		{
			*dst++ = mip_pal[i].alpha;
		}
	}

	// Match Images
	src_image = mips[0];	
	for( i = 0; i < levels; i++ )
	{
		NxTexture*	image = mips[i];
        unsigned char* new_texel_data;

		new_texel_data = new unsigned char[ image->GetTexelDataSize( 0 ) ];
        if( new_texel_data )
        {
            NxPalQuantMatchImage(	new_texel_data,
									image->GetBytesPerRow( 0 ),//RwImageGetStride(new_image),
									image->GetBpp(),//RwImageGetDepth(new_image),
									&pal_quant,
									image );
			
			src_image->SetWidth( i, image->GetWidth( 0 ));
			src_image->SetHeight( i, image->GetHeight( 0 ));
			src_image->SetTexelData( i, (char*) new_texel_data );
        }
        else
        {
            return false;
        }
	}

	src_image->SetPaletteData((char*) final_pal );
	NxPalQuantTerm( &pal_quant );

	return true;
}

