#include <Engine/Engine.h>
#include "Export/CamExporter.h"
#include <Sk/Gamenet/ExportMsg.h>
#include <process.h>
#include <Export/SceneExport.h>
#include <Export/TextureExporter.h>
#include <Export/ScriptExport.h>
#include <Export/Strip/Strip.h>
#include <Wibble/Wibble.h>
#include <FaceFlags/FaceFlags.h>
#include <fstream.h>
#include <stdmat.h>
#include <Image/Image.h>
#include <Material/NExtMat.h>
#include <Misc/GenCrc.h>
#include <Misc/Util.h>
#include "../UI/OktoAll.h"

#define vGAMEOBJ_CLASS_TOKEN		"GameObject"
#define vBOUNCYOBJ_CLASS_TOKEN		"BouncyObject"

#define PRINT_DEBUG_NODES

bool gbAbort;			// Global abort flag to force export to abort

class SceneExporter : public ISceneExporter
{
public:
	SceneExporter( void );

	void		EnumerateExportableNodes( void );
	NxScene*	ExtractSceneData( bool bLocalCoords=false );
	bool		SaveScene( void );
	bool		SaveGeometry( fstream& file );
	bool		SaveTextures( void );	
	bool		SpawnPostExportProcessor( void );

	// From Exporter
	bool		DoExport( ExportType type );
	bool		Save( void );
	bool		ExportAsSky( void );	

//private:

	void		enumerate_all_nodes( INode* root );
	void		enumerate_selected_nodes( void );
	void		enumerate_visible_nodes( INode* root );
	void		enumerate_selection_set_nodes( void );
	
	Tab <INode *>		m_exportable_nodes;	
	NxScene*			m_scene;
	SceneExportOptions	m_export_options;		
};

static SceneExporter	s_scene_exporter;

ISceneExporter*		GetSceneExporter( void )
{
	return &s_scene_exporter;
}

DWORD WINAPI SceneExportProgressFunc(LPVOID arg) 
{
    return(0);
}

void	SceneExporter::enumerate_all_nodes( INode* root )
{
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		if( IsExportable( child ))
		{
			m_exportable_nodes.Append( 1, &child );
		}
		// Recursively add all children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_all_nodes( child );
		}
	}
}

void	SceneExporter::enumerate_selected_nodes( void )
{
	int i;
	INode* sel_node;
	
	for( i = 0; i < gInterface->GetSelNodeCount(); i++ )
	{
		sel_node = gInterface->GetSelNode( i );
		if( IsExportable( sel_node ))
		{
			m_exportable_nodes.Append( 1, &sel_node );
		}
	}
}

void	SceneExporter::enumerate_visible_nodes( INode* root )
{
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		// Add only visible, non-frozen nodes
		if(	( child->IsHidden() == false ) &&
			( child->IsFrozen() == false ))
		{
			if( IsExportable( child ))
			{
				m_exportable_nodes.Append( 1, &child );			
			}
		}

		// Recursively add all visible children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_visible_nodes( child );
		}
	}
}

void	SceneExporter::enumerate_selection_set_nodes( void )
{
	int i, num_sel_sets;

	// If they haven't entered a set, just return
	if( m_export_options.m_ExportSet == TSTR( "" ))
	{
		return;
	}

	// Find the set that matches the name of the proposed export set
	num_sel_sets = gInterface->GetNumNamedSelSets();
	for( i = 0; i < num_sel_sets; i++ )
	{
		if( m_export_options.m_ExportSet == TSTR( gInterface->GetNamedSelSetName( i )))
		{
			int j, num_nodes;

			num_nodes = gInterface->GetNamedSelSetItemCount( i );
			for( j = 0; j < num_nodes; j++ )
			{
				INode* sel_node;

				sel_node = gInterface->GetNamedSelSetItem( i, j );

				if (IsExportable( sel_node ))
					m_exportable_nodes.Append( 1, &sel_node );
			}

			break;
		}
	}
}

void	SceneExporter::EnumerateExportableNodes( void )
{
	m_exportable_nodes.ZeroCount();
	
	if( m_export_options.m_ExportAll )
	{
		enumerate_all_nodes( gInterface->GetRootNode());
	}
	else if( m_export_options.m_ExportSelected )				
	{
		enumerate_selected_nodes();
	}
	else if( m_export_options.m_ExportVisibleOnly )
	{
		enumerate_visible_nodes( gInterface->GetRootNode());
	}
	else if( m_export_options.m_ExportSelectionSet )
	{
		enumerate_selection_set_nodes();
	}	
}

NxScene*	SceneExporter::ExtractSceneData( bool bLocalCoords )
{
	int i, j;
	INode* node;
	NxScene* scene;
	NxObject* object;
	int num_exportable_nodes;

	gbAbort=false;

	scene = new NxScene;	

	num_exportable_nodes = m_exportable_nodes.Count();
	
	gInterface->ProgressStart(_T("Enumerating exportable objects"), TRUE, 
							SceneExportProgressFunc, NULL );

#ifdef PRINT_DEBUG_NODES
	HWND HWNDmax = gInterface->GetMAXHWnd();
	char bufTitle[256];
	GetWindowText(HWNDmax,bufTitle,255);
#endif

	for( i = 0; i < num_exportable_nodes; i++ )
	{
		CStr class_name;
		CStr lclass_name;
		bool is_model;

		node = m_exportable_nodes[i];
		CStr nodeName = CStr("Enum Nodes Exporting: ") + node->GetName() + "\n";
		OutputDebugString(nodeName);

#ifdef PRINT_DEBUG_NODES
	SetWindowText(HWNDmax,nodeName);
#endif

		// Acquire class and type for highlighting
		class_name = "";
		if( !node->GetUserPropString( "Class", class_name ))
		{
			node->GetUserPropString( "class", class_name );
		}

		lclass_name = class_name;
		lclass_name.toLower();

		is_model = false;		

		OutputDebugString("Pre create object from node\n");

		if (bLocalCoords || lclass_name == CStr("levelobject"))
			object = scene->CreateObjectFromNode( node, 1.0f, true );
		else
			object = scene->CreateObjectFromNode( node );

		OutputDebugString("Post create object from node\n");

		if( object )
		{			
			if(	( stricmp( class_name, vGAMEOBJ_CLASS_TOKEN ) == 0 ) ||
				( stricmp( class_name, vBOUNCYOBJ_CLASS_TOKEN ) == 0 ))
			{		
				object->m_Flags |= NxObject::mDYNAMIC;
				is_model = true;
			}
		
			// Check (and warn) for duplicate object names
			for( j = 0; j < scene->m_Objects.Count(); j++ )
			{
				if(( stricmp( object->m_Name, scene->m_Objects[j]->m_Name )) == 0 )
				{
					char msg[1024];

					sprintf( msg, "Multiple objects are named %s. This will cause problems.", object->m_Name );
					MessageBoxAll( gInterface->GetMAXHWnd(), msg, "Multiple named objects!", MB_OK|MB_TASKMODAL);
				}
			}
			//object->Dump();
			//object->OptimizeVertList();
			// Print the object's contents for debug purposes
			//object->Dump();
			scene->m_Objects.Append( 1, &object );					
		}

		if( gInterface->GetCancel() || gbAbort )
		{
			gInterface->ProgressEnd();
			delete scene;
			return NULL;
		}
		gInterface->ProgressUpdate( i * 100 / num_exportable_nodes );
	}
	
	gInterface->ProgressEnd();	
	
#ifdef PRINT_DEBUG_NODES
	SetWindowText(HWNDmax,bufTitle);
#endif

	return scene;
}

bool	SceneExporter::SaveGeometry( fstream& file )
{
	int i, j, k, num_obj, num_faces, num_verts;

	num_obj = m_scene->m_Objects.Count();
	file.write((const char*) &num_obj, sizeof( int ));
	gInterface->ProgressStart(_T("Exporting Geometry"), TRUE, 
							SceneExportProgressFunc, NULL );
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxFace* face_list;
		NxObject* object;
		Point3 center;
		float radius;
		unsigned long checksum;

		object = m_scene->m_Objects[i];
		vert_list = object->m_Verts;
		face_list = object->m_Faces;
		num_faces = object->m_NumFaces;
		num_verts = object->m_NumVerts;

		// Write out the object's checksum		
		checksum = GenerateCRC( object->m_Name );
		file.write((const char*) &checksum, sizeof( unsigned long ));

		// Write out the flags which describe the object's properties
		object->m_Flags |= NxObject::mHASVERSIONINFO;
		file.write((const char*) &object->m_Flags, sizeof( int ));

		// Write out the number of sets of UVs
		file.write((const char*) &object->m_NumUVSets, sizeof( int ));

		// Write out bounding box/sphere data
		file.write((const char*) &object->m_BoundingBox.pmin, sizeof( Point3 ));
		file.write((const char*) &object->m_BoundingBox.pmax, sizeof( Point3 ));

		center = object->m_BoundingBox.Center();
		radius = (( object->m_BoundingBox.pmax - object->m_BoundingBox.pmin ).Length()) / 2.0f;
		file.write((const char*) &center, sizeof( Point3 ));
		file.write((const char*) &radius, sizeof( float ));

		file.write((const char*) &num_verts, sizeof( int ));
		for( j = 0; j < num_verts; j++ )
		{
			NxVertex* vert;			
			
			vert = &vert_list[j];

			// Write out first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					file.write((const char*) &vert->m_TexCoord[k].x, sizeof( float ));
					file.write((const char*) &vert->m_TexCoord[k].y, sizeof( float ));
				}
			}

			// Write out colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				file.write((const char*) &vert->m_Color.r, sizeof( float ));
				file.write((const char*) &vert->m_Color.g, sizeof( float ));
				file.write((const char*) &vert->m_Color.b, sizeof( float ));
				file.write((const char*) &vert->m_Color.a, sizeof( float ));
			}				

			// Write out normals
			if( object->m_Flags & NxObject::mNORMALS )
			{					
				file.write((const char*) &vert->m_Normal.x, sizeof( float ));
				file.write((const char*) &vert->m_Normal.y, sizeof( float ));
				file.write((const char*) &vert->m_Normal.z, sizeof( float ));
			}

			// Write out vc wibble data
			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				file.write((const char*) &vert->m_WibbleIndex, sizeof( char ));
				file.write((const char*) &vert->m_WibbleOffset, sizeof( char ));	
			}

			// Write out positions				
			/*
			vert->m_Pos.x = ((int) (( vert->m_Pos.x * 10 ) + 0.5f )) / 10.0f;
			vert->m_Pos.y = ((int) (( vert->m_Pos.y * 10 ) + 0.5f )) / 10.0f;
			vert->m_Pos.z = ((int) (( vert->m_Pos.z * 10 ) + 0.5f )) / 10.0f;
			*/
			
			file.write((const char*) &vert->m_Pos.x, sizeof( float ));
			file.write((const char*) &vert->m_Pos.y, sizeof( float ));
			file.write((const char*) &vert->m_Pos.z, sizeof( float ));		
		}

		file.write((const char*) &num_faces, sizeof( int ));
		for( j = 0; j < num_faces; j++ )
		{
			// Write out the material checksum for this face
			file.write((const char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Write out the face normal
			file.write((const char*) &face_list[j].m_Normal, sizeof( Point3 ));

			// Versioning data isn't saved with NxObject.  So we are using this flag
			// to indicate the addition of CAS Face data
			face_list[j].m_FaceFlags |= mFD_CASFACEFLAGSEXIST;

			// Write out the face flags
			file.write((const char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Write out the CAS face flags
			file.write((const char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));

			// Write out the vertex indices
			file.write((const char*) face_list[j].m_Vertex, sizeof( int ) * 3 );
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Write out CAS face removal flags
			file.write((const char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		// Write out KBias if appropriate
		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Write out the KBias
			file.write((const char*)&object->m_KBias,sizeof( float ));
		}

		// Write out LOD data if appropriate
		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Write out the LOD info
			file.write((const char*)&object->m_LODVersion,sizeof(unsigned int));
			file.write((const char*)&object->m_LODFlags,sizeof(int));

			if (object->m_LODFlags & NxObject::mMASTER)
			{
				file.write((const char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));
				file.write((const char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
			}

			if (object->m_LODFlags & NxObject::mSLAVE)
			{
				file.write((const char*)&object->m_LODSlave.m_masterCRC,sizeof(int));
			}
		}

		// Write out version info
		file.write((const char*)&object->m_Version,sizeof(unsigned int));

		// Write out relative info v1
		file.write((const char*)&object->m_ParentCRC,sizeof(unsigned long));
		file.write((const char*)&object->m_NumChildren,sizeof(int));
		file.write((const char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			file.close();
			return false;
		}
		gInterface->ProgressUpdate( i * 100 / num_obj );
	}

	gInterface->ProgressEnd();
	
	return true;
}

bool	SceneExporter::SaveScene( void )
{
	fstream file;
	TSTR scene_name, scene_dir, base_name;
	char* project_root;
	int version;
	DWORD attribs;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_UPDATE || 
		m_export_type == vQUICK_UPDATE_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SceneName;
	}	

	scene_dir = TSTR( project_root ) + TSTR( "/data/levels/" ) + base_name;
	CreateDirectory( scene_dir, NULL );	
	CreateDirectory( "c:/temp", NULL );	// for image processing

	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".scn" );
	attribs = GetFileAttributes( scene_name );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", scene_name );
			if( MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_OKCANCEL ) == IDCANCEL )
			{
				return false;
			}
		}
	}

	// break the lock, if necessary
	SetFileAttributes( scene_name, FILE_ATTRIBUTE_NORMAL );
	file.open( scene_name, ios::out | ios::binary );

	version = NxMaterial::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	file.write((const char*) &m_export_options.m_ExportAsSky, sizeof( bool ));
	if( SaveMaterials( file, m_scene ) == false )
	{
		file.close();
		return false;
	}

	if( SaveGeometry( file ) == false )
	{
		file.close();
		return false;	
	}	

	file.close();
	return true;
}

bool	SceneExporter::SaveTextures( void )
{
	TSTR texfile_name, usgfile_name, base_name;
	char* project_root;	
	ITextureExporter* tex_exp;
	
	tex_exp = GetTextureExporter();
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_UPDATE ||
		m_export_type == vQUICK_UPDATE_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SceneName;
	}

	texfile_name = TSTR( project_root ) + TSTR( "/data/levels/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".tex" );
	usgfile_name = TSTR( project_root ) + TSTR( "/data/levels/" ) + base_name + TSTR( "/" ) + base_name + TSTR( ".usg" );
	
	if (m_export_options.m_DirWarn)
	    if (!tex_exp->VerifyTexturePaths( TSTR( "levels/" ) + base_name ))
			return false;

	return tex_exp->SaveTextureDictionary( texfile_name, usgfile_name );
}

bool	SceneExporter::ExportAsSky( void )
{
	return m_export_options.m_ExportAsSky;
}

bool	SceneExporter::Save( void )
{
	bool bVal;

	bVal = SaveScene();

	if (m_export_type != vQUICK_EXPORT_NOTEX &&
		m_export_type != vQUICK_UPDATE_NOTEX &&
		m_export_type != vQUICK_VIEW_NOTEX   &&
		m_export_type != vRESET_AND_RELOAD_NOTEX)
		bVal = bVal && SaveTextures();

	return( bVal );	
}

bool	SceneExporter::SpawnPostExportProcessor( void )
{
	int result;
	char *args[4];
	char path[_MAX_PATH], conv_path[_MAX_PATH];	
	TSTR scene_name, scene_dir, base_name;
	char* project_root;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_UPDATE ||
		m_export_type == vQUICK_UPDATE_NOTEX )
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_SceneName;
	}	

	scene_dir = TSTR( project_root ) + TSTR( "/data/levels/" ) + base_name;	
	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".scn" );

	sprintf( path, "-f%s", scene_name );
	sprintf( conv_path, "%s\\bin\\win32\\sceneconv.exe", project_root );

	args[0] = conv_path;
	args[1] = "-pp";
	args[2] = path;	
	args[3] = NULL; 

	result = ( spawnv( _P_WAIT, args[0], args ) == 0 );					
	return true;
}

bool	SceneExporter::DoExport( ExportType type )
{
	ITextureExporter* tex_exp;
	IScriptExporter* script_exp;

	// Reset all warning mesages
	MessageBoxResetAll();

	GetSceneExportOptions( &m_export_options );
	if(	m_export_options.m_QuickUpdateSelectedOnly && (type == vQUICK_UPDATE || type == vQUICK_UPDATE_NOTEX) )
	{
		m_export_options.m_ExportSelected = true;
	}

	tex_exp = GetTextureExporter();
	tex_exp->Reset();	

	EnumerateExportableNodes();

	// Scripts should be exported even if their aren't any scene exportable nodes
	// since the script export does it's own node enumeration
//	if(( type != vQUICK_VIEW ) && ( type != vQUICK_UPDATE ) && ( type != vQUICK_EXPORT_NOTEX ) &&
//	   ( type != vQUICK_UPDATE_NOTEX) && (type != vQUICK_VIEW_NOTEX))
	if( type == vSCRIPT_EXPORT || type == vQUICK_VIEW || type == vQUICK_EXPORT || type == vRESET_AND_RELOAD ||
		type == vQUICK_UPDATE_NOTEX || type == vQUICK_VIEW_NOTEX || type == vRESET_AND_RELOAD_NOTEX || type == vQUICK_EXPORT_NOTEX)
	{
		script_exp = GetScriptExporter();
		if( script_exp->ExportScripts( m_export_options ) == false )
		{
			return false;
		}

		// If that was it, just return now
		if( type == vSCRIPT_EXPORT )
		{
			return true;
		}
	}

	if( m_exportable_nodes.Count() == 0 )
	{
		// Nothing to export
		return false;
	}

	if( type != vQUICK_UPDATE &&
		type != vQUICK_UPDATE_NOTEX)
	{
		// Export the camera animations  aml
		if (m_export_options.m_ExportCameras)
		{
			CameraExporter* camExporter=GetCamExporter();
			assert(camExporter);

			if (!camExporter->DoExport( vCAM_EXPORT ))
				return false;
		}
	}

	// At this point, we have gathered a list of nodes which represent the exportable geometry.
	// Now parse these nodes and store relavent data in our own generic database
	m_scene = ExtractSceneData( m_export_options.m_ExportFromOrigin );
	// If something went wrong or it was cancelled, don't continue
	if( m_scene == NULL )
	{
		return false;
	}
	
	m_export_type = type;
	
	// Load the textures needed for the current export set
	if (m_export_type != vQUICK_EXPORT_NOTEX &&
		m_export_type != vQUICK_UPDATE_NOTEX &&
		m_export_type != vQUICK_VIEW_NOTEX   &&
		m_export_type != vRESET_AND_RELOAD_NOTEX)
		if (!tex_exp->LoadTextureData())
			return false;

	// Here output the scene and its materials, either via the network or to disk
	if( Save())
	{
		if( SpawnPostExportProcessor())
		{
			if( m_export_type == vRESET_AND_RELOAD ||
				m_export_type == vRESET_AND_RELOAD_NOTEX)
			{
				char exec_str[128];
				char *viewer_root;

				if( viewer_root = getenv( vSKATE4_ENVIRON_VAR ))
				{			
					char viewer_app[_MAX_DIR + _MAX_FNAME];

					sprintf( viewer_app, "%s\\build\\NGPSgnu\\%s", viewer_root, vVIEWER_APP_NAME );
					sprintf( exec_str, "ps2run -r %s", viewer_app );
					WinExec( exec_str, SW_HIDE );
				}
			}
			else if( m_export_type == vQUICK_VIEW ||
				     m_export_type == vQUICK_VIEW_NOTEX)
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				sprintf( msg.m_Filename, m_export_options.m_SceneName );
				gClient->EnqueueMessageToServer( Net::vMSG_ID_QUICKVIEW, sizeof( Net::MsgQuickview ),
													&msg );
			}
			// TODO: in the future NOTEX should probably be a seperate message
			//       but for now we just won't update textures on the client side
			else if( m_export_type == vQUICK_UPDATE ||
				     m_export_type == vQUICK_UPDATE_NOTEX)
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				sprintf( msg.m_Filename, m_export_options.m_SceneName );
				sprintf( msg.m_UpdateFilename, "quick" );
				gClient->EnqueueMessageToServer( Net::vMSG_ID_INCREMENTAL_UPDATE, sizeof( Net::MsgQuickview ),
													&msg );
			}
		}
	}
	
	// We're done with the scene. KILL IT!
	delete m_scene;
	m_scene = NULL;

	return true;
}

SceneExporter::SceneExporter( void )
{	
}

NxScene::NxScene( void )
{
	NxMaterial default_mat;

	m_Materials.Append( 1, &default_mat );
}

NxScene::~NxScene( void )
{
}