/*
	ObjAnimExport.h
	This is adapted from the camera exporter to export information for animation of objects within the level
*/

#ifndef __OBJANIMEXPORTER__
#define __OBJANIMEXPORTER__

#include "AnimExporter.h"
#undef assert
#include "../PropEdit/RTFEditor.h"
#include "max.h"
#include "notify.h"
#include "../UI/ResizeWindow.h"
#include "../UI/ProgressBar.h"
#include "../misc/llist.h"
#include "skeleton.h"
#include "Export.h"
#include "ExportOptions.h"
#include "../UI/ModalDlg.h"

class GenExporter;
class ListView;
class RTFEditor;
struct ExportInfo;

struct ObjSpecificExportSettings  // vNAPP_CAMERA_EXPORT_CAMSETTINGS_ID
{
	float fRotTolerance;	// Rotation compression tolerance
	float fTranTolerance;	// Translation compression tolerance
	int   start;			// Start frame
	int   end;              // End frame
	char  strUserName[256];	// Name of the camera assigned by user (export identifier)
};

class ObjAddKeyDlg: public MSDlgWindow
{
	Interface*        ip;					// MAX Interface pointer
	CStr              scriptFile;			// The script filename of the currently selected script

	void (*fpOK)(ObjAddKeyDlg*,void*);		// Callback executed when user presses OK
	void *pOKData;							// Callback retained data

	ICustEdit*       IEdit;
	ICustEdit*       IEditFile;
	RTFEditor*       scriptEditor;

	ISpinnerControl* ISpin;
	LinkList<CStr>   scriptFileDB;			// Stores the names of all the script files referenced in the script list

	bool             bAbort;				// True if script search was aborted
	bool             bScriptsLoaded;		// True if recursive search done for scripts

	BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	bool MatchContent(FILE* fp,char* wildcard,char* buf,int* line);
	void Find(char* sdir,char* prefix,char* search);
	int  CountFiles(char* sdir,char* prefix);

	void SelChange();

public:
	void  Show();

	void SetOKCB(void(*Func)(ObjAddKeyDlg*,void*),void* pData);
	
	void  SetLimits(int min,int max);
	int   GetTime();
	float GetTimeSec();
	void  SetTime(int time);

	CStr  GetScript();
	void  SetScript(CStr strScript);
	void  SpinChange();

	CStr  GetScriptCode();
	void  SetScriptCode(CStr strScriptCode);

	void  Refresh();

	CStr GetScriptFile();
	inline void SetScriptFile(CStr file) { scriptFile=file; }

	ObjAddKeyDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip);
	~ObjAddKeyDlg();
};

struct ObjScriptKey
{
	CStr  name;		// For user-defined scripts name = [User] and file contains the script
	CStr  file;
	int   frame;
};

struct ObjInfo
{
	INode*                  node;
	LinkList<ObjScriptKey>  ObjScriptKeys;
};

class ObjectExporter: public Exporter, public AnimExporter
{
	struct NodeData
	{
		Gfx::SAnimQFrame*         QFrames;
		Gfx::SAnimTFrame*         TFrames;
		int*                      numCompressedQFrames;
		int*                      numCompressedTFrames;
		int                       numNodes;
		
		INode*                    node;

		ObjSpecificExportSettings settings;

		NodeData()
		{
			QFrames = NULL;
			TFrames = NULL;
			numCompressedQFrames = NULL;
			numCompressedTFrames = NULL;
			numNodes = 0;
		}

		~NodeData()
		{
			if (QFrames)
				delete [] QFrames;

			if (TFrames)
				delete [] TFrames;

			if (numCompressedQFrames)
				delete [] numCompressedQFrames;

			if (numCompressedTFrames)
				delete [] numCompressedTFrames;
		}
	};

	LinkList<NodeData>     nodeList;		// Stored data to be exported

	int GetKeyCount(char* buf);

protected:
	GenKey<ObjScriptKey>* GetObjScriptKeys(INode* node,int* numKeys);

	void BuildNodeData(INodeTab& nodes,bool bCompress=false,bool bOneFrame=true);

	bool ExportNode(FILE* fpExport,FILE* fpDebug,INode* node,int numExtFrames,
					float errorQ,float errorT,
					bool bUseCompression, bool bCompressTime, bool bOneFrame);
	
	bool Export(char* filename,LinkList<ObjInfo>* objDB,int start,int end,
							float errorQ,float errorT,
							bool bCompress,bool bOneFrame, bool bDebug, bool bCompressTime);

	CStr GetSceneName();
	void SendNetMessage(char* camname,int delay);

	bool NodeSelected(INode* node);

	int GetNumScriptKeys(INode* node);
	int GetNumScriptKeys( INodeTab& nodes );
	int GetNumQFrames();
	int GetNumTFrames();
	void GetFrameRange( int& min, int& max);

public:
	ObjectExporter();
	~ObjectExporter();

	bool ExportQN(FILE* fp);
	bool DoExport( ExportType type );
	bool Save();

	void BuildPS2File(char* camname);
	void DoQBRUpdate();

};

ObjectExporter* GetObjExporter();

class ObjAnimExportDlg: public ResizeDlgWindow, public ObjectExporter
{
	Interface*   ip;						// MAX Interface pointer
	ExportOptionsDlg* expopt;				// Stores the export options dialog, so we can force it to close
	GenExporter* genExporter;				// Pointer the generalized keyframe data exporter (used for FOV keys)

	RTFEditor*   scriptEditor;

	LinkList<CStr> scriptDB;				// Script buffers attached to each export node

	ObjSpecificExportSettings objsettings_defaults;

	ListView*    lview;						// UI Interfaces
	ListView*    objList;

	ICustEdit* editQErr;
	ICustEdit* editTErr;
	ICustEdit* editStart;
	ICustEdit* editEnd;
	ICustEdit* editKeyTime;
	ICustEdit* editSearch;

	ISpinnerControl* spinQErr;
	ISpinnerControl* spinTErr;
	ISpinnerControl* spinFErr;
	ISpinnerControl* spinStart;
	ISpinnerControl* spinEnd;
	ISpinnerControl* spinKeyTime;

	ProgressBar*     progBar;
	
	CStr             strObjName;			// Name of the current camera
	int              lastItem;				// Item described by strCamName

	ObjAddKeyDlg*    pAddKeyDlg;			// Add key dialog

	LinkList<ObjInfo> objDB;				// Database of info attached to each camera

	Tab<INode*>      selSet;				// List of nodes to apply changes to

	INode*           lastCamNode;			// The last camera node selected
	bool             bLockobjList;			// Locks update calls for the camera list (objList)
	bool             bLockKeyList;			// Locks update calls for the keyframe list (lview)
	bool             bLockNodeSelect;		// Locks update calls from camera list selection
	bool             bLockLoadSave;
	bool             bLockSelChange;		// Lock update on any selection changes
	bool             bAbort;				// True if the export was aborted

	BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
	bool DoExport();
	CStr GetSaveFile();

	static void MAXSelChange(void *param,NotifyInfo *info);
	static void AddKeyCB(ObjAddKeyDlg* pDlg,void* pData);

	void SelChange(bool bNoApply=false);	// MAX selection change
	void CamSelChange(NMLISTVIEW* pNotify);	// Camera list box selection change

	void AddScript();
	void DeleteScript();
	void ListViewUpdate(NMLISTVIEW* pNotify);

	void SaveToNode(char* strSys,INode* node=NULL);		// Saves the script keyframe info to the current camera node
	void LoadFromNode(char* strSys,INode* node=NULL);	// Loads the script keyframe info from the current camera node

	void SaveSettings();					// Save the general export settings to the scene
	void LoadSettings();					// Loads the general export settings from the scene
	
	void Apply();
	void AcquireSelSet();
	void GetRange();
	void UpdateRange();
	void UpdateLabel(NMLVDISPINFO* pInfo);	// Process changing the name of a field in the listview window (objList)
	void UpdateName();

	void SaveObjSelection();
	void LoadObjSelection();

	void UpdateScript();

	bool ExportSingleCam(int i);

	void SendNetMessage(int i=-1);
	void BuildPS2File(int i=-1);

	static void CancelCB(void* pData);
	static int Sort(DWORD dw1,DWORD dw2,ListView* pData,void* pExtData);

	//void Resize(int width,int height);

	void UpdateKeyTime();

	static void CloseExportDlg(void *param, NotifyInfo *info);

	bool ObjectExists(char* name);

	void AddObj(char* sysName);
	void AddObj(Link<ObjInfo>* objlink);
	void RemoveObj(char* sysName);
	void UpdateAllObjSel();

	CStr GetCurSystem();

	void RemoveSystem(CStr sysName,INode* root=NULL);
	void AnimSysSelChange();

public:
	ObjAnimExportDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip,ExportOptionsDlg* expopt=NULL);
	~ObjAnimExportDlg();

	void GetObjData(char* sysName);
	void GetObjSysData();
	void ResortList();

	void Cleanup(char* path);
	void Archive(char* path);

	void  AddSystem();
	void  RemoveSystem();

	void Show();
	void Hide();
};

class SystemNameDlg: public ModalDlgWindow
{
	ICustEdit* IEdit;
	CStr       name;

	BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

public:
	SystemNameDlg(HINSTANCE hInstance,HWND hwndParent);
	~SystemNameDlg();

	void SetName(CStr name);
	CStr GetName();
};

#endif
