#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <Export/ModelExport.h>
#include <Export/ScriptExport.h>
#include <Export/TextureExporter.h>
#include <Export/Strip/Strip.h>
#include <Misc/GenCrc.h>
#include <process.h>
#include "../UI/OKtoAll.h"
#include "../misc/mdir.h"
#include "skeleton.h"
#undef __TRIG__
#include "trig.h"

extern Interface* gInterface;

class ModelExporter : public IModelExporter
{
public:
	NxModel*		ExtractModelData( Tab <INode *>& exportable_nodes, bool skinned, float scale );

	bool	SpawnPostExportProcessor( void );
	void	EnumerateExportableNodes( void );
	bool	SaveTextures( void );
	bool	SaveModel( void );
	bool	SaveGeometry( fstream& file );

	bool    BuildModelSkelData(NxModel* model,INode* node);
	int		FindBone(NxModel* model,unsigned long crc);

	// From Exporter
	bool	DoExport( ExportType type );
	bool	Save( void );

private:

	void		enumerate_visible_nodes( INode* root );
	
	Tab <INode *>		m_exportable_nodes;	
	ModelExportOptions	m_export_options;
	NxModel*			m_model;
};

static	ModelExporter	s_model_exporter;

IModelExporter*	GetModelExporter( void )
{
	return &s_model_exporter;
}

DWORD WINAPI ModelExportProgressFunc(LPVOID arg) 
{
    return(0);
}

int		ModelExporter::FindBone(NxModel* model,unsigned long crc)
{
	int nBones = model->nBones;

	for(int i=0;i<nBones;i++)
	{
		if (model->bones[i].crc == crc)
			return i;
	}

	return -1;
}

void DumpMatrix(char* name, Mth::Matrix mat)
{
	// Don't dump unless the file already exists
	FILE* fp = fopen("c:\\modeldbg.txt","r");

	if (!fp)
		return;

	fclose(fp);

	fp = fopen("c:\\modeldbg.txt","a");

	fprintf(fp,"New Matrix (%s):\n",name);

	for(int i=0;i<4;i++)
	{
		for(int j=0;j<4;j++)
		{
			fprintf(fp,"%f ",mat[i][j]);
		}

		fprintf(fp,"\n");
	}

	fprintf(fp,"\n");

	fclose(fp);
}

void DumpMatrix(char* name, Matrix3 mat)
{
	// Don't dump unless the file already exists
	FILE* fp = fopen("c:\\modeldbg.txt","r");

	if (!fp)
		return;

	fclose(fp);
	
	fp = fopen("c:\\modeldbg.txt","a");

	fprintf(fp,"New Matrix (%s):\n",name);

	for(int i=0;i<4;i++)
	{
		Point3 row;
		row = mat.GetRow(i);

		fprintf(fp,"%f %f %f",row.x,row.y,row.z);

		fprintf(fp,"\n");
	}

	fprintf(fp,"\n");

	fclose(fp);
}

bool	ModelExporter::BuildModelSkelData(NxModel* model,INode* node)
{
	ModelExportOptions meo;
	GetModelExportOptions(&meo);

	if (!meo.m_IncSkelData)
		return false;

	// Allocate memory for bone data
	if (model->m_Objects.Count()==0)
		return false;

	// All objects in the model should be connected in the hierarchy
	// (at least to the root anyway, so it should be safe to pick the first one)
	CSkeletonData csData(node,false);
	CStr name = node->GetName();

	int nKids = node->NumberOfChildren();

	model->nBones = csData.GetCount();

	model->bones = new NxModel::BoneDesc[model->nBones];

	// Now go through all the bones and build up a list of corresponding parent matricies
	for(int i=0;i<model->nBones;i++)
	{
		INode* bone = csData.GetBone(i);
		CStr   name = bone->GetName();

		model->bones[i].crc = GenerateCRC(name);

		Matrix3 tm = bone->GetNodeTM(0);

		// Rotate the root bone
		if (i==0)
			tm.RotateX(DegToRad(-90));

		// Swap coordSystem
		//tm.RotateX(DegToRad(-90));

		tm.NoScale();
		INode* parentNode = bone->GetParentNode();
		CStr   parentName = parentNode->GetName();

		Matrix3 parentTM;

		if (parentNode == gInterface->GetRootNode())
			parentTM.IdentityMatrix();
		else
			parentTM = parentNode->GetNodeTM(0);

		parentTM.NoScale();
		DumpMatrix(parentName + " PARENT",parentTM);
		
		DumpMatrix(name + "Orig TM",tm);

		tm = tm * Inverse(parentTM);

		DumpMatrix(name + "Post TM",tm);

		/*
		quat=Quat(tm);

		// make sure they're all anims start off with a positive w.
		// (only do this on frame 0!)
		if (quat.w<0.0f)
		{
			quat.x=-quat.x;
			quat.y=-quat.y;
			quat.z=-quat.z;
			quat.w=-quat.w;
		}

		prevQuat = quat;
		qframes[i].q=Mth::Quat(quat.x,quat.y,quat.z,quat.w);
		*/
		
		// MAX is right-handed as well, so this should be OK in theory
		model->bones[i].mat.Ident();
		
		Point3 row;
		row = tm.GetRow(0);
		model->bones[i].mat[Mth::RIGHT][0] = row.x;
		model->bones[i].mat[Mth::RIGHT][1] = row.y;
		model->bones[i].mat[Mth::RIGHT][2] = row.z;
		
		row = tm.GetRow(1);
		model->bones[i].mat[Mth::UP][0] = row.x;
		model->bones[i].mat[Mth::UP][1] = row.y;
		model->bones[i].mat[Mth::UP][2] = row.z;

		row = tm.GetRow(2);
		model->bones[i].mat[Mth::AT][0] = row.x;
		model->bones[i].mat[Mth::AT][1] = row.y;
		model->bones[i].mat[Mth::AT][2] = row.z;

		row = tm.GetRow(3);
		model->bones[i].mat[Mth::POS][0] = row.x;
		model->bones[i].mat[Mth::POS][1] = row.y;
		model->bones[i].mat[Mth::POS][2] = row.z;

		DumpMatrix(name,model->bones[i].mat);
	}

	return true;
}

bool	ModelExporter::SaveGeometry( fstream& file )
{
	int i, j, k, num_obj, num_faces, num_verts;

	// TODO: Hierarchy matrix save

	num_obj = m_model->m_Objects.Count();

	file.write((const char*) &num_obj, sizeof( int ));
	gInterface->ProgressStart(_T("Exporting Geometry"), TRUE, 
							ModelExportProgressFunc, NULL );
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxFace* face_list;
		NxObject* object;
		Point3 center;
		float radius;
		unsigned long checksum;

		object = m_model->m_Objects[i];
		vert_list = object->m_Verts;
		face_list = object->m_Faces;
		num_faces = object->m_NumFaces;
		num_verts = object->m_NumVerts;

		
		// Write out the object's checksum		
		checksum = GenerateCRC( object->m_Name );
		file.write((const char*) &checksum, sizeof( unsigned long ));		

		if (object->m_Flags & NxObject::mINVISIBLE)
		{
			int zzz;
			zzz=0;
		}

		// Write out the flags which describe the object's properties
		object->m_Flags |= NxObject::mHASVERSIONINFO;

		if (object->m_Flags & NxObject::mINVISIBLE)
		{
			int zzz;
			zzz=0;
		}

		if (m_model->nBones > 0)
			object->m_Flags |= NxObject::mSKELETALMODEL;

		file.write((const char*) &object->m_Flags, sizeof( int ));

		// Write out the index of the object's parent matrix
		if (object->m_Flags & NxObject::mSKELETALMODEL)
		{
			int index = FindBone(m_model,checksum);
			file.write((const char*) &index, sizeof( int ));
		}

		// Write out the number of sets of UVs
		file.write((const char*) &object->m_NumUVSets, sizeof( int ));

		// Write out bounding box/sphere data
		file.write((const char*) &object->m_BoundingBox.pmin, sizeof( Point3 ));
		file.write((const char*) &object->m_BoundingBox.pmax, sizeof( Point3 ));

		center = object->m_BoundingBox.Center();
		radius = (( object->m_BoundingBox.pmax - object->m_BoundingBox.pmin ).Length()) / 2.0f;
		file.write((const char*) &center, sizeof( Point3 ));
		file.write((const char*) &radius, sizeof( float ));

		file.write((const char*) &num_verts, sizeof( int ));
		for( j = 0; j < num_verts; j++ )
		{
			NxVertex* vert;			
			
			vert = &vert_list[j];

			// Write out first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					file.write((const char*) &vert->m_TexCoord[k].x, sizeof( float ));
					file.write((const char*) &vert->m_TexCoord[k].y, sizeof( float ));
				}
			}

			// Write out colors
			if( object->m_Flags & NxObject::mCOLORED )
			{
				file.write((const char*) &vert->m_Color.r, sizeof( float ));
				file.write((const char*) &vert->m_Color.g, sizeof( float ));
				file.write((const char*) &vert->m_Color.b, sizeof( float ));
				file.write((const char*) &vert->m_Color.a, sizeof( float ));
			}				

			// Write out normals
			if( object->m_Flags & NxObject::mNORMALS )
			{					
				file.write((const char*) &vert->m_Normal.x, sizeof( float ));
				file.write((const char*) &vert->m_Normal.y, sizeof( float ));
				file.write((const char*) &vert->m_Normal.z, sizeof( float ));
			}

			// Write out weighting info
			if( object->m_Flags & NxObject::mSKINNED )
			{
				file.write((const char*) &vert->m_Weight[0], sizeof( float ));
				file.write((const char*) &vert->m_Weight[1], sizeof( float ));
				file.write((const char*) &vert->m_Weight[2], sizeof( float ));

				file.write((const char*) &vert->m_WeightedIndex[0], sizeof( int ));
				file.write((const char*) &vert->m_WeightedIndex[1], sizeof( int ));
				file.write((const char*) &vert->m_WeightedIndex[2], sizeof( int ));			
			}

			// Write out vc wibble data
			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				file.write((const char*) &vert->m_WibbleIndex, sizeof( char ));
				file.write((const char*) &vert->m_WibbleOffset, sizeof( char ));	
			}

			// Write out positions				
			file.write((const char*) &vert->m_Pos.x, sizeof( float ));
			file.write((const char*) &vert->m_Pos.y, sizeof( float ));
			file.write((const char*) &vert->m_Pos.z, sizeof( float ));		
		}

		file.write((const char*) &num_faces, sizeof( int ));
		for( j = 0; j < num_faces; j++ )
		{
			// Write out the material checksum for this face
			file.write((const char*) &face_list[j].m_MatChecksum, sizeof( unsigned long ));

			// Write out the face normal
			file.write((const char*) &face_list[j].m_Normal, sizeof( Point3 ));

			// Versioning data isn't saved with NxObject.  So we are using this flag
			// to indicate the addition of CAS Face data
			face_list[j].m_FaceFlags |= mFD_CASFACEFLAGSEXIST;

			// Write out the face flags
			file.write((const char*) &face_list[j].m_FaceFlags, sizeof( FlagType ));

			// Write out the CAS face flags
			file.write((const char*) &face_list[j].m_CASFaceFlags, sizeof( CASFlagType ));

			// Write out the vertex indices
			file.write((const char*) face_list[j].m_Vertex, sizeof( int ) * 3 );
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Write out CAS face removal flags
			file.write((const char*)&object->m_CASRemoveFlags, sizeof( int ));
		}

		// Write out KBias if appropriate
		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Write out the KBias
			file.write((const char*)&object->m_KBias,sizeof( float ));
		}

		// Write out LOD data if appropriate
		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Write out the LOD info
			file.write((const char*)&object->m_LODVersion,sizeof(unsigned int));
			file.write((const char*)&object->m_LODFlags,sizeof(int));

			if (object->m_LODFlags & NxObject::mMASTER)
			{
				file.write((const char*)&object->m_LODMaster.m_NumLODLevels,sizeof(int));
				file.write((const char*)object->m_LODMaster.m_LODLevels,sizeof(NxLODLevel) * object->m_LODMaster.m_NumLODLevels);
			}

			if (object->m_LODFlags & NxObject::mSLAVE)
			{
				file.write((const char*)&object->m_LODSlave.m_masterCRC,sizeof(int));
			}
		}

		// Write out version info
		file.write((const char*)&object->m_Version,sizeof(unsigned int));

		// Write out relative info v1
		file.write((const char*)&object->m_ParentCRC,sizeof(unsigned long));
		file.write((const char*)&object->m_NumChildren,sizeof(int));
		file.write((const char*)object->m_ChildCRCs,sizeof(unsigned long) * object->m_NumChildren);

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			file.close();
			return false;
		}
		gInterface->ProgressUpdate( i * 100 / num_obj );
	}

	// Write out model matrix hierarchy information
	// This is only read if an object(s) were read in with the mSKELETALMODEL flag
	if (m_model->nBones > 0)
	{
		int version = NxModel::MODEL_VERSION;
		file.write((const char*) &version, sizeof( int ));

		file.write((const char*)&m_model->nBones,sizeof( int ));
		file.write((const char*)m_model->bones,sizeof(NxModel::BoneDesc) * m_model->nBones);
	}

	gInterface->ProgressEnd();
	
	return true;
}

bool	ModelExporter::SaveModel( void )
{
	fstream file;
	TSTR mdl_name, mdl_dir, base_name;
	char* project_root;
	int version;
	DWORD attribs;
	bool sky_export;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_ModelName;
	}	

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		mdl_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name;
	}
	else
		mdl_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name;

	//CreateDirectory( mdl_dir, NULL );	
	MDir(mdl_dir);	// Make directory hierarchy
	CreateDirectory( "c:/temp", NULL );	// for image processing

	mdl_name = mdl_dir + TSTR( "/" ) + base_name + TSTR( ".mdl" );
	attribs = GetFileAttributes( mdl_name );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", mdl_name );
			if( MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_OKCANCEL ) == IDCANCEL )
			{
				return false;
			}
		}
	}

	// break the lock, if necessary
	SetFileAttributes( mdl_name, FILE_ATTRIBUTE_NORMAL );
	file.open( mdl_name, ios::out | ios::binary );

	version = NxMaterial::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	file.write((const char*) &version, sizeof( int ));
	sky_export = false;
	file.write((const char*) &sky_export, sizeof( bool ));

	if( SaveMaterials( file, m_model ) == false )
	{
		file.close();
		return false;
	}

	if( SaveGeometry( file ) == false )
	{
		file.close();
		return false;	
	}	

	file.close();
	return true;	
}

bool	ModelExporter::SpawnPostExportProcessor( void )
{
	int result;
	char *args[4];
	char path[_MAX_PATH], conv_path[_MAX_PATH];	
	TSTR scene_name, scene_dir, base_name;
	char* project_root;
	
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	/*if( m_export_type == vQUICK_VIEW )
	{
		base_name = "quick";
	}
	else*/
	{
		base_name = m_export_options.m_ModelName;;
	}	

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name;
	}
	else
		scene_dir = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name;

	scene_name = scene_dir + TSTR( "/" ) + base_name + TSTR( ".mdl" );

	sprintf( path, "-f%s", scene_name );
	sprintf( conv_path, "%s\\bin\\win32\\sceneconv.exe", project_root );

	args[0] = conv_path;
	args[1] = "-pp";
	args[2] = path;	
	args[3] = NULL; 

	result = ( spawnv( _P_WAIT, args[0], args ) == 0 );					
	return true;
}

bool	ModelExporter::SaveTextures( void )
{
	TSTR texfile_name, usgfile_name, base_name;
	char* project_root;	
	ITextureExporter* tex_exp;
	
	tex_exp = GetTextureExporter();
	project_root = getenv( vSKATE4_ENVIRON_VAR );
	if( project_root == NULL )
	{
		MessageBox( gInterface->GetMAXHWnd(), "You must set up your SKATE4_PATH environment variable",
						"Error!", MB_OK|MB_TASKMODAL);
		return false;
	}

	if( m_export_type == vQUICK_VIEW ||
		m_export_type == vQUICK_VIEW_NOTEX)
	{
		base_name = "quick";
	}
	else
	{
		base_name = m_export_options.m_ModelName;
	}

	if (m_export_options.m_ExportToSceneDir)
	{
		SceneExportOptions seo;
		GetSceneExportOptions(&seo);

		texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".tex" );
		usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + seo.m_SceneName + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".usg" );
	}
	else
	{
		texfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".tex" );
		usgfile_name = TSTR( project_root ) + TSTR( "/data/models/" ) + m_export_options.m_ModelDir + TSTR("/") + base_name + TSTR("/") + base_name + TSTR( ".usg" );
	}
	
	if (m_export_options.m_DirWarn)
	    if (!tex_exp->VerifyTexturePaths( TSTR( "models/" ) + base_name ))
			return false;

	return tex_exp->SaveTextureDictionary( texfile_name, usgfile_name );
}

bool	ModelExporter::Save( void )
{
	return( SaveModel() && SaveTextures());
}

bool	ModelExporter::DoExport( ExportType type )
{
	CStr export_name;
	ITextureExporter* tex_exp;

	// Ensure that an object is select if doing skeletal animation export
	if (m_export_options.m_IncSkelData)
	{
		if (gInterface->GetSelNodeCount()==0)
		{
			MessageBox(gInterface->GetMAXHWnd(),"You must select the model object you are exporting if you are exporting with skeletal animation data.","Model Skeletal Animation Export",MB_ICONWARNING|MB_OK);
			return false;
		}
	}

	// Reset all warning mesages
	MessageBoxResetAll();

	// Write out a .qn file for the model
	IScriptExporter* scriptExp = GetScriptExporter();
	
	tex_exp = GetTextureExporter();
	tex_exp->Reset();	

	GetModelExportOptions( &m_export_options );

	EnumerateExportableNodes();
	m_export_type = type;

	// Dump model .qn
	if (m_export_options.m_ExportQN)
	{
		SceneExportOptions options;
		GetSceneExportOptions(&options);
		//options.m_SceneName=m_export_options.m_ModelName;

		//scriptExp->ExportScriptsFromSet(m_export_options.m_ModelName,m_exportable_nodes,true);
		scriptExp->ExportScripts(options,true);
	}
	

	if( m_exportable_nodes.Count() == 0 )
	{
		// Nothing to export
		return false;
	}

	// Check (and warn) for duplicate object names
	int i,j;

	for( i = 0; i < m_exportable_nodes.Count(); i++)
	{
		for( j = 0; j < m_exportable_nodes.Count(); j++ )
		{
			if (j != i)
			{
				CStr name1 = m_exportable_nodes[i]->GetName();
				CStr name2 = m_exportable_nodes[j]->GetName();

				if( name1 == name2 )
				{
					char msg[1024];

					sprintf( msg, "Multiple objects are named %s. This will cause problems.  Aborting export.", (char*)m_exportable_nodes[i]->GetName() );
					MessageBoxAll( gInterface->GetMAXHWnd(), msg, "Multiple named objects!", MB_OK|MB_TASKMODAL);
					return false;
				}
			}
		}
	}

	// At this point, we have gathered a list of nodes which represent the exportable geometry.
	// Now parse these nodes and store relavent data in our own generic database
	m_model = ExtractModelData( m_exportable_nodes, false, 1.0f );
	// If something went wrong or it was cancelled, don't continue
	if( m_model == NULL )
	{
		return false;
	}

	// Load the textures needed for the current export set
	tex_exp->LoadTextureData();

	// Here output the model and its materials, either via the network or to disk
	if( Save())
	{		
		if( SpawnPostExportProcessor())
		{
			// Here do quickview stuff
			if( m_export_type == vRESET_AND_RELOAD ||
				m_export_type == vRESET_AND_RELOAD_NOTEX)
			{
				char exec_str[128];
				char *viewer_root;

				if( viewer_root = getenv( vSKATE4_ENVIRON_VAR ))
				{			
					char viewer_app[_MAX_DIR + _MAX_FNAME];

					sprintf( viewer_app, "%s\\build\\NGPSgnu\\%s", viewer_root, vVIEWER_APP_NAME );
					sprintf( exec_str, "ps2run -r %s", viewer_app );
					WinExec( exec_str, SW_HIDE );
				}
			}
			else if( m_export_type == vQUICK_VIEW ||
				     m_export_type == vQUICK_VIEW_NOTEX)
			{
				Net::MsgQuickview msg;

				// The scn, tex and usg files have been exported to temp "quick" files.
				// Tell the engine, via the network, to load these files into the current scene
				sprintf( msg.m_Filename, m_export_options.m_ModelName );
				gClient->EnqueueMessageToServer( Net::vMSG_ID_QUICKVIEW, sizeof( Net::MsgQuickview ),
													&msg );
			}
		}
	}

	return true;
}

NxModel*		ModelExporter::ExtractModelData( Tab <INode *>& exportable_nodes, bool skinned, float scale )
{
	int i;
	INode* node;
	NxModel* model;
	int num_exportable_nodes;
	model = new NxModel;
	model->m_Skinned = skinned;
	model->m_DynamicallyLit = true;

	num_exportable_nodes = exportable_nodes.Count();
	
	gInterface->ProgressStart(_T("Enumerating exportable objects"), TRUE, 
							ModelExportProgressFunc, NULL );
	for( i = 0; i < num_exportable_nodes; i++ )
	{
		NxObject* object;

		node = exportable_nodes[i];
		CStr nodeName = CStr("Enum Nodes Exporting: ") + node->GetName() + "\n";
		OutputDebugString(nodeName);

		//object = model->CreateObjectFromNode( node, scale );

		////  Objects with the class LevelObject should be in local coordinates /////////
		CStr class_name, lclass_name;

		if( !node->GetUserPropString( "Class", class_name ))
		{
			node->GetUserPropString( "class", class_name );
		}

		lclass_name = class_name;
		lclass_name.toLower();

		// Determine if object should be exported using local coordinate system
		// If we're exporting with skeleton data everything must export to local coords
		if (m_export_options.m_IncSkelData)
		{
			ULONG flags = GeoDatabase::LOCAL_COORDS | GeoDatabase::SKEL_EXPORT | GeoDatabase::OFFSET_PIVOT;

			if (m_export_options.m_FlipVerts)
				flags |= GeoDatabase::FLIP_VERTS;

			object = model->CreateObjectFromNode( node, scale, flags );
		}
		else
		{
			if (lclass_name == CStr("levelobject"))
				object = model->CreateObjectFromNode( node, scale, GeoDatabase::LOCAL_COORDS );
			else
				object = model->CreateObjectFromNode( node, scale );
		}
		/////////////////////////////////////////////////////////////////////////////////

		if( object )
		{
			model->m_Objects.Append( 1, &object );		
		}
		
		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			delete model;
			return NULL;
		}
		gInterface->ProgressUpdate( i * 100 / num_exportable_nodes );
	}

	// Build up the hierarchial model data
	if (exportable_nodes.Count()>0 && m_export_options.m_IncSkelData)
	{
		int selCount = gInterface->GetSelNodeCount();
		
		if (selCount > 0)
		{
			INode* node = gInterface->GetSelNode(0);
			BuildModelSkelData(model,node);
		}
		else
		{
			MessageBox(gInterface->GetMAXHWnd(),"Select the model that you wish to export.","No model selected for export",MB_ICONWARNING|MB_OK);
			//BuildModelSkelData(model,exportable_nodes[0]);
		}

		/*
		// Find the first exportable node that has more than 1 child
		int cnt = exportable_nodes.Count();

		for(int i=0;i<cnt;i++)
		{
			if (exportable_nodes[i]->NumberOfChildren()>0)
			{
				BuildModelSkelData(model,exportable_nodes[0]);
				break;
			}
		}
		*/
	}

	gInterface->ProgressEnd();	
	
	return model;
}

void	ModelExporter::enumerate_visible_nodes( INode* root )
{
	INode* child;
	int i;
	
	for( i = 0; i < root->NumberOfChildren(); i++ )
	{
		child = root->GetChildNode( i );
		// Add only visible, non-frozen nodes
		if(	( child->IsHidden() == false ) &&
			( child->IsFrozen() == false ))
		{
			if( IsExportable( child ))
			{
				m_exportable_nodes.Append( 1, &child );			
			}
		}

		// Recursively add all visible children
		if( child->NumberOfChildren() > 0 )
		{
			enumerate_visible_nodes( child );
		}
	}
}

void	ModelExporter::EnumerateExportableNodes( void )
{
	m_exportable_nodes.ZeroCount();
	
	enumerate_visible_nodes( gInterface->GetRootNode());	
}

NxModel::NxModel( void )
{
	NxMaterial default_mat;

	m_Materials.Append( 1, &default_mat );

	nBones = 0;
	bones = NULL;
}

NxModel::~NxModel()
{
	if (bones)
		delete [] bones;
}
