/*
	AnimOptionsDlg.cpp
	1-7-01
*/

#include <Engine/Engine.h>
#include <Sk/Gamenet/ExportMsg.h>
#include <process.h>
#include "../Misc/GenCrc.h"
#include "AnimOptionsDlg.h"
#include "NExt.h"
#include <direct.h>
#include "../path.h"
#include "AppData.h"
#include "../PropEdit/ParseFuncs.h"
#include "../UI/PropList.h"
#include "SceneExportOptions.h"
#include "Resource.h"
//#include "phyexp.h"				// CStudio SDK  Physique Export

#define DEFAULT_COMPRESS_STRING  "0.0"//"0.999700"
#define DEFAULT_TCOMPRESS_STRING "1.000000"

#define ROTMIN  0.0f			// Min/Max for rotation tolerance
#define ROTMAX  1.0f
#define ROTINCR 0.0001f

#define TRANMIN -10.0f		// Min/Max for translation tolerance
#define TRANMAX  10.0f
#define TRANINCR 0.0001f

#ifndef IDC_HAND
#define IDC_HAND            MAKEINTRESOURCE(32649)
#endif

// Column indexes
#define COLID_ANIMNAME  0
#define COLID_START     1
#define COLID_END       2
#define COLID_ROTTOL    3
#define COLID_TRANTOL   4
#define COLID_FILESIZE  5
#define COLID_ACTIVE    6

// Extended Animation Options Dialog //////////////////////////////////////////////////////////

AnimOptions2Dlg::AnimOptions2Dlg(HINSTANCE hInstance,HWND hwndParent) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_ANIMOPTIONS2),hwndParent)
{
	animoptdata.modelname[0]=0;
	animoptdata.suffixname[0]=0;
	animoptdata.bExportQN=false;
	animoptdata.bUseCompression=true;
	animoptdata.bOutputDebug=false;			// Output debug information
	animoptdata.bCompressDoubles=true;		// Compress linear anims to one key
	animoptdata.dirname[0]=0;
	animoptdata.skeletonName[0]=0;

	this->hwndParent = hwndParent;

	CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
	CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);
	EnableWindow(GetDlgItem(hwnd,IDC_USEQNSUFFIX),TRUE);
}

AnimOptions2Dlg::~AnimOptions2Dlg()
{
	
}

BOOL AnimOptions2Dlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_CLOSE:
		Hide();
		return TRUE;
	
	case WM_COMMAND:

		switch(LOWORD(wParam))
		{
		case IDC_USEQNSUFFIX:
			if (IsDlgButtonChecked(hwnd,IDC_USEQNSUFFIX))
				EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),TRUE);
			else
				EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),FALSE);

			return TRUE;

		case IDC_EXPORTQN:
			if (IsDlgButtonChecked(hwnd,IDC_EXPORTQN))
			{
				EnableWindow(GetDlgItem(hwnd,IDC_USEQNSUFFIX),TRUE);

				if (IsDlgButtonChecked(hwnd,IDC_USEQNSUFFIX))
					EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),TRUE);
			}
			else
			{
				EnableWindow(GetDlgItem(hwnd,IDC_USEQNSUFFIX),FALSE);
				EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),FALSE);
			}

			return TRUE;

		case IDOK:
			UpdateOptions();
			Hide();
			return TRUE;

		case IDCANCEL:
			Hide();
			return TRUE;
		}
	}

	return FALSE;
}

void AnimOptions2Dlg::UpdateOptions()
{
	GetDlgItemText(hwnd,IDC_EDITNAME,animoptdata.modelname,80);
	GetDlgItemText(hwnd,IDC_DIRNAME,animoptdata.dirname,80);
	GetDlgItemText(hwnd,IDC_SKELNAME,animoptdata.skeletonName,80);

	SetDlgItemText(hwndParent,IDC_MODELNAME,animoptdata.modelname);

	if (!IsDlgButtonChecked(hwnd,IDC_USEQNSUFFIX))
		animoptdata.suffixname[0]='\0';

	GetDlgItemText(hwnd,IDC_EDITSUFFIX,animoptdata.suffixname,80);

	if (IsDlgButtonChecked(hwnd,IDC_EXPORTQN))
		animoptdata.bExportQN=true;
	else
		animoptdata.bExportQN=false;

	if (IsDlgButtonChecked(hwnd,IDC_USECOMPRESSION))
		animoptdata.bUseCompression=true;
	else
		animoptdata.bUseCompression=false;

	if (IsDlgButtonChecked(hwnd,IDC_COMPDBLS))
		animoptdata.bCompressDoubles=true;
	else
		animoptdata.bCompressDoubles=false;

	if (IsDlgButtonChecked(hwnd,IDC_OUTPUTDBG))
		animoptdata.bOutputDebug=true;
	else
		animoptdata.bOutputDebug=false;

//	if (IsDlgButtonChecked(hwnd,IDC_PREROTATE))
//		animoptdata.bRotateRoot=true;
//	else
		animoptdata.bRotateRoot=false;

//	if (IsDlgButtonChecked(hwnd,IDC_COMPTIME))
		// Always on now
		animoptdata.bCompressTime=true;
//	else
//		animoptdata.bCompressTime=false;
}

void AnimOptions2Dlg::SetOptions(AnimOptionData* data)
{
	animoptdata=*data;
	SetOptions();
}

void AnimOptions2Dlg::SetOptions()
{
	SetDlgItemText(hwnd,IDC_EDITNAME,animoptdata.modelname);
	SetDlgItemText(hwndParent,IDC_MODELNAME,animoptdata.modelname);

	if (strlen(animoptdata.suffixname)>0)
	{
		CheckDlgButton(hwnd,IDC_USEQNSUFFIX,BST_CHECKED);
		EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),TRUE);
		SetDlgItemText(hwnd,IDC_EDITSUFFIX,animoptdata.suffixname);
	}
	else
	{
		CheckDlgButton(hwnd,IDC_USEQNSUFFIX,BST_UNCHECKED);
		EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),FALSE);
		SetDlgItemText(hwnd,IDC_EDITSUFFIX,"");
	}

	if (strlen(animoptdata.dirname)>0)
		SetDlgItemText(hwnd,IDC_DIRNAME,animoptdata.dirname);

	if (strlen(animoptdata.skeletonName)>0)
		SetDlgItemText(hwnd,IDC_SKELNAME,animoptdata.skeletonName);

	if (animoptdata.bExportQN)
		CheckDlgButton(hwnd,IDC_EXPORTQN,BST_CHECKED);
	else
	{
		CheckDlgButton(hwnd,IDC_EXPORTQN,BST_UNCHECKED);
		EnableWindow(GetDlgItem(hwnd,IDC_USEQNSUFFIX),FALSE);
		EnableWindow(GetDlgItem(hwnd,IDC_EDITSUFFIX),FALSE);
	}

	if (animoptdata.bUseCompression)
		CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_CHECKED);
	else
		CheckDlgButton(hwnd,IDC_USECOMPRESSION,BST_UNCHECKED);

	if (animoptdata.bCompressDoubles)
		CheckDlgButton(hwnd,IDC_COMPDBLS,BST_CHECKED);
	else
		CheckDlgButton(hwnd,IDC_COMPDBLS,BST_UNCHECKED);

	if (animoptdata.bOutputDebug)
		CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_CHECKED);
	else
		CheckDlgButton(hwnd,IDC_OUTPUTDBG,BST_UNCHECKED);

//	if (animoptdata.bRotateRoot)
//		CheckDlgButton(hwnd,IDC_PREROTATE,BST_CHECKED);
//	else
//		CheckDlgButton(hwnd,IDC_PREROTATE,BST_UNCHECKED);

//	if (animoptdata.bCompressTime)
//		CheckDlgButton(hwnd,IDC_COMPTIME,BST_CHECKED);
//	else
//		CheckDlgButton(hwnd,IDC_COMPTIME,BST_UNCHECKED);
}

// Main Animation Options Dialog ////////////////////////////////////////////////////////////

AnimOptionsDlg::AnimOptionsDlg(HINSTANCE hInstance,HWND hwndParent,Interface* ip) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_ANIMOPTIONS),hwndParent)
{
	this->ip=ip;
	Show();

	bUpdated = false;

	listview=new ListView(GetDlgItem(hwnd,IDC_LIST1));
	listview->AddColumn("Animation Name",120);
	listview->AddColumn("Start",40);
	listview->AddColumn("End",40);
	listview->AddColumn("Rot Tolerance",60);
	listview->AddColumn("Tran Tolerance",60);
	listview->AddColumn("File Size",54);
	listview->AddColumn("Active",25);

	// Load cursors
	//cursorStd=LoadCursor(hInstance,IDC_ARROW);
	
	cursorMove=LoadCursor(NULL,IDC_HAND);

	CheckDlgButton(hwnd,IDC_EXPORTALL,BST_CHECKED);
	bLockUpdate=false;
	bDrag=false;

	optionsDlg=new AnimOptions2Dlg(hInstance,hwnd);

	// Initialize Property lists
	pTolList = new PropList(hInstance,1);
	pTolList->Attach(GetDlgItem(hwnd,IDC_TOLLIST));
	pTolList->HasApply(FALSE);
	pTolList->SetFocusCB(TolFocusCB,this);
	pTolList->SetChangeCB(TolChangeCB,this);

	pTolListTran = new PropList(hInstance,1);
	pTolListTran->Attach(GetDlgItem(hwnd,IDC_TOLLISTTRAN));
	pTolListTran->HasApply(FALSE);
	pTolListTran->SetFocusCB(TolFocusTranCB,this);
	pTolListTran->SetChangeCB(TolChangeTranCB,this);

	lastTolID = -1;
	lastTolIDTran = -1;

	// Test
	/*
	for(int i=0;i<40;i++)
	{
		char name[256];
		sprintf(name,"Item #%i",i);
		//pTolList->AddEdit(name);
		pTolList->AddSpinEdit(name,0.0f,100.0f);
		pTolListTran->AddSpinEdit(name,0.0f,100.0f);
	}

	pTolList->BuildUI();
	pTolListTran->BuildUI();
	*/

	LoadFromScene();

	IEditRot  = GetICustEdit(GetDlgItem(hwnd,IDC_COMPRESSION));
	IEditTran = GetICustEdit(GetDlgItem(hwnd,IDC_TCOMPRESSION));

	IEditEnd   = GetICustEdit(GetDlgItem(hwnd,IDC_EDITEND));
	IEditStart = GetICustEdit(GetDlgItem(hwnd,IDC_EDITSTART));
	IEditAnim  = GetICustEdit(GetDlgItem(hwnd,IDC_EDITANIM));

	ISpinRot  = GetISpinner(GetDlgItem(hwnd,IDC_ROTSPIN));
	ISpinTran = GetISpinner(GetDlgItem(hwnd,IDC_TRANSPIN));

	ISpinStart = GetISpinner(GetDlgItem(hwnd,IDC_STARTSPIN));
	ISpinEnd   = GetISpinner(GetDlgItem(hwnd,IDC_ENDSPIN));

	ISpinRot->LinkToEdit(GetDlgItem(hwnd,IDC_COMPRESSION),EDITTYPE_FLOAT);
	ISpinTran->LinkToEdit(GetDlgItem(hwnd,IDC_TCOMPRESSION),EDITTYPE_FLOAT);

	ISpinStart->LinkToEdit(GetDlgItem(hwnd,IDC_EDITSTART),EDITTYPE_INT);
	ISpinEnd->LinkToEdit(GetDlgItem(hwnd,IDC_EDITEND),EDITTYPE_INT);

	//Interval interval = ip->GetAnimRange();

	ISpinRot->SetLimits(0.0f,1.0f);
	ISpinTran->SetLimits(-10.000000f,10.000000f);

	//ISpinStart->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());
	//ISpinEnd->SetLimits(interval.Start()/GetTicksPerFrame(),interval.End()/GetTicksPerFrame());

	ISpinStart->SetLimits(0,999999);
	ISpinEnd->SetLimits(0,999999);

	ISpinRot->SetScale(0.01f);
	ISpinTran->SetScale(0.000001f);

	SendDlgItemMessage(hwnd,IDC_BONESEL,CB_RESETCONTENT,0,0);
	AddBones(ip->GetRootNode());
	SendDlgItemMessage(hwnd,IDC_BONESEL,CB_SETCURSEL,0,0);

	CheckRadioButton(hwnd,IDC_PRESET,IDC_MANUAL,IDC_PRESET);
	CheckRadioButton(hwnd,IDC_PRESETTRANS,IDC_MANUALTRANS,IDC_PRESETTRANS);

	LoadTolerances();
	UpdateTolerances();

	progBar=NULL;
}

AnimOptionsDlg::~AnimOptionsDlg()
{
	Hide();

	if (ISpinRot)
		ReleaseISpinner(ISpinRot);

	if (ISpinTran)
		ReleaseISpinner(ISpinTran);

	if (ISpinStart)
		ReleaseISpinner(ISpinStart);

	if (ISpinEnd)
		ReleaseISpinner(ISpinEnd);

	if (IEditRot)
		ReleaseICustEdit(IEditRot);

	if (IEditTran)
		ReleaseICustEdit(IEditTran);

	if (IEditEnd)
		ReleaseICustEdit(IEditEnd);

	if (IEditStart)
		ReleaseICustEdit(IEditStart);

	if (IEditAnim)
		ReleaseICustEdit(IEditAnim);

	if (listview)
		delete listview;

	if (optionsDlg)
		delete optionsDlg;

	if (progBar)
		delete progBar;

	if (pTolList)
		delete pTolList;
}

void AnimOptionsDlg::AssignTime(const int ctrlID)
{
	CStr strTime;
	TimeValue t=ip->GetTime();
	TimeToString(t,strTime);

	SetDlgItemText(hwnd,ctrlID,(char*)strTime);
}

BOOL AnimOptionsDlg::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	LPNMHDR nmhdr;
	int     index;
	POINT   pt;

	if (bDrag)
	{
		SendDlgItemMessage(hwnd,IDC_LIST1,msg,wParam,lParam);
	}

	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();
		
		return TRUE;

	case WM_LBUTTONUP:
		if (bDrag)
		{
			GetCursorPos(&pt);
			ScreenToClient(GetDlgItem(hwnd,IDC_LIST1),&pt);

			index=listview->HitTest(pt.x,pt.y);
			MoveItems(index);

			// Completion of drag event
			ReleaseCapture();
			bDrag=false;
		}

		return TRUE;

	case WM_MOUSEMOVE:
		if (bDrag)
		{
			// Forward message to child if dragging to get colored select
			//SendDlgItemMessage(hwnd,IDC_LIST1,WM_MOUSEMOVE,wParam,lParam);
		}
		return TRUE;

	case WM_NOTIFY:
		nmhdr=(LPNMHDR)lParam;

		switch(nmhdr->code)
		{
		case LVN_ITEMACTIVATE:
			GoToAnim();
			return TRUE;

		case HDN_ITEMCLICK:
			FieldSort(((LPNMHEADER)nmhdr)->iItem);
			return TRUE;

		case LVN_ITEMCHANGED:
			SelChange((LPNMLISTVIEW)lParam);
			EnableUI(true);
			return TRUE;

		// Always allow inlist editing of the labels
		case LVN_BEGINLABELEDIT:
			return FALSE;

		case LVN_ENDLABELEDIT:
			UpdateLabelEdit((NMLVDISPINFO*)lParam);
			return TRUE;

		case LVN_BEGINDRAG:
			SetCursor(cursorMove);
			SetCapture(hwnd);
			bDrag=true;
		}
		return FALSE;

	case CC_SPINNER_CHANGE:
		switch(LOWORD(wParam))
		{
		case IDC_ROTSPIN:
			SliderUpdate(IEditRot->GetFloat(),3);
			return TRUE;

		case IDC_TRANSPIN:
			SliderUpdate(IEditTran->GetFloat(),4);
			return TRUE;

		case IDC_STARTSPIN:
			UpdateStart();
			return TRUE;

		case IDC_ENDSPIN:
			UpdateEnd();
			return TRUE;
		}
		return FALSE;

	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_PRESETLIST:
				UpdatePresetSel();
			}
			return TRUE;

		case EN_CHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_COMPRESSION:
				SliderUpdate(IEditRot->GetFloat(),3);
				return TRUE;

			case IDC_TCOMPRESSION:
				SliderUpdate(IEditTran->GetFloat(),4);
				return TRUE;

			case IDC_EDITANIM:
				UpdateName();
				return TRUE;

			case IDC_EDITSTART:
				UpdateStart();
				return TRUE;

			case IDC_EDITEND:
				UpdateEnd();
				return TRUE;
			}

			return TRUE;
		}

		switch(LOWORD(wParam))
		{
		case IDC_ADDALLBONES:
			AddAllBones();
			return TRUE;

		case IDC_REMOVEALLBONES:
			RemoveAllBones();
			return TRUE;
		
		case IDC_ADDBONE:
			AddBone();
			return TRUE;

		case IDC_REMOVEBONE:
			RemoveBone();
			return TRUE;

		case IDC_PRESET:
			SetPresetTol();
			return TRUE;

		case IDC_MANUAL:
			SetManualTol();
			return TRUE;

		case IDC_PRESETTRANS:
			SetPresetTolTran();
			return TRUE;

		case IDC_MANUALTRANS:
			SetManualTolTran();
			return TRUE;

		case IDC_ASSIGNSTART:
			AssignTime(IDC_EDITSTART);
			return TRUE;

		case IDC_ASSIGNEND:
			AssignTime(IDC_EDITEND);
			return TRUE;

		case IDC_NEWANIM:
			NewAnimation();
			return TRUE;

		case IDC_REMOVESEL:
			RemoveSelection();
			return TRUE;

		case IDC_EXPORTRANGE:
			ExportRange();
			return TRUE;

		case IDC_IMPORTRANGE:
			ImportRange();
			return TRUE;

		case IDC_EXPORT:
			Export();
			return TRUE;

		case IDC_ACTIVEANIM:
			ToggleActive();
			return TRUE;

		case IDC_OPTIONS:
			optionsDlg->Show();
			optionsDlg->SetOptions();
			return TRUE;

		case IDC_FINDANIM:
			FindAnim();
			return TRUE;

		case IDC_NETVIEW:
		case IDC_REALTIME:
			{
				int index = listview->GetSel();
				NMLISTVIEW nmlv;
				ZeroMemory(&nmlv,sizeof(NMLISTVIEW));

				nmlv.iItem = index;
				nmlv.uNewState |= LVIS_SELECTED;
				
				SelChange(&nmlv);
			}
			return TRUE;

		}
		return TRUE;

	case WM_CLOSE:
		if (MaybeSaveAnr()==ANRCANCEL)
			return FALSE;

		SaveToScene();
		Hide();
		return TRUE;
	}

	return FALSE;
}

void AnimOptionsDlg::SelChange(LPNMLISTVIEW item)
{
	char buf[256];
	int index=item->iItem;
	
	if (!(item->uNewState & LVIS_SELECTED))
	{
		// Verify animation range
		char strVal[64];
		int  valStart,valEnd;
		listview->GetItem(index,strVal,64,1);
		valStart=atoi(strVal);
		listview->GetItem(index,strVal,64,2);
		valEnd=atoi(strVal);

		if (valEnd<valStart)
		{
			// Inform and don't do any network update if the info is invalid
			MessageBox(ip->GetMAXHWnd(),"The end frame for this animation is less than the start frame.","Invalid animation Range",MB_ICONWARNING|MB_OK);
			return;
		}

		return;
	}

	bLockUpdate=true;

	// Assign edit box content
	listview->GetItem(index,buf,256,0);
	//SetDlgItemText(hwnd,IDC_EDITANIM,buf);
	IEditAnim->SetText(buf);

	listview->GetItem(index,buf,256,1);
	//SetDlgItemText(hwnd,IDC_EDITSTART,buf);
	IEditStart->SetText(buf);

	listview->GetItem(index,buf,256,2);
	//SetDlgItemText(hwnd,IDC_EDITEND,buf);
	IEditEnd->SetText(buf);

	// Assign rotation slider content
	listview->GetItem(index,buf,256,3);
	ISpinRot->SetValue((float)atof(buf),FALSE);
	IEditRot->SetText(buf);

	// Assign translation slider content
	listview->GetItem(index,buf,256,4);
	ISpinTran->SetValue((float)atof(buf),FALSE);
	IEditTran->SetText(buf);

	// Assign Active checkbox
	listview->GetItem(index,buf,256,6);
	if (atoi(buf)==1)
		CheckDlgButton(hwnd,IDC_ACTIVEANIM,BST_CHECKED);
	else
		CheckDlgButton(hwnd,IDC_ACTIVEANIM,BST_UNCHECKED);

	UpdateTolerances();

	bLockUpdate=false;

	// Verify animation range
	char strVal[64];
	int  valStart,valEnd;
	listview->GetItem(index,strVal,64,1);
	valStart=atoi(strVal);
	listview->GetItem(index,strVal,64,2);
	valEnd=atoi(strVal);

	if (valEnd<valStart)
	{
		// Inform and don't do any network update if the info is invalid
		MessageBox(ip->GetMAXHWnd(),"The end frame for this animation is less than the start frame.","Invalid animation Range",MB_ICONWARNING|MB_OK);
		return;
	}

	// Update via network if appropriate
	if (IsDlgButtonChecked(hwnd,IDC_NETVIEW)==BST_CHECKED)
	{
		char buf[256];
		Net::MsgViewObjSetAnim msg;

		listview->GetItem(index,buf,256);
		msg.m_AnimName=GenerateCRC(buf);

		gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_ANIM, sizeof( Net::MsgViewObjSetAnim ),
										 &msg );
	}

	if (IsDlgButtonChecked(hwnd,IDC_REALTIME)==BST_CHECKED)
	{
		char buf[256];
		Net::MsgViewObjSetAnimFile msg;

		listview->GetItem(index,buf,256);
		CStr name=buf;
		CStr strEnv=getenv(APP_ENV);

		if (strEnv==CStr(""))
		{
			char strErr[256];
			sprintf(strErr,"Your '%s' environment variable is not set.  Cannot continue.",APP_ENV);
			MessageBox(hwnd,strErr,"Environment variable not set",MB_ICONWARNING|MB_OK);
			return;
		}

		animopt=optionsDlg->GetOptions();
		//CStr FullFilename=strEnv+CStr(ANIM_PATH)+CStr(animopt.modelname)+CStr("\\")+CStr(animopt.modelname)+CStr("_")+name+CStr(".ska");
		//CStr PartFilename=CStr(QNANIM_PATH)+CStr(animopt.modelname)+CStr("\\")+CStr(animopt.modelname)+CStr("_")+name+CStr(".ska");

		//CStr FullFilename=strEnv+CStr(ANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+CStr(animopt.modelname)+CStr("_")+name+CStr(".ska");
		//CStr PartFilename=CStr(QNANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+CStr(animopt.modelname)+CStr("_")+name+CStr(".ska");

		CStr FullFilename=strEnv+CStr(ANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+name+CStr(".ska");
		CStr PartFilename=CStr(QNANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+name+CStr(".ska");

		assert(PartFilename.Length()<128);

		strcpy(msg.m_Filename,(char*)PartFilename);
		msg.m_checksum=GenerateCRC(name);
		//msg.m_checksum=GetChecksum(FullFilename);

		gClient->EnqueueMessageToServer( Net::vMSG_ID_VIEWOBJ_SET_ANIM_FILE, sizeof( Net::MsgViewObjSetAnimFile ),
			                             &msg );
	}
}

void AnimOptionsDlg::EnableUI(bool bActive)
{
	EnableWindow(GetDlgItem(hwnd,IDC_REMOVESEL),bActive);
	EnableWindow(GetDlgItem(hwnd,IDC_EDITANIM),bActive);
	EnableWindow(GetDlgItem(hwnd,IDC_EDITSTART),bActive);
	EnableWindow(GetDlgItem(hwnd,IDC_EDITEND),bActive);
}

void AnimOptionsDlg::NewAnimation()
{
	int index;

	index = listview->AddItem("NewItem");
	listview->AddSubItem("0");
	listview->AddSubItem("0");
	listview->AddSubItem(DEFAULT_COMPRESS_STRING);
	EnableUI(true);

	Link<AnimListData>* tdata = TolDB.Add();

	tdata->data.animName = "NewItem";

	// Reset preset selection
	SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_SETCURSEL,(WPARAM)-1,0);

	listview->SetItemData(index,(DWORD)tdata);
}

void AnimOptionsDlg::RemoveSelection()
{
	int count=listview->GetCount();

	for(int i=0;i<count;i++)
	{
		if (listview->IsSelected(i))
		{
			Link<AnimListData>* link = (Link<AnimListData>*)listview->GetItemData(i);
			TolDB.Remove(link);
			listview->DeleteItem(i);
			i=-1;
		}

		count=listview->GetCount();
	}
	
	EnableUI(false);
}

bool AnimOptionsDlg::ExportRange(char* file)
{
	OPENFILENAME ofn;
	char filename[2048]="";

	if (!file)
	{
		if (anrfile.Length()>0 && anrfile.Length()<2047)
			strcpy(filename,anrfile);

		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=hwnd;
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Animation Ranges (*.anr)\0*.anr\0Text Files (*.txt)\0*.txt\0All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=0;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Export Animation Ranges";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".anr");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetSaveFileName(&ofn);

		// Exit if cancelled
		if (strlen(filename)==0)
			return false;
	}
	else
		strcpy(filename,file);

	FILE* fp=fopen(filename,"w");
	
	if (!fp)
	{
		char ErrMsg[256];
		sprintf(ErrMsg,"Couldn't write file '%s'",filename);
		MessageBox(hwnd,ErrMsg,"Export Animation Ranges",MB_ICONWARNING|MB_OK);
		return false;
	}

	animopt=optionsDlg->GetOptions();

	// Write this out as an animation version 5 file
	//fprintf(fp,"AnV4\n");
	//fprintf(fp,"AnV5\n");
	fprintf(fp,"AnV6\n");
	
	if (strlen(animopt.modelname)>0)
		fprintf(fp,"%s\n",animopt.modelname);
	else
		fprintf(fp,"[none]\n");

	if (strlen(animopt.suffixname)>0)
		fprintf(fp,"%s\n",animopt.suffixname);
	else
		fprintf(fp,"[none]\n");

	if (strlen(animopt.dirname)>0)
		fprintf(fp,"%s\n",animopt.dirname);
	else
		fprintf(fp,"[none]\n");

	if (strlen(animopt.skeletonName)>0)
		fprintf(fp,"%s\n",animopt.skeletonName);
	else
		fprintf(fp,"[none]\n");

	char  name[256];
	char  valStart[256];
	char  valEnd[256];
	char  valCompress[256];
	char  valTCompress[256];
	char  valActive[80];
	int   ivalStart,ivalEnd;
	int   ivalActive;
	int   index=-1;
	float fCompress;
	float fTCompress;

	int  numItems=listview->GetCount();

	for(int i=0;i<numItems;i++)
	{
//		if (!IsDlgButtonChecked(hwnd,IDC_EXPORTSEL) ||
//			(IsDlgButtonChecked(hwnd,IDC_EXPORTSEL) && listview->IsSelected(i)))
		{
			listview->GetItem(i,name,256);
			listview->GetItem(i,valStart,256,1);
			listview->GetItem(i,valEnd,256,2);
			listview->GetItem(i,valCompress,256,3);
			listview->GetItem(i,valTCompress,256,4);
			listview->GetItem(i,valActive,256,6);

			ivalStart=atoi(valStart);
			ivalEnd=atoi(valEnd);
			ivalActive=atoi(valActive);
			fCompress=0.999700f+atof(valCompress)*0.000300f;	// Convert to actual val
			fTCompress=atof(valTCompress);

			fprintf(fp,"%s %i %i %f %f %i\n",name,ivalStart,ivalEnd,fCompress,fTCompress,ivalActive);
			Link<TolData>* link = (Link<TolData>*)listview->GetItemData(i);
			link->data.Save(fp);
		}
	}

	fclose(fp);
	return true;
}

bool AnimOptionsDlg::ImportRange(char* file)
{
	OPENFILENAME ofn;
	char filename[2048]="";

	if (!file)
	{
		if (anrfile.Length()>0 && anrfile.Length()<2047)
			strcpy(filename,anrfile);

		// Inform the user that this will clear the current list
		int rVal=MessageBox(hwnd,"This operation will clear the current animation list.\nAre you sure?","Import Animation Ranges",MB_YESNO|MB_ICONQUESTION);
		if (rVal==IDNO)
			return false;

		// Pop open a file dialog
		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=hwnd;
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Animation Ranges (*.anr)\0*.anr\0Text Files (*.txt)\0*.txt\0All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=0;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Export Animation Ranges";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".anr");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetOpenFileName(&ofn);

		// Exit if cancelled
		if (strlen(filename)==0)
			return false;
	}
	else
		strcpy(filename,file);

	FILE* fp=fopen(filename,"r");

	if (!fp)
	{
		char ErrMsg[256];
		sprintf(ErrMsg,"Couldn't read file '%s'",filename);
		MessageBox(hwnd,ErrMsg,"Import Animation Ranges",MB_ICONWARNING|MB_OK);
		return false;
	}

	anrfile = filename;
	char bufTitle[256];
	sprintf(bufTitle,"Animation Exporter - [%s]",(char*)anrfile);
	SetWindowText(hwnd,bufTitle);

	// Clear out tolerance database
	TolDB.Clear();

	char verID[6];
	verID[0]=fgetc(fp);
	verID[1]=fgetc(fp);
	verID[2]=fgetc(fp);
	verID[3]=fgetc(fp);
	verID[4]=fgetc(fp);	// Should be CR

	if (optionsDlg)
		animopt = optionsDlg->GetOptions();

	// Check if this is an animation range version 2 format file
	// TODO: Switch ID string to check by integer version
	if (strstr(verID,"AnV2") || strstr(verID,"AnV3") || strstr(verID,"AnV4") || strstr(verID,"AnV5") || strstr(verID,"AnV6"))
	{
		char modelname[80];
		char suffixname[80];

		fscanf(fp,"%s\n%s\n",modelname,suffixname);

		strcpy(animopt.modelname, modelname);
		
		if (strcmp(suffixname,"[none]")==0)
			animopt.suffixname[0]='\0';
		else
			strcpy(animopt.suffixname, suffixname);

		if (strstr(verID,"AnV3") || strstr(verID,"AnV4") || strstr(verID,"AnV5") || strstr(verID,"AnV6"))
		{
			char dirname[80];
			fscanf(fp,"%s\n",dirname);

			strcpy(animopt.dirname, dirname);
		}
		else
			strcpy(animopt.dirname,"");

		if (strstr(verID,"AnV4") || strstr(verID,"AnV5") || strstr(verID,"AnV6"))
		{
			char skeletonName[80];
			fscanf(fp,"%s\n",skeletonName);

			strcpy(animopt.skeletonName,skeletonName);
		}
		else
			strcpy(animopt.skeletonName,"");

		optionsDlg->SetOptions(&animopt);
	}
	else
	{
		// If this isn't an animation range version 2 or 3 file return to top of file
		fseek(fp,0,SEEK_SET);
	}

	TolDB.Clear();
	listview->Clear();

	char  name[256];
	char  valStart[80];
	char  valEnd[80];
	char  valCompress[80];
	char  valTCompress[80];
	char  valActive[80];
	int   ivalStart,ivalEnd;
	int   iActive;					// 1 if the animation is active (0 if not)
	float fCompress=0.0f;
	float fTCompress=0.0f;

	// Scan in each animation
	while(!feof(fp))
	{
		Link<AnimListData>* tdata = TolDB.Add();
		tdata->data.toldata.Reset();

		if (strstr(verID,"AnV6"))
		{
			if (fscanf(fp,"%s %i %i %f %f %i\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress,&iActive) != 6)
				break;			
		}
		else
		{
			if (fscanf(fp,"%s %i %i %f %f\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress) != 5)
				break;

			iActive=1;
		}

		if (strstr(verID,"AnV5") || strstr(verID,"AnV6"))
		{
			tdata->data.toldata.Load(fp);
		}

		_itoa(ivalStart,valStart,10);
		_itoa(ivalEnd,valEnd,10);
		_itoa(iActive,valActive,10);

		sprintf(valCompress,"%f",fCompress);
		sprintf(valTCompress,"%f",fTCompress);

		int index = listview->AddItem(name);
		listview->AddSubItem(valStart);
		listview->AddSubItem(valEnd);

		tdata->data.animName = name;
		tdata->data.start    = ivalStart;
		tdata->data.end      = ivalEnd;
		tdata->data.rotTol   = fCompress;
		tdata->data.tranTol  = fTCompress;
		tdata->data.active   = (iActive==1) ? true : false;

		// Add extended tolerance data
		listview->SetItemData(index,(DWORD)tdata);

		if (fCompress==0.0f)
			listview->AddSubItem(DEFAULT_COMPRESS_STRING);		// To maintain backward compat.
		else
		{
			// Compute scaled compress val
			fCompress=(fCompress-0.999700f)/0.000300f;
			sprintf(valCompress,"%f",fCompress);
			listview->AddSubItem(valCompress);
		}

		if (fTCompress==0.0f)
			listview->AddSubItem(DEFAULT_TCOMPRESS_STRING);		// To maintain backward compat
		else
			listview->AddSubItem(valTCompress);

		listview->AddSubItem(valActive,6);
	}

	fclose(fp);
	bUpdated = false;
	return true;
}

void AnimOptionsDlg::SaveToScene()
{
	// From now on, we'll just save the name of the anr file instead
	ReferenceTarget* scene=ip->GetScenePointer();
	
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_ANR_FILE_ID);
	char* anrChunk = (char*)malloc(anrfile.Length()+1);
	strcpy(anrChunk,(char*)anrfile);

	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_ANR_FILE_ID,strlen(anrChunk)+1,anrChunk);

	/*
	AnimOptionData* animoptdata=(AnimOptionData*)malloc(sizeof(AnimOptionData));
	*animoptdata=optionsDlg->GetOptions();

	ReferenceTarget* scene=ip->GetScenePointer();

	// Export the options chunk
	animoptdata->version=ANIMOPTDATA_VER;
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_OPTIONS_ID);
	scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_OPTIONS_ID,sizeof(AnimOptionData),animoptdata);

	// Build up the animation data
	CStr  strAnim;
	char  strBuf[256];
	char  name[256];
	char  valStart[256];
	char  valEnd[256];
	char  valCompress[256];
	char  valTCompress[256];
	int   ivalStart,ivalEnd;
	int   index=-1;
	float fCompress;
	float fTCompress;

	int  numItems=listview->GetCount();

	for(int i=0;i<numItems;i++)
	{
		listview->GetItem(i,name,256);
		listview->GetItem(i,valStart,256,1);
		listview->GetItem(i,valEnd,256,2);
		listview->GetItem(i,valCompress,256,3);
		listview->GetItem(i,valTCompress,256,4);

		ivalStart=atoi(valStart);
		ivalEnd=atoi(valEnd);
		fCompress=atof(valCompress);
		fCompress=0.999700f+atof(valCompress)*0.000300f;	// Convert to actual val
		fTCompress=atof(valCompress);

		sprintf(strBuf,"%s %i %i %f %f\n",name,ivalStart,ivalEnd,fCompress,fTCompress);
		strAnim+=strBuf;
	}
	
	if (strAnim.Length()>0)
	{
		char* AnimData=(char*)malloc(strAnim.Length()+1);
		strcpy(AnimData,(char*)strAnim);

		// Export the animations to the scene
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_ANIMATIONS_ID);
		scene->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_ANIMATIONS_ID,strlen(AnimData)+1,AnimData);
	}
	*/
}

bool AnimOptionsDlg::LoadFromScene()
{
	AppDataChunk*   appdata;

	ReferenceTarget* scene = ip->GetScenePointer();

	appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_ANR_FILE_ID);

	if (!appdata || !appdata->data)
		return false;

	ImportRange((char*)appdata->data);

	/*
	AppDataChunk*   appdata;

	ReferenceTarget* scene=ip->GetScenePointer();
	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_OPTIONS_ID);

	if (!appdata)
		return false;

	// Display the model name
	SetDlgItemText(hwnd,IDC_MODELNAME,((AnimOptionData*)appdata->data)->modelname);

	if (optionsDlg)
	{
		AnimOptionData* opts=(AnimOptionData*)appdata->data;


		if (opts->version==ANIMOPTDATA_VER)
			optionsDlg->SetOptions((AnimOptionData*)appdata->data);
		else
		{
			// Using old version before version stamping
			AnimOptionData aniopts;
			char*    pOpts = (char*)&aniopts;

			if (opts->version<0x0003)
			{
				aniopts.bCompressTime=false;
				aniopts.bRotateRoot=false;
			}

			if (opts->version>ANIMOPTDATA_VER || opts->version==0)
				pOpts+=sizeof(WORD);				// Skip version
				
			aniopts.version=ANIMOPTDATA_VER;
			memcpy(pOpts,appdata->data,appdata->length);
			strcpy(aniopts.dirname, aniopts.modelname);

			optionsDlg->SetOptions(&aniopts);
		}
	}

	appdata=scene->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_ANIM_EXPORT_ANIMATIONS_ID);
	
	if (!appdata)
		return false;

	int len=appdata->length;

	for(int i=0;i<listview->GetCount();i++)
	{
		Link<AnimListData>* link = (Link<AnimListData>*)listview->GetItemData(i);
		TolDB.Remove(link);
	}
	listview->Clear();

	char* AnimData=(char*)appdata->data;

	char  name[256];
	char  valStart[256];
	char  valEnd[256];
	char  valCompress[256];
	char  valTCompress[256];
	int   ivalStart,ivalEnd;
	float fCompress=0.0f;
	float fTCompress=0.0f;

	int numItems=CountChar(AnimData,'\n');

	// Clear out tolerance database
	TolDB.Clear();

	// Scan in each animation
	for(i=0;i<numItems;i++)
	{
		sscanf(AnimData,"%s %i %i %f %f\n",name,&ivalStart,&ivalEnd,&fCompress,&fTCompress);

		_itoa(ivalStart,valStart,10);
		_itoa(ivalEnd,valEnd,10);
		sprintf(valCompress,"%f",fCompress);
		sprintf(valTCompress,"%f",fTCompress);

		int index = listview->AddItem(name);
		listview->AddSubItem(valStart);
		listview->AddSubItem(valEnd);

		// Add extended tolerance data
		Link<AnimListData>* tdata = TolDB.Add();
		listview->SetItemData(index,(DWORD)tdata);

		tdata->data.animName = name;
		tdata->data.start    = ivalStart;
		tdata->data.end      = ivalEnd;
		tdata->data.rotTol   = fCompress;
		tdata->data.tranTol  = fTCompress;
		//tdata->data.active  = (iActive==1) ? true : false;

		if (fCompress==0.0f)
			listview->AddSubItem(DEFAULT_COMPRESS_STRING);		// To maintain backward compat.
		else
		{
			// Compute scaled compress val
			fCompress=(fCompress-0.999700f)/0.000300f;
			sprintf(valCompress,"%f",fCompress);

			listview->AddSubItem(valCompress);
		}

		if (fTCompress==0.0f)
			listview->AddSubItem(DEFAULT_TCOMPRESS_STRING);		// To maintain backward compat.
		else
			listview->AddSubItem(valTCompress);

		if (!IsInstr(AnimData,"\n"))
			return false;

		AnimData+=Instr(AnimData,"\n")+1;
	}
	*/

	return true;
}


void AnimOptionsDlg::UpdateName()
{
	if (bLockUpdate)
		return;

	bUpdated = true;

	char txtBuf[256];
	int index=listview->GetSel();
	//GetDlgItemText(hwnd,IDC_EDITANIM,txtBuf,256);
	IEditAnim->GetText(txtBuf,256);

	CStr strText=txtBuf;

	// Don't allow first character to be number
	char check[2];
	check[0]=txtBuf[0];
	check[1]='\0';

	if (HasNum(check))
	{
		strText=CStr("_")+strText;
		//SetDlgItemText(hwnd,IDC_EDITANIM,strText);
		strText=ReplaceStr(strText,"\t","");
		IEditAnim->SetText(strText);

		// Advance cursor to end of line
		int buflen=strlen(strText);
		SendDlgItemMessage(hwnd,IDC_EDITANIM,EM_SETSEL,(WPARAM)buflen,(LPARAM)buflen);
	}

	// Don't allow spaces or tabs
	if (IsInstr(txtBuf," ") || IsInstr(txtBuf,"\t"))
	{
		strText=ReplaceStr(strText," ","");
		strText=ReplaceStr(strText,"\t","");

		//SetDlgItemText(hwnd,IDC_EDITANIM,strText);
		IEditAnim->SetText(strText);
	}

	listview->AddSubItem(strText,0,index);

	Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(index);
	tdata->data.animName = strText;
}

void AnimOptionsDlg::UpdateStart()
{
	bUpdated = true;

	if (bLockUpdate)
		return;
	
	char txtBuf[256];
	int index=listview->GetSel();
	//GetDlgItemText(hwnd,IDC_EDITSTART,txtBuf,256);
	IEditStart->GetText(txtBuf,256);

	// Convert to value
	int val=atoi(txtBuf);
	_itoa(val,txtBuf,10);

	//SetDlgItemText(hwnd,IDC_EDITSTART,txtBuf);
	//IEditStart->SetText(txtBuf);

	// Advance cursor to end of line
	int buflen=strlen(txtBuf);
	SendDlgItemMessage(hwnd,IDC_EDITSTART,EM_SETSEL,(WPARAM)buflen,(LPARAM)buflen);

	listview->AddSubItem(txtBuf,1,index);

	Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(index);
	tdata->data.start = val;
}

void AnimOptionsDlg::UpdateEnd()
{
	bUpdated = true;

	if (bLockUpdate)
		return;

	char txtBuf[256];
	int index=listview->GetSel();
	//GetDlgItemText(hwnd,IDC_EDITEND,txtBuf,256);
	IEditEnd->GetText(txtBuf,256);

	// Convert to value
	int val=atoi(txtBuf);
	_itoa(val,txtBuf,10);

	//SetDlgItemText(hwnd,IDC_EDITEND,txtBuf);
	//IEditEnd->SetText(txtBuf);

	// Advance cursor to end of line
	int buflen=strlen(txtBuf);
	SendDlgItemMessage(hwnd,IDC_EDITEND,EM_SETSEL,(WPARAM)buflen,(LPARAM)buflen);

	listview->AddSubItem(txtBuf,2,index);

	Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(index);
	tdata->data.end = val;
}

void AnimOptionsDlg::UpdateLabelEdit(NMLVDISPINFO* dispinfo)
{
	bUpdated = true;
	bLockUpdate=true;

	int index=dispinfo->item.iItem;

	if (dispinfo->item.pszText)
	{
		listview->AddSubItem(dispinfo->item.pszText,0,index);
		//SetDlgItemText(hwnd,IDC_EDITANIM,dispinfo->item.pszText);
		IEditAnim->SetText(dispinfo->item.pszText);
	}

	Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(index);
	tdata->data.animName = dispinfo->item.pszText;

	bLockUpdate=false;
}

void AnimOptionsDlg::GoToAnim()
{
	char valStart[256];
	char valEnd[256];

	int ivalStart,ivalEnd;

	// Construct an interval for the animation
	int index=listview->GetSel();

	listview->GetItem(index,valStart,256,1);
	listview->GetItem(index,valEnd,256,2);

	ivalStart=atoi(valStart);
	ivalEnd=atoi(valEnd);

	if (ivalEnd==0)
		return;

	if (ivalStart>ivalEnd)
		return;

	Interval theInterval( ivalStart * GetTicksPerFrame(), ivalEnd * GetTicksPerFrame() );
	ip->SetAnimRange( theInterval );
	ip->SetTime( ivalStart * GetTicksPerFrame() );
}

void AnimOptionsDlg::MoveItems(int index)
{
	char     name[256];
	char     valStart[256];
	char     valEnd[256];
	char     valCompress[256];
	char     valTCompress[256];
	DWORD    itemData;
	Tab<int> selList;

	selList.ZeroCount();

	int count=listview->GetCount();
	int row;
	int selcount=listview->GetSelCount();

	// Abort if nothing selected to move
	if (selcount==0)
		return;

	// Determine direction
	// if the first item is moving up in the list search in reverse
	// if the first item is moving down in the list search forward
	for(int i=0;i<count;i++)
	{
		if (listview->IsSelected(i))
			break;
	}

	// Can't move to same position
	if (i==index)
		return;

	// Move isn't a valid index
	if (index<0)
		return;

	// We're moving down in the list
	if (i<index)
	{
		// search forward
		for(i=0;i<count;i++)
		{
			if (listview->IsSelected(i))
			{
				// Get the previous item contents
				listview->GetItem(i,name,256);
				listview->GetItem(i,valStart,256,1);
				listview->GetItem(i,valEnd,256,2);
				listview->GetItem(i,valCompress,256,3);
				listview->GetItem(i,valTCompress,256,4);
				itemData = listview->GetItemData(i);

				// Delete the previous item
				listview->DeleteItem(i);

				// Insert it at it's new position
				row=listview->AddItem(name,index);
				//listview->SetItemData(row,1);
				selList.Append(1,&row);

				listview->AddSubItem(valStart,1,row);
				listview->AddSubItem(valEnd,2,row);
				listview->AddSubItem(valCompress,3,row);
				listview->AddSubItem(valTCompress,4,row);
				listview->SetItemData(row,itemData);
				i=-1;
			}
		}
	}
	else
	{
		// We must be moving up in the list
		// search reverse
		for(i=count;i>=0;i--)
		{
			if (listview->IsSelected(i))
			{
				// Get the previous item contents
				listview->GetItem(i,name,256);
				listview->GetItem(i,valStart,256,1);
				listview->GetItem(i,valEnd,256,2);
				listview->GetItem(i,valCompress,256,3);
				listview->GetItem(i,valTCompress,256,4);
				itemData = listview->GetItemData(i);

				// Delete the previous item
				listview->DeleteItem(i);

				// Insert it at it's new position
				row=listview->AddItem(name,index);		
				//listview->SetItemData(row,1);
				selList.Append(1,&row);
				
				listview->AddSubItem(valStart,1,row);
				listview->AddSubItem(valEnd,2,row);
				listview->AddSubItem(valCompress,3,row);
				listview->AddSubItem(valTCompress,4,row);
				listview->SetItemData(row,itemData);
				
				i=count=listview->GetCount();
			}
		}
	}

	// Reselect moved items
	for(i=0;i<selList.Count();i++)
	{
		listview->Select(selList[i]);
	}

	//EnableUI(false);
}

INode* AnimOptionsDlg::GetRootEx(INode* node)
{
	assert(node!=NULL);
	CStr name=node->GetName();
	INode* lastNode=node;

	while(node && !node->IsRootNode())
	{
		lastNode=node;
		node=node->GetParentNode();
	}
	
	// node should be the node above scene root
	return lastNode;
}

int AnimOptionsDlg::CountNodesEx(INode* node)
{
	int numChildren=node->NumberOfChildren();
	int count;

	if (!node->IsHidden())
		count=1;
	else
		count=0;

	for(int i=0;i<numChildren;i++)
	{
		INode* child=node->GetChildNode(i);

		if (!child->IsHidden())
			count+=CountNodesEx(node->GetChildNode(i));
	}

	return count;
}

bool AnimOptionsDlg::VerifyRoot()
{
	// Verify to ensure that selection has only one root
	int    numNodes=ip->GetSelNodeCount();
	INode* root=NULL;

	for(int i=0;i<numNodes;i++)
	{
		INode* node=ip->GetSelNode(i);
		
		if (root==NULL)
			root=GetRootEx(node);
		else
			if (GetRootEx(node)!=root)
				return false;
	}

	return true;
}

void AnimOptionsDlg::Export()
{
	char strBuf[256];
	INode* node;
	CStr   strEnv=getenv(APP_ENV);
	FILE*  fpqn;						// For .qn file export
	
	bRemoveReadOnly=false;

	if (strEnv==CStr(""))
	{
		char errMsg[256];
		sprintf(errMsg,"Environment variable '%s' not set.  Can't continue.",APP_ENV);
		MessageBox(hwnd,errMsg,"Export Animations",MB_OK|MB_ICONSTOP);
		return;
	}

	// Notify user what is being exported
	animopt=optionsDlg->GetOptions();

	// Ensure there is a selection in the scene
	if (ip->GetSelNodeCount()==0)
	{
		MessageBox(hwnd,"No nodes selected in the scene to export","Export Animations",MB_OK|MB_ICONWARNING);
		return;
	}

	// Ensure an animation(s) has been selected to export
	if (IsDlgButtonChecked(hwnd,IDC_EXPORTSEL) && listview->GetSelCount()==0)
	{
		MessageBox(hwnd,"No animations selected to export","Export Animations",MB_OK|MB_ICONWARNING);
		return;
	}

	// Check if there's a chain of nodes selected to export
	int  numSel=ip->GetSelNodeCount();
	bool bAbort=true;

	for(int i=0;i<numSel;i++)
	{
		node=ip->GetSelNode(i);
		node=GetRootEx(node);

		int numTransforms=CountNodesEx(node);				// Subtract one for the scene root

		if (numTransforms>1)
		{
			bAbort=false;
			break;
		}
	}

	if (bAbort)
	{
		MessageBox(hwnd,"Selected node(s) don't contain any parents.\nExport aborted.","Export Animations",MB_OK|MB_ICONWARNING);
		return;
	}

	// Ensure that no selected nodes could evaluate to more than one root
	if (!VerifyRoot())
	{
		MessageBox(hwnd,"The current selection contains multiple root nodes.\nExport aborted.","Export Animations",MB_OK|MB_ICONWARNING);
		return;
	}

	node=ip->GetSelNode(0);

	// Go through and export each animation
	int count=listview->GetCount();

	// Auto-generate .qn file
	if (animopt.bExportQN)
	{
		CStr strQNFile;

		// Ensure we don't add an extra underscore in the event that the suffix isn't used
		if (strlen(animopt.suffixname)==0)
			strQNFile=strEnv+CStr(SCRIPT_PATH)+CStr(animopt.modelname)+CStr(".qn");
		else
			strQNFile=strEnv+CStr(SCRIPT_PATH)+CStr(animopt.modelname)+CStr("_")+CStr(animopt.suffixname)+CStr(".qn");

		fpqn=fopen(strQNFile,"w");

		if (!fpqn)
		{
			char errMsg[256];
			sprintf(errMsg,"Couldn't open file '%s' for output.\n.QN auto-generation failed.  Export aborted.",(char*)strQNFile);
			MessageBox(hwnd,errMsg,"Export Animations",MB_OK|MB_ICONWARNING);
			return;
		}

		// Ensure we don't add an extra underscore in the event that the suffix isn't used
		if (strlen(animopt.suffixname)==0)
			fprintf(fpqn,"script animload_%s\n",animopt.modelname);
		else
			fprintf(fpqn,"script animload_%s_%s\n",animopt.modelname,animopt.suffixname);
	}

	sprintf(strBuf,"Exporting animation %s...",animopt.modelname);

	// Setup progress bar for animation export
	if (!IsDlgButtonChecked(hwnd,IDC_EXPORTALL))
	{
		progBar=new ProgressBar(hInstance,hwnd,strBuf,0,listview->GetSelCount());
		progBar->AllowCancel(true);
	}
	else
	{
		progBar=new ProgressBar(hInstance,hwnd,strBuf,0,listview->GetCount());
		progBar->AllowCancel(true);
	}

	// Export animations
	for(i=0;i<count;i++)
	{
		if (IsDlgButtonChecked(hwnd,IDC_EXPORTALL) || listview->IsSelected(i))
		{
			char name[256];
			char valStart[256];
			char valEnd[256];
			char valCompress[256];
			char valTCompress[256];
			char bufProgText[256];

			// Update progress bar
			progBar->SetVal(progBar->GetVal()+1);
			progBar->Update();
			if (progBar->WasCanceled())
			{
				delete progBar;
				progBar=NULL;
				return;
			}

			listview->GetItem(i,name,256);
			listview->GetItem(i,valStart,256,1);
			listview->GetItem(i,valEnd,256,2);
			listview->GetItem(i,valCompress,256,3);
			listview->GetItem(i,valTCompress,256,4);

			// Warn and abort if animation out of range
			int vstart = atoi(valStart);
			int vend   = atoi(valEnd);

			if (vend<vstart)
			{
				char strErr[256];
				sprintf(strErr,"The animation '%s' has an end frame less than its start frame.\nOperation aborted.",(char*)name);
				MessageBox(ip->GetMAXHWnd(),strErr,"Animation frames out of range",MB_ICONWARNING|MB_OK);

				delete progBar;
				progBar=NULL;
				return;
			}


			//sprintf(bufProgText,"Exporting %s_%s.ska...",animopt.modelname,name);
			sprintf(bufProgText,"Exporting %s.ska (%s)...",name,animopt.modelname);
			progBar->SetCaption(bufProgText);

			start=atoi(valStart);
			end=atoi(valEnd);
			fCompress=atof(valCompress);
			fTCompress=atof(valTCompress);

			// Scale the rotation compression value appropriately
			fCompress=0.999700f+0.000300f*fCompress;

			// Open export file
			_mkdir(strEnv+CStr(ANIM_PATH)+CStr(animopt.dirname));

			//CStr Filename=strEnv+CStr(ANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+CStr(animopt.modelname)+CStr("_")+name+CStr(".ska");
			CStr Filename=strEnv+CStr(ANIM_PATH)+CStr(animopt.dirname)+CStr("\\")+name+CStr(".ska");
			
			Link<AnimListData>*    tdata = (Link<AnimListData>*)listview->GetItemData(i);

			if (!AnimExporter::ExportAnim(progBar->GetHWND(),
				                          node,
										  Filename,
										  animopt.modelname,
										  start,
										  end,
										  fCompress,
										  fTCompress,
										  &tdata->data.toldata,
										  animopt.bCompressDoubles,
										  animopt.bOutputDebug,
										  animopt.bUseCompression,
										  animopt.bCompressTime,
										  false))
										  //animopt.bRotateRoot))
			{
				if (progBar)
				{
					delete progBar;
					progBar=NULL;
				}

				char errMsg[256];
				sprintf(errMsg,"Unable to open file '%s' for output.",Filename);
				MessageBox(hwnd,errMsg,"Export Animations",MB_OK|MB_ICONWARNING);
				return;
			}

			// Run AnimConv to create a PS2 file for the animation so it will update properly
			BuildPS2File(Filename);

			// Update the UI with the file size
			int filesize=GetFileSize(Filename +".ps2");
			char strFileSize[256];
			_itoa(filesize,strFileSize,10);
			listview->AddSubItem(strFileSize,5,i);
			listview->Update(i);

			tdata->data.fileSize = filesize;

			// Add .qn line
			if (animopt.bExportQN)
			{
				fprintf(fpqn,"\tLoadAnim name=\"%s%s\\%s.ska\" descChecksum=%s\n",
					QNANIM_PATH,animopt.modelname,animopt.modelname,name);
			}
		}
	}

	// close .qn export
	if (animopt.bExportQN)
	{
		fprintf(fpqn,"endscript\n");
		fclose(fpqn);
	}

	// Shutdown the progress bar
	delete progBar;
	progBar=NULL;
}

void AnimOptionsDlg::SliderUpdate(float val,int pos)
{
	// pos == 3 for Rotation Tolerance
	// pos == 4 for Translation Tolerance

	char strVal[256];

	int index=listview->GetSel();
	if (index==-1)
		return;

	sprintf(strVal,"%f",val);

	int count=listview->GetCount();
	for(int i=0;i<count;i++)
	{
		if (listview->IsSelected(i))
		{
			listview->AddSubItem(strVal,pos,i);

			Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(i);
			if (pos==3)
				tdata->data.rotTol = val;
			
			if (pos==4)
				tdata->data.tranTol = val;
		}
	}
}

int AnimOptionsDlg::GetFileSize(char* Filename)
{
	FILE* fp=fopen(Filename,"r");
	int   size;

	if (fp)
	{
		fseek(fp,0,SEEK_END);
		size=(int)ftell(fp);
		fclose(fp);
	}

	return size;
}

void AnimOptionsDlg::FindAnim()
{
	int   time  = ip->GetTime()/GetTicksPerFrame();
	int  count  = listview->GetCount();
	bool bFound = false;

	for(int i=0;i<count;i++)
	{
		char strBuf[256];
		int start,end;

		listview->GetItem(i,strBuf,255,1);
		start=atoi(strBuf);
		listview->GetItem(i,strBuf,255,2);
		end=atoi(strBuf);

		if (time>=start && time<=end)
		{
			listview->Select(i);
			listview->SetSel(i);
			bFound=true;
		}
		else
			listview->UnSelect(i);
	}

	if (!bFound)
	{
		MessageBox(hwnd,"Couldn't find an animation that matches your current frame.","Find Animation",MB_ICONWARNING|MB_OK);
	}
}

void AnimOptionsDlg::AddBones(INode* root)
{
	int kids = root->NumberOfChildren();

	for(int i=0;i<kids;i++)
	{
		INode* node = root->GetChildNode(i);
		AddBones(node);
	}

	if (root!=ip->GetRootNode())
	{
		CStr name = root->GetName();

		// Ignore names that end with _zz
		CStr subname = name.Substr(name.Length()-3,name.Length());

		if (subname!=CStr("_zz"))
		{
			int index = SendDlgItemMessage(hwnd,IDC_BONESEL,CB_ADDSTRING,0,(LPARAM)(char*)name);        
			SendDlgItemMessage(hwnd,IDC_BONESEL,CB_SETITEMDATA,(WPARAM)index,(LPARAM)root);
		}
	}
}

bool AnimOptionsDlg::LoadTolerances()
{
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\AnimTol.ini";

	if (!TolFile.Load(appDir))
	{
		char strErr[256];
		sprintf(strErr,"Couldn't load tolerance config file '%s'",(char*)appDir);
		MessageBox(hwnd,strErr,"Couldn't open config file",MB_ICONWARNING|MB_OK);
		return false;
	}

	// Add the presets to the preset selection list
	SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_RESETCONTENT,0,0);

	Link<GlobalTolRecord>* link = TolFile.DB.GetHead();

	while(link)
	{
		int index;

		index = SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_ADDSTRING,0,(LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_SETITEMDATA,(WPARAM)index,(LPARAM)&link->data);
		link = link->next;
	}

	// Load sub animation tolerance file
	appDir = ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\SubAnimTol.ini";

	if (!SubTolFile.Load(appDir))
	{
		char strErr[256];
		sprintf(strErr,"Couldn't load sub tolerance config file '%s'",(char*)appDir);
		MessageBox(hwnd,strErr,"Couldn't open config file",MB_ICONWARNING|MB_OK);
		return false;	
	}

	return true;
}

bool AnimOptionsDlg::SaveTolerances()
{

	return false;
}

CStr AnimOptionsDlg::GetValueViaName(LinkList<CStr>* list,CStr name)
{
	Link<CStr>* curnode = list->GetHead();

	while(curnode)
	{
		if (Instr(curnode->data,name)==0)
			return curnode->data;

		curnode=curnode->next;
	}

	return CStr("");
}

void AnimOptionsDlg::UpdateTolerances()
{
	pTolList->Clear();
	pTolListTran->Clear();

	/*
	if (listview->GetSelCount()!=1)
	{
		return;
	}
	*/

	int index = listview->GetSelItem();

	if (index==-1)
		return;

	int tolListIndex;

	Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(index);
	Link<TolRecord>* currec = tdata->data.toldata.DB.GetHead();

	// Reconstruct the property lists based on the selected animation
	while(currec)
	{
		// Add to rotation property list
		if (currec->data.bHasRot)
		{
			if (!currec->data.bRotPreset)
			{
				pTolList->AddSpinEdit(currec->data.name,ROTMIN,ROTMAX,ROTINCR,currec->data.name);
				tolListIndex = pTolList->GetCurIndex();

				char buf[256];
				sprintf(buf,"%f",currec->data.valRot);

				pTolList->SetValue(tolListIndex,buf);
			}
			else
			{
				ListData* ldata = pTolList->AddList(currec->data.name,currec->data.name);
				tolListIndex = pTolList->GetCurIndex();
				BuildSubPresetList(ldata);

				pTolList->SetValue(tolListIndex,GetValueViaName(&ldata->list,currec->data.rotPresetName));
			}
		}

		// Add to translation property list
		if (currec->data.bHasTran)
		{
			if (!currec->data.bTranPreset)
			{
				pTolListTran->AddSpinEdit(currec->data.name,TRANMIN,TRANMAX,TRANINCR,currec->data.name);
				tolListIndex = pTolListTran->GetCurIndex();

				char buf[256];
				sprintf(buf,"%f",currec->data.valTran);

				pTolListTran->SetValue(tolListIndex,buf);
			}
			else
			{
				ListData* ldata = pTolListTran->AddList(currec->data.name,currec->data.name);
				tolListIndex = pTolListTran->GetCurIndex();
				BuildSubPresetList(ldata);

				pTolListTran->SetValue(tolListIndex,GetValueViaName(&ldata->list,currec->data.tranPresetName));
			}
		}

		currec = currec->next;
	}

	pTolList->BuildUI();
	pTolListTran->BuildUI();
	pTolList->ResetUpdateStatus();
	pTolListTran->ResetUpdateStatus();
}

void AnimOptionsDlg::TolFocusCB(PropList* plist,int id,void* pData)
{
	AnimOptionsDlg* pthis = (AnimOptionsDlg*)pData;

	// Previous color is stored in the property flags
	if (pthis->lastTolID != -1)
		plist->SetColor(pthis->lastTolID,PROPCOLOR_GRAY);

	plist->SetColor(id,PROPCOLOR_WHITE);

	switch(plist->GetType(id))
	{
	case PROPTYPE_SPINEDIT:
		CheckRadioButton(pthis->hwnd,IDC_PRESET,IDC_MANUAL,IDC_MANUAL);
		break;
	case PROPTYPE_LIST:
		CheckRadioButton(pthis->hwnd,IDC_PRESET,IDC_MANUAL,IDC_PRESET);
		break;
	}

	pthis->lastTolID = id;
}

void AnimOptionsDlg::TolFocusTranCB(PropList* plist,int id,void* pData)
{
	AnimOptionsDlg* pthis = (AnimOptionsDlg*)pData;

	// Previous color is stored in the property flags
	if (pthis->lastTolIDTran != -1)
		plist->SetColor(pthis->lastTolIDTran,PROPCOLOR_GRAY);

	plist->SetColor(id,PROPCOLOR_WHITE);

	switch(plist->GetType(id))
	{
	case PROPTYPE_SPINEDIT:
		CheckRadioButton(pthis->hwnd,IDC_PRESETTRANS,IDC_MANUALTRANS,IDC_MANUALTRANS);
		break;
	case PROPTYPE_LIST:
		CheckRadioButton(pthis->hwnd,IDC_PRESETTRANS,IDC_MANUALTRANS,IDC_PRESETTRANS);
		break;
	}

	pthis->lastTolIDTran = id;
}

void AnimOptionsDlg::TolChangeCB(PropList* plist,void* pData)
{
	AnimOptionsDlg* pthis = (AnimOptionsDlg*)pData;

	if (pthis->bLockUpdate)
		return;

	pthis->bUpdated = true;

	int modID = plist->GetLastMod();

	int index = pthis->listview->GetSelItem();
	if (index>-1)
	{
		// Add to listview data list
		Link<AnimListData>*  tlink = (Link<AnimListData>*)pthis->listview->GetItemData(index);
		TolRecord            trec;
		trec.name = plist->GetName(modID);
			
		Link<TolRecord>* treclink = tlink->data.toldata.DB.Find(&trec);
		TolRecord*          ptrec = &treclink->data;

		if (!ptrec)
			return;

		ptrec->bHasRot = true;
		
		switch(plist->GetType(modID))
		{
		case PROPTYPE_SPINEDIT:
			{
				CStr  strVal;
				plist->GetValue(modID,strVal);
				ptrec->valRot = atof((char*)strVal);
				ptrec->bRotPreset=false;
			}
			break;
		case PROPTYPE_LIST:
			{		
				char name[256];
				plist->GetValue(modID,ptrec->rotPresetName);
				sscanf(ptrec->rotPresetName,"%s (%f)",&name,&ptrec->valRot);
				ptrec->bRotPreset=true;
			}
			break;
		}
	}

	pthis->UpdateOtherTolerances();
}

void AnimOptionsDlg::TolChangeTranCB(PropList* plist,void* pData)
{
	AnimOptionsDlg* pthis = (AnimOptionsDlg*)pData;
	pthis->bUpdated = true;

	if (pthis->bLockUpdate)
		return;

	int modID = plist->GetLastMod();

	int index = pthis->listview->GetSelItem();
	if (index>-1)
	{
		// Add to listview data list
		Link<AnimListData>*  tlink = (Link<AnimListData>*)pthis->listview->GetItemData(index);
		TolRecord            trec;
		trec.name = plist->GetName(modID);
			
		Link<TolRecord>* treclink = tlink->data.toldata.DB.Find(&trec);
		TolRecord*          ptrec = &treclink->data;

		if (!ptrec)
			return;

		ptrec->bHasTran = true;
		
		switch(plist->GetType(modID))
		{
		case PROPTYPE_SPINEDIT:
			{
				CStr  strVal;
				plist->GetValue(modID,strVal);
				ptrec->valTran = atof((char*)strVal);
				ptrec->bTranPreset=false;
			}
			break;
		case PROPTYPE_LIST:
			{		
				char name[256];
				plist->GetValue(modID,ptrec->tranPresetName);
				sscanf(ptrec->tranPresetName,"%s (%f)",&name,&ptrec->valTran);
				ptrec->bTranPreset=true;
			}
			break;
		}
	}

	pthis->UpdateOtherTolerances();
}

void AnimOptionsDlg::BuildSubPresetList(ListData* ldata)
{
	// Add preset rotation tolerances to list
	Link<TolRecord>* link = SubTolFile.rotDB.GetHead();

	while(link)
	{
		char buf[256];
		sprintf(buf,"%s (%f)",(char*)link->data.name,link->data.valRot);

		ldata->list.Add(&CStr(buf));
		link=link->next;
	}
}

void AnimOptionsDlg::SetPresetTol()
{
	int modID = lastTolID;	

	if (modID==-1 || modID>pTolList->NumProps()-1)
		return;
	
	CStr        name;
	CStr        value;

	// Switch it to a preset
	name = pTolList->GetName(modID);
	pTolList->GetValue(modID,value);
	pTolList->SaveValues();
	pTolList->RemoveProp(modID);

	ListData* ldata = pTolList->AddList(name,name);
	BuildSubPresetList(ldata);

	int index = pTolList->GetCurIndex();

	pTolList->BuildUI();
	pTolList->RestoreValues();

	//pTolList->SetColor(index,PROPCOLOR_WHITE);
	//lastTolID=index;
	pTolList->SetMod(index,true);
	TolChangeCB(pTolList,this);
	TolFocusCB(pTolList,index,this);
}

void AnimOptionsDlg::SetManualTol()
{
	int modID = lastTolID;
	
	if (modID==-1 || modID>pTolList->NumProps()-1)
		return;

	CStr        name;
	CStr        value;

	// Switch it to a manual spinedit control
	name = pTolList->GetName(modID);
	pTolList->GetValue(modID,value);
	pTolList->SaveValues();
	pTolList->RemoveProp(modID);

	pTolList->AddSpinEdit(name,TRANMIN,TRANMAX,TRANINCR,name);
	int index = pTolList->GetCurIndex();

	pTolList->BuildUI();
	pTolList->RestoreValues();

	//pTolList->SetColor(index,PROPCOLOR_WHITE);
	//lastTolID=index;
	pTolList->SetMod(index,true);
	TolChangeCB(pTolList,this);
	TolFocusCB(pTolList,index,this);
}

void AnimOptionsDlg::SetPresetTolTran()
{
	int modID = lastTolIDTran;	

	if (modID==-1)
		return;
	
	CStr        name;
	CStr        value;

	// Switch it to a preset
	name = pTolListTran->GetName(modID);
	pTolListTran->GetValue(modID,value);
	pTolListTran->SaveValues();
	pTolListTran->RemoveProp(modID);

	ListData* ldata = pTolListTran->AddList(name,name);
	int index = pTolListTran->GetCurIndex();

	// Add preset rotation tolerances to list
	Link<TolRecord>* link = SubTolFile.tranDB.GetHead();

	while(link)
	{
		char buf[256];
		sprintf(buf,"%s (%f)",(char*)link->data.name,link->data.valTran);

		ldata->list.Add(&CStr(buf));
		link=link->next;
	}

	pTolListTran->BuildUI();
	pTolListTran->RestoreValues();

	//TolListTran->SetColor(index,PROPCOLOR_WHITE);
	//lastTolIDTran=index;
	pTolListTran->SetMod(index,true);
	TolChangeTranCB(pTolListTran,this);
	TolFocusTranCB(pTolListTran,index,this);
}

void AnimOptionsDlg::SetManualTolTran()
{
	int modID = lastTolIDTran;
	
	if (modID==-1)
		return;

	CStr        name;
	CStr        value;

	// Switch it to a manual spinedit control
	name = pTolListTran->GetName(modID);
	pTolListTran->GetValue(modID,value);
	pTolListTran->SaveValues();
	pTolListTran->RemoveProp(modID);

	pTolListTran->AddSpinEdit(name,0.0f,100.0f,1.0f,name);
	int index = pTolListTran->GetCurIndex();

	pTolListTran->BuildUI();
	pTolListTran->RestoreValues();

	//pTolListTran->SetColor(index,PROPCOLOR_WHITE);
	//lastTolIDTran=index;
	pTolListTran->SetMod(index,true);
	TolChangeTranCB(pTolListTran,this);
	TolFocusTranCB(pTolListTran,index,this);
}


bool AnimOptionsDlg::AddBone(INode* node,bool bNoBuild)
{
	int scrollVal;

	assert(node);

	if (IsDlgButtonChecked(hwnd,IDC_ADDROT)==BST_CHECKED)
	{
		if (pTolList->PropExists(node->GetName()))
			return false;

		int index = listview->GetSelItem();
		if (index>-1)
		{
			// Add to listview data list
			Link<AnimListData>*  tlink = (Link<AnimListData>*)listview->GetItemData(index);
			TolRecord            trec;
			trec.name = node->GetName();
			
			Link<TolRecord>* treclink  = tlink->data.toldata.DB.AddToTailFindUnique(&trec);
			TolRecord*           ptrec = &treclink->data;

			ptrec->bHasRot = true;
			ptrec->valRot  = IEditRot->GetFloat();

			// Add to property list
			pTolList->ResetIndexToEnd();
			scrollVal = pTolList->GetScrollVal();
			pTolList->SaveValues();
			pTolList->DestroyUI();

			pTolList->AddSpinEdit((char*)node->GetName(),ROTMIN,ROTMAX,ROTINCR,(char*)node->GetName());
			index = pTolList->GetCurIndex();

			char buf[256];
			sprintf(buf,"%f",IEditRot->GetFloat());

			pTolList->SetValue(index,buf);		// Set to default rotation tolerance

			if (!bNoBuild)
			{
				pTolList->BuildUI();
				pTolList->RestoreValues();
				pTolList->SetScrollVal(scrollVal);
			}
		}
	}

	if (IsDlgButtonChecked(hwnd,IDC_ADDTRANS)==BST_CHECKED)
	{
		if (pTolListTran->PropExists(node->GetName()))
			return false;

		int index = listview->GetSelItem();
		if (index>-1)
		{
			// Add to listview data list
			Link<AnimListData>*  tlink = (Link<AnimListData>*)listview->GetItemData(index);
			TolRecord            trec;
			trec.name = node->GetName();
			
			Link<TolRecord>* treclink  = tlink->data.toldata.DB.AddToTailFindUnique(&trec);
			TolRecord*           ptrec = &treclink->data;

			ptrec->bHasTran = true;
			ptrec->valTran  = IEditTran->GetFloat();

			// Add to property list
			pTolListTran->ResetIndexToEnd();
			scrollVal = pTolListTran->GetScrollVal();
			pTolListTran->SaveValues();
			pTolListTran->DestroyUI();

			pTolListTran->AddSpinEdit((char*)node->GetName(),TRANMIN,TRANMAX,TRANINCR,(char*)node->GetName());
			index = pTolListTran->GetCurIndex();

			char buf[256];
			sprintf(buf,"%f",IEditTran->GetFloat());

			pTolListTran->SetValue(index,buf);	// Set to default translation tolerance

			if (!bNoBuild)
			{
				pTolListTran->BuildUI();
				pTolListTran->RestoreValues();
				pTolListTran->SetScrollVal(scrollVal);
			}
		}
	}

	return true;
}

void AnimOptionsDlg::AddBone()
{
	int index = SendDlgItemMessage(hwnd,IDC_BONESEL,CB_GETCURSEL,0,0);

	if (index==CB_ERR)
		return;

	INode* node = (INode*)SendDlgItemMessage(hwnd,IDC_BONESEL,CB_GETITEMDATA,(WPARAM)index,0);
	
	if (!AddBone(node))
	{
		char strErr[256];
		sprintf(strErr,"A duplicate entry exists in one of the selected set.  Please correct before adding.");
		MessageBox(hwnd,strErr,"Duplicate Set Exists",MB_ICONWARNING|MB_OK);
	}
}

void AnimOptionsDlg::RemoveBone()
{
	if (IsDlgButtonChecked(hwnd,IDC_ADDROT)==BST_CHECKED)
	{
		if (lastTolID!=-1)
		{
			int index = listview->GetSelItem();
			if (index>-1)
			{
				// Add to listview data list
				Link<AnimListData>* tlink = (Link<AnimListData>*)listview->GetItemData(index);
				TolRecord      trec;
				trec.name = pTolList->GetName(lastTolID);

				
				Link<TolRecord>* treclink  = tlink->data.toldata.DB.Find(&trec);

				if (treclink)
				{
					TolRecord* ptrec = &treclink->data;
					ptrec->bHasRot = false;
				}
			}

			pTolList->SaveValues();
			pTolList->RemoveProp(lastTolID);
			pTolList->BuildUI();
			pTolList->RestoreValues();
			lastTolID--;
		}
	}

	if (IsDlgButtonChecked(hwnd,IDC_ADDTRANS)==BST_CHECKED)
	{
		if (lastTolIDTran!=-1)
		{
			int index = listview->GetSelItem();
			if (index>-1)
			{
				// Add to listview data list
				Link<AnimListData>*  tlink = (Link<AnimListData>*)listview->GetItemData(index);
				TolRecord            trec;
				trec.name = pTolListTran->GetName(lastTolIDTran);
				
				Link<TolRecord>* treclink  = tlink->data.toldata.DB.Find(&trec);

				if (treclink)
				{
					TolRecord* ptrec = &treclink->data;
					ptrec->bHasTran = false;
				}
			}

			pTolListTran->SaveValues();
			pTolListTran->RemoveProp(lastTolIDTran);
			pTolListTran->BuildUI();
			pTolListTran->RestoreValues();
			lastTolIDTran--;
		}
	}
}

void AnimOptionsDlg::AddAllBones()
{
	int count = SendDlgItemMessage(hwnd,IDC_BONESEL,CB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		INode* node = (INode*)SendDlgItemMessage(hwnd,IDC_BONESEL,CB_GETITEMDATA,(WPARAM)i,0);
		AddBone(node,true);
	}

	pTolList->BuildUI();
	pTolList->RestoreValues();

	pTolListTran->BuildUI();
	pTolListTran->RestoreValues();
}

void AnimOptionsDlg::RemoveAllBones()
{
	if (IsDlgButtonChecked(hwnd,IDC_ADDROT)==BST_CHECKED)
	{
		pTolList->Clear();
		lastTolID=-1;
	}

	if (IsDlgButtonChecked(hwnd,IDC_ADDTRANS)==BST_CHECKED)
	{
		pTolListTran->Clear();
		lastTolIDTran=-1;
	}
}

void AnimOptionsDlg::UpdatePresetSel()
{
	pTolList->Clear();
	pTolListTran->Clear();
	lastTolID=-1;
	lastTolIDTran=-1;

	int index = SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_GETCURSEL,0,0);
	
	int lvIndex = listview->GetSelItem();

	if (lvIndex==-1)
	{
		MessageBox(hwnd,"You must select an animation before setting presets.","Can't set preset",MB_OK|MB_ICONWARNING);
		return;
	}

	GlobalTolRecord* record = (GlobalTolRecord*)SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_GETITEMDATA,(WPARAM)index,0);

	Link<TolData>* tlink = (Link<TolData>*)listview->GetItemData(lvIndex);
	tlink->data.DB.Clear();

	if (record)
	{
		Link<TolRecord>* curlink  = record->DB.GetHead();
		Link<TolRecord>* treclink = tlink->data.DB.Find(&curlink->data);
		TolRecord*          ptrec = &treclink->data;

		while(curlink)
		{
			// Update rotation tolerances
			if (curlink->data.bHasRot)
			{			
				int index;

				pTolList->AddSpinEdit(curlink->data.name,ROTMIN,ROTMAX,ROTINCR,curlink->data.name);
				index = pTolList->GetCurIndex();
				
				char buf[256];
				sprintf(buf,"%f",curlink->data.valRot);

				pTolList->SetValue(index,buf);
			}

			// Update translation tolerances
			if (curlink->data.bHasTran)
			{
				int index;

				pTolListTran->AddSpinEdit(curlink->data.name,TRANMIN,TRANMAX,TRANINCR,curlink->data.name);
				index = pTolListTran->GetCurIndex();
				
				char buf[256];
				sprintf(buf,"%f",curlink->data.valRot);

				pTolListTran->SetValue(index,buf);
			}

			tlink->data.DB.Add(&curlink->data);
			curlink=curlink->next;
		}

		pTolList->BuildUI();
		pTolListTran->BuildUI();
	}
}

void AnimOptionsDlg::ToggleActive()
{
	int index = listview->GetCount();

	for(int i=0;i<index;i++)
	{
		if (listview->IsSelected(i))
		{
			if (IsDlgButtonChecked(hwnd,IDC_ACTIVEANIM)==BST_UNCHECKED)
			{
				listview->AddSubItem("0",6,i);
			
				Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(i);
				tdata->data.active = false;
			}
			else
			{
				listview->AddSubItem("1",6,i);

				Link<AnimListData>* tdata = (Link<AnimListData>*)listview->GetItemData(i);
				tdata->data.active = true;
			}
		}
	}
}

int AnimOptionsDlg::Sort(DWORD dw1,DWORD dw2,ListView* pList,void* pData)
{
	int itemID = (int)pData;

	Link<AnimListData>* link1 = (Link<AnimListData>*)dw1;
	Link<AnimListData>* link2 = (Link<AnimListData>*)dw2;

	switch(itemID)
	{
	case COLID_ANIMNAME:
		{
			if (link1->data.animName < link2->data.animName)
				return -1;

			if (link1->data.animName > link2->data.animName)
				return 1;
		}
		break;

	case COLID_START:
		{
			if (link1->data.start < link2->data.start)
				return -1;

			if (link1->data.start > link2->data.start)
				return 1;
		}
		break;

	case COLID_END:
		{
			if (link1->data.end < link2->data.end)
				return -1;

			if (link1->data.end > link2->data.end)
				return 1;
		}
		break;

	case COLID_ROTTOL:
		{
			if (link1->data.rotTol < link2->data.rotTol)
				return -1;

			if (link1->data.rotTol > link2->data.rotTol)
				return 1;
		}
		break;

	case COLID_TRANTOL:
		{
			if (link1->data.tranTol < link2->data.tranTol)
				return -1;

			if (link1->data.tranTol > link2->data.tranTol)
				return 1;
		}
		break;

	case COLID_FILESIZE:
		{
			if (link1->data.fileSize < link2->data.fileSize)
				return -1;

			if (link1->data.fileSize > link2->data.fileSize)
				return 1;
		}
		break;

	case COLID_ACTIVE:
		{
			if (link1->data.active < link2->data.active)
				return -1;

			if (link1->data.active > link2->data.active)
				return 1;
		}
		break;
	}

	return 0;
}

void AnimOptionsDlg::FieldSort(int item)
{
	listview->Sort(Sort,(void*)item);
}

void AnimOptionsDlg::UpdateOtherTolerances()
{
	// Update Rotations
	int  count = listview->GetCount();

	for(int i=0;i<count;i++)
	{
		if (listview->IsSelected(i))
		{
			Link<AnimListData>* link = (Link<AnimListData>*)listview->GetItemData(i);
			
			if (!link)
				return;
			
			// Scan through all of our items, if any of them exists in the animation being modified
			// update their values to match, if it doesn't exist, add it to the other animation
			
			int numProps = pTolList->NumProps();

			for(int i=0;i<numProps;i++)
			{
				if (pTolList->WasUpdated(i))
				{
					CStr value;
					CStr name;
					CStr namevalue;
					bool bIsPreset=false;

					name = pTolList->GetName(i);
					pTolList->GetValue(i,value);

					if (pTolList->GetType(i)==PROPTYPE_LIST)
					{
						char buf[256];
						float val;

						bIsPreset=true;
						namevalue = value;

						sscanf((char*)value,"%s (%f)",buf,&val);
						sprintf(buf,"%f",val);
						value = buf;
					}

					Link<TolRecord>* sublink = link->data.toldata.DB.GetHead();

					bool bUpdated = false;

					while(sublink)
					{
						if (sublink->data.name == name)
						{
							sublink->data.bHasRot = true;
							sublink->data.valRot  = atof(value);
							
							if (bIsPreset)
							{
								sublink->data.rotPresetName = namevalue;
								sublink->data.bRotPreset = true;
							}

							bUpdated = true;
						}

						sublink = sublink->next;
					}

					if (!bUpdated)
					{
						// If we didn't find the modified property, add it
						TolRecord trec;
						trec.name = name;
						trec.bHasRot = true;
						trec.valRot = atof(value);

						if (bIsPreset)
						{
							sublink->data.rotPresetName = namevalue;
							sublink->data.bRotPreset = true;
						}

						link->data.toldata.DB.AddToTail(&trec);
					}
				}
			}

			// Now do the same for translation tolerances
			numProps = pTolListTran->NumProps();

			for(i=0;i<numProps;i++)
			{
				if (pTolListTran->WasUpdated(i))
				{
					CStr value;
					CStr name;
					CStr namevalue;
					bool bIsPreset=false;

					name = pTolListTran->GetName(i);
					pTolListTran->GetValue(i,value);

					if (pTolListTran->GetType(i)==PROPTYPE_LIST)
					{
						char buf[256];
						float val;

						bIsPreset=true;
						namevalue = value;

						sscanf((char*)value,"%s (%f)",buf,&val);
						sprintf(buf,"%f",val);
						value = buf;
					}

					Link<TolRecord>* sublink = link->data.toldata.DB.GetHead();

					bool bUpdated = false;

					while(sublink)
					{
						if (sublink->data.name == name)
						{
							sublink->data.bHasTran = true;
							sublink->data.valTran  = atof(value);

							if (bIsPreset)
							{
								sublink->data.tranPresetName = namevalue;
								sublink->data.bTranPreset = true;
							}

							bUpdated = true;
						}

						sublink = sublink->next;
					}

					if (!bUpdated)
					{
						// If we didn't find the modified property, add it
						TolRecord trec;
						trec.name = name;
						trec.bHasTran = true;
						trec.valTran = atof(value);

						if (bIsPreset)
						{
							sublink->data.tranPresetName = namevalue;
							sublink->data.bTranPreset = true;
						}

						link->data.toldata.DB.AddToTail(&trec);
					}
				}
			}
		}
	}
}

AnimOptionsDlg::AnrSaveMode AnimOptionsDlg::MaybeSaveAnr()
{
	if (bUpdated)
	{
		int bResult = MessageBox(hwnd,"You've made changes to the animation data.  Do you want to save the .anr file?","Animation Data Modified",MB_ICONQUESTION|MB_YESNOCANCEL);

		if (bResult == IDYES)
		{
			ExportRange(anrfile);
			return ANRYES;
		}

		if (bResult == IDCANCEL)
			return ANRCANCEL;
	}

	return ANRNO;
}

void AnimOptionsDlg::BuildPS2File(CStr Filename)
{
	SceneExportOptions seo;
	GetSceneExportOptions(&seo);

	// First verify that the animation actually exists
	// Execute animation converter
	char  ancPath[256];
	char* args[4];
	CStr  strFileCmd = CStr("-f") + Filename;
	sprintf( ancPath, "%s\\bin\\win32\\animconv.exe", getenv(APP_ENV) );

	args[0] = ancPath;
	args[1] = strFileCmd;
	args[2] = "-pp";
	args[3] = NULL;

	int result = ( spawnvp( _P_WAIT, ancPath, args ) == 0 );
}
