/*
	AnimExporter.h
	Animation Exporter

	Encapsulates animation export functionality

	1-11-01
*/

#ifndef __ANIMEXPORTER__
#define __ANIMEXPORTER__

#include <core/math.h>
#include "skeleton.h"
#include "GenExport.h"
#include "max.h"

#define  OUTPUT_FPS     60		// Number of frames per second to output animation as

class TolData;

namespace Gfx
{
	struct SAnimQFrame
	{
		Mth::Quat		q;
		float			time;
		bool            bFlip[3];

		SAnimQFrame()
		{
			bFlip[0]=false;
			bFlip[1]=false;
			bFlip[2]=false;
		}
	};

	struct SAnimTFrame
	{
		float			t[3];	
		float			time;
	};
}

class AnimExporter
{
protected:
	Gfx::SAnimQFrame* Qframes;		// Uncompressed Rotation list  (Quaternions) for current node
	Gfx::SAnimTFrame* Tframes;		// Uncompressed Translation list for current node

	Gfx::SAnimQFrame* compQframes;	// Compressed Rotation list (Quaternions) for current node
	Gfx::SAnimTFrame* compTframes;	// Compressed Translation list for current node

	int* numCompressedQFrames;
	int* numCompressedTFrames;

	FILE*      fpExport;			// File stream being exported
	FILE*      fpDebug;				// Debug File stream

	Interface* ip;
	int        start,end;			// Start and end frames of MAX animation
	int        curFrame;			// Current position within the frames arrays
	int        numFrames;			// Number of animation frames in memory

	bool       bRemoveReadOnly;		// True if read only flag should be removed on all
	bool       bSwapCoordSystem;	// True if coordinates from MAX should be adjusted to game

	CSkeletonData* skeleton;		// Skeleton data for current animation
	INode*         rootbone;		// The root object of the skeleton

	bool       bCompressTime;		// True if times should be given as integers
	bool       bRotateRoot;			// True if root node should be rotated 90 degrees

	void BuildNodeQFrames(Gfx::SAnimQFrame* qframes,INode* node);
	void BuildNodeTFrames(Gfx::SAnimTFrame* qframes,INode* node);

	int BuildAllQFrames(Gfx::SAnimQFrame* qframes,INode* node,int numFrames,int ID);
	int BuildAllTFrames(Gfx::SAnimTFrame* qframes,INode* node,int numFrames,int ID);

	void GetQFrames(Gfx::SAnimQFrame* qframes,INode* root,bool bOneNode=false);	// Builds a list of qframes for the given node and it's children
	void GetTFrames(Gfx::SAnimTFrame* tframes,INode* root,bool bOneNode=false);	// Builds a list of tframes for the given node and it's children

	static void FreeFrames(Gfx::SAnimQFrame** frame);	// Frees list returned by GetQFrame
	static void FreeFrames(Gfx::SAnimTFrame** frame);	// Frees list returned by GetTFrame

	int GetNodeName(CStr* strName,INode* node,int ID);
	int GetParentName(CStr* strName,INode* node,int ID);

	void RemoveDblKeys(int numNodes,int numFrames);
	void RemoveDblKeys(int numNodes, int numFrames,
						 Gfx::SAnimQFrame* compQframes,
						 Gfx::SAnimTFrame* compTframes,
						 int* numCompressedQFrames,
						 int* numCompressedTFrames);

	bool VerifyRoot();
	INode* GetRoot(INode* node);
	bool ExportNodes(INode* root);
	int  CountNodes(INode* node);
	void GetNodeNames(CStr* strNames,INode* root);
	void GetParentNames(CStr* strNames,INode* root);

	int  BuildFrames(INode* node);									// Builds frames for the given node

	bool GetAnim(INode* root,int start,int end,float errorQ,float errorT,TolData* tdata,bool bCompress=true);
	bool GetAnim(INode* root,int start,int end,float errorQ,float errorT,TolData* tdata,bool bCompress,
				   int** numCompressedQFrames, Gfx::SAnimQFrame** compQFrames,
				   int** numCompressedTFrames, Gfx::SAnimTFrame** compTFrames,
				   int*  numNodes);

	void HideDummy(INode* node);
	void UnhideDummy(INode* node);

	void UnhideDummies();
	void HideDummies();

	int CompressRawQFrameData( Gfx::SAnimQFrame* p_newdata, Gfx::SAnimQFrame* p_data, int num_keys, double error );
	int CompressRawTFrameData( Gfx::SAnimTFrame* p_newdata, Gfx::SAnimTFrame* p_data, int num_keys, double error );
	int CompressFloatKeys( GenKey<float>* newvals, GenKey<float>* vals, int num_keys, double error );

	void FixEulerFlips(Gfx::SAnimQFrame* pdata,int numKeys);

	bool ApproxEqual(float v1,float v2);

public:
	AnimExporter();
	~AnimExporter();

	bool Export(INode* root,char* Filename,char* ModelName,int start,int end);
	bool ExportAnim(HWND     hwnd,
		            INode*   root,
					char*    Filename,
					char*    ModelName,
					int      start,
					int      end,
					float    errorQ,
					float    errorT,
					TolData* tdata,
					bool     bOneFrame=false,
					bool     bDebug=false,
					bool     bUseCompression=false,
					bool     bCompressTime=false,
					bool     bRotateRoot=false);

	// Retrieves the animation checksum from a .ska file
	unsigned int GetChecksum(char* Filename);
};

#endif
