/**********************************************************************
 *<
	FILE: PolyEdit.h

	DESCRIPTION:   Editable Polygon Mesh Object

	CREATED BY: Steve Anderson

	HISTORY: created March 2000

 *>	Copyright (c) 2000, All Rights Reserved.
 **********************************************************************/

#ifndef __POLYOBJED__
#define __POLYOBJED__

#include "iparamm2.h"
#include "sbmtlapi.h"
#include "istdplug.h"
#include "nsclip.h"
#include "iFnPub.h"
#include "iEPoly.h"
#include <FaceFlags/FaceFlags.h>

#define DEF_FALLOFF 20.0f

#define EPOLYOBJ_CLASS_ID Class_ID(0x1bf8338d,0x192f6098)

namespace EditPoly
{

// Table converts editable poly selection levels to MNMesh ones.
static int meshSelLevel[] = { MNM_SL_OBJECT, MNM_SL_VERTEX, MNM_SL_EDGE,
								MNM_SL_EDGE, MNM_SL_FACE, MNM_SL_FACE };
// Table converts editable poly selection levels to display flags.
static DWORD dispFlags[] = { 0, MNDISP_VERTTICKS|MNDISP_SELVERTS,
		MNDISP_SELEDGES, MNDISP_SELEDGES, MNDISP_SELFACES,
		MNDISP_SELFACES };

#define DEF_PICKBOX_SIZE	4

// ID's for toolbar
#define IDC_SELVERTEX	0x4015
#define IDC_SELEDGE	0x4016
#define IDC_SELBORDER	0x4017
#define IDC_SELFACE		0x4018
#define IDC_SELELEMENT	0x4019

// Parameter block 2 enumerations:
// One parameter block:
enum { ep_block };

// Several rollouts:
enum { ep_select, ep_softsel, ep_geom, ep_subdiv, ep_surface, ep_vertex,
	ep_edge, ep_face, ep_displacement, ep_advanced_displacement };

extern int surfIDs[];
extern int surfDlgs[];
extern ParamMap2UserDlgProc *surfProcs[];

class CreateVertCMode;
class CreateEdgeCMode;
class CreateFaceCMode;
class AttachPickMode;
class DivideEdgeCMode;
class DivideFaceCMode;
class ExtrudeCMode;
class ChamferCMode;
class BevelCMode;
class CutVertCMode;
class CutEdgeCMode;
class CutFaceCMode;
class WeldCMode;
class EditTriCMode;

#define CID_CREATEVERT	CID_USER+0x1000
#define CID_CREATEEDGE  CID_USER+0x1001
#define CID_CREATEFACE  CID_USER+0x1002
#define CID_CAP			CID_USER+0x1003
#define CID_ATTACH		CID_USER+0x1010
#define CID_DIVIDEEDGE	CID_USER+0x1018
#define CID_DIVIDEFACE	CID_USER+0x1019
#define CID_EXTRUDE		CID_USER+0x1020
#define CID_POLYCHAMFER	CID_USER+0x1028
#define CID_BEVEL		CID_USER+0x102C
#define CID_CUTVERT		CID_USER+0x1038
#define CID_CUTEDGE		CID_USER+0x1039
#define CID_CUTFACE		CID_USER+0x103A
#define CID_WELD		CID_USER+0x1040
#define CID_FLIP		CID_USER+0x1044
#define CID_EDITTRI		CID_USER+0x1048

#define MAX_MATID	0xffff

// Named selection set levels:
#define NS_VERTEX 0
#define NS_EDGE 1
#define NS_FACE 2

// Conversion from selLevel to named selection level:
static int namedSetLevel[] = { NS_VERTEX, NS_VERTEX, NS_EDGE, NS_EDGE, NS_FACE, NS_FACE };
static int namedClipLevel[] = { CLIP_VERT, CLIP_VERT, CLIP_EDGE, CLIP_EDGE, CLIP_FACE, CLIP_FACE };
static int hitLevel[] = {0, SUBHIT_MNVERTS,SUBHIT_MNEDGES,
					SUBHIT_MNEDGES|SUBHIT_OPENONLY, 
					SUBHIT_MNFACES, SUBHIT_MNFACES};

// Flags:
// Disp Result keeps track of "Show End Result" button for this Editable Mesh.
#define EPOLY_DISP_RESULT 0x0001
// In Render indicates if we're in the middle of a render.
#define EPOLY_IN_RENDER 0x0002
// Aborted indicates that the user escaped out of a MeshSmooth subdivision.
#define EPOLY_ABORTED 0x0004
// Force indicates that the user wishes to force a recomputation of the subdivision result.
#define EPOLY_FORCE 0x0008

// References:
#define EPOLY_PBLOCK 0
#define EPOLY_MASTER_CONTROL_REF  1
#define EPOLY_VERT_BASE_REF 2

class EPolyShortcutCB;
class TempMoveRestore;
class EPolyBackspaceUser;

class EditPolyObject : public PolyObject, public EventUser, public ISubMtlAPI,
			public EPoly, public AttachMatDlgUser, public IMeshSelect, 
			public IMeshSelectData {
public:
	// Class vars
	static HWND hFaceFlags;		// aml

	static Interface *ip;
	static EditPolyObject *editObj;
	static IParamMap2 *pSelect, *pSoftsel, *pGeom, *pSubdiv, *pSurface, *pDisp;
	static bool rsSelect, rsSoftsel, rsGeom, rsSubdiv, rsSurface, rsDisp;
	static BOOL rsFaceFlags /* aml */;
	static bool inExtrude, inBevel, inChamfer;

	// Command modes
	static MoveModBoxCMode *moveMode;
	static RotateModBoxCMode *rotMode;
	static UScaleModBoxCMode *uscaleMode;
	static NUScaleModBoxCMode *nuscaleMode;
	static SquashModBoxCMode *squashMode;
	static SelectModBoxCMode *selectMode;
	static CreateVertCMode *createVertMode;
	static CreateEdgeCMode * createEdgeMode;
	static CreateFaceCMode* createFaceMode;
	static AttachPickMode* attachPickMode;
	static DivideEdgeCMode* divideEdgeMode;
	static DivideFaceCMode *divideFaceMode;
	static ExtrudeCMode *extrudeMode;
	static ChamferCMode *chamferMode;
	static BevelCMode* bevelMode;
	static CutVertCMode *cutVertMode;
	static CutEdgeCMode *cutEdgeMode;
	static CutFaceCMode *cutFaceMode;
	static WeldCMode *weldMode;
	static EditTriCMode *editTriMode;

	static int attachMat;
	static int pickBoxSize;
	static BOOL condenseMat;
	static Quat sliceRot;
	static Point3 sliceCenter;
	static float sliceSize;
	static bool sliceMode;
	static EPolyBackspaceUser backspacer;

	IParamBlock2 *pblock;

	// Cache for computing coord. systems
	// methods in tridata.cpp
	MNTempData *tempData;
	MNTempData *TempData ();
	void InvalidateTempData (PartID parts=PART_TOPO|PART_GEOM);

	static TempMoveRestore *tempMove;

	GenericNamedSelSetList selSet[3];
	MasterPointControl	*masterCont;		// Master track controller
	Tab<Control*> cont;
	int selLevel;
	Interval arValid;
	DWORD ePolyFlags;
	BitArray vsel, esel, fsel;
	bool sliceInitialized;
	MNMesh subdivResult;
	Interval subdivValid;

	EditPolyObject();
	~EditPolyObject();
	void ResetClassParams (BOOL fileReset);

	// Flag methods.
	void SetFlag (DWORD fl, BOOL val=TRUE) { if (val) ePolyFlags |= fl; else ePolyFlags &= ~fl; }
	void ClearFlag (DWORD fl) { ePolyFlags &= (~fl); }
	bool GetFlag (DWORD fl) { return (ePolyFlags&fl) ? TRUE : FALSE; }

	MNMesh *GetMeshPtr () { return &mm; }

	// Temp (drag) move methods:
	void DragMoveInit ();
	void DragMoveRestore ();
	void DragMove (Tab<Point3> & delta, EPoly *epol, TimeValue t);
	void DragMap (int mp, Tab<UVVert> & mapDelta, EPoly *epol, TimeValue t);
	void DragMoveAccept (TimeValue t);
	void DragMoveClear ();
	void ApplyDelta (Tab<Point3> & delta, EPoly *epol, TimeValue t);
	void ApplyMapDelta (int mp, Tab<UVVert> & mapDelta, EPoly *epol, TimeValue t);

	// Animatable methods
	void DeleteThis() {delete this;}
	Class_ID ClassID() {return EPOLYOBJ_CLASS_ID;}
	void GetClassName(TSTR& s) {s = GetString(IDS_SCA_E_POLY);}
	void BeginEditParams(IObjParam  *ip, ULONG flags,Animatable *prev);
	void EndEditParams(IObjParam *ip, ULONG flags,Animatable *next);
	int NumSubs() { return 2; }		// only the pblock and the master point controller
	Animatable* SubAnim(int i) { return GetReference(i); }
	TSTR SubAnimName(int i);
	BOOL AssignController(Animatable *control,int subAnim);
	int SubNumToRefNum(int subNum) {return subNum;}
	BOOL SelectSubAnim(int subNum);

	// PBlock2 methods:
	int NumParamBlocks () { return 1; }
	IParamBlock2 *GetParamBlock (int i) { return pblock; }
	IParamBlock2 *GetParamBlockByID (short id) { return (pblock->ID() == id) ? pblock : NULL; }

	// Reference methods
	int NumRefs() {return EPOLY_VERT_BASE_REF+cont.Count();}
	RefTargetHandle GetReference(int i);
	void SetReference(int i, RefTargetHandle rtarg);
	void CreateContArray();
	void SynchContArray(int newNV);
	void AllocContArray(int count);
	void ReplaceContArray(Tab<Control *> &nc);
	//BOOL BypassTreeView();
	void DeletePointConts(BitArray &set);
	void PlugControllersSel(TimeValue t,BitArray &set);
	BOOL PlugControl(TimeValue t,int i);
	void SetPtCont(int i, Control *c);
	void SetPointAnim(TimeValue t, int i, Point3 pt);
	BOOL CloneVertCont(int from, int to);
	RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, PartID& partID, RefMessage message);

	void UpdateGeometry (TimeValue time);
	void UpdateSoftSelection (TimeValue time);
	void UpdateSubdivResult (TimeValue time);

	// BaseObject methods
	ObjectState Eval(TimeValue time);
	//Object versions:
	int Display(TimeValue t, INode* inode, ViewExp *vpt, int flags);
	void GetWorldBoundBox (TimeValue t, INode * inode, ViewExp* vp, Box3& box);
	// Gizmo versions:
	bool ShowGizmo ();
	int Display(TimeValue t, INode* inode, ViewExp *vpt, int flags, ModContext *mc);
	void GetWorldBoundBox (TimeValue t, INode * inode, ViewExp* vp, Box3& box, ModContext *mc);

	void GetLocalBoundBox (TimeValue t, INode* inode, ViewExp* vp, Box3& box);
	DWORD CurrentHitLevel (int *selByVert=NULL);
	int HitTest(TimeValue t, INode* inode, int type, int crossing, int flags, IPoint2 *p, ViewExp *vpt, ModContext* mc);
	void SelectSubComponent (HitRecord *hitRec, BOOL selected, BOOL all, BOOL invert=FALSE);
	void ClearSelection (int selLevel);
	void SelectAll (int selLevel);
	void InvertSelection (int selLevel);
	void InvalidateDistanceCache ();
	void InvalidateSoftSelectionCache ();
	void ActivateSubobjSel (int level, XFormModes& modes );		
	void GetSubObjectCenters(SubObjAxisCallback *cb,TimeValue t,INode *node,ModContext *mc);
	void GetSubObjectTMs(SubObjAxisCallback *cb,TimeValue t,INode *node,ModContext *mc);
	bool getShowVerts ();
	void ShowEndResultChanged (BOOL showEndResult);

	// Object methods		
	TCHAR *GetObjectName() { return GetString(IDS_SCA_E_POLY);}
	BOOL IsSubClassOf(Class_ID classID);
	Object *CollapseObject(){return this;}
	Object *ConvertToType (TimeValue t, Class_ID obtype);
	int RenderBegin(TimeValue t, ULONG flags);
	int RenderEnd(TimeValue t);
	Mesh* GetRenderMesh(TimeValue t, INode *inode, View &view,  BOOL& needDelete);

	// Named subobject selections:
	BOOL SupportsNamedSubSels() { return true; }
	void ActivateSubSelSet(TSTR &setName);
	void NewSetFromCurSel(TSTR &setName);
	void RemoveSubSelSet(TSTR &setName);
	void SetupNamedSelDropDown();
	void InvalidateNamedSelDropDown ();
	void UpdateNamedSelDropDown ();
	int NumNamedSelSets();
	TSTR GetNamedSelSetName(int i);
	void SetNamedSelSetName(int i,TSTR &newName);
	void NewSetByOperator(TSTR &newName,Tab<int> &sets,int op);
	void EpfnNamedSelectionCopy (TSTR setName);
	void NamedSelectionCopy (int index);
	void EpfnNamedSelectionPaste (BOOL useDlgToRename);
	BOOL GetUniqueSetName(TSTR &name, bool useDlg);
	int SelectNamedSet();
	void IncreaseNamedSetSize (int nsl, int oldsize, int increase);
	void DeleteNamedSetArray (int nsl, BitArray & del);

	// Reference methods
	RefTargetHandle Clone(RemapDir& remap = NoRemap());
	IOResult Load(ILoad *iload);
	IOResult Save(ISave *isave);

	// Local methods
	void ExitAllCommandModes (bool exSlice=TRUE);
	GenericNamedSelSetList &GetSelSet();
	int GetSubobjectLevel();
	void SetSubobjectLevel(int level);
	void RefreshSelType ();

	// Operations -- in polyedops.cpp
	// Shift-cloning:
	void CloneSelSubComponents(TimeValue t);
	void AcceptCloneSelSubComponents(TimeValue t);
	// Transform stuff:
	void Transform (int sl, TimeValue t, Matrix3& partm, Matrix3 tmAxis, BOOL localOrigin, Matrix3 xfrm, int type);
	void Move(TimeValue t, Matrix3& partm, Matrix3& tmAxis, Point3& val, BOOL localOrigin=FALSE);
	void Rotate(TimeValue t, Matrix3& partm, Matrix3& tmAxis, Quat& val, BOOL localOrigin=FALSE);
	void Scale(TimeValue t, Matrix3& partm, Matrix3& tmAxis, Point3& val, BOOL localOrigin=FALSE);
	void TransformStart(TimeValue t);
	void TransformHoldingFinish (TimeValue t);
	void TransformFinish(TimeValue t);
	void TransformCancel(TimeValue t);

	// Useful many places
	void CollapseDeadVerts ();
	void CollapseDeadEdges ();
	void CollapseDeadFaces ();
	void CollapseDeadStructs ();

	// Select panel ops:
	bool EpfnHide (int msl, DWORD flag=MN_SEL);
	bool EpfnUnhideAll (int msl);

	// Topological & Geometric ops from the Edit Geometry panel:
	int EpfnCreateVertex(Point3 pt, bool pt_local=false, bool select=true);
	int EpfnCreateEdge (int v1, int v2, bool select=true);
	int EpfnCreateFace (int *v, int deg, bool select=true);
	bool EpfnCapHoles (int msl=MNM_SL_EDGE, DWORD targetFlags=MN_SEL);
	bool EpfnDelete (int msl, DWORD delFlag=MN_SEL, bool delIsoVerts=false);
	bool EpfnDeleteIsoVerts ();
	bool DeleteVerts (DWORD flag=MN_SEL);
	bool DeleteEdges (DWORD flag=MN_SEL);
	bool DeleteFaces (DWORD flag=MN_SEL, bool delIsoVerts=false);
	
	// from AttachMatDlgUser
	int GetAttachMat() { return attachMat; }
	void SetAttachMat(int value) { attachMat = value; }
	BOOL GetCondenseMat() { return condenseMat; }
	void SetCondenseMat(BOOL sw) { condenseMat = sw; }

	void EpfnAttach (INode *node, bool & canUndo, INode *myNode, TimeValue t);
	void EpfnMultiAttach (INodeTab &nodeTab, INode *myNode, TimeValue t);
	bool EpfnDetachToElement (int msl, DWORD flag, bool keepOriginal);
	bool EpfnDetachToObject (TSTR name, int msl, DWORD flag, bool keepOriginal, INode *myNode, TimeValue t);
	bool EpfnSplitEdges (DWORD flag=MN_SEL);
	bool EpfnBreakVerts (DWORD flag=MN_SEL);
	int EpfnDivideFace (int face, Tab<float> &bary, bool select=true);
	int EpfnDivideEdge (int edge, float prop, bool select=true);
	bool EpfnCollapse (int msl, DWORD flag);

	void EpfnExtrudeFaces (float amount, DWORD flag, TimeValue t);
	void EpfnBevelFaces (float height, float outline, DWORD flag, TimeValue t);
	void EpfnChamferVertices (float amount, TimeValue t);
	void EpfnChamferEdges (float amount, TimeValue t);

	// Extrude/Bevel/Chamfer methods:
	bool DoExtrusion(int msl, DWORD flag=MN_SEL);
	void DoChamfer(int msl);
	void EpfnBeginExtrude (int msl, DWORD flag, TimeValue t);
	void EpfnEndExtrude (bool accept, TimeValue t);
	void EpfnDragExtrude (float amount, TimeValue t);
	void EpfnBeginBevel (int msl, DWORD flag, bool doExtrude, TimeValue t);
	void EpfnEndBevel (bool accept, TimeValue t);
	void EpfnDragBevel (float outline, float height, TimeValue t);
	void EpfnBeginChamfer (int msl, TimeValue t);
	void EpfnEndChamfer (bool accept, TimeValue t);
	void EpfnDragChamfer (float amount, TimeValue t);

	bool EpfnCreateShape (TSTR name, bool smooth, INode *myNode, DWORD edgeFlag=MN_SEL);
	bool EpfnMakePlanar (int msl, DWORD flag=MN_SEL, TimeValue t=0);
	bool EpfnMoveToPlane (Point3 planeNormal, float planeOffset, int msl, DWORD flag=MN_SEL, TimeValue t=0);
	bool EpfnAlignToGrid (int msl, DWORD flag=MN_SEL);
	bool EpfnAlignToView (int msl, DWORD flag=MN_SEL);
	bool EpfnMeshSmooth (int msl, DWORD flag=MN_SEL);
	bool EpfnTessellate (int msl, DWORD flag=MN_SEL);
	bool EpfnSlice (Point3 planeNormal, Point3 planeCenter, bool flaggedFacesOnly=false, DWORD faceFlags=MN_SEL);
	bool EpfnInSlicePlaneMode () { return sliceMode; }
	int EpfnCutVertex (int startv, Point3 destination, Point3 projDir);
	int EpfnCutEdge (int e1, float prop1, int e2, float prop2, Point3 projDir);
	int EpfnCutFace (int f1, Point3 p1, Point3 p2, Point3 projDir);
	bool EpfnWeldVerts (int vert1, int vert2, Point3 destination);
	bool EpfnWeldEdges (int edge1, int edge2);
	bool EpfnWeldFlaggedVerts (DWORD flag);
	bool EpfnWeldFlaggedEdges (DWORD flag);
	void EpfnForceSubdivision ();
	void EpfnSetDiagonal (int face, int corner1, int corner2);
	bool EpfnRetriangulate (DWORD flag=MN_SEL);
	bool EpfnFlipNormals (DWORD flag=MN_SEL);
	void EpfnSelectByMat (int index, bool clear, TimeValue t);
	void EpfnSelectBySmoothGroup (DWORD bits, BOOL clear, TimeValue t);
	void EpfnAutoSmooth (TimeValue t);

	// Slice plane accessors:
	void EpResetSlicePlane ();
	void EpGetSlicePlane (Point3 & planeNormal, Point3 & planeCenter, float *planeSize=NULL);
	void EpSetSlicePlane (Point3 & planeNormal, Point3 & planeCenter, float planeSize);

	// Vertex/Edge Data accessors:
	float GetVertexDataValue (int channel, int *numSel, bool *uniform, DWORD vertFlags, TimeValue t);
	float GetEdgeDataValue (int channel, int *numSel, bool *uniform, DWORD edgeFlags, TimeValue t);
	void SetVertexDataValue (int channel, float w, DWORD vertFlags, TimeValue t);
	void SetEdgeDataValue (int channel, float w, DWORD edgeFlags, TimeValue t);
	void ResetVertexData (int channel);
	void ResetEdgeData (int channel);
	void BeginPerDataModify (int mnSelLevel, int channel);
	bool InPerDataModify ();
	void EndPerDataModify (bool success);
	void UpdatePerDataDisplay (TimeValue t, int mnSelLev, int channel);

	// EPoly color accessors:
	Color GetVertexColor (bool *uniform=NULL, int *num=NULL, int mp=0, DWORD flag=MN_SEL, TimeValue=0);
	void SetVertexColor (Color clr, int mp=0, DWORD flag=MN_SEL, TimeValue t=0);
	Color GetFaceColor (bool *uniform=NULL, int *num=NULL, int mp=0, DWORD flag=MN_SEL, TimeValue t=0);
	void SetFaceColor (Color clr, int mp=0, DWORD flag=MN_SEL, TimeValue t=0);
	void BeginVertexColorModify (int mp=0);
	bool InVertexColorModify ();
	void EndVertexColorModify (bool success);

	// Support for above.
	void InitVertColors (int mp=0);

	void EpfnSelectVertByColor (BOOL add, BOOL sub, int mp=0, TimeValue t=0);

	// Face Property Accessors:
	int GetMatIndex (bool *determined);
	void SetMatIndex (int index, DWORD flag=MN_SEL);
	void GetSmoothingGroups (DWORD faceFlag, DWORD *anyFaces, DWORD *allFaces=NULL);
	// Note: needed "bool" version locally, but need to maintain support for EPoly interface.
	// "bool" version returns true if any faces were selected; false otherwise.
	void SetSmoothBits (DWORD bits, DWORD bitmask, DWORD flag) { LocalSetSmoothBits (bits, bitmask, flag); }
	bool LocalSetSmoothBits (DWORD bits, DWORD bitmask, DWORD flag);

	// Psuedo-command-mode, mixed in with real ones in polymodes.cpp:
	void EnterSliceMode ();
	void ExitSliceMode ();

	// UI code -- polyedui.cpp
	void InvalidateDialogElement (int elem);
	HWND GetDlgHandle(int dlgID);
	void UpdateSurfType ();
	void SetSoftSelDlgEnables();
	void SetGeomDlgEnables();
	void SetSubdivDlgEnables ();
	float GetPolyFaceThresh();
	void InvalidateSurfaceUI();
	void InvalidateNumberSelected ();
	void SetNumSelLabel();
	BOOL SplitSharedVertCol();

	// Copy displacement parameters from pblock to polyobject:
	void SetDisplacementParams ();
	// Copy displacement parameters from polyobject to pblock:
	void UpdateDisplacementParams ();
	// Engage a displacement approximation preset:
	void UseDisplacementPreset (int presetNumber);

	// NS: New SubObjType API
	int NumSubObjTypes();
	ISubObjType *GetSubObjType(int i);

	// IMeshSelect methods:
	DWORD GetSelLevel();
	void SetSelLevel(DWORD level);
	void LocalDataChanged();

	// IMeshSelectData methods:
	BitArray GetVertSel() { mm.getVertexSel(vsel); return vsel; }
	BitArray GetFaceSel() { mm.getFaceSel(fsel); return fsel; }
	BitArray GetEdgeSel() { if (mm.GetFlag (MN_MESH_FILLED_IN)) mm.getEdgeSel(esel); return esel; }
	bool EpGetVerticesByFlag (BitArray & vset, DWORD flags, DWORD fmask=0x0) { return mm.getVerticesByFlag (vset, flags, fmask); }
	bool EpGetEdgesByFlag (BitArray & eset, DWORD flags, DWORD fmask=0x0) { return mm.getEdgesByFlag (eset, flags, fmask); }
	bool EpGetFacesByFlag (BitArray & fset, DWORD flags, DWORD fmask=0x0) { return mm.getFacesByFlag (fset, flags, fmask); }
	BitArray GetSel (int nsl);
	void SetVertSel(BitArray &set, IMeshSelect *imod, TimeValue t);
	void SetFaceSel(BitArray &set, IMeshSelect *imod, TimeValue t);
	void SetEdgeSel(BitArray &set, IMeshSelect *imod, TimeValue t);
	void EpSetVertexFlags (BitArray &vset, DWORD flags, DWORD fmask=0x0, bool undoable=true);
	void EpSetEdgeFlags (BitArray &eset, DWORD flags, DWORD fmask = 0x0, bool undoable=true);
	void EpSetFaceFlags (BitArray &fset, DWORD flags, DWORD fmask = 0x0, bool undoable=true);
	void SetSel (int nsl, BitArray & set, IMeshSelect *imod, TimeValue t);
	GenericNamedSelSetList & GetNamedVertSelList () { return selSet[NS_VERTEX]; }
	GenericNamedSelSetList & GetNamedEdgeSelList () { return selSet[NS_EDGE]; }
	GenericNamedSelSetList & GetNamedFaceSelList () { return selSet[NS_FACE]; }

	// EPoly methods:
	void LocalDataChanged (DWORD parts);
	void RefreshScreen ();
	void EnterCommandMode (int mode);
	CommandMode *getCommandMode (int mode);
	IParamBlock2 *getParamBlock () { return pblock; }
	void EpActionToggleCommandMode (int mode);
	void EpActionEnterPickMode (int mode);
	void EpActionButtonOp (int opcode);
	void EpActionExitCommandModes () { ExitAllCommandModes (TRUE); }
	void MoveSelection(int level, TimeValue t, Matrix3& partm, Matrix3& tmAxis, Point3& val, BOOL localOrigin);
	void RotateSelection(int level, TimeValue t, Matrix3& partm, Matrix3& tmAxis, Quat& val, BOOL localOrigin);
	void ScaleSelection(int level, TimeValue t, Matrix3& partm, Matrix3& tmAxis, Point3& val, BOOL localOrigin);
	bool Editing () { return (ip && (editObj==this)) ? TRUE : FALSE; }
	int GetEPolySelLevel() { return selLevel; }
	int GetMNSelLevel() { return meshSelLevel[selLevel]; }
	void SetEPolySelLevel (int sl) { if (ip) ip->SetSubObjectLevel (sl); else selLevel=sl; }
	int EpfnPropagateComponentFlags (int slTo, DWORD flTo, int slFrom, DWORD flFrom, bool ampersand=FALSE, bool set=TRUE, bool undoable=FALSE);

	// Neversoft Extensions
	void SelectByFaceFlags( FlagType &flags, bool exact_match );
	int	GetFaceFlagsSet( HWND hwnd, FlagType &diff_flags );
	void SetFaceFlagMask( HWND parent, FlagType mask, FlagType diff_flags );
	void SetFaceFlags( FlagType flags );
	void ClearFaceFlags( FlagType flags );
	void DisplayFaceFlags( void );

	// ISubMtlAPI methods:
	void*	GetInterface(ULONG id);
	MtlID	GetNextAvailMtlID(ModContext* mc);
	BOOL	HasFaceSelection(ModContext* mc);
	void	SetSelFaceMtlID(ModContext* mc, MtlID id, BOOL bResetUnsel = FALSE);
	int		GetSelFaceUniqueMtlID(ModContext* mc);
	int		GetSelFaceAnyMtlID(ModContext* mc);
	int		GetMaxMtlID(ModContext* mc);

	// Access to EPoly interface:
	BaseInterface *GetInterface (Interface_ID id);

	// EventUser methods:
	void Notify() { EpActionButtonOp (epop_delete); /*delete key was pressed*/}
};

// Backspace event user:
class EPolyBackspaceUser : public EventUser {
	EditPolyObject *epo;
public:
	void Notify();
	void SetEPoly (EditPolyObject *e) { epo = e; }
};

// --- Command Modes & Mouse Procs -------------------------------

// Virtual mouse procs: these are useful for multiple actual procs.
class PickEdgeMouseProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;

	PickEdgeMouseProc(EditPolyObject* e, IObjParam *i) {ep=e;ip=i;}
	HitRecord *HitTestEdges(IPoint2 &m, ViewExp *vpt, float *prop, Point3 *snapPoint);
	virtual int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m );
	virtual void EdgePick(int edge, float prop)=0;
};

class PickFaceMouseProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	int face;	// Face we picked
	Tab<float> bary;
	//Tab<int> tri;	// triangulation of face
	//int tid;	// triangle where we hit
	//double bary[3];	// barycentric coordinates of hit.

	PickFaceMouseProc(EditPolyObject* e, IObjParam *i) {ep=e;ip=i;}
	HitRecord *HitTestFaces(IPoint2 &m, ViewExp *vpt);
	void ProjectHitToFace (IPoint2 &m, ViewExp *vpt, HitRecord *hr, Point3 *snapPoint);
	virtual int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m);
	virtual void FacePick()=0;
};

// Used in both create edge and edit triangulation:
class ConnectVertsMouseProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	int v1, v2;
	Tab<int> neighbors;
	IPoint2 m1, lastm;

	ConnectVertsMouseProc (EPoly *e, IObjParam *i) { ep=e; ip=i; }
	HitRecord *HitTestVertices (IPoint2 & m, ViewExp *vpt);
	void DrawDiag (HWND hWnd, const IPoint2 & m);
	void SetV1 (int vv);
	int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m);
	virtual void VertConnect () {}
};

// Actual procs & command modes:

class CreateVertMouseProc : public MouseCallBack {
private:
	EPoly *ep;
	IObjParam *ip;
public:
	CreateVertMouseProc (EPoly* mod, IObjParam *i) {ep=mod;ip=i;}
	int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m);
};

class CreateVertCMode : public CommandMode {
private:
	ChangeFGObject fgProc;
	CreateVertMouseProc proc;
	EditPolyObject* ep;

public:
	CreateVertCMode(EditPolyObject* mod, IObjParam *i) : fgProc(mod), proc(mod,i) {ep=mod;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CREATEVERT; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=1; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

class CreateEdgeMouseProc : public ConnectVertsMouseProc {
public:
	CreateEdgeMouseProc (EditPolyObject *e, IObjParam *i) : ConnectVertsMouseProc (e,i) {}
	void VertConnect ();
};

class CreateEdgeCMode : public CommandMode {
private:
	ChangeFGObject fgProc;
	CreateEdgeMouseProc proc;
	EditPolyObject* ep;

public:
	CreateEdgeCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CREATEEDGE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=2; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

class CreateFaceMouseProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	Tab<int> vts;
	Tab<IPoint2> mpts;
	IPoint2 oldm;
	int pt;

	CreateFaceMouseProc(EditPolyObject* e, IObjParam *i);
	void DrawEstablishedFace (GraphicsWindow *gw);
	void DrawCreatingFace (HWND hWnd, const IPoint2 & m);
	BOOL HitTestVerts(IPoint2 m, ViewExp *vpt,int &v);
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m );
	void Backspace ();
};

class CreateFaceCMode : public CommandMode {
public:
	ChangeFGObject fgProc;
	EditPolyObject *ep;
	CreateFaceMouseProc proc;

	CreateFaceCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CREATEFACE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=999999; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

class AttachPickMode : public PickModeCallback, public PickNodeCallback {
public:
	EditPolyObject* ep;
	IObjParam *ip;

	AttachPickMode(EditPolyObject* o, IObjParam *i) { ep=o; ip=i; }
	BOOL HitTest(IObjParam *ip,HWND hWnd,ViewExp *vpt,IPoint2 m,int flags);
	BOOL Pick(IObjParam *ip,ViewExp *vpt);
	void EnterMode(IObjParam *ip);
	void ExitMode(IObjParam *ip);		

	BOOL Filter(INode *node);
	BOOL RightClick(IObjParam *ip,ViewExp *vpt) {return TRUE;}
	PickNodeCallback *GetFilter() {return this;}
};

class AttachHitByName : public HitByNameDlgCallback {
public:
	EditPolyObject *ep;
	bool inProc;

	AttachHitByName (EditPolyObject *e) {ep=e; inProc=FALSE;}
	TCHAR *dialogTitle() { return GetString(IDS_ATTACH_LIST); }
	TCHAR *buttonText() { return GetString(IDS_ATTACH); }
	int filter(INode *node);	
	void proc(INodeTab &nodeTab);	
};

class DivideEdgeProc : public PickEdgeMouseProc {
public:
	DivideEdgeProc(EditPolyObject* e, IObjParam *i) : PickEdgeMouseProc(e,i) {}
	void EdgePick(int edge, float prop);
};

class DivideEdgeCMode : public CommandMode {
private:
	ChangeFGObject fgProc;		
	DivideEdgeProc proc;
	EditPolyObject* ep;

public:
	DivideEdgeCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_DIVIDEEDGE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=1; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

class DivideFaceProc : public PickFaceMouseProc {
public:
	DivideFaceProc(EditPolyObject* e, IObjParam *i) : PickFaceMouseProc(e,i) {}
	void FacePick();
};

class DivideFaceCMode : public CommandMode {
private:
	ChangeFGObject fgProc;		
	DivideFaceProc proc;
	EditPolyObject* ep;

public:
	DivideFaceCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_DIVIDEFACE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=1; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

class ExtrudeProc : public MouseCallBack {
public:
	MoveTransformer moveTrans;
	EditPolyObject *eo;
	Interface *ip;
	IPoint2 om;

	ExtrudeProc(EditPolyObject* o, IObjParam *i) : moveTrans(i) {eo=o;ip=i;}
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m);
};

class ExtrudeSelectionProcessor : public GenModSelectionProcessor {
protected:
	HCURSOR GetTransformCursor();
public:
	ExtrudeSelectionProcessor(ExtrudeProc *mc, EditPolyObject *o, IObjParam *i) 
		: GenModSelectionProcessor(mc,o,i) {}
};

class ExtrudeCMode : public CommandMode {
private:
	ChangeFGObject fgProc;
	ExtrudeSelectionProcessor mouseProc;
	ExtrudeProc eproc;
	EditPolyObject* eo;

public:
	ExtrudeCMode(EditPolyObject* o, IObjParam *i) :
		fgProc(o), mouseProc(&eproc,o,i), eproc(o,i) {eo=o;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_EXTRUDE; }
	MouseCallBack *MouseProc(int *numPoints) { *numPoints=2; return &mouseProc; }
	ChangeForegroundCallback *ChangeFGProc() { return &fgProc; }
	BOOL ChangeFG( CommandMode *oldMode ) { return oldMode->ChangeFGProc() != &fgProc; }
	void EnterMode();
	void ExitMode();
};

class ChamferMouseProc : public MouseCallBack {
private:
	MoveTransformer moveTrans;
	EditPolyObject *eo;
	Interface *ip;
	IPoint2 om;
public:
	ChamferMouseProc (EditPolyObject* o, IObjParam *i) : moveTrans(i) {eo=o;ip=i;}
	int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m);
};

class ChamferSelectionProcessor : public GenModSelectionProcessor {
protected:
	HCURSOR GetTransformCursor();
public:
	EditPolyObject *eto;
	ChamferSelectionProcessor(ChamferMouseProc *mc, EditPolyObject *o, IObjParam *i) 
		: GenModSelectionProcessor(mc,o,i) {eto=o;}
};

class ChamferCMode : public CommandMode {
private:
	ChangeFGObject fgProc;
	ChamferSelectionProcessor mouseProc;
	ChamferMouseProc eproc;
	EditPolyObject* eo;

public:
	ChamferCMode (EditPolyObject* o, IObjParam *i) :
		fgProc(o), mouseProc(&eproc,o,i), eproc(o,i) {eo=o;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_POLYCHAMFER; }
	MouseCallBack *MouseProc(int *numPoints) { *numPoints=2; return &mouseProc; }
	ChangeForegroundCallback *ChangeFGProc() { return &fgProc; }
	BOOL ChangeFG( CommandMode *oldMode ) { return oldMode->ChangeFGProc() != &fgProc; }
	void EnterMode();
	void ExitMode();
};

class BevelMouseProc : public MouseCallBack {
private:
	MoveTransformer moveTrans;
	EditPolyObject *eo;
	Interface *ip;
	IPoint2 m0, m1;
	bool m0set, m1set;
	float height;
public:
	BevelMouseProc(EditPolyObject* o, IObjParam *i) : moveTrans(i) {eo=o;ip=i; m0set=FALSE; m1set=FALSE; }
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m);
};

class BevelSelectionProcessor : public GenModSelectionProcessor {
protected:
	HCURSOR GetTransformCursor();
public:
	BevelSelectionProcessor(BevelMouseProc *mc, EditPolyObject *o, IObjParam *i) 
		: GenModSelectionProcessor(mc,o,i) {}
};

class BevelCMode : public CommandMode {
private:
	ChangeFGObject fgProc;
	BevelSelectionProcessor mouseProc;
	BevelMouseProc eproc;
	EditPolyObject* eo;

public:
	BevelCMode(EditPolyObject* o, IObjParam *i) :
		fgProc(o), mouseProc(&eproc,o,i), eproc(o,i) {eo=o;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_BEVEL; }
	MouseCallBack *MouseProc(int *numPoints) { *numPoints=3; return &mouseProc; }
	ChangeForegroundCallback *ChangeFGProc() { return &fgProc; }
	BOOL ChangeFG( CommandMode *oldMode ) { return oldMode->ChangeFGProc() != &fgProc; }
	void EnterMode();
	void ExitMode();
};

class CutVertProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	int v1;
	IPoint2 m1, oldm2;

	CutVertProc (EPoly* e, IObjParam *i) { ep=e; ip=i; v1=-1;}
	HitRecord *HitTestVerts(IPoint2 &m, ViewExp *vpt);
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m );
	void DrawCutter (HWND hWnd,IPoint2 &m);
};

class CutVertCMode : public CommandMode {
private:
	ChangeFGObject fgProc;	
	CutVertProc proc;
	EditPolyObject* ep;

public:
	CutVertCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CUTEDGE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=999; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
	void AbandonCut ();
};

class CutEdgeProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	int e1;
	bool e1set;
	float prop1;
	IPoint2 m1, oldm2;

	CutEdgeProc(EPoly *e, IObjParam *i) { ep=e; ip=i; e1set = FALSE;}
	HitRecord *HitTestEdges(IPoint2 &m, ViewExp *vpt);
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m );
	void DrawCutter (HWND hWnd,IPoint2 &m);
};

class CutEdgeCMode : public CommandMode {
private:
	ChangeFGObject fgProc;	
	CutEdgeProc proc;
	EditPolyObject* ep;

public:
	CutEdgeCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CUTEDGE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=999; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
	void AbandonCut ();
};

class CutFaceProc : public MouseCallBack {
public:
	EPoly *ep;
	IObjParam *ip;
	int f1;
	Point3 p1;
	IPoint2 m1, oldm2;

	CutFaceProc(EPoly *e, IObjParam *i) { ep=e; ip=i; f1 = -1; }
	HitRecord *HitTestFaces(IPoint2 &m, ViewExp *vpt);
	int proc(HWND hwnd, int msg, int point, int flags, IPoint2 m );
	void DrawCutter (HWND hWnd,IPoint2 &m);
	void ProjectHitToFace (IPoint2 &m, ViewExp *vpt, HitRecord *hr,
		Point3 *snapPoint, int & face, Point3 & hitPoint);
};

class CutFaceCMode : public CommandMode {
private:
	ChangeFGObject fgProc;	
	CutFaceProc proc;
	EditPolyObject* ep;

public:
	CutFaceCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_CUTEDGE; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=999; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
	void AbandonCut ();
};

class WeldMouseProc : public MouseCallBack {
public:
	EditPolyObject *ep;
	IObjParam *ip;
	int targ1;
	IPoint2 m1, oldm2;

	WeldMouseProc(EditPolyObject *e, IObjParam *i) { ep=e; ip=i; targ1=-1; }
	void DrawWeldLine (HWND hWnd, IPoint2 &m);
	HitRecord *HitTest (IPoint2 &m, ViewExp *vpt);
	int proc (HWND hwnd, int msg, int point, int flags, IPoint2 m);
};

class WeldCMode : public CommandMode {
public:
	ChangeFGObject fgProc;
	WeldMouseProc mproc;
	EditPolyObject* ep;

	WeldCMode(EditPolyObject* mod, IObjParam *i) :
		fgProc(mod), mproc(mod,i) {ep=mod;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_WELD; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=2; return &mproc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc; }
	BOOL ChangeFG( CommandMode *oldMode ) {return oldMode->ChangeFGProc() != &fgProc;}
	void EnterMode();
	void ExitMode();
};

class EditTriProc : public ConnectVertsMouseProc {
public:
	EditTriProc(EditPolyObject* e, IObjParam *i) : ConnectVertsMouseProc(e,i) { }
	void VertConnect ();
};

class EditTriCMode : public CommandMode {
public:
	ChangeFGObject fgProc;
	EditTriProc proc;
	EditPolyObject* ep;

	EditTriCMode(EditPolyObject* e, IObjParam *i) : fgProc(e), proc(e,i) {ep=e;}
	int Class() { return MODIFY_COMMAND; }
	int ID() { return CID_EDITTRI; }
	MouseCallBack *MouseProc(int *numPoints) {*numPoints=1; return &proc;}
	ChangeForegroundCallback *ChangeFGProc() {return &fgProc;}
	BOOL ChangeFG(CommandMode *oldMode) {return oldMode->ChangeFGProc()!= &fgProc;}
	void EnterMode();
	void ExitMode();
};

// --- Restore objects ---------------------------------------------

// Not really a restore object, just used in some drag moves.
class TempMoveRestore {
public:
	Tab<Point3> init;
	Tab<UVVert> *mapInit;
	BitArray active;

	TempMoveRestore (EditPolyObject *em);
	~TempMoveRestore ();
	void Restore (EditPolyObject *em);
};

class ComponentFlagRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	int sl;
	Tab<DWORD> undo, redo;

	ComponentFlagRestore (EditPolyObject *editObj, int selLevel);
	void Restore (int isUndo);
	void Redo ();
	TSTR Description () { return TSTR(_T("Component flag change restore")); }
	int Size () { return sizeof(void *) + sizeof(int) + 2*sizeof(Tab<DWORD>); }
};

class CueDragRestore : public RestoreObj {
public:
	EditPolyObject *editObj;

	CueDragRestore (EditPolyObject *eo) { editObj = eo; }
	void Restore(int isUndo) { editObj->DragMoveRestore(); }
	void Redo() { }
	TSTR Description() {return TSTR(_T("Cue drag move Restore"));}
	int Size() { return sizeof(int) + sizeof(void *); }
};

class PerDataRestore;

// Made to track "local" topological changes to an MNMesh.
// (No collapseDead's allowed between Before and After.)
// (Creations are ok.)
class TopoChangeRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	MNMesh start;
	Tab<int> vertID, edgeID, faceID;
	Tab<MNVert> vertsBefore, vertsAfter;
	Tab<int> *vfacBefore, *vfacAfter;
	Tab<int> *vedgBefore, *vedgAfter;
	Tab<MNEdge> edgesBefore, edgesAfter;
	Tab<MNFace> facesBefore, facesAfter;
	int numvBefore, numvAfter, numeBefore, numeAfter, numfBefore, numfAfter;
	Tab<int> numMvBefore, numMvAfter;
	Tab<int> *mapVertID, *mapFaceID;
	Tab<UVVert> *mapVertsBefore, *mapVertsAfter;
	Tab<MNMapFace> *mapFacesBefore, *mapFacesAfter;
	Tab<float> *vertexDataBefore, *vertexDataAfter;
	Tab<float> *edgeDataBefore, *edgeDataAfter;
	int vertexDataCount, edgeDataCount;

	TopoChangeRestore (EditPolyObject *editObj);
	~TopoChangeRestore ();
	void Before () { start = ep->mm; }
	void After ();
	void Restore (int isUndo);
	void Redo ();
	TSTR Description () { return TSTR(_T("EPoly Topology Change Restore")); }
	int Size () { return sizeof(void *) + 9*sizeof (Tab<int>) + sizeof (MNMesh) + 4*sizeof(Tab<int> *) + 6*sizeof(int); }
};

class MapChangeRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	int mp;
	Tab<UVVert> preVerts;
	Tab<MNMapFace> preFaces;
	Tab<int> vertID, faceID;
	Tab<UVVert> vbefore, vafter;
	Tab<MNMapFace> fbefore, fafter;
	int numvBefore, numvAfter;
	int numfBefore, numfAfter;
	DWORD mapFlagsBefore, mapFlagsAfter;

	MapChangeRestore (EditPolyObject *editObj, int mapchan);
	~MapChangeRestore ();
	bool ReduceMem ();
	void Restore (int isUndo);
	void Redo ();
	TSTR Description () { return TSTR(_T("Map change restore")); }
	int Size () { return sizeof(void *) + 4*sizeof(int) + 8*sizeof(Tab<int>) + 2*sizeof(DWORD); }
};

// MapVertRestore is used when only the values of the map vertices change.
// (Not their number, or any map faces.)
class MapVertRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	int mp;
	// Used to store the original map vertices before ReduceMem()
	Tab<UVVert> preVerts;
	// Used to store only the changed map vertices after ReduceMem();
	Tab<int> vertID;
	Tab<UVVert> vbefore, vafter;

	MapVertRestore (EditPolyObject *editObj, int mapchan);
	bool ReduceMem ();
	void Restore (int isUndo);
	void Redo ();
	TSTR Description () { return TSTR(_T("Map vertex restore")); }
	int Size () { return sizeof(void *) + sizeof(int) + sizeof(Tab<int>) + 3*sizeof(Tab<UVVert>); }
};

// Following intended only for operations that purely create, not those that modify existing topology.
class CreateOnlyRestore : public RestoreObj {
public:
	EditPolyObject *epol;
	int ovnum, oenum, ofnum;
	Tab<int> omvnum, omfnum;
	Tab<MNVert> nverts;
	Tab<int> *nvfac;
	Tab<int> *nvedg;
	Tab<MNEdge> nedges;
	Tab<MNFace> nfaces;
	Tab<MNMapFace> *nmfaces;
	Tab<UVVert> *nmverts;
	BitArray omapsUsed, nmapsUsed;
	bool afterCalled;

	CreateOnlyRestore (EditPolyObject *ep);
	~CreateOnlyRestore ();
	void After ();
	void Restore (BOOL isUndo);
	void Redo ();
	TSTR Description () { return TSTR(_T("Creation restore")); }
	int Size () { return sizeof(void *) + 3*sizeof(int) + 3*sizeof(Tab<MNVert>); }
};

class MeshVertRestore : public RestoreObj {
public:		   	
	Tab<Point3> undo, redo;
	EditPolyObject *ep;

	MeshVertRestore(EditPolyObject *ep);
	void Restore(int isUndo);
	void Redo();
	void EndHold() {ep->ClearAFlag(A_HELD);}
	TSTR Description() {return TSTR(_T("Mesh Geometry Change"));}
};

class MtlIDRestore : public RestoreObj {	
public:		   	
	Tab<MtlID> undo, redo;
	EditPolyObject *ep;		

	MtlIDRestore (EditPolyObject *ep);
	void After ();
	void Restore(int isUndo);
	void Redo();
	void EndHold() {ep->ClearAFlag(A_HELD);}
	TSTR Description() {return TSTR(_T("Mesh Face Mat"));}
};

class SmGroupRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	Tab<DWORD> osmg, nsmg;

	SmGroupRestore (EditPolyObject *e);
	void After ();
	void Restore (int isUndo);
	void Redo ();
	TSTR Description() {return TSTR(_T("MatID Restore"));}
};

class CollapseDeadVertsRestore : public RestoreObj {	
public:
	Tab<MNVert> undo;
	Tab<int> *undovedg, *undovfac;
	Tab<int> positions;
	int startNum;
	EditPolyObject *ep;		

	CollapseDeadVertsRestore (EditPolyObject *ep);
	void Restore(int isUndo);
	void Redo();
	void MyDebugPrint ();
	TSTR Description() {return TSTR(_T("Collapse Dead Verts Restore"));}
	int Size() { return sizeof(Tab<int>) + sizeof(Tab<MNVert>) + sizeof(void *); }
};

class CollapseDeadEdgesRestore : public RestoreObj {	
public:
	Tab<MNEdge> undo;
	Tab<int> positions;
	EditPolyObject *ep;		

	CollapseDeadEdgesRestore (EditPolyObject *ep);
	void Restore(int isUndo);
	void Redo();
	TSTR Description() {return TSTR(_T("CollapseDeadEdgesRestore"));}
	int Size() { return sizeof(Tab<int>) + sizeof(Tab<MNEdge>) + sizeof(void *); }
};

class CollapseDeadFacesRestore : public RestoreObj {	
public:
	Tab<MNFace> undo;
	Tab<int> positions;
	Tab<MNMapFace *> undoMap;
	EditPolyObject *ep;		

	CollapseDeadFacesRestore (EditPolyObject *ep);
	~CollapseDeadFacesRestore ();
	void Restore(int isUndo);
	void Redo();
	TSTR Description() {return TSTR(_T("Collapse Dead Faces Restore"));}
	int Size() { return sizeof(Tab<int>) + sizeof(Tab<MNFace>) + sizeof(Tab<MNMapFace *>) + sizeof(void *); }
};

class PerDataRestore : public RestoreObj {
public:
	EditPolyObject *ep;
	int msl, channel, num;
	Tab<float> oldData;
	Tab<float> newData;

	PerDataRestore (EditPolyObject *eo, int mnSelLev, int chan);
	void After ();
	void Restore (int isUndo);
	void Redo ();
	int Size () { return sizeof(EditPolyObject *) + 3*sizeof(int) + 2*sizeof(Tab<float>); }
	TSTR Description() { return _T("PerData Change Restore"); }
};

class InitVertColorRestore : public RestoreObj {
public:
	EditPolyObject *epol;
	int mapChannel;
	InitVertColorRestore (EditPolyObject *e, int mp) { epol = e; mapChannel=mp; }
	void Restore (int isUndo) { epol->mm.M(mapChannel)->SetFlag (MN_DEAD); }
	void Redo () { epol->InitVertColors (mapChannel); }
	TSTR Description () { return TSTR(_T("Initializing vertex colors")); }
	int Size () { return sizeof(void *); }
};

class AppendSetRestore : public RestoreObj {
public:
	BitArray set;
	TSTR name;
	GenericNamedSelSetList *setList;
	EditPolyObject *ep;

	AppendSetRestore(GenericNamedSelSetList *sl,EditPolyObject *e) {
		setList = sl; ep = e;
	}
	void Restore(int isUndo) {
		set  = *setList->sets[setList->Count()-1];
		name = *setList->names[setList->Count()-1];
		setList->DeleteSet(setList->Count()-1);
		if (ep->ip) {
			ep->ip->NamedSelSetListChanged();
			ep->UpdateNamedSelDropDown ();
		}
	}
	void Redo() {
		setList->AppendSet(set, 0, name);
		if (ep->ip) {
			ep->ip->NamedSelSetListChanged();
			ep->UpdateNamedSelDropDown ();
		}
	}
			
	TSTR Description() {return TSTR(_T("Append Set"));}
};

class DeleteSetRestore : public RestoreObj {
public:
	BitArray set;
	TSTR name;
	int index;
	GenericNamedSelSetList *setList;
	EditPolyObject *ep;

	DeleteSetRestore(TSTR nm, GenericNamedSelSetList *sl,EditPolyObject *e) {
		setList = sl;
		ep = e;
		set  = *(sl->GetSet(nm));
		name = nm;
	}
	void Restore(int isUndo) {
		setList->AppendSet (set, 0, name);
		if (ep->ip) {
			ep->ip->NamedSelSetListChanged();
			ep->UpdateNamedSelDropDown ();
		}
	}
	void Redo() {
		setList->RemoveSet (name);
		if (ep->ip) {
			ep->ip->NamedSelSetListChanged();
			ep->UpdateNamedSelDropDown ();
		}
	}
			
	TSTR Description() {return TSTR(_T("Delete Set"));}
};

class SetNameRestore : public RestoreObj {
public:
	TSTR undo, redo;
	int index;
	GenericNamedSelSetList *setList;
	EditPolyObject *ep;
	SetNameRestore(int i,GenericNamedSelSetList *sl,EditPolyObject *e) {
		index = i; setList = sl; ep = e;
		undo = *setList->names[index];
	}

	void Restore(int isUndo) {			
		redo = *setList->names[index];
		*setList->names[index] = undo;
		if (ep->ip) ep->ip->NamedSelSetListChanged();
	}
	void Redo() {
		*setList->names[index] = redo;
		if (ep->ip) ep->ip->NamedSelSetListChanged();
	}
			
	TSTR Description() {return TSTR(_T("Set Name"));}
};

class TransformPlaneRestore : public RestoreObj {
public:
	Point3 oldSliceCenter, newSliceCenter;
	Quat oldSliceRot, newSliceRot;
	float oldSliceSize, newSliceSize;
	EditPolyObject *eo;
	TransformPlaneRestore (EditPolyObject *eto) {
		eo = eto;
		oldSliceCenter = eo->sliceCenter;
		oldSliceRot = eo->sliceRot;
		oldSliceSize = eo->sliceSize;
	}
	void Restore (int isUndo) {
		newSliceCenter = eo->sliceCenter;
		newSliceRot = eo->sliceRot;
		newSliceSize = eo->sliceSize;
		eo->sliceCenter = oldSliceCenter;
		eo->sliceRot = oldSliceRot;
		eo->sliceSize = oldSliceSize;
		eo->NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	}
	void Redo () {
		oldSliceCenter = eo->sliceCenter;
		oldSliceRot = eo->sliceRot;
		oldSliceSize = eo->sliceSize;
		eo->sliceCenter = newSliceCenter;
		eo->sliceRot = newSliceRot;
		eo->sliceSize = newSliceSize;
		eo->NotifyDependents(FOREVER, PART_DISPLAY, REFMSG_CHANGE);
	}
	int Size () {
		return 2*(sizeof(Point3) + sizeof(Quat) + sizeof(float))
			+ sizeof (EditPolyObject *);
	}
	TSTR Description () { return TSTR (_T("Slice Plane transform")); }
};

// PolyEdUI.cpp functions:
bool GetCreateShapeName (Interface *ip, TSTR &name, bool &ccSmooth);
bool GetDetachObjectName (Interface *ip, TSTR &name, bool &elem, bool &clone);
BOOL GetCloneObjectName (Interface *ip, TSTR &name);
bool GetSelectByMaterialParams (Interface *ip, MtlID &matId, bool &clear);
bool GetSelectBySmoothParams (Interface *ip, DWORD usedBits, DWORD &smBits, bool &clear);

// triops.cpp
bool CreateCurveFromMeshEdges (MNMesh & mesh, INode *node, Interface *ip, TSTR & name, bool curved, DWORD flag);
BOOL CALLBACK DispApproxDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam);

}	// end namespace EditPoly

#endif //__TRIOBJED__
