/**********************************************************************
 *<
	FILE: PolyEdit.cpp

	DESCRIPTION:   Editable Polygon Mesh Object

	CREATED BY: Steve Anderson

	HISTORY: created November 1999

 *>	Copyright (c) 1999, All Rights Reserved.
 **********************************************************************/

#include "EPoly.h"
#include "PolyEdit.h"
#include "MeshDLib.h"
#include "macrorec.h"

namespace EditPoly
{

#define vTRI_OBJ_FACE_FLAGS_CHUNK	0x5000	// WARNING! Change this in triobjed.cpp to maintain compat

//--- Class vars -------------------------------

Interface *			EditPolyObject::ip              = NULL;
EditPolyObject *	EditPolyObject::editObj			= NULL;
IParamMap2 *		EditPolyObject::pSelect		= NULL;
IParamMap2 *		EditPolyObject::pSoftsel	= NULL;
IParamMap2 *		EditPolyObject::pGeom		= NULL;
IParamMap2 *		EditPolyObject::pSubdiv		= NULL;
IParamMap2 *		EditPolyObject::pSurface	= NULL;
IParamMap2 *		EditPolyObject::pDisp   	= NULL;
bool				EditPolyObject::rsSelect		= FALSE;
bool				EditPolyObject::rsSoftsel		= TRUE;
bool				EditPolyObject::rsGeom			= FALSE;
bool				EditPolyObject::rsSubdiv		= TRUE;
bool				EditPolyObject::rsSurface		= FALSE;
bool				EditPolyObject::rsDisp			= FALSE;
bool				EditPolyObject::inExtrude    = FALSE;
bool				EditPolyObject::inBevel = FALSE;
bool				EditPolyObject::inChamfer = FALSE;

// New for FaceFlags   aml
HWND				EditPolyObject::hFaceFlags	= NULL;
BOOL				EditPolyObject::rsFaceFlags = FALSE;

////////////////////////////////////////////////////////////////////

MoveModBoxCMode*    EditPolyObject::moveMode        = NULL;
RotateModBoxCMode*  EditPolyObject::rotMode			= NULL;
UScaleModBoxCMode*  EditPolyObject::uscaleMode      = NULL;
NUScaleModBoxCMode* EditPolyObject::nuscaleMode     = NULL;
SquashModBoxCMode*  EditPolyObject::squashMode      = NULL;
SelectModBoxCMode*  EditPolyObject::selectMode      = NULL;
CreateVertCMode*	EditPolyObject::createVertMode  = NULL;
CreateEdgeCMode*	EditPolyObject::createEdgeMode	= NULL;
CreateFaceCMode*	EditPolyObject::createFaceMode	= NULL;
AttachPickMode*		EditPolyObject::attachPickMode	= NULL;
DivideEdgeCMode*	EditPolyObject::divideEdgeMode	= NULL;
DivideFaceCMode *	EditPolyObject::divideFaceMode	= NULL;
ExtrudeCMode *		EditPolyObject::extrudeMode		= NULL;
ChamferCMode *		EditPolyObject::chamferMode		= NULL;
BevelCMode*			EditPolyObject::bevelMode		= NULL;
CutVertCMode *		EditPolyObject::cutVertMode		= NULL;
CutEdgeCMode *		EditPolyObject::cutEdgeMode		= NULL;
CutFaceCMode *		EditPolyObject::cutFaceMode		= NULL;
WeldCMode *			EditPolyObject::weldMode		= NULL;
EditTriCMode *		EditPolyObject::editTriMode		= NULL;
int EditPolyObject::pickBoxSize     = DEF_PICKBOX_SIZE;
int EditPolyObject::attachMat = 0;
int EditPolyObject::condenseMat = 0;
Quat				EditPolyObject::sliceRot(0.0f,0.0f,0.0f,1.0f);
Point3				EditPolyObject::sliceCenter(0.0f,0.0f,0.0f);
float				EditPolyObject::sliceSize = 100.0f;
bool				EditPolyObject::sliceMode		= false;
EPolyBackspaceUser EditPolyObject::backspacer;

TempMoveRestore * EditPolyObject::tempMove = NULL;

//--- EditPolyObject methods ----------------------------------

EditPolyObject::EditPolyObject() {
	tempData = NULL;
	pblock = NULL;
	masterCont = NULL;
	selLevel = EP_SL_OBJECT;
	arValid.SetEmpty();
	ePolyFlags = 0;
	sliceInitialized = false;
	GetEditablePolyDesc()->MakeAutoParamBlocks(this);

	theHold.Suspend();
	MakeRefByID (FOREVER, EPOLY_MASTER_CONTROL_REF, NewDefaultMasterPointController());
	theHold.Resume();
}

EditPolyObject::~EditPolyObject() {
	if (tempData) delete tempData;

	theHold.Suspend ();
	DeleteAllRefsFromMe ();
	theHold.Resume ();
}

void EditPolyObject::ResetClassParams (BOOL fileReset) {
	EditPolyObject::rsSelect		= FALSE;
	EditPolyObject::rsSoftsel		= TRUE;
	EditPolyObject::rsGeom			= FALSE;
	EditPolyObject::rsSubdiv		= TRUE;
	EditPolyObject::rsSurface		= FALSE;
	EditPolyObject::rsDisp			= FALSE;
	EditPolyObject::inExtrude    = FALSE;
	EditPolyObject::inBevel = FALSE;
	EditPolyObject::inChamfer = FALSE;
	EditPolyObject::pickBoxSize     = DEF_PICKBOX_SIZE;
	EditPolyObject::attachMat = 0;
	EditPolyObject::condenseMat = 0;
	EditPolyObject::sliceRot = Quat(0.0f,0.0f,0.0f,1.0f);
	EditPolyObject::sliceCenter = Point3(0.0f,0.0f,0.0f);
	EditPolyObject::sliceSize = 100.0f;
	EditPolyObject::sliceMode		= false;
}

RefTargetHandle EditPolyObject::Clone(RemapDir& remap) {
	EditPolyObject *npol = new EditPolyObject;
	npol->mm = mm;
	npol->ReplaceReference(EPOLY_PBLOCK,pblock->Clone(remap));
	for (int i=0; i<cont.Count(); i++) {
		if (cont[i] == NULL) continue;
		npol->MakeRefByID (FOREVER, EPOLY_VERT_BASE_REF + i, cont[i]->Clone (remap));
	}
	npol->SetDisplacementParams ();
	BaseClone(this, npol, remap);
	return npol;
}

BOOL EditPolyObject::IsSubClassOf(Class_ID classID) {
	return classID==ClassID() || classID==polyObjectClassID;	
}

Object *EditPolyObject::ConvertToType (TimeValue t, Class_ID obtype) {
	int useSubdiv=false;
	if (pblock) pblock->GetValue (ep_surf_subdivide, t, useSubdiv, FOREVER);
	if (!useSubdiv) return PolyObject::ConvertToType (t, obtype);

	UpdateSubdivResult (t);
	PolyObject *pobj = new PolyObject;
	pobj->mm = subdivResult;
	pobj->SetChannelValidity (GEOM_CHAN_NUM, subdivValid);
	pobj->SetChannelValidity (TOPO_CHAN_NUM, subdivValid);
	pobj->SetChannelValidity (TEXMAP_CHAN_NUM, FOREVER);
	pobj->SetChannelValidity (VERT_COLOR_CHAN_NUM, FOREVER);
	pobj->SetDisplacementDisable (GetDisplacementDisable ());
	pobj->SetDisplacementSplit (GetDisplacementSplit());
	pobj->SetDisplacementParameters (GetDisplacementParameters());
	pobj->SetDisplacement (GetDisplacement ());
	pobj->mm.dispFlags = mm.dispFlags;
	Object *ret = pobj->ConvertToType (t, obtype);
	if (ret != pobj) delete pobj;
	return ret;
}

int EditPolyObject::RenderBegin(TimeValue t, ULONG flags) {
	SetFlag(EPOLY_IN_RENDER);
	if (!pblock) return 0;
	int update, useRIter, riter, iter, recalc, useRSharp;
	float sharp, rsharp;
	pblock->GetValue (ep_surf_update, t, update, FOREVER);
	recalc = (update==1);
	if (!recalc) {
		pblock->GetValue (ep_surf_use_riter, t, useRIter, FOREVER);
		if (useRIter) {
			pblock->GetValue (ep_surf_iter, t, iter, FOREVER);
			pblock->GetValue (ep_surf_riter, t, riter, FOREVER);
			if (iter != riter) recalc = 1;
		}
	}
	if (!recalc) {
		pblock->GetValue (ep_surf_use_rthresh, t, useRSharp, FOREVER);
		if (useRSharp) {
			pblock->GetValue (ep_surf_thresh, t, sharp, FOREVER);
			pblock->GetValue (ep_surf_rthresh, t, rsharp, FOREVER);
			if (sharp != rsharp) recalc = 1;
		}
	}
	if (recalc) {
		subdivValid.SetEmpty ();
		NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
	}
	return 0;
}

int EditPolyObject::RenderEnd(TimeValue t) {
	ClearFlag(EPOLY_IN_RENDER);
	if (!pblock) return 0;
	int update, useRIter, riter, iter, useRSharp, recalc=FALSE;
	float sharp, rsharp;
	pblock->GetValue (ep_surf_update, t, update, FOREVER);
	recalc = (update==1);
	if (!recalc) {
		pblock->GetValue (ep_surf_use_riter, t, useRIter, FOREVER);
		if (useRIter) {
			pblock->GetValue (ep_surf_iter, t, iter, FOREVER);
			pblock->GetValue (ep_surf_riter, t, riter, FOREVER);
			if (iter != riter) recalc = 1;
		}
	}
	if (!recalc) {
		pblock->GetValue (ep_surf_use_rthresh, t, useRSharp, FOREVER);
		if (useRSharp) {
			pblock->GetValue (ep_surf_thresh, t, sharp, FOREVER);
			pblock->GetValue (ep_surf_rthresh, t, rsharp, FOREVER);
			if (sharp != rsharp) recalc = 1;
		}
	}
	if (recalc) {
		subdivValid.SetEmpty ();
		NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
	}
	return 0;
}

Mesh* EditPolyObject::GetRenderMesh(TimeValue t, INode *inode, View &view,  BOOL& needDelete) {
	int useSubdiv=false;
	if (pblock) pblock->GetValue (ep_surf_subdivide, t, useSubdiv, FOREVER);
	if (!useSubdiv) return PolyObject::GetRenderMesh (t, inode, view, needDelete);

	UpdateSubdivResult (t);
	needDelete = TRUE;
	Mesh *ret;
	BOOL needDisp = FALSE;
	if (!GetDisplacementDisable() && (view.flags & RENDER_MESH_DISPLACEMENT_MAP)) {
		// Find out if we need displacement mapping:
		// Get the material
		Mtl* pMtl = inode ? inode->GetMtl() : NULL;

		if (pMtl) {
			for (int i = 0; i< subdivResult.numf; i++) {
				if (pMtl->Requirements(subdivResult.f[i].material)&MTLREQ_DISPLACEMAP) {
					needDisp = TRUE;
					break;
				}
			}
		}
	}

	if (!needDisp) {
		ret = new class Mesh;
		subdivResult.OutToTri (*ret);
		return ret;
	}

	// Otherwise we have displacement mapping.
	// Create a TriObject from this, and get its render mesh.
	TriObject *tobj = (TriObject *) ConvertToType (t, triObjectClassID);
	if (!tobj) {
		// shouldn't happen
		DbgAssert(0);
		ret = new class Mesh;
		subdivResult.OutToTri (*ret);
		return ret;
	}
	BOOL subNeedDelete;
	ret = tobj->GetRenderMesh (t, inode, view, subNeedDelete);
	if (!subNeedDelete) {
		// Shouldn't happen, because of displacement parameters.
		Mesh *ret2 = ret;
		ret = new Mesh (*ret2);
	}
	delete tobj;
	return ret;
}

int EditPolyObject::GetSubobjectLevel() {
	return selLevel;
}

void EditPolyObject::SetSubobjectLevel(int level) {
	selLevel = level;
	mm.selLevel = meshSelLevel[level];
	mm.dispFlags = dispFlags[level];
	subdivResult.selLevel = meshSelLevel[level];
	subdivResult.dispFlags = dispFlags[level];

	if (ip && ip->GetShowEndResult()) mm.ClearDispFlag (MNDISP_SELVERTS|MNDISP_VERTTICKS);

	InvalidateTempData (PART_SUBSEL_TYPE);
	if (editObj) RefreshSelType ();
	InvalidateNumberSelected ();
	NotifyDependents (FOREVER, PART_SUBSEL_TYPE|PART_DISPLAY, REFMSG_CHANGE);
	if (ip) ip->RedrawViews (ip->GetTime());
}

bool CheckNodeSelection (Interface *ip, INode *inode) {
	if (!ip) return FALSE;
	if (!inode) return FALSE;
	int i, nct = ip->GetSelNodeCount();
	for (i=0; i<nct; i++) if (ip->GetSelNode (i) == inode) return TRUE;
	return FALSE;
}

bool EditPolyObject::ShowGizmo () {
	if (!ip) return FALSE;
	if (editObj != this) return FALSE;
	if (selLevel == EP_SL_OBJECT) return false;
	if (ip->GetShowEndResult()) return TRUE;
	int subdiv;
	pblock->GetValue (ep_surf_subdivide, TimeValue(0), subdiv, FOREVER);
	if (subdiv) return TRUE;
	return FALSE;
}

int EditPolyObject::Display(TimeValue t, INode* inode, ViewExp *vpt, int flags, ModContext *mc) {
	if (!ShowGizmo()) return 0;

	// Set up GW
	GraphicsWindow *gw = vpt->getGW();
	Matrix3 tm = inode->GetObjectTM(t);

	DWORD savedLimits = gw->getRndLimits();
	gw->setRndLimits((savedLimits & ~GW_ILLUM) | GW_ALL_EDGES);
	gw->setTransform(tm);

	if (sliceMode) {
		// Draw rectangle representing slice plane.
		gw->setColor(LINE_COLOR,GetUIColor(COLOR_SEL_GIZMOS));

		Point3 rp[5];
		Matrix3 rotMatrix;
		sliceRot.MakeMatrix (rotMatrix);
		rotMatrix.SetTrans (sliceCenter);
		rp[0] = Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
		rp[1] = Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[2] = Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[3] = Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
		gw->polyline (4, rp, NULL, NULL, TRUE, NULL);
	}

	// We need to draw a "gizmo" version of the polymesh:
	Point3 colSel=GetSubSelColor();
	Point3 colTicks=GetUIColor (COLOR_VERT_TICKS);
	Point3 colGiz=GetUIColor(COLOR_GIZMOS);
	Point3 colGizSel=GetUIColor(COLOR_SEL_GIZMOS);
	gw->setColor (LINE_COLOR, colGiz);
	Point3 rp[3];
	int i;
	int es[3];
	bool edgeLev = (meshSelLevel[selLevel] == MNM_SL_EDGE);
	bool faceLev = (meshSelLevel[selLevel] == MNM_SL_FACE);
	bool diagonals = (ip->GetCommandMode() == editTriMode);
	if (diagonals) {
		for (i=0; i<mm.numf; i++) {
			if (mm.f[i].GetFlag (MN_DEAD)) continue;
			if (mm.f[i].GetFlag (MN_HIDDEN)) continue;
			if (mm.f[i].deg < 4) continue;
			es[0] = GW_EDGE_INVIS;
			if (mm.f[i].GetFlag (MN_SEL)) gw->setColor (LINE_COLOR, colGizSel);
			else gw->setColor (LINE_COLOR, colGiz);

			for (int j=0; j<mm.f[i].deg-3; j++) {
				int jj = j*2;
				rp[0] = mm.v[mm.f[i].vtx[mm.f[i].diag[jj]]].p;
				rp[1] = mm.v[mm.f[i].vtx[mm.f[i].diag[jj+1]]].p;
			}
			gw->polyline (2, rp, NULL, NULL, FALSE, es);
		}
	}
	for (i=0; i<mm.nume; i++) {
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (mm.f[mm.e[i].f1].GetFlag (MN_HIDDEN) && 
			((mm.e[i].f2 < 0) || mm.f[mm.e[i].f2].GetFlag (MN_HIDDEN))) continue;
		if (mm.e[i].GetFlag (MN_EDGE_INVIS)) {
			if (!edgeLev) continue;
			es[0] = GW_EDGE_INVIS;
		} else {
			es[0] = GW_EDGE_VIS;
		}
		if (!sliceMode) {
			if (edgeLev) {
				if (mm.e[i].GetFlag (MN_SEL)) gw->setColor (LINE_COLOR, colGizSel);
				else gw->setColor (LINE_COLOR, colGiz);
			}
			if (faceLev) {
				if (mm.f[mm.e[i].f1].GetFlag (MN_SEL) ||
					((mm.e[i].f2 >= 0) && mm.f[mm.e[i].f2].GetFlag (MN_SEL)))
					gw->setColor (LINE_COLOR, colGizSel);
				else gw->setColor (LINE_COLOR, colGiz);
			}
		}
		rp[0] = mm.v[mm.e[i].v1].p;
		rp[1] = mm.v[mm.e[i].v2].p;
		gw->polyline (2, rp, NULL, NULL, FALSE, es);
	}

	if ((ip->GetCommandMode()==createFaceMode) ||
		(ip->GetCommandMode()==createEdgeMode) ||
		(ip->GetCommandMode()==editTriMode) ||
		(selLevel == EP_SL_VERTEX)) {
		int softSel, useEdgeDist, edgeIts, affectBack;
		float falloff, pinch, bubble;
		Interval frvr = FOREVER;
		pblock->GetValue (ep_ss_use, t, softSel, frvr);
		if (softSel) {
			pblock->GetValue (ep_ss_use, t, useEdgeDist, frvr);
			if (useEdgeDist) pblock->GetValue (ep_ss_use, t, edgeIts, frvr);
			pblock->GetValue (ep_ss_use, t, affectBack, frvr);
			pblock->GetValue (ep_ss_falloff, t, falloff, frvr);
			pblock->GetValue (ep_ss_pinch, t, pinch, frvr);
			pblock->GetValue (ep_ss_bubble, t, bubble, frvr);
		}
		float *ourvw = NULL;
		if (softSel) {
			ourvw = TempData()->VSWeight (useEdgeDist, edgeIts, !affectBack, falloff, pinch, bubble)->Addr(0);
		}
		for (i=0; i<mm.numv; i++) {
			if (mm.v[i].GetFlag (MN_HIDDEN)) continue;
			if (mm.v[i].GetFlag (MN_SEL)) gw->setColor (LINE_COLOR, colSel);
			else {
				if (ourvw) gw->setColor (LINE_COLOR, SoftSelectionColor(ourvw[i]));
				else gw->setColor (LINE_COLOR, colTicks);
			}

			if(getUseVertexDots()) gw->marker (&(mm.v[i].p), getVertexDotType() ? DOT_MRKR : SM_DOT_MRKR);
			else gw->marker (&(mm.v[i].p), PLUS_SIGN_MRKR);
		}
	}

	if ((editObj==this) && (ip->GetCommandMode () == createFaceMode)) {
		gw->setColor(LINE_COLOR,GetUIColor(COLOR_SUBSELECTION));
		createFaceMode->proc.DrawEstablishedFace (gw);
	}

	gw->setRndLimits(savedLimits);
	return 0;	
}

void EditPolyObject::GetWorldBoundBox(TimeValue t,INode* inode, ViewExp *vpt, Box3& box, ModContext *mc) {
	box.Init();
	if (!ShowGizmo()) return;
	Matrix3 tm = inode->GetObjectTM(t);
	// We need to draw a "gizmo" version of the mesh:
	box = mm.getBoundingBox (&tm);
	if (!sliceMode) return;
	if (!CheckNodeSelection (ip, inode)) return;
	Matrix3 rotMatrix;
	sliceRot.MakeMatrix (rotMatrix);
	rotMatrix.SetTrans (sliceCenter);
	rotMatrix *= tm;
	box += Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
	box += Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
}

int EditPolyObject::Display (TimeValue t, INode* inode, ViewExp *vpt, int flags) {
	BOOL showSubdiv=false;
	if (pblock) pblock->GetValue (ep_surf_subdivide, t, showSubdiv, FOREVER);
	if (showSubdiv) {
		// Show subdivision result
		UpdateSubdivResult (t);
		GraphicsWindow *gw = vpt->getGW();
		gw->setTransform(inode->GetObjectTM(t));
		DWORD oldDispFlags = subdivResult.dispFlags;
		if (editObj == this) subdivResult.dispFlags &= ~(MNDISP_SELVERTS|MNDISP_VERTTICKS);
		switch (inode->GetVertexColorType()) {
		case nvct_color:
			subdivResult.SetDisplayVertexColors (0);
			break;
		case nvct_illumination:
			subdivResult.SetDisplayVertexColors (MAP_SHADING);
			break;
		case nvct_alpha:
			subdivResult.SetDisplayVertexColors (MAP_ALPHA);
			break;
		}
		subdivResult.render (gw, inode->Mtls(),
			(flags&USE_DAMAGE_RECT) ? &vpt->GetDammageRect() : NULL, 
			COMP_ALL | ((flags&DISP_SHOWSUBOBJECT)?COMP_OBJSELECTED:0),
			inode->NumMtls());
		subdivResult.dispFlags = oldDispFlags;
		// Don't forget the gizmo - if this node is selected, that is:
		if (inode->Selected ()) Display (t, inode, vpt, flags, NULL);
	} else {
		// Commented out 4/26/01 - was causing vertex ticks to show up on unselected nodes.
		// GraphicsWindow *gw = vpt->getGW();
		// DWORD oldLimits = gw->getRndLimits ();
		//if (mm.GetDispFlag (MNDISP_VERTTICKS)) {
			//gw->setRndLimits (oldLimits | GW_VERT_TICKS);
		//}
		PolyObject::Display (t, inode, vpt, flags);
		//gw->setRndLimits (oldLimits);
	}
	if (!CheckNodeSelection (ip, inode)) return 0;
	GraphicsWindow *gw = vpt->getGW();
	Matrix3 tm = inode->GetObjectTM(t);
	int savedLimits;

	gw->setRndLimits((savedLimits = gw->getRndLimits()) & ~GW_ILLUM);
	gw->setTransform(tm);

	if (sliceMode) {
		// Draw rectangle representing slice plane.
		gw->setColor(LINE_COLOR,GetUIColor(COLOR_SEL_GIZMOS));

		Point3 rp[5];
		Matrix3 rotMatrix;
		sliceRot.MakeMatrix (rotMatrix);
		rotMatrix.SetTrans (sliceCenter);
		rp[0] = Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
		rp[1] = Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[2] = Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[3] = Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
		gw->polyline (4, rp, NULL, NULL, TRUE, NULL);
	}

	if ((editObj==this) && (ip->GetCommandMode () == createFaceMode)) {
		gw->setColor(LINE_COLOR,GetUIColor(COLOR_SUBSELECTION));
		createFaceMode->proc.DrawEstablishedFace (gw);
	}

	gw->setRndLimits(savedLimits);
	return 0;
}

void EditPolyObject::GetWorldBoundBox (TimeValue t, INode* inode, ViewExp* vp, Box3& box ) {
	PolyObject::GetWorldBoundBox (t, inode, vp, box);

	if (!sliceMode) return;
	if (!CheckNodeSelection (ip, inode)) return;
	Matrix3 tm = inode->GetObjectTM(t);
	Matrix3 rotMatrix;
	sliceRot.MakeMatrix (rotMatrix);
	rotMatrix.SetTrans (sliceCenter);
	rotMatrix *= tm;
	box += Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
	box += Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
}

void EditPolyObject::GetLocalBoundBox (TimeValue t, INode* inode, ViewExp* vp, Box3& box ) {
	PolyObject::GetLocalBoundBox (t, inode, vp, box);

	if (!sliceMode) return;
	if (!CheckNodeSelection (ip, inode)) return;
	Matrix3 rotMatrix;
	sliceRot.MakeMatrix (rotMatrix);
	rotMatrix.SetTrans (sliceCenter);
	box += Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
	box += Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
	box += Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
}

// What subobject level should we be hit-testing on?
DWORD EditPolyObject::CurrentHitLevel (int *selByVert) {
	// Under normal circumstances, pretty much the level we're at.
	DWORD hitLev = hitLevel[selLevel];

	// But there are exceptions...
	int sbv;
	if (selByVert == NULL) selByVert = &sbv;
	pblock->GetValue (ep_by_vertex, TimeValue(0), *selByVert, FOREVER);

	// Many edge and face command modes override the select-by-vertex:
	if (*selByVert) {
		if (cutEdgeMode == ip->GetCommandMode()) *selByVert = FALSE;
		if (divideEdgeMode == ip->GetCommandMode()) *selByVert = FALSE;
		if (divideFaceMode == ip->GetCommandMode()) *selByVert = FALSE;
	}
	if (*selByVert) {
		hitLev = SUBHIT_MNVERTS;
		if (selLevel != EP_SL_VERTEX) hitLev |= SUBHIT_MNUSECURRENTSEL;
		if (selLevel == EP_SL_BORDER) hitLev |= SUBHIT_OPENONLY;
	}

	if (ip->GetCommandMode()==createFaceMode) hitLev = SUBHIT_MNVERTS|SUBHIT_OPENONLY;
	if (ip->GetCommandMode()==createEdgeMode) hitLev = SUBHIT_MNVERTS;
	if (ip->GetCommandMode()==editTriMode) hitLev = SUBHIT_MNVERTS;
	if (ip->GetCommandMode()==cutEdgeMode) hitLev = SUBHIT_MNEDGES;
	if ((ip->GetCommandMode()==weldMode) && (selLevel != EP_SL_VERTEX)) hitLev |= SUBHIT_OPENONLY;
	return hitLev;
}

int EditPolyObject::HitTest (TimeValue t, INode* inode, int type, int crossing, 
		int flags, IPoint2 *p, ViewExp *vpt, ModContext* mc) {
	Interval valid;
	int savedLimits, res = 0;
	GraphicsWindow *gw = vpt->getGW();
	HitRegion hr;
	
	// Setup GW
	MakeHitRegion(hr,type, crossing, pickBoxSize, p);
	gw->setHitRegion(&hr);
	Matrix3 mat = inode->GetObjectTM(t);
	gw->setTransform(mat);
	gw->setRndLimits(((savedLimits = gw->getRndLimits()) | GW_PICK) & ~GW_ILLUM);

	int ignoreBack;
	pblock->GetValue (ep_ignore_backfacing, t, ignoreBack, FOREVER);
	if (ignoreBack) gw->setRndLimits(gw->getRndLimits() |  GW_BACKCULL);
	else gw->setRndLimits(gw->getRndLimits() & ~GW_BACKCULL);
	gw->clearHitCode();	

	if (sliceMode && CheckNodeSelection (ip, inode)) {
		Point3 rp[5];
		Matrix3 rotMatrix;
		sliceRot.MakeMatrix (rotMatrix);
		rotMatrix.SetTrans (sliceCenter);
		rp[0] = Point3(-sliceSize,-sliceSize,0.0f)*rotMatrix;
		rp[1] = Point3(-sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[2] = Point3(sliceSize,sliceSize,0.0f)*rotMatrix;
		rp[3] = Point3(sliceSize,-sliceSize,0.0f)*rotMatrix;
		gw->polyline (4, rp, NULL, NULL, TRUE, NULL);
		if (gw->checkHitCode()) {
			vpt->LogHit (inode, mc, gw->getHitDistance(), 0, NULL);
			res = 1;
		}
		gw->setRndLimits (savedLimits);
		return res;
	}

	SubObjHitList hitList;
	MeshSubHitRec *rec;
	int selByVert;
	DWORD hitLev = CurrentHitLevel (&selByVert);
	DWORD thisFlags = hitLev | flags;
	res = mm.SubObjectHitTest(gw, gw->getMaterial(), &hr, thisFlags, hitList);

	rec = hitList.First();
	while (rec) {
		vpt->LogHit(inode,mc,rec->dist,rec->index,NULL);
		rec = rec->Next();
	}

	gw->setRndLimits(savedLimits);	
	return res;
}

void EditPolyObject::SelectSubComponent (HitRecord *hitRec, BOOL selected, BOOL all, BOOL invert) {
	if (selLevel == EP_SL_OBJECT) return;
	if (!ip) return;
	if (sliceMode) return;
	ip->ClearCurNamedSelSet ();
	TimeValue t = ip->GetTime();
	int selByVert;
	pblock->GetValue (ep_by_vertex, t, selByVert, FOREVER);
	if (cutEdgeMode == ip->GetCommandMode()) selByVert = FALSE;
	if (divideEdgeMode == ip->GetCommandMode()) selByVert = FALSE;
	if (divideFaceMode == ip->GetCommandMode()) selByVert = FALSE;

	BitArray nsel, currSel;
	HitRecord *hr;

	switch (selLevel) {
	case EP_SL_VERTEX:
		mm.getVertexSel (nsel);
		for (hr=hitRec; hr!=NULL; hr = hr->Next()) {
			nsel.Set (hr->hitInfo, invert ? !mm.v[hr->hitInfo].GetFlag (MN_SEL) : selected);
			if (!all) break;
		}
		SetVertSel (nsel, this, t);
		// JBW: macro-recorder
		macroRecorder->FunctionCall(_T("select"), 1, 0, mr_index, mr_prop, _T("verts"), mr_reftarg, this, mr_bitarray, &nsel);
		break;

	case EP_SL_EDGE:
		mm.getEdgeSel (nsel);
		for (hr=hitRec; hr != NULL; hr=hr->Next()) {
			if (selByVert) {
				// Consider: what if vedg is out of kilter?
				Tab<int> & ve = mm.vedg[hr->hitInfo];
				for (int i=0; i<ve.Count(); i++) nsel.Set (ve[i], invert ? !mm.e[ve[i]].GetFlag (MN_SEL) : selected);
			} else {
				nsel.Set (hr->hitInfo, invert ? !nsel[hr->hitInfo] : selected);
			}
			if (!all) break;
		}
		SetEdgeSel (nsel, this, t);
		// JBW: macro-recorder
		macroRecorder->FunctionCall(_T("select"), 1, 0, mr_index, mr_prop, _T("edges"), mr_reftarg, this, mr_bitarray, &nsel);
		break;

	case EP_SL_BORDER:
		nsel.SetSize (mm.nume);
		for (hr=hitRec; hr != NULL; hr=hr->Next()) {
			if (selByVert) {
				// Consider: what if vedg is out of kilter?
				Tab<int> & ve = mm.vedg[hr->hitInfo];
				for (int i=0; i<ve.Count(); i++) mm.BorderFromEdge (ve[i], nsel);
			} else mm.BorderFromEdge (hr->hitInfo, nsel);
			if (!all) break;
		}
		mm.getEdgeSel (currSel);
		if (invert) nsel ^= currSel;
		else {
			if (selected) nsel |= currSel;
			else nsel = currSel & ~nsel;
		}
		SetEdgeSel (nsel, this, t);

		// JBW: macro-recorder
		macroRecorder->FunctionCall(_T("select"), 1, 0, mr_index, mr_prop, _T("edges"), mr_reftarg, this, mr_bitarray, &nsel);
		break;

	case EP_SL_FACE:
		mm.getFaceSel (nsel);
		for (hr=hitRec; hr != NULL; hr = hr->Next()) {
			if (selByVert) {
				Tab<int> & vf = mm.vfac[hr->hitInfo];
				for (int i=0; i<vf.Count(); i++) {
					nsel.Set (vf[i], invert ? !mm.f[vf[i]].GetFlag (MN_SEL) : selected);
				}
			} else {
				nsel.Set (hr->hitInfo, invert ? !mm.f[hr->hitInfo].GetFlag (MN_SEL) : selected);
			}
			if (!all) break;
		}
		SetFaceSel (nsel, this, t);

		// JBW: macro-recorder
		macroRecorder->FunctionCall(_T("select"), 1, 0, mr_index, mr_prop, _T("faces"), mr_reftarg, this, mr_bitarray, &fsel);
		break;

	case EP_SL_ELEMENT:
		nsel.SetSize (mm.numf);
		for (hr=hitRec; hr != NULL; hr=hr->Next()) {
			if (selByVert) {
				Tab<int> & vf = mm.vfac[hr->hitInfo];
				for (int i=0; i<vf.Count(); i++) mm.ElementFromFace (vf[i], nsel);
			} else mm.ElementFromFace (hr->hitInfo, nsel);
			if (!all) break;
		}
		mm.getFaceSel (currSel);
		if (invert) nsel ^= currSel;
		else {
			if (selected) nsel |= currSel;
			else nsel = currSel & ~nsel;
		}
		SetFaceSel (nsel, this, t);

		// JBW: macro-recorder
		macroRecorder->FunctionCall(_T("select"), 1, 0, mr_index, mr_prop, _T("faces"), mr_reftarg, this, mr_bitarray, &nsel);
		break;
	};
	LocalDataChanged(PART_SELECT);

	DisplayFaceFlags();	
	RefreshScreen ();
}

void EditPolyObject::ClearSelection(int sl) {
	if (sliceMode) return;
	BitArray sel;
	switch (sl) {
	case EP_SL_OBJECT: return;
	case EP_SL_VERTEX:
		sel.SetSize (mm.numv);
		sel.ClearAll ();
		SetVertSel (sel, this, ip->GetTime());
		break;
	case EP_SL_EDGE:
	case EP_SL_BORDER:
		sel.SetSize (mm.nume);
		sel.ClearAll();
		SetEdgeSel (sel, this, ip->GetTime());
		break;
	default:
		sel.SetSize (mm.numf);
		sel.ClearAll ();
		SetFaceSel (sel, this, ip->GetTime());
		// Neversoft addition for face flags
		if( hFaceFlags )
		{
			SetFaceFlagMask( hFaceFlags, 0, 0 );
		}
		break;
	}
	LocalDataChanged (PART_SELECT);
	RefreshScreen ();
}

void EditPolyObject::SelectAll(int sl) {
	if (sl == EP_SL_OBJECT) return;
	if (sliceMode) return;
	BitArray sel;
	int i;
	switch (sl) {
	case EP_SL_VERTEX: 
		sel.SetSize (mm.numv);
		sel.SetAll();
		SetVertSel (sel, this, ip->GetTime());
		break;
	case EP_SL_EDGE:
		sel.SetSize (mm.nume);
		sel.SetAll();
		SetEdgeSel (sel, this, ip->GetTime());
		break;
	case EP_SL_BORDER:
		sel.SetSize (mm.nume);
		for (i=0; i<mm.nume; i++) sel.Set (i, mm.e[i].f2<0);
		SetEdgeSel (sel, this, ip->GetTime());
		break;
	default:
		sel.SetSize (mm.numf);
		sel.SetAll(); 
		SetFaceSel (sel, this, ip->GetTime());
		break;
	}
	LocalDataChanged (PART_SELECT);
	RefreshScreen ();
}

void EditPolyObject::InvertSelection(int sl) {
	if (sl == EP_SL_OBJECT) return;
	if (sliceMode) return;
	BitArray sel;
	switch (sl) {
	case EP_SL_VERTEX:
		mm.getVertexSel (sel);
		sel = ~sel;
		SetVertSel (sel, this, ip->GetTime());
		break;
	case EP_SL_EDGE:
	case EP_SL_BORDER:
		mm.getEdgeSel (sel);
		sel = ~sel;
		if (sl == EP_SL_BORDER) {
			for (int i=0; i<mm.nume; i++) if (mm.e[i].f2>-1) sel.Clear(i);
		}
		SetEdgeSel (sel, this, ip->GetTime());
		break;
	default:
		mm.getFaceSel (sel);
		sel = ~sel;
		SetFaceSel (sel, this, ip->GetTime());
		break;
	}
	LocalDataChanged (PART_SELECT);
	RefreshScreen ();
}

BOOL EditPolyObject::SelectSubAnim(int subNum) {
	if (subNum<2) return FALSE;	// cannot select master point controller or pblock.
	subNum -= 2;
	if (subNum >= mm.numv) return FALSE;

	BOOL add = GetKeyState(VK_CONTROL)<0;
	BOOL sub = GetKeyState(VK_MENU)<0;
	BitArray nvs;

	if (add || sub) {
		mm.getVertexSel (nvs);
		if (sub) nvs.Clear (subNum);
		else nvs.Set (subNum);
	} else {
		nvs.SetSize (mm.numv);
		nvs.ClearAll ();
		nvs.Set (subNum);
	}

	if (ip) SetVertSel (nvs, this, ip->GetTime());
	else SetVertSel (nvs, this, TimeValue(0));
	LocalDataChanged (PART_SELECT);
	RefreshScreen ();
	return TRUE;
}

void EditPolyObject::ActivateSubobjSel (int level, XFormModes& modes) {
	// Register or unregister delete key notification
	if (ip) {
		if (selLevel==EP_SL_OBJECT && level!=EP_SL_OBJECT) {
			ip->RegisterDeleteUser(this);
			backspacer.SetEPoly (this);
			backspaceRouter.Register (&backspacer);
		}
		if (selLevel!=EP_SL_OBJECT && level==EP_SL_OBJECT) {
			ip->UnRegisterDeleteUser(this);
			backspacer.SetEPoly (NULL);
			backspaceRouter.UnRegister (&backspacer);
		}
	}

	if (meshSelLevel[selLevel] != meshSelLevel[level]) {
		ExitAllCommandModes (level == EP_SL_OBJECT);
	}
	if ((selLevel == EP_SL_EDGE) && (level==EP_SL_BORDER)) {
		// Don't want to leave Create mode on when the button becomes Cap Borders.
		if (createEdgeMode) ip->DeleteMode (createEdgeMode);
	}

	// Set the mesh's level
	SetSubobjectLevel(level);

	// Neversoft Extension : Update face flags UI
	DisplayFaceFlags();

	// Fill in modes with our sub-object modes
	if (level!=EP_SL_OBJECT)
		modes = XFormModes(moveMode,rotMode,nuscaleMode,uscaleMode,squashMode,selectMode);

	// Setup named selection sets
	if (ip && (level != EP_SL_OBJECT)) {
		GenericNamedSelSetList &set = GetSelSet();
		ip->ClearSubObjectNamedSelSets();
		for (int i=0; i<set.Count(); i++) {
			ip->AppendSubObjectNamedSelSet(*(set.names[i]));
		}
	}
	if (ip) ip->PipeSelLevelChanged();
	UpdateNamedSelDropDown ();
	InvalidateTempData (PART_TOPO|PART_GEOM|PART_SELECT|PART_SUBSEL_TYPE);
	NotifyDependents(FOREVER, SELECT_CHANNEL|DISP_ATTRIB_CHANNEL|SUBSEL_TYPE_CHANNEL, REFMSG_CHANGE);
}

GenericNamedSelSetList &EditPolyObject::GetSelSet() {
	return selSet[namedSetLevel[selLevel]];
}

void EditPolyObject::GetSubObjectCenters(SubObjAxisCallback *cb,
										TimeValue t, INode *node,ModContext *mc) {
	Matrix3 tm = node->GetObjectTM(t);

	if (sliceMode) {
		cb->Center (sliceCenter*tm, 0);
		return;
	}

	if (selLevel == EP_SL_OBJECT) return;
	if (selLevel == EP_SL_VERTEX) {
		BitArray sel = mm.VertexTempSel();
		Point3 cent(0,0,0);
		int ct=0;
		for (int i=0; i<mm.numv; i++) {
			if (sel[i]) {
				cent += mm.v[i].p;
				ct++;
			}
		}
		if (ct) {
			cent /= float(ct);			
			cb->Center(cent*tm,0);
		}
		return;
	}

	Tab<Point3> *centers = TempData()->ClusterCenters(meshSelLevel[selLevel]);
	DbgAssert (centers);
	if (!centers) return;
	for (int i=0; i<centers->Count(); i++) cb->Center((*centers)[i]*tm,i);
}

void EditPolyObject::GetSubObjectTMs (SubObjAxisCallback *cb,TimeValue t,
		INode *node,ModContext *mc) {
	Matrix3 tm, otm = node->GetObjectTM(t);

	if (sliceMode) {
		Matrix3 rotMatrix(1);
		sliceRot.MakeMatrix (rotMatrix);
		rotMatrix.SetTrans (sliceCenter);
		rotMatrix *= otm;
		cb->TM (rotMatrix, 0);
		return;
	}

	switch (selLevel) {
	case EP_SL_OBJECT:
		break;

	case EP_SL_VERTEX:
		if (ip->GetCommandMode()->ID()==CID_SUBOBJMOVE) {
			if (!mm.GetFlag (MN_MESH_NO_BAD_VERTS)) mm.EliminateBadVerts ();
			Matrix3 otm = node->GetObjectTM(t);
			for (int i=0; i<mm.numv; i++) {
				if (!mm.v[i].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
				mm.GetVertexSpace (i, tm);
				tm = otm * tm;
				tm.SetTrans(mm.v[i].p*otm);
				cb->TM(tm,i);
			}
		} else {
			for (int i=0; i<mm.numv; i++) if (mm.v[i].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) break;
			if (i >= mm.numv) return;
			Point3 norm(0,0,0), *nptr=TempData()->VertexNormals ()->Addr(0);
			Point3 cent(0,0,0);
			int ct=0;

			// Compute average face normal
			for (; i<mm.numv; i++) {
				if (!mm.v[i].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
				cent += mm.v[i].p;
				norm += nptr[i];
				ct++;
			}
			cent /= float(ct);
			norm = Normalize (norm/float(ct));

			cent = cent * otm;
			norm = Normalize(VectorTransform(otm,norm));
			Matrix3 mat;
			MatrixFromNormal(norm,mat);
			mat.SetTrans(cent);
			cb->TM(mat,0);
		}
		break;

	case EP_SL_EDGE:
	case EP_SL_BORDER:
		int i, ct;
		ct = TempData()->ClusterNormals(MNM_SL_EDGE)->Count();
		for (i=0; i<ct; i++) {
			tm = TempData()->ClusterTM (i) * otm;
			cb->TM(tm,i);
		}
		break;

	default:
		ct = TempData()->ClusterNormals(MNM_SL_FACE)->Count();
		for (i=0; i<ct; i++) {
			tm = TempData()->ClusterTM (i) * otm;
			cb->TM(tm,i);
		}
		break;
	}
}

bool EditPolyObject::getShowVerts () {
	if (selLevel==EP_SL_VERTEX) return TRUE;
	if (ip->GetCommandMode() == createFaceMode) return TRUE;
	return FALSE;
}

void EditPolyObject::ShowEndResultChanged (BOOL showEndResult) {
	if ((!ip) || (editObj != this)) return;
	if (getShowVerts()){
		if (showEndResult) mm.dispFlags = 0;
		else mm.dispFlags = MNDISP_VERTTICKS | MNDISP_SELVERTS;
	}
	NotifyDependents (FOREVER, PART_DISPLAY, REFMSG_CHANGE);
}

//--- Saving/Loading --------------------------------

#define VERTCOUNT_CHUNKID		0x3002
#define GENSELSET_ID_CHUNK     0x3003
#define GENSELSET_CHUNK     0x3004
#define EPOBJ_SELLEVEL_CHUNK 0x4038

IOResult EditPolyObject::Load(ILoad *iload) {
	ulong nb;
	IOResult res;
	bool selLevLoaded = FALSE;

	while (iload->PeekNextChunkID() == GENSELSET_ID_CHUNK) {
		int which;
		iload->OpenChunk();
		res = iload->Read(&which, sizeof(int), &nb);
		iload->CloseChunk ();
		if (res!=IO_OK) return res;

		if (iload->PeekNextChunkID() != GENSELSET_CHUNK) break;
		iload->OpenChunk ();
		res = selSet[which].Load(iload);
		iload->CloseChunk();
		if (res!=IO_OK) return res;
	}

	if (iload->PeekNextChunkID()==VERTCOUNT_CHUNKID) {		
		int ct;
		iload->OpenChunk();
		iload->Read(&ct,sizeof(ct),&nb);
		iload->CloseChunk();
		AllocContArray(ct);
		for (int i=0; i<ct; i++) SetPtCont(i, NULL);
	}

	if (iload->PeekNextChunkID() == EPOBJ_SELLEVEL_CHUNK) {
		iload->OpenChunk ();
		res = iload->Read (&selLevel, sizeof(DWORD), &nb);
		if (res != IO_OK) return res;
		iload->CloseChunk ();
		selLevLoaded = TRUE;
	}

	// Neversoft extension. Output the face flags data
	if (iload->PeekNextChunkID() == vTRI_OBJ_FACE_FLAGS_CHUNK) {
		iload->OpenChunk ();		

		MNMesh& mesh = GetMesh();
	
		// Get the face-data channel from the incoming mesh
		IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
		if( pFDMgr )
		{
			int i, num_faces, value;
			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			if( fdc == NULL )
			{					
				// The mesh does not have our face-data channel so we will add it here
				fdc = new FaceFlagsData();
				pFDMgr->AddFaceDataChan( fdc );
			}
			
			res = iload->Read(&num_faces,sizeof(int), &nb);
			fdc->FacesCreated( 0, num_faces );
			for( i = 0; i < num_faces; i++ )
			{
				res = iload->Read( &value,sizeof(int), &nb );
				fdc->SetValue( i, value );			
			}		
		}
		iload->CloseChunk ();
	}	


	IOResult ret = PolyObject::Load (iload);
	if (!selLevLoaded) {
		switch (mm.selLevel) {
		case MNM_SL_OBJECT: selLevel = EP_SL_OBJECT; break;
		case MNM_SL_VERTEX: selLevel = EP_SL_VERTEX; break;
		case MNM_SL_EDGE: selLevel = EP_SL_EDGE; break;
		case MNM_SL_FACE: selLevel = EP_SL_FACE; break;
		}
	}
	return ret;
}

IOResult EditPolyObject::Save(ISave *isave) {	
	int ct = cont.Count();
	ulong nb;

	for (int j=0; j<3; j++) {
		if (!selSet[j].Count()) continue;
		isave->BeginChunk(GENSELSET_ID_CHUNK);
		isave->Write (&j, sizeof(j), &nb);
		isave->EndChunk ();
		isave->BeginChunk (GENSELSET_CHUNK);
		selSet[j].Save(isave);
		isave->EndChunk();
	}
	
	if (ct) {
		isave->BeginChunk(VERTCOUNT_CHUNKID);
		isave->Write(&ct,sizeof(ct),&nb);
		isave->EndChunk();
	}

	isave->BeginChunk (EPOBJ_SELLEVEL_CHUNK);
	isave->Write (&selLevel, sizeof(DWORD), &nb);
	isave->EndChunk ();

	// Neversoft extension. Output the face flags data
	MNMesh& mesh = GetMesh();
	
	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if( pFDMgr )
	{
		
		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
		if( fdc )
		{					
			int i, num_faces;
			FlagType value;

			num_faces = mesh.FNum();
			isave->BeginChunk( vTRI_OBJ_FACE_FLAGS_CHUNK );
			isave->Write ( &num_faces, sizeof(int), &nb );	

			for( i = 0; i < num_faces; i++ )
			{
				fdc->GetValue( i, value );
				isave->Write ( &value, sizeof(int), &nb );
			}
			isave->EndChunk ();
		}
	}

	return PolyObject::Save(isave);
}

static namedSelSetValid = FALSE;

void EditPolyObject::ActivateSubSelSet(TSTR &setName) {
	if (selLevel == EP_SL_OBJECT) return;

	BitArray *sset;
	int nsl = namedSetLevel[selLevel];
	sset = selSet[nsl].GetSet(setName);
	if (sset==NULL) return;
	theHold.Begin ();
	SetSel (nsl, *sset, this, ip->GetTime());
	theHold.Accept (GetString (IDS_SELECT));
	namedSelSetValid = TRUE;
	LocalDataChanged ();
	RefreshScreen ();
}

void EditPolyObject::NewSetFromCurSel(TSTR &setName) {
	if (selLevel == EP_SL_OBJECT) return;
	BitArray *sset;
	int nsl = namedSetLevel[selLevel];
	sset = selSet[nsl].GetSet(setName);
	if (sset) *sset = GetSel (nsl);
	else {
		BitArray sel = GetSel(nsl);
		selSet[nsl].AppendSet (sel, 0, setName);
	}
}

void EditPolyObject::RemoveSubSelSet(TSTR &setName) {
	GenericNamedSelSetList &set = GetSelSet();
	BitArray *ssel = set.GetSet(setName);
	if (ssel) {
		if (theHold.Holding()) theHold.Put (new DeleteSetRestore (setName, &set, this));
		set.RemoveSet (setName);
	}
	ip->ClearCurNamedSelSet();
	namedSelSetValid = TRUE;
}

void EditPolyObject::SetupNamedSelDropDown() {
	// Setup named selection sets
	if (selLevel == EP_SL_OBJECT) return;
	GenericNamedSelSetList &set = GetSelSet();
	ip->ClearSubObjectNamedSelSets();
	for (int i=0; i<set.Count(); i++) ip->AppendSubObjectNamedSelSet(*(set.names[i]));
	namedSelSetValid = TRUE;
}

void EditPolyObject::InvalidateNamedSelDropDown () {
	namedSelSetValid = FALSE;
}

void EditPolyObject::UpdateNamedSelDropDown () {
	if (!ip) return;
	if (namedSelSetValid) return;
	DWORD nsl = namedSetLevel[selLevel];
	GenericNamedSelSetList & ns = selSet[nsl];
	for (int i=0; i<ns.Count(); i++) {
		if (*(ns.sets[i]) == GetSel (nsl)) break;
	}
	if (i<ns.Count()) ip->SetCurNamedSelSet (*(ns.names[i]));
	namedSelSetValid = TRUE;
}

int EditPolyObject::NumNamedSelSets() {
	GenericNamedSelSetList &set = GetSelSet();
	return set.Count();
}

TSTR EditPolyObject::GetNamedSelSetName(int i) {
	GenericNamedSelSetList &set = GetSelSet();
	return *set.names[i];
}

void EditPolyObject::SetNamedSelSetName(int i,TSTR &newName) {
	GenericNamedSelSetList &set = GetSelSet();
	if (theHold.Holding()) theHold.Put(new SetNameRestore(i,&set,this));
	*set.names[i] = newName;
}

void EditPolyObject::NewSetByOperator (TSTR &newName,Tab<int> &sets,int op) {
	GenericNamedSelSetList &set = GetSelSet();
	BitArray bits = *set.sets[sets[0]];

	for (int i=1; i<sets.Count(); i++) {
		switch (op) {
		case NEWSET_MERGE:
			bits |= *set.sets[sets[i]];
			break;

		case NEWSET_INTERSECTION:
			bits &= *set.sets[sets[i]];
			break;

		case NEWSET_SUBTRACT:
			bits &= ~(*set.sets[sets[i]]);
			break;
		}
	}
	
	set.AppendSet(bits,0,newName);
	if (theHold.Holding()) theHold.Put(new AppendSetRestore(&set,this));
	if (!bits.NumberSet()) RemoveSubSelSet(newName);
}

void EditPolyObject::EpfnNamedSelectionCopy (TSTR setName) {
	GenericNamedSelSetList & selSet = GetSelSet();
	int i;
	for (i=0; i<selSet.names.Count(); i++) {
		if (*selSet.names[i] == setName) break;
	}
	if (i>=selSet.names.Count()) return;
	NamedSelectionCopy (i);
}

void EditPolyObject::NamedSelectionCopy(int index) {
	if (selLevel == EP_SL_OBJECT) return;
	if (index<0) return;
	int nsl = namedSetLevel[selLevel];
	GenericNamedSelSetList & setList = selSet[nsl];
	MeshNamedSelClip *clip = new MeshNamedSelClip(*(setList.names[index]));
	BitArray *bits = new BitArray(*setList.sets[index]);
	clip->sets.Append(1,&bits);
	SetMeshNamedSelClip (clip, namedClipLevel[selLevel]);

	// Enable the paste button
	HWND hSel = GetDlgHandle (ep_select);
	if (hSel) {
		ICustButton *but;
		but = GetICustButton(GetDlgItem(hSel, IDC_PASTE_NS));
		but->Enable();
		ReleaseICustButton(but);
	}
}

void EditPolyObject::EpfnNamedSelectionPaste(BOOL useDlgToRename) {
	if (selLevel==EP_SL_OBJECT) return;
	int nsl = namedSetLevel[selLevel];
	MeshNamedSelClip *clip = GetMeshNamedSelClip(namedClipLevel[selLevel]);
	if (!clip) return;
	TSTR name = clip->name;
	if (!GetUniqueSetName(name,true)) return;

	theHold.Begin ();
	GenericNamedSelSetList & setList = selSet[nsl];
	setList.AppendSet (*clip->sets[0], 0, name);
	if (theHold.Holding()) theHold.Put(new AppendSetRestore(&setList,this));
	ActivateSubSelSet(name);
	theHold.Accept (GetString (IDS_PASTE_NS));
	ip->SetCurNamedSelSet(name);
	SetupNamedSelDropDown();
}

static INT_PTR CALLBACK PickSetNameDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam) {
	static TSTR *name;
	ICustEdit *edit;
	TCHAR buf[256];

	switch (msg) {
	case WM_INITDIALOG:
		name = (TSTR*)lParam;
		edit =GetICustEdit(GetDlgItem(hWnd,IDC_SET_NAME));
		edit->SetText(*name);
		ReleaseICustEdit(edit);
		break;

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDOK:
			edit =GetICustEdit(GetDlgItem(hWnd,IDC_SET_NAME));
			edit->GetText(buf,256);
			*name = TSTR(buf);
			ReleaseICustEdit(edit);
			EndDialog(hWnd,1);
			break;

		case IDCANCEL:
			EndDialog(hWnd,0);
			break;
		}
		break;

	default:
		return FALSE;
	}
	return TRUE;
}

BOOL EditPolyObject::GetUniqueSetName(TSTR &name, bool useDlg) {
	TSTR startname = name;
	TCHAR buffer[80];
	for (int loopCount=0; loopCount<100; loopCount++) {
		GenericNamedSelSetList & setList = selSet[namedSetLevel[selLevel]];

		BOOL unique = TRUE;
		for (int i=0; i<setList.Count(); i++) {
			if (name==*setList.names[i]) {
				unique = FALSE;
				break;
			}
		}
		if (unique) break;

		if (useDlg) {
			if (!ip) return FALSE;
			if (!DialogBoxParam (hInstance, MAKEINTRESOURCE(IDD_NAMEDSET_PASTE),
				ip->GetMAXHWnd(), PickSetNameDlgProc, (LPARAM)&name)) return FALSE;
		} else {
			sprintf (buffer, "%s%d", startname, loopCount);
			name = TSTR(buffer);
		}
	}
	return TRUE;
}

static INT_PTR CALLBACK PickSetDlgProc (HWND hWnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch (msg) {
	case WM_INITDIALOG:	{
		GenericNamedSelSetList *setList = (GenericNamedSelSetList *)lParam;
		for (int i=0; i<setList->Count(); i++) {
			int pos  = SendDlgItemMessage(hWnd,IDC_NS_LIST,LB_ADDSTRING,0,
				(LPARAM)(TCHAR*)*(setList->names[i]));
			SendDlgItemMessage(hWnd,IDC_NS_LIST,LB_SETITEMDATA,pos,i);
		}
		break;
	}

	case WM_COMMAND:
		switch (LOWORD(wParam)) {
		case IDC_NS_LIST:
			if (HIWORD(wParam)!=LBN_DBLCLK) break;
			// fall through

		case IDOK:
			int sel;
			sel = SendDlgItemMessage(hWnd,IDC_NS_LIST,LB_GETCURSEL,0,0);
			if (sel!=LB_ERR) {
				int res =SendDlgItemMessage(hWnd,IDC_NS_LIST,LB_GETITEMDATA,sel,0);
				EndDialog(hWnd,res);
				break;
			}
			// fall through

		case IDCANCEL:
			EndDialog(hWnd,-1);
			break;
		}
		break;

	default:
		return FALSE;
	}
	return TRUE;
}

int EditPolyObject::SelectNamedSet() {
	GenericNamedSelSetList & setList = selSet[namedSetLevel[selLevel]];
	return DialogBoxParam (hInstance, MAKEINTRESOURCE(IDD_NAMEDSET_SEL),
		ip->GetMAXHWnd(), PickSetDlgProc, (LPARAM)&setList);
}

class NamedSetSizeIncrease : public RestoreObj {
public:
	EditPolyObject *ep;
	int nsl;
	int oldsize, increase;
	NamedSetSizeIncrease () { ep=NULL; }
	NamedSetSizeIncrease (EditPolyObject *eto, int n, int old, int inc) { ep=eto; nsl=n; oldsize=old; increase=inc; }
	void Restore (int isUndo) { ep->selSet[nsl].SetSize (oldsize); }
	void Redo () { ep->selSet[nsl].SetSize (oldsize+increase); }
	int GetSize () { return 3*sizeof(int) + sizeof (void *); }
	TSTR Description () { return _T("Named Selection Set Size Increase"); }
};

void EditPolyObject::IncreaseNamedSetSize (int nsl, int oldsize, int increase) {
	if (increase == 0) return;
	if (selSet[nsl].Count() == 0) return;
	if (theHold.Holding())
		theHold.Put (new NamedSetSizeIncrease (this, nsl, oldsize, increase));
	selSet[nsl].SetSize (oldsize + increase);
}

class NamedSetDelete : public RestoreObj {
public:
	EditPolyObject *ep;
	int nsl;
	Tab<BitArray *> oldSets;
	BitArray del;

	NamedSetDelete (EditPolyObject *eto, int n, BitArray &d);
	~NamedSetDelete ();
	void Restore (int isUndo);
	void Redo () { ep->selSet[nsl].DeleteSetElements (del, (nsl==NS_EDGE) ? 3 : 1); }
	int GetSize () { return 3*sizeof(int) + sizeof (void *); }
	TSTR Description () { return _T("Named Selection Set Subset Deletion"); }
};

NamedSetDelete::NamedSetDelete (EditPolyObject *eto, int n, BitArray &d) {
	ep = eto;
	nsl = n;
	del = d;
	oldSets.SetCount (ep->selSet[nsl].Count());
	for (int i=0; i<ep->selSet[nsl].Count(); i++) {
		oldSets[i] = new BitArray;
		(*oldSets[i]) = (*(ep->selSet[nsl].sets[i]));
	}
}

NamedSetDelete::~NamedSetDelete () {
	for (int i=0; i<oldSets.Count(); i++) delete oldSets[i];
}

void NamedSetDelete::Restore (int isUndo) {
	int i, max = oldSets.Count();
	if (ep->selSet[nsl].Count() < max) max = ep->selSet[nsl].Count();
	for (i=0; i<max; i++) *(ep->selSet[nsl].sets[i]) = *(oldSets[i]);
}

void EditPolyObject::DeleteNamedSetArray (int nsl, BitArray &del) {
	if (del.NumberSet() == 0) return;
	if (selSet[nsl].Count() == 0) return;
	selSet[nsl].Alphabetize ();
	if (theHold.Holding()) 
		theHold.Put (new NamedSetDelete (this, nsl, del));
	selSet[nsl].DeleteSetElements (del, (nsl==NS_EDGE) ? 3 : 1);
}

void EditPolyObject::CreateContArray() {
	if (cont.Count()) return;
	AllocContArray (mm.numv);
	for (int i=0; i<cont.Count(); i++) SetPtCont(i, NULL);
}

void EditPolyObject::SynchContArray(int nv) {
	int i, cct = cont.Count();
	if (!cct) return;
	if (cct == nv) return;
	if (masterCont) masterCont->SetNumSubControllers(nv, TRUE);
	if (cct>nv) {
		cont.Delete (nv, cct-nv);
		return;
	}
	cont.Resize (nv);
	Control *dummy=NULL;
	for (i=cct; i<nv; i++) cont.Append (1, &dummy);
}

void EditPolyObject::AllocContArray(int count) {
	cont.SetCount (count);
	if (masterCont) masterCont->SetNumSubControllers (count);
}

void EditPolyObject::ReplaceContArray(Tab<Control *> &nc) {
	AllocContArray (nc.Count());
	for(int i=0; i<nc.Count(); i++) SetPtCont (i, nc[i]);
}

RefTargetHandle EditPolyObject::GetReference(int i) {
	if (i == EPOLY_PBLOCK) return pblock;
	if (i == EPOLY_MASTER_CONTROL_REF) return masterCont;
	if (i >= (cont.Count() + EPOLY_VERT_BASE_REF)) return NULL;
	return cont[i - EPOLY_VERT_BASE_REF];
}

void EditPolyObject::SetReference(int i, RefTargetHandle rtarg) {
	if (i == EPOLY_PBLOCK) {
		pblock = (IParamBlock2 *) rtarg;
		return;
	}

	if(i == EPOLY_MASTER_CONTROL_REF) {
		masterCont = (MasterPointControl*)rtarg;
		if (masterCont) masterCont->SetNumSubControllers(cont.Count());
		return;
	}

	if(i < (mm.numv + EPOLY_VERT_BASE_REF)) {
		if (!cont.Count()) CreateContArray();
		SetPtCont(i - EPOLY_VERT_BASE_REF, (Control*)rtarg); 
	}
}

TSTR EditPolyObject::SubAnimName(int i) {
	if (i == EPOLY_PBLOCK) return GetString (IDS_PBLOCK);
	if (i == EPOLY_MASTER_CONTROL_REF) return GetString(IDS_MASTERCONT);
	TSTR buf;
	if(i < (cont.Count() + EPOLY_VERT_BASE_REF))
		buf.printf(GetString(IDS_WHICH_POINT), i+1-EPOLY_VERT_BASE_REF);
	return buf;
}

void EditPolyObject::DeletePointConts(BitArray &set) {
	if (!cont.Count()) return;

	BOOL deleted = FALSE;
	Tab<Control*> nc;
	nc.SetCount(cont.Count());
	int ix=0;
	for (int i=0; i<cont.Count(); i++) {
		if (!set[i]) nc[ix++] = cont[i];
		else deleted = TRUE;
	}
	nc.SetCount(ix);
	nc.Shrink();
	ReplaceContArray(nc);
}

void EditPolyObject::PlugControllersSel(TimeValue t,BitArray &set) {
	BOOL res = FALSE;
	if (!Animating() || t==0) return;
	for (int i=0; i<mm.numv; i++) if (set[i] && PlugControl(t,i)) res = TRUE;
	if (res) NotifyDependents(FOREVER,0,REFMSG_SUBANIM_STRUCTURE_CHANGED);
}

BOOL EditPolyObject::PlugControl(TimeValue t,int i) {
	if (!Animating() || t==0) return FALSE;
	if (!cont.Count()) CreateContArray();
	if (cont[i]) return FALSE;

	MakeRefByID (FOREVER, i + EPOLY_VERT_BASE_REF, NewDefaultPoint3Controller());
	SuspendAnimate();
	AnimateOff();
	theHold.Suspend ();
	cont[i]->SetValue (t, &mm.v[i].p);
	theHold.Resume ();
	ResumeAnimate ();
	masterCont->SetSubController (i, cont[i]);
	return TRUE;
}

void EditPolyObject::SetPtCont (int i, Control *c) {
	cont[i]=c;
	if (masterCont) masterCont->SetSubController(i, c);
}

void EditPolyObject::SetPointAnim (TimeValue t, int i, Point3 pt) {
	if (cont.Count() && cont[i]) cont[i]->SetValue(t,&pt);
	else mm.v[i].p = pt;
}

void EditPolyObject::InvalidateDistanceCache () {
	InvalidateSoftSelectionCache ();
	if (!tempData) return;
	tempData->InvalidateDistances ();
}

void EditPolyObject::InvalidateSoftSelectionCache () {
	arValid = NEVER;
	if (!tempData) return;
	tempData->InvalidateSoftSelection ();
	int affectRegion;
	pblock->GetValue (ep_ss_use, TimeValue(0), affectRegion, FOREVER);
	if (affectRegion) NotifyDependents (FOREVER, PART_SELECT, REFMSG_CHANGE);
}

void EditPolyObject::UpdateGeometry (TimeValue time) {
	arValid = NEVER;
	geomValid = FOREVER;
	for (int i=0; i<cont.Count(); i++) {
		if (cont[i]) cont[i]->GetValue (time, mm.P(i), geomValid);
	}
	mm.InvalidateGeomCache ();
	mm.freeRVerts ();
	InvalidateTempData (PART_GEOM);
}

void EditPolyObject::UpdateSoftSelection (TimeValue time) {
	if (!geomValid.InInterval (time)) UpdateGeometry (time);

	arValid = geomValid;

	int useSoftSel, useEdgeDist, edgeIts, affectBack;
	float falloff, pinch, bubble;
	pblock->GetValue (ep_ss_use, time, useSoftSel, arValid);
	if (!useSoftSel) {
		mm.freeVSelectionWeights ();
		return;
	}

	pblock->GetValue (ep_ss_edist_use, time, useEdgeDist, arValid);
	if (useEdgeDist) {
		pblock->GetValue (ep_ss_edist, time, edgeIts, arValid);
	}
	pblock->GetValue (ep_ss_affect_back, time, affectBack, arValid);
	pblock->GetValue (ep_ss_falloff, time, falloff, arValid);
	pblock->GetValue (ep_ss_pinch, time, pinch, arValid);
	pblock->GetValue (ep_ss_bubble, time, bubble, arValid);

	mm.SupportVSelectionWeights ();
	float *vsw = mm.getVSelectionWeights();
	float *myVSW = TempData()->VSWeight (useEdgeDist, edgeIts, !affectBack,
		falloff, pinch, bubble)->Addr(0);
	memcpy (vsw, myVSW, mm.numv*sizeof(float));
}

class Epomop : public MeshOpProgress {
public:
	EditPolyObject *epo;
	bool testEscape;
	Epomop () { epo = NULL; testEscape=false; }
	Epomop (EditPolyObject *e, int order) { epo=e; SuperInit (order); }
	void SuperInit (int order);	// My localized init
	void Init (int numSteps) {}
	BOOL Progress (int step);
};

void Epomop::SuperInit (int total) {
	if (epo) epo->ClearFlag (EPOLY_ABORTED);
	if (total<200) { testEscape=false; return; }
	testEscape = true;
	SetCursor (LoadCursor (NULL, IDC_WAIT));
	GetAsyncKeyState (VK_ESCAPE);	// to clear the escape-key.
}

BOOL Epomop::Progress(int p) {
	if (!testEscape) return TRUE;
	if (GetAsyncKeyState(VK_ESCAPE)) {
		if (epo) epo->SetFlag(EPOLY_ABORTED);
		return FALSE;
	}
	else return TRUE;
}

void EditPolyObject::UpdateSubdivResult (TimeValue t) {
	BOOL useSubdiv=false;
	if (pblock) pblock->GetValue (ep_surf_subdivide, t, useSubdiv, FOREVER);
	if (!useSubdiv) return;

	if (subdivValid.InInterval (t)) {
		subdivResult.selLevel = mm.selLevel;
		return;
	}

	bool inRender = GetFlag (EPOLY_IN_RENDER);
	bool forceUpdate = GetFlag (EPOLY_FORCE);
	int update;
	pblock->GetValue (ep_surf_update, t, update, subdivValid);
	if (!forceUpdate && ((update==2) || ((update==1) && !inRender))) return;

	if (!mm.GetFlag (MN_MESH_FILLED_IN)) {
		mm.FillInMesh ();
	}
	if (!geomValid.InInterval(t)) UpdateGeometry (t);

	BOOL useRIter=false, useRThresh=false, sepSmooth, sepMat, smoothResult;
	int iter;
	float thresh;
	pblock->GetValue (ep_surf_subdiv_smooth, t, smoothResult, subdivValid);
	if (inRender) {
		pblock->GetValue (ep_surf_use_riter, t, useRIter, subdivValid);
		pblock->GetValue (ep_surf_use_rthresh, t, useRThresh, subdivValid);
	}
	if (useRIter) pblock->GetValue (ep_surf_riter, t, iter, subdivValid);
	else pblock->GetValue (ep_surf_iter, t, iter, subdivValid);
	if (useRThresh) pblock->GetValue (ep_surf_rthresh, t, thresh, subdivValid);
	else pblock->GetValue (ep_surf_thresh, t, thresh, subdivValid);
	pblock->GetValue (ep_surf_sep_smooth, t, sepSmooth, subdivValid);
	pblock->GetValue (ep_surf_sep_mat, t, sepMat, subdivValid);

	subdivResult = mm;
	subdivValid = geomValid;
	ClearFlag (EPOLY_FORCE);

	subdivResult.EliminateBadVerts ();	// don't trust NO_BAD_VERTS flag.
	subdivResult.SetMapSeamFlags ();
	subdivResult.FenceOneSidedEdges ();
	if (sepMat) subdivResult.FenceMaterials ();
	if (sepSmooth) subdivResult.FenceSmGroups ();

	DWORD subdivFlags = MN_SUBDIV_NEWMAP;
	int nstart = subdivResult.TargetVertsBySelection (MNM_SL_OBJECT);

	// Order-of-magnitude calculation:
	// We approximately multiply the number of active components by four each iteration.
	int order = nstart;
	order *=4;
	if (iter>1) order *=4;
	if (iter>2) order *=4;
	if (iter>3) order *=4;
	Epomop epomop(this, order);

	int oldfnum = subdivResult.numf;
	for (int i=0; i<iter; i++) {
		if (!epomop.Progress(i)) break;
		// Can following line be removed?
		subdivResult.OrderVerts ();
		if (thresh<1) subdivResult.DetargetVertsBySharpness (1.0f-thresh);
		if (!epomop.Progress(i)) break;

		subdivResult.CubicNURMS (&epomop, NULL, subdivFlags);
		subdivResult.CollapseDeadFaces ();
		// Can following line be removed?
		// Certainly it should go after FillInMesh below...  But are either needed?
		subdivResult.EliminateBadVerts();

		if (GetFlag (EPOLY_ABORTED)) break;

		if (!subdivResult.GetFlag (MN_MESH_FILLED_IN)) subdivResult.FillInMesh ();
		subdivResult.SetMapSeamFlags ();
		subdivResult.FenceOneSidedEdges ();
		if (sepMat) subdivResult.FenceMaterials ();
		if (sepSmooth) subdivResult.FenceSmGroups ();
	}

	if (GetFlag (EPOLY_ABORTED)) {
		subdivValid.SetInstant(t);
		if (!GetFlag (EPOLY_IN_RENDER)) pblock->SetValue (ep_surf_update, t, 2);	// Set to manual update.
	} else {
		// Apply smoothing:
		if (smoothResult) {
			subdivResult.ClearEFlags (MN_WHATEVER);
			subdivResult.PropegateComponentFlags (MNM_SL_EDGE, MN_WHATEVER,
				MNM_SL_EDGE, MN_EDGE_NOCROSS);
			float *ec;
			if (ec = subdivResult.edgeFloat (EDATA_CREASE)) {
				for (i=0; i<subdivResult.nume; i++) {
					if (ec[i]) subdivResult.e[i].SetFlag (MN_WHATEVER);
				}
			}
			subdivResult.SmoothByCreases (MN_WHATEVER);
		}
	}
}

ObjectState EditPolyObject::Eval(TimeValue time) {
	if (!mm.GetFlag (MN_MESH_FILLED_IN)) {
		mm.FillInMesh ();
	}
	if (!geomValid.InInterval(time)) UpdateGeometry (time);
	if (!arValid.InInterval (time)) UpdateSoftSelection (time);

	// RB 7/11/97: There's no reason for any of the other intervals
	// to be anything but FOREVER. There are some cases where they
	// can end up as an instant point in time
	topoValid   = FOREVER;
	texmapValid = FOREVER;
	selectValid = FOREVER;
	vcolorValid = FOREVER;

	return ObjectState(this);
}

RefResult EditPolyObject::NotifyRefChanged (Interval changeInt, RefTargetHandle hTarget,
										   PartID& partID, RefMessage message) {
	switch (message) {
	case REFMSG_CHANGE:
		if (hTarget == pblock) {
			// if this was caused by a NotifyDependents from pblock, LastNotifyParamID()
			// will contain ID to update, else it will be -1 => inval whole rollout
			int pid = pblock->LastNotifyParamID ();
			if (editObj==this) InvalidateDialogElement (pid);
			if ((pid >= ep_surf_subdivide) && (pid <= ep_surf_sep_mat))
				subdivValid.SetEmpty ();
			if (pid==ep_surf_subdiv_smooth) subdivValid.SetEmpty ();
			switch (pid) {
			case ep_vert_color_selby:
				if (editObj==this) InvalidateSurfaceUI();
				break;
			case ep_sd_use:
			case ep_sd_split_mesh:
			case ep_sd_method:
			case ep_sd_tess_steps:
			case ep_sd_tess_edge:
			case ep_sd_tess_distance:
			case ep_sd_tess_angle:
			case ep_sd_view_dependent:
			case ep_asd_style:
			case ep_asd_min_iters:
			case ep_asd_max_iters:
			case ep_asd_max_tris:
				SetDisplacementParams ();
				break;
			}
		}
		geomValid.SetEmpty();
		break;
	}
	return REF_SUCCEED;
}

BOOL EditPolyObject::CloneVertCont(int from, int to) {	
	if (cont.Count() && cont[from]) {
		RemapDir *remap = NewRemapDir();
		ReplaceReference(to,remap->CloneRef(cont[from]));
		remap->DeleteThis();
		return TRUE;
	}
	return FALSE;
}

BOOL EditPolyObject::AssignController (Animatable *control, int subAnim) {
	ReplaceReference (subAnim, (Control*)control);
	if (subAnim==EPOLY_MASTER_CONTROL_REF) {
		int n = cont.Count();
		masterCont->SetNumSubControllers(n);
		for (int i=0; i<n; i++) if (cont[i]) masterCont->SetSubController(i,cont[i]);
	}
	NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
	NotifyDependents(FOREVER,0,REFMSG_SUBANIM_STRUCTURE_CHANGED);
	return TRUE;
}

#ifdef _SUBMTLASSIGNMENT

// CCJ 1/19/98
// Return the sub object material assignment interface
// This is used by the node when assigning materials.
// If an open face selection mode is active, the material
// will be assigned to the selected faces only.
// A multi/sub-object material is created and the material
// is assigned to the matierial ID created for the selected
// faces.
void* EditPolyObject::GetInterface(ULONG id) {
	switch (id) {
	case I_SUBMTLAPI: return (ISubMtlAPI*)this;
	case I_MESHSELECT: return (IMeshSelect*)this;
	case I_MESHSELECTDATA: return (IMeshSelectData*)this;
	}
	//JH 3/8/99
	//This previously called Object"s implementation
	return Object::GetInterface(id);
}

// Return a material ID that is currently not used by the object.
// If the current face selection share once single MtlID that is not
// used by any other faces, you should use it.
MtlID EditPolyObject::GetNextAvailMtlID(ModContext* mc) {
	int mtlID = GetSelFaceUniqueMtlID(mc);

	if (mtlID == -1) {
		int i;
		BitArray b;
		mtlID = mm.numf;
		b.SetSize (mm.numf, FALSE);
		b.ClearAll ();
		for (i=0; i<mm.numf; i++) {
			int mid = mm.f[i].material;
			if (mid < mm.numf) b.Set(mid);
		}

		for (i=0; i<mm.numf; i++) {
			if (!b[i]) {
				mtlID = i;
				break;
			}
		}
	}

	return (MtlID)mtlID;
}

// Indicate if you are active in the modifier panel and have an 
// active face selection
BOOL EditPolyObject::HasFaceSelection(ModContext* mc) {
	// Are we the edited object?
	if (ip == NULL)  return FALSE;
	// Is Face selection active?
	if (selLevel < EP_SL_FACE) return FALSE;
	// Do we have any selected faces?
	for (int i=0; i<mm.numf; i++) {
		if (mm.f[i].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) break;
	}
	return (i<mm.numf);
}

// Set the selected faces to the specified material ID.
// If bResetUnsel is TRUE, then you should set the remaining
// faces material ID's to 0
void EditPolyObject::SetSelFaceMtlID(ModContext* mc, MtlID id, BOOL bResetUnsel) {
	if (theHold.Holding() && !TestAFlag(A_HELD))
		theHold.Put(new MtlIDRestore(this));

	for (int i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (mm.f[i].GetFlag (MN_SEL)) mm.f[i].material = id;
		else if (bResetUnsel) mm.f[i].material = 0;
	}

	LocalDataChanged (PART_TOPO);
}

// Return the material ID of the selected face(s).
// If multiple faces are selected they should all have the same MtlID -
// otherwise you should return -1.
// If faces other than the selected share the same material ID, then 
// you should return -1.
int EditPolyObject::GetSelFaceUniqueMtlID(ModContext* mc) {
	int	i;
	int	mtlID;

	mtlID = GetSelFaceAnyMtlID(mc);

	if (mtlID == -1) return mtlID;
	for (i=0; i<mm.numf; i++) {
		if (mm.f[i].GetFlag (MN_SEL|MN_DEAD)) continue;
		if (mm.f[i].material != mtlID) continue;
		mtlID = -1;
		break;
	}

	return mtlID;
}

// Return the material ID of the selected face(s).
// If multiple faces are selected they should all have the same MtlID,
// otherwise you should return -1.
int EditPolyObject::GetSelFaceAnyMtlID(ModContext* mc) {
	int				mtlID = -1;
	BOOL			bGotFirst = FALSE;
	int				i;

	for (i=0; i<mm.numf; i++) {
		if (!mm.f[i].FlagMatch (MN_SEL|MN_DEAD, MN_SEL)) continue;
		if (bGotFirst) {
			if (mtlID != mm.f[i].material) {
				mtlID = -1;
				break;
			}
		} else {
			mtlID = mm.f[i].material;
			bGotFirst = TRUE;
		}
	}

	return mtlID;
}

// Return the highest MtlID used by the object.
int EditPolyObject::GetMaxMtlID(ModContext* mc) {
	MtlID mtlID = 0;
	for (int i=0; i<mm.numf; i++) mtlID = max(mtlID, mm.f[i].material);
	return mtlID;
}

#endif // _SUBMTLASSIGNMENT

BaseInterface *EditPolyObject::GetInterface (Interface_ID id) {
	if (id == EPOLY_INTERFACE) return (EPoly *)this;
	return FPMixinInterface::GetInterface(id);
}

void EditPolyObject::LocalDataChanged () {
	LocalDataChanged (PART_SELECT);
}

void EditPolyObject::SetSelLevel (DWORD lev) {
	DWORD sl;
	switch (lev) {
	case IMESHSEL_OBJECT:
		sl = EP_SL_OBJECT;
		break;
	case IMESHSEL_VERTEX:
		sl = EP_SL_VERTEX;
		break;
	case IMESHSEL_EDGE:
		if (meshSelLevel[selLevel] == MNM_SL_EDGE) sl = selLevel;
		else sl = EP_SL_EDGE;
		break;
	case IMESHSEL_FACE:
		if (meshSelLevel[selLevel] == MNM_SL_FACE) sl = selLevel;
		else sl = EP_SL_FACE;
		break;
	}
	if (ip) ip->SetSubObjectLevel (sl);
	else InvalidateTempData (PART_SUBSEL_TYPE);
}

DWORD EditPolyObject::GetSelLevel () {
	switch (selLevel) {
	case EP_SL_OBJECT:
		return IMESHSEL_OBJECT;
	case EP_SL_VERTEX:
		return IMESHSEL_VERTEX;
	case EP_SL_EDGE:
	case EP_SL_BORDER:
		return IMESHSEL_EDGE;
	case EP_SL_FACE:
	case EP_SL_ELEMENT:
		return IMESHSEL_FACE;
	}
	return IMESHSEL_OBJECT;
}

BitArray EditPolyObject::GetSel (int nsl) {
	BitArray ret;
	switch (nsl) {
	case NS_VERTEX:
		mm.getVertexSel (ret);
		break;
	case NS_EDGE:
		mm.getEdgeSel (ret);
		break;
	case NS_FACE:
		mm.getFaceSel (ret);
		break;
	}
	return ret;
}

void EditPolyObject::SetVertSel(BitArray &set, IMeshSelect *imod, TimeValue t) {
	if (ip) ip->ClearCurNamedSelSet();
	if (theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));
	mm.VertexSelect (set);
	mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_SEL, MNM_SL_VERTEX, MN_HIDDEN, FALSE, FALSE);
}

void EditPolyObject::SetFaceSel(BitArray &set, IMeshSelect *imod, TimeValue t) {
	if (ip) ip->ClearCurNamedSelSet();
	if (theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_FACE));
	mm.FaceSelect (set);
	mm.PropegateComponentFlags (MNM_SL_FACE, MN_SEL, MNM_SL_FACE, MN_HIDDEN, FALSE, FALSE);
	DisplayFaceFlags();
}

void EditPolyObject::SetEdgeSel(BitArray &set, IMeshSelect *imod, TimeValue t) {
	if (ip) ip->ClearCurNamedSelSet();
	if (theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_EDGE));
	mm.EdgeSelect (set);
}

void EditPolyObject::EpSetVertexFlags (BitArray &vset, DWORD flags, DWORD fmask, bool undoable) {
	if ((flags & MN_SEL) && ip) ip->ClearCurNamedSelSet();
	if (undoable && theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_VERTEX));

	fmask = fmask | flags;
	int i, max = (mm.numv > vset.GetSize()) ? vset.GetSize() : mm.numv;
	for (i=0; i<max; i++) {
		mm.v[i].ClearFlag (fmask);
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (!vset[i]) continue;
		mm.v[i].SetFlag (flags);
	}
	if (flags & (MN_HIDDEN|MN_SEL)) {
		mm.PropegateComponentFlags (MNM_SL_VERTEX, MN_SEL, MNM_SL_VERTEX, MN_HIDDEN, FALSE, FALSE);
		DWORD parts = PART_DISPLAY|PART_SELECT;
		if (flags & MN_HIDDEN) parts |= PART_GEOM;	// for bounding box change
		LocalDataChanged (parts);
	}
}

void EditPolyObject::EpSetEdgeFlags (BitArray &eset, DWORD flags, DWORD fmask, bool undoable) {
	if ((flags & MN_SEL) && ip) ip->ClearCurNamedSelSet();
	if (undoable && theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_EDGE));

	fmask = fmask | flags;
	int i, max = (mm.nume > eset.GetSize()) ? eset.GetSize() : mm.nume;
	for (i=0; i<max; i++) {
		mm.e[i].ClearFlag (fmask);
		if (mm.e[i].GetFlag (MN_DEAD)) continue;
		if (!eset[i]) continue;
		mm.e[i].SetFlag (flags);
	}
	if (flags & MN_SEL) LocalDataChanged (PART_SELECT|PART_DISPLAY);
}

void EditPolyObject::EpSetFaceFlags (BitArray &fset, DWORD flags, DWORD fmask, bool undoable) {
	if ((flags & MN_SEL) && ip) ip->ClearCurNamedSelSet();
	if (undoable && theHold.Holding()) theHold.Put (new ComponentFlagRestore (this, MNM_SL_FACE));

	fmask = fmask | flags;
	int i, max = (mm.numf > fset.GetSize()) ? fset.GetSize() : mm.numf;
	for (i=0; i<max; i++) {
		mm.f[i].ClearFlag (fmask);
		if (mm.f[i].GetFlag (MN_DEAD)) continue;
		if (!fset[i]) continue;
		mm.f[i].SetFlag (flags);
	}
	if (flags & (MN_HIDDEN|MN_SEL)) {
		mm.PropegateComponentFlags (MNM_SL_FACE, MN_SEL, MNM_SL_FACE, MN_HIDDEN, FALSE, FALSE);
		DWORD parts = PART_DISPLAY|PART_SELECT;
		if (flags & MN_HIDDEN) parts |= PART_GEOM;	// for bounding box change
		LocalDataChanged (parts);
	}
}

void EditPolyObject::SetSel (int nsl, BitArray & set, IMeshSelect *imod, TimeValue t) {
	switch (nsl) {
	case NS_VERTEX: SetVertSel (set, imod, t); return;
	case NS_EDGE: SetEdgeSel (set, imod, t); return;
	}
	SetFaceSel (set, imod, t);
}

void EditPolyObject::ApplyMapDelta (int mp, Tab<UVVert> & mapDelta, EPoly *epol, TimeValue t) {
	MapVertRestore *mvr = NULL;
	if (theHold.Holding ()) mvr = new MapVertRestore (this, mp);
	if (mm.M(mp)->GetFlag (MN_DEAD)) return;
	UVVert *mv = mm.M(mp)->v;
	for (int i=0; i<mm.M(mp)->numv; i++) mv[i] += mapDelta[i];
	if (mvr) {
		if (!mvr->ReduceMem()) {	// no changes occurred.
			delete mvr;
			return;
		}
		theHold.Put (mvr);
	}
	epol->LocalDataChanged ((mp<1) ? PART_VERTCOLOR : PART_TEXMAP);
}

void EditPolyObject::ApplyDelta (Tab<Point3> & delta, EPoly *epol, TimeValue t) {

	MNMesh& mesh=GetMesh();

	// Get the face-data channel from the incoming mesh
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
	if( pFDMgr )
	{
		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>(pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
		if( fdc == NULL ) 
		{
			
			// The mesh does not have our face-data channel so we will add it here
			fdc = new FaceFlagsData();
			fdc->FacesCreated( 0, mesh.FNum() );		

			pFDMgr->AddFaceDataChan( fdc );			
		}
	}
	
	if (Animating() && t) {
		BOOL addedCont = FALSE;
		for (int i=0; i<mm.numv; i++) {
			if (mm.v[i].GetFlag (MN_DEAD)) continue;
			if (delta[i] == Point3(0,0,0)) continue;
			if (!PlugControl(t, i)) continue;
			addedCont = TRUE;
			// Set initial position:
			cont[i]->SetValue (TimeValue(0), &(mm.v[i].p));
		}
		if (addedCont) NotifyDependents(FOREVER,0,REFMSG_SUBANIM_STRUCTURE_CHANGED);
	}
	if (theHold.Holding ()) theHold.Put (new MeshVertRestore (this));
	for (int i=0; i<mm.numv; i++) {
		if (delta[i] == Point3(0,0,0)) continue;
		mm.v[i].p += delta[i];
		if (cont.Count() && cont[i]) cont[i]->SetValue (t, &(mm.v[i].p));
	}
	epol->LocalDataChanged (PART_GEOM);
	epol->RefreshScreen ();
}

MNTempData *EditPolyObject::TempData () {
	if (!tempData) tempData = new MNTempData(&(mm));
	return tempData;
}

void EditPolyObject::InvalidateTempData (PartID parts) {
	if (!tempMove) {
		if (tempData) tempData->Invalidate (parts);
		if (parts & (PART_TOPO|PART_GEOM|PART_SELECT|PART_SUBSEL_TYPE))
			InvalidateSoftSelectionCache ();
	}
	// we NEVER call InvalidateTopoCache, since that trashes the edge list.
	if (parts & PART_TOPO) mm.freeRVerts();
	if (parts & PART_GEOM) mm.InvalidateGeomCache ();
}

static int dragRestored;

void EditPolyObject::DragMoveInit () {
	if (tempMove) delete tempMove;
	tempMove = new TempMoveRestore (this);
	dragRestored = TRUE;
}

void EditPolyObject::DragMoveRestore () {
	if (!tempMove) return;
	if (dragRestored) return;
	tempMove->Restore (this);
	dragRestored = TRUE;
	LocalDataChanged (PART_GEOM|PART_TEXMAP|PART_VERTCOLOR);
}

void EditPolyObject::DragMove (Tab<Point3> & delta, EPoly *epol, TimeValue t) {
	if (!tempMove) {
		ApplyDelta (delta, epol, t);
		return;
	}
	for (int i=0; i<mm.numv; i++) {
		if (mm.v[i].GetFlag (MN_DEAD)) continue;
		if (delta[i] == Point3(0,0,0)) continue;
		tempMove->active.Set (i);
		mm.v[i].p += delta[i];
	}
	if (theHold.Holding ()) theHold.Put (new CueDragRestore(this));
	dragRestored = FALSE;
	epol->LocalDataChanged (PART_GEOM);
}

void EditPolyObject::DragMap (int mp, Tab<UVVert> & mapDelta, EPoly *epol, TimeValue t) {
	if (!tempMove) {
		ApplyMapDelta (mp, mapDelta, epol, t);
		return;
	}
	if (mm.M(mp)->GetFlag (MN_DEAD)) return;
	UVVert *mv = mm.M(mp)->v;
	int numMapVerts = mm.M(mp)->numv;
	for (int i=0; i<numMapVerts; i++) mv[i] += mapDelta[i];
	epol->LocalDataChanged ((mp<1) ? PART_VERTCOLOR : PART_TEXMAP);
}

void EditPolyObject::DragMoveAccept (TimeValue t) {
	if (!tempMove) return;
	if (!tempMove->active.NumberSet()) {
		delete tempMove;
		tempMove = NULL;
		return;
	}

	// Check for polygons whose triangulation might now be invalid.
	// Also check to see if we need to add controllers.
	DWORD partsChanged = PART_GEOM|PART_TEXMAP|PART_VERTCOLOR;
	bool addedCont = false, needControllers = (Animating() && t) ? true : false;
	TopoChangeRestore *tchange = NULL;
	if (theHold.Holding ()) {
		tchange = new TopoChangeRestore (this);
		tchange->Before ();
	}
	for (int i=0; i<tempMove->active.GetSize(); i++) {
		if (!tempMove->active[i]) continue;
		if (needControllers && PlugControl (t,i)) {
			addedCont = true;
			cont[i]->SetValue (0, tempMove->init[i]);
		}

		if (!mm.vfac) {
			DbgAssert (false);
			continue;
		}
		// For each face using this vertex:
		for (int j=0; j<mm.vfac[i].Count(); j++) {
			int fj = mm.vfac[i][j];
			// For each triangle using this vertex:
			Tab<int> tri;
			Point3 fnorm = mm.GetFaceNormal (fj, true);
			mm.f[fj].GetTriangles (tri);
			for (int k=0; k<tri.Count(); k+=3) {
				for (int kk=0; kk<3; kk++) {
					if (mm.f[fj].vtx[tri[k+kk]] == i) break;
				}
				if (kk==3) continue;

				// Check to see if triangle has flipped normal:
				Point3 A = mm.v[mm.f[fj].vtx[tri[k+1]]].p - mm.v[mm.f[fj].vtx[tri[k]]].p;
				Point3 B = mm.v[mm.f[fj].vtx[tri[k+2]]].p - mm.v[mm.f[fj].vtx[tri[k]]].p;
				if (DotProd (A^B, fnorm) < -.001f) break;
			}
			if (k<tri.Count()) {
				mm.RetriangulateFace (fj);
				partsChanged |= PART_TOPO;
			}
		}
	}
	if (tchange) {
		if (partsChanged & PART_TOPO) {
			tchange->After ();
			theHold.Put (tchange);
		} else {
			delete tchange;
		}
	}


	if (addedCont) NotifyDependents(FOREVER,0,REFMSG_SUBANIM_STRUCTURE_CHANGED);

	if (cont.Count()) {
		for (i=0; i<tempMove->active.GetSize(); i++) {
			if (!tempMove->active[i]) continue;
			if (cont[i]) cont[i]->SetValue (t, &(mm.v[i].p));
		}
	}
	if (theHold.Holding()) {
		MeshVertRestore *mvr = new MeshVertRestore(this);
		memcpy (mvr->undo.Addr(0), tempMove->init.Addr(0), mm.numv*sizeof(Point3));
		theHold.Put (mvr);
		for (int mp=-NUM_HIDDENMAPS; mp<mm.numm; mp++) {
			if (mm.M(mp)->GetFlag (MN_DEAD)) continue;
			int nmp = NUM_HIDDENMAPS + mp;
			MapVertRestore *mpvr = new MapVertRestore (this, mp);
			memcpy (mpvr->preVerts.Addr(0), tempMove->mapInit[nmp].Addr(0),
				mpvr->preVerts.Count()*sizeof(UVVert));
			if (!mpvr->ReduceMem ()) delete mpvr;	// No changes in this channel
			else theHold.Put (mpvr);
		}
	}
	delete tempMove;
	tempMove = NULL;
	dragRestored = TRUE;
	LocalDataChanged (partsChanged);
}

void EditPolyObject::DragMoveClear () {
	if (tempMove) delete tempMove;
	tempMove=NULL;
}

// Class EPolyBackspaceUser: Used to process backspace key input.
void EPolyBackspaceUser::Notify() {
	if (!epo) return;
	if (!epo->createFaceMode) return;
	if (epo->ip->GetCommandMode () != epo->createFaceMode) return;
	epo->createFaceMode->proc.Backspace ();
}

// Neversoft Extensions   aml
void EditPolyObject::SelectByFaceFlags( FlagType &flags, bool exact_match )
{
	int i;
	theHold.Begin();
	BitArray nfs;
	
	//nfs.SetSize (GetMesh().getNumFaces());
	nfs.SetSize (GetMesh().FNum());
	nfs.ClearAll ();
	
	IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>( GetMesh().GetInterface( FACEDATAMGR_INTERFACE ));
	
	if( pFDMgr )
	{	
		FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));

		if (!fdc)
			return;

		//for( i = 0; i < GetMesh().getNumFaces(); i++ ) 
		for( i = 0; i < GetMesh().FNum(); i++ ) 
		{			
			FlagType value;

			fdc->GetValue( i, value );
			if( exact_match )
			{
				if( value == flags )
				{
					nfs.Set(i);
				}
			}
			else
			{
				if( value & flags )
				{
					nfs.Set(i);
				}
			}
			
		}
	}
	
	SetFaceSel (nfs, this, ip->GetTime());
	theHold.Accept( "Select by Face Flags" );
	LocalDataChanged ();
	ip->RedrawViews (ip->GetTime());
}

}	// end namespace EditPoly