#include <windows.h>
#include <stdlib.h>
#include <process.h>
#include <stdio.h>
#include <assert.h>
#include <string.h>
#include <iostream.h>
#include <fstream.h>

#include "png.h"
#include "Makefont.h"

#define streams		1
#define hilevel		1
#define entire		1

PNGInfo						ThePNGInfo;
FontInfo					TheFontInfo;

char Update[100] = "1/16/02, by Ryan McMahon, Neversoft";

typedef enum {
	NGPS = 0,
	NGC,
	XBOX,

	MAX_PLATFORM
} PLATFORM;

PLATFORM	g_platform = NGPS;

#define writeBig32(_f,_p)								\
{														\
	_f->write( &((char *)(_p))[3], sizeof( char ) );		\
	_f->write( &((char *)(_p))[2], sizeof( char ) );		\
	_f->write( &((char *)(_p))[1], sizeof( char ) );		\
	_f->write( &((char *)(_p))[0], sizeof( char ) );		\
}

#define writeBig16(_f,_p)								\
{														\
	_f->write( &((char *)(_p))[1], sizeof( char ) );		\
	_f->write( &((char *)(_p))[0], sizeof( char ) );		\
}

void writeNGC8( ofstream * p_file, unsigned char * p_image, RGBA * p_palette, int width, int height )
{
	int lp, lp2;

//	// Find number of unique colors.
//	int num_unique = 0;
//	for ( lp = 0; lp < 256; lp ++ )
//	{
//		bool found = false;
//		for ( lp2 = 0; lp2 < lp; lp2++ )
//		{
//			if (	( p_palette[lp].r == p_palette[lp2].r ) &&
//					( p_palette[lp].g == p_palette[lp2].g ) &&
//					( p_palette[lp].b == p_palette[lp2].b ) &&
//					( p_palette[lp].a == p_palette[lp2].a ) )
//			{
//				found = true;
//				break;
//			}
//		}
//		if ( !found ) num_unique++;
//	}
//	printf( "\nNum unique: %d", num_unique );

	// Convert from basic PS2 format to funky GC format.
	// GC format is like this:

	// 00 01 02 03 04 05 06 07 | 32 33 34 35 36 37 38 39 |
	// 08 09 10 11 12 13 14 15 | 40 41 42 43 44 45 46 47 |
	// 16 17 18 19 20 21 22 23 | 48 49 50 51 52 53 54 55 |
	// 24 25 26 27 28 29 30 31 | 56 57 58 59 60 61 62 63 | ...and so on...
	unsigned char * p8 = (unsigned char *)malloc ( width * height );
	if ( p8 ) {
		for ( lp = 0; lp < height; lp += 4 ) {
			for ( lp2 = 0; lp2 < width; lp2 += 8 ) {
				p8[ 0+(lp2*4)+(lp*width)] = p_image[0+(width*0)+lp2+(lp*width)];
				p8[ 1+(lp2*4)+(lp*width)] = p_image[1+(width*0)+lp2+(lp*width)];
				p8[ 2+(lp2*4)+(lp*width)] = p_image[2+(width*0)+lp2+(lp*width)];
				p8[ 3+(lp2*4)+(lp*width)] = p_image[3+(width*0)+lp2+(lp*width)];
				p8[ 4+(lp2*4)+(lp*width)] = p_image[4+(width*0)+lp2+(lp*width)];
				p8[ 5+(lp2*4)+(lp*width)] = p_image[5+(width*0)+lp2+(lp*width)];
				p8[ 6+(lp2*4)+(lp*width)] = p_image[6+(width*0)+lp2+(lp*width)];
				p8[ 7+(lp2*4)+(lp*width)] = p_image[7+(width*0)+lp2+(lp*width)];

				p8[ 8+(lp2*4)+(lp*width)] = p_image[0+(width*1)+lp2+(lp*width)];
				p8[ 9+(lp2*4)+(lp*width)] = p_image[1+(width*1)+lp2+(lp*width)];
				p8[10+(lp2*4)+(lp*width)] = p_image[2+(width*1)+lp2+(lp*width)];
				p8[11+(lp2*4)+(lp*width)] = p_image[3+(width*1)+lp2+(lp*width)];
				p8[12+(lp2*4)+(lp*width)] = p_image[4+(width*1)+lp2+(lp*width)];
				p8[13+(lp2*4)+(lp*width)] = p_image[5+(width*1)+lp2+(lp*width)];
				p8[14+(lp2*4)+(lp*width)] = p_image[6+(width*1)+lp2+(lp*width)];
				p8[15+(lp2*4)+(lp*width)] = p_image[7+(width*1)+lp2+(lp*width)];

				p8[16+(lp2*4)+(lp*width)] = p_image[0+(width*2)+lp2+(lp*width)];
				p8[17+(lp2*4)+(lp*width)] = p_image[1+(width*2)+lp2+(lp*width)];
				p8[18+(lp2*4)+(lp*width)] = p_image[2+(width*2)+lp2+(lp*width)];
				p8[19+(lp2*4)+(lp*width)] = p_image[3+(width*2)+lp2+(lp*width)];
				p8[20+(lp2*4)+(lp*width)] = p_image[4+(width*2)+lp2+(lp*width)];
				p8[21+(lp2*4)+(lp*width)] = p_image[5+(width*2)+lp2+(lp*width)];
				p8[22+(lp2*4)+(lp*width)] = p_image[6+(width*2)+lp2+(lp*width)];
				p8[23+(lp2*4)+(lp*width)] = p_image[7+(width*2)+lp2+(lp*width)];

				p8[24+(lp2*4)+(lp*width)] = p_image[0+(width*3)+lp2+(lp*width)];
				p8[25+(lp2*4)+(lp*width)] = p_image[1+(width*3)+lp2+(lp*width)];
				p8[26+(lp2*4)+(lp*width)] = p_image[2+(width*3)+lp2+(lp*width)];
				p8[27+(lp2*4)+(lp*width)] = p_image[3+(width*3)+lp2+(lp*width)];
				p8[28+(lp2*4)+(lp*width)] = p_image[4+(width*3)+lp2+(lp*width)];
				p8[29+(lp2*4)+(lp*width)] = p_image[5+(width*3)+lp2+(lp*width)];
				p8[30+(lp2*4)+(lp*width)] = p_image[6+(width*3)+lp2+(lp*width)];
				p8[31+(lp2*4)+(lp*width)] = p_image[7+(width*3)+lp2+(lp*width)];
			}
		}
	}
	p_file->write((const char*)p8, width * height );

	// Fix palette alpha.
	for ( lp = 0; lp < 256; lp ++ )
	{
		p_palette[lp].a = ( p_palette[lp].a >= 128 ) ? 255 : ( p_palette[lp].a * 2 );
	}

	for ( lp = 0; lp < 256; lp ++ )
	{
		// High bits.
		unsigned short pal;
		pal =	( ( p_palette[lp].r <<  4 ) & 0x0f00 ) |
				( ( p_palette[lp].g <<  0 ) & 0x00f0 ) |
				( ( p_palette[lp].b >>  4 ) & 0x000f ) |
				( ( p_palette[lp].a <<  7 ) & 0x7000 );
		writeBig16( p_file, &pal );
	}
//	for ( lp = 0; lp < 256; lp ++ )
//	{
//		// High bits.
//		unsigned short pal;
//		pal =	( ( p_palette[lp].r <<  8 ) & 0x0f00 ) |
//				( ( p_palette[lp].g <<  4 ) & 0x00f0 ) |
//				( ( p_palette[lp].b >>  0 ) & 0x000f ) |
//				( ( p_palette[lp].a << 10 ) & 0x7000 );
//		writeBig16( p_file, &pal );
//	}

	// Write padding for alignment purposes.
	unsigned int dummy = 0;
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
	writeBig32( p_file, &dummy );
}

void print_usage_message_and_quit()
{
	printf("---------------------------------\n");
	printf("Font Grabber Usage:\n\n");
	printf("Makefont [options] $infile$.png $outfile$.fnt $mapfile$.txt\n");
	printf("options: -c == print out capture details\n");
	printf("         -p == print out palette details\n\n");
	printf("other: see font building document in SourceSafe\n");
	printf("last update: %s\n", Update);
	printf("---------------------------------\n");
	exit(1);
}


int main(int argc, char *argv[]) 
{
	if (argc < 3) 
	{
		print_usage_message_and_quit();
		return 1;
	}

	bool print_capture_info = false; 
	bool print_palette_info = false; 
	
	// scan for options
	int count;
	for (count = 1; count < argc; count++)
	{
		if (*argv[count] == '-') 
		{
			switch(*(argv[count] + 1)) 
			{
			case 'c':
				print_capture_info = true;
				printf("Font Grabber, last updated %s\n", Update);
				break;
			case 'i':
				print_palette_info = true;
				break;
			case 'p':
				switch(*(argv[count] + 2)) 
				{
					case 'p':
						g_platform = NGPS;
						break;
					case 'g':
						g_platform = NGC;
						break;
					case 'x':
						g_platform = XBOX;
						break;
				}
				break;
			default:
				printf("invalid option\n");
				print_usage_message_and_quit();
				return(1);
			} // end switch
		} // end if

		else
			break;
	}
	
	if (argc < count + 3)
	{
		print_usage_message_and_quit();
		return(1);
	}
	
	char *p_png_name = argv[count];
	char *p_fnt_name = argv[count+1];
	char *p_map_name = argv[count+2];

	//if (print_capture_info || print_palette_info)
		printf("Creating font %s from PNG %s", p_fnt_name, p_png_name);

	LoadAsciiTable(p_map_name);
	
	if (!load_png(p_png_name, &ThePNGInfo))
		Assert(0, "load of PNG %s failed", p_png_name);

	BuildOutputPalette();
	
	if (print_capture_info)
	{
		printf("Load succeeded\n");
		printf("w,h = %d,%d\n", ThePNGInfo.mWidth, ThePNGInfo.mHeight);
		printf("BPP = %d\n", ThePNGInfo.mBpp);
	}
		
	TheFontInfo.mNumChars = 0;
	int line = 0;
	
	while(1)
	{
		line = FindDimsNextCharacter(line);
		if (line == -1) break;

		if (print_capture_info)
		{
			CharInfo &entry = TheFontInfo.mCharTab[TheFontInfo.mNumChars-1];
			if (entry.mAscii >= 0)
				printf("character dims for    %c ", (char) entry.mAscii);
			else
				printf("character dims for (%.2d)", (char) entry.mAscii);
			printf(" are: %d,%d,%d,%d\n", entry.mX, entry.mY, entry.mW, entry.mH);
		}

		/*
		if (entry.mAscii == 'u')
		{
			Uint32 *pRGBA = (Uint32 *) ThePNGInfo.mpRawPaletteData;
			Uint8 *pTexel = (Uint8 *) ThePNGInfo.mpTexelData + entry.mY * ThePNGInfo.mWidth + entry.mX;
			for (int j = 0; j < entry.mH; j++)
			{
				for (int i = 0; i < entry.mW; i++)
				{
					printf("%.2x ", *pTexel);
					pTexel++;
				}
				pTexel += ThePNGInfo.mWidth - entry.mW;

				printf("\n");
			}
		}
		*/
	}
	
	if (print_capture_info)
	{
		printf("total characters: %d\n", TheFontInfo.mNumChars);
		printf("max height: %d\n", TheFontInfo.mMaxHeight);
		printf("baseline: %d\n", TheFontInfo.mBaseLine);
		printf("highest pixel: %d\n", TheFontInfo.mHighestPixel);
		printf("lowest pixel: %d\n", TheFontInfo.mLowestPixel);
		printf("transparent-keyed palette entry: %d\n", ThePNGInfo.mTransIndex);
	}

	if (print_palette_info)
	{
		for (int col = 0; col < ThePNGInfo.mNumUsedPaletteEntries; col++)
			printf("palette entry %d: 0x%x, full_trans=%d\n", col, *((Uint32 *) ThePNGInfo.mpOutPaletteData + col), ThePNGInfo.mpTransTab[col]);
	}
	
	SaveToFile(p_fnt_name);
	
	printf("\n");
	return 0;
}



void sendOut(Uint8 *pImage, int out_x, int out_y, int n)
{
	Uint8 *pOut = TheFontInfo.mpOutBuf + out_y * TheFontInfo.mOutW + out_x;
	
	CharInfo &char_entry = TheFontInfo.mCharTab[n];
	Uint8 *pIn = (Uint8 *) ThePNGInfo.mpTexelData + char_entry.mY * ThePNGInfo.mWidth + char_entry.mX;

	for (int j = 0; j < char_entry.mH; j++)
	{
		for (int i = 0; i < char_entry.mW; i++)
			*(pOut++) = *(pIn++);
		pOut += TheFontInfo.mOutW - char_entry.mW;
		pIn += ThePNGInfo.mWidth - char_entry.mW;
	}
}


void SaveToFile(char *file) 
{
	// create output buffer

	TheFontInfo.mOutW = 256;
	TheFontInfo.mOutH = 256;
	// this number will probably increase
	TheFontInfo.mOutUsedH = 32;

	// We reserve space for a 256X256 image buffer, although we may output
	// fewer rows than 256 to disk, as determined by Out.usedH
	int out_buf_size = TheFontInfo.mOutW * TheFontInfo.mOutH;
	TheFontInfo.mpOutBuf = new Uint8[out_buf_size];
	// fill with transparent color
	memset(TheFontInfo.mpOutBuf, (char) ThePNGInfo.mTransIndex, out_buf_size);

	// reserve space for sub-texture descriptors
	OutCharInfo *pOutChar = new OutCharInfo[TheFontInfo.mNumChars];

	// reserve space for extra character info
	// (which isn't included in the texture part of the FNT file)
	ExtraCharInfo *pOutExtra = new ExtraCharInfo[TheFontInfo.mNumChars];

	int out_x = 1, out_y = 1;
	// used to count characters successfully written to disk
	// (some might be incorrect)
	int written_char = 0;

	// create image data
	for (int n = 1; n < TheFontInfo.mNumChars; n++)
	{
		CharInfo &char_entry = TheFontInfo.mCharTab[n];
		
		// skip any characters that have been read incorrectly
		if (char_entry.mW <= 0 || char_entry.mH <= 0 || char_entry.mW > TheFontInfo.mOutW)
		{
			printf("Character %d not exported. Dimensions incorrect:\n", n);
			printf("    w = %d, h = %d\n", char_entry.mW, char_entry.mH);
			continue;
		}

		// skip any characters not in ascii map file
		if (char_entry.mAscii == 0)
		{
			printf("Character %d not in ascii map file, skipped\n", n);
			continue;
		}
		
		if (out_x + char_entry.mW > TheFontInfo.mOutW - 1)
		{
			// want to leave 1 pixel horizontal buffer line between chars
			out_x = 1;
			out_y += TheFontInfo.mLowestPixel - TheFontInfo.mHighestPixel + 2;
			if (out_y + char_entry.mH > TheFontInfo.mOutH - 1)
				Assert(0, "not enough room in output bitmap\n");
			while (out_y + char_entry.mH > TheFontInfo.mOutUsedH) 
				TheFontInfo.mOutUsedH *= 2;
		}
		
		pOutChar[written_char].x = out_x;
		pOutChar[written_char].y = out_y;
		pOutChar[written_char].w = char_entry.mW;
		pOutChar[written_char].h = char_entry.mH;
		sendOut(TheFontInfo.mpOutBuf, out_x, out_y, n);

		pOutExtra[written_char].baseline = TheFontInfo.mBaseLine - char_entry.mY;
		pOutExtra[written_char].ascii = char_entry.mAscii;
		
		written_char++;

		// want to leave 1 pixel vertical line between chars
		out_x += char_entry.mW + 1;
	}
	int num_written_chars = written_char;
	// for accounting purposes
	if (out_x > 0)
		out_y += TheFontInfo.mLowestPixel - TheFontInfo.mHighestPixel + 2;
	
	/*
	if (OptPrint)
		printf("Final texture size: %d by %d by %dbpp (used height = %d)\n",
			Out.w, Out.usedH, BMP.bpp, out_y);	
	*/
	
	/*
	=======================================
	Do actual file writing now
	=======================================
	*/

	ofstream to(file, ios::out | ios::binary);

	// for writing garbage characters
	char blank[256];
	
	// compute sizes
	int out_size = (TheFontInfo.mOutW * TheFontInfo.mOutUsedH + 3) & (~3);
	int palette_size = 1024;
	int texture_size = 16 + out_size + palette_size + 4 + num_written_chars * 8;
	
	int font_size = 16 + 4 * num_written_chars + texture_size;
	
	/*
	=======================================
	Write FNT file specific section of file
	=======================================
	*/

	int default_height = TheFontInfo.mLowestPixel - TheFontInfo.mHighestPixel + 1;
	int max_char_height_above_base = TheFontInfo.mBaseLine - TheFontInfo.mHighestPixel;
	
	to.write((char *) &font_size, 4);
	to.write((char *) &num_written_chars, 4);
	to.write((char *) &default_height, 4);
	to.write((char *) &max_char_height_above_base, 4);
	to.write((char *) pOutExtra, 4 * num_written_chars);
	
	// should be nicely padded at this point

	/*
	=======================================
	Write Image::Texture section of file
	=======================================
	*/
	
	Sint16 temp_out_w = TheFontInfo.mOutW;
	Sint16 temp_out_h = TheFontInfo.mOutUsedH;
	Sint16 temp_bpp = 8;

	// write header
	to.write((char *) &texture_size, 4);
	to.write((char *) &temp_out_w, 2);
	to.write((char *) &temp_out_h, 2);
	to.write((char *) &temp_bpp, 2);
	// padding
	to.write(blank, 6);

	switch ( g_platform )
	{
		case NGC:
			writeNGC8( &to, TheFontInfo.mpOutBuf, ThePNGInfo.mpOutPaletteData, temp_out_w, temp_out_h );
			break;
		case NGPS:
		case XBOX:
		default:
			// write image data
			to.write((char *) TheFontInfo.mpOutBuf, out_size);

			// write palette data
			if (palette_size)
				to.write((char *) ThePNGInfo.mpOutPaletteData, palette_size);
			break;
	}

	// write total number of subtextures in texture
	to.write((char *) &num_written_chars, 4);

	// write subtexture descriptors
	to.write((char *) pOutChar, num_written_chars * 8);
	
	to.close();
}



/*
	Loads a file to set the ascii value for each character
*/
void LoadAsciiTable(char *file)
{
	ifstream from(file, ios::nocreate | ios::in);
	if (!from.good())
		Assert(0, "error opening file %s\n", file);

	// Fill with zeros in case there are fewer characters in the ascii map file
	// than in the BMP. Extra BMP characters won't be exported.
	for (int n = 0; n < 256; n++)
		TheFontInfo.mCharTab[n].mAscii = 0;

	ThePNGInfo.mUseGrayScaleTransKeying = false;

	n = 1; // the first character is for calibration, not used
	while(1) 
	{
		unsigned char c = from.get();
		if (from.eof())
			break;

		if (c == '\\')
		{
			c = from.get();
			if (c == '\\')
				TheFontInfo.mCharTab[n++].mAscii = '\\';
			else if (c >= '0' && c <= '9')
				TheFontInfo.mCharTab[n++].mAscii = -((int) (c - '0') + 1);
			else if (c >= 'A' && c <= 'V')
				TheFontInfo.mCharTab[n++].mAscii = -((int) (c - 'A') + 11);
			else if (c >= 'a' && c <= 'v')
				TheFontInfo.mCharTab[n++].mAscii = -((int) (c - 'a') + 11);
			else if (c == 'g' || c == 'G')
				ThePNGInfo.mUseGrayScaleTransKeying = true;
			else
				Assert(0, "unexpected character after \\ in input file\n");
		}
		else if (c != '\n') 
			TheFontInfo.mCharTab[n++].mAscii = c;
	}
}



// returns line from which to scan for next, -1 if end reached
int FindDimsNextCharacter(int line)
{
	Assert(TheFontInfo.mNumChars < 256, "more than 256 characters in source PNG");
	CharInfo *p_entry = &TheFontInfo.mCharTab[TheFontInfo.mNumChars];
	
	line = ScanForNextOccupiedLine(line);
	if (line == -1)
		return -1;
	
	p_entry->mX = line;

	int best_top = ThePNGInfo.mHeight;
	int best_bottom = 0;

	int top, bottom;
	
	while(ScanVerticalLine(line, &top, &bottom))
	{
		if (top < best_top)
			best_top = top;
		if (bottom > best_bottom)
			best_bottom = bottom;
		line++;
		Assert(line < ThePNGInfo.mWidth, "unexpectedly hit end of character strip");
	}

	p_entry->mY = best_top;
	p_entry->mW = line - p_entry->mX;
	p_entry->mH = best_bottom - best_top + 1;

	if (TheFontInfo.mNumChars == 0)
	{
		TheFontInfo.mBaseLine = best_bottom;
		TheFontInfo.mHighestPixel = ThePNGInfo.mHeight;
		TheFontInfo.mLowestPixel = 0;
	}
	if (TheFontInfo.mMaxHeight < p_entry->mH)
		TheFontInfo.mMaxHeight = p_entry->mH;

	if (TheFontInfo.mHighestPixel > best_top)
		TheFontInfo.mHighestPixel = best_top;
	if (TheFontInfo.mLowestPixel < best_bottom)
		TheFontInfo.mLowestPixel = best_bottom;
	
	TheFontInfo.mNumChars++;
	
	return line;
}



// returns number of occupied line, or -1 if none found
int ScanForNextOccupiedLine(int line)
{
	int top, bottom;
	while(1)
	{
		if (ScanVerticalLine(line, &top, &bottom))
			return line;
		line++;
		if (line == ThePNGInfo.mWidth)
			return -1;
	};
	return -1;
}


// returns 'false' if line is totally transparent
bool ScanVerticalLine(int lineNum, int *pTopVal, int *pBottomVal)
{
	int w = ThePNGInfo.mWidth;
	int h = ThePNGInfo.mHeight;
	Uint8 trans_index = ThePNGInfo.mTransIndex;
	Uint8 *p_data = (Uint8 *) ThePNGInfo.mpTexelData + lineNum;

	bool empty_line = true;

	for (int i = 0; i < h; i++)
	{
		if (ThePNGInfo.mpTransTab[*p_data] == 0)
		{
			// not a transparent pixel
			if (empty_line)
			{
				*pTopVal = i;
				empty_line = false;
			}
			*pBottomVal = i;
		}
		p_data += w;
	}

	return !empty_line;
}




void BuildOutputPalette()
{
	PNGInfo *pInfo = &ThePNGInfo;
	
	// copy raw palette to output one
	pInfo->mpOutPaletteData = new RGBA[pInfo->mNumPaletteEntries];	
	memcpy(pInfo->mpOutPaletteData, pInfo->mpRawPaletteData, pInfo->mNumPaletteEntries * 4);
	
	bool trans_color_found = false;
	
	for( int i = 0; i < pInfo->mNumPaletteEntries; i++ )
	{
		pInfo->mpTransTab[i] = 0;

		RGBA &entry = pInfo->mpOutPaletteData[i];
		
		if (ThePNGInfo.mUseGrayScaleTransKeying)
		{
			// make RGB values = (255, 255, 255), alpha = highest of original RGB
			
			if (entry.a > 0)
			{
				entry.a = (entry.r > entry.g) ? entry.r : entry.g;
				if (entry.a < entry.b) entry.a = entry.b;
				entry.a = ((int) entry.a + 1) / 2;
			}
			else if (!trans_color_found)
			{
				// entry.a == 0
				pInfo->mTransIndex = i;
				trans_color_found = true;
			}
			entry.r = 255;
			entry.g = 255;
			entry.b = 255;
		}
		// else, not using gray scale keying -- use RGB values as is

		if (entry.a == 0)
		{
			// fully transparent color
			pInfo->mpTransTab[i] = 1;
			if (!trans_color_found)
			{
				pInfo->mTransIndex = i;
				trans_color_found = true;
			}
		}
	} // end for

	Assert(trans_color_found, "must have at least one fully transparent palette entry");	
}




enum
{
	v32_BIT,
	v24_BIT,
	v16_BIT,
	v8_BIT,
	v4_BIT,		
	v8_BIT_GRAY,
	v4_BIT_GRAY
};

bool load_png( char* path, PNGInfo *pInfo)
{
	png_structp png_ptr;
	png_infop info_ptr;

	if( read_png( path, &png_ptr, &info_ptr ))
	{	
		bool has_trans_chunk;
		int pixel_data_size, palette_data_size;
		char* p_dst;
		int i;

		has_trans_chunk = false;
		// Check if this image has a valid transparency chunk
		if( png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS ))
		{
			has_trans_chunk = true;
		}

		pInfo->mWidth = info_ptr->width;
		pInfo->mHeight = info_ptr->height;

		if ( info_ptr->color_type == PNG_COLOR_TYPE_PALETTE)
		{			
			if( has_trans_chunk )
			{
				pInfo->mPaletteFormat = v32_BIT;					
				pInfo->mPaletteBpp = 32;
			}
			else
			{
				pInfo->mPaletteFormat = v24_BIT;
				pInfo->mPaletteBpp = 24;
			}
			
			if( info_ptr->bit_depth == 8 )
			{
				pInfo->mNumPaletteEntries = 256;
				palette_data_size = pInfo->mNumPaletteEntries * ( pInfo->mPaletteBpp >> 3 );
				pixel_data_size = pInfo->mWidth * pInfo->mHeight;
				pInfo->mpTexelData = new char[pixel_data_size];
				pInfo->mpRawPaletteData = new char[palette_data_size];					
				pInfo->mpTransTab = new char[palette_data_size];					
				pInfo->mBpp = 8;
			}
			else
			{
				Assert(0, "not 8 bits per pixel");
				return false;
			}

			// First copy texel data
			p_dst = pInfo->mpTexelData;
			for( i = 0; i < pInfo->mHeight; i++ )
			{
				memcpy( p_dst, info_ptr->row_pointers[i], info_ptr->rowbytes );
				p_dst += info_ptr->rowbytes;
			}

			pInfo->mNumUsedPaletteEntries = (info_ptr->num_palette > png_ptr->num_trans) ? info_ptr->num_palette : png_ptr->num_trans; 
			
			// Now copy palette data
			p_dst = pInfo->mpRawPaletteData;
			for( i = 0; i < pInfo->mNumPaletteEntries; i++ )
			{
				if( i < info_ptr->num_palette )
				{
					*p_dst++ = info_ptr->palette[i].red;
					*p_dst++ = info_ptr->palette[i].green;
					*p_dst++ = info_ptr->palette[i].blue;
				}
				else
				{
					*p_dst++ = 0;
					*p_dst++ = 0;
					*p_dst++ = 0;
				}
				
				if( has_trans_chunk )
				{
					// Note: for NS games, alpha=128 means fully opaque

					if( i < png_ptr->num_trans )
					{
						*p_dst++ = (png_ptr->trans[i] + 1) / 2;
					}
					else
					{
						*p_dst++ = (char) 128;
					}
				}			
				else
					Assert(0, "no transparency chunk found in PNG");
			} // end for
		}
		else
		{
			Assert(0, "unrecognized color type -- not palettized");
		}
		
		png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
		return true;
	}
	
	return false;
}




void PNGAPI nx_png_warning(png_structp png_ptr, png_const_charp message)
{
	printf( "PNG Warning: %s\n", message );
}




/* Read a PNG file.  You may want to return an error code if the read
 * fails (depending upon the failure).  There are two "prototypes" given
 * here - one where we are given the filename, and we need to open the
 * file, and the other where we are given an open file (possibly with
 * some or all of the magic bytes read - see comments above).
 */
bool read_png( char *file_name, png_structp* in_png_ptr, png_infop* in_info_ptr )  /* We need to open the file */
{
   png_structp png_ptr;
   png_infop info_ptr;
   unsigned int sig_read = 0;      
   FILE *fp;

   if(( fp = fopen( file_name, "rb" )) == NULL )
   {
      return false;
   }

   /* Create and initialize the png_struct with the desired error handler
    * functions.  If you want to use the default stderr and longjump method,
    * you can supply NULL for the last three parameters.  We also supply the
    * the compiler header file version, so that we know if the application
    * was compiled with a compatible version of the library.  REQUIRED
    */
   *in_png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING, NULL, NULL, nx_png_warning );
      //(png_voidp) user_error_ptr, user_error_fn, user_warning_fn);
   png_ptr = *in_png_ptr;

   if (png_ptr == NULL)
   {
      fclose(fp);
      return false;
   }

   /* Allocate/initialize the memory for image information.  REQUIRED. */
   *in_info_ptr = png_create_info_struct(png_ptr);
   info_ptr = *in_info_ptr;
   if (info_ptr == NULL)
   {
      fclose(fp);
      png_destroy_read_struct(&png_ptr, (png_infopp)NULL, (png_infopp)NULL);
      return false;
   }

   /* Set error handling if you are using the setjmp/longjmp method (this is
    * the normal method of doing things with libpng).  REQUIRED unless you
    * set up your own error handlers in the png_create_read_struct() earlier.
    */

   if (setjmp(png_jmpbuf(png_ptr)))
   {
      /* Free all of the memory associated with the png_ptr and info_ptr */
      png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);
      fclose(fp);
      /* If we get here, we had a problem reading the file */
      return false;
   }

   /* One of the following I/O initialization methods is REQUIRED */
#ifdef streams /* PNG file I/O method 1 */
   /* Set up the input control if you are using standard C streams */
   png_init_io(png_ptr, fp);

#else no_streams /* PNG file I/O method 2 */
   /* If you are using replacement read functions, instead of calling
    * png_init_io() here you would call:
    */
   png_set_read_fn(png_ptr, (void *)user_io_ptr, user_read_fn);
   /* where user_io_ptr is a structure you want available to the callbacks */
#endif no_streams /* Use only one I/O method! */

   /* If we have already read some of the signature */
   png_set_sig_bytes(png_ptr, sig_read);

#ifdef hilevel
   /*
    * If you have enough memory to read in the entire image at once,
    * and you need to specify only transforms that can be controlled
    * with one of the PNG_TRANSFORM_* bits (this presently excludes
    * dithering, filling, setting background, and doing gamma
    * adjustment), then you can read the entire image (including
    * pixels) into the info structure with this call:
    */
      
   png_read_png(png_ptr, info_ptr, 0, NULL);
#else
   /* OK, you're doing it the hard way, with the lower-level functions */

   /* The call to png_read_info() gives us all of the information from the
    * PNG file before the first IDAT (image data chunk).  REQUIRED
    */
   png_read_info(png_ptr, info_ptr);

   png_get_IHDR(png_ptr, info_ptr, &width, &height, &bit_depth, &color_type,
       &interlace_type, NULL, NULL);

/**** Set up the data transformations you want.  Note that these are all
 **** optional.  Only call them if you want/need them.  Many of the
 **** transformations only work on specific types of images, and many
 **** are mutually exclusive.
 ****/

   /* tell libpng to strip 16 bit/color files down to 8 bits/color */
   png_set_strip_16(png_ptr);

   /* Strip alpha bytes from the input data without combining with the
    * background (not recommended).
    */
   png_set_strip_alpha(png_ptr);

   /* Extract multiple pixels with bit depths of 1, 2, and 4 from a single
    * byte into separate bytes (useful for paletted and grayscale images).
    */
   png_set_packing(png_ptr);

   /* Change the order of packed pixels to least significant bit first
    * (not useful if you are using png_set_packing). */
   png_set_packswap(png_ptr);

   /* Expand paletted colors into true RGB triplets */
   if (color_type == PNG_COLOR_TYPE_PALETTE)
      png_set_palette_rgb(png_ptr);

   /* Expand grayscale images to the full 8 bits from 1, 2, or 4 bits/pixel */
   if (color_type == PNG_COLOR_TYPE_GRAY && bit_depth < 8)
      png_set_gray_1_2_4_to_8(png_ptr);

   /* Expand paletted or RGB images with transparency to full alpha channels
    * so the data will be available as RGBA quartets.
    */
   if (png_get_valid(png_ptr, info_ptr, PNG_INFO_tRNS))
      png_set_tRNS_to_alpha(png_ptr);

   /* Set the background color to draw transparent and alpha images over.
    * It is possible to set the red, green, and blue components directly
    * for paletted images instead of supplying a palette index.  Note that
    * even if the PNG file supplies a background, you are not required to
    * use it - you should use the (solid) application background if it has one.
    */

   png_color_16 my_background, *image_background;

   if (png_get_bKGD(png_ptr, info_ptr, &image_background))
      png_set_background(png_ptr, image_background,
                         PNG_BACKGROUND_GAMMA_FILE, 1, 1.0);
   else
      png_set_background(png_ptr, &my_background,
                         PNG_BACKGROUND_GAMMA_SCREEN, 0, 1.0);

   /* Some suggestions as to how to get a screen gamma value */

   /* Note that screen gamma is the display_exponent, which includes
    * the CRT_exponent and any correction for viewing conditions */
   if (/* We have a user-defined screen gamma value */)
   {
      screen_gamma = user-defined screen_gamma;
   }
   /* This is one way that applications share the same screen gamma value */
   else if ((gamma_str = getenv("SCREEN_GAMMA")) != NULL)
   {
      screen_gamma = atof(gamma_str);
   }
   /* If we don't have another value */
   else
   {
      screen_gamma = 2.2;  /* A good guess for a PC monitors in a dimly
                              lit room */
      screen_gamma = 1.7 or 1.0;  /* A good guess for Mac systems */
   }

   /* Tell libpng to handle the gamma conversion for you.  The final call
    * is a good guess for PC generated images, but it should be configurable
    * by the user at run time by the user.  It is strongly suggested that
    * your application support gamma correction.
    */

   int intent;

   if (png_get_sRGB(png_ptr, info_ptr, &intent))
      png_set_gamma(png_ptr, screen_gamma, 0.45455);
   else
   {
      double image_gamma;
      if (png_get_gAMA(png_ptr, info_ptr, &image_gamma))
         png_set_gamma(png_ptr, screen_gamma, image_gamma);
      else
         png_set_gamma(png_ptr, screen_gamma, 0.45455);
   }

   /* Dither RGB files down to 8 bit palette or reduce palettes
    * to the number of colors available on your screen.
    */
   if (color_type & PNG_COLOR_MASK_COLOR)
   {
      int num_palette;
      png_colorp palette;

      /* This reduces the image to the application supplied palette */
      if (/* we have our own palette */)
      {
         /* An array of colors to which the image should be dithered */
         png_color std_color_cube[MAX_SCREEN_COLORS];

         png_set_dither(png_ptr, std_color_cube, MAX_SCREEN_COLORS,
            MAX_SCREEN_COLORS, NULL, 0);
      }
      /* This reduces the image to the palette supplied in the file */
      else if (png_get_PLTE(png_ptr, info_ptr, &palette, &num_palette))
      {
         png_uint_16p histogram;

         png_get_hIST(png_ptr, info_ptr, &histogram);

         png_set_dither(png_ptr, palette, num_palette,
                        max_screen_colors, histogram, 0);
      }
   }

   /* invert monochrome files to have 0 as white and 1 as black */
   png_set_invert_mono(png_ptr);

   /* If you want to shift the pixel values from the range [0,255] or
    * [0,65535] to the original [0,7] or [0,31], or whatever range the
    * colors were originally in:
    */
   if (png_get_valid(png_ptr, info_ptr, PNG_INFO_sBIT))
   {
      png_color_8p sig_bit;

      png_get_sBIT(png_ptr, info_ptr, &sig_bit);
      png_set_shift(png_ptr, sig_bit);
   }

   /* flip the RGB pixels to BGR (or RGBA to BGRA) */
   if (color_type & PNG_COLOR_MASK_COLOR)
      png_set_bgr(png_ptr);

   /* swap the RGBA or GA data to ARGB or AG (or BGRA to ABGR) */
   png_set_swap_alpha(png_ptr);

   /* swap bytes of 16 bit files to least significant byte first */
   png_set_swap(png_ptr);

   /* Add filler (or alpha) byte (before/after each RGB triplet) */
   png_set_filler(png_ptr, 0xff, PNG_FILLER_AFTER);

   /* Turn on interlace handling.  REQUIRED if you are not using
    * png_read_image().  To see how to handle interlacing passes,
    * see the png_read_row() method below:
    */
   number_passes = png_set_interlace_handling(png_ptr);

   /* Optional call to gamma correct and add the background to the palette
    * and update info structure.  REQUIRED if you are expecting libpng to
    * update the palette for you (ie you selected such a transform above).
    */
   png_read_update_info(png_ptr, info_ptr);

   /* Allocate the memory to hold the image using the fields of info_ptr. */

   /* The easiest way to read the image: */
   png_bytep row_pointers[height];

   for (row = 0; row < height; row++)
   {
      row_pointers[row] = png_malloc(png_ptr, png_get_rowbytes(png_ptr,
         info_ptr));
   }

   /* Now it's time to read the image.  One of these methods is REQUIRED */
#ifdef entire /* Read the entire image in one go */
   png_read_image(png_ptr, row_pointers);

#else no_entire /* Read the image one or more scanlines at a time */
   /* The other way to read images - deal with interlacing: */

   for (pass = 0; pass < number_passes; pass++)
   {
#ifdef single /* Read the image a single row at a time */
      for (y = 0; y < height; y++)
      {
         png_read_rows(png_ptr, &row_pointers[y], NULL, 1);
      }

#else no_single /* Read the image several rows at a time */
      for (y = 0; y < height; y += number_of_rows)
      {
#ifdef sparkle /* Read the image using the "sparkle" effect. */
         png_read_rows(png_ptr, &row_pointers[y], NULL, number_of_rows);
#else no_sparkle /* Read the image using the "rectangle" effect */
         png_read_rows(png_ptr, NULL, &row_pointers[y], number_of_rows);
#endif no_sparkle /* use only one of these two methods */
      }

      /* if you want to display the image after every pass, do
         so here */
#endif no_single /* use only one of these two methods */
   }
#endif no_entire /* use only one of these two methods */

   /* read rest of file, and get additional chunks in info_ptr - REQUIRED */
   png_read_end(png_ptr, info_ptr);
#endif hilevel

   /* At this point you have read the entire image */

   /* clean up after the read, and free any memory allocated - REQUIRED */
   //png_destroy_read_struct(&png_ptr, &info_ptr, (png_infopp)NULL);

   /* close the file */
   fclose(fp);

   /* that's it */
   return true;
}




void Assert(int condition, char *pMessage, ...)
{
	char pad[128];
	
	if (condition) return;
	
	va_list args;
	
	va_start( args, pMessage );
	vsprintf( pad, pMessage, args);
	va_end( args );

	printf("ERROR: ");
	printf( pad );
	printf("\n");
	exit(1);
}



