// localtext.cpp
// KSH 5 Sep 2001

// A utility for extracting all the text from all the script files so that it can be
// translated.
// It can also replace the english text with the translated text.


#include "stdafx.h"
#include <stdarg.h>
#include "stdio.h"
#include "string.h"
#include "stdlib.h"
#include "conio.h"
#include "process.h"
#include "time.h"
#include "resource.h"
#include "sys/types.h"
#include "sys/stat.h"
#include "\skate5\code\gel\scripting\tokens.h"

using namespace Script;
#include <windows.h>
#include <windowsx.h>
#include <commctrl.h>

typedef unsigned long uint32;
typedef unsigned char uint8;

// This is so that skiptoken.cpp can be #included.
#define Dbg_MsgAssert( _c, _params )							\
																\
if( !( _c ))													\
{																\
	MsgAssert_PrintfIntoBuffer _params;							\
	Assert( 0, pMsgAssert_Buffer );								\
}

#define MSGASSERT_BUFFER_SIZE 1000
static char pMsgAssert_Buffer[MSGASSERT_BUFFER_SIZE];
void MsgAssert_PrintfIntoBuffer(const char* p_text, ...)
{
	pMsgAssert_Buffer[0]=0;
	if (p_text)
	{
		va_list args;
		va_start( args, p_text );

		sprintf( pMsgAssert_Buffer, p_text, args);
		
		va_end( args );
	}
}

void Assert(int Condition, char *pMessage);

#include <c:/skate5/code/gel/scripting/skiptoken.cpp>

// The q file format is used to store the text & corresponding translations, because that
// way qcomp can be used to get it into a format that is easier to parse. (The qb format)
// Also it means comments are supported without any extra parsing code needed here.
// localtext.q is not stored in the data\scripts directory however, otherwise it would end
// up getting parsed for text too.
#define LOCALTEXTDOTQ "localtext.q"
#define LOCALTEXTDOTQB "localtext.qb"

uint8 *gpLocalTextToken=NULL;

#define MAX_QB_FILES 500
#define MAX_QB_CHARS 100
char ppQBNames[MAX_QB_FILES][MAX_QB_CHARS+1];
int NumQBFiles=0;

// Checksums of 'English', 'French', etc.
// Using checksums rather than an enum because they'll be stored as checksums in localtext.qb,
// so might as well represent them the same way here. Saves a few switch statements.
#define ENGLISH 0xd37cfdff
#define BRITISH 0x4852c080
#define FRENCH 0x508a31a1
#define GERMAN 0x5dcbbf5e
#define SPANISH 0xcbe70acb
#define ITALIAN 0xa8469630



//#define FIND_ME 0x12c7cd6b

// -------------------- Checksum stuff ------------------------

unsigned long CRC_TABLE[256] = // CRC polynomial 0xedb88320
{
      0x00000000, 0x77073096, 0xee0e612c, 0x990951ba,
      0x076dc419, 0x706af48f, 0xe963a535, 0x9e6495a3,
      0x0edb8832, 0x79dcb8a4, 0xe0d5e91e, 0x97d2d988,
      0x09b64c2b, 0x7eb17cbd, 0xe7b82d07, 0x90bf1d91,
      0x1db71064, 0x6ab020f2, 0xf3b97148, 0x84be41de,
      0x1adad47d, 0x6ddde4eb, 0xf4d4b551, 0x83d385c7,
      0x136c9856, 0x646ba8c0, 0xfd62f97a, 0x8a65c9ec,
      0x14015c4f, 0x63066cd9, 0xfa0f3d63, 0x8d080df5,
      0x3b6e20c8, 0x4c69105e, 0xd56041e4, 0xa2677172,
      0x3c03e4d1, 0x4b04d447, 0xd20d85fd, 0xa50ab56b,
      0x35b5a8fa, 0x42b2986c, 0xdbbbc9d6, 0xacbcf940,
      0x32d86ce3, 0x45df5c75, 0xdcd60dcf, 0xabd13d59,
      0x26d930ac, 0x51de003a, 0xc8d75180, 0xbfd06116,
      0x21b4f4b5, 0x56b3c423, 0xcfba9599, 0xb8bda50f,
      0x2802b89e, 0x5f058808, 0xc60cd9b2, 0xb10be924,
      0x2f6f7c87, 0x58684c11, 0xc1611dab, 0xb6662d3d,
      0x76dc4190, 0x01db7106, 0x98d220bc, 0xefd5102a,
      0x71b18589, 0x06b6b51f, 0x9fbfe4a5, 0xe8b8d433,
      0x7807c9a2, 0x0f00f934, 0x9609a88e, 0xe10e9818,
      0x7f6a0dbb, 0x086d3d2d, 0x91646c97, 0xe6635c01,
      0x6b6b51f4, 0x1c6c6162, 0x856530d8, 0xf262004e,
      0x6c0695ed, 0x1b01a57b, 0x8208f4c1, 0xf50fc457,
      0x65b0d9c6, 0x12b7e950, 0x8bbeb8ea, 0xfcb9887c,
      0x62dd1ddf, 0x15da2d49, 0x8cd37cf3, 0xfbd44c65,
      0x4db26158, 0x3ab551ce, 0xa3bc0074, 0xd4bb30e2,
      0x4adfa541, 0x3dd895d7, 0xa4d1c46d, 0xd3d6f4fb,
      0x4369e96a, 0x346ed9fc, 0xad678846, 0xda60b8d0,
      0x44042d73, 0x33031de5, 0xaa0a4c5f, 0xdd0d7cc9,
      0x5005713c, 0x270241aa, 0xbe0b1010, 0xc90c2086,
      0x5768b525, 0x206f85b3, 0xb966d409, 0xce61e49f,
      0x5edef90e, 0x29d9c998, 0xb0d09822, 0xc7d7a8b4,
      0x59b33d17, 0x2eb40d81, 0xb7bd5c3b, 0xc0ba6cad,
      0xedb88320, 0x9abfb3b6, 0x03b6e20c, 0x74b1d29a,
      0xead54739, 0x9dd277af, 0x04db2615, 0x73dc1683,
      0xe3630b12, 0x94643b84, 0x0d6d6a3e, 0x7a6a5aa8,
      0xe40ecf0b, 0x9309ff9d, 0x0a00ae27, 0x7d079eb1,
      0xf00f9344, 0x8708a3d2, 0x1e01f268, 0x6906c2fe,
      0xf762575d, 0x806567cb, 0x196c3671, 0x6e6b06e7,
      0xfed41b76, 0x89d32be0, 0x10da7a5a, 0x67dd4acc,
      0xf9b9df6f, 0x8ebeeff9, 0x17b7be43, 0x60b08ed5,
      0xd6d6a3e8, 0xa1d1937e, 0x38d8c2c4, 0x4fdff252,
      0xd1bb67f1, 0xa6bc5767, 0x3fb506dd, 0x48b2364b,
      0xd80d2bda, 0xaf0a1b4c, 0x36034af6, 0x41047a60,
      0xdf60efc3, 0xa867df55, 0x316e8eef, 0x4669be79,
      0xcb61b38c, 0xbc66831a, 0x256fd2a0, 0x5268e236,
      0xcc0c7795, 0xbb0b4703, 0x220216b9, 0x5505262f,
      0xc5ba3bbe, 0xb2bd0b28, 0x2bb45a92, 0x5cb36a04,
      0xc2d7ffa7, 0xb5d0cf31, 0x2cd99e8b, 0x5bdeae1d,
      0x9b64c2b0, 0xec63f226, 0x756aa39c, 0x026d930a,
      0x9c0906a9, 0xeb0e363f, 0x72076785, 0x05005713,
      0x95bf4a82, 0xe2b87a14, 0x7bb12bae, 0x0cb61b38,
      0x92d28e9b, 0xe5d5be0d, 0x7cdcefb7, 0x0bdbdf21,
      0x86d3d2d4, 0xf1d4e242, 0x68ddb3f8, 0x1fda836e,
      0x81be16cd, 0xf6b9265b, 0x6fb077e1, 0x18b74777,
      0x88085ae6, 0xff0f6a70, 0x66063bca, 0x11010b5c,
      0x8f659eff, 0xf862ae69, 0x616bffd3, 0x166ccf45,
      0xa00ae278, 0xd70dd2ee, 0x4e048354, 0x3903b3c2,
      0xa7672661, 0xd06016f7, 0x4969474d, 0x3e6e77db,
      0xaed16a4a, 0xd9d65adc, 0x40df0b66, 0x37d83bf0,
      0xa9bcae53, 0xdebb9ec5, 0x47b2cf7f, 0x30b5ffe9,
      0xbdbdf21c, 0xcabac28a, 0x53b39330, 0x24b4a3a6,
      0xbad03605, 0xcdd70693, 0x54de5729, 0x23d967bf,
      0xb3667a2e, 0xc4614ab8, 0x5d681b02, 0x2a6f2b94,
      0xb40bbe37, 0xc30c8ea1, 0x5a05df1b, 0x2d02ef8d
};

uint32 GenerateChecksum(const char *pString)
{
	// Initializing the CRC to all one bits avoids failure of detection
	// should entire data stream get cyclically bit-shifted by one position.
	// The calculation of the probability of this happening is left as
	// an exercise for the reader.
	unsigned long rc = 0xffffffff;
	int Len=strlen(pString);
	for (int i=0; i<Len; i++)
	{
		char ch=pString[i];
        // Convert to lower case.
        if (ch>='A' && ch<='Z') ch='a'+ch-'A';
		rc = CRC_TABLE[(rc^ch) & 0xff] ^ ((rc>>8) & 0x00ffffff);
	}
	Assert(rc,"Checksum is zero !!!");
	
	// Add the name and checksum to the hash table for fast lookup.
	//AddChecksum(pString,rc);	

	return rc;
}

const char *GetLanguageName(uint32 Language)
{
	switch (Language)
	{
		case ENGLISH: return "English"; break;
		case FRENCH:  return "French"; break;
		case BRITISH: return "British"; break;
		case SPANISH: return "Spanish"; break;
		case ITALIAN: return "Italian"; break;
		case GERMAN:  return "German"; break;
		default: Assert(0,"Bad Language sent to GetLanguageName"); break;
	}
	return "";
}

#define MAX_CHARS_PER_STRING 500

class CTranslation
{
public:
	CTranslation *mpNext;
	CTranslation *mpPrevious;

	char mpEnglish[MAX_CHARS_PER_STRING+1];
	char mpFrench[MAX_CHARS_PER_STRING+1];
	char mpBritish[MAX_CHARS_PER_STRING+1];
	char mpSpanish[MAX_CHARS_PER_STRING+1];
	char mpGerman[MAX_CHARS_PER_STRING+1];
	char mpItalian[MAX_CHARS_PER_STRING+1];

	// True if the source text may not need translation, ie it's probably a filename or something.
	bool mCrap;

	// True if this text does occur in some q file.
	// This could be false in the case of some text already in localtext.q not being in any q file
	// any more.
	bool mOccursInSomeQFile;

	// How many times this piece of text is used in each of the qb files.
	int mNumTimesOccursInQB[MAX_QB_FILES];

	// The checksum of the section in localtext.q where this was defined.
	uint32 mSectionChecksum;

	CTranslation();
	~CTranslation();

	const char *GetText(uint32 Language);
	void SetText(const char *pText, uint32 Language);

	bool CheckPercents();
};

CTranslation *pTranslations=NULL;

CTranslation::CTranslation()
{
	mpEnglish[0]=0;
	mpFrench[0]=0;
	mpBritish[0]=0;
	mpSpanish[0]=0;
	mpGerman[0]=0;
	mpItalian[0]=0;
	mCrap=false;
	mOccursInSomeQFile=false;
	mSectionChecksum=0;

	for (int i=0; i<MAX_QB_FILES; ++i)
	{
		mNumTimesOccursInQB[i]=0;
	}

	mpNext=pTranslations;
	pTranslations=this;
	mpPrevious=NULL;
	if (mpNext)
	{
		mpNext->mpPrevious=this;
	}
}

CTranslation::~CTranslation()
{
	if (mpNext)
	{
		mpNext->mpPrevious=mpPrevious;
	}
	if (mpPrevious)
	{
		mpPrevious->mpNext=mpNext;
	}
	else
	{
		pTranslations=mpNext;
	}
}

int CountPercents(const char *pString)
{
	int NumPercents=0;
	while (*pString)
	{
		if (*pString++=='%')
		{
			++NumPercents;
		}
	}
	return NumPercents;
}

bool CheckNumPercents(const char *pA, const char *pB,int NumPercentsInA)
{
	if (pB[0] && CountPercents(pB)!=NumPercentsInA)
	{
		printf("Mismatch in the number of percent signs:\n\"%s\"\n\"%s\"\n",pA,pB);
		return false;
	}
	return true;
}

// Checks to make sure that each of the strings contains the same number of percent signs.
// This is because if a translation of some text that is used in a printf is missing a percent,
// a crash could occur in the game.
// Returns true if everything OK.
bool CTranslation::CheckPercents()
{
	bool Ok=true;
	// Count the percents in the English.
	int NumPercentsInEnglish=CountPercents(mpEnglish);
	if (!CheckNumPercents(mpEnglish,mpFrench,NumPercentsInEnglish))
	{
		Ok=false;
	}
	if (!CheckNumPercents(mpEnglish,mpGerman,NumPercentsInEnglish))
	{
		Ok=false;
	}
	if (!CheckNumPercents(mpEnglish,mpBritish,NumPercentsInEnglish))
	{
		Ok=false;
	}
	if (!CheckNumPercents(mpEnglish,mpSpanish,NumPercentsInEnglish))
	{
		Ok=false;
	}
	if (!CheckNumPercents(mpEnglish,mpItalian,NumPercentsInEnglish))
	{
		Ok=false;
	}

	return Ok;
}

void CheckPercents()
{
	bool Ok=true;
	CTranslation *pTrans=pTranslations;
	while (pTrans)
	{
		// Don't assert straight away, so that all the errors get listed at once.
		if (!pTrans->CheckPercents())
		{
			Ok=false;
		}
		pTrans=pTrans->mpNext;
	}
	Assert(Ok,"Mismatch in the number of percent signs");
}

CTranslation *GetLastTransInList()
{
	CTranslation *pTrans=pTranslations;
	if (pTrans)
	{
		while (pTrans->mpNext)
		{
			pTrans=pTrans->mpNext;
		}
	}
	return pTrans;
}

const char *CTranslation::GetText(uint32 Language)
{
	switch (Language)
	{
		case ENGLISH: return mpEnglish; break;
		case FRENCH:  return mpFrench; break;
		case BRITISH: return mpBritish; break;
		case SPANISH: return mpSpanish; break;
		case GERMAN:  return mpGerman; break;
		case ITALIAN: return mpItalian; break;
		default: Assert(0,"Bad Language sent to GetText"); break;
	}
	return NULL;
}

char *Crap_Exception[] = {
	"Buildings/Trailers",
	"Pipes/Tunnels",
	"Rock/Other",
	"LOGIN/PROFILE",
	"Login/Create",
	"Signs\\ndestroyed",
	"Donuts\\nCollected",
	"Stickers\\nRemoved",
	"Leis\\nCollected",
	"Guards\\nDistracted",
	"Trash\\nCollected",
	"Skater/Story",
	"Rooms\\nRecovered",
	"On/Off",
	"STORY/SKATER",
};

// Crapness detector.
// Returns true if the text is something that probably won't need translating,
// such as a filename or animation name.
bool MaybeCrap(const char *pText)
{
	// The first test of crapness is whether the string has no spaces in it.
	if (strchr(pText,' '))
	{
		// It contains spaces, so probably will need translating.
		return false;
	}

	if (strchr(pText,'='))
	{
		// It contains '=', so probably will need translating.
		return false;
	}

	// There are no spaces in the string ...
	if ( strchr(pText,'\\') || strchr(pText,'/') || strchr(pText,'_'))
	{
		//temp hack to get quick results for the translators
		for( int j = 0; j < sizeof(Crap_Exception)/sizeof(char*); ++j)
		{
			if(!strcmp(pText,Crap_Exception[j]))
				return false;
		}

		// If the string contains a backslash, it is probably a filename.
		// If it contains an underscore, it is probably some sort of id.
		return true;
	}

	int Len=strlen(pText);
	char ch=pText[Len-1];
	if (ch>='0' && ch<='9')
	{
		// If it ends in a number, it is probably some sort of id,
		// such as the particle system id's.
		return true;
	}

	if (Len>=4)
	{
		if (pText[Len-4]=='.')
		{
			// It is probably a filename that got past the
			// earlier backslash test, eg "foo.png"
			return true;
		}
		if (stricmp(pText+Len-4,"Anim")==0)
		{
			// A special case, to catch the camera anim names which all
			// end in Anim. I can't think of any other words ending in
			// anim that would need translating, so this is probably safe.
			return true;
		}
	}

	return false;
}

void CTranslation::SetText(const char *pText, uint32 Language)
{
	Assert(pText!=NULL,"NULL pText sent to SetText");

	if (strlen(pText)>MAX_CHARS_PER_STRING)
	{
		printf("String too long:\n\"%s\"\n",pText);
		Assert(0,"String too long");
	}

	switch (Language)
	{
		case ENGLISH: strcpy(mpEnglish,pText);break;
		case FRENCH:  strcpy(mpFrench,pText); break;
		case BRITISH: strcpy(mpBritish,pText); break;
		case SPANISH: strcpy(mpSpanish,pText);break;
		case GERMAN:  strcpy(mpGerman,pText);break;
		case ITALIAN: strcpy(mpItalian,pText);break;
		default: Assert(0,"Bad Language sent to SetText"); break;
	}
}

CTranslation *FindTranslation(const char *pText, uint32 Language)
{
	CTranslation *pTrans=pTranslations;
	while (pTrans)
	{
		if (strcmp(pTrans->GetText(Language),pText)==0)
		{
			return pTrans;
		}

		pTrans=pTrans->mpNext;
	}

	return NULL;
}

void DeleteTranslations()
{
	// Don't call Assert anywhere in here, otherwise it'll infinitely recurse,
	// because Assert calls CleanUp which calls this.

	CTranslation *pTrans=pTranslations;
	while (pTrans)
	{
		CTranslation *pNext=pTrans->mpNext;
		delete pTrans;
		pTrans=pNext;
	}
	Assert(pTranslations==NULL,"Eh? pTranslations not NULL at the end of DeleteTranslations ?");
}

void CleanUp()
{
	DeleteTranslations();
}

uint32 GetLineNumber(uint8 *p_token)
{
    while (true)
    {
        switch (*p_token)
		{
		case ESCRIPTTOKEN_ENDOFLINE:
			return 0;
			break;
			
		case ESCRIPTTOKEN_ENDOFLINENUMBER:
			++p_token;
			return p_token[0]+(p_token[1]<<8)+(p_token[2]<<16)+(p_token[3]<<24);
            break;
		
		case ESCRIPTTOKEN_ENDOFFILE:
			return -1;
			break;
		
		default:
			p_token=SkipToken(p_token);
			break;		
		}	
    }

	return 0;
}

void Assert(int Condition, char *pMessage)
{
	if (!Condition)
	{
		printf("LocalText assertion failed:\n%s",pMessage);
		if (gpLocalTextToken)
		{
			printf("\nlocaltext.q line %d\n",GetLineNumber(gpLocalTextToken));
		}
		CleanUp();
		exit(1);
	}
}


uint8 *SkipEndOfLines(uint8 *pToken)
{
	while (*pToken==ESCRIPTTOKEN_ENDOFLINE || *pToken==ESCRIPTTOKEN_ENDOFLINENUMBER)
	{
		pToken=SkipToken(pToken);
	}
	return pToken;
}

uint8 *SkipTokenAndEndOfLines(uint8 *pToken)
{
	pToken=SkipToken(pToken);
	pToken=SkipEndOfLines(pToken);
	return pToken;
}

int FindLineNumber(uint8 *pToken)
{
	int Line=-1;
	while (!(*pToken==ESCRIPTTOKEN_ENDOFLINE || 
			 *pToken==ESCRIPTTOKEN_ENDOFLINENUMBER ||
			 *pToken==ESCRIPTTOKEN_ENDOFFILE))
	{
		pToken=SkipToken(pToken);
	}
	if (*pToken==ESCRIPTTOKEN_ENDOFLINENUMBER)
	{
		Line=*(uint32*)(pToken+1);
	}
	return Line;
}

void Extract(const char *pQBName, uint32 Language)
{
	printf("Extracting from %s\n",pQBName);

	// Search backwards through the qb name to find the name without the
	// path, and store it in the array of qb names.
	// Not storing the path too because it will make the comments in the
	// outputted localtext.q too long and hard to read.
	const char *pName=pQBName+strlen(pQBName)-1;
	while (pName>pQBName && *pName!='\\')
	{
		--pName;
	}
	if (*pName=='\\')
	{
		++pName;
	}
	Assert(strlen(pName)<=MAX_QB_CHARS,"QB name too long");

	Assert(NumQBFiles<MAX_QB_FILES,"Too many qb files");
	strcpy(ppQBNames[NumQBFiles++],pName);


	// Open the file and load it into memory
	FILE *ifp=fopen(pQBName,"rb");
	Assert(ifp!=NULL,"Could not open source file");
	fseek(ifp,0,SEEK_END);
	long SourceSize=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	uint8 *pQB=(uint8*)malloc(SourceSize);
	Assert(pQB!=NULL,"Could not allocate memory to hold input file");
	Assert(fread(pQB,SourceSize,1,ifp)==1,"fread failed");
	fclose(ifp);
	ifp=NULL;
	// The qb is now stored in pQB


	// Scan through it looking for strings ...
	uint8 *pToken=pQB;
	bool Ignore=false;
	int EndGapCheck=0;
	while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
	{
		if (*pToken==ESCRIPTTOKEN_NAME)
		{
			uint32 Ch=*(uint32*)(pToken+1);
			if (Ch==0x2de8c60e	||	// printf
				Ch==0xf87bcb53	||	// AddTextureToVram
				Ch==0x30986839		// RemoveTextureFromVram
				)
			{
				Ignore=true;
			}
		}

		// Keep a look out for any string parameters called text in EndGap calls,
		// and if found, do not include them in localtext.q
		if (*pToken==ESCRIPTTOKEN_ENDOFLINE || *pToken==ESCRIPTTOKEN_ENDOFLINENUMBER)
		{
			// Don't ignore the next string if an end-of-line is hit.
			// This is because printf or AddTextureToVram may not be followed by a string,
			// eg we could have AddTextureToVram <icon_texture>
			// and we don't want any string in the next command to be ignored.
			// It's better to include unnecessary strings than accidentally ignoring ones
			// that might need to be translated.
			Ignore=false;

			// End of line is taken to be the end of the EndGap command.
			// This is not always the case since parameters may be spread across lines using
			// curly brackets, but this will catch almost all cases.
			// It does not matter too much if some gap names do get into localtext.q,
			// they can always be removed manually.
			EndGapCheck=0;
		}
		bool GotGapName=false;
		switch (EndGapCheck)
		{
		case 0:
			if (*pToken==ESCRIPTTOKEN_NAME)
			{
				uint32 Ch=*(uint32*)(pToken+1);
				if (Ch==0xe5399fb2/*EndGap*/ || Ch==0x2877b61a/*IsLatestTrick*/ || Ch==0xe366436b/*GetNumberOfTrickOccurrences*/ || Ch==0x65b9ec39 /*StartGap*/)
				{
					// Found EndGap, so onto stage 1
					++EndGapCheck;
				}
			}
			break;	
		case 1:
			// Stage 1 is where we know we are in an EndGap parameter list, and we are
			// keeping a lookout for the name 'text'
			if (*pToken==ESCRIPTTOKEN_NAME)
			{
				uint32 Ch=*(uint32*)(pToken+1);
				if (Ch==0xc4745838/*Text*/ || Ch==0x3eafa520/*tricktext*/)
				{
					// Found the word 'text' in the EndGap command, so onto stage 2
					++EndGapCheck;
				}
			}
			break;
		case 2:
			if (*pToken==ESCRIPTTOKEN_EQUALS)
			{
				// The word 'text' is followed by an equals! Onto stage 3 ...
				++EndGapCheck;
			}
			else
			{
				// Oh dear, it is just a spurious parameter, so go back to stage 1 ...
				// (Note: This case should never really happen, but in theory it might)
				EndGapCheck=1;
			}
			break;
		case 3:
			if (*pToken==ESCRIPTTOKEN_STRING)
			{
				// Got a text=(some string) in an EndGap command, so we've definitely
				// found a gap name. So ignore it!
				GotGapName=true;
				EndGapCheck=0;
				//const char *pText=(const char *)pToken+5;
				//printf("Gap: %s\n",pText);
			}
			else
			{
				// Got a spurious parameter named 'text', but not of type string.
				// This should never happen, but in theory it might, so just go back to
				// stage 1.
				EndGapCheck=1;
			}
			break;
		default:
			Assert(0,"Bad EndGapCheck value");
			break;
		}

		if (*pToken==ESCRIPTTOKEN_STRING || *pToken==ESCRIPTTOKEN_LOCALSTRING)
		{
			if (Ignore)
			{
				Ignore=false;
				// Ignore any string that follows special calls such as printf
			}
			else if (GotGapName)
			{
				// Ignore gap names
			}
			else
			{
				const char *pText=(const char *)pToken+5;


				// Found a string!

				// See if we've seen it already ...
				CTranslation *pTrans=FindTranslation(pText,Language);
				if (!pTrans)
				{
					// Nope, haven't seen it before, so create a new translation entry.
					pTrans=new CTranslation;

					// Fill it in.
					pTrans->SetText(pText,Language);

					// Determine whether the text may be crap that does not require translation.
					pTrans->mCrap=MaybeCrap(pText);
				}

				if (pTrans)
				{
					// Increment the usage for this file.
					++pTrans->mNumTimesOccursInQB[NumQBFiles-1];

					pTrans->mOccursInSomeQFile=true;
				}
			}
		}

		pToken=SkipToken(pToken);
	}

	free(pQB);
}

void Extract(uint32 Language)
{
	printf("Extracting text from all qb files ...\n");

	char *pPath=getenv("PROJ_ROOT");
	Assert(pPath!=NULL,"The environment variable PROJ_ROOT is not set.");

	#define MAXDIRBUFCHARS 100
	char DirNameBuf[MAXDIRBUFCHARS+1];
	Assert(strlen(pPath)+strlen("\\bin\\win32\\qmake.txt")<=MAXDIRBUFCHARS,"Path too long.");
	strcpy(DirNameBuf,pPath);
	strcat(DirNameBuf,"\\bin\\win32\\qmake.txt");

	FILE *listifp=fopen(DirNameBuf,"r");
	Assert(listifp!=NULL,"Could not open qmake.txt.\n\nYou need to run qcompall.bat to generate qmake.txt");

    // Check each of the files listed.
    while (true)
    {
		#define MAXFILENAMECHARS 100
        // File name buffer.
        char pNameBuf[MAXFILENAMECHARS+1];
        pNameBuf[MAXFILENAMECHARS]=0;

        // Load each character into the buffer until end-of-line or end-of-file.
        int i=0;
        while (true)
        {
            Assert(i<=MAXFILENAMECHARS,"Too many chars in filename.");
			char ch=fgetc(listifp);

            // Check if end-of-file or end-of-line.
            if (ch==EOF || ch=='\n')
            {
                // Terminate the current string.
                pNameBuf[i]=0;
                break;
            }

			pNameBuf[i]=ch;
            ++i;
        }

        if (pNameBuf[0])
		{
			// Derive the .qb filename.
	        char pQBNameBuf[MAXFILENAMECHARS+1];
			strcpy(pQBNameBuf,pNameBuf);
			int Len=strlen(pQBNameBuf);
			Assert(Len>3,"Very short filename?");

			if (stricmp(&pQBNameBuf[Len-2],".q")==0)
				// Append a 'b' if it is a .q
				strcat(pQBNameBuf,"b");
			else if (stricmp(&pQBNameBuf[Len-3],".qn")==0)
				// Change to qb if it is a .qn
				pQBNameBuf[Len-1]='b';
			else
			{
				printf("Bad: '%s'\n",pQBNameBuf);
				Assert(0,"Filename does not have .q or .qn extension");
			}

			// Only call extract on it if it is not localtext.q, just in case localtext.q
			// got created somehow in one of the scripts directories, which would happen if
			// localtext.exe was run from one of those directories.
			if (stricmp(pQBNameBuf,LOCALTEXTDOTQB))
			{
				Extract(pQBNameBuf,Language);
			}
		}
        else
            break;
    }
	
	fclose (listifp);

	printf("Done.\n");
}

// Given a string, this will generate the string that will need to be written
// out to the q file such that when qcomp'd it will give the original string.
// For example, if the string contains quotes, then the string written out to the
// q file will need to have backslashes inserted before the quotes.
// Note that FixString will not insert the surrounding quotes, so the first char
// of the returned string will not be ".
const char *FixString(const char *p_string)
{
	static char sp_out[MAX_CHARS_PER_STRING+1];
	char *p_dest=sp_out;
	const char *p_source=p_string;
	while (*p_source)
	{
		if (*p_source=='"')
		{
			*p_dest++='\\';
			*p_dest++='"';
			++p_source;
		}
		else if (p_source[0]=='\\' && p_source[1]==0 && !(p_source > p_string && p_source[-1]=='\\'))
		{
			// If the source ends in a backslash, an extra backslash will need to
			// be inserted, otherwise the backslash will kill the terminating quote.
			*p_dest++='\\';
			*p_dest++='\\';
			++p_source;
		}
		else
		{
			*p_dest++=*p_source++;
		}
	}
	*p_dest++=0;

	return sp_out;
}

void WriteLocalTextDotQ()
{
	FILE *ifp=fopen(LOCALTEXTDOTQ ,"wb");
	Assert(ifp!=NULL,"Could not open localtext.q for writing.");

	/////////////////////////////////////////////////////////////////
	// NeedsAllTranslations
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This array contains all the English text in the q files which have not had any translations\n");
	fprintf(ifp,"// specified yet.\n");
	fprintf(ifp,"// Each of these entries needs a French=, British=, etc.\n");
	fprintf(ifp,"// Eg:\n");
	fprintf(ifp,"//  {\n");
	fprintf(ifp,"//      English=\"Hello\"\n");
	fprintf(ifp,"//      French=\"Bonjour\"\n");
	fprintf(ifp,"//  }\n");
	fprintf(ifp,"//\n");
	fprintf(ifp,"// Doing a 'localtext extract' will automatically move any translations added to the NeedsAllTranslations\n");
	fprintf(ifp,"// array to the Translations array further down, so there's no need to move them manually.\n\n");

	fprintf(ifp,"NeedsAllTranslations=\n[\n");
	CTranslation *pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (!pTrans->mCrap && 
			pTrans->mOccursInSomeQFile &&
			pTrans->mSectionChecksum!=0xc34206b2 && // DoNotTranslate
			pTrans->mpFrench[0]==0 &&
			pTrans->mpBritish[0]==0 &&
			pTrans->mpSpanish[0]==0 &&
			pTrans->mpGerman[0]==0 &&
			pTrans->mpItalian[0]==0
			)
		{
			fprintf(ifp,"    {\n");
			fprintf(ifp,"        // NeedsAllTranslations section\n");

			// Write out the comment indicating which qb files use this piece of text.
			fprintf(ifp,"        // ");
			for (int i=0; i<MAX_QB_FILES; ++i)
			{
				if (pTrans->mNumTimesOccursInQB[i])
				{
					if (pTrans->mNumTimesOccursInQB[i]>1)
					{
						fprintf(ifp,"%s(%d times) ",ppQBNames[i],pTrans->mNumTimesOccursInQB[i]);
					}
					else
					{
						fprintf(ifp,"%s ",ppQBNames[i]);
					}
				}
			}
			fprintf(ifp,"\n");

			fprintf(ifp,"        English=\"%s\"\n",FixString(pTrans->mpEnglish));
			if (pTrans->mpFrench[0])  fprintf(ifp,"        French=\"%s\"\n",FixString(pTrans->mpFrench));
			if (pTrans->mpBritish[0]) fprintf(ifp,"        British=\"%s\"\n",FixString(pTrans->mpBritish));
			if (pTrans->mpSpanish[0]) fprintf(ifp,"        Spanish=\"%s\"\n",FixString(pTrans->mpSpanish));
			if (pTrans->mpGerman[0])  fprintf(ifp,"        German=\"%s\"\n",FixString(pTrans->mpGerman));
			if (pTrans->mpItalian[0]) fprintf(ifp,"        Italian=\"%s\"\n",FixString(pTrans->mpItalian));
			fprintf(ifp,"    }\n");
		}
		pTrans=pTrans->mpPrevious;
	}

	fprintf(ifp,"]\n\n");

	/////////////////////////////////////////////////////////////////
	// NeedsTranslation
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This array contains all the English text in the q files which are\n");
	fprintf(ifp,"// missing some translations.\n");
	fprintf(ifp,"// Each of these entries needs a French=, British=, etc.\n");
	fprintf(ifp,"// Eg:\n");
	fprintf(ifp,"//  {\n");
	fprintf(ifp,"//      English=\"Hello\"\n");
	fprintf(ifp,"//      French=\"Bonjour\"\n");
	fprintf(ifp,"//  }\n");
	fprintf(ifp,"//\n");
	fprintf(ifp,"// Doing a 'localtext extract' will automatically move any translations added to the NeedsTranslation\n");
	fprintf(ifp,"// array to the Translations array further down, so there's no need to move them manually.\n\n");

	fprintf(ifp,"NeedsTranslation=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (!pTrans->mCrap && 
			pTrans->mOccursInSomeQFile &&
			pTrans->mSectionChecksum!=0xc34206b2 && // DoNotTranslate
			// Need to have at least one translation,
			(pTrans->mpFrench[0] || pTrans->mpSpanish[0] || pTrans->mpGerman[0] || pTrans->mpItalian[0]) &&
			// ... but not all of them
			(	pTrans->mpFrench[0]==0 ||
				pTrans->mpSpanish[0]==0 ||
				pTrans->mpGerman[0]==0 ||
				pTrans->mpItalian[0]==0)
			)
		{
			fprintf(ifp,"    {\n");
			fprintf(ifp,"        // NeedsTranslation section\n");

			// Write out the comment indicating which qb files use this piece of text.
			fprintf(ifp,"        // ");
			for (int i=0; i<MAX_QB_FILES; ++i)
			{
				if (pTrans->mNumTimesOccursInQB[i])
				{
					if (pTrans->mNumTimesOccursInQB[i]>1)
					{
						fprintf(ifp,"%s(%d times) ",ppQBNames[i],pTrans->mNumTimesOccursInQB[i]);
					}
					else
					{
						fprintf(ifp,"%s ",ppQBNames[i]);
					}
				}
			}
			fprintf(ifp,"\n");

			fprintf(ifp,"        English=\"%s\"\n",FixString(pTrans->mpEnglish));
			if (pTrans->mpFrench[0])  fprintf(ifp,"        French=\"%s\"\n",FixString(pTrans->mpFrench));
			if (pTrans->mpBritish[0]) fprintf(ifp,"        British=\"%s\"\n",FixString(pTrans->mpBritish));
			if (pTrans->mpSpanish[0]) fprintf(ifp,"        Spanish=\"%s\"\n",FixString(pTrans->mpSpanish));
			if (pTrans->mpGerman[0])  fprintf(ifp,"        German=\"%s\"\n",FixString(pTrans->mpGerman));
			if (pTrans->mpItalian[0]) fprintf(ifp,"        Italian=\"%s\"\n",FixString(pTrans->mpItalian));
			fprintf(ifp,"    }\n");
		}
		pTrans=pTrans->mpPrevious;
	}

	fprintf(ifp,"]\n\n");

	/////////////////////////////////////////////////////////////////
	// Translations
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This array contains all the English text in the q files which have had translations specified.\n");
	fprintf(ifp,"Translations=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		// Took out the test for mCrap because sometimes things that get put in crap are not
		// really crap and do require translations. If the test were not commented out and someone
		// had moved one of the 'crap' entries into the 'translations' section and added a translation,
		// then the next localtext extract would move it back into crap and lose the translation,
		// which would be bad.
		if (//!pTrans->mCrap && 
			pTrans->mOccursInSomeQFile &&
			pTrans->mSectionChecksum!=0xc34206b2 && // DoNotTranslate
			// Need to have all the translations present
			(pTrans->mpFrench[0] &&	pTrans->mpSpanish[0] && pTrans->mpGerman[0] && pTrans->mpItalian[0])
			)
		{
			fprintf(ifp,"    {\n");

			fprintf(ifp,"        // Translations section\n");
			// Write out the comment indicating which qb files use this piece of text.
			fprintf(ifp,"        // ");
			for (int i=0; i<MAX_QB_FILES; ++i)
			{
				if (pTrans->mNumTimesOccursInQB[i])
				{
					if (pTrans->mNumTimesOccursInQB[i]>1)
					{
						fprintf(ifp,"%s(%d times) ",ppQBNames[i],pTrans->mNumTimesOccursInQB[i]);
					}
					else
					{
						fprintf(ifp,"%s ",ppQBNames[i]);
					}
				}
			}
			fprintf(ifp,"\n");

			fprintf(ifp,"        English=\"%s\"\n",FixString(pTrans->mpEnglish));
			if (pTrans->mpFrench[0])  fprintf(ifp,"        French=\"%s\"\n",FixString(pTrans->mpFrench));
			if (pTrans->mpBritish[0]) fprintf(ifp,"        British=\"%s\"\n",FixString(pTrans->mpBritish));
			if (pTrans->mpSpanish[0]) fprintf(ifp,"        Spanish=\"%s\"\n",FixString(pTrans->mpSpanish));
			if (pTrans->mpGerman[0])  fprintf(ifp,"        German=\"%s\"\n",FixString(pTrans->mpGerman));
			if (pTrans->mpItalian[0]) fprintf(ifp,"        Italian=\"%s\"\n",FixString(pTrans->mpItalian));
			fprintf(ifp,"    }\n");
		}
		pTrans=pTrans->mpPrevious;
	}

	fprintf(ifp,"]\n\n");


	/////////////////////////////////////////////////////////////////
	// DoNotTranslate
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This array contains all text that used to be in the NeedsTranslation section\n");
	fprintf(ifp,"// but no longer needs to be translated and will not be swapped out of the qb files.\n");
	fprintf(ifp,"DoNotTranslate=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (pTrans->mSectionChecksum==0xc34206b2) // DoNotTranslate
		{
			fprintf(ifp,"    {\n");
			fprintf(ifp,"        // DoNotTranslate section\n");

			// Write out the comment indicating which qb files use this piece of text.
			fprintf(ifp,"        // ");
			for (int i=0; i<MAX_QB_FILES; ++i)
			{
				if (pTrans->mNumTimesOccursInQB[i])
				{
					if (pTrans->mNumTimesOccursInQB[i]>1)
					{
						fprintf(ifp,"%s(%d times) ",ppQBNames[i],pTrans->mNumTimesOccursInQB[i]);
					}
					else
					{
						fprintf(ifp,"%s ",ppQBNames[i]);
					}
				}
			}
			fprintf(ifp,"\n");

			fprintf(ifp,"        English=\"%s\"\n",FixString(pTrans->mpEnglish));
			if (pTrans->mpFrench[0])  fprintf(ifp,"        French=\"%s\"\n",FixString(pTrans->mpFrench));
			if (pTrans->mpBritish[0]) fprintf(ifp,"        British=\"%s\"\n",FixString(pTrans->mpBritish));
			if (pTrans->mpSpanish[0]) fprintf(ifp,"        Spanish=\"%s\"\n",FixString(pTrans->mpSpanish));
			if (pTrans->mpGerman[0])  fprintf(ifp,"        German=\"%s\"\n",FixString(pTrans->mpGerman));
			if (pTrans->mpItalian[0]) fprintf(ifp,"        Italian=\"%s\"\n",FixString(pTrans->mpItalian));
			fprintf(ifp,"    }\n");
		}
		pTrans=pTrans->mpPrevious;
	}
	fprintf(ifp,"]\n\n");

	/////////////////////////////////////////////////////////////////
	// Unused
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This array contains any old text that may have had a translation specified once but no longer\n");
	fprintf(ifp,"// occurs in any q file.\n");
	fprintf(ifp,"Unused=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (!pTrans->mCrap && 
			!pTrans->mOccursInSomeQFile &&
			pTrans->mSectionChecksum!=0xc34206b2) // DoNotTranslate
		{
			fprintf(ifp,"    {\n");

			fprintf(ifp,"        English=\"%s\" // (Unused)\n",FixString(pTrans->mpEnglish));
			if (pTrans->mpFrench[0])  fprintf(ifp,"        French=\"%s\"\n",FixString(pTrans->mpFrench));
			if (pTrans->mpBritish[0]) fprintf(ifp,"        British=\"%s\"\n",FixString(pTrans->mpBritish));
			if (pTrans->mpSpanish[0]) fprintf(ifp,"        Spanish=\"%s\"\n",FixString(pTrans->mpSpanish));
			if (pTrans->mpGerman[0])  fprintf(ifp,"        German=\"%s\"\n",FixString(pTrans->mpGerman));
			if (pTrans->mpItalian[0]) fprintf(ifp,"        Italian=\"%s\"\n",FixString(pTrans->mpItalian));
			fprintf(ifp,"    }\n");
		}
		pTrans=pTrans->mpPrevious;
	}

	fprintf(ifp,"]\n\n");

	/////////////////////////////////////////////////////////////////
	// Crap
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This is text that is used in some q files but does not require translating, eg filenames.\n");
	fprintf(ifp,"Crap=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (pTrans->mCrap && 
			pTrans->mOccursInSomeQFile && 
			// Don't write out "\" because it confuses qcomp
			strcmp(pTrans->mpEnglish,"\\") &&
			// Don't put it in the crap section if it has translations specified.
			!(pTrans->mpFrench[0] ||	pTrans->mpBritish[0] || pTrans->mpSpanish[0] || pTrans->mpGerman[0] || pTrans->mpItalian[0]))
		{
			fprintf(ifp,"    \"%s\" // (Crap)\n",FixString(pTrans->mpEnglish));
		}
		pTrans=pTrans->mpPrevious;
	}
	fprintf(ifp,"]\n\n");

	/////////////////////////////////////////////////////////////////
	// UnusedCrap
	/////////////////////////////////////////////////////////////////
	fprintf(ifp,"// This is text that does not need translating, and does not occur in any q file any more either.\n");
	fprintf(ifp,"UnusedCrap=\n[\n");
	pTrans=GetLastTransInList();
	while (pTrans)
	{
		if (pTrans->mCrap &&
			!pTrans->mOccursInSomeQFile && 
			// Don't write out "\" because it confuses qcomp
			strcmp(pTrans->mpEnglish,"\\"))
		{
			fprintf(ifp,"    \"%s\" // (Unused crap)\n",FixString(pTrans->mpEnglish));
		}
		pTrans=pTrans->mpPrevious;
	}
	fprintf(ifp,"]\n\n");

	fclose(ifp);
}

// pToken must point to the next token after the ESCRIPTTOKEN_STARTARRAY and any end-of-lines.
// The pointer returned will point to the ESCRIPTTOKEN_ENDARRAY
uint8 *GetTranslationsFromQBStructureArray(uint8 *pToken, uint32 SectionChecksum)
{
	while (*pToken!=ESCRIPTTOKEN_ENDARRAY)
	{
		// Must be an array of structures.
		Assert(*pToken==ESCRIPTTOKEN_STARTSTRUCT,"Expected a {");
		pToken=SkipTokenAndEndOfLines(pToken);

		const char *pEnglish="";
		const char *pFrench="";
		const char *pBritish="";
		const char *pSpanish="";
		const char *pGerman="";
		const char *pItalian="";

		while (*pToken!=ESCRIPTTOKEN_ENDSTRUCT)
		{
			gpLocalTextToken=pToken;

			if (*pToken==ESCRIPTTOKEN_NAME)
			{
				uint32 Checksum=*(uint32*)(pToken+1);
				pToken=SkipTokenAndEndOfLines(pToken);
				Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected an =");
				pToken=SkipTokenAndEndOfLines(pToken);
				Assert(*pToken==ESCRIPTTOKEN_STRING || *pToken==ESCRIPTTOKEN_LOCALSTRING,"Expected a string");

				switch (Checksum)
				{
				case ENGLISH: pEnglish=(const char*)(pToken+5); break;
				case FRENCH:  pFrench=(const char*)(pToken+5); break;
				case BRITISH: pBritish=(const char*)(pToken+5); break;
				case SPANISH: pSpanish=(const char*)(pToken+5); break;
				case GERMAN:  pGerman=(const char*)(pToken+5); break;
				case ITALIAN: pItalian=(const char*)(pToken+5); break;
				default:
					// Actually, printing the line number won't be that useful here, because inside arrays
					// there are no end-of-line tokens, so it will end up printing the line number of the
					// end of the array. Need to fix qcomp so that it does include end-of-lines for arrays
					// (Missing them off was a memory optimization)
					printf("Bad language name, checksum=0x%08x (Line %d)\n",Checksum,FindLineNumber(pToken));
					Assert(0,"Bad language name");
					break;
				}

				pToken=SkipTokenAndEndOfLines(pToken);
			}
			else
			{
				pToken=SkipTokenAndEndOfLines(pToken);
			}
		}
		pToken=SkipTokenAndEndOfLines(pToken);

		CTranslation *pTrans=FindTranslation(pEnglish,ENGLISH);
		if (pTrans==NULL)
		{
			pTrans=new CTranslation;
		}
		pTrans->mSectionChecksum=SectionChecksum;

		bool Error=false;

		if (pTrans->mpEnglish[0]==0)
		{
			pTrans->SetText(pEnglish,ENGLISH);
		}
		else
		{
			if (pEnglish[0])
			{
				if (strcmp(pTrans->mpEnglish,pEnglish))
				{
					Error=true;
				}
			}
		}

		if (pTrans->mpFrench[0]==0)
		{
			pTrans->SetText(pFrench,FRENCH);
		}
		else
		{
			if (pFrench[0])
			{
				if (strcmp(pTrans->mpFrench,pFrench))
				{
					Error=true;
				}
			}
		}
		if (pTrans->mpBritish[0]==0)
		{
			pTrans->SetText(pBritish,BRITISH);
		}
		else
		{
			if (pBritish[0])
			{
				if (strcmp(pTrans->mpBritish,pBritish))
				{
					Error=true;
				}
			}
		}
		if (pTrans->mpSpanish[0]==0)
		{
			pTrans->SetText(pSpanish,SPANISH);
		}
		else
		{
			if (pSpanish[0])
			{
				if (strcmp(pTrans->mpSpanish,pSpanish))
				{
					Error=true;
				}
			}
		}
		if (pTrans->mpGerman[0]==0)
		{
			pTrans->SetText(pGerman,GERMAN);
		}
		else
		{
			if (pGerman[0])
			{
				if (strcmp(pTrans->mpGerman,pGerman))
				{
					Error=true;
				}
			}
		}
		if (pTrans->mpItalian[0]==0)
		{
			pTrans->SetText(pItalian,ITALIAN);
		}
		else
		{
			if (pItalian[0])
			{
				if (strcmp(pTrans->mpItalian,pItalian))
				{
					Error=true;
				}
			}
		}

		// In theory, none of the English text in any of the structures in localtext.q
		// should be crap, because localtext.q was generated by localtext.exe, and it would
		// have already put all the crap text into the 'crap' section, which is an array of
		// strings, not structures.
		// However, if MaybeCrap gets modified, strings which were not previously crap may
		// now be, so will need to be flagged as such so that they get re-exported into
		// the crap section.
		// (I recently modified MaybeCrap to detect filenames that were using / instead of \)
		// Note: If a string is newly discovered to be crap, any translation it had will be lost.
		if (pTrans->mpEnglish[0])
		{
			pTrans->mCrap=MaybeCrap(pTrans->mpEnglish);
		}

		if (Error)
		{
			// Error! 
			printf("\nTwo different translations for the English text \"%s\":\n\n",pEnglish);

			printf("English=\"%s\"\n",pTrans->mpEnglish);
			printf("French=\"%s\"\n",pTrans->mpFrench);
			printf("British=\"%s\"\n",pTrans->mpBritish);
			printf("Spanish=\"%s\"\n",pTrans->mpSpanish);
			printf("German=\"%s\"\n\n",pTrans->mpGerman);
			printf("Italian=\"%s\"\n\n",pTrans->mpItalian);

			printf("and\n\n");

			printf("English=\"%s\"\n",pEnglish);
			printf("French=\"%s\"\n",pFrench);
			printf("British=\"%s\"\n",pBritish);
			printf("Spanish=\"%s\"\n",pSpanish);
			printf("German=\"%s\"\n\n",pGerman);
			printf("Italian=\"%s\"\n\n",pItalian);

			Assert(0,"Two different translations for the same English text in localtext.q");
		}
	}
	return pToken;
}

// pToken must point to the next token after the ESCRIPTTOKEN_STARTARRAY and any end-of-lines.
// The pointer returned will point to the ESCRIPTTOKEN_ENDARRAY
uint8 *GetTranslationsFromQBStringArray(uint8 *pToken, uint32 SectionChecksum)
{
	while (*pToken!=ESCRIPTTOKEN_ENDARRAY)
	{
		// Must be an array of strings.
		Assert(*pToken==ESCRIPTTOKEN_STRING || *pToken==ESCRIPTTOKEN_LOCALSTRING,"Expected array to contain strings");
		const char *pEnglish=(const char*)(pToken+5);

		CTranslation *pTrans=FindTranslation(pEnglish,ENGLISH);
		if (pTrans!=NULL)
		{
			if( pTrans->mCrap )
			{
				printf ("WARNING!!!\n");
				printf("\nTwo entries for the English (maybe crap) text '%s'\nExisting entry is:\n\n",pEnglish);
				printf("English=\"%s\"\n",pTrans->mpEnglish);
				printf("French=\"%s\"\n",pTrans->mpFrench);
				printf("British=\"%s\"\n",pTrans->mpBritish);
				printf("Spanish=\"%s\"\n",pTrans->mpSpanish);
				printf("German=\"%s\"\n",pTrans->mpGerman);
				printf("Italian=\"%s\"\n\n",pTrans->mpItalian);
			}
			else
			{
				printf("\nTwo entries for the English (maybe crap) text '%s'\nExisting entry is:\n\n",pEnglish);

				printf("English=\"%s\"\n",pTrans->mpEnglish);
				printf("French=\"%s\"\n",pTrans->mpFrench);
				printf("British=\"%s\"\n",pTrans->mpBritish);
				printf("Spanish=\"%s\"\n",pTrans->mpSpanish);
				printf("German=\"%s\"\n",pTrans->mpGerman);
				printf("Italian=\"%s\"\n\n",pTrans->mpItalian);

				Assert(0,"Two entries for the same English text in localtext.q");
			}
		}

		pTrans=new CTranslation;
		pTrans->mSectionChecksum=SectionChecksum;
		pTrans->SetText(pEnglish,ENGLISH);
		// The string array is either Crap or UnusedCrap, so set the flag.
		
		pTrans->mCrap=true;

		pToken=SkipTokenAndEndOfLines(pToken);
	}
	return pToken;
}

#define MAX_JUMP_OFFSETS 2000
const uint8 *ppJumpOffsets[MAX_JUMP_OFFSETS];

void AddJumpOffset(const uint8 *pJumpOffset)
{
	if (*(long*)pJumpOffset<0)
	{
		printf("%d\n",*(long*)pJumpOffset);
	}
	// Negative jump offsets will be more complex to update, so I hope there aren't any ...
	// There should not be any at the moment, but maybe that might change later.
	Assert(*(long*)pJumpOffset>=0,"Oops, negative jump offset ... ");

	bool Added=false;
	for (int i=0; i<MAX_JUMP_OFFSETS; ++i)
	{
		if (ppJumpOffsets[i]==NULL)
		{
			ppJumpOffsets[i]=pJumpOffset;
			Added=true;
			break;
		}
	}
	Assert(Added,"Need to increase MAX_JUMP_OFFSETS");
}

char LastFile[256] = { 0 };

void SwapText(const char *pQBName, uint32 FromLanguage, uint32 ToLanguage)
{
	printf("Translating %s\n",pQBName);

	strncpy(LastFile,pQBName,strlen(pQBName));

	// Open the file and load it into memory
	FILE *ifp=fopen(pQBName,"rb");
	Assert(ifp!=NULL,"Could not open source file");
	fseek(ifp,0,SEEK_END);
	long SourceSize=ftell(ifp);
	fseek(ifp,0,SEEK_SET);
	uint8 *pSourceQB=(uint8*)malloc(SourceSize);
	Assert(pSourceQB!=NULL,"Could not allocate memory to hold input file");
	Assert(fread(pSourceQB,SourceSize,1,ifp)==1,"fread failed");
	fclose(ifp);
	ifp=NULL;
	// The qb is now stored in pSourceQB

	#define OUTPUT_BUFFER_MARGIN 5000
	long DestSize=SourceSize+OUTPUT_BUFFER_MARGIN;
	uint8 *pDestQB=(uint8*)malloc(DestSize);
	Assert(pDestQB!=NULL,"Could not allocate memory to hold input file");

	// Initialise the list of jump offsets.
	for (int i=0; i<MAX_JUMP_OFFSETS; ++i)
	{
		ppJumpOffsets[i]=NULL;
	}


	uint8 *pToken=pSourceQB;
	uint8 *pOut=pDestQB;
	// Certain strings must not be translated, eg Target="..." Prefix="..." and Desc="..."
	int TargetCheck=0;
	while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
	{
		bool GotTargetEquals=false;
		switch (TargetCheck)
		{
		case 0:
			if (*pToken==ESCRIPTTOKEN_NAME)
			{
				uint32 Ch=*(uint32*)(pToken+1);
				if (Ch==0xb990d003/*Target*/ || 
					Ch==0x6c4e7971/*Prefix*/ || 
					Ch==0xf44a53ab/*Desc*/ ||
					Ch==0x9458ff20/*GripTapeDescription*/)
				{
					++TargetCheck;
				}
			}
			break;
		case 1:
			if (*pToken==ESCRIPTTOKEN_EQUALS)
			{
				++TargetCheck;
			}
			else
			{
				if (*pToken==ESCRIPTTOKEN_ENDOFLINE || *pToken==ESCRIPTTOKEN_ENDOFLINENUMBER)
				{
				}
				else
				{
					TargetCheck=0;
				}
			}
			break;
		case 2:
			if (*pToken==ESCRIPTTOKEN_STRING)
			{
				GotTargetEquals=true;
			}
			TargetCheck=0;
			break;
		default:
			Assert(0,"Bad TargetCheck value");
			break;
		}


		bool WrittenToken=false;

		// Do not replace the Target= strings, since these are script names.
		if (!GotTargetEquals && (*pToken==ESCRIPTTOKEN_STRING || *pToken==ESCRIPTTOKEN_LOCALSTRING))
		{
			const char *pOldText=(const char *)pToken+5;
			
			if (pOldText[0])
			{
				CTranslation *pTrans=FindTranslation(pOldText,FromLanguage);
				if (pTrans && pTrans->mSectionChecksum!=0xc34206b2/*DoNotTranslate*/)
				{
					const char *pNewText=pTrans->GetText(ToLanguage);
					if (pNewText[0])
					{
						//printf("Changing \"%s\"\n      To \"%s\"\n",pOldText,pNewText);
						int NewLen=strlen(pNewText);
						int OldLen=strlen(pOldText);

						// Update any jump offsets that jump across this string.
						int d=NewLen-OldLen;
						uint8 *pAfter=pOut+5+OldLen+1;

						for (int i=0; i<MAX_JUMP_OFFSETS; ++i)
						{
							const uint8 *pJumpOffset=ppJumpOffsets[i];
							if (pJumpOffset)
							{
								const uint8 *pJumpsTo=pJumpOffset+4+*(uint32*)pJumpOffset;
								if (pJumpsTo<pAfter)
								{
									// It jumped to before this string, so no need to ever check
									// this jump offset again, so remove it.
									ppJumpOffsets[i]=NULL;
								}
								else
								{
									// Update the offset by adding the difference in string size.
									*(uint32*)pJumpOffset+=d;
								}
							}
						}

						// Write out the new string
						Assert(pOut+5+NewLen+1<pOut+DestSize,"Output buffer overflow! Need to increase OUTPUT_BUFFER_MARGIN");
						*pOut++=*pToken;
						*(uint32*)pOut=NewLen+1;
						pOut+=4;
						strcpy((char*)pOut,pNewText);
						pOut+=NewLen+1;

						pToken=SkipToken(pToken);
						WrittenToken=true;
					} // if (pNewText[0])
				} // if (pTrans)
			} // if (pOldText[0])
		} // if (*pToken==ESCRIPTTOKEN_STRING || *pToken==ESCRIPTTOKEN_LOCALSTRING)

		if (!WrittenToken)
		{
			// Write the token out to the output buffer.
			const uint8 *pLastToken=pToken;
			pToken=SkipToken(pToken);

			int NumBytes=pToken-pLastToken;
			Assert(pOut+NumBytes<pOut+DestSize,"Output buffer overflow! Need to increase OUTPUT_BUFFER_MARGIN");
			const uint8 *pStart=pOut;
			for (int i=0; i<NumBytes; ++i)
			{
				*pOut++=*pLastToken++;
			}

			// If the token was a jump instruction, add the jump offset to the table of jump
			// offsets for possible updating later.
			if (*pStart==ESCRIPTTOKEN_JUMP)
			{
				// Calculate what the pointer to the offset will be in the output buffer.
				const uint8 *pJumpOffset=pStart+1;
				// Add it in to the list of jump offsets.
				AddJumpOffset(pJumpOffset);
			}
			else if (*pStart==ESCRIPTTOKEN_KEYWORD_RANDOM || *pStart==ESCRIPTTOKEN_KEYWORD_RANDOM2)
			{
				uint32 NumOffsets=*(uint32*)(pStart+1);

				// The 2*NumOffsets is to skip over the weights section.
				const uint8 *pJumpOffset=pStart+5+2*NumOffsets;
				for (uint32 i=0; i<NumOffsets; ++i)
				{
					AddJumpOffset(pJumpOffset);
					pJumpOffset+=4;
				}
			}
		}
	}
	// Terminate the output buffer.
	Assert(pOut<pOut+DestSize,"Output buffer overflow! Need to increase OUTPUT_BUFFER_MARGIN");
	*pOut++=ESCRIPTTOKEN_ENDOFFILE;

	// Write out the new qb file.
	FILE *ofp=fopen(pQBName,"wb");
	if (ofp==NULL)
	{
		printf("Could not open %s for writing.\n",pQBName);
		Assert(0,"Could not open .qb file for writing. \n\nIt may have been checked in to SourceSafe by mistake.\nMake sure it is not in SourceSafe, then make your local copy writable.");
	}
	Assert(fwrite(pDestQB,pOut-pDestQB,1,ofp)==1,"fwrite failed");
	fclose(ofp);
	ofp=NULL;

	free(pSourceQB);
	free(pDestQB);
}

void SwapText(uint32 FromLanguage, uint32 ToLanguage, const char *pQBOnlyNameBuf = NULL)
{
	printf("Swapping text from %s to %s\n",GetLanguageName(FromLanguage),GetLanguageName(ToLanguage));

	char *pPath=getenv("PROJ_ROOT");
	Assert(pPath!=NULL,"The environment variable PROJ_ROOT is not set.");

	#define MAXDIRBUFCHARS 100
	char DirNameBuf[MAXDIRBUFCHARS+1];
	Assert(strlen(pPath)+strlen("\\bin\\win32\\qmake.txt")<=MAXDIRBUFCHARS,"Path too long.");
	strcpy(DirNameBuf,pPath);
	strcat(DirNameBuf,"\\bin\\win32\\qmake.txt");

	FILE *listifp=fopen(DirNameBuf,"r");
	Assert(listifp!=NULL,"Could not open qmake.txt.\n\nYou need to run qcompall.bat to generate qmake.txt");

    // Check each of the files listed.
    while (true)
    {
		#define MAXFILENAMECHARS 100
        // File name buffer.
        char pNameBuf[MAXFILENAMECHARS+1];
        pNameBuf[MAXFILENAMECHARS]=0;

        // Load each character into the buffer until end-of-line or end-of-file.
        int i=0;
        while (true)
        {
            Assert(i<=MAXFILENAMECHARS,"Too many chars in filename.");
			char ch=fgetc(listifp);

            // Check if end-of-file or end-of-line.
            if (ch==EOF || ch=='\n')
            {
                // Terminate the current string.
                pNameBuf[i]=0;
                break;
            }

			pNameBuf[i]=ch;
            ++i;
        }

        if (pNameBuf[0])
		{
			// Derive the .qb filename.
	        char pQBNameBuf[MAXFILENAMECHARS+1];
			strcpy(pQBNameBuf,pNameBuf);
			int Len=strlen(pQBNameBuf);
			Assert(Len>3,"Very short filename?");

			if (stricmp(&pQBNameBuf[Len-2],".q")==0)
				// Append a 'b' if it is a .q
				strcat(pQBNameBuf,"b");
			else if (stricmp(&pQBNameBuf[Len-3],".qn")==0)
				// Change to qb if it is a .qn
				pQBNameBuf[Len-1]='b';
			else
				Assert(0,"Filename does not have .q or .qn extension");

			// Only call extract on it if it is not localtext.q, just in case localtext.q
			// got created somehow in one of the scripts directories, which would happen if
			// localtext.exe was run from one of those directories.
			if (stricmp(pQBNameBuf,LOCALTEXTDOTQB))
			{   
				if (pQBOnlyNameBuf)
				{	
					
					int length1 = strlen(pQBOnlyNameBuf);
					int length2 = strlen(pQBNameBuf);
					if(length2 >= length1 && !stricmp(pQBOnlyNameBuf,&(pQBNameBuf[length2-length1])))
						SwapText(pQBNameBuf,FromLanguage,ToLanguage);
				}
				else
					SwapText(pQBNameBuf,FromLanguage,ToLanguage);
			}
		}
        else
            break;
    }
	
	fclose (listifp);

	printf("Done.\n");
}

HWND CreateListView (HWND hWndParent, HINSTANCE hInst, int ID_LISTVIEW)
{
	RECT rcl;
	INITCOMMONCONTROLSEX icex;
	HWND hWndListView;

	// Ensure that the common control DLL is loaded. 
	icex.dwSize = sizeof(INITCOMMONCONTROLSEX);
	icex.dwICC  = ICC_LISTVIEW_CLASSES;
	InitCommonControlsEx(&icex); 

	GetClientRect(hWndParent, &rcl);

	// Create a list-view control in icon view.
	hWndListView = CreateWindowEx(WS_EX_CLIENTEDGE, WC_LISTVIEW, "", WS_VISIBLE | WS_CHILD | LVS_SHOWSELALWAYS | LVS_LIST |
							 LVS_SORTASCENDING | WS_BORDER | WS_VSCROLL &~WS_HSCROLL,
							0, 0, 150, rcl.bottom - rcl.top, hWndParent, 
							(HMENU)0, hInst, NULL);
	if(hWndListView == NULL)
		return NULL;
	
	
	char *pPath=getenv("PROJ_ROOT");
	Assert(pPath!=NULL,"The environment variable PROJ_ROOT is not set.");

	#define MAXDIRBUFCHARS 100
	char DirNameBuf[MAXDIRBUFCHARS+1];
	Assert(strlen(pPath)+strlen("\\bin\\win32\\qmake.txt")<=MAXDIRBUFCHARS,"Path too long.");
	strcpy(DirNameBuf,pPath);
	strcat(DirNameBuf,"\\bin\\win32\\qmake.txt");

	FILE *listifp=fopen(DirNameBuf,"r");
	Assert(listifp!=NULL,"Could not open qmake.txt.\n\nYou need to run qcompall.bat to generate qmake.txt");

    // Check each of the files listed.
	int index=0;
    while (true)
    {
		#define MAXFILENAMECHARS 100
        // File name buffer.
        char pNameBuf[MAXFILENAMECHARS+1];
        pNameBuf[MAXFILENAMECHARS]=0;

        // Load each character into the buffer until end-of-line or end-of-file.
        int i=0;
        while (true)
        {
            Assert(i<=MAXFILENAMECHARS,"Too many chars in filename.");
			char ch=fgetc(listifp);

            // Check if end-of-file or end-of-line.
            if (ch==EOF || ch=='\n')
            {
                // Terminate the current string.
                pNameBuf[i]=0;
                break;
            }

			pNameBuf[i]=ch;
            ++i;
        }

        if (pNameBuf[0])
		{
			// Derive the .qb filename.
	        char pQBNameBuf[MAXFILENAMECHARS+1];
			strcpy(pQBNameBuf,pNameBuf);
			int Len=strlen(pQBNameBuf);
			Assert(Len>3,"Very short filename?");

			if (stricmp(&pQBNameBuf[Len-2],".q")==0)
				// Append a 'b' if it is a .q
				strcat(pQBNameBuf,"b");
			else if (stricmp(&pQBNameBuf[Len-3],".qn")==0)
				// Change to qb if it is a .qn
				pQBNameBuf[Len-1]='b';
			else
				Assert(0,"Filename does not have .q or .qn extension");
			
			LV_ITEM item;
			ZeroMemory(&item,sizeof(item));
			item.mask = LVIF_TEXT ;
			item.iItem = index++;
			item.iSubItem = 0;
			item.pszText = (strrchr(pQBNameBuf,'\\')+1);
			ListView_InsertItem(hWndListView,&item); 
			
		}
		else
            break;
	}
	
	return (hWndListView);
}


LRESULT WndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static HWND hQBListView;
	static HWND hSelAllButton;
	static HWND hSwapFrame;
	switch(msg)
	{
	case WM_CREATE:
		{
			hQBListView = CreateListView(hwnd,GetModuleHandle(NULL),10000);
			hSelAllButton = CreateWindow(WC_BUTTON, "", WS_VISIBLE | WS_CHILD,
							155, 0, 150, 50, hwnd,
							(HMENU)IDC_SELALL_BUTTON, GetModuleHandle(NULL), NULL);
			SetWindowText(hSelAllButton,"Select All");

			hSwapFrame = CreateWindow(WC_BUTTON,"", WS_VISIBLE | WS_CHILD | BS_GROUPBOX | WS_GROUP,
							155, 200, 150, 50, hwnd,
							(HMENU)(IDC_SELALL_BUTTON+1), GetModuleHandle(NULL), NULL);
			SetWindowText(hSwapFrame,"Swap");
		}
	case WM_SIZING:
	case WM_SIZE:
		return 0;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_SELALL_BUTTON:
			SetFocus(hQBListView);
			ListView_SetItemState( hQBListView, -1, LVIS_SELECTED, LVIS_SELECTED );
		default:
			return 0;
		}
	case WM_DESTROY:
		break;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance,LPSTR  lpszCmdParam, int  nCmdShow )
{
	static char szAppName[] = "LocalText";
	HWND        hwnd;
	MSG         msg;
	WNDCLASS    wc;

	wc.style         = CS_HREDRAW | CS_VREDRAW;
    wc.lpfnWndProc   = (WNDPROC)WndProc; 
	wc.cbClsExtra    = 0; 
    wc.cbWndExtra    = 0;
    wc.hInstance     = (HINSTANCE)hInstance;
	wc.hIcon         = LoadIcon( NULL, IDI_APPLICATION );
    wc.hCursor       = LoadCursor( NULL, IDC_ARROW );
    wc.hbrBackground = (HBRUSH)GetStockObject( WHITE_BRUSH );
	wc.lpszMenuName  = NULL;
    wc.lpszClassName = szAppName;

    RegisterClass( &wc );
	
	hwnd = CreateWindow(szAppName, "LocalText",WS_OVERLAPPEDWINDOW & ~WS_THICKFRAME,  CW_USEDEFAULT,                 
      CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, (HINSTANCE)hInstance, NULL);

	ShowWindow( hwnd, SW_SHOW );
	UpdateWindow( hwnd );
    
	while( GetMessage( &msg, NULL, 0, 0 ) )
	{
		TranslateMessage( &msg );
		DispatchMessage( &msg );
	}

	return msg.wParam;
}

int main(int argc, char* argv[])
{
	//HINSTANCE hInstance = (HINSTANCE)GetModuleHandle(NULL);

	//WinMain(hInstance,0,0,0);

	if (argc==1)
	{
		printf("\n");
		printf("LocalText.exe, %s\n\n",__DATE__);
		printf("Usage:\n\n");
		printf(">localtext extract english\n"); 
		printf("This will extract all the text from all the qb files in the data directory,\n");
		printf("label each string as English and add them to localtext.q if they are not\n");
		printf("there already.\n\n");
		printf(">localtext swap english french [file.qb]\n");
		printf("This will swap all english text in the qb's for the equivalent french using\n");
		printf("the translations in localtext.q for all *.qb files or the one specified\n\n");
		printf(">localtext qswap english french [file.qb]\n");
		printf("This will swap all french text in the qb's with the french in localtext.q.\n");
		printf("This swap will use English as the base language.\n\n");
		exit(1);
	}

	bool DoExtract=false;
	bool DoSwap=false;
	bool DoQSwap=false;
	if (stricmp(argv[1],"extract")==0) DoExtract=true;
	if (stricmp(argv[1],"swap")==0) DoSwap=true;
	if (stricmp(argv[1],"qswap")==0) {DoQSwap=true; DoSwap=true;}
	Assert(DoExtract || DoSwap,"Requires the flag 'Extract' or 'Swap' to be the first argument.");
	
	bool DoQBR = true;
	if(strchr(argv[argc-1],'-'))
		DoQBR = false;

	// Check if the localtext.q file exists in the current directory, and if it does qcomp it and
	// then load and parse the qb.
	FILE *qifp=fopen(LOCALTEXTDOTQ,"r");
	if (qifp)
	{
		fclose(qifp);

		if (DoQBR && !DoQSwap)
		{
			// localtext.q does exist, so qcomp it.
			printf("Calling qcomp on localtext.q ...\n");
			char *args[4];
			args[0]="qcomp";
			args[1]=LOCALTEXTDOTQ;
			args[2]="-nomapfile";
			args[3]=NULL;
			int ErrorCode=spawnvp(P_WAIT,args[0],&args[0]);
			if (ErrorCode<0)
			{
				Assert(0,"Could not run qcomp.exe ...\n");
			}
			if (ErrorCode>0)
			{
				Assert(0,"QComp.exe returned an error.\n");
			}
			printf("Done.\n");
			// Successfully qcomp'd it, so now parse the qb.
		}
		// Open the file and load it into memory
		printf("Parsing localtext.qb ...\n");
		FILE *ifp=fopen(LOCALTEXTDOTQB,"rb");
		Assert(ifp!=NULL,"Could not open localtext.qb");
		fseek(ifp,0,SEEK_END);
		long SourceSize=ftell(ifp);
		fseek(ifp,0,SEEK_SET);
		uint8 *pQB=(uint8*)malloc(SourceSize);
		Assert(pQB!=NULL,"Could not allocate memory to hold input file");
		Assert(fread(pQB,SourceSize,1,ifp)==1,"fread failed");
		fclose(ifp);
		ifp=NULL;
		// The qb is now stored in pQB

		uint8 *pToken=pQB;
		while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
		{
			if (*pToken==ESCRIPTTOKEN_NAME)
			{
				uint32 Checksum=*(uint32*)(pToken+1);
				switch (Checksum)
				{
				case 0x39482578: // Translations
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Translations' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Translations=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
					break;
				case 0x6bd3b706: // Unused
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Unused' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Unused=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
					break;
				case 0x1264e951: // NeedsAllTranslations
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'NeedsAllTranslations' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'NeedsAllTranslations=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
					break;
				case 0x3130c2e5: // NeedsTranslation
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'NeedsTranslation' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'NeedsTranslation=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
					break;
				case 0xc34206b2: // DoNotTranslate
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'DoNotTranslate' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'DoNotTranslate=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
					break;
				case 0x8cbeefea: // Crap
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Crap' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Crap=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStringArray(pToken,Checksum);
					break;

				case 0xc70a564c: // UnusedCrap
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'UnusedCrap' to be followed by =");
					pToken=SkipTokenAndEndOfLines(pToken);
					Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'UnusedCrap=' to be followed by [");
					pToken=SkipTokenAndEndOfLines(pToken);
					pToken=GetTranslationsFromQBStringArray(pToken,Checksum);
					break;

				default:
					break;
				}
			}
			pToken=SkipTokenAndEndOfLines(pToken);
		}
		free(pQB);
		gpLocalTextToken=NULL;
		printf("Done.\n");
	}


	if (DoExtract)
	{
		uint32 Language=ENGLISH;
		if (argc>=3)
		{
			if (stricmp(argv[2],"english")==0) Language=ENGLISH;
			else if (stricmp(argv[2],"french")==0) Language=FRENCH;
			else if (stricmp(argv[2],"british")==0) Language=BRITISH;
			else if (stricmp(argv[2],"spanish")==0) Language=SPANISH;
			else if (stricmp(argv[2],"german")==0) Language=GERMAN;
			else if (stricmp(argv[2],"italian")==0) Language=ITALIAN;
			else Assert(0,"Bad language following 'extract'");
		}
		Extract(Language);

		WriteLocalTextDotQ();
	}

	// Make sure there is no mismatch in the number of percent signs for any translation,
	// otherwise that will make printf crash in the game.
	CheckPercents();
	char *pQBName = NULL;
	if (DoSwap)
	{
		uint32 FromLanguage=ENGLISH;
		if (argc>=3)
		{
			if (stricmp(argv[2],"english")==0) FromLanguage=ENGLISH;
			else if (stricmp(argv[2],"french")==0) FromLanguage=FRENCH;
			else if (stricmp(argv[2],"british")==0) FromLanguage=BRITISH;
			else if (stricmp(argv[2],"spanish")==0) FromLanguage=SPANISH;
			else if (stricmp(argv[2],"german")==0) FromLanguage=GERMAN;
			else if (stricmp(argv[2],"italian")==0) FromLanguage=ITALIAN;
			else Assert(0,"Bad language following 'swap'");
		}
		uint32 ToLanguage=FRENCH;
		if (argc>=4)
		{
			if (stricmp(argv[3],"english")==0) ToLanguage=ENGLISH;
			else if (stricmp(argv[3],"french")==0) ToLanguage=FRENCH;
			else if (stricmp(argv[3],"british")==0) ToLanguage=BRITISH;
			else if (stricmp(argv[3],"spanish")==0) ToLanguage=SPANISH;
			else if (stricmp(argv[3],"german")==0) ToLanguage=GERMAN;
			else if (stricmp(argv[3],"italian")==0) ToLanguage=ITALIAN;
			else Assert(0,"Bad language following 'swap'");
		}

		if (argc>=5)
		{	
			if( !strchr(argv[4],'-') )
				pQBName = argv[4];
		}

		if (DoQSwap)
		{
			SwapText(ToLanguage,FromLanguage,pQBName);
			CleanUp();
			FILE *qifp=fopen(LOCALTEXTDOTQ,"r");
			if (qifp)
			{
				fclose(qifp);

				
				// localtext.q does exist, so qcomp it.
				printf("Calling qcomp on localtext.q ...\n");
				char *args[4];
				args[0]="qcomp";
				args[1]=LOCALTEXTDOTQ;
				args[2]="-nomapfile";
				args[3]=NULL;
				int ErrorCode=spawnvp(P_WAIT,args[0],&args[0]);
				if (ErrorCode<0)
				{
					Assert(0,"Could not run qcomp.exe ...\n");
				}
				if (ErrorCode>0)
				{
					Assert(0,"QComp.exe returned an error.\n");
				}
				printf("Done.\n");
				// Successfully qcomp'd it, so now parse the qb.
		
				// Open the file and load it into memory
				printf("Parsing localtext.qb ...\n");
				FILE *ifp=fopen(LOCALTEXTDOTQB,"rb");
				Assert(ifp!=NULL,"Could not open localtext.qb");
				fseek(ifp,0,SEEK_END);
				long SourceSize=ftell(ifp);
				fseek(ifp,0,SEEK_SET);
				uint8 *pQB=(uint8*)malloc(SourceSize);
				Assert(pQB!=NULL,"Could not allocate memory to hold input file");
				Assert(fread(pQB,SourceSize,1,ifp)==1,"fread failed");
				fclose(ifp);
				ifp=NULL;
				// The qb is now stored in pQB

				uint8 *pToken=pQB;
				while (*pToken!=ESCRIPTTOKEN_ENDOFFILE)
				{
					if (*pToken==ESCRIPTTOKEN_NAME)
					{
						uint32 Checksum=*(uint32*)(pToken+1);
						switch (Checksum)
						{
						case 0x39482578: // Translations
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Translations' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Translations=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
							break;
						case 0x6bd3b706: // Unused
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Unused' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Unused=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
							break;
						case 0x1264e951: // NeedsAllTranslations
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'NeedsAllTranslations' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'NeedsAllTranslations=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
							break;
						case 0x3130c2e5: // NeedsTranslation
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'NeedsTranslation' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'NeedsTranslation=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
							break;
						case 0xc34206b2: // DoNotTranslate
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'DoNotTranslate' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'DoNotTranslate=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStructureArray(pToken,Checksum);
							break;
						case 0x8cbeefea: // Crap
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'Crap' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'Crap=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStringArray(pToken,Checksum);
							break;

						case 0xc70a564c: // UnusedCrap
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_EQUALS,"Expected 'UnusedCrap' to be followed by =");
							pToken=SkipTokenAndEndOfLines(pToken);
							Assert(*pToken==ESCRIPTTOKEN_STARTARRAY,"Expected 'UnusedCrap=' to be followed by [");
							pToken=SkipTokenAndEndOfLines(pToken);
							pToken=GetTranslationsFromQBStringArray(pToken,Checksum);
							break;

						default:
							break;
						}
					}
					pToken=SkipTokenAndEndOfLines(pToken);
				}
				free(pQB);
				gpLocalTextToken=NULL;
				printf("Done.\n");
			}

			SwapText(FromLanguage,ToLanguage,pQBName);

			if(LastFile[0])
			{
				printf("Calling rq on %s ...\n",LastFile);
				char args[256] = "rq ";
				strncat(args, LastFile,strlen(LastFile) );

				int ErrorCode=system(args);
				if (ErrorCode<0)
				{
					Assert(0,"Could not run rq.exe ...\n");
				}
				if (ErrorCode>0)
				{
					Assert(0,"rq.exe returned an error.\n");
				}
				printf("Done.\n");
			}
		}
		else
		{
			SwapText(FromLanguage,ToLanguage,pQBName);
		}
	}

	/*
	const char *p_found_text=NULL;
	CTranslation *p_search=GetLastTransInList();
	while (p_search)
	{
		if (GenerateChecksum(p_search->mpBritish)==FIND_ME)
		{
			p_found_text=p_search->mpBritish;
		}
		if (GenerateChecksum(p_search->mpFrench)==FIND_ME)
		{
			p_found_text=p_search->mpFrench;
		}
		if (GenerateChecksum(p_search->mpEnglish)==FIND_ME)
		{
			p_found_text=p_search->mpEnglish;
		}
		if (GenerateChecksum(p_search->mpGerman)==FIND_ME)
		{
			p_found_text=p_search->mpGerman;
		}
		p_search=p_search->mpPrevious;
	}
	if (p_found_text)
	{
		printf("Found! Text='%s'\n",p_found_text);
	}
	*/

	CleanUp();

	return 0;
}
