/*
	WorldDB.cpp
	Functions for generating and managing the world database
	This includes spatial partitioning functions
*/

#include "stdafx.h"
#include "WorldDB.h"

#define vNUM_PLATFORMS  3

NxObject nxobj;

WorldDB::WorldDB()
{

}

WorldDB::~WorldDB()
{

}

void WorldDB::AddMaterial(NxMaterial* pMtl)
{
	mtlList.Add(pMtl);
}

NxMaterial* WorldDB::GetMaterial(unsigned long crc)
{
	Link<NxMaterial>* link = mtlList.GetHead();

	while(link)
	{
		if (link->data.GetCRC() == crc)
			return &link->data;

		link = link->next;
	}

	return NULL;
}

bool WorldDB::LoadScene(char* filename)
{
	int mtlVersion, meshVersion, vertVersion;

	FILE* fp = fopen(filename, "rb");

	fread(&mtlVersion, sizeof(int), 1, fp);
	fread(&meshVersion, sizeof(int), 1, fp);
	fread(&vertVersion, sizeof(int), 1, fp);
	fread(&m_sky, sizeof(bool), 1, fp);

	if (!LoadMaterials(fp, mtlVersion))
		return false;

	if (!LoadObjects(fp, meshVersion))
		return false;

	fclose(fp);

	return true;
}

bool WorldDB::LoadMaterials(FILE* fp, int version)
{
	int nMaterials;

	fread(&nMaterials, sizeof(int), 1, fp);

	for(int i = 0; i < nMaterials; i++)
	{
		NxMaterial* cur_mat;
		int j, flags;

		//cur_mat = new NxMaterial;
		Link<NxMaterial>* link = mtlList.Add();
		cur_mat = &link->data;

		fread(&cur_mat->m_Checksum, sizeof(unsigned long), 1, fp);
		fread(&flags, sizeof(int), 1, fp);

		if (version >= 0x000F)
			fread(&cur_mat->m_NewChecksum, sizeof(unsigned long), 1, fp);
		else
			cur_mat->m_NewChecksum = cur_mat->m_Checksum;

		// CHECK: m_GroupFlags ?  NxMaterial::mTRANSPARENT

		if (flags & NxMaterial::mINVISIBLE)
			cur_mat->m_Invisible = true;

		if (flags & NxMaterial::mTWO_SIDED)
			cur_mat->m_TwoSided = true;

		if (flags & NxMaterial::mONE_SIDED)
			cur_mat->m_OneSided = true;

		fread(&cur_mat->m_Terrain, sizeof(int), 1, fp);

		if (version >= 0x0002)
			fread(&cur_mat->m_AlphaCutoff, sizeof(int), 1, fp);

		if (version >= 0x0005)
			fread(&cur_mat->m_AlphaCutoff, sizeof(int), 1, fp);

		if (version >= 0x0004)
		{
			fread(&cur_mat->m_DrawOrder, sizeof(float), 1, fp);
			fread(&cur_mat->m_Sorted, sizeof(bool), 1, fp);
		}

		if (version >= 0x0008)
			fread(&cur_mat->m_grassify, sizeof(bool), 1, fp);

		if (version >= 0x0009)
		{
			fread(&cur_mat->m_grassHeight, sizeof(float), 1, fp);
			fread(&cur_mat->m_grassLayers, sizeof(int), 1, fp);
		}

		if (version >= 0x000E)
			fread(&cur_mat->m_water, sizeof(bool), 1, fp);

		if (version >= 0x000D)
			fread(&cur_mat->m_BasePass, sizeof(int), 1, fp);

		// Specular data
		if (version >= 0x000A)
		{
			if (flags & NxMaterial::mSPECULAR)
			{
				fread(&cur_mat->m_SpecularPower, sizeof(float), 1, fp);
				fread(&cur_mat->m_SpecularColor, sizeof(float), 1, fp);
			}
		}

		if( flags & NxMaterial::mVC_WIBBLE )
		{
			int k, num_sequences;

			fread(&num_sequences, sizeof(int), 1, fp);

			for( j = 0; j < num_sequences; j++ )
			{
				VCWibbleSequence* new_seq = &cur_mat->m_WibbleSequences[j];

				new_seq->m_Index = j + 1;	// sequences are one-based
				fread(&new_seq->m_NumFrames, sizeof(int), 1, fp);
				new_seq->m_WibbleFrames = new WibbleKeyframe[new_seq->m_NumFrames];
				for( k = 0; k < new_seq->m_NumFrames; k++ )
				{
					fread(&new_seq->m_WibbleFrames[k].m_Time, sizeof(int), 1, fp);
					fread(&new_seq->m_WibbleFrames[k].m_Color, 4 * sizeof(float), 1, fp);
				}
			}
		}

		fread(&cur_mat->m_NumPasses, sizeof(int), 1, fp);
		for(j = 0; j < cur_mat->m_NumPasses; j++)
		{
			NxMaterialPass* pass;

			pass = &cur_mat->m_Passes[j];
			int k, flags, mmag, mmin;
			unsigned long tex_checksums[vNUM_PLATFORMS];
			
			if (version >= 0x000C)
			{
				fread(&pass->m_Flags, sizeof(unsigned long), 1, fp);

				if (pass->m_Flags & NxMaterialPass::ANIMATED_TEXTURE)
				{
					NxAnimatedTexture* anim_tex;
					int frame;

					anim_tex = &pass->m_AnimatedTexture;
					fread(&anim_tex->m_NumKeyframes, sizeof(int), 1, fp);
					fread(&anim_tex->m_Period, sizeof(int), 1, fp);
					fread(&anim_tex->m_Iterations, sizeof(int), 1, fp);
					fread(&anim_tex->m_Phase, sizeof(int), 1, fp);
					
					for( frame = 0; frame < anim_tex->m_NumKeyframes; frame++ )
					{
						NxAnimatedTextureKeyframe* key_frame;

						key_frame = &anim_tex->m_Keyframes[frame];
						fread(&key_frame->m_Time, sizeof(unsigned int), 1, fp);
						fread(&key_frame->m_TexChecksum, sizeof(unsigned long), vNUM_PLATFORMS, fp);
					}
				}
				else
				{
					fread(&tex_checksums, sizeof( unsigned long ), vNUM_PLATFORMS, fp);
					for( k = 0; k < vNUM_PLATFORMS; k++ )
					{
						pass->m_TexChecksum[k] = tex_checksums[k];
					}
				}
			}
			else
			{
				fread(&tex_checksums, sizeof( unsigned long ), vNUM_PLATFORMS, fp);
				for( k = 0; k < vNUM_PLATFORMS; k++ )
				{
					pass->m_TexChecksum[k] = tex_checksums[k];
				}				
			}

			fread(&flags, sizeof(int), 1, fp);
			fread(&pass->m_BlendMode, sizeof(int), 1, fp);

			if (version >= 0x0002)
			{
				fread(&pass->m_AddressModeU, sizeof(int), 1, fp);
				fread(&pass->m_AddressModeV, sizeof(int), 1, fp);
			}
			else
			{
				pass->m_AddressModeU = 0;
				pass->m_AddressModeV = 0;
			}

			fread(&pass->m_FixedAlpha, sizeof(int), 1, fp);

			if (version >= 0x0003)
				fread(&pass->m_HasColor, sizeof(bool), 1, fp);
			else
				pass->m_HasColor = true;

			fread(&pass->m_Color[0], sizeof(float), 1, fp);
			fread(&pass->m_Color[1], sizeof(float), 1, fp);
			fread(&pass->m_Color[2], sizeof(float), 1, fp);
			fread(&pass->m_Color[3], sizeof(float), 1, fp);

			fread(&pass->m_Diffuse[0], sizeof(float), 1, fp);
			fread(&pass->m_Diffuse[1], sizeof(float), 1, fp);
			fread(&pass->m_Diffuse[2], sizeof(float), 1, fp);

			fread(&pass->m_Specular[0], sizeof(float), 1, fp);
			fread(&pass->m_Specular[1], sizeof(float), 1, fp);
			fread(&pass->m_Specular[2], sizeof(float), 1, fp);

			fread(&pass->m_Ambient[0], sizeof(float), 1, fp);
			fread(&pass->m_Ambient[1], sizeof(float), 1, fp);
			fread(&pass->m_Ambient[2], sizeof(float), 1, fp);

			// Read in UV wibble params
			if( flags & NxMaterial::mUV_WIBBLE)
			{
				// Flag that this pass has UV wibbling enabled
				pass->m_UVWibbleEnabled = true;

				fread(&pass->m_UVel, sizeof(float), 1, fp);
				fread(&pass->m_VVel, sizeof(float), 1, fp);
				fread(&pass->m_UFrequency, sizeof(float), 1, fp);
				fread(&pass->m_VFrequency, sizeof(float), 1, fp);
				fread(&pass->m_UAmplitude, sizeof(float), 1, fp);
				fread(&pass->m_VAmplitude, sizeof(float), 1, fp);
				fread(&pass->m_UPhase, sizeof(float), 1, fp);
				fread(&pass->m_VPhase, sizeof(float), 1, fp);
			}

			if (version >= 0x000B)
			{
				if (flags & NxMaterial::mENVIRONMENT)
				{
					fread(&pass->m_EnvTileU, sizeof(float), 1, fp);
					fread(&pass->m_EnvTileV, sizeof(float), 1, fp);
				}
			}

			// Read in VC wibble sequence
			if (flags & NxMaterial::mVC_WIBBLE)
			{
				int nSequences;
				fread(&nSequences, sizeof(int), 1, fp);
			}

			if (flags & NxMaterial::mENVIRONMENT)
			{
				pass->m_MappingMode = vMAPPING_ENVIRONMENT;
			}

			// Read in MMAG and MMIN
			fread(&mmag, sizeof(int), 1, fp);
			fread(&mmin, sizeof(int), 1, fp);
			pass->m_MagFilteringMode = (FilteringMode)mmag;
			pass->m_MinFilteringMode = (FilteringMode)mmin;

			// Read in K
			fread(&pass->m_MipMapK, sizeof(float), 1, fp);
			if (version < 0x0002)
			{
				float mip_map_l;
				fread(&mip_map_l, sizeof(float), 1, fp);	// Trashed ???
			}

			// In version 0x000C we moved the flags to the top of the loop
			if (( version >= 0x000A ) && ( version < 0x000C ))
			{
				fread(&pass->m_Flags, sizeof(unsigned long), 1, fp);
			}
		}

		// If the user has created a material that blends with the frame-buffer but has not flagged it as
		// transparent, we will nudge it just a bit so that it renders right after opaque materials
		if(	( cur_mat->m_Passes[0].m_BlendMode != vBLEND_MODE_DIFFUSE ) /*&&
			( !( cur_mat->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT ))*/)
		{
			//cur_mat->m_GroupFlags |= NxMaterial::mGROUP_TRANSPARENT;
			cur_mat->m_DrawOrder = 1.0f;
		}

		// Keep the materials in sorted order based on draw order id
		/*
		if( cur_mat->m_Sorted )
		{
			// Sort "sorted" materials to the end of the list
			// I add one to make sure that we won't get a value of 0, which would merge sorted materials
			// in with unsorted ones.  Then I multiply * 1000 to make sure the sorted materials
			// sort to the end of the list
			cur_mat->SetPri(( cur_mat->m_DrawOrder + 1.0f ) * 1000.0f );
		}
		else
		{
			cur_mat->SetPri( cur_mat->m_DrawOrder );
		}
		*/
	}

	return true;
}

bool WorldDB::LoadObjects(FILE* fp, int version)
{
	int  i, j, k;
	bool bReadSkeletalInfo = false;
	int  numRemovedObjects = 0;

	int  nObjects = 0;

	fread(&nObjects, sizeof(int), 1, fp);

	for(i = 0; i < nObjects; i++)
	{
		NxVertex* vert_list;
		NxFace*   face_list;
		NxObject* object;

		Link<NxObject>* link = objList.Add();
		object = &link->data;

		fread(&object->m_Checksum, sizeof(int), 1, fp);
		fread(&object->m_Flags, sizeof(int), 1, fp);

		if (object->m_Flags & NxObject::mSKELETALMODEL)
		{
			fread(&object->m_ParentMatrixIndex, sizeof(int), 1, fp);
			bReadSkeletalInfo = true;
		}

		if (object->m_Flags & NxObject::mSKINNED)
		{
			fread(&object->m_SkeletonChecksum, sizeof(int), 1, fp);

			object->mp_CASData = new NxCASData[NxObject::vMAXCASDATA];
		}

		fread(&object->m_NumUVSets, sizeof(int), 1, fp);

		// Reading bounding box/sphere data
		fread(&object->m_BoundingBox.m_Min, sizeof(float), 3, fp);
		fread(&object->m_BoundingBox.m_Max, sizeof(float), 3, fp);

		//object->m_CollisionBoundingBox = object->m_BoundingBox;

		fread(&object->m_BoundingSphere.m_Center, sizeof(float), 3, fp);
		fread(&object->m_BoundingSphere.m_Radius, sizeof(float), 3, fp);
		
		fread(&object->m_NumVerts, sizeof(int), 1, fp);

		object->m_Verts = new NxVertex[ object->m_NumVerts ];
		vert_list = object->m_Verts;

		for(j = 0; j < object->m_NumVerts; j++)
		{
			NxVertex* vert;

			vert = &vert_list[j];

			// Read in first texture coordinate set
			if( object->m_Flags & NxObject::mTEXTURED )
			{
				for( k = 0; k < object->m_NumUVSets; k++ )
				{
					fread(&vert->m_TexCoord[k].x, sizeof(float), 1, fp);
					fread(&vert->m_TexCoord[k].y, sizeof(float), 1, fp);
					vert->m_TexCoord[j].z = 0.0f;
				}
			}

			// Read in colors
			if( object->m_Flags & NxObject::mCOLORED)
			{
				fread(&vert->m_Color[0], sizeof(float), 1, fp);
				fread(&vert->m_Color[1], sizeof(float), 1, fp);
				fread(&vert->m_Color[2], sizeof(float), 1, fp);
				fread(&vert->m_Color[3], sizeof(float), 1, fp);
			}

			// Read in normals
			if( object->m_Flags & ( NxObject::mNORMALS | NxObject::mSS_NORMALS ))
			{					
				fread(&vert->m_Normal.x, sizeof(float), 1, fp);
				fread(&vert->m_Normal.y, sizeof(float), 1, fp);
				fread(&vert->m_Normal.z, sizeof(float), 1, fp);
			}

			if( object->m_Flags & NxObject::mSKINNED)
			{
				fread(&vert->m_Weight[0], sizeof(float), 1, fp);
				fread(&vert->m_Weight[1], sizeof(float), 1, fp);
				fread(&vert->m_Weight[2], sizeof(float), 1, fp);

				fread(&vert->m_WeightedIndex[0], sizeof(int), 1, fp);
				fread(&vert->m_WeightedIndex[1], sizeof(int), 1, fp);
				fread(&vert->m_WeightedIndex[2], sizeof(int), 1, fp);
				
				if( object->m_Flags & NxObject::mHAS4WEIGHTS )
				{
					fread(&vert->m_Weight[3], sizeof(float), 1, fp);
					fread(&vert->m_WeightedIndex[3], sizeof(int), 1, fp);
				}
				else
				{
					vert->m_Weight[3]        = 0.0f;
					vert->m_WeightedIndex[3] = 0;					
				}
			}

			if( object->m_Flags & NxObject::mVCWIBBLE )
			{
				fread(&vert->m_WibbleIndex, sizeof(char), 1, fp);
				fread(&vert->m_WibbleOffset, sizeof(char), 1, fp);
			}

			// Read in positions				
			fread(&vert->m_Pos.x, sizeof(float), 1, fp);
			fread(&vert->m_Pos.y, sizeof(float), 1, fp);
			fread(&vert->m_Pos.z, sizeof(float), 1, fp);

			// Round vert positions to nearest tenth of an inch
			vert->m_Pos.x = ((int) (( vert->m_Pos.x * 16.0f ) + ((vert->m_Pos.x >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
			vert->m_Pos.y = ((int) (( vert->m_Pos.y * 16.0f ) + ((vert->m_Pos.y >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
			vert->m_Pos.z = ((int) (( vert->m_Pos.z * 16.0f ) + ((vert->m_Pos.z >0.0f) ? 0.5f : -0.5f) )) / 16.0f;
		}
		
		// Copy the verts over into the original verts field.
		object->m_OriginalVerts = new NxVertex[object->m_NumVerts];
		memcpy( object->m_OriginalVerts, object->m_Verts, sizeof( NxVertex ) * object->m_NumVerts );
		
		fread(&object->m_NumFaces, sizeof(int), 1, fp);
		object->m_Faces = new NxFace[ object->m_NumFaces ];
		face_list = object->m_Faces;		
		for( j = 0; j < object->m_NumFaces; j++ )
		{
			FlagType min_flags;

			// Read in the material checksum for this face
			fread(&face_list[j].m_MatChecksum, sizeof( unsigned long ), 1, fp);

			// Read in the face normal
			fread(&face_list[j].m_Normal, sizeof(float), 3, fp);

			// Read in the face flags
			fread(&face_list[j].m_FaceFlags, sizeof(FlagType), 1, fp);

			// Read in the minimalist face flags
			if( version >= 0x0002 )
			{
				fread(&min_flags, sizeof(FlagType), 1, fp);

				/*
				if( should_generate_optimized_level())
				{
					face_list[j].m_FaceFlags = min_flags;
				}
				*/
			}		

			if (face_list[j].m_FaceFlags & NxFace::mFD_CASFACEFLAGSEXIST)	// (versioning data isn't exported for NxObject)
			{
				// Read in CAS face flags
				fread(&face_list[j].m_CASFaceFlags, sizeof( CASFlagType ), 1, fp);
			}

			face_list[j].m_PassFlags = 0;
			if( !( face_list[j].m_FaceFlags & NxFace::mFD_PASS_1_DISABLED ))
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_1;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_2_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_2;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_3_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_3;
			}
			if( face_list[j].m_FaceFlags & NxFace::mFD_PASS_4_ENABLED )
			{
				face_list[j].m_PassFlags |= NxMaterial::mGROUP_PASS_4;
			}

			// Read in the vertex indices
			fread(&face_list[j].m_Vertex, sizeof(int), 3, fp);

			// Read in the lightmap data for this face
			if (face_list[j].m_FaceFlags & NxFace::mFD_LIGHTMAPPED)
			{
				fread(&face_list[j].m_LightmapChecksum, sizeof(unsigned long), 1, fp);
				fread(&face_list[j].m_LightmapUV, sizeof(UVCoord), 3, fp);
			}
			else
			{
				face_list[j].m_LightmapChecksum = 0;
				face_list[j].m_LightmapUV[0].u  = 0;
				face_list[j].m_LightmapUV[0].v  = 0;
				face_list[j].m_LightmapUV[1]    = face_list[j].m_LightmapUV[0];
				face_list[j].m_LightmapUV[2]    = face_list[j].m_LightmapUV[0];
			}
		}

		if (object->m_Flags & NxObject::mHASCASREMOVEFLAGS)
		{
			// Read in CAS face removal flags
			fread(&object->m_CASRemoveFlags, sizeof(int), 1, fp);
		}

		if (object->m_Flags & NxObject::mHASKBIAS)
		{
			// Read in KBias
			fread(&object->m_KBias, sizeof(int), 1, fp);
		}

		if (object->m_Flags & NxObject::mHASLODINFO)
		{
			// Read in the LOD info
			unsigned int version;
			fread(&version, sizeof(unsigned int), 1, fp);

			if (version >= 0x0001)
			{
				fread(&object->m_LODFlags, sizeof(int), 1, fp);

				if (object->m_LODFlags & NxObject::mMASTER)
				{
					fread(&object->m_LODMaster.m_NumLODLevels, sizeof(int), 1, fp);

					if (object->m_LODMaster.m_LODLevels)
						delete [] object->m_LODMaster.m_LODLevels;

					object->m_LODMaster.m_LODLevels = new NxLODLevel[object->m_LODMaster.m_NumLODLevels];

					fread(&object->m_LODMaster.m_LODLevels, sizeof(NxLODLevel), object->m_LODMaster.m_NumLODLevels, fp);

					for (int lidx = 0; lidx < object->m_LODMaster.m_NumLODLevels; lidx++)
					{
						//printf("Reading LOD %x with distance of %f\n", object->m_LODMaster.m_LODLevels[lidx].m_ObjectCRC, object->m_LODMaster.m_LODLevels[lidx].m_Distance);
						assert(object->m_Checksum != object->m_LODMaster.m_LODLevels[lidx].m_ObjectCRC);
					}
				}

				if (object->m_LODFlags & NxObject::mSLAVE)
				{
					fread(&object->m_LODSlave.m_masterCRC, sizeof(unsigned long), 1, fp);
					assert(object->m_Checksum != object->m_LODSlave.m_masterCRC);
				}
			}
		}

		if (object->m_Flags & NxObject::mHASINTLODINFO)
		{
			fread(&object->m_LODLevels, sizeof(int), 1, fp);
			
			object->m_LODinfo = new NxLODInfo[object->m_LODLevels];

			for(int lodlev = 0; lodlev < object->m_LODLevels; lodlev++)
			{
				fread(&object->m_LODinfo[lodlev].numFaces, sizeof(int), 1, fp);

				object->m_LODinfo[lodlev].faces = new NxLODFace[object->m_LODinfo[lodlev].numFaces];

				fread(&object->m_LODinfo[lodlev].faces, 
					  sizeof(NxLODFace), object->m_LODinfo[lodlev].numFaces, fp);

				// If userdefined LOD distances are included, load them in
				if (object->m_Flags & NxObject::mHASLODDISTS)
					fread(&object->m_LODinfo[lodlev].distance, sizeof(float), 1, fp);
			}
		}

		if (object->m_Flags & NxObject::mBILLBOARD)
		{
			fread(&object->m_BillboardType, sizeof(NxObject::BillboardType), 1, fp);
			fread(&object->m_BillboardOrigin, sizeof(float), 3, fp);
			fread(&object->m_PivotPos, sizeof(float), 3, fp);
			fread(&object->m_PivotAxis, sizeof(float), 3, fp);
		}

		if (object->m_Flags & NxObject::mHASVERSIONINFO)
		{
			unsigned int version;
			fread(&version, sizeof(unsigned int), 1, fp);

			if (version >= 0x0001)
			{
				// Read in relative info
				fread(&object->m_ParentCRC, sizeof(unsigned long), 1, fp);
				fread(&object->m_NumChildren, sizeof(unsigned long), 1, fp);
				
				object->m_ChildCRCs = new unsigned long[object->m_NumChildren];
				fread(&object->m_ChildCRCs, sizeof(unsigned long), object->m_NumChildren, fp);
			}
			// The mSS_NORMALS flag is not valid before object version 4
			if( version < 0x0004)
			{
				object->m_Flags &= ~NxObject::mSS_NORMALS;
			}
		}
	}



	return true;
}

bool WorldDB::SaveScene(char* filename)
{

	return true;
}
