/*
	FuncTrace.cpp
	FuncTrace parses c/cpp files and inserts debug lines to isolate last calls
	in the event of stack corruption problems

	Note: To save from writting a rather elaborate parser, we'll just consider
	      functions to be lines containing 1 "::", 1 "(" and 1 ")" and expect "{"
		  to follow immediately on the next line

	aml - 7-09-03
*/

#include <windows.h>
#include <stdio.h>
#include "ParseFuncs.h"

char funcName[2048];

bool ExtractName(char* name, char* src)
{
	char* pos    = strstr(src, "(");

	if (!pos)
		return false;

	if (pos == src)
		return false;
	
	char* endpos = pos;

	if (!pos)
		return false;

	// Scan backwards to the first space, tab or beginning of the line
	while(*pos != 9   &&
		  *pos != ' ' &&
		   pos != src)
	{
		pos--;
	}

	if (*pos == 9   ||
		*pos == ' ')
		pos++;

	int idx = 0;

	// Copy the characters into the name field
	while(pos != endpos)
	{
		name[idx++] = *pos;
		pos++;
	}

	name[idx] = 0;
	return true;
}

// Must have at least 2 letters in a row
bool IsNumber(char chr)
{
	if (chr >= '0' &&
		chr <= '9')
		return true;

	return false;
}

bool HasLetters(char* buf)
{
	int len = strlen(buf);
	int cnt = 0;

	for(int i = 0; i < len; i++)
	{
		if (tolower(buf[i]) >= 'a' &&
			tolower(buf[i]) <= 'z')
		{
			// Deny letters preceeded immediately by numbers
			if (i > 0 && IsNumber(buf[i - 1]))
				return false;

			return true;
		}
	}

	return false;
}

bool IsSymbol(char chr)
{
	char symbols[] = "!@#$%^&*()-+={}[]|\\/?<>,.`~:";

	int len = strlen(symbols);

	for(int i = 0; i < len; i++)
		if (chr == symbols[i])
			return true;

	return false;
}

void CopyTruncate(char* dest, char* src)
{
	int idx = 0;
	int len = strlen(src);

	for(int i = 0; i < len; i++)
	{
		if (src[i] == 0 ||
			src[i] == 9 ||
			src[i] == ' ')
		{
			dest[idx] = 0;
			return;
		}

		dest[idx++] = src[i];
	}

	dest[idx] = 0;
}

char* GetNextWord(char* src)
{
	int   len = strlen(src);
	char* pos = src;
	bool  bAcceptNext = false;

	for(int i = 0; i < len; i++)
	{
		if (src[i] == ',')
			return NULL;

		if (src[i] == ' ' ||
			src[i] == 9)
		{
			bAcceptNext = true;
		}
		else
		{
			if (bAcceptNext)
				return src + i;
		}
	}

	return NULL;
}

bool IsFuncParamList(char* lineBuf)
{
	if (strcmp(lineBuf, "void") == 0)
		return true;

	char* word = lineBuf;
	char  tword[1024];

	// If there are two words in a row its valid
	CopyTruncate(tword, word);

	if (HasLetters(tword))
	{
		word = GetNextWord(word);

		if (!word)
			return false;

		CopyTruncate(tword, word);

		if (HasLetters(tword))
			return true;
	}

	return false;
}

bool IsFunction(FILE* fpIn, FILE* fpOut, char* lineBuf)
{
	char params[2048] = "";

	if (strstr(lineBuf, ";"))
		return false;

	ExtractName(funcName, lineBuf);

	// Handle parameters spread over multiple lines
	if (CountChar(lineBuf, '(') == 1 &&
		CountChar(lineBuf, ')') == 0)
	{
		char* pos = strstr(lineBuf, "(");
		strcat(params, pos + 1);

		fputs(lineBuf, fpOut);
		fgets(lineBuf, 2048, fpIn);
		//fputs(lineBuf, fpOut);
		pos = strstr(lineBuf, ")");

		while(!pos)
		{
			strcat(params, lineBuf);
			fgets(lineBuf, 2048, fpIn);
			fputs(lineBuf, fpOut);
			pos = strstr(lineBuf, ")");
		}

		strcat(params, lineBuf);
		ReplaceStr(params, params, ")", "");

		StripToken(params);
		return IsFuncParamList(params);
	}
	else if (CountChar(lineBuf, '(') == 1 &&
		     CountChar(lineBuf, ')') == 1)
	{
		char* posS = strstr(lineBuf,"(") + 1;
		char* posE = strstr(posS,")");
		int idx = 0;

		while(posS != posE)
		{
			params[idx++] = *posS;
			posS++;
		}

		if (strlen(params) == 0)
		{
			if (strstr(lineBuf, "::"))
				return true;

			return false;
		}

		StripToken(params);
		return IsFuncParamList(params);
	}

	return false;
}

int main(int argc, char* argv[])
{
	if (argc == 1)
	{
		printf("Syntax: FuncTrace [filename]\n");
		return -1;
	}

	// Make a backup file first
	char backupName[256];
	strcpy(backupName, argv[1]);
	strcat(backupName, ".bak");

	if (!CopyFile(argv[1], backupName, FALSE))
	{
		printf("Failed to copy file '%s' to '%s'.", argv[1], backupName);
		return -2;
	}

	printf("Backed up as '%s'\n", backupName);

	FILE* fpSrc  = fopen(backupName, "r");

	if (!fpSrc)
	{
		printf("Failed to open '%s'.\n", backupName);
		return -3;
	}

	FILE* fpDest = fopen(argv[1], "w");

	if (!fpDest)
	{
		printf("Failed to open '%s'.\n", argv[1]);
		return -4;
	}

	printf("Processing '%s'...", argv[1]);

	char lineBuf[2048];		// 2kb
	char lineBuf2[2048];

	while(!feof(fpSrc))
	{
		fgets(lineBuf, 2048, fpSrc);

		// Check if this qualifies as a function
		/*
		if (CountStr(lineBuf, "::") == 1 &&
			CountChar(lineBuf, '(') == 1 &&
			CountChar(lineBuf, ')') == 1)
		*/
		if (IsFunction(fpSrc, fpDest, lineBuf))
		{
			long offset = ftell(fpSrc);
			fgets(lineBuf2, 2048, fpSrc);

			if (CountChar(lineBuf, '{') == 1 ||
				CountChar(lineBuf2, '{') == 1)
			{
				// Qualifies as function, insert start macro
				if (CountChar(lineBuf, '{') == 1)
				{
					fputs(lineBuf, fpDest);
					fprintf(fpDest, "\tFUNCSTART(\"%s\");\n", funcName);
					fputs(lineBuf2, fpDest);
				}
				else
				{
					fputs(lineBuf, fpDest);
					fputs(lineBuf2, fpDest);
					fprintf(fpDest, "\tFUNCSTART(\"%s\");\n", funcName);
				}

				// Scan to function end
				int nestLevel = CountChar(lineBuf, '{')  - CountChar(lineBuf, '}') +
					            CountChar(lineBuf2, '{') - CountChar(lineBuf2, '}');

				while(!feof(fpSrc))
				{
					long offset = ftell(fpSrc);
					fgets(lineBuf, 2048, fpSrc);

					char* posReturn = strstr(lineBuf, "return");
					if (posReturn)
					{
						int nTabs = CountChar(lineBuf, 9);
						char tabs[256] = "";
						for(int tbs = 0; tbs < nTabs; tbs++)
							strcat(tabs, "\t");

						fprintf(fpDest, "%s{\n%s\tFUNCEND(\"%s\");\n\t", tabs, tabs, funcName);
						fputs(lineBuf, fpDest);
						fprintf(fpDest, "%s}\n", tabs);

						/*
						fprintf(fpDest, "{  /////////////// Auto inserted return block ////////\nFUNCEND(\"%s\");\n", funcName);
						fputs(lineBuf, fpDest);
						fprintf(fpDest, "}  ///////////////////////////////////////////////////\n");
						*/
						nestLevel = nestLevel + CountChar(lineBuf, '{') - CountChar(lineBuf, '}');
					}
					else
						fseek(fpSrc, offset, SEEK_SET);

					char chr = fgetc(fpSrc);
					
					if (chr == '{')
						nestLevel++;
					
					if (chr == '}')
						nestLevel--;

					if (nestLevel == 0)
					{
						fprintf(fpDest, "\tFUNCEND(\"%s\");\n", funcName);
						fputc(chr, fpDest);
						break;
					}

					fputc(chr, fpDest);
				}

				continue;
			}
			else
				fseek(fpSrc, offset, SEEK_SET);
		}

		fputs(lineBuf, fpDest);
	}

	printf("\nCompleted instrumenting functions.\n");

	fclose(fpSrc);
	fclose(fpDest);

	return 0;
}
