//****************************************************************************
//* MODULE:         Tools/ConvertAssets
//* FILENAME:       ConvertAssets.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  ?/??/????
//****************************************************************************

#include <process.h>

#include <core/defines.h>
#include <core/string/cstring.h>

#include "FileDatabase.h"
#include "Utility.h"
#include "VirtualFile.h"
#include <direct.h>

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

#define vMAJOR_REV		2
#define vMINOR_REV		0

static char* proj_path;
static char* cmd_options;
static char start_path[_MAX_PATH];
static bool s_skip_exe_dependency_check = true;

IoUtils::CFileDatabase theFileDatabase(16);

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

inline int get_target_timestamp( const char* pFileName )
{
	return theFileDatabase.GetFileTime( pFileName );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CDependencyList
{
public:
	CDependencyList()
	{
		Clear();
	}

	virtual ~CDependencyList()
	{
	}

public:
	void			Clear()
	{
		m_fileCount = 0;
	}

	void			AddFile( char* pFileName )
	{
		Dbg_MsgAssert( m_fileCount < vMAX_DEPENDENCIES, ( "Too many files in dependency list" ) );
		m_fileName[m_fileCount] = pFileName;
		m_fileCount++;
	}

	bool			UpToDate( int timeStamp )
	{
		if ( s_skip_exe_dependency_check )
		{
			// this is so that we don't have to rebuild all
			// the files after a cosmetic change to sceneconv...
			return true;
		}

		for ( int i = 0; i < m_fileCount; i++ )
		{
			// can't use the file database, because
			// the tools will not be in the data directory
			int fileTime = Utils::GetTimeStamp( m_fileName[i].getString() );

			if ( fileTime == -1 )
			{
				// the file doesn't exist then the file is considered
				// to be out of date, just to be on the safe side...
				// (although i'm not really sure why the file wouldn't exist)
				return false;
			}

			if ( fileTime > timeStamp )
			{
				return false;
			}
		}

		return true;
	}

protected:
	enum
	{
		vMAX_DEPENDENCIES = 16
	};

	Str::String		m_fileName[vMAX_DEPENDENCIES];
	int				m_fileCount;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void lowercase( char* pSrc )
{
	while ( *pSrc )
	{
		if ( *pSrc >= 'A' && *pSrc <= 'Z' )
		{
			*pSrc = *pSrc - 'A' + 'a';
		}
		pSrc++;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool DependenciesUpToDate( const char* dep_file_path, int target_time )
{
	int checkingNumFiles = 0;

	IoUtils::CVirtualInputFile theDEPFile;

	if ( Utils::GetTimeStamp(dep_file_path) == -1 )
	{
		// the DEP file doesn't exist
		return false;
	}

	if ( !theDEPFile.Load( dep_file_path ) )
	{
		// the DEP file exists, but is empty
		return true;
	}

	char fileNameBuf[_MAX_PATH];
	while ( theDEPFile.ReadString( fileNameBuf, _MAX_PATH ) )
	{
		if ( Utils::GetTimeStamp( fileNameBuf ) > target_time )
		{
			return false;
		}
	
		checkingNumFiles++;
	}

//	printf( "Checked %d files in %s\n", checkingNumFiles, dep_file_path );
	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_font_converter( char* pTitle, char* pExeName, char* pRoot, char* pSourceExt, char* pTargetExt, int platform )
{
	printf( "%s ", pTitle );

	int num_items_to_update = 0;
	int num_items = 0;

	int exe_time = Utils::GetTimeStamp( pExeName );

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );		

		char infoFileName[_MAX_PATH];
		strcpy( infoFileName, pInfo->pFileName );
		lowercase( infoFileName );
		if ( strstr( infoFileName, "\\fonts" ) && !stricmp( ext, pSourceExt ) )
		{

			bool up_to_date = false;

			// See if it exists
			char dest_file_path[_MAX_PATH];
			sprintf( dest_file_path, "%s\\data\\fonts\\%s%s.%s", proj_path, cur_file, pTargetExt, Utils::GetPlatformExt(platform) );
			int target_time = get_target_timestamp( dest_file_path );

			if ( target_time != -1 )
			{
				// it exists, assume up to date
				up_to_date = true;

				// but set it not up to date if the source file is newer
				int src_time = pInfo->timestamp;
				if( src_time > target_time)
				{
					up_to_date = false;
				}
					
				// or if the tool itself is newer
				else if ( exe_time > target_time ) 
				{
					up_to_date = false;
				}
			}
			else
			{
				printf( "%s not found!\n", dest_file_path );
			}

			if ( !up_to_date )
			{
				char *args[6];
				args[0] = pExeName;
				args[1] = cmd_options;

				args[2] = pInfo->pFileName;
				args[3] = dest_file_path;

				char ascii_file_name[_MAX_PATH];
				sprintf( ascii_file_name, "%s%sascii.txt", drive, path );
				args[4] = ascii_file_name;

				args[5] = NULL;

				// Run the conversion utility
				int errCode = _spawnvp( _P_WAIT, args[0], args );
				if ( errCode != 0 )
				{
					printf( "%s returned error code (%d).\n", args[0], errCode );
					exit( 1 );
				}
			
				num_items_to_update++;
			}

			num_items++;
		}
	}

	printf( "%d of %d items need updating\n", num_items_to_update, num_items );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool try_to_delete( const char* pFileName, const char* drive, const char* path )
{
	bool success = false;

	if (remove(pFileName) == 0)
	{
		success = true;

		// Try to get rid of empty dirs
		char targetDir[_MAX_PATH];
		strcpy(targetDir, drive);
		strcat(targetDir, path);

		Utils::RemoveDirs(targetDir);
	}
	else
	{
		printf( "ERROR: Can't delete %s\n", pFileName );
	}

	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_wav_converter( char* pTitle, char* pExeName, char* pSourceExt, char* pTargetExt, int platform, char* pSubDir, CDependencyList* pDependencyList, bool is_music, bool generate_hed )
{
	// the tool that's being run should automatically
	// be a dependency, so add it to the list just
	// in case it wasn't added already...
	if ( pDependencyList )
	{
		pDependencyList->AddFile( pExeName );
	}

	printf( "%s ", pTitle );

	int num_items_to_update = 0;
	int num_items = 0;
	int num_items_deleted = 0;
	
	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init( 1 * 1024 * 1024 );

	int exe_time = Utils::GetTimeStamp( pExeName );

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		bool hasSourceExt = !stricmp( ext, pSourceExt );
		bool hasTargetExt = !stricmp( ext, pTargetExt );
		bool hasInterleavedExt = is_music && !stricmp( ext, ".ivg" );

		char infoFileName[_MAX_PATH];
		strcpy( infoFileName, pInfo->pFileName );
		lowercase( infoFileName );
		if ( (!pSubDir || strstr( infoFileName, pSubDir )) && (hasSourceExt || hasTargetExt || hasInterleavedExt) )
		{
			bool up_to_date = false;

			// See if it exists
			char dest_file_path[_MAX_PATH];

			if (hasSourceExt)
			{
				if (is_music)
				{
					// Check for left channel of vag
					sprintf( dest_file_path, "%s%s%s%s%s", drive, path, cur_file, "L", pTargetExt);
				}
				else
				{
					sprintf( dest_file_path, "%s%s%s%s", drive, path, cur_file, pTargetExt);
				}
				// Change wav directory to vag
				char *vagdir = strstr(dest_file_path, "\\wav\\");
				if ( !vagdir ) vagdir = strstr(dest_file_path, "\\Wav\\");
				if ( !vagdir ) vagdir = strstr(dest_file_path, "\\WAV\\");
				if (vagdir)
				{
					vagdir[1] = pTargetExt[1];
					vagdir[2] = pTargetExt[2];
					vagdir[3] = pTargetExt[3];
				}
			}
			else
			{
				if( platform == Utils::vPLATFORM_XBOX )
				{
					sprintf( dest_file_path, "%s%s%s%s", drive, path, cur_file, pSourceExt);
				}
				else if ( platform == Utils::vPLATFORM_PS2 )
				{
					if (is_music)
					{
						if (hasTargetExt)	// only do this to vag files
						{
							char *p_last_char = &(cur_file[strlen(cur_file) - 1]);
							Utils::Assert((toupper(*p_last_char) == 'L') || (toupper(*p_last_char) == 'R'), "Vag file %s doesn't end in L or R", cur_file);

							*p_last_char = '\0';		// Get rid or L or R (ok to modify cur_file since it is no longer used)
						}
						sprintf( dest_file_path, "%s%s%s%s", drive, path, cur_file, pSourceExt);
					}
					else
					{
						sprintf( dest_file_path, "%s%s%s%s", drive, path, cur_file, pSourceExt);
					}
					// Change vag directory to wav
					char *wavdir = strstr(dest_file_path, "\\vag\\");
					if ( !wavdir ) wavdir = strstr(dest_file_path, "\\Vag\\");
					if ( !wavdir ) wavdir = strstr(dest_file_path, "\\VAG\\");
					if (wavdir)
					{
						wavdir[1] = pSourceExt[1];
						wavdir[2] = pSourceExt[2];
						wavdir[3] = pSourceExt[3];
					}
				}
			}

			int target_time = get_target_timestamp( dest_file_path );

			if ( target_time != -1 )
			{
				// it exists, assume up to date
				up_to_date = true;

				// but set it not up to date if the source file is newer
				int src_time = pInfo->timestamp;
				if( src_time > target_time)
				{
					up_to_date = false;
				}
				else if ( !pDependencyList->UpToDate( target_time ) )
				{
					up_to_date = false;
				}
			}
			else
			{
				if(( platform != Utils::vPLATFORM_NGC ) && ( platform != Utils::vPLATFORM_XBOX ))
				{
					// Delete unnecessary target files
					if (hasTargetExt || hasInterleavedExt)
					{
						if ( try_to_delete(pInfo->pFileName, drive, path) )
						{
							num_items_deleted++;
							printf( "Deleted %s\n", pInfo->pFileName );
						}
					}
				}
			}

			if (hasSourceExt)
			{
				if ( !up_to_date )
				{
					if ( platform == Utils::vPLATFORM_NGC )
					{
						char *args[6];
						char arg[_MAX_PATH];
						char arg2[_MAX_PATH];
						int errCode;

						// Make sure directories are there.
						char * ps = pInfo->pFileName;
						char dirname[_MAX_PATH];
						char * pd = dirname;
						while ( *ps != '\0' )
						{
							while ( ( *ps != '\\' ) && ( *ps != '\0' ) ) *pd++ = *ps++;
							if ( *ps == '\0' ) break;
							*pd++ = *ps++;
							if ( *ps == ':' ) continue;
							*pd = '\0';

							char *dir = strstr( dirname, "\\wav\\" );
							if ( !dir ) dir = strstr( dirname, "\\Wav\\" );
							if ( !dir ) dir = strstr( dirname, "\\WAV\\" );
							if ( dir )
							{
								dir[1] = pTargetExt[1];
								dir[2] = pTargetExt[2];
								dir[3] = pTargetExt[3];
							}

//							printf( "\nMD: %s", dirname );
							_mkdir(dirname);
//
						}

						args[0] = pExeName;

						sprintf( arg, "%s", pInfo->pFileName );
						args[1] = arg;
						sprintf( arg2, "-d%s", pInfo->pFileName );
						char *dir = strstr( arg2, "\\wav\\" );
						if ( !dir ) dir = strstr( arg2, "\\Wav\\" ); 
						if ( !dir ) dir = strstr( arg2, "\\WAV\\" ); 
						if ( dir )
						{
							dir[1] = pTargetExt[1];
							dir[2] = pTargetExt[2];
							dir[3] = pTargetExt[3];
						}
						char *wav = strstr( arg2, ".wav" );
						if ( wav )
						{
							wav[1] = 't';
							wav[2] = 'm';
							wav[3] = 'p';
						}
						args[2] = arg2;
						args[3] = "-R22050";
						args[4] = "-n";
						args[5] = NULL;

//						printf( "\nConvert: %s %s %s %s %s", args[0], args[1], args[2], args[3], args[4] );

						// Run the conversion utility
						errCode = _spawnvp( _P_WAIT, args[0], args );
						if ( errCode != 0 )
						{
							printf( "%s returned error code (%d).\n", args[0], errCode  );
							exit( 1 );
						}

						strcpy( arg2, &arg2[2] );
						args[0] = "wav2dsp";
						args[1] = arg2;
						args[2] = NULL;

//						printf( "\nConvert: %s %s %s", args[0], args[1], args[2] );

						// Run the conversion utility
						errCode = _spawnvp( _P_WAIT, args[0], args );
						if ( errCode != 0 )
						{
							printf( "%s returned error code (%d).\n", args[0], errCode  );
							exit( 1 );
						}

						num_items_to_update++;
					}
					else
					{
						theOutputFile.WriteString( pInfo->pFileName, true );
//						g << pInfo->pFileName << endl;
						num_items_to_update++;
					}
				}

				num_items++;
			}
		}
	}

//	g.close();

	char temp_file_name[_MAX_PATH];
	sprintf( temp_file_name, "%s\\bin\\win32\\ca.txt", proj_path );
	if ( !theOutputFile.Save( temp_file_name, IoUtils::vTEXT_MODE ) )
	{
		Utils::Assert( 0, "Couldn't open temp file %s", temp_file_name );
	}

	// Check for hed file
	if (generate_hed)
	{
		char hed_file_name[_MAX_PATH];
		if (is_music)
		{
			sprintf( hed_file_name, "%s\\data\\music\\musichost.hed", proj_path );
		}
		else
		{
			if ( platform == Utils::vPLATFORM_NGC )
			{
				sprintf( hed_file_name, "%s\\data\\streams\\streamsngc.hed", proj_path );
			}
			else
			{
				sprintf( hed_file_name, "%s\\data\\streams\\streamshost.hed", proj_path );
			}
		}
		if (!theFileDatabase.FileExists(hed_file_name))
		{
			num_items_deleted++;	// Make sure to rebuild hed file since it doesn't exist
		}
	}

	printf( "%d of %d items need updating\n", num_items_to_update, num_items );
	
	if ( num_items_to_update || num_items_deleted )
	{
		char *args[7];
		char arg[_MAX_PATH];
		int errCode;

		if( platform == Utils::vPLATFORM_XBOX )
		{
			if( num_items_to_update )
			{
				args[0] = pExeName;
				args[1] = cmd_options;

				sprintf( arg, "-b%s", temp_file_name );
				args[2] = arg;

				// Is this a (stereo) music or cutscene file, a regular (mono) stream, or a sound?
				if( strstr( pSubDir, "music" ))
				{
					args[3] = "-D";
					args[4] = NULL; 
				}
				else if( strstr( pSubDir, "streams" ))
				{
					args[3] = "-D";
					args[4] = NULL; 
				}
				else if( strstr( pSubDir, "sounds" ))
				{
					// args[1] will be '-px', but we want it to be '-pxc' so that all chunks will be copied to the .pcm file.
					args[1] = "-pxc";
					args[3] = "-D";
					args[4] = NULL;
				}
				else
				{
					printf( "%s is a bad sub directory\n", pSubDir );
					exit( 1 );
				}

				// Run the conversion utility.
				errCode = _spawnvp( _P_WAIT, args[0], args );
				if ( errCode != 0 )
				{
					printf( "%s returned error code (%d).\n", args[0], errCode  );
					exit( 1 );
				}
			}
		}
        else if ( platform == Utils::vPLATFORM_PS2 )
		{
			if ( num_items_to_update )
			{
				args[0] = pExeName;
				args[1] = cmd_options;

				sprintf( arg, "-b%s", temp_file_name );
				args[2] = arg;

				if (is_music)
				{
					args[3] = NULL;
				} else
				{
					args[3] = "-s";
					args[4] = "-L";
					args[5] = "-o22050";
					args[6] = NULL; 
				}

				// Run the conversion utility
				errCode = _spawnvp( _P_WAIT, args[0], args );
				if ( errCode != 0 )
				{
					printf( "%s returned error code (%d).\n", args[0], errCode  );
					exit( 1 );
				}
			}
		}

		// Run the hed generation tool
		if (generate_hed)
		{
			char exeName[_MAX_PATH];
			if (is_music)
			{
				sprintf( exeName, "%s\\bin\\win32\\makemusic.bat", proj_path );
			}
			else
			{
				sprintf( exeName, "%s\\bin\\win32\\makestreams.bat", proj_path );
			}
			args[0] = (char*)exeName;
			if ( platform == Utils::vPLATFORM_NGC )
			{
				args[1] = "update";
			}
			else
			{
				args[1] = "hed";
			}
			args[2] = NULL; 

			// Run the conversion utility
			errCode = _spawnvp( _P_WAIT, args[0], args );
			if ( errCode != 0 )
			{
				printf( "%s returned error code (%d).\n", args[0], errCode  );
				exit(1);
			}
		}

		// need to rebuild the file list, in case any new FAM files were created
		printf( "Re-scanning root directory: %s...\n", start_path );
		theFileDatabase.RefreshFileList( start_path );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/
bool is_target_file( const char* pFileName, char* pSrcExt, char* pTargetExt, int platform )
{
	char filename[_MAX_PATH];
	strcpy( filename, pFileName );
	lowercase( filename );

	char srcExt[_MAX_PATH];
	strcpy( srcExt, pSrcExt );
	lowercase( srcExt );

	char targetExt[_MAX_PATH];
	sprintf( targetExt, "%s.%s", pTargetExt, Utils::GetPlatformExt(platform) );
	lowercase( targetExt );

	if ( strstr( filename, targetExt ) && strstr( filename, srcExt )  )
	{
		return true;
	}

	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_batch_converter( char* pTitle, char* pExeName, char* pSourceExt, char* pTargetExt, int platform, CDependencyList* pDependencyList, char* pExtraArgs = NULL, char** pExtraDependencies = NULL, bool check_dep_file = false )
{
	// the tool that's being run should automatically
	// be a dependency, so add it to the list just
	// in case it wasn't added already...
	if ( pDependencyList )
	{
		pDependencyList->AddFile( pExeName );
	}

	printf( "%s ", pTitle );

	int num_items_to_update = 0;
	int num_items_to_delete = 0;
	int num_items = 0;
	
	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init( 1 * 1024 * 1024 );

	int exe_time = Utils::GetTimeStamp( pExeName );

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		if ( is_target_file( pInfo->pFileName, pSourceExt, pTargetExt, platform ) )
		{
			// if the target exists, but not the source,
			// then delete it
			char src_file_path[_MAX_PATH];
			sprintf( src_file_path, "%s%s%s", drive, path, cur_file );
			int src_time = get_target_timestamp( src_file_path );

			if ( src_time == -1 )
			{
				// if it doesn't contain the source
				// Delete unnecessary target files
	
				lowercase(cur_file);
				// Don't delete files with the extension _net. 
				// as they don't fit into the normal scheme of things
				if (strstr(cur_file,"_net.") == 0)
				{
					if ( try_to_delete(pInfo->pFileName, drive, path) )
					{
						num_items_to_delete++;
						printf( "Deleting unnecessary target file %s\n", pInfo->pFileName );
					}
				}
			}
		}
		else if ( !stricmp( ext, pSourceExt ) )
		{
			bool up_to_date = false;

			// See if it exists
			char dest_file_path[_MAX_PATH];
			sprintf( dest_file_path, "%s%s%s%s.%s", drive, path, cur_file, pTargetExt, Utils::GetPlatformExt(platform) );

			int target_time = get_target_timestamp( dest_file_path );

			if ( target_time != -1 )
			{
				// it exists, assume up to date
				up_to_date = true;

				char** pCurrentDependency = pExtraDependencies;
				if ( pCurrentDependency )
				{
					while ( *pCurrentDependency )
					{
						char tex_file_path[_MAX_PATH];
						sprintf( tex_file_path, "%s%s%s%s", drive, path, cur_file, *pCurrentDependency );
						int tex_time_stamp = Utils::GetTimeStamp( tex_file_path );
						if ( ( tex_time_stamp != -1 ) && ( tex_time_stamp > target_time ) )
						{
							up_to_date = false;
						}
						pCurrentDependency++;
					}
				}

				// if there's a DEP file, then check its contents
				if ( check_dep_file )
				{
					char dep_file_path[_MAX_PATH];
					sprintf( dep_file_path, "%s%s%s.dep.%s", drive, path, cur_file, Utils::GetPlatformExt(platform) );
					if ( Utils::GetTimeStamp( dep_file_path ) != -1 )
					{
						if ( !DependenciesUpToDate( dep_file_path, target_time ) )
						{
							printf( "DEP file %s contains out of date files\n", dep_file_path );
							up_to_date = false;
						}
					}
					else
					{
						// no dep file found, so force rebuild
						up_to_date = false;
					}
				}

				// but set it not up to date if the source file is newer
				int src_time = pInfo->timestamp;
				if( src_time > target_time)
				{
					up_to_date = false;
				}	
				else if ( pDependencyList && !pDependencyList->UpToDate( target_time ) )
				{
					up_to_date = false;
				}
			}
			else
			{
//				printf( "%s not found!\n", dest_file_path );
			}

			if ( !up_to_date )
			{
				theOutputFile.WriteString( pInfo->pFileName, true );
//				g << pInfo->pFileName << endl;
				num_items_to_update++;
			}

			num_items++;
		}
	}
	
//	g.close();

	char temp_file_name[_MAX_PATH];
	sprintf( temp_file_name, "%s\\bin\\win32\\ca.txt", proj_path );
	if ( !theOutputFile.Save( temp_file_name, IoUtils::vTEXT_MODE ) )
	{
		Utils::Assert( 0, "Couldn't open temp file %s", temp_file_name );
	}

	printf( "%d of %d items need updating\n", num_items_to_update, num_items );

	if ( num_items_to_update )
	{
		char *args[5];
		args[0] = pExeName;
		args[1] = cmd_options;

		char arg[_MAX_PATH];
		sprintf( arg, "-b%s", temp_file_name );
		args[2] = arg;

		args[3] = pExtraArgs;

		args[4] = NULL; 

		// Run the conversion utility
		int errCode = _spawnvp( _P_WAIT, args[0], args );
		if ( errCode != 0 )
		{
			printf( "%s returned error code (%d).\n", args[0], errCode  );
			exit( 1 );
		}
	}

	if ( num_items_to_delete )
	{
		printf( "%d of %d items deleted\n", num_items_to_delete, num_items );

		// should rebuild the file database here
		// (or deleting files should update the file database manually)
		printf( "Re-scanning root directory: %s...\n", start_path );
		theFileDatabase.RefreshFileList( start_path );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_file_converter( char* pTitle, char* pExeName, char* pSourceExt, char* pTargetExt, int platform, CDependencyList* pDependencyList )
{
	// the tool that's being run should automatically
	// be a dependency, so add it to the list just
	// in case it wasn't added already...
	if ( pDependencyList )
	{
		pDependencyList->AddFile( pExeName );
	}

	printf( "%s ", pTitle );

	int num_items_to_update = 0;
	int num_items_to_delete = 0;
	int num_items = 0;

	int exe_time = Utils::GetTimeStamp( pExeName );

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		if ( is_target_file( pInfo->pFileName, pSourceExt, pTargetExt, platform ) )
		{
			// if the target exists, but not the source,
			// then delete it
			char src_file_path[_MAX_PATH];
			sprintf( src_file_path, "%s%s%s", drive, path, cur_file );
			int src_time = get_target_timestamp( src_file_path );

			if ( src_time == -1 )
			{
				// if it doesn't contain the source
				// Delete unnecessary target files
	
				if ( try_to_delete(pInfo->pFileName, drive, path) )
				{
					num_items_to_delete++;
					printf( "Deleting unnecessary target file %s\n", pInfo->pFileName );
				}
			}
		}
		else if ( !stricmp( ext, pSourceExt ) )
		{
			bool up_to_date = false;

			// See if it exists
			char dest_file_path[_MAX_PATH];
			sprintf( dest_file_path, "%s%s%s%s.%s", drive, path, cur_file, pTargetExt, Utils::GetPlatformExt(platform) );
			int target_time = get_target_timestamp( dest_file_path );

			if ( target_time != -1 )
			{
				// it exists, assume up to date
				up_to_date = true;

				// but set it not up to date if the source file is newer
				int src_time = pInfo->timestamp;
				if( src_time > target_time)
				{
					up_to_date = false;
				}
					
				// or if the tool itself is newer
				else if ( exe_time > target_time ) 
				{
					up_to_date = false;
				}
				else if ( pDependencyList && !pDependencyList->UpToDate( target_time ) )
				{
					up_to_date = false;
				}
			}
			else
			{
//				printf( "%s not found!\n", dest_file_path );
			}

			if ( !up_to_date )
			{
				char *args[4];
				args[0] = pExeName;
				args[1] = cmd_options;

				char srcFileName[_MAX_PATH];
				sprintf( srcFileName, "-f%s", pInfo->pFileName );
				args[2] = srcFileName;

				args[3] = NULL; 

				// Run the conversion utility
				int errCode = _spawnvp( _P_WAIT, args[0], args );
				if ( errCode != 0 )
				{
					printf( "%s returned error code (%d).\n", args[0], errCode  );
					exit(1);
				}
			
				num_items_to_update++;
			}

			num_items++;
		}
	}

	printf( "%d of %d items need updating\n", num_items_to_update, num_items );

	if ( num_items_to_delete )
	{
		printf( "%d of %d items deleted\n", num_items_to_delete, num_items );

		// should rebuild the file database here
		// (or deleting files should update the file database manually)
		printf( "Re-scanning root directory: %s...\n", start_path );
		theFileDatabase.RefreshFileList( start_path );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void run_once( const char* pTitle, const char* pExeName, char* pArg)
{
	printf( "%s ", pTitle );

	char *args[4];
	args[0] = (char*)pExeName;
	args[1] = cmd_options;

	args[2] = pArg;

	args[3] = NULL; 

	// Run the conversion utility
	int errCode = _spawnvp( _P_WAIT, args[0], args );
	if ( errCode != 0 )
	{
		printf( "%s returned error code (%d).\n", args[0], errCode  );
		exit(1);
	}

	printf( "Done.\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

/*
void run_on_each( char* pTitle, char* pExeName, char* pSourceExt )
{
	printf( "%s ", pTitle );

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		// standardize it so that the strstrs work properly
		lowercase( cur_file );

		if ( !stricmp( ext, pSourceExt ) )
		{
			char *args[4];
			args[0] = (char*)pExeName;
			args[1] = pInfo->pFileName;
			args[2] = cur_file; //NULL; //cmd_options;
			args[3] = cmd_options;
			args[4] = NULL;

			// Run the conversion utility
			int errCode = _spawnvp( _P_WAIT, args[0], args );
			if ( errCode != 0 )
			{
				printf( "%s returned error code (%d).\n", args[0], errCode  );
				exit(1);
			}
		}
	}

	printf( "Done.\n" );
}
*/

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_title()
{
#ifdef _DEBUG
	printf( "\nConvertAssets v%d.%d (Debug) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#else
	printf( "\nConvertAssets v%d.%d (Release) Built %s %s\n", vMAJOR_REV, vMINOR_REV ,__DATE__, __TIME__ );
#endif
	printf( "Neversoft Entertainment\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void remove_unused_fams( int platform )
{
	int num_items_to_delete = 0;
	int num_lip_wavs_found = 0;

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		lowercase( cur_file );
		lowercase( ext );

		// if a _LIP.WAV file exists w/o an accompanying FAM file,
		// then delete the .VAG file so that the .FAM file will
		// be rebuilt in a later step
		if ( !strcmp( ext, ".wav" ) && strstr(cur_file, "_lip") )
		{
			// used later for removing unnecessary FAMs
			num_lip_wavs_found++;

			uint32 checksum = Crc::GenerateCRCFromString(cur_file);

			// build the fam file name
			char famFileName[_MAX_PATH];
			strcpy( famFileName, pInfo->pFileName );

			// replace the "data\streams" with "data\fam"
			lowercase(famFileName);
			char* pDirectory = strstr( famFileName, "data\\streams" );
			if ( !pDirectory )
			{
				Utils::Assert( 0, "Expected to find 'data\\streams' in %s", famFileName );
			}

			// look for a filename with the checksum as the name
			bool found = false;
			sprintf( pDirectory, "data\\fam\\%08x.fam", checksum );
			found = ( Utils::GetTimeStamp( famFileName ) != -1 );

			if ( !found )
			{
				// build the fam file name
				char targetFileName[_MAX_PATH];
				strcpy( targetFileName, pInfo->pFileName );

				switch ( platform )
				{
					case Utils::vPLATFORM_XBOX:
					{
						char* pDirectory = NULL;
						while ( pDirectory = strstr( targetFileName, "wav" ) )
						{
							pDirectory[0] = 'p';
							pDirectory[1] = 'c';
							pDirectory[2] = 'm';
						}
					}
					break;
					case Utils::vPLATFORM_PS2:
					{
						char* pDirectory = NULL;
						while ( pDirectory = strstr( targetFileName, "wav" ) )
						{
							pDirectory[0] = 'v';
							pDirectory[1] = 'a';
							pDirectory[2] = 'g';
						}
					}
					break;
					case Utils::vPLATFORM_NGC:
					{
						char* pDirectory = NULL;
						while ( pDirectory = strstr( targetFileName, "wav" ) )
						{
							pDirectory[0] = 'd';
							pDirectory[1] = 's';
							pDirectory[2] = 'p';
						}
					}
					break;
					default:
						Utils::Assert( 0, "Unrecognized platform %d", platform );
				}

				_splitpath( targetFileName, drive, path, cur_file, ext );
				if ( Utils::GetTimeStamp( targetFileName ) != -1 )
				{
					if ( try_to_delete(targetFileName, drive, path) )
					{
						num_items_to_delete++;
						printf( "Couldn't find %s...  deleting %s to force rebuild of FAM file\n", famFileName, targetFileName );
					}
				}
			}
		}
	}

	uint32* pLipWavChecksums = NULL;
	if ( num_lip_wavs_found )
	{
		pLipWavChecksums = new uint32[num_lip_wavs_found];
		int lip_wav_index = 0;
	
		// reset list traversal
		theFileDatabase.IterateStart();

		while ( pInfo = theFileDatabase.IterateNext() )
		{
			char ext[_MAX_EXT];
			char cur_file[_MAX_FNAME];
			char drive[_MAX_PATH];
			char path[_MAX_PATH];
	
			_splitpath( pInfo->pFileName, drive, path, cur_file, ext );
	
			lowercase( cur_file );
			lowercase( ext );

			// build list of LIP WAV checksums
			if ( !strcmp( ext, ".wav" ) && strstr(cur_file, "_lip") )
			{
				pLipWavChecksums[lip_wav_index++] = Crc::GenerateCRCFromString( cur_file );
			}
		}
	}

	// reset list traversal
	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );
		
		lowercase( cur_file );
		lowercase( ext );

		// if a FAM file exists w/o an corresponding _LIP.WAV file,
		// then delete it
		if ( !strcmp( ext, ".fam" ) )
		{
			char checksumString[256];
			sprintf( checksumString, "0x%s", cur_file );
			uint32 checksum;
			char dummy[_MAX_PATH];
			
			if ( sscanf( checksumString, "0x%x%s", &checksum, &dummy ) != 1 )
			{
				// it's a string, not number filename
				continue;
			}
	
			bool found = false;

			for ( int i = 0; i < num_lip_wavs_found; i++ )
			{
				Utils::Assert( pLipWavChecksums != NULL, "No lip wav checksums?" );
				if ( pLipWavChecksums[i] == checksum )
				{
					found = true;
				}
			}

			if ( !found )
			{
				if ( try_to_delete(pInfo->pFileName, drive, path) )
				{
					num_items_to_delete++;
					printf( "Deleted %s (%08x)\n", pInfo->pFileName, checksum );
				}
			}
		}
	}

	if ( pLipWavChecksums )
	{
		delete[] pLipWavChecksums;
	}
	
	if ( num_items_to_delete )
	{
		printf( "%d items deleted\n", num_items_to_delete );

		// should rebuild the file database here
		// (or deleting files should update the file database manually)
		printf( "Re-scanning root directory: %s...\n", start_path );
		theFileDatabase.RefreshFileList( start_path );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void remove_unused_files( char** ppSourceModelExtensions, char* pTargetExtension, int platform )
{
	int num_items_to_delete = 0;

	IoUtils::SFileInfo* pInfo = NULL;

	theFileDatabase.IterateStart();

	while ( pInfo = theFileDatabase.IterateNext() )
	{
		char ext[_MAX_EXT];
		char cur_file[_MAX_FNAME];
		char drive[_MAX_PATH];
		char path[_MAX_PATH];

		_splitpath( pInfo->pFileName, drive, path, cur_file, ext );

		lowercase( cur_file );
		// First need to find the base filename, removing any _net extension

/*
		// MICK:  This more complex behaviour ended up being TOO complex,
		// as there was a file with the extension _NET naturally, so it ended
		// up deleting that one....
		// so I made it simppler by just not deleting files with the string "_net." in them

		// If a file ends in _net
		// then remove the _net to get the original source file name
		// (the "_net" is appended during the sceneconv process as a xxx.scn file is 
		// converted both into a xxx.scn.ps2 and a xxx_net.scn.ps2 file)
		int	cf = strlen(cur_file);
		// for files with a double extension like level_net.scn.ps2
		// then cur_file will have the .ps2 entension removed
		// soskip back to the first . from the end (like in level_net.scn)
		while (cf > 0 && cur_file[cf] != '.')
		{
			cf--;
		}
		// if there was a second extension
		if (cf >0)
		{
			// then need to check if BEFORE that was _net
			if (strncmp(&cur_file[cf-4],"_net",4) == 0)
			{
				// if there was, then we need to overwrite it with the previous extension
				// copy the extension into an temp buffer
				char temp_ext[128];
				strcpy(temp_ext,&cur_file[cf]);
				// then copy that extension to the point at which _net starts								
				strcpy(&cur_file[cf-4],temp_ext);
			}
		}
*/		
		
		char targetExt[_MAX_PATH];
		if ( platform == Utils::vPLATFORM_NONE )
		{
			sprintf( targetExt, "%s", pTargetExtension );
		}
		else
		{
			sprintf( targetExt, "%s.%s", pTargetExtension, Utils::GetPlatformExt(platform) );
		}
		lowercase( targetExt );

		// if a file with the target extension exists,
		// but the *none* of the source files exist
		if ( strstr( pInfo->pFileName, targetExt ) )
		{
			bool source_file_exists = false;
			
			char srcName[_MAX_PATH];
			char** pCurr = ppSourceModelExtensions;
			while ( *pCurr )
			{
				sprintf( srcName, "%s%s%s", drive, path, cur_file );
				char* pExt = strstr( srcName, "." );
				if ( pExt )
				{
					*pExt = NULL;
				}
				strcat(	srcName, *pCurr );
//				printf( "Looking for %s\n", srcName );
				lowercase( srcName );
				if ( get_target_timestamp( srcName ) != -1 )
				{
					source_file_exists = true;
					break;
				}
				pCurr++;
			}

			if ( !source_file_exists )
			{
				// Simply don't delete it if it ends with a _net. extension
				if (strstr(cur_file,"_net.") == 0)
				{
					if ( try_to_delete(pInfo->pFileName, drive, path) )
					{
						num_items_to_delete++;
						printf( "Deleted %s as %s no longer exists (cur_file = [%s])\n", pInfo->pFileName,srcName, cur_file );
					}
				}
			}
		}
	}
	
	if ( num_items_to_delete )
	{
		printf( "%d items deleted\n", num_items_to_delete );

		// should rebuild the file database here
		// (or deleting files should update the file database manually)
		printf( "Re-scanning root directory: %s...\n", start_path );
		theFileDatabase.RefreshFileList( start_path );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void main( int argc, char *argv[] )
{	
	char tempExeName[_MAX_PATH];

	print_title();

	int platform = Utils::vPLATFORM_NONE;

	char* pPlatform;
	pPlatform = Utils::TestArg( argc, argv, 'P' );

	if ( pPlatform )
	{
		switch ( *pPlatform )
		{
			case 'X':
			case 'x':
				platform = Utils::vPLATFORM_XBOX;
				break;
			case 'G':
			case 'g':
				platform = Utils::vPLATFORM_NGC;
				break;
			case 'P':
			case 'p':
				platform = Utils::vPLATFORM_PS2;
				break;
		}
	}

	if( platform == Utils::vPLATFORM_NONE ) 
	{
		printf( "Usage: ConvertAssets -p[p: Playstation/g: Gamecube/x: Xbox]\n" );
		printf( "Example : ConvertAssets -pp\n" );
		exit(1);
	}

	proj_path = getenv( "PROJ_ROOT" );
	if( proj_path == NULL )
	{		
		printf( "You must first define your PROJ_ROOT environment variable\n" );
		exit(1);
	}

	if ( Utils::TestArg( argc, argv, 'D' ) )
	{
		printf( "(*******************************************************************)\n" );
		printf( "(**************** Skipping .EXE dependency check... ****************)\n" );
		printf( "(*******************************************************************)\n" );
		s_skip_exe_dependency_check = true;
	}
	else
	{
		s_skip_exe_dependency_check = false;
	}

	cmd_options = argv[1];	

	char exeName[_MAX_PATH];

	// build the entire file list here...
	sprintf( start_path, "%s\\data", proj_path ); 
	printf( "Scanning root directory: %s...\n", start_path );
	theFileDatabase.RefreshFileList( start_path );
	printf( "Found %d files...\n", theFileDatabase.GetSize() );

	printf( "Removing unnecessary files...\n" );
	
	char* pQSourceExtensions[] = {
		".q",
		".qn",
		NULL
	};

	remove_unused_files( pQSourceExtensions, ".qb", Utils::vPLATFORM_NONE );

	char* pModelSourceExtensions[] = {
		".mdl",
		".skin",
		".scn",
		NULL
	};

	remove_unused_files( pModelSourceExtensions, ".geom", platform );
	remove_unused_files( pModelSourceExtensions, ".tex", platform );
	remove_unused_files( pModelSourceExtensions, ".usg", platform );
	remove_unused_files( pModelSourceExtensions, ".col", platform );
	remove_unused_files( pModelSourceExtensions, ".cas", platform );
	remove_unused_files( pModelSourceExtensions, ".wgt", platform );

	char* pImageSourceExtensions[] = {
		".png",
		NULL
	};

	remove_unused_files( pImageSourceExtensions, ".img", platform );

//	GJ TODO:  The following doesn't work, because fonts are exported to
//	a different directory.
//	remove_unused_files( pImageSourceExtensions, ".fnt", platform );

//	FAMs are handled specially because of their odd naming convention
//	and because they are exported to a separate directory
	remove_unused_fams( platform );

	// TODO:  Make the exe names and the file extensions scriptable...

	sprintf( exeName, "%s\\bin\\win32\\convertsounds.bat", proj_path );
	run_once( "Converting sounds...", exeName, NULL );

//	sprintf( exeName, "%s\\bin\\win32\\qcomp.bat", proj_path );
//	sprintf( args, "-update bin\\win32" );
//	run_once( "Compiling .Q and .QN files...", exeName, args );

	sprintf( exeName, "%s\\bin\\win32\\makefont.exe", proj_path );
	run_font_converter( "Converting fonts...", exeName, "data\\fonts", ".png", ".fnt", platform );

	CDependencyList theSceneConvDependencyList;
	char dllName[256];
	switch ( platform )
	{
		case Utils::vPLATFORM_XBOX:
			sprintf( dllName, "%s\\bin\\win32\\%s", proj_path, "xboxdll.dll" );
			theSceneConvDependencyList.AddFile( dllName );
			break;
		case Utils::vPLATFORM_NGC:
			sprintf( dllName, "%s\\bin\\win32\\%s", proj_path, "ngcdll.dll" );
			theSceneConvDependencyList.AddFile( dllName );
			break;
		case Utils::vPLATFORM_PS2:
			sprintf( dllName, "%s\\bin\\win32\\%s", proj_path, "ps2dll.dll" );
			theSceneConvDependencyList.AddFile( dllName );
			break;
		default:
			// automatically added by run_batch_converter
//			sprintf( dllName, "%s\\bin\\win32\\%s", proj_path, "sceneconv.exe" );
//			theSceneConvDependencyList.AddFile( dllName );
			break;
	}

	char* pExtraModelDependencies[] = {
		".tex",
		".usg",
		".wgt",
		NULL
	};

	sprintf( exeName, "%s\\bin\\win32\\sceneconv.exe", proj_path );
	run_batch_converter( "Converting scenes...", exeName, ".scn", ".scn", platform, &theSceneConvDependencyList, NULL, pExtraModelDependencies );

	sprintf( exeName, "%s\\bin\\win32\\sceneconv.exe", proj_path );
	run_batch_converter( "Converting skins...", exeName, ".skin", ".skin", platform, &theSceneConvDependencyList, NULL, pExtraModelDependencies );

	sprintf( exeName, "%s\\bin\\win32\\sceneconv.exe", proj_path );
	run_batch_converter( "Converting models...", exeName, ".mdl", ".mdl", platform, &theSceneConvDependencyList, NULL, pExtraModelDependencies );

	CDependencyList theSkeletonDependencyList;

	sprintf( exeName, "%s\\bin\\win32\\skeconv.exe", proj_path );
	run_batch_converter( "Converting skeletons...", exeName, ".ske", ".ske", platform, &theSkeletonDependencyList );

	CDependencyList thePngDependencyList;

	sprintf( exeName, "%s\\bin\\win32\\pngconv.exe", proj_path );
	run_batch_converter( "Converting textures...", exeName, ".png", ".img", platform, &thePngDependencyList );

	CDependencyList theAnimDependencyList;
	sprintf( tempExeName, "%s\\data\\anims\\standardkeyq.bin", proj_path );
	theAnimDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\data\\anims\\standardkeyt.bin", proj_path );
	theAnimDependencyList.AddFile( tempExeName );

	sprintf( exeName, "%s\\bin\\win32\\animconv.exe", proj_path );
	run_batch_converter( "Converting anims...", exeName, ".ska", ".ska", platform, &theAnimDependencyList );

	CDependencyList theCutsceneDependencyList;
	sprintf( tempExeName, "%s\\bin\\win32\\animconv.exe", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\sceneconv.exe", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	theCutsceneDependencyList.AddFile( dllName );
	sprintf( tempExeName, "%s\\bin\\win32\\cifconv.exe", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\qcomp.exe", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\qcompall.bat", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\libconv.ini", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\libconv_ps2.ini", proj_path );
	theCutsceneDependencyList.AddFile( tempExeName );
//	proposed format for adding a new type of wildcard dependency
//	this makes it dependent on <sourcefile> AND the list
//	inside <sourcefile>.dep
//	char ruleString[512];
//	sprintf( ruleString, "%%s (%%s.dep)", proj_path );
//	theCutsceneDependencyList.AddRule( ruleString );

	sprintf( exeName, "%s\\bin\\win32\\libconv.exe", proj_path );
	char libConvIni[512];
	strcpy( libConvIni, "" );
	if ( platform == Utils::vPLATFORM_PS2 )
	{
		// playstation uses a special ini file.
		sprintf( libConvIni, "-i%s\\bin\\win32\\libconv_ps2.ini", proj_path );
	}
	run_batch_converter( "Converting cutscenes...", exeName, ".cut", ".cut", platform, &theCutsceneDependencyList, libConvIni, NULL, true );

	switch ( platform )
	{
		case Utils::vPLATFORM_XBOX:
		{
			CDependencyList theStreamDependencyList, theMusicDependencyList, theSoundDependencyList;
			sprintf( exeName, "%s\\bin\\win32\\Nswav2vag.exe", proj_path );
			run_wav_converter( "Converting streams...", exeName, ".wav", ".pcm", platform, "\\streams", &theStreamDependencyList, false, false );
			run_wav_converter( "Converting music...", exeName, ".wav", ".pcm", platform, "\\music", &theMusicDependencyList, false, false );
			run_wav_converter( "Converting sounds...", exeName, ".wav", ".pcm", platform, "\\sounds", &theSoundDependencyList, false, false );
			break;
		}
		case Utils::vPLATFORM_NGC:
			{
				CDependencyList theStreamDependencyList, theSoundDependencyList;
				sprintf( exeName, "%s\\bin\\win32\\Nswav2vag.exe", proj_path );
				run_wav_converter( "Converting streams...", exeName, ".wav", ".dsp", platform, "\\streams", &theStreamDependencyList, false, true );
				run_wav_converter( "Converting sounds...", exeName, ".wav", ".dsp", platform, "\\sounds", &theSoundDependencyList, false, false );
//				run_wav_converter( "Converting music...", exeName, ".wav", ".dtk", platform, "\\music", &theMusicDependencyList, true );
			}
			break;
		case Utils::vPLATFORM_PS2:
			{
				CDependencyList theStreamDependencyList, theMusicDependencyList, theSoundDependencyList;
				sprintf( exeName, "%s\\bin\\win32\\Nswav2vag.exe", proj_path );
				run_wav_converter( "Converting streams...", exeName, ".wav", ".vag", platform, "\\streams", &theStreamDependencyList, false, true );
				run_wav_converter( "Converting music...", exeName, ".wav", ".vag", platform, "\\music", &theMusicDependencyList, true, true );
				run_wav_converter( "Converting sounds...", exeName, ".wav", ".vag", platform, "\\sounds", &theSoundDependencyList, false, false );
			}
			break;
		default:
			break;
	}

	// this needs to come after the streams, which will generate FAM files
	CDependencyList theFacialAnimDependencyList;
	sprintf( tempExeName, "%s\\data\\anims\\standardkeyq.bin", proj_path );
	theFacialAnimDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\data\\anims\\standardkeyt.bin", proj_path );
	theFacialAnimDependencyList.AddFile( tempExeName );
	sprintf( tempExeName, "%s\\bin\\win32\\animconv.ini", proj_path );
	theFacialAnimDependencyList.AddFile( tempExeName );
	sprintf( exeName, "%s\\bin\\win32\\animconv.exe", proj_path );
	run_batch_converter( "Converting frame amplitude files...", exeName, ".fam", ".fam", platform, &theFacialAnimDependencyList );
}