//****************************************************************************
//* MODULE:         Tools/CifConv
//* FILENAME:       main.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  1/29/2003
//****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include <core/defines.h>
#include <core/math.h>

#include "Converter.h"
#include "Utility.h"
#include "VirtualFile.h"

#include "..\LibMaker\LibMaker.h"

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

#define	vMAJOR_REVISION		1
#define	vMINOR_REVISION		0
#define vEXE_NAME			"CifConv"

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

struct SObjectInfo
{
	uint32	objectName;
	uint32	internalModelName;
	uint32	skeletonName;
	uint32	bonedAnimName;
	char	externalModelName[256];
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

#define vCUTSCENEOBJECTFLAGS_ISSKATER		(1<<31)
#define vCUTSCENEOBJECTFLAGS_ISHEAD			(1<<30)

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void print_object_info( SObjectInfo* pObjectInfo )
{
	printf( "-------------------\n" );
	printf( "objectName = %08x\n", pObjectInfo->objectName );
	printf( "skeletonName = %08x\n", pObjectInfo->skeletonName );
	printf( "bonedAnimName = %08x\n", pObjectInfo->bonedAnimName );

	if ( pObjectInfo->internalModelName )
	{
		printf( "internalModelName = %08x\n", pObjectInfo->internalModelName );
	}
	else
	{
		printf( "externalModelName = %s\n", pObjectInfo->externalModelName );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void write_object_info( IoUtils::CVirtualOutputFile* pOutputFile, SObjectInfo* pObjectInfo, bool reverseByteOrder, uint32 extraFlags )
{
		pOutputFile->Write( (const char*)&pObjectInfo->objectName, sizeof( int ), reverseByteOrder );

		if ( pObjectInfo->internalModelName )
		{
			pOutputFile->Write( (const char*)&pObjectInfo->internalModelName, sizeof( int ), reverseByteOrder );
		}
		else
		{
			pOutputFile->Write( (const char*)&pObjectInfo->objectName, sizeof( int ), reverseByteOrder );
		}
		pOutputFile->Write( (const char*)&pObjectInfo->skeletonName, sizeof( int ), reverseByteOrder );
		pOutputFile->Write( (const char*)&pObjectInfo->bonedAnimName, sizeof( int ), reverseByteOrder );
//		pOutputFile->Write( (const char*)&pObjectInfo->externalModelName, sizeof( int ), reverseByteOrder );

		pOutputFile->Write( (const char*)&extraFlags, sizeof( int ), reverseByteOrder );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CCifConverter : public Utils::CBaseConverter
{
public:
	CCifConverter();
	virtual ~CCifConverter();
	
public:
	virtual void	Reset();
	virtual void	PrintUsage();

protected:
	virtual bool	ReadInputFile( IoUtils::CVirtualInputFile* pInputFile );
	virtual bool	WriteOutputFile( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder );

public:
	bool			WriteLibFile( bool reverseByteOrder );

protected:
	bool			is_head( int objectIdx );
	bool			is_skater( int objectIdx );

protected:
	enum
	{
		MAX_OBJECTS = 1024
	};
	SObjectInfo		m_objectList[MAX_OBJECTS];
	int				m_numObjects;
	int				m_versionNum;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CCifConverter::CCifConverter()
{
	m_debugMode = true;

	Reset();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CCifConverter::~CCifConverter()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CCifConverter::Reset()
{
	// reset state so that it can be run again
	m_numObjects = 0;
	m_versionNum = 1;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CCifConverter::ReadInputFile( IoUtils::CVirtualInputFile* pInputFile )
{
	pInputFile->Read( (char*)&m_versionNum, sizeof( int ) );
	if ( m_debugMode )
	{
		printf( "VersionNum = %d\n", m_versionNum );
	}

	pInputFile->Read( (char*)&m_numObjects, sizeof( int ) );
	if ( m_debugMode )
	{
		printf( "NumObjects = %d\n", m_numObjects );
	}

	if ( m_numObjects > MAX_OBJECTS )
	{
		char msg[512];
		sprintf( msg, "Too many objects (raise MAXOBJECTS from %d to %d\n", MAX_OBJECTS, m_numObjects );
		Utils::Assert( 0, msg );
		return false;
	}

	pInputFile->Read( (char*)&m_objectList[0], m_numObjects * sizeof( SObjectInfo ) );

	for ( int i = 0; i < m_numObjects; i++ )
	{
		if ( m_debugMode )
		{
			print_object_info( &m_objectList[i] );
		}
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool write_skinned_model( char* pBaseName, uint32 objectName, CLibMaker* pLibMaker, IoUtils::CVirtualOutputFile* pDependencyFile, int platform )
{
	char pFileName[_MAX_PATH];
	char* proj_root = getenv( "PROJ_ROOT" );

	sprintf( pFileName, "%s\\data\\models\\%s.skin.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
	bool success = pLibMaker->AddFile( pFileName, 
		objectName,
		Crc::GenerateCRCFromString( "skin" ) );

	// write it to the dependency file
	if ( success )
	{
		pDependencyFile->WriteString( pFileName, true );
	}

	if ( !success )
	{
		sprintf( pFileName, "%s\\data\\models\\%s.geom.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
		success = pLibMaker->AddFile( pFileName, 
			objectName,
			Crc::GenerateCRCFromString( "geom" ) );

		// write it to the dependency file
		if ( success )
		{
			pDependencyFile->WriteString( pFileName, true );
		}
	}
	
	if ( !success )
	{
		sprintf( pFileName, "%s\\data\\models\\%s.mdl.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
		success = pLibMaker->AddFile( pFileName,
			objectName,
			Crc::GenerateCRCFromString( "mdl" ) );

		// write it to the dependency file
		if ( success )
		{
			pDependencyFile->WriteString( pFileName, true );
		}
	}

	if ( !success )
	{
		printf( "Couldn't find file %s\n", pFileName );
		return false;
	}

	sprintf( pFileName, "%s\\data\\models\\%s.tex.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
	success = pLibMaker->AddFile( pFileName, 
		objectName,
		Crc::GenerateCRCFromString( "tex" ) );

	// write it to the dependency file
	if ( success )
	{
		pDependencyFile->WriteString( pFileName, true );
	}

	sprintf( pFileName, "%s\\data\\models\\%s.cas.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
	pLibMaker->AddFile( pFileName, 
		objectName,
		Crc::GenerateCRCFromString( "cas" ) );

	// write it to the dependency file
	if ( success )
	{
		pDependencyFile->WriteString( pFileName, true );
	}

	sprintf( pFileName, "%s\\data\\models\\%s.wgt.%s", proj_root, pBaseName, Utils::GetPlatformExt(platform) );
	pLibMaker->AddFile( pFileName, 
		objectName,
		Crc::GenerateCRCFromString( "wgt" ) );

	// write it to the dependency file
	if ( success )
	{
		pDependencyFile->WriteString( pFileName, true );
	}

	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CCifConverter::WriteLibFile( bool reverseByteOrder )
{
	CLibMaker theLibMaker;

	// 1-meg buffer for the lib file
	IoUtils::CVirtualOutputFile theDependencyFile;
	theDependencyFile.Init(1*1024*1024);

	for ( int i = 0; i < m_numObjects; i++ )
	{
		if ( *m_objectList[i].externalModelName == 0 )
		{
			// internal model filename
			continue;
		}

		char pBaseName[_MAX_PATH];
		char pDummyExt[_MAX_PATH];
		Utils::SplitFileExt( m_objectList[i].externalModelName, pBaseName, pDummyExt );

		// get the filename into lowercase
		char* pCurr = pBaseName;
		while( *pCurr )
		{
			if ( (*pCurr >= 'A') && (*pCurr <= 'Z') )
			{
				(*pCurr) = (*pCurr) - 'A' + 'a';
			}
			pCurr++;
		}

		// write out all four heads
		char* pShortName = strstr( pBaseName, "head_cas_male" );
		if ( pShortName )
		{
			strcpy( pShortName, "head_cas_male01" );
			write_skinned_model( pBaseName, m_objectList[i].objectName, &theLibMaker, &theDependencyFile, m_platform );

			strcpy( pShortName, "head_cas_male02" );
			write_skinned_model( pBaseName, m_objectList[i].objectName+1, &theLibMaker, &theDependencyFile, m_platform );

			strcpy( pShortName, "head_cas_female01" );
			write_skinned_model( pBaseName, m_objectList[i].objectName+2, &theLibMaker, &theDependencyFile, m_platform );

			strcpy( pShortName, "head_cas_female02" );
			write_skinned_model( pBaseName, m_objectList[i].objectName+3, &theLibMaker, &theDependencyFile, m_platform );
		}
		else
		{
			write_skinned_model( pBaseName, m_objectList[i].objectName, &theLibMaker, &theDependencyFile, m_platform );
		}
	}

	char pInputFileName[_MAX_PATH];
	char pDummyExt[_MAX_PATH];
	Utils::SplitFileExt( m_currentInputFile, pInputFileName, pDummyExt );

	char OutputLibFileName[_MAX_PATH];
	sprintf( OutputLibFileName, "%s.lib", pInputFileName );

	char DependencyFileName[_MAX_PATH];
	sprintf( DependencyFileName, "%s.dep.%s", pInputFileName, Utils::GetPlatformExt(m_platform) );

	// 10-meg buffer for the lib file
	IoUtils::CVirtualOutputFile theOutputLibFile;
	theOutputLibFile.Init(10*1024*1024);

	// this generates a lib file full of platform-specific files;  the lib header
	// is byte-swapped as necessary, and it's up to libconv to undo the byte-swapping
	// before expanding it into the main lib file
	if ( theLibMaker.OutputFiles( &theOutputLibFile, reverseByteOrder ) == 0 )
	{
		printf( "Error while writing output file %s\n", OutputLibFileName );
		exit(1);
	}

	if ( !theOutputLibFile.Save( OutputLibFileName ) )
	{
		printf( "Error while writing output file %s\n", OutputLibFileName );
		return false;
	}
	else
	{
		printf( "Successfully wrote out %s\n", OutputLibFileName );
	}

	if ( !theDependencyFile.Save( DependencyFileName ) )
	{
		printf( "Error while writing dependency file %s\n", DependencyFileName );
		return false;
	}
	else
	{
		printf( "Successfully wrote out %s\n", DependencyFileName );
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CCifConverter::is_head( int objectIdx )
{
	if ( objectIdx < (m_numObjects - 1) )
	{
		SObjectInfo* pInfo = &m_objectList[objectIdx]; 
		SObjectInfo* pNextInfo = &m_objectList[objectIdx + 1]; 
		if ( pInfo->objectName == ( pNextInfo->objectName + 1 ) )
		{
			return true;
		}
	}

	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CCifConverter::is_skater( int objectIdx )
{
	SObjectInfo* pInfo = &m_objectList[objectIdx]; 
	if ( pInfo->internalModelName == Crc::GenerateCRCFromString( "skater" )
		|| pInfo->internalModelName == Crc::GenerateCRCFromString( "player" ) )
	{
		return true;
	}

	if ( is_head( objectIdx ) )
	{
		// if it's the head, then check the next item,
		// which is the associated body
		Utils::Assert( objectIdx < m_numObjects - 1, "out of range" );
		SObjectInfo* pNextInfo = &m_objectList[objectIdx+1]; 
		if ( pNextInfo->internalModelName == Crc::GenerateCRCFromString( "skater" )
			|| pNextInfo->internalModelName == Crc::GenerateCRCFromString( "player" ) )
		{
			return true;
		}
	}

	return false;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CCifConverter::WriteOutputFile( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder )
{
	pOutputFile->Write((const char*) &m_versionNum, sizeof( int ), reverseByteOrder );

	pOutputFile->Write( (const char*)&m_numObjects, sizeof(int), reverseByteOrder );

	for ( int i = 0; i < m_numObjects; i++ )
	{
		uint32 extraFlags = 0;
		
		if ( is_head( i ) )
		{
			extraFlags |= vCUTSCENEOBJECTFLAGS_ISHEAD;
		}

		// either the skater's head or body
		if ( is_skater( i ) ) 
		{
			extraFlags |= vCUTSCENEOBJECTFLAGS_ISSKATER;
		}

		write_object_info( pOutputFile, &m_objectList[i], reverseByteOrder, extraFlags );
	}

	// in addition to the CIF.PS2 file, we need to create a new .LIB file
	// which contains all of the external models, texture dictionaries, etc.
	// note:  this assumes that all of the model assets have already been
	// converted to their platform specific formats!
	if ( !this->WriteLibFile( reverseByteOrder ) )
	{
		return false;
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CCifConverter::PrintUsage()
{
#ifdef _DEBUG
	printf( "%s v%d.%d (%s)\n", vEXE_NAME, vMAJOR_REVISION, vMINOR_REVISION, "Debug" );
#else
	printf( "%s v%d.%d (%s)\n", vEXE_NAME, vMAJOR_REVISION, vMINOR_REVISION, "Release" );
#endif
	printf( "Neversoft Entertainment, 2003\n" );
	printf( "\nUsage: CifConv -b<batchfile> -f<filename> -p[p | g | x]\n" );
	printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
	printf( "Ex: CifConv -f%PROJ_ROOT%\\nj_cutscene01.cif -pp\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void main( int argc, char* argv[] )
{
	CCifConverter theConverter;
	Utils::ConverterMain( &theConverter, argc, argv );
}