//****************************************************************************
//* MODULE:         Tools/AnimConv
//* FILENAME:       animconv.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  1/16/2002
//****************************************************************************

#include "AnimConv.h"

#include <math.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <core/defines.h>
#include <core/math.h>

#include "Utility.h"
#include "VirtualFile.h"

// GJ's suggestions for THPS5:

// 1.  Have the data come in as "uncompressed time" and "non-prerotated root"...
// that way, the artists don't have to worry about setting any checkboxes,
// and we can catch any bugs in handling these flags in one central location

// 2.  Move all animation compression out of the exporter.  We can do the
// compression in the converter so that we don't need to keep re-exporting
// in MAX.  Perhaps we can even test out different compression tolerances
// in-game.

const int vOUTPUT_VERSION = 1;

#define nxBONEDANIMFLAGS_LEGACY				(1<<31)
#define nxBONEDANIMFLAGS_INTERMEDIATE		(1<<30)
#define nxBONEDANIMFLAGS_UNCOMPRESSED		(1<<29)
#define nxBONEDANIMFLAGS_PLATFORM			(1<<28)	
#define nxBONEDANIMFLAGS_CAMERADATA			(1<<27)	
#define nxBONEDANIMFLAGS_COMPRESSEDTIME		(1<<26)
#define nxBONEDANIMFLAGS_PREROTATEDROOT		(1<<25)
#define nxBONEDANIMFLAGS_OBJECTANIMDATA		(1<<24)
#define nxBONEDANIMFLAGS_USECOMPRESSTABLE	(1<<23)
#define nxBONEDANIMFLAGS_HIRESFRAMEPOINTERS	(1<<22)
#define nxBONEDANIMFLAGS_CUSTOMKEYSAT60FPS	(1<<21)
#define nxBONEDANIMFLAGS_CUTSCENEDATA		(1<<20)
#define nxBONEDANIMFLAGS_PARTIALANIM		(1<<19)
#define nxBONEDANIMFLAGS_OLDPARTIALANIM		(1<<18)

// for bone arrays as local variables...  this number was
// chosen arbitrarily and can be increased as necessary
const int			vMAX_BONES = 128;

struct SBonedAnimFileHeader
{
	uint32 	    version;
	uint32    	flags;
	float	    duration;
};

struct SIntermediateFileHeader
{
    uint32      skeletonName;
    int		    numBones;
    int		    numQKeys;
    int		    numTKeys;
    int			numCustomAnimKeys;
};

struct SPlatformFileHeader
{
    uint32      numBones;
    int			numQKeys;
    int			numTKeys;
    int			numCustomAnimKeys;
};

class CAnimQKey
{
public:
	short			timestamp:15;
	short			signBit:1;	// 1 = negative

protected:
	CAnimQKey() {}
};

class CStandardAnimQKey	: public CAnimQKey
{
public:
    short           qx;
    short           qy;
    short           qz;
};

class CHiResAnimQKey : public CAnimQKey
{
public:
    float           qx;
    float           qy;
    float           qz;
};

class CAnimTKey
{
public:
	short			timestamp;

protected:
	CAnimTKey() {}
};

class CStandardAnimTKey	: public CAnimTKey
{
public:
    short           tx;
    short           ty;
    short           tz;
};

class CHiResAnimTKey : public CAnimTKey
{
public:
    float           tx;
    float           ty;
    float           tz;
};

class CHiResAnimFramePointers
{
public:
	unsigned short	numQKeys;
	unsigned short	numTKeys;
};

class CStandardAnimFramePointers
{
public:
	unsigned char	numQKeys;
	unsigned char	numTKeys;
};

struct SIntermediateQKey
{
public:
    uint32      time; 		// in frames
    float       imag[3];
    float       real;
};

struct SIntermediateTKey
{
public:
    uint32      time; 		// in frames
    float       t[3];
};

struct SIntermediateCustomAnimKeyHeader
{
	uint32		timeStamp;
	uint32		keyType;
	uint32		size;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CompressTable::CompressTable()
{
	m_uniqueCount = 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CompressTable::~CompressTable()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CompressTable::Init( char* pFileName )
{
	bool success = false;
	int i = 0;
	
	IoUtils::CVirtualInputFile compressInputFile;
	int bytesRead = compressInputFile.Load( pFileName );

	if ( !bytesRead )
	{
		printf( "Error!  Could not open compress table %s\n", pFileName );
		goto load_error;
	}

	if ( m_uniqueCount != 0 )
	{
		printf( "Error!  Compress table has already been initialized\n", pFileName );
		goto load_error;
	}

	m_uniqueCount = 0;

	for ( i = 0; i < vNUM_COMPRESS_TABLE_BUCKETS; i++ )
	{
		short qx, qy, qz;

		if ( !compressInputFile.Read( (char*)&qx, sizeof(short) ) )
		{
			goto load_error;
		}
		if ( !compressInputFile.Read( (char*)&qy, sizeof(short) ) )
		{
			goto load_error;
		}
		if ( !compressInputFile.Read( (char*)&qz, sizeof(short) ) )
		{
			goto load_error;
		}

		m_uniqueTable[m_uniqueCount].x = qx;
		m_uniqueTable[m_uniqueCount].y = qy;
		m_uniqueTable[m_uniqueCount].z = qz;
		m_uniqueTable[m_uniqueCount].lookupIndex = i;
		m_uniqueCount++;

		short dummy;
		if ( !compressInputFile.Read( (char*)&dummy, sizeof(short) ) )
		{
			goto load_error;
		}
	}

	success = true;

load_error:

	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int CompressTable::InUniqueTable( short x, short y, short z )
{
	SUniqueItem* pItem = &m_uniqueTable[0];

	for ( int i = 0; i < m_uniqueCount; i++ )
	{
		if ( x == pItem->x && y == pItem->y && z == pItem->z )
		{
			return pItem->lookupIndex;
		}

		pItem++;
	}

	return -1;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool g_should_swap_coords( uint32 flags, int boneIndex )
{
	// TODO:  The root should be rotated starting
	// with version 3 of the intermediate SKA format.
	// (Prior to version 3, we didn't want to rotate
	// the root bone, because bone #0 usually
	// referred to a bone that was somewhere
	// lower in the hierarchy...  starting with
	// version 3, bone #0 refers to the real
	// root bone)
	if ( flags & nxBONEDANIMFLAGS_PARTIALANIM )
	{
		return false;
	}

	// should pretty much always swap the coords to
	// get from the MAX -> game coordinate system
	// (not sure why we don't have to do this
	// for non-cutscene cameras;  maybe it's a
	// bug in the exporter...  will investigate later)
	if ( flags & nxBONEDANIMFLAGS_CAMERADATA )
	{
		if ( !( flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
		{
			// we don't need to swap the coords for cam anims...
			return false;
		}
	}

	if ( flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
	{
		// always swap the coords for object anims
		return !(flags & nxBONEDANIMFLAGS_PREROTATEDROOT);
	}

	return (!(flags & nxBONEDANIMFLAGS_PREROTATEDROOT)) && (boneIndex==0);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool g_should_rotate_root( uint32 flags, int boneIndex )
{
	// TODO:  The root should be rotated starting
	// with version 3 of the intermediate SKA format.
	// (Prior to version 3, we didn't want to rotate
	// the root bone, because bone #0 usually
	// referred to a bone that was somewhere
	// lower in the hierarchy...  starting with
	// version 3, bone #0 refers to the real
	// root bone)
	if ( flags & nxBONEDANIMFLAGS_PARTIALANIM )
	{
		return false;
	}

	if ( flags & nxBONEDANIMFLAGS_CAMERADATA )
	{
		if ( !( flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
		{
			// we don't need to rotate the root for cam anims...
			return false;
		}
		else
		{
			// unless it's a cutscene, for some reason...
			// probably due to an exporter bug that we
			// can fix later
			return (!(flags & nxBONEDANIMFLAGS_PREROTATEDROOT)) && (boneIndex==0);
		}
	}

	if ( flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
	{
		if ( !( flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
		{
			// we don't ever need to rotate the root for object anims...
			return false;
		}
		else
		{
	//		return false;

			// unless it's a cutscene, for some reason...
			// probably due to an exporter bug that we
			// can fix later
			return !( flags & nxBONEDANIMFLAGS_PREROTATEDROOT );
		}
	}

	// the only time that it makes sense to rotate the root
	// is for skeletal animations...  this is to compensate
	// for the T-keys for each bone not being swapped...
	// it doesn't make sense to rotate the cameras or
	// the objects, although it seems that we currently
	// need to do it for cutscene cameras and objects...
	// i'll take a look at why this is, later.
	return (!(flags & nxBONEDANIMFLAGS_PREROTATEDROOT)) && (boneIndex==0);
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

float round(float f)
{
	if ( f >= 0.0f )
	{
		return f + 0.5f;
	}
	else
	{
		return f - 0.5f;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CBoneData
{
public:
	CBoneData()
	{
		m_boneName = 0;
		m_numQKeys = 0;
		mp_qKeys = NULL;
		m_numTKeys = 0;
		mp_tKeys = NULL;
	}

	virtual ~CBoneData()
	{
		if ( mp_qKeys )
		{
			delete[] mp_qKeys;
			mp_qKeys = NULL;
		}

		if ( mp_tKeys )
		{
			delete[] mp_tKeys;
			mp_tKeys = NULL;
		}
	}

	void SetQKeyData( int numQKeys, SIntermediateQKey* pQKeyData )
	{
		Utils::Assert( mp_qKeys == NULL, "q-keys already allocated" );
		if ( numQKeys )
		{
			mp_qKeys = new SIntermediateQKey[numQKeys];
			memcpy( mp_qKeys, pQKeyData, numQKeys * sizeof(SIntermediateQKey) );
			m_numQKeys = numQKeys;
		}
	}

	void SetTKeyData( int numTKeys, SIntermediateTKey* pTKeyData )
	{
		Utils::Assert( mp_tKeys == NULL, "t-keys already allocated" );
		if ( numTKeys )
		{
			mp_tKeys = new SIntermediateTKey[numTKeys];
			memcpy( mp_tKeys, pTKeyData, numTKeys * sizeof(SIntermediateTKey) );
			m_numTKeys = numTKeys;
		}
	}

	void SetBoneName( uint32 boneName )
	{
		m_boneName = boneName;
	}

	uint32 GetBoneName() const
	{
		return m_boneName;
	}

	int	GetNumQKeys() const
	{
		return m_numQKeys;
	}

	int GetNumTKeys() const
	{
		return m_numTKeys;
	}

	SIntermediateQKey* GetQKeys()
	{
		return mp_qKeys;
	}

	SIntermediateTKey* GetTKeys()
	{
		return mp_tKeys;
	}

	int CompressQKeys()
	{
		return 0;
	}

	int CompressTKeys()
	{
		return 0;
	}

	bool same_q_key( const SIntermediateQKey& key0, const SIntermediateQKey& key1 )
	{
		return ( key0.imag[X] == key1.imag[X]
				&& key0.imag[Y] == key1.imag[Y]
				&& key0.imag[Z] == key1.imag[Z]
				&& key0.real == key1.real );
	}

	bool same_t_key( const SIntermediateTKey& key0, const SIntermediateTKey& key1 )
	{
		return ( key0.t[X] == key1.t[X]
				&& key0.t[Y] == key1.t[Y]
				&& key0.t[Z] == key1.t[Z] );
	}

	void RemoveQKeys()
	{
		m_numQKeys = 0;
	}

	void RemoveTKeys()
	{
		m_numTKeys = 0;
	}

	int RemoveDuplicateQKeys()
	{
		if ( !m_numQKeys )
		{
			return 0;
		}

		bool* p_should_remove = new bool[m_numQKeys];
		for ( int i = 0; i < m_numQKeys; i++ )
		{
			p_should_remove[i] = false;
		}

		int num_to_remove = 0;

		for ( int i = 1; i < m_numQKeys - 1; i++ )
		{
			if ( same_q_key( mp_qKeys[i], mp_qKeys[i-1] ) && 
				same_q_key( mp_qKeys[i], mp_qKeys[i+1] ) )
			{
				p_should_remove[i] = true;
				num_to_remove++;
			}
		}

		int newCount = 0;
		for ( int i = 0; i < m_numQKeys; i++ )
		{
			mp_qKeys[newCount] = mp_qKeys[i];

			if ( !p_should_remove[i] )
			{
				newCount++;
			}
		}
		m_numQKeys = newCount;

		delete[] p_should_remove;

		return num_to_remove;
	}

	int RemoveDuplicateTKeys()
	{
		if ( !m_numTKeys )
		{
			return 0;
		}

		bool* p_should_remove = new bool[m_numTKeys];
		for ( int i = 0; i < m_numTKeys; i++ )
		{
			p_should_remove[i] = false;
		}

		int num_to_remove = 0;

		for ( int i = 1; i < m_numTKeys - 1; i++ )
		{
			if ( same_t_key( mp_tKeys[i], mp_tKeys[i-1] ) && 
				 same_t_key( mp_tKeys[i], mp_tKeys[i+1] ) )
			{
				p_should_remove[i] = true;
				num_to_remove++;
			}
		}

		int newCount = 0;
		for ( int i = 0; i < m_numTKeys; i++ )
		{
			mp_tKeys[newCount] = mp_tKeys[i];

			if ( !p_should_remove[i] )
			{
				newCount++;
			}
		}
		m_numTKeys = newCount;

		delete[] p_should_remove;

		return num_to_remove;
	}

	void MassageQKeys( bool should_rotate_root, uint32 flags, bool isHiResData )
	{
		// massage the q data
		for ( int i = 0; i < m_numQKeys; i++ )
		{
			if ( should_rotate_root )
			{
				Mth::Quat quat;
				quat[X] = mp_qKeys[i].imag[0];
				quat[Y] = mp_qKeys[i].imag[1];
				quat[Z] = mp_qKeys[i].imag[2];
				quat[W] = mp_qKeys[i].real;

				Mth::Quat rotQuat( Mth::Vector(1,0,0), Mth::DegToRad(90.0f) );
				quat *= rotQuat;

				mp_qKeys[i].imag[0] = quat[X];
				mp_qKeys[i].imag[1] = quat[Y];
				mp_qKeys[i].imag[2] = quat[Z];
				mp_qKeys[i].real = quat[W];
			}

			if ( ( flags & nxBONEDANIMFLAGS_UNCOMPRESSED ) || 
				( flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
			{
				// GJ:  Do some basic compression here
				// just to account for slop in the floats
				// (should be able to do this for
				// non-compressed anims, too...)
				const float EPSILON = .000001f;
				if ( fabs(mp_qKeys[i].imag[X])<EPSILON )
				{
					mp_qKeys[i].imag[X] = 0.0f;
				}
				if ( fabs(mp_qKeys[i].imag[Y])<EPSILON )
				{
					mp_qKeys[i].imag[Y] = 0.0f;
				}
				if ( fabs(mp_qKeys[i].imag[Z])<EPSILON )
				{
					mp_qKeys[i].imag[Z] = 0.0f;
				}
				if ( fabs(mp_qKeys[i].real)<EPSILON )
				{
					mp_qKeys[i].real = 0.0f;
				}

				int64 fixPt;
				float precision = (float)(1<<20);
				fixPt = round( mp_qKeys[i].imag[0] * precision );
				mp_qKeys[i].imag[0] = (float)fixPt / precision;

				fixPt = round( mp_qKeys[i].imag[1] * precision );
				mp_qKeys[i].imag[1] = (float)fixPt / precision;

				fixPt = round( mp_qKeys[i].imag[2] * precision );
				mp_qKeys[i].imag[2] = (float)fixPt / precision;

				fixPt = round( mp_qKeys[i].real * precision );
				mp_qKeys[i].real = (float)fixPt / precision;
			}
			else
			{
				// decrease precision (not sure if this really works)
				if ( isHiResData )
				{
					int64 fixPt;
					float precision = (float)(1<<20);
					fixPt = round( mp_qKeys[i].imag[0] * precision );
					mp_qKeys[i].imag[0] = (float)fixPt / precision;

					fixPt = round( mp_qKeys[i].imag[1] * precision );
					mp_qKeys[i].imag[1] = (float)fixPt / precision;

					fixPt = round( mp_qKeys[i].imag[2] * precision );
					mp_qKeys[i].imag[2] = (float)fixPt / precision;

					fixPt = round( mp_qKeys[i].real * precision );
					mp_qKeys[i].real = (float)fixPt / precision;
				}
			}

			if ( !(flags & nxBONEDANIMFLAGS_COMPRESSEDTIME) )
			{
				// need to convert from float to uint32 if time is not already compressed...
				mp_qKeys[i].time = (short)((*((float*)&mp_qKeys[i].time) * 60.0f) + 0.5f);
			}
		}
	}

	void MassageTKeys( bool should_swap_coords, uint32 flags )
	{	
		// massage the t data
		for ( int i = 0; i < m_numTKeys; i++ )
		{
			if ( should_swap_coords )
			{
				float tx = mp_tKeys[i].t[0];
				float ty = mp_tKeys[i].t[1];
				float tz = mp_tKeys[i].t[2];

				tx = mp_tKeys[i].t[0];
				ty = mp_tKeys[i].t[2];
				tz = -mp_tKeys[i].t[1];

				mp_tKeys[i].t[0] = tx;
				mp_tKeys[i].t[1] = ty;
				mp_tKeys[i].t[2] = tz;
			}

			if ( ( flags & nxBONEDANIMFLAGS_UNCOMPRESSED ) || 
				( flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
			{
				// round to the nearest 1/100th inch
				float round_amount = 100.0f;
			
				// GJ:  Do some basic compression here
				// just to account for slop in the floats
				int64 t0 = round( mp_tKeys[i].t[0] * round_amount);
				mp_tKeys[i].t[0] = (float)t0 / round_amount;
				int64 t1 = round( mp_tKeys[i].t[1] * round_amount);
				mp_tKeys[i].t[1] = (float)t1 / round_amount;
				int64 t2 = round( mp_tKeys[i].t[2] * round_amount);
				mp_tKeys[i].t[2] = (float)t2 / round_amount;
			}

			if ( !(flags & nxBONEDANIMFLAGS_COMPRESSEDTIME) )
			{
				// need to convert from float to uint32 if time is not already compressed...
				mp_tKeys[i].time = (short)((*((float*)&mp_tKeys[i].time) * 60.0f) + 0.5f);
			}
		}
	}

	void PrintContents();

protected:
	int						m_boneName;
	int						m_numQKeys;
	SIntermediateQKey*		mp_qKeys;
	int						m_numTKeys;
	SIntermediateTKey*		mp_tKeys;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CBoneData::PrintContents()
{
	printf( "******************************\n" );
	printf( "------------------------------\n" );
	printf( "Bone Name: %08x\n", m_boneName );
	
	printf( "------------------------------\n" );
	printf( "NumQKeys: %d\n", m_numQKeys );
	
	for ( int i = 0; i < m_numQKeys; i++ )
	{
		printf( "\tQKey[%d] at frame %4d:\t(%f %f %f %f)\n", 
				i,
				mp_qKeys[i].time,
				mp_qKeys[i].imag[X], 
				mp_qKeys[i].imag[Y], 
				mp_qKeys[i].imag[Z], 
				mp_qKeys[i].real );
	}
	
	printf( "------------------------------\n" );
	printf( "NumTKeys: %d\n", m_numTKeys );

	for ( int i = 0; i < m_numTKeys; i++ )
	{
		printf( "\tTKey[%d] at frame %4d:\t(%f %f %f)\n", 
				i,
				mp_tKeys[i].time,
				mp_tKeys[i].t[X], 
				mp_tKeys[i].t[Y], 
				mp_tKeys[i].t[Z] );
	}
	printf( "------------------------------\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::is_hires_data()
{
	return m_flags & ( nxBONEDANIMFLAGS_CAMERADATA | nxBONEDANIMFLAGS_OBJECTANIMDATA );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

AnimConverter::AnimConverter()
{
	m_numCustomKeys = 0;
	mp_customData = NULL;
	m_customDataSize = 0;
	
	m_numBones = 0;
	mp_boneData = NULL;

	Reset();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void AnimConverter::Reset()
{
	m_flags = 0;
	m_duration = 0;
	m_lastCameraTimeStamp = -1;
	m_secondToLastCameraTimeStamp = -1;
	
	m_numCustomKeys = 0;
	if ( mp_customData )
	{
		delete[] mp_customData;
		mp_customData = NULL;
	}
	m_customDataSize = 0;
	
	m_numBones = 0;
	if ( mp_boneData )
	{
		delete[] mp_boneData;
		mp_boneData = NULL;
	}

	for ( int i = 0; i < vMAX_MASKS; i++ )
	{
		m_partialAnimMasks[i] = 0;
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::use_compress_table( CompressTable* pCompressTable )
{
//	comment this in if you don't want to compress
//	return false;

	return ( !is_hires_data() && pCompressTable && pCompressTable->GetCount() );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

AnimConverter::~AnimConverter()
{
	// reset basically frees up memory anyway
	Reset();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimConverter::ReadCustomKey( unsigned char* pData )
{
	SIntermediateCustomAnimKeyHeader* pIntermediateHeader;
	pIntermediateHeader = (SIntermediateCustomAnimKeyHeader*)pData;

	if ( !( m_flags & nxBONEDANIMFLAGS_COMPRESSEDTIME ) )
	{
//		printf( "Custom key at time %f \n", *(float*)&pIntermediateHeader->timeStamp );
		pIntermediateHeader->timeStamp = (short)((*((float*)&pIntermediateHeader->timeStamp) * 60.0f) + 0.5f);
	}

	if ( m_flags & nxBONEDANIMFLAGS_CAMERADATA && !(m_flags & nxBONEDANIMFLAGS_CUSTOMKEYSAT60FPS) )
	{
		// Adam used to export camera data at 30 fps, but the code expects it at 60 fps
		pIntermediateHeader->timeStamp *= 2;
	}

	// correct for letterbox effect on cutscene FOVs
	if ( ( pIntermediateHeader->keyType == 1 ) && ( m_flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
	{
		float* pFloatData = (float*)(pData + sizeof(SIntermediateCustomAnimKeyHeader));
		float widescreenFOV = *pFloatData; // in radians
		float standardFOV = 2 * atan( tan( widescreenFOV/2 ) * 3/4 );
//		printf( "widescreenFOV = %f\n", widescreenFOV );
//		printf( "standardFOV = %f\n", standardFOV );
		*pFloatData = standardFOV;
	}

	if ( m_debugMode )
	{
		printf( "------------------------------\n" );
		printf( "compressed = %08x\n", m_flags & nxBONEDANIMFLAGS_COMPRESSEDTIME );
		printf( "time in frames = %d\n", pIntermediateHeader->timeStamp );
		printf( "type = %08x\n", pIntermediateHeader->keyType );
		printf( "size = %d\n", pIntermediateHeader->size );

		switch( pIntermediateHeader->keyType )
		{
			case 2: // vCHANGE_CAMERA_RT
			{
				Mth::Vector* pVectorData = (Mth::Vector*)(pData + sizeof(SIntermediateCustomAnimKeyHeader));
				Mth::Quat* pQuatData = (Mth::Quat*)(pVectorData + sizeof(Mth::Vector));
				
				printf( "camera r = %f %f %f %f\n", (*pQuatData)[X], (*pQuatData)[Y], (*pQuatData)[Z], (*pQuatData)[W] );
				printf( "camera t = %f %f %f %f\n", (*pVectorData)[X], (*pVectorData)[Y], (*pVectorData)[Z], (*pVectorData)[W] );
			}
			break;
		}
	}

	if ( ( pIntermediateHeader->keyType == 2 ) && ( m_flags & nxBONEDANIMFLAGS_CUTSCENEDATA ) )
	{
		if ( pIntermediateHeader->timeStamp == m_lastCameraTimeStamp)
		{
			// GJ:  There's a bug in the exporter where
			// 2 camera RT keys with different positions
			// are listed for the same time...  this causes
			// a glitch during certain camera transitions.
			// To fix, I will remove the second key if
			// 2 CameraRT packets are found for the same time.

			// set the type to vCHANGE_CAMERA_RT_IGNORE
			pIntermediateHeader->keyType = 3;

			if ( m_debugMode )
			{
				printf( "***********************************************\n" );
				printf( "Ignoring duplicate camera RT at time %d!!!\n", pIntermediateHeader->timeStamp );
				printf( "***********************************************\n" );
			}
		}
		else if ( pIntermediateHeader->timeStamp == ( m_secondToLastCameraTimeStamp + 2 ) )
		{
			// GJ:  There's also a bug where a ChangeCameraRT
			// key is written out for the "next" camera... 
			// this is generally fine when the next camera is
			// stationary, but if the next camera is moving
			// it will create a little pop, because the overridden
			// camera will not interpolate at all in that 1/60th
			// of a frame

			// set the type to vCHANGE_CAMERA_RT_ENDKEY
			pIntermediateHeader->keyType = 7;

			if ( m_debugMode )
			{
				printf( "***********************************************\n" );
				printf( "Ignoring change to camera RT at time %d!!!\n", pIntermediateHeader->timeStamp );
				printf( "***********************************************\n" );
			}
		}

		if ( m_lastCameraTimeStamp != pIntermediateHeader->timeStamp )
		{
			m_secondToLastCameraTimeStamp = m_lastCameraTimeStamp;
			m_lastCameraTimeStamp = pIntermediateHeader->timeStamp;
		}
	}

	return pIntermediateHeader->size;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::Load(const char* pFileName)
{
	char tempFileName[_MAX_PATH];
	strcpy( tempFileName, pFileName );
	strlwr( tempFileName );

	if ( strstr( tempFileName, ".fam" ) )
	{
		return load_facial_anim( tempFileName );
	}
	else
	{
		return load_intermediate( tempFileName );
	}
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::load_intermediate(const char* pFileName)
{
	bool success = false;

	// declare local vars
	long				startPos;
	long				endPos;
	uint32				runningKeyCount;
	int					numQKeysPerBone[vMAX_BONES];
	int					numTKeysPerBone[vMAX_BONES];
	SIntermediateQKey*	p_qKeys = NULL;
	SIntermediateTKey*	p_tKeys = NULL;

	if ( m_debugMode )
	{
		printf( "Processing skeletal animation %s\n", pFileName );
	}

	IoUtils::CVirtualInputFile inputFile;
	int bytesRead = inputFile.Load( pFileName );

	if ( !bytesRead )
	{
		printf( "Could not open %s\n", pFileName );
		goto load_error;
	}

	SBonedAnimFileHeader theFileHeader;
	if ( !inputFile.Read( (char*)&theFileHeader, sizeof(SBonedAnimFileHeader) ) )
	{
		printf( "Couldn't read file header %s\n", pFileName );
		goto load_error;
	}

	// store some header information
	m_version = theFileHeader.version;
	m_duration = theFileHeader.duration;
	m_flags = theFileHeader.flags;

	if ( m_debugMode )
	{
		printf( "theFileHeader:\n" );
		printf( "version = 0x%08x\n", m_version );
		printf( "flags = 0x%08x\n", m_flags );
		printf( "duration = %f\n", m_duration );
	}

	// PATCH:  if it's a cutscene related file, then
	// automatically set its cutscene flag.
	if ( strstr(pFileName,"lctemp") )
	{
		m_flags |= nxBONEDANIMFLAGS_CUTSCENEDATA;
	}

	if ( !( m_flags & nxBONEDANIMFLAGS_INTERMEDIATE ) )
    {
		printf( "%s didn't have intermediate flag set\n", pFileName );
		goto load_error;
    }
	
#if 0
	if ( ( m_flags & nxBONEDANIMFLAGS_PREROTATEDROOT ) )
    {
		printf( "Wasn't expecting pre-rotated root in %s\n", pFileName );
		goto load_error;
    }
#endif

#if 0
	if ( ( m_flags & nxBONEDANIMFLAGS_COMPRESSEDTIME ) )
	{
		printf( "Was expecting time in frames (60fps) in %s\n", pFileName );
		goto load_error;
	}
#endif

	// read in header information
	SIntermediateFileHeader theIntermediateHeader;
	if ( !inputFile.Read( (char*)&theIntermediateHeader, sizeof(SIntermediateFileHeader) ) )
	{
		printf( "Couldn't read intermediate header in %s\n", pFileName );
		goto load_error;
	}

	if ( m_debugMode )
	{
		printf( "theIntermediateFormatHeader:\n" );
		printf( "skeletonName = 0x%08x\n", theIntermediateHeader.skeletonName );
		printf( "numBones = %d\n", theIntermediateHeader.numBones );
		printf( "numQKeys = %d\n", theIntermediateHeader.numQKeys );
		printf( "numTKeys = %d\n", theIntermediateHeader.numTKeys );
		printf( "numCustomAnimKeys = %d\n", theIntermediateHeader.numCustomAnimKeys );
	}

	// sanity check
    if ( (theIntermediateHeader.numQKeys == 0 ) || ( theIntermediateHeader.numTKeys == 0 ) )
    {
        printf( "Need at least 1 Q- and 1 T-key in %s!\n", pFileName );
		goto load_error;
    }

    m_numBones = theIntermediateHeader.numBones;

	Utils::Assert( m_numBones < vMAX_BONES, "Too many bones!" );

	Utils::Assert( mp_boneData == NULL, "Bone data already exists" );
	mp_boneData = new CBoneData[m_numBones];

// TODO:  Remember this only as a safety check
//	m_numQKeys = theIntermediateHeader.numQKeys;
//	m_numTKeys = theIntermediateHeader.numTKeys;

	m_numCustomKeys = theIntermediateHeader.numCustomAnimKeys;

    int dummy[128];

	// skeleton id
	if ( !inputFile.Read( (char*)&dummy[0], sizeof(uint32) ) )
    {
		printf( "Couldn't read skeleton name in %s\n", pFileName );
		goto load_error;
    }

	if ( m_debugMode )
	{
		printf( "skeletonName (dummy) = 0x%08x\n", dummy[0] );
	}

	// num bones
	if ( !inputFile.Read( (char*)&dummy[0], sizeof(uint32) ) )
    {
		printf( "Couldn't read number of bones in %s\n", pFileName );
		goto load_error;
    }

	if ( m_debugMode )
	{
		printf( "numBones (dummy) = %d\n", dummy[0] );
	}

	if ( dummy[0] != m_numBones )
	{
		printf( "Wrong number of bones in %s\n", pFileName );
		goto load_error;
	}

    // read bone names
	for ( int i = 0; i < m_numBones; i++ )
	{
		uint32 boneName;
		if ( !inputFile.Read( (char*)&boneName, sizeof(uint32) ) )
		{
			printf( "Couldn't read bone names in %s\n", pFileName );
			goto load_error;				
		}

		CBoneData* pBoneData = get_bone_data(i);
		Utils::Assert( pBoneData != NULL, "No Bone Data?!?" );
		pBoneData->SetBoneName( boneName );

		if ( m_debugMode )
		{
			printf( "boneName[%d] = (dummy) = 0x%08x\n", i, boneName );
		}
	}

    // read parent data
	if ( !inputFile.Read( (char*)&dummy[0], m_numBones * sizeof(uint32) ) )
    {
		printf( "Couldn't read parent names in %s\n", pFileName );
		goto load_error;
    }

	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			printf( "parentName[%d] = (dummy) = 0x%08x\n", i, dummy[i] );
		}
	}

    // read flip table
	if ( !inputFile.Read( (char*)&dummy[0], m_numBones * sizeof(uint32) ) )
    {
		printf( "Couldn't read flip names in %s\n", pFileName );
		goto load_error;
    }

	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			printf( "flipName[%d] = (dummy) = 0x%08x\n", i, dummy[i] );
		}
	}

	if ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM )
	{
		// version 2 has partial anims in new format (doesn't need bone names any more)
		if ( m_version >= 3 )
		{
			// read in some extra data having to do with partial anims
			if ( !inputFile.Read( (char*)&m_partialAnimTotalBones, sizeof(uint32) ) )
			{
				printf( "Couldn't read total num bones in %s\n", pFileName );
				goto load_error;
			}

			// masks
			int numMasks = (( m_partialAnimTotalBones - 1 )/ 32) + 1;
			Utils::Assert( numMasks < vMAX_MASKS, "Too many masks" );

			if ( !inputFile.Read( (char*)&m_partialAnimMasks[0], numMasks * sizeof(uint32) ) )
			{
				printf( "Couldn't read masks in %s\n", pFileName );	
				goto load_error;
			}
		}
	}

    // read num q frames
	if ( !inputFile.Read( (char*)&numQKeysPerBone[0], m_numBones * sizeof(uint32) ) )
	{
		printf( "Couldn't read number of Q keys in %s\n", pFileName );
		goto load_error;
	}
    
    // read the q data into a temporary buffer
	p_qKeys = new SIntermediateQKey[theIntermediateHeader.numQKeys];
	if ( !inputFile.Read( (char*)&p_qKeys[0], theIntermediateHeader.numQKeys * sizeof(SIntermediateQKey) ) )
    {
		printf( "Couldn't read Q-frames in %s\n", pFileName );
		goto load_error;
	}

	// copy the temporary data into the per-bone data
	runningKeyCount = 0;
 	for ( int i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data( i );
		Utils::Assert( pBoneData != NULL, "No bone data" );
		pBoneData->SetQKeyData( numQKeysPerBone[i], p_qKeys + runningKeyCount );
        runningKeyCount += numQKeysPerBone[i];
	}
	if ( runningKeyCount != theIntermediateHeader.numQKeys )
	{
		printf( "Wrong number of Q keys (%d, %d) in %s\n", runningKeyCount, theIntermediateHeader.numQKeys, pFileName );
		goto load_error;
	}

	// read num t frames
	if ( !inputFile.Read( (char*)&numTKeysPerBone[0], m_numBones * sizeof(uint32) ) )
    {
		printf( "Couldn't read number of T keys for bone %d in %s\n", i, pFileName );
		goto load_error;
	}

	// read the t data into a temporary buffer
	p_tKeys = new SIntermediateTKey[theIntermediateHeader.numTKeys];
	if ( !inputFile.Read( (char*)&p_tKeys[0], theIntermediateHeader.numTKeys * sizeof(SIntermediateTKey) ) )
    {
		printf( "Couldn't read T-frames in %s\n", pFileName );
		goto load_error;
    }

	// copy the temporary data into the per-bone data
	runningKeyCount = 0;
 	for ( int i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data( i );
		Utils::Assert( pBoneData != NULL, "No bone data" );
		pBoneData->SetTKeyData( numTKeysPerBone[i], p_tKeys + runningKeyCount );
        runningKeyCount += numTKeysPerBone[i];
	}
	if ( runningKeyCount != theIntermediateHeader.numTKeys )
	{
		printf( "Wrong number of T keys (%d, %d) in %s\n", runningKeyCount, theIntermediateHeader.numTKeys, pFileName );
		goto load_error;
	}

/*
	if ( m_debugMode )
	{
		printf( "Pre-massage:\n" );
		for ( int i = 0; i < m_numBones; i++ )
		{
			CBoneData* pBoneData = get_bone_data(i);
			Utils::Assert( pBoneData != NULL, "Couldn't find bone data" );
			pBoneData->PrintContents();
		}
	}
*/

	int numQRemoved = 0;
	int numTRemoved = 0;

	// massage the data here
 	for ( int i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data( i );
		Utils::Assert( pBoneData != NULL, "No bone data" );
		
		// should only have to rotate the root for
		// skeletal animations (to compensate for
		// the MAX coordinate system change).
		// object anims and cameras should NOT
		// be rotated (although due to a possible
		// bug in the export data, they will
		// be rotated)
		bool should_rotate_root = g_should_rotate_root(m_flags, i);
		pBoneData->MassageQKeys( should_rotate_root, m_flags, is_hires_data() );

		bool should_swap_coords = g_should_swap_coords(m_flags, i);
		pBoneData->MassageTKeys( should_swap_coords, m_flags );
	
		numQRemoved += pBoneData->RemoveDuplicateQKeys();
		numTRemoved += pBoneData->RemoveDuplicateTKeys();
	}

//	printf( "numQRemoved = %d keys\n", numQRemoved );
//	printf( "numTRemoved = %d keys\n", numTRemoved );
	
	// should just byte-copy the rest of the data
	// (custom keys) into a new buffer, and write it
	// out verbatim to the PS2 file
	startPos = inputFile.TellPos();
	endPos = bytesRead;

	m_customDataSize = endPos - startPos;

	if ( m_debugMode )
	{
		printf( "Custom Data Size = 0x%08x bytes\n", m_customDataSize );
	}
	
	// oops, the file format must have messed up,
	// because the byte swapping code expects
	// everything to be long-sized.
	if ( ( m_customDataSize % 4 ) != 0 )
	{
		printf( "Custom data size not multiple of 4 in %s (%d %d %d)\n", 
			pFileName,
			m_customDataSize,
			endPos,
			startPos );
		Utils::Assert( 0, "Custom data size not multiple of 4!\n" );
		goto load_error;
	}

	if ( m_customDataSize > 0 )
	{
///		printf( "Custom Data Size = %d bytes\n", endPos - startPos );
		mp_customData = new unsigned char[m_customDataSize];

		if ( !inputFile.Read( (char*)mp_customData, m_customDataSize ) )
		{
			printf( "Couldn't read custom keys in %s\n", pFileName );
			goto load_error;
		}

		// massage the time data
		// by compressing its time
		for ( i = 0; i < m_customDataSize; )
		{
			int size = ReadCustomKey( &mp_customData[i] );

			if ( size == 0 )
			{
				// shrink the custom data size to the last valid key
				m_customDataSize = i;
				printf( "Couldn't process custom keys in %s\n", pFileName );
				goto load_error;
			}

			i += size;
		}
	}

	// by this point, all the flags should be set
	m_flags |= (nxBONEDANIMFLAGS_PREROTATEDROOT);
	m_flags |= (nxBONEDANIMFLAGS_COMPRESSEDTIME);

	if ( m_debugMode )
	{
		printf( "Done parsing intermediate format\n" );
	}

	if ( m_debugMode )
	{
//		printf( "Post-massage:\n" );
		for ( int i = 0; i < m_numBones; i++ )
		{
			CBoneData* pBoneData = get_bone_data(i);
			Utils::Assert( pBoneData != NULL, "Couldn't find bone data" );
			pBoneData->PrintContents();
		}
	}

	success = true;

load_error:
	// delete temporary buffers
	if ( p_qKeys )
	{
		delete[] p_qKeys;
	}

	if ( p_tKeys )
	{
		delete[] p_tKeys;
	}

	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::load_facial_anim(const char* pFileName)
{
	bool success = false;

	if ( m_debugMode )
	{
		printf( "Processing facial animation %s\n", pFileName );
	}

	IoUtils::CVirtualInputFile inputFile;
	int bytesRead = inputFile.Load( pFileName );

	if ( !bytesRead )
	{
		printf( "Could not open %s\n", pFileName );
		goto load_error;
	}

	// store some header information
	m_version = 3;
	m_duration = ((float)bytesRead / 60.0f);
	m_flags = nxBONEDANIMFLAGS_INTERMEDIATE	
		| nxBONEDANIMFLAGS_UNCOMPRESSED
		| nxBONEDANIMFLAGS_COMPRESSEDTIME
		| nxBONEDANIMFLAGS_PREROTATEDROOT
		| nxBONEDANIMFLAGS_PARTIALANIM;

	if ( m_debugMode )
	{
		printf( "theFileHeader:\n" );
		printf( "version = 0x%08x\n", m_version );
		printf( "flags = 0x%08x\n", m_flags );
		printf( "duration = %f\n", m_duration );
	}

	// hardcoded number of bones
	m_numBones = 55;
	Utils::Assert( m_numBones < vMAX_BONES, "Too many bones!" );

	Utils::Assert( mp_boneData == NULL, "Bone data already exists" );
	mp_boneData = new CBoneData[m_numBones];

	m_numCustomKeys = 0;

	// specifically target the "bone_jaw"
	int jaw_bone_index = 31;
	CBoneData* pBoneData = get_bone_data(jaw_bone_index);
	Utils::Assert( pBoneData != NULL, "No Bone Data?!?" );
	pBoneData->SetBoneName( Crc::GenerateCRCFromString("Bone_Jaw") );

	m_partialAnimTotalBones = m_numBones;
	m_partialAnimMasks[ jaw_bone_index / 32 ] |= 1 << ( jaw_bone_index % 32 );

	char* proj_path = getenv( "PROJ_ROOT" );
	if( proj_path == NULL )
	{		
		printf( "You must first define your PROJ_ROOT environment variable\n" );
	}

	int max_angle = 45;

	// override max_angle, if the animconv.ini exists:
	char animconv_ini_name[_MAX_PATH];
	sprintf( animconv_ini_name, "%s\\bin\\win32\\animconv.ini", proj_path );

	FILE* fp = fopen(animconv_ini_name, "r");
	if ( !fp )
	{
		printf( "Couldn't open file %s to get max jaw rotation angle\n", animconv_ini_name );
		goto load_error;
	}

	char max_angle_string[255];
	fread( max_angle_string, 255, 1, fp );
	if ( !sscanf( max_angle_string, "%d", &max_angle ) )
	{
		printf( "Couldn't parse file %s to get max jaw rotation angle %s\n", animconv_ini_name );
		goto load_error;
	}
	fclose( fp );

	bool* pRemoveBytes = new bool[bytesRead];
	for ( int i = 0; i < bytesRead; i++ )
	{
		pRemoveBytes[i] = false;
	}

	// this looks for 3 consecutive bytes that travel in the
	// same direction...  in this case, we can remove the
	// middle byte
	uint8* pAllBytes = new uint8[bytesRead];
	inputFile.Read( (char*)pAllBytes, bytesRead );
	for ( int i = 1; i < bytesRead - 1; i++ )
	{
		if ( ( pAllBytes[i] > pAllBytes[i-1] && pAllBytes[i] < pAllBytes[i+1] )
			|| ( pAllBytes[i] < pAllBytes[i-1] && pAllBytes[i] > pAllBytes[i+1] ) )
		{
			pRemoveBytes[i] = true;
		}
	}

	int curr = 0;
	SIntermediateQKey* p_qKeys = new SIntermediateQKey[bytesRead];
	for ( int i = 0; i < bytesRead; i++ )
	{
		uint8 currByte = pAllBytes[i];

		Mth::Quat quat( 0.0f, 0.0f, 0.0f, 1.0f );
		Mth::Quat rotQuat( Mth::Vector(1,0,0), Mth::DegToRad((float)-max_angle * (float)((uint8)currByte) / 255.0f) );
		quat *= rotQuat;

		p_qKeys[curr].imag[0] = quat[X];
		p_qKeys[curr].imag[1] = quat[Y];
		p_qKeys[curr].imag[2] = quat[Z];
		p_qKeys[curr].real = quat[W];
		p_qKeys[curr].time = i;

		if ( !pRemoveBytes[i] )
		{
			curr++;
		}	
	}

	bytesRead = curr;

	delete[] pRemoveBytes;
	delete[] pAllBytes;

	pBoneData->SetQKeyData( bytesRead, p_qKeys );
	delete[] p_qKeys;

	{
		SIntermediateTKey tKey;
		// these T-values were chosen to
		// match the values in idle.ska
		// (it's all very kludgy!)
		tKey.t[0] = -0.000004f;
		tKey.t[1] = -1.776662f;
		tKey.t[2] = 0.599481f;
		tKey.time = 0;
		pBoneData->SetTKeyData( 1, &tKey );
	}

	bool should_rotate_root = g_should_rotate_root(m_flags, jaw_bone_index);
	pBoneData->MassageQKeys( should_rotate_root, m_flags, is_hires_data() );

	bool should_swap_coords = g_should_swap_coords(m_flags, jaw_bone_index);
	pBoneData->MassageTKeys( should_swap_coords, m_flags );

	pBoneData->RemoveDuplicateQKeys();
	pBoneData->RemoveDuplicateTKeys();

	m_customDataSize = 0;

	// by this point, all the flags should be set
	m_flags |= (nxBONEDANIMFLAGS_PREROTATEDROOT);
	m_flags |= (nxBONEDANIMFLAGS_COMPRESSEDTIME);

	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			CBoneData* pBoneData = get_bone_data(i);
			Utils::Assert( pBoneData != NULL, "Couldn't find bone data" );
			pBoneData->PrintContents();
		}
	}

	success = true;

load_error:
	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimConverter::get_num_qkeys()
{
	int count = 0;

	for ( int i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data(i);
		Utils::Assert( pBoneData != NULL, "No bone data?!?" );
		count += pBoneData->GetNumQKeys();
	}

	return count;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimConverter::get_num_tkeys()
{
	int count = 0;

	for ( int i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data(i);
		Utils::Assert( pBoneData != NULL, "No bone data?!?" );
		count += pBoneData->GetNumTKeys();
	}

	return count;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CBoneData* AnimConverter::get_bone_data(int bone_index)
{
	Utils::Assert( mp_boneData != NULL, "No bone data?!?" );

	if ( bone_index < 0 || bone_index >= m_numBones )
	{
		// out of range
		return NULL;
	}

	return mp_boneData + bone_index;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/
 
bool AnimConverter::SaveCompressed(const char* pFileName, CompressTable* pQCompressTable, CompressTable* pTCompressTable, bool reverseByteOrder)
{
	if ( !use_compress_table(pQCompressTable) && !use_compress_table(pTCompressTable) )
	{
		// not compressed
		return Save( pFileName, reverseByteOrder );
	}

	bool success = false;

	// declare some local vars here
	int qStart;
	int tStart;
	int qSize;
	int tSize;
	int currPos;

	const int MAX_BUFFER_SIZE = 3072 * 1024;
	IoUtils::CVirtualOutputFile theOutputFile;
	IoUtils::CVirtualOutputFile* p_outputFile = &theOutputFile;
	p_outputFile->Init(MAX_BUFFER_SIZE);

	SBonedAnimFileHeader theFileHeader;
	theFileHeader.version = vOUTPUT_VERSION;
	theFileHeader.duration = m_duration;
	theFileHeader.flags = nxBONEDANIMFLAGS_PLATFORM 
		| nxBONEDANIMFLAGS_COMPRESSEDTIME 
		| nxBONEDANIMFLAGS_PREROTATEDROOT 
		| ( m_flags & nxBONEDANIMFLAGS_CAMERADATA )
		| ( m_flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
		| ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM );

	// early versions of the partial anim have a slightly
	// different format...  this will get phased out when
	// Adam implements some new partial animation data
	// format changes...
	if ( ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM ) && ( m_version < 3 ) )
	{
		theFileHeader.flags |= nxBONEDANIMFLAGS_OLDPARTIALANIM;
	}

	if ( use_compress_table(pQCompressTable) || use_compress_table(pTCompressTable) )
	{
		theFileHeader.flags |= nxBONEDANIMFLAGS_USECOMPRESSTABLE;
		theFileHeader.flags &= ~nxBONEDANIMFLAGS_PLATFORM;
	}

	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.version, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.flags, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.duration, reverseByteOrder );

	SPlatformFileHeader thePlatformHeader;
	thePlatformHeader.numBones = m_numBones;
	thePlatformHeader.numQKeys = get_num_qkeys();
	thePlatformHeader.numTKeys = get_num_tkeys();
	thePlatformHeader.numCustomAnimKeys = m_numCustomKeys;

	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numBones, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numQKeys, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numTKeys, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numCustomAnimKeys, reverseByteOrder );
	
	uint32 qPatchAddress = 0;
	uint32 tPatchAddress = 0;

	uint32 allocPatchAddress = p_outputFile->TellPos();

	uint32 dummy32 = 0;
	IoUtils::write32( p_outputFile, (unsigned int*)&dummy32, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&dummy32, reverseByteOrder );
	
	int i;

	qPatchAddress = p_outputFile->TellPos();
	for ( i = 0; i < m_numBones; i++ )
	{
		short dummy16 = 0;
		IoUtils::write16( p_outputFile, (unsigned short*)&dummy16, reverseByteOrder );
	}

	tPatchAddress = p_outputFile->TellPos();
	for ( i = 0; i < m_numBones; i++ )
	{
		short dummy16 = 0;
		IoUtils::write16( p_outputFile, (unsigned short*)&dummy16, reverseByteOrder );
	}

	// long-align
	p_outputFile->Align(4);

	if ( m_flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
	{
		// IoUtils::write out bone names
		for ( i = 0; i < m_numBones; i++ )
		{
			CBoneData* pBoneData = get_bone_data(i);
			Utils::Assert( pBoneData != NULL, "No bone data?!?" );
			uint32 boneName = pBoneData->GetBoneName();

			IoUtils::write32( p_outputFile, (unsigned int*)&boneName, reverseByteOrder );
		}
	}

	if ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM )
	{
		write_partial_anim_data( p_outputFile, reverseByteOrder );
	}

	// In compressed files, these are taken over by the size table
#if 0
	for ( i = 0; i < m_numBones; i++ )
	{
		if ( is_hires_data() )
		{
			CHiResAnimFramePointers thePlatformPointers;
			thePlatformPointers.numQKeys = mp_perBoneFrames[i].numQKeys;
			thePlatformPointers.numTKeys = mp_perBoneFrames[i].numTKeys;

			IoUtils::write16( p_outputFile, (unsigned short*)&thePlatformPointers.numQKeys, reverseByteOrder );
			IoUtils::write16( p_outputFile, (unsigned short*)&thePlatformPointers.numTKeys, reverseByteOrder );
		}
		else
		{
			CStandardAnimFramePointers thePlatformPointers;
			thePlatformPointers.numQKeys = mp_perBoneFrames[i].numQKeys;
			thePlatformPointers.numTKeys = mp_perBoneFrames[i].numTKeys;

			IoUtils::write8( p_outputFile, (unsigned char*)&thePlatformPointers.numQKeys, reverseByteOrder );
			IoUtils::write8( p_outputFile, (unsigned char*)&thePlatformPointers.numTKeys, reverseByteOrder );
		}
	}
#endif

	// long-align
	p_outputFile->Align(4);

	qStart = p_outputFile->TellPos();

	// IoUtils::write the Q data
	if ( !write_compressed_q_data( p_outputFile, pQCompressTable, reverseByteOrder, qPatchAddress ) )
	{
		goto failure;
	}

	qSize = p_outputFile->TellPos() - qStart;

	tStart = p_outputFile->TellPos();

	// IoUtils::write the T data
	if ( !write_compressed_t_data( p_outputFile, pTCompressTable, reverseByteOrder, tPatchAddress ) )
	{
		goto failure;
	}

	tSize = p_outputFile->TellPos() - tStart;

	// need to patch the sizes...
	// (basically how much to allocate for the Q- and T- keys);
	currPos = p_outputFile->TellPos();
	p_outputFile->SeekPos( allocPatchAddress );
	IoUtils::write32( p_outputFile, (unsigned int*)&qSize, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&tSize, reverseByteOrder );
	p_outputFile->SeekPos( currPos );

	// long-align
	p_outputFile->Align(4);

	if ( ( m_customDataSize % 4 ) != 0 )
	{
		printf( "Custom data size not multiple of 4!" );
		goto failure;
	}

	for ( i = 0; i < m_customDataSize; i+=4 )
	{
//		unsigned int* foo = (unsigned int*)&mp_customData[i];
//		printf( "Addr = %p %d %p\n", foo, m_customDataSize, mp_customData );
//		printf( "[%08x] %d\n", *foo, reverseByteOrder );
		IoUtils::write32( p_outputFile, (unsigned int*)&mp_customData[i], reverseByteOrder );
	}

	if ( !p_outputFile->Save( pFileName ) )
	{
		goto failure;
	}

	success = true;

failure:
	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::write_partial_anim_data( IoUtils::CVirtualOutputFile* p_outputFile, bool reverseByteOrder )
{
	if ( m_version >= 3 )
	{
		IoUtils::write32( p_outputFile, (unsigned int*)&m_partialAnimTotalBones,	reverseByteOrder );

		int numMasks = (( m_partialAnimTotalBones - 1 )/ 32) + 1;		
		for ( int i = 0; i < numMasks; i++ )
		{
			IoUtils::write32( p_outputFile, (unsigned int*)&m_partialAnimMasks[i], reverseByteOrder );
		}
	}
	else
	{
		uint32 boneNames[] = {
			0x1be55811,
			0x63540a6d,
			0x897b494b,
			0xe9beedf4,
			0x96ec22f2,
			0xf1c35735,
			0x1addb5f1,
			0x9bbb8ddd,
			0x82b37cbc,
			0x744087f0,
			0x437e2114,
			0x78a9d53a,
			0x1c37d3df,
			0xe039f273,
			0x7ee14cfe,
			0x4e7a0d0d,
			0xcb0f4e56,
			0x0bcc6a56,
			0xe0d28892,
			0x61b4b0be,
			0x78bc41df,
			0x82a6e859,
			0xe638eebc,
			0x8e4fba93,
			0xb9711c77,
			0x1a36cf10,
			0x84ee719d,
			0xb475306e,
			0x31007335,
			0x5a0e0860,
			0xddec28af,
			0x7268c230,
			0x4177c63c,
			0xb105116e,
			0x2182e17b,
			0xb88bb0c1,
			0x0aa1482b,
			0x7f353d60,
			0xef8a20f1,
			0x853a0003,
			0xdd0cd021,
			0xe0e63a64,
			0x9baac767,
			0x09a99f42,
			0x727c060f,
			0x2703ed42,
			0x1ae90707,
			0x88733b6c,
			0x61a5fa04,
			0xf3a6a221,
			0x98971faf,
			0x0e9f8a27,
			0xe68d351a,
			0xf25452a9,
			0x1a46ed94,
			0
		};

		// <= version 2: we lookup the bone names in the above list,
		// which effectively means that partial anims will only
		// work with the existing 55-bone thps5_human skeleton.
		// (version 3 is more generic, in that the exporter writes
		// out these masks directly)
		int numBones = 55;
		int numMasks = (( numBones - 1 )/ 32) + 1;		
		IoUtils::write32( p_outputFile, (unsigned int*)&numBones, reverseByteOrder );

		uint32 mask[vMAX_MASKS];
		for ( int i = 0; i < vMAX_MASKS; i++ )
		{
			mask[i] = 0;
		}

		int x = 0;
		uint32* pCurr = &boneNames[0];
		while ( *pCurr )
		{
			for ( int j = 0; j < m_numBones; j++ )
			{
				CBoneData* pBoneData = get_bone_data(j);
				if ( *pCurr == pBoneData->GetBoneName() )
				{
					mask[ x / 32 ] |= ( 1 << (x % 32) );
					break;
				}
			}
	
			x++;
			pCurr++;
		}

		for ( int i = 0; i < numMasks; i++ )
		{
	//		printf( "Writing mask[%d] %08x\n", i, mask[i] );
	
			IoUtils::write32( p_outputFile, (unsigned int*)&mask[i], reverseByteOrder );			
		}
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::Save(const char* pFileName, bool reverseByteOrder)
{
	bool success = false;

	const int MAX_BUFFER_SIZE = 3072 * 1024;
	IoUtils::CVirtualOutputFile theOutputFile;
	IoUtils::CVirtualOutputFile* p_outputFile = &theOutputFile;
	p_outputFile->Init(MAX_BUFFER_SIZE);

	SBonedAnimFileHeader theFileHeader;
	theFileHeader.version = vOUTPUT_VERSION;
	theFileHeader.duration = m_duration;
	theFileHeader.flags = nxBONEDANIMFLAGS_PLATFORM 
		| nxBONEDANIMFLAGS_COMPRESSEDTIME 
		| nxBONEDANIMFLAGS_PREROTATEDROOT 
		| ( m_flags & nxBONEDANIMFLAGS_CAMERADATA )
		| ( m_flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
		| ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM );

	int i;

	// check to see if we're going to need to IoUtils::write out
	// shorts instead of chars for the per-bone frame counts
	for ( i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data(i);
		Utils::Assert( pBoneData != NULL, "No bone data?!?" );

		if ( pBoneData->GetNumQKeys() >= 256 || pBoneData->GetNumTKeys() >= 256 )
		{
			theFileHeader.flags |= nxBONEDANIMFLAGS_HIRESFRAMEPOINTERS;
		}
	}

	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.version, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.flags, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&theFileHeader.duration, reverseByteOrder );

	SPlatformFileHeader thePlatformHeader;
	thePlatformHeader.numBones = m_numBones;
	thePlatformHeader.numQKeys = get_num_qkeys();
	thePlatformHeader.numTKeys = get_num_tkeys();
	thePlatformHeader.numCustomAnimKeys = m_numCustomKeys;

	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numBones, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numQKeys, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numTKeys, reverseByteOrder );
	IoUtils::write32( p_outputFile, (unsigned int*)&thePlatformHeader.numCustomAnimKeys, reverseByteOrder );

	if ( m_flags & nxBONEDANIMFLAGS_OBJECTANIMDATA )
	{
		// IoUtils::write out bone names
		for ( i = 0; i < m_numBones; i++ )
		{
			CBoneData* pBoneData = get_bone_data(i);
			Utils::Assert( pBoneData != NULL, "No bone data?!?" );
			uint32 boneName = pBoneData->GetBoneName();

			IoUtils::write32( p_outputFile, (unsigned int*)&boneName, reverseByteOrder );
		}
	}

	if ( m_flags & nxBONEDANIMFLAGS_PARTIALANIM )
	{
		write_partial_anim_data( p_outputFile, reverseByteOrder );
	}

	for ( i = 0; i < m_numBones; i++ )
	{
		CBoneData* pBoneData = get_bone_data(i);
		Utils::Assert( pBoneData != NULL, "No bone data?!?" );

		if ( theFileHeader.flags & nxBONEDANIMFLAGS_HIRESFRAMEPOINTERS )
		{
			CHiResAnimFramePointers thePlatformPointers;
			thePlatformPointers.numQKeys = pBoneData->GetNumQKeys();
			thePlatformPointers.numTKeys = pBoneData->GetNumTKeys();

			IoUtils::write16( p_outputFile, (unsigned short*)&thePlatformPointers.numQKeys, reverseByteOrder );
			IoUtils::write16( p_outputFile, (unsigned short*)&thePlatformPointers.numTKeys, reverseByteOrder );
		}
		else
		{
			CStandardAnimFramePointers thePlatformPointers;
			thePlatformPointers.numQKeys = pBoneData->GetNumQKeys();
			thePlatformPointers.numTKeys = pBoneData->GetNumTKeys();

			IoUtils::write8( p_outputFile, (unsigned char*)&thePlatformPointers.numQKeys, reverseByteOrder );
			IoUtils::write8( p_outputFile, (unsigned char*)&thePlatformPointers.numTKeys, reverseByteOrder );
		}
	}

	// long-align
	p_outputFile->Align(4);

	// IoUtils::write the Q data
	if ( !write_q_data( p_outputFile, reverseByteOrder ) )
	{
		goto failure;
	}

	// IoUtils::write the T data
	if ( !write_t_data( p_outputFile, reverseByteOrder ) )
	{
		goto failure;
	}

	// long-align
	p_outputFile->Align(4);

	if ( ( m_customDataSize % 4 ) != 0 )
	{
		printf( "Custom data size not multiple of 4!" );
		goto failure;
	}

	for ( i = 0; i < m_customDataSize; i+=4 )
	{
		IoUtils::write32( p_outputFile, (unsigned int*)&mp_customData[i], reverseByteOrder );
	}

	if ( p_outputFile->TellPos() > ( 2048 * 1024 ) )
	{
		char msg[1024];
		sprintf( msg, "File %s was unusually large (%d bytes).", pFileName, p_outputFile->TellPos() );
		Utils::Assert( 0, msg );
		goto failure;
	}

	if ( !p_outputFile->Save( pFileName ) )
	{
		goto failure;
	}

	success = true;

failure:
	return success;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::write_q_data( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder )
{
	for ( int j = 0; j < m_numBones; j++ )
	{
		CBoneData* pBoneData = get_bone_data(j);
		SIntermediateQKey* pQKey = pBoneData->GetQKeys();

		for ( int i = 0; i < pBoneData->GetNumQKeys(); i++, pQKey++ )
		{
			int keyStart = pOutputFile->TellPos();

			float magnitude = sqrtf( pQKey->imag[0] * pQKey->imag[0]
									 + pQKey->imag[1] * pQKey->imag[1] 
									 + pQKey->imag[2] * pQKey->imag[2] 
									 + pQKey->real * pQKey->real );

			float qx = ( pQKey->imag[0] / magnitude );
			float qy = ( pQKey->imag[1] / magnitude );
			float qz = ( pQKey->imag[2] / magnitude );
			float qw = ( pQKey->real / magnitude );

			if ( is_hires_data() )
			{
				CHiResAnimQKey theKey;
				theKey.qx = qx;
				theKey.qy = qy;
				theKey.qz = qz;
				theKey.signBit = ( qw < 0.0f ) ? 1 : 0;
				theKey.timestamp = (short)pQKey->time;

				unsigned short bitShort;
				bitShort = ( theKey.timestamp & 0x7fff ) | ( theKey.signBit ?  0x8000 : 0x0000 );
				IoUtils::write16( pOutputFile, (unsigned short*)&bitShort, reverseByteOrder );

				unsigned short pad = 0;
				IoUtils::write16( pOutputFile, (unsigned short*)&pad, reverseByteOrder );

				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.qx, reverseByteOrder );
				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.qy, reverseByteOrder );
				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.qz, reverseByteOrder );
			}
			else
			{
				CStandardAnimQKey theKey;
				theKey.qx = quatToFixedPoint(qx);
				theKey.qy = quatToFixedPoint(qy);
				theKey.qz = quatToFixedPoint(qz);
				theKey.signBit = ( qw < 0.0f ) ? 1 : 0;
				theKey.timestamp = (short)pQKey->time;	

				unsigned short bitShort;
				bitShort = ( theKey.timestamp & 0x7fff ) | ( theKey.signBit ? 0x8000 : 0x0000 );
				IoUtils::write16( pOutputFile, (unsigned short*)&bitShort, reverseByteOrder );

				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qx, reverseByteOrder );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qy, reverseByteOrder );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qz, reverseByteOrder );
			}
		}
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::write_compressed_q_data( IoUtils::CVirtualOutputFile* pOutputFile, CompressTable* pCompressTable, bool reverseByteOrder, int patchAddress )
{
	if ( is_hires_data() )
	{
		Utils::Assert( 0, "Hi-res data is not compressible (yet)\n" );
		return false;
	}

	int frameSizes[vMAX_BONES];
	for ( int i = 0; i < vMAX_BONES; i++ )
	{
		frameSizes[i] = 0;
	}

	for ( int j = 0; j < m_numBones; j++ )
	{
		CBoneData* pBoneData = get_bone_data(j);
		SIntermediateQKey* pQKey = pBoneData->GetQKeys();

		for ( int i = 0; i < pBoneData->GetNumQKeys(); i++, pQKey++ )
		{
			int keyStart = pOutputFile->TellPos();

			float magnitude = sqrtf( pQKey->imag[0] * pQKey->imag[0]
									 + pQKey->imag[1] * pQKey->imag[1] 
									 + pQKey->imag[2] * pQKey->imag[2] 
									 + pQKey->real * pQKey->real );

			float qx = ( pQKey->imag[0] / magnitude );
			float qy = ( pQKey->imag[1] / magnitude );
			float qz = ( pQKey->imag[2] / magnitude );
			float qw = ( pQKey->real / magnitude );

			CStandardAnimQKey theKey;
			theKey.qx = quatToFixedPoint(qx);
			theKey.qy = quatToFixedPoint(qy);
			theKey.qz = quatToFixedPoint(qz);
			theKey.signBit = ( qw < 0.0f ) ? 1 : 0;
			theKey.timestamp = (short)pQKey->time;	

			unsigned short bitShort;

			bitShort = ( theKey.timestamp & 0x7fff ) | ( theKey.signBit ? 0x8000 : 0x0000 );

			int lookup = -1;
			if ( use_compress_table( pCompressTable ) )
			{
				lookup = pCompressTable->InUniqueTable( theKey.qx, theKey.qy, theKey.qz );
			}

			// mark off special flags
			bool doNotCompress = false;

			if ( (theKey.timestamp >= 0x0800) )
			{
				// flags will interfere
				doNotCompress = true;
			}
			else if ( lookup != -1 ) 
			{
				bitShort |= 0x4000;
			}
			else
			{
				if ( !( theKey.qx & 0xff00 ) )
				{
					bitShort |= 0x4000;
					bitShort |= 0x2000;
				}
				if ( !( theKey.qy & 0xff00 ) )
				{
					bitShort |= 0x4000;
					bitShort |= 0x1000;
				}
				if ( !( theKey.qz & 0xff00 ) )
				{
					bitShort |= 0x4000;
					bitShort |= 0x0800;
				}
			}

			IoUtils::write16( pOutputFile, (unsigned short*)&bitShort, 0 );			

			if ( !doNotCompress && lookup != -1 )
			{
				unsigned char lookupChar = (unsigned char)lookup;
				IoUtils::write8( pOutputFile, (unsigned char*)&lookupChar, reverseByteOrder );
				goto calc_q_size;
			}
			else
			{
				if ( !doNotCompress && !( theKey.qx & 0xff00 ) )
				{
					unsigned char qx = theKey.qx;
					IoUtils::write8( pOutputFile, (unsigned char*)&qx, 0 );
				}
				else
					{
					IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qx, 0 );
				}

				if ( !doNotCompress && !( theKey.qy & 0xff00 ) )
				{
					unsigned char qy = theKey.qy;
					IoUtils::write8( pOutputFile, (unsigned char*)&qy, 0 );	
				}
				else
				{
					IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qy, 0 );
				}

				if ( !doNotCompress && !( theKey.qz & 0xff00 ) )
				{	
					unsigned char qz = theKey.qz;
					IoUtils::write8( pOutputFile, (unsigned char*)&qz, 0 );
				}
				else	
				{
					IoUtils::write16( pOutputFile, (unsigned short*)&theKey.qz, 0 );
				}
			}

calc_q_size:
			int keySize = pOutputFile->TellPos() - keyStart;
			frameSizes[j] += keySize;
		}
	}

	// remember the position
	int currPos = pOutputFile->TellPos();

	// patch all the addresses here...
	pOutputFile->SeekPos( patchAddress );
	for ( int i = 0; i < m_numBones; i++ )
	{
		if ( frameSizes[i] > 0xffff )
		{
			printf( "frame offset too large!\n" );
			return false;
		}

		unsigned short frameSizeShort = (unsigned short)frameSizes[i];
		IoUtils::write16( pOutputFile, (unsigned short*)&frameSizeShort, reverseByteOrder );
	}

	// restore the position
	pOutputFile->SeekPos( currPos );

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::write_compressed_t_data( IoUtils::CVirtualOutputFile* pOutputFile, CompressTable* pCompressTable, bool reverseByteOrder, int patchAddress )
{
	if( is_hires_data() )
	{
		printf( "Hi res data is not compressible (yet)\n" );
		return false;
	}
	
	int frameSizes[vMAX_BONES];
	for ( int i = 0; i < vMAX_BONES; i++ )
	{
		frameSizes[i] = 0;
	}

	for ( int j = 0; j < m_numBones; j++ )
	{
		CBoneData* pBoneData = get_bone_data(j);
		SIntermediateTKey* pTKey = pBoneData->GetTKeys();

		for ( int i = 0; i < pBoneData->GetNumTKeys(); i++, pTKey++ )
		{
			int keyStart = pOutputFile->TellPos();

			CStandardAnimTKey theKey;
			theKey.tx = transToFixedPoint(pTKey->t[0]);
			theKey.ty = transToFixedPoint(pTKey->t[1]);
			theKey.tz = transToFixedPoint(pTKey->t[2]);
			theKey.timestamp = (short)pTKey->time;	

			int lookup = -1;
			if ( use_compress_table( pCompressTable ) )
			{
				lookup = pCompressTable->InUniqueTable( theKey.tx, theKey.ty, theKey.tz );
			}

			int lookupX = -1;
			int lookupY = -1;
			int lookupZ = -1;
			bool forceUncompressed = 0;

	#if 0
			// compression scheme:  converts shorts to chars (generates ANIMS.PRE of size 3050150/2140490)
			lookupX = (theKey.tx & 0xffffff00)? -1 : theKey.tx;
			lookupY = (theKey.ty & 0xffffff00)? -1 : theKey.ty;
			lookupZ = (theKey.tz & 0xffffff00)? -1 : theKey.tz;
	#endif

	#if 0
			// compression type:  stores frequently used shorts in lookup table (generates ANIMS.PRE of size 3041795/2131919)
			lookupX = in_small_table( m_tCompressTable, theKey.tx );
			lookupY = in_small_table( m_tCompressTable, theKey.ty );
			lookupZ = in_small_table( m_tCompressTable, theKey.tz );
	#endif

	#if 1
			// compression type:  timestamps as chars (3235513/2230645, without 48-bit table)
			// compression type:  timestamps as chars (3000633/2078475, with 48-bit table, but needs escape code with times > 0x7f)
			lookupX = -1;
			lookupY = -1;
			lookupZ = -1;
			forceUncompressed = 1;
	#endif

	/*
			IoUtils::write16( (unsigned short*)&theKey.timestamp, reverseByteOrder );
			IoUtils::write16( (unsigned short*)&theKey.tx, reverseByteOrder );
			IoUtils::write16( (unsigned short*)&theKey.ty, reverseByteOrder );
			IoUtils::write16( (unsigned short*)&theKey.tz, reverseByteOrder );
	*/

			unsigned char timeStampChar = 0;

			if ( lookup != -1 )
			{
				timeStampChar |= 0x80;
			}

			if ( theKey.timestamp <= 0x3f )
			{
				// this flag means that the timestamp fits completely...
				timeStampChar |= 0x40;
				timeStampChar |= theKey.timestamp;
			}

			// first char tells us whether there's a timestamp built into
			// it and tells us whether the next packet will be a lookup item
			IoUtils::write8( pOutputFile, (unsigned char*)&timeStampChar, reverseByteOrder );

			if ( theKey.timestamp > 0x3f )
			{
				theKey.timestamp &= 0x7fff;
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.timestamp, 0 );
			}

			if ( lookup != -1 )
			{
				if ( lookup < 0 && lookup >= 256 )
				{
					printf( "error\n" );
					return false;
				}
				unsigned char lookupChar = lookup;
				IoUtils::write8( pOutputFile, (unsigned char*)&lookupChar, reverseByteOrder );
			}
			else
			{
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.tx, 0 );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.ty, 0 );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.tz, 0 );
			}

	//calc_t_size:
			int keySize = pOutputFile->TellPos() - keyStart;
			frameSizes[j] += keySize;
		}
	}

	// remember the position
	int currPos = pOutputFile->TellPos();

	// patch all the addresses here...
	pOutputFile->SeekPos( patchAddress );
	for ( i = 0; i < m_numBones; i++ )
	{
		if ( frameSizes[i] > 0xffff )
		{
			printf( "frame offset too large!\n" );
			return false;
		}

		unsigned short frameSizeShort = (unsigned short)frameSizes[i];
		IoUtils::write16( pOutputFile, (unsigned short*)&frameSizeShort, reverseByteOrder );
	}

	// restore the position
	pOutputFile->SeekPos( currPos );
	
	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool AnimConverter::write_t_data( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder )
{
	for ( int j = 0; j < m_numBones; j++ )
	{
		CBoneData* pBoneData = get_bone_data(j);
		SIntermediateTKey* pTKey = pBoneData->GetTKeys();

		for ( int i = 0; i < pBoneData->GetNumTKeys(); i++, pTKey++ )
		{
			if ( is_hires_data() )
			{
				CHiResAnimTKey theKey;
				theKey.tx = pTKey->t[0];
				theKey.ty = pTKey->t[1];
				theKey.tz = pTKey->t[2];
				theKey.timestamp = (short)pTKey->time;	

				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.timestamp, reverseByteOrder );

				unsigned short pad = 0;
				IoUtils::write16( pOutputFile, (unsigned short*)&pad, reverseByteOrder );

				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.tx, reverseByteOrder );
				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.ty, reverseByteOrder );
				IoUtils::write32( pOutputFile, (unsigned int*)&theKey.tz, reverseByteOrder );
			}
			else
			{
				CStandardAnimTKey theKey;
				theKey.tx = transToFixedPoint(pTKey->t[0]);
				theKey.ty = transToFixedPoint(pTKey->t[1]);
				theKey.tz = transToFixedPoint(pTKey->t[2]);
				theKey.timestamp = (short)pTKey->time;	

				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.timestamp, reverseByteOrder );

				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.tx, reverseByteOrder );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.ty, reverseByteOrder );
				IoUtils::write16( pOutputFile, (unsigned short*)&theKey.tz, reverseByteOrder );
			}
		}
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/








#if 0
		// TODO:  Take a look why this assert fires off in the crossbone animation...

		// Some tests to make sure the unit quat compression code 
		// works without too much loss of precision...
		float original_w = ( p_intermediateqKeys[i].real / magnitude ) * signBit;
		float rebuilt_w = sqrtf( 1.0f - qx * qx - qy * qy - qz * qz );
		float diff_w = fabs( original_w - rebuilt_w );
		if ( diff_w > nxUNITQUAT_TOLERANCE_RADIANS )
		{
			Dbg_MsgAssert( 0, ( "%f %f %f %f %f %f %f Compressed W in animation is off by %f degrees (tolerance = %f degrees) in %s", 
								p_intermediateqKeys[i].imag[0],
								p_intermediateqKeys[i].imag[1],
								p_intermediateqKeys[i].imag[2],
								p_intermediateqKeys[i].real,
								magnitude,
								original_w,
								rebuilt_w,
								Mth::RadToDeg(diff_w),
								Mth::RadToDeg(nxUNITQUAT_TOLERANCE_RADIANS),
								m_fileName.getString() ) );
		}

#if 1
		if ( ( bitShort & 0xff00 ) == 0 )
		{
			// fits into a char...
			unsigned char timestamp = bitShort;
			IoUtils::write8( (unsigned char*)&timestamp, reverseByteOrder );
		}
		else
		{
			// otherwise, we need to IoUtils::write out an escape code
			unsigned char timestamp = 0xff;
			IoUtils::write8( (unsigned char*)&timestamp, reverseByteOrder );

			// and then the full short...
			IoUtils::write16( (unsigned short*)&bitShort, reverseByteOrder );			
		}
#else
		// and then the full short...
		IoUtils::write16( (unsigned short*)&bitShort, reverseByteOrder );			
#endif

		if ( /*!doNotCompress && */( lookup != -1 ) )
		{
			unsigned char lookupChar = (unsigned char)lookup; 
			IoUtils::write8( (unsigned char*)&lookupChar, reverseByteOrder );
			goto calc_t_size;
		}
		else
		{
#if 0
			IoUtils::write16( (unsigned short*)&theKey.tx, reverseByteOrder );
			IoUtils::write16( (unsigned short*)&theKey.ty, reverseByteOrder );
			IoUtils::write16( (unsigned short*)&theKey.tz, reverseByteOrder );
#else
			if ( !doNotCompress && ( lookupX != -1 ) )
			{
				unsigned char tx = (unsigned char)lookupX;
				IoUtils::write8( (unsigned char*)&tx, reverseByteOrder );
			}
			else
			{
				IoUtils::write16( (unsigned short*)&theKey.tx, reverseByteOrder );
			}

			if ( !doNotCompress && ( lookupY != -1 ) )
			{
				unsigned char ty = (unsigned char)lookupY;
				IoUtils::write8( (unsigned char*)&ty, reverseByteOrder );
			}
			else
			{
				IoUtils::write16( (unsigned short*)&theKey.ty, reverseByteOrder );
			}

			if ( !doNotCompress && ( lookupZ != -1 ) )
			{
				unsigned char tz = (unsigned char)lookupZ;
				IoUtils::write8( (unsigned char*)&tz, reverseByteOrder );
			}
			else
			{
				IoUtils::write16( (unsigned short*)&theKey.tz, reverseByteOrder );
			}
#endif


#endif



#if 0

// Generally, it's bad to remove duplicate Q keys
// (unless there's a long sequence of the same key...)

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimConverter::remove_duplicate_q()
{
	int currQKey = 0;

	// pack duplicates
	int firstKeyInThisBone = 0;
	int firstKeyInNextBone = 0;
	for ( int i = 0; i < m_numBones; i++ )
	{
		int qCount = 0;

		firstKeyInNextBone = firstKeyInThisBone + mp_perBoneFrames[i].numQKeys;
			
		float imag0 = mp_qKeys[firstKeyInThisBone].imag[0];
		float imag1 = mp_qKeys[firstKeyInThisBone].imag[1];
		float imag2 = mp_qKeys[firstKeyInThisBone].imag[2];
		float real = mp_qKeys[firstKeyInThisBone].real;

		for ( int j = firstKeyInThisBone; j < firstKeyInNextBone; j++ )
		{
			if ( !( mp_qKeys[j].imag[0] == imag0
				&& mp_qKeys[j].imag[1] == imag1
				&& mp_qKeys[j].imag[2] == imag2
				&& mp_qKeys[j].real == real )
				|| j == firstKeyInThisBone )
			{
				imag0 = mp_qKeys[j].imag[0];
				imag1 = mp_qKeys[j].imag[1];
				imag2 = mp_qKeys[j].imag[2];
				real = mp_qKeys[j].real;
				mp_qKeys[currQKey] = mp_qKeys[j];

				currQKey++;
				qCount++;
			}
		}

		mp_perBoneFrames[i].numQKeys = qCount;

		firstKeyInThisBone = firstKeyInNextBone;
	}

	int num_removed = m_numQKeys - currQKey;

	m_numQKeys = currQKey;

	return num_removed;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int AnimConverter::remove_duplicate_t()
{
	int currTKey = 0;

	// pack duplicates
	int firstKeyInThisBone = 0;
	int firstKeyInNextBone = 0;
	for ( int i = 0; i < m_numBones; i++ )
	{
		int tCount = 0;

		firstKeyInNextBone = firstKeyInThisBone + mp_perBoneFrames[i].numTKeys;
			
		float t0 = mp_tKeys[firstKeyInThisBone].t[0];
		float t1 = mp_tKeys[firstKeyInThisBone].t[1];
		float t2 = mp_tKeys[firstKeyInThisBone].t[2];

		for ( int j = firstKeyInThisBone; j < firstKeyInNextBone; j++ )
		{
			if ( !( mp_tKeys[j].t[0] == t0
				&& mp_tKeys[j].t[1] == t1
				&& mp_tKeys[j].t[2] == t2 )
				|| j == firstKeyInThisBone )
			{
				t0 = mp_tKeys[j].t[0];
				t1 = mp_tKeys[j].t[1];
				t2 = mp_tKeys[j].t[2];
				mp_tKeys[currTKey] = mp_tKeys[j];
				currTKey++;
				tCount++;
			}
		}

		mp_perBoneFrames[i].numTKeys = tCount;

		firstKeyInThisBone = firstKeyInNextBone;
	}

	int num_removed = m_numTKeys - currTKey;

	m_numTKeys = currTKey;

	return num_removed;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

#endif

#if 0
	// GJ:  Duplicate keys are actually valid sometimes
	// (like when you want to hold on a certain pose or 
	// a camera position for a few frames)
	if ( 0 )
	{
		int num_q_removed = remove_duplicate_q();
		int num_t_removed = remove_duplicate_t();

		if ( num_q_removed )
		{
//			printf( "Removed %d q-duplicates\n", num_q_removed );
		}
	
		if ( num_t_removed )
		{
//			printf( "Removed %d t-duplicates\n", num_t_removed );
		}
	}
#endif
