/*
	vagio.cpp
	Conversion routines for Sony's VAG Format
*/

#include "vagio.h"
#include <stdio.h>

#ifdef sony_news
#define sEndian(X) (X)
#define lEndian(X) (X)
#else
#define sEndian(X) ((0xff & (X>>8)) | ((0xff & X) << 8))
#define lEndian(X) (((X >> 24) & 0xff) | ((X >> 8) & 0xff00) | ((X & 0xff00) << 8) | ((X & 0xff) << 24))
#endif

#define VAG_TOKEN    0x56414770	// "VAGp"
#define VAG_VERSION  0x20L

#define BLKSIZ       28         // VAG block size is 28 bytes (from sample code)

bool SaveVAG(WAVEFORMATEX* fmt,char* filename,VAGheader* header,int mode,char* data,int size,int channel,bool bLoop)
{
	FILE* fp = NULL;

	if (fmt->nChannels == 1)
	{
		fp=fopen(filename,"wb");
	}
	else
	{
		char channel_filename[256];
		char *p_filetype;

		strcpy(channel_filename, filename);
		p_filetype = strrchr(channel_filename, '.');
		if (p_filetype)
		{
			*p_filetype = '\0';
		}

		// Append channel name
		if (channel == 0)
			strcat(channel_filename, "L");
		else
			strcat(channel_filename, "R");

		// Put filetype back
		p_filetype = strrchr(filename, '.');
		if (p_filetype)
		{
			strcat(channel_filename, p_filetype);
		}

		fp=fopen(channel_filename,"wb");
	}

	if (!fp)
	{
		printf("ERROR: Failed to write '%s'.\n",filename);
		return false;
	}

	int channelSize = size / fmt->nChannels;

	// Compute WAV size
	int numBlks = (channelSize / (BLKSIZ*2) ) + ( (channelSize % (BLKSIZ*2) != 0)  ? 1 : 0 );

	// compressed blocks end up as 16 bytes + start block + end block
//	if (!bLoop)
//		header->size = lEndian( (numBlks * 16 + 16 + 16) );
//	else
		header->size = lEndian( (numBlks * 16 + 16) );

	// Dump the header
	if (fwrite(header,sizeof(VAGheader),1,fp)==0)
	{
		printf("ERROR:d Failed to write VAG header\n");
		return false;
	}

	// Write waveform start block (16 bytes of nothingness)
	char sblock[16];
	memset(&sblock,0,16);
	if (fwrite(sblock,16,1,fp)==0)
	{
		printf("ERROR: Failed to write VAG waveform start block.\n");
		return false;
	}

	EncVagInit( mode );

	// Write waveform data  (each byte is considered a frame)
	for(int i=channel * 2;i<size;)
	{
		short block[BLKSIZ];
		short outblk[8];		// VAG output block (encoded) 16 bytes

		ZeroMemory(&block,sizeof(short)*BLKSIZ);
		ZeroMemory(&outblk,sizeof(short)*8);

		// Fill up vag input block  (28 bytes as shorts)
		for(int bf=0;bf<BLKSIZ;bf++)
		{
			char* blk=(char*)&(block[bf]);
			
			// Swap order within block section to account for change in endian ness
			// NOT changing endianness anymore
			
			if (i>=size)
				blk[0] = 0;
			else
				blk[0] = data[i];
				
			if (i+1>=size)
				blk[1] = 0;
			else
				blk[1] = data[i+1];

			i+=2 * fmt->nChannels;
		}

		// Determine block flag and do VAG encoding
		short flag;
		
		if (!bLoop)
		{
			if (i>=size)
				flag = ENC_VAG_1_SHOT_END;		// END flags on all but end block (following sample) ???
			else
				flag = ENC_VAG_1_SHOT;
		}
		else
		{
			if (i<=BLKSIZ*2)
				flag = ENC_VAG_LOOP_START;
			else if (i>=size)
				flag = ENC_VAG_LOOP_END;
			else
				flag = ENC_VAG_LOOP_BODY;
		}

		EncVag( block, outblk, flag );

		// Write out the encoded VAG block
		if (fwrite(outblk,sizeof(short)*8,1,fp)==0)
		{
			printf("ERROR: Failed to write VAG block at byte pos %i.\n",i);
			return false;
		}
	}

	// Write out end block if this is a non looping waveform
	if (!bLoop)
	{
		short endblk[8];
		EncVagFin( endblk );

		// Write out the VAG end block
		if (fwrite(endblk,sizeof(short)*8,1,fp)==0)
		{
			printf("ERROR: Failed to write VAG end block\n");
			return false;
		}
	}

	fclose(fp);

	return true;
}

bool ConvertVAG(WAVEFORMATEX* fmt,char* name,int mode,char* stream,int size,bool bLoop)
{
	VAGheader header;
	ZeroMemory(&header,sizeof(VAGheader));

	header.format = lEndian( VAG_TOKEN );					// File ID Format token
	header.ver    = lEndian( VAG_VERSION );					// VAG format version
	header.ssa    = 0;										// Start position of waveform data after header
	header.size   = 0;				;						// Size of waveform data (filled in by save VAG)
	header.fs     = lEndian( (long)fmt->nSamplesPerSec );	// Sample rate

	char filename[256];
	char ext[80];

	_splitpath(name,NULL,NULL,filename,ext);

	// We have 16 bytes to specify the file name, make sure we don't go over 16th byte must be NULL
	for(UINT i=0;i<strlen(filename) && i<15;i++)
		header.name[i] = filename[i];
	
	bool success = true;
	for (int channel = 0; channel < fmt->nChannels; channel++)
	{
		success = success && SaveVAG(fmt,name,&header,mode,stream,size,channel,bLoop);
	}

	return success;
}
