/*
	ResizeWindow.h
	Standard Resizeable window
	This class will automatically resize all the child windows on
	the main window to be proportionate to the the size of a
	resizedn window
*/

#ifndef __RESIZEWINDOW__
#define __RESIZEWINDOW__

#include "MSWindow.h"

#define RSW_SCALE_X			0x0001	// Scales the child window on the X axis
#define RSW_SCALE_Y			0x0002	// Scales the child window on the Y axis
#define RSW_LOCK_X			0x0004	// The leftmost X point of the child window shouldn't move
#define RSW_LOCK_Y			0x0008	// The topmost Y point of the child window shouldn't move

#define RSW_ALIGN_TOP		0x0010	// The child window should be aligned to the top of the reference window
									// after resize operations have completed

#define RSW_ALIGN_BOTTOM	0x0020	// The child window should be aligned to the bottom of the reference
									// window after resize operations have completed

#define RSW_ALIGN_LEFT      0x0040	// Like above but to left of reference window
#define RSW_ALIGN_RIGHT     0x0080	// Like above but to right of reference window

struct ResizeRecord
{
	int   ID;			// The window identifier
	int   refIDtop;		// The window identifier referring to the position this resize mode depends on to work
	int   refIDbottom;	// same for align bottom
	int   refIDleft;	// same for align left
	int   refIDright;	// same for alight right
	WORD  mode;			// The resize operation(s) to perform
	POINT ptTL;			// Top left point of the child window in the parent's client space
	POINT ptBR;			// Bottom right point of the child window in the parent's client space
	int   width;		// width of the child window in client space
	int   height;		// height of the child window in client space
};


class ResizeDlgWindow: public MSDlgWindow
{
	int numIDs;
	int lastRecord;
	ResizeRecord* resizeDB;

	int sampleWidth;		// The width of the parent window at time of SnapShot
	int sampleHeight;		// The height of the parent window at time of SnapShot

	ResizeRecord* Find(int ID)
	{
		for(int i=0;i<numIDs;i++)
			if (resizeDB[i].ID==ID)
				return resizeDB+i;

		return NULL;
	}

	static BOOL CALLBACK CountChildren(HWND hwnd,LPARAM lParam)
	{
		int* count=(int*)lParam;
		(*count)++;

		return TRUE;
	}

	static BOOL CALLBACK InitDB(HWND childHwnd,LPARAM lParam)
	{
		ResizeDlgWindow* pthis=(ResizeDlgWindow*)lParam;

		int   rec=pthis->lastRecord;
		RECT  rect;
		POINT ptTL,ptBR;
		pthis->resizeDB[rec].ID=(int)GetWindowLong(childHwnd,GWL_ID);
		
		GetWindowRect(childHwnd,&rect);
		ptTL.x=rect.left;
		ptTL.y=rect.top;
		ptBR.x=rect.right;
		ptBR.y=rect.bottom;

		// Convert the points 
		ScreenToClient(pthis->hwnd,&ptTL);
		ScreenToClient(pthis->hwnd,&ptBR);

		pthis->resizeDB[rec].ptTL=ptTL;
		pthis->resizeDB[rec].ptBR=ptBR;

		pthis->resizeDB[rec].width  = ptBR.x-ptTL.x;
		pthis->resizeDB[rec].height = ptBR.y-ptTL.y;

		// Assign default resize mode
		pthis->resizeDB[rec].mode=RSW_SCALE_X|RSW_SCALE_Y;
		pthis->resizeDB[rec].refIDtop=0;
		pthis->resizeDB[rec].refIDbottom=0;
		pthis->resizeDB[rec].refIDleft=0;
		pthis->resizeDB[rec].refIDright=0;

		// Advance record
		pthis->lastRecord++;

		return TRUE;
	}

protected:

	virtual BOOL DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
	{
		switch(msg)
		{
		case WM_SIZE:
			ResizeWindow(LOWORD(lParam),HIWORD(lParam));
			return TRUE;
		};

		return FALSE;
	}

public:
	ResizeDlgWindow(HINSTANCE hInstance,
					LPCTSTR   lpTemplate,
					HWND      hwndParent) : MSDlgWindow(hInstance,lpTemplate,hwndParent)
	{
		resizeDB=NULL;
		Snapshot();
	}

	~ResizeDlgWindow()
	{
		if (resizeDB)
			delete [] resizeDB;
	}

	// This method captures the original coordinates of each child window
	// so that the relative relationship to the parent can be maintained 
	// when scaling each child window
	void Snapshot()
	{
		// Grab the sample width/height of the parent window
		RECT winRect;
		GetWindowRect(hwnd,&winRect);

		sampleWidth  = winRect.right-winRect.left;
		sampleHeight = winRect.bottom-winRect.top;

		// Determine the number of child windows
		int numChildren=0;
		EnumChildWindows(hwnd,CountChildren,(LPARAM)&numChildren);

		// Allocate the resize database
		if (resizeDB)
			delete [] resizeDB;
		else
			resizeDB=new ResizeRecord[numChildren];

		numIDs = numChildren;
		lastRecord = 0;

		EnumChildWindows(hwnd,InitDB,(LPARAM)this);
	}

	// This should contain a list of window identifiers and their desired resize mode
	// If this command isn't called all child windows will be sized proportionately
	// on both the X and Y axis (SCALE_XY)
	bool SetResizeModes(int num,int ID,int mode,int refIDtop,int refIDbottom,int refIDleft,int refIDright,...)
	{
		if (numIDs>numIDs)
			return false;
		
		int* pos=&ID;

		for(int i=0;i<numIDs;i++)
		{
			ResizeRecord* record=Find(ID);

			if (!record)
				return false;

			record->mode        = mode;
			record->refIDtop    = refIDtop;
			record->refIDbottom = refIDbottom;
			record->refIDleft   = refIDleft;
			record->refIDright  = refIDright;

			// Advance to next argument in variable length list
			pos += 6;	// 6 args per record
			ID          = *pos;
			mode        = *(pos+1);
			refIDtop    = *(pos+2);
			refIDbottom = *(pos+3);
			refIDleft   = *(pos+4);
			refIDright  = *(pos+5);
		}

		return true;
	}

	void ResizeWindow(int width,int height)
	{
		for(int i=0;i<numIDs;i++)
		{
			HWND childHwnd = GetDlgItem(hwnd,resizeDB[i].ID);

			POINT newTL = resizeDB[i].ptTL;
			POINT newBR = resizeDB[i].ptBR;

			if (!(resizeDB[i].mode & RSW_LOCK_X))
				newTL.x = newTL.x*width/sampleWidth;

			if (resizeDB[i].mode & RSW_SCALE_X)
				newBR.x = newBR.x*width/sampleWidth;
			else
				newBR.x = newTL.x+resizeDB[i].width;

			if (!(resizeDB[i].mode & RSW_LOCK_Y))
				newTL.y = newTL.y*height/sampleHeight;

			if (resizeDB[i].mode & RSW_SCALE_Y)
				newBR.y = newBR.y*height/sampleHeight;
			else
				newBR.y = newTL.y+resizeDB[i].height;

			// Resize the child window
			MoveWindow(childHwnd,newTL.x,newTL.y,
				                 newBR.x-newTL.x,
								 newBR.y-newTL.y,
								 FALSE);
		}

		InvalidateRect(hwnd,NULL,TRUE);
	}
};

#endif
