/*
	ColorListBox.cpp
	A list box control that supports colored entries
	aml 4-24-01
*/

#include "ColorListBox.h"
#define COLORLISTBOX_NAME   "ColorListBox"
#define LBS_FLAGS  WS_CHILD|WS_BORDER|WS_VSCROLL|LBS_OWNERDRAWFIXED|LBS_HASSTRINGS|LBS_NOTIFY|WS_VISIBLE

struct LBData
{
	void*    extdata;		// Extended data attached to the item
	COLORREF color;			// The color that the item should be displayed in in the list
};

ColorListBox::ColorListBox(HINSTANCE hInstance)
{
	this->hInstance = hInstance;
	hFont  = NULL;
	hwnd   = NULL;
	hwndLB = NULL;
	Register(hInstance);
}

ColorListBox::ColorListBox(HINSTANCE hInstance, HWND hwnd,DWORD flags)
{
	this->hInstance = hInstance;
	hFont = NULL;
	Register(hInstance);
	Attach(hwnd,flags);
}

ColorListBox::~ColorListBox()
{
	// To ensure we don't have any LBData's floating around
	SendMessage(hwnd,LB_RESETCONTENT,0,0);

	if (hFont)
		DeleteObject(hFont);
}

void ColorListBox::Attach(HWND hwnd, DWORD flags)
{
	if (hwndLB)
		DestroyWindow(hwndLB);

	this->hwnd = hwnd;
	SetWindowLong(hwnd,GWL_USERDATA,(LONG)this);
	BuildUI(flags);
}

void ColorListBox::SetStyle(DWORD flags)
{
	SetWindowLong(hwnd,GWL_STYLE,(LONG)LBS_FLAGS|flags);
}

void ColorListBox::BuildUI(DWORD flags)
{
	// Create the listbox window
	// It will inherit all the flags from the parent window
	
	if (flags == 0)
		flags = GetWindowLong(hwnd,GWL_STYLE);

	RECT  wndRect;

	GetClientRect(hwnd, &wndRect);

	hwndLB = CreateWindow("listbox",
		                  "",
						  LBS_FLAGS|flags,
						  0,
						  0,
						  wndRect.right - wndRect.left,
						  wndRect.bottom - wndRect.top,
						  hwnd,
						  NULL,
						  hInstance,
						  this);

	// Set a font that's more appealing than the default (yuck!)
	if (hFont)
		DeleteObject(hFont);

	//hFont = CreateFont(14,0,0,0,0,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, "");
	//hFont = CreateFont(18,0,0,0,0,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, "");
	//SendMessage(hwndLB,WM_SETFONT,(WPARAM)hFont,MAKELPARAM(FALSE,0));

	ShowWindow(hwndLB, SW_SHOW);
}

void ColorListBox::Register(HINSTANCE hInstance)
{
	WNDCLASS wndclass;
	ZeroMemory(&wndclass,sizeof(WNDCLASS));

	wndclass.style         = CS_HREDRAW | CS_VREDRAW;
	wndclass.lpfnWndProc   = WndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = COLORLISTBOX_NAME;

	RegisterClass(&wndclass);
}

void ColorListBox::UnRegister(HINSTANCE hInstance)
{
	UnregisterClass(COLORLISTBOX_NAME,hInstance);
}

int ColorListBox::AddItem(char* str,COLORREF color,DWORD data)
{
	// We'll let our wndproc do all the work
	int index = SendMessage(hwnd,LB_ADDSTRING,0,(LPARAM)str);

	if (index != LB_ERR)
	{
		SendMessage(hwnd,LB_SETITEMDATA,(WPARAM)index,(LPARAM)data);
		SendMessage(hwnd,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)color);
	}

	return index;
}

void ColorListBox::SelItem(int index)
{
	SendMessage(hwnd,LB_SETSEL,(WPARAM)TRUE,(LPARAM)index);
}

bool ColorListBox::IsSelected(int index)
{
	return (SendMessage(hwnd,LB_GETSEL,(WPARAM)index,0) ? true : false);
}

void ColorListBox::UnSelItem(int index)
{
	SendMessage(hwnd,LB_SETSEL,(WPARAM)FALSE,(LPARAM)index);
}

int  ColorListBox::GetCount()
{
	return SendMessage(hwnd,LB_GETCOUNT,0,0);
}

void ColorListBox::RemoveItem(int index)
{
	// Let our wndproc handle it
	SendMessage(hwnd,LB_DELETESTRING,(WPARAM)index,0);
}

DWORD ColorListBox::GetItemData(int index)
{
	return SendMessage(hwnd,LB_GETITEMDATA,(WPARAM)index,0);
}

void ColorListBox::SetItemData(int index,DWORD val)
{
	SendMessage(hwnd,LB_SETITEMDATA,(WPARAM)index,(LPARAM)val);
}

void ColorListBox::SetItemColor(int index, COLORREF color)
{
	SendMessage(hwnd,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)color);
}

COLORREF ColorListBox::GetItemColor(int index)
{
	COLORREF cref;
	SendMessage(hwnd,LB_GETITEMCOLOR,(WPARAM)index,(LPARAM)&cref);

	return cref;
}

void ColorListBox::Clear()
{
	SendMessage(hwnd,LB_RESETCONTENT,0,0);
}

void ColorListBox::GetItem(int index,int len,char* val)
{
	char* buf;
	int   txtlen = (int)SendMessage(hwnd,LB_GETTEXTLEN,(WPARAM)index,0);

	buf = new char[txtlen+1];
	SendMessage(hwnd,LB_GETTEXT,(WPARAM)index,(LPARAM)buf);

	for(int i=0;i<txtlen && i<len;i++)
		val[i] = buf[i];
	
	val[i] = '\0';
}

LRESULT CALLBACK ColorListBox::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	ColorListBox* pthis = (ColorListBox*)GetWindowLong(hwnd,GWL_USERDATA);

	switch(msg)
	{
	case WM_CREATE:
		{
			SetWindowLong(hwnd, GWL_USERDATA, (LONG)((CREATESTRUCT*)lParam)->lpCreateParams);
			pthis = (ColorListBox*)((CREATESTRUCT*)lParam)->lpCreateParams;

			if (pthis)
				pthis->Attach(hwnd);
		}
		return TRUE;

	case WM_DESTROY:
		{
			if (pthis)
			{
				int count = SendMessage(pthis->hwndLB,LB_GETCOUNT,0,0);

				for(int i=0;i<count;i++)
				{
					LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,(WPARAM)i,0);
					
					if (lbdata && (LRESULT)lbdata != LB_ERR)
						delete lbdata;
				}

				// Forward message
				//return SendMessage(pthis->hwndLB,msg,wParam,lParam);						
			}
		}
		return TRUE;

	// Handle Notifications
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case LBN_DBLCLK:
		case LBN_ERRSPACE:
		case LBN_KILLFOCUS:
		case LBN_SELCANCEL:
		case LBN_SELCHANGE:
		case LBN_SETFOCUS:
			{
				HWND hwndParent = GetParent(hwnd);
				int  ID = (int)GetWindowLong(hwnd,GWL_ID);

				SendMessage(hwndParent,msg,(WPARAM)MAKEWPARAM(ID,HIWORD(wParam)),(LPARAM)hwnd);
				return TRUE;
			}
		}
		break;

	case LB_SELITEMRANGE:
	case LB_ITEMFROMPOINT:
	case LB_SELECTSTRING:
	case LB_SETCURSEL:
	case LB_SETSEL:
	case LB_GETSEL:
	case LB_GETCOUNT:
	case LB_GETCURSEL:
	case LB_GETSELCOUNT:
	case LB_GETSELITEMS:
	case LB_GETTEXT:
	case LB_GETTEXTLEN:
		{
			// Forward message
			return SendMessage(pthis->hwndLB,msg,wParam,lParam);
		}

	case LB_RESETCONTENT:
		{
			int count = SendMessage(pthis->hwndLB,LB_GETCOUNT,0,0);

			for(int i=0;i<count;i++)
			{
				LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,(WPARAM)i,0);
				
				if (lbdata)
					delete lbdata;
			}

			// Forward message
			return SendMessage(pthis->hwndLB,msg,wParam,lParam);
		}

	case LB_DELETESTRING:
		{
			// Delete our extended color data first
			LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,wParam,0);
			if (lbdata)
				delete lbdata;

			// Forward message
			return SendMessage(pthis->hwndLB,msg,wParam,lParam);
		}

	case LB_GETITEMDATA:
		{
			LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,wParam,0);
			if (lbdata && (LRESULT)lbdata != LB_ERR)
				return (LRESULT)lbdata->extdata;

			return LB_ERR;
		}

	case LB_SETITEMDATA:
		{
			LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,wParam,0);
			if (lbdata)
			{
				lbdata->extdata = (void*)lParam;
				return 0;
			}

			return LB_ERR;
		}
		return TRUE;

	case LB_ADDSTRING:
		{
			if (pthis)
			{
				int index = SendMessage(pthis->hwndLB,msg,wParam,lParam);

				if (index != LB_ERR)
				{
					// Add our color information
					// This gets cleaned up in RESETCONTENT, DELETESTRING, and DESTROY messages
					LBData* lbdata = new LBData;
					lbdata->extdata = NULL;
					lbdata->color   = RGB(0,0,0);

					SendMessage(pthis->hwndLB,LB_SETITEMDATA,(WPARAM)index,(LPARAM)lbdata);
				}

				return index;
			}
		
			return LB_ERR;
		}

	case LB_SETITEMCOLOR:
		{
			if (pthis)
			{
				LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,wParam,0);
				
				if (lbdata && (DWORD)lbdata != LB_ERR)
				{
					lbdata->color = (COLORREF)lParam;
					return TRUE;
				}
			}

			return FALSE;
		}

	case LB_GETITEMCOLOR:
		{
			if (pthis)
			{
				LBData* lbdata = (LBData*)SendMessage(pthis->hwndLB,LB_GETITEMDATA,wParam,0);

				if (lbdata && (DWORD)lbdata != LB_ERR)
				{
					if (lParam)
					{
						*((COLORREF*)lParam) = lbdata->color;
						return TRUE;
					}
				}
			}

			return FALSE;
		}

	case WM_DRAWITEM:
		{
			DRAWITEMSTRUCT* drawInfo = (DRAWITEMSTRUCT*)lParam;
			HBRUSH bkgBrush;

			// Don't draw if the space is empty
			if (drawInfo->itemID == -1)
				return TRUE;

			// Get the text we're supposed to render
			int txtLen = SendMessage(drawInfo->hwndItem,LB_GETTEXTLEN,(WPARAM)drawInfo->itemID,0);
			char* itemText = new char[txtLen+1];
			SendMessage(drawInfo->hwndItem,LB_GETTEXT,(WPARAM)drawInfo->itemID,(LPARAM)itemText);

			HDC hdc = drawInfo->hDC;

			if (drawInfo->itemData)
			{
				LBData* lbdata = (LBData*)SendMessage(drawInfo->hwndItem,LB_GETITEMDATA,(WPARAM)drawInfo->itemID,0);
				SetTextColor(hdc, lbdata->color);
			}
			else
				SetTextColor(hdc, RGB(0,0,0));

			if (drawInfo->itemState & ODS_SELECTED)
			{
				bkgBrush = CreateSolidBrush(RGB(195,220,239));
				SetBkColor(hdc, RGB(195,220,239));
			}
			else
			{
				bkgBrush = CreateSolidBrush(RGB(255,255,255));
				SetBkColor(hdc, RGB(255,255,255));
			}

			FillRect(hdc,&drawInfo->rcItem,bkgBrush);
			DrawText(hdc,itemText,strlen(itemText),&drawInfo->rcItem,DT_LEFT);

			if (drawInfo->itemState & ODS_FOCUS)
				DrawFocusRect(hdc,&drawInfo->rcItem);

			if (itemText)
				delete itemText;

			DeleteObject(bkgBrush);
		}
		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}
