/*
	LODBrowser.cpp
	LOD Browser popup window
	6-7-01
*/

#include "PropEdit.h"
#include "LODBrowser.h"
#include "../UI/PropList.h"
#include "ParseFuncs.h"
#include "appdata.h"
#include "Resource.h"
#include "ConfigData.h"
#include "next.h"
#include "../Misc/GenCrc.h"

extern Interface* gInterface;

LODBrowser::LODBrowser(HINSTANCE hInstance,HWND hwndParent,Interface* ip, PropEditor* pe) :
	MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_LODBROWSER),hwndParent)
{
	this->ip = ip;
	this->pe = pe;

	plist = new PropList(hInstance,1);
	plist->Attach(GetDlgItem(hwnd,IDC_LODLIST));
	plist->SetStaticCB(SubPropProc,this);

	numLODs = 0;

	bListUpdateLock = false;
	bLockShow = false;

	/*
	plist->AddNode("Master","The master node that contains the LOD system information.  This is LOD level 0");
	plist->SetChangeCB(ListUpdate,this);
	plist->HasApply(FALSE);
	
	plist->BuildUI();

	nodeMaster = GetNodeMaster();

	if (nodeMaster)
		ReadLODData();

	plist->GetValues(&lastLODlist);
	*/
#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(SelChange,this,NOTIFY_SELECTIONSET_CHANGED);
#endif

	// Acquire the right click popup menu
	hPopupMenu    = LoadMenu(hInstance,MAKEINTRESOURCE(IDR_LODPOPUP));	
	hPopupSubMenu = GetSubMenu(hPopupMenu,0);
}

LODBrowser::~LODBrowser()
{
	UnRegisterNotification(SelChange,this,NOTIFY_SELECTIONSET_CHANGED);
	DestroyMenu(hPopupSubMenu);
	DestroyMenu(hPopupMenu);
}

void LODBrowser::Show()
{
	if (bLockShow)
		return;

	bLockShow = true;
	bListUpdateLock = true;

	plist->Clear();

	plist->AddNode("Master","The master node that contains the LOD system information.  This is LOD level 0");
	plist->SetChangeCB(ListUpdate,this);
	plist->HasApply(FALSE);

	numLODs = 0;

	nodeMaster = GetNodeMaster();

	if (nodeMaster)
		ReadLODData();
	else
		plist->BuildUI();

#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(SelChange,this,NOTIFY_SELECTIONSET_CHANGED);
#endif

	MSDlgWindow::Show();

	bLockShow = false;
	bListUpdateLock = false;
}

void LODBrowser::Hide()
{
	bLockShow = true;

	UnRegisterNotification(SelChange,this,NOTIFY_SELECTIONSET_CHANGED);
	StoreLODData();
	plist->Clear();

	MSDlgWindow::Hide();

	bLockShow = false;
}

INode* LODBrowser::GetNodeMaster()
{
	// The first selected node in the current selection set will be used as the master
	int nNodes = ip->GetSelNodeCount();

	if (nNodes > 0)
	{	
		for(int curnode=0;curnode<nNodes;curnode++)
		{
			INode* node = ip->GetSelNode(curnode++);

			// Determine the node master
			AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);

			if (appdata && appdata->data)
			{
				ULONG masterID = *((ULONG*)appdata->data);
				return ip->GetINodeByHandle(masterID);
			}
		}
	}

	return NULL;
}

BOOL LODBrowser::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_ACTIVATE:
		if (LOWORD(wParam)==WA_INACTIVE)
			EnableAccelerators();
		else
			DisableAccelerators();

		return TRUE;

	case WM_CLOSE:
		Hide();
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDM_ADDLOD:
			AddLOD();
			return TRUE;

		case IDM_SELMASTER:
			SelMaster();
			return TRUE;

		case IDM_SELCHILDREN:
			SelChildren();
			return TRUE;

		case IDM_SELALL:
			SelAll();
			return TRUE;

		case IDM_CLOSE:
			Hide();
			return TRUE;
		}
	}

	return FALSE;
}

void LODBrowser::ListUpdate(PropList* plist,void* pdata)
{
	LODBrowser* pthis = (LODBrowser*)pdata;

	if (pthis->bListUpdateLock)
		return;

	CStr value;
	plist->GetValue(0,value);

	pthis->nodeMaster = pthis->ip->GetINodeByName(value);

	if (!pthis->nodeMaster)
	{
		// Save all the previous values in the list, so we can unlink nodes removed from the system later
		// should the need arise
		pthis->lastLODlist.Clear();
		plist->GetValues(&pthis->lastLODlist);
		return;
	}

	// Check if the master object was updated
	int lastMod = plist->GetLastMod();

	if (lastMod == 0 && pthis->lastLODlist.GetSize() > 0 && value != pthis->lastLODlist[0])
	{
		pthis->ip->ClearPickMode();
		pthis->lastLODlist.Clear();
		pthis->plist->Clear();
		pthis->plist->AddNode("Master","The LOD_Level0");
		pthis->plist->SetValue(0,value);
		pthis->bListUpdateLock = true;
		pthis->ReadLODData();
		pthis->bListUpdateLock = false;
		pthis->plist->ResetUpdateStatus();

		// Save all the previous values in the list, so we can unlink nodes removed from the system later
		// should the need arise
		pthis->lastLODlist.Clear();
		plist->GetValues(&pthis->lastLODlist);
		return;
	}

	// Unlink a node (if necessary)
	if (lastMod != -1 && lastMod < pthis->lastLODlist.GetSize())
	{
		// See if the last modified entry refers to an LOD node object
		if (IsInstr(plist->GetName(lastMod),"LOD_Level"))
		{
			// Now see if it's value has changed
			CStr value;
			plist->GetValue(lastMod,value);

			if (value != pthis->lastLODlist[lastMod])
			{
				// The value has changed unlink the old node from the system
				INode* node = pthis->ip->GetINodeByName(pthis->lastLODlist[lastMod]);

				if (node)
					node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);
			}
		}
	}

	// Reference stage
	// Scan through all the LOD nodes, and make sure they have a vNAPP_LOD_MASTER_ID chunk containing
	// the node ID for the master LOD object

	ULONG masterID    = pthis->nodeMaster->GetHandle();

	int nProps = pthis->plist->NumProps();
	for(int i=1;i<nProps;i++)
	{
		CStr name = pthis->plist->GetName(i);
		if (pthis->plist->GetValue(i,value))
		{
			if (IsInstr(name,"LOD_Level"))
			{
				INode* node = pthis->ip->GetINodeByName(value);

				if (node)
				{
					void* data = malloc(sizeof(ULONG));
					memcpy(data,&masterID,sizeof(ULONG));

					node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);
					node->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID,sizeof(ULONG),data);
				}
			}

			// Assign all LOD properties to the master node
			if (value == CStr(""))
				value = "[None]";

			pthis->nodeMaster->SetUserPropString(name,value);
		}
	}

	CStr propBuffer;
	pthis->nodeMaster->GetUserPropBuffer(propBuffer);
	pthis->UpdateLODDynUI(propBuffer);
	pthis->nodeMaster->SetUserPropBuffer(propBuffer);

	int nNodes = pthis->ip->GetSelNodeCount();

	if (nNodes > 0)
	{
		for(int i=0;i<nNodes;i++)
		{
			INode* node = pthis->ip->GetSelNode(i);

			if (node == pthis->nodeMaster)
			{
				pthis->pe->Update();
				break;
			}
		}
	}

	// Save all the previous values in the list, so we can unlink nodes removed from the system later
	// should the need arise
	pthis->lastLODlist.Clear();
	plist->GetValues(&pthis->lastLODlist);
}

void LODBrowser::StoreLODData()
{
	if (!nodeMaster)
		return;

	int nProps = plist->NumProps();

	for(int i=1;i<nProps;i++)
	{
		CStr value;
		CStr name = plist->GetName(i);

		plist->GetValue(i,value);

		if (value == CStr(""))
			value = "[None]";

		nodeMaster->SetUserPropString(name,value);
	}
}

void LODBrowser::ReadLODData()
{	
	LinkList<ConfigProp> cprops;
	CStr                 propBuffer;
	
	numLODs = 0;

	if (!nodeMaster)
		return;

	nodeMaster->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);

	Link<ConfigProp>* curprop = cprops.GetHead();

	// Set the name of the LOD nodeMaster
	plist->Clear();
	plist->AddNode("Master","The master node that contains the LOD system information.  This is LOD level 0");
	plist->SetValue(0,nodeMaster->GetName());

	while(curprop)
	{
		if (IsInstr(curprop->data.name,"LOD_Level"))
		{
			plist->AddNode(curprop->data.name,"The LOD Representation node for the specified LOD level");
			plist->SetValue(plist->GetCurIndex(),curprop->data.value);
			numLODs++;
		}

		if (IsInstr(curprop->data.name,"LOD_Dist"))
		{
			plist->AddSpinEdit(curprop->data.name,0,5000,1,"Distances greater than this value will use the model representation in the corresponding slot");
			plist->SetValue(plist->GetCurIndex(),curprop->data.value);
		}

		curprop = curprop->next;
	}

	plist->DestroyUI();
	plist->BuildUI();
}

void LODBrowser::AddLOD()
{
	char bufName[256];
	char bufDesc[256];
	sprintf(bufName,"LOD_Level%i",++numLODs);
	sprintf(bufDesc,"This is LOD Level Number %i",numLODs);

	plist->SaveValues();
	plist->DestroyUI();
	plist->AddNode(bufName,bufDesc);

	sprintf(bufName,"LOD_Dist%i",numLODs);
	sprintf(bufDesc,"Distances greater than this value will use the model representation in slot LOD_Level%i",numLODs);

	plist->AddSpinEdit(bufName,0,5000,1,bufDesc);
	plist->BuildUI();
	plist->RestoreValues();
}

void LODBrowser::UpdateLODDynUI(CStr& propBuffer)
{
	// Scan the buffer for any LOD DynUI commands
	// If an LOD DynUI command doesn't exist for a particular level add it
	CStr lcBuffer = propBuffer;
	CStr strAdd;
	lcBuffer.toLower();
	
	for(int i=0;i<numLODs;i++)
	{
		char bufLevelName[256];
		char bufDistName[256];
		CStr eol = "\r\n";

		sprintf(bufLevelName,"LOD_Level%i",i+1);
		sprintf(bufDistName,"LOD_Dist%i",i+1);

		CStr strSearch;
		strSearch = CStr("// @nextparm | ") + CStr(bufLevelName) + CStr(" | node");
		strAdd = strSearch;
		strSearch.toLower();

		if (!IsInstr(lcBuffer,strSearch))
			propBuffer += strAdd + eol;
		
		strSearch = CStr("// @nextparm | ") + CStr(bufDistName) + CStr(" | spinedit | 0 | 5000 | 1");
		strAdd = strSearch;
		strSearch.toLower();

		if (!IsInstr(lcBuffer,strSearch))
			propBuffer += strAdd + eol;

		strSearch = CStr("// @nextint | ") + CStr(bufLevelName);
		strAdd = strSearch;
		strSearch.toLower();

		if (!IsInstr(lcBuffer,strSearch))
			propBuffer += strAdd + eol;

		strSearch = CStr("// @nextint | ") + CStr(bufDistName);
		strAdd = strSearch;
		strSearch.toLower();

		if (!IsInstr(lcBuffer,strSearch))
			propBuffer += strAdd + eol;
	}
}

bool LODBrowser::HasLODData(INode* node)
{
	CStr propBuffer;
	LinkList<ConfigProp> cprops;

	node->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);

	Link<ConfigProp>* curprop = cprops.GetHead();

	while(curprop)
	{
		if (strstr(curprop->data.name, "LOD_Level"))
			return true;

		curprop = curprop->next;
	}

	return false;
}

void LODBrowser::SelChange(void *param, NotifyInfo *info)
{
	OutputDebugString("HANDLER: SelChange (LODBrowser)\n");
	LODBrowser* pthis = (LODBrowser*)param;

	if (IsWindowVisible(pthis->hwnd))
	{
		if (pthis->bListUpdateLock)
			return;

		pthis->bListUpdateLock = true;

		pthis->plist->Clear();
		pthis->plist->AddNode("Master","The master node that contains the LOD system information.  This is LOD level 0");

		pthis->nodeMaster = pthis->GetNodeMaster();

		// If we haven't selected a child check if we've selected a master
		if (!pthis->nodeMaster)
		{
			if (pthis->ip->GetSelNodeCount() == 1)
			{
				INode* node = pthis->ip->GetSelNode(0);
				
				if (pthis->HasLODData(node))
					pthis->nodeMaster = node;
			}
		}

		if (pthis->nodeMaster)
		{
			pthis->plist->SetValue(0,pthis->nodeMaster->GetName());
			pthis->ReadLODData();
		}
		else
		{
			pthis->lastLODlist.Clear();
			pthis->plist->BuildUI();
		}

		pthis->plist->GetValues(&pthis->lastLODlist);

		pthis->bListUpdateLock = false;
	}
}

bool LODBrowser::UpdateLODs(LinkList<ConfigProp>* list,PropList* plist,INode* nodeApply)
{
	// If a LOD node is changed the former LOD node must be removed from the system
	bool bUpdate = false;
	int numProps = plist->NumProps();

	for(int i=0;i<numProps;i++)
	{
		if (plist->WasUpdated(i))
		{
			if (IsInstr(plist->GetName(i),"LOD_Level"))
			{
				// If the value of the node has changed remove any LOD appdata to disconnect it from the
				// LOD system
				CStr value;
				plist->GetValue(i,value);

				ConfigProp cpFind;
				cpFind.name = plist->GetName(i);

				Link<ConfigProp>* cprop = list->Find(&cpFind);

				//if (value != (*list)[i].value)
				if (cprop)
				{
					INode* node = gInterface->GetINodeByName(cprop->data.value);

					if (node)
						node->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);

					ULONG nodeID = nodeApply->GetHandle();
					void* data = malloc(sizeof(ULONG));
					memcpy(data,&nodeID,sizeof(ULONG));

					INode* newnode = gInterface->GetINodeByName(value);

					newnode->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);
					newnode->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID,sizeof(ULONG),data);
					bUpdate = true;
				}
			}
		}
	}

	return bUpdate;
}

int LODBrowser::GetNumLODLevels(INode* node)
{
	CStr propBuffer;
	LinkList<ConfigProp> cprops;
	int  numLevels = 0;

	node->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);
	
	Link<ConfigProp>* curprop = cprops.GetHead();

	while(curprop)
	{
		if (IsInstr(curprop->data.name,"LOD_Level"))
			numLevels++;

		curprop = curprop->next;
	}

	return numLevels;
}

void LODBrowser::GetLODLevels(INode* node, NxLODLevel* levels)
{
	CStr propBuffer;
	LinkList<ConfigProp> cprops;
	int curLevel = 0;

	node->GetUserPropBuffer(propBuffer);
	ParseConfigProps(&cprops,NULL,propBuffer);
	
	Link<ConfigProp>* curprop = cprops.GetHead();

	while(curprop)
	{
		int ID;

		if (IsInstr(curprop->data.name,"LOD_Level"))
		{
			sscanf(curprop->data.name,"LOD_Level%i",&ID);
			
			if (!gInterface->GetINodeByName(curprop->data.value))
				levels[ID-1].m_ObjectCRC = 0;
			else
				levels[ID-1].m_ObjectCRC = GenerateCRC(curprop->data.value);
		}

		if (IsInstr(curprop->data.name,"LOD_Dist"))
		{
			sscanf(curprop->data.name,"LOD_Dist%i",&ID);

			// Convert distance from feet to inches (mult by 12.  12 in. in foot)
			levels[ID-1].m_Distance = atof(curprop->data.value) * 12.0f;
		}

		curprop = curprop->next;
	}
}

void LODBrowser::SelAll()
{
	SelChildren();

	CStr value;
	plist->GetValue(0,value);

	INode* node = gInterface->GetINodeByName(value);

	if (node)
		gInterface->SelectNode(node, FALSE);

	gInterface->ForceCompleteRedraw();
}

void LODBrowser::SelMaster()
{
	CStr value;
	plist->GetValue(0,value);

	INode* node = gInterface->GetINodeByName(value);

	if (node)
		gInterface->SelectNode(node);

	gInterface->ForceCompleteRedraw();
}

void LODBrowser::SelChildren()
{
	if (ip->GetSelNodeCount() == 0)
		return;

	INode* node = ip->GetSelNode(0);

	// We need to select all the nodes that are referenced by "LOD_Level" in the node buffer
	CStr selNodeName;
	char bufKey[256];
	int  val = 1;
	
	sprintf(bufKey,"LOD_Level%i",val);

	ip->ClearNodeSelection();

	while(node->GetUserPropString(bufKey,selNodeName))
	{
		INode* selNode = ip->GetINodeByName(selNodeName);
		
		if (selNode)
			ip->SelectNode(selNode,0);

		sprintf(bufKey,"LOD_Level%i",++val);
	}

	ip->ForceCompleteRedraw();
}

int  LODBrowser::GetLODType(INode* node)
{
	if (GetNumLODLevels(node) > 0)
		return LODTYPE_MASTER;
	
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);

	if (appdata && appdata->data)
	{
		ULONG ID = *((ULONG*)appdata->data);
		
		// This can only be a slave if it points to a valid MASTER node
		if ( gInterface->GetINodeByHandle(ID) )
			return LODTYPE_SLAVE;
	}

	return LODTYPE_NONE;
}

INode* LODBrowser::GetLODMaster(INode* node)
{
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_LOD_MASTER_ID);

	if (appdata && appdata->data)
	{
		ULONG ID = *((ULONG*)appdata->data);
		return gInterface->GetINodeByHandle(ID);
	}

	return NULL;
}

LRESULT LODBrowser::SubPropProc(PropList* plist,HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam,void* pData)
{
	LODBrowser* pthis = (LODBrowser*)pData;

	switch(msg)
	{
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDM_REMOVELOD:
				pthis->RemoveLOD( plist->GetPropFromHWnd(hwnd) );
			break;
		}
		break;

	case WM_RBUTTONDOWN:
		{
			// Don't popup the menu if we've right clicked the master field
			if ( plist->GetPropFromHWnd(hwnd) != 0 )
			{
				// Bring up the popup menu
				POINT pt;
				GetCursorPos(&pt);
				TrackPopupMenu(pthis->hPopupSubMenu,TPM_RIGHTBUTTON,
							   pt.x,pt.y,
							   0,hwnd,NULL);
			}
			return TRUE;
		}
	};

	return FALSE;
}

void LODBrowser::RemoveLOD(int id)
{
	if (id == -1)
		return;

	CStr name = plist->GetName(id);

	if (IsInstr(name,"LOD_Level"))
	{
		plist->RemoveProp(id);
		plist->RemoveProp(id);
	}
	else
	{
		plist->RemoveProp(id-1);
		plist->RemoveProp(id-1);
	}

	numLODs--;

	// Renumber the LOD ranges
	int num = 1;

	for(int i=1;i<plist->NumProps();i++)
	{
		char name[256];

		if (i % 2 == 1)
		{
			sprintf(name,"LOD_Level%i",num);
			plist->SetName(i,name);
		}
		else
		{
			sprintf(name,"LOD_Dist%i",num);
			plist->SetName(i,name);
			num++;
		}
	}

	plist->BuildUI();
	plist->ResetIndexToEnd();

	// Get the property buffer of the master node and remove any LOD references first
	CStr strMaster;
	plist->GetValue(0,strMaster);
	INode* nodeMaster = ip->GetINodeByName(strMaster);

	if (nodeMaster)
	{
		CStr buf;
		nodeMaster->GetUserPropBuffer(buf);
		RemoveLevels(buf);
		nodeMaster->SetUserPropBuffer(buf);
	}

	ListUpdate(plist,this);
}

void LODBrowser::RemoveLevels(CStr& propBuffer)
{
	// Scan through the given property buffer and remove any lines
	// containing LOD_Level or LOD_Dist
	int  pos = 0;
	CStr line;
	CStr buf;

	while(pos <= propBuffer.Length() )
	{
		line = GetRemainLineExact(propBuffer,&pos);
		if (IsInstr(line,"LOD_Level") ||
			IsInstr(line,"LOD_Dist"))
		{
			propBuffer = ReplaceStr(propBuffer,line,"");
			pos = 0;
		}
	}
}

bool LODBrowser::IsActive()
{
	return IsWindowVisible(hwnd);
}
