/*
	D3DRenderer.cpp
	Direct3D Renderer
	aml - 11-21-02
*/

#include "D3DRenderer.h"

//#define DEBUGLINE(line) { FILE* fpDbg = fopen("c:\\D3Ddebug.txt","a"); fprintf(fpDbg, "%s\n", line); fclose(fpDbg); }
#define DEBUGLINE(line)

D3DRenderer::D3DRenderer()
{
	displays   = NULL;
	clearColor = D3DCOLOR_XRGB(0,0,0);
}

D3DRenderer::~D3DRenderer()
{
	if (displays)
		delete [] displays;
}

bool D3DRenderer::InitD3D(HWND hwnd)
{
	this->hwnd = hwnd;

	pD3D = Direct3DCreate8(D3D_SDK_VERSION);
	if (FAILED(pD3D))
		return false;

	// Acquire each display adapter attached to the system
	nDisplays = pD3D->GetAdapterCount();	
	displays = new DisplayDevice[nDisplays];

	for(int i = 0; i < nDisplays; i++)
	{
		displays[i].id = i;

		if (FAILED(pD3D->GetAdapterIdentifier(i, 0, &displays[i].ident)))
			return false;

		if (FAILED(pD3D->GetAdapterDisplayMode(i, &displays[i].mode)))
			return false;

		// Setup D3D presentation parameters
		ZeroMemory(&displays[i].d3dpp, sizeof(D3DPRESENT_PARAMETERS));
		displays[i].d3dpp.Windowed               = TRUE;
		displays[i].d3dpp.SwapEffect             = D3DSWAPEFFECT_DISCARD;
		displays[i].d3dpp.BackBufferFormat       = displays[i].mode.Format;
		displays[i].d3dpp.EnableAutoDepthStencil = TRUE;
		displays[i].d3dpp.AutoDepthStencilFormat = D3DFMT_D16;

		// Create the D3D Device interface
		/*
		if (SUCCEEDED(gpD3D->CreateDevice(i, 
			                D3DDEVTYPE_HAL,
							hwnd,
							D3DCREATE_HARDWARE_VERTEXPROCESSING,
							&displays[i].d3dpp,
							&displays[i].device)))
		{
			displays[i].bHardwareVBuffers = true;
		}
		else
		*/
		{
			displays[i].bHardwareVBuffers = false;

			// Attempt using software vertex processing instead
			if (FAILED(pD3D->CreateDevice(i,
								D3DDEVTYPE_HAL,
								hwnd,
								D3DCREATE_SOFTWARE_VERTEXPROCESSING,
								&displays[i].d3dpp,
								&displays[i].device)))
			{
				return false;
			}
		}

		if (FAILED(displays[i].device->GetDeviceCaps(&displays[i].caps)))
			return false;

		displays[i].hMonitor = pD3D->GetAdapterMonitor(i);

		displays[i].device->SetRenderState( D3DRS_CULLMODE, D3DCULL_NONE );
		displays[i].device->SetRenderState( D3DRS_LIGHTING, FALSE );
		displays[i].device->SetRenderState( D3DRS_ZENABLE, TRUE );

		displays[i].device->SetRenderState( D3DRS_ZFUNC, D3DCMP_LESSEQUAL );

		// Create default texture
		if (FAILED(D3DXCreateTexture(displays[i].device,
			              1,1,0,0,D3DFMT_X8R8G8B8, D3DPOOL_MANAGED, &displays[i].texDefault)))
		{
			return false;
		}

		// Write solid white into the default texture
		D3DLOCKED_RECT lockrect;
		if (FAILED(displays[i].texDefault->LockRect(0,
			                                        &lockrect,
										            NULL,		// Lock entire texture
                                                    0)))
		{
			return false;
		}

		DWORD* color = (DWORD*)lockrect.pBits;
		*color = 0xFFFFFFFF;

		if (FAILED(displays[i].texDefault->UnlockRect(0)))
			return false;
	}

	return true;
}	

void D3DRenderer::ShutdownD3D()
{
	FreeAll();

	for(int i = 0; i < nDisplays; i++)
	{
		if (displays[i].device)
		{
			displays[i].device->Release();
			displays[i].device = NULL;
		}
	}

	if (pD3D)
		pD3D->Release();

	pD3D = NULL;
}

void D3DRenderer::SetTestMatrices()
{
	D3DXMATRIX matX, matY;
	D3DXMatrixRotationY( &matY, timeGetTime()/600.0f );
	//D3DXMatrixRotationX( &matX, timeGetTime()/600.0f );

	//matWorld = matX * matY;
	matWorld = matY;

	D3DXMatrixLookAtLH( &matView, &D3DXVECTOR3( 0.0f, 2.0f, -2.0f ),
								  &D3DXVECTOR3( 0.0f, -0.3f,  1.0f ),
								  &D3DXVECTOR3( 0.0f, 1.0f,  0.0f ) );

	D3DXMatrixPerspectiveFovLH( &matProj, D3DX_PI/4, 1.0f, 1.0f, 1000.0f );
}

void D3DRenderer::CreateTestMesh2(D3DMesh* mesh)
{
	mesh->AllocVerts(100);

	DWORD i;

    for(i=0; i<50; i++ )
    {
        FLOAT theta = (2*D3DX_PI*i)/(50-1);
		D3DXVECTOR3 pos = D3DXVECTOR3( sinf(theta),-1.0f, cosf(theta) );

        mesh->verts[2*i+0].x     = pos.x;
		mesh->verts[2*i+0].y     = pos.y;
		mesh->verts[2*i+0].z     = pos.z;
		mesh->verts[2*i+0].color = 0xffffffff;
        mesh->verts[2*i+0].u     = ((FLOAT)i)/(50-1);
        mesh->verts[2*i+0].v     = 1.0f;

		pos = D3DXVECTOR3( sinf(theta), 1.0f, cosf(theta) );

        mesh->verts[2*i+1].x     = pos.x;
		mesh->verts[2*i+1].y     = pos.y;
		mesh->verts[2*i+1].z     = pos.z;
		mesh->verts[2*i+1].color = 0xff808080;
        mesh->verts[2*i+1].u     = ((FLOAT)i)/(50-1);
        mesh->verts[2*i+1].v     = 0.0f;
    }

	mesh->AllocFaces(33);
	
	for(i = 0; i < 33; i++)
	{
		mesh->faces[i].f[0] = (unsigned short)(i * 3);
		mesh->faces[i].f[1] = (unsigned short)(i * 3) + 1;
		mesh->faces[i].f[2] = (unsigned short)(i * 3) + 2;
	}

	mesh->AllocTextures(1);
	mesh->textures[0].texname = "c:\\banana.png";
	mesh->AllocRefs(nDisplays);
}

void D3DRenderer::CreateTestMesh(D3DMesh* mesh)
{
	D3DVert v[]=
	{
        { -1.0f,-1.0f, 0.0f, 0xffff0000, 0.0f, 0.0f},
        {  1.0f,-1.0f, 0.0f, 0xff0000ff, 1.0f, 0.0f},
        {  0.0f, 1.0f, 0.0f, 0xffffffff, 1.0f, 1.0f},
        { -2.0f,-2.0f, 2.0f, 0xffff0000, 0.0f, 0.0f},
        {  2.0f,-2.0f, 2.0f, 0xff0000ff, 1.0f, 0.0f},
        {  0.0f, 2.0f, 2.0f, 0xffffffff, 1.0f, 1.0f},		
	};

	/*
	Vert v[]=
	{
        { -1.0f,-1.0f, 0.0f, 0xffffffff, 0.0f, 0.0f},
        {  1.0f,-1.0f, 0.0f, 0xffffffff, 1.0f, 0.0f},
        {  0.0f, 1.0f, 0.0f, 0xffffffff, 1.0f, 1.0f},
	};
	*/

	/*
	mesh->AllocVerts(6);
	memcpy(mesh->verts, v, sizeof(D3DVert)*6);
	mesh->AllocFaces(2);
	mesh->faces[0].f[0] = 0;
	mesh->faces[0].f[1] = 1;
	mesh->faces[0].f[2] = 2;

	mesh->faces[1].f[0] = 3;
	mesh->faces[1].f[1] = 4;
	mesh->faces[1].f[2] = 5;

	mesh->AllocTextures(2);
	mesh->textures[0].texname = "c:\\sbanana.png";
	mesh->textures[1].texname = "c:\\banana.png";
	mesh->AllocRefs(nDisplays);
	*/

	mesh->AllocVerts(6);
	memcpy(mesh->verts, v, sizeof(D3DVert)*6);
	mesh->AllocFaces(2);
	mesh->texFaces[0].f[0] = 0;
	mesh->texFaces[0].f[1] = 1;
	mesh->texFaces[0].f[2] = 2;
	mesh->texFaces[0].tex  = 0;

	mesh->texFaces[1].f[0] = 3;
	mesh->texFaces[1].f[1] = 4;
	mesh->texFaces[1].f[2] = 5;
	mesh->texFaces[1].tex  = 1;

	mesh->AllocTextures(2);
	mesh->textures[0].texname = "c:\\sbanana.png";
	mesh->textures[1].texname = "c:\\banana.png";
	mesh->AllocRefs(nDisplays);

	mesh->BuildBuffers();
}

void D3DRenderer::UploadMesh(D3DMesh* mesh)
{
	char str[256];
	sprintf(str, "UploadMesh: numTextures: %i", mesh->nTextures);
	DEBUGLINE(str);

	for(int tex = 0; tex < mesh->nTextures; tex++)
	{
		// Allocate vert and index buffers
		if (mesh->texgroup[tex].vbuf)
			delete [] mesh->texgroup[tex].vbuf;

		if (mesh->texgroup[tex].ibuf)
			delete [] mesh->texgroup[tex].ibuf;

		mesh->texgroup[tex].vbuf = new VertBufRef[nDisplays];
		mesh->texgroup[tex].ibuf = new IndexBufRef[nDisplays];
	}

	for(int disp = 0; disp < nDisplays; disp++)
		UploadMesh(&displays[disp], mesh);
}

/*
void D3DRenderer::UploadMesh(DisplayDevice* disp, D3DMesh* mesh)
{
	// Upload verticies
	Link<VertexBuffer>* vBuffer = disp->AddVertBuffer(mesh->nVerts);
	disp->AssignVertData(vBuffer, mesh->verts, mesh->nVerts);
	mesh->vBuffer[disp->id] = vBuffer;

	// Upload indicies
	Link<IndexBuffer>* iBuffer = disp->AddIndexBuffer(mesh->nFaces);
	disp->AssignIndexData(iBuffer, mesh->faces, mesh->nFaces);
	mesh->iBuffer[disp->id] = iBuffer;

	// Upload textures
	for(int i = 0; i < mesh->nTextures; i++)
	{
		Link<Texture>* texture = disp->AddTexture(mesh->textures[i].texname.c_str());
		mesh->textures[i].textures[disp->id] = texture;
	}
}
*/

void D3DRenderer::UploadMesh(DisplayDevice* disp, D3DMesh* mesh)
{
	for(int tex = 0; tex < mesh->nTextures; tex++)
	{
		// Upload verticies
		Link<VertexBuffer>* vBuffer = disp->AddVertBuffer(mesh->texgroup[tex].nVerts);
		disp->AssignVertData(vBuffer, mesh->texgroup[tex].verts, mesh->texgroup[tex].nVerts);
		mesh->texgroup[tex].vbuf[disp->id] = vBuffer;

		// Upload indicies
		Link<IndexBuffer>* iBuffer = disp->AddIndexBuffer(mesh->texgroup[tex].nFaces);
		disp->AssignIndexData(iBuffer, mesh->texgroup[tex].faces, mesh->texgroup[tex].nFaces);
		mesh->texgroup[tex].ibuf[disp->id] = iBuffer;

		// Upload texture
		Link<Texture>* texture = disp->AddTexture((char*)mesh->textures[tex].texname);
		mesh->textures[tex].textures[disp->id] = texture;
	}

	// Upload verticies
	Link<VertexBuffer>* vBuffer = disp->AddVertBuffer(mesh->nVerts);
	disp->AssignVertData(vBuffer, mesh->verts, mesh->nVerts);
	mesh->vBuffer[disp->id] = vBuffer;

	// Upload indicies
	Link<IndexBuffer>* iBuffer = disp->AddIndexBuffer(mesh->nFaces);
	disp->AssignIndexData(iBuffer, mesh->faces, mesh->nFaces);
	mesh->iBuffer[disp->id] = iBuffer;
}

void D3DRenderer::Render(D3DMesh* mesh)
{
	HMONITOR curMonitor;
	curMonitor = MonitorFromWindow(hwnd, MONITOR_DEFAULTTONULL);

	//int disp = USEDISPLAY;
	for(int disp = 0; disp < nDisplays; disp++)
	{
		// Determine which rendering device to use
		if (displays[disp].hMonitor == curMonitor)
		{
			if (!displays[disp].device)
				continue;

			displays[disp].device->Clear(0,								// No Dirty Rects (Full window refresh)
									NULL,								// No Dirty Rects
									D3DCLEAR_TARGET|D3DCLEAR_ZBUFFER,	// Clear target as opposed to ZBuffer/Stencil
									clearColor,							// Clear Color
									1.0f,								// New Z Value
									0);									// New stencil value

			//displays[disp].device->Reset(&(displays[disp].d3dpp));

			displays[disp].device->BeginScene();

			SetTestMatrices();
			displays[disp].matWorld = matWorld;
			displays[disp].matView  = matView;
			displays[disp].matProj  = matProj;

			displays[disp].SetMatrices();
			DrawMesh(&displays[disp], mesh);

			displays[disp].device->EndScene();
			displays[disp].device->Present(NULL, NULL, NULL, NULL);
		}
	}
}

/*
bool D3DRenderer::DrawMesh(DisplayDevice* disp, D3DMesh* mesh)
{
	disp->device->SetTexture(0, mesh->textures[0].textures[disp->id]->data.texture);
    disp->device->SetTextureStageState( 0, D3DTSS_COLOROP,   D3DTOP_MODULATE );
    disp->device->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
    disp->device->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
    disp->device->SetTextureStageState( 0, D3DTSS_ALPHAOP,   D3DTOP_DISABLE );

	if (FAILED(disp->device->SetStreamSource(0, mesh->vBuffer[disp->id]->data.vertBuffer, sizeof(D3DVert))))
		return false;

	if (FAILED(disp->device->SetVertexShader(Vert::Type)))
		return false;

	if (FAILED(disp->device->SetIndices(mesh->iBuffer[disp->id]->data.indexBuffer, 0)))
		return false;

	//disp->device->DrawPrimitive(D3DPT_TRIANGLESTRIP, 0, 1);
	//disp->device->DrawPrimitive(D3DPT_TRIANGLELIST, 0, mesh->nFaces);
	//disp->device->DrawPrimitive(D3DPT_TRIANGLELIST, 0, 30);

	if (FAILED(disp->device->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, mesh->nVerts, 0, mesh->nFaces)))
		return false;

	return true;
}
*/

bool D3DRenderer::DrawMesh(DisplayDevice* disp, D3DMesh* mesh)
{
	disp->device->SetTextureStageState( 0, D3DTSS_COLOROP,   D3DTOP_MODULATE );
	disp->device->SetTextureStageState( 0, D3DTSS_COLORARG1, D3DTA_TEXTURE );
	disp->device->SetTextureStageState( 0, D3DTSS_COLORARG2, D3DTA_DIFFUSE );
	disp->device->SetTextureStageState( 0, D3DTSS_ALPHAOP,   D3DTOP_DISABLE );

	char str[256];
	sprintf(str, "numTextures: %i", mesh->nTextures);
	DEBUGLINE(str);

	if (mesh->nTextures == 0)
	{
		disp->device->SetTexture(0, disp->texDefault);

		if (FAILED(disp->device->SetStreamSource(0, mesh->vBuffer[disp->id]->data.vertBuffer, sizeof(D3DVert))))
			return false;

		if (FAILED(disp->device->SetVertexShader(D3DVert::Type)))
			return false;

		if (FAILED(disp->device->SetIndices(mesh->iBuffer[disp->id]->data.indexBuffer, 0)))
			return false;

		if (FAILED(disp->device->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, mesh->nVerts, 0, mesh->nFaces)))
			return false;
	}

	for(int tex = 0; tex < mesh->nTextures; tex++)
	{
		if (mesh->textures[tex].texname.Length() > 0)
		{
			if (mesh->textures[tex].textures[disp->id] &&
				mesh->textures[tex].textures[disp->id]->data.texture)
			{
				disp->device->SetTexture(0, mesh->textures[tex].textures[disp->id]->data.texture);
			}
			else
			{
				disp->device->SetTexture(0, disp->texDefault);
			}
		}
		else
		{
			disp->device->SetTexture(0, disp->texDefault);
		}

		if (FAILED(disp->device->SetStreamSource(0, mesh->texgroup[tex].vbuf[disp->id]->data.vertBuffer, sizeof(D3DVert))))
			return false;

		if (FAILED(disp->device->SetVertexShader(D3DVert::Type)))
			return false;

		if (FAILED(disp->device->SetIndices(mesh->texgroup[tex].ibuf[disp->id]->data.indexBuffer, 0)))
			return false;

		if (FAILED(disp->device->DrawIndexedPrimitive(D3DPT_TRIANGLELIST, 0, mesh->texgroup[tex].nVerts, 0, mesh->texgroup[tex].nFaces)))
			return false;
	}

	return true;
}

void D3DRenderer::FreeTextures()
{
	for(int disp = 0; disp < nDisplays; disp++)
		displays[disp].FreeTextures();
}

void D3DRenderer::FreeVertBuffers()
{
	for(int disp = 0; disp < nDisplays; disp++)
		displays[disp].FreeVertBuffers();
}

void D3DRenderer::FreeIndexBuffers()
{
	for(int disp = 0; disp < nDisplays; disp++)
		displays[disp].FreeIndexBuffers();
}

void D3DRenderer::FreeAll()
{
	for(int disp = 0; disp < nDisplays; disp++)
		displays[disp].FreeAll();
}
