/*
	MergeList.cpp
*/

#include "../Trigger/Trigger.h"
#include "../Link/LinkUI.h"
#include "MergeList.h"
#include "Resource.h"
#include "ConflictDlg.h"
#include "MergeRenameDlg.h"
#include "../UI/PropList.h"
#include "../UI/ColorListBox.h"
#include "../PropEdit/ParseFuncs.h"
#include <winbase.h>

extern GUP* gpGUP;

//////////////////// Node Merge List

NodeMergeList::NodeMergeList(HINSTANCE hInstance,HWND hwndParent,
							 LinkList<NodeEntry>* srcList, LinkList<NodeEntry>* mergeList,
							 Interface* ip) :
	ModalDlgWindow(hInstance,MAKEINTRESOURCE(IDD_MERGELIST),hwndParent)
	//MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_MERGELIST),hwndParent)
{
	this->srcList = srcList;
	this->mergeList = mergeList;
	this->ip=ip;

	bRenameTriggers      = false;
	bIncludeTriggerLinks = false;

	// Attach property list
	plist = new PropList(hInstance,1);
	listbox = new ColorListBox(hInstance);
}

NodeMergeList::~NodeMergeList()
{
	delete listbox;
	delete plist;
}

void NodeMergeList::AddItems(LinkList<NodeEntry>* list)
{
	Link<NodeEntry>* link = list->GetHead();

	while(link)
	{
		int index = SendDlgItemMessage(hwnd,IDC_LIST,LB_ADDSTRING,0,(LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)link);

		// If the item is also in the src list color it red (otherwise green)
		if (srcList->Find(&link->data))
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(255,0,0));
		else
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(0,128,0));

		link=link->next;
	}
}

void NodeMergeList::SelAll()
{
	SendDlgItemMessage(hwnd,IDC_LIST,LB_SELITEMRANGE,(WPARAM)TRUE,MAKELPARAM(0,mergeList->GetSize()-1));
	SelChange();
}

void NodeMergeList::SelNone()
{
	SendDlgItemMessage(hwnd,IDC_LIST,LB_SELITEMRANGE,(WPARAM)FALSE,MAKELPARAM(0,mergeList->GetSize()-1));
	SelChange();
}

void NodeMergeList::SelInvert()
{
	int size = mergeList->GetSize();

	for(int i=0;i<size;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);
		else
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)i);
	}

	SelChange();
}

BOOL NodeMergeList::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		{
			// Set stats
			char str[256];
			_itoa(srcList->GetSize(),str,10);
			SetDlgItemText(hwnd,IDC_SRCNUM,str);
			_itoa(mergeList->GetSize(),str,10);
			SetDlgItemText(hwnd,IDC_MERGENUM,str);

			_itoa(0,str,10);
			SetDlgItemText(hwnd,IDC_MERGEDNUM,str);

			// Set total number of objects if all merged
			int iSame,iNew;
			CountNodes(mergeList,iSame,iNew);

			_itoa(iSame+iNew,str,10);
			SetDlgItemText(hwnd,IDC_FLDTOTALOBJS,str);

			// Set total polys in scene
			nSrcFaces  = CountFaces(srcList);

			_itoa(nSrcFaces,str,10);
			SetDlgItemText(hwnd,IDC_FLDTOTALPOLYSCENE,str);

			// Set total polys in merge file
			int nFaces = CountFaces(mergeList);

			_itoa(nFaces,str,10);
			SetDlgItemText(hwnd,IDC_FLDTOTALPOLYMERGE,str);

			// Set total same poly diff
			char sSign[2] = "+";
			int nOrigSrc  = CountUniqueFaces(mergeList,srcList);
			int nOrigMrg  = CountUniqueFaces(srcList,mergeList);
			int nOrigDiff = nOrigMrg - nOrigSrc;

			if (nOrigDiff < 0)
				sSign[0] = '-';

			sprintf(str,"%s%i",sSign,nOrigDiff);
			SetDlgItemText(hwnd,IDC_FLDSAMEPOLYDIFF,str);

			this->hwnd = hwnd;
			plist->Attach(GetDlgItem(hwnd,IDC_STATLIST));
			plist->HasApply(false);

			listbox->Attach(GetDlgItem(hwnd,IDC_LIST),LBS_SORT|LBS_EXTENDEDSEL);

			SetVisible();
			AddItems(mergeList);

			IEdit = GetICustEdit(GetDlgItem(hwnd,IDC_EDIT));
		}
		return TRUE;

	case WM_CLOSE:
		ReleaseICustEdit(IEdit);
		EndDialog(hwnd,-1);
		return TRUE;
	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_MERGE:
			ProcMerge();
			ReleaseICustEdit(IEdit);
			EndDialog(hwnd,0);
			return TRUE;

		case IDC_CANCEL:
			ReleaseICustEdit(IEdit);
			EndDialog(hwnd,-1);
			return TRUE;

		case IDC_SELALL:
			SelAll();
			return TRUE;

		case IDC_SELNONE:
			SelNone();
			return TRUE;

		case IDC_SELINVERT:
			SelInvert();
			return TRUE;

		case IDC_EDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateMatching();
				break;
			}
			break;

		case IDC_LIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				SelChange();
				return TRUE;
			}
		}
	}

	return FALSE;
}

void NodeMergeList::CreateNonMergedDelList()
{
	nonMergeDelList.ZeroCount();

	// Go through nodes deleting any merge nodes not selected
	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	// Pass 1 delete non-triggers
	for(int i=0;i<count;i++)
	{
		// Delete non-triggers first
		if (!SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<NodeEntry>* link = (Link<NodeEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);

			// Only delete the node if it exists, it's possible that a node might recieve a reference message
			// on destruction which could cause one of our nodes to delete itself later in the list causing
			// it to try to delete twice.   (As is the case with TriggerLinks)

			char strErr[256];
			sprintf(strErr,"DelNonMergedNodes: %s (%i)\n",link->data.name,link->data.id);
			OutputDebugString(strErr);

			if (NodeExists(link->data.node) &&
				link->data.node->EvalWorldState(0).obj->ClassID() != vTRIGGER_CLASS_ID)
			{
				// The node should not be added to the deletion list in the event that
				// it exists within the final list
				if (!FinalNodeExists(link->data.node))
					nonMergeDelList.Append(1, &link->data.node);
			}
		}
	}

	// Now go back through and delete the triggers
	OutputDebugString("----------------- PASS 2 -------------------\n");

	for(i=0;i<count;i++)
	{
		if (!SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<NodeEntry>* link = (Link<NodeEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);

			// Only delete the node if it exists, it's possible that a node might recieve a reference message
			// on destruction which could cause one of our nodes to delete itself later in the list causing
			// it to try to delete twice.   (As is the case with TriggerLinks)

			char strErr[256];
			sprintf(strErr,"DelNonMergedNodes: %s (%i)\n",link->data.name,link->data.id);
			OutputDebugString(strErr);

			if (NodeExists(link->data.node) &&
				link->data.node->EvalWorldState(0).obj->ClassID() ==  vTRIGGER_CLASS_ID)
			{
				// The node should not be added to the deletion list in the event that
				// it exists within the final list
				if (!FinalNodeExists(link->data.node))
					nonMergeDelList.Append(1, &link->data.node);
			}

		}
	}
}

void NodeMergeList::DelNonMergedNodes()
{
	int count = nonMergeDelList.Count();

	for(int i=0;i<count;i++)
	{
		char debug[256];
		sprintf(debug,"Delete: %s\n",nonMergeDelList[i]->GetName());
		OutputDebugString(debug);
	}

	ip->ClearNodeSelection();

	ip->SelectNodeTab(nonMergeDelList,TRUE,FALSE);
	gpGUP->ExecuteStringScript("max delete\n");

	//int oldSetCount = ip->GetNumNamedSelSets();

	//ip->AddNewNamedSelSet(nonMergeDelList,CStr("!!@!!DelList"));

	/*
	int newSetCount = ip->GetNumNamedSelSets();

	int numSets = ip->GetNumNamedSelSets();

	for(int i=0;i<numSets;i++)
	{
		CStr name = ip->GetNamedSelSetName(i);

		if (CStr(ip->GetNamedSelSetName(i))==CStr("!!@!!DelList"))
		{
			while(ip->GetNamedSelSetItemCount(i))
			{
				INode* node = ip->GetNamedSelSetItem(i,0);

				if (node->ClassID()!=vLINK_OBJ_CLASS_ID &&
					NodeExists(node) )
					ip->DeleteNode(node);
			}	
		}
	}

	ip->RemoveNamedSelSet(CStr("!!@!!DelList"));
	*/

	/*
	for(int i=0;i<count;i++)
	{
		//nonMergeDelList[i]->NotifyDependents(FOREVER,PART_ALL,REFMSG_TARGET_DELETED);
		ip->DeleteNode(nonMergeDelList[i]);
	}
	*/

	nonMergeDelList.ZeroCount();
}

bool NodeMergeList::NodeExists(ULONG id,INode* root)
{
	if (!root)
		root = ip->GetRootNode();

	int kids = root->NumberOfChildren();

	for(int i=0; i<kids; i++)
	{
		INode* child = root->GetChildNode(i);
		
		if (NodeExists(id,child))
			return true;
	}

	if (root->GetHandle() == id)
		return true;

	return false;
}

bool NodeMergeList::NodeExists(INode* node,INode* root)
{
	if (!root)
		root = ip->GetRootNode();

	int kids = root->NumberOfChildren();

	for(int i=0; i<kids; i++)
	{
		INode* child = root->GetChildNode(i);
		
		if (NodeExists(node,child))
			return true;
	}

	if (node == root)
		return true;

	return false;
}


void NodeMergeList::ProcMerge()
{
	ClearDelLists();
	BuildFinalList();
	//DelNonMergedNodes();
	CreateNonMergedDelList();

	ConflictDlg* conflictDlg = new ConflictDlg(hInstance,hwnd);
	conflictDlg->SetGeomMode();

	// Go through the final list determining if we encounter any duplicates in the source list
	// if so, we'll need to prompt the user to see which one they want to keep
	
	Link<NodeEntry>* curNode = finalList.GetHead();

	while(curNode)
	{
		Link<NodeEntry>* srcLink = srcList->Find(&curNode->data);

		// Debug dump current node being processed
		char strErr[256];
		sprintf(strErr,"Node: %s\n",(char*)curNode->data.name);
		OutputDebugString(strErr);

		Class_ID cid = curNode->data.node->ClassID();

		if (srcLink)
		{
			if (bRenameTriggers &&
				IsInstr(curNode->data.name,"TriggerLink"))
			{
				CStr mergeName = curNode->data.node->GetName();				
				ip->MakeNameUnique(mergeName);
				curNode->data.node->SetName(mergeName);
			}
			else
			{
				//conflictDlg->DisableMerge();
				conflictDlg->EnableMerge();

				conflictDlg->SetItemName(curNode->data.name);

				if (!conflictDlg->ApplyAll())
					conflictDlg->Show();							// This blocks

				switch(conflictDlg->GetResolveMode())
				{
				case RESOLVE_DELSRC:
					if (ip->GetINodeByHandle(srcLink->data.id))
					{
						if (srcLink->data.node->EvalWorldState(0).obj->ClassID() == vTRIGGER_CLASS_ID)
							postDelList.Append(1, &srcLink->data.node);
						else
							preDelList.Append(1, &srcLink->data.node);
					}
					break;

				case RESOLVE_DELMERGE:				
					if (ip->GetINodeByHandle(curNode->data.id))
					{
						if (srcLink->data.node->EvalWorldState(0).obj->ClassID() == vTRIGGER_CLASS_ID)
							postDelList.Append(1, &curNode->data.node);
						else
							preDelList.Append(1, &curNode->data.node);
					}
					break;

				case RESOLVE_MERGE:
					{
						char* renameName = conflictDlg->GetRenameName();
						CStr mergeName;

						if (!renameName)
						{
							MergeRenameDlg* mergeDlg = new MergeRenameDlg(hInstance,hwnd,curNode->data.name);
							mergeDlg->Show();

							mergeName = mergeDlg->GetName();
							conflictDlg->SetRenameName(mergeName);
							ip->MakeNameUnique(mergeName);
							curNode->data.node->SetName(mergeName);

							delete mergeDlg;
						}
						else
						{
							mergeName = renameName;
							ip->MakeNameUnique(mergeName);
							curNode->data.node->SetName(mergeName);
						}
					}
					break;

				case RESOLVE_AUTORENAME:
					{
						CStr mergeName = curNode->data.node->GetName();
						ip->MakeNameUnique(mergeName);
						curNode->data.node->SetName(mergeName);

						if (conflictDlg->GetAutoRename())
							bRenameTriggers = true;
					}
					break;
				}
			}
		}

		curNode = curNode->next;
	}

	delete conflictDlg;
	bRenameTriggers = false;

	//ProcDelLists();
}

Link<NodeEntry>* NodeMergeList::FinalNodeExists(INode* node)
{
	// Scan through the list to see if this link node already exists
	// in the objects to be merged
	Link<NodeEntry>* linkScan = finalList.GetHead();

	while(linkScan)
	{
		if (linkScan->data.node == node)
			return linkScan;

		linkScan = linkScan->next;
	}

	return NULL;
}

void NodeMergeList::BuildFinalList()
{
	finalList.Clear();

	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<NodeEntry>* link = (Link<NodeEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);
			finalList.AddToTailUnique(&link->data);
		}
	}

	// If trigger links are to be included scan through the final list making sure all referenced
	// trigger links are included
	if (bIncludeTriggerLinks)
	{
		Link<NodeEntry>* link = finalList.GetHead();

		while(link)
		{
			// Scan the reference list for this node and add all the appropriate trigger links
			RefList& refList = link->data.node->GetRefList();
			RefListItem* refItem = refList.FirstItem();
			while(refItem)
			{
				if (refItem->maker->ClassID() == vLINK_OBJ_CLASS_ID)
				{
					LinkObject* linkObj = (LinkObject*)refItem->maker;
					if (linkObj->link_node)
					{
						if (!FinalNodeExists(linkObj->link_node))
						{
							NodeEntry ne;
							ne.name  = linkObj->link_node->GetName();
							ne.node  = linkObj->link_node;
							ne.id    = linkObj->link_node->GetHandle();
							ne.flags = 0;

							finalList.AddToTail(&ne);
						}
					}
				}

				refItem = refItem->next;
			}

			link = link->next;
		}
	}
}

void NodeMergeList::CountNodes(LinkList<NodeEntry>* list,int& iSame,int& iNew)
{
	Link<NodeEntry>* link = list->GetHead();

	iSame = 0;
	iNew  = 0;

	while(link)
	{
		if (srcList->Find(&link->data))
			iSame++;
		else
			iNew++;

		link = link->next;
	}
}

int NodeMergeList::CountFaces(LinkList<NodeEntry>* nlist)
{
	Link<NodeEntry>* curNode = nlist->GetHead();
	int count = 0;

	while(curNode)
	{
		count += CountFaces(curNode->data.node);
		curNode = curNode->next;
	}

	return count;
}

int NodeMergeList::CountFaces(INode* node)
{
	int count = 0;

	Object* obj = node->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		Mesh& mesh = triObj->GetMesh();

		count += mesh.getNumFaces();

		if (triObj != obj)
			triObj->DeleteThis();
	}
	
	return count;
}

int NodeMergeList::CountUniqueFaces(LinkList<NodeEntry>* flist,LinkList<NodeEntry>* slist)
{
	Link<NodeEntry>* curNode = flist->GetHead();
	int count = 0;

	while(curNode)
	{
		Link<NodeEntry>* srcLink = slist->Find(&curNode->data);
		
		if (srcLink)
		{
			Object* obj = srcLink->data.node->EvalWorldState(0).obj;

			if (obj && obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
				Mesh& mesh = triObj->GetMesh();

				count += mesh.getNumFaces();

				if (triObj != obj)
					triObj->DeleteThis();
			}
		}

		curNode = curNode->next;
	}

	return count;
}

void NodeMergeList::UpdateNodeProps()
{
	plist->Clear();

	int count    = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);
	int selCount = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSELCOUNT,0,0);
	int pos   = 0;

	if (selCount == 1)
	{
		char str[256];

		// Determine the selected item
		for(int sel=0;sel<count;sel++)
			if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)sel,0))
				break;

		Link<NodeEntry>* link = (Link<NodeEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)sel,0);
		NodeEntry* nentry = &link->data;

		// Same Object?
		plist->AddStatic("Same Object?","True if there is an object of the same name in the original scene");
		
		Link<NodeEntry>* srcLink = srcList->Find(nentry);

		if (srcLink)
			plist->SetValue(pos++,"Yes");
		else
			plist->SetValue(pos++,"No");

		// Number of polys
		plist->AddStatic("Number of Polys","The number of polys in the object");
		int nFaces = CountFaces(nentry->node);

		_itoa(nFaces,str,10);
		plist->SetValue(pos++,str);

		// Number of polys in object that will be replaced
		if (srcLink)
		{
			plist->AddStatic("Num Polys in orig","Number of polys in the original object that would be replaced by this one");
			
			int nOrigFaces = CountFaces(srcLink->data.node);
			_itoa(nOrigFaces,str,10);
			plist->SetValue(pos++,str);

			plist->AddStatic("Poly Difference","Difference between the number of polys of the merge material and the number of polys of the original material that would typically be replaced");
			_itoa(nFaces - nOrigFaces,str,10);
			plist->SetValue(pos++,str);
		}

		plist->BuildUI();
	}
}

void NodeMergeList::SetVisible()
{
	SetWindowText(hwnd, "Merge Geometry");
	SetDlgItemText(hwnd,IDC_TXTSRC,"Number of orig Nodes");
	SetDlgItemText(hwnd,IDC_TXTMERGE,"Number of Merge Nodes");
	SetDlgItemText(hwnd,IDC_TXTMERGED,"Number of Merged Nodes");
	SetDlgItemText(hwnd,IDC_TXTSAME,"Number of Same Nodes");
	SetDlgItemText(hwnd,IDC_TXTNEW,"Number of new Nodes");


	ShowWindow(GetDlgItem(hwnd,IDC_FRMSCNDETAILS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTTOTALOBJS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDTOTALOBJS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTTOTALPOLYSCENE),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDTOTALPOLYSCENE),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTTOTALPOLYMERGE),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDTOTALPOLYMERGE),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTPOLYDIFF),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDPOLYDIFF),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTMERGEDPOLYS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDMERGEDPOLYS),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_TXTSAMEPOLYDIFF),SW_SHOW);
	ShowWindow(GetDlgItem(hwnd,IDC_FLDSAMEPOLYDIFF),SW_SHOW);

	ShowWindow(GetDlgItem(hwnd,IDC_DISPLAYSUBS),SW_HIDE);
}

void NodeMergeList::SelChange()
{
	char str[256];

	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);
	int nSelItems = 0;
	for(int i=0;i<count;i++)
		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
			nSelItems++;

	_itoa(nSelItems,str,10);
	SetDlgItemText(hwnd,IDC_MERGEDNUM,str);

	BuildFinalList();

	int iSame,iNew;
	CountNodes(&finalList,iSame,iNew);

	_itoa(iSame,str,10);
	SetDlgItemText(hwnd,IDC_SAMENUM,str);

	_itoa(iNew,str,10);
	SetDlgItemText(hwnd,IDC_NEWNUM,str);

	int  nMrgFaces    = CountFaces(&finalList);
	int  nOldSrcFaces = CountUniqueFaces(&finalList,srcList);
	int  nFinalFaces  = nMrgFaces - nOldSrcFaces;
	char sSign[]     = "+";

	if (nFinalFaces < 0)
		sSign[0] = ' ';

	sprintf(str,"%s%i",sSign,nFinalFaces);
	SetDlgItemText(hwnd,IDC_FLDPOLYDIFF,str);

	_itoa(nFinalFaces + nSrcFaces,str,10);
	SetDlgItemText(hwnd,IDC_FLDMERGEDPOLYS,str);

	UpdateNodeProps();
}

void NodeMergeList::ProcDelLists()
{
	DelNonMergedNodes();

	int count = preDelList.Count();

	ip->ClearNodeSelection();

	ip->SelectNodeTab(preDelList,TRUE,FALSE);
	gpGUP->ExecuteStringScript("max delete\n");

	count = postDelList.Count();

	ip->ClearNodeSelection();

	ip->SelectNodeTab(postDelList,TRUE,FALSE);
	gpGUP->ExecuteStringScript("max delete\n");

	/*
	for(i=0;i<count;i++)
		if (NodeExists(preDelList[i]))
		{
			char buf[256];
			sprintf(buf, "PreDel: %s\n",preDelList[i]->GetName());
			OutputDebugString(buf);

			ip->DeleteNode(preDelList[i]);
		}
	
	count = postDelList.Count();

	for(i=0;i<count;i++)
		if (NodeExists(preDelList[i]))
		{
			char buf[256];
			sprintf(buf, "PostDel: %s\n",postDelList[i]->GetName());
			OutputDebugString(buf);

			ip->DeleteNode(postDelList[i]);
		}
	*/
}

void NodeMergeList::ClearDelLists()
{
	preDelList.ZeroCount();
	postDelList.ZeroCount();
}

void NodeMergeList::UpdateMatching()
{
	char wildcard[256];

	IEdit->GetText(wildcard,255);

	// Go through the list and update the selection, highlighting anything that
	// matches our wildcard

	int count = listbox->GetCount();

	for(int i=0;i<count;i++)
	{
		char buf[256];
		listbox->GetItem(i,255,buf);

		if (MatchPattern(CStr(buf),CStr(wildcard)+"*"))
			listbox->SelItem(i);
		else
			listbox->UnSelItem(i);
	}
}
