#include <Next.h>
#include <Material/Multi.h>
#include "MatMergeList.h"
#include "ConflictDlg.h"
#include "MergeRenameDlg.h"
#include "Resource.h"
#include "../UI/PropList.h"
#include "../UI/ColorListBox.h"
#include "../PropEdit/ParseFuncs.h"
#include "../Material/TerrainTypes.h"
#include "../Misc/Util.h"

MatMergeList::MatMergeList(HINSTANCE hInstance,HWND hwndParent,
							 LinkList<MatEntry>* srcList, LinkList<MatEntry>* mergeList,
							 Interface* ip) :
	ModalDlgWindow(hInstance,MAKEINTRESOURCE(IDD_MERGELIST),hwndParent)
	//MSDlgWindow(hInstance,MAKEINTRESOURCE(IDD_MERGELIST),hwndParent)
{
	this->srcList = srcList;
	this->mergeList = mergeList;
	this->ip=ip;

	bAutoMerge = false;

	//AddItems(mergeList);

	// Attach property list
	plist = new PropList(hInstance,1);
	listbox = new ColorListBox(hInstance);

	lastNum = 0;

	BuildFinalList();
}

MatMergeList::~MatMergeList()
{
	delete listbox;
	delete plist;
}

bool MatMergeList::MtlExists(MatEntry* mat)
{
	// Check if this material exists in the merge file
	if (mat->flags & MATFLAG_MERGEMULTI)
	{
		// Check for sub materials
		Link<MatEntry>* mparentLink = srcList->Find(mat->parent);

		if (mparentLink)
		{
			if (mparentLink->data.submtls.Find(mat))
				return true;
			else
				return false;
		}
		else
			return false;
	}
	else
	{
		if (srcList->Find(mat))
			return true;
		else
			return false;
	}
}

Link<MatEntry>* MatMergeList::GetSrcMtl(MatEntry* mat)
{
	// Check if this material exists in the merge file
	if (mat->flags & MATFLAG_MERGEMULTI)
	{
		// Check for sub materials
		Link<MatEntry>* mparentLink = srcList->Find(mat->parent);

		if (mparentLink)
		{
			return mparentLink->data.submtls.Find(mat);
		}
		else
			return NULL;
	}
	else
	{
		return srcList->Find(mat);
	}
}

void MatMergeList::CountMats(int& iSame, int& iNew)
{
	Link<MatEntry>* finalLink = finalList.GetHead();

	iSame = 0;
	iNew  = 0;

	while(finalLink)
	{
		if (MtlExists(&finalLink->data))
			iSame++;
		else
			iNew++;

		finalLink = finalLink->next;
	}
}

void MatMergeList::BuildFinalList()
{
	finalList.Clear();

	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<MatEntry>* link = (Link<MatEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);

			if (link->data.flags & MATFLAG_MERGEMULTIMASTER)
			{
				// Add all its subs
				Link<MatEntry>* sublink = link->data.submtls.GetHead();

				while(sublink)
				{
					finalList.AddToTailUnique(&sublink->data);
					//finalList.AddToTail(&sublink->data);
					sublink = sublink->next;
				}
			}
			else
			{
				finalList.AddToTailUnique(&link->data);
				//finalList.AddToTail(&link->data);
			}
		}
	}
}

int MatMergeList::AddItems(LinkList<MatEntry>* list)
{
	Link<MatEntry>* link = list->GetHead();
	int numItems = 0;

	while(link)
	{
		// Check if this is a multi-material
		if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED &&
			link->data.submtls.GetSize()>0)
		{
			Link<MatEntry>* sublink = link->data.submtls.GetHead();

			int index = SendDlgItemMessage(hwnd,IDC_LIST,LB_ADDSTRING,0,(LPARAM)(char*)link->data.name);
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)link);

			numItems++;

			while(sublink)
			{
				//CStr name=CStr("          --> ")+sublink->data.name+CStr(" (")+link->data.name+")";
				CStr name=CStr("    ")+sublink->data.name;//+CStr(" (")+link->data.name+")";

				int index = SendDlgItemMessage(hwnd,IDC_LIST,LB_ADDSTRING,0,(LPARAM)(char*)name);
				SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)sublink);		

				if (MtlExists(&sublink->data))
					SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(255,0,0));
				else
					SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(0,128,0));

				numItems++;

				sublink=sublink->next;
			}
		}
		else
		{
			// Just add the material name if it's not a multi-material
			int index = SendDlgItemMessage(hwnd,IDC_LIST,LB_ADDSTRING,0,(LPARAM)(char*)link->data.name);
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMDATA,(WPARAM)index,(LPARAM)link);

			// If the item is also in the src list color it red (otherwise green)
			if (srcList->Find(&link->data))
				SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(255,0,0));
			else
				SendDlgItemMessage(hwnd,IDC_LIST,LB_SETITEMCOLOR,(WPARAM)index,(LPARAM)RGB(0,128,0));

			numItems++;
		}

		link=link->next;
	}

	return numItems;
}

void MatMergeList::SelAll()
{
	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	SendDlgItemMessage(hwnd,IDC_LIST,LB_SELITEMRANGE,(WPARAM)TRUE,MAKELPARAM(0,count));
	SelChange();
}

void MatMergeList::SelNone()
{
	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	SendDlgItemMessage(hwnd,IDC_LIST,LB_SELITEMRANGE,(WPARAM)FALSE,MAKELPARAM(0,count));
	SelChange();
}

void MatMergeList::SelInvert()
{
	int size = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	for(int i=0;i<size;i++)
	{
		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETSEL,(WPARAM)FALSE,(LPARAM)i);
		else
			SendDlgItemMessage(hwnd,IDC_LIST,LB_SETSEL,(WPARAM)TRUE,(LPARAM)i);
	}

	SelChange();
}

void MatMergeList::UpdateMtlProps()
{
	// Update material properties
	plist->Clear();

	int count    = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);
	int selCount = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSELCOUNT,0,0);
	int pos   = 0;

	if (selCount == 1)
	{
		// Determine the selected item
		for(int sel=0;sel<count;sel++)
			if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)sel,0))
				break;

		plist->AddStatic("Same Material?","True if the material is duplicated in the current scene");
		
		// Check if material exists
		Link<MatEntry>* mat = (Link<MatEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)sel,0);
		Link<MatEntry>* parentOrig = NULL;
		Link<MatEntry>* matOrig = NULL;

		//matOrig = srcList->Find(&mat->data);
		matOrig = GetSrcMtl(&mat->data);

		if (mat->data.parent)
			parentOrig = srcList->Find(mat->data.parent);

		if (MtlExists(&mat->data))
			plist->SetValue(pos++,"Yes");
		else
			plist->SetValue(pos++,"No");
		
		// Determine number of sub materials
		char buf[256];
		int  nSubs;
		Link<MatEntry>* parent;

		// Display the terrain type
		if (mat->data.mtl->ClassID() == NEXT_MATERIAL_CLASS_ID )
		{
			INExtMaterial* nextMat;
			int iTerrainType;

			if (matOrig)
			{
				plist->AddStatic("Orig Terrain Type","The terrain type of the original material");
			
				nextMat = dynamic_cast<INExtMaterial*>(matOrig->data.mtl);
				iTerrainType = nextMat->GetTerrainType();
				plist->SetValue(pos++,terrain_types[iTerrainType]);
			}

			plist->AddStatic("New Terrain Type","The terrain type of the new material");
		
			nextMat = dynamic_cast<INExtMaterial*>(mat->data.mtl);
			iTerrainType = nextMat->GetTerrainType();
			plist->SetValue(pos++,terrain_types[iTerrainType]);
		}


		if (mat->data.flags & MATFLAG_MERGEMULTI)
		{
			plist->AddStatic("Parent Material","The name of the parent material to which this material is a member of");
			plist->SetValue(pos++,mat->data.parent->name);

			parent = mergeList->Find(mat->data.parent);

			if (parent)
			{
				nSubs = parent->data.submtls.GetSize();
				_itoa(nSubs,buf,10);

				plist->AddStatic("Num parent submats","The number of submaterials contained within the parent material");
				plist->SetValue(pos++,buf);

				nSubs = parentOrig->data.submtls.GetSize();
				_itoa(nSubs,buf,10);

				plist->AddStatic("Num orig parent submats","The number of submaterials contained within the original parent material for the material this would be merged with");
				plist->SetValue(pos++,buf);

				// List out the parent subs
				Link<MatEntry>* psub = parent->data.submtls.GetHead();

				while(psub)
				{
					plist->AddStatic(">>> Merge Parent Sub","One of the sub materials contained in the parent material");
					plist->SetColor(pos,PROPCOLOR_GREEN);
					plist->SetValue(pos++,psub->data.name);
					psub = psub->next;
				}
			}

			// Determine number of subs in original parent
			parent = srcList->Find(mat->data.parent);
			
			if (parent)
			{
				// List out the parent subs
				Link<MatEntry>* psub = parent->data.submtls.GetHead();

				while(psub)
				{
					plist->AddStatic(">>> Orig Parent Sub","One of the sub materials contained in the parent material");
					plist->SetColor(pos,PROPCOLOR_GREEN);
					plist->SetValue(pos++,psub->data.name);
					psub = psub->next;
				}
			}
		}

		plist->BuildUI();
	}
}

void MatMergeList::SelChange()
{
	char str[256];

	BuildFinalList();

	_itoa(finalList.GetSize(),str,10);
	SetDlgItemText(hwnd,IDC_MERGEDNUM,str);

	int iSame,iNew;
	CountMats(iSame,iNew);

	_itoa(iSame,str,10);
	SetDlgItemText(hwnd,IDC_SAMENUM,str);

	_itoa(iNew,str,10);
	SetDlgItemText(hwnd,IDC_NEWNUM,str);

	UpdateMtlProps();
}

BOOL MatMergeList::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		{
			// Set stats
			char str[256];
			_itoa(srcList->GetSize(),str,10);
			SetDlgItemText(hwnd,IDC_SRCNUM,str);
			_itoa(mergeList->GetSize(),str,10);
			SetDlgItemText(hwnd,IDC_MERGENUM,str);

			this->hwnd = hwnd;

			listbox->Attach(GetDlgItem(hwnd,IDC_LIST),LBS_SORT|LBS_EXTENDEDSEL);
			
			if (AddItems(mergeList)==0)
				bAutoMerge = true;

			plist->Attach(GetDlgItem(hwnd,IDC_STATLIST));
			plist->HasApply(false);

			_itoa(finalList.GetSize(),str,10);
			SetDlgItemText(hwnd,IDC_MERGEDNUM,str);

			IEdit = GetICustEdit(GetDlgItem(hwnd,IDC_EDIT));
			IEdit->WantReturn(TRUE);

			if (bAutoMerge)
			{
				CheckDlgButton(hwnd,IDC_DISPLAYSUBS,BST_UNCHECKED);
				SelAll();
				BuildFinalList();
				ProcMerge();
				EndDialog(hwnd,0);				
			}
		}
		return TRUE;

	case WM_CLOSE:
		ReleaseICustEdit(IEdit);
		EndDialog(hwnd,-1);
		//Hide();
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_SELALL:
			SelAll();
			return TRUE;

		case IDC_SELNONE:
			SelNone();
			return TRUE;

		case IDC_SELINVERT:
			SelInvert();
			return TRUE;

		case IDC_LIST:
			switch(HIWORD(wParam))
			{
			case LBN_SELCHANGE:
				SelChange();
				return TRUE;
			}
			break;

		case IDC_EDIT:
			switch(HIWORD(wParam))
			{
			case EN_CHANGE:
				UpdateMatching();
				return TRUE;
			}
			break;

		case IDC_DISPLAYSUBS:
			SendDlgItemMessage(hwnd,IDC_LIST,LB_RESETCONTENT,0,0);

			if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
				listbox->Attach(GetDlgItem(hwnd,IDC_LIST),LBS_EXTENDEDSEL);
			else
				listbox->Attach(GetDlgItem(hwnd,IDC_LIST),LBS_SORT|LBS_EXTENDEDSEL);

			AddItems(mergeList);
			return TRUE;

		case IDC_MERGE:
			ProcMerge();
			EndDialog(hwnd,0);
			return TRUE;

		case IDC_CANCEL:
			ReleaseICustEdit(IEdit);
			EndDialog(hwnd,-1);
			return TRUE;
		}
	}

	return FALSE;
}

void MatMergeList::DeleteMtlFromScene(Mtl* mtl)
{
	// We can't remove the materials directly from the material list
	// instead we'll go through the references to this material and set
	// the material on those nodes to NULL
	RefList& refList     = mtl->GetRefList();
	RefListItem* refItem = refList.FirstItem();
	RefListItem* refNext;

	while(refItem)
	{
		refNext = refItem->next;

		if (refItem->maker->ClassID()==Class_ID(BASENODE_CLASS_ID, 0))
		{
			INode* node = (INode*)refItem->maker;

			// Instead of clearing out this material we'll now assign it a material with
			// the same name that exists within the original scene if one exists, if not
			// we'll assign it to NULL
			Mtl*  mtl = node->GetMtl();
			
			if (!mtl)
			{
				refItem = refNext;
				continue;
			}

			MatEntry mentry;
			mentry.name = mtl->GetName();

			Link<MatEntry>* link = srcList->Find(&mentry);

			if (link && link->data.mtl && MtlInScene(link->data.mtl))
				node->SetMtl(link->data.mtl);
			else
				node->SetMtl(NULL);

			node->NotifyDependents(FOREVER,PART_MTL,REFMSG_CHANGE);
		}

		refItem = refNext;
	}
}

void MatMergeList::RemapMtlRefs(Mtl* mtlSrc, Mtl* mtlDest)
{
	// Don't remap if both materials don't exist
	if (!MtlInScene(mtlSrc) ||
		!MtlInScene(mtlDest))
		return;

	// This function will remap references to the source material to
	// use the destination material instead
	RefList& refList     = mtlSrc->GetRefList();
	refList.Cleanup();
	RefListItem* refItem = refList.FirstItem();
	RefListItem* refNext;

	while(refItem)
	{
		refNext = refItem->next;

		if (refItem->maker->ClassID()==Class_ID(BASENODE_CLASS_ID, 0))
		{
			INode* node = (INode*)refItem->maker;
			node->SetMtl(mtlDest);
			node->NotifyDependents(FOREVER,PART_MTL,REFMSG_CHANGE);
		}

		refItem = refNext;
	}
}

// Returns true if the given material entry is selected in the list
bool MatMergeList::IsSelected(MatEntry* me)
{
	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		Link<MatEntry>* link = (Link<MatEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);

		if (SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0) &&
			&link->data == me)
		{
			return true;
		}
	}

	return false;
}

// Revamped to take account for deleting materials that aren't in the visual list box
void MatMergeList::DeleteNonMergedMtls()
{
	// Run through the merge list and delete any materials that correspond to entries not selected in the listbox
	Link<MatEntry>* link = mergeList->GetHead();

	while(link)
	{
		if (!IsSelected(&link->data))
		{
			if (link->data.flags & MATFLAG_MERGEMULTI)
			{
				// Delete sub-material
				assert(link->data.parent!=NULL);
				Mtl* mainmtl = link->data.parent->mtl;
				assert(mainmtl!=NULL);

				// Verify that the material is a multi material (it should be since MATFLAG_MERGEMULTI is set)
				assert((mainmtl->ClassID()==Class_ID(MULTI_CLASS_ID,0) ||
					    mainmtl->ClassID()==vNEXT_MULTI_MATERIAL_CLASS_ID) );

				//Multi* multimat = dynamic_cast<Multi*>(mainmtl);
				Multi* multimat = (Multi*)mainmtl;
				multimat->DeleteSubMtl(link->data.mtl);

				link->data.flags |= MATFLAG_NOMERGE;
			}
			else
			{
				// Delete material from scene
				DeleteMtlFromScene(link->data.mtl);
			}
		}

		link = link->next;
	}
}

/*
void MatMergeList::DeleteNonMergedMtls()
{
	// Run through the list box and delete anything not selected from the scene
	int count = SendDlgItemMessage(hwnd,IDC_LIST,LB_GETCOUNT,0,0);

	for(int i=0;i<count;i++)
	{
		if (!SendDlgItemMessage(hwnd,IDC_LIST,LB_GETSEL,(WPARAM)i,0))
		{
			Link<MatEntry>* link = (Link<MatEntry>*)SendDlgItemMessage(hwnd,IDC_LIST,LB_GETITEMDATA,(WPARAM)i,0);

			if (link->data.flags & MATFLAG_MERGEMULTI)
			{
				// Delete sub-material
				assert(link->data.parent!=NULL);
				Mtl* mainmtl = link->data.parent->mtl;
				assert(mainmtl!=NULL);

				// Verify that the material is a multi material (it should be since MATFLAG_MERGEMULTI is set)
				assert((mainmtl->ClassID()==Class_ID(MULTI_CLASS_ID,0) ||
					    mainmtl->ClassID()==vNEXT_MULTI_MATERIAL_CLASS_ID) );

				//Multi* multimat = dynamic_cast<Multi*>(mainmtl);
				Multi* multimat = (Multi*)mainmtl;
				multimat->DeleteSubMtl(link->data.mtl);

				link->data.flags |= MATFLAG_NOMERGE;
			}
			else
			{
				// Delete material from scene
				DeleteMtlFromScene(link->data.mtl);
			}
		}
	}
}
*/

// This function reassigns the material IDs of the individual
// faces within the mesh to correspond to our new combined material
void MatMergeList::ReassignMtlIDs(Multi* oldMtl, int oldMtlID,
								  Multi* newMtl, int newMtlID)
{
	RefList& refList     = oldMtl->GetRefList();
	RefListItem* refItem = refList.FirstItem();
	RefListItem* refNext;

	int nOldSubs = oldMtl->NumSubs();

	while(refItem)
	{
		refNext = refItem->next;

		if (refItem->maker->ClassID()==Class_ID(BASENODE_CLASS_ID, 0))
		{
			INode* node = (INode*)refItem->maker;
			
			// Reassign the new multimaterial in place of the old
			node->SetMtl(newMtl);
			node->NotifyDependents(FOREVER,PART_MTL,REFMSG_CHANGE);

			Object* obj = node->EvalWorldState(0).obj;

			// Reassign the material IDs of each face in the referencing mesh
			if (obj && obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

				Mesh& mesh   = triobj->GetMesh();
				int   nFaces = mesh.getNumFaces();

				for(int i=0;i<nFaces;i++)
				{
					Face& face = mesh.faces[i];
					
					if (((face.getMatID() - 1) % nOldSubs) + 1 == oldMtlID)
						face.setMatID(newMtlID);
				}
			}
		}

		refItem = refNext;
	}	
}

void MatMergeList::ReassignMtlIDs()
{
	Link<ReassignDB>* rdbLink = reassignList.GetHead();

	while(rdbLink)
	{
		// Before we can update the oldMtl to index the new material properly we have to go through everything
		// using the new material and adjust those mtlID's to account for the fact that we've added submtls
		// It's possible that mtlID's may have referenced a material because the mtlID value flipped back to
		// the first material.  Now that we've added more subs that would no longer be the case
		RefList& preRefList = rdbLink->data.newMtl->GetRefList();
		RefListItem* prerefItem = preRefList.FirstItem();
		
		while(prerefItem)
		{
			if (prerefItem->maker->ClassID()==Class_ID(BASENODE_CLASS_ID,0))
			{
				INode* node = (INode*)prerefItem->maker;
				Object* obj = node->EvalWorldState(0).obj;

				if (obj && obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

					Mesh& mesh = triObj->GetMesh();
					int   nFaces = mesh.getNumFaces();
					int    nSubs = rdbLink->data.norigSubs;

					for(int i=0;i<nFaces;i++)
					{
						Face& face = mesh.faces[i];

						int ID = face.getMatID() % nSubs;
						face.setMatID(ID);
					}

					mesh.InvalidateTopologyCache();
				}

				node->NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
			}

			prerefItem = prerefItem->next;
		}
		///////////

		// Traverse through all the old materials that should be updated to this new material
		Link<ReassignRecord>* recLink = rdbLink->data.DB.GetHead();

		while(recLink)
		{
			// Now traverse all the references to each old material and update them for the new material
			RefList& refList     = recLink->data.oldMtl->GetRefList();
			RefListItem* refItem = refList.FirstItem();
			RefListItem* refNext;

			int nOldSubs = recLink->data.oldMtl->NumSubs();

			while(refItem)
			{
				refNext = refItem->next;

				if (refItem->maker->ClassID()==Class_ID(BASENODE_CLASS_ID, 0))
				{
					// Do mesh mtlID reassignment
					INode* node = (INode*)refItem->maker;
					Object* obj = node->EvalWorldState(0).obj;

					// Reassign the material IDs of each face in the referencing mesh
					if (obj && obj->CanConvertToType(triObjectClassID))
					{
						TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

						Mesh& mesh   = triobj->GetMesh();
						int   nFaces = mesh.getNumFaces();

						for(int i=0;i<nFaces;i++)
						{
							Face& face = mesh.faces[i];
							
							// Traverse all old materials that will be reassigned to our new one updating IDs					
							Link<ReassignEntry>* entryLink = recLink->data.reassignDB.GetHead();

							// Perform each individual ID reassignment for this old material
							while(entryLink)
							{
								// It's not enough to check that a face is equal to a value, as mtlIDs greater
								// than the maximum mtlID in the material wrap back to the first submtl
								//if (face.getMatID() == entryLink->data.oldID)
								int faceMatID = face.getMatID();
								
//								if (((face.getMatID() - 1) % nOldSubs) + 1 == entryLink->data.oldID + 1)
//									face.setMatID(entryLink->data.newID + 1);

								if (((face.getMatID()) % nOldSubs) == entryLink->data.oldID)
									face.setMatID(entryLink->data.newID);

								entryLink = entryLink->next;
							}
						}

						mesh.InvalidateTopologyCache();

						if (triobj!=obj)
							triobj->DeleteThis();
					}

					// Now that we've reassigned all the material IDs for the object it should
					// be safe to set the material
					node->SetMtl(rdbLink->data.newMtl);
					//node->NotifyDependents(FOREVER,PART_MTL,REFMSG_CHANGE);
					node->NotifyDependents(FOREVER,PART_ALL,REFMSG_CHANGE);
				}

				refItem = refNext;
			}

			recLink = recLink->next;
		}

		rdbLink = rdbLink->next;
	}
}

void MatMergeList::UpdateMatching()
{
	char wildcard[256];

	IEdit->GetText(wildcard,255);

	// Go through the list and update the selection, highlighting anything that
	// matches our wildcard

	int count = listbox->GetCount();

	for(int i=0;i<count;i++)
	{
		char buf[256];
		listbox->GetItem(i,255,buf);

		if (MatchPattern(CStr(buf),CStr(wildcard)+"*"))
			listbox->SelItem(i);
		else
			listbox->UnSelItem(i);
	}
}

void MatMergeList::NonSubMerge()
{
	// Run through the final list deleting/merging conflicting materials
	ConflictDlg* conflictDlg = new ConflictDlg(hInstance,hwnd);
	conflictDlg->SetMtlMode();
	Link<MatEntry>* mergelink = finalList.GetHead();

	while(mergelink)
	{
	
		mergelink = mergelink->next;
	}
}

DWORD WINAPI MatMergeListProgressFunc(LPVOID arg) 
{
    return(0);
}

void MatMergeList::ProcMerge()
{
	// If they've chosen to merge in any materials, get the latest textures
	if( finalList.GetHead())
	{
		GetLatestTextures();		
	}

	DeleteNonMergedMtls();

	if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED)
	{
		NonSubMerge();
		//return;
	}

	// Run through the final list deleting/merging conflicting materials
	ConflictDlg* conflictDlg = new ConflictDlg(hInstance,hwnd);
	conflictDlg->SetMtlMode();
	Link<MatEntry>* mergelink = finalList.GetHead();
	Link<MatEntry>* mergelinkNext;

	while(mergelink)
	{
		mergelinkNext = mergelink->next;

		if (!(mergelink->data.flags & MATFLAG_NOMERGE))
		{
			// Perform merge for non sub materials
			Link<MatEntry>* srclink = srcList->Find(&mergelink->data);

			char strErr[256];
			sprintf(strErr,"Mtl: %s\n",(char*)mergelink->data.name);
			OutputDebugString(strErr);

			// Pick the parent materials if in non-sub mode
			if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED)
			{
				if (mergelink->data.parent)
				{
					srclink = srcList->Find(mergelink->data.parent);
					mergelink = mergeList->Find(mergelink->data.parent);
				}
			}

			if (srclink)
			{
				ResolveMode rmode;

				// If we're doing multimaterial level merges only prompt for the action
				// on all of the subs once rather than for each
				if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED &&
					mergelink->data.parent &&
					mergelink->data.parent->submode != RESOLVE_UNDEFINED)
				{
					rmode = mergelink->data.parent->submode;
				}
				else
				{
					//conflictDlg->DisableMerge();
					conflictDlg->EnableMerge();

					CStr name;

					if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) == BST_CHECKED &&
						mergelink->data.flags & MATFLAG_MERGEMULTI &&
						mergelink->data.parent)
					{
						name = mergelink->data.name + CStr(" (") + mergelink->data.parent->name + ")";
					}
					else
					{
						if (mergelink->data.parent)
							name = mergelink->data.parent->name;
						else
							name = mergelink->data.name;
					}


					conflictDlg->SetItemName(name);
					
					if (!conflictDlg->ApplyAll())
						conflictDlg->Show();							// This blocks

					rmode = conflictDlg->GetResolveMode();

					// Store the resolve mode in the parent data if appropriate
					if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED &&
						mergelink->data.parent)
					{
						mergelink->data.parent->submode = rmode;
					}
				}

				switch(rmode)
				{
				case RESOLVE_DELSRC:
					RemapMtlRefs(srclink->data.mtl, mergelink->data.mtl);
					//DeleteMtlFromScene(srclink->data.mtl);
					break;

				case RESOLVE_DELMERGE:				
					RemapMtlRefs(mergelink->data.mtl, srclink->data.mtl);
					//DeleteMtlFromScene(mergelink->data.mtl);
					break;

				case RESOLVE_MERGE:
					{
						CStr mergeName;
						char* renameName = conflictDlg->GetRenameName();

						if (renameName == NULL)
						{
							MergeRenameDlg* mergeDlg = new MergeRenameDlg(hInstance,hwnd,mergelink->data.name);
							mergeDlg->Show();

							mergeName = mergeDlg->GetName();
							conflictDlg->SetRenameName(mergeName);
							//ip->MakeNameUnique(mergeName);
							MakeUniqueMtlName(mergeName);

							// Only assign the name if we're in submtl mode, or it happens to be the parent
							if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
								mergelink->data.mtl->SetName(mergeName);
							else
							{
								if (mergelink->data.flags & MATFLAG_MERGEMULTIMASTER)
									mergelink->data.mtl->SetName(mergeName);
							}

							delete mergeDlg;
						}
						else
						{
							mergeName = renameName;
							//ip->MakeNameUnique(mergeName);
							MakeUniqueMtlName(mergeName);
							
							// Only assign the name if we're in submtl mode, or it happens to be the parent
							if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
								mergelink->data.mtl->SetName(mergeName);
							else
							{
								if (mergelink->data.flags & MATFLAG_MERGEMULTIMASTER &&
									!(mergelink->data.flags & MATFLAG_PROCESSED))
								{
									mergelink->data.mtl->SetName(mergeName);
									mergelink->data.flags |= MATFLAG_PROCESSED;
								}
							}
						}
					}
					break;
				}
			}
			else
			{
				// Perform merge for submaterials
				if (mergelink->data.flags & MATFLAG_MERGEMULTI)
				{
					srclink = srcList->Find(mergelink->data.parent);

					// Now scan the submtls to see if there's a collision
					Link<MatEntry>* srcSubLink;
						
					if (srclink)
						srcSubLink = srclink->data.submtls.Find(&mergelink->data);
					else
						srcSubLink = NULL;

					if (srcSubLink)
					{
						// This sub material exists, we should be able to replace it directly
						// Query user to see which sub material should be kept

						ResolveMode rmode;

						// If we're doing multimaterial level merges only prompt for the action
						// on all of the subs once rather than for each
						if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED &&
							srcSubLink->data.parent &&
							srcSubLink->data.parent->submode != RESOLVE_UNDEFINED)
						{
							rmode = srcSubLink->data.parent->submode;
						}
						else
						{
							//conflictDlg->DisableMerge();
							conflictDlg->EnableMerge();

							CStr name;

							if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) == BST_CHECKED &&
								srcSubLink->data.flags & MATFLAG_MERGEMULTI &&
								srcSubLink->data.parent)
							{
								name = srcSubLink->data.name + CStr(" (") + srcSubLink->data.parent->name + ")";
							}
							else
							{
								if (srcSubLink->data.parent)
									name = srcSubLink->data.parent->name;
								else
									name = srcSubLink->data.name;
							}

							conflictDlg->SetItemName(name);
							
							if (!conflictDlg->ApplyAll())
								conflictDlg->Show();							// This blocks

							rmode = conflictDlg->GetResolveMode();

							// Store the resolve mode in the parent data if appropriate
							if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS) != BST_CHECKED &&
								srcSubLink->data.parent)
							{
								srcSubLink->data.parent->submode = rmode;
							}
						}

						switch(rmode)
						{ 
						case RESOLVE_DELSRC:
							{
								Multi* multi    = (Multi*)srclink->data.mtl;
								Multi* oldMulti = (Multi*)mergelink->data.parent->mtl;
								int mtlID = multi->GetMtlID(srcSubLink->data.mtl);
								assert( mtlID != -1);
								int oldID = oldMulti->GetMtlID(mergelink->data.mtl);

								// TODO: Reassign the referenced source geometry to use the merged submtl

								// Add the reindexing information to the database
								// We can't reindex this immediately, because if we change
								// a referenced node's material it would break the references
								// to other submaterials
								ReassignDB     rdb;
								ReassignRecord rrec;
								ReassignEntry  rentry;

								rdb.newMtl    = multi;
								rdb.norigSubs = multi->NumSubs();

								multi->SetSubMtl(mtlID, mergelink->data.mtl);

								rentry.oldID = oldID;
								rentry.newID = mtlID;

								rrec.oldMtl  = (Multi*)mergelink->data.parent->mtl;

								Link<ReassignDB>*     rdbLink;
								Link<ReassignRecord>* recLink;
								Link<ReassignEntry>*  entryLink;

								rdbLink = reassignList.Find(&rdb);

								if (!rdbLink)
									rdbLink = reassignList.Add(&rdb);

								recLink = rdbLink->data.DB.Find(&rrec);

								if (!recLink)
									recLink = rdbLink->data.DB.Add(&rrec);

								entryLink = recLink->data.reassignDB.Find(&rentry);

								if (!entryLink)
									entryLink = recLink->data.reassignDB.Add(&rentry);

							}
							break;
						case RESOLVE_DELMERGE:
							{
								// TODO: Reassign the referenced merged geometry to use the source submtl
								// Should occur by default

									// ----------------------------------------------------------------------------


									Multi* multi    = (Multi*)srclink->data.mtl;
									Multi* oldMulti = (Multi*)mergelink->data.parent->mtl;
									int oldID = multi->GetMtlID(srcSubLink->data.mtl);
									//assert( oldID != -1);
									int mtlID = oldMulti->GetMtlID(mergelink->data.mtl);

									// Add the reindexing information to the database
									// We can't reindex this immediately, because if we change
									// a referenced node's material it would break the references
									// to other submaterials
									ReassignDB     rdb;
									ReassignRecord rrec;
									ReassignEntry  rentry;

									//rdb.newMtl   = oldMulti;
									rdb.newMtl    = multi;
									rdb.norigSubs = multi->NumSubs();
										
									//multi->SetSubMtl(mtlID, srclink->data.mtl);

									rentry.oldID = oldID;
									rentry.newID = mtlID;

									//rrec.oldMtl  = (Multi*)srclink->data.mtl;
									rrec.oldMtl    = oldMulti;

									Link<ReassignDB>*     rdbLink;
									Link<ReassignRecord>* recLink;
									Link<ReassignEntry>*  entryLink;

									rdbLink = reassignList.Find(&rdb);

									if (!rdbLink)
										rdbLink = reassignList.Add(&rdb);

									recLink = rdbLink->data.DB.Find(&rrec);

									if (!recLink)
										recLink = rdbLink->data.DB.Add(&rrec);

									entryLink = recLink->data.reassignDB.Find(&rentry);

									if (!entryLink)
										entryLink = recLink->data.reassignDB.Add(&rentry);



									// ----------------------------------------------------------------------------

							}
							break;
						case RESOLVE_MERGE:
							{
								MergeRenameDlg* renameDlg;
								char* renameName = conflictDlg->GetRenameName();
								CStr newName;

								if (renameName == NULL)
								{
									if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
									{
										// Prompt the user for a new name for the material
										renameDlg = new MergeRenameDlg(hInstance,hwnd,srcSubLink->data.name);
									}
									else
									{
										if (srcSubLink->data.parent)
											renameDlg = new MergeRenameDlg(hInstance,hwnd,srcSubLink->data.parent->name);
									}
									
									renameDlg->Show();		// blocks
									newName = renameDlg->GetName();
									conflictDlg->SetRenameName(newName);
								}
								else
								{
									newName = renameName;
								}

								// Just to be sure the name is unique
								//ip->MakeNameUnique(newName);
								MakeUniqueMtlName(newName);

								// Now that we have a new name, assign it
								if (srclink)
								{
									// Add the submaterial to the appropriate multi material
									// with a newly assigned material ID
									Multi* multi  = (Multi*)srclink->data.mtl;
									Multi* mmulti = (Multi*)mergelink->data.parent->mtl;
									int oldID = mmulti->GetMtlID(mergelink->data.mtl);
									int maxID = multi->NumSubMtls();

									// Only assign the name if we're in submtl mode, or it happens to be the parent
									if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
										mergelink->data.mtl->SetName(newName);
									else
									{
										if (mergelink->data.flags & MATFLAG_MERGEMULTIMASTER)
											mergelink->data.mtl->SetName(newName);
										else
											if (mergelink->data.parent)
												mergelink->data.parent->mtl->SetName(newName);
									}

									// Add the reindexing information to the database
									// We can't reindex this immediately, because if we change
									// a referenced node's material it would break the references
									// to other submaterials
									ReassignDB     rdb;
									ReassignRecord rrec;
									ReassignEntry  rentry;

									rdb.newMtl    = multi;
									rdb.norigSubs = multi->NumSubs();

									multi->SetSubMtl(maxID,mergelink->data.mtl);

									rentry.oldID = oldID;
									rentry.newID = maxID;

									rrec.oldMtl  = mmulti;

									Link<ReassignDB>*     rdbLink;
									Link<ReassignRecord>* recLink;
									Link<ReassignEntry>*  entryLink;

									rdbLink = reassignList.Find(&rdb);

									if (!rdbLink)
										rdbLink = reassignList.Add(&rdb);

									recLink = rdbLink->data.DB.Find(&rrec);

									if (!recLink)
										recLink = rdbLink->data.DB.Add(&rrec);

									entryLink = recLink->data.reassignDB.Find(&rentry);

									if (!entryLink)
										entryLink = recLink->data.reassignDB.Add(&rentry);
								}

								if (!renameName)
									delete renameDlg;
							}
							break;
						}						
					}
					else
					{
						// This sub material doesn't exist, we'll need to add it and reindex
						// face material IDs if appropriate

						if (srclink)
						{
							// Add the submaterial to the appropriate multi material
							// with a newly assigned material ID
							Multi* multi  = (Multi*)srclink->data.mtl;
							Multi* mmulti = (Multi*)mergelink->data.parent->mtl;
							int oldID = mmulti->GetMtlID(mergelink->data.mtl);
							int maxID = multi->NumSubMtls();

							// Add the reindexing information to the database
							// We can't reindex this immediately, because if we change
							// a referenced node's material it would break the references
							// to other submaterials
							ReassignDB     rdb;
							ReassignRecord rrec;
							ReassignEntry  rentry;

							rdb.newMtl    = multi;
							rdb.norigSubs = multi->NumSubs();

							multi->SetSubMtl(maxID,mergelink->data.mtl);

							rentry.oldID = oldID;
							rentry.newID = maxID;

							rrec.oldMtl  = mmulti;

							Link<ReassignDB>*     rdbLink;
							Link<ReassignRecord>* recLink;
							Link<ReassignEntry>*  entryLink;

							rdbLink = reassignList.Find(&rdb);

							if (!rdbLink)
								rdbLink = reassignList.Add(&rdb);

							recLink = rdbLink->data.DB.Find(&rrec);

							if (!recLink)
								recLink = rdbLink->data.DB.Add(&rrec);

							entryLink = recLink->data.reassignDB.Find(&rentry);

							if (!entryLink)
								entryLink = recLink->data.reassignDB.Add(&rentry);
						}
						else
						{
							// Add a new multi-material with this entry
							// Nothing changes we keep the merge material

						}
					}
				}

			}

			mergelink = mergelinkNext;
		}
	}

	if (IsDlgButtonChecked(hwnd,IDC_DISPLAYSUBS)==BST_CHECKED)
		ReassignMtlIDs();

	delete conflictDlg;
}


/*
void MatMergeList::ProcMerge()
{
	DeleteNonMergedMtls();

	// Run through the final list deleting/merging conflicting materials
	ConflictDlg* conflictDlg = new ConflictDlg(hInstance,hwnd);
	Link<MatEntry>* mergelink = finalList.GetHead();

	while(mergelink)
	{
		if (!(mergelink->data.flags & MATFLAG_NOMERGE))
		{
			// Perform merge for non sub materials
			Link<MatEntry>* srclink = srcList->Find(&mergelink->data);

			char strErr[256];
			sprintf(strErr,"Mtl: %s\n",(char*)mergelink->data.name);
			OutputDebugString(strErr);

			if (srclink &&
				IsInstr(srclink->data.name,"Zoo_Artist_04_24_02"))
				__asm int 3;

			if (srclink)
			{
				do	// Handle same file having multiple mtls named the same
				{
					//conflictDlg->DisableMerge();
					conflictDlg->EnableMerge();

					CStr name;

					if (mergelink->data.flags & MATFLAG_MERGEMULTI &&
						mergelink->data.parent)
					{
						name = mergelink->data.name + CStr(" (") + mergelink->data.parent->name + ")";
					}
					else
					{
						name = mergelink->data.name;
					}

					conflictDlg->SetItemName(name);
					
					if (!conflictDlg->ApplyAll())
						conflictDlg->Show();							// This blocks

					switch(conflictDlg->GetResolveMode())
					{
					case RESOLVE_DELSRC:
						RemapMtlRefs(srclink->data.mtl, mergelink->data.mtl);
						//DeleteMtlFromScene(srclink->data.mtl);
						break;

					case RESOLVE_DELMERGE:				
						RemapMtlRefs(mergelink->data.mtl, srclink->data.mtl);
						//DeleteMtlFromScene(mergelink->data.mtl);
						break;

					case RESOLVE_MERGE:
						{
							CStr mergeName;

							MergeRenameDlg* mergeDlg = new MergeRenameDlg(hInstance,hwnd,mergelink->data.name);
							mergeDlg->Show();

							mergeName = mergeDlg->GetName();
							ip->MakeNameUnique(mergeName);
							mergelink->data.mtl->SetName(mergeName);

							delete mergeDlg;
						}
						break;
					}

				} while(srclink = srcList->FindFrom(&mergelink->data,srclink));
			}
			else
			{
				// Perform merge for submaterials
				if (mergelink->data.flags & MATFLAG_MERGEMULTI)
				{
					srclink = srcList->Find(mergelink->data.parent);

					// Now scan the submtls to see if there's a collision
					Link<MatEntry>* srcSubLink;
						
					if (srclink)
						srcSubLink = srclink->data.submtls.Find(&mergelink->data);
					else
						srcSubLink = NULL;

					if (srcSubLink)
					{
						// This sub material exists, we should be able to replace it directly
						// Query user to see which sub material should be kept

						do		// handle multiple materials with same name in same file
						{
							conflictDlg->EnableMerge();

							CStr name;

							if (srcSubLink->data.flags & MATFLAG_MERGEMULTI &&
								srcSubLink->data.parent)
							{
								name = srcSubLink->data.name + CStr(" (") + srcSubLink->data.parent->name + ")";
							}
							else
							{
								name = srcSubLink->data.name;
							}

							if (IsInstr(name,"Zoo_Artist_04_24_02") &&
								IsInstr(name,"monkey Fence02"))
								__asm int 3;

							conflictDlg->SetItemName(name);

							if (!conflictDlg->ApplyAll())
								conflictDlg->Show();

							switch(conflictDlg->GetResolveMode())
							{ 
							case RESOLVE_DELSRC:
								{
									Multi* multi    = (Multi*)srclink->data.mtl;
									Multi* oldMulti = (Multi*)mergelink->data.parent->mtl;
									int mtlID = multi->GetMtlID(srcSubLink->data.mtl);
									assert( mtlID != -1);
									int oldID = oldMulti->GetMtlID(mergelink->data.mtl);

									multi->SetSubMtl(mtlID, mergelink->data.mtl);
									// TODO: Reassign the referenced source geometry to use the merged submtl

									// Add the reindexing information to the database
									// We can't reindex this immediately, because if we change
									// a referenced node's material it would break the references
									// to other submaterials
									ReassignDB     rdb;
									ReassignRecord rrec;
									ReassignEntry  rentry;

									rdb.newMtl   = multi;

									rentry.oldID = oldID;
									rentry.newID = mtlID;

									rrec.oldMtl  = (Multi*)mergelink->data.parent->mtl;

									Link<ReassignDB>*     rdbLink;
									Link<ReassignRecord>* recLink;
									Link<ReassignEntry>*  entryLink;

									rdbLink = reassignList.Find(&rdb);

									if (!rdbLink)
										rdbLink = reassignList.Add(&rdb);

									recLink = rdbLink->data.DB.Find(&rrec);

									if (!recLink)
										recLink = rdbLink->data.DB.Add(&rrec);

									entryLink = recLink->data.reassignDB.Find(&rentry);

									if (!entryLink)
										entryLink = recLink->data.reassignDB.Add(&rentry);

								}
								break;
							case RESOLVE_DELMERGE:
								{
									// TODO: Reassign the referenced merged geometry to use the source submtl
									// Should occur by default

									// ----------------------------------------------------------------------------


									Multi* multi    = (Multi*)srclink->data.mtl;
									Multi* oldMulti = (Multi*)mergelink->data.parent->mtl;
									int oldID = multi->GetMtlID(srcSubLink->data.mtl);
									//assert( oldID != -1);
									int mtlID = oldMulti->GetMtlID(mergelink->data.mtl);

									multi->SetSubMtl(mtlID, srclink->data.mtl);

									// Add the reindexing information to the database
									// We can't reindex this immediately, because if we change
									// a referenced node's material it would break the references
									// to other submaterials
									ReassignDB     rdb;
									ReassignRecord rrec;
									ReassignEntry  rentry;

									rdb.newMtl   = oldMulti;

									rentry.oldID = oldID;
									rentry.newID = mtlID;

									rrec.oldMtl  = (Multi*)srclink->data.mtl;

									Link<ReassignDB>*     rdbLink;
									Link<ReassignRecord>* recLink;
									Link<ReassignEntry>*  entryLink;

									rdbLink = reassignList.Find(&rdb);

									if (!rdbLink)
										rdbLink = reassignList.Add(&rdb);

									recLink = rdbLink->data.DB.Find(&rrec);

									if (!recLink)
										recLink = rdbLink->data.DB.Add(&rrec);

									entryLink = recLink->data.reassignDB.Find(&rentry);

									if (!entryLink)
										entryLink = recLink->data.reassignDB.Add(&rentry);



									// ----------------------------------------------------------------------------
								}
								break;
							case RESOLVE_MERGE:
								{
									// Prompt the user for a new name for the material
									CStr newName;
									MergeRenameDlg* renameDlg = new MergeRenameDlg(hInstance,hwnd,srcSubLink->data.name);
									
									renameDlg->Show();		// blocks
									newName = renameDlg->GetName();

									// Just to be sure the name is unique
									ip->MakeNameUnique(newName);

									// Now that we have a new name, assign it
									if (srclink)
									{
										// Add the submaterial to the appropriate multi material
										// with a newly assigned material ID
										Multi* multi  = (Multi*)srclink->data.mtl;
										Multi* mmulti = (Multi*)mergelink->data.parent->mtl;
										int oldID = mmulti->GetMtlID(mergelink->data.mtl);
										int maxID = multi->NumSubMtls();
										
										multi->SetSubMtl(maxID,mergelink->data.mtl);
										mergelink->data.mtl->SetName(newName);

										// Add the reindexing information to the database
										// We can't reindex this immediately, because if we change
										// a referenced node's material it would break the references
										// to other submaterials
										ReassignDB     rdb;
										ReassignRecord rrec;
										ReassignEntry  rentry;

										rdb.newMtl   = multi;

										rentry.oldID = oldID;
										rentry.newID = maxID;

										rrec.oldMtl  = mmulti;

										Link<ReassignDB>*     rdbLink;
										Link<ReassignRecord>* recLink;
										Link<ReassignEntry>*  entryLink;

										rdbLink = reassignList.Find(&rdb);

										if (!rdbLink)
											rdbLink = reassignList.Add(&rdb);

										recLink = rdbLink->data.DB.Find(&rrec);

										if (!recLink)
											recLink = rdbLink->data.DB.Add(&rrec);

										entryLink = recLink->data.reassignDB.Find(&rentry);

										if (!entryLink)
											entryLink = recLink->data.reassignDB.Add(&rentry);
									}

									delete renameDlg;
								}
								break;
							}
							
						} while (srcSubLink = srclink->data.submtls.FindFrom(&mergelink->data,srcSubLink));

					}
					else
					{
						// This sub material doesn't exist, we'll need to add it and reindex
						// face material IDs if appropriate

						if (srclink)
						{
							// Add the submaterial to the appropriate multi material
							// with a newly assigned material ID
							Multi* multi  = (Multi*)srclink->data.mtl;
							Multi* mmulti = (Multi*)mergelink->data.parent->mtl;
							int oldID = mmulti->GetMtlID(mergelink->data.mtl);
							int maxID = multi->NumSubMtls();
							
							multi->SetSubMtl(maxID,mergelink->data.parent->mtl);

							// Add the reindexing information to the database
							// We can't reindex this immediately, because if we change
							// a referenced node's material it would break the references
							// to other submaterials
							ReassignDB     rdb;
							ReassignRecord rrec;
							ReassignEntry  rentry;

							rdb.newMtl   = multi;

							rentry.oldID = oldID;
							rentry.newID = maxID;

							rrec.oldMtl  = mmulti;

							Link<ReassignDB>*     rdbLink;
							Link<ReassignRecord>* recLink;
							Link<ReassignEntry>*  entryLink;

							rdbLink = reassignList.Find(&rdb);

							if (!rdbLink)
								rdbLink = reassignList.Add(&rdb);

							recLink = rdbLink->data.DB.Find(&rrec);

							if (!recLink)
								recLink = rdbLink->data.DB.Add(&rrec);

							entryLink = recLink->data.reassignDB.Find(&rentry);

							if (!entryLink)
								entryLink = recLink->data.reassignDB.Add(&rentry);
						}
						else
						{
							// Add a new multi-material with this entry
							// Nothing changes we keep the merge material

						}
					}
				}

			}

			mergelink=mergelink->next;
		}
	}

	ReassignMtlIDs();

	delete conflictDlg;
}
*/

bool MatMergeList::MtlInScene(Mtl* mtl,INode* root)
{
	if (!root)
		root = ip->GetRootNode();

	int kids = root->NumberOfChildren();

	for(int i=0;i<kids;i++)
	{
		INode* child = root->GetChildNode(i);

		if (MtlInScene(mtl,child))
			return true;
	}

	if (root->GetMtl()==mtl)
		return true;

	return false;
}

bool MatMergeList::MtlInScene(CStr name,INode* root)
{
	if (!root)
		root = ip->GetRootNode();

	int kids = root->NumberOfChildren();

	for(int i=0;i<kids;i++)
	{
		INode* child = root->GetChildNode(i);

		if (MtlInScene(name,child))
			return true;
	}

	Mtl* mtl = root->GetMtl();

	if (mtl)
	{
		if (mtl->GetName() == name)
			return true;

		if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID ||
			mtl->ClassID() == Class_ID(MULTI_CLASS_ID,0))
		{
			Multi* multi = (Multi*)mtl;
			int numsubs = multi->NumSubs();

			for(int sub = 0; sub < numsubs; sub++)
			{
				Mtl* submtl = multi->GetSubMtl(sub);

				if (submtl && submtl->GetName() == name)
					return true;
			}

		}
	}

	return false;
}

bool IsNum(char val)
{
	char num[] = "0123456789";

	for(int j=0;j<10;j++)
		if (val == num[j])
			return true;

	return false;
}

int SeperateNameNum(char* buf)
{
	int val;

	int len = strlen(buf);
	int pos = len;

	for(int i=len-1;i>=0;i--)
	{
		if (!IsNum(buf[i]))
			break;
	}

	if (i==-1)
		i = 0;

	val = atoi(&buf[i+1]);
	buf[i+1] = '\0';

	return val;
}

bool MatMergeList::MakeUniqueMtlName(CStr& name)
{
	bool bModified = false;
	int num;

	// Keep incrementing the name number until the material name is unique
	while (MtlInScene(name))
	{
		char buf[256];
		num = SeperateNameNum(name);
		
		if (num<lastNum)
			num = lastNum;
		
		sprintf(buf,"%i",num+1);
		name += CStr(buf);
		bModified = true;

		lastNum = num;
	}

	return bModified;
}
