/*
	LightmapGen.h
	This class is responsible for generating lightmaps
	aml - 9-18-02
*/

#ifndef __LIGHTMAPGEN__
#define __LIGHTMAPGEN__

//typedef        int int32;
//typedef signed int sint;

#include "../Export/SceneExport.h"
#include "max.h"
#include "../misc/llist.h"
#include "GLRenderer.h"

class ImageData;

enum lightType
{
	LIGHTTYPE_OMNI,
	LIGHTTYPE_DIRECTIONAL,
};

struct FaceObj
{
	NxFace*   face;    // The face being hit by the light
	NxObject* obj;	   // The object containing the face that gets hit by the light

	int operator == (FaceObj& right)
	{
		return (obj == right.obj && face == right.face);
	}
};

struct LightData
{
	Ray               ray;
	Color             color;	// The color of the light
	lightType         type;
	LinkList<FaceObj> pvs;		// Faces that are visible from the light's position
								// When raytracing we need to consider these to account
								// for obscured light and other effects
};

// This is derived from the OpenGL Renderer, as it uses GL to pre-render scene
// locations in order to build up PVS data for each light
class LightmapGen: public GLRenderer
{
	LinkList<LightData> lightList;
	FaceObj*            sceneFaces;				// List of pointer to all NxFaces in the scene
												// Corresponds to PVS indexes

	LinkList<Link<LightData>*>* faceLights;		// Contains references to the lights that affect the surface of
												// each face corresponding 1-to-1 with the sceneFaces list

	ImageData* imgMap;
	NxScene*   scene;
	int        curdir;							// The current direction that is being used to render PVS data
	int        curmap;							// The current lightmap index to be loaded

	ImageData* maps;							// Maintains all the generated lightmaps until saved

	int triMapWidth;
	int triMapHeight;

	int origMapWidth;
	int origMapHeight;

	float GetAng(Point3 v1, Point3 v2);
	bool  PtInLine(Point3 start, Point3 end, Point3 pt);
	bool  PtInTri(Point3 tri0, Point3 tri1, Point3 tri2, Point3 pt);

	bool  RayTriIntersect(Point3  tri0, Point3 tri1, Point3 tri2,
		                  Ray     ray,
					      Point3* intersect = NULL);

	bool  LineSegTriIntersect(Point3  tri0, Point3 tri1, Point3 tri2,
		                      Point3  segStart, Point3 segEnd,
							  Point3* intersect = NULL);

	float Dist(Point3 pt1, Point3 pt2);
	float Dist2(Point3 pt1, Point3 pt2);

	inline float Interp(float start, float end, float per)
	{
		return start + (end - start) * per;
	}

	inline float GetPer(float start, float end, float cur)
	{
		return (cur - start) / (end - start);
	}

	void ProcLight(NxScene* scene, INode* node);
	
	bool IsLit(Point3     worldPos,
			   LightData* ldata,
			   Point3     excludeTri0,
			   Point3     excludeTri1,
			   Point3     excludeTri2);

	static void glLightUpdate(GLRenderer* gl, void* pData);

	int NumSceneTris();
	int NumLightMaps();

public:
	LightmapGen();
	~LightmapGen();

	// This is the amount of space that a single triangles map
	// should take up (this should typically be very small) and
	// exist as
	void             SetTriMapSize(int width, int height);
	Link<LightData>* AddLight(Point3 pos, Point3 dir, Color color);
	Link<LightData>* AddLight(Point3 pos, Color color);
	void             ClearLights();
	void             GenerateLightmaps(NxScene* scene);

	// Triangle goes in UVs on the map for the object come out
	void LightTri(Point3     vert[3],
		          UVVert*    uv,
				  ImageData* img);

	void LightTri(Point3 vert[3],
						 UVCoord*   uv,
						 ImageData* img,
						 LinkList< Link<LightData>* >* lights );

	bool SaveLightmaps();
};

#endif
