/*
	ImageData.cpp
	Class for storing PNG format image data
*/

#include "ImageData.h"
#include "Targa.h"
#include "colors.h"
#include <stdlib.h>
#include <assert.h>
#include "BitArray.h"
#include <conio.h>

ImageData::ImageData()
{
	width            = 0;
	height           = 0;
	bit_depth        = 0;
	color_type       = 0;
	interlace_type   = 0;
	compression_type = 0;
	filter_type      = 0;
	pixel_depth      = 0;

	palette     = NULL;
	nPalEntries = 0;

	pals    = NULL;
	nPals   = 0;

	buf     = NULL;
	pRows   = NULL;

	tRNSpal     = NULL;
	tRNSentries = 0;
	
	tRNStrans_values.index = 0;
	tRNStrans_values.red   = 0;
	tRNStrans_values.green = 0;
	tRNStrans_values.blue  = 0;
	tRNStrans_values.gray  = 0;

	histogram = NULL;

	btRNStransValid = false;
}

ImageData::ImageData(ImageData& src)
{
	CopyNoImage(&src);

	buf = (unsigned char*)malloc(imageSize);
	memcpy(buf,src.buf,imageSize);
}

ImageData& ImageData::operator= (ImageData& src)
{
	CopyNoImage(&src);

	if (buf)
		free(buf);

	buf = (unsigned char*)malloc(imageSize);
	memcpy(buf,src.buf,imageSize);

	return *this;
}

void ImageData::CopyNoImage(ImageData* src)
{
	bit_depth        = src->bit_depth;
	color_type       = src->color_type;
	compression_type = src->compression_type;
	filter_type      = src->filter_type;
	height           = src->height;
	width            = src->width;
	nChannels        = src->nChannels;
	interlace_type   = src->interlace_type;
	imageSize        = src->imageSize;
	nPals            = src->nPals;
	tRNSentries      = src->tRNSentries;
	tRNStrans_values = src->tRNStrans_values;
	btRNStransValid  = src->btRNStransValid;
	pixel_depth      = src->pixel_depth;

	if (palette)
	{
		free(palette);
		palette = NULL;
		nPalEntries = 0;
	}

	if (buf)
	{
		free(buf);
		buf = NULL;
	}

	FreePals();

	if (tRNSpal)
	{
		free(tRNSpal);
		tRNSpal = NULL;
		tRNSentries = 0;
	}

	if (histogram)
		free(histogram);

	// Copy tRNS palette
	if (src->tRNSentries>0)
	{
		tRNSpal = (png_byte*)malloc(sizeof(png_byte)*src->tRNSentries);
		memcpy(tRNSpal,src->tRNSpal,sizeof(png_byte)*src->tRNSentries);
	}

	// If a palette exists, copy it
	if (src->palette)
	{
		nPalEntries = src->nPalEntries;
		
		if (src->histogram)
			histogram = (png_uint_16*)malloc(sizeof(png_uint_16)*nPalEntries);

		palette     = (png_color*)malloc(sizeof(png_color) * nPalEntries);

		for(int i=0;i<nPalEntries;i++)
		{
			palette[i]   = src->palette[i];

			if (src->histogram)
				histogram[i] = src->histogram[i];
		}
	}

	if (src->nPals > 0)
	{
		pals = (png_sPLT_tp)malloc(sizeof(png_sPLT_t)*nPals);

		for(int i=0;i<nPals;i++)
		{
			pals[i].depth    = src->pals[i].depth;
			pals[i].nentries = src->pals[i].nentries;

			pals[i].name = (char*)malloc(strlen(src->pals[i].name)+1);
			strcpy(pals[i].name,src->pals[i].name);
			
			// Copy all the palette entries
			for(int i=0;i<pals[i].nentries;i++)
				pals[i].entries[i] = src->pals[i].entries[i];
		}
	}
}

ImageData::~ImageData()
{
	if (pRows)
		free(pRows);

	if (buf)
		free(buf);

	if (palette)
		free(palette);

	if (histogram)
		free(histogram);

	if (pals)
		FreePals();
}

void ImageData::FreePals()
{
	if (pals)
	{
		for(int i=0;i<nPals;i++)
		{
			if (pals[i].name)
				free(pals[i].name);

			if (pals[i].nentries > 0)
				free(pals[i].entries);
		}

		nPals = 0;
	}
}

void ImageData::GetStdPal(png_structp png_ptr, png_infop info_ptr)
{
	png_colorp tmppal;
	png_get_PLTE(png_ptr,info_ptr,&tmppal,&nPalEntries);

	// Copy palette data
	if (palette)
		free(palette);

	palette = (png_colorp)malloc(sizeof(png_color)*nPalEntries);

	for(int i=0;i<nPalEntries;i++)
		palette[i] = tmppal[i];
}

void ImageData::GetPal(png_structp png_ptr, png_infop info_ptr)
{
	png_sPLT_tp ppal;
	nPals = png_get_sPLT(png_ptr,info_ptr,&ppal);

	if (nPals == 0)
	{
		FreePals();
		return;
	}

	pals = (png_sPLT_tp)malloc(sizeof(png_sPLT_t)*nPals);

	for(int i=0;i<nPals;i++)
	{
		pals[i].depth    = ppal->depth;
		pals[i].nentries = ppal->nentries;

		pals[i].name = (char*)malloc(strlen(ppal->name)+1);
		strcpy(pals[i].name,ppal->name);
		
		// Copy all the palette entries
		for(int i=0;i<pals[i].nentries;i++)
			pals[i].entries[i] = ppal->entries[i];
	}
}

bool ImageData::ReadPNG(char* filename,unsigned short* hist)
{
	FILE* fp = fopen(filename,"rb");

	if (!fp)
		return false;

	// Read in the PNG Sig to verify this is valid PNG file
	unsigned char IHDR[8];
	fread(IHDR,1,8,fp);

	// Have the PNG library verify the header information
	if (png_sig_cmp(IHDR,0,8))
	{
		fclose(fp);
		return false;
	}

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
		                                         NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);
	png_set_sig_bytes(png_ptr,8);

	// Do a simple read of the PNG file without doing any transforms on the image
	png_read_info(png_ptr, info_ptr);
	
	// Retrieve the header data
	png_get_IHDR(png_ptr,info_ptr,&width,&height,&bit_depth,&color_type,&interlace_type,&compression_type,&filter_type);

	nChannels = png_get_channels(png_ptr,info_ptr);

	pixel_depth = info_ptr->pixel_depth;

	GetStdPal(png_ptr,info_ptr);
	GetPal(png_ptr,info_ptr);

	// Allocate memory for each row of the png
	imageSize = (int)((float)(width)*(float)(height)*(((float)nChannels*(float)bit_depth)/8.0f));
	buf = (unsigned char*)malloc(imageSize);

	// Setup pointers to each row of the image
	unsigned char* pPos = buf;
	pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// libpng appers to store each pixel component R,G,B,A as a byte
	// well, not sure about A yet
	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += (int)((float)width*((float)nChannels*(float)bit_depth/8.0f));
	}

	/*
	png_bytep* pRowsRev = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}
	*/

	if (nPalEntries>0)
	{
		png_uint_16p tmphist = NULL;
		png_get_hIST(png_ptr,info_ptr,&tmphist);

		if (hist)
		{
			if (histogram)
				free(histogram);

			histogram = (png_uint_16*)malloc(sizeof(png_uint_16)*nPalEntries);

			for(int i=0;i<nPalEntries;i++)
				histogram[i] = hist[i];			
		}
		else if (tmphist)
		{
			if (histogram)
				free(histogram);

			histogram = (png_uint_16*)malloc(sizeof(png_uint_16)*nPalEntries);

			for(int i=0;i<nPalEntries;i++)
				histogram[i] = tmphist[i];
		}
	}

	// The dithering in the png library currently isn't sufficient for our purposes
	//png_set_dither(png_ptr,palette,nPalEntries,200,histogram,1);

	png_read_image(png_ptr,pRows);

	GetPal(png_ptr,info_ptr);

	// Read in tRNS data
	png_bytep      trans = NULL;
	int            numTrans = 0;
	png_color_16p  trans_values = NULL;

	png_get_tRNS(png_ptr,info_ptr,&trans,&numTrans,&trans_values);

	tRNSentries = numTrans;

	if (numTrans>0)
	{
		tRNSpal = (png_byte*)malloc(sizeof(png_byte)*numTrans);
		memcpy(tRNSpal,trans,sizeof(png_byte)*numTrans);
	}

	if (trans_values)
		tRNStrans_values = *trans_values;

	png_read_end(png_ptr,end_info);
	png_destroy_read_struct(&png_ptr,&info_ptr,&end_info);

	fclose(fp);

	return true;
}

bool ImageData::WritePNG(char* Filename)
{
	FILE* fp = fopen(Filename, "wb");

	if (!fp)
		return false;

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		                                          NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);

	// Color type modes
	//
	// PNG_COLOR_TYPE_GRAY
	// PNG_COLOR_TYPE_PALETTE
	// PNG_COLOR_TYPE_RGB
	// PNG_COLOR_TYPE_RGB_ALPHA
	// PNG_COLOR_TYPE_GRAY_ALPHA

	png_set_IHDR(png_ptr,info_ptr,
	             width,height,
				 bit_depth,
				 color_type,
				 interlace_type,
				 compression_type,
				 filter_type);

	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	if (nPalEntries > 0)
		png_set_PLTE(png_ptr,info_ptr,palette,nPalEntries);

	if (nPals > 0)
		png_set_sPLT(png_ptr,info_ptr,pals,nPals);

	// Build up row pointers
	png_bytep* pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);
	unsigned char* pPos = (unsigned char*)buf;

	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += (int)((float)width*(((float)nChannels*(float)bit_depth)/8.0f));
	}

	// Set tRNS data
	if (btRNStransValid)
		png_set_tRNS(png_ptr,info_ptr,tRNSpal,tRNSentries,&tRNStrans_values);
	else
		if (tRNSentries>0)
			png_set_tRNS(png_ptr,info_ptr,tRNSpal,tRNSentries,NULL);

	/*
	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}
	*/

	if (histogram)
		png_set_hIST(png_ptr,info_ptr,histogram);

	png_write_info(png_ptr,info_ptr);

	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	png_write_image(png_ptr,pRows);

	png_write_end(png_ptr,info_ptr);
	png_destroy_write_struct(&png_ptr,&info_ptr);

	free(pRows);

	fclose(fp);

	return true;
}

// Dumps out the image in Targa format for a quick test (me likes targa)
bool ImageData::DumpTGA(char* Filename)
{
	int depth = bit_depth * nChannels;

	// Not currently supporting palletized Targas
	if (depth == 8)
		return false;

	// Construct the Targa Header
	TGAHeader header;
	header.IDLength        = 0;						// No image ID field
	header.ColorMapType    = 0;						// Expecting only non-paletted data for now

	switch(depth)
	{
	case 8:
		header.ImageType   = TGATYPE_RAWPAL;
		break;

	default:
		header.ImageType   = TGATYPE_RAWTRUECOLOR;	// Only uncompressed true color images for now	
		break;
	}
	
	header.CMapStart       = 0;						// No palette
	header.CMapLength      = 0;						// No palette
	header.CMapDepth       = 0;						// No palette
	header.XOffset         = 0;
	header.YOffset         = 0;
	header.Width           = (unsigned short)width;
	header.Height          = (unsigned short)height;
	header.PixelDepth      = (unsigned char)depth;
	header.ImageDescriptor = 0;

	FILE* fp;

	fp = fopen(Filename,"wb");

	if (!fp)
		return false;

	// Write out the Targa Header
	fwrite(&header,sizeof(TGAHeader),1,fp);

	// Dump the raw image data
	int size = width * height;
	
	switch(depth)
	{
	case 8:
		{
			color8* pPos = (color8*)buf;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->index,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 16:
		{
			color16_555* pPos = (color16_555*)buf;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->color,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 24:
		{
			color24* pPos = (color24*)buf;
			
			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->b,sizeof(unsigned char),1,fp);
				fwrite(&pPos->g,sizeof(unsigned char),1,fp);
				fwrite(&pPos->r,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 32:
		{
			color32* pPos = (color32*)buf;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->b,sizeof(unsigned char),1,fp);
				fwrite(&pPos->g,sizeof(unsigned char),1,fp);
				fwrite(&pPos->r,sizeof(unsigned char),1,fp);
				fwrite(&pPos->a,sizeof(unsigned char),1,fp);
			}

			break;
		}
	}

	fclose(fp);
	return true;
}

color32 ImageData::GetColor32(unsigned int x, unsigned int y)
{
	// This needs to account for greyscale images with lower bitdepths
	// Determine the source format
	BArray ba(buf,pixel_depth);

	int idx = y*width+x;

	// Handle Palettized data
	if (nPalEntries > 0)
	{
		int index;
		ba.GetVal(&index,sizeof(int),idx);

		color32 outColor;
		outColor.r = palette[index].red;
		outColor.g = palette[index].green;
		outColor.b = palette[index].blue;

		if (tRNSentries > index)
			outColor.a = tRNSpal[index];

		return outColor;
	}
	else
	{
		
		BArray bac(buf,pixel_depth / nChannels);
		
		color32 outColor;
		bac.GetVal(&outColor.r,sizeof(outColor.r),idx * nChannels);
		bac.GetVal(&outColor.g,sizeof(outColor.g),idx * nChannels + 1);
		bac.GetVal(&outColor.b,sizeof(outColor.b),idx * nChannels + 2);

		if (nChannels > 3)
			bac.GetVal(&outColor.a,sizeof(outColor.a),idx * nChannels + 3);
		else
			outColor.a = 255;

		return outColor;
	}

/*
	// Determine the source format
	int depth = bit_depth * nChannels;

	switch(depth)
	{
	case 8:
		{
			// Palettized image
			color8* color = (color8*)buf;
			int     index = color[y*width+x].index;

			color32 outColor;
			outColor.r = palette[index].red;
			outColor.g = palette[index].green;
			outColor.b = palette[index].blue;
			outColor.a = 255;

			return outColor;
		}
		break;

	case 16:
		{
			color16_555* color   = (color16_555*)buf;
			color16_555  inColor = color[y*width+x];

			color32 outColor;
			outColor.r = inColor.R();
			outColor.g = inColor.G();
			outColor.b = inColor.B();
			outColor.a = 255;

			return outColor;
		}
		break;

	case 24:
		{
			color24* color   = (color24*)buf;
			color24  inColor = color[y*width+x];

			color32 outColor;
			outColor.r = inColor.r;
			outColor.g = inColor.g;
			outColor.b = inColor.b;
			outColor.a = 255;

			return outColor;
		}

	case 32:
		{
			color32* color = (color32*)buf;
			return color[y*width+x];
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);
*/
	color32 errColor;
	errColor.r = 0;
	errColor.g = 0;
	errColor.b = 0;
	errColor.a = 0;

	return errColor;
}

color32 ImageData::GetColor32(unsigned int idx)
{
	// This needs to account for greyscale images with lower bitdepths
	// Determine the source format
	BArray ba(buf,pixel_depth);

	// Handle Palettized data
	if (nPalEntries > 0)
	{
		int index;
		ba.GetVal(&index,sizeof(int),idx);

		color32 outColor;
		outColor.r = palette[index].red;
		outColor.g = palette[index].green;
		outColor.b = palette[index].blue;

		//printf("[%i,%i,%i, a=%i]\n",outColor.r,outColor.g,outColor.b,tRNSpal[index]);
		//while( getch() == 0 ) {}

		if (tRNSentries > index)
			outColor.a = tRNSpal[index];
		else
			outColor.a = 255;

		return outColor;
	}
	else
	{
		
		BArray bac(buf,pixel_depth / nChannels);
		
		color32 outColor;
		bac.GetVal(&outColor.r,sizeof(outColor.r),idx * nChannels);
		bac.GetVal(&outColor.g,sizeof(outColor.g),idx * nChannels + 1);
		bac.GetVal(&outColor.b,sizeof(outColor.b),idx * nChannels + 2);

		if (nChannels > 3)
			bac.GetVal(&outColor.a,sizeof(outColor.a),idx * nChannels + 3);
		else
			outColor.a = 255;

		return outColor;
	}
/*
	// Determine the source format
	int depth = bit_depth * nChannels;

	switch(depth)
	{
	case 8:
		{
			// Palettized image
			color8* color = (color8*)buf;
			int     index = color[idx].index;

			color32 outColor;
			outColor.r = palette[index].red;
			outColor.g = palette[index].green;
			outColor.b = palette[index].blue;
			
			if (index < tRNSentries)
				outColor.a = tRNSpal[index];
			else
				outColor.a = 255;

			return outColor;
		}
		break;

	case 16:
		{
			color16_555* color   = (color16_555*)buf;
			color16_555  inColor = color[idx];

			color32 outColor;
			outColor.r = inColor.R();
			outColor.g = inColor.G();
			outColor.b = inColor.B();
			outColor.a = 255;

			return outColor;
		}
		break;

	case 24:
		{
			color24* color   = (color24*)buf;
			color24  inColor = color[idx];

			color32 outColor;
			outColor.r = inColor.r;
			outColor.g = inColor.g;
			outColor.b = inColor.b;
			outColor.a = 255;

			return outColor;
		}

	case 32:
		{
			color32* color = (color32*)buf;
			return color[idx];
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);
*/
	color32 errColor;
	errColor.r = 0;
	errColor.g = 0;
	errColor.b = 0;
	errColor.a = 0;

	return errColor;
}

color24 ImageData::GetColor24(unsigned int x, unsigned int y)
{
	// This needs to account for greyscale images with lower bitdepths
	// Determine the source format
	BArray ba(buf,pixel_depth);

	int idx = y*width+x;

	// Handle Palettized data
	if (nPalEntries > 0)
	{
		int index;
		ba.GetVal(&index,sizeof(int),idx);

		color24 outColor;
		outColor.r = palette[index].red;
		outColor.g = palette[index].green;
		outColor.b = palette[index].blue;

		return outColor;
	}
	else
	{
		
		BArray bac(buf,pixel_depth / nChannels);
		
		color24 outColor;
		bac.GetVal(&outColor.r,sizeof(outColor.r),idx * nChannels);
		bac.GetVal(&outColor.g,sizeof(outColor.g),idx * nChannels + 1);
		bac.GetVal(&outColor.b,sizeof(outColor.b),idx * nChannels + 2);
		return outColor;
	}

/*
	int depth = bit_depth * nChannels;

	switch(depth)
	{
	case 8:
		{
			// Palettized image
			color8* color = (color8*)buf;
			int     index = color[y*width+x].index;

			color24 outColor;
			outColor.r = palette[index].red;
			outColor.g = palette[index].green;
			outColor.b = palette[index].blue;

			return outColor;
		}
		break;

	case 16:
		{
			color16_555* color   = (color16_555*)buf;
			color16_555  inColor = color[y*width+x];

			color24 outColor;
			outColor.r = inColor.R();
			outColor.g = inColor.G();
			outColor.b = inColor.B();

			return outColor;
		}
		break;

	case 24:
		{
			color24* color   = (color24*)buf;
			return color[y*width+x];
		}

	case 32:
		{
			color32* color   = (color32*)buf;
			color32  inColor = color[y*width+x];
			
			color24 outColor;
			outColor.r = inColor.r;
			outColor.g = inColor.g;
			outColor.b = inColor.b;
			
			return outColor;
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);
*/
	color24 errColor;
	errColor.r = 0;
	errColor.g = 0;
	errColor.b = 0;

	return errColor;
}

color24 ImageData::GetColor24(unsigned int idx)
{
	// This needs to account for greyscale images with lower bitdepths
	// Determine the source format

	// Handle Palettized data
	if (nPalEntries > 0)
	{
		int pdepth;

		if (pixel_depth > 8)
			pdepth = 8;
		else
			pdepth = pixel_depth;

		BArray ba(buf,pdepth);

		int index;
		ba.GetVal(&index,sizeof(int),idx);

		color24 outColor;
		outColor.r = palette[index].red;
		outColor.g = palette[index].green;
		outColor.b = palette[index].blue;

		return outColor;
	}
	else
	{
		
		BArray bac(buf,pixel_depth / nChannels);
		
		color24 outColor;
		bac.GetVal(&outColor.r,sizeof(outColor.r),idx * nChannels);
		bac.GetVal(&outColor.g,sizeof(outColor.g),idx * nChannels + 1);
		bac.GetVal(&outColor.b,sizeof(outColor.b),idx * nChannels + 2);
		return outColor;
	}
/*
	switch(depth)
	{
	case 8:
		{
			// Palettized image
			color8* color = (color8*)buf;
			int     index = color[idx].index;

			color24 outColor;
			outColor.r = palette[index].red;
			outColor.g = palette[index].green;
			outColor.b = palette[index].blue;

			return outColor;
		}
		break;

	case 16:
		{
			color16_555* color   = (color16_555*)buf;
			color16_555  inColor = color[idx];

			color24 outColor;
			outColor.r = inColor.R();
			outColor.g = inColor.G();
			outColor.b = inColor.B();

			return outColor;
		}
		break;

	case 24:
		{
			color24* color   = (color24*)buf;
			return color[idx];
		}

	case 32:
		{
			color32* color   = (color32*)buf;
			color32  inColor = color[idx];
			
			color24 outColor;
			outColor.r = inColor.r;
			outColor.g = inColor.g;
			outColor.b = inColor.b;
			
			return outColor;
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);
*/
	color24 errColor;
	errColor.r = 0;
	errColor.g = 0;
	errColor.b = 0;

	return errColor;
}

color16_555 ImageData::GetColor16(unsigned int x, unsigned int y)
{
	int depth = bit_depth * nChannels;

	switch(depth)
	{
	case 8:
		{
			color8*  color = (color8*)buf;
			color8 inColor = color[y*width+x];

			color16_555 outColor;
			outColor.R(palette[inColor.index].red);
			outColor.G(palette[inColor.index].green);
			outColor.B(palette[inColor.index].blue);

			return outColor;
		}

	case 16:
		{
			color16_555*   color = (color16_555*)buf;
			return color[y*width+x];
		}

	case 24:
		{
			color24*  color = (color24*)buf;
			color24 inColor = color[y*width+x];

			color16_555 outColor;
			outColor.R(inColor.r);
			outColor.G(inColor.g);
			outColor.B(inColor.b);

			return outColor;
		}

	case 32:
		{
			color32*  color = (color32*)buf;
			color32 inColor = color[y*width+x];

			color16_555 outColor;
			outColor.R(inColor.r);
			outColor.G(inColor.g);
			outColor.B(inColor.b);

			return outColor;
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);

	color16_555 errColor;
	errColor.R(0);
	errColor.G(0);
	errColor.B(0);

	return errColor;
}

color16_555 ImageData::GetColor16(unsigned int idx)
{
	int depth = bit_depth * nChannels;

	switch(depth)
	{
	case 8:
		{
			color8*  color = (color8*)buf;
			color8 inColor = color[idx];

			color16_555 outColor;
			outColor.R(palette[inColor.index].red);
			outColor.G(palette[inColor.index].green);
			outColor.B(palette[inColor.index].blue);

			return outColor;
		}

	case 16:
		{
			color16_555*   color = (color16_555*)buf;
			return color[idx];
		}

	case 24:
		{
			color24*  color = (color24*)buf;
			color24 inColor = color[idx];

			color16_555 outColor;
			outColor.R(inColor.r);
			outColor.G(inColor.g);
			outColor.B(inColor.b);

			return outColor;
		}

	case 32:
		{
			color32*  color = (color32*)buf;
			color32 inColor = color[idx];

			color16_555 outColor;
			outColor.R(inColor.r);
			outColor.G(inColor.g);
			outColor.B(inColor.b);

			return outColor;
		}
	}

	printf("ERROR!: Unknown format\n");
	assert(0);

	color16_555 errColor;
	errColor.R(0);
	errColor.G(0);
	errColor.B(0);

	return errColor;
}

varColor ImageData::GetColor(unsigned int x, unsigned int y, int depth)
{
	varColor color;

	switch(depth)
	{
	case 32:
		color.c32 = GetColor32(x,y);
		break;

	case 24:
		color.c24 = GetColor24(x,y);
		break;

	case 16:
		color.c16_555 = GetColor16(x,y);
		break;
	}

	return color;
}


varColor ImageData::GetColor(unsigned int idx, int depth)
{
	varColor color;

	switch(depth)
	{
	case 32:
		color.c32 = GetColor32(idx);
		break;

	case 24:
		color.c24 = GetColor24(idx);
		break;

	case 16:
		color.c16_555 = GetColor16(idx);
		break;
	}

	return color;
}

void ImageData::BuildHist()
{
	if (palette)
	{
		// Allocate memory for the histogram
		if (histogram)
			free(histogram);

		histogram = (png_uint_16*)calloc(sizeof(png_uint_16)*nPalEntries,1);

		color8* color = (color8*)buf;
		int size = width * height;

		for(int i=0;i<size;i++)
		{
			histogram[color->index]++;
			color++;
		}
	}
}

bool ImageData::SavePal(char* filename)
{
	FILE* fp = fopen(filename,"w");

	if (!fp)
		return false;

	fprintf(fp,"%i\n",nPalEntries);

	for(int i=0;i<nPalEntries;i++)
	{
		unsigned char alpha;

		if (i < tRNSentries)
			alpha = tRNSpal[i];
		else
			alpha = 255;

		fprintf(fp,"[%i]: %i,%i,%i,%i\n",i,palette[i].red,palette[i].green,palette[i].blue,alpha);
	}

	fclose(fp);
	return true;
}

bool ImageData::LoadPal(char* filename)
{
	FILE* fp = fopen(filename,"r");

	if (!fp)
		return false;

	// Free any current palette data from memory
	if (palette)
	{
		free(palette);
		palette = NULL;
		nPalEntries = 0;
	}

	if (tRNSpal)
	{
		free(tRNSpal);
		tRNSpal = NULL;
		tRNSentries = 0;
	}

	fscanf(fp,"%i\n",&nPalEntries);

	tRNSentries = nPalEntries;

	palette = (png_color*)malloc(sizeof(png_color)*nPalEntries);
	tRNSpal = (png_byte*)malloc(sizeof(png_byte)*nPalEntries);

	int  dummy;
	bool tRNSRequired = false;

	for(int i=0;i<nPalEntries;i++)
	{
		unsigned char alpha;
		fscanf(fp,"[%i]: %i,%i,%i,%i\n",&dummy,&palette[i].red,&palette[i].green,&palette[i].blue,&alpha);

		tRNSpal[i] = alpha;
		
		if (alpha != 255)
			tRNSRequired = true;
	}

	fclose(fp);

	if (!tRNSRequired)
	{
		free(tRNSpal);
		tRNSpal = NULL;
		tRNSentries = 0;
	}

	return true;
}

void ImageData::Convert32(ImageData* dest)
{
	int size = width * height;

	dest->CopyNoImage(this);
	dest->bit_depth = 8;
	dest->pixel_depth = 32;
	dest->nChannels = 4;	// RGBA
	
	dest->color_type = PNG_COLOR_TYPE_RGB_ALPHA;
	dest->imageSize = size * 4;

	// Allocate space for the resampled image
	if (dest->buf)
		free(dest->buf);

	dest->buf = (unsigned char*)malloc(dest->imageSize);

	// Resample image
	
	for(int i=0;i<size;i++)
		((color32*)dest->buf)[i] = GetColor32(i);
}

void ImageData::Convert24(ImageData* dest)
{
	int size = width * height;

	dest->CopyNoImage(this);
	dest->bit_depth = 8;
	dest->pixel_depth = 24;
	dest->nChannels = 3;	// RGB

	dest->color_type = PNG_COLOR_TYPE_RGB;
	dest->imageSize = size * 3;

	// Allocate space for the resampled image
	if (dest->buf)
		free(dest->buf);

	dest->buf = (unsigned char*)malloc(dest->imageSize);

	// Resample image
	for(int i=0;i<size;i++)
		((color24*)dest->buf)[i] = GetColor24(i);
}

void ImageData::Convert16(ImageData* dest)
{
	int size = width * height;

	dest->CopyNoImage(this);
	dest->bit_depth = 8;
	dest->pixel_depth = 16;
	dest->nChannels = 3;	// RGB
	
	dest->color_type = PNG_COLOR_TYPE_RGB;
	dest->imageSize = size * 2;

	// Allocate space for the resampled image
	if (dest->buf)
		free(dest->buf);

	dest->buf = (unsigned char*)malloc(dest->imageSize);

	// Resample image
	for(int i=0;i<size;i++)
		((color16_555*)dest->buf)[i] = GetColor16(i);
}
