/*
	GLRenderer.cpp
	A Quick OpenGL renderer for displaying scene previews
	(or Lightmapped geometry for now)
*/

#include "GLRenderer.h"
#include "max.h"

#include <gl/gl.h>
#include <gl/glu.h>
#include <gl/glext.h>
#include "ImageData.h"

extern HINSTANCE  hInstance;
extern Interface* gInterface;

GLRenderer::GLRenderer()
{
	hwnd      = NULL;
	hglrc     = NULL;
	hdc       = NULL;
	fpUpdate  = NULL;
	
	bMultiTextureSupport = false;

	lastTexID = 1;
}

GLRenderer::~GLRenderer()
{
	if (hwnd && hdc)
	{
		ReleaseDC(hwnd, hdc);
		DestroyWindow(hwnd);
	}
}

void GLRenderer::DrawModel(Mesh*  mesh,
		                   Point3 pos)
{
	// No optimizations for now
	Point3* verts = mesh->verts;
	Face*   faces = mesh->faces;
	TVFace* tvFaces = mesh->tvFace;
	UVVert* tVerts = mesh->tVerts;

	glMatrixMode(GL_MODELVIEW);

	for(int i = 0; i < mesh->numFaces; i++)
	{
		glBegin(GL_TRIANGLES);

		glTexCoord2f(-tVerts[tvFaces[i].t[0]].x,
				     -tVerts[tvFaces[i].t[0]].y);

		glVertex3f(verts[faces[i].v[0]].x,
				   verts[faces[i].v[0]].z,
				   verts[faces[i].v[0]].y);

		glTexCoord2f(-tVerts[tvFaces[i].t[1]].x,
				     -tVerts[tvFaces[i].t[1]].y);

		glVertex3f(verts[faces[i].v[1]].x,
				   verts[faces[i].v[1]].z,
				   verts[faces[i].v[1]].y);

		glTexCoord2f(-tVerts[tvFaces[i].t[2]].x,
				     -tVerts[tvFaces[i].t[2]].y);

		glVertex3f(verts[faces[i].v[2]].x,
				   verts[faces[i].v[2]].z,
				   verts[faces[i].v[2]].y);

		glEnd();
	}	
}

void GLRenderer::Update()
{
	wglMakeCurrent(hdc, hglrc);

	// Initialize OpenGL extensions
	glActiveTextureARB = (PFNGLACTIVETEXTUREARBPROC)wglGetProcAddress("glActiveTextureARB");
	glMultiTexCoord2fARB = (PFNGLMULTITEXCOORD2FARBPROC)wglGetProcAddress("glMultiTexCoord2fARB");

	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	gluPerspective(30.0, 1.0, 1.0, 5000.0);

	glEnable(GL_DEPTH_TEST);
	glEnable(GL_TEXTURE_2D);
	glEnable(GL_BLEND);
	glDepthFunc(GL_LESS);

	// Enable multitexturing if the video card supports it
	const unsigned char* ext = glGetString(GL_EXTENSIONS);

	if (strstr((const char*)ext, "GL_ARB_multitexture"))
	{
		int nTexUnits;

		glGetIntegerv(GL_MAX_TEXTURE_UNITS_ARB, &nTexUnits);

		if (nTexUnits > 1)
		{
			glActiveTextureARB(GL_TEXTURE0_ARB);
			glEnable(GL_TEXTURE_2D);
			glActiveTextureARB(GL_TEXTURE1_ARB);
			glEnable(GL_TEXTURE_2D);

			bMultiTextureSupport = true;
		}
	}

	gluLookAt(0.0f,15.0f,-70.0f,
			  0.0f,15.0f,1.0f,
			  0.0f,1.0f,0.0f);

	glViewport(0,
			   0,
			   640,//rect.right - rect.left,
			   480);//rect.bottom - rect.top);

	glClearColor(0.0f,0.0f,0.0f,1.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	glColor3f(1.0f,1.0f,1.0f);
	glMatrixMode(GL_MODELVIEW);

	if (fpUpdate)
		fpUpdate(this, pUpdateData);

	SwapBuffers(hdc);

	wglMakeCurrent(NULL,NULL);	
}

void GLRenderer::SetCallback(void (*fp)(GLRenderer*,void*), void* pData)
{
	fpUpdate = fp;
	pUpdateData = pData;
}

LRESULT CALLBACK GLRenderer::GLWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	GLRenderer* pthis = (GLRenderer*)GetWindowLong(hwnd, GWL_USERDATA);

	switch(msg)
	{
	case WM_PAINT:
		{
			pthis->Update();
		}
		return TRUE;

	case WM_RBUTTONDOWN:
		pthis->Update();
		return TRUE;

	case WM_LBUTTONDOWN:
		pthis->Update();
		//DestroyWindow(hwnd);
		return TRUE;

	case WM_CREATE:

		return TRUE;
	}

	return DefWindowProc(hwnd,msg,wParam,lParam);
}

HWND GLRenderer::InitWindow()
{
	// Creates a preview window for the pieces and sets it up for OpenGL
	WNDCLASS wndclass;
	ZeroMemory(&wndclass,sizeof(WNDCLASS));

	wndclass.style         = CS_HREDRAW | CS_VREDRAW | CS_OWNDC;
	wndclass.lpfnWndProc   = GLWndProc;
	wndclass.cbClsExtra    = 0;
	wndclass.cbWndExtra    = 0;
	wndclass.hInstance     = hInstance;
	wndclass.hIcon         = NULL;
	wndclass.hCursor       = LoadCursor(NULL,IDC_ARROW);
	wndclass.hbrBackground = (HBRUSH)GetStockObject(WHITE_BRUSH);
	wndclass.lpszMenuName  = NULL;
	wndclass.lpszClassName = "OpenGLContainer";

	RegisterClass(&wndclass);

	hwnd = CreateWindow("OpenGLContainer",
	                    "Part Preview",
					    WS_OVERLAPPEDWINDOW,
						0,
						0,
						640,
						480,
						gInterface->GetMAXHWnd(),
						NULL,
						hInstance,
						NULL);

	GetWindowRect(hwnd, &rect);
	SetWindowLong(hwnd, GWL_USERDATA, (LONG)this);

	hdc = GetDC(hwnd);
	InitContext(hdc);

	//InvalidateRect(hwnd,NULL,TRUE);
	Update();
	return hwnd;
}

int GLRenderer::UploadTexture(ImageData* img)
{
	ImageData img32;
	img->Convert32(&img32);

	int ID = -1;

	int width  = img32.width;
	int height = img32.height;

	wglMakeCurrent(hdc, hglrc);

	glBindTexture(GL_TEXTURE_2D, lastTexID);

	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_REPEAT);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);
	//glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_NEAREST);
	glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_NEAREST);

	glPixelStorei(GL_UNPACK_ROW_LENGTH,0);
	glPixelStorei(GL_UNPACK_SKIP_ROWS,0);
	glPixelStorei(GL_UNPACK_SKIP_PIXELS,0);

	glPixelStorei(GL_UNPACK_ALIGNMENT,1);

	glTexImage2D(GL_TEXTURE_2D,
					0,					// Mip level 0
					4,					// Number of color components RBGA (4)
					width,
					height,
					0,					// No border
					GL_RGBA,
					GL_UNSIGNED_BYTE,
					img32.buf);

	ID = lastTexID;
	lastTexID++;

	wglMakeCurrent(NULL, NULL);

	return ID;
}

HDC GLRenderer::InitContext(HDC context)
{
	if (!context)
	{
		hdc = CreateDC("DISPLAY",
					   NULL,
					   NULL,
					   NULL);
	}
	else
	{
		hdc = context;
	}

	PIXELFORMATDESCRIPTOR pfd;
	ZeroMemory(&pfd, sizeof(PIXELFORMATDESCRIPTOR));

	pfd.nSize    = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags  = PFD_DOUBLEBUFFER   |
		           PFD_SUPPORT_OPENGL |
				   PFD_DRAW_TO_WINDOW;

	pfd.iPixelType = PFD_TYPE_RGBA;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;
	pfd.iLayerType = PFD_MAIN_PLANE;

	int nPixelFormat = ChoosePixelFormat(hdc, &pfd);
	
	if (nPixelFormat == 0)
		return false;

	if (!SetPixelFormat(hdc, nPixelFormat, &pfd))
		return false;

	hglrc = wglCreateContext(hdc);

	return hdc;
}

void GLRenderer::ShowWindow()
{
	::ShowWindow(hwnd, SW_SHOW);
}

void GLRenderer::MakeCurrent()
{
	wglMakeCurrent(hdc, hglrc);
}

void GLRenderer::EndCurrent()
{
	wglMakeCurrent(NULL, NULL);
}
