#include <next.h>
#include <Export/Export.h>
#include <Export/ExportOptions.h>
#include <Export/SceneExport.h>
#include <Export/SceneExportOptions.h>
#include <Export/TextureExporter.h>
#include <Texture/NExtTexture.h>
#include <Image/Image.h>
#include <Misc/GenCrc.h>
#include <Misc/Util.h>
#include "texture/bmtex.h"
#include "../../genlib/utility.h"
#include <process.h>

#include <fstream.h>
#include <stdmat.h>
#include "../PropEdit/ParseFuncs.h"
#include "../UI/OKtoAll.h"

extern int CalcNumMips(int width, int height, int minWidth = 1, int minHeight = 1);

class TextureExporter : public ITextureExporter
{
	bool            bWarn;
	bool			m_dirty;
	bool            m_reqInitalExport;

	CStr            export_name;			// Name of the last texture dictionary exported
											// If the name changed then the dictionary will need exported again

public:
	TextureExporter() { bWarn = true; m_last_num_exported_textures = 0; m_dirty = true; m_reqInitalExport = false; }

	void			Reset( void );	
	bool			AddTexture( Texmap* map, int flags, unsigned long checksum_array[vNUM_PLATFORMS],
									bool allow_mips );
	bool			LoadTextureData( void );
	bool            VerifyTexturePaths( char* path );

	bool			SaveTextureDictionary( char* dict_path, 
		                                   char* usg_path,
										   IoUtils::CVirtualOutputFile* udef_file     = NULL, 
										   IoUtils::CVirtualOutputFile* udef_usg_file = NULL);

	NxTexture*		GetTextureByChecksum( unsigned long checksum );	
	bool			AllTexturesAreValid( void );
	
	void			OnPostOpen( void );

	void			LoadTextureDictionary( char* path );
	
	NxTexture*		GetTextureFromDatabase( NxTexture* src_tex );
	bool			IsSameTexture( NxTexture* src, NxTexture* dst );

	void			MarkAsClean( void );
	void			MarkAsDirty( void );

	void            ResetInitialExport( void );

	bool			ShouldExportTextureDictionary( char* expName );

	Tab< NxTexture* >	m_Textures;	
	Tab< NxTexture* >	m_TextureDb;
	Tab< NxTexture* >	m_ExportTextures;
	

private:

	unsigned long	add_texture( INExtTexture* tex, int platform, int flags, int plat_use_flags,
									bool allow_mips );

	char			m_tex_buff[1024 * 1024 * 4];
	int				m_last_num_exported_textures;

};


static TextureExporter	s_texture_exporter;

ITextureExporter*	GetTextureExporter( void )
{
	return &s_texture_exporter;
}


DWORD WINAPI TextureExportProgressFunc(LPVOID arg) 
{
    return(0);
}

void	TextureExporter::Reset( void )
{
	
	//int i;

	/*for( i = 0; i < m_Textures.Count(); i++ )
	{
		delete m_Textures[i];
	}*/
	
	m_Textures.Delete(0,m_Textures.Count());	// aml: Tab class has ownership, it has to do the delete
	m_Textures.ZeroCount();
	
	m_ExportTextures.Delete(0,m_Textures.Count());	// aml: Tab class has ownership, it has to do the delete
	m_ExportTextures.ZeroCount();

	m_WarnedAboutDupeMaterials = false;
	bWarn = true;

	MessageBoxResetAll();
}

int	get_max_mip_level( BitmapInfo* bitmap )
{
	int x_power, y_power, larger, smaller;

	if( !IsPowerOfTwo( bitmap->Width()) ||
		!IsPowerOfTwo( bitmap->Height()))
	{
		return 0;
	}

	x_power = GetLog2( bitmap->Width());
	y_power = GetLog2( bitmap->Height());
	larger = ( x_power < y_power ) ? y_power : x_power;
	smaller = ( x_power < y_power ) ? x_power : y_power;
	
	if( smaller >= 2 )
	{
		return ( smaller - 2 );
	}
	else
	{
		return 0;
	}
}

unsigned long	TextureExporter::add_texture( INExtTexture* tex, int platform, int flags, int plat_use_flags,
												bool allow_mips )
{
	NxTexture* new_tex;
	BitmapInfo* map;
	int i, num_mip_levels;

	new_tex = new NxTexture;

	map = tex->GetBaseMap( platform, allow_mips );

	// set to generated textures if appropriate
	if ( map->Flags() & MAP_IS_GENERATED )
	{
		flags |= NxTexture::mPOST_GENERATED_TEXTURE;
	}

	if( map == NULL )
	{
		delete new_tex;
		return 0;
	}
	if( !allow_mips || ( tex->GetMipType( platform ) == vMIP_TYPE_NONE ))
	{
		num_mip_levels = 0;
	}
	else if( tex->GetMipType( platform ) == vMIP_TYPE_AUTO )
	{
		num_mip_levels = get_max_mip_level( map );
		flags |= NxTexture::mAUTO_GENERATE_MIPMAPS;
	}
	else
	{
		int j;

		num_mip_levels = tex->GetNumMipLevels( platform );
		for( j = 1; j <= num_mip_levels; j++ )
		{
			BitmapInfo* mip_map;

			mip_map = tex->GetMipMap( platform, j );
			if( mip_map == NULL )
			{
				// This shouldn't happen. If so, then it means we don't actually have as many levels as we thought
				// we did. In that case, adjust the number of levels accordingly
				num_mip_levels = j - 1;
				break;
			}

			new_tex->SetName((char*) mip_map->Name(), j );
		}		
	}	

	// Assign the UseBasePal flags
	for( i = 0; i<num_mip_levels; i++)
		new_tex->SetUseBasePal(i, tex->UseBasePal(platform,i));


	if( tex->ChangeTransColor())
	{
		flags |= NxTexture::mCHANGE_FULLY_TRANSPARENT_COLOR;
	}

	if( tex->Force24BitPalette())
	{
		flags |= NxTexture::mFORCE_BYTE_PER_COMPONENT;
	}

	if( tex->ShouldCompress( vPLAT_NGC ))
	{
		flags |= NxTexture::mCOMPRESS_NGC;
	}
	if( tex->ShouldCompress( vPLAT_XBOX ))
	{
		flags |= NxTexture::mCOMPRESS_XBOX;
	}

	new_tex->SetName((char*) map->Name(), 0);	
	new_tex->SetNumMipLevels( num_mip_levels );
	new_tex->SetMap( map );	
	new_tex->SetFlags( flags );
	new_tex->SetTransparentColor( tex->GetTransColor());
	new_tex->SetPlatformFlags( plat_use_flags );
	new_tex->GenerateChecksum();	

	// First check to see if we already have this texture
	for( i = 0; i < m_Textures.Count(); i++ )
	{
		if( *m_Textures[i] == *new_tex )
		{
			// If the new texture has more mip levels than the previous texture with the same base name
			// is the new one instead
			if( m_Textures[i]->GetNumMipLevels() < num_mip_levels )
			{
				char msg[1024];

				sprintf( msg, "Two NxTextures share the same base texture %s but have different mipmaps.  Using the one with more mipmap levels defined", new_tex->GetName( 0 ));
				// Temporarily disabled for THPS4
				//MessageBox( gInterface->GetMAXHWnd(), msg, "Warning!", MB_OK );
				m_Textures.Append( 1, &new_tex );
				m_Textures.Delete( i, 1 );
				return new_tex->GetChecksum();
			}
			else
			{			
				// The final flags should be a combination of all of the operations the user
				// wanted to perform on this texture
				m_Textures[i]->SetFlags( flags );
				delete new_tex;
				return m_Textures[i]->GetChecksum();
			}
		}
	}	
	
	m_Textures.Append( 1, &new_tex );
	return new_tex->GetChecksum();
}

bool	platform_override( INExtTexture* tex, int src_plat, int dst_plat, bool allow_mips )
{
	BitmapInfo* bm_src, *bm_dst;

	bm_src = tex->GetBaseMap( src_plat, allow_mips );
	assert( bm_src );
	bm_dst = tex->GetBaseMap( dst_plat, allow_mips );
	
	// If there's a different base texture defined, the dst platform is overriding for sure
	if( bm_dst && ( stricmp( bm_src->Name(), bm_dst->Name()) != 0 ))
	{
		return true;
	}

	// If we're not allowing mips for this image, then no need to check if the
	// mips are the same
	if( !allow_mips )
	{
		return false;
	}

	// Even with the same base texture, the dst platform may want to override mip maps
	if(	tex->GetMipType( src_plat ) != tex->GetMipType( dst_plat ))
	{
		return true;
	}

	if(	src_plat == vPLAT_PS2 )
	{
		// At this point the only way they could be the same is if the source is using manual mips and the
		// dst is sharing them
		if( tex->GetMipType( src_plat ) == vMIP_TYPE_MANUAL )
		{
			return ( tex->UsePS2Mips( dst_plat ) == false );		
		}	
	}

	return false;
}

bool	TextureExporter::AddTexture( Texmap* map, int flags, unsigned long checksum_array[vNUM_PLATFORMS],
										bool allow_mips )
{
	BitmapInfo* bitmap_ps2, *bitmap_ngc, *bitmap_xbox;
	int plat_use_flags;

	if( map == NULL )
	{
		return false;
	}
	
	if( map->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		return false;
	}

	INExtTexture* next_tex = dynamic_cast< INExtTexture* >( map );		

	// For now force all NGC and XBOX to use 32-bit RGBA regardless
	bitmap_ps2 = next_tex->GetBaseMap( vPLAT_PS2, allow_mips );

	if (bitmap_ps2 == NULL)
		return false;

	// Don't add the map if there's no defined device (Bad/Empty map?)
	if (strlen(bitmap_ps2->Device()) == 0)
		return false;

	char bufDrive[_MAX_DRIVE];
	char bufPath[_MAX_DIR];
	char bufFilename[_MAX_FNAME];
	char bufExt[_MAX_EXT];

	_splitpath((char*)bitmap_ps2->Name(),bufDrive,bufPath,bufFilename,bufExt);


	TSTR strOverride = TSTR(bufDrive) + TSTR(bufPath) + TSTR("mips\\") + TSTR(bufFilename);
	strOverride += "_auto32m0.png";

	// Only one texture should be assigned so texture usage isn't duplicated
	// (If XBOX and NGC are both assigned auto they will get the texture object
	//  for the first type used, should be NGC, if they are assigned none or
	//  manual the override does not apply and processing should be done as it
	//  always has been (Manual mips/ or none))
	if( allow_mips )
	{
		if (next_tex->GetMipType( vPLAT_NGC ) == vMIP_TYPE_AUTO)
		{
			next_tex->AssignOverrideTexName( vPLAT_NGC, strOverride );
			flags |= NxTexture::mAUTO_GENERATE_NGC_MIPS;
		}
		else if (next_tex->GetMipType( vPLAT_XBOX ) == vMIP_TYPE_AUTO)
		{
			next_tex->AssignOverrideTexName( vPLAT_XBOX, strOverride );
			flags |= NxTexture::mAUTO_GENERATE_XBOX_MIPS;
		}
	}
	

	bitmap_ngc = next_tex->GetBaseMap( vPLAT_NGC, allow_mips );
	bitmap_xbox = next_tex->GetBaseMap( vPLAT_XBOX, allow_mips );

	if( bitmap_ps2 == NULL )
	{
		return false;
	}

	plat_use_flags = mPLAT_PS2;
	if( platform_override( next_tex, vPLAT_PS2, vPLAT_NGC, allow_mips ) == false )
	{
		plat_use_flags |= mPLAT_NGC;
	}
	if( platform_override( next_tex, vPLAT_PS2, vPLAT_XBOX, allow_mips ) == false )
	{
		plat_use_flags |= mPLAT_XBOX;
	}

	checksum_array[vPLAT_PS2] = add_texture( next_tex, vPLAT_PS2, flags, plat_use_flags, allow_mips );

	// If neither NGC nor Xbox share this texture, check if THEY'RE using the same
	// texture as eachother
	if(	(	( plat_use_flags & ( mPLAT_NGC | mPLAT_XBOX )) == 0 ) &&
		( platform_override( next_tex, vPLAT_XBOX, vPLAT_NGC, allow_mips ) == false ))
	{
		
		checksum_array[vPLAT_NGC] = add_texture( next_tex, vPLAT_NGC, flags, mPLAT_NGC | mPLAT_XBOX, allow_mips );		
		checksum_array[vPLAT_XBOX] = checksum_array[vPLAT_NGC];
	}
	else
	{			
		if( platform_override( next_tex, vPLAT_PS2, vPLAT_NGC, allow_mips ))
		{
			checksum_array[vPLAT_NGC] = add_texture( next_tex, vPLAT_NGC, flags, mPLAT_NGC, allow_mips );
		}
		else
		{
			checksum_array[vPLAT_NGC] = checksum_array[vPLAT_PS2];		
		}

		if( platform_override( next_tex, vPLAT_PS2, vPLAT_XBOX, allow_mips ))
		{
			checksum_array[vPLAT_XBOX] = add_texture( next_tex, vPLAT_XBOX, flags, mPLAT_XBOX, allow_mips );
		}
		else
		{
			checksum_array[vPLAT_XBOX] = checksum_array[vPLAT_PS2];		
		}
	}

	return true;
}

bool TextureExporter::LoadTextureData( void )
{	
	int i, j, num_textures, export_pass;
	unsigned long checksum;
	Tab< NxTexture* >	m_RedundantTextures;	
	
	// Ok, we need to export the normal textures, followed by the
	// mPOST_GENERATED_TEXTURE flag

	num_textures = m_Textures.Count();

	gInterface->ProgressStart(_T("Loading Texture Data"), TRUE, 
										TextureExportProgressFunc, NULL );

	MarkAsClean();

	char buf[256];
	sprintf(buf, "nTextures: %i\n", num_textures);
	OutputDebugString(buf);

	for( i = 0; i < num_textures; i++ )
	{
		NxTexture* texture, *db_tex;
		BitmapInfo* bitmap;	
		int width, height;
		bool bExportPostGenMips;
		WIN32_FIND_DATA find_data;
		HANDLE file_handle;

		texture = m_Textures[i];
	
		sprintf(buf, "Processing texture %s (%i)...\n", (char*)texture->GetName(0), i);
		OutputDebugString(buf);
		
		bitmap = texture->GetMap();
		assert( bitmap );

		width = bitmap->Width();
		height = bitmap->Height();
		
		file_handle = FindFirstFile( texture->GetName( 0 ), &find_data );  
		if( file_handle != INVALID_HANDLE_VALUE )
		{
			texture->SetFileTime( 0, find_data.ftLastWriteTime );
			FindClose( file_handle );
		}		

		if( texture->ShouldAutoGenerateMipMaps() == false )
		{
			for( j = 1; j <= texture->GetNumMipLevels(); j++ )
			{
				file_handle = FindFirstFile( texture->GetName( j ), &find_data );  
				if( file_handle != INVALID_HANDLE_VALUE )
				{
					texture->SetFileTime( j, find_data.ftLastWriteTime );
					FindClose( file_handle );
				}		
			}

			// Determine the base filename
			/*char ult_mip_path[_MAX_PATH];
			char mip_path[_MAX_PATH];
			char drive[_MAX_DRIVE];
			char cur_path[_MAX_PATH];
			char cur_file[_MAX_FNAME];			
			char ext[_MAX_EXT];

			_splitpath( texture->GetName( 0 ), drive, cur_path, cur_file, ext );
			sprintf( ult_mip_path, "%s%smips", drive, cur_path );

			if( texture->GetFlags() & NxTexture::mPOST_GENERATED_TEXTURE )
			{
				int snamelen = strlen(cur_file);
				cur_file[snamelen - 1] = 0;

				num_mip_levels = CalcNumMips( width, height, 1, 1);
				texture->SetNumMipLevels( num_mip_levels );

				for( j = 1; j <= texture->GetNumMipLevels(); j++ )
				{
					char bufLevel[20];
					sprintf(bufLevel,"%d",j);

					CStr filename = CStr(drive) + CStr(cur_path) + cur_file + CStr(bufLevel) + ext;
					texture->SetName( filename, j );
				}			
				
			}
			else
			{
				for( j = 1; j <= texture->GetNumMipLevels(); j++ )
				{
					sprintf( mip_path, "%s\\%s_autom%d%s", ult_mip_path, cur_file, j, ext );
					texture->SetName( mip_path, j );
				}			
			}*/
		}		

		if(( db_tex = GetTextureFromDatabase( texture )))
		{
			m_ExportTextures.Append( 1, &db_tex );
			m_RedundantTextures.Append( 1, &texture ); // Don't need this anymore since we already have a fully-loaded equivalent			
			continue;
		}

		MarkAsDirty();
		for( export_pass = 0; export_pass < 2 ; export_pass++)
		{
			if (export_pass == 0 && !(texture->GetFlags() & NxTexture::mPOST_GENERATED_TEXTURE) ||
				export_pass == 1 && (texture->GetFlags() & NxTexture::mPOST_GENERATED_TEXTURE))
			{
				checksum = texture->GetChecksum();

				// If this is pass 2 we won't have width and height yet
				if (export_pass == 0)
				{					
					if( !IsPowerOfTwo( width ) ||
						!IsPowerOfTwo( height ))
					{
						char error_msg[128];
						
						sprintf( error_msg, "Texture %s has a dimensions %d %d. They must be powers of two.", texture->GetName( 0 ),
									width, height );				
						MessageBoxAll( gInterface->GetMAXHWnd(), error_msg, "Texture Dimension Error!", MB_OK );		
						texture->SetValidity( false );
						ReportWarning( error_msg );
						continue;
					}

					texture->SetWidth( 0, width );
					texture->SetHeight( 0, height );
					
					bExportPostGenMips = true;
				}
				else
				{
					bExportPostGenMips = false;
				}

				if( texture->LoadImage( bExportPostGenMips ))
				{
					switch( texture->GetPixelFormat())
					{
						case NxTexture::v32_BIT:						
						case NxTexture::v24_BIT:						
						case NxTexture::v16_BIT:						
						case NxTexture::v8_BIT:						
						case NxTexture::v4_BIT:
							break;
						default:
							texture->SetValidity( false );
							continue;
					}

					switch( texture->GetPaletteFormat())
					{
						case NxTexture::v32_BIT:
						case NxTexture::v16_BIT:
						case NxTexture::v24_BIT:
							break;
						default:
							texture->SetValidity( false );
							continue;
					}
				}
				else
				{
					texture->SetValidity( false );
				}

				if( gInterface->GetCancel())
				{
					gInterface->ProgressEnd();
					return false;
				}
			}			
		}

		m_ExportTextures.Append( 1, &texture );			
		//m_TextureDb.Append( 1, &texture );

		gInterface->ProgressUpdate( i * 100 / num_textures );
	}

	OutputDebugString("PreProgressEnd\n");

	gInterface->ProgressEnd();

	OutputDebugString("PreTexDelete\n");

	for( i = 0; i < m_RedundantTextures.Count(); i++ )
	{
		delete m_RedundantTextures[i];
	}

	OutputDebugString("Delete 2\n");

	m_Textures.Delete(0,m_Textures.Count());	// aml: Tab class has ownership, it has to do the delete
	m_Textures.ZeroCount();

	m_Textures = m_ExportTextures;

	OutputDebugString("Pre return\n");

	return true;
}

bool TextureExporter::AllTexturesAreValid( void )
{
	int i, num_textures;
	NxTexture* texture;

	num_textures = m_ExportTextures.Count();
	for( i = 0; i < num_textures; i++ )
	{		
		texture = m_ExportTextures[i];
		if( texture->IsValid() == false )
		{
			return false;
		}
	}

	return true;
}

bool TextureExporter::SaveTextureDictionary( char* dict_path, 
											 char* usg_path, 
											 IoUtils::CVirtualOutputFile* udef_file, 
											 IoUtils::CVirtualOutputFile* udef_usg_file)
{
	//fstream file, usg_file;
	IoUtils::CVirtualOutputFile *file, *usg_file;
	int i, j, num_textures, version;
	unsigned long checksum;
	DWORD attribs;
	bool all_valid;
	bool noUSG;

	if (!usg_path && !udef_usg_file)
		noUSG = true;
	else
		noUSG = false;
	
	if( ShouldExportTextureDictionary(dict_path) == false )
	{
		return true;
	}

	export_name = dict_path;

	attribs = GetFileAttributes( dict_path );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			int result;
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", dict_path );
			result = MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( dict_path, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = dict_path;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );			
				process_result = ExecuteCommand( args );
				attribs = GetFileAttributes( dict_path );
				if(( attribs & FILE_ATTRIBUTE_READONLY ) || ( attribs == -1 ))
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Textures",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}

	// Do the same check for the usage file
	attribs = GetFileAttributes( usg_path );
	if( attribs != -1 )
	{
		if( attribs & FILE_ATTRIBUTE_READONLY )
		{
			char message[256];
			int result;
			
			sprintf( message, "You are about to save over a Read-Only file, %s, perhaps you should check-out the file first", usg_path );
			result = MessageBoxAll( gInterface->GetMAXHWnd(), message, "Read-only Warning", MB_CHECKOUT_UNLOCK );
			if( result == IDWRITABLE )
			{
				// break the lock, if necessary
				SetFileAttributes( usg_path, FILE_ATTRIBUTE_NORMAL );
			}
			else if( result == IDCHECKOUT )
			{
				char *args[4];
				int process_result;

				args[0] = "p4";
				args[1] = "edit";
				args[2] = usg_path;
				args[3] = NULL;
				//process_result = spawnvp( _P_WAIT, args[0], args );
				process_result = ExecuteCommand( args );
				attribs = GetFileAttributes( usg_path );
				if(( attribs & FILE_ATTRIBUTE_READONLY ) || ( attribs == -1 ))
				{
					MessageBox(NULL,"An error occurred while trying to check the file out. Perhaps someone else has it checked out already?","Export Textures",MB_ICONSTOP|MB_OK);
					return false;
				}
			}
			else
			{
				return false;
			}
		}
	}
	
	// Determine if we should create our own virtual file or use
	// a virtual file provided to us by the caller
	if (udef_file)
		file = udef_file;
	else
	{
		file = new IoUtils::CVirtualOutputFile;
		file->Init( vMAX_TEXTURE_FILE_SIZE );		// reserve 50 Megs for the texture dictionary
	}

	if (udef_usg_file)
		usg_file = udef_usg_file;
	else
	{
		if (!noUSG)
		{
			usg_file = new IoUtils::CVirtualOutputFile;
			usg_file->Init( vMAX_USAGE_FILE_SIZE );		// reserve 500 KB for the usage file
		}
	}

	//file.open( dict_path, ios::out | ios::binary );
	//usg_file.open( usg_path, ios::out );
	
	version = vTEXTURE_VERSION_NUMBER;
	file->Write((const char*) &version, sizeof( int ));

	all_valid = AllTexturesAreValid();
	file->Write((const char*) &all_valid, sizeof( bool ));

	num_textures = m_ExportTextures.Count();
	file->Write((const char*) &num_textures, sizeof( int ));

	gInterface->ProgressStart(_T("Exporting Texture Dictionary"), TRUE, 
							TextureExportProgressFunc, NULL );
	for( i = 0; i < num_textures; i++ )
	{		
		NxTexture* texture;
		int max_level;
		char basename[_MAX_FNAME], ext[_MAX_EXT];
		char usg_line[128];
		
		texture = m_ExportTextures[i];

		// Verify that the texture names don't have spaces
		CStr texname = texture->GetName( 0 );
		char* nstr = strrchr((char*)texname,'\\');
		if (!nstr)
			nstr = (char*)texname;
		else
			nstr++;

		if (strstr(nstr," "))
		{
			char strErr[256];
			sprintf(strErr,"A texture '%s' contains spaces.  This is unsupported.",nstr);
			MessageBoxAll(gInterface->GetMAXHWnd(),strErr,"Texture contains spaces",MB_ICONWARNING|MB_OK);
			ReportWarning( strErr );
		}

		if( texture->IsValid())
		{
			int width, height;
			int pixel_mode, clut_mode, flags, plat_flags, name_len;
	
			name_len = strlen( texture->GetName( 0 ));
			file->Write((const char*) &name_len, sizeof( int ));
			file->Write((const char*) texture->GetName( 0 ), name_len );

			plat_flags = texture->GetPlatformFlags();
			file->Write((const char*) &plat_flags, sizeof( int ));

			checksum = texture->GetChecksum();
			file->Write((const char*) &checksum, sizeof( unsigned long ));
			width = texture->GetWidth( 0 );
			height = texture->GetHeight( 0 );
			file->Write((const char*) &width, sizeof( int ));
			file->Write((const char*) &height, sizeof( int ));
			pixel_mode = texture->GetPixelFormat();
			clut_mode = texture->GetPaletteFormat();
			flags = texture->GetFlags();			
		
			// Get the max mip level
			max_level = texture->GetNumMipLevels();
			
			// Write out pixel/clut storage modes
			file->Write((const char*) &flags, sizeof( int ));			
			file->Write((const char*) &pixel_mode, sizeof( int ));
			file->Write((const char*) &clut_mode, sizeof( int ));			
			file->Write((const char*) &max_level, sizeof( int ));

			if( texture->IsPaletted())
			{				
				if( texture->GetPaletteFormat() == NxTexture::v32_BIT )
				{
					if( texture->ShouldChangeTransparentColor())
					{
						unsigned char* src, *alpha;
						int num_entries;
						unsigned char red, green, blue;
						Color trans_color;

						trans_color = texture->GetTransparentColor();
						red = trans_color.r * 255.0f;
						green = trans_color.g * 255.0f;
						blue = trans_color.b * 255.0f;
						num_entries = texture->GetNumPaletteEntries();
						src = (unsigned char*) texture->GetPaletteData();
						for( j = 0; j < num_entries; j++ )
						{
							alpha = src + ( 3 * sizeof( char ));	// skip to alpha val
							// If the alpha is
							if( *alpha == 0 )
							{
								*src++ = red;
								*src++ = green;
								*src++ = blue;
								src++;
							}
							else
							{
								src += ( 4 * sizeof( char ));
							}
						}							
					}
				}
				
				file->Write((const char*) texture->GetPaletteData(), texture->GetPaletteDataSize());				
			}

			for( j = 0; j <= texture->GetNumMipLevels(); j++ )
			{
				// Write out the width and height of the mip  aml
				int mipWidth, mipHeight, name_len;				
	
				name_len = strlen( texture->GetName( j ));
				file->Write((const char*) &name_len, sizeof( int ));
				file->Write((const char*) texture->GetName( j ), name_len );

				file->Write((const char*) &texture->GetFileTime( j ), sizeof( FILETIME ));

				mipWidth  = texture->GetWidth( j );
				mipHeight = texture->GetHeight( j );

				file->Write((const char*) &mipWidth, sizeof(int));
				file->Write((const char*) &mipHeight, sizeof(int));

				// Write out the texel data
				file->Write((const char*) texture->GetTexelData( j ), 
								texture->GetTexelDataSize( j ));									
			}
		}
		else
		{
			int pad;

			pad = 0;

			file->Write((const char*) &pad, sizeof( int ));		
			file->Write((const char*) &pad, sizeof( int ));

			pad = -1;			
		
			// Write out a header that signifies an empty texture
			checksum = 0xFFFFFFFF;
			file->Write((const char*) &checksum, sizeof( unsigned long ));
			
			file->Write((const char*) &pad, sizeof( int ));
			file->Write((const char*) &pad, sizeof( int ));
			file->Write((const char*) &pad, sizeof( int ));
			file->Write((const char*) &pad, sizeof( int ));
			file->Write((const char*) &pad, sizeof( int ));
			file->Write((const char*) &pad, sizeof( int ));
		}

		if( gInterface->GetCancel())
		{
			gInterface->ProgressEnd();
			if (!udef_file)
				file->Uninit();

			if (!udef_usg_file)
				usg_file->Uninit();
			//file.close();
			//usg_file.close();
			return false;
		}

		if (!noUSG)
		{
			_splitpath( texture->GetName( 0 ), NULL, NULL, basename, ext );
			sprintf( usg_line, "%s%s (0x%x) : %dx%d: %d bpp %d pbpp %d levels\n", basename, ext, 
						texture->GetChecksum(),	texture->GetWidth( 0 ),	texture->GetHeight( 0 ), 
						texture->GetBpp(), texture->GetPaletteBpp(), max_level + 1 );
			usg_file->Write( usg_line, strlen( usg_line ));
		}
		
		gInterface->ProgressUpdate( i * 100 / num_textures );
	}	

	gInterface->ProgressEnd();

	if (!udef_file)
	{
		file->Save( dict_path );
		delete file;
	}

	if (!noUSG)
	{
		if (!udef_usg_file)
		{
			usg_file->Save( usg_path, IoUtils::vTEXT_MODE );
			delete usg_file;
		}
	}

	//file.close();
	//usg_file.close();
	
	m_last_num_exported_textures = m_ExportTextures.Count();
	
	m_TextureDb.Delete(0,m_Textures.Count());
	m_TextureDb.ZeroCount();

	m_TextureDb = m_ExportTextures;
	return true;
}


NxTexture*	TextureExporter::GetTextureByChecksum( unsigned long checksum )
{
	int i;

	for( i = 0; i < m_Textures.Count(); i++ )
	{
		if( m_Textures[i]->GetChecksum() == checksum )
		{
			return m_Textures[i];
		}
	}
	
	return NULL;
}

bool TextureExporter::VerifyTexturePaths( char* path )
{
	int i;

	// Standardize on using backslashes for paths
	CStr strPath = ReplaceStr(path,"/","\\");
	CStr filePath;

	char* texpath = getenv(TEXPATH_ENV);

	// If no texture path is defined default it to q:\sk4
	if (!texpath)
		texpath = "q:\\sk4";

	strPath = CStr(texpath) + CStr("\\") + strPath + CStr("\\tex");

	for( i = 0; i < m_Textures.Count(); i++ )
	{
		// Standardize on using backslashes for paths
		filePath = ReplaceStr(m_Textures[i]->GetName(0),"/","\\");
		filePath = ReplaceStr(filePath,"\\\\Bruce\\Qqq","q:");		// TODO: Hard coded paths for now, will change later
		filePath.toLower();
		strPath.toLower();

		if( !IsInstr(filePath,strPath) )
		{
			strPath.toLower();

			char buf[256];
			sprintf(buf,"WARNING! The texture '%s' does not reside in the expected directory '%s'.\nWould you like to recieve further warnings of this type?",(char*)m_Textures[i]->GetName(0),(char*)strPath);
			if( !bWarn )
			{
				int bResult = MessageBox(gInterface->GetMAXHWnd(),buf,"Directory Warning",MB_ICONWARNING|MB_YESNOCANCEL);

				if (bResult==IDNO)
				{
					bWarn = false;
					return true;
				}

				if (bResult==IDCANCEL)
					return false;
			}
			ReportWarning( buf );
		}
	}

	return true;
}

void	TextureExporter::OnPostOpen( void )
{
	ExportOptions* options;
			
	options = GetExportOptions();		
	
	if( options->m_ExportType == ExportOptions::vEXPORT_SCENE )
	{
		TSTR texfile_name, base_name;
		SceneExportOptions scene_options;
		char* project_root;	
		
		GetSceneExportOptions( &scene_options );
			
		project_root = getenv( APP_ENV );
		if( project_root )
		{
			base_name = scene_options.m_SceneName;
			texfile_name = TSTR( project_root ) + TSTR( "/data/levels/" ) + base_name + TSTR("/") + base_name + TSTR( ".tex" );

			LoadTextureDictionary( texfile_name );
		}
	}	
}

void	TextureExporter::LoadTextureDictionary( char* path )
{
	int i, version, num_textures;
	bool all_valid;
	char* texel_data;

	m_TextureDb.Delete(0,m_Textures.Count());
	m_TextureDb.ZeroCount();

	IoUtils::CVirtualInputFile theInputFile;
	if ( !theInputFile.Load( path ) )
	{
		return;
	}

	// dictionary version
	theInputFile.Read((char*) &version, sizeof( int ));
	
	// For now, don't be backwards-compatible
	if( version != vTEXTURE_VERSION_NUMBER )
	{
		return;
	}

	theInputFile.Read((char*) &all_valid, sizeof( bool ));
	if( all_valid == false )
	{
		return;
	}	

	theInputFile.Read((char*) &num_textures, sizeof( int ));
	for( i = 0; i < num_textures; i++ )
	{		
		NxTexture* texture;
		int j, width, height;
		char name[ 256 ];
		unsigned long checksum;		
		int length, plat_flags, flags, format, levels;
		FILETIME time;
		
		texture = new NxTexture;
			
		theInputFile.Read((char *) &length, sizeof( int ));
		if( length > 0 )
		{
			theInputFile.Read((char *) name, length );
			name[length] = '\0';
			texture->SetName( name, 0 );
		}		

		theInputFile.Read((char *) &plat_flags, sizeof( int ));		
		texture->SetPlatformFlags( plat_flags );
		theInputFile.Read((char*) &checksum, sizeof( unsigned long ));
		texture->SetChecksum( checksum );
				
		theInputFile.Read((char *) &width, sizeof( int ));
		theInputFile.Read((char *) &height, sizeof( int ));
		texture->SetWidth( 0, width );
		texture->SetHeight( 0, height );
		
		// Read in pixel/clut storage modes
		theInputFile.Read((char *) &flags, sizeof( int ));		
		texture->SetFlags( flags );
		theInputFile.Read((char *) &format, sizeof( int ));
		texture->SetPixelFormat( format );
		theInputFile.Read((char *) &format, sizeof( int ));			
		texture->SetPaletteFormat( format );
		theInputFile.Read((char *) &levels, sizeof( int ));
		texture->SetNumMipLevels( levels );

		// Check for a checksum that indicates an invalid texture
		if( checksum == vINVALID_CHECKSUM )
		{
			continue;
		}

		if( texture->IsPaletted())
		{				
			int bytes_per_entry;
			char* palette_data;

			switch( texture->GetPixelFormat())
			{
				case NxTexture::v8_BIT:
					texture->SetNumPaletteEntries( 256 );
					break;
				case NxTexture::v4_BIT:
					texture->SetNumPaletteEntries( 16 );					
					break;
				default:
					return;
			}

			switch( texture->GetPaletteFormat())
			{
				case NxTexture::v32_BIT:
					texture->SetPaletteBpp( 32 );					
					bytes_per_entry = 4;
					break;
				case NxTexture::v24_BIT:
					texture->SetPaletteBpp( 24 );
					bytes_per_entry = 3;
					break;
				case NxTexture::v16_BIT:
					texture->SetPaletteBpp( 16 );
					bytes_per_entry = 2;
					break;
				default:
					return;
			}
			
			palette_data = new char[ texture->GetNumPaletteEntries()* bytes_per_entry ];
			texture->SetPaletteData( palette_data );
			theInputFile.Read((char *) texture->GetPaletteData(), texture->GetNumPaletteEntries() * bytes_per_entry );
		}

		switch( texture->GetPixelFormat())
		{
			case NxTexture::v32_BIT:
				texture->SetBpp( 32 );
				break;
			case NxTexture::v24_BIT:
				texture->SetBpp( 24 );
				break;
			case NxTexture::v16_BIT:
				texture->SetBpp( 16 );
				break;
			case NxTexture::v8_BIT:
				texture->SetBpp( 8 );
				break;
			case NxTexture::v4_BIT:
				texture->SetBpp( 4 );
				break;
			case NxTexture::v8_BIT_GRAY:
				texture->SetBpp( 8 );
				break;
			case NxTexture::v4_BIT_GRAY:
				texture->SetBpp( 4 );
				break;
			default:
				return;
		}

		width = texture->GetWidth( 0 );
		height = texture->GetWidth( 0 );
		for( j = 0; j <= texture->GetNumMipLevels(); j++ )
		{
			char mip_name[ 256 ];
			
			theInputFile.Read((char *) &length, sizeof( int ));
			if( length > 0 )
			{
				theInputFile.Read((char *) mip_name, length );
				mip_name[length] = '\0';
				texture->SetName( mip_name, j );
			}

			theInputFile.Read((char *) &time, sizeof( FILETIME ));
			texture->SetFileTime( j, time );

			theInputFile.Read((char*) &width, sizeof(int));
			theInputFile.Read((char*) &height, sizeof(int));

			texture->SetWidth( j, width );
			texture->SetHeight( j, height );			

			int texel_data_size, bytes_per_row;

			bytes_per_row = (( width * texture->GetBpp()) + 7 ) >> 3;
			texel_data_size = height * bytes_per_row;
			texel_data = new char[ texel_data_size ];			
			// Read in the texel data
			theInputFile.Read((char *) texel_data, texel_data_size );		
			texture->SetTexelData( j, texel_data );
			delete [] texel_data;
		}		

		m_TextureDb.Append( 1, &texture );
	}			
}

bool			TextureExporter::IsSameTexture( NxTexture* src, NxTexture* dst )
{
	int i;
	Color src_color, dst_color;

	// The checksum test should cover the following fields :
	// Basemap filename
	// platform flags
	if( src->GetChecksum() != dst->GetChecksum())
	{
		return false;
	}

	// Which means we still need to test:
	// Num mip levels
	// names of mips
	// texture flags
	// modified times on each mip

	if( src->GetFlags() != dst->GetFlags())
	{
		return false;
	}

	if( src->ShouldAutoGenerateMipMaps() != dst->ShouldAutoGenerateMipMaps())
	{
		return false;
	}


	src_color = src->GetTransparentColor();
	dst_color = dst->GetTransparentColor();
	if( src_color != dst_color )
	{
		return false;
	}

	// If we auto-generate mips, then we only need to check if the basemap has changed
	if( src->ShouldAutoGenerateMipMaps())
	{
		WIN32_FIND_DATA find_data;
		HANDLE file_handle;

		if( stricmp( src->GetName( 0 ), dst->GetName( 0 )))
		{
			return false;
		}		
	
		file_handle = FindFirstFile( src->GetName( 0 ), &find_data );  
		if( file_handle == INVALID_HANDLE_VALUE )
		{
			return false;
		}

		if( CompareFileTime( &find_data.ftLastWriteTime, &dst->GetFileTime( 0 )))
		{
			return false;
		}
		FindClose( file_handle );
	}
	else
	{
		if( src->GetNumMipLevels() != dst->GetNumMipLevels())
		{
			return false;
		}

		// If we are using manual mips, check each mip level to see if it has changed.
		for( i = 0; i <= src->GetNumMipLevels(); i++ )
		{
			WIN32_FIND_DATA find_data;
			HANDLE file_handle;

			if( stricmp( src->GetName( i ), dst->GetName( i )))
			{
				return false;
			}		
		
			file_handle = FindFirstFile( src->GetName( i ), &find_data );  
			if( file_handle == INVALID_HANDLE_VALUE )
			{
				return false;
			}
			FindClose( file_handle );

			if( CompareFileTime( &find_data.ftLastWriteTime, &dst->GetFileTime( i )))
			{
				return false;
			}
		}
	}

	return true;
}

NxTexture*		TextureExporter::GetTextureFromDatabase( NxTexture* src_tex )
{	
	int i, num_textures;
	
	num_textures = m_TextureDb.Count();
	for( i = 0; i < num_textures; i++ )
	{
		NxTexture* db_tex;
		
		db_tex = m_TextureDb[i];
		if( IsSameTexture( src_tex, db_tex ))
		{
			return db_tex;
		}
	}

	return NULL;
}

void	TextureExporter::MarkAsClean( void )
{
	m_dirty = false;
}

void	TextureExporter::MarkAsDirty( void )
{
	m_dirty = true;
}

void	TextureExporter::ResetInitialExport( void )
{
	m_reqInitalExport = true;
}

bool	TextureExporter::ShouldExportTextureDictionary( char* expName )
{
	if( m_dirty || m_reqInitalExport)
	{
		m_reqInitalExport = false;
		return true;
	}

	if( m_last_num_exported_textures != m_ExportTextures.Count())
	{
		return true;
	}

	// Verify dictionary name if the name changed then it weill need reexported again
	if (export_name != CStr(expName))
		return true;

	return false;
}
