/*
	CutsceneExportDlg.h
	Cutscene Exporter Interface
	aml - 1-30-03
*/

#ifndef __CUTSCENEEXPORTDLG_H__
#define __CUTSCENEEXPORTDLG_H__

#include "CamExporter.h"
#include "ObjAnimExport.h"
#include "../UI/MSWindow.h"
#include "../UI/ModalDlg.h"
#include "../UI/ListView.h"
#include "../UI/TrackUI.h"
#include "../misc/saveable.h"
#include "AnimTol.h"			// Animation tolerance config file structs and parsing
#include "ModelExport.h"
#include "CutsceneObjTypes.h"
#include "notetrck.h"
#include "../../../../include/AnimExportFmt.h"
#include "ScriptKeyEditor.h"
#include "ObjKeyEditor.h"
#include "QNEditor.h"
#include "PartialAnimEditor.h"

#define CUTSCENEEXPORTER_CHUNK_VERSION  3
#define CUTSCENEDATA_VERSION            4
#define CUTSCENEOBJ_VERSION             6
#define PERBONETOL_VERSION              2

struct LIBObjPacket
{
	unsigned int offset;		// Offset to this packet from the beginning of the library file
	unsigned int size;			// Size of described packet
	unsigned int nameCRC;
	unsigned int extCRC;		// Extension (CIF, CAM, OBA, SKA, etc)
};

class PerBoneTol: public Saveable
{
public:
	DWORD version;	// version for backward compat with later PerBoneTol structures
	DWORD handle;	// Handle of the node for which these tolerance levels apply to
	float rotTol;	// Translation tolerance for this bone
	float tranTol;	// Rotation tolerance for this bone
	CStr  name;		// Name of the node for which these tolerance levels apply to
					// we now use names instead of handles due to problems with XRefs

	PerBoneTol();

	int  GetSize();
	void Store(void* data);
	void Retrieve(void* data);
};

class CutsceneObj: public Saveable
{
public:
	DWORD   version;					// version for backward compat with later CutsceneObj structures
	CStr    name;						// Name of the object
	DWORD   type;						// Type of the object
	DWORD   handle;						// Handle of this node in the scene
	int     start;						// Time object becomes visible/active
	int     end;						// Time object becomes invisible/inactive
	float   rotTol;						// Rotation tolerance
	float   tranTol;					// Translation tolerance
	CStr    modelName;					// Name of associated model
	CStr    modelName2;					// If this is a hi-res skeleton, this is the name for the upper half of the body
	CStr    skelName;					// Name of associated skeleton
	CStr    skelName2;					// If this is a hi-res skeleton, this is the name for the lower half of the body
	CStr    rootBone;					// Name of associated root bone
	CStr    partialAnimSet;				// Name of associated partial anim set (0 length string if n/a)
	CStr    debugStr;					// Temporary non-saved string for debugging purposes
	LinkList<PerBoneTol> boneTolList;	// Stores tolerances per bone
	LinkList<TrackUIKey> trackKeyDB;	// Per object script key data  (v3)

	bool    bHiresSkeleton;				// If true the skeleton associated with this object needs to be
										// broken into two skeletons having (1 BonedAnim & 1 AnimObj) each
										// The exporter will break these into two objects immediatly before
										// export and delete the extras when done

	bool    bUseSkaterModel;

	bool    bCopy;						// Temporary value in the event that the system makes a copy of the
										// cutscene object to handle breaking up hires skeletons and anims
										// if bCopy is set, this object should be disposed of after export,
										// it also indicates that this should refer to a model break point
										// at the neck bone (for now)

	bool    bExportFromPelvis;			// If true and the object is a hires skeleton it will be exported from the pelvis

	bool    bModified;					// True if this object has been modified since the last cutscene export
										// and should thus be reexported next time the user exports the cutscene

	CutsceneObj*  pBonedAnim;			// In main objDB:
										// If this object is a skin this pointer will point to the associated bonedAnim
										// that was generated when building the boned anim list
										// 
										// In bonedAnimDB: (bCopy is true)
										// This field will reference the original bonedAnim in the bonedAnimDB that
										// this cutscene object was copied from
										//
										// In bonedAnimDB: (bCopy is false)
										// This field will reference the copied bonedAnim in the bonedAnimDB that
										// was made from this cutscene object if one exists

	CutsceneObj*  pObjAnim;				// If this object is a model or a skin this pointer will point to the associated
										// object animation that was generated when building the obj anim list
										// (Main objDB only)

	CutsceneObj*  pOrigObj;				// Points back to the original object in the main objDB that this object
										// was created from (assuming this object exists in objAnimDB or bonedAnimDB)

	CutsceneObj();

	// Acquires the cutscene object properties from PE properties in the node (uses handle to find node)
	void LoadFromNode(CStr cutsceneName, CutsceneObj* csoDefault = NULL);

	// Stores the values in the cutscene object into the node as PE properties (uses handle to find node)
	void SaveToNode(CStr cutsceneName);

	int operator==(CutsceneObj& right)
	{
		return (name   == right.name  &&
			    type   == right.type  &&
				bCopy  == right.bCopy &&
				handle == right.handle);
	}

	int  GetSize();
	void Store(void* data);
	void Retrieve(void* data);

	static DWORD GetTaggedFlags(INode* node);
	       void  SetTaggedFlags(INode* node);
};

class Cutscene: public Saveable
{
public:
	DWORD version;						// version for backward compat with later cutscene structures
	CStr  name;							// Name of the cutscene
	LinkList<CutsceneObj> objDB;		// List of user set objects to export (only Models, Skins, or Cameras)
	LinkList<CutsceneObj> objAnimDB;	// Generated list of all the associated object animations to export
	LinkList<CutsceneObj> bonedAnimDB;	// Generated list of all the boned animations to export
	LinkList<TrackUIKey>  trackKeyDB;	// Global track key information (Gets assigned to TrackUI control)
	LinkList<SaveableStr> nodeNameDB;	// Names of all the nodes that should be included as structs in the QN
	float                 rotTol;		// Rotation tolerance
	float                 tranTol;		// Translation tolerance

	Cutscene();

	// Saves/Retrieves all objects in objDB properties from their user prop buffer data (accessible to PE)
	void StoreObjects();
	void RetrieveObjects();

	void BuildObjAnimDB();
	void BuildBonedAnimDB();

	bool ModelExists(CutsceneObj* ignore, CStr modelName);
	bool VerifyRootBones(HWND hwnd);
	bool VerifyModelNames(HWND hwnd);

	int  GetSize();
	void Store(void* data);
	void Retrieve(void* data);
};

class CutsceneExportDlg: public MSDlgWindow, 
                         public CameraExporter, 
						 public ObjectExporter,
						 public ReferenceMaker
{
	ListView*        pObjList;				// The object list of all objects in the cutscene
	TrackUI*         pCameraTrack;			// The visualization UI for camera start/end times
	//TrackUI*       pObjectTrack;			// The visualization UI for object start/end visibilities
	ICustEdit*       pEditFind;				// Edit box for matching wildcards in object list
	ICustEdit*       pEditFindScn;			// Edit box for matching wildcards in cutscene list
	PropList*        propList;				// List of specific properties based on type of object
	CStr             rootDir;				// Root directory for ModelName skin files
	CStr             configPath;			// Path where the OptFile config files should be stored
	CStr             curCutsceneFilename;	// Filename of the current .cut file being exported
	ScriptKeyEditor* scriptKeyEditor;		// Popup script key editor window when script key is double-clicked 
											// in camera TrackUI control

	ObjKeyEditor*      pObjKeyEditor;		// Object track key editor popup on obj double click
	QNEditor*          pQNEditor;			// Editor for defining the nodes that get included in the cutscene QN
	PartialAnimEditor* partialAnimEditor;	// Partial animation editor window


	LinkList<INode*> QNNodeList;			// List of all the nodes that get output to the cutscene QN as structs

	bool           bLockPropChange;			// True if property change updates shouldn't occur
	bool           bLockExportModeChange;	// True if the export mode should temporarily not be updated
	bool           bLoaded;					// True if cutscene data has been loaded from the MAX file
	
	Interface* ip;

	Link<Cutscene>* linkActiveCutscene;	// The active cutscene

	LinkList<Cutscene> cutsceneDB;

	// For RemoveInvisibleKeys
	bool           bexpModel;				// True if model is being exported, false for skin
	INode*         expNode;					// The node currently being exported (used for Visibility check)
	INode*         root;					// Root node being exported for boned anim export
	int            numNodes;				// Number of nodes being exported for boned anim export
	CutsceneObj*   expCso;					// Cutscene object currently being exported
	///////////

	BOOL DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

	enum
	{
		DISP_ALL,								// Display all objects in scene
		DISP_CUTSCENE,							// Display only objects in cutscene
		DISP_UNASSIGNED_TO_CUTSCENE,			// Display only objects not in current cutscene
		DISP_UNASSIGNED_TO_ANY,					// Display all objects not assigned to any cutscene
	};

	enum
	{
		PROPVIEW_UNASSIGNED,
		PROPVIEW_CUTSCENE,						// Properties for the current cutscene will be displayed
		PROPVIEW_OBJECT,						// Properties for the current object will be displayed
	};

	int  displayMode;							// The current display being shown in the object list
	int  propViewMode;							// The current display mode for properties

	void InitObjList(int selFlags = 0);
	
	// Generate the object list based on display params
	void AddObjListNodesAll(INode* node = NULL);
	void AddObjListNodesCutscene();
	void AddObjListNodesUnassignedToCutscene(INode* node = NULL);
	void AddObjListNodesUnassignedToAnyCutscene(INode* node = NULL);
	
	void AddObjNode(INode* node, char* name = "");	// Actually adds each node to the object list

	void AddCutscene();
	void RemoveCutscene();
	void RenameCutscene();

	void AddCutsceneObj();
	void RemoveCutsceneObj();
	void SelCutsceneObj();
	void GetCutsceneObjSel();

	void FindCutscene();
	void FindObject();

	void AddCamsToTrack();
	int  NumCameras(Cutscene* cutscene);
	bool CamsHaveTimes(Cutscene* cutscene);
	void TrackUpdate(HWND hwnd);
	void PropertyUpdate();

	Link<Cutscene>* GetObjScene(INode* node);
	Link<CutsceneObj>* GetCutsceneObj(Cutscene* cutscene, INode* node);

	void StoreCutsceneProps();
	void RetrieveCutsceneProps();
	void StoreObjectProps();
	void RetrieveObjectProps();

	void UpdateExportMode();
	void ExportModeChange();

	void SaveToMAX();
	void LoadFromMAX();
	void CreateCutsceneList();

	void GetExportParams(int* items, int* nItems);
	int  GetNumPackets();

	int  GetNumType(Cutscene* cutscene, DWORD type);
	int  GetNumCIFObjects(Cutscene* cutscene);

	bool IsExternal(CStr modelName);
	bool IsExternal(CutsceneObj* cso);
	int  GetNumExternal(Cutscene* cutscene, ObjType objType);

	bool DoExport();

	static void TimeChangeCB(TrackUI* trackUI, void* pData);
	static void PropListChangeCB(PropList* propList, void* pData);

	void FindDuration(Cutscene* cutscene, ObjType type, int* startFrame, int* endFrame);

	int  BuildCompositeCamPacket(Cutscene* cutscene, unsigned char* pData, unsigned int* pos);
	int  BuildObjAnimPacket(Cutscene* cutscene, unsigned char* pData, unsigned int* pos);
	int  BuildBonedAnimPacket(Cutscene* cutscene, CutsceneObj* csobj, unsigned char* pData, unsigned int* pos);
	
	int  BuildModelPacket(CutsceneObj* csobj, unsigned char* pData, unsigned int* pos,
		                                      unsigned char* pDataTEX, unsigned int* posTEX);
	
	int  BuildSkinPacket(Cutscene* cutscene, CutsceneObj* csobj, unsigned char* pData, unsigned int* pos,
		                                                         unsigned char* pDataTEX, unsigned int* posTEX);

	void WriteCIFPacket(Cutscene* cutscene, FILE* fp);

	void ExportOptions();

	bool AddHiresDupes(Cutscene* cutscene);
	void RemoveHiresDupes(Cutscene* cutscene);

	// TEMP: Returns true if skeleton is using a 3 bone brow structure
	//       (in which case Bone_Brow_MID is returned)
	bool SetSkeletonExceptions(INode* node, CSkeletonData* skeleton);
	
	bool BoneIsFake(INode* root, CutsceneObj* csobj, INode* node);

	INode* FindRoot(INode* node);

	void BuildExportableNodeList(INodeTab& exportable_nodes, INode* node);
	bool ModelExistsInCutscene(Cutscene* cutscene, CutsceneObj* excludeobj, CStr modelName);
	CutsceneObj* FindObjByModelName(Cutscene* cutscene, char* modelName);
	CutsceneObj* FindObjByName(Cutscene* cutscene, char* name, bool bCopy);

	void EnableFlags(BOOL bMode);
	CStr BuildFlagString(DWORD flags);

	INode* FindBoneRootByModelName(Cutscene* cutscene, char* modelName);
	INode* FindBoneRootBySkin(INode* skinNode);
	CutsceneObj* FindAssocBoneAnim(Cutscene* cutscene, INode* rootBone);

	void RetrieveAllProps();
	int  FindLowestCamStartFrame(Cutscene* cutscene);

	bool RequiresCompression(Cutscene* cutscene, DWORD type);

	bool HasCamera(Cutscene* cutscene);							// Returns true if a camera exists in the cutscene

	void ClearPerBoneTolerances(INode* node = NULL);
	bool HasExportModeData(INode* node);

	CutsceneObj* AddRootBoneToScene(Cutscene* cutscene, INode* skinNode);
	CutsceneObj* GetObjByTypeAndModelName(Cutscene* cutscene, DWORD objType, CStr modelName);
	CutsceneObj* FindObjByName(LinkList<CutsceneObj>* list, CStr name, bool bCopy);

	void  DiscardLegacyFlags();
	DWORD DetermineType(INode* node);

	// Retrieve the index of the first key that occurs before the given time
	int GetKeyBeforeTime(STKey* keys, int nKeys, float time);
	int GetKeyBeforeTime(SQKey* keys, int nKeys, float time);
	int GetKeyAtTime(STKey* keys, int nKeys, float time);
	int GetKeyAtTime(SQKey* keys, int nKeys, float time);

	bool IsValidNoteKey(NoteKey* key, int start, int end);

	void DisplayHelp();
	void SyncTimesToCam();
	void SyncTimesToCamSel();

	INode* GetNoteTrackCam(Cutscene* cutscene, float time, TimeValue* camTime = NULL);
	
	bool ShouldCompress(CutsceneObj* cso);
	void RemoveFlaggedInvisibleKeys(Gfx::SAnimQFrame** qframes, int* nKeys);
	void RemoveFlaggedInvisibleKeys(Gfx::SAnimTFrame** tframes, int* nKeys);

	int GetNumInvisibleKeys(Gfx::SAnimQFrame* qframes, int nKeys);
	int GetNumInvisibleKeys(Gfx::SAnimTFrame* tframes, int nKeys);

	bool SkeletonIsHires(INode* node);
	void EditObjKeys();
	bool CompileScriptKeys(Cutscene* pCutscene);
	int  GetTotalScriptKeys(Cutscene* cutscene);

	bool KeysExist(Cutscene* pCutscene);

	float GetStartTime(Cutscene* pCutscene);
	void  OpenQNEditor();

	CStr GetGlobalScriptName(Cutscene* pCutscene, TrackUIKey* pTrackKey);
	CStr GetObjScriptName(Cutscene* pCutscene, CutsceneObj* pCSO, TrackUIKey* pTrackKey);

	void PartialAnimChanged();

	// Called whenever a cutsceneobj changes in some way
    RefResult NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
								PartID& partID,  RefMessage message);		

	bool ModificationsExist();
	void ResetModifiedFlags();
	void SetModified(bool bMode);
	void SetCameraFlags();
	bool CamerasModified(Cutscene* cutscene);

	void MakeNodeReferences();
	void* LoadPacket(char* filename, unsigned int objName, unsigned int packetType, LIBObjPacket* packetDesc);

	void   SetCamerasModified(Cutscene* cutscene);

	bool IsFlaggedWorldSpace(Cutscene* cutscene, INode* node);

	void FlagAutoAssign();

	// Callback called by MAX when the scene is reset or a new one loaded
	static void CloseDlg(void *param, NotifyInfo *info);
	static void CloseDlgSave(void *param, NotifyInfo *info);

	static void RemoveInvisibleKeys(void* pData);
	static void ScriptKeyUI(Link<TrackUIKey>* link, void* pData);
	
	static void KeyTimeChanged(int time, void* pData);
	static void UpdateKeys(ScriptKeyEditor* pEditor, void* pData);
	static void CancelKeyUpdate(ScriptKeyEditor* pEditor, void* pData);
	static void KeyChanged(ScriptKeyEditor* pEditor, Link<TrackUIKey>* link, void* pData);

	static void ctKeyAdded(TrackUI* pTrackUI, Link<TrackUIKey>* linkKey, void* pData);
	static void ctKeyRemoved(TrackUI* pTrackUI, Link<TrackUIKey>* linkKey, void* pData);
	static void ctKeyChanged(TrackUI* pTrackUI, void* pData);

	static void ObjKeyChanged(ObjKeyEditor* pObjKeyEditor, CutsceneObj* pCutsceneObj, void* pData);

public:
	CutsceneExportDlg(HINSTANCE hInstance, HWND hwndParent);
	~CutsceneExportDlg();

	inline LinkList<Cutscene>* GetCutsceneDB() { return &cutsceneDB; }
	inline LinkList<INode*>*   GetQNNodeList() { return &QNNodeList; }

	void SetActiveCutscene();
	void Refresh();

	void Hide();
	void HideNoSave();		// Hides the cutscene exporter window without saving data
	void Show();

	bool Load(char* filename = NULL);
	bool Save(char* filename = NULL);

	void UpdateKeyData();
	CStr GenerateScript(Cutscene* pCutscene);

	void   PopulatePartialAnimList();
	void   DumpDebugFrames();
};

class CutsceneBoneTolDlg: public ModalDlgWindow
{
	Interface*       ip;
	CutsceneObj*     cutsceneObj;		// The cutscene object that contains the tolerance data being edited
	ListView*        pTolList;			// List containing tolerance values
	ICustEdit*       rotTolEdit;		// Rotation tolerance edit field
	ICustEdit*       transTolEdit;		// Translation tolerance edit field
	ISpinnerControl* rotTolEditSpin;	// Rotation tolerance spinner
	ISpinnerControl* transTolEditSpin;	// Translation tolerance spinner
	bool             bLockTolChange;	// True if tolerance changes are not to cause updates

	AnimTolFile      TolFile;			// Animation tolerance preset file (animtol.ini)

	BOOL DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	
	void Init();
	void AddBoneHierarchy(INode* bone);
	void AddBone();
	void RemoveBone();

	void AddAll();
	void RemoveAll();

	void RotTolUpdate();
	void TranTolUpdate();

	void Store();
	void Retrieve();

	void UpdateTolBoxes();
	void PresetChange();

	void LoadPresets();

public:
	CutsceneBoneTolDlg(HINSTANCE hInstance, HWND hwndParent, Interface* ip, CutsceneObj* obj);
	~CutsceneBoneTolDlg();
};

INode* FindPelvisBone(INode* node);

#endif
