/*
	CutsceneExportDlg.cpp
	Cutscene Exporter Interface
	aml - 1-30-03
*/

#include "CutsceneExportDlg.h"
#include "../UI/InputDlg.h"
#include "resource.h"
#include "../PropEdit/ConfigData.h"
#include "misc/gencrc.h"
#include "appdata.h"
#include "path.h"
#include "../../../../include/AnimExportFmt.h"
#include "../misc/mth.h"
#include "SkinExport.h"
#include "Export.h"
#include "Strip/Strip.h"
#include "TextureExporter.h"
#include "../UI/OptFile.h"
#include "../RCMenu.h"
#include "../misc/HelperFuncs.h"
#include "phyexp.h"					// Physique SDK (for locating root of system from skin object)
#include "memDebug.h"
#include "../misc/util.h"
#include "../misc/MAXUtil.h"
#include "ScriptExport.h"
#include <io.h>
#include <sys/stat.h>
#include "ExportOptions.h"
#include "../Particle/NExtParticle.h"

//#define PI  ((float)3.1415926535)
//#define DEG_TO_RAD (PI/(float)180.0)
//#define RAD_TO_DEG ((float)180.0/PI)
//#define DegToRad(deg) (((float)deg)*DEG_TO_RAD)

#define PACKET_DEBUG

extern Interface* gInterface;

//#define SKATER_CRC            0x5b8ab877
#define SKATER_CRC            0x67e6859a

#define CUTSCENEGRP           CStr("Cutscn/")	// Property tab prefix
#define CAMERA_CRC            0xc4e311fa		// Explicit bone name when exporting cameras

#define NEARKEY_TOL           0.0001f			// This is the difference in time for dupe keys added to
												// ensure the cutscene sequence doesn't interpolate accross camera
												// cuts in-game

#define CUTSCENE_VERSION      0x0001			// Version of the cutscene file format
#define CUTSCENE_BUFFER_SIZE  1024*1024*30		// 30 Meg max size for cutscene files

#define ROT_COL   1
#define TRANS_COL 2

#define NUM_FAKE_BONES        8
#define BONE_BROW_ID          4

#define vMAX_TEXTURE_FILE_SIZE	( 50 * ( 1024 * 1000 ))
#define vMAX_USAGE_FILE_SIZE	( 500 * 1024 )

#define vCIF_FORMAT_VERSION 0x0001

#define SKELNAME1_BONED_DEFAULT  "thps5_human"
#define SKELNAME2_BONED_DEFAULT  "test_facial"

#define DEBUG_BONEDANIM_FILE
#define OBA_DEBUG
#define DUMP_CAMERA_DEBUG
#define DUMP_DEBUG_CUSTKEYS

#define FRAME_INCR       0.016666667f

#define KEYFLAG_INVISIBLE   0x01

char FakeBoneNames[NUM_FAKE_BONES][32] = { "Bone_Neck",
                                           "Bone_Head",
							               "Bone_Jaw",
							               "Cloth_Hat",
							               "Bone_Brow",
							               "Bone_PonyTail_1",
							               "Bone_PonyTail_2",
							               "Cloth_Breast" };

PerBoneTol::PerBoneTol()
{
	handle  = 0;
	rotTol  = 0.0f;
	tranTol = 0.0f;
	version = PERBONETOL_VERSION;
}

int PerBoneTol::GetSize()
{
	return sizeof(DWORD) + sizeof(DWORD) + sizeof(float) + sizeof(float) + GetStringSize(name);
}

void PerBoneTol::Store(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	// Always store most up-to-date version
	//*((DWORD*)pos) = version;
	*((DWORD*)pos) = PERBONETOL_VERSION;
	pos += sizeof(DWORD);
	*((DWORD*)pos) = handle;
	pos += sizeof(DWORD);
	*((float*)pos) = rotTol;
	pos += sizeof(float);
	*((float*)pos) = tranTol;
	pos += sizeof(float);
	WriteString(&pos, name);
}

void PerBoneTol::Retrieve(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	version = *((DWORD*)pos);
	pos += sizeof(DWORD);
	handle = *((DWORD*)pos);
	pos += sizeof(DWORD);
	rotTol = *((float*)pos);
	pos += sizeof(float);
	tranTol = *((float*)pos);
	pos += sizeof(float);
	GetString(&pos, name);
}

Cutscene::Cutscene()
{
	version = CUTSCENEDATA_VERSION;
}

CutsceneObj::CutsceneObj()
{
	type    = 0;
	handle  = 0;
	start   = 0;
	end     = 0;
	rotTol  = 1.0f;
	tranTol = 0.0f;
	version = CUTSCENEOBJ_VERSION;
	
	bHiresSkeleton    = false;
	bCopy             = false;
	bUseSkaterModel   = false;
	bModified         = false;
	bExportFromPelvis = false;

	pBonedAnim = NULL;
	pObjAnim   = NULL;
}

int  CutsceneObj::GetSize()
{
	int size = sizeof(DWORD) +
		       GetStringSize(name) + 
		       sizeof(DWORD) + 
		       sizeof(DWORD) + 
		       sizeof(int) +
		       sizeof(int) +
		       sizeof(bool) +
		       ListSize(&boneTolList);

	if (version >= 3)
		size += ListSize(&trackKeyDB);

	if (version >= 4)
		size += GetStringSize(partialAnimSet);

	if (version >= 5)
		size += sizeof(bool);

	if (version >= 6)
		size += sizeof(bool);

	return size;
}

void CutsceneObj::Store(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	// Always write most up-to-date version
	//*((DWORD*)pos) = version;
	*((DWORD*)pos) = CUTSCENEOBJ_VERSION;
	pos += sizeof(DWORD);
	WriteString(&pos, name);
	*((DWORD*)pos) = type;
	pos += sizeof(ObjType);
	*((DWORD*)pos) = handle;
	pos += sizeof(DWORD);
	*((int*)pos) = start;
	pos += sizeof(int);
	*((int*)pos) = end;
	pos += sizeof(int);
	*((bool*)pos) = bHiresSkeleton;
	pos += sizeof(bool);

	WriteList<PerBoneTol>(&pos, &boneTolList);

	// v3
	WriteList<TrackUIKey>(&pos, &trackKeyDB);

	// v4
	WriteString(&pos, partialAnimSet);

	// v5
	*((bool*)pos) = bModified;
	pos += sizeof(bool);

	// v6
	*((bool*)pos) = bExportFromPelvis;
	pos += sizeof(bool);
}

void CutsceneObj::Retrieve(void* data)
{
	unsigned char* pos = (unsigned char*)data;
	version = *((DWORD*)pos);
	pos += sizeof(DWORD);
	GetString(&pos, name);
	type = *((DWORD*)pos);
	pos += sizeof(ObjType);
	handle = *((DWORD*)pos);
	pos += sizeof(DWORD);
	start = *((int*)pos);
	pos += sizeof(int);
	end = *((int*)pos);
	pos += sizeof(int);
	bHiresSkeleton = *((bool*)pos);
	pos += sizeof(bool);

	GetList<PerBoneTol>(&pos, &boneTolList);

	if (version >= 3)
		GetList<TrackUIKey>(&pos, &trackKeyDB);
	else
		trackKeyDB.Clear();

	if (version >= 4)
		GetString(&pos, partialAnimSet);

	if (version >= 5)
	{
		bModified = *(bool*)pos;
		pos += sizeof(bool);
	}
	else
	{
		// Any old format Cutscene objects automatically flag as modified
		// so the first export re-exports everything.
		bModified = true;
	}

	if (version >= 6)
	{
		bExportFromPelvis = *(bool*)pos;
		pos += sizeof(bool);
	}
	else
	{
		bExportFromPelvis = false;
	}
}

void CutsceneObj::LoadFromNode(CStr cutsceneName, CutsceneObj* csoDefault)
{
	CStr     strCategory = CUTSCENEGRP + cutsceneName;
	CStr     propVal;
	//INode*   node        = gInterface->GetINodeByHandle(handle);
	INode*   node          = gInterface->GetINodeByName(name);
	Interval interval;

	interval = gInterface->GetAnimRange();

	if (node)
	{
		propVal = GetPropValue(node, strCategory + "/start");

		if (propVal == CStr("") || propVal == CStr("!"))
		{
			if (csoDefault)
				start = csoDefault->start;
			else
			{
				start = interval.Start() / GetTicksPerFrame();
				if (start < 0)
					start = 0;
			}
		}
		else
			start = atoi(propVal);

		propVal = GetPropValue(node, strCategory + "/end");

		if (propVal == CStr("") || propVal == CStr("!"))
		{
			if (csoDefault)
				end = csoDefault->end;
			else
				end = interval.End() / GetTicksPerFrame();
		}
		else
			end = atoi(propVal);

		propVal = GetPropValue(node, strCategory + "/rotTol");
		if (propVal == CStr("") || propVal == CStr("!"))
		{
			if (csoDefault)
				rotTol = csoDefault->rotTol;
		}
		else
			rotTol = atof(propVal);

		propVal = GetPropValue(node, strCategory + "/tranTol");
		if (propVal == CStr("") || propVal == CStr("!"))
		{
			if (csoDefault)
				tranTol = csoDefault->tranTol;
		}
		else
			tranTol = atof(propVal);

		propVal = GetPropValue(node, strCategory + "/UseSkaterModel");
		bUseSkaterModel = (propVal == CStr("TRUE")) ? true : false;

		propVal = GetPropValue(node, strCategory + "/HiResSkeleton");
		bHiresSkeleton = (propVal == CStr("TRUE")) ? true : false;

		modelName  = GetPropValue(node, strCategory + "/ModelName");
		if (modelName == CStr("!"))
			modelName = "";

		if (modelName == CStr("") && csoDefault)
			modelName = csoDefault->modelName;

		modelName2 = GetPropValue(node, strCategory + "/ModelName2");
		if (modelName2 == CStr("!"))
			modelName2 = "";

		if (modelName2 == CStr("") && csoDefault)
			modelName2 = csoDefault->modelName2;

		skelName   = GetPropValue(node, strCategory + "/SkeletonName");
		if (skelName == CStr("!"))
			skelName = "";

		if (skelName == CStr("") && csoDefault)
			skelName = csoDefault->skelName;

		skelName2  = GetPropValue(node, strCategory + "/SkeletonName2");
		if (skelName2 == CStr("!"))
			skelName2 = "";
		
		if (skelName2 == CStr("") && csoDefault)
			skelName2 = csoDefault->skelName2;

		rootBone   = GetPropValue(node, strCategory + "/rootBone");
		if (rootBone == CStr("!"))
			rootBone = "";
		
		if (rootBone == CStr("") && csoDefault)
			rootBone = csoDefault->rootBone;

		partialAnimSet = GetPropValue(node, strCategory + "/partialAnimSet");
		if (partialAnimSet == CStr("!"))
			partialAnimSet = "";

		if (partialAnimSet == CStr("") && csoDefault)
			partialAnimSet = csoDefault->partialAnimSet;

		// Node tolerance data is stored in the node directly
		AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_PERBONE_TOLERANCES);

		if (appdata && appdata->data)
		{
			unsigned char* pos = (unsigned char*)appdata->data;
			GetList<PerBoneTol>(&pos, &boneTolList);
		}

		// Acquire any tagged export mode data
		// Override if this is a camera
		if (node->EvalWorldState(0).obj->SuperClassID() == CAMERA_CLASS_ID)
			type = OBJTYPE_CAMERA;
		else
			type = GetTaggedFlags(node);
	}
}

void CutsceneObj::SaveToNode(CStr cutsceneName)
{
	CStr    strCategory = CUTSCENEGRP + cutsceneName;
	CStr    propVal;
	//INode*  node = gInterface->GetINodeByHandle(handle);
	INode*  node = gInterface->GetINodeByName(name);

	if (node)
	{
		AddPropValue(node, strCategory, "start", start, "spinedit | 0 | 999999 | 1");
		AddPropValue(node, strCategory, "end", end, "spinedit | 0 | 999999 | 1");
		AddPropValue(node, strCategory, "rotTol", rotTol, "spinedit | 0.0 | 1.0 | 0.0001");
		AddPropValue(node, strCategory, "tranTol", tranTol, "spinedit | 0.0 | 1.0 | 0.0001");
		
		if (bUseSkaterModel)
			AddPropValue(node, strCategory, "UseSkaterModel", "TRUE", "check");
		else
			AddPropValue(node, strCategory, "UseSkaterModel", "FALSE", "check");

		//if (modelName.Length() > 0)
			AddPropValue(node, strCategory, "ModelName", modelName, "edit");

		//if (bHiresSkeleton && modelName2.Length() > 0)
			AddPropValue(node, strCategory, "ModelName2", modelName2, "edit");

		//if (skelName.Length() > 0)
		{
			AddPropValue(node, strCategory, "SkeletonName", skelName, "edit");

			if (bHiresSkeleton)
				AddPropValue(node, strCategory, "HiResSkeleton", "TRUE", "check");
			else
				AddPropValue(node, strCategory, "HiResSkeleton", "FALSE", "check");
		}

		if (bExportFromPelvis)
			AddPropValue(node, strCategory, "ExportFromPelvis", "TRUE", "check");
		else
			AddPropValue(node, strCategory, "ExportFromPelvis", "FALSE", "check");

		//if (skelName2.Length() > 0)
			AddPropValue(node, strCategory, "SkeletonName2", skelName2, "edit");

		//if (rootBone.Length() > 0)
			AddPropValue(node, strCategory, "rootBone", rootBone, "node");
		
		AddPropValue(node, strCategory, "partialAnimSet", partialAnimSet, "edit");

		// Node tolerance data is stored in the node directly
		int listSize = ListSize(&boneTolList);
		unsigned char* pData = (unsigned char*)malloc(listSize);

		node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_PERBONE_TOLERANCES);
		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_PERBONE_TOLERANCES, listSize, pData);

		WriteList<PerBoneTol>(&pData, &boneTolList);
	}
}

DWORD CutsceneObj::GetTaggedFlags(INode* node)
{
	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);

	if (appdata && appdata->data)
	{
		DWORD exptype = *((ExportType*)appdata->data);

		// Remove legacy flags
		exptype &= ~OBJTYPE_EXTERNAL;
		//exptype &= ~OBJTYPE_OBJANIM;		// OBJANIM is no longer a legacy flag
		exptype &= ~OBJTYPE_BONEDANIM;

		return exptype;	
	}

	return 0;
}

void CutsceneObj::SetTaggedFlags(INode* node)
{
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
	DWORD* pFlags = (DWORD*)malloc(sizeof(DWORD));
	*pFlags = type;

	node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pFlags);
}

int  Cutscene::GetSize()
{
	int size = sizeof(DWORD) +
		       GetStringSize(name) +
		       ListSize(&objDB) +
		       sizeof(float) +
		       sizeof(float);

	if (version >= 3)
		size += ListSize(&trackKeyDB);

	if (version >= 4)
		size += ListSize(&nodeNameDB);

	return size;
}

void Cutscene::Store(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	//*((DWORD*)pos) = version;
	*((DWORD*)pos) = CUTSCENEDATA_VERSION;
	pos += sizeof(DWORD);
	WriteString(&pos, name);
	WriteList<CutsceneObj>(&pos, &objDB);
	*((float*)pos) = rotTol;
	pos += sizeof(float);
	*((float*)pos) = tranTol;
	pos += sizeof(float);

	// Always write most up-to-date version
	//if (version >= 3)
	WriteList<TrackUIKey>(&pos, &trackKeyDB);

	// v4
	WriteList<SaveableStr>(&pos, &nodeNameDB);
}

void Cutscene::Retrieve(void* data)
{
	unsigned char* pos = (unsigned char*)data;

	version = *((DWORD*)pos);
	pos += sizeof(DWORD);
	GetString(&pos, name);
	GetList<CutsceneObj>(&pos, &objDB);
	rotTol = *((float*)pos);
	pos += sizeof(float);
	tranTol = *((float*)pos);
	pos += sizeof(float);

	if (version >= 3)
		GetList<TrackUIKey>(&pos, &trackKeyDB);
	else
		trackKeyDB.Clear();

	if (version >= 4)
		GetList<SaveableStr>(&pos, &nodeNameDB);
	else
		nodeNameDB.Clear();
}

void Cutscene::StoreObjects()
{
	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
		link->data.SaveToNode(name);
		link = link->next;
	}
}

void Cutscene::RetrieveObjects()
{
	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
//////////
		CutsceneObj csoDefault;
		INode*      node     = gInterface->GetINodeByName(link->data.name);
		Interval    interval = gInterface->GetAnimRange();

		// Assign model defaults
		csoDefault.start      = interval.Start() / GetTicksPerFrame();
		csoDefault.end        = interval.End() / GetTicksPerFrame();
		csoDefault.rotTol     = 1.0f;
		csoDefault.tranTol    = 0.0f;
		csoDefault.modelName  = node->GetName();
		csoDefault.modelName2 = "";
		csoDefault.handle     = node->GetHandle();
		csoDefault.skelName   = "THPS5_human";
/////////

		link->data.LoadFromNode(name, &csoDefault);
		link = link->next;
	}
}

bool Cutscene::ModelExists(CutsceneObj* ignore, CStr modelName)
{
	Link<CutsceneObj>* link = objDB.GetHead();

	CStr modelNameLC = modelName;
	modelNameLC.toLower();

	while(link)
	{
		if (link->data.name == modelName)
			return true;

		if (&link->data != ignore)
		{
			// Check case insensitive
			CStr modelNameLC1 = link->data.modelName;
			modelNameLC1.toLower();

			if (modelNameLC == modelNameLC1)
				return true;
		}

		link = link->next;
	}
	
	return false;
}

bool Cutscene::VerifyModelNames(HWND hwnd)
{
	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
		// Check validity of modelName field
		if (!ModelExists(&link->data, link->data.modelName))
		{
			// If the modelname doesn't exist within the cutscene then it must contain
			// .mdl if its a model or .skin if its a skin
			char modelname[256];
			strcpy(modelname, link->data.modelName);
			_strlwr(modelname);

			if (link->data.type & OBJTYPE_MODEL)
			{
				if (!strstr(link->data.modelName, ".mdl"))
				{
					char strErr[1024];
					sprintf(strErr, "WARNING!  The model object '%s' makes reference to a model '%s' that isn't in the scene (external ref) and doesn't contain the .mdl extension", (char*)link->data.name, (char*)link->data.modelName);
					MessageBox(hwnd, strErr, "External ref contains no .mdl extension", MB_ICONWARNING|MB_OK);
					return false;
				}
			}

			if (link->data.type & OBJTYPE_SKIN)
			{
				if (!strstr(link->data.modelName, ".skin"))
				{
					char strErr[1024];
					sprintf(strErr, "WARNING!  The skin object '%s' makes reference to a skin '%s' that isn't in the scene (external ref) and doesn't contain the .skin extension", (char*)link->data.name, (char*)link->data.modelName);
					MessageBox(hwnd, strErr, "External ref contains no .skin extension", MB_ICONWARNING|MB_OK);
					return false;
				}
			}
		}

		// Check validity of modelName2 field
		if (link->data.modelName2.Length() > 0 &&
			!ModelExists(&link->data, link->data.modelName2))
		{
			// If the modelname doesn't exist within the cutscene then it must contain
			// .mdl if its a model or .skin if its a skin
			char modelname2[256];
			strcpy(modelname2, link->data.modelName2);
			_strlwr(modelname2);

			if (link->data.type & OBJTYPE_MODEL)
			{
				if (!strstr(link->data.modelName2, ".mdl"))
				{
					char strErr[1024];
					sprintf(strErr, "WARNING!  The model object '%s' makes reference to a model '%s' that isn't in the scene (external ref) and doesn't contain the .mdl extension", (char*)link->data.name, (char*)link->data.modelName2);
					MessageBox(hwnd, strErr, "External ref contains no .mdl extension", MB_ICONWARNING|MB_OK);
					return false;
				}
			}

			if (link->data.type & OBJTYPE_SKIN)
			{
				if (!strstr(link->data.modelName2, ".skin"))
				{
					char strErr[1024];
					sprintf(strErr, "WARNING!  The skin object '%s' makes reference to a skin '%s' that isn't in the scene (external ref) and doesn't contain the .skin extension", (char*)link->data.name, (char*)link->data.modelName2);
					MessageBox(hwnd, strErr, "External ref contains no .skin extension", MB_ICONWARNING|MB_OK);
					return false;
				}
			}
		}

		link = link->next;
	}

	return true;
}

bool Cutscene::VerifyRootBones(HWND hwnd)
{
	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_SKIN)
		{
			if (link->data.rootBone.Length() > 0)
			{
				INode* node = gInterface->GetINodeByName(link->data.rootBone);
				
				if (!node)
				{
					char strErr[256];
					sprintf(strErr, "The cutscene '%s' uses a root bone '%s' that doesn't exist in object '%s'.", (char*)name, (char*)link->data.rootBone, (char*)link->data.name);
					MessageBox(hwnd, strErr, "Cutscene Export Failed!", MB_ICONWARNING|MB_OK);
					return false;
				}

				// Correct any case-sensitivity issues
				link->data.rootBone = node->GetName();

				// Ensure that the root bone is part of a bone hierarchy
				if (node->NumberOfChildren() == 0)
				{
					char buf[1024];
					sprintf(buf, "WARNING!  You've chosen a rootbone '%s' that has no children for skinned object '%s'.\nSkinned objects should have a hierarchy of bones.  You may want to export this as a model instead.", (char*)link->data.rootBone, (char*)link->data.name);
					MessageBox(hwnd, buf, "Missing bone hierarchy", MB_ICONWARNING|MB_OK);
					return false;
				}
			}
		}

		link = link->next;
	}

	return true;
}

void Cutscene::BuildObjAnimDB()
{
	// This function scans through the user object database and builds a new database
	// containing the unique object animations that should be exported
	Link<CutsceneObj>* newLink;
	objAnimDB.Clear();

	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_MODEL   ||
			link->data.type & OBJTYPE_SKIN    ||
			link->data.type & OBJTYPE_OBJANIM ||
			link->data.bUseSkaterModel)
		{
			// Use all data from the skin
			CutsceneObj cso = link->data;
			cso.type = OBJTYPE_OBJANIM;
	
			// But, the root bone should be used for the object name
			if (link->data.rootBone.Length() > 0)
			{
				INode* rootBone;
				
				rootBone   = gInterface->GetINodeByName(link->data.rootBone);

				CStr nodeName = gInterface->GetINodeByName(link->data.name)->GetName();;
				CStr rootName = rootBone->GetName();

				if (!rootBone)
				{
					char strErr[256];
					sprintf(strErr, "WARNING!  Failed to find rootbone '%s' for object '%s'.", (char*)link->data.rootBone, (char*)link->data.name);
					MessageBox(gInterface->GetMAXHWnd(), strErr, "Cutscene exporter", MB_ICONWARNING|MB_OK);
					continue;
				}

				cso.name   = rootBone->GetName();
				cso.handle = rootBone->GetHandle();
			}

			newLink = objAnimDB.AddToTail(&cso);		// Note: NOT UNIQUE!  Dupes may occur and this is what
														// the engine currently expects.  Garuntees back pointer
														// pOrigObj will always be correct

			link->data.pObjAnim = &newLink->data;
			newLink->data.pOrigObj = &link->data;
		}

		link = link->next;
	}
}

void Cutscene::BuildBonedAnimDB()
{
	// This function scans through the user object database and builds a new database
	// containing the unique boned animations that should be exported
	Link<CutsceneObj> *newLink, *copyLink;
	bonedAnimDB.Clear();

	Link<CutsceneObj>* link = objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_SKIN /*&&
			!link->data.bUseSkaterModel*/)
		{
			// If the object is tagged as a hi-res skeleton an additional
			// copy should be added.   The copy will be the head
			// (Note: Copy needs to come before original in list)
			CutsceneObj cso = link->data;
			cso.type     = OBJTYPE_BONEDANIM;
			cso.debugStr = link->data.name;

			// But, the root bone should be used for the object name
			if (link->data.rootBone.Length() > 0)
			{
				INode* rootBone;
				
				rootBone   = gInterface->GetINodeByName(link->data.rootBone);
				cso.name   = rootBone->GetName();
				cso.handle = rootBone->GetHandle();
			}

			if (link->data.bHiresSkeleton)
			{
				cso.bCopy = true;
				copyLink  = bonedAnimDB.AddToTailFindUnique(&cso);
				copyLink->data.pOrigObj = &link->data;
			}

			cso.bCopy = false;
			
			bool bExists;
			if (bonedAnimDB.Find(&cso))
				bExists = true;
			else
				bExists = false;

			newLink = bonedAnimDB.AddToTailFindUnique(&cso);
			link->data.pBonedAnim = &newLink->data;

			// If we copied the object, then the bonedAnim object needs to
			// reference the original boned anim in the list and vice-versa
			if (link->data.bHiresSkeleton)
			{
				copyLink->data.pBonedAnim = &newLink->data;
				newLink->data.pBonedAnim  = &copyLink->data;
			}

			// Note: this back pointer will always reference only the last object
			//       changed in the event multiple references are made to the same boned anim (.ska)
			newLink->data.pOrigObj = &link->data;
		}

		link = link->next;
	}
}

void CutsceneExportDlg::ctKeyAdded(TrackUI* pTrackUI, Link<TrackUIKey>* linkKey, void* pData)
{
	//linkKey->data.userBuffer = "";
	//linkKey->data.userData = SCRIPTKEY_SCRIPT;
}

void CutsceneExportDlg::ctKeyRemoved(TrackUI* pTrackUI, Link<TrackUIKey>* linkKey, void* pData)
{

}

void CutsceneExportDlg::ctKeyChanged(TrackUI* pTrackUI, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	// Update the current cutscene's global track data to be in sync with the TrackUI control
	int idx = SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETCURSEL, 0, 0);
	if (idx == LB_ERR)
		return;

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)idx, 0);
	if ((INT)link == LB_ERR)
		return;

	// Set the cameras in the current cutscene to modified
	pthis->SetCamerasModified(&link->data);

	link->data.trackKeyDB = *pTrackUI->GetKeyList();
}

void CutsceneExportDlg::SetCamerasModified(Cutscene* cutscene)
{
	Link<Cutscene>* link = cutsceneDB.GetHead();
	bool bNeedsRefresh = false;

	Link<CutsceneObj>* linkCSO = cutscene->objDB.GetHead();

	while(linkCSO)
	{
		if (linkCSO->data.type & OBJTYPE_CAMERA)
		{
			if (!linkCSO->data.bModified)
			{
				linkCSO->data.bModified = true;
				bNeedsRefresh = true;
			}
		}

		linkCSO = linkCSO->next;
	}

	if (bNeedsRefresh)
	{
		InitObjList(displayMode);
	}
}

CutsceneExportDlg::CutsceneExportDlg(HINSTANCE hInstance, HWND hwndParent) :
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_CUTSCENEEXPORT), hwndParent)
{
	ip = gInterface;

	// Setup paths
	char* dir = getenv(APP_ENV);
	if (dir == NULL)
		rootDir = "C:\\";
	else
	{
		rootDir = dir;
		rootDir += MODEL_PATH;
	}

	// Register notifications so we can force the cutscene exporter to close
	// if the user does something like resetting the scene or loading a new one
#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(CloseDlg,this,NOTIFY_FILE_PRE_OPEN);
	RegisterNotification(CloseDlg,this,NOTIFY_SYSTEM_PRE_RESET);
	RegisterNotification(CloseDlgSave,this,NOTIFY_FILE_PRE_SAVE);
#endif

	configPath = gInterface->GetDir(APP_PLUGCFG_DIR);
	configPath += "\\";

	pEditFind    = GetICustEdit(GetDlgItem(hwnd, IDC_EDITFIND));
	pEditFindScn = GetICustEdit(GetDlgItem(hwnd, IDC_EDITFINDCUTSCN));
	pCameraTrack = new TrackUI(hInstance);

	//pCameraTrack->SetKeySaveChunk(vNAPP_CAMERA_TRACK_DATA);
	//pCameraTrack->SetTrackSaveAnimatable(ip->GetRootNode());

	pCameraTrack->Attach(GetDlgItem(hwnd, IDC_CAMERATRACK));
	pCameraTrack->RegisterTimeChangeCB(TimeChangeCB, this);

	// Setup key notifications
	pCameraTrack->SetKeyAddedCB(ctKeyAdded, this);
	pCameraTrack->SetRemoveKeyCB(ctKeyRemoved, this);
	pCameraTrack->SetKeyChangedCB(ctKeyChanged, this);

	TrackUIKey tui;
	tui.time              = 50 * GetTicksPerFrame();
	//tui.userData          = (DWORD)this;
	tui.userData          = SCRIPTKEY_SCRIPT;
	tui.name              = "New Script Key";
	tui.cDisplayColor     = RGB(255,0,0);
	tui.cHighlightColor   = RGB(0,255,0);
	tui.pInternalUserData = NULL;
	//tui.ActionCB        = ScriptKeyUI;

	pCameraTrack->SetDefaultAddKey(&tui);
	pCameraTrack->SetKeyActionCB(ScriptKeyUI, this);

	pObjList = new ListView(GetDlgItem(hwnd, IDC_OBJLIST));

	pObjList->AddColumn("Name");
	pObjList->AddColumn("Type");
	pObjList->AddColumn("Cutscene");
	pObjList->AddColumn("Mod", 20);

	displayMode  = DISP_ALL;
	propViewMode = PROPVIEW_UNASSIGNED;

	bLockPropChange = false;
	bLockExportModeChange = false;

	Interval animRange = ip->GetAnimRange();

	propList = new PropList(hInstance,2);
	propList->Attach(GetDlgItem(hwnd, IDC_PARAMS));
	propList->HasApply(false);
	propList->SetChangeCB(PropListChangeCB, this);

	scriptKeyEditor = new ScriptKeyEditor(hInstance, hwnd);
	partialAnimEditor = new PartialAnimEditor(hInstance, hwnd);
	partialAnimEditor->LoadAnimSets();

	PopulatePartialAnimList();

	//CheckDlgButton(hwnd, IDC_DISPALL, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_DISPSCENE, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_VIEWCUTSCENE, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_EXPSEL, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_RUNLIBCONV, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_PROCVIS, BST_CHECKED);
	CheckDlgButton(hwnd, IDC_PERBONEVIS, BST_UNCHECKED);
	CheckDlgButton(hwnd, IDC_EXPORTFROMPELVIS, BST_CHECKED);

	// Add Export types
	/*
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"Normal");
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"ObjectAnim");
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"Model");
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"Skin");
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"BonedAnim");
	SendDlgItemMessage(hwnd, IDC_EXPORTMODE, CB_ADDSTRING, 0, (LPARAM)"Camera");
	*/

	CameraExporter::bFixReverseKeyInterpolation = true;
	ObjectExporter::bFixReverseKeyInterpolation = true;

	linkActiveCutscene = NULL;
	pObjKeyEditor = NULL;
	pQNEditor = NULL;
	bLoaded = false;

	LoadFromMAX();
	CreateCutsceneList();

	displayMode = DISP_CUTSCENE;
	InitObjList(displayMode);

	RetrieveAllProps();
	PropertyUpdate();
}

CutsceneExportDlg::~CutsceneExportDlg()
{
	// Unregister close notifications
	UnRegisterNotification(CloseDlg,this,NOTIFY_FILE_PRE_OPEN);
	UnRegisterNotification(CloseDlg,this,NOTIFY_SYSTEM_PRE_RESET);
	UnRegisterNotification(CloseDlgSave,this,NOTIFY_FILE_PRE_SAVE);

	ReleaseICustEdit(pEditFind);

	if (pQNEditor)
		delete pQNEditor;

	delete pCameraTrack;
	delete pObjList;
	delete propList;
	delete scriptKeyEditor;
	delete partialAnimEditor;
}

void CutsceneExportDlg::CloseDlg(void *param, NotifyInfo *info)
{
	OutputDebugString("HANDLER: CloseDlg\n");
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)param;

	pthis->HideNoSave();

#ifdef DEBUG_CUTSCENEDB
	MessageBox(gInterface->GetMAXHWnd(), "CloseDlg Notification Called\nCutsceneDB Cleared\nMake NOTE!!", "Debug tracking for Cutscene DB Clears", MB_OK);
#endif

	pthis->bLoaded = false;
	pthis->cutsceneDB.Clear();
}

void CutsceneExportDlg::CloseDlgSave(void *param, NotifyInfo *info)
{
	OutputDebugString("HANDLER: CloseDlgSave\n");
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)param;

	if (pthis->bLoaded)
		pthis->SaveToMAX();

	/*
	if (pthis->cutsceneDB.GetSize() > 0)
		pthis->Hide();
	else
		pthis->HideNoSave();
	*/

#ifdef DEBUG_CUTSCENEDB
	MessageBox(gInterface->GetMAXHWnd(), "CloseDlgSave Notification Called\nCutsceneDB Saved/Cleared\nMake NOTE!!", "Debug tracking for Cutscene DB Clears", MB_OK);
#endif

	pthis->bLoaded = false;
	//pthis->cutsceneDB.Clear();
}

void CutsceneExportDlg::UpdateKeyData()
{
	int idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETCURSEL, 0, 0);

	if (idx < 0)
		return;

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)idx, 0);
	if ((INT)link == LB_ERR)
		return;
	
	pCameraTrack->CreateKeyList(&link->data.trackKeyDB);
	scriptKeyEditor->BuildKeyList(pCameraTrack);
}

void CutsceneExportDlg::Show()
{
	LoadFromMAX();
	CreateCutsceneList();
	InitObjList();
	RetrieveAllProps();
	
	// Load key data for the selected cutscene
	//pCameraTrack->LoadKeyData();
	UpdateKeyData();
	MSDlgWindow::Show();
}

void CutsceneExportDlg::Hide()
{
	SaveToMAX();
	Save("c:\\autoback.cdb");
	MSDlgWindow::Hide();
}

void CutsceneExportDlg::HideNoSave()
{
	MSDlgWindow::Hide();
}

void CutsceneExportDlg::Refresh()
{
	InitObjList(displayMode);
}

void CutsceneExportDlg::InitObjList(int selFlags)
{
	// Add all the objects in the scene to the list
	pObjList->Clear();

	switch(selFlags)
	{
	case DISP_ALL:
		// Display all objects in scene
		AddObjListNodesAll();
		break;

	case DISP_CUTSCENE:
		// Display only objects in cutscene
		AddObjListNodesCutscene();
		break;

	case DISP_UNASSIGNED_TO_CUTSCENE:
		// Display only objects not in current cutscene
		AddObjListNodesUnassignedToCutscene();
		break;

	case DISP_UNASSIGNED_TO_ANY:
		// Display all objects not assigned to any cutscene
		AddObjListNodesUnassignedToAnyCutscene();
		break;
	}

	AddCamsToTrack();
}

CStr CutsceneExportDlg::BuildFlagString(DWORD flags)
{
	CStr strType;

	if (flags & OBJTYPE_EXTERNAL)
		strType += "E";

	if (flags & OBJTYPE_OBJANIM)
		strType += "O";
	
	if (flags & OBJTYPE_MODEL)
		strType += "Model";

	if (flags & OBJTYPE_SKIN)
		strType += "Skin";

	if (flags & OBJTYPE_BONEDANIM)
		strType += "B";

	if (flags & OBJTYPE_CAMERA)
		strType += "Camera";

	if (flags & OBJTYPE_WORLDSPACE)
		strType += "W";

	return strType;	
}

DWORD CutsceneExportDlg::DetermineType(INode* node)
{
	// This will determine if the given node is a model, skin, or particle system
	Object* obj = node->EvalWorldState(0).obj;

	if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
		return OBJTYPE_OBJANIM | OBJTYPE_WORLDSPACE;

	if (FindPhysiqueModifier(node))
		return OBJTYPE_SKIN;

	return OBJTYPE_MODEL;
}

bool CutsceneExportDlg::HasExportModeData(INode* node)
{
	// Cameras are always considered as if they have exportmode data
	if (node->EvalWorldState(0).obj->SuperClassID() == CAMERA_CLASS_ID)
		return true;

	AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);

	if (appdata && appdata->data)
	{
		if (*((DWORD*)appdata->data) != 0)
			return true;
	}
	else
	{
		// If the object doesn't have any tag info but, does contain a tag prefix in its name
		// then appropriate tag data should be added
		CStr LCname = node->GetName();
		LCname.toLower();

		// Mark exportmode data by name
		if (strstr(LCname, "skin_") == (char*)LCname)
		{
			node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
			DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
			//*pMode = OBJTYPE_SKIN | OBJTYPE_OBJANIM | OBJTYPE_BONEDANIM;
			*pMode = OBJTYPE_SKIN | OBJTYPE_OBJANIM;
			node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);
			return true;
		}

		if (strstr(LCname, "model_") == (char*)LCname)
		{
			node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
			DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
			*pMode = OBJTYPE_MODEL | OBJTYPE_OBJANIM;
			//*pMode = OBJTYPE_MODEL;
			node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);
			return true;
		}

		if (strstr(LCname, "camera_") == (char*)LCname)
		{
			node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
			DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
			*pMode = OBJTYPE_CAMERA;
			node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);
			return true;
		}
	}

	return false;
}

void CutsceneExportDlg::AddObjNode(INode* node, char* name)
{
	// If we're only adding tagged objects others should be thrown out
	if (IsDlgButtonChecked(hwnd, IDC_SHOWONLYTAGGED) == BST_CHECKED &&
		!HasExportModeData(node))
		return;

	int idx = pObjList->AddItem(node->GetName());

	Object* obj = node->EvalWorldState(0).obj;

	//if (obj->SuperClassID() == CAMERA_CLASS_ID)
	//	pObjList->AddSubItem("Camera");
	//else
	{
		// If we're in cutscene list view mode and there's only one cutscene
		// selected then we can display the actual export mode details
		if (IsDlgButtonChecked(hwnd, IDC_DISPSCENE) == BST_CHECKED)
		{
			int nSel = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

			if (nSel == 1)
			{
				int item;
				SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

				Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd,
					                                                               IDC_CUTSCNLIST,
																				   LB_GETITEMDATA,
																				   (WPARAM)item,
																				   0);

				if (linkCutscene)
				{
					Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);

					if (linkObj)
					{
						//linkObj->data.type = linkObj->data.GetTaggedFlags(node);
						//linkObj->data.type = DetermineType(node);
	
						if (obj->SuperClassID() == CAMERA_CLASS_ID)
							pObjList->AddSubItem("Camera");
						else
							pObjList->AddSubItem(BuildFlagString(linkObj->data.type));

						// Add the modified state flag
						if (linkObj->data.bModified)
							pObjList->AddSubItem("M",3);
						else
							pObjList->AddSubItem("_",3);
					}
					else
					{
						//pObjList->AddSubItem(BuildFlagString(CutsceneObj::GetTaggedFlags(node)));	
						pObjList->AddSubItem(BuildFlagString(DetermineType(node)));	
					}
				}
			}
		}
		else
		{
			//pObjList->AddSubItem(BuildFlagString(CutsceneObj::GetTaggedFlags(node)));
			pObjList->AddSubItem(BuildFlagString(DetermineType(node)));
		}
	}

	pObjList->AddSubItem(name);

	pObjList->SetItemData(idx, (DWORD)node);
}

void CutsceneExportDlg::AddObjListNodesAll(INode* node)
{
	if (node == NULL)
		node = ip->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		AddObjListNodesAll(child);
	}

	if (node != ip->GetRootNode())
		AddObjNode(node);
}

void CutsceneExportDlg::AddObjListNodesCutscene()
{
	int items[256];
	int nSelCutscenes = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);

	for(int i = 0; i < nSelCutscenes; i++)
	{
		Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd,
			                                                               IDC_CUTSCNLIST,
																		   LB_GETITEMDATA,
																		   (WPARAM)items[i],
																		   0);

		if (linkCutscene)
		{
			// Add the objects in the cutscene database to the list
			Link<CutsceneObj>* curObj = linkCutscene->data.objDB.GetHead();

			while(curObj)
			{
				Link<CutsceneObj>* nextObj = curObj->next;

				// Verify that the objects actually exist in the scene
				//INode* node = ip->GetINodeByHandle(curObj->data.handle);
				INode* node = ip->GetINodeByName(curObj->data.name);

				if (node)
				{
					// The node does exist in the scene add it to the list
					AddObjNode(node, linkCutscene->data.name);
				}
				else
				{
					// The node doesn't exist in the scene, remove it from the DB
					linkCutscene->data.objDB.Remove(curObj);
				}

				curObj = nextObj;
			}
		}
	}
}

void CutsceneExportDlg::AddObjListNodesUnassignedToCutscene(INode* node)
{
	if (node == NULL)
		node = ip->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		AddObjListNodesUnassignedToCutscene(child);
	}

	if (node != ip->GetRootNode())
	{
		// Determine if this node exists in any of the currently selected cutscenes
		int items[256];
		int nSelCutscenes = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);

		bool bNodeExists = false;
		for(int i = 0; i < nSelCutscenes; i++)
		{
			Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd,
				                                                               IDC_CUTSCNLIST, 
																			   LB_GETITEMDATA,
																			   (WPARAM)items[i],
																			   0);

			if (linkCutscene)
			{
				Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);

				if (linkObj)
				{
					bNodeExists = true;
					break;
				}
			}
		}

		if (!bNodeExists)
			AddObjNode(node);
	}
}

void CutsceneExportDlg::AddObjListNodesUnassignedToAnyCutscene(INode* node)
{
	if (node == NULL)
		node = ip->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		AddObjListNodesUnassignedToAnyCutscene(child);
	}

	if (node != ip->GetRootNode() &&
		!GetObjScene(node))
		AddObjNode(node);
}


BOOL CutsceneExportDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case TRACKUI_UPDATE:
		TrackUpdate((HWND)lParam);
		PropertyUpdate();
		break;

	case WM_NOTIFY:
		{
			NMHDR* nmhdr = (NMHDR*)lParam;

			switch(nmhdr->code)
			{
			case LVN_ITEMCHANGED:
				{
					PropertyUpdate();
					UpdateExportMode();
				}
				return TRUE;

			case LVN_ITEMACTIVATE:
   				EditObjKeys();
				return TRUE;
			}
		}
		break;

	case WM_COMMAND:

		switch(HIWORD(wParam))
		{
		case LBN_SELCHANGE:		// LBN_SELCHANGE == CBN_SELCHANGE
			{
				switch(LOWORD(wParam))
				{
				case IDC_CUTSCNLIST:
					{
						//if (linkActiveCutscene)
						//	linkActiveCutscene->data.trackKeyDB = *pCameraTrack->GetKeyList();

						UpdateKeyData();

						InitObjList(displayMode);
						PropertyUpdate();
						UpdateExportMode();

						SetActiveCutscene();
					}
					return TRUE;

				case IDC_PARTIALANIMLIST:
					PartialAnimChanged();
					return TRUE;
				}
			}
			return TRUE;

		case LBN_DBLCLK:
			{
				switch(LOWORD(wParam))
				{
				case IDC_CUTSCNLIST:
					RenameCutscene();
					return TRUE;
				}
			}
			return TRUE;

		case EN_CHANGE:
			{
				switch(LOWORD(wParam))
				{
				case IDC_EDITFINDCUTSCN:
					FindCutscene();
					return TRUE;

				case IDC_EDITFIND:
					FindObject();
					return TRUE;
				}

				break;
			}
		}

		switch(LOWORD(wParam))
		{
		case IDC_EM_E:
			if (IsDlgButtonChecked(hwnd, IDC_EM_E) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_E, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_O:
			if (IsDlgButtonChecked(hwnd, IDC_EM_O) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_O, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_M:
			if (IsDlgButtonChecked(hwnd, IDC_EM_M) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_M, BST_UNCHECKED);

			CheckDlgButton(hwnd, IDC_EM_S, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_S:
			if (IsDlgButtonChecked(hwnd, IDC_EM_S) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_S, BST_UNCHECKED);

			CheckDlgButton(hwnd, IDC_EM_M, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_B:
			if (IsDlgButtonChecked(hwnd, IDC_EM_B) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_B, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_C:
			if (IsDlgButtonChecked(hwnd, IDC_EM_C) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_C, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_EM_WORLDSPACE:
			if (IsDlgButtonChecked(hwnd, IDC_EM_WORLDSPACE) == BST_INDETERMINATE)
				CheckDlgButton(hwnd, IDC_EM_WORLDSPACE, BST_UNCHECKED);

			ExportModeChange();
			return TRUE;

		case IDC_SETMOD:
			SetModified(true);
			return TRUE;

		case IDC_SETUNMOD:
			SetModified(false);
			return TRUE;

		case IDC_VIEWCUTSCENE:
			PropertyUpdate();
			return TRUE;

		case IDC_VIEWOBJECT:
			PropertyUpdate();
			return TRUE;

		case IDC_ADDCUTSCN:
			AddCutscene();
			return TRUE;

		case IDC_REMOVECUTSCN:
			RemoveCutscene();
			return TRUE;

		case IDC_ADDOBJ:
			AddCutsceneObj();
			InitObjList(displayMode);
			return TRUE;

		case IDC_DELOBJ:
			RemoveCutsceneObj();
			InitObjList(displayMode);
			return TRUE;

		case IDC_SELOBJ:
			SelCutsceneObj();
			return TRUE;

		case IDC_GETSEL:
			GetCutsceneObjSel();
			return TRUE;

		// Selection modes
		case IDC_DISPALL:
			displayMode = DISP_ALL;
			InitObjList(displayMode);
			return TRUE;

		case IDC_DISPSCENE:
			displayMode = DISP_CUTSCENE;
			InitObjList(displayMode);
			return TRUE;

		case IDC_DISPUNASSIGNED:
			displayMode = DISP_UNASSIGNED_TO_CUTSCENE;
			InitObjList(displayMode);
			return TRUE;

		case IDC_DISPUNASSIGNEDANY:
			displayMode = DISP_UNASSIGNED_TO_ANY;
			InitObjList(displayMode);
			return TRUE;

		case IDC_SHOWONLYTAGGED:
			InitObjList(displayMode);
			return TRUE;

		case IDC_EXPORT:
			DoExport();
			return TRUE;

		case IDC_EXPOPTIONS:
			ExportOptions();
			return TRUE;

		case IDC_TOLHELP:
			DisplayHelp();
			return TRUE;

		case IDC_SYNCTIMESTOCAM:
			SyncTimesToCam();
			return TRUE;
			
		case IDC_SYNCTIMESTOCAMSEL:
			SyncTimesToCamSel();
			return TRUE;

		case IDC_QNEDITOR:
			OpenQNEditor();
			return TRUE;

		case IDC_PARTIALANIMS:
			{	
				partialAnimEditor->Show();
				PopulatePartialAnimList();
			}
			return TRUE;

		case IDC_LOADCS:
			Load();
			return TRUE;

		case IDC_SAVECS:
			Save();
			return TRUE;

		case IDC_FLAGAUTOASSIGN:
			FlagAutoAssign();
			return TRUE;
		}
		break;

	case WM_CLOSE:
		Hide();
		return TRUE;
	}

	return FALSE;
}

void CutsceneExportDlg::AddCutscene()
{
	char cutsceneName[256];
	pEditFindScn->GetText(cutsceneName, 255);

	if (strlen(cutsceneName) == 0)
	{
		InputDlg* inpdlg = new InputDlg(hInstance, hwnd,
										"Cutscene",
										"",
										"Enter Cutscene Name");
										
		inpdlg->Show();
		CStr strCutscene = inpdlg->GetInput();

		if (!inpdlg->WasCancelled())
		{
			Cutscene newCutscene;
			newCutscene.name = strCutscene;

			Link<Cutscene>* link = cutsceneDB.Add(&newCutscene);
			int idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_ADDSTRING, (WPARAM)0, (LPARAM)(char*)strCutscene);
			SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)link);
		}

		delete inpdlg;
	}
	else
	{
		Cutscene newCutscene;
		newCutscene.name = cutsceneName;

		Link<Cutscene>* link = cutsceneDB.Add(&newCutscene);
		int idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_ADDSTRING, (WPARAM)0, (LPARAM)cutsceneName);
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)link);	
		
		pEditFindScn->SetText("");
	}
}

void CutsceneExportDlg::RemoveCutscene()
{
	int item;
	int nSelItems;

	nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, 1, (LPARAM)&item);

	while(nSelItems > 0)
	{
		Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

		if (link)
			cutsceneDB.Remove(link);

		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_DELETESTRING, (WPARAM)item, 0);

		nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, 1, (LPARAM)&item);
	}

	// If we're in the current cutscene display mode after deleting no cutscenes will be selected
	// and thus we should clear the contents of the now out of date object list
	if (IsDlgButtonChecked(hwnd, IDC_DISPSCENE) == BST_CHECKED)
		pObjList->Clear();
}

void CutsceneExportDlg::AddCutsceneObj()
{
	int cnt = pObjList->GetCount();
	int items[256];
	int nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);

	if (nSelItems == 0)
	{
		MessageBox(hwnd, "You must select a cutscene first", "No cutscene selected", MB_ICONWARNING|MB_OK);
		return;
	}

	for(int i = 0; i < cnt; i++)
	{
		INode* node = (INode*)pObjList->GetItemData(i);

		if (node && pObjList->IsSelected(i))
		{
			// Add the node to all selected cutscenes
			for(int j = 0; j < nSelItems; j++)
			{
				Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, 
					                                                               IDC_CUTSCNLIST,
																				   LB_GETITEMDATA,
																				   (WPARAM)items[j],
																				   0);

				if (linkCutscene)
				{
					CutsceneObj cso;
					Object*     obj = node->EvalWorldState(0).obj;
					cso.bModified   = true;
					cso.name        = node->GetName();
					
					cso.type = cso.GetTaggedFlags(node);

					if (obj->SuperClassID() == CAMERA_CLASS_ID)
						cso.type = OBJTYPE_CAMERA;

					if (cso.type == 0)
					{
						cso.type = DetermineType(node);

						/*
						if (FindBoneRootBySkin(node))
							cso.type = OBJTYPE_BONEDANIM;
						else
							cso.type = OBJTYPE_MODEL;
						*/
					}

					cso.SetTaggedFlags(node);

					// Attempt to acquire cutscene object data from property editor props

					cso.start     = 0;
					cso.end       = 0;
					cso.handle    = node->GetHandle();
					cso.bModified = true;

					// Setup defaults incase the data doesn't already exist
					
					// Skin defaults are different from model defaults
					INode* rootNode = FindBoneRootBySkin(node);

					CutsceneObj csoDefault;
					Interval interval = gInterface->GetAnimRange();

					if (rootNode)
					{
						// Attempt to acquire cutscene object data from property editor props
						csoDefault.start      = interval.Start() / GetTicksPerFrame();
						csoDefault.end        = interval.End() / GetTicksPerFrame();
						csoDefault.rotTol     = 1.0f;
						csoDefault.tranTol    = 0.0f;
						csoDefault.modelName  = node->GetName();
						csoDefault.modelName2 = "";
						csoDefault.handle     = node->GetHandle();
						csoDefault.skelName   = "THPS5_human";
					}
					else
					{
						// Assign model defaults
						csoDefault.start      = interval.Start() / GetTicksPerFrame();
						csoDefault.end        = interval.End() / GetTicksPerFrame();
						csoDefault.rotTol     = 1.0f;
						csoDefault.tranTol    = 0.0f;
						csoDefault.modelName  = node->GetName();
						csoDefault.modelName2 = "";
						csoDefault.handle     = node->GetHandle();
						csoDefault.skelName   = "THPS5_human";
					}

					cso.LoadFromNode(linkCutscene->data.name, &csoDefault);
					cso.SaveToNode(linkCutscene->data.name);
					linkCutscene->data.objDB.AddUnique(&cso);
					MakeRefByID(FOREVER, 0, node);
					//linkCutscene->data.objDB.Add(&cso);
				}
			}
		}
	}
}

void CutsceneExportDlg::RemoveCutsceneObj()
{
	int cnt = pObjList->GetCount();

	for(int i = 0; i < cnt; i++)
	{
		INode* node = (INode*)pObjList->GetItemData(i);

		if (node && pObjList->IsSelected(i))
		{
			// Add the node to all selected cutscenes
			int items[256];
			int nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);

			for(int j = 0; j < nSelItems; j++)
			{
				Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, 
					                                                               IDC_CUTSCNLIST,
																				   LB_GETITEMDATA,
																				   (WPARAM)items[j],
																				   0);

				if (linkCutscene)
				{
					Link<CutsceneObj>* csoLink = GetCutsceneObj(&linkCutscene->data, node);

					if (csoLink)
						linkCutscene->data.objDB.Remove(csoLink);
				}
			}
		}
	}
}

void CutsceneExportDlg::SelCutsceneObj()
{
	int cnt = pObjList->GetCount();

	for(int i = 0; i < cnt; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);

			if (node)
			{
				ip->ClearNodeSelection();
				ip->SelectNode(node, 0);
			}
		}
	}

	ip->ForceCompleteRedraw();
}

// Highlights the nodes in the current cutscene exporter list that
// are selected in the scene
void CutsceneExportDlg::GetCutsceneObjSel()
{
	int cntObjList = pObjList->GetCount();
	
	int  cnt = ip->GetSelNodeCount();
	int  i, j;
	int  firstItem = -1;
	bool bSelected;

	for(i  = 0; i < cntObjList; i++)
	{
		INode* node = (INode*)pObjList->GetItemData(i);
		
		bSelected = false;

		for(j = 0; j < cnt; j++)
		{
			INode* selNode = ip->GetSelNode(j);

			if (node == selNode)
			{
				if (firstItem == -1)
					firstItem = i;

				pObjList->Select(i);
				bSelected = true;
				break;
			}
		}

		if (!bSelected)
			pObjList->UnSelect(i);
	}

	if (firstItem != -1)
		pObjList->ScrollToItem(firstItem);
}

Link<Cutscene>* CutsceneExportDlg::GetObjScene(INode* node)
{
	DWORD dwScanHandle = node->GetHandle();
	Link<Cutscene>* curCutscene = cutsceneDB.GetHead();

	while(curCutscene)
	{
		Link<CutsceneObj>* curObj = curCutscene->data.objDB.GetHead();

		while(curObj)
		{
			if (curObj->data.handle == dwScanHandle)
				return curCutscene;

			curObj = curObj->next;
		}

		curCutscene = curCutscene->next;
	}

	return NULL;
}

Link<CutsceneObj>* CutsceneExportDlg::GetCutsceneObj(Cutscene* cutscene, INode* node)
{
	//DWORD dwScanHandle = node->GetHandle();
	CStr name = node->GetName();
	Link<CutsceneObj>* curObj = cutscene->objDB.GetHead();

	while(curObj)
	{
		//if (curObj->data.handle == dwScanHandle)
		if (curObj->data.name == name)
			return curObj;

		curObj = curObj->next;
	}

	return NULL;
}

void CutsceneExportDlg::FindCutscene()
{
	char searchText[256];
	char itemText[256];

	pEditFindScn->GetText(searchText, 255);

	if (strlen(searchText) > 0)
		strcat(searchText, "*");

	int count = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETCOUNT, 0, 0);
	int idx = 0;

	for(int i = 0; i < count; i++)
	{
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETTEXT, (WPARAM)i, (LPARAM)itemText);

		if (MatchPattern(CStr(itemText), CStr(searchText)))
		{
			if (idx == 0)
				idx = i;

			SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETSEL, (WPARAM)TRUE, (LPARAM)i);
		}
		else
		{
			SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETSEL, (WPARAM)FALSE, (LPARAM)i);
		}
	}

	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETCARETINDEX, (WPARAM)idx, MAKELPARAM(FALSE, 0));
}

void CutsceneExportDlg::FindObject()
{
	char searchText[256];

	pEditFind->GetText(searchText, 255);

	if (searchText[0] != '\0')
		strcat(searchText, "*");

	pObjList->SelectWildcard(searchText);	
}

int CutsceneExportDlg::NumCameras(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();
	int count = 0;

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
			count++;

		link = link->next;
	}

	return count;
}

bool CutsceneExportDlg::CamsHaveTimes(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA &&
			link->data.end != 0)
			return true;

		link = link->next;
	}

	return false;
}

void CutsceneExportDlg::AddCamsToTrack()
{
	pCameraTrack->ClearBar();
	SetCameraFlags();

	// Cameras are only visible if a single cutscene is selected
	int nSel = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (nSel != 1)
		return;

	// Determine the selected item
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
	linkActiveCutscene = link;

	// Assign the initial time for the camera track grap
	pCameraTrack->SetStartTime(FindLowestCamStartFrame(&link->data));

	bool bHaveTimes = CamsHaveTimes(&link->data);
	int nCameras = NumCameras(&link->data);

	if (nCameras == 0)
		return;

	int incr = pCameraTrack->TrackLength() / nCameras;
	int endTime = 0;
	int startTime = 0;

	Link<CutsceneObj>* linkObj = link->data.objDB.GetHead();

	while(linkObj)
	{
		// If we encounter a camera add it to the track bar
		if (linkObj->data.type & OBJTYPE_CAMERA)
		{
			if (bHaveTimes)
			{
				pCameraTrack->AddItem(linkObj->data.end, linkObj->data.name, (DWORD)linkObj);
			}
			else
			{
				endTime += incr;
				pCameraTrack->AddItem(endTime, linkObj->data.name, (DWORD)linkObj);
				linkObj->data.start = startTime;
				linkObj->data.end   = endTime;
				linkObj->data.SaveToNode(linkActiveCutscene->data.name);

				startTime = endTime;
			}
		}

		linkObj = linkObj->next;
	}
}

void CutsceneExportDlg::TrackUpdate(HWND hwndTrack)
{
	if (hwndTrack == GetDlgItem(hwnd, IDC_CAMERATRACK))
	{
		// Update stored camera times
		assert(linkActiveCutscene);

		int cnt   = pCameraTrack->barlist.GetSize();
		int start = pCameraTrack->GetStartTime();//FindLowestCamStartFrame(&linkActiveCutscene->data);

		for(int i = 0; i < cnt; i++)
		{
			Link<CutsceneObj>* link = (Link<CutsceneObj>*)pCameraTrack->barlist[i].userData;
			//INode* node = ip->GetINodeByHandle(link->data.handle);
			INode* node = ip->GetINodeByName(link->data.name);

			link->data.start = start;
			link->data.end   = pCameraTrack->barlist[i].end;
			link->data.SaveToNode(linkActiveCutscene->data.name);
			start = link->data.end;
		}
	}
}

void CutsceneExportDlg::TimeChangeCB(TrackUI* trackUI, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;
	
	if (IsDlgButtonChecked(pthis->hwnd, IDC_VIEWPORTUPDATE) == BST_CHECKED)
	{
		ViewExp* viewexp = pthis->ip->GetActiveViewport();

		Link<TrackUIRange>* tuiRange = trackUI->FindTrack(pthis->ip->GetTime());

		if (!tuiRange)
			return;

		Link<CutsceneObj>* cutsceneObj = (Link<CutsceneObj>*)tuiRange->data.userData;
		//INode* node = pthis->ip->GetINodeByHandle(cutsceneObj->data.handle);
		INode* node = pthis->ip->GetINodeByName(cutsceneObj->data.name);

		if (viewexp)
			viewexp->SetViewCamera(node);
	}
}

void CutsceneExportDlg::PropertyUpdate()
{
	if (IsDlgButtonChecked(hwnd, IDC_VIEWCUTSCENE) == BST_CHECKED)
	{
		propViewMode = PROPVIEW_CUTSCENE;
		int selCount = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

		//StoreCutsceneProps();
		propList->Clear();

		if (selCount != 1)
			return;

		propList->AddSpinEdit("RotationTol",0.0f,1.0f,0.001f,"Root object rotation tolerance");
		propList->AddSpinEdit("TranslationTol",0.0f,10.0f,0.001f,"Root object translation tolerance");
		propList->BuildUI();

		RetrieveCutsceneProps();
		return;
	}

	if (IsDlgButtonChecked(hwnd, IDC_VIEWOBJECT) == BST_CHECKED)
	{
		propViewMode = PROPVIEW_OBJECT;
		propList->Clear();

		// If we're updating an object, we can only update one object's properties at a time
		// User should use prop editor if they want to manage simultaneous data
		int selCount = pObjList->GetSelCount();

		if (selCount != 1)
			return;

		// Only one cutscene may be selected if properties are to be displayed
		int cutsceneSelCount = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

		if (cutsceneSelCount != 1)
			return;

		// Acquire selected cutscene
		int item;
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);
		Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

		// Acquire selected node
		int selObj  = pObjList->GetSelItem();
		INode* node = (INode*)pObjList->GetItemData(selObj);

		Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);

		if (!linkObj)
			return;

		Interval animRange = ip->GetAnimRange();

		propList->AddSpinEdit("RotationTol",0.0f,1.0f,0.001f,"Root object rotation tolerance");
		propList->AddSpinEdit("TranslationTol",0.0f,10.0f,0.001f,"Root object translation tolerance");
		propList->AddSpinEdit("StartFrame",0.0f,999999.0f,1.0f,"Starting frame to export for this object");
		propList->AddSpinEdit("EndFrame",0.0f,999999.0f,1.0f,"Ending frame to export for this object");

		if (linkObj->data.type & OBJTYPE_MODEL ||
			linkObj->data.type & OBJTYPE_SKIN)
		{
			propList->AddCheck("UseSkaterModel", "Check this to use the skater's model");
		}

		// Determine the type of the currently selected object
		if (linkObj->data.type & OBJTYPE_OBJANIM   ||
			linkObj->data.type & OBJTYPE_BONEDANIM ||
			linkObj->data.type & OBJTYPE_MODEL     ||
			linkObj->data.type & OBJTYPE_SKIN)
		{
			//propList->AddEdit("ModelName", "Name of model to associate with this object");
			OptFile* of = propList->AddOptFile("ModelName", "Name of model to associate with this object");
			of->SetRoot(rootDir);
			of->SetCaption("Select Model file");
			of->SetFilter("Skin files (*.skin)\0*.skin\0All files (*.*)\0*.*\0\0");
			of->SetConfigFile(configPath + "ModelName.ini");
		}

		if (linkObj->data.type & OBJTYPE_SKIN ||
			/*linkObj->data.type & OBJTYPE_MODEL ||*/
			linkObj->data.type & OBJTYPE_EXTERNAL)
		{
			// Moved from OBJANIM/BONEDANIM
			propList->AddEdit("SkeletonName", "Name of skeleton to associate with this object");
			propList->AddCheck("HiResSkeleton", "Skeleton has more than 58 bones (and needs segmented accross 2 objects)");
			propList->AddCheck("ExportFromPelvis", "The object should be exported by using OBA transforms from bone_pelvis in the object as opposed to the control_root.  This would be the case if you animated the object by leaving the control root at its origin and putting keys directly on the pelvis.");
			//propList->AddEdit("ModelName2", "Name of the upper half of the model (if HiResSkeleton)");
			OptFile* of = propList->AddOptFile("ModelName2", "Name of the upper half of the model (if HiResSkeleton)");
			of->SetRoot(rootDir);
			of->SetCaption("Select Model file");
			of->SetFilter("Skin files (*.skin)\0*.skin\0All files (*.*)\0*.*\0\0");
			of->SetConfigFile(configPath + "ModelName2.ini");

			propList->AddEdit("SkeletonName2", "Name of skeleton to associate with upper half of hires skeleton");

			HWND hwndProp       = propList->GetHWndFromProp("ModelName2");
			HWND hwndStaticProp = propList->GetStaticHWndFromProp("ModelName2");
			EnableWindow(hwndProp, FALSE);
			EnableWindow(hwndStaticProp, FALSE);

			hwndProp       = propList->GetHWndFromProp("SkeletonName2");
			hwndStaticProp = propList->GetStaticHWndFromProp("SkeletonName2");
			EnableWindow(hwndProp, FALSE);
			EnableWindow(hwndStaticProp, FALSE);

			if (!(linkObj->data.type & OBJTYPE_EXTERNAL))
				propList->AddNode("rootBone", "The root bone to associate with this skin/model");
		}

		//if (linkObj->data.type & OBJTYPE_OBJANIM)
		//	propList->AddEdit("AnimName", "Name of animation script to associate with this object");

		propList->BuildUI();
		RetrieveObjectProps();
	}
}

INode* CutsceneExportDlg::FindBoneRootByModelName(Cutscene* cutscene, char* modelName)
{
	Link<CutsceneObj>* linkObj = cutscene->objDB.GetHead();
	CStr LCmodelName = modelName;
	LCmodelName.toLower();

	while(linkObj)
	{
		if (linkObj->data.type & OBJTYPE_BONEDANIM)
		{
			CStr LClinkModelName = linkObj->data.modelName;
			LClinkModelName.toLower();

			if (LClinkModelName == LCmodelName)
			{
				//INode* node = ip->GetINodeByHandle(linkObj->data.handle);
				INode* node = ip->GetINodeByName(linkObj->data.name);

				CStr name = node->GetName();

				if (node)
					return node;
			}
		}

		linkObj = linkObj->next;
	}

	return NULL;
}

void CutsceneExportDlg::StoreCutsceneProps()
{
	int selCount = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCount != 1)
		return;

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	CStr strVal;
	propList->GetValue("RotationTol", strVal);
	linkCutscene->data.rotTol = atof(strVal);

	propList->GetValue("TranslationTol", strVal);
	linkCutscene->data.tranTol = atof(strVal);
}

void CutsceneExportDlg::RetrieveCutsceneProps()
{
	int selCount = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCount != 1)
		return;

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	bLockPropChange = true;
	char val[256];
	sprintf(val, "%f", linkCutscene->data.rotTol);
	propList->SetValue("RotationTol", val);

	sprintf(val, "%f", linkCutscene->data.tranTol);
	propList->SetValue("TranslationTol", val);
	bLockPropChange = false;
}

void CutsceneExportDlg::StoreObjectProps()
{
	// Ensure that only one cutscene is selected
	int selCountCutscene = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCountCutscene != 1)
		return;

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	// Ensure that only one object within the cutscene is selected
	int selCount = pObjList->GetSelCount();

	if (selCount != 1)
		return;

	int selItem = pObjList->GetSelItem();

	INode* node = (INode*)pObjList->GetItemData(selItem);
	Link<CutsceneObj>* link = GetCutsceneObj(&linkCutscene->data, node);

	if (link)
	{
		CStr strVal;
		propList->GetValue("RotationTol", strVal);
		link->data.rotTol = atof(strVal);

		propList->GetValue("TranslationTol", strVal);
		link->data.tranTol = atof(strVal);

		propList->GetValue("StartFrame", strVal);
		link->data.start = atoi(strVal);

		propList->GetValue("EndFrame", strVal);
		link->data.end = atoi(strVal);

		propList->GetValue("UseSkaterModel", strVal);
		(strVal == CStr("TRUE")) ? link->data.bUseSkaterModel = true : link->data.bUseSkaterModel = false;

		if (!propList->GetValue("ModelName", link->data.modelName))
			link->data.modelName = "";
		
		if (!propList->GetValue("ModelName2", link->data.modelName2))
			link->data.modelName2 = "";

		if (!propList->GetValue("SkeletonName", link->data.skelName))
			link->data.skelName = "";

		if (!propList->GetValue("SkeletonName2", link->data.skelName2))
			link->data.skelName2 = "";

		if (!propList->GetValue("rootBone", link->data.rootBone))
			link->data.rootBone = "";

		if (!propList->GetValue("HiResSkeleton", strVal))
			link->data.bHiresSkeleton = false;
		else
			link->data.bHiresSkeleton = (strVal == CStr("TRUE")) ? true : false;

		if (!propList->GetValue("ExportFromPelvis", strVal))
			link->data.bExportFromPelvis = false;
		else
			link->data.bExportFromPelvis = (strVal == CStr("TRUE")) ? true : false;

		link->data.SaveToNode(linkCutscene->data.name);
	}
}

void CutsceneExportDlg::RetrieveAllProps()
{
	int nCutscenes = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETCOUNT, 0, 0);

	for(int i = 0; i < nCutscenes; i++)
	{
		Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)i, 0);
		Link<CutsceneObj>* linkCSO = linkCutscene->data.objDB.GetHead();

		while(linkCSO)
		{
			CutsceneObj csoDefault;
			INode*      node     = gInterface->GetINodeByName(linkCSO->data.name);
			Interval    interval = gInterface->GetAnimRange();

			// Assign model defaults
			if (node)
			{
				csoDefault.start      = interval.Start() / GetTicksPerFrame();
				csoDefault.end        = interval.End() / GetTicksPerFrame();
				csoDefault.rotTol     = 1.0f;
				csoDefault.tranTol    = 0.0f;
				csoDefault.modelName  = node->GetName();
				csoDefault.modelName2 = "";
				csoDefault.handle     = node->GetHandle();
				csoDefault.skelName   = "THPS5_human";
				
				linkCSO->data.LoadFromNode(linkCutscene->data.name, &csoDefault);
			}
			else
			{
				linkCSO->data.LoadFromNode(linkCutscene->data.name);
			}

			linkCSO = linkCSO->next;
		}
	}
}

void CutsceneExportDlg::RetrieveObjectProps()
{
	// Ensure that only one cutscene is selected
	int selCountCutscene = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCountCutscene != 1)
		return;

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	// Ensure that only one object within the cutscene is selected
	int selCount = pObjList->GetSelCount();

	if (selCount != 1)
		return;

	int selItem = pObjList->GetSelItem();

	INode* node = (INode*)pObjList->GetItemData(selItem);
	Link<CutsceneObj>* link = GetCutsceneObj(&linkCutscene->data, node);

	if (link)
	{
		bLockPropChange = true;
		link->data.LoadFromNode(linkCutscene->data.name);

		char val[256];
		sprintf(val, "%f", link->data.rotTol);
		propList->SetValue("RotationTol", val);
		
		sprintf(val, "%f", link->data.tranTol);
		propList->SetValue("TranslationTol", val);

		sprintf(val, "%i", link->data.start);
		propList->SetValue("StartFrame", val);
		link->data.start = atoi(val);

		sprintf(val, "%i", link->data.end);
		propList->SetValue("EndFrame", val);
		link->data.end = atoi(val);

		if (link->data.modelName == CStr("") &&
			link->data.type & OBJTYPE_MODEL)
			link->data.modelName = node->GetName();

		propList->SetValue("ModelName", link->data.modelName);
		propList->SetValue("ModelName2", link->data.modelName2);

		if (link->data.skelName == CStr("") &&
			link->data.type & OBJTYPE_BONEDANIM)
			link->data.skelName = SKELNAME1_BONED_DEFAULT;

		propList->SetValue("SkeletonName", link->data.skelName);

		if (link->data.skelName2 == CStr("") &&
			link->data.type & OBJTYPE_BONEDANIM)
			link->data.skelName2 = SKELNAME2_BONED_DEFAULT;

		propList->SetValue("SkeletonName2", link->data.skelName2);

		if (link->data.rootBone == CStr("") &&
			(link->data.type & OBJTYPE_MODEL ||
			link->data.type & OBJTYPE_SKIN))
		{
			INode* node = ip->GetINodeByName(link->data.name);
			if (node)
			{
				//CSkeletonData skel(node);
				//INode* rootBone = skel.GetRootBone(node);

				INode* rootBone = FindBoneRootBySkin(node);

				if (rootBone)
					link->data.rootBone = rootBone->GetName();
			}
		}

		propList->SetValue("rootBone", link->data.rootBone);

		HWND hwndProp           = propList->GetHWndFromProp("ModelName2");
		HWND hwndStaticProp     = propList->GetStaticHWndFromProp("ModelName2");
		HWND hwndPropSkel       = propList->GetHWndFromProp("SkeletonName2");
		HWND hwndStaticPropSkel = propList->GetStaticHWndFromProp("SkeletonName2");

		if (link->data.bHiresSkeleton)
		{
			propList->SetValue("HiResSkeleton", "TRUE");
			EnableWindow(hwndProp, TRUE);
			EnableWindow(hwndStaticProp, TRUE);
			EnableWindow(hwndPropSkel, TRUE);
			EnableWindow(hwndStaticPropSkel, TRUE);
		}
		else
		{
			propList->SetValue("HiResSkeleton", "FALSE");
			EnableWindow(hwndProp, FALSE);
			EnableWindow(hwndStaticProp, FALSE);
			EnableWindow(hwndPropSkel, FALSE);
			EnableWindow(hwndStaticPropSkel, FALSE);
		}

		if (link->data.bExportFromPelvis)
			propList->SetValue("ExportFromPelvis", "TRUE");
		else
			propList->SetValue("ExportFromPelvis", "FALSE");

		if (link->data.bUseSkaterModel)
			propList->SetValue("UseSkaterModel", "TRUE");
		else
			propList->SetValue("UseSkaterModel", "FALSE");

		link->data.SaveToNode(linkCutscene->data.name);
		bLockPropChange = false;

		// Retrieve the partial anim setting
		if (link->data.partialAnimSet.Length() > 0)
		{
			int pos = SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_FINDSTRING, (WPARAM)-1, (LPARAM)(char*)link->data.partialAnimSet);
			
			// If the item wasn't found we default to All Bones (entry 0)
			if (pos == -1)
				pos = 0;
			
			SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_SETCURSEL, (WPARAM)pos, 0);
		}
		else
		{
			// Zero length partialAnimSet indicates all bones (entry 0)
			SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_SETCURSEL, (WPARAM)0, 0);
		}
	}
}

int CutsceneExportDlg::FindLowestCamStartFrame(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();
	int  lowestStart = 0;
	bool bFirst = true;

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
		{
			if (bFirst)
			{
				lowestStart = link->data.start;
				bFirst = false;
			}

			if (link->data.start < lowestStart)
				lowestStart = link->data.start;
		}

		link = link->next;
	}

	return lowestStart;
}

void CutsceneExportDlg::PropListChangeCB(PropList* propList, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	if (pthis->bLockPropChange)
		return;

	switch(pthis->propViewMode)
	{
	case PROPVIEW_CUTSCENE:
		pthis->StoreCutsceneProps();
		break;

	case PROPVIEW_OBJECT:
		pthis->StoreObjectProps();
		break;
	}

	int lastMod = propList->GetLastMod();

	if (lastMod != -1)
	{
		CStr name = propList->GetName(lastMod);

		// Determine if the current object being modified in the cutscene exporter
		// is a camera
		int nCutscenes                  = SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);
		int nObjs                       = pthis->pObjList->GetSelCount();
		Link<CutsceneObj>* linkCSO      = NULL;
		Link<Cutscene>*    linkCutscene = NULL;
		INode*             nodeCSO      = NULL;

		if (nCutscenes == 1 && nObjs == 1)
		{
			int item;
			int selCutscene = SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);
			int selObj      = pthis->pObjList->GetSelItem();
			
			SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
			linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(pthis->hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
			nodeCSO = (INode*)pthis->pObjList->GetItemData(selObj);
			linkCSO = pthis->GetCutsceneObj(&linkCutscene->data, nodeCSO);
		}

		if ((name == CStr("StartFrame") || name == CStr("EndFrame")) &&
			linkCSO->data.type & OBJTYPE_CAMERA)
		{
			pthis->AddCamsToTrack();	
			pthis->TrackUpdate(GetDlgItem(pthis->hwnd, IDC_CAMERATRACK));
			//pthis->RetrieveObjectProps();
		}

		if (name == CStr("HiResSkeleton"))
		{
			CStr val;
			BOOL bEnable;
			propList->GetValue("HiResSkeleton", val);
			
			if (val == CStr("TRUE"))
				bEnable = TRUE;
			else
				bEnable = FALSE;

			HWND hwndProp       = propList->GetHWndFromProp("ModelName2");
			HWND hwndStaticProp = propList->GetStaticHWndFromProp("ModelName2");
			EnableWindow(hwndProp, bEnable);
			EnableWindow(hwndStaticProp, bEnable);

			hwndProp       = propList->GetHWndFromProp("SkeletonName2");
			hwndStaticProp = propList->GetStaticHWndFromProp("SkeletonName2");
			EnableWindow(hwndProp, bEnable);
			EnableWindow(hwndStaticProp, bEnable);
		}
	}
}

void CutsceneExportDlg::UpdateExportMode()
{
	// Export mode selection is always disabled if the current cutscene display mode is not selected
	if (IsDlgButtonChecked(hwnd, IDC_DISPSCENE) != BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_EXPORTMODE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_EXPOPTIONS), FALSE);
		return;
	}
	
	// Ensure that only one cutscene is selected
	int selCountCutscene = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCountCutscene != 1)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_EXPORTMODE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_EXPOPTIONS), FALSE);
		return;
	}

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	// If any of the selected objects are cameras, then changes cannot be made
	int    nCount = pObjList->GetCount();
	bool bDisable = false;
	
	// If there is only one object selected in the cutscene object list selected and that object
	// has BonedAnim export properties we can enable the export options button so that
	// the user can select per bone tolerance limits
	if (pObjList->GetSelCount() == 1)
	{
		int selItem = pObjList->GetSelItem();
		INode* node = (INode*)pObjList->GetItemData(selItem);
		Link<CutsceneObj>* cso = GetCutsceneObj(&linkCutscene->data, node);

		if (cso &&
			cso->data.type & OBJTYPE_SKIN)
		{
			EnableWindow(GetDlgItem(hwnd, IDC_EXPOPTIONS), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_PARTIALANIMLIST), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_PARTIALANIMFRAME), TRUE);
		}
		else
		{
			EnableWindow(GetDlgItem(hwnd, IDC_EXPOPTIONS), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_PARTIALANIMLIST), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_PARTIALANIMFRAME), FALSE);
		}
	}
	else
		EnableWindow(GetDlgItem(hwnd, IDC_EXPOPTIONS), FALSE);

	for(int i = 0; i < nCount; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);
			Object* obj = node->EvalWorldState(0).obj;

			if (obj->SuperClassID() == CAMERA_CLASS_ID)
			{
				bDisable = true;
				break;
			}
		}
	}

	EnableFlags(!bDisable);

	// Find the first selected item in the list
	int selItem = pObjList->GetSelItem();

	if (selItem != -1)
	{
		// Assign first selection flags
		INode* node = (INode*)pObjList->GetItemData(selItem);
		Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);
		assert(linkObj);

		if (linkObj->data.type & OBJTYPE_EXTERNAL)
			CheckDlgButton(hwnd, IDC_EM_E, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_E, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_OBJANIM)
			CheckDlgButton(hwnd, IDC_EM_O, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_O, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_MODEL)
			CheckDlgButton(hwnd, IDC_EM_M, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_M, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_SKIN)
			CheckDlgButton(hwnd, IDC_EM_S, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_S, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_BONEDANIM)
			CheckDlgButton(hwnd, IDC_EM_B, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_B, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_CAMERA)
			CheckDlgButton(hwnd, IDC_EM_C, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_C, BST_UNCHECKED);

		if (linkObj->data.type & OBJTYPE_WORLDSPACE)
			CheckDlgButton(hwnd, IDC_EM_WORLDSPACE, BST_CHECKED);
		else
			CheckDlgButton(hwnd, IDC_EM_WORLDSPACE, BST_UNCHECKED);

		// See if the flag state of the other items in the selection vary and thus
		// determine the flag checkboxes that are indeterminate
		for(i = selItem + 1; i < nCount; i++)
		{
			if (pObjList->IsSelected(i))
			{
				INode* node = (INode*)pObjList->GetItemData(i);
				Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);

				if (linkObj->data.type & OBJTYPE_EXTERNAL)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_E) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_E, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_E) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_E, BST_INDETERMINATE);
				}

				if (linkObj->data.type & OBJTYPE_OBJANIM)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_O) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_O, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_O) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_O, BST_INDETERMINATE);				
				}

				if (linkObj->data.type & OBJTYPE_MODEL)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_M) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_M, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_M) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_M, BST_INDETERMINATE);				
				}

				if (linkObj->data.type & OBJTYPE_SKIN)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_S) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_S, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_S) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_S, BST_INDETERMINATE);				
				}

				if (linkObj->data.type & OBJTYPE_BONEDANIM)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_B) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_B, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_B) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_B, BST_INDETERMINATE);				
				}

				if (linkObj->data.type & OBJTYPE_CAMERA)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_C) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_C, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_C) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_C, BST_INDETERMINATE);				
				}

				if (linkObj->data.type & OBJTYPE_WORLDSPACE)
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_WORLDSPACE) == BST_UNCHECKED)
						CheckDlgButton(hwnd, IDC_EM_WORLDSPACE, BST_INDETERMINATE);
				}
				else
				{
					if (IsDlgButtonChecked(hwnd, IDC_EM_WORLDSPACE) == BST_CHECKED)
						CheckDlgButton(hwnd, IDC_EM_WORLDSPACE, BST_INDETERMINATE);
				}
			}
		}
	}
}

void CutsceneExportDlg::EnableFlags(BOOL bMode)
{
	EnableWindow(GetDlgItem(hwnd, IDC_EM_E), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_O), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_M), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_S), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_B), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_C), bMode);
	EnableWindow(GetDlgItem(hwnd, IDC_EM_WORLDSPACE), bMode);
}

void CutsceneExportDlg::ExportModeChange()
{
	bLockExportModeChange = true;

	// Ensure that only one cutscene is selected
	int selCountCutscene = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (selCountCutscene != 1)
	{
		bLockExportModeChange = false;
		return;
	}

	// Find the selected cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	int nCount = pObjList->GetCount();
	int i;

	// If any of the selected objects are cameras, then changes cannot be made
	for(i = 0; i < nCount; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);
			Object* obj = node->EvalWorldState(0).obj;

			if (obj->SuperClassID() == CAMERA_CLASS_ID)
			{
				EnableFlags(FALSE);
				return;
			}
		}
	}

	EnableFlags(TRUE);

	for(i = 0; i < nCount; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);
			Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);

			if (linkObj)
			{
				// Apply the check boxes to the selected item
				// Indeterminate checks will be ignored
				if (IsDlgButtonChecked(hwnd, IDC_EM_E) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_EXTERNAL;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_E) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_EXTERNAL;

				if (IsDlgButtonChecked(hwnd, IDC_EM_O) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_OBJANIM;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_O) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_OBJANIM;

				if (IsDlgButtonChecked(hwnd, IDC_EM_M) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_MODEL;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_M) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_MODEL;

				if (IsDlgButtonChecked(hwnd, IDC_EM_S) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_SKIN;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_S) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_SKIN;

				if (IsDlgButtonChecked(hwnd, IDC_EM_B) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_BONEDANIM;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_B) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_BONEDANIM;

				if (IsDlgButtonChecked(hwnd, IDC_EM_C) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_CAMERA;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_C) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_CAMERA;

				if (IsDlgButtonChecked(hwnd, IDC_EM_WORLDSPACE) == BST_CHECKED)
					linkObj->data.type |= OBJTYPE_WORLDSPACE;
				else if (IsDlgButtonChecked(hwnd, IDC_EM_WORLDSPACE) == BST_UNCHECKED)
					linkObj->data.type &= ~OBJTYPE_WORLDSPACE;

				pObjList->AddSubItem(BuildFlagString(linkObj->data.type), 1, i);
			}

			linkObj->data.SetTaggedFlags(node);
		}
	}

	PropertyUpdate();
	bLockExportModeChange = false;
}

void CutsceneExportDlg::SaveToMAX()
{
	//ReferenceTarget* scene = ip->GetScenePointer();
	INode* scene = ip->GetRootNode();
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA);

	int size = ListSize(&cutsceneDB) + sizeof(DWORD) + GetNodeListSize(&QNNodeList);
	unsigned char* data = (unsigned char*)malloc(size);
	unsigned char* pos = data;
	
	*((DWORD*)pos) = CUTSCENEEXPORTER_CHUNK_VERSION;
	pos += sizeof(DWORD);
	WriteList<Cutscene>(&pos, &cutsceneDB);

	// v2
	WriteNodeList(&pos, &QNNodeList);

	scene->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA, size, data);
}

void CutsceneExportDlg::LoadFromMAX()
{
	//ReferenceTarget* scene   = ip->GetScenePointer();
	INode* scene = ip->GetRootNode();
	AppDataChunk*    appdata;

	appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA);

	if (appdata && appdata->data)
	{
		DWORD version;
		unsigned char* pos = (unsigned char*)appdata->data;
		
		version = *((DWORD*)pos);
		pos += sizeof(DWORD);

		// For now if the cutscene version isn't up to date, just clear it
		/*
		if (version != CUTSCENEEXPORTER_CHUNK_VERSION)
		{
			scene->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA);
			ClearPerBoneTolerances();
			return;
		}
		*/

		GetList<Cutscene>(&pos, &cutsceneDB);

		if (version >= 0x0003)
			GetNodeList(&pos, &QNNodeList);

		// Update version info
		Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

		while(linkCutscene)
		{
			linkCutscene->data.version = CUTSCENEDATA_VERSION;

			Link<CutsceneObj>* linkObj = linkCutscene->data.objDB.GetHead();

			while(linkObj)
			{
				linkObj->data.version = CUTSCENEOBJ_VERSION;
				linkObj = linkObj->next;
			}

			linkCutscene = linkCutscene->next;
		}
	}

	// Users can no longer specify exports other than models and skins
	// this is handled by the objanim lister and the boneanim lister now instead
	DiscardLegacyFlags();
	bLoaded = true;

	// Ensure that cameras have flags properly assigned
	SetCameraFlags();

	MakeNodeReferences();
}

void CutsceneExportDlg::SetCameraFlags()
{
	Link<Cutscene>* link = cutsceneDB.GetHead();

	while(link)
	{
		Link<CutsceneObj>* linkCSO = link->data.objDB.GetHead();

		while(linkCSO)
		{
			INode* node = gInterface->GetINodeByName(linkCSO->data.name);

			if (node)
			{
				Object* obj = node->EvalWorldState(0).obj;

				if (obj->SuperClassID() == CAMERA_CLASS_ID)
					linkCSO->data.type = OBJTYPE_CAMERA;
			}

			linkCSO = linkCSO->next;
		}

		link = link->next;
	}
}

void CutsceneExportDlg::CreateCutsceneList()
{
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_RESETCONTENT, 0, 0);
	
	Link<Cutscene>* link = cutsceneDB.GetHead();
	int idx;

	while(link)
	{
		idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_ADDSTRING, 0, (LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)link);
		link = link->next;
	}
}

void CutsceneExportDlg::GetExportParams(int* items, int* nItems)
{
	if (IsDlgButtonChecked(hwnd, IDC_EXPALL) == BST_CHECKED)
	{
		*nItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETCOUNT, 0, 0);
		assert(*nItems < 256);
		
		for(int i = 0; i < *nItems; i++)
			items[i] = i;
	}
	else
	{
		*nItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);
	}
}

int CutsceneExportDlg::GetNumPackets()
{
	int items[256];
	int nCutscenes;

	GetExportParams(items, &nCutscenes);

	for(int i = 0; i < nCutscenes; i++)
	{

	}

	return 0;
}

// Determines the amount of padding required to ensure that the packet ends on a 4 byte boundary
inline int GetNumPadBytes(int size)
{
	return 4 - (size % 4);
	//return (int)(( ((float)size / 4.0f) - (float)(size / 4) ) * 4.0f);
}

bool CutsceneExportDlg::IsExternal(CStr modelName)
{
	modelName.toLower();

	if (strstr(modelName, ".skin") ||
		strstr(modelName, ".mdl"))
		return true;

	return false;
}

bool CutsceneExportDlg::IsExternal(CutsceneObj* cso)
{
	if (cso->bUseSkaterModel)
		return true;

	return IsExternal(cso->modelName);
}

int CutsceneExportDlg::GetNumExternal(Cutscene* cutscene, ObjType objType)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();
	int cnt = 0;

	while(link)
	{
		if (link->data.type & objType)
		{
			if (IsExternal(&link->data))
				cnt++;
		}

		link = link->next;
	}

	return cnt;
}

void CutsceneExportDlg::DiscardLegacyFlags()
{
	Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

	while(linkCutscene)
	{
		Link<CutsceneObj>* linkObj = linkCutscene->data.objDB.GetHead();

		while(linkObj)
		{
			// Remove all the flags that no longer can be specified directly by the user
			linkObj->data.type &= ~OBJTYPE_EXTERNAL;
			linkObj->data.type &= ~OBJTYPE_OBJANIM;
			linkObj->data.type &= ~OBJTYPE_BONEDANIM;

			linkObj = linkObj->next;
		}

		linkCutscene = linkCutscene->next;
	}
}

int CutsceneExportDlg::GetNumCIFObjects(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();
	int nObjects = 0;

	while(link)
	{
		if (link->data.type & OBJTYPE_MODEL ||
			link->data.type & OBJTYPE_SKIN  /*||
			link->data.type & OBJTYPE_OBJANIM*/)
		{
			nObjects++;

			if (link->data.bHiresSkeleton /*&&
				!link->data.bUseSkaterModel*/)
				nObjects++;
		}

		link = link->next;
	}

	return nObjects;
}

bool CutsceneExportDlg::KeysExist(Cutscene* pCutscene)
{
	if (pCutscene->trackKeyDB.GetSize() > 0)
		return true;

	Link<CutsceneObj>* link = pCutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.trackKeyDB.GetSize() > 0)
			return true;

		link = link->next;
	}

	return false;
}

bool CutsceneExportDlg::ModificationsExist()
{
	// Scan through the cutscenes being exported and verify that at least one modification flag is set
	Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

	while(linkCutscene)
	{
		Link<CutsceneObj>* linkObj = linkCutscene->data.objDB.GetHead();

		while(linkObj)
		{
			if (linkObj->data.bModified)
				return true;

			linkObj = linkObj->next;
		}

		linkCutscene = linkCutscene->next;
	}

	return false;
}

bool CutsceneExportDlg::DoExport()
{
	if (!ModificationsExist())
	{
		MessageBox(hwnd, "No modifications have been made to cutscene data.", "Export Unnecessary", MB_ICONINFORMATION|MB_OK);
		return true;
	}

	// Ensure the export options are set to export as a cutscene
	class ExportOptions* pExpOptions;
	pExpOptions = GetExportOptions();
	pExpOptions->m_ExportType = ExportOptions::vEXPORT_CUTSCENE;

	GetScriptExporter()->ParseScriptIni();

	int items[256];
	int nCutscenes;		// Number of cutscenes being exported
	int i, j;

	GetExportParams(items, &nCutscenes);
	RetrieveAllProps();

	//ProgressBar* pbarCS = new ProgressBar(hInstance, hwnd, "Exporting Cutscene(s)...", 0, nCutscenes);

	for(i = 0; i < nCutscenes; i++)
	{
		//pbarCS->SetVal(i);

		Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd,
			                                                               IDC_CUTSCNLIST,
																		   LB_GETITEMDATA, 
																		   (WPARAM)items[i], 
																		   0);

		// Script keys moved to packet of raw text in .cut file
		//if (!CompileScriptKeys(&linkCutscene->data))
		//	return false;

		// Verify that cameras exist within this cutscene
		if (!HasCamera(&linkCutscene->data))
		{
			char strErr[256];
			sprintf(strErr, "The cutscene '%s' doesn't contain any cameras.  Every exported cutscene must contain a camera.  Export aborted.", (char*)linkCutscene->data.name);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Cutscene Exporter", MB_ICONWARNING|MB_OK);
			continue;
		}

		char* appenv = getenv(APP_ENV);
		
		if (!appenv)
		{
			char strErr[256];
			sprintf(strErr, "The environment variable '%s' was undefined.  Cannot continue.", APP_ENV);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Environment Variable Not Set", MB_ICONWARNING|MB_OK);
			//delete pbarCS;
			return false;
		}

		if (!linkCutscene->data.VerifyModelNames(hwnd))
			return false;

		if (!linkCutscene->data.VerifyRootBones(hwnd))
			return false;

		// Copy the previous cutscene so we can use it to fill in unmodified data
		CStr filename = CStr(appenv) + CStr(CUTSCENE_PATH) + linkCutscene->data.name + ".cut";
		curCutsceneFilename = ReplaceStr(filename, ".cut", ".cut.bak");

		// Ignore warning if the source file doesn't exist (i.e. this is the first export)
		FILE* fp = fopen(filename, "rb");
		if (fp)
		{
			fclose(fp);

			if (!CopyFile(filename, curCutsceneFilename, FALSE))
			{
				char strErr[256];
				sprintf(strErr, "Failed to copy file '%s' to '%s'.", (char*)filename, (char*)curCutsceneFilename);
				MessageBox(gInterface->GetMAXHWnd(), strErr, "Cutscene Exporter", MB_ICONWARNING | MB_OK);
				return false;
			}
			else
			{
				// CopyFile also copies file attributes we need to ensure that the .bak is not set
				// to have a read-only attribute
				_chmod(curCutsceneFilename, _S_IWRITE | _S_IREAD);
			}
		}

		fp = fopen(filename, "wb");

		if (!fp)
		{
			char strErr[256];
			sprintf(strErr, "Failed to open '%s' for writing.\n", filename);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Cutscene Exporter", MB_ICONWARNING | MB_OK);
			//delete pbarCS;
			return false;
		}

		// Build cutscene databases
		linkCutscene->data.BuildObjAnimDB();
		linkCutscene->data.BuildBonedAnimDB();

		// Write out the cutscene version
		const unsigned int CutsceneVer = CUTSCENE_VERSION;
		fwrite(&CutsceneVer, sizeof(unsigned int), 1, fp);

		//  Build camera packet
		unsigned int nCameras    = GetNumType(&linkCutscene->data, OBJTYPE_CAMERA);

		unsigned char* bufCAM = (unsigned char*)malloc(CUTSCENE_BUFFER_SIZE);
		unsigned char* bufOBA;
		unsigned char* bufSKA;
		unsigned char* bufMDL;
		unsigned char* bufSKIN;
		unsigned char* bufTEX = (unsigned char*)malloc(1024 * 1024 * 10);

		unsigned int  posCAM = 0, posOBA = 0, posSKA = 0, posMDL = 0, posSKIN = 0, posTEX = 0;
		
		if (nCameras > 0)
		{
			if (BuildCompositeCamPacket(&linkCutscene->data, bufCAM, &posCAM) == -1)
				return false;
		}
		
		bufOBA = bufCAM + posCAM;

		BuildObjAnimPacket(&linkCutscene->data, bufOBA, &posOBA);
		bufSKA = bufOBA + posOBA;

		// Write LIB header directory
		unsigned int nObjects    = linkCutscene->data.objAnimDB.GetSize();
		unsigned int nCIFObjects = GetNumCIFObjects(&linkCutscene->data);
		unsigned int nBonedAnims = linkCutscene->data.bonedAnimDB.GetSize();
		unsigned int nModels     = GetNumType(&linkCutscene->data, OBJTYPE_MODEL) - GetNumExternal(&linkCutscene->data, OBJTYPE_MODEL);
		unsigned int nSkins      = GetNumType(&linkCutscene->data, OBJTYPE_SKIN) - GetNumExternal(&linkCutscene->data, OBJTYPE_SKIN);
		unsigned int nQN         = (KeysExist(&linkCutscene->data) || QNNodeList.GetSize() > 0) ? 1 : 0;
		unsigned int nPackets    = 1;	// CIF

		if (nCameras > 0)
			nPackets++;

		// If object animations exist then the OBA packet exists
		if (nObjects > 0)
			nPackets++;

		nPackets += nBonedAnims;
		nPackets += nModels * 2;		// Each model also has a tex dictionary packet
		nPackets += nSkins * 2;			// Each skin also has a tex dicrtionary packet	
		nPackets += nQN;

		fwrite(&nPackets, sizeof(unsigned int), 1, fp);

		// Generate LIB Packet headers
		LIBObjPacket  CIFPacket;
		LIBObjPacket  CAMPacket;
		LIBObjPacket  OBAPacket;
		LIBObjPacket* SKAPackets   = new LIBObjPacket[nBonedAnims];
		LIBObjPacket* MDLPackets   = new LIBObjPacket[nModels];
		LIBObjPacket* SKINPackets  = new LIBObjPacket[nSkins];
		LIBObjPacket* TEXPackets   = new LIBObjPacket[nModels + nSkins];
		LIBObjPacket  QNPacket;
		int*          nSKAPadding  = new int[nBonedAnims];
		int*          nMDLPadding  = new int[nModels];
		int*          nSKINPadding = new int[nSkins];
		int*          nTEXPadding  = new int[nModels + nSkins];

		// LIB Packets are aligned no need to pad
		CIFPacket.offset   = sizeof(LIBObjPacket) * nPackets + sizeof(unsigned int) + sizeof(unsigned int);
		CIFPacket.size     = 8 + 272 * nCIFObjects;
		//CIFPacket.size     = sizeof(unsigned int) + nObjects * (sizeof(unsigned int) * 4);
		CIFPacket.nameCRC  = 0;
		CIFPacket.extCRC   = GenerateCRC("CIF");

		// CIF Packets are aligned no need to pad
		CAMPacket.offset   = CIFPacket.offset + CIFPacket.size;
		CAMPacket.size     = posCAM;
		CAMPacket.nameCRC  = 0;
		CAMPacket.extCRC   = GenerateCRC("CAM");

		// Ensure CAM Packet is 4 byte aligned
		int nCAMPadding = GetNumPadBytes(CAMPacket.size);

		OBAPacket.offset   = CAMPacket.offset + CAMPacket.size + nCAMPadding;
		OBAPacket.size     = posOBA;
		OBAPacket.nameCRC  = 0;
		OBAPacket.extCRC   = GenerateCRC("OBA");

		int nOBAPadding = GetNumPadBytes(OBAPacket.size);
		unsigned int lastSize = OBAPacket.offset + OBAPacket.size + nOBAPadding;

		// Construct SKA Packet info
		int curSKAPacket = 0;
		unsigned char* curbufSKA = bufSKA;

		unsigned char pad[4] = "\0\0\0";

		Link<CutsceneObj>* linkCSObj = linkCutscene->data.bonedAnimDB.GetHead();

		while(linkCSObj)
		{	
			if (BuildBonedAnimPacket(&linkCutscene->data, &linkCSObj->data, curbufSKA, &posSKA) == -1)
				return false;

			SKAPackets[curSKAPacket].offset   = lastSize;
			SKAPackets[curSKAPacket].size     = posSKA;
			
			if (linkCSObj->data.pOrigObj && linkCSObj->data.pOrigObj->name.Length() > 0)
			{
				// Force copies to have unique names by adding one
				/*
				if (linkCSObj->data.bCopy)
					SKAPackets[curSKAPacket].nameCRC = GenerateCRC(linkCSObj->data.name) + 1;
				else
					SKAPackets[curSKAPacket].nameCRC = GenerateCRC(linkCSObj->data.name);
				*/

				if (linkCSObj->data.bCopy)
					SKAPackets[curSKAPacket].nameCRC = GenerateCRC(linkCSObj->data.pOrigObj->name) + 1;
				else
					SKAPackets[curSKAPacket].nameCRC = GenerateCRC(linkCSObj->data.pOrigObj->name);
			}
			else
				SKAPackets[curSKAPacket].nameCRC = 0;

			SKAPackets[curSKAPacket].extCRC   = GenerateCRC("SKA");

			// Advance to next SKA packet LIB header
			nSKAPadding[curSKAPacket] = GetNumPadBytes(SKAPackets[curSKAPacket].size);
			curbufSKA += posSKA;
			memcpy(curbufSKA, pad, nSKAPadding[curSKAPacket]);
			curbufSKA += nSKAPadding[curSKAPacket];
			
			lastSize   = SKAPackets[curSKAPacket].offset + SKAPackets[curSKAPacket].size + nSKAPadding[curSKAPacket];
			posSKA     =  0;
			curSKAPacket++;

			linkCSObj = linkCSObj->next;
		}

		// Compute total size of SKA buffers w/ padding
		int SKASize = 0;
		for(j = 0; j < nBonedAnims; j++)
			SKASize += SKAPackets[j].size + nSKAPadding[j];

		// Construct MDL Packet info
		bufMDL = bufSKA + SKASize;
		int curMDLPacket = 0;
		int curTEXPacket = 0;
		unsigned char* curbufMDL = bufMDL;
		unsigned char* curbufTEX = bufTEX;

		linkCSObj = linkCutscene->data.objDB.GetHead();

		while(linkCSObj)
		{
			if (linkCSObj->data.type & OBJTYPE_MODEL &&
				!IsExternal(&linkCSObj->data))
			{
				OutputDebugString("Pre BuildModelPacket\n");

				BuildModelPacket(&linkCSObj->data, curbufMDL, &posMDL,
					                               curbufTEX, &posTEX);

				OutputDebugString("Post BuildModelPacket\n");

				//MDLPackets[curMDLPacket].offset = lastSize;
				MDLPackets[curMDLPacket].size   = posMDL;
				TEXPackets[curTEXPacket].size   = posTEX;

				if (linkCSObj->data.modelName.Length() > 0)
				{
					MDLPackets[curMDLPacket].nameCRC = GenerateCRC(linkCSObj->data.modelName);
					TEXPackets[curTEXPacket].nameCRC = GenerateCRC(linkCSObj->data.modelName);
				}
				else
				{
					char strErr[256];
					sprintf(strErr, "WARNING! You are exporting a model '%s' without a model export name defined.", (char*)linkCSObj->data.modelName);
					MessageBox(hwnd, strErr, "Undefined Model Name", MB_ICONWARNING|MB_OK);
					MDLPackets[curMDLPacket].nameCRC = 0;
					TEXPackets[curTEXPacket].nameCRC = 0;
				}

				MDLPackets[curMDLPacket].extCRC = GenerateCRC("MDL");
				TEXPackets[curTEXPacket].extCRC = GenerateCRC("TEX");

				// Advance to next MDL packet LIB header
				nMDLPadding[curMDLPacket] = GetNumPadBytes(MDLPackets[curMDLPacket].size);
				nTEXPadding[curTEXPacket] = GetNumPadBytes(TEXPackets[curTEXPacket].size);

				curbufMDL += posMDL;
				memcpy(curbufMDL, pad, nMDLPadding[curMDLPacket]);
				curbufMDL += nMDLPadding[curMDLPacket];

				curbufTEX += posTEX;
				memcpy(curbufTEX, pad, nTEXPadding[curTEXPacket]);
				curbufTEX += nTEXPadding[curTEXPacket];

				//lastSize   = MDLPackets[curMDLPacket].offset + MDLPackets[curMDLPacket].size;
				posMDL     = 0;
				posTEX     = 0;
				curMDLPacket++;
				curTEXPacket++;
			}

			linkCSObj = linkCSObj->next;
		}

		// Compute total size of MDL buffers with padding
		int MDLSize = 0;
		for(j = 0; j < nModels; j++)
			MDLSize += MDLPackets[j].size + nMDLPadding[j];

		// Construct SKIN Packet info
		bufSKIN = bufMDL + MDLSize;
		int curSKINPacket = 0;
		unsigned char* curbufSKIN = bufSKIN;

		linkCSObj = linkCutscene->data.objDB.GetHead();

		while(linkCSObj)
		{
			if (linkCSObj->data.type & OBJTYPE_SKIN &&
				!IsExternal(&linkCSObj->data))
			{
				BuildSkinPacket(&linkCutscene->data, &linkCSObj->data, curbufSKIN, &posSKIN,
					                                                   curbufTEX,  &posTEX);

				//SKINPackets[curSKINPacket].offset = lastSize;
				SKINPackets[curSKINPacket].size   = posSKIN;
				TEXPackets[curTEXPacket].size     = posTEX;

				if (linkCSObj->data.modelName.Length() > 0)
				{
					SKINPackets[curSKINPacket].nameCRC = GenerateCRC(linkCSObj->data.modelName);
					TEXPackets[curTEXPacket].nameCRC   = GenerateCRC(linkCSObj->data.modelName);
				}
				else
				{
					SKINPackets[curSKINPacket].nameCRC = 0;
					TEXPackets[curTEXPacket].nameCRC = 0;
				}

				SKINPackets[curSKINPacket].extCRC = GenerateCRC("SKIN");
				TEXPackets[curTEXPacket].extCRC = GenerateCRC("TEX");

				// Advance to next SKIN packet LIB header
				nSKINPadding[curSKINPacket] = GetNumPadBytes(SKINPackets[curSKINPacket].size);
				nTEXPadding[curTEXPacket] = GetNumPadBytes(TEXPackets[curTEXPacket].size);

				curbufSKIN += posSKIN;
				memcpy(curbufSKIN, pad, nSKINPadding[curSKINPacket]);
				curbufSKIN += nSKINPadding[curSKINPacket];

				curbufTEX  += posTEX;
				memcpy(curbufTEX, pad, nTEXPadding[curTEXPacket]);
				curbufTEX  += nTEXPadding[curTEXPacket];

				//lastSize    = SKINPackets[curSKINPacket].offset + SKINPackets[curSKINPacket].size;
				posSKIN     = 0;
				posTEX      = 0;
				curSKINPacket++;
				curTEXPacket++;
			}

			linkCSObj = linkCSObj->next;
		}

		// Construct Texture packet offset size with padding
		int TEXSize = 0;
		for(j = 0; j < curTEXPacket; j++)
		{
			TEXPackets[j].offset = lastSize;
			lastSize = TEXPackets[j].offset + TEXPackets[j].size + nTEXPadding[j];
			TEXSize += TEXPackets[j].size + nTEXPadding[j];
		}

		// Construct Model packet offset size
		for(j = 0; j < nModels; j++)
		{
			MDLPackets[j].offset = lastSize;
			lastSize = MDLPackets[j].offset + MDLPackets[j].size + nMDLPadding[j];
		}

		// Construct SKIN packet offset size
		for(j = 0; j < nSkins; j++)
		{
			SKINPackets[j].offset = lastSize;
			lastSize = SKINPackets[j].offset + SKINPackets[j].size + nSKINPadding[j];
		}

		// Compute total size of SKIN buffers with padding
		int SKINSize = 0;
		for(j = 0; j < nSkins; j++)
			SKINSize += SKINPackets[j].size + nSKINPadding[j];

		// Assign QN Packet LIBObj header data
		CStr QNScript = GenerateScript(&linkCutscene->data);
		
		QNPacket.offset    = lastSize;
		QNPacket.size      = QNScript.Length() + 1;
		QNPacket.nameCRC  = 0;
		QNPacket.extCRC   = GenerateCRC("QN");
		lastSize = QNPacket.offset + QNPacket.size + GetNumPadBytes(QNPacket.size);

		// Write out LIB Packet headers
		fwrite(&CIFPacket, sizeof(LIBObjPacket), 1, fp);
		fwrite(&CAMPacket, sizeof(LIBObjPacket), 1, fp);
		
		if (nObjects > 0)
			fwrite(&OBAPacket, sizeof(LIBObjPacket), 1, fp);

		for(j = 0; j < nBonedAnims; j++)
		{
			fwrite(&SKAPackets[j], sizeof(LIBObjPacket), 1, fp);
		}

		for(j = 0; j < curTEXPacket; j++)
		{
			fwrite(&TEXPackets[j], sizeof(LIBObjPacket), 1, fp);
		}

		for(j = 0; j < nModels; j++)
		{
			fwrite(&MDLPackets[j], sizeof(LIBObjPacket), 1, fp);
		}

		for(j = 0; j < nSkins; j++)
		{
			fwrite(&SKINPackets[j], sizeof(LIBObjPacket), 1, fp);
		}

		if (KeysExist(&linkCutscene->data) || QNNodeList.GetSize() > 0)
			fwrite(&QNPacket, sizeof(LIBObjPacket), 1, fp);

		// Write CIF Packet
		WriteCIFPacket(&linkCutscene->data, fp);

		// Write CAM Packet
		fwrite(bufCAM, posCAM, 1, fp);
		fwrite(pad, nCAMPadding, 1, fp);

		// Write OBA Packet
		fwrite(bufOBA, posOBA, 1, fp);
		fwrite(pad, nOBAPadding, 1, fp);

		// Write SKA Packets
		fwrite(bufSKA, SKASize, 1, fp);

		// Write TEX Packets
		fwrite(bufTEX, TEXSize, 1, fp);

		// Write MDL Packets
		//long fpPos = ftell(fp);
		fwrite(bufMDL, MDLSize, 1, fp);

		// Write SKIN Packets
		fwrite(bufSKIN, SKINSize, 1, fp);

		// Write QN Packet
		if (KeysExist(&linkCutscene->data) || QNNodeList.GetSize() > 0)
			fwrite((char*)QNScript, QNScript.Length()+1, 1, fp);

		free(bufCAM);	//	includes OBA, SKA, MDL, and SKIN
		free(bufTEX);
		fclose(fp);

		delete [] SKAPackets;
		delete [] MDLPackets;
		delete [] SKINPackets;
		delete [] TEXPackets;

		delete [] nSKAPadding;
		delete [] nMDLPadding;
		delete [] nSKINPadding;
		delete [] nTEXPadding;

		ResetModifiedFlags();

		// Launch libconv to convert the file if appropriate
		if (IsDlgButtonChecked(hwnd, IDC_RUNLIBCONV) == BST_CHECKED)
		{
			char path[1024], conv_path[1024];
			char* project_root = getenv( APP_ENV );
			char* args[4];

			if (!project_root)
			{
				char strErr[256];
				sprintf(strErr, "Environment variable '%s' must be defined to continue.  Operation aborted.", APP_ENV);

				MessageBox(hwnd, strErr, "Environment variable not set", MB_ICONSTOP|MB_OK);
				continue;
			}

			sprintf( path, "-f%s", filename );
			sprintf( conv_path, "%s\\bin\\win32\\libconv.exe", project_root );

			args[0] = conv_path;
			switch( GetQuickviewPlatform())
			{
				case vQUICKVIEW_XBOX:
					args[1] = "-px";
					break;
				case vQUICKVIEW_PS2:
					args[1] = "-pp";
					break;
				case vQUICKVIEW_NGC:
					args[1] = "-pg";
					break;
			}
			args[2] = path;	
			args[3] = NULL; 

			//result = spawnv( _P_WAIT, args[0], args );
			int result = ExecuteCommand( args );

			if (result < 0)
			{
				MessageBeep(0xFFFFFFFF);			// PC Speaker beep
				MessageBeep(MB_ICONEXCLAMATION);	// Exclamation sound through sound card
				MessageBox(gInterface->GetMAXHWnd(), "Libconv has failed to convert the cutscene.  Please verify settings.", "LibConv Failed", MB_ICONSTOP|MB_OK);
				return false;
			}
		}
	}

	//delete pbarCS;

	// Give user auditory notification upon completion
	MessageBeep(0xFFFFFFFF);			// PC Speaker beep
	MessageBeep(MB_ICONEXCLAMATION);	// Exclamation sound through sound card
	MessageBox(hwnd, "Cutscene export completed!", "Export Complete", MB_ICONEXCLAMATION|MB_OK);
	InitObjList(displayMode);
	return true;
}

bool CutsceneExportDlg::ModelExistsInCutscene(Cutscene* cutscene, CutsceneObj* excludeobj, CStr modelName)
{
	int        nObjs = pObjList->GetCount();
	CStr LCmodelName = modelName;
	LCmodelName.toLower();

	for(int i = 0; i < nObjs; i++)
	{
		INode* node = (INode*)pObjList->GetItemData(i);

		if (node)
		{
			Link<CutsceneObj>* link = GetCutsceneObj(cutscene, node);

			if (link && link->data.type & OBJTYPE_MODEL)
			{
				CStr LClinkModelName = link->data.modelName;
				LClinkModelName.toLower();

				if (&link->data != excludeobj &&
					LClinkModelName == LCmodelName &&
					!link->data.bCopy)
				{				
					return true;
				}
			}
		}
	}

	return false;
}

CutsceneObj* CutsceneExportDlg::FindObjByName(LinkList<CutsceneObj>* list, CStr name, bool bCopy)
{
	Link<CutsceneObj>* link = list->GetHead();

	while(link)
	{
		if (link->data.name == name && link->data.bCopy == bCopy)
			return &link->data;

		link = link->next;
	}

	return NULL;
}

void CutsceneExportDlg::WriteCIFPacket(Cutscene* cutscene, FILE* fp)
{
	unsigned int nObjects = GetNumCIFObjects(cutscene);
	unsigned int version  = vCIF_FORMAT_VERSION;

	// Write version
	fwrite(&version, sizeof(unsigned int), 1, fp);
	fwrite(&nObjects, sizeof(unsigned int), 1, fp);

	Link<CutsceneObj>* csobj = cutscene->objDB.GetHead();

	while(csobj)
	{
		if (csobj->data.type & OBJTYPE_CAMERA)
		{
			csobj = csobj->next;
			continue;
		}

		if (csobj->data.type & OBJTYPE_MODEL ||
			csobj->data.type & OBJTYPE_SKIN  /*||
			csobj->data.type & OBJTYPE_OBJANIM*/)
		{
			unsigned int objCRC, modelCRC, skelCRC, rootBoneCRC;
			char modelName[256];

			// If a hi-res skeleton is being used then a copy exists
			// and two CIF objects should be exported (Copy [head] should be dumped first)
			if (csobj->data.bHiresSkeleton /*&&
				!csobj->data.bUseSkaterModel*/)
			{
				assert(csobj->data.pObjAnim);
				objCRC = GenerateCRC(csobj->data.name) + 1;
				//objCRC = GenerateCRC(csobj->data.pObjAnim->name) + 1;

				if (csobj->data.modelName2.Length() == 0 || IsExternal(csobj->data.modelName2))
					modelCRC = 0;
				else
					modelCRC = GenerateCRC(csobj->data.modelName2);

				if (csobj->data.skelName2.Length() == 0)
					skelCRC = 0;
				else
					skelCRC = GenerateCRC(csobj->data.skelName2);

				if (csobj->data.rootBone.Length() == 0)
					rootBoneCRC = 0;
				else
				{
					CutsceneObj* obaRootBoneObj = FindObjByName(&cutscene->bonedAnimDB, csobj->data.rootBone, true);
					assert(obaRootBoneObj);
					assert(obaRootBoneObj->pOrigObj);

					rootBoneCRC = GenerateCRC(obaRootBoneObj->pOrigObj->name) + 1;
					//rootBoneCRC = GenerateCRC(csobj->data.rootBone) + 1;
				}

				fwrite(&objCRC, sizeof(unsigned int), 1, fp);
				fwrite(&modelCRC, sizeof(unsigned int), 1, fp);
				fwrite(&skelCRC, sizeof(unsigned int), 1, fp);
				fwrite(&rootBoneCRC, sizeof(unsigned int), 1, fp);

				ZeroMemory(modelName, 256);

				if (modelCRC == 0)
					strcpy(modelName, csobj->data.modelName2);

				fwrite(modelName, 256, 1, fp);				
			}	

			assert(csobj->data.pObjAnim);
			objCRC = GenerateCRC(csobj->data.name);
			//objCRC = GenerateCRC(csobj->data.pObjAnim->name);

			if (csobj->data.bUseSkaterModel)
			{
				modelCRC = SKATER_CRC;
			}
			else
			{
				if (csobj->data.modelName.Length() == 0 || IsExternal(csobj->data.modelName))
					modelCRC = 0;
				else
					modelCRC = GenerateCRC(csobj->data.modelName);
			}

			if (csobj->data.skelName.Length() == 0)
				skelCRC = 0;
			else
				skelCRC = GenerateCRC(csobj->data.skelName);

			if (csobj->data.rootBone.Length() == 0)
				rootBoneCRC = 0;
			else
			{
				CutsceneObj* obaRootBoneObj = FindObjByName(&cutscene->bonedAnimDB, csobj->data.rootBone, false);
				
				if (obaRootBoneObj)
				{
					assert(obaRootBoneObj->pOrigObj);
					rootBoneCRC = GenerateCRC(obaRootBoneObj->pOrigObj->name);
				}
				else
					rootBoneCRC = 0;

				//rootBoneCRC = GenerateCRC(csobj->data.rootBone);
			}

			fwrite(&objCRC, sizeof(unsigned int), 1, fp);

			// If the object is a particle system override all other entries
			INode* node = gInterface->GetINodeByName(csobj->data.name);
			Object* obj = node->EvalWorldState(0).obj;

			/*
			if (obj->ClassID() == PARTICLE_BOX_CLASS_ID)
			{
				unsigned int zero = 0;
				fwrite(&zero, sizeof(unsigned int), 1, fp);
				fwrite(&zero, sizeof(unsigned int), 1, fp);
				fwrite(&zero, sizeof(unsigned int), 1, fp);

				ZeroMemory(modelName, 256);
				fwrite(modelName, 256, 1, fp);
			}
			else
			*/
			{
				fwrite(&modelCRC, sizeof(unsigned int), 1, fp);
				fwrite(&skelCRC, sizeof(unsigned int), 1, fp);
				fwrite(&rootBoneCRC, sizeof(unsigned int), 1, fp);

				ZeroMemory(modelName, 256);

				if (modelCRC == 0)
					strcpy(modelName, csobj->data.modelName);

				fwrite(modelName, 256, 1, fp);
			}
		}

/*
		objCRC = GenerateCRC(csobj->data.pObjAnim->name);

		if (csobj->data.bCopy)
			objCRC++;

		if (csobj->data.bCopy)
		{
			// If this object anim is also being exported as a model then, it should
			// ModelExistsInCutscene won't catch it because the current obj is excluded
			// So check if it's also an OBJTYPE_MODEL
			if (csobj->data.type & OBJTYPE_MODEL ||
				ModelExistsInCutscene(cutscene, &csobj->data, csobj->data.modelName2))
			{
				modelCRC = GenerateCRC(csoLinked->modelName2);
			}
			else
			{
				modelCRC = 0;
				ZeroMemory(modelName, 256);
			}
		}
		else
		{
			// If this object anim is also being exported as a model then, it should
			// ModelExistsInCutscene won't catch it because the current obj is excluded
			// So check if it's also an OBJTYPE_MODEL
			if (csobj->data.type & OBJTYPE_MODEL || 
				ModelExistsInCutscene(cutscene, &csobj->data, csobj->data.modelName))
			{
				modelCRC = GenerateCRC(csobj->data.modelName);
			}
			else
			{
				modelCRC = 0;
				ZeroMemory(modelName, 256);
			}
		}

		skelCRC = 0;

		if (csobj->data.bCopy)
		{
			if (csoLinked->skelName2.Length() > 0)
				skelCRC = GenerateCRC(csoLinked->skelName2);
		}
		else
		{
			if (csoLinked->skelName.Length() > 0)
				skelCRC = GenerateCRC(csoLinked->skelName);
		}

		if (csoLinked->rootBone.Length() > 0)
		{
			CutsceneObj* cso = FindObjByName(cutscene, csoLinked->rootBone, csobj->data.bCopy);
			
			if (cso)
				if (csobj->data.bCopy)
					rootBoneCRC = cso->tmpId + 1;
				else
					rootBoneCRC = cso->tmpId;
			else
				rootBoneCRC = 0;
			//rootBoneCRC = GenerateCRC(csoLinked->rootBone);
		}
		else
			rootBoneCRC = 0;

		fwrite(&objCRC, sizeof(unsigned int), 1, fp);
		fwrite(&modelCRC, sizeof(unsigned int), 1, fp);
		fwrite(&skelCRC, sizeof(unsigned int), 1, fp);
		fwrite(&rootBoneCRC, sizeof(unsigned int), 1, fp);

		ZeroMemory(modelName, 256);

		if (modelCRC == 0)
		{
			if (csobj->data.bCopy)
				strcpy(modelName, csoLinked->modelName2);
			else
				strcpy(modelName, csoLinked->modelName);
		}

		fwrite(modelName, 256, 1, fp);
		*/

		csobj = csobj->next;
	}
}

int CutsceneExportDlg::GetNumType(Cutscene* cutscene, DWORD type)
{
	Link<CutsceneObj>* csobj = cutscene->objDB.GetHead();
	int nCameras = 0;

	while(csobj)
	{
		if (csobj->data.type & type)
			nCameras++;

		csobj = csobj->next;
	}

	return nCameras;
}

///////////////////////////// OBA Packet ///////////////////////////////////////////////////////////////////////////

bool CutsceneExportDlg::HasCamera(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
			return true;

		link = link->next;
	}

	return false;
}

void CutsceneExportDlg::FindDuration(Cutscene* cutscene, ObjType type, int* startFrame, int* endFrame)
{
	Link<CutsceneObj>* csobj = cutscene->objDB.GetHead();

	int  sFrame, eFrame;
	bool bFoundStart = false;

	// Assign start and end frame to the first object we encounter
	while(csobj)
	{
		if (csobj->data.type & OBJTYPE_MODEL ||
			csobj->data.type & OBJTYPE_SKIN)
		{
			sFrame = csobj->data.start;
			eFrame = csobj->data.end;
			bFoundStart = true;
			break;
		}

		csobj = csobj->next;
	}

	if (!bFoundStart)
	{
		sFrame = ip->GetAnimRange().Start() / GetTicksPerFrame();
		eFrame = ip->GetAnimRange().End() / GetTicksPerFrame();

		if (sFrame < 0)
			sFrame = 0;
	}

	while(csobj)
	{
		if (csobj->data.type & OBJTYPE_MODEL ||
			csobj->data.type & OBJTYPE_SKIN)
		{
			if (csobj->data.start < sFrame)
				sFrame = csobj->data.start;

			if (csobj->data.end > eFrame)
				eFrame = csobj->data.end;
		}

		csobj = csobj->next;
	}

	*startFrame = sFrame;
	*endFrame   = eFrame;
}

bool CutsceneExportDlg::RequiresCompression(Cutscene* cutscene, DWORD type)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & type)
		{
			if (link->data.rotTol  != 0.0f ||
				link->data.tranTol != 0.0f)
				return true;
		}

		link = link->next;
	}

	return false;
}

INode* FindPelvisBone(INode* node)
{
	CStr name = node->GetName();
	name.toLower();

	if (name == CStr("bone_pelvis"))
		return node;

	int nChildren = node->NumberOfChildren();

	for(int i = 0; i < nChildren; i++)
	{
		INode* childNode = node->GetChildNode(i);
		INode* retNode   = FindPelvisBone(childNode);
		if (retNode)
			return retNode;
	}

	return NULL;
}

int CutsceneExportDlg::BuildObjAnimPacket(Cutscene* cutscene, unsigned char* pData, unsigned int* pos)
{
	int  numObjAnims = cutscene->objAnimDB.GetSize();
	bool bCompressTime = false;				// Determines if the timestamps are represented in frames or not
	bool bCompress;							// Determines if keys have compression

	bCompress = RequiresCompression(cutscene, OBJTYPE_OBJANIM);

	ObjectExporter::bRotateRoot        = false;
	//ObjectExporter::bTestRotate        = true;
	ObjectExporter::bSwapCoordSystem   = false;
	ObjectExporter::bNoParentTransform = true;

	int  startFrame;
	int  endFrame;

	SHeader header;
	header.version = ANIMFILE_VERSION;
	//header.flags   = INTERMEDIATE_FORMAT | OBJECTANIM | UNCOMPRESSED_FORMAT | PREROTATEDROOT;
	//header.flags   = INTERMEDIATE_FORMAT | OBJECTANIM | UNCOMPRESSED_FORMAT;
	header.flags   = INTERMEDIATE_FORMAT | OBJECTANIM | CUTSCENE;

	if (!bCompress)
		header.flags |= UNCOMPRESSED_FORMAT;

	if (bCompressTime)
		header.flags |= COMPRESSEDTIME;
	

	FindDuration(cutscene, OBJTYPE_OBJANIM, &startFrame, &endFrame);

	int num3DSFrames = endFrame - startFrame;
	int fps3DS       = GetFrameRate();
		
	header.skeletonName = GenerateCRC("OBJECTS");			// hard-coded default for now
	header.duration     = (float)num3DSFrames/(float)fps3DS;

	// Build node list of animation data for each node in the system (cutscene)
	INodeTab nodes;
	
	Link<CutsceneObj>* csobj = cutscene->objAnimDB.GetHead();

	ObjSpecificExportSettings settings;
	
	nodeList.Clear();
	nodes.ZeroCount();

	LinkList<unsigned int> nameList;

	while(csobj)
	{
		INode* node = ip->GetINodeByName(csobj->data.name);

		// If export from Pelvis is selected we'll export the pelvis bone by scanning down the hierarchy instead
		if (IsDlgButtonChecked(hwnd, IDC_EXPORTFROMPELVIS) == BST_CHECKED &&
			csobj->data.bExportFromPelvis)
		{
 			expNode = ip->GetINodeByName(csobj->data.pOrigObj->name);

			CStr expName = expNode->GetName();
			INode* rootSkin   = FindBoneRootBySkin(expNode);
			if (rootSkin)
			{
				INode* nodePelvis = FindPelvisBone(rootSkin);
				CStr pelvisName = nodePelvis->GetName();

				if (!nodePelvis)
				{
					char strErr[256];
					sprintf(strErr, "Failed to find bone_pelvis scanning through '%s' hierarchy.", (char*)expNode->GetName());
					MessageBox(gInterface->GetMAXHWnd(), strErr, "Couldn't find pelvis bone", MB_ICONWARNING|MB_OK);
				}
				else
					expNode = nodePelvis;
			}
			else
				expNode = node;
		}
		else
		{
			// If not exporting from pelvis no changes to the export process should be made
			expNode = node;
		}

		bexpModel = true;
		expCso    = &csobj->data;

		if (node)
		{
			// Hires skeleton dupes are always the animation of the neck bone
			// in the system ss thats where the skeleton will be broken
			/* Neck bone OBAs are no longer exported
			if (csobj->data.bHiresSkeleton)
			{
				if (csobj->data.bCopy)
				{
					node = FindNode("Bone_Neck", node);

					if (!node)
					{
						char strErr[256];
						sprintf(strErr, "Failed to locate 'Bone_Neck' bone in for hires skeleton on object '%s'.", (char*)csobj->data.name);
						MessageBox(ip->GetMAXHWnd(), strErr, "Hires Skeleton Problem", MB_ICONWARNING | MB_OK);
						return 0;
					}
				}
			}
			*/

			// Normal export
			float fCompress = 0.999700f + 0.000300f * csobj->data.rotTol;

			settings.fRotTolerance  = fCompress;
			settings.fTranTolerance = csobj->data.tranTol;
			settings.start          = csobj->data.start;
			settings.end            = csobj->data.end;
				
			strcpy(settings.strUserName, csobj->data.name);
			
			Link<ObjectExporter::NodeData>* linkND;

/*
			if (IsDlgButtonChecked(hwnd, IDC_PROCVIS) == BST_CHECKED)
				linkND = AddNodeData(node, &settings, bCompress, true, RemoveInvisibleKeys, this);
			else
				linkND = AddNodeData(node, &settings, bCompress, true);
*/

			if (IsDlgButtonChecked(hwnd, IDC_PROCVIS) == BST_CHECKED)
				linkND = AddNodeData(expNode, &settings, bCompress, true, RemoveInvisibleKeys, this);
			else
				linkND = AddNodeData(expNode, &settings, bCompress, true);

			// Offset from pivot
			if (expNode != node)
			{
				//UpdateNodeDataRelative(cutscene, &linkND->data, node, csobj->data.start);
				
				// Repatch the original node name (just includes anim data from the pelvis however)
				linkND->data.node = node;
			}

			// If compression was not used, we need to manually remove the duplicated keys
			if (!bCompress)
			{
				for(int i = 0; i < linkND->data.numNodes; i++)
				{
					RemoveFlaggedInvisibleKeys(&linkND->data.QFrames, &linkND->data.numCompressedQFrames[i]);
					RemoveFlaggedInvisibleKeys(&linkND->data.TFrames, &linkND->data.numCompressedTFrames[i]);
				}
			}

			nodes.Append(1, &node);

			// Bone names should match up with the original main objDB name that generated
			// this objanim
			unsigned int nameCRC = GenerateCRC(csobj->data.pOrigObj->name);
			nameList.Add(&nameCRC);
		}

		csobj = csobj->next;
	}

	header.numBones     = numObjAnims;

	// We should now have all the animation data for the object animation nodes in the nodeList
	header.numQKeys     = GetNumQFrames();
	header.numTKeys     = GetNumTFrames();
	
	header.numUserDefinedKeys = 0;			// No script keys for now

	unsigned int datasize;

	if (!header.Write(pData, &datasize))
		return -1;

	pData += datasize;
	*pos  += datasize;

	// Build CRC
	unsigned int uiCRC = (unsigned int)numObjAnims;

	int i, j;

	for(i = 0; i < numObjAnims; i++)
		uiCRC ^= GenerateCRC(nodes[i]->GetName());
	
	// Output Skeleton data
	*((unsigned int*)pData) = uiCRC;		// Dump CRC
	pData += sizeof(unsigned int);
	*pos  += sizeof(unsigned int);

	*((unsigned int*)pData) = numObjAnims;	// Dump number of objects (num bones)
	pData += sizeof(unsigned int);
	*pos  += sizeof(unsigned int);

	// Dump bone names/Object names
	for(i = 0; i < numObjAnims; i++)
	{
		// !!! Added + i * 10 to ensure unique id between bones with same names !!!
		/*
		CStr nodeName = nodes[i]->GetName();
#ifdef USE_UNIQUE_BONE_IDS
		*((unsigned int*)pData) = GenerateCRC(nodeName) + i * 10;
#else
		*((unsigned int*)pData) = GenerateCRC(nodeName);
#endif
		*/

		*((unsigned int*)pData) = nameList[i];

		pData += sizeof(unsigned int);
		*pos  += sizeof(unsigned int);
	}

	// Pad out unused fields (Parent Names, Flip Names)
	for(i = 0; i < numObjAnims * 2; i++)
	{
		*((unsigned int*)pData) = 0;
		pData += sizeof(unsigned int);
		*pos  += sizeof(unsigned int);
	}

////////////////////

#ifndef NDEBUG
#ifdef  OBA_DEBUG
	FILE* fpDebug = fopen("c:\\obadebug.txt", "w");
#endif
#endif

	// Output compressed rotation data
	SQData sqdata(numObjAnims, header.numQKeys);
	int QKeyPos = 0;

	// Traverse the node list dumping out rotation keys
	for(i = 0; i < nodeList.GetSize(); i++)
	{

#ifndef NDEBUG
#ifdef  OBA_DEBUG
		fprintf(fpDebug, "Name: %s\n", (char*)nodeList[i].node->GetName());
#endif
#endif

		sqdata.numKeysPerBone[i] = nodeList[i].numCompressedQFrames[0];

		for(j = 0; j < nodeList[i].numCompressedQFrames[0]; j++)
		{
			int curFrame = j;
	
			// If the compress time flag is specified we need to multiply time time by the FPS
			// rate to convert it into the frame number as opposed to the time into the animation
			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
				float ftmp = (nodeList[i].QFrames[curFrame].time * (float)OUTPUT_FPS);
				*val = Round(ftmp);
			}
			else
				sqdata.theQKeys[QKeyPos].time = nodeList[i].QFrames[curFrame].time;

			sqdata.theQKeys[QKeyPos].qx   = nodeList[i].QFrames[curFrame].q[X];
			sqdata.theQKeys[QKeyPos].qy   = nodeList[i].QFrames[curFrame].q[Y];
			sqdata.theQKeys[QKeyPos].qz   = nodeList[i].QFrames[curFrame].q[Z];
			sqdata.theQKeys[QKeyPos].real = nodeList[i].QFrames[curFrame].q[W];

#ifndef NDEBUG
#ifdef  OBA_DEBUG
			fprintf(fpDebug, "SQ#: %i Time: %f  Frame: %i  Qx: %f  Qy: %f  Qz: %f  Real: %f\n", QKeyPos,
				                                                           sqdata.theQKeys[QKeyPos].time,
																		   (int)(sqdata.theQKeys[QKeyPos].time * (float)OUTPUT_FPS),
				                                                           sqdata.theQKeys[QKeyPos].qx,
																		   sqdata.theQKeys[QKeyPos].qy,
																		   sqdata.theQKeys[QKeyPos].qz,
																		   sqdata.theQKeys[QKeyPos].real);
#endif
#endif

			QKeyPos++;
		}
	}

	if (!sqdata.Write(pData, &datasize))
		return -1;

	pData += datasize;
	*pos  += datasize;

	// Output compressed translation data
	STData stdata(numObjAnims, header.numTKeys);
	int TKeyPos = 0;

	// Traverse the node list dumping out rotation keys
	for(i = 0; i < nodeList.GetSize(); i++)
	{
#ifndef NDEBUG
#ifdef  OBA_DEBUG
		fprintf(fpDebug, "Name: %s\n", (char*)nodeList[i].node->GetName());
#endif
#endif
		
		stdata.numKeysPerBone[i] = nodeList[i].numCompressedTFrames[0];

		for(j = 0; j < nodeList[i].numCompressedTFrames[0]; j++)
		{
			int curFrame = j;

			if (bCompressTime)
			{
				// This is still going to get dumped out as a float, so we need to make sure
				// the compiler makes the representation of the float match an int
				// We can do this since they're both 4 bytes
				int* val = (int*)&stdata.theTKeys[TKeyPos].time;
				float ftmp = (nodeList[i].TFrames[curFrame].time * (float)OUTPUT_FPS);
				*val = Round(ftmp);
			}
			else
				stdata.theTKeys[TKeyPos].time = nodeList[i].TFrames[curFrame].time;

			stdata.theTKeys[TKeyPos].tx   = nodeList[i].TFrames[curFrame].t[0];
			stdata.theTKeys[TKeyPos].ty   = nodeList[i].TFrames[curFrame].t[1];
			stdata.theTKeys[TKeyPos].tz   = nodeList[i].TFrames[curFrame].t[2];

#ifndef NDEBUG
#ifdef  OBA_DEBUG
			fprintf(fpDebug, "ST#: %i Time: %f  Frame: %i  Tx: %f  Ty: %f  Tz: %f\n", TKeyPos,
				                                                      stdata.theTKeys[TKeyPos].time,
																	  (int)(stdata.theTKeys[TKeyPos].time * (float)OUTPUT_FPS),
																	  stdata.theTKeys[TKeyPos].tx,
																	  stdata.theTKeys[TKeyPos].ty,
																	  stdata.theTKeys[TKeyPos].tz);
#endif
#endif

			TKeyPos++;
		}
	}
	
	if (!stdata.Write(pData, &datasize))
		return -1;

	pData += datasize;
	*pos  += datasize;

#ifndef NDEBUG
#ifdef  OBA_DEBUG
	fclose(fpDebug);
#endif
#endif

	// Script keys would be output at this point but, we've forced them to not export for now
	return numObjAnims;
}

///////////////////////// Cam Packet ////////////////////////////////////////////////////////////////////////////
int CutsceneExportDlg::GetKeyBeforeTime(STKey* keys, int nKeys, float time)
{
	for(int i = nKeys - 1; i >= 0; i--)
	{
		if (keys[i].time < time)
			return i;
	}

	return -1;
}

int CutsceneExportDlg::GetKeyBeforeTime(SQKey* keys, int nKeys, float time)
{
	for(int i = nKeys - 1; i >= 0; i--)
	{
		if (keys[i].time < time)
			return i;
	}

	return -1;
}

int CutsceneExportDlg::GetKeyAtTime(STKey* keys, int nKeys, float time)
{
	for(int i = nKeys - 1; i >= 0; i--)
	{
		if (keys[i].time == time)
			return i;
	}

	return -1;
}

int CutsceneExportDlg::GetKeyAtTime(SQKey* keys, int nKeys, float time)
{
	for(int i = nKeys - 1; i >= 0; i--)
	{
		if (keys[i].time == time)
			return i;
	}

	return -1;
}

bool CutsceneExportDlg::IsValidNoteKey(NoteKey* key, int start, int end)
{
	if (key->time > 0 &&
		key->time / GetTicksPerFrame() >  start &&
		key->time / GetTicksPerFrame() <= end)
		return true;

	return false;
}

float CutsceneExportDlg::GetStartTime(Cutscene* pCutscene)
{
	// Determine the object in the cutscene with the lowest start time and return
	Link<CutsceneObj>* link = pCutscene->objDB.GetHead();
	int minStart = 0;

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
			minStart = link->data.start;

		link = link->next;
	}

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
		{
			if (link->data.start < minStart)
				minStart = link->data.start;
		}

		link = link->next;
	}

	return ((float)minStart / (float)GetFrameRate());
}

// Returns true if any of the cameras in the cutscene have been modified
bool CutsceneExportDlg::CamerasModified(Cutscene* cutscene)
{
	Link<CutsceneObj>* linkCSO = cutscene->objDB.GetHead();

	while(linkCSO)
	{
		if (linkCSO->data.type & OBJTYPE_CAMERA && 
			linkCSO->data.bModified)
			return true;

		linkCSO = linkCSO->next;
	}

	return false;
}

int CutsceneExportDlg::BuildCompositeCamPacket(Cutscene* cutscene, unsigned char* pData, unsigned int* pos)
{	
	// Load packet from last exported file if the object wasn't modified and doing so is possible
	if (!CamerasModified(cutscene))
	{
		LIBObjPacket packet;
		void* pLoadedData;

		char buf[256];
		sprintf(buf, "Attempting to load packet for '%s' type '%s'\n", "", "CAM");

#ifdef PACKET_DEBUG
		//MessageBox(gInterface->GetMAXHWnd(), buf, "LoadPacket", MB_OK);
		FILE* fpDebug = fopen("c:\\cutpakdbg.txt", "a");
		fputs(buf, fpDebug);
		fclose(fpDebug);
#endif
		OutputDebugString(buf);

		if (pLoadedData = LoadPacket(curCutsceneFilename, 0, GenerateCRC("CAM"), &packet))
		{
			memcpy(pData, pLoadedData, packet.size);
			*pos += packet.size;
			return 1;
		}
	}

	bool bUseCompression;

	CameraExporter::bSwapCoordSystem = false;	// Swap camera export to game coordinate system
	CameraExporter::bRotateRoot      = false;

	// Alright, we need to scan through all the cameras in the cutscene and build keys for all of them
	Link<CutsceneObj>* csobj = cutscene->objDB.GetHead();

	Gfx::SAnimQFrame* compositeQframes = 0;		// Uncompressed Rotation list  (Quaternions) for current node
	Gfx::SAnimTFrame* compositeTframes = 0;		// Uncompressed Translation list for current node

	int numCompositeQframes = 0;
	int numCompositeTframes = 0;

	int startFrame = ip->GetAnimRange().Start() / GetTicksPerFrame();
	int endFrame   = ip->GetAnimRange().End() / GetTicksPerFrame();

	int numCompositeFOVKeys         = 0;
	GenKey<float>* compositeFOVKeys = NULL;
	
	int nCameras           = GetNumType(cutscene, OBJTYPE_CAMERA);
	int numFOVKeys         = 0;
	GenKey<float>* FOVKeys = NULL;

	bool bFirstCam = true;

	int  numNoteKeys = 0;

	while(csobj)
	{
		// We only care about the object if its a camera
		if (csobj->data.type & OBJTYPE_CAMERA)
		{
			if (bFirstCam)
			{
				startFrame = csobj->data.start;
				endFrame   = csobj->data.end;
			}

			// Determine the starting and ending frames
			if (csobj->data.start < startFrame)
				startFrame = csobj->data.start;

			if (csobj->data.end > endFrame)
				endFrame = csobj->data.end;

			//INode* node = ip->GetINodeByHandle(csobj->data.handle);
			INode* node = ip->GetINodeByName(csobj->data.name);
			
			// We are storing camera changes as note track notes
			// Determine the number of changes that exist
			if (node->NumNoteTracks() > 0)
			{
				// Determine the number of note track keys
				DefNoteTrack* notetrack = (DefNoteTrack*)node->GetNoteTrack(0);

				if (notetrack)
				{
					for(int k = 0; k < notetrack->keys.Count(); k++)
					{
						TimeValue time = notetrack->keys[k]->time;
						int       ticksperframe = GetTicksPerFrame();

						if (IsValidNoteKey(notetrack->keys[k], csobj->data.start, csobj->data.end))
							numNoteKeys++;
					}
				}
			}

			if (node)
			{
				int fps3DS=GetFrameRate();								// Frames per second MAX is set to
				Interval animRange    = gInterface->GetAnimRange();

				int num3DSFrames = csobj->data.end - csobj->data.start;
				int reqFrames    = num3DSFrames*OUTPUT_FPS/fps3DS;		// Number of frames at the game's frame rate
				
				unsigned int wsize = 0;									// Size of last write operation (when storing to mem mapped file)

				if (CameraExporter::skeleton)
					delete CameraExporter::skeleton;

				//skeleton=new CSkeletonData(node,false);
				CameraExporter::skeleton=new CSkeletonData(node,false,true);
				CameraExporter::rootbone=node;

				INode* root=node;
				int numNodes=CameraExporter::skeleton->GetCount();

				//GetAnim(root,start,end,errorQ,errorT,NULL,bUseCompression);
				//CameraExporter::GetAnim(root, csobj->data.start, csobj->data.end, 0.0f, 0.0f, NULL, false);

				if (csobj->data.rotTol  == 1.0f &&
					csobj->data.tranTol == 0.0f)
					bUseCompression = false;
				else
					bUseCompression = true;

				CStr root_name = root->GetName();

				float fCompress = 0.999700f + 0.000300f * csobj->data.rotTol;

				ProgressBar* pbarCam = new ProgressBar(hInstance, hwnd, "Exporting camera", 0, 1);
				pbarCam->AllowCancel(TRUE);

				CameraExporter::GetAnim(root, csobj->data.start, csobj->data.end, fCompress, csobj->data.tranTol, NULL, bUseCompression, pbarCam);

				if (pbarCam->WasCanceled())
				{
					// Copy the .cut file back from the default
					char* appenv = getenv(APP_ENV);
					if (!appenv)
					{
						char strErr[256];
						sprintf(strErr, "The environment variable '%s' is not defined can't continue.", APP_ENV);
						MessageBox(gInterface->GetMAXHWnd(), strErr, "Couldn't find environment variable", MB_ICONWARNING|MB_OK);

						delete pbarCam;
						return -1;
					}

					CStr filename = CStr(appenv) + CStr(CUTSCENE_PATH) + cutscene->name + ".cut";
					CStr curCutsceneFilename = ReplaceStr(filename, ".cut", ".cut.bak");

					if (!CopyFile(curCutsceneFilename, filename, FALSE))
					{
						char strErr[256];
						sprintf(strErr, "Failed to restore '%s' from backup '%s'.", (char*)filename, (char*)curCutsceneFilename);
						MessageBox(gInterface->GetMAXHWnd(), strErr, "Failed to restore .cut file from backup", MB_ICONWARNING|MB_OK);
					}

					delete pbarCam;
					return -1;
				}

				delete pbarCam;
				
				//if (bOneFrame)
				//CameraExporter::RemoveDblKeys(numNodes,num3DSFrames);		// test rm

				// Copy the rotation keyframes into the composite keyframe list
				// Composite frames add + 1 to leave space for dupe key that will be added so cuts between
				// cameras aren't interpolated for the one camera simulating the sequence in-game
				void* keyAddPos;
				if (!compositeQframes)
				{
					compositeQframes = (Gfx::SAnimQFrame*)malloc(sizeof(Gfx::SAnimQFrame) * CameraExporter::numCompressedQFrames[0]);
					keyAddPos = compositeQframes;
				}
				else
				{
					compositeQframes = (Gfx::SAnimQFrame*)realloc(compositeQframes, 
					                                              sizeof(Gfx::SAnimQFrame) * numCompositeQframes +
																  sizeof(Gfx::SAnimQFrame) * (CameraExporter::numCompressedQFrames[0] + 1));

					keyAddPos = compositeQframes + numCompositeQframes;

					// Handle inserted frame for camera cut (first encountered zero frame is assumed cut)
					(*((Gfx::SAnimQFrame*)keyAddPos)).time = 0.0f;

					keyAddPos = compositeQframes + (numCompositeQframes + 1);
				}

				memcpy(keyAddPos, CameraExporter::compQframes, sizeof(Gfx::SAnimQFrame) * CameraExporter::numCompressedQFrames[0]);
				numCompositeQframes += CameraExporter::numCompressedQFrames[0];

				// Copy the translation keyframes into the composite keyframe list
				if (!compositeTframes)
				{
					compositeTframes = (Gfx::SAnimTFrame*)malloc(sizeof(Gfx::SAnimTFrame) * CameraExporter::numCompressedTFrames[0]);
					keyAddPos = compositeTframes;
				}
				else
				{
					compositeTframes = (Gfx::SAnimTFrame*)realloc(compositeTframes, 
					                                              sizeof(Gfx::SAnimTFrame) * numCompositeTframes +
																  sizeof(Gfx::SAnimTFrame) * (CameraExporter::numCompressedTFrames[0] + 1));

					keyAddPos = compositeTframes + numCompositeTframes;

					// Handle inserted frame for camera cut (first encountered zero frame is assumed cut)
					(*((Gfx::SAnimTFrame*)keyAddPos)).time = 0.0f;

					keyAddPos = compositeTframes + (numCompositeTframes + 1);
				}

				memcpy(keyAddPos, CameraExporter::compTframes, sizeof(Gfx::SAnimTFrame) * CameraExporter::numCompressedTFrames[0]);
				numCompositeTframes += CameraExporter::numCompressedTFrames[0];

				// Each camera is exported as a seperate entity from its start to end so, its times
				// don't correlate with the timebar in MAX, we need to recompute times afterwards
				// based on camera starts and fixed sample rate based on FPS
				/*
				FOVKeys = GetFOVKeys(node, 
					                 csobj->data.start, 
									 csobj->data.end, 
									 &numFOVKeys,
									 0.0f, 
									 true);
				*/
				
				FOVKeys = GetFOVKeys(node, 
					                 csobj->data.start, 
									 csobj->data.end, 
									 &numFOVKeys,
									 csobj->data.tranTol, 
									 true);
									 //false);

				if (!compositeFOVKeys)
				{
					compositeFOVKeys = (GenKey<float>*)malloc(sizeof(GenKey<float>) * numFOVKeys);
					keyAddPos = compositeFOVKeys;
				}
				else
				{
					compositeFOVKeys = (GenKey<float>*)realloc(compositeFOVKeys,
						                                       sizeof(GenKey<float>) * numCompositeFOVKeys +
															   sizeof(GenKey<float>) * (numFOVKeys + 1));

					keyAddPos = compositeFOVKeys + numCompositeFOVKeys;

					// Handle inserted frame for camera cut (first encountered zero frame is assumed cut)
					(*((GenKey<float>*)keyAddPos)).time = 0.0f;

					keyAddPos = compositeFOVKeys + (numCompositeFOVKeys + 1);
				}

				memcpy(keyAddPos, FOVKeys, sizeof(GenKey<float>) * numFOVKeys);
				numCompositeFOVKeys += numFOVKeys;
			}
		}

		csobj = csobj->next;
	}

	// Go through the composite keyframes and update the time values ////////////////////////////////////////////
	int   i;
	int   lastFrame;
	float seqFrame;

	// Update rotation keys
	lastFrame = 0;

	for(i = 0; i < numCompositeQframes; i++)
	{
		if (i != 0 && compositeQframes[i].time == 0.0f)
		{
			// If this is the beginning of a new cam, set this as a dup key of the next key
			lastFrame = i;

			compositeQframes[i] = compositeQframes[i - 1];

			seqFrame = (compositeQframes[i+1].time * GetFrameRate()) + lastFrame;
			compositeQframes[i  ].time = (float)seqFrame / (float)GetFrameRate() - NEARKEY_TOL;
			compositeQframes[i+1].time = (float)seqFrame / (float)GetFrameRate();
			i++;
			continue;
		}

		// Compute the current frame in the sequence
		seqFrame = (compositeQframes[i].time * (float)GetFrameRate()) + lastFrame;

		compositeQframes[i].time = (float)seqFrame / (float)GetFrameRate();
	}

	// Update translation keys
	lastFrame = 0;

	for(i = 0; i < numCompositeTframes; i++)
	{
		if (i != 0 && compositeTframes[i].time == 0.0f)
		{
			// If this is the beginning of a new cam, set this as a dup key of the next key
			lastFrame = i;

			compositeTframes[i] = compositeTframes[i - 1];

			// Compute the current frame in the sequence
			seqFrame = (compositeTframes[i+1].time * GetFrameRate()) + lastFrame;
			compositeTframes[i  ].time = (float)seqFrame / (float)GetFrameRate() - NEARKEY_TOL;
			compositeTframes[i+1].time = (float)seqFrame / (float)GetFrameRate();
			i++;
			continue;
		}

		// Compute the current frame in the sequence
		seqFrame = (compositeTframes[i].time * (float)GetFrameRate()) + lastFrame;

		compositeTframes[i].time = (float)seqFrame / (float)GetFrameRate();
	}

	// Update FOV keys
	lastFrame = 0;

	for(i = 0; i < numCompositeFOVKeys; i++)
	{
		if (i != 0 && compositeFOVKeys[i].time == 0.0f)
		{
			// If this is the beginning of a new cam, set this as a dup key of the next key
			lastFrame = i;

			compositeQframes[i] = compositeQframes[i - 1];

			// Compute the current frame in the sequence
			seqFrame = (compositeFOVKeys[i+1].time * GetFrameRate()) + lastFrame;
			compositeFOVKeys[i  ].time = (float)seqFrame / (float)GetFrameRate() - NEARKEY_TOL;
			compositeFOVKeys[i+1].time = (float)seqFrame / (float)GetFrameRate();
			i++;
			continue;
		}

		// Compute the current frame in the sequence
		seqFrame = (compositeFOVKeys[i].time * GetFrameRate()) + lastFrame;

		compositeFOVKeys[i].time = (float)seqFrame / (float)GetFrameRate();
	}

	/////////////////////////////////////////////////////////////////////////////////////////////////

	// Compute total QFrames and TFrames
	int QKeys = numCompositeQframes;
	int TKeys = numCompositeTframes;

	// Output packet header
	SHeader header;
	header.version=ANIMFILE_VERSION;
	header.flags=INTERMEDIATE_FORMAT | CUTSCENE;
	
	if (!bUseCompression)
		header.flags|=UNCOMPRESSED_FORMAT;

	//if (bCompressTime)
	//	header.flags|=COMPRESSEDTIME;

	// Cameras always had pre rotated root, assign the flag for it
	//header.flags |= PREROTATEDROOT;

	header.flags |= CONTAINS_CAMDATA;
	header.flags |= CUSTOMKEYSAT60FPS;

	header.skeletonName=CAMERA_CRC;			// hard-coded default for now
	header.duration = (float)(endFrame - startFrame)/(float)GetFrameRate();
	header.numBones = 1;
	header.numQKeys = QKeys;
	header.numTKeys = TKeys;

	// We're storing camera changes now as Note Track info
	// need to add the total number of note track keys to our user defined keys
	header.numUserDefinedKeys = numCompositeFOVKeys + numNoteKeys * 3 + GetTotalScriptKeys(cutscene);

	unsigned int wsize = 0;

	if (!header.Write(pData, &wsize))
	{
		delete CameraExporter::skeleton;
		CameraExporter::skeleton = NULL;
		return -1;
	}

	pData += wsize;
	*pos  += wsize;

	// Output Skeleton data

	// Output using CSkeleton instead
	wsize = CameraExporter::skeleton->Write(pData);

	if (!wsize)
	{
		delete CameraExporter::skeleton;
		CameraExporter::skeleton=NULL;
		return -1;
	}

	pData += wsize;
	*pos  += wsize;

	///////////////////////////////////////////////////

#ifdef DUMP_CAMERA_DEBUG
	FILE* fpDebug = fopen("c:\\camdebug.txt", "w");
#endif

	// Output compressed rotation data
	SQData sqdata(1,QKeys);

	sqdata.numKeysPerBone[0] = QKeys;
		
	fprintf(fpDebug, "Num Q-Keys: %i\n", QKeys);

	for(int j = 0;j < QKeys; j++)\
	{
		sqdata.theQKeys[j].time = compositeQframes[j].time;
		sqdata.theQKeys[j].qx   = compositeQframes[j].q[X];
		sqdata.theQKeys[j].qy   = compositeQframes[j].q[Y];
		sqdata.theQKeys[j].qz   = compositeQframes[j].q[Z];
		sqdata.theQKeys[j].real = compositeQframes[j].q[W];

#ifdef DUMP_CAMERA_DEBUG
		fprintf(fpDebug, "#: %i  Time: %f  Qx: %f  Qy: %f  Qz: %f\n", j,
			                                                        compositeQframes[j].time,
			                                                        compositeQframes[j].q[X],
														            compositeQframes[j].q[Y],
														            compositeQframes[j].q[Z],
														            compositeQframes[j].q[W]);
#endif
	}

	if (!sqdata.Write(pData, &wsize))
		return -1;

	pData += wsize;
	*pos  += wsize;

	// Output compressed translation data
	STData stdata(1,TKeys);

	stdata.numKeysPerBone[0] = TKeys;

	fprintf(fpDebug, "Num T-Keys: %i\n", TKeys);

	for(i = 0;i < TKeys; i++)
	{
		stdata.theTKeys[i].time = CameraExporter::compTframes[i].time;
		//stdata.theTKeys[i].time = i;										// Really likely to change back to time
																			// But, for now time duplicates frame number
																			// (Not my choice)
		stdata.theTKeys[i].tx   = CameraExporter::compTframes[i].t[0];
		stdata.theTKeys[i].ty   = CameraExporter::compTframes[i].t[1];
		stdata.theTKeys[i].tz   = CameraExporter::compTframes[i].t[2];

#ifdef DUMP_CAMERA_DEBUG
		fprintf(fpDebug, "#: %i  Time: %f  X: %f Y: %f Z: %f\n", i,
			                                                     stdata.theTKeys[i].time,
			                                                     stdata.theTKeys[i].tx,
													             stdata.theTKeys[i].ty,
													             stdata.theTKeys[i].tz);
#endif
	}

	if (!stdata.Write(pData, &wsize))
		return -1;

	pData += wsize;
	*pos  += wsize;

#ifdef DUMP_CAMERA_DEBUG
	fclose(fpDebug);
#endif

	// Acquire FOV keys and output
	//GenKey<float>* CameraExporter::GetFOVKeys(INode* node, int start, int end, int* numKeys,
	//									  float error, bool bOneFrame)

	// Build sorted list of all the user defined keys including FOV and camera changes
	// Build FOV list
	unsigned long  nUserKeyBytes = numCompositeFOVKeys * KEYSIZE_CHANGEFOV + numNoteKeys * 3 * KEYSIZE_CHANGECAM + GetTotalScriptKeys(cutscene) * KEYSIZE_RUNSCRIPT;
	unsigned char* pUserKeys     = (unsigned char*)malloc(nUserKeyBytes);
	unsigned char* pUserKeyPos   = pUserKeys;
	unsigned int   ukSize;

	// Store the start point of the custom keys because they'll need sorted after the first write
	unsigned char* pPosPreKey = pUserKeyPos;

	// Add the FOV Keys
	fprintf(fpDebug, "NumFOVKeys: %i\n", numCompositeFOVKeys);

	for(i = 0; i < numCompositeFOVKeys; i++)
	{
		fprintf(fpDebug, "#: %i  Time: %f   Value: %f (Deg: %f)\n", i,
			                                                        compositeFOVKeys[i].time,
															        compositeFOVKeys[i].key,
																	Mth::RadToDeg(compositeFOVKeys[i].key));
		
		if (!ChangeFovKey(compositeFOVKeys[i].time,compositeFOVKeys[i].key).Write(pUserKeyPos, &ukSize))
		{
			// Free memory
			delete CameraExporter::skeleton;
			CameraExporter::skeleton=NULL;

			free(compositeQframes);
			free(compositeTframes);
			free(compositeFOVKeys);

			delete [] FOVKeys;
			
			return nCameras;
		}
		
		pUserKeyPos += ukSize;
	}

	// Add the Camera change keys from the note track (we assume all keys are at track 0)
	csobj = cutscene->objDB.GetHead();

	while(csobj)
	{
		if (csobj->data.type & OBJTYPE_CAMERA)
		{
			INode* node = ip->GetINodeByName(csobj->data.name);

			if (node->NumNoteTracks() > 0)
			{
				DefNoteTrack* notetrack = (DefNoteTrack*)node->GetNoteTrack(0);

				if (notetrack)
				{
					int nNoteTrackKeys = notetrack->keys.Count();
					float startTime = (float)csobj->data.start / (float)GetFrameRate();		// Start time should be subtracted so sequence starts at 0

					for(int k = 0; k < notetrack->keys.Count(); k++)
					{
							NoteKey* notekey = *(notetrack->keys.Addr(k));

							// The note key should only be processed if its within the range that we're exporting
							if (!IsValidNoteKey(notekey, csobj->data.start, csobj->data.end))
								continue;

							float time = (float)notekey->time / (float)GetTicksPerFrame() / (float)GetFrameRate();

							int refKeyT = GetKeyBeforeTime(stdata.theTKeys, TKeys, time);
							int refKeyQ = GetKeyBeforeTime(sqdata.theQKeys, QKeys, time);

							// The key that is closest to the target time will be used
							float preTime = time - FRAME_INCR;

							if (preTime < 0)
								preTime = 0;

							/*
							if (stdata.theTKeys[refKeyT].time > sqdata.theQKeys[refKeyQ].time)
								preTime = stdata.theTKeys[refKeyT].time;
							else
								preTime = sqdata.theQKeys[refKeyQ].time;

							if (preTime == -1)
								preTime = time;
							*/

							Matrix3 tm;
							Point3  trans;
							Quat    q;

							INode* preCamNode, *camNode, *postCamNode;
							
							if (k == 0)
								preCamNode = node;
							else
							{
								NoteKey* notekey = *(notetrack->keys.Addr(k - 1));
								preCamNode = ip->GetINodeByName(notekey->note);
							}

							if (!preCamNode)
								preCamNode = node;


							camNode     = node;
							postCamNode = ip->GetINodeByName(notekey->note);

							if (!postCamNode)
								postCamNode = node;

							// Add a key just before the pretime
							if (preTime >= FRAME_INCR)
								tm    = preCamNode->GetNodeTM((preTime - FRAME_INCR) * GetFrameRate() * GetTicksPerFrame());
							else
								tm    = preCamNode->GetNodeTM((preTime) * GetFrameRate() * GetTicksPerFrame());

							trans = tm.GetTrans();
							q     = tm;

							float useTime = preTime - FRAME_INCR - startTime;
							if (useTime < 0)
								useTime = 0;

							if (!ChangeCamKey(useTime,
											  trans.x,
											  trans.y,
											  trans.z,
											  q.x,
											  q.y,
											  q.z,
											  q.w).Write(pUserKeyPos, &ukSize))

							{
								// Free memory
								delete CameraExporter::skeleton;
								CameraExporter::skeleton=NULL;

								free(compositeQframes);
								free(compositeTframes);
								free(compositeFOVKeys);
								delete [] FOVKeys;
									
								return nCameras;							
							}

							pUserKeyPos += ukSize;

							// Add a key at the pre time
							tm    = camNode->GetNodeTM(preTime * GetFrameRate() * GetTicksPerFrame());
							trans = tm.GetTrans();
							q     = tm;

							useTime = preTime - startTime;
							if (useTime < 0)
								useTime = 0;

							if (!ChangeCamKey(useTime,
											  trans.x,
											  trans.y,
											  trans.z,
											  q.x,
											  q.y,
											  q.z,
											  q.w).Write(pUserKeyPos, &ukSize))

							{
								// Free memory
								delete CameraExporter::skeleton;
								CameraExporter::skeleton=NULL;

								free(compositeQframes);
								free(compositeTframes);
								free(compositeFOVKeys);
								delete [] FOVKeys;
									
								return nCameras;							
							}

							pUserKeyPos += ukSize;


							// It's possible that the at key time may not exist due to camera compression
							// so, we need to manually evaluate the camera's position/rotation at
							// that time
							tm    = postCamNode->GetNodeTM(time * GetFrameRate() * GetTicksPerFrame());
							trans = tm.GetTrans();
							q = tm;

							// add a 3rd key at the exactcamera change time
							useTime = time - startTime;
							if (useTime < 0)
								useTime = 0;

							if (!ChangeCamKey(useTime,
											  trans.x,
											  trans.y,
											  trans.z,
											  q.x,
											  q.y,
											  q.z,
											  q.w).Write(pUserKeyPos, &ukSize))

							{
								// Free memory
								delete CameraExporter::skeleton;
								CameraExporter::skeleton=NULL;

								free(compositeQframes);
								free(compositeTframes);
								free(compositeFOVKeys);
								delete [] FOVKeys;
									
								return nCameras;							
							}

							pUserKeyPos += ukSize;
					}
				}
			}	// Endif NumNoteTracks > 0
		}

		csobj = csobj->next;
	}

	// Insert all the script keys for the cutscene
	// Insert global keys
	float startTime = GetStartTime(cutscene);
	Link<TrackUIKey>* linkKey = cutscene->trackKeyDB.GetHead();

	while(linkKey)
	{
		// Calc time in seconds
		float time = (float)linkKey->data.time / (float)GetTicksPerFrame() / (float)GetFrameRate();

		if (linkKey->data.GetClass() == CStr(KEYOBJ_KEY_NAME))
		{
			if (linkKey->data.GetType() == CStr(KEYOBJ_ENABLE_KEY_NAME))
			{
				if (!EnableObjKey(time - startTime,
					              GenerateCRC(linkKey->data.GetProp("Node"))).Write(pUserKeyPos, &ukSize))
				{
					// Free memory
					delete CameraExporter::skeleton;
					CameraExporter::skeleton=NULL;

					free(compositeQframes);
					free(compositeTframes);
					free(compositeFOVKeys);
					delete [] FOVKeys;
						
					return nCameras;							
				}
			}

			if (linkKey->data.GetType() == CStr(KEYOBJ_DISABLE_KEY_NAME))
			{
				if (!DisableObjKey(time - startTime,
					               GenerateCRC(linkKey->data.GetProp("Node"))).Write(pUserKeyPos, &ukSize))
				{
					// Free memory
					delete CameraExporter::skeleton;
					CameraExporter::skeleton=NULL;

					free(compositeQframes);
					free(compositeTframes);
					free(compositeFOVKeys);
					delete [] FOVKeys;
						
					return nCameras;							
				}
			}
		}
		else
		{
			if (!RunScriptKey(time - startTime, 
							  GenerateCRC(GetGlobalScriptName(cutscene, &linkKey->data))).Write(pUserKeyPos, &ukSize))
			{
				// Free memory
				delete CameraExporter::skeleton;
				CameraExporter::skeleton=NULL;

				free(compositeQframes);
				free(compositeTframes);
				free(compositeFOVKeys);
				delete [] FOVKeys;
					
				return nCameras;							
			}
		}

		pUserKeyPos += ukSize;

		linkKey = linkKey->next;
	}

	// Insert object keys
	Link<CutsceneObj>* linkKeyCSO = cutscene->objDB.GetHead();

	while(linkKeyCSO)
	{
		Link<TrackUIKey>* linkKey = linkKeyCSO->data.trackKeyDB.GetHead();

		while(linkKey)
		{
			// Calc time in seconds
			float time = (float)linkKey->data.time / (float)GetTicksPerFrame() / (float)GetFrameRate();

			if (!RunScriptKey(time - startTime, 
				              GenerateCRC(GetObjScriptName(cutscene, &linkKeyCSO->data, &linkKey->data))).Write(pUserKeyPos,
							                                                                                    &ukSize))
			{
				// Free memory
				delete CameraExporter::skeleton;
				CameraExporter::skeleton=NULL;

				free(compositeQframes);
				free(compositeTframes);
				free(compositeFOVKeys);
				delete [] FOVKeys;
					
				return nCameras;
			}

			pUserKeyPos += ukSize;

			linkKey = linkKey->next;
		}

		linkKeyCSO = linkKeyCSO->next;
	}

	// Sort the user keys to the appropriate time order (since they're actually just mem, and not in a file yet)
	SortUserKeys((SUserDefinedKey*)pUserKeys, header.numUserDefinedKeys); 
	//memcpy(pPosPreKey, pUserKeys, GetUserKeyMemoryUsage(pUserKeys, header.numUserDefinedKeys));

	// Dump custom keys to debug file
#ifdef DUMP_DEBUG_CUSTKEYS
	FILE* fpCustDebug = fopen("c:\\custkeydbg.txt", "w");

	//unsigned char* pCurKey = (unsigned char*)pUserKeys;
	unsigned char* pCurKey = (unsigned char*)pPosPreKey;

	for(int curCustKey = 0; curCustKey < header.numUserDefinedKeys; curCustKey++)
	{
		SUserDefinedKey* pudk = (SUserDefinedKey*)pCurKey;
		fprintf(fpCustDebug, "%i: Time: %f  Type: %i  Size: %i  ", curCustKey, pudk->timeStamp, pudk->keyType, pudk->size);
		
		switch(pudk->keyType)
		{
		case KEYTYPE_CHANGEPARENTKEY:
			{
				ChangeParentKey* cpk = (ChangeParentKey*)pudk;
				fprintf(fpCustDebug, "newParentBoneId: %x\n", cpk->newParentBoneId);
			}
			break;

		case KEYTYPE_CHANGEFOV:
			{
				ChangeFovKey* cfk = (ChangeFovKey*)pudk;
				fprintf(fpCustDebug, "NewFov: %f\n", cfk->newFov);
			}
			break;

		case KEYTYPE_CHANGECAM:
			{
				ChangeCamKey* cck = (ChangeCamKey*)pudk;

				fprintf(fpCustDebug, "Pos: %f, %f, %f, %f  Rot: %f, %f, %f, %f\n", 
					cck->x,  cck->y,  cck->z,  cck->w, 
					cck->rx, cck->ry, cck->rz, cck->rw);

			}
			break;

		case KEYTYPE_RUNSCRIPT:
			{
				RunScriptKey* rsk = (RunScriptKey*)pudk;

				fprintf(fpCustDebug, "scriptName: %x\n", rsk->scriptName);
			}
			break;

		default:
			fprintf(fpCustDebug, "\n");
			break;
		}

		pCurKey += ((SUserDefinedKey*)pCurKey)->size;
	}

	fclose(fpCustDebug);
#endif

	// Dump the user defined keys out to the buffer
	memcpy(pData, pUserKeys, nUserKeyBytes);
	pData += nUserKeyBytes;
	*pos  += nUserKeyBytes;

	// Free memory
	delete CameraExporter::skeleton;
	CameraExporter::skeleton=NULL;

	free(compositeQframes);
	free(compositeTframes);
	free(compositeFOVKeys);

	delete [] FOVKeys;
	
	return nCameras;
}

void CutsceneExportDlg::ExportOptions()
{
	// Determine the cutscene object that we're setting options on
	int nSel = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	// Must have one and only one cutscene selected
	if (nSel != 1)
		return;

	// Must have one and only one object selected within that cutscene
	int nObjSel = pObjList->GetSelCount();

	if (nObjSel != 1)
		return;

	int idx = pObjList->GetSelItem();

	if (idx == -1)
		return;

	// Acquire the current cutscene
	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* cutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	if (!cutscene)
		return;

	// Acquire object node
	INode* node = (INode*)pObjList->GetItemData(idx);

	if (node)
	{
		Link<CutsceneObj>* cutsceneObj = GetCutsceneObj(&cutscene->data, node);

		if (!cutsceneObj)
			return;

		// Open the Bone Tolerance Editor dialog
		CutsceneBoneTolDlg* boneTolDlg = new CutsceneBoneTolDlg(hInstance, hwnd, ip, &cutsceneObj->data);
		boneTolDlg->Show();	// Modal Dlgs block

		cutsceneObj->data.SaveToNode(cutscene->data.name);
		delete boneTolDlg;
	}
}

void CutsceneExportDlg::BuildExportableNodeList(INodeTab& exportable_nodes, INode* node)
{
	// This node and all of its children will be added to the node list
	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		BuildExportableNodeList(exportable_nodes, child);
	}

	exportable_nodes.Append(1, &node);
}

CutsceneObj* CutsceneExportDlg::GetObjByTypeAndModelName(Cutscene* cutscene, DWORD objType, CStr modelName)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & objType && link->data.modelName == modelName)
			return &link->data;

		link = link->next;
	}

	return NULL;
}

int CutsceneExportDlg::BuildSkinPacket(Cutscene* cutscene, CutsceneObj* csobj, unsigned char* pData, unsigned int* pos,
									                                           unsigned char* pDataTEX, unsigned int* posTEX)
{
	ISkinExporter*  skinExporter  = GetSkinExporter();
	IModelExporter* modelExporter = GetModelExporter();
	IoUtils::CVirtualOutputFile skinBuf, texBuf;

	INodeTab exportable_nodes;
	//INode*   node = ip->GetINodeByHandle(csobj->handle);
	INode*   node = ip->GetINodeByName(csobj->name);

	if (!node)
		return 0;

	skinBuf.Init(1024 * 1024 * 5);

	int version;
	bool sky_export = false;
	version = NxMaterial::vVERSION_NUMBER;
	skinBuf.Write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	skinBuf.Write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	skinBuf.Write((const char*) &version, sizeof( int ));
	skinBuf.Write((const char*) &sky_export, sizeof( bool ));

	// Reset texture export so the next dumped texture library only contains materials for this model
	ITextureExporter* tex_exp = GetTextureExporter();
	tex_exp->Reset();

	BuildExportableNodeList(exportable_nodes, node);

	// The model may be being built from a node that doesn't start at the actual root
	// Scan through the cutscene objects to see if there is a matching boned anim export
	// and acquire the starting skeletal bone from that
	//INode* rootBone = FindBoneRootByModelName(cutscene, csobj->modelName);
	INode* rootBone = ip->GetINodeByName(csobj->rootBone);

	// If the root bone could not be found by the root bone name, try to find it by the model name
	if (!rootBone)
		rootBone = FindBoneRootByModelName(cutscene, csobj->modelName);

	if (!rootBone)
	{
		char strErr[256];
		sprintf(strErr, "Failed to find root bone for '%s'", (char*)csobj->name);
		MessageBox(hwnd, strErr, "Couldn't find root bone", MB_ICONWARNING|MB_OK);
		return 0;
	}

	CStr name = rootBone->GetName();

	NxModel* model = modelExporter->ExtractModelData( exportable_nodes, true, 1.0f, rootBone, true ); // Force Local coords
	
	tex_exp->LoadTextureData();

	/*
	// Flag the model data as being skinned
	int nObjs = model->m_Objects.Count();

	for(int i = 0; i < nObjs; i++)
		model->m_Objects[i]->m_Flags |= NxObject::mSKINNED | NxObject::mHAS4WEIGHTS;
	/////
	*/

	skinExporter->SetExportModel(model);

	skinExporter->SaveMaterials( skinBuf, model );

	int offset = skinBuf.GetBufferSize();
	skinExporter->SaveGeometry( skinBuf );

	int bufSize = skinBuf.GetBufferSize();
	memcpy(pData, skinBuf.GetBuffer(), bufSize);
	*pos += bufSize;

	// Write out the texture dictioary to memory
	texBuf.Init(vMAX_TEXTURE_FILE_SIZE);
	
	CStr texName = csobj->name + ".tex";
	//CStr usgName = csobj->name + ".usg";
	tex_exp->MarkAsDirty();
	//tex_exp->SaveTextureDictionary( texName, usgName, &texBuf, NULL);

	// No .usg file should be generated
	tex_exp->SaveTextureDictionary( texName, NULL, &texBuf, NULL);

	bufSize = texBuf.GetBufferSize();
	memcpy(pDataTEX, texBuf.GetBuffer(), bufSize);
	*posTEX += bufSize;

	skinBuf.Uninit();
	texBuf.Uninit();

	if (model)
		delete model;

	return 1;
}

int CutsceneExportDlg::BuildModelPacket(CutsceneObj* csobj, unsigned char* pData, unsigned int* pos,
										                    unsigned char* pDataTEX, unsigned int* posTEX)
{
	IModelExporter* modelExporter = GetModelExporter();
	IoUtils::CVirtualOutputFile modelBuf, texBuf;

	INodeTab exportable_nodes;
	//INode*   node = ip->GetINodeByHandle(csobj->handle);
	INode*   node = ip->GetINodeByName(csobj->name);

	if (!node)
		return 0;

	expNode = ip->GetINodeByName(csobj->name);

	ITextureExporter* tex_exp = GetTextureExporter();
	tex_exp->Reset();

	//BuildExportableNodeList(exportable_nodes, node);	
	exportable_nodes.Append(1, &node);
	NxModel* model = modelExporter->ExtractModelData( exportable_nodes, false, 1.0f, NULL, true );	// Force local coords

	tex_exp->LoadTextureData();

	if (!model)
		return 0;

	modelExporter->SetExportModel(model);

	modelBuf.Init(1024 * 1024 * 3);  // 3 MB model buffer

	int version;
	bool sky_export = false;

	version = NxMaterial::vVERSION_NUMBER;
	modelBuf.Write((const char*) &version, sizeof( int ));
	version = NxMesh::vVERSION_NUMBER;
	modelBuf.Write((const char*) &version, sizeof( int ));
	version = NxVertex::vVERSION_NUMBER;
	modelBuf.Write((const char*) &version, sizeof( int ));
	modelBuf.Write((const char*) &sky_export, sizeof( bool ));
	
	modelExporter->SaveMaterials( modelBuf, model );
	modelExporter->SaveGeometry( modelBuf );

	int bufSize = modelBuf.GetBufferSize();
	memcpy(pData, modelBuf.GetBuffer(), bufSize);
	*pos += bufSize;

	modelBuf.Uninit();

	// Write out the texture dictioary to memory
	texBuf.Init(vMAX_TEXTURE_FILE_SIZE);
	
	CStr texName = csobj->name + ".tex";
	//CStr usgName = csobj->name + ".usg";
	
	// Since the exporter now checks if it needs to export new tex dictionaries
	// we need to force it to always export when building the .cut file
	tex_exp->MarkAsDirty();
	//tex_exp->SaveTextureDictionary( texName, usgName, &texBuf, NULL);

	// No .usg file should be generated
	tex_exp->SaveTextureDictionary(texName, NULL, &texBuf, NULL);

	bufSize = texBuf.GetBufferSize();
	memcpy(pDataTEX, texBuf.GetBuffer(), bufSize);
	*posTEX += bufSize;

	texBuf.Uninit();
	delete model;
	return 1;
}

bool CutsceneExportDlg::SetSkeletonExceptions(INode* node, CSkeletonData* skeleton)
{
	for(int i = 0; i < NUM_FAKE_BONES; i++)
		skeleton->AddIgnoreException(FindNode(FakeBoneNames[i], node));	

	// In the event that the model does not contain a single brow bone (Bone_Brow)
	// we should change Bone_Brow to Bone_Brow_MID (assuming that the model then uses
	// a three brow bone structure), then rename to Bone_Brow after construction
	// and throw out the middle bone data (placeholder)
	CStr name;
	name = node->GetName();

	if (!FindNode("Bone_Brow", node))
	{
		skeleton->AddIgnoreException(FindNode("Bone_Brow_MID", node));
		return true;
	}

	return false;
}

bool CutsceneExportDlg::BoneIsFake(INode* root, CutsceneObj* csobj, INode* node)
{
	if (!csobj->bHiresSkeleton)
		return false;

	if (csobj->bCopy)
		return false;
	
	// Neck bone is never fake.  Exists in both upper and lower portions of body
	INode* nodeNeck = FindNode("Bone_Neck", node);
	if (node == nodeNeck)
		return false;

	// The brow bone is only fake in the body portion of the skeleton
	INode* nodeBrow = FindNode("Bone_Brow", node);

	if (nodeBrow)
	{
		if ( node == nodeBrow )
		{
			if (csobj->bHiresSkeleton && !csobj->bCopy)
				return true;
			else
				return false;
		}
	}

	nodeBrow = FindNode("Bone_Brow_MID", node);

	if (nodeBrow)
	{
		if ( node == nodeBrow )
		{
			if (csobj->bHiresSkeleton && !csobj->bCopy)
				return true;
			else
				return false;
		}
	}

	for(int i = 0; i < NUM_FAKE_BONES; i++)
		if (FindNode(FakeBoneNames[i], root) == node)
			return true;

	return false;
}

INode* CutsceneExportDlg::FindRoot(INode* node)
{
	if (!node)
		return NULL;

	while(node->GetParentNode() != ip->GetRootNode())
		node = node->GetParentNode();
	
	return node;
}

// Determines the camera being used for a specific time within the cutscene
INode* CutsceneExportDlg::GetNoteTrackCam(Cutscene* cutscene, float time, TimeValue* camTime)
{
	// Scan through the exported cameras
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_CAMERA)
		{
			// Make sure this camera falls within the given timeframe
			if (link->data.start <= time &&
				link->data.end >= time)
			{
				INode* node = gInterface->GetINodeByName(link->data.name);

				if (node)
				{
					// Scan through the camera's notetracks to find which camera should be active
					if (node->NumNoteTracks() > 0)
					{
						// Determine the number of note track keys
						DefNoteTrack* notetrack = (DefNoteTrack*)node->GetNoteTrack(0);

						if (notetrack && notetrack->keys.Count() >= 0)
						{
							int       keyCam = 0;	// The key that identifies the active camera
							TimeValue finalTime  = 0;
						
							for(int k = 0; k < notetrack->keys.Count(); k++)
							{
								TimeValue keytime = notetrack->keys[k]->time;

								if (IsValidNoteKey(notetrack->keys[k], link->data.start, link->data.end))
								{
									if ((keytime / GetTicksPerFrame()) <= time && (keytime / GetTicksPerFrame()) > finalTime)
									{
										keyCam = k;
										finalTime  = keytime;
									}
								}
							}

							// keyCam is now equal to the camera track key closest to our intended time
							INode* cam = gInterface->GetINodeByName(notetrack->keys[keyCam]->note);

							if (camTime)
								*camTime = finalTime;

							return cam;
						}
					}
				}
			}
		}

		link = link->next;
	}

	if (camTime)
		*camTime = -1;

	return NULL;
}

void CutsceneExportDlg::RemoveInvisibleKeys(void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	// Acquire cam
	/*
	INode* nodeCam = GetNoteTrackCam(pthis);
	if (!nodeCam)
		return;
	*/

	// List rebuild method
	/*
	Gfx::SAnimQFrame* QframeCpy = new Gfx::SAnimQFrame[pthis->ObjectExporter::numFrames];
	Gfx::SAnimTFrame* TframeCpy = new Gfx::SAnimTFrame[pthis->ObjectExporter::numFrames];
	int nFramesCpy = 0;
	
	// Run through each node being exported and scan its visibility info
	int nBones = pthis->ObjectExporter::skeleton->GetCount();

	for(int i = 0; i < nBones; i++)
	{
		INode* bone = pthis->ObjectExporter::skeleton->GetBone(i);

		for(int j = 0; j < pthis->ObjectExporter::numFrames; j++)
		{
			int curframe = j + i * nBones;

			// Scan Q/T frames for visibility
			float vis = bone->GetVisibility(pthis->ObjectExporter::Qframes[curframe].time);
			
			assert(pthis->ObjectExporter::Qframes[curframe].time == pthis->ObjectExporter::Tframes[curframe].time);

			// Only visible frames are copied
			if (vis > 0)
			{
				// Copy data
				pthis->ObjectExporter::Qframes[nFramesCpy] = pthis->ObjectExporter::Qframes[curframe];
				pthis->ObjectExporter::Tframes[nFramesCpy] = pthis->ObjectExporter::Tframes[curframe];
				nFramesCpy++;
			}
			else
				OutputDebugString("Detected invisible frame\n");
		}
	}

	delete [] pthis->ObjectExporter::Qframes;
	delete [] pthis->ObjectExporter::Tframes;

	pthis->ObjectExporter::Qframes = QframeCpy;
	pthis->ObjectExporter::Tframes = TframeCpy;

	pthis->ObjectExporter::numFrames = nFramesCpy;
	*/

	/*
	// Duplicate key method 
	// Run through each node being exported and scan its visibility info
	int nBones = pthis->ObjectExporter::skeleton->GetCount();

	for(int i = 0; i < nBones; i++)
	{
		INode* bone = pthis->ObjectExporter::skeleton->GetBone(i);

		for(int j = 0; j < pthis->ObjectExporter::numFrames; j++)
		{
			int curframe = j + i * nBones;

			// Scan Q/T frames for visibility
			float vis = bone->GetVisibility(pthis->ObjectExporter::Qframes[curframe].time);
			
			assert(pthis->ObjectExporter::Qframes[curframe].time == pthis->ObjectExporter::Tframes[curframe].time);

			// If visibility is less than 0 this indicates off
			if (vis < 0)
			{
				// Copy data from previous frame so it gets removed as a dupe
				if (curframe != 0)
				{
					pthis->ObjectExporter::Qframes[curframe] = pthis->ObjectExporter::Qframes[curframe - 1];
					pthis->ObjectExporter::Tframes[curframe] = pthis->ObjectExporter::Tframes[curframe - 1];
					OutputDebugString("Detected invisible frame\n");
				}
			}
		}
	}
	*/

	// Duplicate key method 
	// Run through each node being exported and scan its visibility info
	int nBones;
	
	if (pthis->bexpModel)
		nBones = 1;
	else
		nBones = pthis->ObjectExporter::skeleton->GetCount();
	
	CStr name = pthis->expNode->GetName();

	if (name == CStr("Skin_tomas"))
	{
		int zzz;
		zzz = 0;
	}

	int  TPF = GetTicksPerFrame();
	bool bWasVisible = true;			// Keeps track of prior state.  The first key should be retained on a vis change
	INode* curBone;

	int visOffCount = 0;				// DEBUG: Visibility keys
	int visOffPotentialDrop = 0;
	int totalKeys = 0;

	for(int i = 0; i < nBones; i++)
	{
		if (pthis->bexpModel)
			curBone = pthis->expNode;
		else
			curBone = pthis->ObjectExporter::skeleton->GetBone(i);

		for(int j = 0; j < pthis->ObjectExporter::numFrames; j++)
		{
			totalKeys++;

			//int curframe = j + i * nBones;
			int curframe = j + i * pthis->ObjectExporter::numFrames;

			// Scan Q/T frames for 2
			// Account for relative time from export start frame
			float vis;
			
			TimeValue t = ((pthis->ObjectExporter::Qframes[curframe].time * GetFrameRate()) + pthis->expCso->start) * GetTicksPerFrame();
			
			/*
			char buf[256];
			sprintf(buf, "Vis key check time: %i\n", t);
			OutputDebugString(buf);
			*/

			if (IsDlgButtonChecked(pthis->hwnd, IDC_PERBONEVIS) == BST_CHECKED)
				vis = curBone->GetVisibility(((pthis->ObjectExporter::Qframes[curframe].time * GetFrameRate()) + pthis->expCso->start) * GetTicksPerFrame());
			else
				vis = pthis->expNode->GetVisibility(((pthis->ObjectExporter::Qframes[curframe].time * GetFrameRate()) + pthis->expCso->start) * GetTicksPerFrame());
			
			assert(pthis->ObjectExporter::Qframes[curframe].time == pthis->ObjectExporter::Tframes[curframe].time);

			// If visibility is less than 0 this indicates off
			if (vis <= 0)
			{
				//__asm int 3;
				visOffPotentialDrop++;

				/*
				if (bWasVisible)
				{
					bWasVisible = false;
					continue;
				}
				*/

				int time = pthis->ObjectExporter::Qframes[curframe].time;

				// Copy data from previous frame so it gets removed as a dupe
				//if (curframe != 0)
				// The first key in any bone sequence should be preserved
				if (j != 0)
				{
					visOffCount++;

					// Times should not be modified
					pthis->ObjectExporter::Qframes[curframe].q        = pthis->ObjectExporter::Qframes[curframe - 1].q;
					pthis->ObjectExporter::Qframes[curframe].bFlip[0] = pthis->ObjectExporter::Qframes[curframe - 1].bFlip[0];
					pthis->ObjectExporter::Qframes[curframe].bFlip[1] = pthis->ObjectExporter::Qframes[curframe - 1].bFlip[1];
					pthis->ObjectExporter::Qframes[curframe].bFlip[2] = pthis->ObjectExporter::Qframes[curframe - 1].bFlip[2];

					// We want the compression code to remove this on its own if its active, if its not then we need some way
					// to indicate that this key should be removed after the final keys are grabbed from max
					// We can't remove it immediately because we are a callback in AnimExporter and the state of the array
					// needs to be maintained until the GetAnim function completes
					pthis->ObjectExporter::Qframes[curframe].flags |= KEYFLAG_INVISIBLE;

					pthis->ObjectExporter::Tframes[curframe].t[0] = pthis->ObjectExporter::Tframes[curframe - 1].t[0];
					pthis->ObjectExporter::Tframes[curframe].t[1] = pthis->ObjectExporter::Tframes[curframe - 1].t[1];
					pthis->ObjectExporter::Tframes[curframe].t[2] = pthis->ObjectExporter::Tframes[curframe - 1].t[2];
					
					// We want the compression code to remove this on its own if its active, if its not then we need some way
					// to indicate that this key should be removed after the final keys are grabbed from max
					// We can't remove it immediately because we are a callback in AnimExporter and the state of the array
					// needs to be maintained until the GetAnim function completes
					pthis->ObjectExporter::Tframes[curframe].flags |= KEYFLAG_INVISIBLE;					
				}
			}
			/*
			else
			{
				int zzz;
				zzz = 0;

				int time = pthis->ObjectExporter::Qframes[curframe].time;

				if (!bWasVisible)	
					bWasVisible = true;
					
			}
			*/
		}
	}

	//char buf[256];
	//sprintf(buf, "%i vis keys were dropped of %i potential drop keys and %i total.\n", visOffCount, visOffPotentialDrop, totalKeys);
	//OutputDebugString(buf);
}

// Note: We assume new was used to allocate the qframes array
void CutsceneExportDlg::RemoveFlaggedInvisibleKeys(Gfx::SAnimQFrame** qframes, int* nKeys)
{
	// Pass 1: Count flagged keys
	int nFlaggedKeys = 0;
	int pos = 0;
	int i;

	/*
	for(i = 0; i < (*nKeys); i++)
	{
		if ((*qframes)[i].flags & KEYFLAG_INVISIBLE)
			nFlaggedKeys++;
	}
	*/

	// Pass 2: Copy the keys into our new array
	//Gfx::SAnimQFrame* newQFrames = new Gfx::SAnimQFrame[(*nKeys) - nFlaggedKeys];

	for(i = 0; i < (*nKeys); i++)
	{
		if (!((*qframes)[i].flags & KEYFLAG_INVISIBLE))
		{
			(*qframes)[pos++] = (*qframes)[i];
		}
		else
			nFlaggedKeys++;

		//newQFrames[pos++] = (*qframes)[i];
	}

	//delete [] (*qframes);
	//*qframes = newQFrames;

	*nKeys = (*nKeys) - nFlaggedKeys;

	char buf[256];
	sprintf(buf, "Found %i flagged keys for removal.\n", nFlaggedKeys);
	OutputDebugString(buf);
}

// Note: We assume new was used to allocate the tframes array
void CutsceneExportDlg::RemoveFlaggedInvisibleKeys(Gfx::SAnimTFrame** tframes, int* nKeys)
{
	// Pass 1: Count flagged keys
	int nFlaggedKeys = 0;
	int pos = 0;
	int i;

	/*
	for(i = 0; i < (*nKeys); i++)
	{
		if ((*tframes)[i].flags & KEYFLAG_INVISIBLE)
			nFlaggedKeys++;
	}

	// Pass 2: Copy the keys into our new array
	Gfx::SAnimTFrame* newTFrames = new Gfx::SAnimTFrame[(*nKeys) - nFlaggedKeys];
	*/

	for(i = 0; i < (*nKeys); i++)
	{
		if (!((*tframes)[i].flags & KEYFLAG_INVISIBLE))
		{
			(*tframes)[pos++] = (*tframes)[i];
		}
		else
			nFlaggedKeys++;

		//newTFrames[pos++] = (*tframes)[i];
	}

	//delete [] (*tframes);
	//*tframes = newTFrames;

	*nKeys = (*nKeys) - nFlaggedKeys;
}

int CutsceneExportDlg::GetNumInvisibleKeys(Gfx::SAnimTFrame* tframes, int nKeys)
{
	int nFlaggedKeys = 0;

	for(int i = 0; i < nKeys; i++)
	{
		if (tframes[i].flags & KEYFLAG_INVISIBLE)
			nFlaggedKeys++;
	}

	return nFlaggedKeys;
}

int CutsceneExportDlg::GetNumInvisibleKeys(Gfx::SAnimQFrame* qframes, int nKeys)
{
	int nFlaggedKeys = 0;

	for(int i = 0; i < nKeys; i++)
	{
		if (qframes[i].flags & KEYFLAG_INVISIBLE)
			nFlaggedKeys++;
	}

	return nFlaggedKeys;
}

bool CutsceneExportDlg::ShouldCompress(CutsceneObj* cso)
{
	if (cso->rotTol  == 1.0f &&
		cso->tranTol == 0.0f)
		return false;

	return true;
}

// Returns true if the skeleton exceeds 55 bones
bool CutsceneExportDlg::SkeletonIsHires(INode* node)
{
	CSkeletonData skel(node, true, false, true);	// Force as root

	if (skel.GetCount() > 55)
		return true;

	return false;
}

void CutsceneExportDlg::DumpDebugFrames()
{
	Gfx::SAnimQFrame* Qframes;		// Uncompressed Rotation list  (Quaternions) for current node
	Gfx::SAnimTFrame* Tframes;		// Uncompressed Translation list for current node

	Gfx::SAnimQFrame* compQframes;	// Compressed Rotation list (Quaternions) for current node
	Gfx::SAnimTFrame* compTframes;	// Compressed Translation list for current node

	int* numCompressedQFrames;
	int* numCompressedTFrames;

	int  numNodes;

	//ObjectExporter::numFrames
}

int CutsceneExportDlg::BuildBonedAnimPacket(Cutscene* cutscene, CutsceneObj* csobj, unsigned char* pData, unsigned int* pos)
{
	// Load packet from last exported file if the object wasn't modified and doing so is possible
	if (!csobj->bModified)
	{
		LIBObjPacket packet;
		void* pLoadedData;
		unsigned int objCRC;

		char buf[256];
		if (csobj->pOrigObj && csobj->pOrigObj->name.Length() > 0)
		{
			sprintf(buf, "Attempting to load packet for '%s' type '%s'\n", csobj->pOrigObj->name, "SKA");

#ifdef PACKET_DEBUG
		//MessageBox(gInterface->GetMAXHWnd(), buf, "LoadPacket", MB_OK);
		FILE* fpDebug = fopen("c:\\cutpakdbg.txt", "a");
		fputs(buf, fpDebug);
		fclose(fpDebug);
#endif

			objCRC = GenerateCRC(csobj->pOrigObj->name);
		}
		else
		{
			sprintf(buf, "Attempting to load packet for '%s' type '%s'\n", "", "SKA");

#ifdef PACKET_DEBUG
		//MessageBox(gInterface->GetMAXHWnd(), buf, "LoadPacket", MB_OK);
		FILE* fpDebug = fopen("c:\\cutpakdbg.txt", "a");
		fputs(buf, fpDebug);
		fclose(fpDebug);
#endif

			objCRC = 0;
		}
		
		OutputDebugString(buf);

		//unsigned int objCRC = GenerateCRC(csobj->debugStr);

		// Copied objects have differing CRCs
		if (csobj->bCopy)
		{
			OutputDebugString("Object IS a COPY\n");
			objCRC++;
		}

		if (pLoadedData = LoadPacket(curCutsceneFilename, objCRC, GenerateCRC("SKA"), &packet))
		{
			memcpy(pData, pLoadedData, packet.size);
			*pos += packet.size;
			return 1;
		}
	}

	// Left for convenience in case we decide to switch these options
	bool bOneFrame       = true;
	bool bCompressTime   = true;
	bool bUseCompression;
	bool bThreeBrowSys   = false;

	ObjectExporter::bRotateRoot        = true;
	//ObjectExporter::bTestRotate        = false;
	ObjectExporter::bNoParentTransform = false;
	ObjectExporter::bSwapCoordSystem   = false;
	ObjectExporter::bCompressTime      = true;

	bUseCompression = RequiresCompression(cutscene, OBJTYPE_BONEDANIM);

	//INode* node     = ip->GetINodeByHandle(csobj->handle);
	INode* node = ip->GetINodeByName(csobj->name);

	if (!node)
		return 0;

	expNode   = ip->GetINodeByName(csobj->pOrigObj->name);
	bexpModel = false;
	expCso    = csobj;

	INode* nodeNeck        = FindNode("Bone_Neck", node);
	INode* nodeControlRoot = FindNode("Control_Root", node);

	if (!nodeNeck && csobj->bHiresSkeleton)
		return 0;

	//GetAnim(node, csobj->start, csobj->end, 0.0f, 0.0f, tdata, bUseCompression);
	// Build AnimExporter format tolerance data structure
	TolData tdata;

	Link<PerBoneTol>* link = csobj->boneTolList.GetHead();
	INode* linkNode;

	while(link)
	{
		//linkNode = ip->GetINodeByHandle(link->data.handle);
		linkNode = ip->GetINodeByName(link->data.name);

		if (linkNode)
		{
			Link<TolRecord>* linkTR = tdata.DB.Add();
			linkTR->data.name = linkNode->GetName();
			
			if (link->data.rotTol == 1.0f)
				linkTR->data.bHasRot = false;
			else
				linkTR->data.bHasRot = true;

			if (link->data.tranTol == 1.0f)
				linkTR->data.bHasTran = false;
			else
				linkTR->data.bHasTran = true;

			linkTR->data.valRot      = link->data.rotTol;
			linkTR->data.valTran     = link->data.tranTol;
			linkTR->data.bRotPreset  = false;
			linkTR->data.bTranPreset = false;
		}

		link = link->next;
	}

	// Acquire global compression values
	float fCompress = 0.999700f + 0.000300f * csobj->rotTol;

	if (ObjectExporter::skeleton)
		delete ObjectExporter::skeleton;

	// Handle breaking up of skeleton
	// Hires skeletons not marked as such should still be limited to the 55 bone boundary
	if (csobj->bHiresSkeleton || SkeletonIsHires(node))
	{
		if (!csobj->bCopy)
		{
			ObjectExporter::skeleton = new CSkeletonData(NULL, nodeNeck);	// Prevent Neck hierarchy from inclusion
			
			// Use the rootbone from the current object
			INode* root = gInterface->GetINodeByName(csobj->rootBone);
			
			if (!root)
				root = FindRoot(node);

			bThreeBrowSys = SetSkeletonExceptions(root, ObjectExporter::skeleton);
			ObjectExporter::skeleton->BuildSkeleton(node);
		}
		else
		{
			node = nodeNeck;
			ObjectExporter::skeleton = new CSkeletonData(node, true, false, true);	// Force as root
		}
	}
	else
	{
		// The given node will be forced to be the root of the skeleton
		ObjectExporter::skeleton = new CSkeletonData(node, true, false, true);
		//ObjectExporter::skeleton = new CSkeletonData(node);
	}

	//////// Remove any bones from the skeleton that are not in our list
	if (csobj->partialAnimSet.Length() > 0)
	{
		// Acquire the partialAnimSet specified for this object from the partial anim editor
		PartialAnimSet* pPartialAnimSet = partialAnimEditor->GetPartialAnimSet(csobj->partialAnimSet);
		assert(pPartialAnimSet);
		ObjectExporter::skeleton->RemoveNonPartialAnimBones(pPartialAnimSet);
	}
	//////////////////////////////////////////////////////////////////////

	ObjectExporter::rootbone = ObjectExporter::skeleton->GetRootBone(node);

	INode* root     = ObjectExporter::GetRoot(node);
	int    numNodes = ObjectExporter::CountNodes(root); 

	if (csobj->rotTol  == 1.0f &&
		csobj->tranTol == 0.0f)
		bUseCompression = false;
	else
		bUseCompression = true;

	//ObjectExporter::GetAnim(root, csobj->start, csobj->end, fCompress, csobj->tranTol, &tdata, bUseCompression);
	ProgressBar* pbarBA = new ProgressBar(hInstance, hwnd, "Exporting BonedAnim", 0, 1);
	pbarBA->AllowCancel(TRUE);
	//ObjectExporter::GetAnim(root, csobj->start, csobj->end, csobj->rotTol, csobj->tranTol, &tdata, bUseCompression, pbarBA);
	//ObjectExporter::GetAnim(root, csobj->start, csobj->end, fCompress, csobj->tranTol, &tdata, bUseCompression, pbarBA);
	//rootBone = root;

	// If export from pelvis is true then the first 2 transforms will be the identity
	bool bExportFromPelvis = false;

	if (IsDlgButtonChecked(hwnd, IDC_EXPORTFROMPELVIS) == BST_CHECKED)
	{
		//if (csobj->bCopy)
		//if (csobj->bHiresSkeleton)
		
		if (!csobj->bCopy &&
			csobj->bExportFromPelvis)
			bExportFromPelvis = true;
		
		/*
		CStr rootName = root->GetName();

		INode* rootBone = FindBoneRootBySkin(root);
		if (rootBone)
		{
			CStr rootBoneName = rootBone->GetName();

			INode* bonePelvis = FindPelvisBone(rootBone);

			if (bonePelvis)
				bExportFromPelvis = true;
		}
		*/
	}

	// Process visibility track dawta if appropriate
	CStr rootNodeName = root->GetName();

	//ObjectExporter::bNoParentTransform = false;

	ObjectExporter::pelvisbone = FindPelvisBone(root);

	if (IsDlgButtonChecked(hwnd, IDC_PROCVIS) == BST_CHECKED)
		ObjectExporter::GetAnim(root, csobj->start, csobj->end, fCompress, csobj->tranTol, &tdata, bUseCompression, pbarBA, RemoveInvisibleKeys, this, bExportFromPelvis);
	else
		ObjectExporter::GetAnim(root, csobj->start, csobj->end, fCompress, csobj->tranTol, &tdata, bUseCompression, pbarBA, NULL, NULL, bExportFromPelvis);

	DumpDebugFrames();

	if (pbarBA->WasCanceled())
	{
		// Copy the .cut file back from the default
		char* appenv = getenv(APP_ENV);
		if (!appenv)
		{
			char strErr[256];
			sprintf(strErr, "The environment variable '%s' is not defined can't continue.", APP_ENV);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Couldn't find environment variable", MB_ICONWARNING|MB_OK);

			delete pbarBA;
			return -1;
		}

		CStr filename = CStr(appenv) + CStr(CUTSCENE_PATH) + cutscene->name + ".cut";
		CStr curCutsceneFilename = ReplaceStr(filename, ".cut", ".cut.bak");

		if (!CopyFile(curCutsceneFilename, filename, FALSE))
		{
			char strErr[256];
			sprintf(strErr, "Failed to restore '%s' from backup '%s'.", (char*)filename, (char*)curCutsceneFilename);
			MessageBox(gInterface->GetMAXHWnd(), strErr, "Failed to restore .cut file from backup", MB_ICONWARNING|MB_OK);
		}

		delete pbarBA;
		return -1;
	}

	delete pbarBA;

	int num3DSFrames = csobj->end - csobj->start + 1;	//	TEST: diff range
	//int num3DSFrames = csobj->end - csobj->start;
	int reqFrames    = num3DSFrames * OUTPUT_FPS / GetFrameRate();

	if (!bUseCompression)
	{
		int nQKeys = 0, nTKeys = 0;

		for(int i = 0; i < numNodes; i++)
		{
			int curFrame = i * num3DSFrames;
			
			/*
			//nQKeys += ObjectExporter::numCompressedQFrames[i];
			//nTKeys += ObjectExporter::numCompressedTFrames[i];
			nQKeys += num3DSFrames;
			nTKeys += num3DSFrames;

			int v1, v2;
			v1 = ObjectExporter::numCompressedQFrames[i] = 
				ObjectExporter::numCompressedQFrames[i] - 
				GetNumInvisibleKeys(&ObjectExporter::compQframes[curFrame], ObjectExporter::numCompressedQFrames[i]);

			v2 = ObjectExporter::numCompressedTFrames[i] = 
				ObjectExporter::numCompressedTFrames[i] - 
				GetNumInvisibleKeys(&ObjectExporter::compTframes[curFrame], ObjectExporter::numCompressedTFrames[i]);
			*/

			// If compression was not used, we need to manually remove the duplicated keys
			// Specifying directly without temporary causes error with VC6
			Gfx::SAnimQFrame* pQFrames = &ObjectExporter::compQframes[curFrame];
			Gfx::SAnimTFrame* pTFrames = &ObjectExporter::compTframes[curFrame];
			
			int nQFrames = ObjectExporter::numCompressedQFrames[i];
			int nTFrames = ObjectExporter::numCompressedTFrames[i];

			///////////////////////////////////////////////////////////////////////////////////////////
			char buf[256];
			INode* nodeBone = ObjectExporter::skeleton->GetBone(i);
			if (nodeBone)
				sprintf(buf, "RemoveFlaggedInvisibleKeys for '%s'\n", (char*)nodeBone->GetName());
			else
				sprintf(buf, "RemoveFlaggedInvisibleKeys for UNKNOWN bone\n");

			OutputDebugString(buf);
			///////////////////////////////////////////////////////////////////////////////////////////

			RemoveFlaggedInvisibleKeys(&pQFrames, &ObjectExporter::numCompressedQFrames[i]);
			RemoveFlaggedInvisibleKeys(&pTFrames, &ObjectExporter::numCompressedTFrames[i]);

			nQFrames = ObjectExporter::numCompressedQFrames[i];
			nTFrames = ObjectExporter::numCompressedTFrames[i];

			//ObjectExporter::compQframes = pQFrames;
			//ObjectExporter::compTframes = pTFrames;
		}
	}

	if (bOneFrame)
		ObjectExporter::RemoveDblKeys(numNodes, num3DSFrames);

	// Compute total QFrames and TFrames
	int QKeys = 0;
	int TKeys = 0;
	int i, j;    

	INode* absRoot = FindRoot(node);

	for(i=0;i<numNodes;i++)
	{
		// Fake bones only have 1 Q key and 1 T key
		CStr boneName = ObjectExporter::skeleton->GetBone(i)->GetName();

/*
		if (/*!csobj->bCopy &&*//*
			(i == 0 && boneName != CStr("Bone_Neck")) ||	// The first bone in the sequence will have it's transforms wiped unless its the neck bone
			BoneIsFake(absRoot, ObjectExporter::skeleton->GetBone(i)))
*/

		CStr nodeName = ObjectExporter::skeleton->GetBone(i)->GetName();

		if (i == 0 /*|| BoneIsFake(absRoot, csobj, ObjectExporter::skeleton->GetBone(i))*/)
		{
			QKeys++;
			TKeys++;
		}
		else
		{
			QKeys += ObjectExporter::numCompressedQFrames[i];
			TKeys += ObjectExporter::numCompressedTFrames[i];		
		}
	}

	// Output File Header
	INode* browBone = NULL;

	if (bThreeBrowSys)
	{
		// If we're using a 3 brow system we need to temporarily rename
		// Bone_Brow_MID to Bone_Brow, so the skeleton data gets the
		// appropritate checksums to match up the the THPS5 skeleton
		// as Gary wants it.
		browBone = FindNode("Bone_Brow_MID", FindRoot(node));
		browBone->SetName("Bone_Brow");
	}

	SHeader header;
	header.version = ANIMFILE_VERSION;
	header.flags   = INTERMEDIATE_FORMAT | CUTSCENE;

	if (csobj->partialAnimSet.Length() > 0)
		header.flags |= PARTIALANIM;

	if (!bUseCompression)
		header.flags |= UNCOMPRESSED_FORMAT;

	// !! was commented out !!
	if (ObjectExporter::bRotateRoot)
		header.flags|=PREROTATEDROOT;

	if (bCompressTime)
		header.flags|=COMPRESSEDTIME;

	header.skeletonName       = ObjectExporter::skeleton->GetChecksum();			// hard-coded default for now
	header.duration           = (float)(num3DSFrames-1)/(float)GetFrameRate();
	/// TEST: diff range
	//header.duration           = (float)(num3DSFrames)/(float)GetFrameRate();
	header.numBones           = numNodes;
	header.numQKeys           = QKeys;
	header.numTKeys           = TKeys;
	header.numUserDefinedKeys = 0;

	unsigned int wsize = 0;

	if (!header.Write(pData, &wsize))
	{
		delete ObjectExporter::skeleton;
		ObjectExporter::skeleton = NULL;

		if (bThreeBrowSys && browBone)
			browBone->SetName("Bone_Brow_MID");

		return 0;
	}

	pData += wsize;
	*pos  += wsize;

	// Output skeleton data
	wsize = ObjectExporter::skeleton->Write(pData);

	if (bThreeBrowSys && browBone)
		browBone->SetName("Bone_Brow_MID");

	if (wsize == 0)
	{
		delete ObjectExporter::skeleton;
		ObjectExporter::skeleton = NULL;
		return 0;
	}

	pData += wsize;
	*pos  += wsize;

	/// Output compressed rotation data
	SQData sqdata(numNodes, QKeys);
	int QKeyPos = 0;

#ifdef DEBUG_BONEDANIM_FILE
	FILE* fpBonedAnim = fopen("c:\\boneanims.txt","a");
	fprintf(fpBonedAnim, "Exporting %s...", (char*)csobj->debugStr);
#endif

	for(i = 0; i < numNodes; i++)
	{
		CStr boneName = ObjectExporter::skeleton->GetBone(i)->GetName();

#ifdef DEBUG_BONEDANIM_FILE
		fprintf(fpBonedAnim, "BoneName: %s\n", (char*)boneName);
#endif

		/*
		if ((i == 0 && boneName != CStr("Bone_Neck")) ||	// The first bone in the sequence will have it's transforms wiped unless its the neck bone
			BoneIsFake(node, ObjectExporter::skeleton->GetBone(i)))
		*/

		if (i == 0 /*||
			(BoneIsFake(node, csobj, ObjectExporter::skeleton->GetBone(i)))*/)
		{
			sqdata.numKeysPerBone[i] = 1;
			ZeroMemory(&sqdata.theQKeys[QKeyPos].time, sizeof(float));	// Ensure float representation of 0 is all 0s
																		// in case data is expected to be compressed and
																		// converted to integer
			sqdata.theQKeys[QKeyPos].qx    = 0.0f;
			sqdata.theQKeys[QKeyPos].qy    = 0.0f;
			sqdata.theQKeys[QKeyPos].qz    = 0.0f;
			sqdata.theQKeys[QKeyPos].real  = 1.0f;

#ifdef DEBUG_BONEDANIM_FILE
			CStr boneName = ObjectExporter::skeleton->GetBone(i)->GetName();

			fprintf(fpBonedAnim, "BoneName: %s\n", (char*)boneName);
			fprintf(fpBonedAnim, "#%i  time: %f  qx: %f  qy: %f  qz: %f  real: %f\n", QKeyPos,
																                      sqdata.theQKeys[QKeyPos].time,
					                                                                  sqdata.theQKeys[QKeyPos].qx,
					                                                                  sqdata.theQKeys[QKeyPos].qy,
																                      sqdata.theQKeys[QKeyPos].qz,
																                      sqdata.theQKeys[QKeyPos].real);
#endif

			QKeyPos++;
		}
		else
		{
			sqdata.numKeysPerBone[i] = ObjectExporter::numCompressedQFrames[i];

			for(j = 0; j < ObjectExporter::numCompressedQFrames[i]; j++)
			{
				int curFrame = j + i * num3DSFrames;

				// If the compress time flag is specified we need to multiply time time by the FPS
				// rate to convert it into the frame number as opposed to the time into the animation
				if (bCompressTime)
				{
					// This is still going to get dumped out as a float, so we need to make sure
					// the compiler makes the representation of the float match an int
					// We can do this since they're both 4 bytes
					int* val = (int*)&sqdata.theQKeys[QKeyPos].time;
					float ftmp = ObjectExporter::compQframes[curFrame].time * (float)OUTPUT_FPS;
					*val = Round(ftmp);
				}
				else
					sqdata.theQKeys[QKeyPos].time = ObjectExporter::compQframes[curFrame].time;

				sqdata.theQKeys[QKeyPos].qx   = ObjectExporter::compQframes[curFrame].q[X];
				sqdata.theQKeys[QKeyPos].qy   = ObjectExporter::compQframes[curFrame].q[Y];
				sqdata.theQKeys[QKeyPos].qz   = ObjectExporter::compQframes[curFrame].q[Z];
				sqdata.theQKeys[QKeyPos].real = ObjectExporter::compQframes[curFrame].q[W];

#ifdef DEBUG_BONEDANIM_FILE
				if (bCompressTime)
				{
					fprintf(fpBonedAnim, "#%i time: %i  qx: %f  qy: %f  qz: %f  real: %f\n", QKeyPos,
						                                                  *((int*)&sqdata.theQKeys[QKeyPos].time),
							                                              sqdata.theQKeys[QKeyPos].qx,
								                                          sqdata.theQKeys[QKeyPos].qy,
																		  sqdata.theQKeys[QKeyPos].qz,
																		  sqdata.theQKeys[QKeyPos].real);
				}
				else
				{
					fprintf(fpBonedAnim, "#%i time: %f  qx: %f  qy: %f  qz: %f  real: %f\n", QKeyPos,
						                                                  sqdata.theQKeys[QKeyPos].time,
							                                              sqdata.theQKeys[QKeyPos].qx,
								                                          sqdata.theQKeys[QKeyPos].qy,
																		  sqdata.theQKeys[QKeyPos].qz,
																		  sqdata.theQKeys[QKeyPos].real);
				}
#endif

				QKeyPos++;
			}
		}
	}

	if (!sqdata.Write(pData, &wsize))
	{
		delete ObjectExporter::skeleton;
		ObjectExporter::skeleton = NULL;
		return 0;
	}

	pData += wsize;
	*pos  += wsize;

	// Output compressed translation data
	STData stdata(numNodes, TKeys);
	int TKeyPos = 0;

	for(i = 0; i < numNodes; i++)
	{
		CStr boneName = ObjectExporter::skeleton->GetBone(i)->GetName();

#ifdef DEBUG_BONEDANIM_FILE
		fprintf(fpBonedAnim, "BoneName: %s\n", (char*)boneName);
#endif

		if (i == 0 /*||
			BoneIsFake(node, csobj, ObjectExporter::skeleton->GetBone(i))*/)
		{
			stdata.numKeysPerBone[i] = 1;

			ZeroMemory(&stdata.theTKeys[TKeyPos].time, sizeof(float));
			stdata.theTKeys[TKeyPos].tx = 0.0f;
			stdata.theTKeys[TKeyPos].ty = 0.0f;
			stdata.theTKeys[TKeyPos].tz = 0.0f;

#ifdef DEBUG_BONEDANIM_FILE
			fprintf(fpBonedAnim, "#%i time: %f  tx: %f  ty: %f  tz: %f\n", TKeyPos,
				                                            stdata.theTKeys[TKeyPos].time,
					                                        stdata.theTKeys[TKeyPos].tx,
					                                        stdata.theTKeys[TKeyPos].ty,
															stdata.theTKeys[TKeyPos].tz
															);
#endif

			TKeyPos++;
		}
		else
		{
			stdata.numKeysPerBone[i] = ObjectExporter::numCompressedTFrames[i];

			for(j = 0; j < ObjectExporter::numCompressedTFrames[i]; j++)
			{
				int curFrame = j + i * num3DSFrames;

				// If the compress time flag is specified we need to multiply time time by the FPS
				// rate to convert it into the frame number as opposed to the time into the animation
				if (bCompressTime)
				{
					// This is still going to get dumped out as a float, so we need to make sure
					// the compiler makes the representation of the float match an int
					// We can do this since they're both 4 bytes
					int* val = (int*)&stdata.theTKeys[TKeyPos].time;
					float ftmp = ObjectExporter::compTframes[curFrame].time * (float)OUTPUT_FPS;
					*val = Round(ftmp);
				}
				else
					stdata.theTKeys[TKeyPos].time = ObjectExporter::compTframes[curFrame].time;
		
				stdata.theTKeys[TKeyPos].tx   = ObjectExporter::compTframes[curFrame].t[0];
				stdata.theTKeys[TKeyPos].ty   = ObjectExporter::compTframes[curFrame].t[1];
				stdata.theTKeys[TKeyPos].tz   = ObjectExporter::compTframes[curFrame].t[2];

#ifdef DEBUG_BONEDANIM_FILE
				CStr boneName = ObjectExporter::skeleton->GetBone(i)->GetName();

				if (bCompressTime)
				{
					fprintf(fpBonedAnim, "#%i  time: %i  tx: %f  ty: %f  tz: %f\n", TKeyPos,
						                                            *((int*)&stdata.theTKeys[TKeyPos].time),
																	stdata.theTKeys[TKeyPos].tx,
																	stdata.theTKeys[TKeyPos].ty,
																	stdata.theTKeys[TKeyPos].tz
																	);
				}
				else
				{
					fprintf(fpBonedAnim, "#%i  time: %f  tx: %f  ty: %f  tz: %f\n", TKeyPos,
						                                            stdata.theTKeys[TKeyPos].time,
																	stdata.theTKeys[TKeyPos].tx,
																	stdata.theTKeys[TKeyPos].ty,
																	stdata.theTKeys[TKeyPos].tz
																	);
				}
#endif

				TKeyPos++;
			}
		}
	}

	if (!stdata.Write(pData, &wsize))
	{
		delete ObjectExporter::skeleton;
		ObjectExporter::skeleton = NULL;
		return 0;
	}

	pData += wsize;
	*pos  += wsize;

	delete ObjectExporter::skeleton;
	ObjectExporter::skeleton = NULL;

#ifdef DEBUG_BONEDANIM_FILE
	fclose(fpBonedAnim);
#endif

	return numNodes;
}

INode* CutsceneExportDlg::FindBoneRootBySkin(INode* skinNode)
{
	if (!skinNode)
		return NULL;

	// Attempt to locate the Physique modifier
	Modifier* mod = FindPhysiqueModifier(skinNode);

	if (!mod)
		return NULL;

	// Grab the physique interface
    IPhysiqueExport* pPhysiqueExport = (IPhysiqueExport*)mod->GetInterface(I_PHYINTERFACE);    
	assert( pPhysiqueExport );

	IPhyContextExport* pModContextExport = (IPhyContextExport*)pPhysiqueExport->GetContextInterface(skinNode);
	assert( pModContextExport );

    pModContextExport->ConvertToRigid( true );
    pModContextExport->AllowBlending( true );

    IPhyVertexExport* pVertexExport;
    pVertexExport = pModContextExport->GetVertexInterface( 0 );
	assert( pVertexExport );

	INode* modelNode;

	switch(pVertexExport->GetVertexType())
	{
	case RIGID_BLENDED_TYPE:
		{
			IPhyBlendedRigidVertex* pBlendedVertex = (IPhyBlendedRigidVertex*)pVertexExport;
			
			if (pBlendedVertex->GetNumberNodes() == 0)
				return NULL;

			modelNode = pBlendedVertex->GetNode(0);
		}
		break;

	case RIGID_TYPE:
		{
			IPhyRigidVertex* pRigidVertex = (IPhyRigidVertex*)pVertexExport;
			modelNode = pRigidVertex->GetNode();
		}
		break;

	default:
		return NULL;
	}

	CStr name = modelNode->GetName();

	CSkeletonData cskel = CSkeletonData(modelNode);
	return cskel.GetRootBone(modelNode);
	//return FindRoot(modelNode);
}

CutsceneObj* CutsceneExportDlg::FindAssocBoneAnim(Cutscene* cutscene, INode* rootBone)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.type & OBJTYPE_BONEDANIM)
		{
			INode* node = ip->GetINodeByName(link->data.name);

			if (node)
			{
				CSkeletonData skel(node, true, true);

				if (skel.GetRootBone(node) == rootBone)
					return &link->data;
			}
		}
		
		link = link->next;
	}

	return NULL;
}

CutsceneObj* CutsceneExportDlg::AddRootBoneToScene(Cutscene* cutscene, INode* skinNode)
{
	Link<CutsceneObj>* link = GetCutsceneObj(cutscene, skinNode);
	
	if (!link)
		return NULL;

	CutsceneObj* skinObj = &link->data;
	
	if (skinObj)
	{
		INode* node;

		if (skinObj->rootBone)
			node = ip->GetINodeByName(skinObj->rootBone);
		else
			node = FindBoneRootBySkin(skinNode);

		if (node)
		{
			CutsceneObj cso, csoDefault;

			cso.name   = node->GetName();
			//cso.type = cso.GetTaggedFlags(node);
			cso.type = OBJTYPE_BONEDANIM;
			cso.SetTaggedFlags(node);

			// Attempt to acquire cutscene object data from property editor props

			csoDefault.start      = skinObj->start;
			csoDefault.end        = skinObj->end;
			csoDefault.rotTol     = skinObj->rotTol;
			csoDefault.tranTol    = skinObj->tranTol;
			csoDefault.modelName  = skinObj->modelName;
			csoDefault.modelName2 = skinObj->modelName2;
			csoDefault.handle     = node->GetHandle();

			cso.LoadFromNode(cutscene->name, &csoDefault);
			cso.SaveToNode(cutscene->name);
			return &cutscene->objDB.AddUnique(&cso)->data;
		}
	}

	return NULL;
}

CutsceneObj* CutsceneExportDlg::FindObjByModelName(Cutscene* cutscene, char* modelName)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.modelName == CStr(modelName))
			return &link->data;

		link = link->next;
	}

	return NULL;
}

// This will add a dupe for every type flagged as bHiresSkeleton and tag it as such
bool CutsceneExportDlg::AddHiresDupes(Cutscene* cutscene)
{
	Link<CutsceneObj>* linkCSObj = cutscene->objDB.GetHead();

	// Now since the bHiresSkeleton checkbox resides within the skin (or model) we
	// need to copy the data from the associated object (probably the same object)
	// when doing the copy, rather than the skin object
	while(linkCSObj)
	{
		if ((linkCSObj->data.type & OBJTYPE_SKIN || linkCSObj->data.type & OBJTYPE_MODEL) &&
			!linkCSObj->data.bCopy)
		{
			if (linkCSObj->data.rootBone == CStr(""))
			{
				INode* skinNode = ip->GetINodeByName(linkCSObj->data.name);
				INode* rootBone = FindBoneRootBySkin(skinNode);

				if (rootBone)
					linkCSObj->data.rootBone = rootBone->GetName();
			}

			// Find the associated boned anim that we want to copy
			INode* node = ip->GetINodeByName(linkCSObj->data.rootBone);

			if (!node)
			{
				char strErr[256];
				sprintf(strErr, "Failed to locate root node '%s'.  Aborted.", linkCSObj->data.rootBone);
				MessageBox(hwnd, strErr, "Failed to locate root bone", MB_ICONWARNING|MB_OK);
				return false;
			}

			CutsceneObj* csobj2 = FindAssocBoneAnim(cutscene, node);

			if (!csobj2)
			{
				INode* skinNode = ip->GetINodeByName(linkCSObj->data.name);
				csobj2 = AddRootBoneToScene(cutscene, skinNode);
			}

			if (!csobj2)
			{
				char strErr[256];
				sprintf(strErr, "Failed to find boned anim export packet for '%s'.", linkCSObj->data.rootBone);
				MessageBox(hwnd, strErr, "Failed to locate boned anim packet", MB_ICONWARNING|MB_OK);
				return false;
			}

			if (linkCSObj->data.bHiresSkeleton)
			{
				// Duplicate the associated boned anim (on export, it will export from the upper portion of
				// the skeleton from the neck)
				//CutsceneObj newCSO = linkCSObj->data;
				CutsceneObj newCSO = *csobj2;
				newCSO.bCopy = true;
				linkCSObj = cutscene->objDB.AddAfterLink(linkCSObj, &newCSO);
			}
		}
		else if ((linkCSObj->data.type & OBJTYPE_BONEDANIM /*|| linkCSObj->data.type & OBJTYPE_OBJANIM*/) &&
			     linkCSObj->data.type & OBJTYPE_EXTERNAL &&
			     !linkCSObj->data.bCopy &&
				 linkCSObj->data.bHiresSkeleton)
		{
			CutsceneObj newCSO = linkCSObj->data;
			newCSO.bCopy = true;
			linkCSObj = cutscene->objDB.AddAfterLink(linkCSObj, &newCSO);
		}

		linkCSObj = linkCSObj->next;
	}

	return true;
}

void CutsceneExportDlg::RemoveHiresDupes(Cutscene* cutscene)
{
	Link<CutsceneObj>* linkCSObj = cutscene->objDB.GetHead();
	Link<CutsceneObj>* linkNext;

	while(linkCSObj)
	{
		linkNext = linkCSObj->next;
		
		if (linkCSObj->data.bCopy)
			cutscene->objDB.Remove(linkCSObj);

		linkCSObj = linkNext;
	}
}

void CutsceneExportDlg::ClearPerBoneTolerances(INode* node)
{
	if (!node)
		node = gInterface->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		ClearPerBoneTolerances(child);
	}

	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_PERBONE_TOLERANCES);
}

CutsceneObj* CutsceneExportDlg::FindObjByName(Cutscene* cutscene, char* name, bool bCopy)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		if (link->data.bCopy == bCopy && link->data.name == CStr(name))
			return &link->data;

		link = link->next;
	}

	return NULL;
}

void CutsceneExportDlg::DisplayHelp()
{
	MessageBox(hwnd, "In order to keep consistency with the Animation Exporter the tolerance values work as follows:\n\nRotation:  1.0 is the lowest compression  0.0 is the highest\nTranslation:  0.0 is no compression  10.0 is the highest (This equates to 10 world units away which would be really bad)\n\nTo disable compression set Rotation to 1.0 and Translation to 0.0", "Tolerance Help", MB_ICONINFORMATION|MB_OK);
}

void CutsceneExportDlg::SyncTimesToCamSel()
{
	int nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (nSelItems != 1)
	{
		MessageBox(hwnd, "A single cutscene mustbe selected to sync the export times to the camera", "Sync Export times to camera", MB_ICONWARNING|MB_OK);
		return;
	}

	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
	if (!link)
	{
		MessageBox(hwnd, "Sync failed!", "Sync Times to Camera", MB_ICONWARNING|MB_OK);
		return;
	}

	CStr strCategory = CUTSCENEGRP + link->data.name;

	// Find the first camera and use its start/end times
	Link<CutsceneObj>* linkCSO = link->data.objDB.GetHead();
	float startTime, endTime;

	while(linkCSO)
	{
		if (linkCSO->data.type & OBJTYPE_CAMERA)
		{
			startTime = linkCSO->data.start;
			endTime   = linkCSO->data.end;
			break;
		}

		linkCSO = linkCSO->next;
	}

	// Cycle through the list again updating the times of the selected objects
	int nObjs = pObjList->GetCount();

	for(int i = 0; i < nObjs; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);

			if (node)
			{
				Link<CutsceneObj>* linkCSO = GetCutsceneObj(&link->data, node);

				linkCSO->data.start = startTime;
				linkCSO->data.end   = endTime;

				AddPropValue(node, strCategory, "start", startTime, "spinedit | 0 | 999999 | 1");
				AddPropValue(node, strCategory, "end", endTime, "spinedit | 0 | 999999 | 1");
			}
		}
	}
	
	RetrieveObjectProps();
}

void CutsceneExportDlg::SyncTimesToCam()
{
	int nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (nSelItems != 1)
	{
		MessageBox(hwnd, "A single cutscene mustbe selected to sync the export times to the camera", "Sync Export times to camera", MB_ICONWARNING|MB_OK);
		return;
	}

	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
	if (!link)
	{
		MessageBox(hwnd, "Sync failed!", "Sync Times to Camera", MB_ICONWARNING|MB_OK);
		return;
	}

	CStr strCategory = CUTSCENEGRP + link->data.name;

	// Find the first camera and use its start/end times
	Link<CutsceneObj>* linkCSO = link->data.objDB.GetHead();
	float startTime, endTime;

	while(linkCSO)
	{
		if (linkCSO->data.type & OBJTYPE_CAMERA)
		{
			startTime = linkCSO->data.start;
			endTime   = linkCSO->data.end;
			break;
		}

		linkCSO = linkCSO->next;
	}

	// Cycle through the list again updating the times
	linkCSO = link->data.objDB.GetHead();
	INode* node;

	while(linkCSO)
	{
		linkCSO->data.start = startTime;
		linkCSO->data.end   = endTime;

		node = gInterface->GetINodeByName(linkCSO->data.name);

		if (node)
		{
			AddPropValue(node, strCategory, "start", startTime, "spinedit | 0 | 999999 | 1");
			AddPropValue(node, strCategory, "end", endTime, "spinedit | 0 | 999999 | 1");
		}

		linkCSO = linkCSO->next;
	}

	RetrieveObjectProps();
}

void CutsceneExportDlg::RenameCutscene()
{
	int item, nSelItems;
	nSelItems = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);
	CStr oldName, newName;

	if (nSelItems != 1)
	{
		MessageBox(hwnd, "Only one cutscene may be selected for renaming", "Rename Cutscene", MB_ICONWARNING|MB_OK);
		return;
	}

	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);
	
	oldName = link->data.name;

	InputDlg* inpdlg = new InputDlg(hInstance, hwnd,
									"Cutscene",
									oldName,
									"Enter New Cutscene Name");
									
	inpdlg->Show();

	if (!inpdlg->WasCancelled())
	{
		newName = inpdlg->GetInput();
		link->data.name = newName;

		// Add new entry to the listbox with the new name and delete the old one
		int idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_INSERTSTRING, 0, (LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)link);
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_DELETESTRING, (WPARAM)item+1, 0);
	}

	delete inpdlg;

	// Scan through all the nodes and replace former property references to the previous cutscene
	INodeTab nodes;
	GetAllNodes(nodes);

	ProgressBar* pBar = new ProgressBar(hInstance, hwnd, "Updating Cutscene Prop Refs...", 0, nodes.Count());
	
	CStr propBuffer;

	for(int i = 0; i < nodes.Count(); i++)
	{
		pBar->SetVal(i);

		if (nodes[i])
		{
			nodes[i]->GetUserPropBuffer(propBuffer);
			propBuffer = ReplaceStr(propBuffer, CUTSCENEGRP + oldName + "/", CUTSCENEGRP + newName + "/");
			nodes[i]->SetUserPropBuffer(propBuffer);
		}
	}

	delete pBar;

	SaveToMAX();
	Save("c:\\autoback.cdb");
}

void CutsceneExportDlg::ScriptKeyUI(Link<TrackUIKey>* link, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	pthis->scriptKeyEditor->BuildKeyList(pthis->pCameraTrack);
	pthis->scriptKeyEditor->SetKeyAdjust(link);
	pthis->pCameraTrack->LockAffectKey(link);
	pthis->scriptKeyEditor->SetTimeChangeCB(pthis->KeyTimeChanged,pthis);
	pthis->scriptKeyEditor->SetUpdateCB(pthis->UpdateKeys,pthis);
	pthis->scriptKeyEditor->SetCancelCB(pthis->CancelKeyUpdate,pthis);
	pthis->scriptKeyEditor->SetKeyChangeCB(pthis->KeyChanged,pthis);
	pthis->scriptKeyEditor->Show();
	//delete scriptKeyEditor;
}

void CutsceneExportDlg::KeyTimeChanged(int time, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;
	pthis->pCameraTrack->Refresh();
}

void CutsceneExportDlg::UpdateKeys(ScriptKeyEditor* pEditor, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	pthis->pCameraTrack->UnlockAffectKey();
	pthis->pCameraTrack->CallKeyUpdatedCB();
	pthis->pCameraTrack->Refresh();
}

void CutsceneExportDlg::CancelKeyUpdate(ScriptKeyEditor* pEditor, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	pthis->pCameraTrack->UnlockAffectKey();
	pthis->pCameraTrack->Refresh();
}

void CutsceneExportDlg::KeyChanged(ScriptKeyEditor* pEditor, Link<TrackUIKey>* link, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	pthis->pCameraTrack->LockAffectKey(link);
}

void CutsceneExportDlg::SetActiveCutscene()
{
	int idx = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETCURSEL, 0, 0);
	Link<Cutscene>* link = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, 0, 0);

	linkActiveCutscene = link;
}

void CutsceneExportDlg::EditObjKeys()
{
	// Operation can only be done in display cutscene contents mode
	if (!displayMode == DISP_CUTSCENE)
		return;

	// Get the currently selected cutscene
	int count = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);
	if (count != 1)
		return;

	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS,(WPARAM)1, (LPARAM)&item);

	// Acquire the cutscene link data from the selected item
	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	if (!linkCutscene)
		return;

	// Determine the currently selected cutscene object
	int idx = pObjList->GetSelItem();

	if (idx == -1)
		return;

	INode* node = (INode*)pObjList->GetItemData(idx);
	Link<CutsceneObj>* cutsceneObj = GetCutsceneObj(&linkCutscene->data, node);

	if (pObjKeyEditor)
		delete pObjKeyEditor;

	pObjKeyEditor = new ObjKeyEditor(hInstance, hwnd, &cutsceneObj->data);
	pObjKeyEditor->SetKeyChangedCB(ObjKeyChanged, this);
	pObjKeyEditor->Show();
}

void CutsceneExportDlg::ObjKeyChanged(ObjKeyEditor* pObjKeyEditor, CutsceneObj* pCutsceneObj, void* pData)
{
	CutsceneExportDlg* pthis = (CutsceneExportDlg*)pData;

	if (!pCutsceneObj->bModified)
	{
		pCutsceneObj->bModified = true;
		pthis->InitObjList(pthis->displayMode);
	}
}

//////////////////////////////// CutsceneBoneTolDlg /////////////////////////////////////////////////////

CutsceneBoneTolDlg::CutsceneBoneTolDlg(HINSTANCE hInstance, HWND hwndParent, Interface* ip, CutsceneObj* cutsceneObj) :
	ModalDlgWindow(hInstance, MAKEINTRESOURCE(IDD_CUTSCENEBONETOLDLG), hwndParent)
{
	this->ip          = ip;
	this->cutsceneObj = cutsceneObj;

	rotTolEdit        = NULL;
	transTolEdit      = NULL;
	rotTolEditSpin    = NULL;
	transTolEditSpin  = NULL;

	pTolList          = NULL;

	bLockTolChange    = false;
}

CutsceneBoneTolDlg::~CutsceneBoneTolDlg()
{
	if (pTolList)
		delete pTolList;
}

BOOL CutsceneBoneTolDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case CC_SPINNER_CHANGE:
		switch(LOWORD(wParam))
		{
		case IDC_ROTTOLSPIN:
			RotTolUpdate();
			return TRUE;

		case IDC_TRANTOLSPIN:
			TranTolUpdate();
			return TRUE;
		}

		break;

	case WM_NOTIFY:
		{
			NMHDR* nmhdr = (NMHDR*)lParam;
			switch(nmhdr->code)
			{
			case LVN_ITEMCHANGED:
				UpdateTolBoxes();
				return TRUE;
			}
		}
		break;

	case WM_INITDIALOG:
		Init();
		Retrieve();
		return TRUE;

	case WM_CLOSE:
		Store();
		EndDialog(hwnd, 0);
		return TRUE;

	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case IDC_PRESETLIST:
				PresetChange();
				return TRUE;
			}
			break;
		}

		switch(LOWORD(wParam))
		{
		case IDC_ADDBONE:
			AddBone();
			return TRUE;

		case IDC_REMOVEBONE:
			RemoveBone();
			return TRUE;

		case IDC_ADDALL:
			AddAll();
			return TRUE;

		case IDC_REMOVEALL:
			RemoveAll();
			return TRUE;
		}
	}

	return FALSE;
}

void CutsceneBoneTolDlg::AddBone()
{
	int     sel = SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETCURSEL, 0, 0);
	INode* node = (INode*)SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETITEMDATA, (WPARAM)sel, 0);

	if (node)
	{
		// Ensure that this bone doesn't already exist within the list
		if (pTolList->Find(node->GetName()) == -1)
		{
			int idx = pTolList->AddItem(node->GetName());
			pTolList->AddSubItem("0.0000", ROT_COL, idx);
			pTolList->AddSubItem("0.0000", TRANS_COL, idx);
			pTolList->SetItemData(idx, (DWORD)node);
		}
	}
}

void CutsceneBoneTolDlg::RemoveBone()
{
	while(pTolList->GetSelCount() > 0)
	{
		int selItem = pTolList->GetSelItem();
		pTolList->DeleteItem(selItem);
	}
}

void CutsceneBoneTolDlg::AddBoneHierarchy(INode* bone)
{
	int nKids = bone->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = bone->GetChildNode(i);
		AddBoneHierarchy(child);
	}

	int idx = SendDlgItemMessage(hwnd, IDC_BONELIST, CB_ADDSTRING, 0, (LPARAM)(char*)bone->GetName());
	SendDlgItemMessage(hwnd, IDC_BONELIST, CB_SETITEMDATA, (WPARAM)idx, (LPARAM)bone);
}

void CutsceneBoneTolDlg::Init()
{
	pTolList = new ListView(GetDlgItem(hwnd, IDC_TLIST));

	pTolList->AddColumn("Name");
	pTolList->AddColumn("RotTol");
	pTolList->AddColumn("TransTol");

	rotTolEdit       = GetICustEdit(GetDlgItem(hwnd, IDC_ROTTOL));
	transTolEdit     = GetICustEdit(GetDlgItem(hwnd, IDC_TRANTOL));
	rotTolEditSpin   = GetISpinner(GetDlgItem(hwnd, IDC_ROTTOLSPIN));
	transTolEditSpin = GetISpinner(GetDlgItem(hwnd, IDC_TRANTOLSPIN));

	rotTolEditSpin->SetLimits(0.0f, 1.0f);
	//rotTolEditSpin->SetSpinnerPrecision(4);
	rotTolEditSpin->SetScale(0.0001f);
	rotTolEditSpin->LinkToEdit(GetDlgItem(hwnd, IDC_ROTTOL), EDITTYPE_FLOAT);

	transTolEditSpin->SetLimits(0.0f, 1.0f);
	//transTolEditSpin->SetSpinnerPrecision(4);
	transTolEditSpin->SetScale(0.0001f);
	transTolEditSpin->LinkToEdit(GetDlgItem(hwnd, IDC_TRANTOL), EDITTYPE_FLOAT);

	//INode* node = ip->GetINodeByHandle(cutsceneObj->handle);
	INode* node = ip->GetINodeByName(cutsceneObj->name);

	if (node)
		AddBoneHierarchy(node);

	LoadPresets();
}

void CutsceneBoneTolDlg::AddAll()
{
	RemoveAll();

	int nCount = SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETCOUNT, 0, 0);
	char buf[256];

	for(int i = 0; i < nCount; i++)
	{
		SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETLBTEXT, (WPARAM)i, (LPARAM)buf);
		INode* node = (INode*)SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETITEMDATA, (WPARAM)i, 0);

		int idx = pTolList->AddItem(buf);
		pTolList->AddSubItem("0.0000", ROT_COL, idx);
		pTolList->AddSubItem("0.0000", TRANS_COL, idx);
		pTolList->SetItemData(idx, (DWORD)node);
	}
}

void CutsceneBoneTolDlg::RemoveAll()
{
	pTolList->Clear();
}

void CutsceneBoneTolDlg::RotTolUpdate()
{
	// Update rotation tolerance for all selected items in the tolerance list
	float fRotTol = rotTolEditSpin->GetFVal();

	char buf[256];
	sprintf(buf, "%f", fRotTol);

	int nCount = pTolList->GetCount();

	for(int i = 0; i < nCount; i++)
	{
		if (pTolList->IsSelected(i))
			pTolList->AddSubItem(buf, ROT_COL, i);
	}
}

void CutsceneBoneTolDlg::TranTolUpdate()
{
	// Update translation tolerance for all selected items in the tolerance list
	float fTranTol = transTolEditSpin->GetFVal();

	char buf[256];
	sprintf(buf, "%f", fTranTol);

	int nCount = pTolList->GetCount();

	for(int i = 0; i < nCount; i++)
	{
		if (pTolList->IsSelected(i))
			pTolList->AddSubItem(buf, TRANS_COL, i);
	}
}

void CutsceneBoneTolDlg::Store()
{
	cutsceneObj->boneTolList.Clear();

	int nCount = pTolList->GetCount();
	char buf[256];

	for(int i = 0; i < nCount; i++)
	{
		INode* node = (INode*)pTolList->GetItemData(i);

		if (!node)
			continue;

		Link<PerBoneTol>* link = cutsceneObj->boneTolList.Add();

		link->data.handle = node->GetHandle();
		link->data.name   = node->GetName();

		pTolList->GetItem(i, buf, 255, ROT_COL);
		link->data.rotTol  = atof(buf);

		pTolList->GetItem(i, buf, 255, TRANS_COL);
		link->data.tranTol = atof(buf);
	}
}

void CutsceneBoneTolDlg::Retrieve()
{
	Link<PerBoneTol>* link = cutsceneObj->boneTolList.GetHead();
	
	char buf[256];
	int  idx;
	pTolList->Clear();

	while(link)
	{
		//INode* node = ip->GetINodeByHandle(link->data.handle);
		INode* node = ip->GetINodeByName(link->data.name);

		if (node)
		{
			idx = pTolList->AddItem(node->GetName());
			pTolList->SetItemData(idx, (DWORD)node);

			sprintf(buf, "%f", link->data.rotTol);
			pTolList->AddSubItem(buf, ROT_COL, idx);

			sprintf(buf, "%f", link->data.tranTol);
			pTolList->AddSubItem(buf, TRANS_COL, idx);
		}

		link = link->next;
	}
}

void CutsceneBoneTolDlg::UpdateTolBoxes()
{
	if (bLockTolChange)
		return;

	bLockTolChange = true;
	int idx = pTolList->GetSelItem();

	if (idx == -1)
	{
		bLockTolChange = false;
		return;
	}

	char  buf[256];
	pTolList->GetItem(idx, buf, 255, ROT_COL);
	rotTolEditSpin->SetValue((float)atof(buf), FALSE);

	pTolList->GetItem(idx, buf, 255, TRANS_COL);
	transTolEditSpin->SetValue((float)atof(buf), FALSE);

	bLockTolChange = false;
}

void CutsceneBoneTolDlg::LoadPresets()
{
	CStr appDir=ip->GetDir(APP_PLUGCFG_DIR);
	appDir+="\\AnimTol.ini";

	if (!TolFile.Load(appDir))
	{
		char strErr[256];
		sprintf(strErr,"Couldn't load tolerance config file '%s'",(char*)appDir);
		MessageBox(hwnd,strErr,"Couldn't open config file",MB_ICONWARNING|MB_OK);
		return;
	}

	// Add the presets to the preset selection list
	SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_RESETCONTENT,0,0);

	Link<GlobalTolRecord>* link = TolFile.DB.GetHead();

	while(link)
	{
		int index;

		index = SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_ADDSTRING,0,(LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd,IDC_PRESETLIST,CB_SETITEMDATA,(WPARAM)index,(LPARAM)&link->data);
		link = link->next;
	}
}

void CutsceneBoneTolDlg::PresetChange()
{
	int idx = SendDlgItemMessage(hwnd, IDC_PRESETLIST, CB_GETCURSEL, 0, 0);

	if (idx == CB_ERR)
		return;

	pTolList->Clear();

	// Clear the tolerance list and fill it in with the preset data (assuming the proper bones exist)
	Link<GlobalTolRecord>* linkGTR = (Link<GlobalTolRecord>*)SendDlgItemMessage(hwnd, IDC_PRESETLIST, CB_GETITEMDATA, (WPARAM)idx, 0);

	Link<TolRecord>* link = linkGTR->data.DB.GetHead();
	char buf[256];
	INode* node;
	int    srchIdx;

	while(link)
	{
		// Item may only be assigned if the bone exists within the bone list
		srchIdx = SendDlgItemMessage(hwnd, IDC_BONELIST, CB_FINDSTRING, (WPARAM)-1, (LPARAM)(char*)link->data.name);

		if (srchIdx != CB_ERR)
		{
			node = (INode*)SendDlgItemMessage(hwnd, IDC_BONELIST, CB_GETITEMDATA, (WPARAM)srchIdx, 0);

			if (node)
			{
				idx = pTolList->AddItem(link->data.name);
				pTolList->SetItemData(idx, (DWORD)node);
				
				if (link->data.bHasRot)
					sprintf(buf, "%f", link->data.valRot);
				else
					sprintf(buf, "0.0000");

				pTolList->AddSubItem(buf, ROT_COL, idx);

				if (link->data.bHasTran)
					sprintf(buf, "%f", link->data.valTran);
				else
					sprintf(buf, "0.0000");

				pTolList->AddSubItem(buf, TRANS_COL, idx);
			}
		}

		link = link->next;
	}
}

CStr CutsceneExportDlg::GetGlobalScriptName(Cutscene* pCutscene, TrackUIKey* pTrackKey)
{
	char bufTime[256];
	_itoa(pTrackKey->time / GetTicksPerFrame(), bufTime, 10);
	return CStr("CutKey_") + pCutscene->name + CStr("_") + CStr(bufTime);
}

CStr CutsceneExportDlg::GetObjScriptName(Cutscene* pCutscene, CutsceneObj* pCSO, TrackUIKey* pTrackKey)
{
	char bufTime[256];
	_itoa(pTrackKey->time / GetTicksPerFrame(), bufTime, 10);
	return CStr("CutObjKey_") + pCutscene->name + CStr("_") + pCSO->name + CStr("_") + CStr(bufTime);
}

bool CutsceneExportDlg::IsFlaggedWorldSpace(Cutscene* cutscene, INode* node)
{
	// Scan through the object database for this node and get its worldspace status
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();

	while(link)
	{
		INode* chkNode = gInterface->GetINodeByName(link->data.name);

		if (chkNode == node)
			return (link->data.type & OBJTYPE_WORLDSPACE);

		link = link->next;
	}

	return false;
}

// This function builds the .qn script file for all the keys in the cutscene
CStr CutsceneExportDlg::GenerateScript(Cutscene* pCutscene)
{
	Link<TrackUIKey>* link = pCutscene->trackKeyDB.GetHead();
	CStr scriptBuffer;
	char eolBD[3];
	eolBD[0] = 10;
	eolBD[1] = 13;
	eolBD[2] = 0;

	char eol[3];
	eol[0] = 13;
	eol[1] = 10;
	eol[2] = 0;

	LinkList<CStr> keyObjList;		// This list keeps track of key obj enable disable scripts
									// added and ensures that we don't add the same structure
									// twice to the QN file

	LinkList<CStr> triggerList;

	// Build any all systems global QN node structures
	Link<INode*>* linkNode = (Link<INode*>*)QNNodeList.GetHead();

	GetScriptExporter()->ClearParticleTextureList();

	while(linkNode)
	{
		INode* node = linkNode->data;
		CStr scriptName = node->GetName();

		// Determine if there is a corresponding export node in the cutscene that has been flagged for
		// world space export so we know how to write out the position data in the QN packet for it
		bool bExportFromOrigin = IsFlaggedWorldSpace(pCutscene, node);

		if (node)
		{
			CStr strArray, strScript;
			GetScriptExporter()->OutputStrNode(node, strArray, strScript, &triggerList, bExportFromOrigin);

			scriptBuffer += scriptName + CStr(" = {") + CStr(eol);
			scriptBuffer += strArray + CStr(eol);
			scriptBuffer += CStr("}") + CStr(eol);

			if (strScript.Length() > 0)
				scriptBuffer += CStr(eol) + CStr(eol) + strScript + CStr(eol);

			keyObjList.Add(&scriptName);
		}

		linkNode = linkNode->next;
	}

	// Build any per cutscene global QN node structures
	Link<SaveableStr>* linkNodes = (Link<SaveableStr>*)pCutscene->nodeNameDB.GetHead();

	while(linkNodes)
	{
		INode* node = gInterface->GetINodeByName(linkNodes->data);
		CStr scriptName = node->GetName();

		if (node)
		{
			CStr strArray, strScript;
			GetScriptExporter()->OutputStrNode(node, strArray, strScript, &triggerList);

			scriptBuffer += scriptName + CStr(" = {") + CStr(eol);
			scriptBuffer += strArray + CStr(eol);
			scriptBuffer += CStr("}") + CStr(eol);

			if (strScript.Length() > 0)
				scriptBuffer += CStr(eol) + CStr(eol) + strScript + CStr(eol);

			keyObjList.Add(&scriptName);
		}

		linkNodes = linkNodes->next;
	}

	// Build the global key scripts
	while(link)
	{
		// If this is a KeyObj class its a special case
		// and the property data for the referenced node should be written out as a structure
		if (link->data.GetClass() == CStr(KEYOBJ_KEY_NAME) &&
			link->data.GetType()  == CStr(KEYOBJ_ENABLE_KEY_NAME))
		{
			CStr strArray, strScript;
			INode* node = gInterface->GetINodeByName(link->data.GetProp("Node"));

			if (!node)
			{
				link = link->next;
				continue;
			}

			//CStr scriptName = GetGlobalScriptName(pCutscene, &link->data);
			CStr scriptName = node->GetName();

			if (!keyObjList.Find(&scriptName))
			{
				GetScriptExporter()->OutputStrNode(node, strArray, strScript, &triggerList);

				scriptBuffer += scriptName + CStr(" = {") + CStr(eol);
				scriptBuffer += strArray + CStr(eol);
				scriptBuffer += CStr("}") + CStr(eol);

				if (strScript.Length() > 0)
					scriptBuffer += CStr(eol) + CStr(eol) + strScript + CStr(eol);

				keyObjList.Add(&scriptName);
			}
		}
		else
		{
			scriptBuffer += CStr("script ") + GetGlobalScriptName(pCutscene, &link->data) + CStr(eol);
			scriptBuffer += scriptKeyEditor->GetScriptBuffer(link) + CStr(eol);
			scriptBuffer += CStr("endscript") + CStr(eol) + CStr(eol);
		}

		link = link->next;
	}

	// Build the per object key scripts
	Link<CutsceneObj>* linkCSO = pCutscene->objDB.GetHead();

	while(linkCSO)
	{
		Link<TrackUIKey>* linkKey = linkCSO->data.trackKeyDB.GetHead();

		while(linkKey)
		{
			// If this is a KeyObj class its a special case
			// and the property data for the referenced node should be written out as a structure
			if (linkKey->data.GetClass() == CStr(KEYOBJ_KEY_NAME) &&
				linkKey->data.GetType()  == CStr(KEYOBJ_ENABLE_KEY_NAME))
			{
				CStr strArray, strScript;
				INode* node = ip->GetINodeByName(linkKey->data.GetProp("Node"));

				if (!node)
				{
					linkKey = linkKey->next;
					continue;
				}

				GetScriptExporter()->OutputStrNode(node, strArray, strScript, &triggerList);

				CStr scriptName = GetObjScriptName(pCutscene, &linkCSO->data, &linkKey->data);

				scriptBuffer += scriptName + CStr(" = {") + CStr(eol);
				scriptBuffer += strArray + CStr(eol);
				scriptBuffer += CStr("}") + CStr(eol);

				if (strScript.Length() > 0)
					scriptBuffer += CStr(eol) + CStr(eol) + strScript + CStr(eol);

				keyObjList.Add(&scriptName);
			}
			else
			{
				scriptBuffer += CStr("script ") + GetObjScriptName(pCutscene, &linkCSO->data, &linkKey->data) + CStr(eol);
				scriptBuffer += scriptKeyEditor->GetScriptBuffer(linkKey) + CStr(eol);
				scriptBuffer += CStr("endscript") + CStr(eol) + CStr(eol);
			}

			linkKey = linkKey->next;
		}

		linkCSO = linkCSO->next;
	}

	// Write out TriggerScripts array
	int i;

	/*
	scriptBuffer += CStr("TriggerScripts = [") + CStr(eol);
	
	for(i = 0; i < triggerList.GetSize(); i++)
		scriptBuffer += CStr("\t") + triggerList[i] + CStr(eol);

	scriptBuffer += CStr("]") + CStr(eol);
	*/

	// Write out CutsceneObjectNames array
	if (keyObjList.GetSize() > 0)
	{
		scriptBuffer += CStr("CutsceneObjectNames = [") + CStr(eol);
		
		for(i = 0; i < keyObjList.GetSize(); i++)
			scriptBuffer += CStr("\t") + keyObjList[i] + CStr(eol);

		scriptBuffer += CStr("]") + CStr(eol);
	}

	LinkList<CStr>* particle_tex_list = GetScriptExporter()->GetParticleTextureList();
	Link<CStr>* linkTex = particle_tex_list->GetHead();

	if (linkTex)
	{
		scriptBuffer += CStr(eol) + CStr("CutsceneParticleTextures = [") + CStr(eol);

		while(linkTex)
		{
			CStr name;
			if (strstr(linkTex->data, "\\"))
				name = strrchr(linkTex->data, '\\') + 1;
			else
				name = linkTex->data;

			scriptBuffer += CStr("\t\"particles/") + name + CStr("\"") + CStr(eol);
			linkTex = linkTex->next;
		}

		scriptBuffer += CStr("]") + CStr(eol);
	}

	char rmv[2];
	rmv[0] = 13;
	rmv[1] = 0;

	scriptBuffer = ReplaceStr(scriptBuffer, rmv, "");
	scriptBuffer = ReplaceStr(scriptBuffer, "\r\n", "\n");

	return scriptBuffer;
}

// This function compiles all the key scripts used in a particular cutscene
bool CutsceneExportDlg::CompileScriptKeys(Cutscene* pCutscene)
{
	CStr scriptBuffer = GenerateScript(pCutscene);

	// Dump the cutscene script out to a file
	char* appenv = getenv(APP_ENV);

	if (!appenv)
	{
		char strErr[256];
		sprintf(strErr, "The environment variable '%s' is not set.  Cannot continue.", APP_ENV);
		MessageBox(hwnd, strErr, "Environment Variable not Set", MB_ICONWARNING|MB_OK);
		return false;
	}

	CStr filename = CStr(appenv) + CStr(CUTSCENE_PATH) + pCutscene->name + ".qn";	

	FILE* fp = fopen(filename, "w");

	if (!fp)
	{
		char strErr[256];
		sprintf(strErr, "Failed to open '%s' for output", filename);
		MessageBox(hwnd, strErr, "Couldn't save file", MB_ICONWARNING|MB_OK);
		return false;
	}

	fprintf(fp, (char*)scriptBuffer);
	fclose(fp);

	// Compile the script
	// Execute qcomp to build .qb
	char  path[256];
	char* args[3];
	sprintf( path, "%s\\bin\\win32\\qcomp.exe", getenv(APP_ENV) );

	args[0] = path;
	args[1] = filename;
	args[2] = NULL;

	int rVal = ExecuteCommand( args );

	if (rVal != 0)
	{
		char strErr[256];
		sprintf(strErr, "QComp failed to compile the script file.  You should examine '%s' to ensure the generated code is valid with the current codebase.", (char*)filename);
		MessageBox(hwnd, strErr, "QComp failed to compile", MB_ICONWARNING|MB_OK);
		return false;
	}

	return true;
}

int CutsceneExportDlg::GetTotalScriptKeys(Cutscene* cutscene)
{
	Link<CutsceneObj>* link = cutscene->objDB.GetHead();
	int count = cutscene->trackKeyDB.GetSize();

	while(link)
	{
		count += link->data.trackKeyDB.GetSize();
		link = link->next;
	}

	return count;
}

void CutsceneExportDlg::OpenQNEditor()
{
	if (pQNEditor)
		delete pQNEditor;

	GetScriptExporter()->ParseScriptIni();
	
	pQNEditor = new QNEditor(hInstance, hwnd, this);
	pQNEditor->Show();
}

bool CutsceneExportDlg::Load(char* fname)
{
	OPENFILENAME ofn;
	char filename[2048]="";

	if (!fname)
	{
		// Inform the user that this will clear the current list
		int rVal=MessageBox(hwnd,"This operation will clear the current cutscene database list.  (Object assigned properties will be retained)\nAre you sure?","Load Cutscene Database",MB_YESNO|MB_ICONQUESTION);
		if (rVal==IDNO)
			return false;

		// Pop open a file dialog
		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=hwnd;
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Cutscene Database (*.cdb)\0*.cdb\0All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=0;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Import Cutscene Database";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".cdb");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetOpenFileName(&ofn);
	}
	else
		strcpy(filename, fname);

	// Exit if cancelled
	if (strlen(filename)==0)
		return false;

	FILE* fp = fopen(filename, "rb");

	if (!fp)
		return false;

	// Determine the size of the file (the entire file will be loaded into memory)
	// And assigned into the MAX File
	unsigned long offset;
	fseek(fp, 0, SEEK_END);
	offset = ftell(fp);
	fseek(fp, 0, SEEK_SET);

	unsigned char* pData = (unsigned char*)malloc(offset);
	fread(pData, offset, 1, fp);
	fclose(fp);

	DWORD version;
	unsigned char* pos = (unsigned char*)pData;

	version = *((DWORD*)pos);
	pos += sizeof(DWORD);

	GetList<Cutscene>(&pos, &cutsceneDB);

	if (version >= 0x0003)
		GetNodeList(&pos, &QNNodeList);

	// Update version info
	Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

	while(linkCutscene)
	{
		linkCutscene->data.version = CUTSCENEDATA_VERSION;

		Link<CutsceneObj>* linkObj = linkCutscene->data.objDB.GetHead();

		while(linkObj)
		{
			linkObj->data.version = CUTSCENEOBJ_VERSION;
			linkObj = linkObj->next;
		}

		linkCutscene = linkCutscene->next;
	}

	// Users can no longer specify exports other than models and skins
	// this is handled by the objanim lister and the boneanim lister now instead
	DiscardLegacyFlags();

	// Assign the data into the MAX file
	INode* scene = ip->GetRootNode();
	scene->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA);
	scene->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_DATA, offset, pData);

	// Update display (not trusting reloading on show for now)
	CreateCutsceneList();
	InitObjList();
	RetrieveAllProps();
	
	// Load key data for the selected cutscene
	UpdateKeyData();

	return true;
}

bool CutsceneExportDlg::Save(char* fname)
{
	OPENFILENAME ofn;
	char filename[2048]="";

	if (!fname)
	{
		ofn.lStructSize=sizeof(ofn);
		ofn.hwndOwner=hwnd;
		ofn.hInstance=hInstance;
		ofn.lpstrFilter="Cutscene Database (*.cdb)\0*.cdb\0All Files (*.*)\0*.*\0\0";
		ofn.lpstrCustomFilter=NULL;
		ofn.nMaxCustFilter=0;
		ofn.nFilterIndex=0;
		ofn.lpstrFile=filename;
		ofn.nMaxFile=256;
		ofn.lpstrFileTitle=NULL;
		ofn.nMaxFileTitle=0;
		ofn.lpstrInitialDir=NULL;
		ofn.lpstrTitle="Export Cutscene Database";
		ofn.Flags=OFN_LONGNAMES|OFN_ENABLESIZING;
		ofn.nFileOffset=0;
		ofn.nFileExtension=0;
		ofn.lpstrDefExt=TEXT(".cdb");
		ofn.lCustData=0;
		ofn.lpfnHook=NULL;
		ofn.lpTemplateName=NULL;

		GetSaveFileName(&ofn);
	}
	else
		strcpy(filename, fname);

	// Exit if cancelled
	if (strlen(filename)==0)
		return false;

	FILE* fp = fopen(filename,"wb");

	if (!fp)
		return false;

	int size = ListSize(&cutsceneDB) + sizeof(DWORD) + GetNodeListSize(&QNNodeList);
	unsigned char* data = (unsigned char*)malloc(size);
	unsigned char* pos = data;
	
	*((DWORD*)pos) = CUTSCENEEXPORTER_CHUNK_VERSION;
	pos += sizeof(DWORD);
	WriteList<Cutscene>(&pos, &cutsceneDB);

	// v2
	WriteNodeList(&pos, &QNNodeList);

	fwrite(data, size, 1, fp);
	fclose(fp);
	free(data);

	SaveToMAX();
	return true;
}

void CutsceneExportDlg::PopulatePartialAnimList()
{
	int iItem;

	SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_RESETCONTENT, 0, 0);
	iItem = SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_ADDSTRING, 0, (LPARAM)"All Bones");
	SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_SETITEMDATA, (WPARAM)iItem, 0);

	Link<PartialAnimSet>* link = partialAnimEditor->GetAnimDB()->GetHead();

	while(link)
	{
		iItem = SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_ADDSTRING, 0, (LPARAM)(char*)link->data.name);
		SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_SETITEMDATA, (WPARAM)iItem, (LPARAM)link);
		link = link->next;
	}

	SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_SETCURSEL, 0, 0);
}

void CutsceneExportDlg::PartialAnimChanged()
{
	if (bLockPropChange)
		return;

	bLockPropChange = true;

	// Anim change can only be executed if a single cutscene is selected
	int nSel = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	if (nSel == 1)
	{
		int item;
		SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);

		Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd,
					                                                       IDC_CUTSCNLIST,
																		   LB_GETITEMDATA,
																		   (WPARAM)item,
																		   0);
		int selCount = pObjList->GetSelCount();

		for(int i = 0; i < pObjList->GetCount(); i++)
		{
			if (pObjList->IsSelected(i))
			{
				INode* node = (INode*)pObjList->GetItemData(i);

				if (node)
				{
					Link<CutsceneObj>* linkObj = GetCutsceneObj(&linkCutscene->data, node);
					
					int iSel = SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_GETCURSEL, 0, 0);
					Link<PartialAnimSet>* link = (Link<PartialAnimSet>*)SendDlgItemMessage(hwnd, IDC_PARTIALANIMLIST, CB_GETITEMDATA, (WPARAM)iSel, 0);
					
					if (link)
						linkObj->data.partialAnimSet = link->data.name;
					else
						linkObj->data.partialAnimSet = "";
				}

			}
		}
	}
	else
	{
		MessageBox(hwnd, "You must have one and only one cutscene selected for changes to take affect.", "Invalid number of cutscenes selected", MB_ICONWARNING|MB_OK);
		bLockPropChange = false;
		return;
	}

	StoreObjectProps();
	bLockPropChange = false;
}

void CutsceneExportDlg::MakeNodeReferences()
{
	// Scan through all the nodes in all the cutscenes and create references to them
	// this way, we can recieve notification whenever they're updated

	Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

	while(linkCutscene)
	{
		Link<CutsceneObj>* linkObj = linkCutscene->data.objDB.GetHead();

		while(linkObj)
		{
			INode* node = gInterface->GetINodeByName(linkObj->data.name);

			if (node)
				MakeRefByID(FOREVER, 0, node);

			linkObj = linkObj->next;
		}

		linkCutscene = linkCutscene->next;
	}
}

RefResult CutsceneExportDlg::NotifyRefChanged(Interval changeInt, RefTargetHandle hTarget, 
								              PartID& partID,  RefMessage message)
{
	switch(message)
	{
	case REFMSG_CHANGE:
		{
			INode* node = (INode*)hTarget;

			// Flag the node as being modified in all cutscenes
			Link<Cutscene>* linkCutscene = cutsceneDB.GetHead();

			while(linkCutscene)
			{
				Link<CutsceneObj>* link = GetCutsceneObj(&linkCutscene->data, node);

				if (link)
				{
					if (!link->data.bModified)
					{
						link->data.bModified = true;
						InitObjList(displayMode);
					}
				}

				/*
				char buf[256];
				sprintf(buf, "The node '%s' has been changed.", (char*)node->GetName());
				MessageBox(gInterface->GetMAXHWnd(), buf, "Node Changed", MB_OK);
				*/

				linkCutscene = linkCutscene->next;
			}
		}
	}

	return REF_SUCCEED;
}

void* CutsceneExportDlg::LoadPacket(char* filename, unsigned int objName, unsigned int packetType, LIBObjPacket* packetDesc)
{
	FILE* fpCutscene = fopen(filename, "rb");

	if (!fpCutscene)
	{
		char buf[256];
		sprintf(buf, "Cutscene file '%s' could not be loaded.  Cannot load packet.\n", filename);

#ifdef PACKET_DEBUG
		//MessageBox(gInterface->GetMAXHWnd(), buf, "LoadPacket", MB_OK);
		FILE* fpDebug = fopen("c:\\cutpakdbg.txt", "a");
		fputs(buf, fpDebug);
		fclose(fpDebug);
#endif

		OutputDebugString(buf);
		return NULL;
	}

	// Scan through the packet header and search for the chunk of interest

	// Write out the cutscene version
	unsigned int CutsceneVer;
	fread(&CutsceneVer, sizeof(unsigned int), 1, fpCutscene);

	if (CutsceneVer != CUTSCENE_VERSION)
	{
		OutputDebugString("Cutscene Version does not match cannot load packet\n");
		return NULL;
	}

	unsigned int nPackets;
	fread(&nPackets, sizeof(unsigned int), 1, fpCutscene);

	for(int i = 0; i < nPackets; i++)
	{
		fread(packetDesc, sizeof(LIBObjPacket), 1, fpCutscene);

		if (packetDesc->nameCRC == objName &&
			packetDesc->extCRC  == packetType)
		{
			fseek(fpCutscene, packetDesc->offset, SEEK_SET);
			void* pMem = malloc(packetDesc->size);
			fread(pMem, packetDesc->size, 1, fpCutscene);
			fclose(fpCutscene);

			return pMem;
		}
	}

	char buf[256];
	sprintf(buf, "Packet name 0x%x type 0x%x was not found in header.  Cannot load packet\n", objName, packetType);
	OutputDebugString(buf);

	fclose(fpCutscene);
	return NULL;
}

// A new export has been completed everything should be up-to-date
void CutsceneExportDlg::ResetModifiedFlags()
{
	Link<Cutscene>* link = cutsceneDB.GetHead();

	while(link)
	{
		Link<CutsceneObj>* linkCSO = link->data.objDB.GetHead();

		while(linkCSO)
		{
			linkCSO->data.bModified = false;
			linkCSO = linkCSO->next;
		}

		link = link->next;
	}

	InitObjList(displayMode);
}

void CutsceneExportDlg::SetModified(bool bMode)
{
	// Must be within the cutscene display
	if (displayMode != DISP_CUTSCENE)
		return;

	// Determine current cutscene
	int nSel = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);

	// Only one cutscene may be selected
	if (nSel != 1)
	{
		MessageBox(hwnd, "You must have a single cutscene selected", "Set Modified Flag", MB_ICONWARNING | MB_OK);
		return;
	}

	int item;
	SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)1, (LPARAM)&item);
	Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)item, 0);

	// Assign the modified flag to all selected items
	int nItems = pObjList->GetCount();

	for(int i = 0; i < nItems; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* node = (INode*)pObjList->GetItemData(i);
			Link<CutsceneObj>* linkCSO = GetCutsceneObj(&linkCutscene->data, node);

			linkCSO->data.bModified = bMode;
		}
	}

	// Refresh the list so the user can see the flag change
	InitObjList(displayMode);
}

void CutsceneExportDlg::FlagAutoAssign()
{
	// Can only auto assign if displaying in cutscene mode
	if (displayMode != DISP_CUTSCENE)
	{
		MessageBox(hwnd, "You must be in the cutscene display mode in order to use this option", "AutoAssign", MB_ICONWARNING|MB_OK);
		return;
	}

	// Determine the cutscene that is currently selected
	int  nSelObjCount  = pObjList->GetCount();
	int  nSelCutscenes = SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELCOUNT, 0, 0);
	int *selCutscenes  = new int[nSelCutscenes];

	for(int i = 0; i < nSelObjCount; i++)
	{
		if (pObjList->IsSelected(i))
		{
			INode* pObjNode = (INode*)pObjList->GetItemData(i);

			// Scan through the selected cutscenes and attempt to find this node
			SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETSELITEMS, (WPARAM)nSelCutscenes, (LPARAM)selCutscenes);

			for(int j = 0; j < nSelCutscenes; j++)
			{
				Link<Cutscene>* linkCutscene = (Link<Cutscene>*)SendDlgItemMessage(hwnd, IDC_CUTSCNLIST, LB_GETITEMDATA, (WPARAM)j, 0);
				assert(linkCutscene);

				Link<CutsceneObj>* linkCSO   = GetCutsceneObj(&linkCutscene->data, pObjNode);
				
				if (linkCSO)
					linkCSO->data.type = DetermineType(pObjNode);
			}
		}
	}

	delete [] selCutscenes;

	InitObjList(displayMode);
}
