//****************************************************************************
//* MODULE:         Tools/SkeConv
//* FILENAME:       main.cpp
//* OWNER:          Gary Jesdanun
//* CREATION DATE:  2/19/2003
//****************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <io.h>

#include <core/defines.h>
#include <core/math.h>

#include "Converter.h"
#include "Utility.h"
#include "VirtualFile.h"

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

#define	vMAJOR_REVISION		1
#define	vMINOR_REVISION		0
#define vEXE_NAME			"SkeConv"

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

class CSkeletonConverter : public Utils::CBaseConverter
{
public:
	CSkeletonConverter();
	virtual ~CSkeletonConverter();

public:
	virtual void	Reset();
	virtual void	PrintUsage();

protected:
	virtual bool	ReadInputFile( IoUtils::CVirtualInputFile* pInputFile );
	virtual bool	WriteOutputFile( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder );
	int				get_bone_index_by_name( uint32 name );

protected:
	enum
	{
		vMAX_BONES = 128
	};
	uint32			m_checksum;
	int				m_numBones;
	uint32			m_boneNames[vMAX_BONES];
	uint32			m_parentNames[vMAX_BONES];
	uint32			m_flipNames[vMAX_BONES];
	Mth::Quat		m_neutralPoseQuats[vMAX_BONES];
	Mth::Vector		m_neutralPoseVectors[vMAX_BONES];
	int				m_versionNumber;
	uint32			m_flags;
};

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CSkeletonConverter::CSkeletonConverter()
{
	Reset();
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

CSkeletonConverter::~CSkeletonConverter()
{
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CSkeletonConverter::Reset()
{
	m_versionNumber = 1;
	m_flags = 0;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

int CSkeletonConverter::get_bone_index_by_name( uint32 name )
{
	for ( int i = 0; i < m_numBones; i++ )
	{
		if ( m_boneNames[i] == name )
		{
			return i;
		}
	}

	Utils::Assert( 0, "What the?" );
	return -1;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CSkeletonConverter::ReadInputFile( IoUtils::CVirtualInputFile* pInputFile )
{
	uint32 checksum;
	pInputFile->Read( (char*)&checksum, sizeof(uint32) );

	if ( checksum == CRCD(0x222756d5,"skeleton") )
	{
		// then it's version 2 already
		pInputFile->Read( (char*)&m_versionNumber, sizeof(int) );

		pInputFile->Read( (char*)&m_flags, sizeof(uint32) );
	}

	pInputFile->Read( (char*)&m_numBones, sizeof(int) );
	if ( m_debugMode )
	{
		printf( "NumBones = %08x\n", m_numBones );
	}

	pInputFile->Read( (char*)&m_boneNames, m_numBones * sizeof(uint32) );
	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			printf( "Name[%d] = %08x\n", i, m_boneNames[i] );
		}
	}

	pInputFile->Read( (char*)&m_parentNames, m_numBones * sizeof(uint32) );
	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			printf( "ParentName[%d] = %08x\n", i, m_parentNames[i] );
		}
	}

	pInputFile->Read( (char*)&m_flipNames, m_numBones * sizeof(uint32) );
	if ( m_debugMode )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			printf( "FlipName[%d] = %08x\n", i, m_flipNames[i] );
		}
	}

	// if it's version 2
	if ( m_versionNumber >= 2 )
	{
		for ( int i = 0; i < m_numBones; i++ )
		{
			pInputFile->Read( (char*)&m_neutralPoseQuats[i], sizeof(Mth::Quat) );
			pInputFile->Read( (char*)&m_neutralPoseVectors[i], sizeof(Mth::Vector) );

			if ( m_debugMode )
			{
				printf( "NeutralPoseQuat[%d] = %f %f %f %f\n", i, m_neutralPoseQuats[i][X], m_neutralPoseQuats[i][Y], m_neutralPoseQuats[i][Z], m_neutralPoseQuats[i][W] );
				printf( "NeutralPoseVector[%d] = %f %f %f %f\n", i, m_neutralPoseVectors[i][X], m_neutralPoseVectors[i][Y], m_neutralPoseVectors[i][Z], m_neutralPoseVectors[i][W] );
			}
		}
	}
	else
	{
		printf( "Error!   SKE file is in old format.  Please use SkePatch to update file to V2.\n" );
		return false;
	}

	if ( m_debugMode )
	{
		// loop through and find the object-space positions of all the bones
		Mth::Matrix* pMatrices = new Mth::Matrix[m_numBones];
		for ( int i = 0; i < m_numBones; i++ )
		{	
			pMatrices[i].Ident();

			// don't forget that quatvectomatrix is destructive!
			Mth::Quat q = m_neutralPoseQuats[i];
			Mth::Vector v = m_neutralPoseVectors[i];
			Mth::QuatVecToMatrix( &q, &v, &pMatrices[i] );

			if ( i != 0 )
			{
				// if it's not the root, then apply the parent's xform
				int parentName = m_parentNames[i];
				Mth::Matrix* pParentMatrix = pMatrices + get_bone_index_by_name( parentName );
				pMatrices[i] *= (*pParentMatrix);
			}

			if ( m_boneNames[i] == Crc::GenerateCRCFromString( "bone_neck" ) )
			{
//				pMatrices[i].Ident();
			}

			printf( "Object space[%d]: %f %f %f\n", 
				i, 
				(pMatrices[i])[Mth::POS][X], 
				(pMatrices[i])[Mth::POS][Y], 
				(pMatrices[i])[Mth::POS][Z] );
		}
		
		delete[] pMatrices;
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

bool CSkeletonConverter::WriteOutputFile( IoUtils::CVirtualOutputFile* pOutputFile, bool reverseByteOrder )
{
	pOutputFile->Write( (const char*)&m_versionNumber, sizeof(int), reverseByteOrder );

	pOutputFile->Write( (const char*)&m_flags, sizeof(uint32), reverseByteOrder );

	pOutputFile->Write( (const char*)&m_numBones, sizeof(int), reverseByteOrder );

	for ( int i = 0; i < m_numBones; i++ )
	{
		pOutputFile->Write( (const char*)&m_boneNames[i], sizeof(uint32), reverseByteOrder );
	}

	for ( int i = 0; i < m_numBones; i++ )
	{
		pOutputFile->Write( (const char*)&m_parentNames[i], sizeof(uint32), reverseByteOrder );
	}

	for ( int i = 0; i < m_numBones; i++ )
	{
		pOutputFile->Write( (const char*)&m_flipNames[i], sizeof(uint32), reverseByteOrder );
	}

	for ( int i = 0; i < m_numBones; i++ )
	{
		Mth::Quat myQuat = m_neutralPoseQuats[i];
		pOutputFile->Write( (const char*)&(myQuat[X]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myQuat[Y]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myQuat[Z]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myQuat[W]), sizeof(float), reverseByteOrder );

		Mth::Vector myVector = m_neutralPoseVectors[i];
		pOutputFile->Write( (const char*)&(myVector[X]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myVector[Y]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myVector[Z]), sizeof(float), reverseByteOrder );
		pOutputFile->Write( (const char*)&(myVector[W]), sizeof(float), reverseByteOrder );
	}

	return true;
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void CSkeletonConverter::PrintUsage()
{
#ifdef _DEBUG
	printf( "%s v%d.%d (%s)\n", vEXE_NAME, vMAJOR_REVISION, vMINOR_REVISION, "Debug" );
#else
	printf( "%s v%d.%d (%s)\n", vEXE_NAME, vMAJOR_REVISION, vMINOR_REVISION, "Release" );
#endif
	printf( "Neversoft Entertainment, 2003\n" );
	printf( "\nUsage: SkeConv -b<batchfile> -f<filename> -p[p | g | x]\n" );
	printf( "Where p = PS2, g = GameCube, x = Xbox\n" );	
	printf( "Ex: SkeConv -f%%PROJ_ROOT%%/data/skeletons/anl_rhino.ske -pp\n" );
}

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/

void main( int argc, char* argv[] )
{
	CSkeletonConverter theConverter;
	Utils::ConverterMain( &theConverter, argc, argv );
}