/*
	PCViewerConv.cpp
	aml - 9-27-02
*/

#define NULL 0
#include "PCViewerConv.h"
#include "VirtualFile.h"
#include <stdlib.h>

struct Point3D
{
	float x, y, z;

	Point3D()
	{
		x = y = z = 0.0f;
	}

	Point3D(float f[3])
	{
		x = f[0];
		y = f[1];
		z = f[2];
	}

	int operator== (Point3D& pt)
	{
		return (x == pt.x && y == pt.y && z == pt.z);
	}

	int operator!= (Point3D& pt)
	{
		return !(x == pt.x && y == pt.y && z == pt.z);
	}
};

struct Face3D
{
	int            v[3];			// Indicies into the vert list for this face
	int            numTextures;		// Number of textures applied to this face
	int*           texIDs;			// Indicies into the textures
	unsigned long* texIDsCRC;		// 
	int            texLightmap;		// Index of the lightmap texture in the lightmap texture file
	UVCoord        uvsLightmap[3];	// UV coordinates for the lightmap texture if appropriate
	UVCoord*       uvs;				// UV coordinates 3 per texture

	Face3D()
	{
		texIDs    = NULL;
		texIDsCRC = NULL;
		uvs       = NULL;
	}

	~Face3D()
	{
		if (texIDs)
			delete [] texIDs;

		if (texIDsCRC)
			delete [] texIDsCRC;

		if (uvs)
			delete [] uvs;
	}
};

struct Object3D
{
	Point3D* verts;
	Face3D*  faces;
	
	int numVerts;
	int numFaces;

	Object3D()
	{
		verts = NULL;
		faces = NULL;
	}

	~Object3D()
	{
		if (verts)
			delete [] verts;

		if (faces)
			delete [] faces;
	}
};

// All PCViewer textures are 32-bit RGBA or 8-bit gray scale for Lightmaps
struct TexData
{
	int width;
	int height;

	unsigned char* data;	// Actual texel data (should be allocated with new!)
	unsigned long  crc;		// Only in SceneConv: NxTexture matching CRC
	bool           bUsed;	// True if this texture is being used

	TexData()
	{
		data = NULL;
	}

	~TexData()
	{
		if (data)
			delete [] data;
	}
};

//////////////////////// Assumed to get to compile  aml
bool	gTextureOverride = false;
int		gTextureOverrideMapping[Utils::vNUM_PLATFORMS] = { 0, 1, 2 };
////////////////////////////////////////////////////////////////////////

class PCViewerConverter : public IPCViewerConverter
{
	Object3D*  objs;
	TexData*   tex;
	TexData*   lighttex;
	int        numObjs;
	int        numTextures;		// Number of normal textures

	int		GetTextureIndex( TexData* tex, int nTextures, unsigned long crc );
	int     ConvertTextures();

public:
	bool	SaveCollisionData( char* path );
	bool	ConvertData( void );
	bool	SaveScene( char* path );
	bool	SaveTextureDictionary( char* path, char* usg_path );
	bool	SaveCASFlags( char* path );	
	int     GetPlatformFlags( void );
	int		TextureCount( void );
};

static PCViewerConverter gPCViewerConv;

IPCViewerConverter* GetPCViewerConverter( void )
{
	return &gPCViewerConv;
}

int	PCViewerConverter::GetPlatformFlags( void )
{
	if( gTextureOverride )
	{
		return( 1 << gTextureOverrideMapping[Utils::vPLATFORM_PCVIEWER] );
	}
	return mPLAT_PCVIEWER;
}

bool PCViewerConverter::SaveCollisionData( char* path )
{
	// PC Viewer does not currently have collision supported
	return true;
}

int PCViewerConverter::GetTextureIndex( TexData* tex, int nTextures, unsigned long crc )
{
	for(int i = 0; i < nTextures; i++)
	{
		if (tex[i].crc == crc)
			return i;
	}

	return -1;
}

int PCViewerConverter::ConvertTextures()
{
	// Add all material textures to our texture list
	int nMaterials = m_scene->m_NumMaterials;
	TexData* temp_tex = new TexData[m_num_textures];
	int i;

	for(i = 0; i < m_num_textures; i++)
	{
		// PC Viewer currently only takes input as 32-bit RGBA
		m_textures[i].ConvertTo32BitPixelFormat();

		// We don't currently care about mip levels
		temp_tex[i].width  = m_textures[i].m_Width[0];
		temp_tex[i].height = m_textures[i].m_Height[0];
		
		temp_tex[i].data = new unsigned char[temp_tex[i].width * temp_tex[i].height * 4];
		memcpy(temp_tex[i].data, m_textures[i].m_TexelData[0], temp_tex[i].width * temp_tex[i].height * 4);

		temp_tex[i].crc = m_textures[i].m_Checksum;
		temp_tex[i].bUsed = false;
	}

	// We'll now scan through all the geometry so we can flag what textures are actually used
	numObjs = m_scene->m_NumObjects;
	int nUsedTextures = 0;

	for(i = 0; i < numObjs; i++)
	{
		NxObject* obj = m_scene->m_Objects + i;
		
		int nFaces = obj->m_NumFaces;

		for(int j = 0; j < nFaces; j++)
		{
			unsigned long mtlCRC = obj->m_Faces[j].m_MatChecksum;
			NxMaterial* mtl = m_scene->GetMaterial(mtlCRC);
			
			int nPasses = mtl->m_NumPasses;

			for(int k = 0; k < nPasses; k++)
			{
				// Using XBOX textures for now
				unsigned long texID = mtl->m_Passes[k].GetTextureChecksum( Utils::vPLATFORM_XBOX );

				if (texID != 0)
				{
					int texIdx = GetTextureIndex( temp_tex, m_num_textures, texID );
					
					if (!temp_tex[texIdx].bUsed)
					{
						temp_tex[texIdx].bUsed = true;
						nUsedTextures++;
					}
				}
			}
		}
	}

	// Reconstruct the texture list with only the used textures
	tex = new TexData[nUsedTextures];
	int curtex = 0;

	for(i = 0; i < m_num_textures; i++)
	{
		if (temp_tex[i].bUsed)
		{
			tex[curtex] = temp_tex[i];
			tex[curtex].data = new unsigned char[temp_tex[i].width * temp_tex[i].height * 4];
			memcpy(tex[curtex].data, temp_tex[i].data, temp_tex[i].width * temp_tex[i].height * 4);

			curtex++;
		}
	}

	delete [] temp_tex;

	numTextures = nUsedTextures;
	return curtex;
}

bool PCViewerConverter::ConvertData( void )
{
	int nUsedTextures = ConvertTextures();
	
	// Convert all the game objects into Object3Ds for the previewer
	numObjs = m_scene->m_NumObjects;
	objs = new Object3D[numObjs];
	
	for(int i = 0; i < m_scene->m_NumObjects; i++)
	{
		NxObject* orig_obj = m_scene->m_Objects + i;
		Object3D* new_obj  = objs + i;

		//orig_obj->OptimizeVertList(m_scene);

		// Convert verts
		int nVerts = m_scene->m_Objects[i].m_NumVerts;
		new_obj->verts = new Point3D[nVerts];
		new_obj->numVerts = nVerts;

		for(int curvert = 0; curvert < nVerts; curvert++)
		{
			new_obj->verts[curvert].x = orig_obj->m_Verts[curvert].m_Pos[0];
			new_obj->verts[curvert].y = orig_obj->m_Verts[curvert].m_Pos[1];
			new_obj->verts[curvert].z = orig_obj->m_Verts[curvert].m_Pos[2];
		}

		// Convert faces
		//int nFaces = m_scene->m_Objects[i].m_NumFaces;
		int nFaces = m_scene->m_Objects[i].m_LODinfo[4].numFaces;

		new_obj->faces = new Face3D[nFaces];
		new_obj->numFaces = nFaces;

		int nLODs = orig_obj->m_LODLevels;

		// Compare orig face indicies with LOD level 0
		for(int cmp = 0; cmp < nFaces; cmp++)
		{
			if (m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[0] != m_scene->m_Objects[i].m_LODinfo[4].faces[cmp].v[0] ||
				m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[1] != m_scene->m_Objects[i].m_LODinfo[4].faces[cmp].v[1] ||
				m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[2] != m_scene->m_Objects[i].m_LODinfo[4].faces[cmp].v[2])
			{
				int zzz;
				zzz = 0;
			}

			Point3D vert0, vert1;

			vert0 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[0]].m_Pos;
			vert1 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_LODinfo[0].faces[cmp].v[0]].m_Pos;

			if (vert0 != vert1)
			{
				int zzz;
				zzz = 0;
			}

			vert0 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[1]].m_Pos;
			vert1 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_LODinfo[0].faces[cmp].v[1]].m_Pos;

			if (vert0 != vert1)
			{
				int zzz;
				zzz = 0;
			}


			vert0 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_Faces[cmp].m_Vertex[2]].m_Pos;
			vert1 = m_scene->m_Objects[i].m_Verts[m_scene->m_Objects[i].m_LODinfo[0].faces[cmp].v[2]].m_Pos;

			if (vert0 != vert1)
			{
				int zzz;
				zzz = 0;
			}

		}

		for(int curface = 0; curface < nFaces; curface++)
		{
			new_obj->faces[curface].v[0] = orig_obj->m_LODinfo[4].faces[curface].v[0];
			new_obj->faces[curface].v[1] = orig_obj->m_LODinfo[4].faces[curface].v[1];
			new_obj->faces[curface].v[2] = orig_obj->m_LODinfo[4].faces[curface].v[2];

			/*
			new_obj->faces[curface].v[0] = orig_obj->m_Faces[curface].m_Vertex[0];
			new_obj->faces[curface].v[1] = orig_obj->m_Faces[curface].m_Vertex[1];
			new_obj->faces[curface].v[2] = orig_obj->m_Faces[curface].m_Vertex[2];
			*/

			unsigned long mtlCRC = orig_obj->m_Faces[curface].m_MatChecksum;

			NxMaterial* mtl = m_scene->GetMaterial(mtlCRC);

			int nPasses = mtl->m_NumPasses;

			new_obj->faces[curface].numTextures = nPasses;
			new_obj->faces[curface].texIDs      = new int[nPasses];
			new_obj->faces[curface].texIDsCRC   = new unsigned long[nPasses];
			new_obj->faces[curface].uvs         = new UVCoord[nPasses * 3];

			// Copy UV coords
			for(int curuv = 0; curuv < nPasses; curuv++)
			{
				NxVertex* verts = orig_obj->m_Verts;

				new_obj->faces[curface].uvs[curuv * 3    ].u = verts[orig_obj->m_Faces[curface].m_Vertex[0]].m_TexCoord[curuv][0];
				new_obj->faces[curface].uvs[curuv * 3    ].v = verts[orig_obj->m_Faces[curface].m_Vertex[0]].m_TexCoord[curuv][1];

				new_obj->faces[curface].uvs[curuv * 3 + 1].u = verts[orig_obj->m_Faces[curface].m_Vertex[1]].m_TexCoord[curuv][0];
				new_obj->faces[curface].uvs[curuv * 3 + 1].v = verts[orig_obj->m_Faces[curface].m_Vertex[1]].m_TexCoord[curuv][1];

				new_obj->faces[curface].uvs[curuv * 3 + 2].u = verts[orig_obj->m_Faces[curface].m_Vertex[2]].m_TexCoord[curuv][0];
				new_obj->faces[curface].uvs[curuv * 3 + 2].v = verts[orig_obj->m_Faces[curface].m_Vertex[2]].m_TexCoord[curuv][1];
			}

			for(int curpass = 0; curpass < nPasses; curpass++)
			{
				// We'll use XBOX textures for the PC Previewer for now
				unsigned long texCRC = mtl->m_Passes[curpass].GetTextureChecksum( Utils::vPLATFORM_XBOX );

				int texIdx = GetTextureIndex( tex, numTextures, texCRC);
				tex[texIdx].bUsed = true;

				new_obj->faces[curface].texIDs[curpass]    = texIdx;
				new_obj->faces[curface].texIDsCRC[curpass] = texCRC;
			}

			new_obj->faces[curface].texLightmap = orig_obj->m_Faces[curface].m_LightmapChecksum;
			new_obj->faces[curface].uvsLightmap[0] = orig_obj->m_Faces[curface].m_LightmapUV[0];
			new_obj->faces[curface].uvsLightmap[1] = orig_obj->m_Faces[curface].m_LightmapUV[1];
			new_obj->faces[curface].uvsLightmap[2] = orig_obj->m_Faces[curface].m_LightmapUV[2];
		}
	}
	
	return true;
}

bool PCViewerConverter::SaveScene( char* path )
{
	IoUtils::CVirtualOutputFile file;
	
	// 10-meg buffer
	file.Init(10*1024*1024);

	file.Write((char*)&numObjs, sizeof(int));
	
	for(int i = 0; i < numObjs; i++)
	{
		// Write out vert list
		int nVerts = objs[i].numVerts;

		file.Write((char*)&nVerts, sizeof(int));
		file.Write((char*)objs[i].verts, sizeof(Point3D) * nVerts);

		// Write out the faces
		int nFaces = objs[i].numFaces;

		file.Write((char*)&nFaces, sizeof(int));
		for(int j = 0; j < nFaces; j++)
		{
			int nTextures = objs[i].faces[j].numTextures;

			file.Write((char*)objs[i].faces[j].v, sizeof(int) * 3);
			file.Write((char*)&nTextures, sizeof(int));
			file.Write((char*)objs[i].faces[j].texIDs, sizeof(int) * nTextures);
			file.Write((char*)objs[i].faces[j].uvs, sizeof(UVCoord) * nTextures * 3);
			//file.Write((char*)objs[i].faces[j].texIDsCRC, sizeof(unsigned long) * nTextures);
			file.Write((char*)&objs[i].faces[j].texLightmap, sizeof(int));
			file.Write((char*)&objs[i].faces[j].uvsLightmap, sizeof(UVCoord) * 3);
		}
	}

	file.Save( path );

	return true;
}

bool PCViewerConverter::SaveTextureDictionary( char* path, char* usg_path )
{
	// The PCViewer texture dictionary consists of 2 sections.  The first is all the
	// scene textures (32-bit RGBA), the second is all the lightmaps (greyscale 8-Bit)
	IoUtils::CVirtualOutputFile file;
	int i;

	// 10-meg buffer
	file.Init(10*1024*1024);

	// Dump all the normal textures
	file.Write((char*)&numTextures, sizeof(int));
	for(i = 0; i < numTextures; i++)
	{
		file.Write((char*)&tex[i].width,sizeof(int));
		file.Write((char*)&tex[i].height,sizeof(int));
		file.Write((char*)tex[i].data,tex[i].width * tex[i].height * 4);
	}

	// Dump all the lightmap textures (identical to .lmp layout but within the same file)
	file.Write((char*)&m_NumLightmaps, sizeof(int));

	for(i = 0; i < m_NumLightmaps; i++)
	{
		//bool rVal = m_lightmaps[i].ConvertTo32BitPixelFormat();

		file.Write((char*)&m_lightmaps[i].m_Width[0],sizeof(int));
		file.Write((char*)&m_lightmaps[i].m_Height[0],sizeof(int));

		// Convert to RGBA
		int size = m_lightmaps[i].m_Width[0] * m_lightmaps[i].m_Height[0];
		//file.Write((char*)m_lightmaps[i].m_TexelData[0], size);

		FILE* fp = fopen("c:\\lmap.raw","wb");

		for(int j = 0; j < size; j++)
		{
			unsigned char alpha = 255;
			//file.Write((char*)&(m_lightmaps[i].m_TexelData[0])[j], sizeof(unsigned char));
			//file.Write((char*)&(m_lightmaps[i].m_TexelData[0])[j], sizeof(unsigned char));
			//file.Write((char*)&(m_lightmaps[i].m_TexelData[0])[j], sizeof(unsigned char));
			//file.Write((char*)&alpha, sizeof(unsigned char));

			file.Write((char*)&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char));
			file.Write((char*)&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char));
			file.Write((char*)&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char));
			file.Write((char*)&alpha, sizeof(unsigned char));

			//fwrite(&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char), 1, fp);
			//fwrite(&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char), 1, fp);
			//fwrite(&((m_lightmaps[i].m_TexelData[0])[j]), sizeof(unsigned char), 1, fp);
			//fwrite(&alpha, sizeof(unsigned char), 1, fp);
		}

		fwrite(m_lightmaps[i].m_TexelData[0], size, 1, fp);

		fclose(fp);
	}

	file.Save( path );

	// TODO: Write out .usg file

	return true;
}

bool PCViewerConverter::SaveCASFlags( char* path )
{
	// PC Viewer doesn't support any CAS data
	return true;
}
