#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include <math.h>
#include <assert.h>
#include <Strip/Strip.h>
#include <XBox/XBoxConv.h>
#include <List/Search.h>
#include <windows.h>
#include <nvDXT/DXTLib.h>

#include "Utility.h"
#include "VirtualFile.h"

#include <MiniBall/MiniBall.h>

bool	gTextureOverride = false;
int		gTextureOverrideMapping[Utils::vNUM_PLATFORMS] = { 0, 1, 2 };

class XBoxConverter : public IXBoxConverter
{
public:
	bool	SaveCollisionData( char* path );
	bool	ConvertData( void );
	bool	SaveScene( char* path );
	bool	SaveTextureDictionary( char* path, char* usg_path );
	bool	SaveCASFlags( char* path );
	bool	SaveWeightMap( char* path );
	int		GetPlatformFlags( void );
	unsigned __int64 GetBlendParameters( int blend_mode, int fixed );
	void	LimitMipMaps( void );

private:
	void	OptimizeVertexLists( void );
	void	ApplyMaterialColors( void );
	void	CreateTriStrips( void );
	void	HandleGrassGeneration( NxObject *p_object );
	int		GetDXTForTexture( NxTexture *p_texture );

	bool	SaveMaterials( IoUtils::CVirtualOutputFile* pOutputFile );
	bool	SaveGeometry( IoUtils::CVirtualOutputFile* pOutputFile );	
};
	
static XBoxConverter*	sp_xb_conv = NULL;

unsigned char*	gpCompressedData		= NULL;
int				gDXTFormat;
int				gDXTBytesWritten;

void WriteDTXnFile( DWORD count, void *buffer )
{
	// It seems this gets called three times per texture.
	// First with a 4-byte block - ID sanity check perhaps?
	// Second, with a 124-byte block - header information?
	// Third, with a block dependent on the size of the texture - the actual DXT compressed texture information.
	if(( count != 4 ) && ( count != 124 ))
	{
		memcpy( gpCompressedData, buffer, count );
		gpCompressedData	+= count;
		gDXTBytesWritten	+= count;
	}
}



void ReadDTXnFile(DWORD count, void * buffer)
{
}

IXBoxConverter* GetXBoxConverter( void )
{
	if ( sp_xb_conv == NULL )
	{
		sp_xb_conv = new XBoxConverter;
	}

	return sp_xb_conv;
}

void ResetXBoxConverter()
{
	if ( sp_xb_conv )
	{
		delete sp_xb_conv;
		sp_xb_conv = NULL;
	}
}



void XBoxConverter::OptimizeVertexLists( void )
{
	for( int i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* p_object = &m_scene->m_Objects[i];
		p_object->OptimizeVertList( m_scene );
	}
}



void XBoxConverter::ApplyMaterialColors( void )
{
//	int total = 0;		// Number of multipass materials.
//	int matching = 0;	// Number of multipass materials with matching colors for each pass.
	
	// We are now doing material color modulation in the pixel shader at runtime.
	return;
	
	for( int i = 0; i < m_scene->m_NumObjects; i++ )
	{
		NxObject* object;		

		object = &m_scene->m_Objects[i];
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		for( int j = 0; j < object->m_NumFaces; j++ )
		{
			NxMaterial* material = m_scene->GetMaterial( object->m_Faces[j].m_MatChecksum );
			if( material )
			{
				// Modulate vertex color with the first-pass material color.
				// (Previously was modulating with the material color from all passes).
				float r = 1.0f;
				float g = 1.0f;
				float b = 1.0f;
//				for( int p = 0; p < material->m_NumPasses; ++p )
//				if( material->m_NumPasses > 1 )
//				{
//					++total;
//					++matching;
//					bool match = true;
//					for( int p = 1; p < material->m_NumPasses; ++p )
//					{
//						if( material->m_Passes[p].m_HasColor &&
//							(( material->m_Passes[p].m_Color[0] != material->m_Passes[0].m_Color[0] ) ||
//							( material->m_Passes[p].m_Color[1] != material->m_Passes[0].m_Color[1] ) ||
//							( material->m_Passes[p].m_Color[2] != material->m_Passes[0].m_Color[2] )))
//						{
//							--matching;
//							break;
//						}
//					}
//				}
				
				for( int p = 0; p < 1; ++p )
				{
					r *= material->m_Passes[p].m_Color[0];
					g *= material->m_Passes[p].m_Color[1];
					b *= material->m_Passes[p].m_Color[2];
				}
				
				for( int k = 0; k < 3; k++ )
				{
					object->m_Verts[object->m_Faces[j].m_Vertex[k]].m_Color[0] *= r;
					object->m_Verts[object->m_Faces[j].m_Vertex[k]].m_Color[1] *= g;
					object->m_Verts[object->m_Faces[j].m_Vertex[k]].m_Color[2] *= b;
				}
			}
		}
	}
//	printf( "%d materials of %d total match multipass colors\n", matching, total );
}



void XBoxConverter::CreateTriStrips( void )
{
	int				num_objects	= m_scene->m_NumObjects;
	INxStripper*	stripper	= GetNxStripper();
	
	for( int i = 0; i < num_objects; i++ )
	{
		if( m_scene->m_Objects[i].m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		stripper->Reset();
		stripper->SetObject( &m_scene->m_Objects[i] );
		stripper->SetPlatform( Utils::vPLATFORM_XBOX );
		if( stripper->StripifyForXbox())
		{
			m_scene->m_Objects[i].m_MeshList = *stripper->GetMeshList();
		}
		else
		{
			exit( 666 );
		}
	}
}




bool XBoxConverter::SaveMaterials( IoUtils::CVirtualOutputFile* pOutputFile )
{
	Lst::Search< NxMaterial > sh;
	NxMaterial* material;

	int num_materials = m_scene->m_Materials.CountItems();
	pOutputFile->Write((const char*) &num_materials, sizeof( int ));
	for( material = sh.FirstItem( m_scene->m_Materials ); material;	material = sh.NextItem())
	{
		int					num_sequences, mmag, mmin;
		unsigned __int64	equation;

		// Write out material checksum.
		pOutputFile->Write((const char*)&material->m_Checksum, sizeof( unsigned long ));

		// Write out material name.
		pOutputFile->Write((const char*)&material->m_materialName, sizeof( unsigned long ));

		// Write out how many passes this material contains.
		pOutputFile->Write((const char*)&material->m_NumPasses, sizeof( unsigned long ));
		
		// Write out the alpha cutoff value.
		pOutputFile->Write((const char*)&material->m_AlphaCutoff, sizeof( int ));

		// Write out the sorted flag.
		pOutputFile->Write((const char*)&material->m_Sorted, sizeof( bool ));
		
		// Write out the draw order.
		pOutputFile->Write((const char*)&material->m_DrawOrder, sizeof( float ));
		
		// Write out the one-sided flag (largely ignored, but considered in certain special cases).
		pOutputFile->Write((const char*)&material->m_OneSided, sizeof( bool ));

		// Write out the backface cull flag.
		pOutputFile->Write((const char*)&material->m_TwoSided, sizeof( bool ));

		// Write out the z-bias flag.
		pOutputFile->Write((const char*)&material->m_BasePass, sizeof( int ));

		// Write out the grassify information.
		pOutputFile->Write((const char*)&material->m_grassify, sizeof( bool ));
		if( material->m_grassify )
		{
			pOutputFile->Write((const char*)&material->m_grassHeight, sizeof( float ));
			pOutputFile->Write((const char*)&material->m_grassLayers, sizeof( int ));
		}

		// Write out the specular lighting information.
		pOutputFile->Write((const char*)&material->m_SpecularPower, sizeof( float ));
		if( material->m_SpecularPower > 0.0f )
		{
			pOutputFile->Write((const char*)&material->m_SpecularColor[0], sizeof( float ) * 3 );
		}
		
		// Write out the texture checksum and flags of each pass (and collate texture flags at the same time).		
		for( int p = 0; p < material->m_NumPasses; ++p )
		{
			unsigned long tex_checksum;

			tex_checksum = material->m_Passes[p].GetTextureChecksum( Utils::vPLATFORM_XBOX );
			pOutputFile->Write((const char*) &tex_checksum, sizeof( unsigned long ));
		
			int flags = 0;
			if( tex_checksum != 0 )
			{
				flags |= NxMaterial::mTEXTURED;
			}
			if( material->m_Passes[p].m_MappingMode == vMAPPING_ENVIRONMENT )
			{
				flags |= NxMaterial::mENVIRONMENT;
			}
			if( material->m_Passes[p].m_UVWibbleEnabled )
			{
				flags |= NxMaterial::mUV_WIBBLE;
			}
			if( material->m_GroupFlags & NxMaterial::mGROUP_TRANSPARENT )
			{
				flags |= NxMaterial::mTRANSPARENT;
			}
			if(( p == 0 ) && ( material->m_WibbleSequences.CountItems() > 0 ))
			{
				flags |= NxMaterial::mVC_WIBBLE;
			}
			if( material->m_Passes[p].m_Flags & NxMaterialPass::COLOR_LOCKED )
			{
				// Hmm, really should be using a per-pass flag here.
				flags |= 0x80;
			}
			if( material->m_Passes[p].m_Flags & NxMaterialPass::ANIMATED_TEXTURE )
			{
				// Hmm, really should be using a per-pass flag here.
				flags |= NxMaterial::mANIMATED_TEX;
			}
			if( material->m_Passes[p].m_Flags & NxMaterialPass::IGNORE_VERTEX_ALPHA )
			{
				// Hmm, really should be using a per-pass flag here.
				flags |= 1 << 12;
			}
			if(( p == 0 ) && material->m_water )
			{
				// Water effect test.
				flags |= 1 << 27;
			}

			// For now, everything is smooth.
			flags |= NxMaterial::mSMOOTH;
			pOutputFile->Write((const char*)&flags, sizeof( int ));

			// Write out the material color flag.
			pOutputFile->Write((const char*)&material->m_Passes[p].m_HasColor, sizeof( bool ));
			
			// We now write out the material color for every stage.
			pOutputFile->Write((const char*)&material->m_Passes[p].m_Color[0], sizeof( float ));
			pOutputFile->Write((const char*)&material->m_Passes[p].m_Color[1], sizeof( float ));
			pOutputFile->Write((const char*)&material->m_Passes[p].m_Color[2], sizeof( float ));
			
			// Write out Xbox alpha-blending params.
			equation = GetBlendParameters( material->m_Passes[p].m_BlendMode, material->m_Passes[p].m_FixedAlpha );
			pOutputFile->Write((const char*)&equation, sizeof( unsigned __int64 ));
			
			// Write out UV addressing modes.
			pOutputFile->Write((const char*)&material->m_Passes[p].m_AddressModeU, sizeof( int ));
			pOutputFile->Write((const char*)&material->m_Passes[p].m_AddressModeV, sizeof( int ));
			
			// Write out environment mapping tile multiples.
			pOutputFile->Write((const char*)&material->m_Passes[p].m_EnvTileU, sizeof( float ));
			pOutputFile->Write((const char*)&material->m_Passes[p].m_EnvTileV, sizeof( float ));

			// Write out minification and magnification filtering.
			unsigned int filtering = ((unsigned int)material->m_Passes[p].m_MinFilteringMode ) | (((unsigned int)material->m_Passes[p].m_MagFilteringMode ) << 16 );
			pOutputFile->Write((const char*)&filtering, sizeof( unsigned int ));
		
			// Write out UV wibble params
			if( flags & NxMaterial::mUV_WIBBLE )
			{
				pOutputFile->Write((const char*)&material->m_Passes[p].m_UVel, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_VVel, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_UFrequency, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_VFrequency, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_UAmplitude, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_VAmplitude, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_UPhase, sizeof( float ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_VPhase, sizeof( float ));
			}
		
			// Write out VC wibble sequences
			if(( p == 0 ) && ( flags & NxMaterial::mVC_WIBBLE ))
			{
				num_sequences = material->m_WibbleSequences.CountItems();
				pOutputFile->Write((const char*) &num_sequences, sizeof( int ));

				if( num_sequences > 0 )
				{
					Lst::Search< VCWibbleSequence > sh;
					VCWibbleSequence *p_sequence;

					for( p_sequence = sh.FirstItem( material->m_WibbleSequences ); p_sequence; p_sequence = sh.NextItem())
					{
						pOutputFile->Write((const char*)&p_sequence->m_NumFrames, sizeof( int ));
						pOutputFile->Write((const char*)&p_sequence->m_Phase, sizeof( int ));
						
						for( int k = 0; k < p_sequence->m_NumFrames; ++k )
						{
							int time = p_sequence->m_WibbleFrames[k].m_Time;
							pOutputFile->Write((const char*)&time, sizeof( int ));

							unsigned int color;

							// The wibble colors appear to be in the range [0.0, 255.0] for each component, we want to normalise
							// down as per Ps2 where 128.0 represents the unmodulated intensity.
							color	= (unsigned int)( p_sequence->m_WibbleFrames[k].m_Color[0] * ( 128.0f / 255.0f ));
							color	|= ((unsigned int)( p_sequence->m_WibbleFrames[k].m_Color[1] * ( 128.0f / 255.0f ) )) << 8;
							color	|= ((unsigned int)( p_sequence->m_WibbleFrames[k].m_Color[2] * ( 128.0f / 255.0f ) )) << 16;
							color	|= ((unsigned int)( p_sequence->m_WibbleFrames[k].m_Color[3] * ( 128.0f / 255.0f ) )) << 24;
							
							pOutputFile->Write((const char*)&color, sizeof( unsigned int ));
						}
					}
				}
			}

			// Write out texture wibble sequences if present.
			if( material->m_Passes[p].m_Flags & NxMaterialPass::ANIMATED_TEXTURE )
			{
				pOutputFile->Write((const char*)&material->m_Passes[p].m_AnimatedTexture.m_NumKeyframes, sizeof( int ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_AnimatedTexture.m_Period, sizeof( int ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_AnimatedTexture.m_Iterations, sizeof( int ));
				pOutputFile->Write((const char*)&material->m_Passes[p].m_AnimatedTexture.m_Phase, sizeof( int ));
				for( int ati = 0; ati < material->m_Passes[p].m_AnimatedTexture.m_NumKeyframes; ++ati )
				{
					NxAnimatedTextureKeyframe* p_keyframe = material->m_Passes[p].m_AnimatedTexture.m_Keyframes + ati;
					pOutputFile->Write((const char*)&p_keyframe->m_Time, sizeof( unsigned int ));
					pOutputFile->Write((const char*)&p_keyframe->m_TexChecksum[Utils::vPLATFORM_XBOX], sizeof( unsigned long ));
				}
			}
		
			// Write out MMAG and MMIN		
			mmag = (int) material->m_Passes[p].m_MagFilteringMode;
			mmin = (int) material->m_Passes[p].m_MinFilteringMode;
			pOutputFile->Write((const char*) &mmag, sizeof( int ));
			pOutputFile->Write((const char*) &mmin, sizeof( int ));

			// Write out K and L		
			pOutputFile->Write((const char*) &material->m_Passes[p].m_MipMapK, sizeof( float ));
			pOutputFile->Write((const char*) &material->m_Passes[p].m_MipMapK, sizeof( float ));
//			pOutputFile->Write((const char*) &material->m_Passes[p].m_MipMapL, sizeof( float ));
		}
	}
	return true;
}




bool XBoxConverter::SaveGeometry( IoUtils::CVirtualOutputFile* pOutputFile )
{
	int i;
	int	num_objects	= m_scene->m_NumObjects;

	// Determine how many of these objects are invisible, and adjust the object count accordingly.
	int num_invisible_objects = 0;
	for( i = 0; i < num_objects; i++ )
	{
		NxObject*	object		= m_scene->m_Objects + i;

		// If flagged as a shadow volume, make sure not invisible.
		if( object->m_Flags & NxObject::mSHADOWVOLUME )
		{
			object->m_Flags &= ~NxObject::mINVISIBLE;
		}
		
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			++num_invisible_objects;
		}
	}

	int num_visible_objects = num_objects - num_invisible_objects;

	pOutputFile->Write((const char*)&num_visible_objects, sizeof( int ));
	
	for( i = 0; i < num_objects; i++ )
	{
		NxObject*	object		= m_scene->m_Objects + i;
		NxVertex*	vert_list	= object->m_Verts;
		NxFace*		face_list	= object->m_Faces;
		
		// Skip invisible objects.
		if( object->m_Flags & NxObject::mINVISIBLE )
		{
			continue;
		}
		
		// Check this checksum doesn't already clash.
		for( int check = 0; check < i; ++check )
		{
			NxObject* test_object = m_scene->m_Objects + check;
			if( test_object->m_Checksum == object->m_Checksum )
			{
				// Just set the object checksum to be something random for now...
				object->m_Checksum = rand();
			}
		}

		// Write out the object checksum.
		pOutputFile->Write((const char*)&object->m_Checksum, sizeof( unsigned long ));

		// Write out hierarchy bone index.
		int boneidx = -1;
		int num_hierarchy_objects = m_scene->m_Model.nBones;
		for( uint8 bone_idx = 0; bone_idx < num_hierarchy_objects; bone_idx++ )
		{
			uint32 cur_checksum = m_scene->m_Model.bones[bone_idx].crc;
			if( cur_checksum == object->m_Checksum )
			{
				boneidx = bone_idx;
				break;
			}
		}
		pOutputFile->Write((const char*)&boneidx, sizeof( int ));
		
		// Write out the flags which describe the object's properties
		pOutputFile->Write((const char*)&object->m_Flags, sizeof( int ));
		
		// How many meshes in this object.		
		int num_meshes = object->m_MeshList.m_NumMeshes[0];
		pOutputFile->Write((const char*)&num_meshes, sizeof( int ));

		// Write out bounding box data.
		pOutputFile->Write((const char*)&object->m_BoundingBox.m_Min, 3 * sizeof( float ));
		pOutputFile->Write((const char*)&object->m_BoundingBox.m_Max, 3 * sizeof( float ));

		// ...and bounding sphere data.
		float center[3], diag[3];
		float diag_length = 0.0f;
		for( int a = 0; a < 3; ++a )
		{
			center[a] = ( object->m_BoundingBox.m_Max[a] + object->m_BoundingBox.m_Min[a] ) / 2;
			diag[a] = ( object->m_BoundingBox.m_Max[a] - object->m_BoundingBox.m_Min[a] );
			diag_length += ( diag[a] * diag[a] );
		}
		diag_length = sqrtf( diag_length );
		float radius = diag_length / 2.0f;
		pOutputFile->Write((const char*)&center, 3 * sizeof( float ));
		pOutputFile->Write((const char*)&radius, sizeof( float ));
		
		// Write out billboard data if present.
		if( object->m_Flags & NxObject::mBILLBOARD )
		{
			pOutputFile->Write((const char*)&object->m_BillboardType,	sizeof( NxObject::BillboardType ));
			pOutputFile->Write((const char*)&object->m_BillboardOrigin,	sizeof( float ) * 3 );
			pOutputFile->Write((const char*)&object->m_PivotPos,		sizeof( float ) * 3 );
			pOutputFile->Write((const char*)&object->m_PivotAxis,		sizeof( float ) * 3 );
		}
		
		// How many vertices in this object.
		int num_verts = object->m_NumVerts;
		pOutputFile->Write((const char*)&num_verts, sizeof( int ));

		// Figure vertex data stride.
		int stride	= sizeof( float ) * 3;	// Position.
		stride		+= ( object->m_Flags & NxObject::mTEXTURED ) ? ( sizeof( float ) * 2 * vMAX_MATERIAL_PASSES ) : 0;	// UVs.
		stride		+= ( object->m_Flags & NxObject::mCOLORED ) ? ( sizeof( char ) * 4 ) : 0;
		stride		+= ( object->m_Flags & NxObject::mNORMALS ) ? ( sizeof( float ) * 3 ) : 0;
		pOutputFile->Write((const char*)&stride, sizeof( int ));
		
		// Write out the vertex data. Positions first (always present).
		for( int v = 0; v < num_verts; ++v )
		{
			NxVertex* vert = object->m_Verts + v;
			pOutputFile->Write((const char*) &vert->m_Pos[0], sizeof( float ));
			pOutputFile->Write((const char*) &vert->m_Pos[1], sizeof( float ));
			pOutputFile->Write((const char*) &vert->m_Pos[2], sizeof( float ));
		}

		// Normals if present.
		if( object->m_Flags & NxObject::mNORMALS )
		{					
			for( int v = 0; v < num_verts; ++v )
			{
				NxVertex* vert = object->m_Verts + v;
				pOutputFile->Write((const char*) &vert->m_Normal[0], sizeof( float ));
				pOutputFile->Write((const char*) &vert->m_Normal[1], sizeof( float ));
				pOutputFile->Write((const char*) &vert->m_Normal[2], sizeof( float ));
			}
		}
			
		// Vertex weights if present.
		if( object->m_Flags & NxObject::mSKINNED )
		{
			for( int v = 0; v < num_verts; ++v )
			{
				NxVertex* vert = object->m_Verts + v;

				// Sort the weights, then use only the first 3.
				vert->SortVertexWeights();
				vert->NormalizeVertexWeights( 3 );

				// Ensure weight 4 is zeroed out.
				vert->m_Weight[3] = 0.0f;
				
				// Calculate the packed vertex weight format corresponding to D3DVSDT_NORMPACKED3.
				unsigned int packed_weight = ((((unsigned int)( vert->m_Weight[2] *  511.0f )) & 0x3ff ) << 22L ) |
											 ((((unsigned int)( vert->m_Weight[1] * 1023.0f )) & 0x7ff ) << 11L ) |
											 ((((unsigned int)( vert->m_Weight[0] * 1023.0f )) & 0x7ff ) <<  0L );

				pOutputFile->Write((const char*)&packed_weight, sizeof( unsigned int ));

				// XXXXXXXXXXXXXXX
				if ( vert->m_MeshScalingWeight[0] != 0.0f )
				{
					Utils::Assert( mp_weight_map_vertices != NULL, "weight map verts were uninitialized" );
//					Utils::Assert( m_num_weight_map_vertices == skinned_vertex_index, "weight map index mismatch" );

 					NxVertex* pVertex = &mp_weight_map_vertices[m_num_weight_map_vertices];
					*pVertex = *vert;
					m_num_weight_map_vertices++;
				}
			}
		}

		// Vertex bone indices if present.
		if( object->m_Flags & NxObject::mSKINNED )
		{
			for( int v = 0; v < num_verts; ++v )
			{
				NxVertex* vert = object->m_Verts + v;

				unsigned short indices[4] = { vert->m_WeightedIndex[0], vert->m_WeightedIndex[1], vert->m_WeightedIndex[2], vert->m_WeightedIndex[3] };
				pOutputFile->Write((const char*)&indices[0], sizeof( unsigned short ));
				pOutputFile->Write((const char*)&indices[1], sizeof( unsigned short ));
				pOutputFile->Write((const char*)&indices[2], sizeof( unsigned short ));
				pOutputFile->Write((const char*)&indices[3], sizeof( unsigned short ));
			}
		}
		
		// Texture coordinate sets if present.
		if( object->m_Flags & NxObject::mTEXTURED )
		{
			// Figure out what the maximum number of passes on any of the meshes is for this object.
			int max_passes = 0;
			for( unsigned int m = 0; m < (unsigned int)num_meshes; ++m )
			{
				NxMesh		*p_mesh		= object->m_MeshList.m_Meshes[0][m];
				NxMaterial	*p_material	= m_scene->GetMaterial( p_mesh->m_MatChecksum );
				if( p_material->m_NumPasses > max_passes )
				{
					max_passes = p_material->m_NumPasses;
				}
			}
			
			// Write out how many sets of texture coodinates present.
			int num_tc_sets = max_passes;
			pOutputFile->Write((const char*)&num_tc_sets, sizeof( int ));
			
			if( num_tc_sets > 0 )
			{
				for( int v = 0; v < num_verts; ++v )
				{
					NxVertex* vert = object->m_Verts + v;
					for( int tc = 0; tc < num_tc_sets; ++tc )
					{
						pOutputFile->Write((const char*) &vert->m_TexCoord[tc][0], sizeof( float ));
						pOutputFile->Write((const char*) &vert->m_TexCoord[tc][1], sizeof( float ));
					}
				}
			}
		}

		// Vertex colors if present.
		if( object->m_Flags & NxObject::mCOLORED )
		{
			for( int v = 0; v < num_verts; ++v )
			{
				NxVertex* vert = object->m_Verts + v;
				unsigned int red	= (unsigned int)(( vert->m_Color[0] * 128.0f ) + 0.5f );
				unsigned int green	= (unsigned int)(( vert->m_Color[1] * 128.0f ) + 0.5f );
				unsigned int blue	= (unsigned int)(( vert->m_Color[2] * 128.0f ) + 0.5f );

				// The alpha values are now supplied in the range [0.0, 1.0] with 1.0 meaning fully opaque on all three platforms.
				// However, since we are doing the x2 multiply on all color components in pixel shaders, we actually need to
				// halve the alpha value here.
				unsigned int alpha	= (unsigned int)(( vert->m_Color[3] * 128.0f ) + 0.5f );
//				unsigned int alpha	= (unsigned int)(( vert->m_Color[3] * 255.0f ) + 0.5f );

				// Write out in BGRA format.
				unsigned int bgra = blue | ( green << 8 ) | ( red << 16 ) | ( alpha << 24 );
				pOutputFile->Write((const char*)&bgra, sizeof( unsigned int ));
			}
		}				
	
		// Vertex color wibble animation indices if present.
		if( object->m_Flags & NxObject::mVCWIBBLE )
		{
			for( int v = 0; v < num_verts; ++v )
			{
				NxVertex* vert = object->m_Verts + v;
				
				unsigned char index = (unsigned char)vert->m_WibbleIndex;
				pOutputFile->Write((const char*)&index, sizeof( unsigned char ));
			}
		}				

		for( unsigned int m = 0; m < (unsigned int)object->m_MeshList.m_NumMeshes[0]; ++m )
		{
			NxMesh*	mesh = object->m_MeshList.m_Meshes[0][m];
			
			float bbmin[3];
			float bbmax[3];

			bbmin[0] = bbmax[0] = object->m_Verts[mesh->m_VertStrip[0].m_Index].m_Pos[0];
			bbmin[1] = bbmax[1] = object->m_Verts[mesh->m_VertStrip[0].m_Index].m_Pos[1];
			bbmin[2] = bbmax[2] = object->m_Verts[mesh->m_VertStrip[0].m_Index].m_Pos[2];

			// Dealing with three dimensional space (!)
			const int d = 3;
			Miniball<d>			mb;
			Miniball<d>::Point	mbp;

			// Calculate bounding box and sphere for this mesh, based on highest LOD level.
			for( int ilp = 0; ilp < mesh->m_NumStripVerts; ++ilp )
			{
				NxVertex *p_vert = object->m_Verts + mesh->m_VertStrip[ilp].m_Index;

				float px = p_vert->m_Pos[0];
				float py = p_vert->m_Pos[1];
				float pz = p_vert->m_Pos[2];

				if( px < bbmin[0] )
					bbmin[0] = px;
				if( py < bbmin[1] )
					bbmin[1] = py;
				if( pz < bbmin[2] )
					bbmin[2] = pz;

				if( px > bbmax[0] )
					bbmax[0] = px;
				if( py > bbmax[1] )
					bbmax[1] = py;
				if( pz > bbmax[2] )
					bbmax[2] = pz;

				// Add point to miniball.
				mbp[0] = px;
				mbp[1] = py;
				mbp[2] = pz;
				mb.check_in( mbp );
			}

			// Build the bounding sphere.
			mb.build( true );
			mbp = mb.center();
			float rad = (float)( sqrt( mb.squared_radius()));
			float cen[3] = { (float)mbp[0], (float)mbp[1], (float)mbp[2] };

			// Sanity check the sphere; it seems in rare cases the MiniBall stuff is erroneous.
			for( int ilp = 0; ilp < mesh->m_NumStripVerts; ++ilp )
			{
				NxVertex *p_vert = object->m_Verts + mesh->m_VertStrip[ilp].m_Index;

				float px = p_vert->m_Pos[0] - cen[0];
				float py = p_vert->m_Pos[1] - cen[1];
				float pz = p_vert->m_Pos[2] - cen[2];

				float dist_from_sphere_center = (float)sqrt(( px * px ) + ( py * py ) + ( pz * pz ));
				if(( dist_from_sphere_center > rad ) && fabs( dist_from_sphere_center - rad ) > 0.5f )
				{
					// Do the standard bounding sphere by using the bounding box.
					float diag0 = ( bbmax[0] - bbmin[0] ) * 0.5f;
					float diag1 = ( bbmax[1] - bbmin[1] ) * 0.5f;
					float diag2 = ( bbmax[2] - bbmin[2] ) * 0.5f;

					cen[0]		= bbmin[0] + diag0;
					cen[1]		= bbmin[1] + diag1;
					cen[2]		= bbmin[2] + diag2;

					rad			= sqrtf(( diag0 * diag0 ) + ( diag1 * diag1 ) + ( diag2 * diag2 ));
				}
			}

			// Write the bounding sphere and bounding box.
			pOutputFile->Write((const char*)&cen[0], sizeof( float ) * 3 );
			pOutputFile->Write((const char*)&rad, sizeof( float ));
			pOutputFile->Write((const char*)&bbmin[0], sizeof( float ) * 3 );
			pOutputFile->Write((const char*)&bbmax[0], sizeof( float ) * 3 );

			// An array of mesh pointers to meshes of the same material, but different LOD levels.
			NxMesh *mesh_array[8] = { NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL };
			mesh_array[0] = mesh;
			
			// We need to figure out if there are lower LOD level meshes that correspond to this mesh (i.e. have the same material).
			unsigned int lod_level;
			for( lod_level = 1; lod_level < 8; ++lod_level )
			{
				if( object->m_MeshList.m_NumMeshes[lod_level] == 0 )
					break;

				for( int mitll = 0; mitll < object->m_MeshList.m_NumMeshes[lod_level]; ++mitll )
				{
					if( object->m_MeshList.m_Meshes[lod_level][mitll]->m_MatChecksum == mesh->m_MatChecksum )
					{
						// This level contains a corresponding mesh.
						mesh_array[lod_level] = object->m_MeshList.m_Meshes[lod_level][mitll];
						break;
					}
				}
				if( mesh_array[lod_level] == NULL )
				{
					break;
				}
			}

			// The variable lod_level will now contain the number of usable entries in mesh_array[].
			mesh->m_Flags |= mesh->m_ShadowFlags;			
			mesh->SetWibbleFlags();
			
			NxMaterial *p_material = m_scene->GetMaterial( mesh->m_MatChecksum );
			if( p_material && ( p_material->m_Passes[0].m_Flags & NxMaterialPass::UNLIT ))
			{
				mesh->m_Flags |= NxObject::mUNLIT;
			}

			// Write out the flags which describe the object's properties
			pOutputFile->Write((const char*)&mesh->m_Flags, sizeof( int ));

			// Checksum of the material for this mesh.
			unsigned long mat_checksum = mesh->m_MatChecksum;
			pOutputFile->Write((const char*)&mat_checksum, sizeof( unsigned long ));

			// How many total levels of LOD indices?
			pOutputFile->Write((const char*)&lod_level, sizeof( unsigned int ));

			for( unsigned int write_lod_level = 0; write_lod_level < lod_level; ++write_lod_level )
			{
				// How many vertex indices in this strip.
				int num_strip_verts = mesh_array[write_lod_level]->m_NumStripVerts;
				pOutputFile->Write((const char*)&num_strip_verts, sizeof( int ));
			
				// Write out the vertex indices.
				for( int v = 0; v < num_strip_verts; ++v )
				{
					unsigned short idx = (unsigned short)mesh_array[write_lod_level]->m_VertStrip[v].m_Index;
					pOutputFile->Write((const char*)&idx, sizeof( unsigned short ));
				}
			}
			
			// Set the num_verts value to be the number of verts in the base mesh vert strip.
			num_verts = mesh->m_NumStripVerts;
			
			// Build the CAS flags for this mesh.
			if(( object->m_Flags & NxObject::mSKINNED ) && ( num_verts >= 3 ))
			{
				// First of all, duplicate code from the gameside, that normalises the indices down so that the fewest
				// number of verts per object are required - see mesh.cpp. The only reason we need to do this here is so
				// that the indices exported in the CAS data will match up with those calculated at runtime.
				short mesh_workspace_array[8192];
				memset( mesh_workspace_array, 1, sizeof( short ) * 8192 );

				// Figure the min and max indices for this mesh.
				int				ilp;
				unsigned short	min_index = mesh->m_VertStrip[0].m_Index;
				unsigned short	max_index = mesh->m_VertStrip[0].m_Index;
				for( ilp = 0; ilp < num_verts; ++ilp )
				{
					assert( mesh->m_VertStrip[ilp].m_Index < 8192 );
				
					mesh_workspace_array[mesh->m_VertStrip[ilp].m_Index] = 0;
				
					if( mesh->m_VertStrip[ilp].m_Index > max_index )
					{
						max_index = mesh->m_VertStrip[ilp].m_Index;
					}
					else if( mesh->m_VertStrip[ilp].m_Index < min_index )
					{
						min_index = mesh->m_VertStrip[ilp].m_Index;
					}
				}
	
				// Now figure the wasted space.
				int wasted_verts = 0;
				for( ilp = min_index; ilp <= max_index; ++ilp )
				{
					if( mesh_workspace_array[ilp] != 0 )
						++wasted_verts;
				}

				// Process the workspace array.
				int offset = 0;
				for( ilp = 0; ilp <= max_index; ++ilp )
				{
					if( mesh_workspace_array[ilp] == 0 )
					{
						// This vertex is used.
						mesh_workspace_array[ilp] = offset;
					}
					else
					{
						// This vertex is not used. Update the offset for the next used vertex.
						--offset;
					}
				}
	
				// Copy in index data, normalising the indices for this vertex buffer (i.e. so the lowest index will reference
				// vertex 0 in the buffer built specifically for this mesh).
//				for( int i = 0; i < num_indices; ++i )
//				{
//					mp_index_buffer[i] = p_indices[i] + mesh_workspace_array[p_indices[i]];
//				}
				
				unsigned int index0 = mesh->m_VertStrip[0].m_Index;
				unsigned int index1 = mesh->m_VertStrip[1].m_Index;
				for( unsigned int v = 2; v < (unsigned int)num_verts; ++v )
				{
					unsigned int index2 = mesh->m_VertStrip[v].m_Index;
					
					// Only bother checking if this is a non-degenerate tri.
					if(( index0 != index1 ) && ( index0 != index2 ) && ( index1 != index2 ))
					{
						unsigned long cas_face_flags = mesh->FindCASFaceFlags( index0, index1, index2 );
						if( cas_face_flags > 0 )
						{
							// Calculate normalised index values.
							unsigned int ni0 = index0 + mesh_workspace_array[index0];
							unsigned int ni1 = index1 + mesh_workspace_array[index1];
							unsigned int ni2 = index2 + mesh_workspace_array[index2];
							
							// Store the mesh number and the three indices.
//							if( !object->AddCASData( cas_face_flags, ( m << 16 ) | index0, ( index1 << 16 ) | index2 ))
							if( !object->AddCASData( cas_face_flags, ( m << 16 ) | ni0, ( ni1 << 16 ) | ni2 ))
							{
								printf( "ERROR: Increase number of cas data from %d!\n", NxObject::vMAXCASDATA );
								return false;
							}
						}
					}
					index0 = index1;
					index1 = index2;
				}
			}
		}
	}
	
	// Write out hierarchy data, if any.
	int num_hierarchy_objects = m_scene->m_Model.nBones;
	assert( num_hierarchy_objects < 256 );
	pOutputFile->Write((const char*)&num_hierarchy_objects, sizeof( int ));

	for( uint8 bone_idx = 0; bone_idx < num_hierarchy_objects; ++bone_idx )
	{
		uint32 cur_checksum = m_scene->m_Model.bones[bone_idx].crc;
		uint32 parent_checksum;
		NxObject *p_cur_object = m_scene->FindObject(cur_checksum);

		int16 parent_index = -1;

		assert( p_cur_object );
		parent_checksum = p_cur_object->m_ParentCRC;
		if( parent_checksum )
		{
			// Find parent index.
			for( int i = 0; i < num_hierarchy_objects; ++i )
			{
				if( parent_checksum == m_scene->m_Model.bones[i].crc )
				{
					parent_index = i;
					break;
				}
			}
			assert( parent_index >= 0 );
		}

		// Write the data
		pOutputFile->Write((const char*)&cur_checksum, sizeof( int ));
		pOutputFile->Write((const char*)&parent_checksum, sizeof( int ));
		pOutputFile->Write((const char*)&parent_index, sizeof( short ));
		pOutputFile->Write((const char*)&bone_idx, sizeof( char ));

		// Padding
		uint32 pad = 0;
		pOutputFile->Write((const char*)&pad, sizeof( char ));
		pOutputFile->Write((const char*)&pad, sizeof( int ));

		// Write the matrix
		pOutputFile->Write((const char*)&m_scene->m_Model.bones[bone_idx].mat, 16 * sizeof( float ));

#		if 0
		printf("Bone %d\n", bone_idx);
		printf("Checksum %x\n", cur_checksum);
		printf("Parent Checksum %x\n", parent_checksum);
		printf("Parent Index %x\n", parent_index);
		Mth::Matrix mat = m_scene->m_Model.bones[bone_idx].mat;
		printf("[ %f %f %f %f]\n", mat[0][0], mat[0][1], mat[0][2], mat[0][3]);
		printf("[ %f %f %f %f]\n", mat[1][0], mat[1][1], mat[1][2], mat[1][3]);
		printf("[ %f %f %f %f]\n", mat[2][0], mat[2][1], mat[2][2], mat[2][3]);
		printf("[ %f %f %f %f]\n", mat[3][0], mat[3][1], mat[3][2], mat[3][3]);
#		endif
	}
	
	return true;
}


bool XBoxConverter::ConvertData( void )
{
	NxObject *p_object = &m_scene->m_Objects[0];
	
	// These are done only for faces flagged as "render as separate geometry"
	SeparateMultiPassMeshes();
	ExpandMultiPassMaterials();
	
	// Bakes in the material colors to the vertex colors.
	ApplyMaterialColors();

	OptimizeVertexLists();
	
	// Strips the geometry.
	CreateTriStrips();

	// PS2 specific steps to generate VRAM swapping banks and associated geometry.
//	CreateTextureGroups();
//	CreateMeshGroups();
	
	return true;
}

#define OUTPUT_COLL_BSP

bool XBoxConverter::SaveCollisionData( char* path )
{
	bool success = false;

	IoUtils::CVirtualOutputFile theOutputFile;

	// 10-meg buffer
	theOutputFile.Init(10*1024*1024);

	int i, j, num_obj, version;

	const float one_f = 1.0f;
	const int zero = 0;
	const int one = 1;
	
	printf( "Collision Data...." );

#ifdef OUTPUT_COLL_BSP
	// Reserve space for BSP nodes
	NxCollBSPNode **p_node_array = new NxCollBSPNode *[m_scene->m_NumCollBSPTreeNodes];
	int node_array_index = 0;
	int node_array_offset = 0;
#endif // OUTPUT_COLL_BSP

	version = vXBOX_COLLISION_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	// Number of collision objects
	num_obj = m_scene->m_NumObjects;
	int non_collision_objects = 0;
//	for( i = 0; i < num_obj; i++ )
//	{
//		if (m_scene->m_Objects[i].m_NumCollVerts == 0)
//			non_collision_objects++;
//	}
	int num_collision_objects = num_obj - non_collision_objects;
	theOutputFile.Write((const char*) &num_collision_objects, sizeof( int ));

	// Calculate the size of the vert and face arrays
	int total_num_verts_large = 0, total_num_verts_small = 0;
	int total_num_faces_large = 0, total_num_faces_small = 0;
	float box_diff = 0.0f;
	for ( i = 0; i < num_obj; i++ )
	{
		NxObject* object;
		object = &m_scene->m_Objects[i];

		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		// Figure out if we can use 16-bit verts and write result out
		object->m_CollUseFixedVerts = GENERATE_FIXED_POINT_COLLISION;
		int fixed_diff;
		for (int axis = 0; axis < 3; axis++)
		{
			fixed_diff = (int) (((object->m_CollisionBoundingBox.m_Max[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION)); //+ 0.5f); Lets not round it here
			if (fixed_diff > 0xFFFF)
			{
				object->m_CollUseFixedVerts = 0;
				break;
			}

			for( j = 0; j < object->m_NumCollVerts; j++ )
			{
				// We don't know why this condition exists, but we need to make sure it doesn't happen here.
				if (object->mp_CollVerts[j].m_Pos[axis] < object->m_CollisionBoundingBox.m_Min[axis])
				{
					object->m_CollUseFixedVerts = 0;
					if (box_diff < (object->m_CollisionBoundingBox.m_Min[axis] - object->mp_CollVerts[j].m_Pos[axis]))
					{
						box_diff = (object->m_CollisionBoundingBox.m_Min[axis] - object->mp_CollVerts[j].m_Pos[axis]);
					}
					//printf("Collision object %x has verts outside of bounding box (diff %f)...\n", object->m_Checksum, object->mp_CollVerts[j].m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]);
					//break;
				}
			}
		}

		if (object->m_CollUseFixedVerts)
		{
			total_num_verts_small += m_scene->m_Objects[i].m_NumCollVerts;
#if !GENERATE_FIXED_POINT_COLLISION
			total_num_verts_small += m_scene->m_Objects[i].m_NumCollVerts % 2;		// Make sure we are 128-bit aligned
#endif
		}
		else
		{
			total_num_verts_large += m_scene->m_Objects[i].m_NumCollVerts;
		}

		if (m_scene->m_Objects[i].UseFaceSmall())
		{
			total_num_faces_small += m_scene->m_Objects[i].m_NumCollFaces;
		} else {
			total_num_faces_large += m_scene->m_Objects[i].m_NumCollFaces;
		}
	}

	// Output the size of the vert and face arrays
	int sum_verts = total_num_verts_large + total_num_verts_small;
	theOutputFile.Write((const char*) &sum_verts, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_faces_large, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_faces_small, sizeof( int ));

	// Padding
	theOutputFile.Write((const char*) &total_num_verts_large, sizeof( int ));
	theOutputFile.Write((const char*) &total_num_verts_small, sizeof( int ));
	theOutputFile.Write((const char*) &zero, sizeof( int ));

	int master_vert_idx = 0, master_face_idx = 0, master_face_offset = 0, master_vert_offset = 0;
	int master_large_vert_offset = 0, master_small_vert_offset = total_num_verts_large * NxObject::sVertexLargeSize;
	for( i = 0; i < num_obj; i++ )
	{
		NxObject* object;
		unsigned short num_verts, num_faces;

		object = &m_scene->m_Objects[i];
		
		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		// Write out the flags which describe the object's properties
		theOutputFile.Write((const char*) &object->m_Checksum, sizeof( unsigned long ));			
		
		// Padding for m_Flags
		theOutputFile.Write((const char*) &zero, sizeof( short ));
		
		num_verts = object->m_NumCollVerts;
		theOutputFile.Write((const char*) &num_verts, sizeof( unsigned short ));

		num_faces = object->m_NumCollFaces;
		theOutputFile.Write((const char*) &num_faces, sizeof( unsigned short ));
		
		// m_use_face_small
		if (object->UseFaceSmall())
		{
			theOutputFile.Write((const char*) &one, sizeof( char ));
		} else {
			theOutputFile.Write((const char*) &zero, sizeof( char ));
		}

		// m_use_fixed_verts
		theOutputFile.Write((const char*) &object->m_CollUseFixedVerts, sizeof( char ));

		// Using offset for pointer mp_faces
		//theOutputFile.Write((const char*) &master_face_idx, sizeof( int ));
		theOutputFile.Write((const char*) &master_face_offset, sizeof( int ));

		// Write out bounding box/sphere data
		theOutputFile.Write((const char*) &object->m_CollisionBoundingBox.m_Min, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &one_f, sizeof( float ));		// Vector pad
		theOutputFile.Write((const char*) &object->m_CollisionBoundingBox.m_Max, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &one_f, sizeof( float ));		// Vector pad

#if 0	// Not exporting bounding sphere now
		float center[3], diag[3];
		float radius, diag_length;

		diag_length = 0;
		for( j = 0; j < 3; j++ )
		{
			center[j] = ( object->m_BoundingBox.m_Max[j] + object->m_BoundingBox.m_Min[j] ) / 2;
			diag[j] = ( object->m_BoundingBox.m_Max[j] - object->m_BoundingBox.m_Min[j] );
			diag_length += ( diag[j] * diag[j] );
		}
		diag_length = sqrtf( diag_length );
		radius = diag_length / 2.0f;
		theOutputFile.Write((const char*) &center, 3 * sizeof( float ));
		theOutputFile.Write((const char*) &radius, sizeof( float ));
#endif

		// Using index for pointers mp_vert_pos and mp_vert_rgba
#if GENERATE_FIXED_POINT_COLLISION
		if (object->m_CollUseFixedVerts)
		{
			theOutputFile.Write((const char*) &master_small_vert_offset, sizeof( int ));
		}
		else
		{
			theOutputFile.Write((const char*) &master_large_vert_offset, sizeof( int ));
		}
#else
		theOutputFile.Write((const char*) &master_vert_offset, sizeof( int ));
		//theOutputFile.Write((const char*) &master_vert_idx, sizeof( int ));
#endif

#ifdef OUTPUT_COLL_BSP
		// Put BSP tree into array and write out first index
		int node_offset = NxCollBSPNode::AssignNodesToArray(object->mp_BSPRoot, p_node_array,
														   node_array_index, node_array_offset);
		assert(node_array_index <= m_scene->m_NumCollBSPTreeNodes);
		theOutputFile.Write((const char*) &node_offset, sizeof( int ));
#else
		// Padding for rest of class
		theOutputFile.Write((const char*) &zero, sizeof( int ));
#endif // OUTPUT_COLL_BSP
		// Intensity index (same as vert index)
		theOutputFile.Write((const char*) &master_vert_idx, sizeof( int ));
		theOutputFile.Write((const char*) &zero, sizeof( int ));

		master_vert_idx += num_verts;
		master_face_idx += num_faces;
		if (object->UseFaceSmall())
		{
			master_face_offset += (num_faces * NxObject::sFaceSmallSize);
		} else {
			master_face_offset += (num_faces * NxObject::sFaceLargeSize);
		}
		if (object->m_CollUseFixedVerts)
		{
			master_vert_offset += (num_verts * NxObject::sVertexSmallSize);
			master_small_vert_offset += (num_verts * NxObject::sVertexSmallSize);

#if !GENERATE_FIXED_POINT_COLLISION
			int align = master_vert_offset % 16;
			if (align)	// Align to 128 bits for now
			{
				master_vert_offset += (16 - align);
			}
#endif
		}
		else
		{
			master_vert_offset += (num_verts * NxObject::sVertexLargeSize);
			master_large_vert_offset += (num_verts * NxObject::sVertexLargeSize);
		}
	}

	// Export array of vert positions and colors
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;
			unsigned char red, green, blue, alpha, intensity;
			
			vert = &vert_list[j];

			// Calc color values
			red = (unsigned char) (( vert->m_Color[0] * 255.0f ) + 0.5f );
			green = (unsigned char) (( vert->m_Color[1] * 255.0f ) + 0.5f );
			blue = (unsigned char) (( vert->m_Color[2] * 255.0f ) + 0.5f );
			alpha = (unsigned char) (( vert->m_Color[3] * 255.0f ) + 0.5f );
			intensity = (unsigned char)( ( (int)red + (int)green + (int)blue ) / 3 );

			if (object->m_CollUseFixedVerts)
			{
#if !GENERATE_FIXED_POINT_COLLISION		// Output the small ones last for the new format
				for (int axis = 0; axis < 3; axis++)
				{
					float delta = ((vert->m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION); // + 0.5f; Let's not round here
					unsigned short fixed_pos = (unsigned short) delta;	// Convert to integer
					theOutputFile.Write((const char*) &fixed_pos, sizeof( short ));
				}
				// Write out the colors of the verts (potentially used to tint the skater)
				theOutputFile.Write((const char*) &zero, sizeof( char ));
				theOutputFile.Write((const char*) &intensity, sizeof( char ));
#endif
			}
			else
			{					
				// Write out positions				
				theOutputFile.Write((const char*) &vert->m_Pos[0], sizeof( float ));
				theOutputFile.Write((const char*) &vert->m_Pos[1], sizeof( float ));
				theOutputFile.Write((const char*) &vert->m_Pos[2], sizeof( float ));	

#if !GENERATE_FIXED_POINT_COLLISION
				// Write out the colors of the verts (potentially used to tint the skater)
				theOutputFile.Write((const char*) &zero, sizeof( char ));
				theOutputFile.Write((const char*) &intensity, sizeof( char ));
				theOutputFile.Write((const char*) &blue, sizeof( char ));
				theOutputFile.Write((const char*) &alpha, sizeof( char ));
#endif
			}
		}

#if !GENERATE_FIXED_POINT_COLLISION
		// Pad fixed verts if necessary.  Assumes fixed vert takes up 8 bytes.
		if (object->m_CollUseFixedVerts && (object->m_NumCollVerts % 2))
		{
			theOutputFile.Write((const char*) &zero, sizeof( int ));	// 4 bytes
			theOutputFile.Write((const char*) &zero, sizeof( int ));	// 4 bytes
		}
#endif
	}

#if GENERATE_FIXED_POINT_COLLISION
	// Export array of small vert positions
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;	
			
			vert = &vert_list[j];
			if (object->m_CollUseFixedVerts)
			{
				for (int axis = 0; axis < 3; axis++)
				{
					float delta = ((vert->m_Pos[axis] - object->m_CollisionBoundingBox.m_Min[axis]) * COLLISION_SUB_INCH_PRECISION); // + 0.5f; Let's not round here
					unsigned short fixed_pos = (unsigned short) delta;	// Convert to integer
					theOutputFile.Write((const char*) &fixed_pos, sizeof( short ));
				}
			}
		}
	}

	// Export array of vert colors
	for( i = 0; i < num_obj; i++ )
	{
		NxVertex* vert_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		vert_list = object->mp_CollVerts;

		for( j = 0; j < object->m_NumCollVerts; j++ )
		{
			NxVertex* vert;				
			unsigned char red, green, blue, alpha, intensity;
			
			vert = &vert_list[j];

			// Write out the colors of the verts (potentially used to tint the skater)
			red = (unsigned char) (( vert->m_Color[0] * 255.0f ) + 0.5f );
			green = (unsigned char) (( vert->m_Color[1] * 255.0f ) + 0.5f );
			blue = (unsigned char) (( vert->m_Color[2] * 255.0f ) + 0.5f );
			alpha = (unsigned char) (( vert->m_Color[3] * 255.0f ) + 0.5f );
			intensity = (unsigned char)( ( (int)red + (int)green + (int)blue ) / 3 );
			//theOutputFile.Write((const char*) &red, sizeof( char ));
			//theOutputFile.Write((const char*) &green, sizeof( char ));
			//theOutputFile.Write((const char*) &blue, sizeof( char ));
			//theOutputFile.Write((const char*) &alpha, sizeof( char ));
			theOutputFile.Write((const char*) &intensity, sizeof( char ));
		}	
	}

	// Pad to 32-bit boundary
	int remainder = (master_vert_offset + total_num_verts_large + total_num_verts_small) % 4;
	if (remainder) {
		for (i = 0; i < (4 - remainder); i++)
		{
			theOutputFile.Write((const char*) &zero, sizeof( char ));
		}
	}
#endif

	// Export array of faces
	int total_small_faces = 0;
	int total_large_faces = 0;
	for( i = 0; i < num_obj; i++ )
	{
		NxFace* face_list;
		NxObject* object;

		object = &m_scene->m_Objects[i];
		face_list = object->mp_CollFaces;

		// Skip empty objects
//		if (object->m_NumCollVerts == 0)
//			continue;

		for( j = 0; j < object->m_NumCollFaces; j++ )
		{
			NxFace* face;
			NxMaterial* material;

			face = &face_list[j];

			// Write out the face's flags
			//theOutputFile.Write((const char*) &face->m_FaceFlags, sizeof( FlagType ));
			unsigned short flags = (unsigned short) face->m_FaceFlags;
			theOutputFile.Write((const char*) &flags, sizeof( unsigned short ));

			// Write out the face's terrain type. We used to write out the material's checksum and we let
			// the game find the terrain type via indirection, but it seems wasteful when we can get it here.
			material = m_scene->GetMaterial( face->m_MatChecksum );
			short terrain = 0;
			if( material )
			{
				terrain = (short) material->m_Terrain;
			}
			theOutputFile.Write((const char *) &terrain, sizeof( short ));

			// Write out the face's vertex indices
			unsigned short vidx0, vidx1, vidx2;
			vidx0 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[0]];
			vidx1 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[1]];
			vidx2 = (unsigned short) object->mp_ToCollVertIdxs[face->m_Vertex[2]];

			if (object->UseFaceSmall())
			{
				theOutputFile.Write((const char*) &vidx0, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &vidx1, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &vidx2, sizeof( unsigned char ));
				theOutputFile.Write((const char*) &zero, sizeof( unsigned char ));

				total_small_faces++;
			} else {
				theOutputFile.Write((const char*) &vidx0, sizeof( unsigned short ));
				theOutputFile.Write((const char*) &vidx1, sizeof( unsigned short ));
				theOutputFile.Write((const char*) &vidx2, sizeof( unsigned short ));
				//theOutputFile.Write((const char*) &zero, sizeof( unsigned short ));

				total_large_faces++;
			}

			// Write out the face's material checksum -- used to get at the material's properties
			//theOutputFile.Write((const char*) &face->m_MatChecksum, sizeof( unsigned long ));
		}
	}

	// Pad to 32-bit boundary
	if (total_large_faces % 2) {
		theOutputFile.Write((const char*) &zero, sizeof( short ));
	}

#ifdef OUTPUT_COLL_BSP
	// Export BSP node arrray size
	theOutputFile.Write((const char*) &node_array_offset, sizeof( int ));

	// Export array of BSP nodes
	assert(node_array_index == m_scene->m_NumCollBSPTreeNodes);
	int master_face_index = 0;
	int node_offset_check = 0;
	for( i = 0; i < node_array_index; i++ )
	{
		NxCollBSPNode *p_bsp_node = p_node_array[i];
		NxCollBSPLeaf *p_bsp_leaf = NULL;
		node_offset_check += 8;
		if (p_bsp_node->m_split_axis == 3)
		{
			p_bsp_leaf = static_cast<NxCollBSPLeaf *>(p_bsp_node);
		}

		if (p_bsp_leaf)
		{
			// m_split axis must always be in line with the low byte of m_split_point
			theOutputFile.Write((const char*) &(p_bsp_node->m_split_axis), sizeof( char ));
			theOutputFile.Write((const char*) &zero, sizeof( char ));
			theOutputFile.Write((const char*) &(p_bsp_leaf->m_num_faces), sizeof( unsigned short ));
		} else {
			//theOutputFile.Write((const char*) &zero, sizeof( unsigned short ));
			int iSplitPoint = (int) (p_bsp_node->m_split_point * COLLISION_SUB_INCH_PRECISION);
			iSplitPoint = (iSplitPoint << NxCollBSPNode::NUM_AXIS_BITS) | p_bsp_node->m_split_axis;
			theOutputFile.Write((const char*) &iSplitPoint, sizeof( int ));
			//theOutputFile.Write((const char*) &(p_bsp_node->m_split_point), sizeof( float ));
		}

		if (p_bsp_leaf)
		{
			// Write out index to master face index array
			theOutputFile.Write((const char*) &master_face_index, sizeof( int ));
			master_face_index += p_bsp_leaf->m_num_faces;
		} else {
			// Write out indexes to master node array
			theOutputFile.Write((const char*) &(p_bsp_node->mp_less_branch->m_array_offset), sizeof( int ));
			//theOutputFile.Write((const char*) &(p_bsp_node->mp_greater_branch->m_array_offset), sizeof( int ));
		}
	}

	assert(node_array_offset == node_offset_check);

	// Export array of BSP face indexes
	for( i = 0; i < node_array_index; i++ )
	{
		// Check for leaf
		if (p_node_array[i]->m_split_axis == 3)
		{
			NxCollBSPLeaf *p_bsp_leaf = static_cast<NxCollBSPLeaf *>(p_node_array[i]);
			for (int j = 0; j < p_bsp_leaf->m_num_faces; j++)
			{
				theOutputFile.Write((const char*) &(p_bsp_leaf->mp_face_idx_array[j]), sizeof( unsigned short ));
			}
		}
	}

	if (p_node_array)
	{
		delete p_node_array;
	}
#endif // OUTPUT_COLL_BSP
	
	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

// Don't do anything. Xbox uses all mips down to 1x1. This overrides the virtual base class function
// which limits the mips to 8x8
void XBoxConverter::LimitMipMaps( void )
{	
//	SceneConverter::LimitMipMaps();

	int i, j;

	for( i = 0; i < m_num_textures; i++ )
	{
		for( j = 0; j <= m_textures[i].m_MipLevels; j++ )
		{
			if( m_textures[i].m_Width[j] <= 4 )
			{
				m_textures[i].m_MipLevels = j;
				break;
			}
			else if( m_textures[i].m_Height[j] <= 4 )
			{
				m_textures[i].m_MipLevels = j;
				break;
			}
		}		
	}
}

bool XBoxConverter::SaveScene( char* path )
{
	bool success = false;
	
	IoUtils::CVirtualOutputFile theOutputFile;

	// 10-meg buffer
	theOutputFile.Init(10*1024*1024);

	int version;
	
	printf( "Scene...." );

	version = vXBOX_MATERIAL_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));
	version = vXBOX_MESH_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));
	version = vXBOX_VERTEX_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	if( SaveMaterials( &theOutputFile ) == false )
	{
		goto save_error;
	}

	if( SaveGeometry( &theOutputFile ) == false )
	{
		goto save_error;
	}

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}

unsigned __int64 XBoxConverter::GetBlendParameters( int blend_mode, int fixed )
{
	BlendModes mode;
	unsigned __int64 equation, fixed_val;
	
	mode		= (BlendModes) blend_mode;
	fixed_val	= fixed;
	equation	= mode | ( fixed_val << 32 );
	
/*	switch( mode )
	{
		case vBLEND_MODE_DIFFUSE:				// ( 0 - 0 ) * 0 + Src
			a = vPARAM_ZERO;
			b = vPARAM_ZERO;
			c = vPARAM_DST;
			d = vPARAM_SRC;
			break;
		case vBLEND_MODE_ADD:					// ( Src - 0 ) * Src + Dst
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_ADD_FIXED:				// ( Src - 0 ) * Fixed + Dst
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_SUBTRACT:				// ( 0 - Src ) * Src + Dst
			a = vPARAM_ZERO;
			b = vPARAM_SRC;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_SUB_FIXED:				// ( 0 - Src ) * Fixed + Dst
			a = vPARAM_ZERO;
			b = vPARAM_SRC;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND:					// ( Src * Dst ) * Src + Dst	
			a = vPARAM_SRC;
			b = vPARAM_DST;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BLEND_FIXED:			// ( Src * Dst ) * Fixed + Dst	
			a = vPARAM_SRC;
			b = vPARAM_DST;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_MODULATE:				// ( Dst - 0 ) * Src + 0
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_ZERO;
			break;
		case vBLEND_MODE_MODULATE_FIXED:			// ( Dst - 0 ) * Fixed + 0	
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_ZERO;
			break;
		case vBLEND_MODE_BRIGHTEN:				// ( Dst - 0 ) * Src + Dst
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_SRC;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_BRIGHTEN_FIXED:			// ( Dst - 0 ) * Fixed + Dst	
			a = vPARAM_DST;
			b = vPARAM_ZERO;
			c = vPARAM_FIXED;
			d = vPARAM_DST;
			break;
		case vBLEND_MODE_ADD_PREVIOUS_ALPHA:	// ( Src - 0 ) * Dst + Src
			a = vPARAM_SRC;
			b = vPARAM_ZERO;
			c = vPARAM_DST;
			d = vPARAM_SRC;
			break;
	}

	fixed_val = fixed;
	fixed_val <<= 32;

	equation = a | ( b << 2 ) | ( c << 4 ) | ( d << 6 ) | fixed_val;
*/
		
	return equation;
}















/*
Log2 
 
  Purpose: to calculate the log(base 2) of a number that is a power of 2
  Parameters:
    IN Value: the number to take the Log of. 
  Returns: 
    (1 << returnvalue) == Value
  Note: Value must be a power of 2, otherwise strange results will occur.
*/
#pragma warning ( disable: 4035 )
static inline DWORD __fastcall Log2( DWORD Value )
{ 
	__asm { bsf eax, [Value] }; 
}
#pragma warning ( default: 4035 )

/*
GetMasks2

  Purpose: produces the coordinate masks for manuvering through swizzled textures

  Parameters:
	IN Width: the width of the texture to be swizzled
	IN Height: the height of the texture to be swizzled
	OUT pMaskU: the mask for the u-coodinate
	OUT pMaskV: the mask for the v-coordinate

  Notes: This is a much faster algorithm for getting the masks than the 
    more generic algorithm used in the Swizzler class defined in xgraphics.h.
	This algorithm works only for 2d textures. Swizzler's works for 2d and 3d.

*/
static inline void GetMasks2( int Width, int Height, DWORD *pMaskU, DWORD *pMaskV )
{
	DWORD LogWidth, LogHeight, Log;

	LogWidth = Log2(Width); 
	LogHeight = Log2(Height);

	Log = min(LogWidth, LogHeight);

    DWORD LowerMask = (1 << (Log << 1)) - 1;
    DWORD UpperMask = ~LowerMask;

    DWORD MaskU = (LogWidth > LogHeight) ? (0x55555555 | UpperMask)
                                        : (0x55555555 & LowerMask);

    DWORD MaskV = (LogWidth < LogHeight) ? (0xaaaaaaaa | UpperMask)
                                        : (0xaaaaaaaa & LowerMask);

	MaskU &= ((1 << (LogWidth + LogHeight)) - 1); //we're letting u & v just loop, so
	MaskV &= ((1 << (LogWidth + LogHeight)) - 1); //we need to limit the bits to the ones we need.

	*pMaskU = MaskU;
	*pMaskV = MaskV;
}



static void swiz2d_8bit (void* pSrc, void* pDest, int Width, int Height) 
{
	DWORD MaskU, MaskV, AddValU, AddValV;

	GetMasks2(Width, Height, &MaskU, &MaskV);

	AddValU = (-64) & MaskU;
	AddValV = (-128) & MaskV;

	//eax = scratch
	//ebx = u
	//ecx = y
	//edx = pitch
	//esi = pSrc 
	//edi = pDest 

	_asm {
		mov esi, pSrc;					//source + (0, 0)
		mov edi, pDest;					//dest

		mov edx, [Width];				//edx = width
		xor ebx, ebx;					//swiz(u)
		xor ecx, ecx;					//swiz(v)

		align 16						//seems to help speed a little

Start:

			  movq mm0, [esi];				//00 01 02 03 04 05 06 07
			  movq mm1, [esi + edx];		//10 11 12 13 14 15 16 17
			  add esi, edx;					//pSrc + (u, v + 1)
			  mov eax, ebx;					//eax = swiz(u)
			  movq mm4, [esi + edx];		//20 21 22 23 24 25 26 27
			  movq mm5, [esi + edx * 2];	//30 31 32 33 34 35 36 37
			  or eax, ecx;					//eax = swiz(u) | swiz(b)
			  movq mm6, mm4;				//20 21 22 23 24 25 26 27
			  movq mm2, mm0;				//00 01 02 03 04 05 06 07
			  
			  punpckhwd mm6, mm5;			//24 25 34 35 26 27 36 37

			  lea esi, [esi + edx * 4];		//pSrc + (u, v + 5)
			  punpckhwd mm2, mm1			//04 05 14 15 06 07 16 17
			  punpcklwd mm4, mm5;			//20 21 30 31 22 23 30 33

			  movq mm3, [esi];				//50 51 52 53 54 55 56 57
			  movq mm5, [esi+edx];			//60 61 62 63 64 65 66 67
			  movq mm7, [esi+edx*2];		//70 71 72 73 74 75 76 77
			  sub esi, edx					//pSrc + (u, v + 4)
			  punpcklwd mm0, mm1;			//00 01 10 11 02 03 12 13
			  movq mm1, [esi];				//40 41 42 43 44 45 46 47

			  movq [edi + eax      ], mm0;	//00 01 10 11 02 03 12 13
			  movq [edi + eax +   8], mm4;	//20 21 30 31 22 23 30 33
			  movq [edi + eax +  16], mm2;  //04 05 14 15 06 07 16 17
			  movq [edi + eax +  24], mm6;	//24 25 34 35 26 27 36 37
		  
			  movq mm0, mm1					//40 41 42 43 44 45 46 47
			  movq mm4, mm5					//60 61 62 63 64 65 66 67

			  punpcklwd mm0, mm3			//40 41 50 51 42 43 52 53
			  punpcklwd mm4, mm7			//60 61 70 71 62 63 72 73
			  punpckhwd mm1, mm3			//44 45 54 55 46 47 56 57
			  punpckhwd mm5, mm7			//64 65 74 75 66 67 76 77

			  movq [edi + eax +  32], mm0;	//40 41 50 51 42 43 52 53
			  movq [edi + eax +  40], mm4;	//60 61 70 71 62 63 72 73
			  movq [edi + eax +  48], mm1;  //44 45 54 55 46 47 56 57
			  movq [edi + eax +  56], mm5;	//64 65 74 75 66 67 76 77

			  sub esi, edx
			  sub esi, edx
			  sub esi, edx
			  sub esi, edx					//pSrc + (u, v)

			  sub ebx, [AddValU];			//part 1 of: swiz(u) += 8

			  //this number of nops seems to be optimal.
			  _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop}
			  _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop}
			  _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop} _asm{nop}
			  _asm{nop} _asm{nop} _asm{nop} _asm{nop} //_asm{nop} //_asm{nop} _asm{nop} _asm{nop}

			  add esi, 8;					//u += 8
			  and ebx, [MaskU];				//(continued): swiz(u) += 8;
			jnz Start;						//if swiz(u) hasn't looped back to 0, repeat

		  sub ecx, [AddValV];			//part 1 of: swiz(v) += 8		

		  lea esi, [esi + edx * 8]		//pSrc + (u, v + 9) //(v has been incrimented by u looping)
		  sub esi, edx;					//pSrc + (u, v + 8)
		//(it has already looped around to 1 row below where we were. This moves
		//it to the second row down, since we are dealing with 8x8 blocks)

		  and ecx, [MaskV];				//(continued): swiz(v) += 8
		  jnz Start;					//if v is not done, keep going

		emms;		//done with mmx
	}
}


static void swiz2d_16bit (void* pSrc, void* pDest, int Width, int Height) 
{
	DWORD MaskU, MaskV, AddValU, AddValV;

	if(((DWORD)pDest) & 15 )
	{
		assert( 0 );
//		RIP(("Error: Destination memory must be 16-byte aligned"));
	}

	GetMasks2(Width, Height, &MaskU, &MaskV);

	AddValU = (-16) & MaskU;
	AddValV = (-8) & MaskV;

	//eax = scratch
	//ebx = u
	//ecx = y
	//edx = pitch
	//esi = pSrc (this is not changed)
	//edi = pDest (this is incrimented)

	_asm {
		mov esi, pSrc;					
		mov edi, pDest;

		mov edx, [Width];
		xor ebx, ebx;
		xor ecx, ecx;
		add edx, edx;					//edx = width * 2

Start:
		mov eax, ebx;					//eax = u
		movlps xmm0, [esi];				//xmm0 = {a, b, e, f, ?, ?, ?, ?}
		or eax, ecx;					//eax = u | v
		movlps xmm1, [esi + edx];		//xmm1 = {c, d, g, h, ?, ?, ?, ?}
		unpcklps xmm0, xmm1;			//xmm1 = {a, b, c, d, e, f, g, h}
		sub ebx, [AddValU];				//part 1 of adding 4 to u

		movaps [edi + eax*2], xmm0;		//dest[u | v] = {a, b, c, d, e, f, g, h}

		add esi, 8;						//move source pointer to next block
		and ebx, [MaskU];				//ebx = the next u coordinate
		jnz Start;						//if u hasn't looped back to 0, repeat

		sub ecx, [AddValV];				//part 1 of adding 2 to v		
		add esi, edx;					//move src pointer to next row
		//(this moves it to the second row down, since it has already looped
		//to the first row down)

		and ecx, [MaskV];				//ecx = the next v coordinate
		jnz Start;						//if v is not done, keep going
	}
}



static void swiz2d_32bit( void *pSrc, void *pDest, int Width, int Height )
{
	DWORD MaskU, MaskV, AddValU, AddValV;

	if(((DWORD)pDest) & 15 )
	{
		assert( 0 );
	}

	GetMasks2(Width, Height, &MaskU, &MaskV);

	AddValU = (-4) & MaskU;
	AddValV = (-8) & MaskV;

	//eax = scratch
	//ebx = u
	//ecx = y
	//edx = pitch
	//esi = pSrc (this is not changed)
	//edi = pDest (this is incrimented)

	_asm {
		mov esi, pSrc;
		mov edi, pDest;

		mov edx, [Width];
		xor ebx, ebx;
		xor ecx, ecx;
		shl edx, 2;						//edx = width * 4

Start:
			mov eax, ebx;				//eax = u

			movlps xmm0, [esi];			//xmm0 = {a, b}

			or eax, ecx;				//eax = u | v

			movhps xmm0, [esi+edx];		//xmm0 = {a, b, c, d}

			sub ebx, [AddValU];			//part 1 of adding 2 to u

			movaps [edi + eax*4], xmm0;	//dest[u | v] = {a,b,c,d}

			add esi, 8;					//move source pointer to next block
		
			and ebx, [MaskU];			//part 2 of "u += 2"
			jnz Start;					//if u hasn't looped to 0, keep going

		  sub ecx, [AddValV];			//part 1 of "v += 2"

		  add esi, edx;					//move source pointer to 2 rows down

		  and ecx, [MaskV];				//part 1 of "v += 2"
		  jnz Start;					//if v hasn't looped to 0, repeat
	}
}



static unsigned int	swizzle_table[4096];
static bool			swizzle_table_generated = false;

#define TWIDDLE(_u, _v) ((swizzle_table[(_v)] << 1) | (swizzle_table[(_u)]))

/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/
static void generateSwizzleTable( void )
{
	if( !swizzle_table_generated )
	{
		for( unsigned int i = 0, value = 0; i < 4096; i++ )
		{
			swizzle_table[i] = value;
			value += 0x2AAAAAAB;
			value &= 0x55555555;
		}
		swizzle_table_generated = true;
	}
}


/******************************************************************/
/*                                                                */
/*                                                                */
/******************************************************************/
void SwizzleTexture( void *dstBuffer, void *srcBuffer, int width, int height, int depth, int stride )
{
	int tilesX, tilesY;
    int tilesSizeX, tilesSizeY;
    int tileSize;

	// Tiny textures.
	if( width <= 2 || height <= 1 ) 
	{		
		memcpy( dstBuffer, srcBuffer, ( width * height * depth ) / 8 );
		return;
	} 

	// We use the fast versions where possible.
	if( depth == 8 )
	{
		if(( width >= 16 ) && ( height >= 8 ))
		{
			swiz2d_8bit( srcBuffer, dstBuffer, width, height );
			return;
		}
	}
	else if( depth == 16 )
	{
		if(( width >= 16 ) && ( height >= 8 ))
		{
			swiz2d_16bit( srcBuffer, dstBuffer, width, height );
			return;
		}
	}
	else if( depth == 32 )
	{
		if(( width >= 8 ) && ( height >= 8 ))
		{
			swiz2d_32bit( srcBuffer, dstBuffer, width, height );
			return;
		}
	}
	else
	{
		exit( 0 );
	}
	
	// Okay, use the slow version.	
	generateSwizzleTable();

	if( width > height )
    {
        tilesX = width / height;
        tilesY = 1;

        tilesSizeX = width / tilesX;
        tilesSizeY = height;
    }
    else
    {
        tilesX = 1;
        tilesY = height / width;

        tilesSizeX = width;
        tilesSizeY = height / tilesY;
    }

    tileSize = tilesSizeX * tilesSizeY;

	switch (depth)
	{
		case 4:
	    case 8:
        {
			int j;

			for( j = 0; j < tilesY; j++)
            {
                int i;

                for (i = 0; i < tilesX; i++)
                {
					int y;
                    unsigned char *base;

                    base = (unsigned char *)(((unsigned char *)dstBuffer) +
                                       ((tileSize * tilesX) * j) +
                                       (tileSize * i));

                    for (y = 0; y < tilesSizeY; y++)
            {
                unsigned char    *srcPixel;
                int     x;

                        srcPixel = (unsigned char *)(((unsigned char *)srcBuffer) +
                                               (stride * (tilesSizeY * j)) +
                                               (tilesSizeX * i) +
                                               (stride * y));

                        for (x = 0; x < tilesSizeX; x++)
                {
                    unsigned char    *dstPixel;
                        dstPixel = (unsigned char *)(base + TWIDDLE(x, y));
		                *dstPixel = *srcPixel;

                    srcPixel++;
                }
            }
        }
            }
        }
        break;

    case 16:
        {
            int j;

            for (j = 0; j < tilesY; j++)
            {
                int i;

                for (i = 0; i < tilesX; i++)
                {
            int y;
                    unsigned char *base;

                    base = (unsigned char *)(((unsigned short *)dstBuffer) +
                                       ((tileSize * tilesX) * j) +
                                       (tileSize * i));

                    for (y = 0; y < tilesSizeY; y++)
            {
                unsigned short    *srcPixel;
                int     x;

                        srcPixel = (unsigned short *)(((unsigned char *)srcBuffer) +
                                                (stride * (tilesSizeY * j)) +
                                                (2 * tilesSizeX * i) +
                                                (stride * y));

                        for (x = 0; x < tilesSizeX; x++)
                {
                    unsigned short    *dstPixel;
                    dstPixel = (unsigned short *)(base + (TWIDDLE(x, y) << 1));
                    *dstPixel = *srcPixel;

                    srcPixel++;
                }
            }
        }
            }
        }
        break;

    case 24:
    case 32:
        {
            int j;

            for (j = 0; j < tilesY; j++)
            {
                int i;

                for (i = 0; i < tilesX; i++)
                {
            int y;
                    unsigned char *base;

                    base = (unsigned char *)(((unsigned int *)dstBuffer) +
                                       ((tileSize * tilesX) * j) +
                                       (tileSize * i));

                    for (y = 0; y < tilesSizeY; y++)
            {
                unsigned int    *srcPixel;
                int     x;

                        srcPixel = (unsigned int *)(((unsigned char *)srcBuffer) +
                                                (stride * (tilesSizeY * j)) +
                                                (4 * tilesSizeX * i) +
                                                (stride * y));

                        for (x = 0; x < tilesSizeX; x++)
                {
                    unsigned int    *dstPixel;
                    dstPixel = (unsigned int *)(base + (TWIDDLE(x, y) << 2));
                    *dstPixel = *srcPixel;

                    srcPixel++;
                }
            }
        }
            }
        }
        break;

    default:
		exit( 0 );
        break;
    }
}



int XBoxConverter::GetDXTForTexture( NxTexture *p_texture )
{
	int has_holes = 0;
				
	unsigned int *pTex = (unsigned int *)p_texture->m_TexelData[0];
	for( int lp = 0; lp < ( p_texture->m_Width[0] * p_texture->m_Height[0] ); ++lp )
	{
		unsigned int alpha = (( pTex[lp] >> 24 ) & 0xFF );
		if(( alpha > 0 ) && ( alpha < 255 ))
		{
			// Texture has meaningful alpha. Requires DXT 5.
			return 5;
		}
		else if( alpha == 0 )
		{
			has_holes = 1;
		}
	}

	// If a texture has holes, requires DXT1A.
	if( has_holes )
		return 2;
	
	// Standard DXT.
	return 1;
}



bool XBoxConverter::SaveTextureDictionary( char* path, char* usg_path )
{
	bool success = false;

	int		i, version, texture_levels, texel_depth, palette_depth;
	
	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init(30*1024*1024);

	IoUtils::CVirtualOutputFile theUsageFile;
	theUsageFile.Init(1*1024*1024);

	printf( "Texture Dictionary....\n" );

	// Write dictionary version.
	version = vXBOX_TEXTURE_VERSION_NUMBER;
	theOutputFile.Write((const char*) &version, sizeof( int ));

	// Grrr - count how many invalid textures - these will not be saved out.
	int invalid = 0;
	for( i = 0; i < m_num_textures; ++i )
	{
		// Only export textures that were meant for this platform
		if( ( m_textures[i].m_Checksum == vINVALID_CHECKSUM ) ||
			( m_textures[i].m_PlatFlags & GetPlatformFlags()) == 0 )
		{
			++invalid;
		}

		// Track this filename and associated CRCs so we can determine duplicate entries
		AddUSGTexture(&m_textures[i]);
	}
	
	// Write number of textures.
	int num_textures = m_num_textures - invalid;
	theOutputFile.Write((const char*)&num_textures, sizeof( int ));

	// Recurse through textures, writing details per texture.
	for( i = 0; i < m_num_textures; ++i )
	{
		NxTexture* p_texture = &m_textures[i];
		
		// Skip invalid textures.
		if( ( p_texture->m_Checksum == vINVALID_CHECKSUM ) ||
			( p_texture->m_PlatFlags & GetPlatformFlags()) == 0 )
		{
			continue;
		}		

		// Check the flag to see whether compression is allowed on this texture.
//		bool	allow_dxt_compression	= true;
		bool	allow_dxt_compression	= ( p_texture->m_Flags & NxTexture::mCOMPRESS_XBOX ) != 0;
		int		dxt_compression_type	= 0;
		
//		if( strstr( path, "sky" ) || strstr( path, "Sky" ) || strstr( path, "SKY" ))
//		{
//			allow_dxt_compression = false;
//		}
		
		char usg_string[256]; 
		sprintf( usg_string, "%s(0x%x): %dx%d: %d bpp %d pbpp %d levels ",
				 p_texture->m_Name, p_texture->m_Checksum, p_texture->m_Width[0], p_texture->m_Height[0],
				 p_texture->m_Bpp, p_texture->m_PaletteBpp, p_texture->m_MipLevels + 1 );
		theUsageFile.Write((const char*)usg_string, strlen( usg_string ));
		
		if( allow_dxt_compression )
		{
			// Make sure this texture is in a format we can use to DXT compress.
			// At the same time determine the type of compression appropriate.
			if( p_texture->IsPaletted())
			{
				if( p_texture->m_PaletteFormat == NxTexture::v16_BIT )
				{
					// A 16bit palette implies only 1 bit alpha.
					dxt_compression_type = 2;
				}
				else if( p_texture->m_PaletteFormat == NxTexture::v24_BIT )
				{
					// A 24bit palette implies no alpha.
					dxt_compression_type = 1;
				}
				else if( p_texture->m_PaletteFormat == NxTexture::v32_BIT )
				{
					// A 32bit palette implies full alpha.
					dxt_compression_type  = 5;
				}
			}
			else
			{
				if( p_texture->m_PixelFormat == NxTexture::v16_BIT )
				{
					// A 16bit pixel implies only 1 bit alpha.
					dxt_compression_type = 2;
				}
				else if( p_texture->m_PixelFormat == NxTexture::v24_BIT )
				{
					// A 24bit pixel implies no alpha.
					dxt_compression_type = 1;
				}
				else if( p_texture->m_PixelFormat == NxTexture::v32_BIT )
				{
					// A 32bit pixel implies full alpha.
					dxt_compression_type  = 5;
				}
			}

			assert( dxt_compression_type > 0 );
			
			// Now convert to 32 bit texture.
			p_texture->ConvertTo32BitPixelFormat();
			p_texture->Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat();

			// Check to see whether alpha is really necessary.
			if( dxt_compression_type == 5 )
			{
				int new_dxt_compression_type = GetDXTForTexture( p_texture );
				if( new_dxt_compression_type != 5 )
				{
					dxt_compression_type = new_dxt_compression_type;
				}
			}
			
			// Go through and compress each MIP level independently.
			for( int m = 0; m <= p_texture->m_MipLevels; ++m )
			{
				// Grab another buffer to hold the DXT compressed version of the texture.
				unsigned int *pPixelData32	= new unsigned int[p_texture->m_TexelDataSize[m] / 4];
				gpCompressedData			= (unsigned char*)pPixelData32;

				CompressionOptions compression_options;
				compression_options.bBinaryAlpha			= 0;								// Zero or one.
				compression_options.bMipMapsInImage			= false;							// mip have been loaded in during read
				compression_options.MipMapType				= dNoMipMaps;						// dNoMipMaps, dUseExistingMipMaps, dGenerateMipMaps
				compression_options.bNormalMap				= false;							// only renormalize MIP maps
				compression_options.bDuDvMap				= false;							// Is a DuDv map
				compression_options.bAlphaBorder			= false;							// make an alpha border
				compression_options.bBorder					= false;							// make a color border
				compression_options.bFade					= false;							// fade to color over MIP maps
				compression_options.bFadeAlpha				= false;							// fade to color over MIP maps
				compression_options.bDitherColor			= true;								// enable dithering during 16 bit conversion
				compression_options.TextureType				= dTextureType2D;					// regular decal, cube or volume: dTextureType2D, dTextureTypeCube, dTextureTypeImage
				compression_options.TextureFormat			= ( dxt_compression_type == 1 ) ? dDXT1 : (( dxt_compression_type == 2 ) ? dDXT1a : dDXT5 );	// dDXT1, dDXT1a, dDXT3, dDXT5, d4444, d1555, d565, d8888, d888, d555
				compression_options.bSwapRGB				= false;
						
				gDXTBytesWritten = 0;
				
				HRESULT hr = nvDXTcompress( (unsigned char*)( p_texture->m_TexelData[m] ),		// pointer to data (24 or 32 bit)
											p_texture->m_Width[m],								// width in texels
											p_texture->m_Height[m],								// height in texels
											p_texture->m_Width[m] * 4,							// pitch
											&compression_options,
											4,													// depth: 3 or 4
											NULL );												// callback for generated levels

				// Set the new texel data pointer and size for this level.
				delete [] p_texture->m_TexelData[m];
				p_texture->m_TexelData[m]		= (char*)pPixelData32;
				p_texture->m_TexelDataSize[m]	= gDXTBytesWritten;
			}

			sprintf( usg_string, "DXT: %s\n", ( dxt_compression_type == 1 ) ? "1" : ( dxt_compression_type == 2 ) ? "1a" : "5" );
			theUsageFile.Write((const char*)usg_string, strlen( usg_string ));
		}
		else
		{
			// Make sure this texture is in a format we can deal with on Xbox.
			if( p_texture->IsPaletted())
			{
				if( p_texture->m_PaletteFormat == NxTexture::v16_BIT )
				{
					p_texture->Convert16BitPaletteFormatTo32BitPaletteFormat();
				}
				else if( p_texture->m_PaletteFormat == NxTexture::v24_BIT )
				{
					p_texture->Convert24BitPaletteFormatTo32BitPaletteFormat();
				}

				if( p_texture->m_PaletteFormat == NxTexture::v32_BIT )
				{
					p_texture->Convert32BitRGBAPaletteFormatTo32BitBGRAPaletteFormat();
				}
			}
			
			if( p_texture->m_PixelFormat == NxTexture::v4_BIT )
			{
				p_texture->Convert4BitPixelFormatTo8BitPixelFormat();
			}
			else if( p_texture->m_PixelFormat == NxTexture::v24_BIT )
			{
				p_texture->ConvertTo32BitPixelFormat();
			}

			if( p_texture->m_PixelFormat == NxTexture::v32_BIT )
			{
				p_texture->Convert32BitRGBAPixelFormatTo32BitBGRAPixelFormat();
			}

			sprintf( usg_string, "DXT: %s\n", "None" );
			theUsageFile.Write((const char*)usg_string, strlen( usg_string ));
		}
		
		// Write out texture checksum.
		theOutputFile.Write((const char*) &p_texture->m_Checksum,				sizeof( unsigned long ));

		// Write out level 0 width and height.
		theOutputFile.Write((const char *) &p_texture->m_Width[0],				sizeof( int ));
		theOutputFile.Write((const char *) &p_texture->m_Height[0],				sizeof( int ));

		// Write out number of MIP maps.
		// The member m_MipLevels is 0 for no levels *other* than the base texture.
		texture_levels = p_texture->m_MipLevels + 1;
		theOutputFile.Write((const char *)&texture_levels,						sizeof( int ));
		
		// Write out texel depth.
		texel_depth =	( p_texture->m_PixelFormat == NxTexture::v4_BIT ) ? 4 :
						(( p_texture->m_PixelFormat == NxTexture::v8_BIT ) ? 8 :
						(( p_texture->m_PixelFormat == NxTexture::v16_BIT ) ? 16 :
						(( p_texture->m_PixelFormat == NxTexture::v32_BIT ) ? 32 : 0 )));
		theOutputFile.Write((const char *) &texel_depth,							sizeof( int ));

		// Write out palette depth.
		palette_depth = ( p_texture->m_PaletteFormat == NxTexture::v16_BIT ) ? 16 :
						(( p_texture->m_PaletteFormat == NxTexture::v24_BIT ) ? 24 :
						(( p_texture->m_PaletteFormat == NxTexture::v32_BIT ) ? 32 : 0 ));
		theOutputFile.Write((const char *) &palette_depth,						sizeof( int ));

		// Write out DXT compression type.
//		dxt_compression_type = 0;
		theOutputFile.Write((const char *) &dxt_compression_type,				sizeof( int ));

		// Write out palette size.
		theOutputFile.Write((const char *) &p_texture->m_TotalPaletteDataSize,	sizeof( int ));

		// Write out palette.
		if( p_texture->IsPaletted())
		{
			theOutputFile.Write((const char *)p_texture->m_PaletteData,			p_texture->m_TotalPaletteDataSize );
		}

		// The member m_MipLevels is 0 for no levels *other* than the base texture.
		for( int m = 0; m <= p_texture->m_MipLevels; ++m )
		{
			// Write out level n size.
			int texel_size = p_texture->m_TexelDataSize[m];
			theOutputFile.Write((const char *)&texel_size,						sizeof( int ));

			// Swizzle level n texel data if required. No swizzling required for textures that have been DXT compressed,
			// or for MIP levels smaller than 8x8.
			if(( dxt_compression_type == 0 ) && ( p_texture->m_Width[m] >= 8 ) && ( p_texture->m_Height[m] >= 8 ))
			{
				switch( p_texture->m_PixelFormat )
				{
					case NxTexture::v8_BIT:
					{
						// Grab array for new data, must be 16byte aligned.
						char *p_new_texel_data			= new char[p_texture->m_TexelDataSize[m] + 16];
						char *p_new_texel_data_aligned	= (char*)(((int)p_new_texel_data + 15 ) & ~15 );

						// Swizzle texture.
//						swiz2d_8bit( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_Width[m], p_texture->m_Height[m] );
						SwizzleTexture( p_new_texel_data_aligned, p_texture->m_TexelData[m], p_texture->m_Width[m], p_texture->m_Height[m], 8, p_texture->m_Width[m] );
						
						// Copy new texture data over old.
						memcpy( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_TexelDataSize[m] );

						// And remove new.
						delete [] p_new_texel_data;
						break;
					}
				
					case NxTexture::v16_BIT:
					{
						// Grab array for new data, must be 16byte aligned.
						char *p_new_texel_data			= new char[p_texture->m_TexelDataSize[m] + 16];
						char *p_new_texel_data_aligned	= (char*)(((int)p_new_texel_data + 15 ) & ~15 );

						// Swizzle texture.
//						swiz2d_16bit( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_Width[m], p_texture->m_Height[m] );
						SwizzleTexture( p_new_texel_data_aligned, p_texture->m_TexelData[m], p_texture->m_Width[m], p_texture->m_Height[m], 16, p_texture->m_Width[m] );
						
						// Copy new texture data over old.
						memcpy( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_TexelDataSize[m] );

						// And remove new.
						delete [] p_new_texel_data;
						break;
					}
				
					case NxTexture::v32_BIT:
					{
						// Grab array for new data, must be 16byte aligned.
						char *p_new_texel_data			= new char[p_texture->m_TexelDataSize[m] + 16];
						char *p_new_texel_data_aligned	= (char*)(((int)p_new_texel_data + 15 ) & ~15 );

						// Swizzle texture.
//						swiz2d_32bit( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_Width[m], p_texture->m_Height[m] );
						SwizzleTexture( p_new_texel_data_aligned, p_texture->m_TexelData[m], p_texture->m_Width[m], p_texture->m_Height[m], 32, p_texture->m_Width[m] );
						
						// Copy new texture data over old.
						memcpy( p_texture->m_TexelData[m], p_new_texel_data_aligned, p_texture->m_TexelDataSize[m] );

						// And remove new.
						delete [] p_new_texel_data;
						break;
					}
				}
			}
			
			// Write out level n texel data.
			theOutputFile.Write((const char *)p_texture->m_TexelData[m], p_texture->m_TexelDataSize[m] );
		}
	}

	WriteUSGDuplicates(&theUsageFile);
	
	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		goto save_error;
	}

	// break the lock, if necessary
	SetFileAttributes( usg_path, FILE_ATTRIBUTE_NORMAL );
	if ( !theUsageFile.Save( usg_path, IoUtils::vTEXT_MODE ) )
	{
		goto save_error;
	}

	success = true;

save_error:

	return success;
}



bool XBoxConverter::SaveCASFlags( char* path )
{
	// Exit if this is not a CAS object.
	NxObject *object = get_cas_object();
	if( !object )
	{
		return false;
	}

	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init(1*1024*1024);
	
	int version;

	printf( "CAS flags...." );

	version = vXBOX_CASFLAGS_VERSION_NUMBER;
	theOutputFile.Write((const char*)&version, sizeof( int ));

	unsigned int removalMask = 0;
	if( object->m_Flags & NxObject::mHASCASREMOVEFLAGS )
	{
		removalMask = object->m_CASRemoveFlags;
	}
	theOutputFile.Write((const char*)&removalMask, sizeof( int ));
	
	theOutputFile.Write((const char*) &object->m_NumCASData, sizeof( int ));

	for ( int i = 0; i < object->m_NumCASData; i++ )
	{
		theOutputFile.Write((const char*) &object->mp_CASData[i].m_Mask,		sizeof( unsigned long ));
		theOutputFile.Write((const char*) &object->mp_CASData[i].m_Data,		sizeof( unsigned long ));
		theOutputFile.Write((const char*) &object->mp_CASData[i].m_Data1,	sizeof( unsigned long ));
	}

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		return false;
	}

	return true;
}



bool XBoxConverter::SaveWeightMap( char* path )
{
	NxObject* object = get_cas_object();

	if( !object )
	{
		// Not a create-a-skater object.
		return false;
	}

	if( !m_weight_map_loaded )
	{
		return false;
	}

	IoUtils::CVirtualOutputFile theOutputFile;
	theOutputFile.Init( 1 * 1024 * 1024 );

	printf( "Mesh scaling weight map...." );

	int version = vXBOX_WEIGHTMAP_VERSION_NUMBER;
	theOutputFile.Write((const char*)&version, sizeof( int ));

	Utils::Assert( mp_weight_map_vertices != NULL, "weight map vertices uninitialized?" );

	NxVertex* pVertices = mp_weight_map_vertices;
	int numVertices = m_num_weight_map_vertices;

	theOutputFile.Write((const char*)&numVertices, sizeof( int ));

	for( int i = 0; i < numVertices; i++ )
	{
		NxVertex* pVertex = &pVertices[i];
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[0], sizeof( float ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[1], sizeof( float ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeight[2], sizeof( float ));
	}

	for( int i = 0; i < numVertices; i++ )
	{
		NxVertex* pVertex = &pVertices[i];
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[0], sizeof( unsigned char ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[1], sizeof( unsigned char ));
		theOutputFile.Write((const char*) &pVertex->m_MeshScalingWeightedIndex[2], sizeof( unsigned char ));
	}

	// break the lock, if necessary
	SetFileAttributes( path, FILE_ATTRIBUTE_NORMAL );
	if ( !theOutputFile.Save( path ) )
	{
		return false;
	}

	return true;
}



int	XBoxConverter::GetPlatformFlags( void )
{
	if( gTextureOverride )
	{
		return( 1 << gTextureOverrideMapping[Utils::vPLATFORM_XBOX] );
	}
	return mPLAT_XBOX;
}
