/*
	MipGen.cpp
	Mip Map Generation and conversion utility
	7-8-01
*/

#define  HASHTABLE_NOASSERTIFCLASH			// Prevent the hashtable from asserting if we have duplicates
#define  HASHINCLUDED
#include <core/hashtable.h>
#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include "Png.h"
#include "Targa.h"

#define MASK_R555  0x7C00
#define MASK_G555  0x3E0
#define MASK_B555  0x1F

#define MASK_R565  0xF800
#define MASK_G565  0x7E0
#define MASK_B565  0x1F

class MipGen
{
public:

};

struct Point
{
	int x,y;

	inline Point(int sx, int sy)
	{
		x = sx;
		y = sy;
	}
};

template<class T>
struct Freq
{
	T            data;
	unsigned int freq;		// How frequent this data appeared within the sample area

	Freq() { freq = 0; }
};

struct color8		// Palletized texture
{
	unsigned char index;

	int operator== (color8& right)
	{
		return (index == right.index);
	}
};

struct color16_555
{
	enum
	{
		RED,
		GREEN,
		BLUE,
	};
	
	__int16 color;		// Contains the packed color in RGB555 format 5 bits (red, green, and blue)	

	inline unsigned char R()
	{
		return (color >> 10);
	}

	// Pack the red component into 555 format
	inline void R(unsigned char val)
	{
		color &= ~MASK_R555;		// Remove current red component
		color |= (val << 10);		// Pack in new value
	}

	inline unsigned char G()
	{
		return ((color & MASK_G555) >> 5);
	}

	inline void G(unsigned char val)
	{
		color &= ~MASK_G555;		// Remove current green component
		color |= (val << 5);		// Pack in new value
	}

	inline unsigned char B()
	{
		return (color & MASK_B555);
	}

	inline void B(unsigned char val)
	{
		color &= ~MASK_B555;		// Remove current blue component
		color |= val;				// Pack in new value
	}

	inline unsigned char operator[](int id)
	{
		switch(id)
		{
		case RED:
			return R();

		case GREEN:
			return G();			

		case BLUE:
			return B();
		}
	}

	int operator== (color16_555& right)
	{
		return (color == right.color);
	}
};

struct color16_565
{
	enum
	{
		RED,
		GREEN,
		BLUE,
	};

	__int16 color;		// Contains the packed color in RGB565 format 5 bits red, 6 bits green, 5 bits blue)

	inline unsigned char R()
	{
		return (color >> 10);
	}

	inline void R(unsigned char val)
	{
		color &= ~MASK_R565;		// Remove red component
		color |= (val << 10);		// Pack in new value
	}

	inline unsigned char G()
	{
		return ((color & MASK_G565) >> 5);
	}

	inline void G(unsigned char val)
	{
		color &= ~MASK_G565;		// Remove green component
		color |= (val << 5);		// Pack in new value
	}

	inline unsigned char B()
	{
		return (color & MASK_B565);
	}

	inline void B(unsigned char val)
	{
		color &= ~MASK_B565;		// Remove blue component
		color |= val;				// Pack in new value
	}

	inline unsigned char operator[](int id)
	{
		switch(id)
		{
		case RED:
			return R();

		case GREEN:
			return G();			

		case BLUE:
			return B();
		}
	}

	int operator== (color16_565& right)
	{
		return (color == right.color);
	}
};

struct color24
{
	unsigned char r,g,b;

	int operator== (color24& right)
	{
		return (r == right.r &&
			    g == right.g &&
				b == right.b);
	}
};

struct color32
{
	unsigned char r,g,b,a;

	int operator== (color32& right)
	{
		return (r == right.r &&
			    g == right.g &&
				b == right.b &&
				a == right.a);
	}
};

// Dumps out the image in Targa format for a quick test (me likes targa)
bool DumpTGA(char* Filename,void* pMem,int width,int height,int depth)
{
	// Construct the Targa Header
	TGAHeader header;
	header.IDLength        = 0;						// No image ID field
	header.ColorMapType    = 0;						// Expecting only non-paletted data for now

	switch(depth)
	{
	case 8:
		header.ImageType   = TGATYPE_RAWPAL;
		break;

	default:
		header.ImageType   = TGATYPE_RAWTRUECOLOR;	// Only uncompressed true color images for now	
		break;
	}
	
	header.CMapStart       = 0;						// No palette
	header.CMapLength      = 0;						// No palette
	header.CMapDepth       = 0;						// No palette
	header.XOffset         = 0;
	header.YOffset         = 0;
	header.Width           = width;
	header.Height          = height;
	header.PixelDepth      = (unsigned char)depth;
	header.ImageDescriptor = 0;

	FILE* fp;

	fp = fopen(Filename,"wb");

	if (!fp)
		return false;

	// Write out the Targa Header
	fwrite(&header,sizeof(TGAHeader),1,fp);

	// Dump the raw image data
	int size = width * height;
	
	switch(depth)
	{
	case 8:
		{
			color8* pPos = (color8*)pMem;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->index,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 16:
		{
			color16_555* pPos = (color16_555*)pMem;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->color,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 24:
		{
			color24* pPos = (color24*)pMem;
			
			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->b,sizeof(unsigned char),1,fp);
				fwrite(&pPos->g,sizeof(unsigned char),1,fp);
				fwrite(&pPos->r,sizeof(unsigned char),1,fp);
			}

			break;
		}
	case 32:
		{
			color32* pPos = (color32*)pMem;

			for(int i=0;i<size;i++,pPos++)
			{
				fwrite(&pPos->b,sizeof(unsigned char),1,fp);
				fwrite(&pPos->g,sizeof(unsigned char),1,fp);
				fwrite(&pPos->r,sizeof(unsigned char),1,fp);
				fwrite(&pPos->a,sizeof(unsigned char),1,fp);
			}

			break;
		}
	}

	fclose(fp);
	return true;
}

// Builds a histogram for the image so we can quantize it
// list of all colors used in the image and the frequencies of each

Lst::HashTable< Freq<color8> >* BuildHist8(void* srcBuf,
										   unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth)
{
	Lst::HashTable< Freq<color8> >* list = new Lst::HashTable< Freq<color8> >(8);
	unsigned int x, y;
	color8 color;

	color8* sBuf = (color8*)srcBuf;

	for(y=0;y<srcHeight;y++)
		for(x=0;x<srcWidth;x++)
		{
			color = sBuf[y*srcDepth+x];

			// TODO: access palette

			unsigned int key = 0;/*( (color.R() << 24) | (color.G() << 16) | (color.B() << 8) );*/
			Freq<color8>* freqColor = list->GetItem(key,false);

			// Update frequency if appropriate
			if (freqColor)
				freqColor->freq++;
			else
			{
				freqColor = new Freq<color8>;
				freqColor->data = color;
				freqColor->freq = 1;

				list->PutItem(key,freqColor);
			}
		}

	return list;
}

Lst::HashTable< Freq<color16_555> >* BuildHist16(void* srcBuf,
												 unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth)
{
	Lst::HashTable< Freq<color16_555> >* list = new Lst::HashTable< Freq<color16_555> >(8);
	unsigned int x, y;
	color16_555 color;

	color16_555* sBuf = (color16_555*)srcBuf;

	for(y=0;y<srcHeight;y++)
		for(x=0;x<srcWidth;x++)
		{
			color = sBuf[y*srcDepth+x];
			unsigned int key = ( (color.R() << 24) | (color.G() << 16) | (color.B() << 8) );
			Freq<color16_555>* freqColor = list->GetItem(key,false);

			// Update frequency if appropriate
			if (freqColor)
				freqColor->freq++;
			else
			{
				freqColor = new Freq<color16_555>;
				freqColor->data = color;
				freqColor->freq = 1;

				list->PutItem(key,freqColor);
			}
		}

	return list;
}

Lst::HashTable< Freq<color24> >* BuildHist24(void* srcBuf,
											 unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth)
{
	Lst::HashTable< Freq<color24> >* list = new Lst::HashTable< Freq<color24> >(32);
	unsigned int x, y;
	color24 color;

	color24* sBuf = (color24*)srcBuf;

	for(y=0;y<srcHeight;y++)
		for(x=0;x<srcWidth;x++)
		{
			color = sBuf[y*srcDepth+x];
			unsigned int key = ( (color.r << 24) | (color.g << 16) | (color.b << 8) );
			Freq<color24>* freqColor = list->GetItem(key,false);

			// Update frequency if appropriate
			if (freqColor)
				freqColor->freq++;
			else
			{
				freqColor = new Freq<color24>;
				freqColor->data = color;
				freqColor->freq = 1;

				list->PutItem(key,freqColor);
			}
		}

	return list;
}

void DumpFreq24(Lst::HashTable< Freq<color24> >* colors)
{
	FILE* fp = fopen("c:\\hist.txt","w");

	colors->IterateStart();

	Freq<color24>* entry = colors->IterateNext();
	int nUniqueColors = 0;

	while(entry)
	{
		fprintf(fp,"[%i,%i,%i]: %i\n",entry->data.r,entry->data.g,entry->data.b,entry->freq);
		entry = colors->IterateNext();

		nUniqueColors++;
	}

	fprintf(fp,"Unique Colors: %i\n",nUniqueColors);

	fclose(fp);
}

Lst::HashTable< Freq<color32> >* BuildHist32(void* srcBuf,
						  unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth)
{
	Lst::HashTable< Freq<color32> >* list = new Lst::HashTable< Freq<color32> >(8);
	unsigned int x, y;
	color32 color;

	color32* sBuf = (color32*)srcBuf;

	for(y=0;y<srcHeight;y++)
		for(x=0;x<srcWidth;x++)
		{
			color = sBuf[y*srcDepth+x];
			unsigned int key = ( (color.r << 24) | (color.g << 16) | (color.b << 8) | (color.a) );
			Freq<color32>* freqColor = list->GetItem(key,false);

			// Update frequency if appropriate
			if (freqColor)
				freqColor->freq++;
			else
			{
				freqColor = new Freq<color32>;
				freqColor->data = color;
				freqColor->freq = 1;

				list->PutItem(key,freqColor);
			}
		}

	return list;
}

// Reduce the size of the image using the closest neighbor method (unoptimized)
void* ReduceImageSizeCN(void* srcBuf,
					    unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth,
					    unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * (srcDepth / 8);
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y;

	unsigned char* Dbuf = (unsigned char*)destBuf;
	unsigned char* Sbuf = (unsigned char*)srcBuf;

	switch(srcDepth)
	{
	case 8:
		{
			color8* Dbuf = (color8*)destBuf;
			color8* Sbuf = (color8*)srcBuf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcWidth+(float)x*Xstep)];
		}
		break;
	case 16:
		{
			// We will assume that all bit depths of 16 are packed 555 RGB
			color16_555* Dbuf = (color16_555*)destBuf;
			color16_555* Sbuf = (color16_555*)srcBuf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcWidth+(float)x*Xstep)];
		}
		break;
	case 24:
		{
			color24* Dbuf = (color24*)destBuf;
			color24* Sbuf = (color24*)srcBuf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)( (float)y*(float)Ystep*(float)srcWidth+((float)x*(float)Xstep) )];
		}
		break;
	case 32:
		{
			color32* Dbuf = (color32*)destBuf;
			color32* Sbuf = (color32*)srcBuf;

			for(y=0;y<destHeight;y++)
				for(x=0;x<destWidth;x++)
					Dbuf[y*destWidth+x] = Sbuf[(int)((float)y*Ystep*srcWidth+(float)x*Xstep)];
		}
		break;
	}

	return destBuf;
}

void* Proc8bitBoxReduce(void* srcBuf,
					    unsigned int srcWidth, unsigned int srcHeight,
					    unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i,j;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	

	color8* Dbuf = (color8*)destBuf;
	color8* Sbuf = (color8*)srcBuf;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			Freq<color8> colors[9];
			unsigned int highFreq = 0;				// Highest color freq
			color8       bestColor;					// Color with highest freq

			Point ptSrc((int)((float)y*Ystep),(int)((float)x*Xstep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];
			colors[0].data   = bestColor;
			colors[0].freq   = 1;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color8 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];
					bool   bFound = false;

					// Check if color exists and update the frequency
					for(j=0;j<i+1;j++)
					{
						if (colors[j].data == color)
						{
							colors[j].freq++;

							// Track the most frequent color
							if (colors[j].freq > highFreq)
							{
								highFreq  = colors[j].freq;
								bestColor = colors[j].data;
							}

							bFound = true;
							break;
						}
					}

					// If we couldn't find it, add a new color
					if (!bFound)
						colors[i+1].data = color;
				}
			}

			// Use the color with the highest frequency
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

void* Proc16bitBoxReduce(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight,
						 unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 2;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color16_555* Dbuf = (color16_555*)destBuf;
	color16_555* Sbuf = (color16_555*)srcBuf;

	color16_555 bestColor;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area

			Point ptSrc((int)((float)y*Ystep),(int)((float)x*Xstep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];

			freqR = bestColor.R();
			freqG = bestColor.G();
			freqB = bestColor.B();

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color16_555 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];

					freqR += color.R();
					freqG += color.G();
					freqB += color.B();
				}
			}

			// Find the average color for the surrounding area
			bestColor.R((unsigned char)(freqR / 8));
			bestColor.G((unsigned char)(freqG / 8));
			bestColor.B((unsigned char)(freqB / 8));

			// Use the average color of the region
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

void* Proc24bitBoxReduce(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight,
						 unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 3;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color24* Dbuf = (color24*)destBuf;
	color24* Sbuf = (color24*)srcBuf;

	color24 bestColor;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int nSamples = 1;	// Number of samples computed from matrix

			Point ptSrc((int)((float)y*Ystep),(int)((float)x*Xstep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color24 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
					nSamples++;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / nSamples);
			bestColor.g = (unsigned char)(freqG / nSamples);
			bestColor.b = (unsigned char)(freqB / nSamples);

			// Use the average color of the region
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

void* Proc32bitBoxReduce(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight,
						 unsigned int destWidth, unsigned int destHeight)
{
	// Allocate memory for destination buffer (owned by caller)
	int memSize = destWidth * destHeight * 4;
	char* destBuf = (char*)malloc(memSize);

	// Calc reduction ratios
	float Xstep = (float)srcWidth / (float)destWidth;
	float Ystep = (float)srcHeight / (float)destHeight;

	unsigned int x,y,i;

	// Matrix for computing neighboring pixels
	const Point Matrix[8] = { Point(-1,-1),
							  Point( 0,-1),
							  Point( 1,-1),
							  Point(-1, 0),
							  Point( 1, 0),
							  Point(-1, 1),
							  Point( 0, 1),
							  Point( 1, 1) };
	
	color32* Dbuf = (color32*)destBuf;
	color32* Sbuf = (color32*)srcBuf;

	color32 bestColor;

	for(y=0;y<destHeight;y++)
		for(x=0;x<destWidth;x++)
		{
			unsigned int freqR = 0;		// Total Red component around sampled area
			unsigned int freqG = 0;		// Total Green component around sampled area
			unsigned int freqB = 0;		// Total Blue component around sampled area
			unsigned int freqA = 0;		// Total Alpha component around sampled area

			Point ptSrc((int)((float)x*Xstep),(int)((float)y*Ystep));

			// Assign the sample point color
			bestColor = Sbuf[ptSrc.y*srcWidth+ptSrc.x];

			freqR = bestColor.r;
			freqG = bestColor.g;
			freqB = bestColor.b;

			// Find neighboring pixel values
			for(i=0;i<8;i++)
			{
				Point ptSample(ptSrc.x+Matrix[i].x,
							   ptSrc.y+Matrix[i].y);

				if (ptSample.x > -1 && ptSample.x < (int)srcWidth &&
					ptSample.y > -1 && ptSample.y < (int)srcHeight)
				{
					// Find sample color for this location in the matrix
					color32 color  = Sbuf[(int)((float)ptSample.y*srcWidth+(float)ptSample.x)];

					freqR += color.r;
					freqG += color.g;
					freqB += color.b;
				}
			}

			// Find the average color for the surrounding area
			bestColor.r = (unsigned char)(freqR / 9);
			bestColor.g = (unsigned char)(freqG / 9);
			bestColor.b = (unsigned char)(freqB / 9);

			// Use the average color of the region
			Dbuf[y*destWidth+x] = bestColor;
		}

	return Dbuf;
}

// Box filtered picking method to average surrounding pixels to determine the best color to use
// for the image size reduction (unoptimized)
void* ReduceImageSizeBox(void* srcBuf,
						 unsigned int srcWidth, unsigned int srcHeight, unsigned int srcDepth,
						 unsigned int destWidth, unsigned int destHeight)
{
	switch(srcDepth)
	{
	case 8:
		return Proc8bitBoxReduce(srcBuf,srcWidth,srcHeight,destWidth,destHeight);

	case 16:
		return Proc16bitBoxReduce(srcBuf,srcWidth,srcHeight,destWidth,destHeight);

	case 24:
		return Proc24bitBoxReduce(srcBuf,srcWidth,srcHeight,destWidth,destHeight);

	case 32:
		return Proc32bitBoxReduce(srcBuf,srcWidth,srcHeight,destWidth,destHeight);
	}

	// Unknown format
	return NULL;
}

bool WritePNG(char* Filename,
			  void* buf,
			  unsigned int width, unsigned int height, unsigned int chandepth, unsigned int nChannels)
{
	FILE* fp = fopen(Filename, "wb");

	if (!fp)
		return false;

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_write_struct(PNG_LIBPNG_VER_STRING,
		                                          NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);

	// Color type modes
	//
	// PNG_COLOR_TYPE_GRAY
	// PNG_COLOR_TYPE_PALETTE
	// PNG_COLOR_TYPE_RGB
	// PNG_COLOR_TYPE_RGB_ALPHA
	// PNG_COLOR_TYPE_GRAY_ALPHA

	png_set_IHDR(png_ptr,info_ptr,
	             width,height,
				 chandepth,
				 PNG_COLOR_TYPE_RGB_ALPHA,
				 PNG_INTERLACE_NONE,
				 PNG_COMPRESSION_TYPE_BASE,
				 PNG_FILTER_TYPE_BASE);


	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	// Build up row pointers
	png_bytep* pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);
	unsigned char* pPos = (unsigned char*)buf;

	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += width*((nChannels*chandepth)/8);
	}

	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}

	png_write_info(png_ptr,info_ptr);

	//png_write_png(png_ptr,info_ptr,PNG_TRANSFORM_IDENTITY,NULL);

	png_write_image(png_ptr,pRows);

	png_write_end(png_ptr,info_ptr);
	png_destroy_write_struct(&png_ptr,&info_ptr);

	free(pRows);
	return true;
}

bool ReadPNG(char* Filename)
{
	FILE* fp = fopen(Filename,"rb");

	if (!fp)
		return false;

	// Read in the PNG Sig to verify this is valid PNG file
	unsigned char IHDR[8];
	fread(IHDR,1,8,fp);

	// Have the PNG library verify the header information
	if (png_sig_cmp(IHDR,0,8))
	{
		fclose(fp);
		return false;
	}

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
		                                         NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);
	png_set_sig_bytes(png_ptr,8);

	// Do a simple read of the PNG file without doing any transforms on the image
	png_read_info(png_ptr, info_ptr);
	
	// Retrieve the header data
	unsigned long width,height;
	int bit_depth,color_type,interlace_type,compression_type,filter_type;
	png_get_IHDR(png_ptr,info_ptr,&width,&height,&bit_depth,&color_type,&interlace_type,&compression_type,&filter_type);

	int channels;
	channels = png_get_channels(png_ptr,info_ptr);

	// Allocate memory for each row of the png
	//void* pMem = malloc(width*height*(bit_depth/8));
	int memSize = (width)*(height)*((channels*bit_depth)/8);
	//void* pMem = calloc(1,memSize);
	void* pMem = malloc(memSize);

	// Setup pointers to each row of the image
	unsigned char* pPos = (unsigned char*)pMem;
	png_bytep* pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// libpng appers to store each pixel component R,G,B,A as a byte
	// well, not sure about A yet
	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += width*(channels*bit_depth/8);
	}

	png_bytep* pRowsRev = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}

	/*
	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += width*(channels*bit_depth/8);
	}
	*/

	png_read_image(png_ptr,pRows);

	/*
	int x;
	x=0;
	*/

	// Reduce the size of the image to 64x64
	//void* pImage = ReduceImageSizeCN(pMem,width,height,bit_depth * channels,512,512);
	void* pImage = ReduceImageSizeBox(pMem,width,height,bit_depth * channels,512,512);

	// Dump out test Targa
	//DumpTGA("c:\\test.tga",pMem,width,height,bit_depth * channels);
	//DumpTGA("c:\\test.tga",pImage,512,512,bit_depth * channels);
	WritePNG("c:\\output.png",pImage,512,512,bit_depth,channels);

	Lst::HashTable< Freq<color24> >* table = BuildHist24(pImage,512,512,bit_depth * channels);
	DumpFreq24(table);

	free(pRows);
	free(pMem);
	free(pImage);

	png_read_end(png_ptr,end_info);

	png_destroy_read_struct(&png_ptr,&info_ptr,&end_info);

	return true;
}

///////////////////////////////// READPNGLL()   (Low-level PNG read) ///////////////////////////////////////
bool ReadPNGLL(char* Filename)
{
	FILE *fp;

	fp = fopen(Filename,"rb");

	// Read in the PNG Sig to verify this is valid PNG file
	unsigned char IHDR[8];
	fread(IHDR,1,8,fp);

	// Have the PNG library verify the header information
	if (png_sig_cmp(IHDR,0,8))
	{
		fclose(fp);
		return false;
	}

	// Initialize the LibPng library
	// (Use default error handling callbacks)
	png_structp png_ptr = png_create_read_struct(PNG_LIBPNG_VER_STRING,
		                                         NULL,NULL,NULL);

	png_infop info_ptr = png_create_info_struct(png_ptr);

	if (!info_ptr)
	{
		png_destroy_read_struct(&png_ptr,
			                    (png_infopp)NULL,
								(png_infopp)NULL);

		fclose(fp);
		return false;
	}

	png_infop end_info = png_create_info_struct(png_ptr);

	if (!end_info)
	{
		png_destroy_read_struct(&png_ptr,
			                    &info_ptr,
								(png_infopp)NULL);

		return false;
	}

	png_init_io(png_ptr,fp);
	png_set_sig_bytes(png_ptr,8);

	// Do a simple read of the PNG file without doing any transforms on the image
	png_read_info(png_ptr, info_ptr);
	
	// Retrieve the header data
	unsigned long width,height;
	int bit_depth,color_type,interlace_type,compression_type,filter_type;
	png_get_IHDR(png_ptr,info_ptr,&width,&height,&bit_depth,&color_type,&interlace_type,&compression_type,&filter_type);

	int channels;
	channels = png_get_channels(png_ptr,info_ptr);

	// Allocate memory for each row of the png
	int memSize = (width)*(height)*((channels*bit_depth)/8);
	void* pMem = malloc(memSize);

	// Setup pointers to each row of the image
	unsigned char* pPos = (unsigned char*)pMem;
	png_bytep* pRows = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// libpng appers to store each pixel component R,G,B,A as a byte
	// well, not sure about A yet
	for(unsigned int i=0;i<height;i++)
	{
		pRows[i] = pPos;
		pPos += width*(channels*bit_depth/8);
	}

	png_bytep* pRowsRev = (png_bytep*)malloc(sizeof(png_bytep)*height);

	// Copy the scanline data in reverse order to flip the image
	unsigned int halfsize = height/2;
	
	for(i=0;i<halfsize;i++)
	{
		png_bytep ptemp = pRows[height-i-1];
		pRows[height-i-1] = pRows[i];
		pRows[i] = ptemp;
	}

	// TODO: Do low-level read




	return true;
}
/////////////////////////////////// END READPNGLL()
/*
int main(int argc,char* argv[])
{
	printf("MipGen - MipMap Generation Utility\n");
	printf("NeverSoft Entertainment  2002\n\n");

	printf("MipGen [options] [SourceFile] {...}\n");
	printf("[SourceFile] Specifies the source file(s) to use for the conversion\n");
	printf("             You may string together as many source files as necessary to perform the operation\n");
	printf("-d[DestFile]       Specifies the destination file to use for output\n");
	printf("-p[file]           Use the palette information from this image to quantize the other files\n");
	printf("-s[width]x[height] Scale the image to the given size (i.e. 8x8, 16x16, 32x32, 64x64, 128x128, etc.\n");
	printf("\n");

	//bool read_png( char *file_name, png_structp* in_png_ptr, png_infop* in_info_ptr )  /* We need to open the file *//*

	//read_png("asdf",NULL,NULL);

	//ReadPNG("c:\\png\\test32.png");
	//ReadPNG("q:\\sk4\\levels\\lon\\tex\\grass0.png");
	//ReadPNG("c:\\test.png");
	//ReadPNG("c:\\alphatest.png");
	//ReadPNG("c:\\1chan.png");
	ReadPNG("c:\\input.png");

	return 0;
}
*/