#include "image.h"
#include "quant.h"
#include <stdmat.h>
#include <process.h>
#include <stdio.h>
#include <fstream.h>

#define TOP_TRANSPARENT		0x01
#define BOTTOM_TRANSPARENT	0x02
#define LEFT_TRANSPARENT	0x04
#define RIGHT_TRANSPARENT	0x08

void	NxTexture::initialize( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		m_width[i] = 0;
		m_height[i] = 0;
		m_texel_data[i] = NULL;
		m_bpp[i] = 0;
		m_pixel_format[i] = v32_BIT;
	}

	m_grayscale = false;
	m_palette_bpp = 0;	
	m_palette_format = v32_BIT;	
	m_palette_data = NULL;
	sprintf( m_name[0], "None" );
	m_map = NULL;	
	m_mip_levels = 0;
	m_flags = 0;
	m_transparent_color.r = 0.0f;
	m_transparent_color.g = 0.0f;
	m_transparent_color.b = 0.0f;
	m_valid = true;
	m_platform_flags = 0;
	
	for(i=0;i<vMAX_MIP_LEVELS; i++)
		m_useBasePal[i]=FALSE;
}

NxTexture::NxTexture( void )
{
	initialize();
}

NxTexture::~NxTexture( void )
{
	int i;

	for( i = 0; i < vMAX_MIP_LEVELS; i++ )
	{
		if( m_texel_data[i] )
		{			
			delete [] m_texel_data[i];
		}
	}

	if( m_palette_data )
	{
		delete [] m_palette_data;
	}
}

bool NxTexture::ShouldAutoGenerateMipMaps( void )
{
	return (( m_flags & mAUTO_GENERATE_MIPMAPS ) != 0 );
}

void NxTexture::SetNumMipLevels( int num_mip_levels )
{
	m_mip_levels = num_mip_levels;
}

int	NxTexture::GetNumMipLevels( void )
{
	return m_mip_levels;
}

int	NxTexture::GetPixelFormat( void )
{
	return m_pixel_format[0];
}

int	NxTexture::GetPaletteFormat( void )
{
	return m_palette_format;
}

char* NxTexture::GetTexelData( int mip_level )
{
	return m_texel_data[mip_level];
}

char* NxTexture::GetPaletteData( void )
{
	return m_palette_data;
}

void NxTexture::SetTexelData( int mip_level, char* data )
{
	m_texel_data[mip_level] = new char[ GetTexelDataSize( mip_level ) ];
	memcpy( m_texel_data[mip_level], data, GetTexelDataSize( mip_level ));
}

void NxTexture::SetPaletteData( char* data )
{
	memcpy( m_palette_data, data, GetPaletteDataSize());
}

void	NxTexture::SetName( char* name, int level )
{
	assert( name );
	strcpy( m_name[level], name );
}

char*	NxTexture::GetName( int level )
{
	return m_name[level];
}

int	NxTexture::GetWidth( int mip_level )
{
	return m_width[mip_level];
}

int NxTexture::GetHeight( int mip_level )
{
	return m_height[mip_level];
}

void NxTexture::SetWidth( int mip_level, int width )
{
	m_width[mip_level] = width;
}
	
void NxTexture::SetHeight( int mip_level, int height )
{
	m_height[mip_level] = height;
}

bool NxTexture::IsPaletted( void )
{
	return (( m_pixel_format[0] == v8_BIT ) || ( m_pixel_format[0] == v4_BIT ));
}

int NxTexture::GetBpp( void )
{
	return m_bpp[0];
}

int NxTexture::GetPaletteBpp( void )
{
	return m_palette_bpp;
}

int NxTexture::GetNumPaletteEntries( void )
{
	return m_num_palette_entries;
}

int	NxTexture::GetBytesPerRow( int mip_level )
{
	int bytes_per_row;

	bytes_per_row = ( GetWidth( mip_level ) * GetBpp() + 7 ) >> 3;
	return bytes_per_row;
}

int	NxTexture::GetTexelDataSize( int mip_level )
{
	return ( GetHeight( mip_level ) * GetBytesPerRow( mip_level ));	
}

int NxTexture::GetPaletteDataSize( void )
{
	int bits_of_palette_data;

	bits_of_palette_data = GetPaletteBpp() * m_num_palette_entries;
	assert(( bits_of_palette_data % 8 ) == 0 );

	return bits_of_palette_data >> 3;
}

void	NxTexture::create_transparent_mip_edges( int trans_flags, unsigned char trans_index )
{
	int i, j;
	char* dst;
	unsigned char byte;

	assert( GetPaletteFormat() == v32_BIT );
	assert(( GetPixelFormat() == v4_BIT ) || ( GetPixelFormat() == v8_BIT ));

	if( GetPixelFormat() == v4_BIT )
	{			
		for( i = 1; i <= m_mip_levels; i++ )
		{
			if( trans_flags & TOP_TRANSPARENT )
			{		
				byte = trans_index;
				byte |= ( trans_index << 4 );
				dst = m_texel_data[i];	
				for( j = 0; j < (( m_width[i] + 1 ) >> 1 ); j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & BOTTOM_TRANSPARENT )
			{
				byte = trans_index;
				byte |= ( trans_index << 4 );
				dst = m_texel_data[i] + ((( m_width[i] * ( m_height[i] - 1 )) + 1 ) >> 1 );	
				for( j = 0; j < (( m_width[i] + 1 ) >> 1 ); j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & LEFT_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_height[i]; j++ )
				{									
					byte = *dst;
					byte &= 0x0F;
					byte |= ( trans_index << 4 );
					*dst = byte;
					dst += (( m_width[i] + 1 ) >> 1 );
				}			
			}
			if( trans_flags & RIGHT_TRANSPARENT )
			{
				dst = m_texel_data[i] + (( m_width[i] - 1 ) >> 1 );	
				for( j = 0; j < m_height[i]; j++ )
				{									
					byte = *dst;
					byte &= 0xF0;
					byte |= trans_index;
					*dst = byte;
					dst += (( m_width[i] + 1 ) >> 1 );
				}			
			}
		}
	}
	else	// 8-bit
	{
		byte = trans_index;
		for( i = 1; i <= m_mip_levels; i++ )
		{
			if( trans_flags & TOP_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_width[i]; j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & BOTTOM_TRANSPARENT )
			{
				dst = m_texel_data[i] + ( m_width[i] * ( m_height[i] - 1 ));	
				for( j = 0; j < m_width[i]; j++ )
				{									
					*dst++ = byte;
				}
			}
			if( trans_flags & LEFT_TRANSPARENT )
			{
				dst = m_texel_data[i];	
				for( j = 0; j < m_height[i]; j++ )
				{									
					*dst = byte;
					dst += m_width[i];
				}
			}
			if( trans_flags & RIGHT_TRANSPARENT )
			{
				dst = m_texel_data[i] + ( m_width[i] - 1 );	
				for( j = 0; j < m_height[i]; j++ )
				{									
					*dst = byte;
					dst += m_width[i];
				}
			}
		}
	}
}

int NxTexture::get_transparent_edge_flags( void )
{
	int trans_flags;
	bool all_trans;
	char* dst;
	int i;

	assert( GetPaletteFormat() == v32_BIT );
	assert(( GetPixelFormat() == v4_BIT ) || ( GetPixelFormat() == v8_BIT ));

	trans_flags = 0;
	if( GetPixelFormat() == v4_BIT )
	{
		dst = m_texel_data[0];	
		all_trans = true;
		// First, test the top row
		for( i = 0; i < (( m_width[0] + 1 ) >> 1 ); i++ )
		{
			unsigned char byte, lo_nibble, hi_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= TOP_TRANSPARENT;
		}

		// Next, test the bottom row
		dst = m_texel_data[0] + ((( m_width[0] * ( m_height[0] - 1 )) + 1 ) >> 1 );	
		all_trans = true;
		for( i = 0; i < (( m_width[0] + 1 ) >> 1 ); i++ )
		{
			unsigned char byte, lo_nibble, hi_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= BOTTOM_TRANSPARENT;
		}

		// Next, test the left column
		dst = m_texel_data[0];
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte, hi_nibble;

			byte = *dst;
			hi_nibble = byte >> 4;

			if( m_palette_data[ ( hi_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += (( m_width[0] + 1 ) >> 1 );
		}

		if( all_trans )
		{
			trans_flags |= LEFT_TRANSPARENT;
		}

		// Lastly, test the right column
		dst = m_texel_data[0] + (( m_width[0] - 1 ) >> 1 );
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte, lo_nibble;

			byte = *dst;
			lo_nibble = byte & 0x0F;

			if( m_palette_data[ ( lo_nibble * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += (( m_width[0] + 1 ) >> 1 );
		}

		if( all_trans )
		{
			trans_flags |= RIGHT_TRANSPARENT;
		}
	}
	else	// 8-bit
	{
		dst = m_texel_data[0];	
		all_trans = true;
		// First, test the top row
		for( i = 0; i < m_width[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;
			
			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= TOP_TRANSPARENT;
		}

		// Next, test the bottom row
		dst = m_texel_data[0] + ( m_width[0] * ( m_height[0] - 1 ));	
		all_trans = true;
		for( i = 0; i < m_width[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;

			if( m_palette_data[ ( byte* 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}			

			dst++;		
		}

		if( all_trans )
		{
			trans_flags |= BOTTOM_TRANSPARENT;
		}

		// Next, test the left column
		dst = m_texel_data[0];
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;			

			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += m_width[0];
		}

		if( all_trans )
		{
			trans_flags |= LEFT_TRANSPARENT;
		}

		// Lastly, test the right column
		dst = m_texel_data[0] + ( m_width[0] - 1 );
		all_trans = true;
		for( i = 0; i < m_height[0]; i++ )
		{
			unsigned char byte;

			byte = *dst;	

			if( m_palette_data[ ( byte * 4 ) + 3 ] != 0 )
			{
				all_trans = false;
				break;
			}

			dst += m_width[0];
		}

		if( all_trans )
		{
			trans_flags |= RIGHT_TRANSPARENT;
		}
	}

	return trans_flags;
}

bool NxTexture::auto_generate_8_bit_palette_mipmaps( bool has_alpha )
{
	char* path;
	int i, j, k, width, height;
	int trans_flags;
	unsigned char trans_index;

	path = GetName( 0 );
	if( load_png( path, 0 ) == false )
	{
		char error_msg[128];

		sprintf( error_msg, "Could not load %s", path );
		MessageBox( NULL, error_msg, "8-bit PNG Load Error!", MB_OK );
		return false;
	}			
	
	assert( GetPixelFormat() == v8_BIT );

	if( has_alpha )
	{
		trans_flags = get_transparent_edge_flags();
		// If one of the edges is transparent, we'll need to fill the corresponding edges on mips
		// so we'll need a transparent palette entry
		if( trans_flags != 0 )
		{
			for( i = 0; i < m_num_palette_entries; i++ )
			{
				if( m_palette_data[ (i * 4) + 3 ] == 0 )
				{
					trans_index = i;
					break;
				}
			}
		}
	}

	width = m_width[0];
	height = m_height[0];
	for( i = 1; i <= m_mip_levels; i++ )
	{	
		int tex_data_size, dst_idx, src_idx, src_tex_x, src_tex_y;
		char* src_texel;
		float tex_x, tex_y, ratio_x, ratio_y;

		width >>= 1;
		height >>= 1;

		if(	( width < vMIN_MIP_WIDTH ) &&
			( height < vMIN_MIP_HEIGHT ))
		{
			m_mip_levels = i - 1;
			break;
		}
		if( width < vMIN_MIP_WIDTH )
		{
			width = vMIN_MIP_WIDTH;
		}
		if( height < vMIN_MIP_HEIGHT )
		{
			height = vMIN_MIP_HEIGHT;
		}

		m_width[i] = width;
		m_height[i] = height;

		tex_data_size = GetTexelDataSize( i );
		m_texel_data[i] = new char[tex_data_size];

		src_texel = GetTexelData( 0 );
		tex_y = 0.5f;
		dst_idx = 0;
		for( j = 0; j < height; j++ )
		{
			tex_x = 0.5f;
			for( k = 0; k < width; k++ )
			{				
				ratio_x = (float) tex_x / width;
				ratio_y = (float) tex_y / height;
				
				src_tex_x = (int) ( m_width[0] * ratio_x );
				src_tex_y = (int) ( m_height[0] * ratio_y );

				src_idx = ( src_tex_y * m_width[0] ) + src_tex_x;
					 
				m_texel_data[i][dst_idx] = src_texel[src_idx];

				tex_x += 1.0f;
				dst_idx++;
			}
			tex_y += 1.0f;
		}
	}

	if( has_alpha && ( trans_flags != 0 ))
	{
		create_transparent_mip_edges( trans_flags, trans_index );
	}

	//Dump();
	return true;
}

bool NxTexture::auto_generate_4_bit_palette_mipmaps( bool has_alpha )
{
	int size;
	char* dst;
	char* path;
	int i, j, k, width, height;
	int trans_flags;	
	unsigned char trans_index;

	path = GetName( 0 );
	if( load_png( path, 0 ) == false )
	{
		char error_msg[128];

		sprintf( error_msg, "Could not load %s", path );
		MessageBox( NULL, error_msg, "4-bit PNG Load Error!", MB_OK );
		return false;
	}			
	
	assert( GetPixelFormat() == v4_BIT );

	// Earlier, we swapped the nibbles to be in the order that the target platforms wanted them.
	// But here, before we down-sample, temporarily revert the nibble ordering
	size = GetTexelDataSize( 0 );
	dst = m_texel_data[0];
	for( i = 0; i < size; i++ )
	{
		unsigned char byte, lo_nibble;
		
		byte = *dst;
		lo_nibble = byte & 0x0F;
		byte >>= 4;
		byte |= ( lo_nibble << 4 );
		*dst++ = byte;
	}

	if( has_alpha )
	{
		// Figure out if any of the four edges of this texture are completely transparent. If so, 
		// the mips need to be transparent on those edges as well just in case the artist wants to
		// use clamping on this texture
		trans_flags = get_transparent_edge_flags();
		// If one of the edges is transparent, we'll need to fill the corresponding edges on mips
		// so we'll need a transparent palette entry
		if( trans_flags != 0 )
		{
			for( i = 0; i < m_num_palette_entries; i++ )
			{
				if( m_palette_data[ (i * 4) + 3 ] == 0 )
				{
					trans_index = i;
					break;
				}
			}
		}
	}

	width = m_width[0];
	height = m_height[0];
	for( i = 1; i <= m_mip_levels; i++ )
	{	
		int tex_data_size, dst_idx, src_idx, src_tex_x, src_tex_y, src_byte, dst_byte;
		char* src_texel;
		unsigned char src_color;
		float tex_x, tex_y, ratio_x, ratio_y;

		width >>= 1;
		height >>= 1;

		if(	( width < vMIN_MIP_WIDTH ) &&
			( height < vMIN_MIP_HEIGHT ))
		{
			m_mip_levels = i - 1;
			break;
		}			
		if( width < vMIN_MIP_WIDTH )
		{
			width = vMIN_MIP_WIDTH;
		}
		if( height < vMIN_MIP_HEIGHT )
		{
			height = vMIN_MIP_HEIGHT;
		}
		
		m_width[i] = width;
		m_height[i] = height;

		tex_data_size = GetTexelDataSize( i );
		m_texel_data[i] = new char[tex_data_size];

		// Clear out the texel data
		for( j = 0; j < tex_data_size; j++ )
		{
			m_texel_data[i][j] = 0;
		}

		src_texel = GetTexelData( 0 );
		tex_y = 0.5f;
		dst_idx = 0;
		for( j = 0; j < height; j++ )
		{
			tex_x = 0.5f;
			for( k = 0; k < width; k++ )
			{	
				ratio_x = (float) tex_x / width;
				ratio_y = (float) tex_y / height;
				
				src_tex_x = (int) ( m_width[0] * ratio_x );
				src_tex_y = (int) ( m_height[0] * ratio_y );

				src_idx = ( src_tex_y * m_width[0] ) + src_tex_x;

				src_byte = src_idx >> 1;	// 4 bits per texel, so byte index is halved
				dst_byte = dst_idx >> 1;	// 4 bits per texel, so byte index is halved
					 
				// Decide which nibble to access
				if( src_idx % 2 )
				{
					src_color = src_texel[src_byte];
					src_color &= 0xF;
				}
				else
				{
					src_color = src_texel[src_byte];
					src_color >>= 4;					
				}

				// Decide which nibble to write to
				if( dst_idx % 2 )
				{
					m_texel_data[i][dst_byte] |= src_color;
				}
				else
				{
					m_texel_data[i][dst_byte] |= ( src_color << 4 );					
				}				

				tex_x += 1.0f;
				dst_idx++;
			}
			tex_y += 1.0f;
		}
	}
	
	if( has_alpha && ( trans_flags != 0 ))
	{
		create_transparent_mip_edges( trans_flags, trans_index );
	}

	// Now swap nibble order so that they are in the order that the targets expect them
	for( i = 0; i <= m_mip_levels; i++ )
	{
		size = GetTexelDataSize( i );
		dst = m_texel_data[i];
	
		for( j = 0; j < size; j++ )
		{
			unsigned char byte, lo_nibble;
			
			byte = *dst;
			lo_nibble = byte & 0x0F;
			byte >>= 4;
			byte |= ( lo_nibble << 4 );
			*dst++ = byte;
		}
	}

	//Dump();
	return true;
}

bool NxTexture::LoadImage( void )
{
	char ext[_MAX_EXT];
	int i;
	char* path;

	path = GetName( 0 );

	_splitpath( path, NULL, NULL, NULL, ext );
	if( stricmp( ext, ".png" ) == 0 )
	{
		if( m_mip_levels == 0 )
		{
			if( load_png( path, 0 ) == false )
			{
				char error_msg[128];
			
				sprintf( error_msg, "Could not load %s", path );
				MessageBox( NULL, error_msg, "LoadImage Error!", MB_OK );
				return false;
			}				
		}
		else
		{
			char palette_path[_MAX_PATH];
			char ult_mip_path[_MAX_PATH];
			char temp_mip_path[_MAX_PATH];
			char mip_path[_MAX_PATH];
			char drive[_MAX_DRIVE];
			char cur_path[_MAX_PATH];
			char cur_file[_MAX_FNAME];			
			int width, height;
			NxTexture tmp_tex;
			bool should_regen;
			
			if( !tmp_tex.load_png( path, 0 ))
			{
				return false;
			}

			
			// First, we need to load the image to see if it's paletted and has alpha. We handle
			// these differently
			if( ( ShouldAutoGenerateMipMaps()) &&
				( tmp_tex.IsPaletted()))								
			{				
				if(( tmp_tex.GetPaletteFormat() == v32_BIT ) || tmp_tex.IsGrayscale())
				{
					if( tmp_tex.GetPixelFormat() == v8_BIT )
					{			
						return auto_generate_8_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
					}
					else if( tmp_tex.GetPixelFormat() == v4_BIT )
					{
						return auto_generate_4_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
					}
				}
			}			

			_splitpath( path, drive, cur_path, cur_file, NULL );
			sprintf( ult_mip_path, "%s%s\\mips", drive, cur_path );
			CreateDirectory( ult_mip_path, NULL );
			
			if( ShouldAutoGenerateMipMaps())
			{
				sprintf( mip_path, "c:/temp/%s_autom0%s", cur_file, ext );
				// Copy the highest level of detail to the mips directory
				CopyFile( path, mip_path, false );
			}
			width = tmp_tex.m_width[0];		// aml
			height = tmp_tex.m_height[0];	// aml
			m_width[0]  = width;
			m_height[0] = height;

			should_regen = false;
			for( i = 1; i <= m_mip_levels; i++ )
			{	
				width >>= 1;
				height >>= 1;

				if(	( width < vMIN_MIP_WIDTH ) &&
					( height < vMIN_MIP_HEIGHT ))
				{
					m_mip_levels = i - 1;
					break;
				}			

				if( width < vMIN_MIP_WIDTH )
				{
					width = vMIN_MIP_WIDTH;
				}
				if( height < vMIN_MIP_HEIGHT )
				{
					height = vMIN_MIP_HEIGHT;
				}
				
				// First generate the mips, if they need to be generated
				if( ShouldAutoGenerateMipMaps())
				{
					sprintf( ult_mip_path, "%s%smips\\%s_autom%d%s", drive, cur_path, cur_file, i, ext );
					sprintf( temp_mip_path, "c:/temp/%s_autom%d%s", cur_file, i, ext );

					// Only generate it if it's actually newer
					{
						int result;
						char* args[11];
						char width_str[32], height_str[32];
						char alch_path[_MAX_PATH];
						char* project_root;

						should_regen = true;
						project_root = "c:\\";
						if( project_root == NULL )
						{
							MessageBox( NULL, "You must set up your SKATE4_PATH environment variable",
											"Error!", MB_OK|MB_TASKMODAL);
							return false;
						}

						sprintf( alch_path, "%s\\bin\\win32\\alchlong.exe", project_root );
						sprintf( width_str, "%d", width );
						sprintf( height_str, "%d", height );
						args[0] = alch_path;
						args[1] = "---n";
						args[2] = "-o";
						args[3] = "-d4";	// Stevenson & Arce dithering
						args[4] = "-Xd";	// Lanczos filtering when scaling X
						args[5] = width_str;
						args[6] = "-Yd";	// Lanczos filtering when scaling Y
						args[7] = height_str;
						args[8] = path;
						args[9] = temp_mip_path;
						args[10] = NULL;

						result = ( spawnv( _P_WAIT, args[0], args ) == 0 );				
					}
				}				
			}

			if( ShouldAutoGenerateMipMaps())
			{
				sprintf( palette_path, "%s%smips\\auto%s.pal", drive, cur_path, cur_file );			
				// Only generate palette and re-palettize if palette is out of date
				//if( FileIsNewer( palette_path, path ))
				if( should_regen )
				{
					int result;
					char* pal_args[10];
					char pal_file[_MAX_PATH];

					char* first_args[7];
					char* second_args[10];
					char mip_wildcard[_MAX_PATH];
					char dest_dir[_MAX_PATH];
					char alch_path[_MAX_PATH];
					char* project_root;

					project_root = "c:\\";
					if( project_root == NULL )
					{
						MessageBox( NULL, "You must set up your SKATE4_PATH environment variable",
										"Error!", MB_OK|MB_TASKMODAL);
						return false;
					}

					sprintf( alch_path, "%s\\bin\\win32\\alchlong.exe", project_root );

					// Create palette file for the basetexture
					sprintf( pal_file, "%s%smips\\autoBase%s.pal", drive, cur_path, cur_file );
						
					pal_args[0] = alch_path;
					pal_args[1] = "-o";
					pal_args[2] = path;
					pal_args[3] = "-l";
					pal_args[4] = pal_file;
					pal_args[5] = NULL;

					result = ( spawnvp( _P_WAIT, alch_path, pal_args ) == 0 );

					// Create a generic palette for all the mips
					sprintf( mip_wildcard, "c:\\temp\\%s_autom*%s", cur_file, ext );
					first_args[0] = alch_path;
					first_args[1] = "-o";
					first_args[2] = "--";
					first_args[3] = mip_wildcard;				
					first_args[4] = "-L";
					first_args[5] = palette_path;				
					first_args[6] = NULL;

					result = ( spawnvp( _P_WAIT, first_args[0], first_args ) == 0 );				

					// Now apply the created palette
					sprintf( dest_dir, "%s%smips\\", drive, cur_path );
					second_args[0] = alch_path;
					second_args[1] = "-o";
					second_args[2] = "---n";
					second_args[3] = "-8";
					second_args[4] = "--";
					second_args[5] = mip_wildcard;
					second_args[6] = "-f";

					// Apply the appropriate palette file
					second_args[7] = palette_path;

					for(int i=0;i<m_mip_levels;i++)
					{
						if (UsesBasePal(i))
						{
							second_args[7] = pal_file;
							break;
						}
					}

					second_args[8] = dest_dir;
					second_args[9] = NULL;

					result = ( spawnvp( _P_WAIT, second_args[0], second_args ) == 0 );
				}

				// Now load each mip level
				for( i = 0; i <= m_mip_levels; i++ )
				{
					sprintf( mip_path, "%s%smips\\%s_autom%d%s", drive, cur_path, cur_file, i, ext );
					if( load_png( mip_path, i ) == false )
					{
						return false;
					}			
				
					// If any of the images we load are grayscale, abort the alchemy path. Auto-generate
					// the mips ourselves. Alchemy does a terrible job on grayscale images
					if( IsGrayscale())
					{
						if( tmp_tex.GetPixelFormat() == v8_BIT )
						{			
							return auto_generate_8_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
						}
						else if( tmp_tex.GetPixelFormat() == v4_BIT )
						{
							return auto_generate_4_bit_palette_mipmaps( tmp_tex.GetPaletteFormat() == v32_BIT );						
						}					
					}

					if( i > 0 )
					{
						int expwidth  = m_width[i-1] >> 1;
						int expheight = m_height[i-1] >> 1;

						if (expwidth<vMIN_MIP_WIDTH)
							expwidth=vMIN_MIP_WIDTH;

						if (expheight<vMIN_MIP_HEIGHT)
							expheight=vMIN_MIP_HEIGHT;

						if( ( m_width[i] != expwidth) ||
							( m_height[i] != expheight))
						{
							char error_msg[128];

							sprintf( error_msg, "Mipmap %s is not half-size of its parent in each dimension!\nExpected Width: %i\nExpected Height: %i\nActual Width: %i\nActual Height: %i\nMip Level: %i\nMip Path: %s", 
								path, 
								m_width[i - 1] >> 1,
								m_height[i - 1] >> 1,
								m_width[i],
								m_height[i],
								i,
								mip_path
							);

							MessageBox( NULL, error_msg, "MipMap Error", MB_OK|MB_ICONWARNING );							
						}
					}
				}
			}			
			else
			{
				NxTexture	mips[vMAX_MIP_LEVELS];
				NxTexture*	all_levels[vMAX_MIP_LEVELS];
				int prev_width, prev_height;

				if( load_png( GetName(0), 0 ) == false )
				{
					char error_msg[128];
		
					sprintf( error_msg, "Could not load %s", GetName(i));
					MessageBox( NULL, error_msg, "Mip Load Error!", MB_OK );
					return false;
				}

				prev_width = m_width[0];
				prev_height = m_height[0];
				all_levels[0] = this;
				// Now load each mip level
				for( i = 1; i <= m_mip_levels; i++ )
				{
					if( mips[i-1].load_png( GetName(i), 0 ) == false )
					{
						char error_msg[128];
			
						sprintf( error_msg, "Could not load %s", GetName(i));
						MessageBox( NULL, error_msg, "Mip Load Error!", MB_OK );
						return false;
					}

					if( mips[i-1].GetBpp() != GetBpp())
					{
						char error_msg[128];
			
						sprintf( error_msg, "Mipmap %s is not in the same format as its basemap : %s", GetName(i), GetName( 0 ));
						MessageBox( NULL, error_msg, "Mip Format Error!", MB_OK );
						return false;
					}

					all_levels[i] = &mips[i-1];
					if( ( mips[i-1].m_width[0] != ( prev_width >> 1 )) ||
						( mips[i-1].m_height[0] != ( prev_height >> 1 )))
					{
						char error_msg[128];

						sprintf( error_msg, "Mipmap %s is not half-size of its parent in each dimension!\nExpected Width: %i\nExpected Height: %i\nActual Width: %i\nActual Height: %i\nMip Level: %i\nMip Path: %s\nMip Name: %s", 
							path, 
							prev_width >> 1,
							prev_height >> 1,
							mips[i-1].m_width[0],
							mips[i-1].m_height[0],
							i,
							mip_path,
							(char*)GetName(i)
							);

						MessageBox( NULL, error_msg, "Half-size MipMap Error", MB_OK|MB_ICONWARNING );							
						return false;
					}										

					prev_width = mips[i-1].m_width[0];
					prev_height = mips[i-1].m_height[0];
				}

				if( NxPalettizeMips( all_levels, m_mip_levels + 1 ) == false )
				{
					char error_msg[128];

					sprintf( error_msg, "Internal palette quantizer error on %s. Notify Steve\n", path );
					MessageBox( NULL, error_msg, "Quantizer MipMap Error", MB_OK|MB_ICONWARNING );							
					return false;
				}
			}
		}
		return true;
	}
	else
	{
		char error_msg[256];
		sprintf(error_msg,"The file format is unsupported for '%s'.  You must use a valid .PNG format file.",path);

		MessageBox(NULL, error_msg, "Bitmap format unrecognized",MB_ICONWARNING|MB_OK);
	}

	return false;
}

BitmapInfo*	NxTexture::GetMap( void )
{
	return m_map;
}

void	NxTexture::SetMap( BitmapInfo* map )
{
	m_map = map;
}

bool	NxTexture::ShouldCompressComponents( void )
{
	return (( m_flags & mFORCE_BYTE_PER_COMPONENT ) == 0 );
}

bool	NxTexture::ShouldChangeTransparentColor( void )
{
	return (( m_flags & mCHANGE_FULLY_TRANSPARENT_COLOR ) != 0 );
}

Color	NxTexture::GetTransparentColor( void )
{
	return m_transparent_color;
}

void	NxTexture::SetTransparentColor( Color trans_color )
{
	m_transparent_color = trans_color;
}

void	NxTexture::SetFlags( int flags )
{
	m_flags |= flags;
}

int		NxTexture::GetFlags( void )
{
	return m_flags;
}

#define FILENAME_ONLY	// Garrett: removes the directory path before the checksum calculation

void	NxTexture::GenerateChecksum( void )
{
	m_checksum = 0;
	if( m_map == NULL )
	{		
		return;
	}

#ifdef FILENAME_ONLY
	// Find base name
	const char *pFileName = GetName(0);
	int idx	= strlen(pFileName);
	while ((idx > 0) && pFileName[idx - 1] != '\\' && pFileName[idx - 1] != '/')
		--idx;

	const char *p_base_name = &(pFileName[idx]);
	m_checksum = 0;
#else
	m_checksum = 0;		
#endif
}

unsigned long	NxTexture::GetChecksum( void )
{
	return m_checksum;
}

bool	NxTexture::operator==( NxTexture& texture )
{
	return( m_checksum == texture.GetChecksum());
}

int	NxTexture::GetTotalDataSize( void )
{
	int i;
	int total_size;

	total_size = GetPaletteDataSize();
	for( i = 0; i <= GetNumMipLevels(); i++ )
	{
		total_size += GetTexelDataSize( i );
	}

	return total_size;
}

void	NxTexture::SetValidity( bool valid )
{
	m_valid = valid;
}

bool	NxTexture::IsValid( void )
{
	return m_valid;
}

bool	NxTexture::IsGrayscale( void )
{
	return m_grayscale;
}

void	NxTexture::SetPlatformFlags( int flags )
{
	m_platform_flags = flags;
}

int		NxTexture::GetPlatformFlags( void )
{
	return m_platform_flags;
}

BOOL	NxTexture::UsesBasePal(int mip)
{
	return m_useBasePal[mip];
}

void	NxTexture::SetUseBasePal(int mip,BOOL bUseBasePal)
{
	m_useBasePal[mip] = bUseBasePal;
}

void	NxTexture::Dump( void )
{
	fstream file;
	int i, size, offset;
	char filename[_MAX_PATH];

	// For now, only write out 24-bit textures
	if( GetPaletteFormat() != v24_BIT )
	{
		return;
	}

	for( i = 0; i <= m_mip_levels; i++ )
	{
		unsigned short field;
		int j;
		char* pal;

		sprintf( filename, "c:/temp/auto%d.bmp", i );
		file.open( filename, ios::out | ios::binary );

		field = 0x4d42;
		file.write((const char*) &field, sizeof( unsigned short ));

		size = 26 /*combined header size*/ + GetTexelDataSize( i ) + GetPaletteDataSize();
		file.write((const char*) &size, sizeof( int ));

		field = 0;
		file.write((const char*) &field, sizeof( unsigned short ));
		file.write((const char*) &field, sizeof( unsigned short ));

		offset = 26 /*combined header size*/ + GetPaletteDataSize();
		file.write((const char*) &offset, sizeof( int ));

		size = 12;	// header size
		file.write((const char*) &size, sizeof( unsigned int ));

		field = GetWidth( i );
		file.write((const char*) &field, sizeof( unsigned short ));

		field = GetHeight( i );
		file.write((const char*) &field, sizeof( unsigned short ));

		field = 1;	// num planes
		file.write((const char*) &field, sizeof( unsigned short ));
	
		field = GetBpp();
		file.write((const char*) &field, sizeof( unsigned short ));

		pal = GetPaletteData();
		for( j = 0; j < GetNumPaletteEntries(); j++ )
		{
			file.write((const char*) &pal[( j * 3 ) + 2], sizeof( char ));
			file.write((const char*) &pal[( j * 3 ) + 1], sizeof( char ));
			file.write((const char*) &pal[(j * 3 )], sizeof( char ));
		}

		file.write((const char*) GetTexelData( i ), GetTexelDataSize( i ));
		file.close();
	}	
}