#include "FuncEnter.h"

/*
	WibbleWatcher.cpp
	Provides functionality to verify wibble data integrity
	aml - 11-7-02
*/

#include "WibbleWatcher.h"
#include "Wibble/Wibble.h"
#include "DebugSrc/DebugTool.h"
#include "UI/OKtoAll.h"

#define MAXNUM_CHECKED_NODES 5

extern Interface* gInterface;
static bool bWibbleWarn = true;
static UINT timerID = 0;
static bool bLockWibbleTimer = false;

extern void PurgeWibbleDataSel();
extern void WibbleRepairSel();

#define POLYCONV_WARNING   "!! Poly Conversion Exception !!"

#define CHECK_POLY_CONVERSION

INode* gLastPolyConvErrorNode = NULL;

void ResetWibbleWatcher()
{ FUNC_ENTER("ResetWibbleWatcher"); 
	bWibbleWarn = true;
}

void CALLBACK WibbleWatcherTimerProc(HWND hwnd, UINT msg, UINT id, DWORD time)
{ FUNC_ENTER("WibbleWatcherTimerProc"); 
	if (bLockWibbleTimer)
		return;

	bLockWibbleTimer = true;
	
	int nNodes = gInterface->GetSelNodeCount();

	if (nNodes > MAXNUM_CHECKED_NODES)
		nNodes = MAXNUM_CHECKED_NODES;

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = gInterface->GetSelNode(i);

		Object* obj = node->EvalWorldState(gInterface->GetTime()).obj;

		if (obj)
		{
			if (obj->CanConvertToType(triObjectClassID))
			{
				TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
				Mesh& mesh = triObj->GetMesh();

				VerifyWibbleIntegrity(mesh, "Periodic", (char*)node->GetName());

				if (triObj != obj)
					triObj->DeleteThis();
			}

#ifdef CHECK_POLY_CONVERSION
			// Attempt to get a polygonal representation if allowed.  Seems to be
			// some problems occuring when this happens this should help us detect
			// this when (or very shortly after it occurs)
			if (obj && obj->CanConvertToType(polyObjectClassID))
			{
				PolyObject* polyobj = NULL;

				try
				{
					polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);
				}
				catch(...)
				{
					if (gLastPolyConvErrorNode != node)
					{
						if (MessageExists(POLYCONV_WARNING))
						{
							gLastPolyConvErrorNode = node;
							//gInterface->ClearNodeSelection();
						}

						char msg[2048];
						sprintf(msg, "WARNING!  An exception has occurred while trying to convert the currently selected node '%s' into a polygonal representation.\n!!! STOP!  AND MAKE NOTE OF THE LAST OPERATIONS YOU PERFORMED ON THIS GEOMETRY NOW !!!\nAnd let Adam/Jason know so we can track down this problem.  Save your work (in a new file) the system may be unstable at this point.\nThanks,\nYour friendly neighborhood exception handler.", (char*)node->GetName());
						MessageBoxAll(gInterface->GetMAXHWnd(), msg, "!! Poly Conversion Exception !!", MB_ICONSTOP|MB_OK);
					}
				}

				if (polyobj && polyobj != obj)
					polyobj->DeleteThis();
			}
#endif

		}
	}

	bLockWibbleTimer = false;
}

void InstallPeriodicWibbleWatcher()
{ FUNC_ENTER("InstallPeriodicWibbleWatcher"); 
	// We will check the currently selected meshes (up to the first 5) every second
	if (timerID != 0)
		KillTimer(NULL, timerID);
	
	timerID = SetTimer(NULL, 0, 1000, WibbleWatcherTimerProc);
}

void UnInstallPeriodicWibbleWatcher()
{ FUNC_ENTER("UnInstallPeriodicWibbleWatcher"); 
	if (timerID != 0)
		KillTimer(NULL, timerID);
}

bool VerifyWibbleIntegrity(Mesh& mesh, char* strMode, char* strNode, int line, char* strFile)
{ FUNC_ENTER("VerifyWibbleIntegrity"); 
	if (!bWibbleWarn)
		return true;

	int nFaces = mesh.numFaces;
	int nVerts = mesh.getNumMapVerts( vWIBBLE_INDEX_CHANNEL );

	UVVert* verts = mesh.mapVerts( vWIBBLE_INDEX_CHANNEL );
	TVFace* faces = mesh.mapFaces( vWIBBLE_INDEX_CHANNEL );

	if (!verts || !faces)
		return true;

	// Ensure that all wibble data is within range
	for(int cface = 0; cface < nFaces; cface++)
	{
		for(int cvert = 0; cvert < 3; cvert++)
		{
			if (faces[cface].t[cvert] < 0 ||
				faces[cface].t[cvert] > nVerts)
			{
				if (strcmp(strMode, "Periodic")==0)
				{
					// Temporarily disable the timer
					UnInstallPeriodicWibbleWatcher();
				}

				char sErr[512];
				sprintf(sErr, "%s Wibble Wacked (INDEX) @ face %i vert %i of %i.\nNode: %s\nLine: %i\nFile: %s\n\nContact Adam\nYou may cancel to disable future warnings but you need to purge wibble data within the current selection set. Go to Debug Tool to reenable warnings if you cancel.  Do you want to fix/purge wibble data now?", strMode, cface, cvert, nVerts, strNode, line, strFile);
				int id = MessageBox(gInterface->GetMAXHWnd(), sErr, "Wibble Watcher", MB_ICONSTOP|MB_YESNOCANCEL);

				if (id == IDCANCEL)
					bWibbleWarn = false;

				if (id == IDYES)
				{
					id = MessageBox(gInterface->GetMAXHWnd(), "Do you want to fix wibbles in selection? (No will purge)", "Wibble Watcher", MB_ICONQUESTION|MB_YESNOCANCEL);

					if (id == IDCANCEL)
						bWibbleWarn = false;

					if (id == IDNO)
						PurgeWibbleDataSel();

					if (id == IDYES)
						WibbleRepairSel();
				}

				if (strcmp((char*)strMode, "Periodic")==0)
				{
					// Reenable the timer
					InstallPeriodicWibbleWatcher();
				}

				return false;
			}
		}
	}

	nVerts = mesh.getNumMapVerts( vWIBBLE_OFFSET_CHANNEL );
	verts  = mesh.mapVerts( vWIBBLE_OFFSET_CHANNEL );
	faces  = mesh.mapFaces( vWIBBLE_OFFSET_CHANNEL );
	
	if (!verts || !faces)
		return true;

	for(cface = 0; cface < nFaces; cface++)
	{
		for(int cvert = 0; cvert < 3; cvert++)
		{
			if (faces[cface].t[cvert] < 0 ||
				faces[cface].t[cvert] > nVerts)
			{
				if (strcmp(strMode, "Periodic")==0)
				{
					// Temporarily disable the timer
					UnInstallPeriodicWibbleWatcher();
				}

				char sErr[512];
				sprintf(sErr, "%s Wibble Wacked (OFFSET) @ face %i vert %i of %i.\nNode: %s\nLine: %i\nFile: %s\nContact Adam\nYou may cancel to disable future warnings but you need to purge wibble data within the current selection set. Go to Debug Tool to reenable warnings if you cancel.  Do you want to fix/purge wibble data now?", strMode, cface, cvert, nVerts, strNode, line, strFile);
				int id = MessageBox(gInterface->GetMAXHWnd(), sErr, "Wibble Watcher", MB_ICONSTOP|MB_YESNOCANCEL);

				if (id == IDCANCEL)
					bWibbleWarn = false;

				if (id == IDYES)
				{
					id = MessageBox(gInterface->GetMAXHWnd(), "Do you want to fix wibbles in selection? (No will purge)", "Wibble Watcher", MB_ICONQUESTION|MB_YESNOCANCEL);

					if (id == IDCANCEL)
						bWibbleWarn = false;

					if (id == IDNO)
						PurgeWibbleDataSel();

					if (id == IDYES)
						WibbleRepairSel();
				}

				if (strcmp(strMode, "Periodic")==0)
				{
					// Reenable the timer
					InstallPeriodicWibbleWatcher();
				}

				return false;
			}
		}
	}

	return true;
}
