/*
	ThumbnailTree.cpp
	Thumbnail TreeView control
	
	Custom Drawn TreeView control that displays thumbnails next to each entry
	and supports multiple selections

	9-10-03 - aml
*/

#include "ThumbNailTree.h"
#include "../ImageData.h"
#include "../Misc/Util.h"
#include <commctrl.h>
#include <assert.h>

// This is allocated for each created node in the tree and contains additional
// data that the ThumbNailTree uses to render each entry in the hierarchy
struct TNTEntryData
{
	bool bSelected;				// True if we're considering this item part of the current selection
								// TV doesn't support multiple selections but we're adding this in ourselves

	bool bNoRender;				// Shouldn't be rendered if set

	ImageData img;
	char      imgFilename[256];	// Filename of the image being displayed for this entry
	HWND      hwndCombo;
	void*     extdata;			// The data added by a user of the control
	
	HANDLE    hLoadThread;		// Handle for the thread that's loading the image
	bool      bLoaded;			// True if the image has been loaded into memory (img is valid)

	HWND      hwndTree;			// The tree that contains the entry that the thread is currently processing
	HTREEITEM hItem;			// handle of the entry being loaded in the tree

	TNTEntryData()
	{
		imgFilename[0] = 0;

		hwndCombo   = NULL;
		extdata     = NULL;

		hLoadThread = NULL;
		bLoaded     = false;
		bNoRender   = false;

		bSelected   = false;
	}
};

struct TreeData
{
	WNDPROC        wndproc;			// Original wndproc for the subclassed tree
	bool           bSelPointMove;	// True if the user has just moved the selection point with the arrow keys
	bool           bSelPointClear;	// Clear old selections (false if SHIFT is depressed)
	ThumbNailTree* pthis;			// Pointer to the tree object that wraps the tree window this data is for

	TreeData()
	{
		wndproc        = NULL;
		bSelPointMove  = false;
		bSelPointClear = true;
	}
};

ThumbNailTree::ThumbNailTree(HINSTANCE hInstance) :
	UIControl(hInstance, CLASS)
{
	fpSelChange    = NULL;
	pSelChangeData = NULL;
}

ThumbNailTree::ThumbNailTree(HINSTANCE hInstance, HWND hwnd) :
	UIControl(hInstance, CLASS)
{
	fpSelChange    = NULL;
	pSelChangeData = NULL;
}

	/*
ThumbNailTree::ThumbNailTree(ThumbNailTree& right) :
	UIControl(right.hInstance, CLASS)
{

}

ThumbNailTree& ThumbNailTree::operator = (ThumbNailTree& right)
{

	return *this;
}
	*/

ThumbNailTree::~ThumbNailTree()
{
	// Uninstall subclass
	TreeData* tdata = (TreeData*)GetWindowLong(hwndTree, GWL_USERDATA);
	SetWindowLong(hwndTree, GWL_WNDPROC, (LONG)tdata->wndproc);
	DestroyWindow(hwndTree);

	delete tdata;
}

DWORD WINAPI ThumbNailTree::LoadImageThread(void* pData)
{
	TNTEntryData* pEntry = (TNTEntryData*)pData;
	ImageData img;
	
	if (!img.ReadPNG((char*)pEntry->imgFilename, NULL))
	{
		pEntry->bLoaded = true;
		delete pData;
		return 0;
	}

	img.Convert24(&pEntry->img);
	SwapRGB(pEntry->img.buf, pEntry->img.imageSize);
	Flip(pEntry->img.buf, pEntry->img.width, pEntry->img.height);
	pEntry->bLoaded = true;

	// Redraw the control once the image finishes loading
	RECT rect;
	*(HTREEITEM*)&rect = pEntry->hItem;
	SendMessage(pEntry->hwndTree, TVM_GETITEMRECT, (WPARAM)FALSE, (LPARAM)&rect);
	InvalidateRect(pEntry->hwndTree, &rect, FALSE);
	return 1;
}

DWORD WINAPI FreeThread(void* pData)
{
	// This thread waits for loading to complete and then frees any data the
	// load thread allocated to prevent the system from having to wait for loading
	// to complete to shutdown the ThumbNailTree.  Can't simply terminate the
	// load thread because this seems to put file I/O and libPNG into a bad state
	// and subsequent calls end up putting future load threads into deadlock
	TNTEntryData* pEntry = (TNTEntryData*)pData;
	while(!pEntry->bLoaded) {}	// TODO: Update to event signal objects

	delete pEntry;
	return 0;
}

void ClearItemSelections(HWND hwndTree, HTREEITEM hItem)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	assert(pEntry);

	pEntry->bSelected = false;

	// Process any children of this item
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);
		if (hItemChild)
			ClearItemSelections(hwndTree, hItemChild);
	}

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hItem, TVGN_NEXT);

	if (hItemSibling)
		ClearItemSelections(hwndTree, hItemSibling);
}

int GetSelCount(HWND hwndTree, HTREEITEM hItem)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	assert(pEntry);

	int selCount = 0;

	// Process any children of this item
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);
		if (hItemChild)
			selCount += GetSelCount(hwndTree, hItemChild);
	}

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hItem, TVGN_NEXT);

	if (hItemSibling)
		selCount += GetSelCount(hwndTree, hItemSibling);

	if (pEntry->bSelected)
		selCount++;

	return selCount;
}

int GetSelCount(HWND hwndTree)
{
	HTREEITEM hItem = TreeView_GetRoot(hwndTree);	
	return GetSelCount(hwndTree, hItem);
}

int   ThumbNailTree::GetSelCount()
{
	return ::GetSelCount(hwndTree);
}

void ClearSelectionsByHWND(HWND hwndTree)
{
	HTREEITEM hItem;
	hItem = TreeView_GetRoot(hwndTree);

	if (hItem)
		ClearItemSelections(hwndTree, hItem);
}

void ThumbNailTree::ClearSelections()
{
	ClearSelectionsByHWND(hwndTree);
}

void ClearItemsChildSib(HWND hwndTree, HTREEITEM hItem)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;

	// Process any children of this item
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);
		if (hItemChild)
			ClearItemsChildSib(hwndTree, hItemChild);
	}

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hItem, TVGN_NEXT);

	if (hItemSibling)
		ClearItemsChildSib(hwndTree, hItemSibling);

	if (pEntry)
	{
		// We're not allowed to free this until any threads operating on it
		// have completed.  To prevent waiting we'll have a background thread
		// do this, and remove the entry immediately
		tvitem.mask   = TVIF_PARAM;
		tvitem.lParam = NULL;
		TreeView_SetItem(hwndTree, &tvitem);

		if (pEntry->bLoaded || pEntry->imgFilename[0] == 0)
			delete pEntry;
		else
			CreateThread(NULL, 0, FreeThread, pEntry, 0, NULL);
	}
}

void ClearItemsChild(HWND hwndTree, HTREEITEM hItem)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;

	// Process any children of this item
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);
		if (hItemChild)
			ClearItemsChildSib(hwndTree, hItemChild);
	}

	if (pEntry)
	{
		// We're not allowed to free this until any threads operating on it
		// have completed.  To prevent waiting we'll have a background thread
		// do this, and remove the entry immediately
		tvitem.mask   = TVIF_PARAM;
		tvitem.lParam = NULL;
		TreeView_SetItem(hwndTree, &tvitem);

		if (pEntry->bLoaded || pEntry->imgFilename[0] == 0)
			delete pEntry;
		else
			CreateThread(NULL, 0, FreeThread, pEntry, 0, NULL);
	}
}

void ClearItems(HWND hwndTree)
{
	HTREEITEM hItem;
	hItem = TreeView_GetRoot(hwndTree);

	if (hItem)
	{
		ClearItemsChildSib(hwndTree, hItem);
		TreeView_DeleteAllItems(hwndTree);
	}
}

void ThumbNailTree::Clear()
{
	ClearItems(hwndTree);
}

void ThumbNailTree::RemoveNode(void* pData)
{
	HTREEITEM hItem = (HTREEITEM)pData;

	// Clean up our attached data
	ClearItemsChild(hwndTree, hItem);
	TreeView_DeleteItem(hwndTree, hItem);
}

bool ThumbNailTree::TraverseTreeItems(void* hItem, bool (*fpFunc)(ThumbNailTree*, void*, void*), void* pData)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = (HTREEITEM)hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	assert(pEntry);

	// Process any children
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);

		if (hItemChild)
			if (!TraverseTreeItems(hItemChild, fpFunc, pData))
				return false;
	}

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hItem, TVGN_NEXT);

	if (hItemSibling)
		if (!TraverseTreeItems(hItemSibling, fpFunc, pData))
			return false;

	assert(fpFunc);
	return fpFunc(this, (void*)hItem, pData);
}

bool ThumbNailTree::TraverseTree(bool (*fpFunc)(ThumbNailTree*, void* , void*), void* pData)
{
	HTREEITEM hRootItem = TreeView_GetRoot(hwndTree);
	
	return TraverseTreeItems(hRootItem, fpFunc, pData);
}

void GetSelItems(HWND hwndTree, HTREEITEM hItem, HTREEITEM* hItemArray, int* pnItems, int itemMax)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	assert(pEntry);

	int selCount = 0;

	// Process any children of this item
	if (tvitem.cChildren)
	{
		HTREEITEM hItemChild = TreeView_GetNextItem(hwndTree, hItem, TVGN_CHILD);

		if (hItemChild)
			GetSelItems(hwndTree, hItemChild, hItemArray, pnItems, itemMax);
	}

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hItem, TVGN_NEXT);

	if (hItemSibling)
		GetSelItems(hwndTree, hItemSibling, hItemArray, pnItems, itemMax);

	if (pEntry->bSelected)
	{
		// Don't fill the item array past its limit
		if (*pnItems == itemMax)
			return;

		hItemArray[*pnItems] = hItem;
		(*pnItems)++;
	}
}

int GetSelItems(HWND hwndTree, HTREEITEM* hItemArray, int itemMax)
{
	HTREEITEM hItem = TreeView_GetRoot(hwndTree);
	int nItems = 0;

	if (hItem)
		GetSelItems(hwndTree, hItem, hItemArray, &nItems, itemMax);

	return nItems;
}

void  ThumbNailTree::GetSelItems(void** selList, int selMax)
{
	::GetSelItems(hwndTree, (HTREEITEM*)selList, selMax);
}

// This function returns true if itemB comes after itemA in the enumeration
bool VerifyOrder(HWND hwndTree, HTREEITEM hItemA, HTREEITEM hItemB)
{
	HTREEITEM hItemNext = hItemA;

	while(hItemNext = TreeView_GetNextItem(hwndTree, hItemNext, TVGN_NEXT))
	{
		if (hItemNext == hItemB)
			return true;
	}

	return false;
}

bool tntSelectRange(HWND hwndTree, HTREEITEM hStartItem, HTREEITEM hEndItem)
{
	TVITEM tvitem;
	char name[256];
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT|TVIF_PARAM|TVIF_CHILDREN;
	tvitem.hItem = hStartItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = 256;

	TreeView_GetItem(hwndTree, &tvitem);

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	assert(pEntry);

	pEntry->bSelected = true;
	
	if (hStartItem == hEndItem)
		return true;

	int selCount = 0;

	// Process any siblings
	HTREEITEM hItemSibling = TreeView_GetNextItem(hwndTree, hStartItem, TVGN_NEXT);

	if (hItemSibling)
		if (tntSelectRange(hwndTree, hItemSibling, hEndItem))
			return true;

	return false;
}

LRESULT ThumbNailTree::SubWndProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	TreeData* tdata = (TreeData*)GetWindowLong(hwnd, GWL_USERDATA);

	// Need to completely revamp selection system
	switch(msg)
	{
	case WM_KEYDOWN:
		{
			switch(wParam)
			{
			case VK_UP:
				{
					// If up and shift are pressed only the last selected item should be deselected
					if (!GetAsyncKeyState(VK_SHIFT))
					{
						ClearSelectionsByHWND(hwnd);
						tdata->bSelPointClear = true;
					}
					else
					{
						HTREEITEM hItem = TreeView_GetSelection(hwnd);
						TVITEM tvitem;
						ZeroMemory(&tvitem, sizeof(TVITEM));
						tvitem.mask  = TVIF_PARAM;
						tvitem.hItem = hItem;
						
						TreeView_GetItem(hwnd, &tvitem);
						TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
						pEntry->bSelected = false;

						tdata->bSelPointClear = false;

						if (tdata->pthis->fpSelChange)
							tdata->pthis->fpSelChange(tdata->pthis, tdata->pthis->pSelChangeData);
					}

					tdata->bSelPointMove = true;					
				}
				break;

			case VK_DOWN:
				{
					// If down and shift are pressed the selection set should not be cleared
					if (!GetAsyncKeyState(VK_SHIFT))
					{
						ClearSelectionsByHWND(hwnd);
						tdata->bSelPointClear = true;
					}
					else
						tdata->bSelPointClear = false;

					tdata->bSelPointMove = true;
				}
				break;
			}
			break;
		}
	case WM_LBUTTONDOWN:
		{
			tdata->bSelPointClear = true;

			// Clicking in a region to the right of the item in the list
			// should still select that item within the list
			POINT pt;
			GetCursorPos(&pt);
			ScreenToClient(hwnd, &pt);

			TVHITTESTINFO hittest;
			hittest.pt = pt;

			SendMessage(hwnd, TVM_HITTEST, 0, (LPARAM)&hittest);

			// If the user is expanding or collapsing no changes need occur
			if (hittest.flags & TVHT_ONITEMBUTTON)
				break;

			if (!(wParam & MK_CONTROL))
				ClearSelectionsByHWND(hwnd);

			// Make sure the item is marked selected
			TVITEM tvitem;
			ZeroMemory(&tvitem, sizeof(TVITEM));
			tvitem.mask  = TVIF_PARAM;
			tvitem.hItem = hittest.hItem;
			
			if (SendMessage(hwnd, TVM_GETITEM, 0, (LPARAM)&tvitem))
			{
				TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;				
				pEntry->bSelected = !pEntry->bSelected;
			}

			if (hittest.flags & TVHT_ONITEMRIGHT)
			{
				// If shift is down, we need to select a range of items
				if (wParam & MK_SHIFT)
				{
					HTREEITEM hItem = TreeView_GetSelection(hwnd);

					// Need to handle range selection going in both directions top-bottom / bottom-top
					if (VerifyOrder(hwnd, hItem, hittest.hItem))
						tntSelectRange(hwnd, hItem, hittest.hItem);
					else
						tntSelectRange(hwnd, hittest.hItem, hItem);
				}
				else
					SendMessage(hwnd, TVM_SELECTITEM, (WPARAM)TVGN_CARET, (LPARAM)hittest.hItem);
			}
			else if (hittest.flags & TVHT_ONITEM)
			{
				if (wParam & MK_SHIFT)
				{
					HTREEITEM hItem = TreeView_GetSelection(hwnd);

					// Need to handle range selection going in both directions top-bottom / bottom-top
					if (VerifyOrder(hwnd, hItem, hittest.hItem))
						tntSelectRange(hwnd, hItem, hittest.hItem);
					else
						tntSelectRange(hwnd, hittest.hItem, hItem);
				}

				// Don't need to manually select for non-ranged since the original TreeView
				// control's wndproc that we subclassed will do it for us
			}

			InvalidateRect(hwnd, NULL, FALSE);

			if (tdata->pthis->fpSelChange)
				tdata->pthis->fpSelChange(tdata->pthis, tdata->pthis->pSelChangeData);

			break;
		}
	}

	return CallWindowProc(tdata->wndproc, hwnd, msg, wParam, lParam);
}

void ThumbNailTree::BuildControlUI()
{
	RECT wndRect;
	GetWindowRect(hwnd, &wndRect);

	unsigned int width  = wndRect.right  - wndRect.left;
	unsigned int height = wndRect.bottom - wndRect.top;

	hwndTree = CreateWindow("SysTreeView32",
		                    "",
							WS_CHILD|WS_VISIBLE|TVS_HASLINES|TVS_HASBUTTONS|TVS_LINESATROOT,
							0,
							0,
							width,
							height,
							hwnd,
							NULL,
							hInstance,
							NULL);

	if (!hwndTree)
		return;

	// We want to add multiple selections to the tree view control, this will require
	// subclassing it
	TreeData* tdata = new TreeData;
	tdata->pthis    = this;
	tdata->wndproc  = (WNDPROC)GetWindowLong(hwndTree, GWL_WNDPROC);
	SetWindowLong(hwndTree, GWL_USERDATA, (LONG)tdata);
	SetWindowLong(hwndTree, GWL_WNDPROC, (LONG)SubWndProc);

	SendMessage(hwndTree, TVM_SETITEMHEIGHT, (WPARAM)70, 0);
}

void* ThumbNailTree::AddNode(char* name, void* pData)
{
	TVINSERTSTRUCT tvis;
	ZeroMemory(&tvis, sizeof(TVINSERTSTRUCT));

	TNTEntryData* pEntry = new TNTEntryData;
	pEntry->extdata      = pData;

	tvis.hParent      = NULL;
	tvis.hInsertAfter = TVI_LAST;

	tvis.item.mask             = TVIF_TEXT|TVIF_PARAM;
	//tvis.item.hItem          =
	//tvis.item.state          =
	//tvis.item.stateMask      =
	tvis.item.pszText          = name;
	tvis.item.cchTextMax       = strlen(name);
	//tvis.item.iImage         =
	//tvis.item.iSelectedImage =
	//tvis.item.cChildren      =
	tvis.item.lParam           = (DWORD)pEntry;
	//tvis.item.iIntegral      =   

	pEntry->imgFilename[0] = 0;
	pEntry->hwndTree       = hwndTree;
	pEntry->hItem          = (HTREEITEM)SendMessage(hwndTree, TVM_INSERTITEM, 0, (LPARAM)&tvis);

	// Return value is actually a HTREEITEM (but, that is hidden outside of this interface)
	return (void*)pEntry->hItem;
}

void* ThumbNailTree::AddNode(char* name, char* filename, void* pData)
{
	TVINSERTSTRUCT tvis;
	ZeroMemory(&tvis, sizeof(TVINSERTSTRUCT));

	TNTEntryData* pEntry = new TNTEntryData;
	pEntry->extdata      = pData;

	tvis.hParent      = NULL;
	tvis.hInsertAfter = TVI_LAST;

	tvis.item.mask             = TVIF_TEXT|TVIF_PARAM;
	//tvis.item.hItem          =
	//tvis.item.state          =
	//tvis.item.stateMask      =
	tvis.item.pszText          = name;
	tvis.item.cchTextMax       = strlen(name);
	//tvis.item.iImage         =
	//tvis.item.iSelectedImage =
	//tvis.item.cChildren      =
	tvis.item.lParam           = (DWORD)pEntry;
	//tvis.item.iIntegral      =   

	strcpy(pEntry->imgFilename, filename);

	pEntry->hwndTree  = hwndTree;
	pEntry->hItem     = (HTREEITEM)SendMessage(hwndTree, TVM_INSERTITEM, 0, (LPARAM)&tvis);

	// Start a new thread to load in the thumbnail image
	pEntry->hLoadThread = CreateThread(NULL, 0, LoadImageThread, pEntry, 0, NULL);

	// Return value is actually a HTREEITEM (but, that is hidden outside of this interface)
	return (void*)pEntry->hItem;
}

void* ThumbNailTree::AddNode(void* pParent, char* name, void* pData)
{
	HTREEITEM hParent = (HTREEITEM)pParent;

	TVINSERTSTRUCT tvis;
	ZeroMemory(&tvis, sizeof(TVINSERTSTRUCT));

	TNTEntryData* pEntry = new TNTEntryData;
	pEntry->extdata      = pData;

	tvis.hParent      = hParent;
	tvis.hInsertAfter = TVI_LAST;

	tvis.item.mask             = TVIF_TEXT|TVIF_PARAM;
	//tvis.item.hItem          =
	//tvis.item.state          =
	//tvis.item.stateMask      =
	tvis.item.pszText          = name;
	tvis.item.cchTextMax       = strlen(name);
	//tvis.item.iImage         =
	//tvis.item.iSelectedImage =
	//tvis.item.cChildren      =
	tvis.item.lParam           = (DWORD)pEntry;
	//tvis.item.iIntegral      =   

	pEntry->imgFilename[0] = 0;
	pEntry->hwndTree       = hwndTree;
	pEntry->hItem          = (HTREEITEM)SendMessage(hwndTree, TVM_INSERTITEM, 0, (LPARAM)&tvis);

	// Return value is actually a HTREEITEM (but, that is hidden outside of this interface)
	return (void*)pEntry->hItem;
}

void* ThumbNailTree::AddNode(void* pParent, char* name, char* filename, void* pData)
{
	HTREEITEM hParent = (HTREEITEM)pParent;

	TVINSERTSTRUCT tvis;
	ZeroMemory(&tvis, sizeof(TVINSERTSTRUCT));

	TNTEntryData* pEntry = new TNTEntryData;
	pEntry->extdata      = pData;

	tvis.hParent      = hParent;
	tvis.hInsertAfter = TVI_LAST;

	tvis.item.mask             = TVIF_TEXT|TVIF_PARAM;
	//tvis.item.hItem          =
	//tvis.item.state          =
	//tvis.item.stateMask      =
	tvis.item.pszText          = name;
	tvis.item.cchTextMax       = strlen(name);
	//tvis.item.iImage         =
	//tvis.item.iSelectedImage =
	//tvis.item.cChildren      =
	tvis.item.lParam           = (DWORD)pEntry;
	//tvis.item.iIntegral      =   
	
	strcpy(pEntry->imgFilename, filename);

	pEntry->hwndTree = hwndTree;
	pEntry->hItem    = (HTREEITEM)SendMessage(hwndTree, TVM_INSERTITEM, 0, (LPARAM)&tvis);

	// Start a new thread to load in the thumbnail image
	pEntry->hLoadThread = CreateThread(NULL, 0, LoadImageThread, pEntry, 0, NULL);

	return (void*)pEntry->hItem;
}

// Determine how many levels deep this item exists within the tree
int ThumbNailTree::GetItemDepth(void* pData)
{
	HTREEITEM hItem = (HTREEITEM)pData;
	int depth = 0;

	while(hItem)
	{
		hItem = TreeView_GetParent(hwndTree, hItem);
		depth++;
	}

	return depth;
}

void RenderImage(int x, int y, HDC hdc, TNTEntryData* pEntry)
{
	// Image shouldn't be renderered until loading is complete
	if (pEntry->bLoaded)
	{
		// Draw the PNG bitmap into the item rect
		HDC        hdcBmp = CreateCompatibleDC(hdc);
		BITMAPINFO bmi;
		bmi.bmiHeader.biSize          = sizeof(BITMAPINFOHEADER);
		bmi.bmiHeader.biWidth         = pEntry->img.width;
		bmi.bmiHeader.biHeight        = pEntry->img.height;
		bmi.bmiHeader.biPlanes        = 1;
		bmi.bmiHeader.biBitCount      = pEntry->img.pixel_depth;
		bmi.bmiHeader.biCompression   = BI_RGB;
		bmi.bmiHeader.biSizeImage     = pEntry->img.imageSize;
		bmi.bmiHeader.biXPelsPerMeter = 0;
		bmi.bmiHeader.biYPelsPerMeter = 0;
		bmi.bmiHeader.biClrUsed       = 0;
		bmi.bmiHeader.biClrImportant  = 0;

		unsigned char* pBmpData;
		HBITMAP   hBmp = CreateDIBSection(hdcBmp, &bmi, DIB_RGB_COLORS, (void**)&pBmpData, NULL, 0);

		SelectObject(hdcBmp, hBmp);
		memcpy(pBmpData, pEntry->img.buf, pEntry->img.imageSize);

		StretchBlt(hdc,
				   x,		// X
				   y,		// Y
				   64,		// width
				   64,		// height
				   hdcBmp,
				   0,
				   0,
				   pEntry->img.width,
				   pEntry->img.height,
				   SRCCOPY);

		//if (drawInfo->itemState & ODS_FOCUS)
		//	DrawFocusRect(hdc,&drawInfo->rcItem);

		// Create a solid black pen to draw a border around the texture
		/*
		POINT pt;

		HPEN  hPen  = CreatePen(PS_SOLID, 1, 0);			
		SelectObject(hdc, hPen);

		MoveToEx(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 3, &pt);
		LineTo(hdc, drawInfo->rcItem.left + 66, drawInfo->rcItem.top + 3);
		LineTo(hdc, drawInfo->rcItem.left + 66, drawInfo->rcItem.top + 66);
		LineTo(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 66);
		LineTo(hdc, drawInfo->rcItem.left + 3, drawInfo->rcItem.top + 3);

		if (itemText)
			delete itemText;

		DeleteObject(hPen);
		*/

		DeleteObject(hBmp);
		DeleteDC(hdcBmp);
	}
}

LRESULT ThumbNailTree::WndProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{
	switch(msg)
	{
	case WM_NOTIFY:
		{
			NMHDR* nmhdr = (NMHDR*)lParam;
			
			switch(nmhdr->code)
			{
			// If the selection set has changed this should be forwarded to the parent
			case TVN_SELCHANGED:
				{
					/*
					if (!GetAsyncKeyState(VK_SHIFT))
					{
						// Ensure current selection is flagged selected first
						HTREEITEM hItem = TreeView_GetSelection(hwndTree);
						
						if (hItem)
						{
							TVITEM tvitem;
							ZeroMemory(&tvitem, sizeof(TVITEM));
							tvitem.hItem = hItem;
							tvitem.mask  = TVIF_PARAM;

							TreeView_GetItem(hwndTree, &tvitem);
							
							TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
							pEntry->bSelected = true;
						}
					}
					*/

					//SendMessage(GetParent(hwnd), msg, wParam, lParam);
				}
				return TRUE;

			case NM_CUSTOMDRAW:
				{
					NMTVCUSTOMDRAW* nmtvcd = (NMTVCUSTOMDRAW*)lParam;

					if (nmtvcd->nmcd.dwDrawStage == CDDS_PREPAINT)
						return CDRF_NOTIFYITEMDRAW;

					if (nmtvcd->nmcd.dwDrawStage == CDDS_POSTPAINT)
						return CDRF_NOTIFYITEMDRAW;

					if (nmtvcd->nmcd.dwDrawStage == CDDS_ITEMPOSTPAINT)
					{
						int indent = SendMessage(hwndTree, TVM_GETINDENT, 0, 0);

						RECT   rect   = nmtvcd->nmcd.rc;

						HTREEITEM hItem = (HTREEITEM)nmtvcd->nmcd.dwItemSpec;
						
						TVITEM tvitem;
						ZeroMemory(&tvitem, sizeof(TVITEM));
						//tvitem.mask  = TVIF_CHILDREN|TVIF_IMAGE|TVIF_PARAM|TVIF_SELECTEDIMAGE|TVIF_STATE|TVIF_TEXT;
						char buf[256];
						tvitem.mask       = 0xFFFFFFFF;
						tvitem.hItem      = hItem;
						tvitem.pszText    = buf;
						tvitem.cchTextMax = 256;
						TreeView_GetItem(hwndTree, &tvitem);

						rect.left = indent * GetItemDepth(tvitem.hItem);

						HDC hdc = nmtvcd->nmcd.hdc;

						HBRUSH hBrush;
						hBrush = (HBRUSH)GetStockObject(WHITE_BRUSH);
						FillRect(hdc, &rect, hBrush);
						//hBrush = (HBRUSH)GetStockObject(BLACK_BRUSH);
						//FrameRect(nmtvcd->nmcd.hdc, &rect, hBrush);

						// Draw the thumbnail border
						HPEN  hPen  = CreatePen(PS_SOLID, 1, 0);			
						SelectObject(hdc, hPen);

						POINT pt;

						TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
						assert(pEntry);

						TreeData* tdata = (TreeData*)GetWindowLong(hwndTree, GWL_USERDATA);

						if (nmtvcd->nmcd.uItemState & CDIS_SELECTED &&
							!pEntry->bSelected &&
							tdata->bSelPointMove)
						{
							if (tdata->bSelPointClear)
								ClearSelectionsByHWND(hwndTree);

							pEntry->bSelected = true;
							tdata->bSelPointMove = false;
							InvalidateRect(hwnd, NULL, FALSE);

							if (tdata->pthis->fpSelChange)
								tdata->pthis->fpSelChange(tdata->pthis, tdata->pthis->pSelChangeData);
						}

						if (pEntry->bSelected)
						{
							LOGBRUSH logbrush;
							logbrush.lbStyle = BS_SOLID;
							logbrush.lbColor = RGB(188,202,248);
							logbrush.lbHatch = 0;

							HBRUSH hBrush = CreateBrushIndirect(&logbrush);
							FillRect(hdc, &rect, hBrush);
							DeleteObject(hBrush);
							SetBkColor(hdc, logbrush.lbColor);
						}
						else
							SetBkColor(hdc, RGB(255,255,255));

						HFONT hFont;

						if (pEntry->imgFilename[0])
						{
							if (!pEntry->bNoRender)
								RenderImage(rect.left + 3, rect.top + 3, hdc, pEntry);

							MoveToEx(hdc, rect.left + 3, rect.top + 3, &pt);
							LineTo(hdc, rect.left + 66, rect.top + 3);
							LineTo(hdc, rect.left + 66, rect.top + 66);
							LineTo(hdc, rect.left + 3, rect.top + 66);
							LineTo(hdc, rect.left + 3, rect.top + 3);

							RECT rectText = rect;
							rectText.left += 74;

							hFont = CreateFont(14,0,0,0,  0  ,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, _T(""));
							SelectObject(hdc, hFont);
							DrawText(hdc, buf, strlen(buf), &rectText, DT_LEFT|DT_VCENTER|DT_SINGLELINE);
							DeleteObject(hFont);
						}
						else
						{
							hFont = CreateFont(16,0,0,0,FW_ULTRABOLD,0,0,0,0,0,0,0, VARIABLE_PITCH | FF_SWISS, _T(""));
							SelectObject(hdc, hFont);
							RECT rectText = rect;
							rectText.left += 10;
							DrawText(hdc, buf, strlen(buf), &rectText, DT_LEFT|DT_VCENTER|DT_SINGLELINE);
						}

						DeleteObject(hPen);
					}

					if (nmtvcd->nmcd.dwDrawStage == CDDS_ITEMPREPAINT)
					{
						// Draw a box around each entry
						//HPEN hSolidPen  = (HPEN)CreatePen(PS_SOLID, 0, 0);
						

						//return CDRF_SKIPDEFAULT;

						//DeleteObject(hSolidPen);
						return CDRF_NOTIFYPOSTPAINT;
					}
				}
				break;
			}
		}
	};

	return DefWindowProc(hwnd, msg, wParam, lParam);
}

bool ThumbNailTree::GetItemName(void* pData, char* name, int len)
{
	HTREEITEM hItem = (HTREEITEM)pData;

	TVITEM tvitem;
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask = TVIF_TEXT;
	tvitem.hItem = hItem;
	tvitem.pszText = name;
	tvitem.cchTextMax = len;

	if (TreeView_GetItem(hwndTree, &tvitem))
		return true;

	return false;
}

void* ThumbNailTree::GetItemData(void* pData)
{
	HTREEITEM hItem = (HTREEITEM)pData;

	TVITEM tvitem;
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask       = TVIF_PARAM;
	tvitem.hItem      = hItem;

	if (TreeView_GetItem(hwndTree, &tvitem))
	{
		TNTEntryData* pData = (TNTEntryData*)tvitem.lParam;
		return (void*)pData->extdata;
	}

	return NULL;
}

void ThumbNailTree::RegisterSelChangeCB(void (*fpFunc)(ThumbNailTree*,void*), void* pData)
{
	fpSelChange    = fpFunc;
	pSelChangeData = pData;
}

bool ThumbNailTree::SetThumbnailFile(void* pItem, char* filename)
{
	HTREEITEM hItem = (HTREEITEM)pItem;

	TVITEM tvitem;
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask  = TVIF_PARAM;
	tvitem.hItem = hItem;
	
	if (!TreeView_GetItem(hwndTree, &tvitem))
		return false;

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;
	
	if (filename)
	{
		strcpy(pEntry->imgFilename, filename);

		// Wait for previous thread to finish loading
		while(!pEntry->bLoaded) {}
		pEntry->bLoaded     = false;
		pEntry->bNoRender   = false;
		pEntry->hLoadThread = CreateThread(NULL, 0, LoadImageThread, pEntry, 0, NULL);
	}
	else
	{
		pEntry->bNoRender = true;
		InvalidateRect(hwndTree, NULL, TRUE);
	}

	return true;
}

char* ThumbNailTree::GetThumbnailFile(void* pItem)
{
	HTREEITEM hItem = (HTREEITEM)pItem;

	TVITEM tvitem;
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask  = TVIF_PARAM;
	tvitem.hItem = hItem;
	
	if (!TreeView_GetItem(hwndTree, &tvitem))
		return NULL;

	TNTEntryData* pEntry = (TNTEntryData*)tvitem.lParam;

	return pEntry->imgFilename;
}

void* ThumbNailTree::GetChildItem(void* pItem)
{
	return TreeView_GetNextItem(hwndTree, pItem, TVGN_CHILD);
}

void* ThumbNailTree::GetParentItem(void* pItem)
{
	return TreeView_GetNextItem(hwndTree, pItem, TVGN_PARENT);
}

void* ThumbNailTree::GetNextItem(void* pItem)
{
	return TreeView_GetNextItem(hwndTree, pItem, TVGN_NEXT);
}

int ThumbNailTree::GetChildIndex(void* pItem)
{
	HTREEITEM hParent = TreeView_GetParent(hwndTree, pItem);
	HTREEITEM hChild  = (HTREEITEM)GetChildItem(hParent);
	int index = 0;
	
	while(hChild && pItem != hChild)
	{
		hChild = (HTREEITEM)GetNextItem(hChild);
		index++;
	}

	return index;
}

bool ThumbNailTree::SetItemName(void* pItem, char* name)
{
	TVITEM tvitem;
	ZeroMemory(&tvitem, sizeof(TVITEM));
	tvitem.mask       = TVIF_TEXT;
	tvitem.hItem      = (HTREEITEM)pItem;
	tvitem.pszText    = name;
	tvitem.cchTextMax = strlen(name) + 1;

	return (TreeView_SetItem(hwndTree, &tvitem) ? true : false);
}

