/*
	RCMenu.cpp
	This file is responsible for adding NExt specific extensions to the
	MAX right-click quad menu

	aml - 3-21-03
*/

#include "Export/CutsceneExportDlg.h"
#include "Next.h"
#include "PropEdit/PropEdit.h"
#include "RCMenu.h"
#include "Resource.h"

#include "Max.h"
#include "istdplug.h"
#include "iparamb2.h"
#include "iparamm2.h"
#include "utilapi.h"
#include "guplib.h"
#include "MaxIcon.h"
#include "Notify.h"

#include "iFnPub.h"
#include "ActionTable.h"
#include "iMenuMan.h"
#include "appdata.h"
#include "UI/ModalDlg.h"
#include "misc/HelperFuncs.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

extern Interface*         gInterface;
extern NExtClassDesc      NExtDesc;
extern PropEditor*        pPropEdit;
extern CutsceneExportDlg* gpCutsceneExporter;

class CutsceneSelectDlg: public ModalDlgWindow
{
	BOOL DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

	LinkList<Cutscene>*       cutsceneList;
	LinkList<Link<Cutscene>*> selList;

	void AddCutscenes();
	void GetSelection();

public:
	CutsceneSelectDlg(HINSTANCE hInstance, HWND hwndParent, LinkList<Cutscene>* csList);
	~CutsceneSelectDlg();

	inline LinkList<Link<Cutscene>*>* GetSelList() { FUNC_ENTER("CutsceneSelectDlg::GetSelList");  return &selList; }
};

CutsceneSelectDlg::CutsceneSelectDlg(HINSTANCE hInstance, HWND hwndParent, LinkList<Cutscene>* csList) :
	ModalDlgWindow(hInstance, MAKEINTRESOURCE(IDD_SELCUTSCENE), hwndParent)
{ FUNC_ENTER("CutsceneSelectDlg::CutsceneSelectDlg"); 
	cutsceneList = csList;
}

CutsceneSelectDlg::~CutsceneSelectDlg()
{ FUNC_ENTER("CutsceneSelectDlg::~CutsceneSelectDlg"); 

}

void CutsceneSelectDlg::AddCutscenes()
{ FUNC_ENTER("CutsceneSelectDlg::AddCutscenes"); 
	Link<Cutscene>* curlink = cutsceneList->GetHead();

	SendDlgItemMessage(hwnd, IDC_CUTSCENELIST, LB_RESETCONTENT, 0, 0);

	while(curlink)
	{
		int idx = SendDlgItemMessage(hwnd, IDC_CUTSCENELIST, LB_ADDSTRING, 0, (LPARAM)(char*)curlink->data.name);
		SendDlgItemMessage(hwnd, IDC_CUTSCENELIST, LB_SETITEMDATA, (WPARAM)idx, (LPARAM)curlink);
		curlink = curlink->next;
	}
}

void CutsceneSelectDlg::GetSelection()
{ FUNC_ENTER("CutsceneSelectDlg::GetSelection"); 
	int items[256];
	int cnt = SendDlgItemMessage(hwnd, IDC_CUTSCENELIST, LB_GETSELCOUNT, 0, 0);
	SendDlgItemMessage(hwnd, IDC_CUTSCENELIST, LB_GETSELITEMS, (WPARAM)256, (LPARAM)items);
	
	selList.Clear();

	for(int i = 0; i < cnt; i++)
	{
		Link<Cutscene>* linkCS = (Link<Cutscene>*)SendDlgItemMessage(hwnd, 
			                                                         IDC_CUTSCENELIST, 
																	 LB_GETITEMDATA, 
																	 (WPARAM)items[i], 0);
		selList.AddToTail(&linkCS);
	}
}

BOOL CutsceneSelectDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("CutsceneSelectDlg::DlgProc"); 
	switch(msg)
	{
	case WM_CLOSE:
		EndDialog(hwnd, 0);
		return TRUE;

	case WM_INITDIALOG:
		AddCutscenes();
		return TRUE;

	case WM_COMMAND:
		{
			switch(LOWORD(wParam))
			{
			case IDC_SELECT:
				GetSelection();
				EndDialog(hwnd, 0);
				return TRUE;
			}
		}
	}

	return FALSE;
}

//Implementation of the Functions Published
class RCMenuActions: public FPStaticInterface
{
public:
	DECLARE_DESCRIPTOR(RCMenuActions) // Needed for static interfaces

	BEGIN_FUNCTION_MAP
		FN_ACTION( fnIdOpenPE,        OpenPropEditor );
		FN_ACTION( fnIdModel,         SetModelAction );
		FN_ACTION( fnIdSkin,          SetSkinAction );
		FN_ACTION( fnIdAddToCutscene, AddToCutscene );

		FN_PRED( fnIdIsAlwaysOn,               fnIsAlwaysOn );
		FN_PRED( fnIdIsAlwaysOff,              fnIsAlwaysOff );
		FN_PRED( fnIdIsExportModelChecked,     fnIsExportModelChecked );
		FN_PRED( fnIdIsExportSkinChecked,      fnIsExportSkinChecked );
		FN_PRED( fnIdIsExportEnabled,          fnIsExportEnabled );

	END_FUNCTION_MAP

	// FMap translators //////
	BOOL fnIsAlwaysOn()               { FUNC_ENTER("RCMenuActions::fnIsAlwaysOn");  return TRUE;                               }
	BOOL fnIsAlwaysOff()              { FUNC_ENTER("RCMenuActions::fnIsAlwaysOff");  return FALSE;                              }
	BOOL fnIsExportModelChecked()     { FUNC_ENTER("RCMenuActions::fnIsExportModelChecked");  return IsTypeChecked(OBJTYPE_MODEL);       }
	BOOL fnIsExportSkinChecked()      { FUNC_ENTER("RCMenuActions::fnIsExportSkinChecked");  return IsTypeChecked(OBJTYPE_SKIN);        }

	FPStatus SetModelAction()         { FUNC_ENTER("RCMenuActions::SetModelAction");  return SetTypeFlags(OBJTYPE_MODEL);        }
	FPStatus SetSkinAction()          { FUNC_ENTER("RCMenuActions::SetSkinAction");  return SetTypeFlags(OBJTYPE_SKIN);         }
	/////////////////////////////

	BOOL     IsTypeChecked(DWORD exptype);
	FPStatus SetTypeFlags(DWORD exptype);
	
	BOOL     fnIsExportEnabled();

	FPStatus OpenPropEditor();
	FPStatus AddToCutscene();
	void     AddNodeToCutscene(Cutscene* cutscene, INode* node);
};

void RCMenuActions::AddNodeToCutscene(Cutscene* cutscene, INode* node)
{ FUNC_ENTER("RCMenuActions::AddNodeToCutscene"); 
	CutsceneObj cso;
	
	Object*  obj = node->EvalWorldState(0).obj;
	cso.name     = node->GetName();
	
	cso.type = cso.GetTaggedFlags(node);

	if (obj->SuperClassID() == CAMERA_CLASS_ID)
		cso.type = OBJTYPE_CAMERA;

	cso.SetTaggedFlags(node);

	// Attempt to acquire cutscene object data from property editor props
	cso.handle = node->GetHandle();

	INode* rootNode = FindBoneRootBySkin(node);

	CutsceneObj csoDefault;
	Interval interval = gInterface->GetAnimRange();

	if (rootNode)
	{
		// Attempt to acquire cutscene object data from property editor props
		csoDefault.start      = interval.Start() / GetTicksPerFrame();
		csoDefault.end        = interval.End() / GetTicksPerFrame();
		csoDefault.rotTol     = 1.0f;
		csoDefault.tranTol    = 0.0f;
		csoDefault.modelName  = node->GetName();
		csoDefault.modelName2 = "";
		csoDefault.handle     = node->GetHandle();
		csoDefault.skelName   = "THPS5_human";
	}
	else
	{
		// Assign model defaults
		csoDefault.start      = interval.Start() / GetTicksPerFrame();
		csoDefault.end        = interval.End() / GetTicksPerFrame();
		csoDefault.rotTol     = 1.0f;
		csoDefault.tranTol    = 0.0f;
		csoDefault.modelName  = node->GetName();
		csoDefault.modelName2 = "";
		csoDefault.handle     = node->GetHandle();
		csoDefault.skelName   = "THPS5_human";
	}

	cso.LoadFromNode(cutscene->name, &csoDefault);
	cso.SaveToNode(cutscene->name);

	cutscene->objDB.AddUnique(&cso);
}

FPStatus RCMenuActions::AddToCutscene()
{ FUNC_ENTER("RCMenuActions::AddToCutscene"); 
	// Check available cutscenes, if more than one exists
	LinkList<Cutscene>* cutsceneList = gpCutsceneExporter->GetCutsceneDB();

	if (cutsceneList->GetSize() == 0)
	{
		MessageBox(gInterface->GetMAXHWnd(), "No cutscenes exists in this file", "Add to Cutscene", MB_ICONWARNING|MB_OK);
		return FPS_OK;
	}

	// If only one item exists in the cutscene list, we don't need to prompt
	if (cutsceneList->GetSize() == 1)
	{
		// Add all the selected nodes to the cutscene
		Cutscene* cutscene = &cutsceneList->GetHead()->data;

		int nNodes = gInterface->GetSelNodeCount();

		for(int i = 0; i < nNodes; i++)
			AddNodeToCutscene(cutscene, gInterface->GetSelNode(i));
	}
	else
	{
		// Prompt the user to select what cutscenes they want to add the node to
		CutsceneSelectDlg* csDlg = new CutsceneSelectDlg(hInstance, gInterface->GetMAXHWnd(), cutsceneList);
		csDlg->Show();	// Blocks
		
		LinkList<Link<Cutscene>*>* csList = csDlg->GetSelList();
		Link<Link<Cutscene>*>* csLink = csList->GetHead();

		int nNodes = gInterface->GetSelNodeCount();

		while(csLink)
		{
			for(int i = 0; i < nNodes; i++)
				AddNodeToCutscene(&csLink->data->data, gInterface->GetSelNode(i));

			csLink = csLink->next;
		}

		delete csDlg;
	}

	gpCutsceneExporter->Refresh();

	return FPS_OK;
}

FPStatus RCMenuActions::OpenPropEditor()
{ FUNC_ENTER("RCMenuActions::OpenPropEditor"); 
	pPropEdit->Show();
	return FPS_OK;
}

BOOL RCMenuActions::fnIsExportEnabled()
{ FUNC_ENTER("RCMenuActions::fnIsExportEnabled"); 
	int selCount = gInterface->GetSelNodeCount();

	for(int i = 0; i < selCount; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->SuperClassID() == CAMERA_CLASS_ID)
			return FALSE;
	}

	return TRUE;
}

BOOL RCMenuActions::IsTypeChecked(DWORD exptype)
{ FUNC_ENTER("RCMenuActions::IsTypeChecked"); 
	AppDataChunk* appdata;
	int selCount = gInterface->GetSelNodeCount();
	
	for(int i = 0; i < selCount; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;

		// Cameras cannot be set.  If this is a cam skip it
		if (obj->SuperClassID() == CAMERA_CLASS_ID)
			continue;

		appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);

		if (appdata && appdata->data)
		{
			if (*((DWORD*)appdata->data) & exptype)
				return TRUE;
		}
		else
		{
			CStr LCname = node->GetName();
			LCname.toLower();

			if (strstr(LCname, "skin_") == (char*)LCname)
			{
				node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
				DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
				*pMode = OBJTYPE_SKIN;
				node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);
				
				if (*pMode & exptype)
					return TRUE;
			}

			if (strstr(LCname, "model_") == (char*)LCname)
			{
				node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
				DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
				*pMode = OBJTYPE_MODEL;
				node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);

				if (*pMode & exptype)
					return TRUE;
			}

			if (strstr(LCname, "camera_") == (char*)LCname)
			{
				node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
				DWORD* pMode = (DWORD*)malloc(sizeof(DWORD));
				*pMode = OBJTYPE_CAMERA;
				node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE, sizeof(DWORD), pMode);

				if (*pMode & exptype)
					return TRUE;
			}
		}
	}

	return FALSE;
}

FPStatus RCMenuActions::SetTypeFlags(DWORD exptype)
{ FUNC_ENTER("RCMenuActions::SetTypeFlags"); 
	int selCount = gInterface->GetSelNodeCount();
	
	// Determine if we should be setting or removing the flags
	bool bRemove = false;

	if (selCount > 0)
	{
		// If the flag is turned on for any of the nodes in the selection set, then we're removing the flag
		for(int i = 0; i < selCount; i++)
		{
			INode* node  = gInterface->GetSelNode(i);
			DWORD  flags = 0;

			AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);

			if (appdata && appdata->data)
			{
				flags = *((DWORD*)appdata->data);
				if (flags & exptype)
				{
					bRemove = true;
					break;
				}
			}
		}
	}

	// Go through and set/remove the appropriate flag
	for(int i = 0; i < selCount; i++)
	{
		INode* node = gInterface->GetSelNode(i);
		Object* obj = node->EvalWorldState(0).obj;
		DWORD flags = 0;

		// If the user's selection contains a camera the camera object cannot be updated
		if (obj->SuperClassID() == CAMERA_CLASS_ID)
			continue;

		AppDataChunk* appdata = node->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);

		if (appdata && appdata->data)
			flags = *((DWORD*)appdata->data);

		node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_CUTSCENE_EXPORTMODE);
		
		// Set/Remove the given flag from the object
		/*
		if (bRemove)
			flags &= ~exptype;
		else
			flags |= exptype;
		*/

		// From now on we're only allowing a single type to be set
		flags = exptype;

		DWORD* appdataExptype = (DWORD*)malloc(sizeof(DWORD));
		*appdataExptype = flags;

		node->AddAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, 
							  vNAPP_CUTSCENE_EXPORTMODE, 
							  sizeof(DWORD), 
							  appdataExptype);
	}

	return FPS_OK;
}

static RCMenuActions exptagmenuai( vNEXT_RCMENU_INTERFACE_ID, _T("NExtRC_Interface"), 0, &NExtDesc, FP_ACTIONS,
	kActionMainUIContext,

	fnIdOpenPE, _T("OpenPE"), IDS_OPENPEACTION_DESCTEXT, 0,
		f_category, _T("Property Editor"), IDS_FPACTIONS_CATEGORY,
		f_menuText, IDS_MENU_OPENPE,
		end,

	fnIdModel, _T("Model"), IDS_TAGACTION_DESCTEXT, 0,
		f_category, _T("Export Mode Tag"), IDS_FPACTIONS_CATEGORY,
		f_menuText, IDS_MENU_MODEL,
		f_isEnabled, fnIdIsExportEnabled,
		f_isChecked, fnIdIsExportModelChecked,
		f_isVisible, fnIdIsAlwaysOn,
		end,

	fnIdSkin, _T("Skin"), IDS_TAGACTION_DESCTEXT, 0,
		f_category, _T("Export Mode Tag"), IDS_FPACTIONS_CATEGORY,
		f_menuText, IDS_MENU_SKIN,
		f_isEnabled, fnIdIsExportEnabled,
		f_isChecked, fnIdIsExportSkinChecked,
		f_isVisible, fnIdIsAlwaysOn,
		end,

	fnIdAddToCutscene, _T("AddToCutscene"), IDS_ADDTOCUTACTION_DESCTEXT, 0,
		f_category, _T("AddToCutscene"), IDS_FPACTIONS_CATEGORY,
		f_menuText, IDS_MENU_ADDTOCUTSCENE,
		end,

	end
);

void CreateRCMenus()
{ FUNC_ENTER("CreateRCMenus"); 
	IMenuManager* pMenuMan = GetCOREInterface()->GetMenuManager();
	ActionTable* pGlobalActionTable = exptagmenuai.action_table;

	ActionItem* pActionOpenPE        = pGlobalActionTable->GetAction( fnIdOpenPE );
	ActionItem* pActionModel         = pGlobalActionTable->GetAction( fnIdModel );
	ActionItem* pActionSkin          = pGlobalActionTable->GetAction( fnIdSkin );
	ActionItem* pActionAddToCutscene = pGlobalActionTable->GetAction( fnIdAddToCutscene );

	assert(pActionOpenPE);
	assert(pActionModel);
	assert(pActionSkin);
	assert(pActionAddToCutscene);

	// Create menu items to hold each action
	IMenuItem* pMenuOpenPE = GetIMenuItem();
	pMenuOpenPE->SetActionItem(pActionOpenPE);
	
	IMenuItem* pMenuModel = GetIMenuItem();
	pMenuModel->SetActionItem(pActionModel);

	IMenuItem* pMenuSkin = GetIMenuItem();
	pMenuSkin->SetActionItem(pActionSkin);

	IMenuItem* pMenuAddToCutscene = GetIMenuItem();
	pMenuAddToCutscene->SetActionItem(pActionAddToCutscene);

	// Add the sub-menu items to the right click menu
	IMenuManager* menuMan = GetCOREInterface()->GetMenuManager();
	IQuadMenu* mainQuad = menuMan->FindQuadMenu("Default Viewport Quad");
	IMenu* quadMenu = mainQuad->GetMenu(0);
	quadMenu->AddItem( pMenuOpenPE );
	quadMenu->AddItem( pMenuModel );
	quadMenu->AddItem( pMenuSkin );
	quadMenu->AddItem( pMenuAddToCutscene );
}

// Called by MAX whenever the menu needs to be recreated due to a interface setting change
static void InterfaceChanged(void *param, NotifyInfo *info) 
{ FUNC_ENTER("InterfaceChanged"); 
	OutputDebugString("HANDLER: InterfaceChanged\n");
	CreateRCMenus();
}

void InstallNExtRCMenus()
{ FUNC_ENTER("InstallNExtRCMenus"); 
	TSTR name = _T("NExt RightClick Extensions");
	GetCOREInterface()->GetActionManager()->RegisterActionContext(vNEXT_RCMENU_CONTEXT,name.data());
	
#ifndef DISABLE_NOTIFICATIONS
	RegisterNotification(InterfaceChanged, NULL, NOTIFY_COLOR_CHANGE);
#endif

	CreateRCMenus();
}

void UnInstallNExtRCMenus()
{ FUNC_ENTER("UnInstallNExtRCMenus"); 
	UnRegisterNotification(InterfaceChanged, NULL, NOTIFY_COLOR_CHANGE);
}
