/*
	NSPiece.cpp
	This is the structure to maintain data loaded from .NSP files
	(NeverSoft Piece files)
*/

#include "NSPiece.h"

#define NO_PREINCLUDES
#define NO_BASEPREINCLUDES
#include "FuncEnter.h"

NSPiece::NSPiece()
{ FUNC_ENTER("NSPiece::NSPiece"); 
	texID        = NULL;
	texname      = NULL;
	texnameID    = NULL;
	texnameIDGL  = NULL;
	nTextures    = 0;
	nFlagTypes   = 0;
	flags        = NULL;
	pieceName    = "UnnamedPiece";
}

NSPiece::~NSPiece()
{ FUNC_ENTER("NSPiece::~NSPiece"); 
	if (texID)
		delete [] texID;

	if (texname)
		delete [] texname;

	if (texnameID)
		delete [] texnameID;

	if (texnameIDGL)
		delete [] texnameIDGL;

	if (flags)
		delete [] flags;
}

void NSPiece::AllocNames(int i)
{ FUNC_ENTER("NSPiece::AllocNames"); 
	if (texname)
		delete [] texname;

	if (texnameID)
		delete [] texnameID;

	if (texnameIDGL)
		delete [] texnameIDGL;

	texname      = new CStr[i];
	texnameID    = new int[i];
	texnameIDGL  = new int[i];
	nTextures    = i;
}

void NSPiece::Alloc()
{
	if (texID)
		delete [] texID;

	texID = new int[mesh.numFaces];
}

// This function scans through the material IDs used within the mesh and then
// assigns the appropriate OpenGL texture ID name based on the texnameID array
void NSPiece::AssignTexIDs()
{ FUNC_ENTER("NSPiece::AssignTexIDs"); 
	for(int curface = 0; curface < mesh.numFaces; curface++)
	{
		int  mtlID     = mesh.getFaceMtlIndex(curface);
		bool bAssigned = false;
		
		// Find the material ID
		for(int i = 0; i < nTextures; i++)
		{
			if (texnameID[i] == mtlID ||
				texnameID[i] == -1)
			{
				bAssigned = true;
				texID[curface] = texnameIDGL[i];
				break;
			}
		}

		if (!bAssigned)
		{
			texID[curface] = 0;

		}
	}
}

int NSPiece::FindTexture(int mtlID)
{ FUNC_ENTER("NSPiece::FindTexture"); 
	for(int i = 0; i < nTextures; i++)
	{
		if (texnameID[i] == mtlID)
			return i;
	}

	return 0;
}

void NSPiece::AllocFlags(int nFlags)
{ FUNC_ENTER("NSPiece::AllocFlags"); 
	if (flags)
		delete [] flags;

	flags = new FlagType[nFlags];
	nFlagTypes = nFlags;
}

void NSPiece::BuildD3DMesh(int nDisp)
{ FUNC_ENTER("NSPiece::BuildD3DMesh"); 
	BuildD3DMeshToIndexed(nDisp);
}

void NSPiece::BuildD3DMeshAsIndexed(int nDisp)
{ FUNC_ENTER("NSPiece::BuildD3DMeshAsIndexed"); 
	D3Dmesh.AllocVerts(mesh.numVerts);
	D3Dmesh.AllocFaces(mesh.numFaces);
	D3Dmesh.AllocTextures(nTextures);

	int i;
	for(i = 0; i < nTextures; i++)
	{
		D3Dmesh.textures[i].texname = texname[i];
	}

	for(i = 0; i < mesh.numVerts; i++)
	{
		D3Dmesh.verts[i].x     = mesh.verts[i].y;
		D3Dmesh.verts[i].y     = mesh.verts[i].z;
		D3Dmesh.verts[i].z     = mesh.verts[i].x;
		D3Dmesh.verts[i].color = 0xffffffff;
		D3Dmesh.verts[i].u     = 0.0f;
		D3Dmesh.verts[i].v     = 0.0f;
	}

	for(i = 0; i < mesh.numFaces; i++)
	{
		D3Dmesh.texFaces[i].f[0] = mesh.faces[i].v[0];
		D3Dmesh.texFaces[i].f[1] = mesh.faces[i].v[1];
		D3Dmesh.texFaces[i].f[2] = mesh.faces[i].v[2];
		D3Dmesh.texFaces[i].tex  = FindTexture(mesh.getFaceMtlIndex(i));

		// This won't look right for textured faces that share verts, but could be sufficient
		// for preview favoring speed over card memory usage over appearance
		D3Dmesh.verts[mesh.faces[i].v[0]].u = mesh.tVerts[mesh.tvFace[i].t[0]].x;
		D3Dmesh.verts[mesh.faces[i].v[0]].v = mesh.tVerts[mesh.tvFace[i].t[0]].y;
		D3Dmesh.verts[mesh.faces[i].v[1]].u = mesh.tVerts[mesh.tvFace[i].t[1]].x;
		D3Dmesh.verts[mesh.faces[i].v[1]].v = mesh.tVerts[mesh.tvFace[i].t[1]].y;
		D3Dmesh.verts[mesh.faces[i].v[2]].u = mesh.tVerts[mesh.tvFace[i].t[2]].x;
		D3Dmesh.verts[mesh.faces[i].v[2]].v = mesh.tVerts[mesh.tvFace[i].t[2]].y;
	}

	D3Dmesh.AllocRefs(nDisp);
	D3Dmesh.BuildBuffers();
}

void NSPiece::BuildD3DMeshToIndexed(int nDisp)
{ FUNC_ENTER("NSPiece::BuildD3DMeshToIndexed"); 
	// We need to ensure a unique vert for every face so that
	// our texture coordinates are correct, since we must specify
	// texture coordinates with each vert in D3D's pipeline
	D3Dmesh.AllocVerts(mesh.numFaces * 3);
	D3Dmesh.AllocFaces(mesh.numFaces);
	D3Dmesh.AllocTextures(nTextures);

	int i;
	for(i = 0; i < nTextures; i++)
	{
		D3Dmesh.textures[i].texname = texname[i];
	}

	for(i = 0; i < mesh.numFaces; i++)
	{
		// Assign vert data with included UV data
		D3Dmesh.verts[i*3  ].x     = mesh.verts[mesh.faces[i].v[0]].y;
		D3Dmesh.verts[i*3  ].y     = mesh.verts[mesh.faces[i].v[0]].z;
		D3Dmesh.verts[i*3  ].z     = mesh.verts[mesh.faces[i].v[0]].x;
		D3Dmesh.verts[i*3  ].color = 0xffffffff;

		if (mesh.tVerts)
		{
			D3Dmesh.verts[i*3  ].u = mesh.tVerts[mesh.tvFace[i].t[0]].x;
			D3Dmesh.verts[i*3  ].v = mesh.tVerts[mesh.tvFace[i].t[0]].y;
		}
		else
		{
			D3Dmesh.verts[i*3  ].u = 0.0f;
			D3Dmesh.verts[i*3  ].v = 0.0f;
		}

		D3Dmesh.verts[i*3+1].x     = mesh.verts[mesh.faces[i].v[1]].y;
		D3Dmesh.verts[i*3+1].y     = mesh.verts[mesh.faces[i].v[1]].z;
		D3Dmesh.verts[i*3+1].z     = mesh.verts[mesh.faces[i].v[1]].x;
		D3Dmesh.verts[i*3+1].color = 0xffffffff;

		if (mesh.tVerts)
		{
			D3Dmesh.verts[i*3+1].u = mesh.tVerts[mesh.tvFace[i].t[1]].x;
			D3Dmesh.verts[i*3+1].v = mesh.tVerts[mesh.tvFace[i].t[1]].y;
		}
		else
		{
			D3Dmesh.verts[i*3+1].u = 1.0f;
			D3Dmesh.verts[i*3+1].v = 0.0f;
		}

		D3Dmesh.verts[i*3+2].x     = mesh.verts[mesh.faces[i].v[2]].y;
		D3Dmesh.verts[i*3+2].y     = mesh.verts[mesh.faces[i].v[2]].z;
		D3Dmesh.verts[i*3+2].z     = mesh.verts[mesh.faces[i].v[2]].x;
		D3Dmesh.verts[i*3+2].color = 0xffffffff;

		if (mesh.tVerts)
		{
			D3Dmesh.verts[i*3+2].u = mesh.tVerts[mesh.tvFace[i].t[2]].x;
			D3Dmesh.verts[i*3+2].v = mesh.tVerts[mesh.tvFace[i].t[2]].y;
		}
		else
		{
			D3Dmesh.verts[i*3+2].u = 1.0f;
			D3Dmesh.verts[i*3+2].v = 1.0f;
		}

		// Index list matches up 1:1 in this case
		// May change system later to draw straight verts w/o index list in this situation
		D3Dmesh.faces[i].f[0]      = i*3;
		D3Dmesh.faces[i].f[1]      = i*3+1;
		D3Dmesh.faces[i].f[2]      = i*3+2;

		D3Dmesh.texFaces[i].f[0]   = i*3;
		D3Dmesh.texFaces[i].f[1]   = i*3+1;
		D3Dmesh.texFaces[i].f[2]   = i*3+2;
		D3Dmesh.texFaces[i].tex    = FindTexture(mesh.getFaceMtlIndex(i));
	}

	D3Dmesh.AllocRefs(nDisp);
	D3Dmesh.BuildBuffers();
}
