/*
	MtlEditor.cpp
	Custom Neversoft Material Editor
	(Allows changing groups of material properties)

	aml - 9-9-03
*/

#include <core/hashtable.h>
#include "MtlEditor.h"
#include "UI/ThumbnailList.h"
#include "UI/ThumbnailTree.h"
#include "UI/MSWindow.h"
#include "resource.h"
#include "Material/NExtMat.h"
#include "Material/NExtMultiMat.h"
#include "Material/Multi.h"
#include "Texture/NExtTexture.h"
#include "CustCont.h"
#include "Material/TerrainTypes.h"
#include "MemDebug.h"

extern HINSTANCE hInstance;
extern char blend_modes[vNUM_BLEND_MODES][32];
//extern char mag_filtering_modes[2][32];
//extern char min_filtering_modes[6][40];
extern char filtering_modes[2][32];

static MtlEditorClassDesc gMtlEditorDesc;
ClassDesc* GetMtlEditorDesc( void ) { return &gMtlEditorDesc; }

#define vNUM_PASSES 4

static bool bLockMtlUpdates                  = false;
static bool bLockMtlFlagUpdates              = false;
static bool bLockMtlPassUpdates[vNUM_PASSES] = {false};
static bool bLockTexPropUpdates              = false;

enum MapModes
{
	MAPMODE_EXPLICIT,
	MAPMODE_ENVIRONMENTMAP,
};

//////////////////////////////////////////////////////////////////////////////
enum MtlTreeItemType
{
	MTITYPE_UNDEFINED,
	MTITYPE_MULTI,		// Material is a multimaterial
	MTITYPE_MTL,		// Material is a normal Next material
	MTITYPE_TEX,		// Material is a texture
};

struct MtlTreeItemData
{
	MtlTreeItemType type;

	union
	{
		INExtTexture*   pTex;
		INExtMaterial*  pMtl;
	};
};

struct WndHookData
{
	DWORD   extdata;
	WNDPROC origproc;
	bool    bHighlighted;
};

// Unfortunately all MAX specific UI controls make use of the window's user data field
// preventing us from overtaking it to subclass their windows.  To get around this we'll
// have to look up our associated data in a hash table
static Lst::HashTable<WndHookData> hookDB(8);

class MtlEditorDlg: public MSDlgWindow
{
	Interface*     ip;
	ThumbNailList* ptlist;
	ThumbNailTree* ptTree;
	IRollupWindow* pRollupWindow;
	int            nSelMaterials;
	void**         selMtlList;

	///  Mtl Props
	ICustEdit        *pDrawOrder, 
		             *pGrassHeight, 
					 *pGrassLayers,
					 *pBasePass, 
					 *pAlphaCutoff,
					 *pSpecularPower;

	ISpinnerControl  *pDrawOrderSpin,
		             *pGrassHeightSpin,
		             *pGrassLayersSpin,
					 *pBasePassSpin,
					 *pAlphaCutoffSpin,
					 *pSpecularPowerSpin;

	IColorSwatch     *pSpecularColor;
	///////////////////

	///  Mtl Pass
	IColorSwatch     *pHasColorColor[vNUM_PASSES],
		             *pAmbient[vNUM_PASSES],
					 *pDiffuse[vNUM_PASSES],
					 *pSpecular[vNUM_PASSES];

	ICustEdit        *pTileU[vNUM_PASSES],
		             *pTileV[vNUM_PASSES],
					 *pFixedSourceAlpha[vNUM_PASSES],
					 *pUVelocity[vNUM_PASSES],
					 *pUFrequency[vNUM_PASSES],
					 *pUAmplitude[vNUM_PASSES],
					 *pUPhase[vNUM_PASSES],
					 *pVVelocity[vNUM_PASSES],
					 *pVFrequency[vNUM_PASSES],
					 *pVAmplitude[vNUM_PASSES],
					 *pVPhase[vNUM_PASSES];

	ISpinnerControl  *pTileUSpin[vNUM_PASSES],
		             *pTileVSpin[vNUM_PASSES],
					 *pFixedSourceAlphaSpin[vNUM_PASSES],
					 *pUVelocitySpin[vNUM_PASSES],
					 *pUFrequencySpin[vNUM_PASSES],
					 *pUAmplitudeSpin[vNUM_PASSES],
					 *pUPhaseSpin[vNUM_PASSES],
					 *pVVelocitySpin[vNUM_PASSES],
					 *pVFrequencySpin[vNUM_PASSES],
					 *pVAmplitudeSpin[vNUM_PASSES],
					 *pVPhaseSpin[vNUM_PASSES];

	ICustButton      *pMap[vNUM_PASSES];
	////////////////////////////////////////

	///  Tex Pass
	ICustButton      *pPS2BaseMap,
		             *pNGCBaseMap,
					 *pXBOXBaseMap,
					 *pPS2Mip[5],
					 *pNGCMip[5],
					 *pXBOXMip[5];

	ICustEdit        *pKBiasEdit,
		             *pCropPlaceUEdit,
					 *pCropPlaceWEdit,
					 *pCropPlaceVEdit,
					 *pCropPlaceHEdit;
	
	ISpinnerControl  *pKBiasSpin,
		             *pCropPlaceUSpin,
					 *pCropPlaceWSpin,
					 *pCropPlaceVSpin,
					 *pCropPlaceHSpin;

	IColorSwatch     *pTransColor;	

	////////////////////////////////////////

	BOOL DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

	static void UpdateMtlPropStates(HWND hwnd);
	static void UpdateMtlFlagStates(HWND hwnd);
	static void UpdateMtlPassStates(HWND hwnd, int pass);
	void UpdateTexPropStates(HWND hwnd);

	static void MtlPropsUpdate(HWND hwnd, MtlEditorDlg* pMtlEditor);
	static void MtlFlagsUpdate(HWND hwnd, MtlEditorDlg* pMtlEditor);
	static void MtlPassUpdate(HWND hwnd, MtlEditorDlg* pMtlEditor, int pass);
	static void TexPassUpdate(HWND hwnd, MtlEditorDlg* pMtlEditor);

	BOOL MtlPassDlgProc(int pass, HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);

	static BOOL CALLBACK MtlPropsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK MtlFlagsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK MtlPass1DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK MtlPass2DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK MtlPass3DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK MtlPass4DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	
	static BOOL CALLBACK TexPropsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	
	static void TreeSelChange(ThumbNailTree* pTree, void* pData);

	void TreeSelChange();
	void BuildMaterialPanels();
	void BuildTexturePanels();

	void ApplyMtlProps(HWND hwnd, int id);
	void ApplyMtlFlagProps(HWND hwnd, int id);
	void ApplyMtlPassProps(HWND hwnd, int id, int pass);
	void ApplyTexProps(HWND hwnd, int id);

	static void GetFileFromMapString(char* strMap, char* filename);
	static void FlagItem(MtlEditorDlg* pthis, HWND hwnd, DWORD id, char* name = "", int pass = -1);

	bool GetBitmapInfo(int id, BitmapInfo* bi);

	// Window hook methods
	void HookWindow(HWND hwnd);
	static LRESULT CALLBACK SubClassDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam);
	static BOOL CALLBACK wndHookEnum(HWND hwnd, LPARAM lParam);
	void HookAllWindows(HWND hwnd);

	static bool MtlUpdate(ThumbNailTree* pTree, void* pItem, void* pthis);

public:
	void RefreshMtls();
	bool UpdateMtls();
	
	MtlEditorDlg(Interface* ip, HWND hwndParent = NULL);
	~MtlEditorDlg();
};

void MtlEditorDlg::UpdateMtlPropStates(HWND hwnd)
{
	if (IsDlgButtonChecked(hwnd, IDC_TRANSPARENT) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID_SPINNER), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_SORTED), TRUE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_SORTED), FALSE);
	}

	if (IsDlgButtonChecked(hwnd, IDC_GRASSIFY) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_EDITHEIGHT), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPINHEIGHT), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_EDITLAYERS), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPINLAYERS), TRUE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_EDITHEIGHT), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPINHEIGHT), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_EDITLAYERS), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPINLAYERS), FALSE);
	}

	if (IsDlgButtonChecked(hwnd, IDC_USEDEFAC) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF_SPINNER), FALSE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF_SPINNER), TRUE);
	}

	if (IsDlgButtonChecked(hwnd, IDC_USESPECULAR) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_SPECPOW), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPECPOWSPIN), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPECCOLOR), TRUE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_SPECPOW), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPECPOWSPIN), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_SPECCOLOR), FALSE);
	}
}

void MtlEditorDlg::UpdateMtlPassStates(HWND hwnd, int pass)
{
	if (IsDlgButtonChecked(hwnd, IDC_ENABLED) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_IGNORE_VERTEX_ALPHA), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_ABSENT_IN_NET_GAMES), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_MAP), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_MAPPING_MODE), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BLEND_MODE), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_COLOR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_UV_ENABLE), TRUE);

		if (IsDlgButtonChecked(hwnd, IDC_UV_ENABLE) == BST_CHECKED)
		{
			EnableWindow(GetDlgItem(hwnd, IDC_U_VEL), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_VEL_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_VEL), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_VEL_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE_SPINNER), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE), TRUE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE_SPINNER), TRUE);
		}
		else
		{
			EnableWindow(GetDlgItem(hwnd, IDC_U_VEL), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_VEL_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_VEL), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_VEL_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE_SPINNER), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE), FALSE);
			EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE_SPINNER), FALSE);
		}
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_IGNORE_VERTEX_ALPHA), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ABSENT_IN_NET_GAMES), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_MAP), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_MAPPING_MODE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BLEND_MODE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_COLOR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_UV_ENABLE), FALSE);

		EnableWindow(GetDlgItem(hwnd, IDC_U_VEL), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_VEL_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_FREQUENCY_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_AMPLITUDE_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_U_PHASE_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_VEL), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_VEL_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_FREQUENCY_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_AMPLITUDE_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_V_PHASE_SPINNER), FALSE);
	}

	int mapMode   = SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_GETCURSEL, 0, 0);
	int blendMode = SendDlgItemMessage(hwnd, IDC_BLEND_MODE, CB_GETCURSEL, 0, 0);

	if (mapMode == MAPMODE_ENVIRONMENTMAP)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_U), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_U_SPINNER), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_V), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_V_SPINNER), TRUE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_U), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_U_SPINNER), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_V), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ENV_TILE_V_SPINNER), FALSE);
	}

	if (blendMode == vBLEND_MODE_ADD_FIXED      ||
		blendMode == vBLEND_MODE_SUB_FIXED      ||
		blendMode == vBLEND_MODE_BLEND_FIXED    ||
		blendMode == vBLEND_MODE_MODULATE_FIXED ||
		blendMode == vBLEND_MODE_BRIGHTEN_FIXED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_FIXED), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_FIXED_SPINNER), TRUE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_FIXED), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_FIXED_SPINNER), FALSE);
	}

	// Enabled and absent in net games are always disabled on pass 0
	if (pass == 0)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_ENABLED), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_ABSENT_IN_NET_GAMES), FALSE);
		CheckDlgButton(hwnd, IDC_ENABLED, TRUE);
		CheckDlgButton(hwnd, IDC_ABSENT_IN_NET_GAMES, FALSE);
	}
}

void MtlEditorDlg::UpdateTexPropStates(HWND hwnd)
{
	/*
	ShowWindow(GetDlgItem(hwnd, IDC_CROPPLACEFRAME), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_BM_CLIP), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_BM_CROP_IMAGE2), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_BM_CROP), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_BM_PLACE), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CROPPLACEU), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CROPPLACEW), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CROPPLACEV), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CROPPLACEH), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_X), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_XSPIN), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_W), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_WSPIN), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_Y), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_YSPIN), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIP_H), SW_HIDE);
	ShowWindow(GetDlgItem(hwnd, IDC_CLIPH_SPIN), SW_HIDE);
	*/

	char buf[1024];
	pNGCBaseMap->GetText(buf, 1023);

	if (strcmp(buf, "Use PS2") != 0)
	{
		CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_NGC, BST_UNCHECKED);
		EnableWindow(GetDlgItem(hwnd, IDC_USE_PS2_MIPMAPS_NGC), FALSE);
	}
	else
		EnableWindow(GetDlgItem(hwnd, IDC_USE_PS2_MIPMAPS_NGC), TRUE);

	pXBOXBaseMap->GetText(buf, 1023);

	if (strcmp(buf, "Use PS2") != 0)
	{
		CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_XBOX, BST_UNCHECKED);
		EnableWindow(GetDlgItem(hwnd, IDC_USE_PS2_MIPMAPS_XBOX), FALSE);
	}
	else
		EnableWindow(GetDlgItem(hwnd, IDC_USE_PS2_MIPMAPS_XBOX), TRUE);


	if (IsDlgButtonChecked(hwnd, IDC_USE_PS2_MIPMAPS_NGC) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_1_NGC_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_2_NGC_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_3_NGC_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_4_NGC_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_5_NGC_CLEAR), FALSE);

		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_NGC), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_NGC), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_NGC), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_NGC), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_NGC), FALSE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_1_NGC_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_2_NGC_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_3_NGC_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_4_NGC_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_5_NGC_CLEAR), TRUE);

		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_NGC), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_NGC), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_NGC), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_NGC), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_NGC), TRUE);
	}

	if (IsDlgButtonChecked(hwnd, IDC_USE_PS2_MIPMAPS_XBOX) == BST_CHECKED)
	{
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_1_XBOX_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_2_XBOX_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_3_XBOX_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_4_XBOX_CLEAR), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_5_XBOX_CLEAR), FALSE);

		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_XBOX), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_XBOX), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_XBOX), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_XBOX), FALSE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_XBOX), FALSE);
	}
	else
	{
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_1_XBOX_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_2_XBOX_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_3_XBOX_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_4_XBOX_CLEAR), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_5_XBOX_CLEAR), TRUE);

		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_XBOX), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_XBOX), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_XBOX), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_XBOX), TRUE);
		EnableWindow(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_XBOX), TRUE);
	}
}

#define DINV 3

void MtlEditorDlg::FlagItem(MtlEditorDlg* pthis, HWND hwnd, DWORD id, char* name, int pass)
{
	HWND hwndItem = GetDlgItem(hwnd, id);
	WndHookData* pData = hookDB.GetItem((DWORD)hwndItem);

	if (pData)
		pData->bHighlighted = true;

	char str[256], str2[256];
	strcpy(str, name);

	if (!pData)
		strcat(str, " (No entry?)");

	if (pass != -1)
		sprintf(str2, "%s (pass %i)", str, pass + 1);
	else
		strcpy(str2, str);

	// Add conflict to conflict list if it doesn't already exist
	if (strlen(name) > 0)
	{
		if (SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLST, LB_FINDSTRING, (WPARAM)-1, (LPARAM)str2) == LB_ERR)
			SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLST, LB_ADDSTRING, 0, (LPARAM)str2);
	}
}

#define FLAGCONFLICTS(pthis, hwnd, id, name, orig)				 \
	if ((orig) != (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)) \
		FlagItem(pthis, hwnd, id, name);

#define FLAGCONFLICTS2(pass, pthis, hwnd, id, name, orig)	     \
	if ((orig) != (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)) \
		FlagItem(pthis, hwnd, id, name, pass);

	

//if (pData->pMtl->GetUseSpecular() != (IsDlgButtonChecked(hwnd, IDC_USESPECULAR) == BST_CHECKED))
//	FlagItem(pthis, hwnd, IDC_USESPECULAR, "Use Specular");


// Unfortunately there is no support for param maps in user created rollouts so we need to
// reimplement all the interfaces (<sigh>...)
void MtlEditorDlg::MtlPropsUpdate(HWND hwnd, MtlEditorDlg* pthis)
{
	bLockMtlUpdates = true;

	// Scan through the materials and assign the properties as appropriate
	//SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLIST, LB_RESETCONTENT, 0, 0);

	for(int i = 0; i < pthis->nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)pthis->ptTree->GetItemData(pthis->selMtlList[i]);

		// Selected types in the ThumbnailTree that are not materials will be ignored
		if (pData->type == MTITYPE_MTL)
		{
			CStr name = pData->pMtl->GetName();

			if (i == 0)
			{
				// Handle Checkboxes
				if (pData->pMtl->IsOneSided())
					CheckDlgButton(hwnd, IDC_ONE_SIDED, BST_CHECKED);

				if (pData->pMtl->IsInvisible())
					CheckDlgButton(hwnd, IDC_INVISIBLE, BST_CHECKED);

				if (pData->pMtl->IsFrontFacing())
					CheckDlgButton(hwnd, IDC_FRONT_FACING, BST_CHECKED);

				if (pData->pMtl->IsTransparent())
					CheckDlgButton(hwnd, IDC_TRANSPARENT, BST_CHECKED);

				if (pData->pMtl->IsDecal())
					CheckDlgButton(hwnd, IDC_DECAL, BST_CHECKED);

				if (pData->pMtl->IsSorted())
					CheckDlgButton(hwnd, IDC_SORTED, BST_CHECKED);

				if (pData->pMtl->IsTwoSided())
					CheckDlgButton(hwnd, IDC_2SIDED, BST_CHECKED);

				if (pData->pMtl->IsWater())
					CheckDlgButton(hwnd, IDC_WATER, BST_CHECKED);

				if (pData->pMtl->IsGrassified())
					CheckDlgButton(hwnd, IDC_GRASSIFY, BST_CHECKED);

				if (pData->pMtl->GetUseSpecular())
					CheckDlgButton(hwnd, IDC_USESPECULAR, BST_CHECKED);

				if (pData->pMtl->GetAlphaCutoff() == 1)
					CheckDlgButton(hwnd, IDC_USEDEFAC, BST_CHECKED);

				if (pData->pMtl->IsUsingMultipassInViewport())
					CheckDlgButton(hwnd, IDC_RENDERMULTI, BST_CHECKED);

				// Handle Spinners
				float fDrawOrder = pData->pMtl->GetDrawOrderId();
				if (fDrawOrder > 1000.0f)
					fDrawOrder -= 1000.0f;

				pthis->pDrawOrderSpin->SetValue(fDrawOrder, TRUE);
				pthis->pGrassHeightSpin->SetValue(pData->pMtl->GetGrassHeight(), TRUE);
				pthis->pGrassLayersSpin->SetValue(pData->pMtl->GetGrassLayers(), TRUE);
				pthis->pBasePassSpin->SetValue(pData->pMtl->GetBasePass() + 1, TRUE);
				pthis->pAlphaCutoffSpin->SetValue(pData->pMtl->GetAlphaCutoff(), TRUE);
				pthis->pSpecularPowerSpin->SetValue(pData->pMtl->GetSpecularPower(), TRUE);
				pthis->pSpecularColor->SetColor(pData->pMtl->GetSpecularColor(), TRUE);

				UpdateMtlPropStates(hwnd);
			}
			else
			{
				// Flag any items that have conflicts
				// Handle Checkboxes
				FLAGCONFLICTS(pthis, hwnd, IDC_ONE_SIDED,    "One-Sided",    pData->pMtl->IsOneSided());
				FLAGCONFLICTS(pthis, hwnd, IDC_INVISIBLE,    "Invisible",    pData->pMtl->IsInvisible());
				FLAGCONFLICTS(pthis, hwnd, IDC_FRONT_FACING, "Font-Facing",  pData->pMtl->IsFrontFacing());
				FLAGCONFLICTS(pthis, hwnd, IDC_TRANSPARENT,  "Transparent",  pData->pMtl->IsTransparent());
				FLAGCONFLICTS(pthis, hwnd, IDC_DECAL,        "Decal",        pData->pMtl->IsDecal());
				FLAGCONFLICTS(pthis, hwnd, IDC_SORTED,       "Sorted",       pData->pMtl->IsSorted());
				FLAGCONFLICTS(pthis, hwnd, IDC_2SIDED,       "Two-sided",    pData->pMtl->IsTwoSided());
				FLAGCONFLICTS(pthis, hwnd, IDC_WATER,        "Water",        pData->pMtl->IsWater());
				FLAGCONFLICTS(pthis, hwnd, IDC_GRASSIFY,     "Grassify",     pData->pMtl->IsGrassified());
				FLAGCONFLICTS(pthis, hwnd, IDC_USESPECULAR,  "Use Specular", pData->pMtl->GetUseSpecular());
				
				FLAGCONFLICTS(pthis, hwnd, IDC_RENDERMULTI, "Renter Multi-pass in viewport", pData->pMtl->IsUsingMultipassInViewport());

				if (pData->pMtl->GetAlphaCutoff() == 1 && IsDlgButtonChecked(hwnd, IDC_USEDEFAC) != BST_CHECKED ||
					pData->pMtl->GetAlphaCutoff() != 1 && IsDlgButtonChecked(hwnd, IDC_USEDEFAC) == BST_CHECKED)
					FlagItem(pthis, hwnd, IDC_USEDEFAC, "Use default Alpha Cut-Off");

				// Handle Spinners
				float fDrawOrder = pData->pMtl->GetDrawOrderId();
				if (fDrawOrder > 1000.0f)
					fDrawOrder -= 1000.0f;

				if (pthis->pDrawOrderSpin->GetFVal() != fDrawOrder)
					FlagItem(pthis, hwnd, IDC_DRAW_ORDER_ID, "Draw Order");

				if (pthis->pGrassHeightSpin->GetFVal() != pData->pMtl->GetGrassHeight())
					FlagItem(pthis, hwnd, IDC_EDITHEIGHT, "Grass Height");

				if (pthis->pGrassLayersSpin->GetIVal() != pData->pMtl->GetGrassLayers())
					FlagItem(pthis, hwnd, IDC_EDITLAYERS, "Grass Layers");

				if (pthis->pBasePassSpin->GetIVal() != pData->pMtl->GetBasePass() + 1)
					FlagItem(pthis, hwnd, IDC_BASE_PASS, "Base Pass");

				if (pthis->pAlphaCutoffSpin->GetIVal() != pData->pMtl->GetAlphaCutoff())
					FlagItem(pthis, hwnd, IDC_ALPHA_CUTOFF, "Alpha Cutoff");

				if (pthis->pSpecularPowerSpin->GetFVal() != pData->pMtl->GetSpecularPower())
					FlagItem(pthis, hwnd, IDC_SPECPOW, "Specular Power");
				
				if (pthis->pSpecularColor->GetColor() != pData->pMtl->GetSpecularColor())
					FlagItem(pthis, hwnd, IDC_SPECCOLOR, "Specular Color");
			}
		}
	}

	//if (pthis->nSelMaterials > 1 && SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLST, LB_GETCOUNT, 0, 0) == 0)
	//	SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLST, LB_ADDSTRING, 0, (LPARAM)"No Conflicts");

	bLockMtlUpdates = false;
}

void MtlEditorDlg::MtlFlagsUpdate(HWND hwnd, MtlEditorDlg* pthis)
{
	bLockMtlFlagUpdates = true;

	// Scan through the materials and assign the properties as appropriate
	//SendDlgItemMessage(pthis->hwnd, IDC_CONFLICTLIST, LB_RESETCONTENT, 0, 0);

	for(int i = 0; i < pthis->nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)pthis->ptTree->GetItemData(pthis->selMtlList[i]);

		// Selected types in the ThumbnailTree that are not materials will be ignored
		if (pData->type == MTITYPE_MTL)
		{
			if (i == 0)
			{
				int index = pData->pMtl->GetTerrainType();
				SendDlgItemMessage(hwnd, IDC_TERRAIN, CB_SETCURSEL, (WPARAM)index, 0);
			}
			else
			{
				int index = SendDlgItemMessage(hwnd, IDC_TERRAIN, CB_GETCURSEL, 0, 0);

				if (index != CB_ERR)
					if (pData->pMtl->GetTerrainType() != index)
						FlagItem(pthis, hwnd, IDC_TERRAIN, "Terrain");
			}
		}
	}

	bLockMtlFlagUpdates = false;
}

void MtlEditorDlg::GetFileFromMapString(char* strMap, char* filename)
{
	char* pos    = strMap;
	char* posOut = filename;

	while(*pos != 0)
	{
		if (*pos == '(')
		{
			pos++;

			while(*pos != ')' && *pos != 0)
			{
				*posOut = *pos;
				pos++;
				posOut++;
			}
		}

		pos++;
	}

	*posOut = 0;
}

void MtlEditorDlg::MtlPassUpdate(HWND hwnd, MtlEditorDlg* pthis, int pass)
{
	bLockMtlPassUpdates[pass] = true;

	for(int i = 0; i < pthis->nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)pthis->ptTree->GetItemData(pthis->selMtlList[i]);

		// Selected types in the ThumbnailTree that are not materials will be ignored
		if (pData->type == MTITYPE_MTL)
		{
			if (i == 0)
			{
				// Handle checkboxes
				if (pData->pMtl->PassEnabled(pass))
					CheckDlgButton(hwnd, IDC_ENABLED, BST_CHECKED);

				if (pData->pMtl->GetColorLocked(pass))
					CheckDlgButton(hwnd, IDC_COLORLOCKED, BST_CHECKED);

				if (pData->pMtl->ColorEnabled(pass))
					CheckDlgButton(hwnd, IDC_HASCOLOR, BST_CHECKED);

				if (!pData->pMtl->GetVertexAlphaEnabledUISetting(pass))
					CheckDlgButton(hwnd, IDC_IGNORE_VERTEX_ALPHA, BST_CHECKED);

				if (pData->pMtl->PassEnabledInNetGames(pass))
					CheckDlgButton(hwnd, IDC_ABSENT_IN_NET_GAMES, BST_CHECKED);

				if (pData->pMtl->UVWibbleDataEnabled(pass))
					CheckDlgButton(hwnd, IDC_UV_ENABLE, BST_CHECKED);

				// Handle Spinners/Edit controls
				pthis->pTileUSpin[pass]->SetValue(pData->pMtl->GetEnvTileU(pass),TRUE);
				pthis->pTileVSpin[pass]->SetValue(pData->pMtl->GetEnvTileV(pass),TRUE);

				pthis->pFixedSourceAlphaSpin[pass]->SetValue(pData->pMtl->GetFixedValue(pass),TRUE);
				pthis->pUVelocitySpin[pass]->SetValue(pData->pMtl->GetUVel(pass),TRUE);
				pthis->pUFrequencySpin[pass]->SetValue(pData->pMtl->GetUFrequency(pass),TRUE);
				pthis->pUAmplitudeSpin[pass]->SetValue(pData->pMtl->GetUAmplitude(pass),TRUE);
				pthis->pUPhaseSpin[pass]->SetValue(pData->pMtl->GetUPhase(pass),TRUE);

				pthis->pVVelocitySpin[pass]->SetValue(pData->pMtl->GetVVel(pass),TRUE);
				pthis->pVFrequencySpin[pass]->SetValue(pData->pMtl->GetVFrequency(pass),TRUE);
				pthis->pVAmplitudeSpin[pass]->SetValue(pData->pMtl->GetVAmplitude(pass),TRUE);
				pthis->pVPhaseSpin[pass]->SetValue(pData->pMtl->GetVPhase(pass),TRUE);

				// Handle Colors
				pthis->pHasColorColor[pass]->SetColor(pData->pMtl->GetColor(pass));
				pthis->pAmbient[pass]->SetColor(pData->pMtl->GetAmbientColor(pass));
				pthis->pDiffuse[pass]->SetColor(pData->pMtl->GetDiffuseColor(pass));
				pthis->pSpecular[pass]->SetColor(pData->pMtl->GetSpecularColor(pass));

				// Handle Map
				char strMapName[1024] = "";
				Texmap* tmap = pData->pMtl->GetTexmap(pass);

				if (tmap)
				{
					strcpy(strMapName, tmap->GetName());

					if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
					{
						INExtTexture* next_tex = dynamic_cast<INExtTexture*>(tmap);
						BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);

						if (bmi)
						{
							char name[256];
							char ext[256];

							_splitpath(bmi->Name(), NULL, NULL, name, ext);
							strcat(name, ext);

							strcat(strMapName, " (");
							strcat(strMapName, name);
							strcat(strMapName, ")");
						}
					}
				}
	
				pthis->pMap[pass]->SetText(strMapName);

				// Handle Combo Boxes
				int mappingMode = pData->pMtl->GetMappingMode(pass);
				SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_SETCURSEL, (WPARAM)mappingMode, 0);
				
				int blendMode = pData->pMtl->GetBlendMode(pass);
				SendDlgItemMessage(hwnd, IDC_BLEND_MODE, CB_SETCURSEL, (WPARAM)blendMode, 0);
			}
			else
			{
				// Flag any items that have conflicts
				// Handle Checkboxes
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_ENABLED,             "Pass Enabled",         pData->pMtl->PassEnabled(pass));
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_COLORLOCKED,         "Color Locked",         pData->pMtl->GetColorLocked(pass));
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_HASCOLOR,            "Has Color",            pData->pMtl->ColorEnabled(pass));
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_IGNORE_VERTEX_ALPHA, "Ignore vertex alpha",  !pData->pMtl->GetVertexAlphaEnabledUISetting(pass));
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_ABSENT_IN_NET_GAMES, "Absent in net games",  pData->pMtl->PassEnabledInNetGames(pass));
				FLAGCONFLICTS2(pass, pthis, hwnd, IDC_UV_ENABLE,           "UV Properties Enable", pData->pMtl->UVWibbleDataEnabled(pass));

				// Handle Spinners/Edit controls
				if (pthis->pTileUSpin[pass]->GetFVal() != pData->pMtl->GetEnvTileU(pass))
					FlagItem(pthis, hwnd, IDC_ENV_TILE_U, "TileU", pass);

				if (pthis->pTileVSpin[pass]->GetFVal() != pData->pMtl->GetEnvTileV(pass))
					FlagItem(pthis, hwnd, IDC_ENV_TILE_V, "TileV", pass);
				
				if (pthis->pFixedSourceAlphaSpin[pass]->GetIVal() != pData->pMtl->GetFixedValue(pass))
					FlagItem(pthis, hwnd, IDC_FIXED, "Fixed Value", pass);

				if (pthis->pUVelocitySpin[pass]->GetFVal() != pData->pMtl->GetUVel(pass))
					FlagItem(pthis, hwnd, IDC_U_VEL, "U Velocity", pass);

				if (pthis->pUFrequencySpin[pass]->GetFVal() != pData->pMtl->GetUFrequency(pass))
					FlagItem(pthis, hwnd, IDC_U_FREQUENCY, "U Frequency", pass);

				if (pthis->pUAmplitudeSpin[pass]->GetFVal() != pData->pMtl->GetUAmplitude(pass))
					FlagItem(pthis, hwnd, IDC_U_AMPLITUDE, "U Amplitude", pass);

				if (pthis->pUPhaseSpin[pass]->GetFVal() != pData->pMtl->GetUPhase(pass))
					FlagItem(pthis, hwnd, IDC_U_PHASE, "U Phase", pass);

				if (pthis->pVVelocitySpin[pass]->GetFVal() != pData->pMtl->GetVVel(pass))
					FlagItem(pthis, hwnd, IDC_V_VEL, "V Velocity", pass);

				if (pthis->pVFrequencySpin[pass]->GetFVal() != pData->pMtl->GetVFrequency(pass))
					FlagItem(pthis, hwnd, IDC_V_FREQUENCY, "V Frequency", pass);

				if (pthis->pVAmplitudeSpin[pass]->GetFVal() != pData->pMtl->GetVAmplitude(pass))
					FlagItem(pthis, hwnd, IDC_V_AMPLITUDE, "V Amplitude", pass);

				if (pthis->pVPhaseSpin[pass]->GetFVal() != pData->pMtl->GetVPhase(pass))
					FlagItem(pthis, hwnd, IDC_V_PHASE, "V Phase", pass);

				// Handle Colors
				if (pthis->pHasColorColor[pass]->GetColor() != pData->pMtl->GetColor(pass))
					FlagItem(pthis, hwnd, IDC_COLOR, "Has Color Color", pass);

				if (pthis->pAmbient[pass]->GetColor() != pData->pMtl->GetAmbientColor(pass))
					FlagItem(pthis, hwnd, IDC_COLOR_A, "Ambient Color", pass);

				if (pthis->pDiffuse[pass]->GetColor() != pData->pMtl->GetDiffuseColor(pass))
					FlagItem(pthis, hwnd, IDC_COLOR_D, "Diffuse Color", pass);
				
				if (pthis->pSpecular[pass]->GetColor() != pData->pMtl->GetSpecularColor(pass))
					FlagItem(pthis, hwnd, IDC_COLOR_S, "Specular Color", pass);
				
				// Handle map
				char strMapName[1024];
				Texmap* tmap = pData->pMtl->GetTexmap(pass);
				if (tmap)
				{
					//strcpy(strMapName, tmap->GetName());

					if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
					{
						INExtTexture* next_tex = dynamic_cast<INExtTexture*>(tmap);
						BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);

						if (bmi)
						{
							char name[256];
							char ext[256];

							_splitpath(bmi->Name(), NULL, NULL, name, ext);
							strcat(name, ext);

							//strcat(strMapName, " (");
							//strcat(strMapName, name);
							//strcat(strMapName, ")");
							strcpy(strMapName, name);
						}
					}
				}
	
				char buf[1024];
				char bufName[256];

				pthis->pMap[pass]->GetText(buf, 1024);
				GetFileFromMapString(buf, bufName);

				if (strcmp(bufName, strMapName) != 0)
					FlagItem(pthis, hwnd, IDC_MAP, "Map", pass);
			}

		}
	}

	bLockMtlPassUpdates[pass] = false;
}

void MtlEditorDlg::TexPassUpdate(HWND hwnd, MtlEditorDlg* pthis)
{
	bLockTexPropUpdates = true;

	BitmapInfo bi;

	for(int i = 0; i < pthis->nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)pthis->ptTree->GetItemData(pthis->selMtlList[i]);

		// Selected types in the ThumbnailTree that are not materials will be ignored
		if (pData->type == MTITYPE_TEX)
		{
			if (i == 0)
			{
				if (pData->pTex->ShouldCompress(vPLAT_NGC))
					CheckDlgButton(hwnd, IDC_COMPRESS_NGC, TRUE);
				else
					CheckDlgButton(hwnd, IDC_COMPRESS_NGC, FALSE);

				if (pData->pTex->ShouldCompress(vPLAT_XBOX))
					CheckDlgButton(hwnd, IDC_COMPRESS_XBOX, TRUE);
				else
					CheckDlgButton(hwnd, IDC_COMPRESS_XBOX, FALSE);

				if (pData->pTex->Force24BitPalette())
					CheckDlgButton(hwnd, IDC_24_BIT, TRUE);
				else
					CheckDlgButton(hwnd, IDC_24_BIT, FALSE);

				if (pData->pTex->ChangeTransColor())
					CheckDlgButton(hwnd, IDC_CHANGE_TRANS, TRUE);
				else
					CheckDlgButton(hwnd, IDC_CHANGE_TRANS, FALSE);

				switch(pData->pTex->GetMipType(vPLAT_PS2))
				{
				case vMIP_TYPE_NONE:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_PS2,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_PS2,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_PS2, BST_UNCHECKED);
					break;

				case vMIP_TYPE_AUTO:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_PS2,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_PS2,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_PS2, BST_UNCHECKED);
					break;

				case vMIP_TYPE_MANUAL:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_PS2, BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_PS2,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_PS2,   BST_UNCHECKED);
					break;
				}

				switch(pData->pTex->GetMipType(vPLAT_NGC))
				{
				case vMIP_TYPE_NONE:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_NGC,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_NGC,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_NGC, BST_UNCHECKED);
					break;

				case vMIP_TYPE_AUTO:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_NGC,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_NGC,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_NGC, BST_UNCHECKED);
					break;

				case vMIP_TYPE_MANUAL:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_NGC, BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_NGC,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_NGC,   BST_UNCHECKED);
					break;
				}

				switch(pData->pTex->GetMipType(vPLAT_XBOX))
				{
				case vMIP_TYPE_NONE:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_XBOX,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_XBOX,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_XBOX, BST_UNCHECKED);
					break;

				case vMIP_TYPE_AUTO:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_XBOX,   BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_XBOX,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_XBOX, BST_UNCHECKED);
					break;

				case vMIP_TYPE_MANUAL:
					CheckDlgButton(hwnd, IDC_MIP_TYPE_MANUAL_XBOX, BST_CHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_XBOX,   BST_UNCHECKED);
					CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_XBOX,   BST_UNCHECKED);
					break;
				}

				if (pData->pTex->UsePS2Mips(vPLAT_NGC))
					CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_NGC, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_NGC, BST_UNCHECKED);

				if (pData->pTex->UsePS2Mips(vPLAT_XBOX))
					CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_XBOX, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_XBOX, BST_UNCHECKED);

				// Handle paletize options
				if (pData->pTex->UseBasePal(vPLAT_PS2, 0))
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP1, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP1, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_PS2, 1))
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP2, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP2, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_PS2, 2))
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP3, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP3, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_PS2, 3))
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP4, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_PS2_MIP4, BST_UNCHECKED);

				///

				if (pData->pTex->UseBasePal(vPLAT_NGC, 0))
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP1, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP1, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_NGC, 1))
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP2, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP2, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_NGC, 2))
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP3, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP3, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_NGC, 3))
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP4, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_NGC_MIP4, BST_UNCHECKED);

				///

				if (pData->pTex->UseBasePal(vPLAT_XBOX, 0))
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP1, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP1, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_XBOX, 1))
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP2, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP2, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_XBOX, 2))
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP3, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP3, BST_UNCHECKED);

				if (pData->pTex->UseBasePal(vPLAT_XBOX, 3))
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP4, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_PAL_XBX_MIP4, BST_UNCHECKED);

				///

				if (pData->pTex->GetCropPlaceMode() == vPLACE)
				{
					CheckDlgButton(hwnd, IDC_BM_CLIP, BST_CHECKED);
					CheckDlgButton(hwnd, IDC_BM_PLACE, BST_CHECKED);
				}

				if (pData->pTex->GetCropPlaceMode() == vCROP)
				{
					CheckDlgButton(hwnd, IDC_BM_CLIP, BST_CHECKED);
					CheckDlgButton(hwnd, IDC_BM_CROP, BST_CHECKED);
				}

				if (pData->pTex->GetCropPlaceApply())
					CheckDlgButton(hwnd, IDC_BM_CLIP, BST_CHECKED);
				else
					CheckDlgButton(hwnd, IDC_BM_CLIP, BST_UNCHECKED);

				pthis->pKBiasSpin->SetValue(pData->pTex->GetMipMapK(), TRUE);
				pthis->pCropPlaceUSpin->SetValue(pData->pTex->GetCropPlaceU(), TRUE);
				pthis->pCropPlaceWSpin->SetValue(pData->pTex->GetCropPlaceW(), TRUE);
				pthis->pCropPlaceVSpin->SetValue(pData->pTex->GetCropPlaceV(), TRUE);
				pthis->pCropPlaceHSpin->SetValue(pData->pTex->GetCropPlaceH(), TRUE);

				// Set trans color
				pthis->pTransColor->SetColor(pData->pTex->GetTransColor(), TRUE);

				// Handle texture names
				BitmapInfo* bmi;
				bmi = pData->pTex->GetBaseMap(vPLAT_PS2);

				if (bmi)
					pthis->pPS2BaseMap->SetText((char*)bmi->Filename());
				else
					pthis->pPS2BaseMap->SetText("None");

				if (pData->pTex->UsePS2Mips(vPLAT_NGC))
					pthis->pNGCBaseMap->SetText("Use PS2");
				else
				{
					bmi = pData->pTex->GetBaseMap(vPLAT_NGC);

					if (bmi)
						pthis->pNGCBaseMap->SetText((char*)bmi->Filename());
					else
						pthis->pNGCBaseMap->SetText("None");
				}

				if (pData->pTex->UsePS2Mips(vPLAT_XBOX))
					pthis->pXBOXBaseMap->SetText("Use PS2");
				else
				{
					bmi = pData->pTex->GetBaseMap(vPLAT_XBOX);

					if (bmi)
						pthis->pXBOXBaseMap->SetText((char*)bmi->Filename());
					else
						pthis->pXBOXBaseMap->SetText("None");
				}
				
				int j;

				for(j = 0; j < 5; j++)
				{
					bmi = pData->pTex->GetManualMap(vPLAT_PS2, j);

					if (bmi)
						pthis->pPS2Mip[j]->SetText((char*)bmi->Filename());
					else
						pthis->pPS2Mip[j]->SetText("None");
				}

				for(j = 0; j < 5; j++)
				{
					bmi = pData->pTex->GetManualMap(vPLAT_NGC, j);

					if (bmi)
						pthis->pNGCMip[j]->SetText((char*)bmi->Filename());
					else
						pthis->pNGCMip[j]->SetText("None");
				}

				for(j = 0; j < 5; j++)
				{
					bmi = pData->pTex->GetManualMap(vPLAT_XBOX, j);

					if (bmi)
						pthis->pXBOXMip[j]->SetText((char*)bmi->Filename());
					else
						pthis->pXBOXMip[j]->SetText("None");
				}

				// Handle combo box settings
				int minFilter = pData->pTex->GetMinFilterMode();
				SendDlgItemMessage(hwnd, IDC_MIN_FILTERING_MODE, CB_SETCURSEL, (WPARAM)minFilter, 0);

				int magFilter = pData->pTex->GetMagFilterMode();
				SendDlgItemMessage(hwnd, IDC_MAG_FILTERING_MODE, CB_SETCURSEL, (WPARAM)magFilter, 0);

				int mipFilter = pData->pTex->GetMipFilterMode();
				SendDlgItemMessage(hwnd, IDC_MIN_MIP_FILTERING_MODE, CB_SETCURSEL, (WPARAM)mipFilter, 0);

				// Handle base map settings
				BitmapInfo *biPS2, *bi;

				biPS2 = pData->pTex->GetBaseMap(vPLAT_PS2);

				if (biPS2)
					pthis->pPS2BaseMap->SetText((char*)biPS2->Filename());
				else
					pthis->pPS2BaseMap->SetText("None");

				bi = pData->pTex->GetBaseMap(vPLAT_NGC);

				if (bi == biPS2)
					pthis->pNGCBaseMap->SetText("Use PS2");
				else
				{
					if (bi)
						pthis->pNGCBaseMap->SetText((char*)bi->Filename());	
					else
						pthis->pNGCBaseMap->SetText("None");
				}

				bi = pData->pTex->GetBaseMap(vPLAT_XBOX);

				if (bi == biPS2)
					pthis->pXBOXBaseMap->SetText("Use PS2");
				else
				{
					if (bi)
						pthis->pXBOXBaseMap->SetText((char*)bi->Filename());
					else
						pthis->pXBOXBaseMap->SetText("None");
				}
				
				// Handle mip map changes
				for(j = 0; j < 5; j++)
				{
					bi = pData->pTex->GetManualMap(vPLAT_PS2, j);

					if (bi)
						pthis->pPS2Mip[j]->SetText((char*)bi->Filename());
					else
						pthis->pPS2Mip[j]->SetText("None");
				}

				for(j = 0; j < 5; j++)
				{
					bi = pData->pTex->GetManualMap(vPLAT_NGC, j);

					if (bi)
						pthis->pNGCMip[j]->SetText((char*)bi->Filename());
					else
						pthis->pNGCMip[j]->SetText("None");
				}

				for(j = 0; j < 5; j++)
				{
					bi = pData->pTex->GetManualMap(vPLAT_XBOX, j);

					if (bi)
						pthis->pXBOXMip[j]->SetText((char*)bi->Filename());
					else
						pthis->pXBOXMip[j]->SetText("None");
				}
			}
			else
			{
				// Flag conflicts
				// Handle buttons
				FLAGCONFLICTS(pthis, hwnd, IDC_COMPRESS_NGC,  "Compress NGC",       pData->pTex->ShouldCompress(vPLAT_NGC));
				FLAGCONFLICTS(pthis, hwnd, IDC_COMPRESS_XBOX, "Compress XBOX",      pData->pTex->ShouldCompress(vPLAT_XBOX));
				FLAGCONFLICTS(pthis, hwnd, IDC_24_BIT,        "Force 24-bit",       pData->pTex->Force24BitPalette());
				FLAGCONFLICTS(pthis, hwnd, IDC_CHANGE_TRANS,  "Change Trans Color", pData->pTex->ChangeTransColor());
				
				FLAGCONFLICTS(pthis, hwnd, IDC_USE_PS2_MIPMAPS_NGC,  "Use PS2 Mips (NGC)",    pData->pTex->UsePS2Mips(vPLAT_NGC));
				FLAGCONFLICTS(pthis, hwnd, IDC_USE_PS2_MIPMAPS_XBOX, "Use PS2 Mips (XBOX)",   pData->pTex->UsePS2Mips(vPLAT_XBOX));
				
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_PS2_MIP1,         "Use BasePal PS2 Mip 1", pData->pTex->UseBasePal(vPLAT_PS2, 0));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_PS2_MIP2,         "Use BasePal PS2 Mip 2", pData->pTex->UseBasePal(vPLAT_PS2, 1));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_PS2_MIP3,         "Use BasePal PS2 Mip 3", pData->pTex->UseBasePal(vPLAT_PS2, 2));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_PS2_MIP4,         "Use BasePal PS2 Mip 4", pData->pTex->UseBasePal(vPLAT_PS2, 3));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_PS2_MIP5,         "Use BasePal PS2 Mip 5", pData->pTex->UseBasePal(vPLAT_PS2, 4));

				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_NGC_MIP1,         "Use BasePal NGC Mip 1", pData->pTex->UseBasePal(vPLAT_NGC, 0));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_NGC_MIP2,         "Use BasePal NGC Mip 2", pData->pTex->UseBasePal(vPLAT_NGC, 1));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_NGC_MIP3,         "Use BasePal NGC Mip 3", pData->pTex->UseBasePal(vPLAT_NGC, 2));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_NGC_MIP4,         "Use BasePal NGC Mip 4", pData->pTex->UseBasePal(vPLAT_NGC, 3));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_NGC_MIP5,         "Use BasePal NGC Mip 5", pData->pTex->UseBasePal(vPLAT_NGC, 4));

				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_XBX_MIP1,        "Use BasePal XBOX Mip 1", pData->pTex->UseBasePal(vPLAT_XBOX, 0));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_XBX_MIP2,        "Use BasePal XBOX Mip 2", pData->pTex->UseBasePal(vPLAT_XBOX, 1));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_XBX_MIP3,        "Use BasePal XBOX Mip 3", pData->pTex->UseBasePal(vPLAT_XBOX, 2));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_XBX_MIP4,        "Use BasePal XBOX Mip 4", pData->pTex->UseBasePal(vPLAT_XBOX, 3));
				FLAGCONFLICTS(pthis, hwnd, IDC_PAL_XBX_MIP5,        "Use BasePal XBOX Mip 5", pData->pTex->UseBasePal(vPLAT_XBOX, 4));

				if (pData->pTex->GetCropPlaceMode() == vPLACE)
				{
					if (IsDlgButtonChecked(hwnd, IDC_BM_PLACE) != BST_CHECKED)
						FlagItem(pthis, hwnd, IDC_BM_PLACE, "Place Mode");
				}

				if (pData->pTex->GetCropPlaceMode() == vCROP)
				{
					if (IsDlgButtonChecked(hwnd, IDC_BM_CROP) != BST_CHECKED)
						FlagItem(pthis, hwnd, IDC_BM_CROP, "Crop Mode");
				}

				FLAGCONFLICTS(pthis, hwnd, IDC_BM_CLIP,            "Crop/Place Apply", pData->pTex->GetCropPlaceApply());

				// Handle MipType fields
				switch(pData->pTex->GetMipType(vPLAT_PS2))
				{
				case vMIP_TYPE_NONE:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_NONE_PS2) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_PS2, "PS2 MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_PS2);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_PS2);
					}
					break;

				case vMIP_TYPE_AUTO:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_AUTO_PS2) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_PS2, "PS2 MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_PS2);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_PS2);
					}
					break;

				case vMIP_TYPE_MANUAL:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_MANUAL_PS2) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_PS2, "PS2 MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_PS2);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_PS2);
					}
					break;
				}

				switch(pData->pTex->GetMipType(vPLAT_NGC))
				{
				case vMIP_TYPE_NONE:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_NONE_NGC) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_NGC, "NGC MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_NGC);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_NGC);
					}
					break;

				case vMIP_TYPE_AUTO:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_AUTO_NGC) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_NGC, "NGC MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_NGC);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_NGC);
					}
					break;

				case vMIP_TYPE_MANUAL:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_MANUAL_NGC) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_NGC, "NGC MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_NGC);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_NGC);
					}
					break;
				}

				switch(pData->pTex->GetMipType(vPLAT_XBOX))
				{
				case vMIP_TYPE_NONE:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_NONE_XBOX) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_XBOX, "XBOX MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_XBOX);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_XBOX);
					}
					break;

				case vMIP_TYPE_AUTO:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_AUTO_XBOX) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_XBOX, "XBOX MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_XBOX);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_XBOX);
					}
					break;

				case vMIP_TYPE_MANUAL:
					if (IsDlgButtonChecked(hwnd, IDC_MIP_TYPE_MANUAL_XBOX) != BST_CHECKED)
					{
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_NONE_XBOX, "XBOX MipType");
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_AUTO_XBOX);
						FlagItem(pthis, hwnd, IDC_MIP_TYPE_MANUAL_XBOX);
					}
					break;
				}

				// Handle spinners
				if (pData->pTex->GetMipMapK() != pthis->pKBiasSpin->GetFVal())
					FlagItem(pthis, hwnd, IDC_MIPMAP_K, "MipMap KBias");

				if (pData->pTex->GetCropPlaceU() != pthis->pCropPlaceUSpin->GetFVal())
					FlagItem(pthis, hwnd, IDC_CLIP_X, "Crop/Place U");

				if (pData->pTex->GetCropPlaceW() != pthis->pCropPlaceWSpin->GetFVal())
					FlagItem(pthis, hwnd, IDC_CLIP_W, "Crop/Place W");

				if (pData->pTex->GetCropPlaceV() != pthis->pCropPlaceVSpin->GetFVal())
					FlagItem(pthis, hwnd, IDC_CLIP_Y, "Crop/Place V");

				if (pData->pTex->GetCropPlaceH() != pthis->pCropPlaceHSpin->GetFVal())
					FlagItem(pthis, hwnd, IDC_CLIP_H, "Crop/Place H");

				// Set trans color
				COLORREF colorA = pthis->pTransColor->GetColor();
				COLORREF colorB = pData->pTex->GetTransColor();

				if (colorA != colorB)
					FlagItem(pthis, hwnd, IDC_TRANS_COLOR, "Transparent Color");

				// Handle combo boxes
				int minFilter = SendDlgItemMessage(hwnd, IDC_MIN_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				if (pData->pTex->GetMinFilterMode() != minFilter)
					FlagItem(pthis, hwnd, IDC_MIN_FILTERING_MODE, "MinFilter");

				int magFilter = SendDlgItemMessage(hwnd, IDC_MAG_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				if (pData->pTex->GetMagFilterMode() != magFilter)
					FlagItem(pthis, hwnd, IDC_MAG_FILTERING_MODE, "MagFilter");

				int mipFilter = SendDlgItemMessage(hwnd, IDC_MIN_MIP_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				if (pData->pTex->GetMipFilterMode() != mipFilter)
					FlagItem(pthis, hwnd, IDC_MIN_MIP_FILTERING_MODE, "MipFilter");

				// Handle base maps
				BitmapInfo *bmiPS2, *bmi;
				char newBuf[1024];
				char buf[1024];

				pthis->pPS2BaseMap->GetText(buf, 1023);

				bmiPS2 = pData->pTex->GetBaseMap(vPLAT_PS2);

				if (bmiPS2)
					strcpy(newBuf, bmiPS2->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_NAME_PS2, "PS2 Basemap");

				pthis->pNGCBaseMap->GetText(buf, 1023);

				bmi = pData->pTex->GetBaseMap(vPLAT_NGC);

				if (bmiPS2 == bmi)
					strcpy(newBuf, "Use PS2");
				else
				{
					if (bmi)
						strcpy(newBuf, bmi->Filename());
					else
						strcpy(newBuf, "None");
				}

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_NAME_NGC, "NGC Basemap");

				pthis->pXBOXBaseMap->GetText(buf, 1023);

				bmi = pData->pTex->GetBaseMap(vPLAT_XBOX);

				if (bmiPS2 == bmi)
					strcpy(newBuf, "Use PS2");
				else
				{
					if (bmi)
						strcpy(newBuf, bmi->Filename());
					else
						strcpy(newBuf, "None");
				}

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_NAME_XBOX, "XBOX Basemap");
					
				// Handle mip maps
				pthis->pPS2Mip[0]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_PS2, 0);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_1_PS2, "PS2 Mip 1");

				pthis->pPS2Mip[1]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_PS2, 1);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_2_PS2, "PS2 Mip 2");

				pthis->pPS2Mip[2]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_PS2, 2);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(bmi->Filename(), buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_3_PS2, "PS2 Mip 3");

				pthis->pPS2Mip[3]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_PS2, 3);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_4_PS2, "PS2 Mip 4");

				pthis->pPS2Mip[4]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_PS2, 4);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(bmi->Filename(), buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_5_PS2, "PS2 Mip 5");

				////
				pthis->pNGCMip[0]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_NGC, 0);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_1_NGC, "NGC Mip 1");

				pthis->pNGCMip[1]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_NGC, 1);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_2_NGC, "NGC Mip 2");

				pthis->pNGCMip[2]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_NGC, 2);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_3_NGC, "NGC Mip 3");

				pthis->pNGCMip[3]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_NGC, 3);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_4_NGC, "NGC Mip 4");

				pthis->pNGCMip[4]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_NGC, 4);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_5_NGC, "NGC Mip 5");
				////

				pthis->pXBOXMip[0]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_XBOX, 0);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_1_XBOX, "XBOX Mip 1");

				pthis->pXBOXMip[1]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_XBOX, 1);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_2_XBOX, "XBOX Mip 2");

				pthis->pXBOXMip[2]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_XBOX, 2);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_3_XBOX, "XBOX Mip 3");

				pthis->pXBOXMip[3]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_XBOX, 3);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_4_XBOX, "XBOX Mip 4");

				pthis->pXBOXMip[4]->GetText(buf, 1023);
				bmi = pData->pTex->GetManualMap(vPLAT_XBOX, 4);

				if (bmi)
					strcpy(newBuf, bmi->Filename());
				else
					strcpy(newBuf, "None");

				if (_stricmp(newBuf, buf) != 0)
					FlagItem(pthis, hwnd, IDC_BMTEX_MIP_NAME_5_XBOX, "XBOX Mip 5");

				////
			}
		}
	}

	bLockTexPropUpdates = false;
}

void MtlEditorDlg::HookWindow(HWND hwnd)
{
	WndHookData* pData  = new WndHookData;
	pData->extdata      = (DWORD)GetWindowLong(hwnd, GWL_USERDATA);
	pData->origproc     = (WNDPROC)GetWindowLong(hwnd, GWL_WNDPROC);
	pData->bHighlighted = false;

	hookDB.PutItem((DWORD)hwnd, pData);

	LONG style = GetWindowLong(hwnd, GWL_EXSTYLE);
	SetWindowLong(hwnd, GWL_WNDPROC, (LONG)SubClassDlgProc);
}

BOOL CALLBACK MtlEditorDlg::wndHookEnum(HWND hwnd, LPARAM lParam)
{
	MtlEditorDlg* pthis = (MtlEditorDlg*)lParam;
	pthis->HookWindow(hwnd);

	return TRUE;
}

void MtlEditorDlg::HookAllWindows(HWND hwnd)
{
	EnumChildWindows(hwnd, wndHookEnum, (LPARAM)this);
}

LRESULT MtlEditorDlg::SubClassDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	//WndHookData* pData = (WndHookData*)GetWindowLong(hwnd, GWL_USERDATA);
	WndHookData* pData = hookDB.GetItem((DWORD)hwnd, false);
	WNDPROC origWndProc = pData->origproc;

	switch(msg)
	{
	case WM_DESTROY:
		{
			// Uninstall hook on window destruction
			//SetWindowLong(hwnd, GWL_USERDATA, pData->extdata);
			SetWindowLong(hwnd, GWL_WNDPROC, (LONG)pData->origproc);
			hookDB.FlushItem((DWORD)hwnd);
			delete pData;
		}
		break;

	case BM_SETSTATE:
		{
			LRESULT rVal = CallWindowProc(pData->origproc, hwnd, msg, wParam, lParam);
			InvalidateRect(hwnd, NULL, TRUE);
			return rVal;
		}
		break;

	case WM_NCPAINT:
	case WM_PAINT:
		{
			// Intercept paint message handling so we can draw over the control immediately
			// when it finishes rendering
			LRESULT rVal = CallWindowProc(pData->origproc, hwnd, msg, wParam, lParam);

			// Do custom paint
			if (pData->bHighlighted)
			{
				HDC  hdc;
				RECT rc;

				hdc = GetWindowDC(hwnd);
				GetWindowRect(hwnd, &rc);
				OffsetRect(&rc, -rc.left, -rc.top);

				if (!IsRectEmpty(&rc))
				{
					PatBlt(hdc, rc.left, rc.top, rc.right - rc.left, DINV,  DSTINVERT);
					PatBlt(hdc, rc.left, rc.bottom - DINV, DINV, -(rc.bottom - rc.top - 2 * DINV), DSTINVERT);
					PatBlt(hdc, rc.right - DINV, rc.top + DINV, DINV, rc.bottom - rc.top - 2 * DINV, DSTINVERT);
					PatBlt(hdc, rc.right, rc.bottom - DINV, -(rc.right - rc.left), DINV, DSTINVERT);     
				} 

				ReleaseDC(hwnd, hdc); 
			}

			return rVal;
		}
	}

	return CallWindowProc(origWndProc, hwnd, msg, wParam, lParam);
}

void MtlEditorDlg::ApplyMtlProps(HWND hwnd, int id)
{
	bLockMtlUpdates = true;

	for(int i = 0; i < nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)ptTree->GetItemData(selMtlList[i]);

		switch(id)
		{
		case IDC_ONE_SIDED:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetOneSided(true);
			else
				pData->pMtl->SetOneSided(false);
			break;

		case IDC_INVISIBLE:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetInvisible(true);
			else
				pData->pMtl->SetInvisible(false);
			break;

		case IDC_FRONT_FACING:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetFrontFacing(true);
			else
				pData->pMtl->SetFrontFacing(false);
			break;

		case IDC_TRANSPARENT:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetTransparent(true);
			else
				pData->pMtl->SetTransparent(false);
			break;

		case IDC_DECAL:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetDecal(true);
			else
				pData->pMtl->SetDecal(false);
			break;

		case IDC_2SIDED:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetTwoSided(true);
			else
				pData->pMtl->SetTwoSided(false);
			break;

		case IDC_WATER:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetWater(true);
			else
				pData->pMtl->SetWater(false);
			break;

		case IDC_SORTED:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetSorted(true);
			else
				pData->pMtl->SetSorted(false);
			break;

		case IDC_GRASSIFY:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetGrassified(true);
			else
				pData->pMtl->SetGrassified(false);
			break;

		case IDC_USESPECULAR:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetUseSpecular(true);
			else
				pData->pMtl->SetUseSpecular(false);
			break;

		case IDC_USEDEFAC:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetAlphaCutoff(1);
			else
				pData->pMtl->SetAlphaCutoff(pAlphaCutoffSpin->GetIVal());
			break;

		case IDC_RENDERMULTI:
			if (IsDlgButtonChecked(hwnd, id) == BST_CHECKED)
				pData->pMtl->SetUsingMultipassInViewport(true);
			else
				pData->pMtl->SetUsingMultipassInViewport(false);
			break;

			////////////////////////////////////////////////////////////////
		case IDC_CUTOFFFUNC:
			// Deprecated
			break;

		case IDC_DRAW_ORDER_ID_SPINNER:
			pData->pMtl->SetDrawOrderId(pDrawOrderSpin->GetFVal());
			break;

		case IDC_SPINHEIGHT:
			pData->pMtl->SetGrassHeight(pGrassHeightSpin->GetFVal());
			break;

		case IDC_SPINLAYERS:
			pData->pMtl->SetGrassLayers(pGrassLayersSpin->GetIVal());
			break;

		case IDC_BASE_PASS_SPINNER:
			pData->pMtl->SetBasePass(pBasePassSpin->GetIVal());
			break;

		case IDC_ALPHA_CUTOFF_SPINNER:
			pData->pMtl->SetAlphaCutoff(pAlphaCutoffSpin->GetIVal());
			break;

		case IDC_SPECPOWSPIN:
			pData->pMtl->SetSpecularPower(pSpecularPowerSpin->GetFVal());
			break;

		case IDC_SPECCOLOR:
			pData->pMtl->SetSpecularColor(pSpecularColor->GetColor());
			break;
			//////////////////////////////////////////////////////////////////
		}
	}

	bLockMtlUpdates = false;
}

void MtlEditorDlg::ApplyMtlFlagProps(HWND hwnd, int id)
{
	bLockMtlFlagUpdates = true;

	for(int i = 0; i < nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)ptTree->GetItemData(selMtlList[i]);

		switch(id)
		{
		case IDC_TERRAIN:
			{
				int index = SendDlgItemMessage(hwnd, IDC_TERRAIN, CB_GETCURSEL, 0, 0);

				if (index != CB_ERR)
					pData->pMtl->SetTerrainType(index);
			}
			break;
		}
	}

	bLockMtlFlagUpdates = false;
}

void MtlEditorDlg::ApplyMtlPassProps(HWND hwnd, int id, int pass)
{
	bLockMtlPassUpdates[pass] = true;

	BitmapInfo bi;

	for(int i = 0; i < nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)ptTree->GetItemData(selMtlList[i]);

		switch(id)
		{
		// Handle Checkboxes
		case IDC_ENABLED:
			if (IsDlgButtonChecked(hwnd, IDC_ENABLED) == BST_CHECKED)
				pData->pMtl->SetPassEnabled(pass, TRUE);
			else
				pData->pMtl->SetPassEnabled(pass, FALSE);
			break;

		case IDC_COLORLOCKED:
			if (IsDlgButtonChecked(hwnd, IDC_COLORLOCKED) == BST_CHECKED)
				pData->pMtl->SetColorLocked(pass, TRUE);
			else
				pData->pMtl->SetColorLocked(pass, FALSE);
			break;

		case IDC_HASCOLOR:
			if (IsDlgButtonChecked(hwnd, IDC_HASCOLOR) == BST_CHECKED)
				pData->pMtl->SetColorEnabled(pass, TRUE);
			else
				pData->pMtl->SetColorEnabled(pass, FALSE);
			break;

		case IDC_IGNORE_VERTEX_ALPHA:
			if (IsDlgButtonChecked(hwnd, IDC_IGNORE_VERTEX_ALPHA) == BST_CHECKED)
				pData->pMtl->SetVertexAlphaEnabled(pass, FALSE);
			else
				pData->pMtl->SetVertexAlphaEnabled(pass, TRUE);
			break;

		case IDC_ABSENT_IN_NET_GAMES:
			if (IsDlgButtonChecked(hwnd, IDC_ABSENT_IN_NET_GAMES) == BST_CHECKED)
				pData->pMtl->SetPassEnabledInNetGames(pass, FALSE);
			else
				pData->pMtl->SetPassEnabledInNetGames(pass, TRUE);
			break;

		case IDC_UV_ENABLE:
			if (IsDlgButtonChecked(hwnd, IDC_UV_ENABLE) == BST_CHECKED)
				pData->pMtl->SetUVWibbleDataEnabled(pass, TRUE);
			else
				pData->pMtl->SetUVWibbleDataEnabled(pass, FALSE);
			break;

		// Handle Edit/Spin Values ////////////////////////////////////////////////////////
		case IDC_ENV_TILE_U_SPINNER:
			pData->pMtl->SetEnvTileU(pass, pTileUSpin[pass]->GetFVal());
			break;

		case IDC_ENV_TILE_V_SPINNER:
			pData->pMtl->SetEnvTileV(pass, pTileVSpin[pass]->GetFVal());
			break;

		case IDC_FIXED_SPINNER:
			pData->pMtl->SetFixedValue(pass, pFixedSourceAlphaSpin[pass]->GetIVal());
			break;

		case IDC_U_VEL_SPINNER:
			pData->pMtl->SetUVel(pass, pUVelocitySpin[pass]->GetFVal());
			break;

		case IDC_U_FREQUENCY_SPINNER:
			pData->pMtl->SetUFrequency(pass, pUFrequencySpin[pass]->GetFVal());
			break;

		case IDC_U_AMPLITUDE_SPINNER:
			pData->pMtl->SetUAmplitude(pass, pUAmplitudeSpin[pass]->GetFVal());
			break;

		case IDC_U_PHASE_SPINNER:
			pData->pMtl->SetUPhase(pass, pUPhaseSpin[pass]->GetFVal());
			break;

		case IDC_V_VEL_SPINNER:
			pData->pMtl->SetVVel(pass, pVVelocitySpin[pass]->GetFVal());
			break;

		case IDC_V_FREQUENCY_SPINNER:
			pData->pMtl->SetVFrequency(pass, pVFrequencySpin[pass]->GetFVal());
			break;

		case IDC_V_AMPLITUDE_SPINNER:
			pData->pMtl->SetVAmplitude(pass, pVAmplitudeSpin[pass]->GetFVal());
			break;

		case IDC_V_PHASE_SPINNER:
			pData->pMtl->SetVPhase(pass, pVPhaseSpin[pass]->GetFVal());
			break;

		// Handle Color Values //////////////////////////////////////////////////////////
		case IDC_COLOR:
			pData->pMtl->SetColor(pass, pHasColorColor[pass]->GetColor());
			break;

		case IDC_COLOR_A:
			pData->pMtl->SetAmbientColor(pass, pAmbient[pass]->GetColor());
			break;

		case IDC_COLOR_D:
			pData->pMtl->SetDiffuseColor(pass, pDiffuse[pass]->GetColor());
			break;

		case IDC_COLOR_S:
			pData->pMtl->SetSpecularColor(pass, pSpecular[pass]->GetColor());
			break;

		// Handle combo list Values /////////////////////////////////////////////////////
		case IDC_MAPPING_MODE:
			{
				int index = SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_GETCURSEL, 0, 0);
				pData->pMtl->SetMappingMode(pass, index);
			}
			break;

		case IDC_BLEND_MODE:
			{
				int index = SendDlgItemMessage(hwnd, IDC_BLEND_MODE, CB_GETCURSEL, 0, 0);
				pData->pMtl->SetBlendMode(pass, (BlendModes)index);
			}
			break;
	break;
		}
	}

	bLockMtlPassUpdates[pass] = false;
}

bool MtlEditorDlg::GetBitmapInfo(int id, BitmapInfo* bi)
{
	switch(id)
	{
	// Basemaps
	case IDC_BMTEX_NAME_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Basemap"));
		break;

	case IDC_BMTEX_NAME_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Basemap"));
		break;

	case IDC_BMTEX_NAME_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Basemap"));
		break;

	// Mipmaps
	case IDC_BMTEX_MIP_NAME_1_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Mipmap Level 1"));
		break;

	case IDC_BMTEX_MIP_NAME_2_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Mipmap Level 2"));
		break;

	case IDC_BMTEX_MIP_NAME_3_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Mipmap Level 3"));
		break;

	case IDC_BMTEX_MIP_NAME_4_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Mipmap Level 4"));
		break;

	case IDC_BMTEX_MIP_NAME_5_PS2:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose PS2 Mipmap Level 5"));
		break;

	case IDC_BMTEX_MIP_NAME_1_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Mipmap Level 1"));
		break;

	case IDC_BMTEX_MIP_NAME_2_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Mipmap Level 2"));
		break;

	case IDC_BMTEX_MIP_NAME_3_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Mipmap Level 3"));
		break;

	case IDC_BMTEX_MIP_NAME_4_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Mipmap Level 4"));
		break;

	case IDC_BMTEX_MIP_NAME_5_NGC:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose NGC Mipmap Level 5"));
		break;

	case IDC_BMTEX_MIP_NAME_1_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose XBOX Mipmap Level 1"));
		break;

	case IDC_BMTEX_MIP_NAME_2_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose XBOX Mipmap Level 2"));
		break;

	case IDC_BMTEX_MIP_NAME_3_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose XBOX Mipmap Level 3"));
		break;

	case IDC_BMTEX_MIP_NAME_4_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose XBOX Mipmap Level 4"));
		break;

	case IDC_BMTEX_MIP_NAME_5_XBOX:
		TheManager->SelectFileInput(bi, hwnd, _T("Choose XBOX Mipmap Level 5"));
		break;

	default:
		return false;
	}

	SetFocus(hwnd);
	return true;
}

void MtlEditorDlg::ApplyTexProps(HWND hwndUI, int id)
{
	bLockTexPropUpdates = true;

	BitmapInfo bi, biClear;
	GetBitmapInfo(id, &bi);

	biClear.SetName( "None" );	
	biClear.SetWidth( 0 );
	biClear.SetHeight( 0 );

	for(int i = 0; i < nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)ptTree->GetItemData(selMtlList[i]);

		switch(id)
		{
		// Handle Map clear functions
		case IDC_BMTEX_NAME_PS2_CLEAR:
			{
				pData->pTex->SetBaseMap(vPLAT_PS2, &biClear);
				pPS2BaseMap->SetText("None");
				UpdateMtls();
			}
			break;

		case IDC_BMTEX_NAME_NGC_CLEAR:
			{
				pData->pTex->SetBaseMap(vPLAT_NGC, &biClear);
				pNGCBaseMap->SetText("Use PS2");
			}
			break;

		case IDC_BMTEX_NAME_XBOX_CLEAR:
			{
				pData->pTex->SetBaseMap(vPLAT_XBOX, &biClear);
				pXBOXBaseMap->SetText("Use PS2");
			}
			break;

		case IDC_BMTEX_MIP_1_PS2_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 0, &biClear);
				pPS2Mip[0]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_2_PS2_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 1, &biClear);
				pPS2Mip[1]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_3_PS2_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 2, &biClear);
				pPS2Mip[2]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_4_PS2_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 3, &biClear);
				pPS2Mip[3]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_5_PS2_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 4, &biClear);
				pPS2Mip[4]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_1_NGC_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 0, &biClear);
				pNGCMip[0]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_2_NGC_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 1, &biClear);
				pNGCMip[1]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_3_NGC_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 2, &biClear);
				pNGCMip[2]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_4_NGC_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 3, &biClear);
				pNGCMip[3]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_5_NGC_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 4, &biClear);
				pNGCMip[4]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_1_XBOX_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 0, &biClear);
				pXBOXMip[0]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_2_XBOX_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 1, &biClear);
				pXBOXMip[1]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_3_XBOX_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 2, &biClear);
				pXBOXMip[2]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_4_XBOX_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 3, &biClear);
				pXBOXMip[3]->SetText("None");
			}
			break;

		case IDC_BMTEX_MIP_5_XBOX_CLEAR:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 4, &biClear);
				pXBOXMip[4]->SetText("None");
			}
			break;

		// Handle Map Buttons
		case IDC_BMTEX_NAME_PS2:
			{
				pData->pTex->SetBaseMap(vPLAT_PS2, &bi);
				pPS2BaseMap->SetText((char*)bi.Name());
				UpdateMtls();
			}
			break;

		case IDC_BMTEX_NAME_NGC:
			{
				pData->pTex->SetBaseMap(vPLAT_NGC, &bi);
				pNGCBaseMap->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_NAME_XBOX:
			{
				pData->pTex->SetBaseMap(vPLAT_XBOX, &bi);
				pXBOXBaseMap->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_1_PS2:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 0, &bi);
				pPS2Mip[0]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_2_PS2:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 1, &bi);
				pPS2Mip[1]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_3_PS2:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 2, &bi);
				pPS2Mip[2]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_4_PS2:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 3, &bi);
				pPS2Mip[3]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_5_PS2:
			{
				pData->pTex->SetManualMap(vPLAT_PS2, 4, &bi);
				pPS2Mip[4]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_1_NGC:
			{
				pData->pTex->SetManualMap(vPLAT_NGC, 0, &bi);
				pNGCMip[0]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_2_NGC:
			{
				pData->pTex->SetManualMap(vPLAT_NGC, 1, &bi);
				pNGCMip[1]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_3_NGC:
			{
				pData->pTex->SetManualMap(vPLAT_NGC, 2, &bi);
				pNGCMip[2]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_4_NGC:
			{
				pData->pTex->SetManualMap(vPLAT_NGC, 3, &bi);
				pNGCMip[3]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_5_NGC:
			{
				pData->pTex->SetManualMap(vPLAT_NGC, 4, &bi);
				pNGCMip[4]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_1_XBOX:
			{
				pData->pTex->SetManualMap(vPLAT_XBOX, 0, &bi);
				pXBOXMip[0]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_2_XBOX:
			{
				pData->pTex->SetManualMap(vPLAT_XBOX, 1, &bi);
				pXBOXMip[1]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_3_XBOX:
			{
				pData->pTex->SetManualMap(vPLAT_XBOX, 2, &bi);
				pXBOXMip[2]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_4_XBOX:
			{
				pData->pTex->SetManualMap(vPLAT_XBOX, 3, &bi);
				pXBOXMip[3]->SetText((char*)bi.Name());
			}
			break;

		case IDC_BMTEX_MIP_NAME_5_XBOX:
			{
				pData->pTex->SetManualMap(vPLAT_XBOX, 4, &bi);
				pXBOXMip[4]->SetText((char*)bi.Name());
			}
			break;

		// Handle Checkboxes
		case IDC_COMPRESS_NGC:
			if (IsDlgButtonChecked(hwndUI, IDC_COMPRESS_NGC) == BST_CHECKED)
				pData->pTex->SetShouldCompress(vPLAT_NGC, true);
			else
				pData->pTex->SetShouldCompress(vPLAT_NGC, false);
			break;

		case IDC_COMPRESS_XBOX:
			if (IsDlgButtonChecked(hwndUI, IDC_COMPRESS_XBOX) == BST_CHECKED)
				pData->pTex->SetShouldCompress(vPLAT_XBOX, true);
			else
				pData->pTex->SetShouldCompress(vPLAT_XBOX, false);
			break;

		case IDC_24_BIT:
			if (IsDlgButtonChecked(hwndUI, IDC_24_BIT) == BST_CHECKED)
				pData->pTex->SetForce24BitPalette(true);
			else
				pData->pTex->SetForce24BitPalette(false);
			break;

		case IDC_CHANGE_TRANS:
			if (IsDlgButtonChecked(hwndUI, IDC_CHANGE_TRANS) == BST_CHECKED)
				pData->pTex->SetChangeTransColor(true);
			else
				pData->pTex->SetChangeTransColor(false);
			break;

		case IDC_USE_PS2_MIPMAPS_NGC:
			if (IsDlgButtonChecked(hwndUI, IDC_USE_PS2_MIPMAPS_NGC) == BST_CHECKED)
				pData->pTex->SetUsePS2Mips(vPLAT_NGC, true);
			else
				pData->pTex->SetUsePS2Mips(vPLAT_NGC, false);
			break;

		case IDC_USE_PS2_MIPMAPS_XBOX:
			if (IsDlgButtonChecked(hwndUI, IDC_USE_PS2_MIPMAPS_XBOX) == BST_CHECKED)
				pData->pTex->SetUsePS2Mips(vPLAT_XBOX, true);
			else
				pData->pTex->SetUsePS2Mips(vPLAT_XBOX, false);
			break;

			///
		case IDC_MIP_TYPE_NONE_PS2:
			pData->pTex->SetMipType(vPLAT_PS2, vMIP_TYPE_NONE);
			break;

		case IDC_MIP_TYPE_AUTO_PS2:
			pData->pTex->SetMipType(vPLAT_PS2, vMIP_TYPE_AUTO);
			break;

		case IDC_MIP_TYPE_MANUAL_PS2:
			pData->pTex->SetMipType(vPLAT_PS2, vMIP_TYPE_MANUAL);
			break;

		case IDC_MIP_TYPE_NONE_NGC:
			pData->pTex->SetMipType(vPLAT_NGC, vMIP_TYPE_NONE);
			break;

		case IDC_MIP_TYPE_AUTO_NGC:
			pData->pTex->SetMipType(vPLAT_NGC, vMIP_TYPE_AUTO);
			break;

		case IDC_MIP_TYPE_MANUAL_NGC:
			pData->pTex->SetMipType(vPLAT_NGC, vMIP_TYPE_MANUAL);
			break;

		case IDC_MIP_TYPE_NONE_XBOX:
			pData->pTex->SetMipType(vPLAT_XBOX, vMIP_TYPE_NONE);
			break;

		case IDC_MIP_TYPE_AUTO_XBOX:
			pData->pTex->SetMipType(vPLAT_XBOX, vMIP_TYPE_AUTO);
			break;

		case IDC_MIP_TYPE_MANUAL_XBOX:
			pData->pTex->SetMipType(vPLAT_XBOX, vMIP_TYPE_MANUAL);
			break;
			///

		case IDC_PAL_PS2_MIP1:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_PS2_MIP1))
				pData->pTex->SetUseBasePal(vPLAT_PS2, 0, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_PS2, 0, FALSE);
			break;

		case IDC_PAL_PS2_MIP2:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_PS2_MIP2))
				pData->pTex->SetUseBasePal(vPLAT_PS2, 1, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_PS2, 1, FALSE);
			break;

		case IDC_PAL_PS2_MIP3:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_PS2_MIP3))
				pData->pTex->SetUseBasePal(vPLAT_PS2, 2, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_PS2, 2, FALSE);			
			break;

		case IDC_PAL_PS2_MIP4:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_PS2_MIP4))
				pData->pTex->SetUseBasePal(vPLAT_PS2, 3, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_PS2, 3, FALSE);
			break;

		case IDC_PAL_PS2_MIP5:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_PS2_MIP5))
				pData->pTex->SetUseBasePal(vPLAT_PS2, 4, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_PS2, 4, FALSE);
			break;

		case IDC_PAL_NGC_MIP1:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_NGC_MIP1))
				pData->pTex->SetUseBasePal(vPLAT_NGC, 0, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_NGC, 0, FALSE);
			break;

		case IDC_PAL_NGC_MIP2:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_NGC_MIP2))
				pData->pTex->SetUseBasePal(vPLAT_NGC, 1, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_NGC, 1, FALSE);
			break;

		case IDC_PAL_NGC_MIP3:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_NGC_MIP3))
				pData->pTex->SetUseBasePal(vPLAT_NGC, 2, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_NGC, 2, FALSE);

			break;

		case IDC_PAL_NGC_MIP4:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_NGC_MIP4))
				pData->pTex->SetUseBasePal(vPLAT_NGC, 3, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_NGC, 3, FALSE);
			break;

		case IDC_PAL_NGC_MIP5:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_NGC_MIP5))
				pData->pTex->SetUseBasePal(vPLAT_NGC, 4, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_NGC, 4, FALSE);
			break;

		case IDC_PAL_XBX_MIP1:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_XBX_MIP1))
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 0, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 0, FALSE);
			break;

		case IDC_PAL_XBX_MIP2:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_XBX_MIP2))
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 1, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 1, FALSE);
			break;

		case IDC_PAL_XBX_MIP3:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_XBX_MIP3))
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 2, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 2, FALSE);
			break;

		case IDC_PAL_XBX_MIP4:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_XBX_MIP4))
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 3, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 3, FALSE);
			break;

		case IDC_PAL_XBX_MIP5:
			if (IsDlgButtonChecked(hwndUI, IDC_PAL_XBX_MIP5))
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 4, TRUE);
			else
				pData->pTex->SetUseBasePal(vPLAT_XBOX, 4, FALSE);
			break;

		///

		case IDC_BM_CLIP:
			if (IsDlgButtonChecked(hwndUI, IDC_BM_CLIP))
				pData->pTex->SetCropPlaceApply(true);
			else
				pData->pTex->SetCropPlaceApply(false);
			break;

		case IDC_BM_CROP:
			pData->pTex->SetCropPlaceMode(vCROP);
			break;

		case IDC_BM_PLACE:
			pData->pTex->SetCropPlaceMode(vPLACE);
			break;

		///

		// Handle Spinners
		case IDC_MIPMAP_K_SPINNER:
			pData->pTex->SetMipMapK(pKBiasSpin->GetFVal());
			break;

		case IDC_CLIP_XSPIN:	// U:
			pData->pTex->SetCropPlaceU(pCropPlaceUSpin->GetFVal());
			break;

		case IDC_CLIP_WSPIN:	// W:
			pData->pTex->SetCropPlaceW(pCropPlaceWSpin->GetFVal());
			break;

		case IDC_CLIP_YSPIN:	// V:
			pData->pTex->SetCropPlaceV(pCropPlaceVSpin->GetFVal());
			break;

		case IDC_CLIPH_SPIN:	// H:
			pData->pTex->SetCropPlaceH(pCropPlaceHSpin->GetFVal());
			break;

		// Handle color swatches
		case IDC_TRANS_COLOR:
			pData->pTex->SetTransColor(pTransColor->GetColor());
			break;

		// Handle combo boxes
		case IDC_MIN_FILTERING_MODE:
			{
				int minFilter = SendDlgItemMessage(hwnd, IDC_MIN_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				pData->pTex->SetMinFilterMode(minFilter);
			}
			break;

		case IDC_MAG_FILTERING_MODE:
			{
				int magFilter = SendDlgItemMessage(hwnd, IDC_MAG_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				pData->pTex->SetMagFilterMode(magFilter);
			}
			break;

		case IDC_MIN_MIP_FILTERING_MODE:
			{
				int mipFilter = SendDlgItemMessage(hwnd, IDC_MIN_MIP_FILTERING_MODE, CB_GETCURSEL, 0, 0);
				pData->pTex->SetMipFilterMode(mipFilter);
			}
			break;
		}
	}

	bLockTexPropUpdates = false;
}

BOOL MtlEditorDlg::MtlPropsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;

	switch(msg)
	{
	case WM_INITDIALOG:
		{
			pthis = (MtlEditorDlg*)lParam;
			pthis->HookAllWindows(hwnd);

			// Initialize all the UI controls
			pthis->pDrawOrder         = GetICustEdit(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID));
			pthis->pGrassHeight       = GetICustEdit(GetDlgItem(hwnd, IDC_EDITHEIGHT));
			pthis->pGrassLayers       = GetICustEdit(GetDlgItem(hwnd, IDC_EDITLAYERS));
			pthis->pBasePass          = GetICustEdit(GetDlgItem(hwnd, IDC_BASE_PASS));
			pthis->pAlphaCutoff       = GetICustEdit(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF));
			pthis->pSpecularPower     = GetICustEdit(GetDlgItem(hwnd, IDC_SPECPOW));

			pthis->pDrawOrderSpin     = GetISpinner(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID_SPINNER));
			pthis->pGrassHeightSpin   = GetISpinner(GetDlgItem(hwnd, IDC_SPINHEIGHT));
			pthis->pGrassLayersSpin   = GetISpinner(GetDlgItem(hwnd, IDC_SPINLAYERS));
			pthis->pBasePassSpin      = GetISpinner(GetDlgItem(hwnd, IDC_BASE_PASS_SPINNER));
			pthis->pAlphaCutoffSpin   = GetISpinner(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF_SPINNER));
			pthis->pSpecularPowerSpin = GetISpinner(GetDlgItem(hwnd, IDC_SPECPOWSPIN));

			pthis->pSpecularColor     = GetIColorSwatch(GetDlgItem(hwnd, IDC_SPECCOLOR));

			// Link spinners to edit controls
			pthis->pDrawOrderSpin->LinkToEdit(GetDlgItem(hwnd, IDC_DRAW_ORDER_ID), EDITTYPE_FLOAT);
			pthis->pGrassHeightSpin->LinkToEdit(GetDlgItem(hwnd, IDC_EDITHEIGHT), EDITTYPE_FLOAT);
			pthis->pGrassLayersSpin->LinkToEdit(GetDlgItem(hwnd, IDC_EDITLAYERS), EDITTYPE_INT);
			pthis->pBasePassSpin->LinkToEdit(GetDlgItem(hwnd, IDC_BASE_PASS), EDITTYPE_INT);
			pthis->pAlphaCutoffSpin->LinkToEdit(GetDlgItem(hwnd, IDC_ALPHA_CUTOFF), EDITTYPE_INT);
			pthis->pSpecularPowerSpin->LinkToEdit(GetDlgItem(hwnd, IDC_SPECPOW), EDITTYPE_FLOAT);

			// Set limits
			pthis->pDrawOrderSpin->SetLimits(0.0f, 1000.0f);
			pthis->pGrassHeightSpin->SetLimits(0.0f, 1000.0f);
			pthis->pGrassLayersSpin->SetLimits(0, 255);
			pthis->pBasePassSpin->SetLimits(1, 12);
			pthis->pAlphaCutoffSpin->SetLimits(0, 255);
			pthis->pSpecularPowerSpin->SetLimits(0.0f, 1000.0f);

			// Set Defaults
			pthis->pDrawOrderSpin->SetValue(500.0f, TRUE);
			pthis->pGrassHeightSpin->SetValue(1.0f, TRUE);
			pthis->pGrassLayersSpin->SetValue(1, TRUE);
			pthis->pBasePassSpin->SetValue(1, TRUE);
			pthis->pAlphaCutoffSpin->SetValue(1, TRUE);
			pthis->pSpecularPowerSpin->SetValue(16.0f, TRUE);

			// Update Enable/Disable states
			UpdateMtlPropStates(hwnd);
			MtlPropsUpdate(hwnd, pthis);
		}
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_ONE_SIDED:
		case IDC_INVISIBLE:
		case IDC_FRONT_FACING:
		case IDC_TRANSPARENT:
		case IDC_DECAL:
		case IDC_2SIDED:
		case IDC_WATER:
		case IDC_SORTED:
		case IDC_GRASSIFY:
		case IDC_USESPECULAR:
		case IDC_USEDEFAC:
		case IDC_RENDERMULTI:
		case IDC_CUTOFFFUNC:
			if (!bLockMtlUpdates)
			{
				if (pthis)
					pthis->ApplyMtlProps(hwnd, LOWORD(wParam));
			}

			UpdateMtlPropStates(hwnd);
		}
		break;

	case CC_SPINNER_CHANGE:
		if (!bLockMtlUpdates)
		{
			if (pthis)
				pthis->ApplyMtlProps(hwnd, LOWORD(wParam));
		}
		break;

	case CC_COLOR_CHANGE:
		if (!bLockMtlUpdates)
		{
			if (pthis)
				pthis->ApplyMtlProps(hwnd, LOWORD(wParam));
		}
		break;

	case WM_CLOSE:
		{
			ReleaseICustEdit(pthis->pDrawOrder);
			ReleaseICustEdit(pthis->pGrassHeight);
			ReleaseICustEdit(pthis->pGrassLayers);
			ReleaseICustEdit(pthis->pBasePass);
			ReleaseICustEdit(pthis->pAlphaCutoff);
			ReleaseICustEdit(pthis->pSpecularPower);

			pthis->pDrawOrder     = NULL;
			pthis->pGrassHeight   = NULL;
			pthis->pGrassLayers   = NULL;
			pthis->pBasePass      = NULL;
			pthis->pAlphaCutoff   = NULL;
			pthis->pSpecularPower = NULL;

			ReleaseISpinner(pthis->pDrawOrderSpin);
			ReleaseISpinner(pthis->pGrassHeightSpin);
			ReleaseISpinner(pthis->pGrassLayersSpin);
			ReleaseISpinner(pthis->pBasePassSpin);
			ReleaseISpinner(pthis->pAlphaCutoffSpin);
			ReleaseISpinner(pthis->pSpecularPowerSpin);
		
			pthis->pDrawOrderSpin     = NULL;
			pthis->pGrassHeightSpin   = NULL;
			pthis->pGrassLayersSpin   = NULL;
			pthis->pBasePassSpin      = NULL;
			pthis->pAlphaCutoffSpin   = NULL;
			pthis->pSpecularPowerSpin = NULL;

			ReleaseIColorSwatch(pthis->pSpecularColor);

			pthis->pSpecularColor = NULL;

			pthis = NULL;
		}
		return TRUE;
	}

	return FALSE;
}

BOOL MtlEditorDlg::MtlFlagsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;
	
	switch(msg)
	{
	case WM_INITDIALOG:
		{
			pthis = (MtlEditorDlg*)lParam;
			pthis->HookAllWindows(hwnd);

			// Fill in the terrain types
			for(int i = 0; i < vNUM_TERRAIN_TYPES; i++)
				SendDlgItemMessage(hwnd, IDC_TERRAIN, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)terrain_types[i]);

			MtlFlagsUpdate(hwnd, pthis);
		}
		return TRUE;

	case WM_COMMAND:
		{
			switch(HIWORD(wParam))
			{
			case CBN_SELCHANGE:
				if (!bLockMtlFlagUpdates)
				{
					if (pthis)
						pthis->ApplyMtlFlagProps(hwnd, LOWORD(wParam));
				}
			}
		}
		break;

	case WM_CLOSE:
		{
			pthis = NULL;
		}
		return TRUE;
	}

	return FALSE;
}

BOOL MtlEditorDlg::MtlPassDlgProc(int pass, HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		{
			bLockMtlPassUpdates[pass] = true;
			HookAllWindows(hwnd);

			pHasColorColor[pass]    = GetIColorSwatch(GetDlgItem(hwnd, IDC_COLOR));
			pAmbient[pass]          = GetIColorSwatch(GetDlgItem(hwnd, IDC_COLOR_A));
			pDiffuse[pass]          = GetIColorSwatch(GetDlgItem(hwnd, IDC_COLOR_D));
			pSpecular[pass]         = GetIColorSwatch(GetDlgItem(hwnd, IDC_COLOR_S));

			pTileU[pass]            = GetICustEdit(GetDlgItem(hwnd, IDC_ENV_TILE_U));
			pTileV[pass]            = GetICustEdit(GetDlgItem(hwnd, IDC_ENV_TILE_V));
			pFixedSourceAlpha[pass] = GetICustEdit(GetDlgItem(hwnd, IDC_FIXED));
			pUVelocity[pass]        = GetICustEdit(GetDlgItem(hwnd, IDC_U_VEL));
			pUFrequency[pass]       = GetICustEdit(GetDlgItem(hwnd, IDC_U_FREQUENCY));
			pUAmplitude[pass]       = GetICustEdit(GetDlgItem(hwnd, IDC_U_AMPLITUDE));
			pUPhase[pass]           = GetICustEdit(GetDlgItem(hwnd, IDC_U_PHASE));
			pVVelocity[pass]        = GetICustEdit(GetDlgItem(hwnd, IDC_V_VEL));
			pVFrequency[pass]       = GetICustEdit(GetDlgItem(hwnd, IDC_V_FREQUENCY));
			pVAmplitude[pass]       = GetICustEdit(GetDlgItem(hwnd, IDC_V_AMPLITUDE));
			pVPhase[pass]           = GetICustEdit(GetDlgItem(hwnd, IDC_V_PHASE));
			
			pTileUSpin[pass]            = GetISpinner(GetDlgItem(hwnd, IDC_ENV_TILE_U_SPINNER));
			pTileVSpin[pass]            = GetISpinner(GetDlgItem(hwnd, IDC_ENV_TILE_V_SPINNER));
			pFixedSourceAlphaSpin[pass] = GetISpinner(GetDlgItem(hwnd, IDC_FIXED_SPINNER));
			pUVelocitySpin[pass]        = GetISpinner(GetDlgItem(hwnd, IDC_U_VEL_SPINNER));
			pUFrequencySpin[pass]       = GetISpinner(GetDlgItem(hwnd, IDC_U_FREQUENCY_SPINNER));
			pUAmplitudeSpin[pass]       = GetISpinner(GetDlgItem(hwnd, IDC_U_AMPLITUDE_SPINNER));
			pUPhaseSpin[pass]           = GetISpinner(GetDlgItem(hwnd, IDC_U_PHASE_SPINNER));
			pVVelocitySpin[pass]        = GetISpinner(GetDlgItem(hwnd, IDC_V_VEL_SPINNER));
			pVFrequencySpin[pass]       = GetISpinner(GetDlgItem(hwnd, IDC_V_FREQUENCY_SPINNER));
			pVAmplitudeSpin[pass]       = GetISpinner(GetDlgItem(hwnd, IDC_V_AMPLITUDE_SPINNER));
			pVPhaseSpin[pass]           = GetISpinner(GetDlgItem(hwnd, IDC_V_PHASE_SPINNER));

			pMap[pass]                  = GetICustButton(GetDlgItem(hwnd, IDC_MAP));

			// Link spinners to edit controls
			pTileUSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_ENV_TILE_U), EDITTYPE_FLOAT);
			pTileVSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_ENV_TILE_V), EDITTYPE_FLOAT);
			pFixedSourceAlphaSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_FIXED), EDITTYPE_INT);
			pUVelocitySpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_U_VEL), EDITTYPE_FLOAT);
			pUFrequencySpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_U_FREQUENCY), EDITTYPE_FLOAT);
			pUAmplitudeSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_U_AMPLITUDE), EDITTYPE_FLOAT);
			pUPhaseSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_U_PHASE), EDITTYPE_FLOAT);
			pVVelocitySpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_V_VEL), EDITTYPE_FLOAT);
			pVFrequencySpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_V_FREQUENCY), EDITTYPE_FLOAT);
			pVAmplitudeSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_V_AMPLITUDE), EDITTYPE_FLOAT);
			pVPhaseSpin[pass]->LinkToEdit(GetDlgItem(hwnd, IDC_V_PHASE), EDITTYPE_FLOAT);

			// Set Limits
			pTileUSpin[pass]->SetLimits(1.0f, 8.0f);
			pTileVSpin[pass]->SetLimits(1.0f, 8.0f);
			pFixedSourceAlphaSpin[pass]->SetLimits(0, 255);
			pUVelocitySpin[pass]->SetLimits(-1024.0f, 1024.0f);
			pUFrequencySpin[pass]->SetLimits(0.0f, 100.0f);
			pUAmplitudeSpin[pass]->SetLimits(0.0f, 100.0f);
			pUPhaseSpin[pass]->SetLimits(0.0f, 100.0f);
			pVVelocitySpin[pass]->SetLimits(-1024.0f, 1024.0f);
			pVFrequencySpin[pass]->SetLimits(0.0f, 100.0f);
			pVAmplitudeSpin[pass]->SetLimits(0.0f, 100.0f);
			pVPhaseSpin[pass]->SetLimits(0.0f, 100.0f);

			// Set Defaults
			pTileUSpin[pass]->SetValue(3.0f, TRUE);
			pTileVSpin[pass]->SetValue(3.0f, TRUE);
			pFixedSourceAlphaSpin[pass]->SetValue(0, TRUE);
			pUVelocitySpin[pass]->SetValue(0.0f, TRUE);
			pUFrequencySpin[pass]->SetValue(1.0f, TRUE);
			pUAmplitudeSpin[pass]->SetValue(1.0f, TRUE);
			pUPhaseSpin[pass]->SetValue(0.0f, TRUE);
			pVVelocitySpin[pass]->SetValue(0.0f, TRUE);
			pVFrequencySpin[pass]->SetValue(1.0f, TRUE);
			pVAmplitudeSpin[pass]->SetValue(1.0f, TRUE);
			pVPhaseSpin[pass]->SetValue(0.0f, TRUE);

			pHasColorColor[pass]->SetColor(RGB(128,128,128));
			pAmbient[pass]->SetColor(0);
			pDiffuse[pass]->SetColor(0);
			pSpecular[pass]->SetColor(0);

			if (pass == 0)
			{
				CheckDlgButton(hwnd, IDC_ENABLED, BST_CHECKED);
				CheckDlgButton(hwnd, IDC_ABSENT_IN_NET_GAMES, BST_UNCHECKED);
			}

			// Populate Mapping Modes
			SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)"Explicit");
			SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)"EnvironmentMap");
			SendDlgItemMessage(hwnd, IDC_MAPPING_MODE, CB_SETCURSEL, MAPMODE_EXPLICIT, 0);

			// Populate Blend Modes
			for(int i = 0; i < vNUM_BLEND_MODES; i++)
				SendDlgItemMessage(hwnd, IDC_BLEND_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)blend_modes[i]);

			SendDlgItemMessage(hwnd, IDC_BLEND_MODE, CB_SETCURSEL, vBLEND_MODE_DIFFUSE, 0);

			bLockMtlPassUpdates[pass] = false;
			MtlPassUpdate(hwnd, this, pass);
			UpdateMtlPassStates(hwnd, pass);
		}
		return TRUE;

	case WM_CLOSE:
		{
			ReleaseICustEdit(pTileU[pass]);
			ReleaseICustEdit(pTileV[pass]);
			ReleaseICustEdit(pFixedSourceAlpha[pass]);
			ReleaseICustEdit(pUVelocity[pass]);
			ReleaseICustEdit(pUFrequency[pass]);
			ReleaseICustEdit(pUAmplitude[pass]);
			ReleaseICustEdit(pUPhase[pass]);
			ReleaseICustEdit(pVVelocity[pass]);
			ReleaseICustEdit(pVFrequency[pass]);
			ReleaseICustEdit(pVAmplitude[pass]);
			ReleaseICustEdit(pVPhase[pass]);

			pTileU[pass]            = NULL;
			pTileV[pass]            = NULL;
			pFixedSourceAlpha[pass] = NULL;
			pUVelocity[pass]        = NULL;
			pUFrequency[pass]       = NULL;
			pUAmplitude[pass]       = NULL;
			pUPhase[pass]           = NULL;
			pVVelocity[pass]        = NULL;
			pVFrequency[pass]       = NULL;
			pVAmplitude[pass]       = NULL;
			pVPhase[pass]           = NULL;

			ReleaseISpinner(pTileUSpin[pass]);
			ReleaseISpinner(pTileVSpin[pass]);
			ReleaseISpinner(pFixedSourceAlphaSpin[pass]);
			ReleaseISpinner(pUVelocitySpin[pass]);
			ReleaseISpinner(pUFrequencySpin[pass]);
			ReleaseISpinner(pUAmplitudeSpin[pass]);
			ReleaseISpinner(pUPhaseSpin[pass]);
			ReleaseISpinner(pVVelocitySpin[pass]);
			ReleaseISpinner(pVFrequencySpin[pass]);
			ReleaseISpinner(pVAmplitudeSpin[pass]);
			ReleaseISpinner(pVPhaseSpin[pass]);

			pTileUSpin[pass]            = NULL;
			pTileVSpin[pass]            = NULL;
			pFixedSourceAlphaSpin[pass] = NULL;
			pUVelocity[pass]            = NULL;
			pUFrequency[pass]           = NULL;
			pVVelocity[pass]            = NULL;
			pVFrequency[pass]           = NULL;
			pVAmplitude[pass]           = NULL;
			pVPhase[pass]               = NULL;

			ReleaseIColorSwatch(pHasColorColor[pass]);
			ReleaseIColorSwatch(pAmbient[pass]);
			ReleaseIColorSwatch(pDiffuse[pass]);
			ReleaseIColorSwatch(pSpecular[pass]);

			pHasColorColor[pass] = NULL;
			pAmbient[pass]       = NULL;
			pDiffuse[pass]       = NULL;
			pSpecular[pass]      = NULL;

			bLockMtlPassUpdates[pass] = true;
		}
		return TRUE;

	case WM_COMMAND:
	case CC_SPINNER_CHANGE:
	case CC_COLOR_CHANGE:
		UpdateMtlPassStates(hwnd, pass);

		if (!bLockMtlPassUpdates[pass])
			ApplyMtlPassProps(hwnd, LOWORD(wParam), pass);

		UpdateMtlPassStates(hwnd, pass);
	}

	return FALSE;
}

BOOL MtlEditorDlg::MtlPass1DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;

	if (msg == WM_INITDIALOG)
		pthis = (MtlEditorDlg*)lParam;

	return pthis->MtlPassDlgProc(0, hwnd, msg, wParam, lParam);
}

BOOL MtlEditorDlg::MtlPass2DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;

	if (msg == WM_INITDIALOG)
		pthis = (MtlEditorDlg*)lParam;

	return pthis->MtlPassDlgProc(1, hwnd, msg, wParam, lParam);
}

BOOL MtlEditorDlg::MtlPass3DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;

	if (msg == WM_INITDIALOG)
		pthis = (MtlEditorDlg*)lParam;

	return pthis->MtlPassDlgProc(2, hwnd, msg, wParam, lParam);
}

BOOL MtlEditorDlg::MtlPass4DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg     *pthis;

	if (msg == WM_INITDIALOG)
		pthis = (MtlEditorDlg*)lParam;

	return pthis->MtlPassDlgProc(3, hwnd, msg, wParam, lParam);
}

BOOL MtlEditorDlg::TexPropsDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	static MtlEditorDlg* pthis = NULL;

	switch(msg)
	{
	case WM_INITDIALOG:
		pthis = (MtlEditorDlg*)lParam;
		pthis->HookAllWindows(hwnd);

		pthis->pPS2BaseMap  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_NAME_PS2));
		pthis->pNGCBaseMap  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_NAME_NGC));
		pthis->pXBOXBaseMap = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_NAME_XBOX));

		pthis->pPS2Mip[0]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_PS2));
		pthis->pPS2Mip[1]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_PS2));
		pthis->pPS2Mip[2]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_PS2));
		pthis->pPS2Mip[3]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_PS2));
		pthis->pPS2Mip[4]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_PS2));

		pthis->pNGCMip[0]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_NGC));
		pthis->pNGCMip[1]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_NGC));
		pthis->pNGCMip[2]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_NGC));
		pthis->pNGCMip[3]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_NGC));
		pthis->pNGCMip[4]   = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_NGC));

		pthis->pXBOXMip[0]  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_1_XBOX));
		pthis->pXBOXMip[1]  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_2_XBOX));
		pthis->pXBOXMip[2]  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_3_XBOX));
		pthis->pXBOXMip[3]  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_4_XBOX));
		pthis->pXBOXMip[4]  = GetICustButton(GetDlgItem(hwnd, IDC_BMTEX_MIP_NAME_5_XBOX));

		pthis->pKBiasEdit      = GetICustEdit(GetDlgItem(hwnd, IDC_MIPMAP_K));
		pthis->pCropPlaceUEdit = GetICustEdit(GetDlgItem(hwnd, IDC_CLIP_X));
		pthis->pCropPlaceWEdit = GetICustEdit(GetDlgItem(hwnd, IDC_CLIP_W));
		pthis->pCropPlaceVEdit = GetICustEdit(GetDlgItem(hwnd, IDC_CLIP_Y));
		pthis->pCropPlaceHEdit = GetICustEdit(GetDlgItem(hwnd, IDC_CLIP_H));

		pthis->pKBiasSpin      = GetISpinner(GetDlgItem(hwnd, IDC_MIPMAP_K_SPINNER));
		pthis->pCropPlaceUSpin = GetISpinner(GetDlgItem(hwnd, IDC_CLIP_XSPIN));
		pthis->pCropPlaceWSpin = GetISpinner(GetDlgItem(hwnd, IDC_CLIP_WSPIN));
		pthis->pCropPlaceVSpin = GetISpinner(GetDlgItem(hwnd, IDC_CLIP_YSPIN));
		pthis->pCropPlaceHSpin = GetISpinner(GetDlgItem(hwnd, IDC_CLIPH_SPIN));

		pthis->pTransColor = GetIColorSwatch(GetDlgItem(hwnd, IDC_TRANS_COLOR));

		pthis->pKBiasSpin->LinkToEdit(GetDlgItem(hwnd, IDC_MIPMAP_K), EDITTYPE_FLOAT);
		pthis->pCropPlaceUSpin->LinkToEdit(GetDlgItem(hwnd, IDC_CLIP_X), EDITTYPE_FLOAT);
		pthis->pCropPlaceWSpin->LinkToEdit(GetDlgItem(hwnd, IDC_CLIP_W), EDITTYPE_FLOAT);
		pthis->pCropPlaceVSpin->LinkToEdit(GetDlgItem(hwnd, IDC_CLIP_Y), EDITTYPE_FLOAT);
		pthis->pCropPlaceHSpin->LinkToEdit(GetDlgItem(hwnd, IDC_CLIP_H), EDITTYPE_FLOAT);

		// Set Limits
		pthis->pKBiasSpin->SetLimits(-100.0f, 100.0f);
		pthis->pCropPlaceUSpin->SetLimits(0.0f, 1.0f);
		pthis->pCropPlaceWSpin->SetLimits(0.0f, 1.0f);
		pthis->pCropPlaceVSpin->SetLimits(0.0f, 1.0f);
		pthis->pCropPlaceHSpin->SetLimits(0.0f, 1.0f);

		// Populate Combo lists
		int i;

		for(i = 0; i < 2; i++)
			SendDlgItemMessage(hwnd, IDC_MIN_FILTERING_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)filtering_modes[i]);

		SendDlgItemMessage(hwnd, IDC_MIN_FILTERING_MODE, CB_SETCURSEL, (WPARAM)1, 0);

		for(i = 0; i < 2; i++)
			SendDlgItemMessage(hwnd, IDC_MAG_FILTERING_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)filtering_modes[i]);

		SendDlgItemMessage(hwnd, IDC_MAG_FILTERING_MODE, CB_SETCURSEL, (WPARAM)1, 0);

		for(i = 0; i < 2; i++)
			SendDlgItemMessage(hwnd, IDC_MIN_MIP_FILTERING_MODE, CB_INSERTSTRING, (WPARAM)-1, (LPARAM)filtering_modes[i]);

		SendDlgItemMessage(hwnd, IDC_MIN_MIP_FILTERING_MODE, CB_SETCURSEL, 0, 0);

		// Set Default Button Captions
		pthis->pPS2BaseMap->SetText("None");
		pthis->pNGCBaseMap->SetText("Use PS2");
		pthis->pXBOXBaseMap->SetText("Use PS2");

		pthis->pPS2Mip[0]->SetText("None");
		pthis->pPS2Mip[1]->SetText("None");
		pthis->pPS2Mip[2]->SetText("None");
		pthis->pPS2Mip[3]->SetText("None");
		pthis->pPS2Mip[4]->SetText("None");

		pthis->pNGCMip[0]->SetText("None");
		pthis->pNGCMip[1]->SetText("None");
		pthis->pNGCMip[2]->SetText("None");
		pthis->pNGCMip[3]->SetText("None");
		pthis->pNGCMip[4]->SetText("None");

		pthis->pXBOXMip[0]->SetText("None");
		pthis->pXBOXMip[1]->SetText("None");
		pthis->pXBOXMip[2]->SetText("None");
		pthis->pXBOXMip[3]->SetText("None");
		pthis->pXBOXMip[4]->SetText("None");

		CheckDlgButton(hwnd, IDC_COMPRESS_NGC, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_COMPRESS_XBOX, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_MIP_TYPE_NONE_PS2, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_NGC, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_MIP_TYPE_AUTO_XBOX, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_NGC, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_USE_PS2_MIPMAPS_XBOX, BST_CHECKED);
		CheckDlgButton(hwnd, IDC_BM_CROP, BST_CHECKED);

		pthis->pKBiasSpin->SetValue(-8.0f, TRUE);
		pthis->pCropPlaceUSpin->SetValue(0.0f, TRUE);
		pthis->pCropPlaceWSpin->SetValue(1.0f, TRUE);
		pthis->pCropPlaceVSpin->SetValue(0.0f, TRUE);
		pthis->pCropPlaceHSpin->SetValue(1.0f, TRUE);

		TexPassUpdate(hwnd, pthis);
		pthis->UpdateTexPropStates(hwnd);
		return TRUE;

	case WM_CLOSE:
		ReleaseICustButton(pthis->pPS2BaseMap);
		ReleaseICustButton(pthis->pNGCBaseMap);
		ReleaseICustButton(pthis->pXBOXBaseMap);

		pthis->pPS2BaseMap  = NULL;
		pthis->pNGCBaseMap  = NULL;
		pthis->pXBOXBaseMap = NULL;

		ReleaseICustButton(pthis->pPS2Mip[0]);
		ReleaseICustButton(pthis->pPS2Mip[1]);
		ReleaseICustButton(pthis->pPS2Mip[2]);
		ReleaseICustButton(pthis->pPS2Mip[3]);
		ReleaseICustButton(pthis->pPS2Mip[4]);

		pthis->pPS2Mip[0] = NULL;
		pthis->pPS2Mip[1] = NULL;
		pthis->pPS2Mip[2] = NULL;
		pthis->pPS2Mip[3] = NULL;
		pthis->pPS2Mip[4] = NULL;

		ReleaseICustButton(pthis->pNGCMip[0]);
		ReleaseICustButton(pthis->pNGCMip[1]);
		ReleaseICustButton(pthis->pNGCMip[2]);
		ReleaseICustButton(pthis->pNGCMip[3]);
		ReleaseICustButton(pthis->pNGCMip[4]);

		pthis->pNGCMip[0] = NULL;
		pthis->pNGCMip[1] = NULL;
		pthis->pNGCMip[2] = NULL;
		pthis->pNGCMip[3] = NULL;
		pthis->pNGCMip[4] = NULL;

		ReleaseICustButton(pthis->pXBOXMip[0]);
		ReleaseICustButton(pthis->pXBOXMip[1]);
		ReleaseICustButton(pthis->pXBOXMip[2]);
		ReleaseICustButton(pthis->pXBOXMip[3]);
		ReleaseICustButton(pthis->pXBOXMip[4]);

		pthis->pXBOXMip[0] = NULL;
		pthis->pXBOXMip[1] = NULL;
		pthis->pXBOXMip[2] = NULL;
		pthis->pXBOXMip[3] = NULL;
		pthis->pXBOXMip[4] = NULL;

		ReleaseICustEdit(pthis->pKBiasEdit);
		ReleaseICustEdit(pthis->pCropPlaceUEdit);
		ReleaseICustEdit(pthis->pCropPlaceWEdit);
		ReleaseICustEdit(pthis->pCropPlaceVEdit);
		ReleaseICustEdit(pthis->pCropPlaceHEdit);

		pthis->pKBiasEdit      = NULL;
		pthis->pCropPlaceUEdit = NULL;
		pthis->pCropPlaceWEdit = NULL;
		pthis->pCropPlaceVEdit = NULL;
		pthis->pCropPlaceHEdit = NULL;

		ReleaseIColorSwatch(pthis->pTransColor);

		pthis->pTransColor = NULL;

		ReleaseISpinner(pthis->pKBiasSpin);
		ReleaseISpinner(pthis->pCropPlaceUSpin);
		ReleaseISpinner(pthis->pCropPlaceWSpin);
		ReleaseISpinner(pthis->pCropPlaceVSpin);
		ReleaseISpinner(pthis->pCropPlaceHSpin);

		pthis->pKBiasSpin      = NULL;
		pthis->pCropPlaceUSpin = NULL;
		pthis->pCropPlaceWSpin = NULL;
		pthis->pCropPlaceVSpin = NULL;
		pthis->pCropPlaceHSpin = NULL;

		pthis = NULL;
		return TRUE;

	case WM_COMMAND: 
	case CC_SPINNER_CHANGE:
	case CC_COLOR_CHANGE:
		if (pthis)
		{
			pthis->UpdateTexPropStates(hwnd);

			if (!bLockTexPropUpdates)
				pthis->ApplyTexProps(hwnd, LOWORD(wParam));

			pthis->UpdateTexPropStates(hwnd);
		}

		return TRUE;
	}

	return FALSE;
}

void MtlEditorDlg::TreeSelChange()
{
	if (selMtlList)
		delete [] selMtlList;

	SendDlgItemMessage(hwnd, IDC_CONFLICTLST, LB_RESETCONTENT, 0, 0);

	nSelMaterials = ptTree->GetSelCount();
	selMtlList = new void*[nSelMaterials];
	
	ptTree->GetSelItems(selMtlList, nSelMaterials);

	// Check the depth of the items to determine if the selection
	// set contains just NExt materials, just textures, or both
	MtlTreeItemType type = MTITYPE_UNDEFINED;
	
	pRollupWindow->Hide();

	for(int i = 0; i < nSelMaterials; i++)
	{
		MtlTreeItemData* pData = (MtlTreeItemData*)ptTree->GetItemData(selMtlList[i]);
		
		// No operations can currently be performed if a multi material is selected
		if (pData->type == MTITYPE_MULTI)
			return;
		
		if (pData->type == MTITYPE_MTL)
		{
			// No operations can currently be performed if both textures and materials are selected
			// user must select either or
			if (type == MTITYPE_TEX)
				return;

			type = MTITYPE_MTL;
		}

		if (pData->type == MTITYPE_TEX)
		{
			// No operations can currently be performed if both textures and materials are selected
			// user must select either or
			if (type == MTITYPE_MTL)
				return;

			type = MTITYPE_TEX;
		}
	}

	bLockMtlUpdates = true;

	switch(type)
	{
	case MTITYPE_MTL:
		BuildMaterialPanels();
		break;

	case MTITYPE_TEX:
		BuildTexturePanels();
		break;
	}

	ptTree->SetFocus();

	bLockMtlUpdates = false;
}

void MtlEditorDlg::BuildMaterialPanels()
{
	// Destroy all rollups in display
	pRollupWindow->DeleteRollup(0, pRollupWindow->GetNumPanels());

	// Construct material specific panels
	pRollupWindow->AppendRollup(hInstance, 
								MAKEINTRESOURCE(IDD_NEXT_MATERIAL_PROPS_PANEL), 
								MtlPropsDlgProc, 
								"Properties",
								(LPARAM)this);

	/* We're not going to support manipulating wibble data for now
	pRollupWindow->AppendRollup(hInstance, 
								MAKEINTRESOURCE(IDD_NEXT_VCWIBBLE_PANEL), 
								PropsDlgProc,
								"VC Wibble Sequences");
	*/

	pRollupWindow->AppendRollup(hInstance,
								MAKEINTRESOURCE(IDD_SKATEFLAGS_MAT_PANEL),
								MtlFlagsDlgProc,
								"Design_Properties",
								(LPARAM)this);

	pRollupWindow->AppendRollup(hInstance, 
								MAKEINTRESOURCE(IDD_NEXT_MATERIAL_PASS_PANEL),
								MtlPass1DlgProc,
								"First Pass",
								(LPARAM)this);

	pRollupWindow->AppendRollup(hInstance,
								MAKEINTRESOURCE(IDD_NEXT_MATERIAL_PASS_PANEL), 
								MtlPass2DlgProc, 
								"Second Pass",
								(LPARAM)this);

	pRollupWindow->AppendRollup(hInstance, 
								MAKEINTRESOURCE(IDD_NEXT_MATERIAL_PASS_PANEL),
								MtlPass3DlgProc,
								"Third Pass",
								(LPARAM)this);

	pRollupWindow->AppendRollup(hInstance, 
								MAKEINTRESOURCE(IDD_NEXT_MATERIAL_PASS_PANEL),
								MtlPass4DlgProc, 
								"Fourth Pass",
								(LPARAM)this);

	pRollupWindow->Show();
}

void MtlEditorDlg::BuildTexturePanels()
{
	// Destroy all panels
	pRollupWindow->DeleteRollup(0, pRollupWindow->GetNumPanels());

	// Construct texture specific panels
	pRollupWindow->AppendRollup(hInstance,
		                        MAKEINTRESOURCE(IDD_BMTEX),
								TexPropsDlgProc,
								"Texture Parameters",
								(LPARAM)this);

	pRollupWindow->Show();
}

BOOL MtlEditorDlg::DlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		{
		}
		return TRUE;
	}

	return FALSE;
}

void MtlEditorDlg::TreeSelChange(ThumbNailTree* pTree, void* pData)
{
	MtlEditorDlg* pthis = (MtlEditorDlg*)pData;
	pthis->TreeSelChange();
}

MtlEditorDlg::MtlEditorDlg(Interface* ip, HWND hwndParent) :
	MSDlgWindow(hInstance, MAKEINTRESOURCE(IDD_MTLBROWSER), hwndParent)
{
	pPS2BaseMap  = NULL;
	pNGCBaseMap  = NULL;
	pXBOXBaseMap = NULL;


	this->ip = ip;
	/*
	ptlist = new ThumbNailList(hInstance);
	ptlist->Attach(GetDlgItem(hwnd, IDC_MTLLIST));
	*/

	ptTree = new ThumbNailTree(hInstance);
	ptTree->Attach(GetDlgItem(hwnd, IDC_THUMBTREE));

	ptTree->RegisterSelChangeCB(TreeSelChange, this);

	pRollupWindow = GetIRollup(GetDlgItem(hwnd, IDC_ROLLUPCONTAINER));

	selMtlList    = NULL;
	nSelMaterials = 0;

	/*
	void* pEntry = ptTree->AddNode("Blah");

	ptTree->AddNode(pEntry, "SubBlahA");
	ptTree->AddNode(pEntry, "SubBlahB");
	ptTree->AddNode(pEntry, "SubBlahC");

	pEntry = ptTree->AddNode("Blah2");
	ptTree->AddNode(pEntry, "SubA");
	void* pEntry2 = ptTree->AddNode(pEntry, "SubB");
	ptTree->AddNode(pEntry, "SubC");

	ptTree->AddNode(pEntry2, "111");
	ptTree->AddNode(pEntry2, "222");
	*/
}

MtlEditorDlg::~MtlEditorDlg()
{
	/*
	if (ptlist)
		delete ptlist;

	if (ptTree)
		delete ptTree;
	*/

	ReleaseIRollup(pRollupWindow);
}

bool MtlEditorDlg::MtlUpdate(ThumbNailTree* pTree, void* pItem, void* pthis)
{
	MtlTreeItemData* pTreeItemData;
	pTreeItemData = (MtlTreeItemData*)pTree->GetItemData(pItem);

	if (pTreeItemData)
	{
		if (pTreeItemData->type == MTITYPE_TEX)
		{
			BitmapInfo* bmi = pTreeItemData->pTex->GetBaseMap(vPLAT_PS2);

			if (bmi)
			{
				if (_stricmp(bmi->Name(), pTree->GetThumbnailFile(pItem)) != 0)
				{
					// Filename has changed the thumbnail needs reloaded
					pTree->SetThumbnailFile(pItem, (char*)bmi->Name());

					int idx = pTree->GetChildIndex(pItem);
					char name[1024];
					sprintf(name, "Pass %i: %s", idx, bmi->Name());
					pTree->SetItemName(pItem, name);
				}
			}
			else
			{
				pTree->SetThumbnailFile(pItem, NULL);
			}
		}
		else if (pTreeItemData->type == MTITYPE_MTL)
		{
			// Since the material image is currently based on the first texture we should check
			// if the image of the first child texture has changed for this material and update appropriately
			void* pItemChild = pTree->GetChildItem(pItem);

			if (pItemChild)
			{
				MtlTreeItemData* pTreeChildData = (MtlTreeItemData*)pTree->GetItemData(pItemChild);

				BitmapInfo* bmi = pTreeChildData->pTex->GetBaseMap(vPLAT_PS2);

				if (bmi)
				{
					if (_stricmp(bmi->Name(), pTree->GetThumbnailFile(pItem)) != 0)
					{
						// Filename has changed the thumbnail needs reloaded
						pTree->SetThumbnailFile(pItem, (char*)bmi->Name());
					}
				}
				else
				{
					pTree->SetThumbnailFile(pItem, NULL);
				}
			}
		}
	}

	return true;
}

bool MtlEditorDlg::UpdateMtls()
{
	ptTree->TraverseTree(MtlUpdate, this);	

	return true;
}

void MtlEditorDlg::RefreshMtls()
{
	//ptTree->Clear();

	// Add all the materials in the scene to our material browser
	MtlBaseLib* mtlLib = ip->GetSceneMtls();

	for(int i = 0; i < mtlLib->Count(); i++)
	{
		MtlBaseHandle mbase = (*mtlLib)[i];

		if (mbase->ClassID() == NEXT_MATERIAL_CLASS_ID)
		{
			INExtMaterial* next_mat = dynamic_cast< INExtMaterial* > ( mbase );

			if (next_mat)
			{
				CStr name    = next_mat->GetName();
				int  nPasses = next_mat->GetNumPasses();

				Texmap* tmap = next_mat->GetTexmap(0);

				if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
				{
					INExtTexture* next_tex = dynamic_cast< INExtTexture* >(tmap);
					BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);
					void* pMtlNode;

					//ptlist->AddItem(next_mat->GetName(), (char*)bmi->Name());
					MtlTreeItemData* pData = new MtlTreeItemData;
					pData->type = MTITYPE_MTL;
					pData->pMtl = next_mat;
					pMtlNode = ptTree->AddNode(next_mat->GetName(), (char*)bmi->Name(), pData);

					for(int pass = 0; pass < 4; pass++)
					{
						Texmap* tmap = next_mat->GetTexmap(pass);

						if (tmap && tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
						{
							INExtTexture* next_tex = dynamic_cast< INExtTexture* >(tmap);
							BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);
							
							// Texture must exist for the material to be added to the list
							if (bmi)
							{								
								char buf[256];
								sprintf(buf, "Pass %i : %s", pass, (char*)bmi->Name());

								MtlTreeItemData* pData = new MtlTreeItemData;
								pData->type = MTITYPE_TEX;
								pData->pTex = next_tex;
								ptTree->AddNode(pMtlNode, buf, (char*)bmi->Name(), pData);
							}
						}
					}

				}
			}
		}

		// Also handle multi materials
		if (mbase->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
		{
			Multi* multimat = (Multi*)mbase;

			int nSubs = multimat->NumSubs();

			MtlTreeItemData* pData = new MtlTreeItemData;
			pData->type = MTITYPE_MULTI;
			void* pMainEntry = ptTree->AddNode(multimat->GetName(), pData);

			for(int i = 0; i < nSubs; i++)
			{
				Mtl* mtl = multimat->GetSubMtl(i);

				if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
				{
					INExtMaterial* next_mat = dynamic_cast< INExtMaterial* > ( mtl );

					if (next_mat)
					{
						void* pSubEntry;

						CStr name    = next_mat->GetName();
						int  nPasses = next_mat->GetNumPasses();

						Texmap* tmap = next_mat->GetTexmap(0);

						if (tmap && tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
						{
							INExtTexture* next_tex = dynamic_cast< INExtTexture* >(tmap);
							BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);
							
							// Texture must exist for the material to be added to the list
							if (bmi)
							{
								//ptlist->AddItem(CStr("SUB ") + next_mat->GetName(), (char*)bmi->Name());
								MtlTreeItemData* pData = new MtlTreeItemData;
								pData->type = MTITYPE_MTL;
								pData->pMtl = next_mat;
								pSubEntry = ptTree->AddNode(pMainEntry, next_mat->GetName(), (char*)bmi->Name(), pData);
							}
						}

						for(int pass = 0; pass < 4; pass++)
						{
							Texmap* tmap = next_mat->GetTexmap(pass);

							if (tmap && tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
							{
								INExtTexture* next_tex = dynamic_cast< INExtTexture* >(tmap);
								BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);
								
								// Texture must exist for the material to be added to the list
								if (bmi)
								{								
									char buf[256];
									sprintf(buf, "Pass %i : %s", pass, (char*)bmi->Name());

									MtlTreeItemData* pData = new MtlTreeItemData;
									pData->type = MTITYPE_TEX;
									pData->pTex = next_tex;
									ptTree->AddNode(pSubEntry, buf, (char*)bmi->Name(), pData);
								}
							}
						}
					}					
				}
			}
		}
	}
}

////////////////////////////////////////////////////////////////////////////////////////////////////

MtlEditor::MtlEditor()
{
	pMtlEditorDlg = NULL;
}


MtlEditor::~MtlEditor()
{

}

BOOL CALLBACK MtlEdDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{
	switch(msg)
	{
	case WM_INITDIALOG:
		return TRUE;
	}

	return FALSE;
}

void MtlEditor::BeginEditParams(Interface *ip,IUtil *iu)
{
	REGISTERCLASS(ThumbNailList, hInstance);
	REGISTERCLASS(ThumbNailTree, hInstance);

	if (!pMtlEditorDlg)
		pMtlEditorDlg = new MtlEditorDlg(ip);

	pMtlEditorDlg->Show();
	pMtlEditorDlg->RefreshMtls();

	//HWND hwndDlg = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_MTLBROWSER), NULL, MtlEdDlgProc);
}

void MtlEditor::EndEditParams(Interface *ip,IUtil *iu)
{
	if (pMtlEditorDlg)
		delete pMtlEditorDlg;
}

void MtlEditor::SelectionSetChanged(Interface *ip,IUtil *iu)
{

}
