#include "FuncEnter.h"

/*
	DebugTool.cpp
	Debugging plugin for Max
	6-17-01
*/

#include "Next.h"
#include "DebugTool.h"
#include "Resource.h"
#include "Appdata.h"
#include "../Wibble/Wibble.h"
#include "modstack.h"
#include "imagedata.h"
#include "../LightMap/LightMapGen.h"
#include "GLRenderer.h"
#include "ImageData.h"
#include "../Material/NExtMat.h"
#include "../Texture/NExtTexture.h"
#include "../misc/maxutil.h"
#include "../UI/ProgressBar.h"
#include "../UI/ThumbnailList.h"
#include "../misc/maxutil.h"
#include "WibbleWatcher.h"
#include "tvnode.h"
#include "memDebug.h"
#include "FlagReassignerPatch2.h"
#include "SymbolDebug.h"

#define PROFILE_CHUNKID_MAX 10000		// Profile first 1000 chunks (hoping they're linear)

#define FORCE_RECONSTRUCT

GLRenderer* glRenderer;
HWND        hwndRender;

static DebugToolClassDesc theDebugToolDesc;
ClassDesc2* GetDebugToolDesc() { FUNC_ENTER("GetDebugToolDesc");  return &theDebugToolDesc; }

void PurgeObjAnimData(INode* root=NULL);
bool PurgeWibbleData(INode* root=NULL);
void PurgeWibbleDataSel();
void SelWibbleNodes(INode* root=NULL);
void SaveSelection();
void LoadSelection();
void PurgeAllData(INode* root=NULL);
void PurgeVertColorsChannel(INode* root=NULL,int channel=0);
void PurgeVertColorsSel();
void FixVertColorsSel();
void LightmapTest();
void GLTest();
void GLTestEnd();
void GLUpdateFunc(GLRenderer* gl, void* pData);
void ProfileFileUsage();
void OpenTestDlg();
void ClearPS2Materials();
void WibbleRepair(INode* node);
void WibbleRepair();
void WibbleRepairSel();
void AddTrackTest();
void TrackDoTest();
void FixVertexNormals();
void PurgeScriptCache(INode* node = NULL);
void PurgeNodeCache(INode* node = NULL);
void GenerateDebugPos();
void ClearPerBoneTolerances(INode* node = NULL);
INode* FixPolyConvNode(INode* node);
void FixPolyConvProblem(INode* node = NULL, int* nFoundObjs = NULL);

DebugTool::DebugTool()
{ FUNC_ENTER("DebugTool::DebugTool"); 

}

DebugTool::~DebugTool()
{ FUNC_ENTER("DebugTool::~DebugTool"); 

}

BOOL __stdcall DebugTool::DlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam)
{ FUNC_ENTER("DebugTool::DlgProc"); 
	switch(msg)
	{
	case WM_CREATE:
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{
		case IDC_PURGE:
			PurgeObjAnimData();
			MessageBox(gInterface->GetMAXHWnd(),"The object animation data has been purged from the scene","Purge Obj Anim Data",MB_ICONINFORMATION|MB_OK);
			return TRUE;

		case IDC_PURGEALL:
			PurgeAllData();
			return TRUE;


		case IDC_PURGEVERTCOLORS:
			PurgeVertColorsSel();
			return TRUE;

		case IDC_PURGEWIBBLE:
			PurgeWibbleData();
			return TRUE;

		case IDC_PURGEWIBBLESEL:
			PurgeWibbleDataSel();
			return TRUE;

		case IDC_FIXVERTCOL:
			FixVertColorsSel();
			return TRUE;

		case IDC_SELWIBBLE:
			gInterface->ClearNodeSelection();
			SelWibbleNodes();
			return TRUE;

		case IDC_SAVESEL:
			SaveSelection();
			return TRUE;

		case IDC_LOADSEL:
			gInterface->ClearNodeSelection();
			LoadSelection();
			return TRUE;

		case IDC_LIGHTMAPTEST:
			LightmapTest();
			return TRUE;

		case IDC_GLTEST:
			GLTest();
			return TRUE;

		case IDC_GLTEST_END:
			GLTestEnd();
			return TRUE;

		case IDC_PROFILESIZE:
			ProfileFileUsage();
			return TRUE;

		case IDC_TESTDLG:
			OpenTestDlg();
			return TRUE;

		case IDC_CLEARPS2MAT:
			ClearPS2Materials();
			return TRUE;

		case IDC_INSTALLWATCHER:
			InstallPeriodicWibbleWatcher();
			MessageBox(gInterface->GetMAXHWnd(), "Periodic Wibble Watcher Installed", "WibbleWatcher", MB_ICONINFORMATION|MB_OK);
			return TRUE;

		case IDC_UNINSTALLWATCHER:
			UnInstallPeriodicWibbleWatcher();
			MessageBox(gInterface->GetMAXHWnd(), "Periodic Wibble Watcher UN-Installed", "WibbleWatcher", MB_ICONINFORMATION|MB_OK);
			return TRUE;

		case IDC_RESETWATCHER:
			ResetWibbleWatcher();
			MessageBox(gInterface->GetMAXHWnd(), "You will now recieve wibble warning messages again (as long as the watcher is active)", "WibbleWatcher", MB_ICONINFORMATION|MB_OK);
			return TRUE;

		case IDC_WIBBLEREPAIR:
			WibbleRepair();
			return TRUE;

		case IDC_WIBBLEREPAIRSEL:
			WibbleRepairSel();
			return TRUE;

		case IDC_ADDTRACKTEST:
			AddTrackTest();
			return TRUE;

		case IDC_TRACKDOTEST:
			TrackDoTest();
			return TRUE;

		case IDC_FIXNORMALS:
			FixVertexNormals();
			return TRUE;

		case IDC_PURGESCRIPTCACHE:
			PurgeScriptCache();
			MessageBox(gInterface->GetMAXHWnd(), "Script cache data has been purged", "Data was purged", MB_OK|MB_ICONINFORMATION);
			return TRUE;

		case IDC_PURGENODECACHE:
			PurgeNodeCache();
			MessageBox(gInterface->GetMAXHWnd(), "Node cache data has been purged", "Data was purged", MB_OK|MB_ICONINFORMATION);
			return TRUE;

		case IDC_GENDEBUGPOS:
			GenerateDebugPos();
			return TRUE;

		case IDC_CLEAR_PERBONE_TOL:
			ClearPerBoneTolerances();
			MessageBox(gInterface->GetMAXHWnd(), "Per bone tolerances have been cleared", "Data was purged", MB_OK|MB_ICONINFORMATION);
			return TRUE;

		case IDC_FIXPOLYCONV:
			{
				int nObjs = 0;
				FixPolyConvProblem(NULL, &nObjs);

				char buf[256];
				sprintf(buf, "Fixed %i nodes with Fix Poly Conv problem.", nObjs);
				MessageBox(hwnd, buf, "Fix Poly Conv problem", MB_ICONINFORMATION|MB_OK);
			}
			return TRUE;

		case IDC_RESETCOLFLAGS:
			{
				if (DoFlagReassignment(true))
					MessageBox(gInterface->GetMAXHWnd(), "Camera Collision flag update completed!", "Camera Collision flag update", MB_ICONINFORMATION|MB_OK);

				return TRUE;
			}

		case IDC_STARTSYMBOLS:
			{
				if (SymbolsActive())
				{
					MessageBox(gInterface->GetMAXHWnd(), "Symbolic Debug mode is already active", "Start Symbolic Debug Mode", MB_ICONINFORMATION|MB_OK);
					return TRUE;
				}

				InitSymbols();
				MessageBox(gInterface->GetMAXHWnd(), "Symbolic Debug mode is now active", "Start Symbolic Debug Mode", MB_ICONINFORMATION|MB_OK);
				return TRUE;
			}

		case IDC_ENDSYMBOLS:
			{
				if (!SymbolsActive())
				{
					MessageBox(gInterface->GetMAXHWnd(), "Symbolic Debug mode was not active", "End Symbolic Debug Mode", MB_ICONINFORMATION|MB_OK);
					return TRUE;
				}

				ShutdownSymbols();
				MessageBox(gInterface->GetMAXHWnd(), "Symbolic Debug mode has been disabled", "End Symbolic Debug Mode", MB_ICONINFORMATION|MB_OK);
				return TRUE;
			}
		}
		break;
	}

	return FALSE;
}

void DebugTool::BeginEditParams(Interface* ip,IUtil* iu)
{ FUNC_ENTER("DebugTool::BeginEditParams"); 
	this->ip = ip;
	this->iu = iu;

	hwnd = ip->AddRollupPage(hInstance,MAKEINTRESOURCE(IDD_DEBUG),DlgProc,"Debug Tool");
	SelectionSetChanged(ip,iu);
}

void DebugTool::EndEditParams(Interface* ip, IUtil* iu)
{ FUNC_ENTER("DebugTool::EndEditParams"); 
	ip->DeleteRollupPage(hwnd);
}

void DebugTool::SelectionSetChanged(Interface* ip, IUtil* iu)
{ FUNC_ENTER("DebugTool::SelectionSetChanged"); 
	int nSelNodes = ip->GetSelNodeCount();

	if (nSelNodes>0)
	{
		// Update the name of the selected node
		INode* node = ip->GetSelNode(0);
		SetDlgItemText(hwnd, IDC_NODENAME, (char*)node->GetName());

		// Update the type of the selected node
		Object* obj = node->EvalWorldState(0).obj;

		//if (obj->ClassID() == vTRIGGER_CLASS_ID)
		//	strcpy(strType, "Trigger");
		//else if (obj->ClassID() == 

		CStr className;
		obj->GetClassName(className);
		SetDlgItemText(hwnd, IDC_NODETYPE, (char*)className);

		char strClassID[256];
		sprintf(strClassID, "%x,%x", obj->ClassID().PartA(), obj->ClassID().PartB());
		SetDlgItemText(hwnd, IDC_NODECLASS, strClassID);
	}

	if (nSelNodes>1 || nSelNodes==0)
		return;

	INode* node = ip->GetSelNode(0);
	Object* obj = node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);

		Mesh& mesh = triObj->GetMesh();


		UVVert* colors = mesh.mapVerts(0);
		TVFace* faces  = mesh.mapFaces(0);

		//int			curVCChan;	// current map channel to use for colors (default = 0)
		//VertColor *	curVCArray;	// possible external color array (default = NULL)
		//TVFace	  * curVCFace;	// possible external face array (default = NULL)

		// Verify that only one face is selected
		if (mesh.faceSel.NumberSet() != 1)
			return;

		// Find the first selected bit
		int selFace;

		for(int i=0;i<mesh.numFaces;i++)
		{
			if (mesh.faceSel[i])
			{
				selFace = i;
				break;
			}
		}

		// Now that we have the selected face, display it's color data in the window
		char buf[256];
		int index;

		sprintf(buf,"%i",selFace);
		SetDlgItemText(hwnd,IDC_FACEID,buf);

		index = faces[selFace].t[0];

		if (index < mesh.numCVerts)
		{
			sprintf(buf,"%f (%f)",colors[index].x,colors[index].x*255.0f);
			SetDlgItemText(hwnd,IDC_VERTR0,buf);

			sprintf(buf,"%f (%f)",colors[index].y,colors[index].y*255.0f);
			SetDlgItemText(hwnd,IDC_VERTG0,buf);

			sprintf(buf,"%f (%f)",colors[index].z,colors[index].z*255.0f);
			SetDlgItemText(hwnd,IDC_VERTB0,buf);
		}

		index = faces[selFace].t[1];
		if (index < mesh.numCVerts)
		{
			sprintf(buf,"%f (%f)",colors[index].x,colors[index].x*255.0f);
			SetDlgItemText(hwnd,IDC_VERTR1,buf);
			
			sprintf(buf,"%f (%f)",colors[index].y,colors[index].y*255.0f);
			SetDlgItemText(hwnd,IDC_VERTG1,buf);
			
			sprintf(buf,"%f (%f)",colors[index].z,colors[index].z*255.0f);
			SetDlgItemText(hwnd,IDC_VERTB1,buf);
		}

		index = faces[selFace].t[2];
		if (index < mesh.numCVerts)
		{
			sprintf(buf,"%f (%f)",colors[index].x,colors[index].x*255.0f);
			SetDlgItemText(hwnd,IDC_VERTR2,buf);
		
			sprintf(buf,"%f (%f)",colors[index].y,colors[index].y*255.0f);
			SetDlgItemText(hwnd,IDC_VERTG2,buf);
		
			sprintf(buf,"%f (%f)",colors[index].z,colors[index].z*255.0f);
			SetDlgItemText(hwnd,IDC_VERTB2,buf);
		}

		// Find the first selected bit
		int selVert;
		int curFace = 0;

		for(i=0;i<mesh.numVerts;i++)
		{
			if (mesh.vertSel[i])
			{
				selVert = i;
				break;
			}
		}

	BOOL init=FALSE;

	TVFace *cf = mesh.mapFaces(0);
	UVVert *cv = mesh.mapVerts(0);
	Color col;

	for (i=0; i<mesh.getNumFaces(); i++) {
		DWORD *tt = cf[i].t;
		DWORD *vv = mesh.faces[i].v;
		for (int j=0; j<3; j++) {
			if (!mesh.vertSel[vv[j]]) continue;
				col = cv[tt[j]];

			sprintf(buf,"%i",vv[j]);
			SetDlgItemText(hwnd,IDC_VERTID,buf);
		}
	}

	sprintf(buf,"%f (%f)",col.r,col.r*255.0f);
	SetDlgItemText(hwnd,IDC_VR,buf);

	sprintf(buf,"%f (%f)",col.g,col.g*255.0f);
	SetDlgItemText(hwnd,IDC_VG,buf);

	sprintf(buf,"%f (%f)",col.b,col.b*255.0f);
	SetDlgItemText(hwnd,IDC_VB,buf);

	/*
		if (selVert < mesh.numCVerts)
		{
			sprintf(buf,"%f (%f)",colors[selVert].x,colors[selVert].x*255.0f);
			SetDlgItemText(hwnd,IDC_VR,buf);
		
			sprintf(buf,"%f (%f)",colors[selVert].y,colors[selVert].y*255.0f);
			SetDlgItemText(hwnd,IDC_VG,buf);
		
			sprintf(buf,"%f (%f)",colors[selVert].z,colors[selVert].z*255.0f);
			SetDlgItemText(hwnd,IDC_VB,buf);
		}
	*/
	}
}

void PurgeObjAnimData(INode* root)
{ FUNC_ENTER("PurgeObjAnimData"); 
	if (!root)
	{
		root = gInterface->GetRootNode();
		ReferenceTarget *scene = gInterface->GetScenePointer();

		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID);
		scene->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		PurgeObjAnimData(child);
	}

	if (root == gInterface->GetRootNode())
		return;

	// Purge the data
	root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_OBJSETTINGS_ID);
	root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_SETTINGS_ID);
	root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORT_ID);
	root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_SELECTED_ID);
	root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_OBJECT_EXPORTFLAGGED);
}

void PurgeAllData(INode* root)
{ FUNC_ENTER("PurgeAllData"); 
	CStr name;
	if (root)
		name = root->GetName();

	if (!root)
	{
		root = gInterface->GetRootNode();
		ReferenceTarget *scene = gInterface->GetScenePointer();

		Object* obj = root->EvalWorldState(0).obj;

		if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			IDerivedObject* dobj = (IDerivedObject*)obj;
			Object* objref = dobj->GetObjRef();
		}

		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triobj->GetMesh();

			mesh.freeAllVData();

			mesh.InvalidateTopologyCache();
			mesh.InvalidateGeomCache();

			if (triobj != obj)
				triobj->DeleteThis();
		}
	}

	//Object* objref = root->GetObjectRef();

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		PurgeWibbleData(child);
	}

	if (root == gInterface->GetRootNode())
	{
		MessageBox(gInterface->GetMAXHWnd(),"Wibble Channel has been purged.","Purge Wibble Channel",MB_ICONINFORMATION|MB_OK);
		return;
	}

	// Purge the data
	Object* obj;

	obj = root->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
		Mesh& mesh = triobj->GetMesh();

		mesh.freeAllVData();

		mesh.InvalidateTopologyCache();
		mesh.InvalidateGeomCache();

		if (triobj != obj)
			triobj->DeleteThis();
	}

	SClass_ID scid = obj->SuperClassID();
	Class_ID cid = obj->ClassID();

	obj = root->GetObjectRef();

	if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		while(obj)
		{
			if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
			{
				if (obj && obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
					Mesh& mesh = triobj->GetMesh();

					mesh.freeAllVData();

					mesh.InvalidateTopologyCache();
					mesh.InvalidateGeomCache();

					if (triobj != obj)
						triobj->DeleteThis();
				}

				// Get down to the base object
				IDerivedObject* dobj = (IDerivedObject*)obj;
				obj = dobj->GetObjRef();
			}
			else
				break;
		}
	}
}

void PurgeVertColorsSel()
{ FUNC_ENTER("PurgeVertColorsSel"); 
	int numNodes = gInterface->GetSelNodeCount();

	for(int i=0;i<numNodes;i++)
	{
		INode* node = gInterface->GetSelNode(i);

		// Clear out the color per vertex data stored in the mesh
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh* mesh = &triObj->mesh;

			// Purge vertex color channel
			PurgeVertColorsChannel(node, mesh->curVCChan);
			mesh->setNumVertCol(0);
			mesh->setNumVCFaces(0);
		}
	}
}

void FixVertColorsSel()
{ FUNC_ENTER("FixVertColorsSel"); 
	int numNodes = gInterface->GetSelNodeCount();

	for(int i=0;i<numNodes;i++)
	{
		INode* node = gInterface->GetSelNode(i);

		// Clear out the color per vertex data stored in the mesh
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh* mesh = &triObj->mesh;

			// Purge vertex color channel
			//PurgeVertColorsChannel(node, mesh->curVCChan);

			//mesh->setNumVertCol(mesh->numFaces * 3, TRUE);

			int nColorVerts = mesh->getNumVertCol();

			if (mesh->vcFaceData)
			{
				for(int i=0; i < mesh->numFaces; i++)
				{
					TVFace* face = &mesh->vcFaceData[i];

					for(int j=0; j < 3; j++)
					{
						if (face->t[j] > nColorVerts ||
							face->t[j] < 0)
							face->t[j] = 0;
					}
				}
			}

			//mesh->setNumVertCol(0);
			//mesh->setNumVCFaces(0);
		}
	}

	MessageBox(gInterface->GetMAXHWnd(), "Vert Color Fix completed", "Fix Vert Colors (Sel)", MB_ICONINFORMATION | MB_OK);
}

void PurgeVertColorsChannel(INode* root, int channel)
{ FUNC_ENTER("PurgeVertColorsChannel"); 
	CStr name;
	if (root)
		name = root->GetName();

	if (!root)
	{
		root = gInterface->GetRootNode();
		ReferenceTarget *scene = gInterface->GetScenePointer();

		Object* obj = root->EvalWorldState(0).obj;

		if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			IDerivedObject* dobj = (IDerivedObject*)obj;
			Object* objref = dobj->GetObjRef();
		}

		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triobj->GetMesh();

			mesh.setNumMapVerts( 0, 0);
			mesh.setMapSupport( 0, FALSE );
			mesh.freeMapVerts( 0 );
			mesh.freeMapFaces( 0 );

			mesh.InvalidateTopologyCache();
			mesh.InvalidateGeomCache();

			if (triobj != obj)
				triobj->DeleteThis();
		}
	}

	//Object* objref = root->GetObjectRef();

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		PurgeWibbleData(child);
	}

	if (root == gInterface->GetRootNode())
	{
		MessageBox(gInterface->GetMAXHWnd(),"Wibble Channel has been purged.","Purge Wibble Channel",MB_ICONINFORMATION|MB_OK);
		return;
	}

	// Purge the data
	Object* obj;

	obj = root->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
		Mesh& mesh = triobj->GetMesh();
		
		mesh.setNumMapVerts( channel, 0);
		mesh.setMapSupport( channel, FALSE );
		mesh.freeMapVerts( channel );
		mesh.freeMapFaces( channel );

		mesh.InvalidateTopologyCache();
		mesh.InvalidateGeomCache();

		if (triobj != obj)
			triobj->DeleteThis();
	}

	SClass_ID scid = obj->SuperClassID();
	Class_ID cid = obj->ClassID();

	obj = root->GetObjectRef();

	if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		IDerivedObject* dobj = (IDerivedObject*)obj;

		while(obj)
		{
			if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
			{
				if (obj && obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
					Mesh& mesh = triobj->GetMesh();
					
					mesh.setNumMapVerts( channel, 0);
					mesh.setMapSupport( channel, FALSE );
					mesh.freeMapVerts( channel );
					mesh.freeMapFaces( channel );

					mesh.InvalidateTopologyCache();
					mesh.InvalidateGeomCache();

					if (triobj != obj)
						triobj->DeleteThis();
				}

				// Get down to the base object
				IDerivedObject* dobj = (IDerivedObject*)obj;
				obj = dobj->GetObjRef();
			}
			else
				break;
		}
	}
}

void PurgeObjectWibbleData(Object* obj)
{ FUNC_ENTER("PurgeObjectWibbleData"); 
	if (obj && obj->CanConvertToType(polyObjectClassID))
	{
		try
		{
			PolyObject* polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);
			MNMesh& mesh = polyobj->GetMesh();

			mesh.ClearMap(vWIBBLE_INDEX_CHANNEL);
			mesh.ClearMap(vWIBBLE_OFFSET_CHANNEL);

			if (polyobj != obj)
				polyobj->DeleteThis();
		}
		catch(...)
		{
		}
	}

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
		Mesh& mesh = triobj->GetMesh();

		mesh.setNumMapVerts( vWIBBLE_INDEX_CHANNEL, 0);
		mesh.setNumMapVerts( vWIBBLE_OFFSET_CHANNEL, 0);

		mesh.setMapSupport( vWIBBLE_INDEX_CHANNEL, FALSE );
		mesh.setMapSupport( vWIBBLE_OFFSET_CHANNEL, FALSE );

		mesh.InvalidateTopologyCache();
		mesh.InvalidateGeomCache();

		if (triobj != obj)
			triobj->DeleteThis();
	}
}

bool PurgeWibbleData(INode* root)
{ FUNC_ENTER("PurgeWibbleData"); 
	CStr name;
	if (root)
		name = root->GetName();

	if (!root)
	{
		root = gInterface->GetRootNode();
		ReferenceTarget *scene = gInterface->GetScenePointer();

		Object* obj = root->EvalWorldState(0).obj;

		// VERIFY: shouldn't obj = objref
		if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
		{
			IDerivedObject* dobj = (IDerivedObject*)obj;
			Object* objref = dobj->GetObjRef();
		}
		///////////////

		// Attempt to clear mesh as both a poly object and a triobject in case the
		// object is originally one or the other
		if (obj && obj->CanConvertToType(polyObjectClassID))
		{
			// If we fail to convert to a poly object this object suffers from the poly
			// conversion problem and we will auto fix it
			try
			{
				PolyObject* polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);
				MNMesh& mesh = polyobj->GetMesh();

				mesh.ClearMap(vWIBBLE_INDEX_CHANNEL);
				mesh.ClearMap(vWIBBLE_OFFSET_CHANNEL);

				if (polyobj != obj)
					polyobj->DeleteThis();
			}
			catch(...)
			{
				FixPolyConvNode(root);
				return true;
			}
		}


		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triobj->GetMesh();

			mesh.setNumMapVerts( vWIBBLE_INDEX_CHANNEL, 0);
			mesh.setNumMapVerts( vWIBBLE_OFFSET_CHANNEL, 0);

			mesh.setMapSupport( vWIBBLE_INDEX_CHANNEL, FALSE );
			mesh.setMapSupport( vWIBBLE_OFFSET_CHANNEL, FALSE );

			mesh.InvalidateTopologyCache();
			mesh.InvalidateGeomCache();

			if (triobj != obj)
				triobj->DeleteThis();
		}
	}

	//Object* objref = root->GetObjectRef();

	int nKids = root->NumberOfChildren();

	//static int cnt = 0;
	bool bRet = false;

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		char strBuf[256];
		sprintf(strBuf, "Node %s being purged!\n", (char*)child->GetName());
		OutputDebugString(strBuf);
		
		if (PurgeWibbleData(child))
			bRet = true;
	}

	/*
	if (root->GetName() == CStr("g_gb_glass_05"))
	{
		__asm int 3;
	}

	char strBuf[256];
	sprintf(strBuf, "PurgeWibble: %i '%s'\n", cnt++, (char*)root->GetName());
	OutputDebugString(strBuf);

	if (cnt >= 2945)
		__asm int 3;
	*/

	if (root == gInterface->GetRootNode())
	{
		// Reexecute purge if we had to clean up a poly object
		if (bRet)
			return PurgeWibbleData(root);

		MessageBox(gInterface->GetMAXHWnd(),"Wibble Channel has been purged.","Purge Wibble Channel",MB_ICONINFORMATION|MB_OK);
		return false;
	}

	// Purge the data
	Object* obj;

	obj = root->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
		Mesh& mesh = triobj->GetMesh();
		
		mesh.setNumMapVerts( vWIBBLE_INDEX_CHANNEL, 0);
		mesh.setNumMapVerts( vWIBBLE_OFFSET_CHANNEL, 0);

		mesh.setMapSupport( vWIBBLE_INDEX_CHANNEL, FALSE );
		mesh.setMapSupport( vWIBBLE_OFFSET_CHANNEL, FALSE );

		mesh.InvalidateTopologyCache();
		mesh.InvalidateGeomCache();

		if (triobj != obj)
			triobj->DeleteThis();
	}

	if (obj && obj->CanConvertToType(polyObjectClassID))
	{
		// If we fail to convert to a poly object this object suffers from the poly
		// conversion problem and we will auto fix it
		try
		{
			PolyObject* polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);

			MNMesh& mesh = polyobj->GetMesh();

			mesh.ClearMap(vWIBBLE_INDEX_CHANNEL);
			mesh.ClearMap(vWIBBLE_OFFSET_CHANNEL);

			if (polyobj != obj)
				polyobj->DeleteThis();
		}
		catch(...) 
		{
			FixPolyConvNode(root);
			return true;
		}
	}

	SClass_ID scid = obj->SuperClassID();
	Class_ID cid = obj->ClassID();

	obj = root->GetObjectRef();

	if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
	{
		while(obj)
		{
			if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
			{
				IDerivedObject* dobj = (IDerivedObject*)obj;

				if (obj && obj->CanConvertToType(polyObjectClassID))
				{
					try
					{
						PolyObject* polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);
						MNMesh& mesh = polyobj->GetMesh();

						mesh.ClearMap(vWIBBLE_INDEX_CHANNEL);
						mesh.ClearMap(vWIBBLE_OFFSET_CHANNEL);

						if (polyobj != obj)
							polyobj->DeleteThis();
					}
					catch(...)
					{
						FixPolyConvNode(root);
						return true;
					}
				}
				else
				if (obj && obj->CanConvertToType(triObjectClassID))
				{
					TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
					Mesh& mesh = triobj->GetMesh();
					
					mesh.setNumMapVerts( vWIBBLE_INDEX_CHANNEL, 0);
					mesh.setNumMapVerts( vWIBBLE_OFFSET_CHANNEL, 0);

					mesh.setMapSupport( vWIBBLE_INDEX_CHANNEL, FALSE );
					mesh.setMapSupport( vWIBBLE_OFFSET_CHANNEL, FALSE );

					mesh.InvalidateTopologyCache();
					mesh.InvalidateGeomCache();

					if (triobj != obj)
						triobj->DeleteThis();
				}

				// Get down to the base object
				//do 
				//{
					obj = dobj->GetObjRef();

					while(obj)
					{
						PurgeObjectWibbleData(obj);
						if (obj && obj->SuperClassID() == GEN_DERIVOB_CLASS_ID)
							obj = ((IDerivedObject*)obj)->GetObjRef();
						else
							obj = NULL;
					}


				//} while(obj);
			}
			else
				break;
		}
	}

	return false;
}

void PurgeWibbleDataSel()
{ FUNC_ENTER("PurgeWibbleDataSel"); 
	int nNodes = gInterface->GetSelNodeCount();
	
	for(int i=0;i<nNodes;i++)
	{
		INode* node = gInterface->GetSelNode(i);

		//Object* obj = node->EvalWorldState(0).obj;

		CStr name = CStr("ClearSel: ") + node->GetName() + CStr("\n");
		OutputDebugString(name);

		PurgeWibbleData(node);
	}

	MessageBox(gInterface->GetMAXHWnd(),"The current selections wibble data has been purged.","Purge Wibble Data",MB_ICONINFORMATION|MB_OK);
}

void SelWibbleNodes(INode* root)
{ FUNC_ENTER("SelWibbleNodes"); 
	if (!root)
	{
		root = gInterface->GetRootNode();
		ReferenceTarget *scene = gInterface->GetScenePointer();

		Object* obj = root->EvalWorldState(0).obj;

		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh& mesh = triobj->GetMesh();

			if ( (mesh.mapSupport( vWIBBLE_INDEX_CHANNEL ) &&
				  mesh.getNumMapVerts( vWIBBLE_INDEX_CHANNEL ) > 0) ||
				 (mesh.mapSupport( vWIBBLE_OFFSET_CHANNEL ) &&
				  mesh.getNumMapVerts( vWIBBLE_OFFSET_CHANNEL ) > 0) )
			{
				gInterface->SelectNode(root,0);
			}

			if (triobj != obj)
				triobj->DeleteThis();
		}
	}

	int nKids = root->NumberOfChildren();

	for(int i=0;i<nKids;i++)
	{
		INode* child = root->GetChildNode(i);
		SelWibbleNodes(child);
	}

	if (root == gInterface->GetRootNode())
	{
		gInterface->ForceCompleteRedraw();
		return;
	}

	// Purge the data
	Object* obj = root->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
		Mesh& mesh = triobj->GetMesh();

		int indexVerts  = mesh.getNumMapVerts( vWIBBLE_INDEX_CHANNEL );
		int offsetVerts = mesh.getNumMapVerts( vWIBBLE_OFFSET_CHANNEL );

		if ( (mesh.mapSupport( vWIBBLE_INDEX_CHANNEL ) &&
			  mesh.getNumMapVerts( vWIBBLE_INDEX_CHANNEL ) > 0) ||
			 (mesh.mapSupport( vWIBBLE_OFFSET_CHANNEL ) &&
			  mesh.getNumMapVerts( vWIBBLE_OFFSET_CHANNEL ) > 0) )
		{
			gInterface->SelectNode(root,0);
		}

		if (triobj != obj)
			triobj->DeleteThis();
	}
}

void SaveSelection()
{ FUNC_ENTER("SaveSelection"); 
	FILE* fp = fopen("c:\\savedsel.txt","w");

	if (!fp)
	{
		MessageBox(gInterface->GetMAXHWnd(),"Failed!","SaveSelection",MB_ICONWARNING|MB_OK);
		return;
	}

	for(int i=0;i<gInterface->GetSelNodeCount();i++)
	{
		INode* node = gInterface->GetSelNode(i);
		CStr name = node->GetName();

		fprintf(fp,"%s\n",(char*)name);
	}

	fclose(fp);
}

void LoadSelection()
{ FUNC_ENTER("LoadSelection"); 
	FILE* fp = fopen("c:\\savedsel.txt","r");

	if (!fp)
	{
		MessageBox(gInterface->GetMAXHWnd(),"Failed!","SaveSelection",MB_ICONWARNING|MB_OK);
		return;
	}

	while(!feof(fp))
	{
		char buf[512];
		fgets(buf,511,fp);

		int slen = strlen(buf);
		buf[slen-1] = 0;

		INode* node = gInterface->GetINodeByName(buf);

		if (node)
		{
			gInterface->SelectNode(node,0);
		}
	}

	fclose(fp);
	gInterface->ForceCompleteRedraw();
}

struct GLRenderData
{
	ImageData* img;				// The original texture to use
	ImageData* imgLightmap;		// The lightmap to apply

	Point3*    verts;			// Verts of model
	UVVert*    uvs;				// UV texture coordinates of model
	UVVert*    uvs2;			// 1-to-1 corresponding UV texture coordinates of lightmap

	Face*      faces;
	TVFace*    tvFaces;

	int        numFaces;
	int        numVerts;
	int        numTVerts;

	int        texID;		// GL Name for the original texture
	int        texID2;		// GL Name for the lightmap texture

	GLRenderData()
	{
		img = NULL;
		imgLightmap = NULL;
		verts = NULL;
		uvs = NULL;
		uvs2 = NULL;
		faces = NULL;
		tvFaces = NULL;

		numFaces  = 0;
		numVerts  = 0;
		numTVerts = 0;
	}

	~GLRenderData()
	{ FUNC_ENTER("GLRenderData::~GLRenderData"); 
		FreeAll();
	}

	void FreeAll()
	{ FUNC_ENTER("GLRenderData::FreeAll"); 
		if (img)
			delete img;

		if (imgLightmap)
			delete imgLightmap;
		
		if (verts)
			free(verts);

		if (uvs)
			free(uvs);

		if (uvs2)
			free(uvs2);

		if (faces)
			free(faces);

		if (tvFaces)
			free(tvFaces);
	}
};

GLRenderData rdata;

void glUpdate(GLRenderer* gl, void* pData)
{ FUNC_ENTER("glUpdate"); 
	GLRenderData* rd = (GLRenderData*)pData;

	// No optimizations for now
	Point3* verts = rd->verts;
	Face*   faces = rd->faces;
	TVFace* tvFaces = rd->tvFaces;
	UVVert* tVerts = rd->uvs;

	glMatrixMode(GL_MODELVIEW);

	for(int i = 0; i < rd->numFaces; i++)
	{

		if (gl->SupportsMultiTexture())
		{
			gl->glActiveTextureARB(GL_TEXTURE0_ARB);
			glBindTexture(GL_TEXTURE_2D, rd->texID);

			gl->glActiveTextureARB(GL_TEXTURE1_ARB);
			glBindTexture(GL_TEXTURE_2D, rd->texID2);
		}
		else
		{
			glBindTexture(GL_TEXTURE_2D, rd->texID2);
		}

		//glColor3f(1.0f,0.0f,0.0f);

		glBegin(GL_TRIANGLES);

//		glTexCoord2f(0.0f,0.0f);

//		glTexCoord2f(-tVerts[tvFaces[i].t[0]].x,
//				     -tVerts[tvFaces[i].t[0]].y);

		if (gl->SupportsMultiTexture())
		{
			gl->glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 
									 rd->uvs[tvFaces[i].t[0]].x,
									 rd->uvs[tvFaces[i].t[0]].y);

			gl->glMultiTexCoord2fARB(GL_TEXTURE1_ARB,
									 rd->uvs2[tvFaces[i].t[0]].x,
									 rd->uvs2[tvFaces[i].t[0]].y);
		}
		else
		{
			glTexCoord2f(rd->uvs2[tvFaces[i].t[0]].x,
						 rd->uvs2[tvFaces[i].t[0]].y);
		}

		glVertex3f(verts[faces[i].v[0]].x,
				   verts[faces[i].v[0]].z,
				   verts[faces[i].v[0]].y);

//		glTexCoord2f(1.0f,0.0f);

//		glTexCoord2f(-tVerts[tvFaces[i].t[1]].x,
//				     -tVerts[tvFaces[i].t[1]].y);

		if (gl->SupportsMultiTexture())
		{
			gl->glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 
									 rd->uvs[tvFaces[i].t[1]].x,
									 rd->uvs[tvFaces[i].t[1]].y);

			gl->glMultiTexCoord2fARB(GL_TEXTURE1_ARB,
									 rd->uvs2[tvFaces[i].t[1]].x,
									 rd->uvs2[tvFaces[i].t[1]].y);
		}
		else
		{
			glTexCoord2f(rd->uvs2[tvFaces[i].t[1]].x,
						 rd->uvs2[tvFaces[i].t[1]].y);
		}

		glVertex3f(verts[faces[i].v[1]].x,
				   verts[faces[i].v[1]].z,
				   verts[faces[i].v[1]].y);

//		glTexCoord2f(1.0f,1.0f);

//		glTexCoord2f(-tVerts[tvFaces[i].t[2]].x,
//				     -tVerts[tvFaces[i].t[2]].y);

		if (gl->SupportsMultiTexture())
		{
			gl->glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 
									 rd->uvs[tvFaces[i].t[2]].x,
									 rd->uvs[tvFaces[i].t[2]].y);

			gl->glMultiTexCoord2fARB(GL_TEXTURE1_ARB,
									 rd->uvs2[tvFaces[i].t[2]].x,
									 rd->uvs2[tvFaces[i].t[2]].y);
		}
		else
		{
			glTexCoord2f(rd->uvs2[tvFaces[i].t[2]].x,
						 rd->uvs2[tvFaces[i].t[2]].y);
		}

		glVertex3f(verts[faces[i].v[2]].x,
				   verts[faces[i].v[2]].z,
				   verts[faces[i].v[2]].y);

		glEnd();
	}	
}

void LightmapTest()
{ FUNC_ENTER("LightmapTest"); 
	int nNodes = gInterface->GetSelNodeCount();

	if (nNodes != 2)
	{
		MessageBox(gInterface->GetMAXHWnd(), "You must have 2 nodes selected.  A geom node and a light node.", "Invalid Lightmap test selection", MB_OK);
		return;
	}

	INode *node1, *node2, *geomNode, *lightNode;

	geomNode  = NULL;
	lightNode = NULL;

	node1 = gInterface->GetSelNode(0);
	node2 = gInterface->GetSelNode(1);

	Object* obj1 = node1->EvalWorldState(0).obj;
	Object* obj2 = node2->EvalWorldState(0).obj;

	if (obj1->ClassID() == Class_ID(OMNI_LIGHT_CLASS_ID, 0))
		lightNode = node1;
	
	if (obj2->ClassID() == Class_ID(OMNI_LIGHT_CLASS_ID, 0))
		lightNode = node2;

	if (obj1->SuperClassID() == GEOMOBJECT_CLASS_ID)
		geomNode = node1;

	if (obj2->SuperClassID() == GEOMOBJECT_CLASS_ID)
		geomNode = node2;

	if (!geomNode)
	{
		MessageBox(gInterface->GetMAXHWnd(), "No geom node was selected", "Invalid Lightmap Test Selection", MB_OK);
		return;
	}

	if (!lightNode)
	{
		MessageBox(gInterface->GetMAXHWnd(), "No light node was selected", "Invalid Lightmap Test Selection", MB_OK);
		return;
	}

	LightmapGen lmapgen;

	//if (glRenderer)
	//	delete glRenderer;

	if (!glRenderer)
	{
		glRenderer = new GLRenderer;
		hwndRender = glRenderer->InitWindow();
		glRenderer->ShowWindow();
	}
	else
	{
		
		if (!IsWindowVisible(hwndRender))
		{
			hwndRender = glRenderer->InitWindow();
			glRenderer->ShowWindow();
		}
		
	}

	// Allocate render data for a single triangle
	rdata.FreeAll();

	rdata.verts   = (Point3*)malloc(sizeof(Point3)*3);
	rdata.faces   = (Face*)malloc(sizeof(Face)*1);
	rdata.uvs     = (UVVert*)malloc(sizeof(UVVert)*3);
	rdata.uvs2    = (UVVert*)malloc(sizeof(UVVert)*3);
	rdata.tvFaces = (TVFace*)malloc(sizeof(TVFace)*1);

	Matrix3 lightTM  = lightNode->GetNodeTM(0);
	Matrix3 geomTM   = geomNode->GetNodeTM(0);
	Object* lightObj = lightNode->EvalWorldState(0).obj;

	Point3 trans = lightTM.GetTrans();
	Color  color;

	color.r = 128;
	color.g = 128;
	color.b = 128;

	lmapgen.AddLight(trans, color);

	// Light each triangle of the geom node
	Object* geomObj = geomNode->EvalWorldState(0).obj;

	if (geomObj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)geomObj->ConvertToType(0, triObjectClassID);

		Mesh& mesh = triObj->GetMesh();

		for(int i = 0; i < mesh.numFaces; i++)
		{
			UVVert uv[3];
			Point3 pts[3];

			pts[0] = mesh.verts[mesh.faces[i].v[0]] * geomTM;
			pts[1] = mesh.verts[mesh.faces[i].v[1]] * geomTM;
			pts[2] = mesh.verts[mesh.faces[i].v[2]] * geomTM;

			if (rdata.img)
				delete rdata.img;

			rdata.img = new ImageData;

			// Load the texture used for this triangle from the associated material
			Mtl* mtl = geomNode->GetMtl();

			if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				INExtMaterial* next_mat = dynamic_cast<INExtMaterial*>(mtl);
				Texmap* tmap = next_mat->GetTexmap(0);
				
				if (tmap->ClassID() == NEXT_TEXTURE_CLASS_ID)
				{
					INExtTexture* next_tex = dynamic_cast< INExtTexture* >(tmap);
					BitmapInfo* bmi = next_tex->GetBaseMap(vPLAT_PS2);
					
					rdata.img->ReadPNG((char*)bmi->Name());
				}
			}

			if (rdata.imgLightmap)
				delete rdata.imgLightmap;

			rdata.imgLightmap = new ImageData;

			lmapgen.LightTri(pts,
							 uv,
							 rdata.imgLightmap);

			// Assign data to render data
			rdata.faces[0].v[0] = 0;
			rdata.faces[0].v[1] = 1;
			rdata.faces[0].v[2] = 2;

			rdata.verts[0] = pts[0];
			rdata.verts[1] = pts[1];
			rdata.verts[2] = pts[2];

			rdata.tvFaces[0].t[0] = 0;
			rdata.tvFaces[0].t[1] = 1;
			rdata.tvFaces[0].t[2] = 2;

			rdata.numTVerts = mesh.numTVerts;

			if (mesh.numTVerts > 0)
			{
				rdata.uvs[0] = mesh.tVerts[0];
				rdata.uvs[1] = mesh.tVerts[1];
				rdata.uvs[2] = mesh.tVerts[2];
			}
			else
			{
				rdata.uvs[0].x = 0.0f;
				rdata.uvs[0].y = 0.0f;
				rdata.uvs[0].z = 0.0f;

				rdata.uvs[1] = rdata.uvs[0];
				rdata.uvs[2] = rdata.uvs[0];
			}

			rdata.uvs2[0] = uv[0];
			rdata.uvs2[1] = uv[1];
			rdata.uvs2[2] = uv[2];

			rdata.numFaces = 1;
			rdata.numVerts = 3;

			rdata.texID = glRenderer->UploadTexture(rdata.img);
			rdata.texID2 = glRenderer->UploadTexture(rdata.imgLightmap);

			delete rdata.img;
			rdata.img = NULL;

			delete rdata.imgLightmap;
			rdata.imgLightmap = NULL;

			//glUpdate(glRenderer, &rdata);

			glRenderer->SetCallback(glUpdate, &rdata);
		}

		if (geomObj != triObj)
			triObj->DeleteThis();
	}

	//MessageBox(gInterface->GetMAXHWnd(), "Completed Lightmap test","Lightmap Test Completed", MB_OK);
}

void GLTest()
{ FUNC_ENTER("GLTest"); 
	glRenderer = new GLRenderer;
	glRenderer->InitWindow();
	glRenderer->ShowWindow();
	glRenderer->SetCallback(GLUpdateFunc, NULL);	
	glRenderer->Update();
}

void GLTestEnd()
{ FUNC_ENTER("GLTestEnd"); 
	if (glRenderer)
	{
		delete glRenderer;
		glRenderer = NULL;
	}
}

void GLUpdateFunc(GLRenderer* gl, void* pData)
{ FUNC_ENTER("GLUpdateFunc"); 
	glBegin(GL_QUADS);

		glVertex3f(0.0f,0.0f,0.0f);
		glVertex3f(1.0f,0.0f,0.0f);
		glVertex3f(1.0f,1.0f,0.0f);
		glVertex3f(0.0f,1.0f,0.0f);

	glEnd();
}

void ProfileFileUsage()
{ FUNC_ENTER("ProfileFileUsage"); 
	INodeTab nodelist;
	GetAllNodes(nodelist);
	ProgressBar pbar(hInstance, gInterface->GetMAXHWnd(), "Profiling file usage...", 0, PROFILE_CHUNKID_MAX);

	FILE* fp = fopen("c:\\fileprofile.txt","w");

	if (!fp)
	{
		MessageBox(gInterface->GetMAXHWnd(), "Failed to create 'fileprofile.txt'", "ProfileFileUsage", MB_OK);
		return;
	}

	fprintf(fp, "AppData Usage Profile:\n\n");

	INode* sceneroot = gInterface->GetRootNode();
	nodelist.Append(1, &sceneroot);

	int nNodes = nodelist.Count();

	int grandTotal = 0;

	// Run through all sub chunk IDs
	DWORD scid;

	for(scid = 0; scid < PROFILE_CHUNKID_MAX; scid++)
	{
		int totalLen = 0;

		for(int i = 0; i < nNodes; i++)
		{
			AppDataChunk* appdata = nodelist[i]->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, scid);

			if (appdata)
				totalLen += appdata->length;
		}

		if (totalLen > 0)
		{
			fprintf(fp, "%i: %i bytes\n", scid, totalLen);
		}

		grandTotal += totalLen;

		pbar.SetVal(scid);
		pbar.Update();
	}

	ReferenceTarget* scene = gInterface->GetScenePointer();
	
	for(scid = 0; scid < PROFILE_CHUNKID_MAX; scid++)
	{
		int totalLen = 0;

		AppDataChunk* appdata = scene->GetAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, scid);

		if (appdata)
			totalLen += appdata->length;

		if (totalLen > 0)
		{
			fprintf(fp, "SceneRoot: %i: %i bytes\n", scid, totalLen);
		}

		grandTotal += totalLen;
	}

	int pbufTotal = 0;
	int vertMem = 0;
	int totalFaceData = 0;
	int totalCASFaceData = 0;

	for(int i = 0; i < nNodes; i++)
	{
		CStr propBuf;
		nodelist[i]->GetUserPropBuffer(propBuf);
		pbufTotal += propBuf.Length();

		Object* obj = nodelist[i]->EvalWorldState(0).obj;

		if (obj && obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
			Mesh& mesh = triobj->GetMesh();

			IFaceDataMgr* pFDMgr = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
			if ( pFDMgr == NULL )
			{
				return;
			}

			FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
			if( fdc != NULL ) 
			{
				int fdcount = fdc->m_data.Count();
				fprintf(fp, "Node: '%s' FaceData count: %i\n", (char*)nodelist[i]->GetName(), fdcount);
				totalFaceData += fdcount;
			}

			CASFaceFlagsData* casfdc = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
			if( casfdc != NULL ) 
			{
				int fdcount = casfdc->m_data.Count();
				fprintf(fp, "Node: '%s' CASFaceData count: %i\n", (char*)nodelist[i]->GetName(), fdcount);
				totalCASFaceData += fdcount;
			}

			int nChannels = mesh.getNumVData();

			for(int channel = 0; channel < nChannels; channel++)
			{
				void* pmem = mesh.vertexData(channel);

				if (pmem)
				{
					int size = _msize(pmem);

					fprintf(fp, "Node: '%s' vertmem: %i.\n", (char*)nodelist[i]->GetName(), size);
					vertMem += size;
				}

				int nMapVerts = mesh.getNumMapVerts(channel);
				int nVerts = mesh.numVerts;

				if (nMapVerts > nVerts)
				{
					fprintf(fp, "WARNING! Node: '%s'   has %i verts and %i mapverts on channel %i of %i.\n",
						    (char*)nodelist[i]->GetName(),
							nVerts,
							nMapVerts,
							channel,
							nChannels);
				}
			}
		}
	}

	fprintf(fp,"\nGrand total: %i bytes.\n", grandTotal);
	fprintf(fp,"PropBuffer total: %i bytes.\n", pbufTotal);
	fprintf(fp,"Total FaceData: %i\n", totalFaceData);
	fprintf(fp,"Total CAS FaceData: %i\n", totalCASFaceData);

	fclose(fp);
}

BOOL CALLBACK TestDlgProc(HWND hwnd, UINT msg, WPARAM wParam, LPARAM lParam)
{ FUNC_ENTER("TestDlgProc"); 
	switch(msg)
	{
	case WM_CLOSE:
		EndDialog(hwnd, 0);
		return TRUE;
	}

	return FALSE;
}

void OpenTestDlg()
{ FUNC_ENTER("OpenTestDlg"); 
	HWND hwndTestDlg;
	ThumbNailList* tnl;

	REGISTERCLASS(ThumbNailList, hInstance);
	hwndTestDlg = CreateDialog(hInstance, MAKEINTRESOURCE(IDD_TESTDLG), gInterface->GetMAXHWnd(), TestDlgProc);	
	ShowWindow(hwndTestDlg, SW_SHOW);

	tnl = new ThumbNailList(hInstance);
	tnl->Attach(GetDlgItem(hwndTestDlg, IDC_TNLIST));

	SendDlgItemMessage(hwndTestDlg, IDC_TNLIST, LB_ADDSTRING, 0, (LPARAM)"c:\\png\\test32.png");
	SendDlgItemMessage(hwndTestDlg, IDC_TNLIST, LB_ADDSTRING, 0, (LPARAM)"c:\\png\\board2.png");
	SendDlgItemMessage(hwndTestDlg, IDC_TNLIST, LB_ADDSTRING, 0, (LPARAM)"c:\\png\\def_hand.png");
	SendDlgItemMessage(hwndTestDlg, IDC_TNLIST, LB_ADDSTRING, 0, (LPARAM)"c:\\png\\def_leg.png");
	SendDlgItemMessage(hwndTestDlg, IDC_TNLIST, LB_ADDSTRING, 0, (LPARAM)"c:\\png\\def_shorts.png");
}

void ClearPS2Materials()
{ FUNC_ENTER("ClearPS2Materials"); 
	// Scan through all the scene materials, if we encounter one that's a PS2 material remove it from the scene
	MtlBaseLib* mtlLib = gInterface->GetSceneMtls();

	int nMaterials = mtlLib->Count();

	Class_ID   cid;
	SClass_ID  scid;
	ClassDesc* cdesc;

	for(int i = 0; i < nMaterials; i++)
	{
		cid  = (*mtlLib)[i]->ClassID();
		scid = (*mtlLib)[i]->SuperClassID();

		cdesc = GetClassDesc(scid, cid);
		
		if (strcmp(cdesc->ClassName(), "PS2 Material")==0)
		{
			char sErr[256];
			sprintf(sErr, "Found a PS2 Material called '%s'.", (char*)((*mtlLib)[i]->GetName()));
			MessageBox(gInterface->GetMAXHWnd(), sErr, "Found PS2 Material", MB_ICONINFORMATION|MB_OK);
		}
	}

	MessageBox(gInterface->GetMAXHWnd(), "Operation completed.", "ClearPS2Materials", MB_ICONINFORMATION|MB_OK);
}

void WibbleRepair(INode* node)
{ FUNC_ENTER("WibbleRepair"); 
	// This routine attempts to remove excess wibble data from a node where
	// there are more wibble verts than actual verts in the mesh.  This used
	// to occur do to a failure to clean up wibble data when verts were deleted

	// Since we don't know what vert(s) were deleted we'll simply keep the verts up
	// to the current number of verts in the mesh, and reconstruct the face list such
	// that faces that reference verts beyond the number of verts that currently exist
	// will be set to 0.  The face list will also be reset to match the original.

	Object* obj = node->EvalWorldState(0).obj;
	
	if (obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triobj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

		if (triobj)
		{
			Mesh& mesh = triobj->GetMesh();

			// Discard any verts contained beyond the number within the mesh
			mesh.setNumMapVerts(vWIBBLE_OFFSET_CHANNEL, mesh.numVerts, TRUE);
			mesh.setNumMapVerts(vWIBBLE_INDEX_CHANNEL, mesh.numVerts, TRUE);

			// Scan through the faces and reassign the face index data such that
			// it doesn't index vert data that no longer exists
			mesh.setNumMapFaces(vWIBBLE_OFFSET_CHANNEL, mesh.numFaces, TRUE);
			mesh.setNumMapFaces(vWIBBLE_INDEX_CHANNEL, mesh.numFaces, TRUE);

			TVFace* facesOffset = mesh.mapFaces(vWIBBLE_OFFSET_CHANNEL);
			TVFace* facesIndex  = mesh.mapFaces(vWIBBLE_INDEX_CHANNEL);

			if (facesOffset && facesIndex)
			{
				for(int i = 0; i < mesh.numFaces; i++)
				{
					for(int j = 0; j < 3; j++)
					{
						if (facesOffset[i].t[j] < 0 ||
							facesOffset[i].t[j] > mesh.numVerts)
						{
							facesOffset[i].t[j] = 0;
						}

						if (facesIndex[i].t[j] < 0 ||
							facesIndex[i].t[j] > mesh.numVerts)
						{
							facesIndex[i].t[j] = 0;
						}
					}
				}
			}
		}
	}
}

void WibbleRepair()
{ FUNC_ENTER("WibbleRepair"); 
	INodeTab nodes;
	GetAllNodes(nodes);

	for(int i = 0; i < nodes.Count(); i++)
		WibbleRepair(nodes[i]);

	MessageBox(gInterface->GetMAXHWnd(), "Wibble Repair (all) Completed.", "Wibble Repair", MB_ICONINFORMATION|MB_OK);
}

void WibbleRepairSel()
{ FUNC_ENTER("WibbleRepairSel"); 
	int nSelNodes = gInterface->GetSelNodeCount();

	if (nSelNodes == 0)
		return;

	INode* selNode = gInterface->GetSelNode(0);
	WibbleRepair(selNode);

	MessageBox(gInterface->GetMAXHWnd(), "Wibble Repair (sel) Completed.", "Wibble Repair", MB_ICONINFORMATION|MB_OK);
}

void AddTrackTest()
{ FUNC_ENTER("AddTrackTest"); 
	ITrackViewNode* tvn = gInterface->GetTrackViewRootNode();
	ITrackViewNode* tvnNewTrack = CreateITrackViewNode();

	tvn->AddNode(tvnNewTrack, "MyTrack(Muhahaha)", vNEXT_CLASS_ID);

	// Lets create ourselves a new float controller and attach it
	Control* c = (Control*)gInterface->CreateInstance(CTRL_FLOAT_CLASS_ID, Class_ID(LININTERP_FLOAT_CLASS_ID,0));
	SuspendAnimate();
	AnimateOn();
	
	float fkey = 0.0f;
	c->SetValue(0, &fkey);
	c->SetValue(100, &fkey);

	ResumeAnimate();

	if (c)
	{
		tvn->AddController(c, "MyFlontCont", vNEXT_CLASS_ID);
	}
}

void TrackDoTest()
{ FUNC_ENTER("TrackDoTest"); 

}

void FixVertexNormals()
{ FUNC_ENTER("FixVertexNormals"); 
	if (gInterface->GetSelNodeCount() != 1)
		return;

	INode* node = gInterface->GetSelNode(0);

	Object* obj = node->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(triObjectClassID))
	{
		TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);
		
		if (triObj)
		{
			Mesh& mesh = triObj->GetMesh();
			mesh.checkNormals(true);

			if (triObj != obj)
				triObj->DeleteThis();

			MessageBox(gInterface->GetMAXHWnd(), "FixVertexNormals completed OK!", "FixVertexNormals", MB_OK|MB_ICONINFORMATION);
		}
	}
}

void PurgeScriptCache(INode* node)
{ FUNC_ENTER("PurgeScriptCache"); 
	if (!node)
		node = gInterface->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child;
		child = node->GetChildNode(i);

		PurgeScriptCache(child);
	}

	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CRC);
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTBUFFER);
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_OUTPUTSCRIPTBUFFER);
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_SCENE_TERRAINLIST);
}

void PurgeNodeCache(INode* node)
{ FUNC_ENTER("PurgeNodeCache"); 
	if (!node)
		node = gInterface->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child;
		child = node->GetChildNode(i);

		PurgeNodeCache(child);
	}

	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CONFIGPROPS_ORIGCRC);
	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_NODE_CONFIGPROPS_CACHEBUFFER);
}

void GenerateDebugPos()
{ FUNC_ENTER("GenerateDebugPos"); 
	int nNodes = gInterface->GetSelNodeCount();

	FILE* fp = fopen("c:\\NodePos.txt", "a");

	Interval interval = gInterface->GetAnimRange();

	for(int i = 0; i < nNodes; i++)
	{
		INode* node = gInterface->GetSelNode(i);

		TimeValue start = interval.Start() / GetTicksPerFrame();
		TimeValue end   = interval.End() / GetTicksPerFrame();

		fprintf(fp, "Node: %s\n", (char*)node->GetName());

		for(float t = start; t < end; t++)
		{
			Matrix3 tm       = node->GetNodeTM(t * GetTicksPerFrame());
			Matrix3 parentTM;
			Point3  trans;
			Quat    q(tm);

			/*
			INode* nodeParent = node->GetParentNode();
			
			if (nodeParent)
			{
				Matrix3 parentTM = nodeParent->GetNodeTM(t * GetTicksPerFrame());
				tm = tm * Inverse(parentTM);
			}
			*/

			trans = tm.GetTrans();

			fprintf(fp, "time: %f  tx: %f ty: %f tz: %f qx: %f  qy: %f  qz: %f real: %f\n", t, trans.x, trans.y, trans.z,
				                                                                            q.x, q.y, q.z, q.w);
		}
	}

	fclose(fp);

	MessageBox(gInterface->GetMAXHWnd(), "Dumped nodepos.txt", "PosDump", MB_ICONINFORMATION|MB_OK);
}

void ClearPerBoneTolerances(INode* node)
{ FUNC_ENTER("ClearPerBoneTolerances"); 
	if (!node)
		node = gInterface->GetRootNode();

	int nKids = node->NumberOfChildren();

	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		ClearPerBoneTolerances(child);
	}

	node->RemoveAppDataChunk(vNEXT_CLASS_ID, GUP_CLASS_ID, vNAPP_PERBONE_TOLERANCES);
}

bool BadChannel(Mesh& mesh, int channel)
{ FUNC_ENTER("BadChannel"); 
	TVFace* faces = mesh.mapFaces(channel);
	int i, j;

	if (!faces)
		return true;

	for(i = 0; i < mesh.numFaces; i++)
	{
		for(j = 0; j < 3; j++)
		{
			if (faces[i].t[j] < 0 ||
				faces[i].t[j] >= mesh.numFaces)
				return true;
		}
	}

	return false;
}

// This function will recursively scan through the nodes in the scene checking to see if any of them can
// be converted to editable polys.  If it fails it will copy the relevant data for the node delete it,
// and reconstruct a new clean node in its place
INode* FixPolyConvNode(INode* node)
{ FUNC_ENTER("FixPolyConvNode"); 
	//bool rVal = false;
	INode* newNode = NULL;

	CStr nodeName = node->GetName();

	// Attempt to convert this node's object to an editable poly
	Object* obj = node->EvalWorldState(0).obj;

	if (obj && obj->CanConvertToType(polyObjectClassID))
	{
#ifndef FORCE_RECONSTRUCT
		PolyObject* polyobj;

		try
		{
			polyobj = (PolyObject*)obj->ConvertToType(0, polyObjectClassID);
		}
		catch(...)
#endif
		{
			// An error occurred converting this to an editable poly
			// Convert to tri object and duplicate
			TriObject* triObj = (TriObject*)obj->ConvertToType(0, triObjectClassID);

			if (triObj)
			{
				Mesh& mesh = triObj->GetMesh();

				// Manually duplicate the portions of the mesh that we care about into a new TriObject
				//TriObject* newTriObj = (TriObject*)gInterface->CreateInstance(0, triObjectClassID);
				TriObject* newTriObj = CreateNewTriObject();

				int i, j;

				// Copy verts
				int nVerts = mesh.getNumVerts();
				newTriObj->mesh.setNumVerts(nVerts);
		
				for(i = 0; i < nVerts; i++)
					newTriObj->mesh.setVert(i, mesh.getVert(i));
			
				// Copy faces
				int nFaces = mesh.getNumFaces();
				newTriObj->mesh.setNumFaces(nFaces);

				for(i = 0; i < nFaces; i++)
					newTriObj->mesh.faces[i] = mesh.faces[i];

				/*
				FILE* fp = fopen("c:\\TVDebug.txt", "w");

				// Copy texture verts
				int nTVerts = mesh.getNumTVerts();
				newTriObj->mesh.setNumTVerts(nTVerts);
				newTriObj->mesh.setNumTVFaces(nFaces);

				for(i = 0; i < nTVerts; i++)
				{
					fprintf(fp, "%i:  (%f, %f, %f)\n", i, mesh.tVerts[i].x,
					                                      mesh.tVerts[i].y,
														  mesh.tVerts[i].z);

					//newTriObj->mesh.setTVert(i, mesh.getTVert(i));
					UVVert uvvert = mesh.getTVert(i);
					newTriObj->mesh.setTVert(i, uvvert);
				}

				// Copy texture faces (Number of TVerts should be same as numFaces)
				for(i = 0; i < nFaces; i++)
				{
					fprintf(fp, "%i:  t0: %i  t1: %i  t2: %i\n", i, mesh.tvFace[i].t[0],
					                                                mesh.tvFace[i].t[1],
																	mesh.tvFace[i].t[2]);

					newTriObj->mesh.tvFace[i] = mesh.tvFace[i];
				}

				fclose(fp);
				*/

				// Copy vertex color verts
				/*
				int nVertColors = mesh.getNumVertCol();
				newTriObj->mesh.setNumVertCol(nVertColors);

				for(i = 0; i < nVertColors; i++)
					newTriObj->mesh.vertCol[i] = mesh.vertCol[i];

				// Copy vertex face verts
				newTriObj->mesh.setNumVCFaces(nFaces);

				for(i = 0; i < nFaces; i++)
					newTriObj->mesh.vcFace[i] = mesh.vcFace[i];
				*/

				// Copy the mapping channels
				int nChannels = mesh.getNumMaps();			
				//int nChannels = 0;
				//newTriObj->mesh.setNumMaps(nChannels);

				for(i = -NUM_HIDDENMAPS; i < nChannels; i++)
				{
					if (!mesh.mapSupport(i) || BadChannel(mesh, i))
						continue;

					newTriObj->mesh.setMapSupport(i);

					// Copy map channel verts
					int nMapVerts = mesh.getNumMapVerts(i);
					newTriObj->mesh.setNumMapVerts(i, nMapVerts);
					
					UVVert* destVerts = newTriObj->mesh.mapVerts(i);
					UVVert* srcVerts  = mesh.mapVerts(i);

					for(j = 0; j < nMapVerts; j++)
						destVerts[j] = srcVerts[j];

					// Copy map channel faces
					newTriObj->mesh.setNumMapFaces(i, nFaces);

					TVFace* destFaces = newTriObj->mesh.mapFaces(i);
					TVFace* srcFaces  = mesh.mapFaces(i);

					for(j = 0; j < nFaces; j++)
					{
						destFaces[j].t[0] = srcFaces[j].t[0];
						destFaces[j].t[1] = srcFaces[j].t[1];
						destFaces[j].t[2] = srcFaces[j].t[2];
					}
				}

				// Copy face data channel
				IFaceDataMgr* pFDMgr    = static_cast<IFaceDataMgr*>(mesh.GetInterface( FACEDATAMGR_INTERFACE ));
				IFaceDataMgr* pFDMgrNew = static_cast<IFaceDataMgr*>(newTriObj->mesh.GetInterface( FACEDATAMGR_INTERFACE ));
				
				if (pFDMgr && pFDMgrNew)
				{
					FaceFlagsData* fdc = dynamic_cast<FaceFlagsData*>( pFDMgr->GetFaceDataChan( vFACE_FLAGS_CHANNEL_ID ));
					FaceFlagsData* fdcNew;
					
					if( fdc ) 
					{
						fdcNew = new FaceFlagsData();
						pFDMgrNew->AddFaceDataChan( fdcNew );
						fdcNew->FacesCreated( 0, nFaces );

						// Now copy the flags from the old mesh to the new
						for(j = 0; j < nFaces; j++)
							fdcNew->m_data[j] = fdc->m_data[j];
					}

					// Copy CAS data channel
					CASFaceFlagsData* fdcCAS    = dynamic_cast<CASFaceFlagsData*>( pFDMgr->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));
					CASFaceFlagsData* fdcNewCAS = dynamic_cast<CASFaceFlagsData*>( pFDMgrNew->GetFaceDataChan( vCASFACE_FLAGS_CHANNEL_ID ));

					if (fdcCAS)
					{
						fdcNewCAS = new CASFaceFlagsData();
						pFDMgrNew->AddFaceDataChan( fdcNewCAS );
						fdcNewCAS->FacesCreated( 0, nFaces );

						// Now copy the flags from the old mesh to the new
						for(j = 0; j < nFaces; j++)
							fdcNewCAS->m_data[j] = fdcCAS->m_data[j];
					}
				}

				// Create a new node in the scene for this object
				newNode = gInterface->CreateObjectNode(newTriObj);
				newNode->SetName(node->GetName());

				Matrix3 tm = node->GetNodeTM(0);
				newNode->SetNodeTM(0, tm);

				// Copy offset positions
				newNode->SetObjOffsetPos(node->GetObjOffsetPos());
				newNode->SetObjOffsetRot(node->GetObjOffsetRot());
				newNode->SetObjOffsetScale(node->GetObjOffsetScale());
				
				// Copy wireframe color
				newNode->SetWireColor(node->GetWireColor());

				// Copy property buffer
				CStr propBuffer;
				node->GetUserPropBuffer(propBuffer);
				newNode->SetUserPropBuffer(propBuffer);

				// Assign material
				newNode->SetMtl(node->GetMtl());

				// Link to the children of the former node
				node->Delete(0, TRUE);
			}

			//rVal = true;
		}

#ifndef FORCE_RECONSTRUCT
		if (polyobj && polyobj != obj)
			polyobj->DeleteThis();
#endif

	}

	return newNode;
}

bool NodeExistsInList(INodeTab& list, INode* node)
{ FUNC_ENTER("NodeExistsInList"); 
	for(int i = 0; i < list.Count(); i++)
	{
		if (list[i] == node)
			return true;
	}

	return false;
}

void FixPolyConvProblem(INode* node, int* nFoundObjs)
{ FUNC_ENTER("FixPolyConvProblem"); 
	/*
	if (!node)
		node = gInterface->GetRootNode();

	int nKids = node->NumberOfChildren();
	
	for(int i = 0; i < nKids; i++)
	{
		INode* child = node->GetChildNode(i);
		
		if (FixPolyConvNode(child))
			if (nFoundObjs)
				(*nFoundObjs)++;
	}
	*/

	int nNodes = gInterface->GetSelNodeCount();
	int i;

	INodeTab procNodes;

	for(i = 0; i < nNodes; i++)
	{
		INode* selNode = gInterface->GetSelNode(i);

		if (!NodeExistsInList(procNodes, selNode))
		{
			if (FixPolyConvNode(selNode))
				if (nFoundObjs)
					(*nFoundObjs)++;

			// The selection set may have changed at this point
			// The only safe way to be sure is to start back at index 0
			// We'll skip nodes that we've already processed
			procNodes.Append(1, &selNode);
			nNodes = gInterface->GetSelNodeCount();
			i = -1;
		}
	}
}
