/*
	DynUILink.cpp
	Functions to provide property list dynamic UI info to MAXScript
	for automated parameter based script generation
	4-29-01  aml
*/

#include "../NExt/NExt.h"
#include "../NExt/appdata.h"
#include "../NExt/FaceFlags/FaceFlags.h"					// cross dependent with NExt.gup
#include "../NExt/Material/NExtMat.h"
#include "../NExt/Material/NExtMultiMat.h"
#include "../NExt/Material/Multi.h"
#include "../NExt/misc/llist.h"
#include "../NExt/PropEdit/ParseFuncs.h"
#include "../NExt/PropEdit/SharedData.h"
#include "../NExt/Texture/NExtTexture.h"
#include "../NExt/Texture/BMTex.h"

extern CRITICAL_SECTION  gCriticalSection;

#define ip   MAXScript_interface

#include "MAXScrpt/MAXScrpt.h"
#include "MAXScrpt/definsfn.h"

def_visible_primitive(NSgetPos, "nsGetPos");				  // Converts MAX position to a game position
def_visible_primitive(NSgetRot, "nsGetRot");				  // Converts MAX rotation to a game rotation
def_visible_primitive(NSgetNode, "nsGetNode");				  // Retrieves the MAX node being exported (or manipulated)
def_visible_primitive(NSclearScripts, "nsClearScripts");	  // Clears all script buffers
def_visible_primitive(NSget,"NSget");						  // Returns a parameter value given it's name
def_visible_primitive(NSgetColor,"NSgetColor");				  // Retrieves color value stored within colorpopup array
def_visible_primitive(NSgetColorSize,"NSgetColorSize");		  // Retrieves the number of colors within a color array
def_visible_primitive(NSset,"NSset");						  // Updates a PE parameter value
def_visible_primitive(NSgetType,"NSgetType");				  // Returns the type of a parameter value
def_visible_primitive(NSout,"NSout");						  // Outputs a line to the dynUI script buffer
def_visible_primitive(NSoutNCR,"NSoutncr");					  // Outputs a line to the dynUI script buffer w/o a CR
def_visible_primitive(NSoutOnce,"NSoutOnce");				  // Outputs a line to the script buffer only if it doesn't already exist
def_visible_primitive(NSoutOnceNCR,"NSoutOnceNCR");			  // Outputs a line to the script buffer only if it doesn't already exist w/o a CR
def_visible_primitive(NSopen,"NSopen");						  // Signals plugin that template buffer should be started
def_visible_primitive(NSclose,"NSclose");					  // Signals plugin that template buffer is completed
															  // (ready for preprocessing and network load request)
def_visible_primitive(NSConvVal,"NSConvVal");				  // Converts floating point value to non-exponential form
def_visible_primitive(NSGetLevel,"NSGetLevel");				  // Retrieves the level name being exported
def_visible_primitive(NSGetTexName,"NSGetTexName");			  // Retrieves the name of the texture filename on the given material
def_visible_primitive(NSGetEmitScript,"NSGetEmitScript");	  // Retrieves the emitscript name for SFP Nodes
def_visible_primitive(NSGetFixedAlpha,"NSGetFixedAlpha");	  // Retrieves the fixed alpha value from the material
def_visible_primitive(NSGetMaxParticles,"NSGetMaxParticles"); // Retrieves the number of particles containing this texture and type combination
def_visible_primitive(NSReplace,"NSReplace");				  // Substring replacement function from ParseFuncs
def_visible_primitive(NSGetBlendMode,"NSGetBlendMode");		  // Retrieves a string indicating the blend mode of a NExtMaterial pass
def_visible_primitive(NSGetWidth,"NSGetWidth");				  // Retrieves the width of a node's geometry
def_visible_primitive(NSGetHeight,"NSGetHeight");			  // Retrieves the height of a nodes' geometry
def_visible_primitive(NSGetNExtMtl,"NSGetNExtMtl");           // Retrieves the first NExtMaterial used on a given node (even if sub)
def_visible_primitive(NSGetUsePS2MipsNGC,"NSGetUsePS2MipsNGC");	// Retrieves the Use PS2 mips field from a next material for NGC
def_visible_primitive(NSGetUsePS2MipsXBOX,"NSGetUsePS2MipsXBOX");	// Retrieves the Use PS2 mips field from a next material for XBOX
def_visible_primitive(NSPutUsePS2MipsNGC,"NSPutUsePS2MipsNGC");	// Assigns the Use PS2 mips field from a next material for NGC
def_visible_primitive(NSPutUsePS2MipsXBOX,"NSPutUsePS2MipsXBOX");	// Assigns the Use PS2 mips field from a next material for XBOX

#include "../NExt/PropEdit/PreProcessor.h"
#include "MAXScrpt/maxobj.h"
#include "MAXScrpt/Numbers.h"
#include "MAXScrpt/Colorval.h"
#include "MAXScrpt/maxmats.h"
#include "iparamb2.h"

/*
struct ScriptBuf
{
	CStr scriptName;
	CStr buffer;

	bool operator ==(ScriptBuf& right)
	{
		if (scriptName==right.scriptName)
			return true;

		return false;
	}
};
*/

struct StoredValue
{
	CStr  name;

	union
	{
		float tFl;
	};
	//Value val;
};

static CStr      dynScriptBuf;					// This is the script attached to the object(s) in the PE
												// It will also be reevaluated per object at export time

static LinkList<ScriptBuf> bufList;				// This is the list of script buffers MAXScript has control of
												// These are interpreted by the ScriptExporter at export time
												// and are used for adding additional scripts and cumulative
												// data to those scripts.  (such as, LoadTextures, LoadCameras, etc)

static Link<ScriptBuf>*    curbuf;				// The active buffer to modify, NULL if dynScriptBuf

HANDLE ghMemSize=NULL;							// Memory mapping containing the size of the script buffer mem map
int*   gpMemSize=NULL;							// Pointer to memory map size data (windowed)
HANDLE ghMemBuf=NULL;							// Memory mapping containing the script buffers
char*  gpMemBuf=NULL;							// Pointer to memory map script buffer data (windowed)
HANDLE ghMemStdSize=NULL;						// Memory mapping containing the size of the script buffer for the attached obj
int*   gpMemStdSize=NULL;						// Pointer to memory map data containing the size of the buffer for the attached obj
HANDLE ghMemStdBuf=NULL;						// Memory mapping containing the buffer attached to the object
char*  gpMemStdBuf=NULL;						// Pointer to the memory map data containing the buffer for the attached obj

void SaveScriptBuffers()
{
	if (gpMemSize)
		UnmapViewOfFile(gpMemSize);

	if (ghMemSize)
		CloseHandle(ghMemSize);

	// Save the script buffers to shared memory
	ghMemSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                          NULL,
								  PAGE_READWRITE,
								  0,
								  sizeof(int),
								  "NExtDLXBufSize");

	if (!ghMemSize)
		return;

	gpMemSize = (int*)MapViewOfFile(ghMemSize,
		                            FILE_MAP_ALL_ACCESS,
							        0,0,0);

	if (!gpMemSize)
	{
		CloseHandle(ghMemSize);
		ghMemSize=NULL;
		return;
	}

	// Compute the size of the chunk
	int count = bufList.GetSize();
	int size  = sizeof(int);		// num entries

	for(int i=0;i<count;i++)
	{
		size+=sizeof(int);						// Size of script name
		size+=bufList[i].scriptName.Length()+1;

		size+=sizeof(int);						// Size of buffer
		size+=bufList[i].buffer.Length()+1;
	}

	*gpMemSize = size;

	if (gpMemBuf)
		UnmapViewOfFile(gpMemBuf);

	if (ghMemBuf)
		CloseHandle(ghMemBuf);

	// Allocate the buffer and fill it with data
	ghMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                         NULL,
								 PAGE_READWRITE,
								 0,
								 size,
								 "NExtDLXBuf");

	if (!ghMemBuf)
	{
		UnmapViewOfFile(gpMemSize);
		CloseHandle(ghMemSize);
		gpMemSize=NULL;
		ghMemSize=NULL;
		return;
	}

	char* gpMemBuf = (char*)MapViewOfFile(ghMemBuf,
		                                  FILE_MAP_ALL_ACCESS,
									      0,0,0);

	if (!gpMemBuf)
	{
		UnmapViewOfFile(gpMemSize);
		CloseHandle(ghMemSize);
		CloseHandle(ghMemBuf);
		gpMemSize=NULL;
		ghMemSize=NULL;
		ghMemBuf=NULL;
		return;
	}

	char* pos = gpMemBuf;

	*((int*)pos) = count;		// Write num entries
	pos+=sizeof(int);

	for(i=0;i<count;i++)
	{
		*((int*)pos) = bufList[i].scriptName.Length()+1;	// Write size of script name
		pos+=sizeof(int);									// Advance
		strcpy(pos,bufList[i].scriptName);					// Copy script name into buffer
		pos+=bufList[i].scriptName.Length()+1;				// Advance

		*((int*)pos) = bufList[i].buffer.Length()+1;		// Write size of buffer
		pos+=sizeof(int);									// Advance
		strcpy(pos,bufList[i].buffer);						// Copy script buffer into new allocated buffer
		pos+=bufList[i].buffer.Length()+1;					// Advance
	}

	// Assign the data chunk
	//root->AddAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID, vNAPP_SE_MAXSCRIPT_INPUT_ID, size, buf);
}

void LoadScriptBuffers()
{
	// Retrieve script buffers

	// Acquire the script buffer data from shared memory
	HANDLE hMemPropDataSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                        NULL,
												PAGE_READONLY,
												0,
												sizeof(int),
												"NExtDLXBufSize");

	if (!hMemPropDataSize)
		return;

	int* size = (int*)MapViewOfFile(hMemPropDataSize,
		                            FILE_MAP_READ,
							        0,0,0);

	if (!size)
	{
		CloseHandle(hMemPropDataSize);
		return;
	}

	HANDLE hMemPropData = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                    NULL,
											PAGE_READONLY,
											0,
											*size,
											"NExtDLXBuf");

	if (!hMemPropData)
	{
		UnmapViewOfFile(size);
		CloseHandle(hMemPropDataSize);
		return;
	}

	char* origpos = (char*)MapViewOfFile(hMemPropData,
		                                 FILE_MAP_READ,
									     0,0,0);

	char* pos = origpos;

	if (pos)
	{
		int count = *((int*)pos);
		pos+=sizeof(int);

		bufList.Clear();

		for(int i=0;i<count;i++)
		{
			ScriptBuf newScript;

			// Read in the script name
			int namelen = *((int*)pos);
			pos+=sizeof(int);

			char* strChr = new char[namelen];
			strcpy(strChr,pos);
			newScript.scriptName = strChr;
			delete strChr;
			pos+=namelen;

			// Read in the script buffer
			int buflen = *((int*)pos);
			pos+=sizeof(int);

			strChr = new char[buflen];
			strcpy(strChr,pos);
			newScript.buffer = strChr;
			delete strChr;
			pos+=buflen;

			bufList.AddToTail(&newScript);
		}

		UnmapViewOfFile(origpos);
	}

	UnmapViewOfFile(size);
	CloseHandle(hMemPropDataSize);
	CloseHandle(hMemPropData);
}

Value* NSgetPos_cf(Value **arg_list, int count)
{
	check_arg_count(NSgetPos, 1, count);
	type_check(arg_list[0], MAXNode, "< First parameter should be a Node >");

	INode* node = arg_list[0]->to_node();
	Matrix3 mat;
	Point3  pos;
	float   tmp;

	mat=node->GetNodeTM(0);		// Get node's transform matrix at time 0

	// Get node position in game coordinate system
	pos=mat.GetTrans();
	tmp=pos.z;
	//pos.z=pos.y;
	pos.z= -pos.y;
	pos.y= tmp;

	return new Point3Value(pos);
}

Value* NSgetRot_cf(Value **arg_list, int count)
{
	float   rot[3];				// Euler rotation of node
	Matrix3 mat;

	check_arg_count(NSgetPos, 1, count);
	type_check(arg_list[0], MAXNode, "< First parameter should be a Node >");

	INode* node = arg_list[0]->to_node();

	mat=node->GetNodeTM(0);		// Get node's transform matrix at time 0

// From NExtScriptMan::GetRotation (skate3)
#if 0
	Interval valid = FOREVER;
	Matrix3 tmat( 1 );

	Control *c = node->GetTMController();
	tmat = n->GetParentTM( 0 );
	c->GetValue( 0, &tmat, valid, CTRL_ABSOLUTE);

	Quat q(tmat);
	QuatToEuler( q, rot );	
#else
	//Interval valid = FOREVER;
	//
	//Quat q(node->GetNodeTM( 0 ));
	//QuatToEuler( q, rot );	

	Matrix3 tm = node->GetNodeTM(0);
	Quat q(tm);
	QuatToEuler( q, rot, EULERTYPE_XYZ );	

	// Switch rotations to game coord system
	// Max:   X: right Y: depth  Z: up
	// Game:  X: right Y: up     Z: -depth

	// Ordering: XYZ in MAX is XZY in game:
	// X is same, so same order
	// Y, Z is swaped so, order is swapped XZY
	Matrix3 tm2;
	tm2.IdentityMatrix();
	tm2.RotateX(rot[0]);
	tm2.RotateZ(-rot[1]);
	tm2.RotateY(rot[2]);

	Quat q2(tm2);
	QuatToEuler(q2, rot, EULERTYPE_XYZ);
#endif

	return new Point3Value(rot);
}

Value* NSgetNode_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);

	HANDLE hMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                               NULL,
									   PAGE_READONLY,
									   0,
									   sizeof(INode*),
									   "NExtGUPCurrentNode");

	if (!hMemBuf)
	{
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	INode** node = (INode**)MapViewOfFile(hMemBuf,
		                                  FILE_MAP_READ,
										  0,0,0);

	if (!node)
	{
		CloseHandle(hMemBuf);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	INode* nd = *node;

	UnmapViewOfFile(node);
	CloseHandle(hMemBuf);

	LeaveCriticalSection(&gCriticalSection);

	//CStr name = nd->GetName();

	return new MAXNode(nd);
}

Value* NSclearScripts_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	bufList.Clear();

	//INode* root = gInterface->GetRootNode();
	//root->RemoveAppDataChunk(vNEXT_CLASS_ID,GUP_CLASS_ID,vNAPP_SE_MAXSCRIPT_INPUT_ID);

	// Force file to contain 0 for mem size on reset
	if (gpMemSize)
		*gpMemSize = 0;

	if (gpMemSize)
	{
		UnmapViewOfFile(gpMemSize);
		gpMemSize=NULL;
	}

	if (ghMemSize)
	{
		CloseHandle(ghMemSize);
		ghMemSize=NULL;
	}

	if (gpMemBuf)
	{
		UnmapViewOfFile(gpMemBuf);
		gpMemBuf=NULL;
	}

	if (ghMemBuf)
	{
		CloseHandle(ghMemBuf);
		ghMemBuf=NULL;
	}

	LeaveCriticalSection(&gCriticalSection);

	return &ok;
}

Value* NSopen_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);

	if (count==0)
	{
		dynScriptBuf="";
		curbuf = NULL;
		LeaveCriticalSection(&gCriticalSection);
		return &ok;
	}
	else
	{
		check_arg_count(NSopen, 1, count);
		type_check(arg_list[0], String, "< First arg should name script to create/expand in .qn file >");

		LoadScriptBuffers();

		char* scriptName = arg_list[0]->to_string();
	
		ScriptBuf sbuf;
		sbuf.scriptName = scriptName;
		
		curbuf = bufList.Find(&sbuf);

		if (curbuf == NULL)
		{
			// The script doesn't exist, create it
			ScriptBuf newScriptBuf;
			newScriptBuf.scriptName = scriptName;
			newScriptBuf.buffer = "";
			curbuf = bufList.AddToTail(&newScriptBuf);

			LeaveCriticalSection(&gCriticalSection);
			return &false_value;
		}
	}

	LeaveCriticalSection(&gCriticalSection);
	return &true_value;
}

Value* NSclose_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);

	if (!curbuf)
	{
		if (gpMemStdSize)
			UnmapViewOfFile(gpMemStdSize);

		if (ghMemStdSize)
			CloseHandle(ghMemStdSize);

		ghMemStdSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
			                             NULL,
										 PAGE_READWRITE,
										 0,
										 sizeof(int),
										 "NExtDLXScriptBufSize");

		if (!ghMemStdSize)
		{
			LeaveCriticalSection(&gCriticalSection);
			return &undefined;
		}

		gpMemStdSize = (int*)MapViewOfFile(ghMemStdSize,
			                               FILE_MAP_ALL_ACCESS,
										   0,0,0);

		if (!gpMemStdSize)
		{
			CloseHandle(ghMemStdSize);
			ghMemStdSize=NULL;
			LeaveCriticalSection(&gCriticalSection);
			return &undefined;		
		}

		// Assign size of buffer to shared memory
		*gpMemStdSize = dynScriptBuf.Length()+1;

		if (gpMemStdBuf)
			UnmapViewOfFile(gpMemStdBuf);

		if (ghMemStdBuf)
			CloseHandle(ghMemStdBuf);

		ghMemStdBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
			                            NULL,
										PAGE_READWRITE,
										0,
										*gpMemStdSize,
										"NExtDLXScriptBuf");

		if (!ghMemStdBuf)
		{
			UnmapViewOfFile(gpMemStdSize);
			CloseHandle(ghMemStdSize);
			gpMemStdSize=NULL;
			ghMemStdSize=NULL;
			LeaveCriticalSection(&gCriticalSection);
			return &undefined;
		}

		gpMemStdBuf = (char*)MapViewOfFile(ghMemStdBuf,
			                               FILE_MAP_ALL_ACCESS,
										   0,0,0);

		if (!gpMemStdBuf)
		{
			UnmapViewOfFile(gpMemStdSize);
			CloseHandle(ghMemStdSize);
			CloseHandle(ghMemStdBuf);
			gpMemStdSize=NULL;
			ghMemStdSize=NULL;
			ghMemStdBuf=NULL;
			LeaveCriticalSection(&gCriticalSection);
			return &undefined;
		}

		// Copy the buffer into shared memory
		strcpy(gpMemStdBuf,(char*)dynScriptBuf);
	}
	else
	{
		SaveScriptBuffers();
	}

	LeaveCriticalSection(&gCriticalSection);

	return &ok;
}

Value* NSgetType_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSgetType, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string identifying type to retrieve >");

	char* propname = arg_list[0]->to_string();

	// Retrieve the property data from shared memory
	HANDLE hMemBufSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                   NULL,
										   PAGE_READONLY,
										   0,
										   sizeof(int),
										   "NExtGUPPropDataSize");

	if (!hMemBufSize)
	{
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int* pMemBufSize = (int*)MapViewOfFile(hMemBufSize,
		                                   FILE_MAP_READ,
										   0,0,0);

	if (!pMemBufSize)
	{
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	HANDLE hMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                               NULL,
									   PAGE_READONLY,
									   0,
									   *pMemBufSize,
									   "NExtGUPPropertyData");

	if (!hMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	char* pMemBuf = (char*)MapViewOfFile(hMemBuf,
		                                 FILE_MAP_READ,
										 0,0,0);

	if (!pMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		CloseHandle(hMemBuf);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int   num = *((int*)pMemBuf);
	char* pos = ((char*)pMemBuf)+sizeof(int);

	PropertyData* pdata;
	
	for(int i=0;i<num;i++)
	{
		pdata = (PropertyData*)pos;

		if (strcmp(pdata->name,propname)==0)
		{
			PropType type = pdata->type;

			UnmapViewOfFile(pMemBufSize);
			UnmapViewOfFile(pMemBuf);
			CloseHandle(hMemBufSize);
			CloseHandle(hMemBuf);

			LeaveCriticalSection(&gCriticalSection);
			return new Integer(type);
		}

		pos+=sizeof(PropertyData);
	}

	UnmapViewOfFile(pMemBufSize);
	UnmapViewOfFile(pMemBuf);
	CloseHandle(hMemBufSize);
	CloseHandle(hMemBuf);

	LeaveCriticalSection(&gCriticalSection);

	return &undefined;
}

CStr ReplaceTags(CStr str)
{
	EnterCriticalSection(&gCriticalSection);

	// Retrieve the property data from shared memory
	HANDLE hMemBufSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                   NULL,
										   PAGE_READONLY,
										   0,
										   sizeof(int),
										   "NExtGUPPropDataSize");

	if (!hMemBufSize)
	{
		LeaveCriticalSection(&gCriticalSection);
		return str;
	}

	int* pMemBufSize = (int*)MapViewOfFile(hMemBufSize,
		                                   FILE_MAP_READ,
										   0,0,0);

	if (!pMemBufSize)
	{
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return str;
	}

	HANDLE hMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                               NULL,
									   PAGE_READONLY,
									   0,
									   *pMemBufSize,
									   "NExtGUPPropertyData");

	if (!hMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return str;
	}

	char* pMemBuf = (char*)MapViewOfFile(hMemBuf,
		                                 FILE_MAP_READ,
										 0,0,0);

	if (!pMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		CloseHandle(hMemBuf);
		LeaveCriticalSection(&gCriticalSection);
		return str;
	}

	int   num = *((int*)pMemBuf);
	char* pos = ((char*)pMemBuf)+sizeof(int);

	PropertyData* pdata;
	
	CStr buf = str;
	CStr tag;

	for(int i=0;i<num;i++)
	{
		pdata = (PropertyData*)pos;

		char* name = strrchr(pdata->name,'/');

		if (name)
			name++;
		else
			name = pdata->name;

		tag = CStr("<")+CStr(name)+CStr(">");
		buf = ReplaceStr(buf,tag,pdata->value);

		pos+=sizeof(PropertyData);
	}

	UnmapViewOfFile(pMemBuf);
	UnmapViewOfFile(pMemBufSize);
	CloseHandle(hMemBuf);
	CloseHandle(hMemBufSize);

	LeaveCriticalSection(&gCriticalSection);
	return buf;
}

Value* NSset_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSset, 2, count);
	type_check(arg_list[0], String, "< First parameter should be string identifying name of property to set >");

	TSTR propname  = arg_list[0]->to_string();
	TSTR propvalue;

	// The second argument may not necessarily be able to convert to a string, we need to detect
	// if its not a string and if that's the case manually convert it into one for the proplist
	     if (arg_list[1]->tag == class_tag(String))
		 {
			 propvalue = arg_list[1]->to_string();
		 }
	else if (arg_list[1]->tag == class_tag(Float))
		{
			char buf[256];
			sprintf(buf,"%f",arg_list[1]->to_float());
			propvalue = buf;
		}
	else if (arg_list[1]->tag == class_tag(Integer))
		{
			char buf[256];
			sprintf(buf,"%i",arg_list[1]->to_int());
			propvalue = buf;
		}
	else if (arg_list[1]->tag == class_tag(ColorValue))
		{
			char buf[256];
			Color color = dynamic_cast<ColorValue*>(arg_list[1])->to_color();
			sprintf(buf,"[%i,%i,%i]",color.r,color.g,color.b);
			propvalue = buf;
		}

	// Retrieve the property data from shared memory
	HANDLE hMemBufSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                   NULL,
										   PAGE_READONLY,
										   0,
										   sizeof(int),
										   "NExtGUPPropDataSize");

	if (!hMemBufSize)
	{
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int* pMemBufSize = (int*)MapViewOfFile(hMemBufSize,
		                                   FILE_MAP_READ,
										   0,0,0);

	if (!pMemBufSize)
	{
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	HANDLE hMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                               NULL,
									   PAGE_READWRITE,
									   0,
									   *pMemBufSize,
									   "NExtGUPPropertyData");

	if (!hMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	char* pMemBuf = (char*)MapViewOfFile(hMemBuf,
		                                 FILE_MAP_ALL_ACCESS,
										 0,0,0);

	if (!pMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		CloseHandle(hMemBuf);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int   num = *((int*)pMemBuf);
	char* pos = ((char*)pMemBuf)+sizeof(int);

	PropertyData* pdata;
	
	for(int i=0;i<num;i++)
	{
		pdata = (PropertyData*)pos;

		char* name = strrchr(pdata->name,'/');
		if (name)
			name++;
		else
			name = pdata->name;

		if (strcmp(name,propname)==0)
		{
			strcpy(pdata->value,propvalue);
		}

		pos+=sizeof(PropertyData);
	}

	UnmapViewOfFile(pMemBuf);
	UnmapViewOfFile(pMemBufSize);
	CloseHandle(hMemBuf);
	CloseHandle(hMemBufSize);

	LeaveCriticalSection(&gCriticalSection);
	return &undefined;
}

Value* NSgetColor_cf(Value **arg_list, int count)
{
	check_arg_count(NSgetColor, 2, count);
	type_check(arg_list[0], String, "< First parameter should be string identifying value to retrieve >");
	type_check(arg_list[1], Integer, "< Second parameter should be integer specifying color array index >");
	Value* val = NSget_cf(arg_list,1);
	CStr str = val->to_string();

	// Parse the returned value
	int iColor = arg_list[1]->to_int();
	int nColors = CountChar(str,'@');

	if (iColor >= nColors || iColor < 0)
	{
		throw RuntimeError("< Acquired color that doesn't exist >");
		return &undefined;
	}

	char srcBuf[256];
	char destBuf[256];
	char colorBuf[256];
	char alphaBuf[256];

	strcpy(srcBuf,str);

	GetOption(srcBuf,destBuf,'/',iColor);
	GetOption(destBuf,colorBuf,'@',0);
	GetOption(destBuf,alphaBuf,'@',1);

	int r,g,b,a;
	sscanf(colorBuf,"[%i,%i,%i]",&r,&g,&b);
	a = atoi(alphaBuf);

	float fr,fg,fb,fa;
	fr = (float)r / 255.0f;
	fg = (float)g / 255.0f;
	fb = (float)b / 255.0f;
	fa = (float)a / 255.0f;

	return new ColorValue(fr,fg,fb,fa);
}

Value* NSgetColorSize_cf(Value **arg_list, int count)
{
	check_arg_count(NSgetColor, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string identifying value to retrieve >");
	Value* val = NSget_cf(arg_list,1);
	CStr str = val->to_string();

	int nColors = CountChar(str,'@');

	return new Integer(nColors);
}

Value* NSget_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSget, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string identifying value to retrieve >");
	
	char* propname = arg_list[0]->to_string();

	// Retrieve the property data from shared memory
	HANDLE hMemBufSize = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                   NULL,
										   PAGE_READONLY,
										   0,
										   sizeof(int),
										   "NExtGUPPropDataSize");

	if (!hMemBufSize)
	{
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int* pMemBufSize = (int*)MapViewOfFile(hMemBufSize,
		                                   FILE_MAP_READ,
										   0,0,0);

	if (!pMemBufSize)
	{
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	HANDLE hMemBuf = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                               NULL,
									   PAGE_READONLY,
									   0,
									   *pMemBufSize,
									   "NExtGUPPropertyData");

	if (!hMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	char* pMemBuf = (char*)MapViewOfFile(hMemBuf,
		                                 FILE_MAP_READ,
										 0,0,0);

	if (!pMemBuf)
	{
		UnmapViewOfFile(pMemBufSize);
		CloseHandle(hMemBufSize);
		CloseHandle(hMemBuf);
		LeaveCriticalSection(&gCriticalSection);
		return &undefined;
	}

	int   num = *((int*)pMemBuf);
	char* pos = ((char*)pMemBuf)+sizeof(int);

	PropertyData* pdata;
	
	for(int i=0;i<num;i++)
	{
		pdata = (PropertyData*)pos;

		char* name = strrchr(pdata->name,'/');
		if (name)
			name++;
		else
			name = pdata->name;

		if (strcmp(name,propname)==0)
		{
			switch(pdata->type)
			{
			case PROPTYPE_UNDEFINED:
			case PROPTYPE_EDIT:
			case PROPTYPE_LIST:
			case PROPTYPE_EXTLIST:
			case PROPTYPE_SCRIPT:
			case PROPTYPE_FILE:
			case PROPTYPE_STATIC:
			case PROPTYPE_MULTI:
			case PROPTYPE_FLAGS:
			case PROPTYPE_MTL:
			case PROPTYPE_COLORPOPUP:
				{
					char buf[256];
					strcpy(buf,pdata->value);

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return new String(buf);
				}

			case PROPTYPE_NODE:
				{
					char buf[256];
					strcpy(buf,pdata->value);

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return new MAXNode(ip->GetINodeByName(buf));
				}

			case PROPTYPE_SLIDER:
			case PROPTYPE_SLIDERNUM:
				{
					float val = atof(pdata->value);

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return new Float(val);
				}

			case PROPTYPE_COLOR:
				{
					int   r,g,b;
					float fr,fg,fb;
					sscanf(pdata->value,"[%i,%i,%i]",&r,&g,&b);
					fr = (float)r/255.0f;
					fg = (float)g/255.0f;
					fb = (float)b/255.0f;

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return new ColorValue(fr,fg,fb);
				}

			case PROPTYPE_CHECK:
				{
					if (strcmp(pdata->value,"TRUE")==0)
					{
						UnmapViewOfFile(pMemBuf);
						UnmapViewOfFile(pMemBufSize);
						CloseHandle(hMemBuf);
						CloseHandle(hMemBufSize);

						LeaveCriticalSection(&gCriticalSection);
						return &true_value;
					}

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return &false_value;
				}

			case PROPTYPE_SPINEDIT:
				{
					float fval;
					sscanf(pdata->value,"%f",&fval);

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);
					
					LeaveCriticalSection(&gCriticalSection);
					return new Float(fval);
				}

			default:
				{
					char buf[256];
					strcpy(buf,pdata->value);

					UnmapViewOfFile(pMemBuf);
					UnmapViewOfFile(pMemBufSize);
					CloseHandle(hMemBuf);
					CloseHandle(hMemBufSize);

					LeaveCriticalSection(&gCriticalSection);
					return new String(buf);
				}
			}
		}

		pos+=sizeof(PropertyData);
	}

	UnmapViewOfFile(pMemBuf);
	UnmapViewOfFile(pMemBufSize);
	CloseHandle(hMemBuf);
	CloseHandle(hMemBufSize);

	LeaveCriticalSection(&gCriticalSection);
	return &undefined;
}

Value* NSout_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSgetFaceMtl, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string to add to output buffer >");

	char  eol[3];
	eol[0] = 13;
	eol[1] = 10;
	eol[2] = 0;

	char* bufline = arg_list[0]->to_string();

	if (!curbuf)
		dynScriptBuf+=CStr(bufline)+eol;
	else
	{
		CStr buf = ReplaceTags(bufline);
		curbuf->data.buffer+=buf+eol;
	}

	LeaveCriticalSection(&gCriticalSection);
	return &ok;
}

Value* NSoutOnce_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSgetFaceMtl, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string to add to output buffer >");

	char  eol[3];
	eol[0] = 13;
	eol[1] = 10;
	eol[2] = 0;

	char* bufline = arg_list[0]->to_string();

	if (!curbuf)
	{
		if (!IsInstr(dynScriptBuf,bufline))
			dynScriptBuf+=CStr(bufline)+eol;
	}
	else
	{
		CStr buf = ReplaceTags(bufline);

		if (!IsInstr(curbuf->data.buffer,bufline))
			curbuf->data.buffer+=buf+eol;
	}

	LeaveCriticalSection(&gCriticalSection);
	return &ok;
}

Value* NSoutNCR_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSgetFaceMtl, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string to add to output buffer >");

	char* bufline = arg_list[0]->to_string();

	if (!curbuf)
		dynScriptBuf+=CStr(bufline);
	else
	{
		CStr buf = ReplaceTags(bufline);
		curbuf->data.buffer+=buf;
	}

	LeaveCriticalSection(&gCriticalSection);
	return &ok;
}

Value* NSoutOnceNCR_cf(Value **arg_list, int count)
{
	EnterCriticalSection(&gCriticalSection);
	check_arg_count(NSgetFaceMtl, 1, count);
	type_check(arg_list[0], String, "< First parameter should be string to add to output buffer >");

	char* bufline = arg_list[0]->to_string();

	if (!curbuf)
	{
		if (!IsInstr(dynScriptBuf,bufline))
			dynScriptBuf+=CStr(bufline);
	}
	else
	{
		CStr buf = ReplaceTags(bufline);

		if (!IsInstr(curbuf->data.buffer,bufline))
			curbuf->data.buffer+=buf;
	}

	LeaveCriticalSection(&gCriticalSection);
	return &ok;
}

Value* NSConvVal_cf(Value **arg_list, int count)
{
	check_arg_count(NSConvValue, 1, count);
	type_check(arg_list[0], Float, "< First parameter should be float to convert >");
	
	// This option converts a value displayed in exponential form to a form
	// that satisfies the scripting language
	float val = arg_list[0]->to_float();

	char  buf[256];
	sprintf(buf,"%.5f",val);

	return new String(buf);
}

void GetSharedData(SharedData* sdata)
{
	HANDLE hMemSharedData = CreateFileMapping((HANDLE)0xFFFFFFFF,
		                                      NULL,
											  PAGE_READONLY,
											  0,
											  sizeof(SharedData),
											  "NExtSharedData");

	if (!hMemSharedData)
	{
		CloseHandle(hMemSharedData);
		return;
	}

	SharedData* psdata = (SharedData*)MapViewOfFile(hMemSharedData,
		                                            FILE_MAP_READ,
									                0,0,0);

	memcpy(sdata,psdata,sizeof(SharedData));

	UnmapViewOfFile(psdata);
	CloseHandle(hMemSharedData);
}

Value* NSGetLevel_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetLevel, 0, count);

	SharedData sdata;
	GetSharedData(&sdata);

	return new String(sdata.LevelName);
}

INExtMaterial* GetNExtMtl(INode* node)
{
	Mtl* mtl = node->GetMtl();

	if (!mtl)
		return NULL;

	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		return dynamic_cast<INExtMaterial*>(mtl);
	}

	if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
	{
		// We'll choose the sub material by the mtlID of the first face in the mesh
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh* mesh = &triObj->mesh;
			
			int mtlID = mesh->getFaceMtlIndex(0);

			Multi* multi = (Multi*)mtl;
			Mtl* submtl = multi->GetSubMtl(mtlID % multi->NumSubMtls());

			if (submtl && submtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				if (triObj != obj)
					triObj->DeleteThis();

				return dynamic_cast<INExtMaterial*>(submtl);
			}

			if (triObj != obj)
				triObj->DeleteThis();

			return NULL;
		}
		else
			return NULL;
	}

	return NULL;
}

Mtl* GetNExtActualMtl(INode* node)
{
	Mtl* mtl = node->GetMtl();

	if (!mtl)
		return NULL;

	if (mtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
	{
		return mtl;
	}

	if (mtl->ClassID() == vNEXT_MULTI_MATERIAL_CLASS_ID)
	{
		// We'll choose the sub material by the mtlID of the first face in the mesh
		Object* obj = node->EvalWorldState(0).obj;

		if (obj->CanConvertToType(triObjectClassID))
		{
			TriObject* triObj = (TriObject*)obj->ConvertToType(0,triObjectClassID);
			Mesh* mesh = &triObj->mesh;
			
			int mtlID = mesh->getFaceMtlIndex(0);

			Multi* multi = (Multi*)mtl;
			Mtl* submtl = multi->GetSubMtl(mtlID % multi->NumSubMtls());

			if (submtl && submtl->ClassID() == NEXT_MATERIAL_CLASS_ID)
			{
				if (triObj != obj)
					triObj->DeleteThis();

				return mtl;
			}

			if (triObj != obj)
				triObj->DeleteThis();

			return NULL;
		}
		else
			return NULL;
	}

	return NULL;
}

Value* NSGetTexName_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetTexName, 1, count);
	int texnum = 0;

	INode* node = arg_list[0]->to_node();

	if (count>1)
	{
		texnum = arg_list[1]->to_int() - 1;
	}

	if (!node)
	{
		throw RuntimeError("Must specify a node");
		return NULL;
	}

	INExtMaterial* nxMtl = GetNExtMtl(node);

	if (!nxMtl)
		return &undefined;

	// Get first pass texture name
	Texmap* tmap = nxMtl->GetTexmap(0);

	// Only bitmap textures, abort if proceedural
	if (tmap->ClassID() != Class_ID(BMTEX_CLASS_ID,0) &&
		tmap->ClassID() != NEXT_TEXTURE_CLASS_ID)
		return &undefined;
	
	BitmapTex* bmtmap = (BitmapTex*)tmap;
	CStr texName = ExtractFile(bmtmap->GetMapName());

	return new String(texName);
}

Value* NSGetFixedAlpha_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetEmitScript, 2, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be an int specifying the pass >");

	Mtl* mtl  = arg_list[0]->to_mtl();
	int  pass = arg_list[1]->to_int();

	if (pass < 0 || pass > 4)
	{
		throw RuntimeError("Pass must be a positive value from 1 - 4");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	INExtMaterial* nxMtl = dynamic_cast<INExtMaterial*>(mtl);

	int fixedVal = nxMtl->GetFixedValue(pass - 1);

	return new Integer(fixedVal);
}

Value* NSGetEmitScript_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetEmitScript, 0, count);

	SharedData sdata;
	GetSharedData(&sdata);

	return new String(sdata.EmitScript);
}

Value* NSGetMaxParticles_cf(Value **arg_list, int count)
{
	// This function retrieves the current number of particle systems using the material
	// and type that have been assigned to the current node being exported

	check_arg_count(NSGetMaxParticles, 0, count);

	SharedData sdata;
	GetSharedData(&sdata);

	return new Integer(sdata.maxParticles);
}

Value* NSReplace_cf(Value **arg_list, int count)
{
	check_arg_count(NSReplace, 3, count);
	type_check(arg_list[0], String, "< First parameter must be a string >");
	type_check(arg_list[1], String, "< First parameter must be a string >");
	type_check(arg_list[2], String, "< First parameter must be a string >");

	TSTR strSrc     = arg_list[0]->to_string();
	TSTR strSearch  = arg_list[1]->to_string();
	TSTR strReplace = arg_list[2]->to_string();

	return new String(ReplaceStr(strSrc,strSearch,strReplace));
}

Value* NSGetBlendMode_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetBlendMode, 2, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be a pass >");

	Mtl* mtl = arg_list[0]->to_mtl();
	int pass = arg_list[1]->to_int();

	if (pass < 0 || pass > 4)
	{
		throw RuntimeError("Pass must be a positive value from 1 - 4");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	INExtMaterial* nxMtl = dynamic_cast<INExtMaterial*>(mtl);

	int iBlendMode = nxMtl->GetBlendMode(pass - 1);
	TSTR strBlendMode;


	switch(iBlendMode)
	{
	case vBLEND_MODE_DIFFUSE:
		strBlendMode = "Diffuse";
		break;
	
	case vBLEND_MODE_ADD:
		strBlendMode = "Add";
		break;
	
	case vBLEND_MODE_ADD_FIXED:
		strBlendMode = "FixAdd";
		break;

	case vBLEND_MODE_SUBTRACT:
		strBlendMode = "Subtract";
		break;

	case vBLEND_MODE_SUB_FIXED:
		strBlendMode = "FixSubtract";
		break;

	case vBLEND_MODE_BLEND:
		strBlendMode = "Blend";
		break;

	case vBLEND_MODE_BLEND_FIXED:
		strBlendMode = "FixBlend";
		break;

	case vBLEND_MODE_MODULATE:
		strBlendMode = "Modulate";
		break;

	case vBLEND_MODE_MODULATE_FIXED:
		strBlendMode = "FixModulate";
		break;

	case vBLEND_MODE_BRIGHTEN:
		strBlendMode = "Brighten";
		break;

	case vBLEND_MODE_BRIGHTEN_FIXED:
		strBlendMode = "FixBrighten";
		break;

	default:
		strBlendMode = "Unknown";
		break;
	}

	return new String(strBlendMode);
}

bool GetExtents(INode* node, Point3& min, Point3& max)
{
	// Convert to a tri object and scan the mesh to determine it's XYZ extents  aml
	Object* obj=node->EvalWorldState(0).obj;

	if (obj->CanConvertToType(Class_ID(TRIOBJ_CLASS_ID,0)))
	{
		TriObject* triobj=(TriObject*)obj->ConvertToType(0,Class_ID(TRIOBJ_CLASS_ID,0));

		Mesh& mesh=triobj->GetMesh();

		if (mesh.numVerts==0)
			return false;

		// Assign min/max to start
		min=max=mesh.verts[0];

		min=min;
		max=max;

		for(int i=0;i<mesh.numVerts;i++)
		{
			Point3 pt=mesh.verts[i];

			if (pt.x<min.x)
				min.x=pt.x;

			if (pt.y<min.y)
				min.y=pt.y;

			if (pt.z<min.z)
				min.z=pt.z;

			if (pt.x>max.x)
				max.x=pt.x;

			if (pt.y>max.y)
				max.y=pt.y;

			if (pt.z>max.z)
				max.z=pt.z;
		}

		if (obj!=triobj)
			triobj->DeleteThis();

		return true;
	}

	return false;
}

Value* NSGetWidth_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetBlendMode, 1, count);
	type_check(arg_list[0], MAXNode, "< First parameter must be a node >");
	
	INode* node = arg_list[0]->to_node();

	if (!node)
	{
		throw RuntimeError("Node is invalid!");
		return &undefined;
	}

	Point3 min,max;
	GetExtents(node,min,max);

	int width = max.x - min.x;

	if (width == 0)
		width = max.z - min.z;

	return new Float(width);
}

Value* NSGetHeight_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetBlendMode, 1, count);
	type_check(arg_list[0], MAXNode, "< First parameter must be a node >");
	
	INode* node = arg_list[0]->to_node();

	if (!node)
	{
		throw RuntimeError("Node is invalid!");
		return &undefined;
	}

	Point3 min,max;
	GetExtents(node,min,max);

	int height = max.y - min.y;

	if (height == 0)
		height = max.z - min.z;

	return new Float(height);
}

Value* NSGetNExtMtl_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetNExtMtl, 1, count);
	type_check(arg_list[0], MAXNode, "< First parameter must be a node >");

	INode* node = arg_list[0]->to_node();

	if (!node)
	{
		throw RuntimeError("Node is invalid!");
		return &undefined;
	}

	Mtl* mtl = GetNExtActualMtl(node);

	if (!mtl)
		return &undefined;

	return MAXClass::make_wrapper_for(mtl);
}

Value* NSGetUsePS2MipsNGC_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetUsePS2MipsNGC, 2, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be an integer >");

	Mtl* mtl  = arg_list[0]->to_mtl();
	int  pass = arg_list[1]->to_int();

	if (pass < 1 || pass > 4)
	{
		throw RuntimeError("Pass must be within the range of 1 - 4");
		return &undefined;
	}

	if (!mtl)
	{
		throw RuntimeError("No material defined");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	// Retrieve the actual value
	INExtMaterial* nextMtl = dynamic_cast<INExtMaterial*>(mtl);

	Texmap* tmap = nextMtl->GetTexmap(	pass - 1 );

	if (tmap->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		// No exceptions because we don't want to terminate the script
		// in this particular case
		return &undefined;
	}

	IParamBlock2* pblock = tmap->GetParamBlock(0);

	PBBitmap* pb_bm;

	pblock->GetValue( bmtex_bitmap_ngc, 0, pb_bm, FOREVER );

	CStr name = pb_bm->bi.Name();

	if (name == CStr("Use PS2") &&
		pb_bm->bi.Width() == 0 &&
		pb_bm->bi.Height() == 0)
	{
		return &true_value;
	}

	return &false_value;
}

Value* NSGetUsePS2MipsXBOX_cf(Value **arg_list, int count)
{
	check_arg_count(NSGetUsePS2MipsXBOX, 2, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be an integer >");

	Mtl* mtl  = arg_list[0]->to_mtl();
	int  pass = arg_list[1]->to_int();

	if (pass < 1 || pass > 4)
	{
		throw RuntimeError("Pass must be within the range of 1 - 4");
		return &undefined;
	}

	if (!mtl)
	{
		throw RuntimeError("No material defined");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	// Retrieve the actual value
	INExtMaterial* nextMtl = dynamic_cast<INExtMaterial*>(mtl);

	Texmap* tmap = nextMtl->GetTexmap(	pass - 1 );

	if (tmap->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		// No exceptions because we don't want to terminate the script
		// in this particular case
		return &undefined;
	}

	IParamBlock2* pblock = tmap->GetParamBlock(0);

	PBBitmap* pb_bm;

	pblock->GetValue( bmtex_bitmap_xbox, 0, pb_bm, FOREVER );

	CStr name = pb_bm->bi.Name();

	if (name == CStr("Use PS2") &&
		pb_bm->bi.Width() == 0 &&
		pb_bm->bi.Height() == 0)
	{
		return &true_value;
	}

	return &false_value;
}

Value* NSPutUsePS2MipsNGC_cf(Value **arg_list, int count)
{
	check_arg_count(NSPutUsePS2MipsNGC, 3, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be an integer >");
	type_check(arg_list[2], Boolean, "< Third parameter must be true/false >");

	Mtl* mtl     = arg_list[0]->to_mtl();
	int  pass    = arg_list[1]->to_int();
	int  bEnable = arg_list[2]->to_bool();

	if (pass < 1 || pass > 4)
	{
		throw RuntimeError("Pass must be within the range of 1 - 4");
		return &undefined;
	}

	if (!mtl)
	{
		throw RuntimeError("No material defined");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	// Retrieve the actual value
	INExtMaterial* nextMtl = dynamic_cast<INExtMaterial*>(mtl);

	Texmap* tmap = nextMtl->GetTexmap(	pass - 1 );

	if (tmap->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		// No exceptions because we don't want to terminate the script
		// in this particular case
		return &undefined;
	}

	INExtTexture* nextTex = dynamic_cast<INExtTexture*>(tmap);

	IParamBlock2* pblock = tmap->GetParamBlock(0);

	if (pblock)
	{
		pblock->SetValue( bmtex_bitmap_use_ps2_mips_ngc, 0, bEnable );

		if (bEnable)
		{
			BitmapInfo info;
			
			info.SetName( "Use PS2" );
			info.SetWidth( 0 );
			info.SetHeight( 0 );

			PBBitmap pb_bm( info );

			pblock->SetValue( bmtex_bitmap_ngc, 0, &pb_bm );
		}
	}

	return &ok;
}

Value* NSPutUsePS2MipsXBOX_cf(Value **arg_list, int count)
{
	check_arg_count(NSPutUsePS2MipsXBOX, 3, count);
	type_check(arg_list[0], MAXMaterial, "< First parameter must be a material >");
	type_check(arg_list[1], Integer, "< Second parameter must be an integer >");
	type_check(arg_list[2], Boolean, "< Third parameter must be true/false >");

	Mtl* mtl     = arg_list[0]->to_mtl();
	int  pass    = arg_list[1]->to_int();
	int  bEnable = arg_list[2]->to_bool();

	if (pass < 1 || pass > 4)
	{
		throw RuntimeError("Pass must be within the range of 1 - 4");
		return &undefined;
	}

	if (!mtl)
	{
		throw RuntimeError("No material defined");
		return &undefined;
	}

	if( mtl->ClassID() != NEXT_MATERIAL_CLASS_ID )
	{	
		throw RuntimeError("This material is not a NExt material");
		return &undefined;
	}

	// Retrieve the actual value
	INExtMaterial* nextMtl = dynamic_cast<INExtMaterial*>(mtl);

	Texmap* tmap = nextMtl->GetTexmap(	pass - 1 );

	if (tmap->ClassID() != NEXT_TEXTURE_CLASS_ID )
	{
		// No exceptions because we don't want to terminate the script
		// in this particular case
		return &undefined;
	}

	INExtTexture* nextTex = dynamic_cast<INExtTexture*>(tmap);

	IParamBlock2* pblock = tmap->GetParamBlock(0);

	if (pblock)
	{
		pblock->SetValue( bmtex_bitmap_use_ps2_mips_xbox, 0, bEnable );

		if (bEnable)
		{
			BitmapInfo info;
			
			info.SetName( "Use PS2" );
			info.SetWidth( 0 );
			info.SetHeight( 0 );

			PBBitmap pb_bm( info );

			pblock->SetValue( bmtex_bitmap_xbox, 0, &pb_bm );
		}
	}

	return &ok;
}
