/*
	RichText.h
	Support class for RichText Edit Controls

	Adam Lippmann
	12-13-01
*/

#ifndef __RICHTEXT__
#define __RICHTEXT__

#include <windows.h>
#include <richedit.h>
#include <stdio.h>
#include "max.h"
#include "strclass.h"

#define RICHTEXT_SIZELIMIT 131072       //  128K of memory

//////  Rich Text Language control codes
#define  RTF_HDR           "{\\rtf "
#define  RTF_COLOR_HEADER  "{\\rtf {\\fonttbl{\\f0\\fmodern\\fprq1\\fcharset0 Courier New;}}{\\colortbl\\red0\\green0\\blue0;\\red255\\green0\\blue0;\\red0\\green150\\blue0;\\red0\\green0\\blue255;}\\f0\\fs18 "
#define  RTF_BLACK         "\\cf0 "
#define  RTF_RED           "\\cf1 "
#define  RTF_GREEN         "\\cf2 "
#define  RTF_BLUE          "\\cf3 "
#define  RTF_BOLD          "\\b "
#define  RTF_BOLDOFF       "\\b0 "
#define  RTF_ITALIC        "\\i "
#define  RTF_ITALICOFF     "\\i0 "
#define  RTF_UNDERLINE     "\\ul "
#define  RTF_UNDERLINEOFF  "\\ul0 "
#define  RTF_ENDLINE       "\\par "
#define  RTF_DEFAULT       "\\cf0 \\b0 \\i0 \\ul0 "
#define  RTF_END           "}"

#define  SRTF_HDR           CStr(RTF_HDR)
#define  SRTF_COLOR_HEADER  CStr(RTF_COLOR_HEADER)
#define  SRTF_BLACK         CStr(RTF_BLACK)
#define  SRTF_RED           CStr(RTF_RED)
#define  SRTF_GREEN         CStr(RTF_GREEN)
#define  SRTF_BLUE          CStr(RTF_BLUE)
#define  SRTF_BOLD          CStr(RTF_BOLD)
#define  SRTF_BOLDOFF       CStr(RTF_BOLDOFF)
#define  SRTF_ITALIC        CStr(RTF_ITALIC)
#define  SRTF_ITALICOFF     CStr(RTF_ITALICOFF)
#define  SRTF_UNDERLINE     CStr(RTF_UNDERLINE)
#define  SRTF_UNDERLINEOFF  CStr(RTF_UNDERLINEOFF)
#define  SRTF_ENDLINE       CStr(RTF_ENDLINE)
#define  SRTF_DEFAULT       CStr(RTF_DEFAULT)
#define  SRTF_END           CStr(RTF_END)
////////////////////////////////////////////////////

#define  RTS_WHOLEWORD		FR_WHOLEWORD
#define  RTS_MATCHCASE		FR_MATCHCASE

enum SetLineMode
{
	SETLINE_RTF,
	SETLINE_TXT,
};

class RichText
{
	struct StreamData
	{
		char*     pBuf;			// The text buffer to stream into the control
		int       bufsize;		// The size of the text buffer to stream into the control
		int       pos;			// The position that the stream function is currently at loading the buffer
		RichText* pRichText;	// Pointer to this RichText class
	};

	HINSTANCE hInstance;		// hInstance of the application
	HWND      hwnd;				// Handle to the RichText edit control
	HWND      hwndSearch;		// Handle to the search dialog
	HWND      hwndReplace;		// Handle to the search and replace dialog
	HLOCAL    hMem;				// Memory allocated for the RichText control's text buffer

	// Notification callbacks
	void(*fpNotifyComplete)(void*);		// Notification function called when a search operation completes
	void *pNotifyCompleteData;			// Pointer to notification data

	void(*fpNotifyCancelled)(void*);	// Notification function called when a search is cancelled
	void *pNotifyCancelledData;			// Pointer to notification data

	void(*fpNotifyReplaceAll)(void*);	// Notificatioin function called when user presses ReplaceAll
	void *pNotifyReplaceAllData;		// Pointer to notification data

	//////////////////

	static DWORD CALLBACK EditStreamCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf);
	static DWORD CALLBACK LoadStreamCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf);
	static DWORD CALLBACK SaveStreamCB(DWORD dwCookie,LPBYTE buf,LONG numBytes,LONG* bytesTransf);
	static DWORD CALLBACK SaveStreamTextCB(DWORD dwCookie,LPBYTE bug,LONG numBytes,LONG* bytesTransf);

	static BOOL CALLBACK RedirectSearchDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
	BOOL SearchDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	static BOOL CALLBACK RedirectReplaceDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);
	BOOL ReplaceDlgProc(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

public:
	RichText(HINSTANCE hInstance,HWND hwndRichText);
	~RichText();

	BOOL ProcMessage(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	int  GetLineNum();								// Retrieves the current line number the cursor is on
	
	// Sets the specified line of text (-1 for the cursor's line)
	void SetLine(char* newLine,int lineNum=-1,SetLineMode mode=SETLINE_RTF);
	
	CStr GetLine(int lineNum=-1);					// Gets the specified line of text (-1 for the cursor's line)
	CStr GetWord();									// Returns a string containing the word the cursor is over

	int   GetCurPos();								// Retrieves the current index position of the cursor
	POINT GetScreenPos(int index=-1);				// Retrieves the current screen position that the cursor is located at
	void  GoToEnd();								// Moves the cursor position to the end of the current line

	bool  HasText();								// Returns true if there's text in the control
	void  Clear();									// Clears the content of the editor

	inline void InsertText(char* text)
		{ SendMessage(hwnd,EM_REPLACESEL,(WPARAM)TRUE,(LPARAM)text); }

	CStr GetText();									// Gets the full contents of the richedit control as standard text
	void GetText(char* str,int count);
	void SetText(CStr str);

	int  Search(CStr strSearch,DWORD flags=0,BOOL bSearchRange=FALSE);
	void Replace(CStr strReplace);
	void Highlight(int start,int end);
	void HighlightLine(int index);
	void HighlightLineNum(int linenum);

	void ClipboardCut();							// Standard clipboard functionality (cut,copy,paste)
	void ClipboardCopy();
	void ClipboardPaste();

	// Pops up a search dialog to be used with the RichText control
	void SearchDlg(void(*NotifyComplete)(void*)=NULL,void* pDataComplete=NULL,
		           void(*NotifyCancelled)(void*)=NULL,void* pDataCancelled=NULL);

		void FindNext();
		void CloseSearch();

	// Pops up a search & replace dialog to be used with the RichText control
	void ReplaceDlg(void(*NotifyComplete)(void*)=NULL,void* pDataComplete=NULL,
		            void(*NotifyCancelled)(void*)=NULL,void* pDataCancelled=NULL,
					void(*NotifyReplaceAll)(void*)=NULL,void* pDataReplaceAll=NULL);
	
		bool RFindNext(bool bWarn=TRUE);
		bool Replace(bool bWarn=TRUE);
		void ReplaceAll();
		void CloseReplace();

	// Neither of these functions will actually load or save the file
	void LoadDlg(char* Title,char* Filter,char* DefaultExt,CStr& Filename);
	int  SaveDlg(char* Title,char* Filter,char* DefaultExt,CStr& Filename);

	// This function will perform a disk save of the control's contents
	bool SaveDlg(char* Title,char* DefaultExt);
	bool SaveTXT(char* Filename);
	bool SaveTXT(FILE* fp);
	bool SaveRTF(char* Filename);

	inline void Disable() { EnableWindow(hwnd,FALSE); }
	inline void Enable()  { EnableWindow(hwnd,TRUE);  }

	void DisableText();
	void EnableText();

	inline HWND GetHWND() { return hwnd; }
};

#endif
