/*
	RTFEditor.h
	This class provides an RTF text editor with syntax highlighting
	and parameter completion features
*/

#ifndef __RTFEDITOR__
#define __RTFEDITOR__

#include "MapFileParser.h"
#include "Max.h"

class RichText;
class PropList;

class RTFEditor: public MapFileParser
{
	BOOL            bAllocRichText;					// True if the RichText control was allocated and needs freed
	BOOL            bAllocPropList;					// True if the property list was allocated and needs freed
	BOOL            bAllowAccelerators;				// True if MAX accelerator processing is currently allowed

	void Init();

protected:	
	Lst::StringHashTable<KeywordDesc>   keywordDB;	// Keyword information database (For syntax highlighting)
													// These include the internal script commands (if endif, etc.)

	RichText*       pRichText;						// The RichText Edit field in the center of the window

	BOOL            bSyntaxHighlighting;			// True if syntax highlighting is enabled
	BOOL            bScriptChanged;					// True if the script has been changed
	BOOL            bChangeMade;					// True if change to script, class, or object type (or properties)
	BOOL            bLockParamPopup;				// True if the next EN_CHANGE message should not close the parameter popup window
	BOOL            bLockParamComplete;				// True if parameter auto-completion isn't allowed to update the editor window

	PropList*       pPopupParams;					// Popup script parameter list

	CStr            FuncName;						// The last function that a parameter popup was created for

	HMENU		    hMenu;							// The menu containing the below popup menu
	HMENU           hPopupMenu;						// The popup menu when you right click in the RichText edit box

	void(*fpRTFChange)(RTFEditor*,void*);			// Callback executed whenever the contents of the editor change
	void*           pChangeData;					// Retained data for the callback

	static void PopupParamsApplyCB(PropList* pPropList,void* pData);
	static void PopupParamsChangeCB(PropList* pPropList,void* pData);
	static void PopupParamsExtListCB(CStr name,CStr value,void* pData);

public:
	RTFEditor(HINSTANCE hInstance, HWND hwnd);
	RTFEditor(RichText* pRichText);
	RTFEditor();
	~RTFEditor();

	void SetChangeCB(void(*fpCallback)(RTFEditor*,void*),void* pData);

	bool AddKeyword(CStr Name,COLORREF color);
	void DeleteKeyword(CStr Name);

	void SyntaxCheck(int lineNum=-1);							// Performs syntax checking on given line of text in RTF box
	CStr ParseEscapes(CStr line);								// Adds necessary RTF escape sequences for block commands \{ \}
	CStr SyntaxRTF(CStr line);									// Converts standard text string to it's syntax highlighted
																// RTF format equivalent

	CStr BuildTextRTF(CStr text);								// Returns string of syntax parsed RTF
	void SetText(CStr text);									// Loads straight text into the RichEdit control as syntax
																// parsed RTF

	CStr GetText();

	CStr GetWord(char* lineBuf,int* xPos,int* wrdStart=0);		// Returns the word on the given line at the specified position

	void ProcParamList();										// Opens a parameter description popup window for the supplied
																// type (currently will only do scripts)

	LRESULT ProcMessage(HWND hwnd,UINT msg,WPARAM wParam,LPARAM lParam);

	void SpawnWinHelp();										// Spawn windows help system for the word the cursor is on

	CStr GetParamCompleteText(PropList* pPropList);				// Gets parameter completion text string

	void InitScriptDB();

	CStr LoadTextStr(char* Filename);		// Loads text from file into a string

	bool LoadText(char* Filename);			// Loads text from file into RichEdit control as syntax
											// highlighted RTF

	void GoToScriptDefinition(CStr keyword);

	void AllowChanges(BOOL mode);
};

#endif
